/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.runtime.checkpoint.CheckpointException;
import org.apache.flink.runtime.checkpoint.CheckpointFailureReason;
import org.apache.flink.runtime.checkpoint.CheckpointPlan;
import org.apache.flink.runtime.checkpoint.CheckpointPlanCalculator;
import org.apache.flink.runtime.checkpoint.CheckpointPlanCalculatorContext;
import org.apache.flink.runtime.checkpoint.DefaultCheckpointPlan;
import org.apache.flink.runtime.execution.ExecutionState;
import org.apache.flink.runtime.executiongraph.Execution;
import org.apache.flink.runtime.executiongraph.ExecutionJobVertex;
import org.apache.flink.runtime.executiongraph.ExecutionVertex;
import org.apache.flink.runtime.executiongraph.InternalExecutionGraphAccessor;
import org.apache.flink.runtime.jobgraph.DistributionPattern;
import org.apache.flink.runtime.jobgraph.IntermediateResultPartitionID;
import org.apache.flink.runtime.jobgraph.JobEdge;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.util.Preconditions;

public class DefaultCheckpointPlanCalculator
implements CheckpointPlanCalculator {
    private final JobID jobId;
    private final CheckpointPlanCalculatorContext context;
    private final List<ExecutionJobVertex> jobVerticesInTopologyOrder = new ArrayList<ExecutionJobVertex>();
    private final List<ExecutionVertex> allTasks = new ArrayList<ExecutionVertex>();
    private final List<ExecutionVertex> sourceTasks = new ArrayList<ExecutionVertex>();
    private final boolean allowCheckpointsAfterTasksFinished;

    public DefaultCheckpointPlanCalculator(JobID jobId, CheckpointPlanCalculatorContext context, Iterable<ExecutionJobVertex> jobVerticesInTopologyOrderIterable, boolean allowCheckpointsAfterTasksFinished) {
        this.jobId = (JobID)Preconditions.checkNotNull((Object)jobId);
        this.context = (CheckpointPlanCalculatorContext)Preconditions.checkNotNull((Object)context);
        this.allowCheckpointsAfterTasksFinished = allowCheckpointsAfterTasksFinished;
        Preconditions.checkNotNull(jobVerticesInTopologyOrderIterable);
        jobVerticesInTopologyOrderIterable.forEach(jobVertex -> {
            this.jobVerticesInTopologyOrder.add((ExecutionJobVertex)jobVertex);
            this.allTasks.addAll(Arrays.asList(jobVertex.getTaskVertices()));
            if (jobVertex.getJobVertex().isInputVertex()) {
                this.sourceTasks.addAll(Arrays.asList(jobVertex.getTaskVertices()));
            }
        });
    }

    @Override
    public CompletableFuture<CheckpointPlan> calculateCheckpointPlan() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (this.context.hasFinishedTasks() && !this.allowCheckpointsAfterTasksFinished) {
                    throw new CheckpointException("Some tasks of the job have already finished and checkpointing with finished tasks is not enabled.", CheckpointFailureReason.NOT_ALL_REQUIRED_TASKS_RUNNING);
                }
                this.checkAllTasksInitiated();
                CheckpointPlan result = this.context.hasFinishedTasks() ? this.calculateAfterTasksFinished() : this.calculateWithAllTasksRunning();
                this.checkTasksStarted(result.getTasksToWaitFor());
                return result;
            }
            catch (Throwable throwable) {
                throw new CompletionException(throwable);
            }
        }, (Executor)this.context.getMainExecutor());
    }

    private void checkAllTasksInitiated() throws CheckpointException {
        for (ExecutionVertex task : this.allTasks) {
            if (task.getCurrentExecutionAttempt() != null) continue;
            throw new CheckpointException(String.format("task %s of job %s is not being executed at the moment. Aborting checkpoint.", task.getTaskNameWithSubtaskIndex(), this.jobId), CheckpointFailureReason.NOT_ALL_REQUIRED_TASKS_RUNNING);
        }
    }

    private void checkTasksStarted(List<Execution> toTrigger) throws CheckpointException {
        for (Execution execution : toTrigger) {
            if (execution.getState() == ExecutionState.RUNNING) continue;
            throw new CheckpointException(String.format("Checkpoint triggering task %s of job %s is not being executed at the moment. Aborting checkpoint.", execution.getVertex().getTaskNameWithSubtaskIndex(), this.jobId), CheckpointFailureReason.NOT_ALL_REQUIRED_TASKS_RUNNING);
        }
    }

    private CheckpointPlan calculateWithAllTasksRunning() {
        List executionsToTrigger = this.sourceTasks.stream().map(ExecutionVertex::getCurrentExecutionAttempt).collect(Collectors.toList());
        List<Execution> tasksToWaitFor = this.createTaskToWaitFor(this.allTasks);
        return new DefaultCheckpointPlan(Collections.unmodifiableList(executionsToTrigger), Collections.unmodifiableList(tasksToWaitFor), Collections.unmodifiableList(this.allTasks), Collections.emptyList(), Collections.emptyList(), this.allowCheckpointsAfterTasksFinished);
    }

    private CheckpointPlan calculateAfterTasksFinished() {
        Map<JobVertexID, BitSet> taskRunningStatusByVertex = this.collectTaskRunningStatus();
        ArrayList<Execution> tasksToTrigger = new ArrayList<Execution>();
        ArrayList<Execution> tasksToWaitFor = new ArrayList<Execution>();
        ArrayList<ExecutionVertex> tasksToCommitTo = new ArrayList<ExecutionVertex>();
        ArrayList<Execution> finishedTasks = new ArrayList<Execution>();
        ArrayList<ExecutionJobVertex> fullyFinishedJobVertex = new ArrayList<ExecutionJobVertex>();
        for (ExecutionJobVertex jobVertex : this.jobVerticesInTopologyOrder) {
            BitSet taskRunningStatus = taskRunningStatusByVertex.get(jobVertex.getJobVertexId());
            if (taskRunningStatus.cardinality() == 0) {
                fullyFinishedJobVertex.add(jobVertex);
                for (ExecutionVertex task : jobVertex.getTaskVertices()) {
                    finishedTasks.add(task.getCurrentExecutionAttempt());
                }
                continue;
            }
            List<JobEdge> prevJobEdges = jobVertex.getJobVertex().getInputs();
            boolean someTasksMustBeTriggered = this.someTasksMustBeTriggered(taskRunningStatusByVertex, prevJobEdges);
            for (int i = 0; i < jobVertex.getTaskVertices().length; ++i) {
                ExecutionVertex task;
                task = jobVertex.getTaskVertices()[i];
                if (taskRunningStatus.get(task.getParallelSubtaskIndex())) {
                    boolean hasRunningPrecedentTasks;
                    tasksToWaitFor.add(task.getCurrentExecutionAttempt());
                    tasksToCommitTo.add(task);
                    if (!someTasksMustBeTriggered || (hasRunningPrecedentTasks = this.hasRunningPrecedentTasks(task, prevJobEdges, taskRunningStatusByVertex))) continue;
                    tasksToTrigger.add(task.getCurrentExecutionAttempt());
                    continue;
                }
                finishedTasks.add(task.getCurrentExecutionAttempt());
            }
        }
        return new DefaultCheckpointPlan(Collections.unmodifiableList(tasksToTrigger), Collections.unmodifiableList(tasksToWaitFor), Collections.unmodifiableList(tasksToCommitTo), Collections.unmodifiableList(finishedTasks), Collections.unmodifiableList(fullyFinishedJobVertex), this.allowCheckpointsAfterTasksFinished);
    }

    private boolean someTasksMustBeTriggered(Map<JobVertexID, BitSet> runningTasksByVertex, List<JobEdge> prevJobEdges) {
        for (JobEdge jobEdge : prevJobEdges) {
            BitSet upstreamRunningStatus;
            DistributionPattern distributionPattern = jobEdge.getDistributionPattern();
            if (!this.hasActiveUpstreamVertex(distributionPattern, upstreamRunningStatus = runningTasksByVertex.get(jobEdge.getSource().getProducer().getID()))) continue;
            return false;
        }
        return true;
    }

    private boolean hasActiveUpstreamVertex(DistributionPattern distribution, BitSet upstreamRunningTasks) {
        return distribution == DistributionPattern.ALL_TO_ALL && upstreamRunningTasks.cardinality() > 0 || distribution == DistributionPattern.POINTWISE && upstreamRunningTasks.cardinality() == upstreamRunningTasks.size();
    }

    private boolean hasRunningPrecedentTasks(ExecutionVertex vertex, List<JobEdge> prevJobEdges, Map<JobVertexID, BitSet> taskRunningStatusByVertex) {
        InternalExecutionGraphAccessor executionGraphAccessor = vertex.getExecutionGraphAccessor();
        for (int i = 0; i < prevJobEdges.size(); ++i) {
            if (prevJobEdges.get(i).getDistributionPattern() != DistributionPattern.POINTWISE) continue;
            for (IntermediateResultPartitionID consumedPartitionId : vertex.getConsumedPartitionGroup(i)) {
                ExecutionVertex precedentTask = executionGraphAccessor.getResultPartitionOrThrow(consumedPartitionId).getProducer();
                BitSet precedentVertexRunningStatus = taskRunningStatusByVertex.get(precedentTask.getJobvertexId());
                if (!precedentVertexRunningStatus.get(precedentTask.getParallelSubtaskIndex())) continue;
                return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    Map<JobVertexID, BitSet> collectTaskRunningStatus() {
        HashMap<JobVertexID, BitSet> runningStatusByVertex = new HashMap<JobVertexID, BitSet>();
        for (ExecutionJobVertex vertex : this.jobVerticesInTopologyOrder) {
            BitSet runningTasks = new BitSet(vertex.getTaskVertices().length);
            for (int i = 0; i < vertex.getTaskVertices().length; ++i) {
                if (vertex.getTaskVertices()[i].getCurrentExecutionAttempt().isFinished()) continue;
                runningTasks.set(i);
            }
            runningStatusByVertex.put(vertex.getJobVertexId(), runningTasks);
        }
        return runningStatusByVertex;
    }

    private List<Execution> createTaskToWaitFor(List<ExecutionVertex> tasks) {
        ArrayList<Execution> tasksToAck = new ArrayList<Execution>(tasks.size());
        for (ExecutionVertex task : tasks) {
            tasksToAck.add(task.getCurrentExecutionAttempt());
        }
        return tasksToAck;
    }
}

