/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.storm.hbase.security;

import static org.apache.storm.Config.TOPOLOGY_AUTO_CREDENTIALS;

import java.io.IOException;
import java.net.InetAddress;
import java.util.List;
import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.security.UserProvider;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.storm.security.auth.kerberos.AutoTGT;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class provides util methods for storm-hbase connector communicating
 * with secured HBase.
 */
public class HBaseSecurityUtil {
    private static final Logger LOG = LoggerFactory.getLogger(HBaseSecurityUtil.class);

    public static final String STORM_KEYTAB_FILE_KEY = "storm.keytab.file";
    public static final String STORM_USER_NAME_KEY = "storm.kerberos.principal";
    public static final String HBASE_CREDENTIALS_CONFIG_KEYS = "hbaseCredentialsConfigKeys";

    public static final String HBASE_CREDENTIALS = "HBASE_CREDENTIALS";
    public static final String HBASE_KEYTAB_FILE_KEY = "hbase.keytab.file";
    public static final String HBASE_PRINCIPAL_KEY = "hbase.kerberos.principal";

    private static volatile UserProvider legacyProvider = null;

    private HBaseSecurityUtil() {
    }

    public static UserProvider login(Map<String, Object> conf, Configuration hbaseConfig) throws IOException {
        //Allowing keytab based login for backward compatibility.
        if (UserGroupInformation.isSecurityEnabled()) {
            List<String> autoCredentials = (List) conf.get(TOPOLOGY_AUTO_CREDENTIALS);
            if ((autoCredentials == null)
                    || (!autoCredentials.contains(AutoHBase.class.getName()) && !autoCredentials.contains(AutoTGT.class.getName()))) {
                LOG.info("Logging in using keytab as neither AutoHBase or AutoTGT is specified for " + TOPOLOGY_AUTO_CREDENTIALS);
                //insure that if keytab is used only one login per process executed
                if (legacyProvider == null) {
                    synchronized (HBaseSecurityUtil.class) {
                        if (legacyProvider == null) {
                            legacyProvider = UserProvider.instantiate(hbaseConfig);
                            String keytab = (String) conf.get(STORM_KEYTAB_FILE_KEY);
                            if (keytab != null) {
                                hbaseConfig.set(STORM_KEYTAB_FILE_KEY, keytab);
                            }
                            String userName = (String) conf.get(STORM_USER_NAME_KEY);
                            if (userName != null) {
                                hbaseConfig.set(STORM_USER_NAME_KEY, userName);
                            }
                            legacyProvider.login(STORM_KEYTAB_FILE_KEY, STORM_USER_NAME_KEY,
                                    InetAddress.getLocalHost().getCanonicalHostName());
                        }
                    }
                }
            }
            return legacyProvider;
        } else {
            return null;
        }
    }
}
