/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ranger.biz;

import org.apache.ranger.common.RESTErrorUtil;
import org.apache.ranger.common.db.RangerTransactionSynchronizationAdapter;
import org.apache.ranger.db.RangerDaoManager;
import org.apache.ranger.db.XXGroupDao;
import org.apache.ranger.db.XXRoleDao;
import org.apache.ranger.db.XXRoleRefGroupDao;
import org.apache.ranger.db.XXRoleRefRoleDao;
import org.apache.ranger.db.XXRoleRefUserDao;
import org.apache.ranger.db.XXUserDao;
import org.apache.ranger.entity.XXGroup;
import org.apache.ranger.entity.XXRole;
import org.apache.ranger.entity.XXRoleRefGroup;
import org.apache.ranger.entity.XXRoleRefRole;
import org.apache.ranger.entity.XXRoleRefUser;
import org.apache.ranger.entity.XXUser;
import org.apache.ranger.plugin.model.RangerRole;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.junit.jupiter.MockitoExtension;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * @generated by Cursor
 * @description <Unit Test for TestRoleRefUpdater class>
 */
@ExtendWith(MockitoExtension.class)
@TestMethodOrder(MethodOrderer.MethodName.class)
public class TestRoleRefUpdater {
    private static void setField(Object target, Class<?> declaring, String fieldName, Object value) throws Exception {
        Field f = declaring.getDeclaredField(fieldName);
        f.setAccessible(true);
        f.set(target, value);
    }

    private RangerRole buildRole(Long id, List<String> users, List<String> groups, List<String> roles) {
        RangerRole rr = new RangerRole("role", null, null, null, null);
        rr.setId(id);
        for (String u : users) {
            rr.getUsers().add(new RangerRole.RoleMember(u, false));
        }
        for (String g : groups) {
            rr.getGroups().add(new RangerRole.RoleMember(g, false));
        }
        for (String r : roles) {
            rr.getRoles().add(new RangerRole.RoleMember(r, false));
        }
        return rr;
    }

    @Test
    public void test01_cleanupRefTables_handlesNullAndDeletesIds() throws Exception {
        RoleRefUpdater updater = new RoleRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        XXRoleRefUserDao urd = mock(XXRoleRefUserDao.class);
        XXRoleRefGroupDao grd = mock(XXRoleRefGroupDao.class);
        XXRoleRefRoleDao rrd = mock(XXRoleRefRoleDao.class);
        when(dao.getXXRoleRefUser()).thenReturn(urd);
        when(dao.getXXRoleRefGroup()).thenReturn(grd);
        when(dao.getXXRoleRefRole()).thenReturn(rrd);
        setField(updater, RoleRefUpdater.class, "daoMgr", dao);

        // null id -> false
        RangerRole roleNull = new RangerRole("r", null, null, null, null);
        roleNull.setId(null);
        Assertions.assertFalse(updater.cleanupRefTables(roleNull));

        // with id -> deletes
        RangerRole role = new RangerRole("r", null, null, null, null);
        role.setId(5L);
        when(urd.findIdsByRoleId(5L)).thenReturn(Arrays.asList(1L, 2L));
        when(grd.findIdsByRoleId(5L)).thenReturn(Collections.singletonList(7L));
        when(rrd.findIdsByRoleId(5L)).thenReturn(Collections.singletonList(9L));
        Assertions.assertTrue(updater.cleanupRefTables(role));
        verify(urd, times(1)).deleteRoleRefUserByIds(any());
        verify(grd, times(1)).deleteRoleRefGroupByIds(any());
        verify(rrd, times(1)).deleteRoleRefRoleByIds(any());
    }

    @Test
    public void test02_createNewRoleMapping_createsAssociationsForExistingPrincipals() throws Exception {
        RoleRefUpdater updater = new RoleRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        XXUserDao xUserDao = mock(XXUserDao.class);
        XXGroupDao xGroupDao = mock(XXGroupDao.class);
        XXRoleDao xRoleDao = mock(XXRoleDao.class);
        XXRoleRefUserDao urd = mock(XXRoleRefUserDao.class);
        XXRoleRefGroupDao grd = mock(XXRoleRefGroupDao.class);
        XXRoleRefRoleDao rrd = mock(XXRoleRefRoleDao.class);
        when(dao.getXXUser()).thenReturn(xUserDao);
        when(dao.getXXGroup()).thenReturn(xGroupDao);
        when(dao.getXXRole()).thenReturn(xRoleDao);
        when(dao.getXXRoleRefUser()).thenReturn(urd);
        when(dao.getXXRoleRefGroup()).thenReturn(grd);
        when(dao.getXXRoleRefRole()).thenReturn(rrd);
        setField(updater, RoleRefUpdater.class, "daoMgr", dao);
        setField(updater, RoleRefUpdater.class, "rangerTransactionSynchronizationAdapter",
                mock(RangerTransactionSynchronizationAdapter.class));
        setField(updater, RoleRefUpdater.class, "xaBizUtil", mock(RangerBizUtil.class));

        XXUser xu = new XXUser();
        xu.setId(11L);
        XXGroup xg = new XXGroup();
        xg.setId(22L);
        XXRole xr = new XXRole();
        xr.setId(33L);
        when(xUserDao.findByUserName("u1")).thenReturn(xu);
        when(xGroupDao.findByGroupName("g1")).thenReturn(xg);
        when(xRoleDao.findByRoleName("r1")).thenReturn(xr);

        RangerRole role = buildRole(5L, Collections.singletonList("u1"), Collections.singletonList("g1"),
                Collections.singletonList("r1"));
        updater.createNewRoleMappingForRefTable(role, true);
        verify(urd, times(1)).create(any(XXRoleRefUser.class));
        verify(grd, times(1)).create(any(XXRoleRefGroup.class));
        verify(rrd, times(1)).create(any(XXRoleRefRole.class));
    }

    @Test
    public void test03_createNewRoleMapping_missingPrincipalThrowsUnlessCreateAllowed() throws Exception {
        RoleRefUpdater updater = new RoleRefUpdater();
        RangerDaoManager dao = mock(RangerDaoManager.class);
        XXUserDao xUserDao = mock(XXUserDao.class);
        RESTErrorUtil rest = mock(RESTErrorUtil.class);
        RangerTransactionSynchronizationAdapter adapter = mock(RangerTransactionSynchronizationAdapter.class);
        RangerBizUtil biz = mock(RangerBizUtil.class);
        XXRoleRefUserDao urd = mock(XXRoleRefUserDao.class);
        XXRoleRefGroupDao grd = mock(XXRoleRefGroupDao.class);
        XXRoleRefRoleDao rrd = mock(XXRoleRefRoleDao.class);
        setField(updater, RoleRefUpdater.class, "daoMgr", dao);
        setField(updater, RoleRefUpdater.class, "restErrorUtil", rest);
        setField(updater, RoleRefUpdater.class, "rangerTransactionSynchronizationAdapter", adapter);
        setField(updater, RoleRefUpdater.class, "xaBizUtil", biz);
        when(dao.getXXUser()).thenReturn(xUserDao);
        when(dao.getXXRoleRefUser()).thenReturn(urd);
        when(dao.getXXRoleRefGroup()).thenReturn(grd);
        when(dao.getXXRoleRefRole()).thenReturn(rrd);
        when(urd.findIdsByRoleId(7L)).thenReturn(Collections.emptyList());
        when(grd.findIdsByRoleId(7L)).thenReturn(Collections.emptyList());
        when(rrd.findIdsByRoleId(7L)).thenReturn(Collections.emptyList());
        when(biz.checkAdminAccess()).thenReturn(false);

        RangerRole role = buildRole(7L, Collections.singletonList("missingUser"), Collections.emptyList(),
                Collections.emptyList());
        RuntimeException expected = new RuntimeException("missing");
        when(rest.createRESTException(anyString(), any())).thenThrow(expected);
        Assertions.assertThrows(RuntimeException.class, () -> updater.createNewRoleMappingForRefTable(role, false));

        // allow creation: should schedule associator
        when(biz.checkAdminAccess()).thenReturn(true);
        updater.createNewRoleMappingForRefTable(role, true);
        verify(adapter, times(1)).executeOnTransactionCommit(any(Runnable.class));
    }
}
