/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.security.MessageDigest;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.ratis.io.MD5Hash;
import org.apache.ratis.util.AtomicFileOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class MD5FileUtil {
    public static final Logger LOG = LoggerFactory.getLogger(MD5FileUtil.class);
    public static final String MD5_SUFFIX = ".md5";
    private static final String LINE_REGEX = "([0-9a-f]{32}) [ *](.+)";
    private static final Pattern LINE_PATTERN = Pattern.compile("([0-9a-f]{32}) [ *](.+)");

    static Matcher getMatcher(String md5) {
        return Optional.ofNullable(md5).map(LINE_PATTERN::matcher).filter(Matcher::matches).orElse(null);
    }

    static String getDoesNotMatchString(String line) {
        return "\"" + line + "\" does not match the pattern " + LINE_REGEX;
    }

    public static void verifySavedMD5(File dataFile, MD5Hash expectedMD5) throws IOException {
        MD5Hash storedHash = MD5FileUtil.readStoredMd5ForFile(dataFile);
        if (!expectedMD5.equals(storedHash)) {
            throw new IOException("File " + dataFile + " did not match stored MD5 checksum  (stored: " + storedHash + ", computed: " + expectedMD5);
        }
    }

    private static String readFirstLine(File f) throws IOException {
        String string;
        BufferedReader reader = new BufferedReader(new InputStreamReader(FileUtils.newInputStream(f, new OpenOption[0]), StandardCharsets.UTF_8));
        try {
            string = Optional.ofNullable(reader.readLine()).map(String::trim).orElse(null);
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ioe) {
                throw new IOException("Failed to read file: " + f, ioe);
            }
        }
        reader.close();
        return string;
    }

    public static MD5Hash readStoredMd5ForFile(File dataFile) throws IOException {
        File md5File = MD5FileUtil.getDigestFileForFile(dataFile);
        if (!md5File.exists()) {
            return null;
        }
        String md5 = MD5FileUtil.readFirstLine(md5File);
        Matcher matcher = Optional.ofNullable(MD5FileUtil.getMatcher(md5)).orElseThrow(() -> new IOException("Invalid MD5 file " + md5File + ": the content " + MD5FileUtil.getDoesNotMatchString(md5)));
        String storedHash = matcher.group(1);
        File referencedFile = new File(matcher.group(2));
        if (!referencedFile.getName().equals(dataFile.getName())) {
            throw new IOException("MD5 file at " + md5File + " references file named " + referencedFile.getName() + " but we expected it to reference " + dataFile);
        }
        return new MD5Hash(storedHash);
    }

    public static MD5Hash computeMd5ForFile(File dataFile) throws IOException {
        int bufferSize = SizeInBytes.ONE_MB.getSizeInt();
        MessageDigest digester = MD5Hash.getDigester();
        try (FileChannel in = FileUtils.newFileChannel(dataFile, StandardOpenOption.READ);){
            long fileSize = in.size();
            int offset = 0;
            while ((long)offset < fileSize) {
                int readSize = Math.toIntExact(Math.min(fileSize - (long)offset, (long)bufferSize));
                digester.update(in.map(FileChannel.MapMode.READ_ONLY, offset, readSize));
                offset += readSize;
            }
        }
        return new MD5Hash(digester.digest());
    }

    public static MD5Hash computeAndSaveMd5ForFile(File dataFile) {
        MD5Hash md5;
        try {
            md5 = MD5FileUtil.computeMd5ForFile(dataFile);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to compute MD5 for file " + dataFile, e);
        }
        try {
            MD5FileUtil.saveMD5File(dataFile, md5);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to save MD5 " + md5 + " for file " + dataFile, e);
        }
        return md5;
    }

    public static void saveMD5File(File dataFile, MD5Hash digest) throws IOException {
        String digestString = StringUtils.bytes2HexString(digest.getDigest());
        MD5FileUtil.saveMD5File(dataFile, digestString);
    }

    private static void saveMD5File(File dataFile, String digestString) throws IOException {
        String md5Line = digestString + " *" + dataFile.getName() + "\n";
        if (MD5FileUtil.getMatcher(md5Line.trim()) == null) {
            throw new IllegalArgumentException("Invalid md5 string: " + MD5FileUtil.getDoesNotMatchString(digestString));
        }
        File md5File = MD5FileUtil.getDigestFileForFile(dataFile);
        try (AtomicFileOutputStream afos = new AtomicFileOutputStream(md5File);){
            afos.write(md5Line.getBytes(StandardCharsets.UTF_8));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Saved MD5 " + digestString + " to " + md5File);
        }
    }

    public static File getDigestFileForFile(File file) {
        return new File(file.getParentFile(), file.getName() + MD5_SUFFIX);
    }
}

