/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.hop.pipeline.transforms.cassandrasstableoutput.writer;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import static org.junit.jupiter.api.Assumptions.assumeFalse;
import static org.mockito.ArgumentMatchers.anyMap;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.cassandra.io.sstable.CQLSSTableWriter;
import org.apache.commons.lang.SystemUtils;
import org.apache.hop.core.row.IRowMeta;
import org.apache.hop.core.row.IValueMeta;
import org.apache.hop.core.row.value.ValueMetaBase;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

class Cql3SsTableWriterTest {
  @BeforeAll
  static void notOnWindows() {
    assumeFalse(SystemUtils.IS_OS_WINDOWS);
  }

  public static final String KEY_FIELD = "KEY_FIELD";
  public static final String TABLE = "TABLE";
  public static final String KEY_SPACE = "KEY_SPACE";
  public static final String DIRECTORY_PATH = "directory_path";
  public static final int BUFFER_SIZE = 10;
  public static final AtomicBoolean checker = new AtomicBoolean(true);
  public static final String COLUMN = "someColumn";

  class CQL3SSTableWriterStub extends Cql3SSTableWriter {
    @Override
    CQLSSTableWriter getCQLSSTableWriter() {
      assertEquals(DIRECTORY_PATH, getDirectory());
      assertEquals(KEY_SPACE, getKeyspace());
      assertEquals(TABLE, getTable());
      assertEquals(BUFFER_SIZE, getBufferSize());
      assertEquals(KEY_FIELD, getPrimaryKey());
      CQLSSTableWriter ssWriter = mock(CQLSSTableWriter.class);
      try {
        doAnswer(
                new Answer<Void>() {
                  @Override
                  public Void answer(InvocationOnMock invocation) throws Throwable {
                    checker.set(false);
                    return null;
                  }
                })
            .when(ssWriter)
            .close();
      } catch (IOException e) {
        fail(e.toString());
      }

      try {
        doAnswer(
                new Answer<Void>() {
                  @Override
                  public Void answer(InvocationOnMock invocation) throws Throwable {
                    checker.set(true);
                    return null;
                  }
                })
            .when(ssWriter)
            .addRow(anyMap());
      } catch (Exception e) {
        fail(e.toString());
      }

      return ssWriter;
    }
  }

  @Test
  void testInit() throws Exception {
    Cql3SSTableWriter writer = getCql3SSTableWriter();
    writer.init();
  }

  private Cql3SSTableWriter getCql3SSTableWriter() {
    Cql3SSTableWriter writer = new CQL3SSTableWriterStub();
    writer.setPrimaryKey(KEY_FIELD);
    IRowMeta rmi = mock(IRowMeta.class);
    IValueMeta one = new ValueMetaBase(KEY_FIELD, ValueMetaBase.TYPE_INTEGER);
    IValueMeta two = new ValueMetaBase(COLUMN, ValueMetaBase.TYPE_STRING);
    List<IValueMeta> valueMetaList = new ArrayList<>();
    valueMetaList.add(one);
    valueMetaList.add(two);
    String[] fieldNames = new String[] {"key", "two"};
    doReturn(valueMetaList).when(rmi).getValueMetaList();
    doReturn(fieldNames).when(rmi).getFieldNames();
    writer.setRowMeta(rmi);
    writer.setBufferSize(BUFFER_SIZE);
    writer.setTable(TABLE);
    writer.setKeyspace(KEY_SPACE);
    writer.setDirectory(DIRECTORY_PATH);
    writer.setPartitionerClass("org.apache.cassandra.dht.Murmur3Partitioner");
    return writer;
  }

  @Test
  void testProcessRow() throws Exception {
    Cql3SSTableWriter writer = getCql3SSTableWriter();
    writer.init();
    Map<String, Object> input = new HashMap<>();
    input.put(KEY_FIELD, 1);
    input.put(COLUMN, "someColumnValue");
    checker.set(false);
    writer.processRow(input);
    assertTrue(checker.get());
  }

  @Test
  void testClose() throws Exception {
    Cql3SSTableWriter writer = getCql3SSTableWriter();
    writer.init();
    checker.set(true);
    writer.close();
    assertFalse(checker.get());
  }

  @Test
  void testBuildCreateTableCQLStatement() throws Exception {
    Cql3SSTableWriter writer = getCql3SSTableWriter();
    writer.init();
    assertEquals(
        "CREATE TABLE KEY_SPACE.TABLE (\"KEY_FIELD\" bigint,\"someColumn\" varchar,PRIMARY KEY "
            + "(\"KEY_FIELD\" ));",
        writer.buildCreateTableCQLStatement());
  }

  @Test
  void testBuildInsertCQLStatement() throws Exception {
    Cql3SSTableWriter writer = getCql3SSTableWriter();
    writer.init();
    assertEquals(
        "INSERT INTO KEY_SPACE.TABLE (\"key\",\"two\") VALUES (?,?);",
        writer.buildInsertCQLStatement());
  }
}
