/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.queryhistory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.DriverContext;
import org.apache.hadoop.hive.ql.ServiceContext;
import org.apache.hadoop.hive.ql.queryhistory.RecordEnricher;
import org.apache.hadoop.hive.ql.queryhistory.repository.QueryHistoryRepository;
import org.apache.hadoop.hive.ql.queryhistory.schema.Record;
import org.apache.hadoop.hive.ql.queryhistory.schema.Schema;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.hadoop.util.Time;
import org.apache.hive.common.util.ReflectionUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryHistoryService {
    private static final Logger LOG = LoggerFactory.getLogger(QueryHistoryService.class);
    private static QueryHistoryService instance = null;
    protected final Queue<Record> queryHistoryQueue = new LinkedBlockingQueue<Record>();
    private ExecutorService flushExecutor;
    private final ScheduledExecutorService periodicFlushExecutor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("QueryHistoryService periodic flush check").setDaemon(true).build());
    private final HiveConf conf;
    private final ServiceContext serviceContext;
    private final int batchSize;
    private final int maxQueueSizeInMemoryBytes;
    private final int flushInterval;
    private final QueryHistoryRepository repository;

    public static QueryHistoryService newInstance(HiveConf inputConf, ServiceContext serviceContext) {
        if (instance == null) {
            instance = new QueryHistoryService(inputConf, serviceContext);
        }
        return instance;
    }

    @VisibleForTesting
    QueryHistoryService(HiveConf inputConf, ServiceContext serviceContext) {
        LOG.info("Creating QueryHistoryService instance");
        this.conf = new HiveConf(inputConf);
        this.serviceContext = serviceContext;
        this.batchSize = HiveConf.getIntVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_QUERY_HISTORY_BATCH_SIZE);
        this.maxQueueSizeInMemoryBytes = HiveConf.getIntVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_QUERY_HISTORY_MAX_MEMORY_BYTES);
        this.flushInterval = HiveConf.getIntVar((Configuration)this.conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_QUERY_HISTORY_FLUSH_INTERVAL_SECONDS);
        this.repository = this.createRepository(this.conf);
        this.printConfigInformation();
        LOG.info("QueryHistoryService created [batchSize: {}, maxQueueSizeInMemoryBytes: {}]", (Object)this.batchSize, (Object)this.maxQueueSizeInMemoryBytes);
    }

    public void start() {
        this.repository.init(this.conf, new Schema());
        this.flushExecutor = Executors.newFixedThreadPool(1, new ThreadFactoryBuilder().setNameFormat("QueryHistoryService repository thread").setDaemon(true).setThreadFactory(r -> new QueryHistoryThread(r, this.conf)).build());
        if (this.flushInterval > 0) {
            this.periodicFlushExecutor.scheduleAtFixedRate(this::doFlush, 0L, this.flushInterval, TimeUnit.SECONDS);
        }
    }

    public static QueryHistoryService getInstance() {
        if (instance == null) {
            throw new RuntimeException("QueryHistoryService is not present when asked for the singleton instance");
        }
        return instance;
    }

    private void printConfigInformation() {
        if (this.batchSize == 0) {
            LOG.info("Query History batch size is 0: every record will be flushed synchronously after the query (not recommended)");
        } else {
            LOG.info("Query History batch size is {}: records will be flushed after every {} queries", (Object)this.batchSize, (Object)this.batchSize);
        }
        if (this.maxQueueSizeInMemoryBytes == 0) {
            LOG.info("Query History max queue memory byte size is 0: the service won't check the consumed memory to make flush decision (not recommended)");
        } else {
            LOG.info("Query History max queue memory byte size is {}: records will be flushed when the queue consumes more than {} bytes in the memory", (Object)this.maxQueueSizeInMemoryBytes, (Object)this.maxQueueSizeInMemoryBytes);
        }
        if (this.flushInterval == 0) {
            LOG.info("Query History flush interval size is 0: the service won't check for records to flush periodically");
        } else {
            LOG.info("Query History flush interval is {}s: service will attempt to flush records in every {}s", (Object)this.flushInterval, (Object)this.flushInterval);
        }
    }

    private QueryHistoryRepository createRepository(HiveConf conf) {
        try {
            return (QueryHistoryRepository)ReflectionUtil.newInstance((Class)conf.getClassByName(conf.get(HiveConf.ConfVars.HIVE_QUERY_HISTORY_REPOSITORY_CLASS.varname)), (Configuration)conf);
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public void logQuery(DriverContext driverContext) {
        long recordCreateStart = Time.monotonicNow();
        Record record = this.createRecord(driverContext);
        long recordCreateEnd = Time.monotonicNow();
        this.queryHistoryQueue.add(record);
        LOG.info("Created query history record in {}ms, current queue size: {}", (Object)(recordCreateEnd - recordCreateStart), (Object)this.queryHistoryQueue.size());
        CompletableFuture<?> flushFuture = this.flushRecordsIfNeeded();
        this.waitForRecordsToBeFlushed(this.batchSize == 0, flushFuture);
    }

    private CompletableFuture<?> flushRecordsIfNeeded() {
        if (this.needToFlush()) {
            return this.doFlush();
        }
        LOG.debug("Not flushing history records yet, current queue size: {}, batchSize: {}", (Object)this.queryHistoryQueue.size(), (Object)this.batchSize);
        return CompletableFuture.completedFuture(null);
    }

    private void waitForRecordsToBeFlushed(boolean shouldWait, CompletableFuture<?> flushFuture) {
        if (shouldWait) {
            flushFuture.join();
        }
    }

    private CompletableFuture<Void> doFlush() {
        LOG.debug("Submitting a job to flush {} history records", (Object)this.queryHistoryQueue.size());
        return CompletableFuture.runAsync(() -> this.repository.flush(this.queryHistoryQueue), this.flushExecutor);
    }

    @VisibleForTesting
    boolean needToFlush() {
        boolean needToFlushAccordingToQueueMemorySize;
        boolean needToFlushAccordingToQueueSize;
        boolean bl = needToFlushAccordingToQueueSize = this.queryHistoryQueue.size() >= this.batchSize;
        if (needToFlushAccordingToQueueSize) {
            LOG.info("Need to flush query history records as queue size ({}) >= batch size ({})", (Object)this.queryHistoryQueue.size(), (Object)this.batchSize);
            return true;
        }
        if (this.maxQueueSizeInMemoryBytes == 0) {
            return false;
        }
        long queueMemorySize = this.queryHistoryQueue.stream().mapToLong(Record::getEstimatedSizeInMemoryBytes).sum();
        boolean bl2 = needToFlushAccordingToQueueMemorySize = queueMemorySize > (long)this.maxQueueSizeInMemoryBytes;
        if (needToFlushAccordingToQueueMemorySize) {
            LOG.info("Need to flush query history records as queue size in memory ({} bytes) is greater than the limit ({} bytes)", (Object)queueMemorySize, (Object)this.maxQueueSizeInMemoryBytes);
        }
        return needToFlushAccordingToQueueMemorySize;
    }

    public void stop() {
        LOG.info("Stopping QueryHistoryService, leftover records to flush: {}", (Object)this.queryHistoryQueue.size());
        this.periodicFlushExecutor.shutdownNow();
        CompletableFuture<Void> flushFuture = this.doFlush();
        this.waitForRecordsToBeFlushed(true, flushFuture);
        this.flushExecutor.shutdownNow();
    }

    private Record createRecord(DriverContext driverContext) {
        return new RecordEnricher(driverContext, this.serviceContext, SessionState.get(), SessionState.getPerfLogger()).createRecord();
    }

    @VisibleForTesting
    public QueryHistoryRepository getRepository() {
        return this.repository;
    }

    private static class QueryHistoryThread
    extends Thread {
        private final HiveConf conf;

        public QueryHistoryThread(Runnable runnable, HiveConf conf) {
            super(runnable);
            conf.setBoolVar(HiveConf.ConfVars.HIVE_CLI_TEZ_INITIALIZE_SESSION, false);
            this.conf = conf;
        }

        @Override
        public void run() {
            SessionState session = SessionState.start(this.conf);
            LOG.info("Session for QueryHistoryService started, sessionId: {}", (Object)session.getSessionId());
            super.run();
        }
    }
}

