/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.errorprone.annotations.RestrictedApi;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.trace.TraceUtil;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class PrefetchExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(PrefetchExecutor.class);
    public static final String PREFETCH_DELAY = "hbase.hfile.prefetch.delay";
    public static final String PREFETCH_DELAY_VARIATION = "hbase.hfile.prefetch.delay.variation";
    public static final float PREFETCH_DELAY_VARIATION_DEFAULT_VALUE = 0.2f;
    private static final Map<Path, Future<?>> prefetchFutures = new ConcurrentSkipListMap();
    private static final Map<Path, Runnable> prefetchRunnable = new ConcurrentSkipListMap<Path, Runnable>();
    private static final ScheduledExecutorService prefetchExecutorPool;
    private static int prefetchDelayMillis;
    private static float prefetchDelayVariation;
    private static final Pattern prefetchPathExclude;

    public static void request(Path path, Runnable runnable) {
        if (!prefetchPathExclude.matcher(path.toString()).find()) {
            long delay = prefetchDelayMillis > 0 ? (long)((float)prefetchDelayMillis * (1.0f - prefetchDelayVariation / 2.0f) + (float)prefetchDelayMillis * (prefetchDelayVariation / 2.0f) * ThreadLocalRandom.current().nextFloat()) : 0L;
            try {
                LOG.debug("Prefetch requested for {}, delay={} ms", (Object)path, (Object)delay);
                Runnable tracedRunnable = TraceUtil.tracedRunnable((Runnable)runnable, (String)"PrefetchExecutor.request");
                ScheduledFuture<?> future = prefetchExecutorPool.schedule(tracedRunnable, delay, TimeUnit.MILLISECONDS);
                prefetchFutures.put(path, future);
                prefetchRunnable.put(path, runnable);
            }
            catch (RejectedExecutionException e) {
                prefetchFutures.remove(path);
                prefetchRunnable.remove(path);
                LOG.warn("Prefetch request rejected for {}", (Object)path);
            }
        }
    }

    public static void complete(Path path) {
        prefetchFutures.remove(path);
        prefetchRunnable.remove(path);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Prefetch completed for {}", (Object)path.getName());
        }
        LOG.debug("Prefetch completed for {}", (Object)path);
    }

    public static void cancel(Path path) {
        Future<?> future = prefetchFutures.get(path);
        if (future != null) {
            future.cancel(true);
            prefetchFutures.remove(path);
            prefetchRunnable.remove(path);
            LOG.debug("Prefetch cancelled for {}", (Object)path);
        }
    }

    public static void interrupt(Path path) {
        Future<?> future = prefetchFutures.get(path);
        if (future != null) {
            prefetchFutures.remove(path);
            future.cancel(true);
            LOG.debug("Prefetch cancelled for {}", (Object)path);
        }
    }

    public static boolean isCompleted(Path path) {
        Future<?> future = prefetchFutures.get(path);
        if (future != null) {
            return future.isDone();
        }
        return true;
    }

    private PrefetchExecutor() {
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    static ScheduledExecutorService getExecutorPool() {
        return prefetchExecutorPool;
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    static Map<Path, Future<?>> getPrefetchFutures() {
        return prefetchFutures;
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    static Map<Path, Runnable> getPrefetchRunnable() {
        return prefetchRunnable;
    }

    static boolean isPrefetchStarted() {
        AtomicBoolean prefetchStarted = new AtomicBoolean(false);
        for (Map.Entry<Path, Future<?>> entry : prefetchFutures.entrySet()) {
            Path k = entry.getKey();
            Future<?> v = entry.getValue();
            ScheduledFuture sf = (ScheduledFuture)prefetchFutures.get(k);
            long waitTime = sf.getDelay(TimeUnit.MILLISECONDS);
            if (waitTime >= 0L) continue;
            prefetchStarted.set(true);
            break;
        }
        return prefetchStarted.get();
    }

    public static int getPrefetchDelay() {
        return prefetchDelayMillis;
    }

    public static void loadConfiguration(Configuration conf) {
        prefetchDelayMillis = conf.getInt(PREFETCH_DELAY, 1000);
        prefetchDelayVariation = conf.getFloat(PREFETCH_DELAY_VARIATION, 0.2f);
        prefetchFutures.forEach((k, v) -> {
            ScheduledFuture sf = (ScheduledFuture)prefetchFutures.get(k);
            if (sf.getDelay(TimeUnit.MILLISECONDS) <= 0L) {
                PrefetchExecutor.interrupt(k);
                PrefetchExecutor.request(k, prefetchRunnable.get(k));
            }
            LOG.debug("Reset called on Prefetch of file {} with delay {}, delay variation {}", new Object[]{k, prefetchDelayMillis, Float.valueOf(prefetchDelayVariation)});
        });
    }

    static {
        Configuration conf = HBaseConfiguration.create();
        prefetchDelayMillis = conf.getInt(PREFETCH_DELAY, 1000);
        prefetchDelayVariation = conf.getFloat(PREFETCH_DELAY_VARIATION, 0.2f);
        int prefetchThreads = conf.getInt("hbase.hfile.thread.prefetch", 4);
        prefetchExecutorPool = new ScheduledThreadPoolExecutor(prefetchThreads, new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                String name = "hfile-prefetch-" + EnvironmentEdgeManager.currentTime();
                Thread t = new Thread(r, name);
                t.setDaemon(true);
                return t;
            }
        });
        prefetchPathExclude = Pattern.compile("(/" + ".tmp".replace(".", "\\.") + '/' + ")|(" + '/' + "compaction.dir".replace(".", "\\.") + '/' + ")");
    }
}

