/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 * 
 */
package org.apache.directory.server.operations.lookup;


import static org.junit.jupiter.api.Assertions.assertNotNull;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.directory.Attributes;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;

import org.apache.directory.api.ldap.model.entry.Entry;
import org.apache.directory.api.ldap.model.message.Control;
import org.apache.directory.api.ldap.util.JndiUtils;
import org.apache.directory.ldap.client.api.LdapConnection;
import org.apache.directory.server.annotations.CreateLdapServer;
import org.apache.directory.server.annotations.CreateTransport;
import org.apache.directory.server.core.annotations.ApplyLdifs;
import org.apache.directory.server.core.annotations.ContextEntry;
import org.apache.directory.server.core.annotations.CreateDS;
import org.apache.directory.server.core.annotations.CreateIndex;
import org.apache.directory.server.core.annotations.CreatePartition;
import org.apache.directory.server.core.integ.AbstractLdapTestUnit;
import org.apache.directory.server.core.integ.ApacheDSTestExtension;
import org.apache.directory.server.core.partition.impl.btree.mavibot.MavibotIndex;
import org.apache.directory.server.core.partition.impl.btree.mavibot.MavibotPartition;
import org.apache.directory.server.integ.ServerIntegrationUtils;
import org.apache.directory.server.ldap.LdapServer;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;


/**
 * Testcase for the lookup operation.
 * 
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
@ExtendWith( { ApacheDSTestExtension.class } )
@CreateDS(allowAnonAccess = true, name = "LookupMavibotPerfIT-class", partitions =
{
    @CreatePartition(
        name = "example",
        type = MavibotPartition.class,
        suffix = "dc=example,dc=com",
        contextEntry = @ContextEntry(
            entryLdif = "dn: dc=example,dc=com\n" +
                "dc: example\n" +
                "objectClass: top\n" +
                "objectClass: domain\n\n"),
        indexes =
            {
                @CreateIndex(type = MavibotIndex.class, attribute = "objectClass"),
                @CreateIndex(type = MavibotIndex.class, attribute = "dc"),
                @CreateIndex(type = MavibotIndex.class, attribute = "ou")
        }),
    })
@CreateLdapServer(transports =
    { @CreateTransport(protocol = "LDAP") })
@ApplyLdifs(
    {
        // Entry # 0
        "dn: cn=The Person,dc=example,dc=com",
        "objectClass: person",
        "objectClass: top",
        "cn: The Person",
        "description: this is a person",
        "sn: Person",
    })
public class LookupMavibotPerfIT extends AbstractLdapTestUnit
{
    /**
     * Evaluate the lookup operation performances
     */
    @Test
    public void testLookupPerfAPI() throws Exception
    {
        LdapConnection connection = ServerIntegrationUtils.getAdminConnection( getLdapServer() );

        Entry entry = connection.lookup( "cn=The Person,dc=example,dc=com" );
        assertNotNull( entry );

        long t0 = System.currentTimeMillis();

        for ( int i = 0; i < 50; i++ )
        {
            for ( int j = 0; j < 10000; j++ )
            {
                entry = connection.lookup( "cn=The Person,dc=example,dc=com" );
            }

            System.out.print( "." );
        }

        long t1 = System.currentTimeMillis();

        System.out.println( "Delta : " + ( t1 - t0 ) );
        connection.close();
    }


    public LdapContext getWiredContext( LdapServer ldapServer, Control[] controls ) throws Exception
    {
        Hashtable<String, Object> env = setDefaultJNDIEnv();
        env.put( Context.PROVIDER_URL, "ldap://localhost:" + ldapServer.getPort() );

        return new InitialLdapContext( env, JndiUtils.toJndiControls(
            ldapServer.getDirectoryService().getLdapCodecService(),
            controls ) );
    }


    /**
     * Evaluate the lookup operation performances
     */
    @Test
    @Disabled
    public void testLookupPerfJNDI() throws Exception
    {
        LdapContext ctx = getWiredContext( ldapServer, null );

        Attributes result = ctx.getAttributes( "cn=The Person,dc=example,dc=com" );

        assertNotNull( result );

        long t0 = System.currentTimeMillis();

        for ( int i = 0; i < 50; i++ )
        {
            for ( int j = 0; j < 10000; j++ )
            {
                ctx.getAttributes( "cn=The Person,dc=example,dc=com" );
            }

            System.out.print( "." );
        }

        long t1 = System.currentTimeMillis();

        System.out.println( "Delta : " + ( t1 - t0 ) );

        ctx.close();
    }
}
