/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.quarkus.component.ldap.it;

import java.io.InputStream;
import java.net.InetAddress;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;

import com.unboundid.ldap.listener.InMemoryDirectoryServer;
import com.unboundid.ldap.listener.InMemoryDirectoryServerConfig;
import com.unboundid.ldap.listener.InMemoryListenerConfig;
import com.unboundid.ldif.LDIFReader;
import com.unboundid.util.ssl.KeyStoreKeyManager;
import com.unboundid.util.ssl.SSLUtil;
import io.quarkus.test.common.QuarkusTestResourceLifecycleManager;

public class LdapTestResource implements QuarkusTestResourceLifecycleManager {

    InMemoryDirectoryServer ldapServer;

    @Override
    public Map<String, String> start() {

        try {
            // Create an LDAP server to handle unencrypted and TLS connections
            InMemoryDirectoryServerConfig dsConfig = new InMemoryDirectoryServerConfig("ou=system");
            InMemoryListenerConfig listenerConfig = InMemoryListenerConfig.createLDAPConfig("ldap",
                    InetAddress.getLoopbackAddress(), 0, null);

            // The keystore is generated by the build process
            Path keystoreFile = Paths.get("target/certs/ldap-keystore.p12");
            if (!Files.isRegularFile(keystoreFile)) {
                /* The test is run from a test-jar within Quarkus Platform, where the Ant script was not run
                 * so let's copy the keystore from test-jar to the local folder */
                Files.createDirectories(keystoreFile.getParent());
                try (InputStream in = LdapTest.class.getClassLoader()
                        .getResourceAsStream(keystoreFile.getFileName().toString())) {
                    Files.copy(in, keystoreFile);
                }
                Path truststorePath = Paths.get("target/certs/ldap-truststore.p12");
                try (InputStream in = LdapTest.class.getClassLoader()
                        .getResourceAsStream(truststorePath.getFileName().toString())) {
                    Files.copy(in, truststorePath);
                }
            }

            SSLUtil serverSSLUtil = new SSLUtil(new KeyStoreKeyManager(keystoreFile.toFile(), "changeit".toCharArray()),
                    null);
            InMemoryListenerConfig sslListenerConfig = InMemoryListenerConfig.createLDAPSConfig("ldaps",
                    InetAddress.getLoopbackAddress(), 0, serverSSLUtil.createSSLServerSocketFactory(),
                    null);
            dsConfig.setListenerConfigs(listenerConfig, sslListenerConfig);
            ldapServer = new InMemoryDirectoryServer(dsConfig);

            // Load the LDIF file from the Camel LDAP tests
            LDIFReader ldifReader = new LDIFReader(
                    LdapTest.class.getClassLoader().getResourceAsStream("LdapRouteTest.ldif"));
            ldapServer.importFromLDIF(true, ldifReader);
            ldapServer.startListening();

            String host = ldapServer.getListenAddress("ldap").getHostAddress();
            int port = ldapServer.getListenPort("ldap");
            int sslPort = ldapServer.getListenPort("ldaps");

            return Map.of(
                    "ldap.host", host,
                    "ldap.port", port + "",
                    "ldap.sslPort", sslPort + "");
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void stop() {
        if (ldapServer != null) {
            try {
                ldapServer.close();
            } catch (Exception e) {
                // ignored
            }
        }
    }
}
