/*!
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { type InfiniteData, useInfiniteQuery, type UseInfiniteQueryOptions } from "@tanstack/react-query";

import { UseDagServiceGetDagTagsKeyFn } from "openapi/queries";
import { DagService } from "openapi/requests/services.gen";
import type { DAGTagCollectionResponse } from "openapi/requests/types.gen";

export const useDagTagsInfinite = <TError = unknown>(
  {
    limit,
    orderBy,
    tagNamePattern,
  }: {
    limit?: number;
    orderBy?: Array<string>;
    tagNamePattern?: string;
  } = {},
  queryKey?: Array<unknown>,
  options?: Omit<
    UseInfiniteQueryOptions<
      DAGTagCollectionResponse,
      TError,
      InfiniteData<DAGTagCollectionResponse>,
      Array<unknown>,
      number
    >,
    "queryFn" | "queryKey"
  >,
) =>
  useInfiniteQuery({
    getNextPageParam: (
      lastPage: DAGTagCollectionResponse,
      _allPages: Array<DAGTagCollectionResponse>,
      lastPageParam: number,
    ) => (lastPageParam < lastPage.total_entries ? lastPage.tags.length + lastPageParam : undefined),
    getPreviousPageParam: (
      firstPage: DAGTagCollectionResponse,
      _allPages: Array<DAGTagCollectionResponse>,
      firstPageParam: number,
    ) => (firstPageParam > 0 ? -firstPage.tags.length + firstPageParam : undefined),
    initialPageParam: 0,
    queryFn: ({ pageParam }: { pageParam: number }) =>
      DagService.getDagTags({ limit, offset: pageParam, orderBy, tagNamePattern }),
    queryKey: UseDagServiceGetDagTagsKeyFn({ limit, orderBy, tagNamePattern }, queryKey),
    ...options,
  });
