/*
 * Copyright (C) 2016 IBM Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#include <linux/bitops.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/mutex.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>
#include <linux/pinctrl/pinconf.h>
#include <linux/pinctrl/pinconf-generic.h>
#include <linux/string.h>
#include <linux/types.h>

#include "../core.h"
#include "../pinctrl-utils.h"
#include "pinctrl-aspeed.h"

/*
 * Uses undefined macros for symbol naming and references, eg GPIOA0, MAC1LINK,
 * TIMER3 etc.
 *
 * Pins are defined in GPIO bank order:
 *
 * GPIOA0: 0
 * ...
 * GPIOA7: 7
 * GPIOB0: 8
 * ...
 * GPIOZ7: 207
 * GPIOAA0: 208
 * ...
 * GPIOAB3: 219
 *
 * Not all pins have their signals defined (yet).
 */

#define A4 2
SSSF_PIN_DECL(A4, GPIOA2, TIMER3, SIG_DESC_SET(SCU80, 2));

#define I2C9_DESC	SIG_DESC_SET(SCU90, 22)

#define C5 4
SIG_EXPR_LIST_DECL_SINGLE(SCL9, I2C9, I2C9_DESC);
SIG_EXPR_LIST_DECL_SINGLE(TIMER5, TIMER5, SIG_DESC_SET(SCU80, 4));
MS_PIN_DECL(C5, GPIOA4, SCL9, TIMER5);

FUNC_GROUP_DECL(TIMER5, C5);

#define B4 5
SIG_EXPR_LIST_DECL_SINGLE(SDA9, I2C9, I2C9_DESC);
SIG_EXPR_LIST_DECL_SINGLE(TIMER6, TIMER6, SIG_DESC_SET(SCU80, 5));
MS_PIN_DECL(B4, GPIOA5, SDA9, TIMER6);

FUNC_GROUP_DECL(TIMER6, B4);
FUNC_GROUP_DECL(I2C9, C5, B4);

#define MDIO2_DESC	SIG_DESC_SET(SCU90, 2)

#define A3 6
SIG_EXPR_LIST_DECL_SINGLE(MDC2, MDIO2, MDIO2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(TIMER7, TIMER7, SIG_DESC_SET(SCU80, 6));
MS_PIN_DECL(A3, GPIOA6, MDC2, TIMER7);

FUNC_GROUP_DECL(TIMER7, A3);

#define D5 7
SIG_EXPR_LIST_DECL_SINGLE(MDIO2, MDIO2, MDIO2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(TIMER8, TIMER8, SIG_DESC_SET(SCU80, 7));
MS_PIN_DECL(D5, GPIOA7, MDIO2, TIMER8);

FUNC_GROUP_DECL(TIMER8, D5);
FUNC_GROUP_DECL(MDIO2, A3, D5);

#define H19 13
#define H19_DESC        SIG_DESC_SET(SCU80, 13)
SIG_EXPR_LIST_DECL_SINGLE(LPCPD, LPCPD, H19_DESC);
SIG_EXPR_LIST_DECL_SINGLE(LPCSMI, LPCSMI, H19_DESC);
MS_PIN_DECL(H19, GPIOB5, LPCPD, LPCSMI);

FUNC_GROUP_DECL(LPCPD, H19);
FUNC_GROUP_DECL(LPCSMI, H19);

#define H20 14
SSSF_PIN_DECL(H20, GPIOB6, LPCPME, SIG_DESC_SET(SCU80, 14));

#define SD1_DESC	SIG_DESC_SET(SCU90, 0)
#define I2C10_DESC	SIG_DESC_SET(SCU90, 23)

#define C4 16
SIG_EXPR_LIST_DECL_SINGLE(SD1CLK, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SCL10, I2C10, I2C10_DESC);
MS_PIN_DECL(C4, GPIOC0, SD1CLK, SCL10);

#define B3 17
SIG_EXPR_LIST_DECL_SINGLE(SD1CMD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SDA10, I2C10, I2C10_DESC);
MS_PIN_DECL(B3, GPIOC1, SD1CMD, SDA10);

FUNC_GROUP_DECL(I2C10, C4, B3);

#define I2C11_DESC	SIG_DESC_SET(SCU90, 24)

#define A2 18
SIG_EXPR_LIST_DECL_SINGLE(SD1DAT0, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SCL11, I2C11, I2C11_DESC);
MS_PIN_DECL(A2, GPIOC2, SD1DAT0, SCL11);

#define E5 19
SIG_EXPR_LIST_DECL_SINGLE(SD1DAT1, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SDA11, I2C11, I2C11_DESC);
MS_PIN_DECL(E5, GPIOC3, SD1DAT1, SDA11);

FUNC_GROUP_DECL(I2C11, A2, E5);

#define I2C12_DESC	SIG_DESC_SET(SCU90, 25)

#define D4 20
SIG_EXPR_LIST_DECL_SINGLE(SD1DAT2, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SCL12, I2C12, I2C12_DESC);
MS_PIN_DECL(D4, GPIOC4, SD1DAT2, SCL12);

#define C3 21
SIG_EXPR_LIST_DECL_SINGLE(SD1DAT3, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SDA12, I2C12, I2C12_DESC);
MS_PIN_DECL(C3, GPIOC5, SD1DAT3, SDA12);

FUNC_GROUP_DECL(I2C12, D4, C3);

#define I2C13_DESC	SIG_DESC_SET(SCU90, 26)

#define B2 22
SIG_EXPR_LIST_DECL_SINGLE(SD1CD, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SCL13, I2C13, I2C13_DESC);
MS_PIN_DECL(B2, GPIOC6, SD1CD, SCL13);

#define A1 23
SIG_EXPR_LIST_DECL_SINGLE(SD1WP, SD1, SD1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(SDA13, I2C13, I2C13_DESC);
MS_PIN_DECL(A1, GPIOC7, SD1WP, SDA13);

FUNC_GROUP_DECL(I2C13, B2, A1);
FUNC_GROUP_DECL(SD1, C4, B3, A2, E5, D4, C3, B2, A1);

#define SD2_DESC	SIG_DESC_SET(SCU90, 1)
#define GPID_DESC       SIG_DESC_SET(HW_STRAP1, 21)
#define GPID0_DESC	SIG_DESC_SET(SCU8C, 8)

#define A18 24
SIG_EXPR_LIST_DECL_SINGLE(SD2CLK, SD2, SD2_DESC);
SIG_EXPR_DECL(GPID0IN, GPID0, GPID0_DESC);
SIG_EXPR_DECL(GPID0IN, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPID0IN, GPID0, GPID);
MS_PIN_DECL(A18, GPIOD0, SD2CLK, GPID0IN);

#define D16 25
SIG_EXPR_LIST_DECL_SINGLE(SD2CMD, SD2, SD2_DESC);
SIG_EXPR_DECL(GPID0OUT, GPID0, GPID0_DESC);
SIG_EXPR_DECL(GPID0OUT, GPID, GPID_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPID0OUT, GPID0, GPID);
MS_PIN_DECL(D16, GPIOD1, SD2CMD, GPID0OUT);

FUNC_GROUP_DECL(GPID0, A18, D16);

#define GPIE_DESC       SIG_DESC_SET(HW_STRAP1, 22)
#define GPIE0_DESC      SIG_DESC_SET(SCU8C, 12)
#define GPIE2_DESC      SIG_DESC_SET(SCU8C, 13)
#define GPIE4_DESC      SIG_DESC_SET(SCU8C, 14)
#define GPIE6_DESC      SIG_DESC_SET(SCU8C, 15)

#define D15 32
SIG_EXPR_LIST_DECL_SINGLE(NCTS3, NCTS3, SIG_DESC_SET(SCU80, 16));
SIG_EXPR_DECL(GPIE0IN, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL(GPIE0IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE0IN, GPIE0, GPIE);
MS_PIN_DECL(D15, GPIOE0, NCTS3, GPIE0IN);

FUNC_GROUP_DECL(NCTS3, D15);

#define C15 33
SIG_EXPR_LIST_DECL_SINGLE(NDCD3, NDCD3, SIG_DESC_SET(SCU80, 17));
SIG_EXPR_DECL(GPIE0OUT, GPIE0, GPIE0_DESC);
SIG_EXPR_DECL(GPIE0OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE0OUT, GPIE0, GPIE);
MS_PIN_DECL(C15, GPIOE1, NDCD3, GPIE0OUT);

FUNC_GROUP_DECL(NDCD3, C15);
FUNC_GROUP_DECL(GPIE0, D15, C15);

#define B15 34
SIG_EXPR_LIST_DECL_SINGLE(NDSR3, NDSR3, SIG_DESC_SET(SCU80, 18));
SIG_EXPR_DECL(GPIE2IN, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL(GPIE2IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE2IN, GPIE2, GPIE);
MS_PIN_DECL(B15, GPIOE2, NDSR3, GPIE2IN);

FUNC_GROUP_DECL(NDSR3, B15);

#define A15 35
SIG_EXPR_LIST_DECL_SINGLE(NRI3, NRI3, SIG_DESC_SET(SCU80, 19));
SIG_EXPR_DECL(GPIE2OUT, GPIE2, GPIE2_DESC);
SIG_EXPR_DECL(GPIE2OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE2OUT, GPIE2, GPIE);
MS_PIN_DECL(A15, GPIOE3, NRI3, GPIE2OUT);

FUNC_GROUP_DECL(NRI3, A15);
FUNC_GROUP_DECL(GPIE2, B15, A15);

#define E14 36
SIG_EXPR_LIST_DECL_SINGLE(NDTR3, NDTR3, SIG_DESC_SET(SCU80, 20));
SIG_EXPR_DECL(GPIE4IN, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL(GPIE4IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE4IN, GPIE4, GPIE);
MS_PIN_DECL(E14, GPIOE4, NDTR3, GPIE4IN);

FUNC_GROUP_DECL(NDTR3, E14);

#define D14 37
SIG_EXPR_LIST_DECL_SINGLE(NRTS3, NRTS3, SIG_DESC_SET(SCU80, 21));
SIG_EXPR_DECL(GPIE4OUT, GPIE4, GPIE4_DESC);
SIG_EXPR_DECL(GPIE4OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE4OUT, GPIE4, GPIE);
MS_PIN_DECL(D14, GPIOE5, NRTS3, GPIE4OUT);

FUNC_GROUP_DECL(NRTS3, D14);
FUNC_GROUP_DECL(GPIE4, E14, D14);

#define C14 38
SIG_EXPR_LIST_DECL_SINGLE(TXD3, TXD3, SIG_DESC_SET(SCU80, 22));
SIG_EXPR_DECL(GPIE6IN, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL(GPIE6IN, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE6IN, GPIE6, GPIE);
MS_PIN_DECL(C14, GPIOE6, TXD3, GPIE6IN);

FUNC_GROUP_DECL(TXD3, C14);

#define B14 39
SIG_EXPR_LIST_DECL_SINGLE(RXD3, RXD3, SIG_DESC_SET(SCU80, 23));
SIG_EXPR_DECL(GPIE6OUT, GPIE6, GPIE6_DESC);
SIG_EXPR_DECL(GPIE6OUT, GPIE, GPIE_DESC);
SIG_EXPR_LIST_DECL_DUAL(GPIE6OUT, GPIE6, GPIE);
MS_PIN_DECL(B14, GPIOE7, RXD3, GPIE6OUT);

FUNC_GROUP_DECL(RXD3, B14);
FUNC_GROUP_DECL(GPIE6, C14, B14);

#define D18 40
SSSF_PIN_DECL(D18, GPIOF0, NCTS4, SIG_DESC_SET(SCU80, 24));

#define ACPI_DESC       SIG_DESC_BIT(HW_STRAP1, 19, 0)

#define B19 41
SIG_EXPR_LIST_DECL_SINGLE(NDCD4, NDCD4, SIG_DESC_SET(SCU80, 25));
SIG_EXPR_DECL(SIOPBI, SIOPBI, SIG_DESC_SET(SCUA4, 12));
SIG_EXPR_DECL(SIOPBI, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(SIOPBI, SIOPBI, ACPI);
MS_PIN_DECL(B19, GPIOF1, NDCD4, SIOPBI);
FUNC_GROUP_DECL(NDCD4, B19);
FUNC_GROUP_DECL(SIOPBI, B19);

#define D17 43
SIG_EXPR_LIST_DECL_SINGLE(NRI4, NRI4, SIG_DESC_SET(SCU80, 27));
SIG_EXPR_DECL(SIOPBO, SIOPBO, SIG_DESC_SET(SCUA4, 14));
SIG_EXPR_DECL(SIOPBO, ACPI, ACPI_DESC);
SIG_EXPR_LIST_DECL_DUAL(SIOPBO, SIOPBO, ACPI);
MS_PIN_DECL(D17, GPIOF3, NRI4, SIOPBO);
FUNC_GROUP_DECL(NRI4, D17);
FUNC_GROUP_DECL(SIOPBO, D17);

FUNC_GROUP_DECL(ACPI, B19, D17);

#define E16 46
SSSF_PIN_DECL(E16, GPIOF6, TXD4, SIG_DESC_SET(SCU80, 30));

#define C17 47
SSSF_PIN_DECL(C17, GPIOF7, RXD4, SIG_DESC_SET(SCU80, 31));

#define AA22 54
SSSF_PIN_DECL(AA22, GPIOG6, FLBUSY, SIG_DESC_SET(SCU84, 6));

#define U18 55
SSSF_PIN_DECL(U18, GPIOG7, FLWP, SIG_DESC_SET(SCU84, 7));

#define UART6_DESC	SIG_DESC_SET(SCU90, 7)
#define ROM16_DESC	SIG_DESC_SET(SCU90, 6)
#define FLASH_WIDE	SIG_DESC_SET(HW_STRAP1, 4)
#define BOOT_SRC_NOR	{ HW_STRAP1, GENMASK(1, 0), 0, 0 }

#define A8 56
SIG_EXPR_DECL(ROMD8, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD8, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD8, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NCTS6, NCTS6, UART6_DESC);
MS_PIN_DECL(A8, GPIOH0, ROMD8, NCTS6);

#define C7 57
SIG_EXPR_DECL(ROMD9, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD9, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD9, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NDCD6, NDCD6, UART6_DESC);
MS_PIN_DECL(C7, GPIOH1, ROMD9, NDCD6);

#define B7 58
SIG_EXPR_DECL(ROMD10, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD10, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD10, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NDSR6, NDSR6, UART6_DESC);
MS_PIN_DECL(B7, GPIOH2, ROMD10, NDSR6);

#define A7 59
SIG_EXPR_DECL(ROMD11, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD11, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD11, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NRI6, NRI6, UART6_DESC);
MS_PIN_DECL(A7, GPIOH3, ROMD11, NRI6);

#define D7 60
SIG_EXPR_DECL(ROMD12, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD12, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD12, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NDTR6, NDTR6, UART6_DESC);
MS_PIN_DECL(D7, GPIOH4, ROMD12, NDTR6);

#define B6 61
SIG_EXPR_DECL(ROMD13, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD13, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD13, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(NRTS6, NRTS6, UART6_DESC);
MS_PIN_DECL(B6, GPIOH5, ROMD13, NRTS6);

#define A6 62
SIG_EXPR_DECL(ROMD14, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD14, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD14, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(TXD6, TXD6, UART6_DESC);
MS_PIN_DECL(A6, GPIOH6, ROMD14, TXD6);

#define E7 63
SIG_EXPR_DECL(ROMD15, ROM16, ROM16_DESC);
SIG_EXPR_DECL(ROMD15, ROM16S, FLASH_WIDE, BOOT_SRC_NOR);
SIG_EXPR_LIST_DECL_DUAL(ROMD15, ROM16, ROM16S);
SIG_EXPR_LIST_DECL_SINGLE(RXD6, RXD6, UART6_DESC);
MS_PIN_DECL(E7, GPIOH7, ROMD15, RXD6);

FUNC_GROUP_DECL(UART6, A8, C7, B7, A7, D7, B6, A6, E7);

#define J3 75
SSSF_PIN_DECL(J3, GPIOJ3, SGPMI, SIG_DESC_SET(SCU84, 11));

#define T4 76
SSSF_PIN_DECL(T4, GPIOJ4, VGAHS, SIG_DESC_SET(SCU84, 12));

#define U2 77
SSSF_PIN_DECL(U2, GPIOJ5, VGAVS, SIG_DESC_SET(SCU84, 13));

#define T2 78
SSSF_PIN_DECL(T2, GPIOJ6, DDCCLK, SIG_DESC_SET(SCU84, 14));

#define T1 79
SSSF_PIN_DECL(T1, GPIOJ7, DDCDAT, SIG_DESC_SET(SCU84, 15));

#define I2C5_DESC	SIG_DESC_SET(SCU90, 18)

#define E3 80
SIG_EXPR_LIST_DECL_SINGLE(SCL5, I2C5, I2C5_DESC);
SS_PIN_DECL(E3, GPIOK0, SCL5);

#define D2 81
SIG_EXPR_LIST_DECL_SINGLE(SDA5, I2C5, I2C5_DESC);
SS_PIN_DECL(D2, GPIOK1, SDA5);

FUNC_GROUP_DECL(I2C5, E3, D2);

#define I2C6_DESC	SIG_DESC_SET(SCU90, 19)

#define C1 82
SIG_EXPR_LIST_DECL_SINGLE(SCL6, I2C6, I2C6_DESC);
SS_PIN_DECL(C1, GPIOK2, SCL6);

#define F4 83
SIG_EXPR_LIST_DECL_SINGLE(SDA6, I2C6, I2C6_DESC);
SS_PIN_DECL(F4, GPIOK3, SDA6);

FUNC_GROUP_DECL(I2C6, C1, F4);

#define I2C7_DESC	SIG_DESC_SET(SCU90, 20)

#define E2 84
SIG_EXPR_LIST_DECL_SINGLE(SCL7, I2C7, I2C7_DESC);
SS_PIN_DECL(E2, GPIOK4, SCL7);

#define D1 85
SIG_EXPR_LIST_DECL_SINGLE(SDA7, I2C7, I2C7_DESC);
SS_PIN_DECL(D1, GPIOK5, SDA7);

FUNC_GROUP_DECL(I2C7, E2, D1);

#define I2C8_DESC	SIG_DESC_SET(SCU90, 21)

#define G5 86
SIG_EXPR_LIST_DECL_SINGLE(SCL8, I2C8, I2C8_DESC);
SS_PIN_DECL(G5, GPIOK6, SCL8);

#define F3 87
SIG_EXPR_LIST_DECL_SINGLE(SDA8, I2C8, I2C8_DESC);
SS_PIN_DECL(F3, GPIOK7, SDA8);

FUNC_GROUP_DECL(I2C8, G5, F3);

#define U1 88
SSSF_PIN_DECL(U1, GPIOL0, NCTS1, SIG_DESC_SET(SCU84, 16));

#define VPI18_DESC	{ SCU90, GENMASK(5, 4), 1, 0 }
#define VPI24_DESC	{ SCU90, GENMASK(5, 4), 2, 0 }
#define VPI30_DESC	{ SCU90, GENMASK(5, 4), 3, 0 }

#define T5 89
#define T5_DESC         SIG_DESC_SET(SCU84, 17)
SIG_EXPR_DECL(VPIDE, VPI18, VPI18_DESC, T5_DESC);
SIG_EXPR_DECL(VPIDE, VPI24, VPI24_DESC, T5_DESC);
SIG_EXPR_DECL(VPIDE, VPI30, VPI30_DESC, T5_DESC);
SIG_EXPR_LIST_DECL(VPIDE, SIG_EXPR_PTR(VPIDE, VPI18),
		SIG_EXPR_PTR(VPIDE, VPI24),
		SIG_EXPR_PTR(VPIDE, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(NDCD1, NDCD1, T5_DESC);
MS_PIN_DECL(T5, GPIOL1, VPIDE, NDCD1);
FUNC_GROUP_DECL(NDCD1, T5);

#define U3 90
#define U3_DESC         SIG_DESC_SET(SCU84, 18)
SIG_EXPR_DECL(VPIODD, VPI18, VPI18_DESC, U3_DESC);
SIG_EXPR_DECL(VPIODD, VPI24, VPI24_DESC, U3_DESC);
SIG_EXPR_DECL(VPIODD, VPI30, VPI30_DESC, U3_DESC);
SIG_EXPR_LIST_DECL(VPIODD, SIG_EXPR_PTR(VPIODD, VPI18),
		SIG_EXPR_PTR(VPIODD, VPI24),
		SIG_EXPR_PTR(VPIODD, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(NDSR1, NDSR1, U3_DESC);
MS_PIN_DECL(U3, GPIOL2, VPIODD, NDSR1);
FUNC_GROUP_DECL(NDSR1, U3);

#define V1 91
#define V1_DESC         SIG_DESC_SET(SCU84, 19)
SIG_EXPR_DECL(VPIHS, VPI18, VPI18_DESC, V1_DESC);
SIG_EXPR_DECL(VPIHS, VPI24, VPI24_DESC, V1_DESC);
SIG_EXPR_DECL(VPIHS, VPI30, VPI30_DESC, V1_DESC);
SIG_EXPR_LIST_DECL(VPIHS, SIG_EXPR_PTR(VPIHS, VPI18),
		SIG_EXPR_PTR(VPIHS, VPI24),
		SIG_EXPR_PTR(VPIHS, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(NRI1, NRI1, V1_DESC);
MS_PIN_DECL(V1, GPIOL3, VPIHS, NRI1);
FUNC_GROUP_DECL(NRI1, V1);

#define U4 92
#define U4_DESC         SIG_DESC_SET(SCU84, 20)
SIG_EXPR_DECL(VPIVS, VPI18, VPI18_DESC, U4_DESC);
SIG_EXPR_DECL(VPIVS, VPI24, VPI24_DESC, U4_DESC);
SIG_EXPR_DECL(VPIVS, VPI30, VPI30_DESC, U4_DESC);
SIG_EXPR_LIST_DECL(VPIVS, SIG_EXPR_PTR(VPIVS, VPI18),
		SIG_EXPR_PTR(VPIVS, VPI24),
		SIG_EXPR_PTR(VPIVS, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(NDTR1, NDTR1, U4_DESC);
MS_PIN_DECL(U4, GPIOL4, VPIVS, NDTR1);
FUNC_GROUP_DECL(NDTR1, U4);

#define V2 93
#define V2_DESC         SIG_DESC_SET(SCU84, 21)
SIG_EXPR_DECL(VPICLK, VPI18, VPI18_DESC, V2_DESC);
SIG_EXPR_DECL(VPICLK, VPI24, VPI24_DESC, V2_DESC);
SIG_EXPR_DECL(VPICLK, VPI30, VPI30_DESC, V2_DESC);
SIG_EXPR_LIST_DECL(VPICLK, SIG_EXPR_PTR(VPICLK, VPI18),
		SIG_EXPR_PTR(VPICLK, VPI24),
		SIG_EXPR_PTR(VPICLK, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(NRTS1, NRTS1, V2_DESC);
MS_PIN_DECL(V2, GPIOL5, VPICLK, NRTS1);
FUNC_GROUP_DECL(NRTS1, V2);

#define W1 94
#define W1_DESC         SIG_DESC_SET(SCU84, 22)
SIG_EXPR_LIST_DECL_SINGLE(VPIB0, VPI30, VPI30_DESC, W1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(TXD1, TXD1, W1_DESC);
MS_PIN_DECL(W1, GPIOL6, VPIB0, TXD1);
FUNC_GROUP_DECL(TXD1, W1);

#define U5 95
#define U5_DESC         SIG_DESC_SET(SCU84, 23)
SIG_EXPR_LIST_DECL_SINGLE(VPIB1, VPI30, VPI30_DESC, U5_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RXD1, RXD1, U5_DESC);
MS_PIN_DECL(U5, GPIOL7, VPIB1, RXD1);
FUNC_GROUP_DECL(RXD1, U5);

#define W4 104
#define W4_DESC         SIG_DESC_SET(SCU88, 0)
SIG_EXPR_LIST_DECL_SINGLE(VPIG0, VPI30, VPI30_DESC, W4_DESC);
SIG_EXPR_LIST_DECL_SINGLE(PWM0, PWM0, W4_DESC);
MS_PIN_DECL(W4, GPION0, VPIG0, PWM0);
FUNC_GROUP_DECL(PWM0, W4);

#define Y3 105
#define Y3_DESC         SIG_DESC_SET(SCU88, 1)
SIG_EXPR_LIST_DECL_SINGLE(VPIG1, VPI30, VPI30_DESC, Y3_DESC);
SIG_EXPR_LIST_DECL_SINGLE(PWM1, PWM1, Y3_DESC);
MS_PIN_DECL(Y3, GPION1, VPIG1, PWM1);
FUNC_GROUP_DECL(PWM1, Y3);

#define AA2 106
#define AA2_DESC        SIG_DESC_SET(SCU88, 2)
SIG_EXPR_DECL(VPIG2, VPI18, VPI18_DESC, AA2_DESC);
SIG_EXPR_DECL(VPIG2, VPI24, VPI24_DESC, AA2_DESC);
SIG_EXPR_DECL(VPIG2, VPI30, VPI30_DESC, AA2_DESC);
SIG_EXPR_LIST_DECL(VPIG2, SIG_EXPR_PTR(VPIG2, VPI18),
		SIG_EXPR_PTR(VPIG2, VPI24),
		SIG_EXPR_PTR(VPIG2, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(PWM2, PWM2, AA2_DESC);
MS_PIN_DECL(AA2, GPION2, VPIG2, PWM2);
FUNC_GROUP_DECL(PWM2, AA2);

#define AB1 107
#define AB1_DESC        SIG_DESC_SET(SCU88, 3)
SIG_EXPR_DECL(VPIG3, VPI18, VPI18_DESC, AB1_DESC);
SIG_EXPR_DECL(VPIG3, VPI24, VPI24_DESC, AB1_DESC);
SIG_EXPR_DECL(VPIG3, VPI30, VPI30_DESC, AB1_DESC);
SIG_EXPR_LIST_DECL(VPIG3, SIG_EXPR_PTR(VPIG3, VPI18),
		SIG_EXPR_PTR(VPIG2, VPI24),
		SIG_EXPR_PTR(VPIG2, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(PWM3, PWM3, AB1_DESC);
MS_PIN_DECL(AB1, GPION3, VPIG3, PWM3);
FUNC_GROUP_DECL(PWM3, AB1);

#define W5 108
#define W5_DESC         SIG_DESC_SET(SCU88, 4)
SIG_EXPR_DECL(VPIG4, VPI18, VPI18_DESC, W5_DESC);
SIG_EXPR_DECL(VPIG4, VPI24, VPI24_DESC, W5_DESC);
SIG_EXPR_DECL(VPIG4, VPI30, VPI30_DESC, W5_DESC);
SIG_EXPR_LIST_DECL(VPIG4, SIG_EXPR_PTR(VPIG4, VPI18),
		SIG_EXPR_PTR(VPIG2, VPI24),
		SIG_EXPR_PTR(VPIG2, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(PWM4, PWM4, W5_DESC);
MS_PIN_DECL(W5, GPION4, VPIG4, PWM4);
FUNC_GROUP_DECL(PWM4, W5);

#define Y4 109
#define Y4_DESC         SIG_DESC_SET(SCU88, 5)
SIG_EXPR_DECL(VPIG5, VPI18, VPI18_DESC, Y4_DESC);
SIG_EXPR_DECL(VPIG5, VPI24, VPI24_DESC, Y4_DESC);
SIG_EXPR_DECL(VPIG5, VPI30, VPI30_DESC, Y4_DESC);
SIG_EXPR_LIST_DECL(VPIG5, SIG_EXPR_PTR(VPIG5, VPI18),
		SIG_EXPR_PTR(VPIG2, VPI24),
		SIG_EXPR_PTR(VPIG2, VPI30));
SIG_EXPR_LIST_DECL_SINGLE(PWM5, PWM5, Y4_DESC);
MS_PIN_DECL(Y4, GPION5, VPIG5, PWM5);
FUNC_GROUP_DECL(PWM5, Y4);

#define AA3 110
#define AA3_DESC        SIG_DESC_SET(SCU88, 6)
SIG_EXPR_LIST_DECL_SINGLE(VPIG6, VPI30, VPI30_DESC, AA3_DESC);
SIG_EXPR_LIST_DECL_SINGLE(PWM6, PWM6, AA3_DESC);
MS_PIN_DECL(AA3, GPION6, VPIG6, PWM6);
FUNC_GROUP_DECL(PWM6, AA3);

#define AB2 111
#define AB2_DESC        SIG_DESC_SET(SCU88, 7)
SIG_EXPR_LIST_DECL_SINGLE(VPIG7, VPI30, VPI30_DESC, AB2_DESC);
SIG_EXPR_LIST_DECL_SINGLE(PWM7, PWM7, AB2_DESC);
MS_PIN_DECL(AB2, GPION7, VPIG7, PWM7);
FUNC_GROUP_DECL(PWM7, AB2);

#define V6 112
SIG_EXPR_LIST_DECL_SINGLE(VPIG8, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 8));
SS_PIN_DECL(V6, GPIOO0, VPIG8);

#define Y5 113
SIG_EXPR_LIST_DECL_SINGLE(VPIG9, VPI24, VPI24_DESC, SIG_DESC_SET(SCU88, 9));
SS_PIN_DECL(Y5, GPIOO1, VPIG9);

FUNC_GROUP_DECL(VPI18, T5, U3, V1, U4, V2, AA22, W5, Y4, AA3, AB2);
FUNC_GROUP_DECL(VPI24, T5, U3, V1, U4, V2, AA22, W5, Y4, AA3, AB2, V6, Y5);
FUNC_GROUP_DECL(VPI30, T5, U3, V1, U4, V2, W1, U5, W4, Y3, AA22, W5, Y4, AA3,
		AB2);

#define Y7 125
SIG_EXPR_LIST_DECL_SINGLE(GPIOP5, GPIOP5);
MS_PIN_DECL_(Y7, SIG_EXPR_LIST_PTR(GPIOP5));

#define AA7 126
SSSF_PIN_DECL(AA7, GPIOP6, BMCINT, SIG_DESC_SET(SCU88, 22));

#define AB7 127
SSSF_PIN_DECL(AB7, GPIOP7, FLACK, SIG_DESC_SET(SCU88, 23));

#define I2C3_DESC	SIG_DESC_SET(SCU90, 16)

#define D3 128
SIG_EXPR_LIST_DECL_SINGLE(SCL3, I2C3, I2C3_DESC);
SS_PIN_DECL(D3, GPIOQ0, SCL3);

#define C2 129
SIG_EXPR_LIST_DECL_SINGLE(SDA3, I2C3, I2C3_DESC);
SS_PIN_DECL(C2, GPIOQ1, SDA3);

FUNC_GROUP_DECL(I2C3, D3, C2);

#define I2C4_DESC	SIG_DESC_SET(SCU90, 17)

#define B1 130
SIG_EXPR_LIST_DECL_SINGLE(SCL4, I2C4, I2C4_DESC);
SS_PIN_DECL(B1, GPIOQ2, SCL4);

#define F5 131
SIG_EXPR_LIST_DECL_SINGLE(SDA4, I2C4, I2C4_DESC);
SS_PIN_DECL(F5, GPIOQ3, SDA4);

FUNC_GROUP_DECL(I2C4, B1, F5);

#define DASH9028_DESC	SIG_DESC_SET(SCU90, 28)

#define H2 134
SIG_EXPR_LIST_DECL_SINGLE(DASHH2, DASHH2, DASH9028_DESC);
SS_PIN_DECL(H2, GPIOQ6, DASHH2);

#define H1 135
SIG_EXPR_LIST_DECL_SINGLE(DASHH1, DASHH1, DASH9028_DESC);
SS_PIN_DECL(H1, GPIOQ7, DASHH1);

#define V20 136
SSSF_PIN_DECL(V20, GPIOR0, ROMCS1, SIG_DESC_SET(SCU88, 24));

#define W21 137
SSSF_PIN_DECL(W21, GPIOR1, ROMCS2, SIG_DESC_SET(SCU88, 25));

#define Y22 138
SSSF_PIN_DECL(Y22, GPIOR2, ROMCS3, SIG_DESC_SET(SCU88, 26));

#define U19 139
SSSF_PIN_DECL(U19, GPIOR3, ROMCS4, SIG_DESC_SET(SCU88, 27));

#define VPOOFF0_DESC	{ SCU94, GENMASK(1, 0), 0, 0 }
#define VPO12_DESC	{ SCU94, GENMASK(1, 0), 1, 0 }
#define VPO24_DESC	{ SCU94, GENMASK(1, 0), 2, 0 }
#define VPOOFF1_DESC	{ SCU94, GENMASK(1, 0), 3, 0 }
#define VPO_OFF_12      { SCU94, 0x2, 0, 0 }
#define VPO_24_OFF      SIG_DESC_SET(SCU94, 1)

#define V21 140
#define V21_DESC	SIG_DESC_SET(SCU88, 28)
SIG_EXPR_DECL(ROMA24, ROM8, V21_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA24, ROM16, V21_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA24, ROM16S, V21_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA24, SIG_EXPR_PTR(ROMA24, ROM8),
		SIG_EXPR_PTR(ROMA24, ROM16),
		SIG_EXPR_PTR(ROMA24, ROM16S));
SIG_EXPR_LIST_DECL_SINGLE(VPOR6, VPO24, V21_DESC, VPO_24_OFF);
MS_PIN_DECL(V21, GPIOR4, ROMA24, VPOR6);

#define W22 141
#define W22_DESC	SIG_DESC_SET(SCU88, 29)
SIG_EXPR_DECL(ROMA25, ROM8, W22_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA25, ROM16, W22_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA25, ROM16S, W22_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA25, SIG_EXPR_PTR(ROMA25, ROM8),
		SIG_EXPR_PTR(ROMA25, ROM16),
		SIG_EXPR_PTR(ROMA25, ROM16S));
SIG_EXPR_LIST_DECL_SINGLE(VPOR7, VPO24, W22_DESC, VPO_24_OFF);
MS_PIN_DECL(W22, GPIOR5, ROMA25, VPOR7);

#define C6 142
SIG_EXPR_LIST_DECL_SINGLE(MDC1, MDIO1, SIG_DESC_SET(SCU88, 30));
SS_PIN_DECL(C6, GPIOR6, MDC1);

#define A5 143
SIG_EXPR_LIST_DECL_SINGLE(MDIO1, MDIO1, SIG_DESC_SET(SCU88, 31));
SS_PIN_DECL(A5, GPIOR7, MDIO1);

FUNC_GROUP_DECL(MDIO1, C6, A5);

#define U21 144
#define U21_DESC        SIG_DESC_SET(SCU8C, 0)
SIG_EXPR_DECL(ROMD4, ROM8, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD4, ROM16, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD4, ROM16S, U21_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD4, SIG_EXPR_PTR(ROMD4, ROM8),
		SIG_EXPR_PTR(ROMD4, ROM16),
		SIG_EXPR_PTR(ROMD4, ROM16S));
SIG_EXPR_DECL(VPODE, VPO12, U21_DESC, VPO12_DESC);
SIG_EXPR_DECL(VPODE, VPO24, U21_DESC, VPO12_DESC);
SIG_EXPR_LIST_DECL_DUAL(VPODE, VPO12, VPO24);
MS_PIN_DECL(U21, GPIOS0, ROMD4, VPODE);

#define T19 145
#define T19_DESC        SIG_DESC_SET(SCU8C, 1)
SIG_EXPR_DECL(ROMD5, ROM8, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD5, ROM16, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD5, ROM16S, T19_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD5, SIG_EXPR_PTR(ROMD5, ROM8),
		SIG_EXPR_PTR(ROMD5, ROM16),
		SIG_EXPR_PTR(ROMD5, ROM16S));
SIG_EXPR_DECL(VPOHS, VPO12, T19_DESC, VPO12_DESC);
SIG_EXPR_DECL(VPOHS, VPO24, T19_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(VPOHS, VPO12, VPO24);
MS_PIN_DECL(T19, GPIOS1, ROMD5, VPOHS);

#define V22 146
#define V22_DESC        SIG_DESC_SET(SCU8C, 2)
SIG_EXPR_DECL(ROMD6, ROM8, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD6, ROM16, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD6, ROM16S, V22_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD6, SIG_EXPR_PTR(ROMD6, ROM8),
		SIG_EXPR_PTR(ROMD6, ROM16),
		SIG_EXPR_PTR(ROMD6, ROM16S));
SIG_EXPR_DECL(VPOVS, VPO12, V22_DESC, VPO12_DESC);
SIG_EXPR_DECL(VPOVS, VPO24, V22_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(VPOVS, VPO12, VPO24);
MS_PIN_DECL(V22, GPIOS2, ROMD6, VPOVS);

#define U20 147
#define U20_DESC        SIG_DESC_SET(SCU8C, 3)
SIG_EXPR_DECL(ROMD7, ROM8, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD7, ROM16, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_DECL(ROMD7, ROM16S, U20_DESC, VPOOFF0_DESC);
SIG_EXPR_LIST_DECL(ROMD7, SIG_EXPR_PTR(ROMD7, ROM8),
		SIG_EXPR_PTR(ROMD7, ROM16),
		SIG_EXPR_PTR(ROMD7, ROM16S));
SIG_EXPR_DECL(VPOCLK, VPO12, U20_DESC, VPO12_DESC);
SIG_EXPR_DECL(VPOCLK, VPO24, U20_DESC, VPO24_DESC);
SIG_EXPR_LIST_DECL_DUAL(VPOCLK, VPO12, VPO24);
MS_PIN_DECL(U20, GPIOS3, ROMD7, VPOCLK);

#define R18 148
#define ROMOE_DESC      SIG_DESC_SET(SCU8C, 4)
SIG_EXPR_LIST_DECL_SINGLE(GPIOS4, GPIOS4);
SIG_EXPR_DECL(ROMOE, ROM8, ROMOE_DESC);
SIG_EXPR_DECL(ROMOE, ROM16, ROMOE_DESC);
SIG_EXPR_DECL(ROMOE, ROM16S, ROMOE_DESC);
SIG_EXPR_LIST_DECL(ROMOE, SIG_EXPR_PTR(ROMOE, ROM8),
		SIG_EXPR_PTR(ROMOE, ROM16),
		SIG_EXPR_PTR(ROMOE, ROM16S));
MS_PIN_DECL_(R18, SIG_EXPR_LIST_PTR(ROMOE), SIG_EXPR_LIST_PTR(GPIOS4));

#define N21 149
#define ROMWE_DESC      SIG_DESC_SET(SCU8C, 5)
SIG_EXPR_LIST_DECL_SINGLE(GPIOS5, GPIOS5);
SIG_EXPR_DECL(ROMWE, ROM8, ROMWE_DESC);
SIG_EXPR_DECL(ROMWE, ROM16, ROMWE_DESC);
SIG_EXPR_DECL(ROMWE, ROM16S, ROMWE_DESC);
SIG_EXPR_LIST_DECL(ROMWE, SIG_EXPR_PTR(ROMWE, ROM8),
		SIG_EXPR_PTR(ROMWE, ROM16),
		SIG_EXPR_PTR(ROMWE, ROM16S));
MS_PIN_DECL_(N21, SIG_EXPR_LIST_PTR(ROMWE), SIG_EXPR_LIST_PTR(GPIOS5));

#define L22 150
#define L22_DESC        SIG_DESC_SET(SCU8C, 6)
SIG_EXPR_DECL(ROMA22, ROM8, L22_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA22, ROM16, L22_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA22, ROM16S, L22_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA22, SIG_EXPR_PTR(ROMA22, ROM8),
		SIG_EXPR_PTR(ROMA22, ROM16),
		SIG_EXPR_PTR(ROMA22, ROM16S));
SIG_EXPR_LIST_DECL_SINGLE(VPOR4, VPO24, L22_DESC, VPO_24_OFF);
MS_PIN_DECL(L22, GPIOS6, ROMA22, VPOR4);

#define K18 151
#define K18_DESC	SIG_DESC_SET(SCU8C, 7)
SIG_EXPR_DECL(ROMA23, ROM8, K18_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA23, ROM16, K18_DESC, VPO_OFF_12);
SIG_EXPR_DECL(ROMA23, ROM16S, K18_DESC, VPO_OFF_12);
SIG_EXPR_LIST_DECL(ROMA23, SIG_EXPR_PTR(ROMA23, ROM8),
		SIG_EXPR_PTR(ROMA23, ROM16),
		SIG_EXPR_PTR(ROMA23, ROM16S));
SIG_EXPR_LIST_DECL_SINGLE(VPOR5, VPO24, K18_DESC, VPO_24_OFF);
MS_PIN_DECL(K18, GPIOS7, ROMA23, VPOR5);

FUNC_GROUP_DECL(ROM8, V20, U21, T19, V22, U20, R18, N21, L22, K18, W21, Y22,
		U19);
FUNC_GROUP_DECL(ROM16, V20, U21, T19, V22, U20, R18, N21, L22, K18,
		A8, C7, B7, A7, D7, B6, A6, E7, W21, Y22, U19);
FUNC_GROUP_DECL(VPO12, U21, T19, V22, U20);
FUNC_GROUP_DECL(VPO24, U21, T19, V22, U20, L22, K18, V21, W22);

#define RMII1_DESC      SIG_DESC_BIT(HW_STRAP1, 6, 0)

#define A12 152
SIG_EXPR_LIST_DECL_SINGLE(GPIOT0, GPIOT0, SIG_DESC_SET(SCUA0, 0));
SIG_EXPR_LIST_DECL_SINGLE(RMII1TXEN, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXCK, RGMII1);
MS_PIN_DECL_(A12, SIG_EXPR_LIST_PTR(GPIOT0), SIG_EXPR_LIST_PTR(RMII1TXEN),
		SIG_EXPR_LIST_PTR(RGMII1TXCK));

#define B12 153
SIG_EXPR_LIST_DECL_SINGLE(GPIOT1, GPIOT1, SIG_DESC_SET(SCUA0, 1));
SIG_EXPR_LIST_DECL_SINGLE(DASHB12, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXCTL, RGMII1);
MS_PIN_DECL_(B12, SIG_EXPR_LIST_PTR(GPIOT1), SIG_EXPR_LIST_PTR(DASHB12),
		SIG_EXPR_LIST_PTR(RGMII1TXCTL));

#define C12 154
SIG_EXPR_LIST_DECL_SINGLE(GPIOT2, GPIOT2, SIG_DESC_SET(SCUA0, 2));
SIG_EXPR_LIST_DECL_SINGLE(RMII1TXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXD0, RGMII1);
MS_PIN_DECL_(C12, SIG_EXPR_LIST_PTR(GPIOT2), SIG_EXPR_LIST_PTR(RMII1TXD0),
		SIG_EXPR_LIST_PTR(RGMII1TXD0));

#define D12 155
SIG_EXPR_LIST_DECL_SINGLE(GPIOT3, GPIOT3, SIG_DESC_SET(SCUA0, 3));
SIG_EXPR_LIST_DECL_SINGLE(RMII1TXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXD1, RGMII1);
MS_PIN_DECL_(D12, SIG_EXPR_LIST_PTR(GPIOT3), SIG_EXPR_LIST_PTR(RMII1TXD1),
		SIG_EXPR_LIST_PTR(RGMII1TXD1));

#define E12 156
SIG_EXPR_LIST_DECL_SINGLE(GPIOT4, GPIOT4, SIG_DESC_SET(SCUA0, 4));
SIG_EXPR_LIST_DECL_SINGLE(DASHE12, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXD2, RGMII1);
MS_PIN_DECL_(E12, SIG_EXPR_LIST_PTR(GPIOT4), SIG_EXPR_LIST_PTR(DASHE12),
		SIG_EXPR_LIST_PTR(RGMII1TXD2));

#define A13 157
SIG_EXPR_LIST_DECL_SINGLE(GPIOT5, GPIOT5, SIG_DESC_SET(SCUA0, 5));
SIG_EXPR_LIST_DECL_SINGLE(DASHA13, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1TXD3, RGMII1);
MS_PIN_DECL_(A13, SIG_EXPR_LIST_PTR(GPIOT5), SIG_EXPR_LIST_PTR(DASHA13),
		SIG_EXPR_LIST_PTR(RGMII1TXD3));

#define E11 164
SIG_EXPR_LIST_DECL_SINGLE(GPIOU4, GPIOU4, SIG_DESC_SET(SCUA0, 12));
SIG_EXPR_LIST_DECL_SINGLE(RMII1RCLK, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXCK, RGMII1);
MS_PIN_DECL_(E11, SIG_EXPR_LIST_PTR(GPIOU4), SIG_EXPR_LIST_PTR(RMII1RCLK),
		SIG_EXPR_LIST_PTR(RGMII1RXCK));

#define D11 165
SIG_EXPR_LIST_DECL_SINGLE(GPIOU5, GPIOU5, SIG_DESC_SET(SCUA0, 13));
SIG_EXPR_LIST_DECL_SINGLE(DASHD11, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXCTL, RGMII1);
MS_PIN_DECL_(D11, SIG_EXPR_LIST_PTR(GPIOU5), SIG_EXPR_LIST_PTR(DASHD11),
		SIG_EXPR_LIST_PTR(RGMII1RXCTL));

#define C11 166
SIG_EXPR_LIST_DECL_SINGLE(GPIOU6, GPIOU6, SIG_DESC_SET(SCUA0, 14));
SIG_EXPR_LIST_DECL_SINGLE(RMII1RXD0, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXD0, RGMII1);
MS_PIN_DECL_(C11, SIG_EXPR_LIST_PTR(GPIOU6), SIG_EXPR_LIST_PTR(RMII1RXD0),
		SIG_EXPR_LIST_PTR(RGMII1RXD0));

#define B11 167
SIG_EXPR_LIST_DECL_SINGLE(GPIOU7, GPIOU7, SIG_DESC_SET(SCUA0, 15));
SIG_EXPR_LIST_DECL_SINGLE(RMII1RXD1, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXD1, RGMII1);
MS_PIN_DECL_(B11, SIG_EXPR_LIST_PTR(GPIOU7), SIG_EXPR_LIST_PTR(RMII1RXD1),
		SIG_EXPR_LIST_PTR(RGMII1RXD1));

#define A11 168
SIG_EXPR_LIST_DECL_SINGLE(GPIOV0, GPIOV0, SIG_DESC_SET(SCUA0, 16));
SIG_EXPR_LIST_DECL_SINGLE(RMII1CRSDV, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXD2, RGMII1);
MS_PIN_DECL_(A11, SIG_EXPR_LIST_PTR(GPIOV0), SIG_EXPR_LIST_PTR(RMII1CRSDV),
		SIG_EXPR_LIST_PTR(RGMII1RXD2));

#define E10 169
SIG_EXPR_LIST_DECL_SINGLE(GPIOV1, GPIOV1, SIG_DESC_SET(SCUA0, 17));
SIG_EXPR_LIST_DECL_SINGLE(RMII1RXER, RMII1, RMII1_DESC);
SIG_EXPR_LIST_DECL_SINGLE(RGMII1RXD3, RGMII1);
MS_PIN_DECL_(E10, SIG_EXPR_LIST_PTR(GPIOV1), SIG_EXPR_LIST_PTR(RMII1RXER),
		SIG_EXPR_LIST_PTR(RGMII1RXD3));

FUNC_GROUP_DECL(RMII1, A12, B12, C12, D12, E12, A13, E11, D11, C11, B11, A11,
		E10);
FUNC_GROUP_DECL(RGMII1, A12, B12, C12, D12, E12, A13, E11, D11, C11, B11, A11,
		E10);

/* Note we account for GPIOY4-GPIOY7 even though they're not valid, thus 216
 * pins becomes 220.
 */
#define ASPEED_G4_NR_PINS 220

/* Pins, groups and functions are sort(1):ed alphabetically for sanity */

static struct pinctrl_pin_desc aspeed_g4_pins[ASPEED_G4_NR_PINS] = {
	ASPEED_PINCTRL_PIN(A1),
	ASPEED_PINCTRL_PIN(A11),
	ASPEED_PINCTRL_PIN(A12),
	ASPEED_PINCTRL_PIN(A13),
	ASPEED_PINCTRL_PIN(A15),
	ASPEED_PINCTRL_PIN(A18),
	ASPEED_PINCTRL_PIN(A2),
	ASPEED_PINCTRL_PIN(A3),
	ASPEED_PINCTRL_PIN(A4),
	ASPEED_PINCTRL_PIN(A5),
	ASPEED_PINCTRL_PIN(A6),
	ASPEED_PINCTRL_PIN(A7),
	ASPEED_PINCTRL_PIN(A8),
	ASPEED_PINCTRL_PIN(AA2),
	ASPEED_PINCTRL_PIN(AA22),
	ASPEED_PINCTRL_PIN(AA3),
	ASPEED_PINCTRL_PIN(AA7),
	ASPEED_PINCTRL_PIN(AB1),
	ASPEED_PINCTRL_PIN(AB2),
	ASPEED_PINCTRL_PIN(AB7),
	ASPEED_PINCTRL_PIN(B1),
	ASPEED_PINCTRL_PIN(B11),
	ASPEED_PINCTRL_PIN(B12),
	ASPEED_PINCTRL_PIN(B14),
	ASPEED_PINCTRL_PIN(B15),
	ASPEED_PINCTRL_PIN(B19),
	ASPEED_PINCTRL_PIN(B2),
	ASPEED_PINCTRL_PIN(B3),
	ASPEED_PINCTRL_PIN(B4),
	ASPEED_PINCTRL_PIN(B6),
	ASPEED_PINCTRL_PIN(B7),
	ASPEED_PINCTRL_PIN(C1),
	ASPEED_PINCTRL_PIN(C11),
	ASPEED_PINCTRL_PIN(C12),
	ASPEED_PINCTRL_PIN(C14),
	ASPEED_PINCTRL_PIN(C15),
	ASPEED_PINCTRL_PIN(C17),
	ASPEED_PINCTRL_PIN(C2),
	ASPEED_PINCTRL_PIN(C3),
	ASPEED_PINCTRL_PIN(C4),
	ASPEED_PINCTRL_PIN(C5),
	ASPEED_PINCTRL_PIN(C6),
	ASPEED_PINCTRL_PIN(C7),
	ASPEED_PINCTRL_PIN(D1),
	ASPEED_PINCTRL_PIN(D11),
	ASPEED_PINCTRL_PIN(D12),
	ASPEED_PINCTRL_PIN(D14),
	ASPEED_PINCTRL_PIN(D15),
	ASPEED_PINCTRL_PIN(D16),
	ASPEED_PINCTRL_PIN(D17),
	ASPEED_PINCTRL_PIN(D18),
	ASPEED_PINCTRL_PIN(D2),
	ASPEED_PINCTRL_PIN(D3),
	ASPEED_PINCTRL_PIN(D4),
	ASPEED_PINCTRL_PIN(D5),
	ASPEED_PINCTRL_PIN(D7),
	ASPEED_PINCTRL_PIN(E10),
	ASPEED_PINCTRL_PIN(E11),
	ASPEED_PINCTRL_PIN(E12),
	ASPEED_PINCTRL_PIN(E14),
	ASPEED_PINCTRL_PIN(E16),
	ASPEED_PINCTRL_PIN(E2),
	ASPEED_PINCTRL_PIN(E3),
	ASPEED_PINCTRL_PIN(E5),
	ASPEED_PINCTRL_PIN(E7),
	ASPEED_PINCTRL_PIN(F3),
	ASPEED_PINCTRL_PIN(F4),
	ASPEED_PINCTRL_PIN(F5),
	ASPEED_PINCTRL_PIN(G5),
	ASPEED_PINCTRL_PIN(H1),
	ASPEED_PINCTRL_PIN(H19),
	ASPEED_PINCTRL_PIN(H2),
	ASPEED_PINCTRL_PIN(H20),
	ASPEED_PINCTRL_PIN(J3),
	ASPEED_PINCTRL_PIN(K18),
	ASPEED_PINCTRL_PIN(L22),
	ASPEED_PINCTRL_PIN(N21),
	ASPEED_PINCTRL_PIN(R18),
	ASPEED_PINCTRL_PIN(T1),
	ASPEED_PINCTRL_PIN(T19),
	ASPEED_PINCTRL_PIN(T2),
	ASPEED_PINCTRL_PIN(T4),
	ASPEED_PINCTRL_PIN(T5),
	ASPEED_PINCTRL_PIN(U1),
	ASPEED_PINCTRL_PIN(U18),
	ASPEED_PINCTRL_PIN(U19),
	ASPEED_PINCTRL_PIN(U2),
	ASPEED_PINCTRL_PIN(U20),
	ASPEED_PINCTRL_PIN(U21),
	ASPEED_PINCTRL_PIN(U3),
	ASPEED_PINCTRL_PIN(U4),
	ASPEED_PINCTRL_PIN(U5),
	ASPEED_PINCTRL_PIN(V1),
	ASPEED_PINCTRL_PIN(V2),
	ASPEED_PINCTRL_PIN(V20),
	ASPEED_PINCTRL_PIN(V21),
	ASPEED_PINCTRL_PIN(V22),
	ASPEED_PINCTRL_PIN(V6),
	ASPEED_PINCTRL_PIN(W1),
	ASPEED_PINCTRL_PIN(W21),
	ASPEED_PINCTRL_PIN(W22),
	ASPEED_PINCTRL_PIN(W4),
	ASPEED_PINCTRL_PIN(W5),
	ASPEED_PINCTRL_PIN(Y22),
	ASPEED_PINCTRL_PIN(Y3),
	ASPEED_PINCTRL_PIN(Y4),
	ASPEED_PINCTRL_PIN(Y5),
	ASPEED_PINCTRL_PIN(Y7),
};

static const struct aspeed_pin_group aspeed_g4_groups[] = {
	ASPEED_PINCTRL_GROUP(ACPI),
	ASPEED_PINCTRL_GROUP(BMCINT),
	ASPEED_PINCTRL_GROUP(DDCCLK),
	ASPEED_PINCTRL_GROUP(DDCDAT),
	ASPEED_PINCTRL_GROUP(FLACK),
	ASPEED_PINCTRL_GROUP(FLBUSY),
	ASPEED_PINCTRL_GROUP(FLWP),
	ASPEED_PINCTRL_GROUP(GPID0),
	ASPEED_PINCTRL_GROUP(GPIE0),
	ASPEED_PINCTRL_GROUP(GPIE2),
	ASPEED_PINCTRL_GROUP(GPIE4),
	ASPEED_PINCTRL_GROUP(GPIE6),
	ASPEED_PINCTRL_GROUP(I2C10),
	ASPEED_PINCTRL_GROUP(I2C11),
	ASPEED_PINCTRL_GROUP(I2C12),
	ASPEED_PINCTRL_GROUP(I2C13),
	ASPEED_PINCTRL_GROUP(I2C3),
	ASPEED_PINCTRL_GROUP(I2C4),
	ASPEED_PINCTRL_GROUP(I2C5),
	ASPEED_PINCTRL_GROUP(I2C6),
	ASPEED_PINCTRL_GROUP(I2C7),
	ASPEED_PINCTRL_GROUP(I2C8),
	ASPEED_PINCTRL_GROUP(I2C9),
	ASPEED_PINCTRL_GROUP(LPCPD),
	ASPEED_PINCTRL_GROUP(LPCPME),
	ASPEED_PINCTRL_GROUP(LPCPME),
	ASPEED_PINCTRL_GROUP(LPCSMI),
	ASPEED_PINCTRL_GROUP(MDIO1),
	ASPEED_PINCTRL_GROUP(MDIO2),
	ASPEED_PINCTRL_GROUP(NCTS1),
	ASPEED_PINCTRL_GROUP(NCTS3),
	ASPEED_PINCTRL_GROUP(NCTS4),
	ASPEED_PINCTRL_GROUP(NDCD1),
	ASPEED_PINCTRL_GROUP(NDCD3),
	ASPEED_PINCTRL_GROUP(NDCD4),
	ASPEED_PINCTRL_GROUP(NDSR1),
	ASPEED_PINCTRL_GROUP(NDSR3),
	ASPEED_PINCTRL_GROUP(NDTR1),
	ASPEED_PINCTRL_GROUP(NDTR3),
	ASPEED_PINCTRL_GROUP(NRI1),
	ASPEED_PINCTRL_GROUP(NRI3),
	ASPEED_PINCTRL_GROUP(NRI4),
	ASPEED_PINCTRL_GROUP(NRTS1),
	ASPEED_PINCTRL_GROUP(NRTS3),
	ASPEED_PINCTRL_GROUP(PWM0),
	ASPEED_PINCTRL_GROUP(PWM1),
	ASPEED_PINCTRL_GROUP(PWM2),
	ASPEED_PINCTRL_GROUP(PWM3),
	ASPEED_PINCTRL_GROUP(PWM4),
	ASPEED_PINCTRL_GROUP(PWM5),
	ASPEED_PINCTRL_GROUP(PWM6),
	ASPEED_PINCTRL_GROUP(PWM7),
	ASPEED_PINCTRL_GROUP(RGMII1),
	ASPEED_PINCTRL_GROUP(RMII1),
	ASPEED_PINCTRL_GROUP(ROM16),
	ASPEED_PINCTRL_GROUP(ROM8),
	ASPEED_PINCTRL_GROUP(ROMCS1),
	ASPEED_PINCTRL_GROUP(ROMCS2),
	ASPEED_PINCTRL_GROUP(ROMCS3),
	ASPEED_PINCTRL_GROUP(ROMCS4),
	ASPEED_PINCTRL_GROUP(RXD1),
	ASPEED_PINCTRL_GROUP(RXD3),
	ASPEED_PINCTRL_GROUP(RXD4),
	ASPEED_PINCTRL_GROUP(SD1),
	ASPEED_PINCTRL_GROUP(SGPMI),
	ASPEED_PINCTRL_GROUP(SIOPBI),
	ASPEED_PINCTRL_GROUP(SIOPBO),
	ASPEED_PINCTRL_GROUP(TIMER3),
	ASPEED_PINCTRL_GROUP(TIMER5),
	ASPEED_PINCTRL_GROUP(TIMER6),
	ASPEED_PINCTRL_GROUP(TIMER7),
	ASPEED_PINCTRL_GROUP(TIMER8),
	ASPEED_PINCTRL_GROUP(TXD1),
	ASPEED_PINCTRL_GROUP(TXD3),
	ASPEED_PINCTRL_GROUP(TXD4),
	ASPEED_PINCTRL_GROUP(UART6),
	ASPEED_PINCTRL_GROUP(VGAHS),
	ASPEED_PINCTRL_GROUP(VGAVS),
	ASPEED_PINCTRL_GROUP(VPI18),
	ASPEED_PINCTRL_GROUP(VPI24),
	ASPEED_PINCTRL_GROUP(VPI30),
	ASPEED_PINCTRL_GROUP(VPO12),
	ASPEED_PINCTRL_GROUP(VPO24),
};

static const struct aspeed_pin_function aspeed_g4_functions[] = {
	ASPEED_PINCTRL_FUNC(ACPI),
	ASPEED_PINCTRL_FUNC(BMCINT),
	ASPEED_PINCTRL_FUNC(DDCCLK),
	ASPEED_PINCTRL_FUNC(DDCDAT),
	ASPEED_PINCTRL_FUNC(FLACK),
	ASPEED_PINCTRL_FUNC(FLBUSY),
	ASPEED_PINCTRL_FUNC(FLWP),
	ASPEED_PINCTRL_FUNC(GPID0),
	ASPEED_PINCTRL_FUNC(GPIE0),
	ASPEED_PINCTRL_FUNC(GPIE2),
	ASPEED_PINCTRL_FUNC(GPIE4),
	ASPEED_PINCTRL_FUNC(GPIE6),
	ASPEED_PINCTRL_FUNC(I2C10),
	ASPEED_PINCTRL_FUNC(I2C11),
	ASPEED_PINCTRL_FUNC(I2C12),
	ASPEED_PINCTRL_FUNC(I2C13),
	ASPEED_PINCTRL_FUNC(I2C3),
	ASPEED_PINCTRL_FUNC(I2C4),
	ASPEED_PINCTRL_FUNC(I2C5),
	ASPEED_PINCTRL_FUNC(I2C6),
	ASPEED_PINCTRL_FUNC(I2C7),
	ASPEED_PINCTRL_FUNC(I2C8),
	ASPEED_PINCTRL_FUNC(I2C9),
	ASPEED_PINCTRL_FUNC(LPCPD),
	ASPEED_PINCTRL_FUNC(LPCPME),
	ASPEED_PINCTRL_FUNC(LPCSMI),
	ASPEED_PINCTRL_FUNC(MDIO1),
	ASPEED_PINCTRL_FUNC(MDIO2),
	ASPEED_PINCTRL_FUNC(NCTS1),
	ASPEED_PINCTRL_FUNC(NCTS3),
	ASPEED_PINCTRL_FUNC(NCTS4),
	ASPEED_PINCTRL_FUNC(NDCD1),
	ASPEED_PINCTRL_FUNC(NDCD3),
	ASPEED_PINCTRL_FUNC(NDCD4),
	ASPEED_PINCTRL_FUNC(NDSR1),
	ASPEED_PINCTRL_FUNC(NDSR3),
	ASPEED_PINCTRL_FUNC(NDTR1),
	ASPEED_PINCTRL_FUNC(NDTR3),
	ASPEED_PINCTRL_FUNC(NRI1),
	ASPEED_PINCTRL_FUNC(NRI3),
	ASPEED_PINCTRL_FUNC(NRI4),
	ASPEED_PINCTRL_FUNC(NRTS1),
	ASPEED_PINCTRL_FUNC(NRTS3),
	ASPEED_PINCTRL_FUNC(PWM0),
	ASPEED_PINCTRL_FUNC(PWM1),
	ASPEED_PINCTRL_FUNC(PWM2),
	ASPEED_PINCTRL_FUNC(PWM3),
	ASPEED_PINCTRL_FUNC(PWM4),
	ASPEED_PINCTRL_FUNC(PWM5),
	ASPEED_PINCTRL_FUNC(PWM6),
	ASPEED_PINCTRL_FUNC(PWM7),
	ASPEED_PINCTRL_FUNC(RGMII1),
	ASPEED_PINCTRL_FUNC(RMII1),
	ASPEED_PINCTRL_FUNC(ROM16),
	ASPEED_PINCTRL_FUNC(ROM8),
	ASPEED_PINCTRL_FUNC(ROMCS1),
	ASPEED_PINCTRL_FUNC(ROMCS2),
	ASPEED_PINCTRL_FUNC(ROMCS3),
	ASPEED_PINCTRL_FUNC(ROMCS4),
	ASPEED_PINCTRL_FUNC(RXD1),
	ASPEED_PINCTRL_FUNC(RXD3),
	ASPEED_PINCTRL_FUNC(RXD4),
	ASPEED_PINCTRL_FUNC(SD1),
	ASPEED_PINCTRL_FUNC(SGPMI),
	ASPEED_PINCTRL_FUNC(SIOPBI),
	ASPEED_PINCTRL_FUNC(SIOPBO),
	ASPEED_PINCTRL_FUNC(TIMER3),
	ASPEED_PINCTRL_FUNC(TIMER5),
	ASPEED_PINCTRL_FUNC(TIMER6),
	ASPEED_PINCTRL_FUNC(TIMER7),
	ASPEED_PINCTRL_FUNC(TIMER8),
	ASPEED_PINCTRL_FUNC(TXD1),
	ASPEED_PINCTRL_FUNC(TXD3),
	ASPEED_PINCTRL_FUNC(TXD4),
	ASPEED_PINCTRL_FUNC(UART6),
	ASPEED_PINCTRL_FUNC(VGAHS),
	ASPEED_PINCTRL_FUNC(VGAVS),
	ASPEED_PINCTRL_FUNC(VPI18),
	ASPEED_PINCTRL_FUNC(VPI24),
	ASPEED_PINCTRL_FUNC(VPI30),
	ASPEED_PINCTRL_FUNC(VPO12),
	ASPEED_PINCTRL_FUNC(VPO24),
};

static struct aspeed_pinctrl_data aspeed_g4_pinctrl_data = {
	.pins = aspeed_g4_pins,
	.npins = ARRAY_SIZE(aspeed_g4_pins),
	.groups = aspeed_g4_groups,
	.ngroups = ARRAY_SIZE(aspeed_g4_groups),
	.functions = aspeed_g4_functions,
	.nfunctions = ARRAY_SIZE(aspeed_g4_functions),
};

static struct pinmux_ops aspeed_g4_pinmux_ops = {
	.get_functions_count = aspeed_pinmux_get_fn_count,
	.get_function_name = aspeed_pinmux_get_fn_name,
	.get_function_groups = aspeed_pinmux_get_fn_groups,
	.set_mux = aspeed_pinmux_set_mux,
	.gpio_request_enable = aspeed_gpio_request_enable,
	.strict = true,
};

static struct pinctrl_ops aspeed_g4_pinctrl_ops = {
	.get_groups_count = aspeed_pinctrl_get_groups_count,
	.get_group_name = aspeed_pinctrl_get_group_name,
	.get_group_pins = aspeed_pinctrl_get_group_pins,
	.pin_dbg_show = aspeed_pinctrl_pin_dbg_show,
	.dt_node_to_map = pinconf_generic_dt_node_to_map_pin,
	.dt_free_map = pinctrl_utils_free_map,
};

static struct pinctrl_desc aspeed_g4_pinctrl_desc = {
	.name = "aspeed-g4-pinctrl",
	.pins = aspeed_g4_pins,
	.npins = ARRAY_SIZE(aspeed_g4_pins),
	.pctlops = &aspeed_g4_pinctrl_ops,
	.pmxops = &aspeed_g4_pinmux_ops,
};

static int aspeed_g4_pinctrl_probe(struct platform_device *pdev)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(aspeed_g4_pins); i++)
		aspeed_g4_pins[i].number = i;

	return aspeed_pinctrl_probe(pdev, &aspeed_g4_pinctrl_desc,
			&aspeed_g4_pinctrl_data);
}

static const struct of_device_id aspeed_g4_pinctrl_of_match[] = {
	{ .compatible = "aspeed,ast2400-pinctrl", },
	{ .compatible = "aspeed,g4-pinctrl", },
	{ },
};

static struct platform_driver aspeed_g4_pinctrl_driver = {
	.probe = aspeed_g4_pinctrl_probe,
	.driver = {
		.name = "aspeed-g4-pinctrl",
		.of_match_table = aspeed_g4_pinctrl_of_match,
	},
};

static int aspeed_g4_pinctrl_init(void)
{
	return platform_driver_register(&aspeed_g4_pinctrl_driver);
}

arch_initcall(aspeed_g4_pinctrl_init);
