/*
 * Copyright (C) 2014 Free Electrons
 *
 * License Terms: GNU General Public License v2
 * Author: Boris BREZILLON <boris.brezillon@free-electrons.com>
 *
 * Allwinner PRCM (Power/Reset/Clock Management) driver
 *
 */

#include <linux/mfd/core.h>
#include <linux/module.h>
#include <linux/of.h>

struct prcm_data {
	int nsubdevs;
	const struct mfd_cell *subdevs;
};

static const struct resource sun6i_a31_ar100_clk_res[] = {
	{
		.start = 0x0,
		.end = 0x3,
		.flags = IORESOURCE_MEM,
	},
};

static const struct resource sun6i_a31_apb0_clk_res[] = {
	{
		.start = 0xc,
		.end = 0xf,
		.flags = IORESOURCE_MEM,
	},
};

static const struct resource sun6i_a31_apb0_gates_clk_res[] = {
	{
		.start = 0x28,
		.end = 0x2b,
		.flags = IORESOURCE_MEM,
	},
};

static const struct resource sun6i_a31_apb0_rstc_res[] = {
	{
		.start = 0xb0,
		.end = 0xb3,
		.flags = IORESOURCE_MEM,
	},
};

static const struct mfd_cell sun6i_a31_prcm_subdevs[] = {
	{
		.name = "sun6i-a31-ar100-clk",
		.of_compatible = "allwinner,sun6i-a31-ar100-clk",
		.num_resources = ARRAY_SIZE(sun6i_a31_ar100_clk_res),
		.resources = sun6i_a31_ar100_clk_res,
	},
	{
		.name = "sun6i-a31-apb0-clk",
		.of_compatible = "allwinner,sun6i-a31-apb0-clk",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_clk_res),
		.resources = sun6i_a31_apb0_clk_res,
	},
	{
		.name = "sun6i-a31-apb0-gates-clk",
		.of_compatible = "allwinner,sun6i-a31-apb0-gates-clk",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_gates_clk_res),
		.resources = sun6i_a31_apb0_gates_clk_res,
	},
	{
		.name = "sun6i-a31-apb0-clock-reset",
		.of_compatible = "allwinner,sun6i-a31-clock-reset",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_rstc_res),
		.resources = sun6i_a31_apb0_rstc_res,
	},
};

static const struct mfd_cell sun8i_a23_prcm_subdevs[] = {
	{
		.name = "sun8i-a23-apb0-clk",
		.of_compatible = "allwinner,sun8i-a23-apb0-clk",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_clk_res),
		.resources = sun6i_a31_apb0_clk_res,
	},
	{
		.name = "sun6i-a31-apb0-gates-clk",
		.of_compatible = "allwinner,sun8i-a23-apb0-gates-clk",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_gates_clk_res),
		.resources = sun6i_a31_apb0_gates_clk_res,
	},
	{
		.name = "sun6i-a31-apb0-clock-reset",
		.of_compatible = "allwinner,sun6i-a31-clock-reset",
		.num_resources = ARRAY_SIZE(sun6i_a31_apb0_rstc_res),
		.resources = sun6i_a31_apb0_rstc_res,
	},
};

static const struct prcm_data sun6i_a31_prcm_data = {
	.nsubdevs = ARRAY_SIZE(sun6i_a31_prcm_subdevs),
	.subdevs = sun6i_a31_prcm_subdevs,
};

static const struct prcm_data sun8i_a23_prcm_data = {
	.nsubdevs = ARRAY_SIZE(sun8i_a23_prcm_subdevs),
	.subdevs = sun8i_a23_prcm_subdevs,
};

static const struct of_device_id sun6i_prcm_dt_ids[] = {
	{
		.compatible = "allwinner,sun6i-a31-prcm",
		.data = &sun6i_a31_prcm_data,
	},
	{
		.compatible = "allwinner,sun8i-a23-prcm",
		.data = &sun8i_a23_prcm_data,
	},
	{ /* sentinel */ },
};

static int sun6i_prcm_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	const struct of_device_id *match;
	const struct prcm_data *data;
	struct resource *res;
	int ret;

	match = of_match_node(sun6i_prcm_dt_ids, np);
	if (!match)
		return -EINVAL;

	data = match->data;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "no prcm memory region provided\n");
		return -ENOENT;
	}

	ret = mfd_add_devices(&pdev->dev, 0, data->subdevs, data->nsubdevs,
			      res, -1, NULL);
	if (ret) {
		dev_err(&pdev->dev, "failed to add subdevices\n");
		return ret;
	}

	return 0;
}

static struct platform_driver sun6i_prcm_driver = {
	.driver = {
		.name = "sun6i-prcm",
		.owner = THIS_MODULE,
		.of_match_table = sun6i_prcm_dt_ids,
	},
	.probe = sun6i_prcm_probe,
};
module_platform_driver(sun6i_prcm_driver);

MODULE_AUTHOR("Boris BREZILLON <boris.brezillon@free-electrons.com>");
MODULE_DESCRIPTION("Allwinner sun6i PRCM driver");
MODULE_LICENSE("GPL v2");
