package Global::Rainbow::Map;

use 5.010;
use constant { false => !1, true => !!1 };
use strict;
use warnings;
use utf8;

use constant {
	stylesheet_xpath  => '//*[@id="style_css_sheet"]',
};

BEGIN {
	$Global::Rainbow::Map::AUTHORITY = 'cpan:TOBYINK';
	$Global::Rainbow::Map::VERSION   = '0.001';
}

use Color::Library 0      ();
use XML::LibXML 0         ();

sub get_template
{
	state $template = XML::LibXML->load_xml(IO => \*DATA);
	return $template;
}

sub clone_template
{
	my ($class) = @_;
	$class->get_template->cloneNode(true);
}

sub new
{
	my ($class, %args) = @_;
	bless \%args => $class;
}

sub countries
{
	my ($self) = @_;
	grep { /^[a-z][a-z0-9-]*[a-z0-9]$/ } sort keys %$self;
}

sub colour_for_country
{
	my ($self, $country) = @_;
	
	my $colour = $self->{$country};
	defined $colour or return;
	
	my $hex;
	if ($colour =~ m{^\#?([0-9a-f]{6})$}i)
	{
		$hex = lc sprintf('#%s', $1);
	}
	elsif ($colour =~ m{^\#?([0-9a-f])([0-9a-f])([0-9a-f])$}i)
	{
		$hex = lc sprintf('#%s%s%s%s%s%s', $1, $1, $2, $2, $3, $3);
	}
	
	defined $hex and return Color::Library::Color->new(
		id         => $hex,
		name       => $hex,
		title      => $hex,
		dictionary => undef,
		value      => $hex,
	);
	
	state $library = Color::Library->dictionary(qw/svg x11/);
	return scalar $library->color($colour);
}

sub generate_css
{
	my ($self) = @_;
	my $str = '';
	
	foreach my $country ($self->countries)
	{
		my $colour = $self->colour_for_country($country)->svg;
		$str .= <<"CSS";
.${country} {
	fill: $colour;
	opacity: 1;
}
CSS
	}
	
	return $str;
}

sub libxml_document
{
	my ($self) = @_;
	my $class  = (ref $self // $self);
	my $doc    = $class->clone_template;
	my $styles = $doc->findnodes($class->stylesheet_xpath)->get_node(1);
	$styles->appendText($self->generate_css);
	return $doc;
}

sub svg
{
	my ($self) = @_;
	$self->libxml_document->toString;
}

# This just doesn't seem to work. :-(
our $png =
q{
	use File::Temp 0          ();
	use SVG::Parser 0         ();
	use SVG::Parser::Expat 0  ();
	use SVG::Parser::SAX 0    ();
	use SVG::Rasterize 0      ();
	
	my ($self, $width, $height) = @_;
	$width  ||= 640;
	$height ||= 480;
	
	my $temp_file = File::Temp->new(UNLINK => 1);
	
	state $parser = SVG::Parser->new;
	my $svg = $parser->parse($self->svg);
	
	my $rasterize = SVG::Rasterize->new(
		svg     => $svg,
		width   => $width,
		height  => $height,
	);
	$rasterize->rasterize;
	$rasterize->write(
		type      => 'png',
		file_name => $temp_file->filename,
	);
	
	return do { local(@ARGV, $/) = $temp_file->filename; <> };
};


=encoding utf8

=head1 NAME

Global::Rainbow::Map - make the world look like a rainbow, but usefully

=head1 SYNOPSIS

 my $map = Global::Rainbow::Map->new(
   gb => 'red',
   ca => 'orange',
   in => 'yellow',
   nz => 'green',
   au => 'blue',
   lk => 'purple',
 );
 print $map->svg;

=head1 DESCRIPTION

Global::Rainbow::Map generates a map of the world (Robinson projection)
with each country shaded whatever colour you like. In fact, you can even
choose colours that don't appear in the rainbow... like magenta!

The possibilities for such a module are clearly endless. If you had a
table listing per-capita alcohol consumption, you could create a map of
the world in different colours to illustrate per-capita alcohol
consumption. Or if you had a table of which countries had won the most
Olympic medals for swimming, then you could create a map of the world
in different colours to illustrate which countries had won the most
Olympic medals for swimming.

The template map used is Wikipedia's public domain blank world map
L<http://en.wikipedia.org/wiki/File:BlankMap-World6.svg>, the 14 Feb
2012 revision.

=head2 Constructor

=over

=item C<< new(%colours) >>

Creates a new map. The hash is a list of country code to colour pairings.

Country codes are typically ISO 3166-1 two letter codes. If you don't
know the code for a particular country, L<Locale::Country> can help
you out. There are a number of additional codes like "eu" (the whole
European Union) and "aq" (Antarctica). These codes are all lower case.

Colours can be hexadecimal codes like '#ff0000' or named colours from
the SVG or X11 palettes. Countries not coloured will remain a pale
grey.

Hash keys which do not match the regular expression
C<< /^[a-z][a-z0-9-]*[a-z0-9]$/ >>, in particular hash keys starting
with a hyphen are reserved for future use.

=back

=head2 Object Methods

The useful ones:

=over

=item C<< svg >>

Returns the coloured SVG map as a string.

=item C<< libxml_document >>

Returns the same coloured SVG map, but as an L<XML::LibXML::Document>.

=back

Somewhat less useful for everyday use, but may be useful for people
subclassing this module...

=over

=item C<< countries >>

Returns the list of country codes which have been explicitly assigned a
colour (i.e. the keys to the hash given in the constructor).

=item C<< colour_for_country($country_code) >>

Returns the colour for a country, as a blessed L<Color::Library::Color>
object.

Will return undef if the country has not been explicitly assigned a
colour.

=item C<< generate_css >>

Generates the CSS that would be inserted into the SVG output.

=back

=head2 Class Methods

Again, these are mostly useful for people subclassing the module.

=over

=item C<< get_template >>

Returns the template used for maps as an L<XML::LibXML::Document>.

Fetching the template and modifying it using DOM methods will affect all
maps subsequently generated for the lifetime of the process. So beware.

=item C<< clone_template >>

As per C<get_template>, but returns a clone of the template.

=back

=head1 BUGS

Please report any bugs to
L<http://rt.cpan.org/Dist/Display.html?Queue=Global-Rainbow-Map>.

=head1 SEE ALSO

L<http://commons.wikimedia.org/wiki/File_talk:BlankMap-World6.svg/Documentation>.

L<http://www.biotele.com/magenta.html>.

=head1 AUTHOR

Toby Inkster E<lt>tobyink@cpan.orgE<gt>.

=head1 COPYRIGHT AND LICENCE

Toby Inkster, 2012. No rights reserved. 

To the extent possible under law, Toby Inkster has waived all copyright
and related or neighboring rights to Global::Rainbow::Map. This work is
published from: United Kingdom.

L<http://creativecommons.org/publicdomain/zero/1.0/>.

=head1 DISCLAIMER OF WARRANTIES

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

=cut

__PACKAGE__

__DATA__
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<svg xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:cc="http://creativecommons.org/ns#" xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#" xmlns:svg="http://www.w3.org/2000/svg" xmlns="http://www.w3.org/2000/svg" xmlns:sodipodi="http://sodipodi.sourceforge.net/DTD/sodipodi-0.dtd" xmlns:inkscape="http://www.inkscape.org/namespaces/inkscape" height="442.84" width="863.21002" viewBox="82.992 45.607 2528.5721 1428.3294" id="svg2985" version="1.1" inkscape:version="0.48.0 r9654" sodipodi:docname="BlankMap-World6.svg">
  <metadata id="metadata5491">
    <rdf:RDF>
      <cc:Work rdf:about="">
        <dc:format>image/svg+xml</dc:format>
        <dc:type rdf:resource="http://purl.org/dc/dcmitype/StillImage"/>
      </cc:Work>
    </rdf:RDF>
  </metadata>
  <sodipodi:namedview pagecolor="#ffffff" bordercolor="#666666" borderopacity="1" objecttolerance="10" gridtolerance="10" guidetolerance="10" inkscape:pageopacity="0" inkscape:pageshadow="2" inkscape:window-width="1024" inkscape:window-height="538" id="namedview5489" showgrid="false" inkscape:zoom="206.8161" inkscape:cx="464.76727" inkscape:cy="247.13556" inkscape:window-x="-8" inkscape:window-y="-8" inkscape:window-maximized="1" inkscape:current-layer="svg2985"/>
  <defs id="defs2987"/>
  <style id="style_css_sheet" type="text/css"> 
/*
 * Below are Cascading Style Sheet (CSS) definitions in use in this file,
 * which allow easily changing how countries are displayed.
 *
 */

/*
 * Circles around small countries
 *
 * Change opacity to 1 to display all circles.
 *
 */
.circlexx
{
   opacity:0;
   fill:#e0e0e0;
   stroke:#000000;
   stroke-width:0.5;
}

/*
 * Smaller circles around French DOMs and Chinese SARs
 *
 * Change opacity to 1 to display all subnational circles.
 *
 */
.subxx
{
   opacity:0;
   stroke-width:0.3;
}

/*
 * Circles around small, unrecognized countries
 *
 * Change opacity to 1 to display all circles.
 *
 */
.unxx
{
   opacity:0;
   fill:#e0e0e0;
   stroke:#000000;
   stroke-width:0.3;
}

/*
 * Circles around small countries, but with no permanent residents 
 *
 * Change opacity to 1 to display all circles.
 *
 */
.noxx
{
   opacity:0;
   fill:#e0e0e0;
   stroke:#000000;
   stroke-width:0.5;
}

/*
 * land
 */
.landxx
{
   fill: #e0e0e0;
   stroke:#ffffff;
   stroke-width:0.5;
   fill-rule:evenodd;
}

/*
 * Styles for coastlines of islands with no borders
 */
.coastxx
{
   fill: #e0e0e0;
   stroke:#ffffff;
   stroke-width:0.3;
   fill-rule:evenodd;
}

/*
 * Styles for nations with limited recognition
 */
.limitxx
{
   fill: #e0e0e0;
   stroke:#ffffff;
   stroke-width:0;
   fill-rule:evenodd;
}

/*
 * Styles for nations with no permanent population.
 */
.antxx
{
   fill: #e0e0e0;
   stroke:#ffffff;
   stroke-width:0;
   fill-rule:evenodd;
}

/*
 * Ocean
 */
.oceanxx
{
   opacity: 1;
   color: #000000;
   fill:#ffffff;
   stroke:#000;
   stroke-width:0.5;
   stroke-miterlimit:1;
}

/*
 * Additional style rules
 *
 * The following are examples of colouring countries.  You can substitute these with your own styles to colour the countries on the map.
 *
 * Color a few countries:
 *
 * .gb, .au, .nc
 * {
 *    fill:       #ff0000;
 * }
 *
 * Color a few small country circles (and the countries):
 *
 * .ms, .ky
 * {
 *    opacity:    1;
 *    fill:       #ff0000;
 * }
 *
 */

</style>
  <path class="oceanxx" d="M 2079.5058,1457.5605 L 615.24779,1457.5605 C 615.13079,1457.5605 615.07079,1457.5535 614.95179,1457.5215 C 611.67679,1456.6515 608.40179,1455.7815 605.12779,1454.9105 C 599.15079,1453.3225 593.22879,1451.5225 587.30779,1449.7385 C 564.39379,1442.8315 541.76579,1434.6995 519.46979,1426.0115 C 495.81079,1416.7935 472.34979,1407.0946 449.19179,1396.6816 C 426.78379,1386.6056 405.17679,1375.0536 383.76379,1363.0356 C 363.11779,1351.4496 343.01079,1338.8296 323.37979,1325.5996 C 302.29179,1311.3876 281.47879,1296.8506 261.05379,1281.7006 C 242.35079,1267.8276 223.73679,1253.6306 206.05279,1238.4686 C 186.98679,1222.1216 167.82379,1205.7706 150.35279,1187.6906 C 132.97779,1169.7106 116.33279,1151.0606 100.15679,1131.9976 C 84.352777,1113.3736 70.086776,1093.4716 56.901786,1072.9246 C 44.046786,1052.8916 32.093776,1032.1356 22.469776,1010.3426 C 17.516776,999.12756 12.408776,987.76556 8.7017761,976.06056 C 5.0657761,964.58166 1.5997761,953.04259 -1.5932239,941.43054 C -8.0102239,918.09356 -14.055224,894.78356 -18.234224,870.92056 C -22.491222,846.60856 -24.894222,822.19856 -26.943222,797.62156 C -28.947222,773.58556 -28.829222,749.40156 -27.225222,725.34856 C -25.648222,701.69856 -23.363222,677.92456 -19.274222,654.56456 C -15.018224,630.25156 -9.3842239,606.45156 -2.8412239,582.66156 C 3.5687761,559.35256 10.483776,536.34856 20.265776,514.19856 C 30.320776,491.42956 42.318786,469.90956 55.502786,448.81656 C 68.512776,428.00356 83.108777,408.26756 98.686787,389.31756 C 113.73479,371.01156 129.68979,353.25256 146.15179,336.20856 C 163.77479,317.96256 182.64579,301.14556 201.89579,284.64756 C 220.33379,268.84456 239.55379,253.98456 258.95979,239.39656 C 278.42779,224.76056 298.39379,210.83356 318.55579,197.17556 C 339.26979,183.14456 360.47879,169.92056 382.15079,157.42556 C 403.46579,145.13656 425.31479,133.98856 447.60479,123.59356 C 469.58579,113.34356 492.14979,104.17656 514.74579,95.375565 C 537.44879,86.532565 560.37879,78.195565 583.63479,70.914565 C 589.69379,69.017565 595.80079,67.276565 601.90879,65.549565 C 604.63079,64.779565 607.36079,64.045565 610.09479,63.318565 C 611.59979,62.918565 613.56579,61.986565 615.13379,61.986565 L 2079.9638,61.986565 C 2079.9798,61.986565 2079.9958,61.986565 2080.0118,61.986565 C 2079.8448,61.986565 2079.6788,61.986565 2079.5118,61.986565 C 2102.8268,68.191565 2125.8778,75.323565 2148.5448,83.611565 C 2171.6378,92.054565 2194.6158,100.87956 2217.2348,110.52956 C 2240.5208,120.46556 2263.4848,130.95256 2285.8828,142.76456 C 2306.9808,153.89056 2327.8178,165.78556 2347.8608,178.72156 C 2368.9418,192.32656 2389.7498,206.29256 2410.2218,220.80056 C 2429.7368,234.62956 2448.9938,248.98756 2467.6208,263.99256 C 2486.8818,279.50856 2506.0408,295.30556 2524.0538,312.27356 C 2541.8708,329.05756 2558.7508,346.67256 2574.9408,365.03656 C 2591.2478,383.53256 2607.1478,402.30656 2621.4508,422.41856 C 2635.3488,441.96256 2648.4468,462.42756 2659.5898,483.67756 C 2671.1238,505.67356 2681.1048,528.35656 2688.6188,552.05056 C 2692.2518,563.50656 2695.3368,575.14756 2698.5608,586.72256 C 2701.7738,598.25356 2704.9238,609.81956 2707.4578,621.52456 C 2712.6258,645.40556 2716.6718,669.47856 2719.1178,693.79656 C 2721.5718,718.19256 2723.1798,742.56356 2722.9808,767.09056 C 2722.7908,790.65456 2720.6908,814.43756 2717.7688,837.81156 C 2714.7368,862.07056 2710.5198,886.16156 2704.6398,909.90056 C 2701.6808,921.84356 2698.3948,933.69556 2695.0378,945.53154 C 2691.7468,957.13466 2688.5558,968.83466 2684.3988,980.16156 C 2676.0918,1002.8036 2666.1688,1024.8836 2654.1338,1045.8046 C 2641.9958,1066.9066 2628.8498,1087.4866 2614.2378,1106.9696 C 2599.1658,1127.0666 2582.6678,1145.9446 2565.8948,1164.6296 C 2549.8298,1182.5266 2532.8918,1199.7316 2514.8648,1215.6616 C 2496.0788,1232.2646 2476.8978,1248.3126 2457.1558,1263.7656 C 2438.2988,1278.5276 2418.9668,1292.7306 2399.3188,1306.4206 C 2378.7018,1320.7866 2357.7538,1334.6106 2336.4758,1347.9776 C 2326.4088,1354.3016 2316.0058,1360.1326 2305.6418,1365.9516 C 2295.0348,1371.9086 2284.3858,1378.0166 2273.4448,1383.3436 C 2250.7338,1394.3986 2227.7768,1404.7266 2204.3888,1414.2665 C 2181.7848,1423.4875 2159.0678,1432.2695 2136.0228,1440.3345 C 2117.4118,1446.8465 2098.5568,1452.4975 2079.5058,1457.5605" id="ocean"/>
  <path class="landxx sd" d="m 1459.2027,675.52234 c 0.3122,-0.31326 0.3651,-0.91731 0.6373,-1.18971 0.427,-0.42681 1.0619,-0.80682 1.4872,-1.23218 0.6662,-0.66615 0,-2.09021 0.6797,-2.76177 0.4806,-0.48064 1.9764,0.2331 2.5495,-0.33992 0.3934,-0.39356 2.687,-1.09444 3.0592,-0.72233 0.5035,0.50373 0.435,2.17714 0.6796,2.42188 0.2125,0.21153 0.9695,0.0769 1.1474,0.255 0.3285,0.32852 0.351,0.95703 0.5522,1.35963 0.132,0.26722 0.4533,0.58074 0.595,0.72231 0.511,0.51084 0.2319,1.88853 0.68,2.33693 0.2929,0.29426 1.2347,0.7373 1.572,0.84975 0.8754,0.29201 4.3922,-0.86554 4.8864,-1.35964 0.1094,-0.10853 0.6705,0.066 0.7646,0.0851 1.396,0.27912 3.4934,0.30296 4.5463,0.12748 0.5291,0.74268 0.8197,1.83922 1.3596,2.3793 0.5882,0.58807 4.5663,0.22376 5.3537,-0.16999 0.2672,-0.13328 0.1224,-0.75732 0.2543,-1.01976 0.3555,-0.71081 1.4482,-1.63761 2.2521,-2.03946 0.9256,-0.46294 3.0875,-0.0709 3.6114,-0.59485 0.3603,-0.36004 -0.3423,-2.33442 0.1706,-2.84677 0.6877,-0.68765 1.9434,-1.24232 2.6345,-1.65708 1.9585,0.46267 2.7862,1.93663 3.9939,3.14424 0.6764,0.6763 1.6346,0.99716 2.2095,1.5721 0.1643,0.16484 0.3761,-0.24886 0.4247,-0.29749 0.1449,-0.14455 1.4845,-0.0425 1.7846,-0.0425 0.3686,0 1.3595,0.0851 1.6146,-0.16999 1.3165,-1.31644 2.375,-3.30984 3.7816,-4.71629 0.6551,-0.65528 0.7328,-2.2626 1.3596,-2.88924 1.1651,-1.16527 2.8545,-2.23226 3.3567,-3.73907 -1.518,-1.18369 0.4034,-4.61026 -0.595,-5.60845 -0.132,-0.13135 -2.1016,-1.84962 -2.0395,-1.91201 0.6661,-0.66584 4.3289,0.39258 4.8012,-0.55237 0.3091,-0.61645 -0.078,-1.44085 0,-1.91202 0.3873,-0.2054 3.9942,-0.0322 3.9942,0 0,0.75704 -0.6761,3.43436 -1.0622,4.20645 -0.3859,0.77169 0.8272,5.43336 1.0622,6.37335 0.1126,0.44979 -0.2929,2.12785 -0.1287,2.29441 0.3534,0.35364 1.2484,0.51144 1.6995,0.8498 1.8045,1.35346 2.8107,3.95777 4.1213,5.26862 0.4237,0.42353 -0.038,2.77247 -0.7646,2.16699 0.1449,4.26951 -0.2576,1.44328 2.2866,2.29265 0.067,-0.63971 0.1984,-1.27253 0.4031,-1.89992 l 0,0 c 0.472,-1.45 1.167,-2.4935 1.11,-4.0565 -0.064,-1.757 -0.2293,-2.9495 0.9077,-4.3875 0.864,-1.091 2.338,-0.179 2.308,1.037 0.608,-0.669 0.8913,-1.20967 1.362,-2.189 1.1884,-2.47234 -0.5621,-2.43817 6e-4,-3.10256 1.3727,-1.62074 0.3397,-4.43611 1.8334,-5.41944 1.437,-0.94598 2.014,-4.113 2.92,-5.552 0.925,-1.468 2.122,-0.79275 3.552,-0.99175 0.951,-0.133 0.7783,-2.98425 0.7953,-3.65325 0.046,-1.888 1.1483,-3.2727 1.5307,-5.069 0.5532,-2.59663 0.136,-5.45923 0.047,-7.27737 -0.096,-1.95077 -0.6974,-4.33449 0.01,-5.65963 0.9908,-1.85604 1.2937,-4.37828 2.138,-6.667 0.3282,-0.88971 0.791,-1.866 0.994,-2.795 0.222,-1.012 -0.3177,-1.80306 -0.1337,-2.80006 0.188,-1.015 0.037,-3.09969 0.9704,-3.65069 0.677,-0.4 2.5582,0.356 2.8182,-0.56 0.506,-1.78 0.2468,-1.96982 2.2088,-2.71282 1.084,-0.411 3.5514,-1.29541 4.148,-2.26337 0.4538,-0.73631 1.6019,-2.72062 1.8715,-3.01368 -0.167,-1.348 -2.4502,-1.19838 -2.5162,-2.43038 -0.934,0.84 -3.559,-3.539 -4.963,-3.778 -2.43,-0.413 -3.153,-5.119 -3.258,-7.158 -0.15,-2.904 -1.023,-5.522 -0.645,-8.377 0.168,-1.264 -0.257,-2.706 -0.76,-3.852 -0.696,-1.586 0.711,-2.303 1.056,-0.739 -0.094,-2.055 -2.453,-3.349 -3.021,-5.256 -0.686,-2.304 0.3,-3.19 -1.822,-4.673 -1.601,-1.119 -2.625,-3.237 -4.298,-4.111 -1.869,-0.977 -3.38,-1.741 -3.89,-3.96 -0.674,0.682 -2.457,3.372 -3.534,2.808 -1.412,-0.74 -1.863,1.128 -2.165,1.987 -0.583,1.655 -1.078,2.598 -2.881,2.907 -1.786,0.307 -2.25,0.682 -2.621,2.477 -0.811,3.925 -4.647,0.93 -6.506,-0.083 -1.612,-0.878 -4.436,-0.208 -6.196,-0.135 -2.299,0.095 -4.63,0.151 -6.907,-0.241 0.7529,-0.69327 0.359,-3.14429 -0.75,-1.366 -1.0967,1.75845 -2.522,1.438 -4.378,1.438 -4.74,0 -9.48,0 -14.219,0 -9.658,0 -19.4512,-0.0901 -29.1082,-0.0901 0.194,5.758 0.4012,11.60913 0.4952,17.37013 -2.493,0 -4.987,0 -7.48,0 l 0.72,37.224 c -1.237,-0.384 -2.527,-0.857 -3.762,-0.212 -1.019,0.534 -1.995,-0.061 -3.033,0.453 -2.421,1.198 0.129,3.012 -1.384,4.94 -1.213,1.544 -1.429,3.224 -3.186,4.157 -1.106,0.586 0.025,3.331 0.704,3.981 -2.519,2.015 -4.146,2.699 -2.339,6.103 0.688,1.296 -0.17,1.842 -1.227,2.588 -1.129,0.797 -1.328,1.823 -1.834,3.04599 0.98,2.64598 2.838,0.15698 4.486,1.11198 0.975,0.568 -0.5,1.501 -0.054,2.266 0.549,0.934 0.841,2.174 0.539,3.243 0.259,-0.131 0.521,-0.251 0.79,-0.36 0.784,1.26 -0.687,2.655 -0.047,3.931 0.634,1.27 2.264,1.44 2.815,2.363 0.809,1.356 -0.895,2.601 -0.295,3.919 0.62,1.365 1.708,2.392 2.75,3.433 1.863,1.863 3.488,4.6214 3.502,7.2884 0.018,0.49621 0.1161,0.98231 0.1161,1.43947 0.01,0.0111 4.4607,0 4.4607,0 0.5104,0.18882 0.6693,-0.23632 1.0644,-0.60637 0.6915,-0.0582 0.8357,0.89901 1.2418,-0.4193 0.5151,-0.32027 0.8137,-0.38143 1.171,-0.87631 z" id="sd"/>
  <path class="landxx ss" d="m 1455.3606,685.71039 c -0.036,0.44732 -0.1385,0.90729 -0.1385,1.15263 0,1.6352 2.1427,1.49933 3.2466,1.6719 0.7589,0.11876 1.6115,0.29246 2.1784,0.85442 0.6058,0.5999 0.8548,1.19197 1.6599,1.70684 0.7997,0.51106 1.2257,0.90903 1.2257,1.62612 0,0.65678 -1.0082,1.37249 -0.4572,1.98837 1.5632,1.74982 3.5335,2.55634 5.5488,3.71624 1.0404,0.5989 0.7418,1.18985 1.3765,2.05724 0.4593,0.62888 1.6006,0.85048 1.8808,1.57414 0.3163,0.81554 -0.7879,1.46034 -0.1867,2.30882 0.3341,0.47314 1.1729,0.83546 1.1729,1.49528 -0.01,1.55117 1.6359,1.82766 2.8677,2.71805 1.0802,0.78057 2.7864,1.088 2.9216,2.67015 0.089,1.04109 -0.2177,2.17404 0.52,3.06241 0.537,0.64584 0.8094,1.75269 1.516,2.16994 0.9814,0.57993 1.3358,0.96027 1.5,2.12487 0.1736,1.22278 1.7801,1.36019 2.3792,2.13773 0.2933,0.38053 1.0953,0.65245 1.5007,0.90486 1.535,0.96526 1.8715,0.31857 3.3116,-0.60563 2.1837,-1.40164 2.4971,0.0982 4.3795,0.25402 1.8963,0.15689 1.3541,-3.66689 3.6268,-1.70843 0.7466,0.6428 0.8609,1.47331 1.3969,2.18502 0.5001,0.66376 0.9022,1.5345 1.4643,2.14738 1.2507,1.3675 4.866,1.8693 3.7488,4.30288 1.0052,-0.23557 2.1413,-0.19166 2.2958,1.07803 0.4424,-1.08701 1.8329,-2.76368 3.2444,-2.4443 0.8583,0.19365 1.2372,1.31953 2.2233,0.87535 0.567,-0.25452 1.5579,-1.4107 2.1178,-0.63811 0.7447,1.02613 1.2885,1.50932 2.6259,1.99145 -0.8635,-2.0463 3.5514,-1.76586 5.5376,-2.80482 1.1088,-0.5799 2.3112,0.78322 3.7649,0.80252 0.5386,0.008 2.2763,-1.27167 2.5402,-1.58396 1.4276,-1.68895 3.2795,-5.24769 5.4852,-5.58075 3.4358,-0.51879 7.602,-0.20038 11.1305,-0.39161 -0.6358,-0.80354 -1.2979,-1.39544 -1.2979,-2.49845 0,-0.8764 0.5578,-1.91852 0.2222,-2.77695 -0.2932,-0.75761 -0.025,-0.46229 -0.884,-1.19788 -0.8853,-0.75827 -1.8642,-0.94393 -2.7111,0.14875 -0.5442,-0.52604 -0.2967,-1.34059 -0.9463,-1.70491 -0.5492,-0.30842 -0.7405,-1.09403 -1.0141,-1.61807 -0.7377,-1.41344 -0.7876,-2.99854 -1.5071,-4.39601 -0.605,-1.17584 -0.5413,-3.091 -1.7353,-3.99502 -1.0944,-0.8286 -1.642,-1.6972 -2.5073,-2.09459 -0.7062,-0.32435 -1.646,-2.57697 -2.5963,-3.7199 -0.8512,-1.0241 -1.5431,-1.87754 -2.9717,-2.61719 -1.2904,-0.66818 -2.7672,-0.38528 -3.9967,-1.09898 -1.8776,-1.08905 0.3359,-1.86929 0.5321,-3.01123 0.2436,-1.40841 -0.032,-2.40505 1.6766,-2.47875 1.1191,-0.0489 2.4206,0.41725 3.4758,0.30457 3.4596,-0.37034 2.0619,-4.43777 1.8851,-6.70862 -0.1097,-1.21479 -0.145,-2.4193 0,-3.60845 -2.5552,-0.85169 -2.1527,1.97454 -2.2976,-2.29497 0.7266,0.60548 1.1883,-1.74346 0.7646,-2.16699 -1.3106,-1.31085 -2.3168,-3.91516 -4.1213,-5.26862 -0.4511,-0.33836 -1.3461,-0.49616 -1.6995,-0.8498 -0.1642,-0.16656 0.2413,-1.84462 0.1287,-2.29441 -0.235,-0.93999 -1.4481,-5.60166 -1.0622,-6.37335 0.3861,-0.77209 1.0622,-3.44941 1.0622,-4.20645 0,-0.0322 -3.6069,-0.2054 -3.9942,0 -0.078,0.47117 0.3091,1.29557 0,1.91202 -0.4723,0.94495 -4.1351,-0.11347 -4.8012,0.55237 -0.062,0.0624 1.9075,1.78066 2.0395,1.91201 0.9984,0.99819 -0.923,4.42476 0.595,5.60845 -0.5022,1.50681 -2.1916,2.5738 -3.3567,3.73907 -0.6268,0.62664 -0.7045,2.23396 -1.3596,2.88924 -1.4066,1.40645 -2.4651,3.39985 -3.7816,4.71629 -0.2551,0.25509 -1.246,0.16999 -1.6146,0.16999 -0.3001,0 -1.6397,-0.10205 -1.7846,0.0425 -0.049,0.0486 -0.2604,0.46233 -0.4247,0.29749 -0.5749,-0.57494 -1.5331,-0.8958 -2.2095,-1.5721 -1.2077,-1.20761 -2.0354,-2.68157 -3.9939,-3.14424 -0.6911,0.41476 -1.9468,0.96943 -2.6345,1.65708 -0.5129,0.51235 0.1897,2.48673 -0.1706,2.84677 -0.5239,0.52395 -2.6858,0.13191 -3.6114,0.59485 -0.8039,0.40185 -1.8966,1.32865 -2.2521,2.03946 -0.1319,0.26244 0.013,0.88648 -0.2543,1.01976 -0.7874,0.39375 -4.7655,0.75806 -5.3537,0.16999 -0.5399,-0.54008 -0.8305,-1.63662 -1.3596,-2.3793 -1.0529,0.17548 -3.1503,0.15164 -4.5463,-0.12748 -0.094,-0.0191 -0.6552,-0.19363 -0.7646,-0.0851 -0.4942,0.4941 -4.011,1.65165 -4.8864,1.35964 -0.3373,-0.11245 -1.2791,-0.55549 -1.572,-0.84975 -0.4481,-0.4484 -0.169,-1.82609 -0.68,-2.33693 -0.1417,-0.14157 -0.463,-0.45509 -0.595,-0.72231 -0.2012,-0.4026 -0.2237,-1.03111 -0.5522,-1.35963 -0.1779,-0.1781 -0.9349,-0.0435 -1.1474,-0.255 -0.2446,-0.24474 -0.1761,-1.91815 -0.6796,-2.42188 -0.3722,-0.37211 -2.6658,0.32877 -3.0592,0.72233 -0.5731,0.57302 -2.0689,-0.14072 -2.5495,0.33992 -0.6797,0.67156 -0.014,2.09562 -0.6797,2.76177 -0.4253,0.42536 -1.0602,0.80537 -1.4872,1.23218 -0.2722,0.2724 -0.3251,0.87645 -0.6373,1.18971 -0.2895,0.28834 -0.4447,0.4974 -1.1755,0.87776 -0.459,1.40437 -0.6556,0.0824 -1.2429,0.41901 -0.1517,0.25406 -0.5894,0.60528 -1.0632,0.60528 l -4.4606,0 c -0.072,1.24447 -0.3634,1.9057 -0.7862,2.21513 -0.609,0.833 -1.231,1.078 -1,2.268 0.264,-0.095 0.526,-0.191 0.79,-0.287 0.213,0.8 -0.227,1.52 -0.5,2.231 1.313,0.239 4.313,-0.149 5.292,0.904 0.2848,0.13355 0.3664,0.47037 0.3003,0.95487 z" id="ss"/>
  <g class="landxx coastxx ge" id="ge">
    <path class="landxx coastxx ge ge-" d="m 1554.3362,384.66839 c 0.025,-3.86382 7.9783,-0.31418 9.497,0.351 1.8035,0.78992 3.2096,1.39466 5.2886,0.99175 0.9803,-0.18997 1.7777,-0.36358 2.9584,0.32625 0.4396,0.25683 1.92,-0.1688 2.716,0.29 1.8133,1.04522 3.3441,1.73612 4.124,2.233 0.345,0.21979 1.1533,0.34912 2.2313,1.45481 0.693,0.71073 -0.2871,0.27594 0.5589,0.61794 2.145,0.865 5.3638,-2.80194 6.7568,0.0801 0.4601,-1.87783 2.4289,-0.52819 2.776,-0.34481 0.375,0.338 0.685,0.75662 1.184,0.92962 0.809,0.28 2.447,-0.25962 2.857,0.93538 0.298,0.87 -0.703,1.51575 0.166,2.27175 1.34,1.166 2.992,1.15325 4.477,1.80325 2.39,1.045 -1.1826,1.79011 -0.017,3.166 0.7235,0.854 4.0538,2.211 3.849,3.321 -0.4571,2.47786 -3.297,-0.0331 -4.047,0.23993 -2.007,0.732 -3.1558,-1.57074 -5.0378,-1.93974 -0.755,-0.148 -1.4712,0.0458 -2.0312,0.50881 -0.528,0.437 -1.98,0.935 -1.727,1.27 -1.228,0.085 -2.455,-0.16 -3.685,-0.082 -1.285,0.082 -2.371,0.66 -3.703,0.655 -1.289,-0.004 -1.8279,-0.16171 -2.9191,-0.28366 -0.1917,-0.0214 -1.8622,0.0645 -1.5756,-0.56999 0.1869,-0.4138 0.025,-0.78639 -0.4159,-0.68033 -0.1243,0.0113 -1.7563,-1.81973 -2.6824,-1.56102 0.639,-1.426 -1.5618,-0.52831 -1.7952,0.35325 -0.238,0.89876 -3.1408,-0.10312 -3.8818,-0.28952 -0.5216,0.093 -0.5402,0.96704 -1.427,0.48727 -0.4376,-0.23678 -1.2699,-0.75253 -1.687,-0.53395 -0.066,-0.14814 0.091,-0.32249 0.287,-0.84605 0.5995,-1.60068 1.287,-2.534 0.172,-4.109 -1.062,-1.499 -1.204,-3.443 -2.239,-4.958 -0.857,-1.255 -2.425,-1.122 -3.336,-2.317 -0.852,-1.119 -2.649,-1.276 -3.896,-1.456 -1.732,-0.251 -2.151,-2.081 -3.796,-2.315" id="ge-"/>
    <g id="xa">
      <path class="limitxx ge xa" d="m 1554.3362,384.66839 c 1.645,0.234 2.064,2.064 3.796,2.315 1.247,0.18 3.044,0.337 3.896,1.456 0.911,1.195 2.4789,1.06204 3.336,2.317 0.262,0.38358 0.4668,0.79456 0.6438,1.21891 0,0 -0.052,0.13909 0.2507,-0.11289 0.3024,-0.25198 0.252,-0.22678 0.4536,-0.57956 0.2016,-0.35278 0.1512,-0.37798 0.4284,-0.57956 0.2772,-0.20159 0.7055,-0.27718 0.8063,-0.60476 0.1008,-0.32758 0.1008,-1.4867 0,-1.91507 -0.1008,-0.42838 0.4032,-0.95754 0.8568,-1.20952 0.4535,-0.25199 0.3527,-0.93234 0.3354,-0.94757 -2.0964,0.38768 -3.5025,-0.21706 -5.306,-1.00698 -1.5187,-0.66518 -9.4719,-4.21482 -9.497,-0.351" id="xa-"/>
      <circle class="unxx ge xa" cx="1561.3726" cy="389.19003" id="xa_" r="4.2338"/>
    </g>
    <g id="xo">
      <path class="limitxx ge xo" d="m 1581.1515,390.3152 c 0.693,0.71073 -0.2873,0.27636 0.5589,0.61794 1.1756,0.47458 2.6743,-0.41344 4.0178,-0.83765 0,0 0.245,0.31103 0.2628,0.5783 0.018,0.26727 0.053,0.55236 0.4276,0.71272 0.3742,0.16036 1.1404,0.55235 1.4255,0.71271 0.2851,0.16036 0.4098,1.3898 0.4098,1.56798 0,0.17818 -0.036,0.87307 -0.6058,1.03344 -0.5702,0.16036 -0.9444,0.48108 -1.1226,0.53453 -0.1781,0.0535 -0.4098,0.16036 -0.8909,-0.44544 -0.481,-0.60581 -1.8352,-1.10471 -1.9243,-1.15817 -0.089,-0.0534 -0.8196,0.4989 -1.176,0.64145 -0.3563,0.14254 -1.5858,-0.33854 -1.6214,-0.55236 -0.036,-0.21381 0.018,-0.74835 -0.071,-0.92653 -0.089,-0.17818 -0.2316,-0.55235 -0.4454,-0.80181 -0.2138,-0.24945 -0.4099,-0.81962 -0.018,-1.14034 0.392,-0.32073 0.7841,-0.55236 0.7732,-0.53677 z" id="xo-"/>
      <circle class="unxx ge xo" cx="1583.5995" cy="395.83105" id="xo_" r="4.2338"/>
    </g>
  </g>
  <path class="landxx pe" d="M 649.70729,798.40855 C 650.70429,796.34955 651.25329,794.94055 652.82029,793.18655 C 653.52029,792.40355 654.06029,791.46455 654.90629,790.82055 C 655.51929,790.35355 656.85129,789.72755 656.90729,788.82755 C 657.90129,789.11155 658.94929,793.17555 658.08529,793.68355 C 657.45629,794.05355 656.73829,793.72055 656.28529,794.36555 C 655.76829,795.10055 655.88429,796.10255 656.90729,796.24755 C 656.06229,797.37355 656.29775,798.80046 657.63075,797.36346 C 658.80075,796.10246 659.39029,796.98555 660.50129,797.92755 C 661.45329,798.73555 662.54329,797.51055 663.29329,798.72655 C 664.06329,799.97655 664.08829,802.04655 665.87529,802.38355 C 667.38429,802.66855 667.47922,802.58983 667.91075,800.92146 C 670.79464,789.77187 673.14786,785.85615 669.83075,797.93746 C 669.34935,799.69076 670.34629,794.76255 671.09229,793.07755 C 671.47529,792.21155 671.77948,791.72909 671.79348,790.73309 C 671.80248,790.10109 672.75029,788.20955 673.10829,789.90755 C 673.34929,789.51755 673.39629,789.10755 673.24929,788.67855 C 673.19129,788.12555 673.99929,787.52755 674.29429,787.13655 C 675.06229,786.11555 676.02329,785.35855 677.18229,784.82755 C 679.70429,783.67155 682.65629,783.24655 684.95829,781.64855 C 689.80029,778.28655 693.15329,773.25355 694.20629,767.51755 C 694.60029,767.93455 695.07929,768.10055 695.64529,768.01755 C 694.93229,766.35755 695.67829,764.88855 694.68729,763.47155 C 694.29729,762.91355 692.35829,761.21655 692.47829,760.52755 C 693.29529,760.66655 694.24829,760.94555 695.02229,760.48655 C 695.78929,760.03255 695.54629,759.88455 696.41129,760.20555 C 697.90329,760.75855 699.00729,761.35055 700.03729,762.59155 C 701.37629,764.20655 702.13329,764.80255 702.77429,766.86755 C 703.04229,767.72855 702.70929,768.43255 703.89629,768.53555 C 705.04729,768.63555 704.74629,769.09555 705.54229,769.80655 C 707.08629,771.18755 708.22329,770.45255 708.76129,772.80955 C 709.17829,774.63555 709.10029,775.07455 711.04729,775.65755 C 713.14529,776.28555 711.16429,778.73355 712.23529,779.95355 C 712.81629,780.61455 713.72229,780.83955 714.57029,780.80455 C 715.09929,780.73355 715.63129,780.68655 716.16529,780.66255 C 716.87329,780.66455 717.70029,781.08055 718.32829,781.05855 C 719.77129,781.00455 721.48029,779.96455 722.10629,778.70555 C 722.76729,777.37655 724.59329,779.91755 725.38329,779.97855 C 726.57429,780.07055 728.48456,778.20501 729.33256,778.82601 C 730.00956,779.32201 730.53529,780.21655 731.18729,780.66355 C 731.81329,781.09355 733.28529,780.97655 733.67029,781.38555 C 735.24429,783.05555 735.26829,783.65755 734.03829,785.60555 C 733.05329,787.16555 731.13929,789.83455 730.86629,791.64955 C 730.53929,793.82055 733.04029,792.12055 733.97929,792.94055 C 734.73729,793.60155 735.29678,794.50945 735.86429,795.33055 C 736.13728,795.72553 736.44813,795.91015 736.31122,796.23756 C 736.22094,796.93804 735.60729,797.17647 735.18029,796.38855 C 734.98729,796.62855 734.79429,796.86855 734.60029,797.10855 C 733.47829,795.58455 730.99129,794.59255 729.57629,796.37955 C 727.96329,798.41755 724.45129,797.87955 722.13329,798.49355 C 720.91929,798.81555 720.33229,800.16355 719.12929,800.53855 C 717.86529,800.93355 718.27629,801.44055 717.43429,802.19855 C 715.41629,804.01755 714.01829,803.49555 713.98829,806.61355 C 713.97729,807.69455 714.04329,808.60055 713.41029,809.54155 C 712.80729,810.43655 712.08729,811.33355 712.32729,812.48055 C 712.57629,813.67055 713.60229,815.36255 712.27229,816.28055 C 711.15329,817.05155 710.09529,817.44555 709.17829,818.52055 C 708.50929,819.30455 708.16129,820.19555 708.37529,821.22855 C 708.49029,821.72855 708.66429,822.20955 708.89629,822.66855 C 708.33729,822.91155 707.75629,823.21655 707.33529,823.66955 C 705.71129,825.41755 710.50329,826.93455 708.53629,828.13755 C 709.53929,828.79255 710.23329,829.98855 710.44529,831.15255 C 710.56929,831.83355 711.03829,832.07055 711.53029,832.46755 C 712.14629,832.96355 712.08829,833.71955 712.49529,834.33755 C 713.21029,835.42055 714.58829,835.93055 715.09029,837.18455 C 715.68929,838.67955 714.54029,839.59255 713.71929,840.66755 C 715.49629,841.59755 718.10829,840.76155 719.87329,841.93455 C 721.01929,842.69555 722.07729,844.67355 721.63929,846.06755 C 724.02629,846.36755 727.08629,846.88455 729.20929,845.43155 C 730.54929,844.51455 732.29629,841.19955 734.10029,841.17755 C 733.88929,842.04455 733.21529,842.08855 733.74529,842.98455 C 734.30729,843.93555 733.36729,844.88455 733.42729,845.97055 C 733.58129,848.79455 733.67829,851.69055 734.02029,854.49755 C 734.94729,853.68455 735.57429,855.34655 736.54029,855.25255 C 737.87229,855.12255 738.80594,854.07423 740.17529,854.08555 C 741.48129,854.09635 742.42716,853.9474 742.80729,855.23755 C 743.38531,857.1993 744.77029,858.81655 745.76429,860.60555 C 746.76529,862.40655 747.76235,864.21041 748.79229,865.99555 C 749.68825,867.54847 750.48235,868.52623 749.0131,869.62574 C 747.71752,870.59528 747.5645,872.34475 748.03629,873.74055 C 748.4436,874.94559 748.14507,876.55626 748.21329,878.72892 C 748.24271,879.66575 749.38329,880.85655 749.42429,881.82855 C 749.42629,881.88855 748.59129,884.19455 748.52429,884.25555 C 747.64829,885.04755 747.12783,886.48309 746.72083,887.58809 C 745.97683,889.60709 747.96829,889.77855 747.97329,891.34755 C 747.97629,892.46155 747.24338,892.92128 747.03538,894.00128 C 745.86651,893.20041 743.24738,890.47412 742.74681,892.5042 C 742.40169,893.90384 743.04755,893.42414 743.19709,895.17686 C 743.27809,895.37686 742.02907,894.02308 742.53106,894.44853 C 742.51849,894.74534 740.86179,896.22205 743.81851,897.26842 C 743.92383,895.8663 745.62872,898.12678 746.42421,897.90055 C 746.87964,897.77103 746.64176,898.40953 746.55181,899.47353 C 747.46167,899.73676 749.0775,900.18287 749.22548,899.60919 C 749.34648,899.63119 750.0712,899.72546 750.27587,899.8804 C 749.69122,903.09688 750.09538,902.38149 750.5333,902.74325 C 750.21012,903.41125 750.67178,903.25889 750.21093,903.85691 C 749.41889,904.88467 748.70629,906.44855 747.75329,907.39555 C 747.33798,907.80824 746.76332,908.02659 746.43529,908.52955 C 745.83332,909.45253 747.08174,909.02182 747.24329,909.79455 C 747.45047,910.78553 747.84866,910.75074 747.49553,911.19414 C 741.76191,918.39354 745.57156,912.17819 745.33311,912.68064 C 744.7971,913.81011 746.19429,915.25955 745.43829,916.55555 C 744.78729,917.67155 743.41229,918.06655 742.19729,918.06655 C 740.83329,918.06655 739.78329,917.04755 738.85829,916.17555 C 737.76729,915.14855 736.57329,914.66755 735.39329,913.83155 C 733.29929,912.34855 733.45629,909.65055 730.94629,908.62055 C 727.97529,907.40055 725.97329,904.49855 722.82829,903.72455 C 721.35629,903.36255 719.84429,902.34555 718.56529,901.54955 C 717.03429,900.59755 715.37829,900.27855 713.81829,899.43755 C 712.29129,898.61255 710.99529,897.35255 709.46429,896.48055 C 707.74229,895.50055 706.02729,894.53955 704.28829,893.58655 C 702.81929,892.78155 701.93429,890.93555 700.77829,889.74655 C 699.24629,888.17055 697.08729,887.34255 696.01629,885.38655 C 695.41229,884.28255 694.64329,882.95655 693.58329,882.29155 C 692.65629,881.70955 693.88729,879.48555 692.26329,879.76555 C 692.28229,879.48455 692.35329,879.21755 692.47929,878.96555 C 692.72129,879.10955 692.96129,879.25655 693.19929,879.40555 C 693.83929,876.20755 692.45629,874.54955 690.85329,871.92955 C 689.39729,869.54955 688.40829,867.03255 686.36729,865.09755 C 685.75529,864.51655 685.13029,864.21055 685.29729,863.41555 C 685.48929,862.50055 684.95329,861.47355 684.51629,860.71155 C 684.08729,859.96255 683.56829,859.15955 682.90329,858.59455 C 682.29129,858.07355 681.24629,857.82655 680.85729,857.06855 C 680.80929,856.67155 680.84529,856.27755 680.96429,855.88855 C 680.84329,855.02555 680.29629,854.03755 679.90329,853.27355 C 679.15029,851.80855 678.12229,850.51055 677.28729,849.09455 C 676.50029,847.76055 676.05829,846.16255 675.50129,844.72555 C 674.20129,841.36955 672.10529,838.78555 671.16329,835.54055 C 669.87629,831.11155 666.11529,828.41555 664.40829,824.04055 C 662.45529,819.03655 658.41729,815.47055 653.52129,813.40655 C 653.09029,813.22455 651.47529,812.83655 651.42029,812.23955 C 651.34429,811.40755 651.16929,810.01955 652.36429,810.16755 C 654.49029,810.43155 653.68329,808.42155 652.99529,807.37455 C 652.02429,805.89755 651.05829,805.28055 651.00729,803.44355 C 651.20429,803.20755 651.39629,802.96755 651.58329,802.72355 C 651.54729,802.44355 650.69829,801.66955 650.50929,801.45555 C 649.51529,800.33555 650.12329,799.67555 649.70729,798.40855" id="pe"/>
  <path class="landxx bf" d="M 1228.8763,669.23755 C 1229.4103,667.85455 1229.8914,666.25336 1229.5714,664.77436 C 1229.2214,663.15736 1230.1951,664.12296 1230.8121,662.91096 C 1231.1931,662.16096 1232.0352,659.87926 1231.1498,659.1943 C 1230.554,658.73335 1231.738,657.69213 1229.9573,657.21455 C 1230.7903,657.25655 1231.312,656.60555 1231.945,656.21755 C 1232.979,655.58555 1234.4589,656.10429 1235.5261,655.45255 C 1236.8311,654.65555 1238.4076,651.56095 1237.2553,650.20055 C 1236.706,649.55201 1237.5062,649.11907 1238.2243,649.20255 C 1239.315,649.32935 1239.2273,648.34255 1239.0653,647.48255 C 1238.9583,646.92055 1237.8463,646.08555 1238.8133,645.74855 C 1239.5573,645.48855 1240.5813,643.38555 1241.0463,642.74155 C 1240.9863,644.00755 1243.9723,645.51955 1245.0063,645.62255 C 1244.7623,644.42155 1245.5423,644.58055 1246.5163,644.61455 C 1246.3853,643.51655 1246.9333,642.02355 1246.3663,641.01355 C 1247.1773,641.16855 1248.0183,641.22555 1248.7463,641.66155 C 1249.8383,640.02355 1249.7053,636.42755 1252.5403,636.13455 C 1253.9373,635.99055 1255.6686,638.20561 1255.9436,635.72961 C 1256.0906,634.40161 1257.5573,634.57255 1258.4663,634.10055 C 1259.6443,633.48955 1260.5472,632.59777 1261.7753,632.01455 C 1263.249,631.31472 1265.9403,627.68455 1267.9993,629.54155 C 1268.39,629.8939 1269.3453,629.07455 1269.9753,629.26555 C 1270.8383,629.52655 1271.7065,629.82631 1272.5183,630.21755 C 1273.7645,630.81816 1271.4716,633.99044 1273.3823,635.03055 C 1274.5625,635.67302 1273.636,637.6749 1274.5213,638.56555 C 1275.0213,639.06855 1276.0337,640.11409 1275.8659,640.77809 C 1276.8052,641.15136 1279.2739,642.81141 1280.8563,644.03755 C 1280.1423,644.38655 1279.5003,644.36855 1278.8463,643.89255 C 1277.8716,646.38055 1281.1483,649.26655 1283.1263,650.20755 C 1284.7152,650.96346 1285.6089,649.06961 1286.7399,649.39861 C 1287.6699,649.66861 1287.8493,651.13955 1288.2063,651.88455 C 1285.6765,652.7547 1287.7883,654.20363 1288.7202,655.94355 C 1289.7132,657.79749 1287.3153,660.48755 1285.4983,660.60555 C 1285.0493,660.63455 1283.3033,661.13955 1282.9783,660.87955 C 1282.0833,660.16555 1281.8843,660.52955 1281.1403,661.19955 C 1279.6713,662.52155 1278.8382,664.25836 1276.9543,664.52755 C 1274.0522,664.94224 1273.5357,664.02316 1271.8222,663.70368 C 1271.0019,663.55072 1270.2251,663.27764 1269.3704,663.26649 C 1268.613,663.25661 1266.6505,665.20901 1266.0903,664.96455 C 1264.734,664.37263 1267.722,664.32043 1257.5181,664.35336 C 1254.1312,664.36429 1252.1229,664.59394 1249.3953,664.34055 C 1249.7283,665.90133 1248.2266,666.70624 1248.6504,667.65305 C 1249.226,668.93897 1249.5093,669.95843 1249.7384,671.30196 C 1249.9332,672.44392 1250.0438,673.86954 1249.9083,675.15355 C 1249.7895,676.27919 1250.6663,677.63455 1249.8943,678.23655 C 1248.6343,677.29655 1247.4373,675.03855 1246.5843,673.70155 C 1245.8633,674.77455 1242.9583,673.48955 1241.8343,673.91755 C 1240.6053,674.38555 1238.0323,674.76255 1238.0143,676.50955 C 1237.2383,676.37655 1234.5156,675.23644 1234.0391,674.64639 C 1233.5942,674.09553 1233.126,673.81479 1233.0413,673.37155 C 1232.8774,672.51371 1232.0833,670.87155 1231.4633,670.53355 C 1230.4193,669.9644 1229.2573,670.79655 1228.8763,669.23755" id="bf"/>
  <g id="fr">
    <g class="landxx coastxx fr frx eu" id="frx">
      <path d="M 1258.5363,351.42953 C 1258.2153,351.09353 1257.8553,350.80453 1257.4563,350.56553 C 1257.5503,351.13153 1257.9293,351.52653 1258.5363,351.42953" id="path2166"/>
      <path d="M 1264.7363,354.66953 C 1264.5373,354.21053 1264.5403,354.04153 1263.9363,354.16553 C 1264.2023,354.33353 1264.4693,354.50153 1264.7363,354.66953" id="path2168"/>
      <path d="M 1270.2063,361.00553 C 1269.9433,360.43353 1269.1693,360.55753 1268.6963,360.64553 C 1269.2233,360.66353 1269.7273,360.78353 1270.2063,361.00553" id="path2172"/>
      <path d="M 1270.4863,363.95753 C 1270.9753,363.25253 1270.4283,362.04653 1269.4863,362.51753 C 1269.9373,362.91553 1270.2713,363.39553 1270.4863,363.95753" id="path2174"/>
      <path class="landxx fr frx eu" d="M 1247.9563,345.09353 C 1248.8483,344.67153 1249.9183,344.40753 1250.9063,344.58953 C 1251.2503,343.07553 1249.2373,344.11653 1248.8163,343.07753 C 1249.7423,342.77053 1250.8213,342.88353 1251.7763,342.93253 C 1251.0263,342.80353 1250.0103,343.07053 1250.5463,342.14053 C 1249.9683,342.10953 1248.0023,343.21953 1247.7643,342.24853 C 1247.4403,340.92353 1249.4573,340.12353 1250.4183,339.93353 C 1251.7103,339.67753 1252.9283,339.57353 1254.2383,339.60953 C 1256.0933,339.66053 1257.2803,337.85953 1259.3353,338.10853 C 1259.1023,338.77353 1261.5713,341.09553 1262.2613,340.81653 C 1262.7253,340.59053 1263.1723,340.33253 1263.6013,340.04453 C 1264.5383,339.52353 1264.4763,340.19653 1265.2593,340.34653 C 1266.0333,340.49453 1266.6253,339.88453 1266.6763,341.13353 C 1267.1823,340.78653 1266.4383,340.37153 1266.8543,339.80653 C 1267.2793,339.22953 1267.4953,340.10553 1267.8683,340.20353 C 1268.7543,340.43553 1269.8883,340.19953 1270.7763,340.05353 C 1268.8903,339.50053 1269.5823,336.83453 1269.7363,335.44553 C 1269.8113,334.76953 1268.7883,334.09153 1268.4413,333.61953 C 1267.8433,332.80653 1268.3953,331.57053 1267.5363,330.90853 C 1268.1903,331.02253 1268.7863,331.34253 1269.4353,331.47753 C 1270.1453,331.62553 1271.0223,330.92853 1271.6453,331.24253 C 1272.6163,331.73053 1271.4053,333.19653 1272.7503,333.96653 C 1273.1383,334.18853 1273.8353,333.79353 1274.2353,333.78953 C 1275.4053,333.77853 1276.5303,334.17453 1277.6673,334.39053 C 1278.7783,334.60153 1279.7453,334.56953 1280.7413,334.00053 C 1281.6443,333.48553 1282.8513,333.50453 1283.6653,332.92553 C 1283.0193,333.18153 1281.9213,333.31153 1281.2923,332.95153 C 1280.0173,332.22053 1282.9563,330.42253 1283.4443,330.20353 C 1284.9943,329.50953 1286.7483,329.59253 1288.3213,328.96253 C 1289.2343,328.59753 1290.4503,326.50953 1291.5843,327.09353 C 1290.5913,326.45453 1291.1983,322.02953 1292.0223,321.20053 C 1292.8333,320.38653 1296.9643,318.91453 1298.0263,319.80953 C 1298.5673,320.26553 1297.5066,322.01981 1299.0912,322.41093 C 1300.7239,322.81393 1300.3433,322.44623 1301.4983,322.79423 C 1302.5753,323.11923 1302.2375,323.9499 1303.0715,324.3539 C 1304.1595,324.8809 1305.2593,324.25553 1305.1233,325.93953 C 1305.7713,325.52053 1306.8953,325.61753 1307.5843,325.85453 C 1308.8783,326.29853 1308.3323,327.89353 1308.0723,328.81953 C 1308.9373,328.96453 1309.9183,329.23353 1310.7793,328.88953 C 1311.6313,328.54853 1311.8263,327.57553 1312.6823,327.23553 C 1312.7563,327.77153 1312.5323,330.16853 1313.2583,330.19453 C 1314.3293,330.23253 1316.4853,331.64553 1316.8183,332.67253 C 1316.7223,332.37653 1320.8893,333.05053 1321.1033,333.06853 C 1322.1163,333.15553 1322.8723,332.94653 1323.6753,333.78353 C 1324.3073,334.44353 1324.7523,335.48053 1325.7133,335.73153 C 1325.9193,335.63353 1326.0843,335.48753 1326.2083,335.29353 C 1326.5183,334.97253 1326.6743,335.39853 1327.0033,335.44353 C 1327.1873,335.51353 1327.2363,335.63953 1327.1503,335.82053 C 1327.0933,336.21253 1328.0113,335.99853 1328.1633,336.01153 C 1329.1863,336.10453 1329.7743,335.47353 1330.7213,336.19453 C 1331.7293,336.96253 1333.8123,337.10453 1335.0733,337.31553 C 1333.5263,338.68053 1332.0983,340.17853 1331.5983,342.20653 C 1331.3553,343.19253 1330.7683,344.09853 1330.6133,345.09253 C 1330.5113,345.74253 1330.7323,346.00653 1330.4713,346.84153 C 1330.1573,347.85053 1330.9253,348.69153 1330.2463,349.31753 C 1329.6743,349.84553 1329.2323,350.35453 1328.3723,350.27753 C 1327.6513,350.21353 1327.5903,349.14653 1326.7223,349.62953 C 1326.0953,349.97853 1326.0353,351.09253 1327.0823,350.92553 C 1326.2953,351.75153 1325.6003,352.73253 1324.7023,353.44553 C 1324.1523,353.80353 1323.6143,354.17953 1323.0883,354.57353 C 1322.6553,354.96353 1322.6973,355.72753 1322.3323,356.18153 C 1321.7093,356.95553 1320.5023,357.31953 1320.3123,358.41353 C 1320.4523,358.78653 1320.5483,359.17053 1320.6023,359.56453 C 1320.4533,360.27353 1319.7943,360.52653 1319.5223,361.14853 C 1320.7103,360.99953 1321.1463,360.59353 1321.4613,359.45753 C 1321.7723,358.33953 1323.8323,358.42553 1324.6653,358.66053 C 1325.6653,358.94353 1324.9863,360.58753 1325.7143,361.38853 C 1326.9783,362.78053 1326.2003,363.30253 1325.0623,364.24553 C 1325.6653,365.10253 1327.9443,366.69053 1327.3153,367.94653 C 1326.8933,368.78953 1323.7483,369.94653 1323.7563,370.09053 C 1323.8123,371.08453 1325.7303,371.93153 1326.3923,372.49853 C 1327.3473,373.31853 1325.0603,374.77953 1325.3913,375.86353 C 1325.7923,377.17953 1328.3083,378.67153 1329.6393,378.13853 C 1331.3113,377.46953 1331.0613,378.44253 1330.4293,379.55553 C 1328.9063,382.23653 1325.9973,383.58853 1323.4023,385.34153 C 1323.5663,385.58553 1323.7363,385.82553 1323.9123,386.06153 C 1322.9433,386.20053 1322.1323,386.69853 1321.2243,387.01653 C 1320.2473,387.35953 1319.6233,386.81953 1318.7013,386.89253 C 1317.3023,387.00253 1316.4443,385.83653 1314.7713,385.98953 C 1315.4423,384.25853 1312.5743,385.58553 1312.5423,384.57953 C 1312.7243,384.26753 1313.0103,384.18553 1313.4003,384.33353 C 1313.3703,383.76653 1313.0303,383.44653 1312.4603,383.46953 C 1312.9383,384.20353 1312.5263,384.31153 1311.8663,384.34453 C 1310.7833,384.39853 1310.6663,384.33453 1310.3703,383.03753 C 1310.7613,385.62053 1310.1013,383.97353 1308.4303,383.97353 C 1307.4643,383.97353 1306.6743,383.36353 1305.7313,383.31753 C 1304.5753,383.26053 1303.3083,384.56653 1302.2813,385.03653 C 1301.4813,385.40353 1298.6623,386.44753 1298.8393,387.53653 C 1298.8203,387.89353 1298.7143,388.22553 1298.5183,388.53253 C 1298.0693,388.93253 1298.1073,389.28853 1298.6333,389.60053 C 1299.0293,390.85253 1298.7703,391.47953 1299.4993,392.68653 C 1298.8123,392.88353 1298.2133,392.53753 1297.5333,392.57853 C 1296.8273,392.62153 1296.4693,393.18353 1295.8393,393.35553 C 1294.3183,393.77053 1293.3913,392.19553 1291.9343,393.25653 C 1291.8323,393.33053 1289.8063,392.38453 1289.5883,392.22153 C 1289.0223,391.79853 1290.3173,391.85253 1289.4053,391.24353 C 1288.8773,390.89053 1287.9263,390.73553 1287.4783,391.31753 C 1287.2703,390.02853 1285.8433,390.56753 1285.0673,389.95553 C 1283.9973,389.11053 1281.8083,388.56853 1282.1483,390.59853 C 1280.9513,390.42553 1279.6003,390.12353 1278.3933,390.34553 C 1277.7223,390.46853 1277.2743,390.65853 1276.7013,390.14253 C 1275.6843,389.22853 1275.7613,389.60453 1274.7283,389.59053 C 1273.7043,389.57653 1268.7553,386.33053 1268.2483,387.64653 C 1267.3133,387.26053 1267.9773,386.58053 1268.0363,385.92553 C 1268.1063,385.14653 1265.8413,385.50153 1265.3683,384.91053 C 1268.2283,383.85153 1268.4703,379.88753 1268.9913,377.30953 C 1269.1963,376.29753 1269.3813,374.70253 1270.2223,373.98053 C 1271.4463,372.92853 1269.8043,372.78053 1269.6883,373.96653 C 1269.5183,371.53053 1270.0083,368.54553 1271.0583,366.33453 C 1272.0693,366.90353 1273.0753,367.57853 1273.5083,368.71053 C 1273.6063,369.21453 1273.7023,369.71853 1273.7983,370.22253 C 1274.0323,370.86353 1274.6733,370.95153 1274.6583,371.73453 C 1274.9483,370.72753 1273.9263,369.95253 1273.8333,368.99853 C 1273.6893,367.53253 1273.1743,366.53953 1271.8743,365.82653 C 1271.3293,365.52853 1270.7213,365.12153 1270.2673,364.68453 C 1269.5283,363.97253 1271.4573,364.32753 1271.8493,364.67953 C 1270.9413,364.23853 1270.8923,360.38553 1271.4193,359.63953 C 1270.2263,360.47453 1267.4193,358.83153 1266.8663,357.72453 C 1266.3773,356.74553 1265.2743,356.23653 1265.0533,355.09353 C 1264.8503,354.05053 1266.7633,352.90853 1264.3603,352.80053 C 1264.6433,352.46753 1264.6983,351.86253 1265.0633,351.60853 C 1265.9773,350.97053 1266.8193,352.04353 1267.6803,352.15253 C 1266.1523,351.14353 1265.5453,351.47853 1263.9513,351.68653 C 1263.1483,351.79153 1261.5923,350.60653 1263.0713,350.42453 C 1262.6583,350.01753 1262.8253,349.75353 1263.5713,349.63253 C 1262.4363,349.35053 1260.8233,350.27453 1259.9013,349.27253 C 1260.4093,349.26553 1260.8893,349.14553 1261.3413,348.91253 C 1260.4373,348.30853 1258.7713,348.55353 1258.5313,349.77653 C 1258.4983,348.97653 1257.6113,348.22853 1258.4613,347.54553 C 1258.1123,348.42153 1256.5413,348.36453 1257.5213,347.25753 C 1255.5273,348.98753 1252.8533,345.04853 1250.8313,347.18553 C 1250.0233,346.06853 1249.3483,345.38253 1247.9563,345.09353" id="path2182"/>
      <path d="M 1337.1663,393.11753 C 1337.5793,392.70253 1337.6613,391.86053 1338.2123,391.58353 C 1339.3123,391.03353 1340.5553,390.89153 1341.5973,390.19853 C 1342.5273,389.57953 1341.8713,387.83453 1343.1363,387.93253 C 1343.2883,390.12853 1344.3433,393.29353 1343.8213,395.42053 C 1343.2723,397.66453 1342.1673,399.80953 1341.1963,401.89953 C 1340.8193,401.30653 1337.2893,400.01753 1339.1763,399.23553 C 1338.6673,399.11653 1338.1873,398.92453 1337.7363,398.66053 C 1338.2333,398.34853 1338.5823,397.82753 1338.5263,397.22053 C 1337.5213,397.45753 1337.3763,396.33353 1337.9563,395.78153 C 1337.0983,395.56553 1336.5583,394.43953 1337.7363,394.19753 C 1337.4923,393.86553 1337.3023,393.50653 1337.1663,393.11753" id="path2184"/>
    </g>
    <g id="gp">
      <g class="landxx coastxx fr gp eu" id="gp-">
        <path d="M 806.52628,618.83753 C 806.18028,619.42153 805.64428,623.09153 804.73928,620.59553 C 804.23028,619.19053 805.02728,617.80753 806.52628,618.83753" id="path2582"/>
        <path d="M 806.52628,618.69353 C 807.15628,618.12053 806.51128,617.00153 807.53628,616.82153 C 807.67228,617.73753 808.43428,618.39553 809.18528,618.83853 C 808.47828,619.08253 806.90828,619.80353 806.52628,618.69353" id="path2584"/>
        <path d="M 808.39528,622.22153 C 807.78728,621.59453 807.90728,621.28353 808.75528,621.28553 C 808.87628,621.69553 808.75628,622.00753 808.39528,622.22153" id="path2586"/>
      </g>
      <circle class="circlexx subxx fr gp eu" cx="809.38892" cy="615.23084" id="gp_" r="4.2338"/>
    </g>
    <g id="mq">
      <path class="landxx coastxx fr mq eu" d="M 810.33629,634.96555 C 809.75329,634.07255 809.03629,634.83255 808.75229,633.98655 C 808.47729,633.16755 808.10629,632.28055 808.09529,631.40355 C 808.08129,630.25455 809.27629,631.25955 809.58929,631.65355 C 810.19929,632.42055 811.05129,634.01455 810.33629,634.96555" id="mq-"/>
      <circle class="circlexx subxx fr mq eu" cx="813.59498" cy="635.36891" id="mq_" r="4.2338"/>
    </g>
    <g id="re">
      <path class="landxx coastxx fr re eu" d="M 1689.4563,941.25855 C 1689.3503,939.27055 1685.4733,940.44855 1685.9143,942.34855 C 1686.5793,945.21455 1691.9433,945.44655 1689.4563,941.25855" id="re-"/>
      <circle class="circlexx subxx fr re eu" cx="1688.1991" cy="941.7735" id="re_" r="4.2338"/>
    </g>
    <g id="yt">
      <path class="landxx coastxx fr yt eu" d="M 1613.4263,871.98855 C 1613.9023,871.34555 1614.5063,869.97055 1613.4263,869.53855 C 1613.1543,870.10155 1613.0153,871.45555 1613.4263,871.98855" id="yt-"/>
      <circle class="circlexx subxx fr yt eu" cx="1614.6058" cy="872.183025" id="yt." r="4.2338"/>
    </g>
    <path class="landxx fr gf eu" d="M 853.32629,739.43755 C 854.61429,739.37755 855.64229,737.16955 856.08129,736.20055 C 856.89929,734.39155 856.09329,732.37855 857.32729,730.73055 C 858.79029,728.77755 857.44629,727.33755 856.20029,725.78255 C 854.90829,724.16955 854.96029,721.28955 854.83629,719.32155 C 854.69229,717.02855 855.02629,715.72555 856.69929,714.12855 C 857.32229,713.53355 858.09829,712.87655 858.29829,711.99755 C 858.55729,710.85855 858.85129,709.43555 860.46629,710.04555 C 861.51429,710.44155 862.23329,711.28955 863.34729,711.59455 C 864.55029,711.92555 866.01129,711.90655 867.02929,712.72055 C 868.22329,713.67755 869.81029,717.32255 871.68529,717.18955 C 871.37029,717.65155 871.15429,718.15555 871.03629,718.70255 C 871.56829,716.30655 875.67329,720.41755 873.33629,722.01455 C 874.46829,721.59755 873.81029,720.53955 874.37429,719.78555 C 875.23029,718.63955 875.89529,721.31055 875.96129,721.87955 C 876.27329,724.57755 874.20329,727.43355 872.80029,729.58155 C 871.18829,732.04955 870.06729,734.13855 869.01629,736.91955 C 868.62629,737.95155 868.12329,738.82455 867.25629,739.52655 C 866.89629,739.81855 866.25129,740.49055 865.77729,740.61555 C 864.99629,740.82255 864.38129,740.19755 863.54629,740.59155 C 863.89029,739.61055 863.27529,739.60655 862.53829,739.83455 C 861.50829,740.15355 860.62829,739.73455 859.64929,739.40355 C 859.15529,739.23655 857.23429,740.62855 856.92729,740.99355 C 856.35729,741.67055 853.70329,739.83455 853.32629,739.43755" id="gf"/>
  </g>
  <path class="landxx ly" d="M 1342.4163,499.67755 C 1341.4533,498.84855 1343.5153,497.87255 1344.0483,497.62555 C 1345.6523,496.88055 1346.5173,496.45355 1347.4893,494.87755 C 1348.2703,493.61055 1349.054,492.99261 1348.749,491.36061 C 1348.486,489.95361 1347.8193,488.53555 1347.9633,487.06755 C 1348.1613,485.05455 1350.6043,485.57026 1350.9943,484.12711 C 1351.4147,482.57157 1352.6828,481.7702 1353.9833,481.15155 C 1354.98,480.67741 1358.2173,479.79355 1358.2433,478.41555 C 1358.2563,477.72555 1357.5803,477.29855 1357.6523,476.56555 C 1357.7823,475.22355 1357.8343,473.87555 1357.9673,472.53355 C 1359.2603,473.43355 1360.9953,473.88855 1362.3843,474.74255 C 1363.8423,475.63855 1365.4973,475.94355 1367.1933,475.75755 C 1368.9093,475.56955 1370.1723,474.48255 1371.9373,475.12055 C 1373.5093,475.68955 1375.1743,475.74855 1376.7043,476.40055 C 1378.0663,476.98155 1379.1653,478.03055 1380.5393,478.56855 C 1381.7413,479.03955 1383.3643,478.54655 1384.4363,479.28755 C 1385.6993,480.16055 1385.9353,481.65855 1386.1783,483.06055 C 1386.4893,484.85455 1387.1933,486.28455 1388.5913,487.48955 C 1391.1163,489.66555 1394.2353,489.15555 1397.2723,489.70655 C 1400.3213,490.26055 1404.6613,491.64555 1407.1033,493.62355 C 1409.0553,495.20555 1410.8613,497.87655 1413.6523,497.62555 C 1416.8773,497.33455 1420.3253,494.49155 1421.0473,491.25355 C 1421.3803,489.75855 1420.3693,488.59555 1419.9113,487.24855 C 1419.4053,485.76355 1419.0353,484.05655 1419.5443,482.51955 C 1420.4783,479.69955 1425.1213,476.20155 1428.0653,476.10255 C 1429.5893,476.05155 1430.4793,474.94855 1431.8893,474.84055 C 1433.3273,474.73055 1434.8613,474.70455 1436.2653,475.04855 C 1437.9073,475.44955 1439.3393,476.24355 1440.8963,476.85355 C 1442.7423,477.57655 1442.2533,477.82755 1442.3443,479.44355 C 1442.4163,480.75255 1444.5303,481.02955 1445.4733,481.09955 C 1447.3033,481.23755 1448.4873,482.06655 1450.1783,482.55655 C 1451.8573,483.04355 1453.6193,482.48055 1455.3233,482.79255 C 1456.2893,482.96855 1457.7453,484.89455 1457.4013,485.90655 C 1457.0393,486.97455 1455.7363,487.24655 1455.5553,488.44855 C 1455.3043,490.10855 1456.3643,491.33955 1456.6563,492.88355 C 1456.9843,494.62055 1455.8645,496.15946 1455.2745,497.66946 C 1454.6475,499.27146 1456.2891,501.17437 1456.5621,502.83237 C 1457.0251,505.65837 1457.5289,508.32982 1457.7059,511.22782 L 1459.4893,586.50855 C 1456.9963,586.50855 1454.5023,586.50855 1452.0093,586.50855 C 1452.0783,587.94755 1452.0243,589.39155 1452.1493,590.82855 L 1392.498,557.06786 L 1378.9703,563.54955 C 1377.8243,563.86255 1376.42,561.99255 1375.752,561.32755 C 1372.96,558.54455 1369.9293,557.97555 1366.2113,557.13055 C 1364.8833,556.82855 1363.1463,556.67455 1361.9633,555.94055 C 1361.0263,555.35855 1360.583,553.71742 1360.195,552.76742 C 1359.056,549.97642 1357.669,549.21693 1354.937,548.18893 C 1354.03,547.84793 1353.1623,547.07755 1352.1693,547.10355 C 1351.1973,547.12855 1350.1483,548.20755 1349.2363,547.22155 C 1348.4953,546.42155 1347.2153,544.36555 1347.1963,543.25855 C 1347.1563,540.94255 1346.9323,540.09655 1345.6583,538.12455 C 1344.6773,536.60455 1341.4153,533.90455 1342.8193,531.88155 C 1344.4393,529.54755 1346.9173,530.44055 1345.8553,526.61255 C 1345.3243,524.69755 1344.9423,523.22655 1345.5983,521.28355 C 1346.1603,519.61855 1346.4933,518.24455 1345.8783,516.52355 C 1345.2323,514.71355 1345.7343,512.90155 1345.8963,511.06255 C 1346.0883,508.88255 1345.7623,506.68355 1345.0383,504.62455 C 1344.4143,502.85855 1343.5113,501.19555 1342.4163,499.67755" id="ly"/>
  <path class="landxx by" d="M 1433.1368,309.64261 C 1433.6053,308.40487 1434.2651,307.23347 1435.7213,306.96855 C 1437.1513,306.70839 1437.5793,305.61555 1437.4223,304.28655 C 1437.0533,301.16755 1434.3533,298.99255 1433.8563,295.91755 C 1435.4893,296.33455 1437.5963,296.03555 1439.3013,296.23055 C 1439.8183,296.28955 1440.5933,295.89355 1441.0933,295.76855 C 1442.1003,295.51655 1441.8443,295.05455 1442.5693,294.44255 C 1443.0163,294.06455 1443.5743,294.42655 1444.0503,294.19755 C 1444.4223,294.01955 1444.6023,293.60255 1444.9963,293.41955 C 1445.7503,293.06955 1447.3013,292.96855 1446.8153,294.26155 C 1447.2373,294.39855 1447.9083,294.35355 1448.1773,293.93455 C 1448.7573,293.03455 1447.1413,292.72655 1446.6753,292.53455 C 1448.3173,290.75055 1447.0363,288.84755 1449.3353,287.63855 C 1450.4033,287.07655 1450.5163,285.92955 1451.9313,286.05555 C 1453.4693,286.19255 1452.7173,285.39755 1453.8753,284.82955 C 1452.2673,284.40155 1450.6289,283.70242 1452.2909,281.78142 C 1453.3519,280.55542 1455.6833,280.19455 1457.1613,280.63855 C 1458.9433,281.17355 1458.9953,278.76755 1460.6293,278.16755 C 1461.4953,277.84855 1462.8229,278.75542 1463.2619,278.40442 C 1464.8843,278.25908 1469.4212,278.46832 1469.7944,279.59402 C 1470.1704,279.93802 1469.5613,280.67855 1469.9903,281.09655 C 1470.5663,281.65955 1472.5753,280.88152 1473.1843,280.58952 C 1475.3233,279.56252 1476.15,280.14005 1478.0696,281.49877 C 1478.5907,281.86757 1479.9677,282.01478 1479.9253,283.11355 C 1479.8743,284.43688 1479.8624,284.72249 1480.5133,285.91655 C 1480.9423,286.70355 1481.1514,286.96817 1480.3651,287.53611 C 1479.2975,288.30729 1481.0306,289.44205 1481.6703,290.05955 C 1483.0433,291.38499 1483.4863,293.49854 1485.6503,294.24755 C 1486.2728,294.46301 1487.2924,294.66276 1487.5674,295.36555 C 1487.9224,296.27267 1486.879,297.17394 1487.7901,297.17489 C 1488.1631,297.12289 1488.6593,297.13955 1489.0103,296.99955 C 1489.5073,296.89755 1490.0938,297.20667 1490.5922,297.26942 C 1492.2644,297.47996 1491.2361,297.85977 1491.9156,298.63358 C 1492.2696,299.03663 1493.3034,299.29155 1493.7774,299.92561 C 1494.7611,301.24147 1493.217,301.35371 1492.773,301.64771 C 1491.977,302.17371 1491.7303,302.52055 1490.5003,302.82055 C 1488.9673,303.19355 1488.0964,301.81633 1486.7551,302.0964 C 1483.4196,302.7929 1485.5375,304.08451 1486.7071,304.97055 C 1487.4513,305.53432 1487.2973,306.69955 1487.4963,307.51055 C 1487.8223,308.83955 1488.8237,309.94102 1489.5407,311.08802 C 1487.9241,311.09518 1488.5962,310.68452 1485.7528,311.58406 C 1484.5722,310.90192 1482.5897,312.09447 1481.9514,314.15721 C 1481.6693,315.06905 1482.2177,315.74926 1482.3637,317.1783 C 1482.5276,318.78271 1480.5286,317.40177 1480.1243,316.91271 C 1478.9522,315.49516 1477.745,317.15056 1476.3163,316.34168 C 1473.0119,318.14677 1473.904,315.86742 1472.363,315.3768 C 1471.445,314.8958 1469.9413,315.80955 1470.0753,316.87155 C 1469.5263,316.62355 1469.5033,315.78755 1468.9953,315.57655 C 1468.4683,315.35655 1467.8023,315.97255 1467.3493,315.64155 C 1466.7113,315.17655 1466.0663,315.21355 1465.3293,315.29655 C 1464.3313,315.40855 1463.8663,314.73355 1463.7343,315.93455 C 1463.1133,315.77655 1462.6631,315.20922 1462.0233,315.14255 C 1461.3043,315.06764 1460.4951,315.70266 1459.9186,314.90708 C 1458.694,313.21694 1456.8468,313.89098 1455.1223,313.53086 C 1453.358,313.16243 1451.3143,312.2891 1449.5213,312.60955 C 1447.1273,312.69473 1445.201,312.80854 1443.1783,312.83855 C 1441.0113,312.83855 1441.0905,313.76968 1439.6125,314.96368 C 1438.8305,315.59568 1438.1169,315.08613 1437.2813,314.92355 C 1436.1558,314.70457 1436.363,315.38408 1436.1158,316.03033 C 1435.3545,317.17672 1436.1862,312.65014 1436.1943,312.32455 C 1436.2555,309.86021 1433.4993,311.27676 1433.1819,309.62008" id="by"/>
  <path class="landxx pk" d="M 1720.2763,501.18955 C 1723.4543,501.90455 1726.3633,503.46455 1729.4743,504.39655 C 1733.1793,505.50655 1736.1583,504.36055 1739.8523,504.28755 C 1741.1653,504.26155 1742.2723,504.79255 1743.5443,504.97855 C 1744.8513,505.16955 1745.7593,504.24255 1746.9063,503.82555 C 1748.4183,503.27555 1749.8353,503.89755 1751.3433,503.81955 C 1752.6153,503.75455 1753.8163,503.10355 1755.0523,502.81855 C 1756.4873,502.48755 1759.2753,502.24855 1760.1853,500.90255 C 1760.7293,500.09955 1759.4343,499.66355 1759.5863,498.88555 C 1759.9543,496.99555 1759.0813,495.47055 1759.2613,493.62955 C 1759.4453,491.76355 1759.8093,492.18455 1760.9093,491.08055 C 1761.5863,490.40255 1761.7543,489.27155 1762.6523,488.80255 C 1763.5903,488.31155 1763.9093,489.30555 1764.8173,489.41055 C 1766.3643,489.59055 1767.7323,489.09755 1769.0863,488.37455 C 1768.5283,487.79155 1767.5183,487.75455 1767.2863,486.86255 C 1768.6163,486.83855 1769.9923,486.00355 1770.7023,484.91555 C 1771.5423,483.62955 1773.6303,485.81655 1773.6953,484.19855 C 1774.6863,484.66455 1775.1003,484.44455 1775.7063,485.47855 C 1776.2173,486.35155 1777.0193,486.05355 1777.7223,485.52555 C 1779.6043,484.11055 1779.1333,481.49955 1778.3813,479.63155 C 1777.8993,478.43255 1778.8403,478.24555 1779.0963,477.21355 C 1779.2953,476.40755 1779.3203,475.18255 1779.0323,474.39455 C 1778.6883,473.45755 1779.6783,473.29355 1780.3413,473.14555 C 1781.4393,472.89955 1782.0183,473.26955 1783.0213,472.46655 C 1784.2433,471.48855 1784.2093,471.43855 1783.7083,470.22955 C 1783.2893,469.21955 1782.6993,468.10855 1781.7483,467.50855 C 1779.8843,466.33055 1779.5963,464.63855 1782.2033,465.34955 C 1783.9793,465.83355 1788.2783,466.40755 1788.7623,463.87955 C 1788.9063,463.13355 1788.9013,462.57455 1788.3193,462.06955 C 1788.0703,461.85355 1786.9513,461.12955 1787.1653,460.68355 C 1787.3573,460.48655 1787.5993,460.40455 1787.8893,460.43755 C 1787.5523,459.72155 1788.1393,459.17155 1788.5343,458.67055 C 1788.8993,458.20855 1790.1893,457.15255 1790.1893,456.58555 C 1790.1963,456.04755 1790.3693,455.56955 1790.7083,455.15255 C 1790.9693,454.78355 1790.0603,454.40955 1789.9393,454.17355 C 1789.8343,453.91155 1789.9023,453.71755 1790.1463,453.59255 C 1790.5483,453.16855 1789.8593,452.30955 1789.7443,451.91155 C 1789.5893,451.38155 1788.6763,450.12355 1788.3123,449.66655 C 1787.7903,449.00955 1786.9383,448.56755 1786.3053,448.02255 C 1785.4463,447.28255 1788.0293,444.11655 1788.8633,444.37655 C 1790.0103,444.73355 1790.0963,443.48055 1790.7763,442.79355 C 1791.4963,442.06655 1792.4323,441.59955 1793.3993,441.29455 C 1795.4763,440.63955 1797.6993,440.63355 1799.8543,440.54955 C 1800.9463,440.50755 1802.4683,439.93755 1803.4763,440.44255 C 1804.4663,440.93755 1804.5423,441.21355 1805.4563,440.53555 C 1806.7083,439.60555 1809.8233,438.05655 1810.7093,440.20555 C 1811.5993,438.60855 1813.9073,439.92055 1814.6133,441.00455 C 1815.2643,442.00655 1817.0673,441.45955 1818.1163,442.43755 C 1818.8603,443.13155 1819.3453,443.92055 1819.6023,444.90355 C 1819.7903,445.61655 1819.7203,446.83955 1819.9973,447.45155 C 1820.2983,448.11955 1821.8303,449.02755 1822.4023,449.54155 C 1823.1393,450.20455 1824.1313,449.09455 1824.8183,448.91755 C 1824.9515,452.02027 1832.1091,453.05688 1834.5827,452.51609 C 1833.9316,453.25941 1829.6694,455.25908 1830.0043,456.25755 C 1830.1566,456.71177 1828.7941,457.93163 1829.3381,457.48973 C 1829.628,457.25424 1828.5554,458.10932 1828.2498,458.59519 C 1827.9428,459.08335 1823.5441,459.66657 1822.7753,460.20055 C 1820.5898,461.71857 1818.651,460.52464 1816.0734,459.98446 C 1813.9953,459.54896 1811.4418,458.16282 1809.4268,459.40482 C 1808.0548,460.24982 1807.1323,462.15455 1808.8693,462.88455 C 1810.4583,463.55255 1808.2543,464.49501 1809.5693,464.91401 C 1810.0033,465.05201 1812.9539,464.92746 1811.8129,466.05246 C 1810.7789,467.07346 1809.431,467.79696 1811.3646,468.96064 C 1813.1757,470.05064 1810.8785,471.55228 1811.8068,472.64528 C 1812.3611,473.29779 1813.2883,473.65555 1813.9583,474.15855 C 1814.2623,474.35755 1814.494,474.63281 1814.4913,474.99555 C 1814.4856,475.75334 1814.7606,475.83817 1815.8493,476.01355 C 1817.9604,476.35362 1816.8836,478.23623 1818.8105,478.65001 C 1819.8994,478.88384 1821.9383,479.09555 1822.7163,480.01255 C 1823.6733,481.14055 1820.6313,482.24155 1820.0183,482.52555 C 1818.4553,483.25055 1816.6451,484.6879 1818.2493,486.26655 C 1819.4189,487.41755 1817.8226,490.49255 1819.8573,490.53555 C 1819.6582,491.36486 1818.2462,492.02102 1817.4758,492.94646 C 1816.5081,494.10904 1814.2841,496.45695 1815.3551,497.33846 C 1817.2908,498.93162 1813.6737,499.82659 1812.9008,500.21455 C 1812.0401,500.6465 1812.0373,502.07255 1811.9023,502.89455 C 1811.6733,504.28955 1811.2953,505.72555 1810.6663,506.99755 C 1809.7293,508.88955 1808.1247,509.14401 1806.7947,510.59501 C 1806.2587,511.17901 1806.0024,512.21709 1805.9344,512.98809 C 1805.8134,514.34909 1804.3563,515.17573 1803.8733,516.37673 C 1802.8453,518.92773 1801.3473,518.10755 1798.9203,518.75555 C 1798.0933,518.97655 1797.1583,519.82655 1796.2673,519.73955 C 1795.1973,519.63555 1794.7113,518.34355 1794.1803,517.59755 C 1792.9323,515.84955 1791.0753,518.50255 1790.5903,519.59555 C 1789.7313,521.53355 1788.3263,523.19555 1787.5453,525.10055 C 1786.5833,527.44655 1787.9083,528.75855 1790.1483,529.32155 C 1792.2373,529.84555 1793.2663,529.95455 1793.0683,532.51255 C 1792.9563,533.97455 1792.6803,537.24555 1795.1903,537.27255 C 1795.9133,537.28055 1796.7483,536.75455 1797.3883,537.33655 C 1798.0243,537.91455 1797.5113,539.48855 1798.0753,540.29755 C 1799.2323,541.96055 1800.5133,543.20455 1801.0583,545.25555 C 1801.2673,546.04155 1801.7173,545.80655 1800.9513,546.35455 C 1800.3403,546.79155 1801.3433,548.24755 1801.8583,548.35155 C 1800.5003,549.37855 1799.0783,550.66455 1797.9273,548.74155 C 1797.3923,547.84755 1796.5773,548.51655 1795.9023,548.78655 C 1795.5633,548.88155 1795.2293,548.99455 1794.9013,549.12355 C 1794.3563,549.40355 1794.7003,550.08155 1794.1133,550.33155 C 1792.1753,551.15155 1789.9993,549.28255 1787.9673,549.39355 C 1786.8813,549.45355 1785.4383,549.64455 1784.4283,549.14355 C 1784.4893,549.94055 1785.2003,552.16755 1783.7563,552.16755 C 1782.5323,552.16755 1780.6673,552.19755 1780.7583,553.89555 C 1780.6773,552.91155 1780.5573,552.93555 1780.3983,553.96755 C 1779.9183,553.54955 1779.6003,552.97555 1779.2493,552.45555 C 1779.5073,552.90755 1779.5773,553.38855 1779.4593,553.89455 C 1778.9303,553.56755 1778.4743,553.15955 1778.0893,552.67055 C 1778.4173,553.98855 1777.3743,553.24555 1776.5793,553.67855 C 1776.5793,553.29455 1776.5793,552.90955 1776.5793,552.52655 C 1775.7023,553.15555 1774.5823,551.96755 1775.5793,551.37555 C 1774.4043,551.59655 1772.7493,549.88555 1773.9893,548.92755 C 1772.6183,548.14055 1773.3553,545.87255 1772.2223,545.30555 C 1771.5873,544.98955 1768.3743,544.85155 1768.4663,543.82555 C 1768.6483,541.74955 1768.1763,541.13655 1767.1483,539.49455 C 1766.3833,538.27655 1764.0923,537.02355 1763.6983,539.20755 C 1764.9363,539.14455 1765.7613,538.04455 1766.4283,539.71155 C 1765.1073,538.85955 1763.6253,539.38455 1762.2653,539.81355 C 1760.8473,540.26055 1759.4893,539.76655 1758.1433,540.17155 C 1757.4123,540.39155 1752.3113,540.32155 1753.6183,541.72755 C 1753.0233,541.90555 1752.6353,541.62455 1752.6083,541.00755 C 1750.6363,542.02755 1749.0693,539.07855 1747.9993,539.56755 C 1748.1633,539.80955 1748.3293,540.04955 1748.4993,540.28855 C 1747.5043,540.49155 1743.4333,539.18455 1744.4693,541.58455 C 1743.4323,541.05755 1735.6453,540.48055 1736.3293,542.44755 C 1735.8053,542.42555 1735.3993,541.66455 1734.9743,541.42255 C 1734.2393,541.00355 1734.2353,541.99155 1733.7853,542.25455 C 1731.9763,543.30855 1730.6543,542.11155 1730.7153,540.51255 C 1730.7973,538.35755 1730.2863,530.56655 1734.4603,531.50455 C 1733.2363,528.29055 1740.6373,530.03155 1740.9983,528.26755 C 1741.1563,527.49755 1741.4393,524.56455 1741.1113,524.30455 C 1740.3223,523.67855 1739.2793,524.13255 1738.3873,524.07555 C 1737.1083,523.99455 1737.2993,521.88755 1737.2683,521.02355 C 1737.2083,519.36055 1736.6843,517.22355 1736.1613,515.63255 C 1735.9193,514.89655 1734.6223,514.85055 1733.9733,514.44855 C 1732.8963,513.78355 1731.7893,513.38155 1730.6483,512.85255 C 1728.1173,511.68055 1727.0763,509.96655 1725.7933,507.62755 C 1724.4283,505.14655 1722.2453,503.17255 1720.2763,501.18955" id="pk"/>
  <g class="landxx coastxx id" id="id" transform="translate(-29.9017, -45.0745)">
    <path d="M 2324.809,838.583 C 2324.301,838.904 2324.444,839.144 2325.239,839.303 C 2325.095,839.063 2324.952,838.823 2324.809,838.583" id="path2528"/>
    <path d="M 2326.318,819.793 C 2325.666,820.012 2325.592,820.755 2325.818,821.303 C 2325.857,820.757 2326.024,820.254 2326.318,819.793" id="path2530"/>
    <path d="M 2327.479,822.093 C 2326.945,821.975 2326.895,822.215 2327.329,822.814 C 2327.378,822.573 2327.429,822.333 2327.479,822.093" id="path2532"/>
    <path d="M 2331.358,812.813 C 2330.45,813.007 2330.596,814.809 2331.574,814.533 C 2332.148,814.372 2331.57,813.066 2331.358,812.813" id="path2540"/>
    <path d="M 2333.088,817.413 C 2333.065,817.559 2333.042,817.706 2333.019,817.852 C 2333.534,817.817 2334.038,817.721 2334.529,817.563 C 2334.048,817.519 2333.567,817.469 2333.088,817.413" id="path2542"/>
    <path d="M 2335.038,810.213 C 2335.041,811.344 2336.938,812.773 2337.76,811.519 C 2338.346,810.626 2335.488,810.257 2335.038,810.213" id="path2544"/>
    <path d="M 2335.898,818.493 C 2335.851,818.733 2335.804,818.973 2335.757,819.213 C 2337.614,819.34 2339.212,820.354 2341.007,820.734 C 2341.768,820.895 2342.579,821.24 2343.372,821.029 C 2344.253,820.795 2345.249,820.268 2346.187,820.443 C 2345.423,819.322 2343.458,819.395 2342.278,819.253 C 2340.148,818.996 2338.042,818.714 2335.898,818.493" id="path2546"/>
    <path d="M 2349.218,876.813 C 2349.145,877.053 2349.072,877.293 2348.998,877.533 C 2350.484,877.039 2352.092,877.262 2353.627,877.254 C 2355.132,877.247 2356.02,876.477 2357.176,875.589 C 2357.684,875.199 2360.837,871.318 2360.203,870.532 C 2358.371,868.264 2357.176,868.028 2354.369,869.199 C 2351.483,870.404 2350.362,874.198 2349.218,876.813" id="path2548"/>
    <path d="M 2358.068,877.393 C 2358.311,877.32 2358.554,877.247 2358.797,877.173 C 2358.424,876.44 2358.461,875.031 2357.254,875.518 C 2355.65,876.164 2356.926,877.356 2358.068,877.393" id="path2550"/>
    <path d="M 2357.718,862.853 C 2357.752,863.421 2358.112,863.792 2358.649,863.933 C 2358.598,863.385 2358.262,862.973 2357.718,862.853" id="path2552"/>
    <path class="landxx id" d="M 2303.638,813.313 C 2303.868,812.747 2303.772,812.243 2303.348,811.802 C 2305.594,811.231 2307.458,811.105 2309.207,809.494 C 2310.83,807.998 2312.52,807.361 2314.702,807.899 C 2316.908,808.443 2318.009,810.072 2319.908,810.891 C 2321.83,811.719 2323.81,810.618 2325.597,812.012 C 2324.231,813.026 2325.544,814.387 2326.128,815.409 C 2326.917,816.79 2326.018,817.942 2325.509,819.216 C 2324.207,822.48 2326.147,826.38 2327.617,829.222 C 2328.121,828.488 2327.43,826.838 2328.257,826.294 C 2329.421,825.527 2329.277,828.96 2329.288,829.286 C 2329.312,829.939 2329.528,830.099 2329.974,830.536 C 2330.615,831.163 2330.314,832.009 2330.883,832.649 C 2331.986,833.887 2334.248,834.549 2335.519,833.231 C 2336.443,832.273 2337.394,831.424 2338.34,830.493 C 2339.334,829.514 2339.525,828.203 2340.853,827.363 C 2342.043,826.61 2341.701,823.819 2343.218,823.81 C 2345.016,823.799 2345.882,823.969 2347.564,823.092 C 2348.769,822.464 2348.826,822.043 2348.289,820.917 C 2347.877,820.05 2350.516,818.822 2351.049,818.539 C 2352.389,817.826 2353.63,817.333 2355.048,818.064 C 2356.899,819.019 2359.107,819.686 2360.766,820.936 C 2362.16,821.986 2363.745,822.565 2365.296,823.35 C 2366.357,823.887 2368.44,825.529 2369.686,825.093 C 2371.258,824.542 2371.915,825.74 2373.525,825.718 C 2375.215,825.695 2376.054,827.441 2377.729,827.214 L 2376.113,858.197 C 2375.939,859.871 2374.667,861.067 2374.74,862.845 C 2374.77,863.578 2375.719,863.974 2375.719,864.434 C 2375.719,865.551 2375.529,866.698 2375.452,867.814 C 2375.091,873.068 2374.691,878.339 2374.199,883.584 C 2372.044,882.497 2371.6,880.033 2369.962,878.471 C 2369.152,877.699 2366.063,875.736 2367.509,874.374 C 2366.566,875.112 2364.884,874.799 2363.755,875.248 C 2362.794,875.63 2361.016,875.718 2361.529,874.155 C 2361.05,874.742 2359.824,875.849 2359.008,875.857 C 2357.774,875.869 2359.215,873.745 2359.343,873.445 C 2359.761,872.466 2361.131,870.269 2360.135,869.735 C 2359.237,869.254 2358.979,868.07 2358.288,867.384 C 2359.465,866.369 2361.025,868.144 2361.958,866.594 C 2360.332,868.217 2358.208,865.659 2357.068,864.655 C 2358.574,862.925 2360.292,864.944 2361.818,865.014 C 2360.49,864.498 2359.362,863.579 2358.148,862.854 C 2359.565,862.605 2358.345,862.169 2357.877,861.68 C 2357.212,860.988 2356.558,860.155 2356.291,859.215 C 2355.947,858.006 2354.827,855.742 2356.418,855.144 C 2354.848,855.724 2354.755,854.029 2356.198,853.774 C 2355.66,853.773 2354.316,854.628 2353.898,853.995 C 2353.397,853.238 2354.402,852.245 2354.038,851.405 C 2353.576,853.251 2352.211,850.305 2351.609,850.045 C 2350.582,849.601 2350.521,849.346 2349.737,848.986 C 2349.131,848.708 2347.67,848.297 2347.306,847.688 C 2346.805,846.848 2343.707,846.78 2342.745,846.196 C 2341.235,845.28 2339.48,844.282 2337.791,843.738 C 2336.161,843.214 2333.759,843.698 2332.337,842.997 C 2331.067,842.372 2329.378,841.507 2328.842,840.097 C 2328.344,838.787 2330.435,838.993 2331.148,838.874 C 2330.278,838.611 2329.183,838.166 2328.486,839.047 C 2327.701,840.039 2327.183,838.512 2326.467,838.294 C 2326.456,839.77 2324.154,838.685 2325.238,837.654 C 2323.664,837.669 2323.049,837.686 2322.648,835.773 C 2322.578,836.329 2322.277,836.783 2321.826,836.212 C 2321.189,835.405 2321.693,834.901 2321.619,834.058 C 2321.484,832.515 2323.02,831.643 2323.078,830.083 C 2322.697,830.893 2321.348,831.18 2321.348,832.19 C 2321.348,833.092 2321.427,834.494 2320.128,834.413 C 2321.255,834.917 2320.361,835.505 2319.988,836.133 C 2319.735,836.36 2319.496,836.6 2319.268,836.853 C 2319.229,837.332 2319.203,837.812 2319.188,838.293 C 2318.837,839.315 2316.757,840.395 2315.674,840.093 C 2314.715,839.825 2314.154,837.581 2314.121,836.742 C 2314.099,836.176 2314.849,836.123 2315.237,835.943 C 2316.264,835.465 2314.687,834.182 2314.947,833.333 C 2313.708,834.08 2312.171,830.081 2310.434,830.011 C 2307.742,829.901 2309.189,828.049 2311.126,827.853 C 2313.14,827.648 2314.591,829.644 2315.987,827.307 C 2317.033,825.558 2320.229,825.001 2319.406,827.853 C 2319.628,827.368 2319.795,826.865 2319.906,826.343 C 2320.192,827.724 2321.777,826.016 2321.716,827.493 C 2321.932,826.294 2322.345,826.961 2322.647,827.573 C 2322.938,826.946 2322.988,826.133 2322.426,825.623 C 2322.924,825.728 2323.407,825.682 2323.876,825.483 C 2323.46,824.982 2323.483,824.502 2323.946,824.043 C 2322.406,825.189 2323.592,823.223 2323.876,822.893 C 2322.226,823.031 2321.337,824.08 2319.564,823.805 C 2318.459,823.634 2316.984,824.679 2316.031,824.328 C 2314.858,823.897 2314.365,824.697 2313.445,823.783 C 2313.127,823.467 2312.22,824.046 2311.914,824.065 C 2310.895,824.128 2310.086,823.199 2309.326,822.673 C 2309.514,822.428 2309.707,822.188 2309.906,821.953 C 2309.094,822.095 2308.362,821.145 2308.652,820.373 C 2308.76,820.085 2309.149,819.691 2309.093,819.358 C 2308.953,818.51 2307.63,819.89 2308.535,818.353 C 2307.799,818.497 2307.614,818.507 2307.675,817.704 C 2306.636,819.211 2304.618,816.576 2303.715,816.624 C 2304.575,817.757 2302.515,817.584 2301.986,817.219 C 2300.52,816.203 2303.37,814.327 2303.638,813.313" id="path2562"/>
    <path d="M 2036.309,784.08 C 2036.623,784.299 2037.76,784.419 2037.583,783.657 C 2037.498,783.289 2036.251,782.523 2035.955,782.225 C 2035.585,781.853 2035.142,781.515 2034.578,781.632 C 2034.865,780.832 2033.767,780.227 2033.208,779.976 C 2032.915,779.844 2032.675,779.246 2032.283,779.568 C 2031.557,780.164 2031.572,780.766 2032.295,781.394 C 2033.5,782.442 2035.618,782.452 2036.309,784.08" id="path2206"/>
    <path d="M 2211.628,849.383 C 2211.292,850.596 2211.937,851.709 2211.769,852.913 C 2212.885,852.393 2213.534,854.541 2214.783,853.612 C 2215.637,852.977 2215.65,852.855 2216.754,852.982 C 2217.792,853.102 2218.485,851.915 2219.48,852.692 C 2219.385,851.004 2218.046,849.547 2218.667,847.815 C 2219.424,845.697 2219.825,844.312 2219.564,841.991 C 2219.345,840.05 2219.781,838.526 2219.958,836.638 C 2220.039,835.776 2219.951,834.92 2220.045,834.056 C 2220.202,832.608 2219.549,832.61 2219.191,831.382 C 2218.495,828.994 2222.632,827.042 2224.439,827.608 C 2226.071,828.119 2225.566,831.767 2224.707,832.721 C 2222.292,835.402 2226.377,838.452 2228.685,839.692 C 2229.664,840.218 2227.204,843.89 2228.201,845.385 C 2228.739,846.191 2230.511,846.604 2231.419,846.737 C 2232.536,846.9 2232.254,845.091 2232.377,844.452 C 2232.655,843.014 2234.688,843.276 2235.699,842.865 C 2236.3,842.62 2238.733,843.863 2237.332,842.113 C 2238.425,843.964 2239.898,840.212 2237.982,839.663 C 2237.638,841.424 2237.168,838.998 2236.252,839.013 C 2236.445,838.749 2236.639,838.486 2236.832,838.223 C 2236.039,838.189 2235.055,837.153 2234.602,836.574 C 2233.864,835.629 2234.832,834.852 2234.163,833.904 C 2234.52,833.992 2234.88,834.065 2235.243,834.124 C 2234.417,833.622 2234.638,832.406 2235.682,832.464 C 2235.49,832.095 2235.237,831.772 2234.925,831.497 C 2234.353,830.869 2234.768,830.199 2234.382,829.514 C 2234.059,828.94 2233.302,828.896 2232.943,828.365 C 2232.535,827.761 2232.401,826.799 2232.066,826.126 C 2231.561,825.111 2231.136,824.117 2230.065,823.593 C 2229.192,823.166 2228.777,821.338 2227.613,821.885 C 2227.666,821.263 2227.689,820.639 2227.682,820.015 C 2228.898,821.047 2229.861,821.989 2231.094,820.287 C 2232.163,818.81 2233.278,818.789 2234.789,817.945 C 2236.863,816.787 2238.381,811.926 2241.1,812.012 C 2241.876,812.037 2242.015,814.28 2242.913,813.335 C 2243.449,812.772 2244.47,811.332 2243.925,810.53 C 2243.019,809.198 2239.496,809.607 2238.413,810.504 C 2239.001,810.566 2239.505,810.806 2239.923,811.224 C 2238.505,811.274 2234.428,810.28 2233.931,812.116 C 2233.598,813.345 2229.899,813.299 2230.133,811.585 C 2228.9,812.341 2227.622,813.595 2227.001,814.913 C 2226.326,816.348 2226.016,817.195 2224.355,816.833 C 2223.267,816.597 2222.235,816.313 2221.936,815.085 C 2221.785,814.466 2221.728,813.84 2221.387,813.281 C 2220.759,812.253 2220.331,813.09 2219.636,812.584 C 2217.399,810.958 2217.302,805.898 2218.27,803.65 C 2218.825,802.361 2219.752,800.916 2221.026,800.233 C 2222.078,799.67 2224.657,800.6 2225.755,800.999 C 2226.534,801.281 2228.026,800.629 2228.724,800.307 C 2230.164,799.64 2230.563,800.66 2231.958,800.872 C 2233.51,801.107 2235.578,800.267 2237.238,800.308 C 2239.225,800.358 2240.502,800.47 2242.092,801.618 C 2244.069,803.046 2250.213,802.083 2251.823,800.2 C 2252.929,798.907 2253.318,797.458 2254.649,796.262 C 2256.082,794.974 2256.016,792.792 2257.562,791.64 C 2257.018,791.296 2256.708,790.696 2256.912,790.056 C 2256.346,790.202 2255.551,789.802 2255.074,790.207 C 2254.357,790.814 2254.398,792 2253.416,792.301 C 2251.917,792.76 2252.87,793.222 2252.162,794.16 C 2251.754,794.701 2250.752,794.492 2250.466,795.233 C 2249.749,797.089 2246.905,797.42 2245.327,797.079 C 2243.99,796.79 2242.409,796.482 2241.035,796.551 C 2240.263,796.59 2239.524,797.445 2238.889,797.189 C 2236.899,796.387 2235.492,796.006 2233.358,795.784 C 2232.261,795.67 2231.622,795.347 2230.456,795.465 C 2229.126,795.599 2228.728,793.785 2228.055,793.798 C 2227.186,793.814 2223.958,792.225 2223.773,794.031 C 2223.521,796.492 2221.289,798.836 2219.484,796.254 C 2219.312,797.061 2219.049,797.674 2218.424,798.225 C 2217.579,798.97 2217.751,800.121 2216.93,800.784 C 2215.363,802.047 2217.408,804.247 2215.803,805.394 C 2215.67,804.828 2215.264,804.444 2214.654,804.604 C 2215.05,806.132 2216.233,805.758 2215.749,807.52 C 2215.308,809.126 2216.775,810.598 2216.314,812.235 C 2215.981,811.625 2215.842,810.921 2215.303,810.434 C 2214.403,812.871 2212.486,814.144 2212.353,816.914 C 2212.293,818.151 2212.402,819.47 2212.313,820.675 C 2212.209,822.087 2210.889,823.002 2210.793,824.513 C 2210.667,826.514 2207.396,827.437 2207.906,829.108 C 2208.193,830.049 2208.947,829.461 2208.135,830.794 C 2207.433,831.948 2208.356,834.447 2208.823,835.635 C 2209.472,834.905 2211.149,834.635 2212.058,834.586 C 2214.047,834.479 2212.47,836.408 2213.117,837.564 C 2214.19,839.484 2213.932,841.126 2213.592,843.337 C 2213.307,845.147 2212.845,847.965 2211.628,849.383" id="path2348"/>
    <path d="M 2231.498,848.593 C 2229.172,849.399 2231.258,852 2231.381,851.888 C 2232.004,851.318 2231.944,849.239 2231.498,848.593" id="path2366"/>
    <path d="M 2232.148,808.703 C 2231.759,807.98 2230.738,807.951 2230.708,808.923 C 2231.229,809.12 2231.709,809.046 2232.148,808.703" id="path2368"/>
    <path d="M 2233.229,807.913 C 2232.769,807.258 2232.199,807.277 2231.718,807.913 C 2232.188,808.245 2232.75,808.233 2233.229,807.913" id="path2370"/>
    <path d="M 2235.678,851.692 C 2235.976,851.322 2236.717,850.528 2236.445,849.986 C 2235.949,848.994 2236.944,848.548 2237.212,847.693 C 2237.358,847.226 2237.964,844.039 2236.68,844.729 C 2236.045,845.07 2235.206,845.519 2234.656,845.976 C 2233.621,846.834 2235.186,847.933 2234.449,848.85 C 2234.165,849.204 2233.422,850.426 2233.66,850.893 C 2234.047,851.648 2235.316,851.424 2235.31,850.534 C 2235.421,850.923 2235.544,851.31 2235.678,851.692" id="path2372"/>
    <path d="M 2240.138,842.763 C 2238.288,842.797 2237.611,846.783 2237.528,848.195 C 2237.442,849.668 2236.865,850.303 2236.324,851.647 C 2235.653,853.312 2236.688,852.868 2237.619,853.853 C 2238.067,853.24 2238.115,852.42 2238.587,851.834 C 2239.063,851.244 2239.94,851.565 2240.392,851.001 C 2241.04,850.195 2239.703,848.776 2238.91,849.602 C 2239.127,848.726 2239.083,847.808 2239.326,846.935 C 2239.486,846.36 2240.552,844.453 2240.79,846.142 C 2241.593,844.807 2240.056,844.033 2240.138,842.763" id="path2378"/>
    <path d="M 2242.229,818.853 C 2242.196,818.342 2242.243,817.839 2242.369,817.343 C 2245.118,818.768 2244.427,813.338 2242.229,816.693 C 2241.512,815.63 2242.95,815.331 2241.495,814.883 C 2240.997,814.73 2240.41,814.667 2239.948,814.967 C 2238.892,815.652 2239.141,817.415 2239.419,818.423 C 2240.227,818.009 2240.686,817.195 2241.069,816.413 C 2241.859,816.981 2241.104,818.77 2242.229,818.853" id="path2386"/>
    <path d="M 2240.068,839.083 C 2239.247,839.427 2239.344,840.932 2240.151,841.247 C 2241.355,841.717 2242.051,839.342 2240.068,839.083" id="path2388"/>
    <path d="M 2242.878,815.043 C 2242.639,815.348 2242.485,815.275 2242.729,815.763 C 2242.778,815.523 2242.828,815.283 2242.878,815.043" id="path2394"/>
    <path d="M 2243.229,819.293 C 2242.912,819.417 2242.745,819.657 2242.729,820.013 C 2242.896,819.773 2243.062,819.533 2243.229,819.293" id="path2396"/>
    <path d="M 2243.309,850.103 C 2243.383,850.615 2243.31,850.819 2243.958,850.824 C 2243.741,850.583 2243.525,850.343 2243.309,850.103" id="path2398"/>
    <path d="M 2244.528,851.973 C 2244.584,852.38 2244.823,852.597 2245.248,852.623 C 2245.008,852.406 2244.769,852.189 2244.528,851.973" id="path2400"/>
    <path d="M 2246.548,855.653 C 2246.031,856.171 2246.198,856.411 2247.048,856.373 C 2246.881,856.133 2246.715,855.893 2246.548,855.653" id="path2404"/>
    <path d="M 2251.229,819.003 C 2249.969,820.839 2250.972,822.439 2253.168,821.733 C 2253.516,821.518 2253.875,821.326 2254.248,821.158 C 2254.666,821.242 2255.088,821.283 2255.515,821.28 C 2256.311,820.881 2257.701,821.198 2258.069,820.154 C 2257.493,820.224 2257.11,819.922 2256.989,819.364 C 2256.059,820.676 2252.619,818.51 2251.229,819.003" id="path2408"/>
    <path d="M 2258.788,780.84 C 2258.047,780.667 2257.973,780.907 2258.568,781.56 C 2258.642,781.32 2258.715,781.08 2258.788,780.84" id="path2414"/>
    <path d="M 2260.298,775.656 C 2260.73,774.754 2259.461,771.879 2258.348,773.135 C 2259.527,773.459 2259.129,775.166 2260.298,775.656" id="path2416"/>
    <path d="M 2258.788,820.083 C 2257.753,821.168 2260.406,821.224 2260.922,821.232 C 2262.325,821.255 2264.402,821.472 2265.479,820.373 C 2263.273,820.048 2260.935,820.865 2258.788,820.083" id="path2418"/>
    <path d="M 2262.888,821.733 C 2261.345,822.067 2262.749,825.584 2263.538,826.132 C 2263.857,825.574 2263.252,822.298 2262.888,821.733" id="path2422"/>
    <path d="M 2267.928,837.793 C 2268.985,837.267 2271.302,836.917 2271.975,835.908 C 2272.827,834.632 2272.359,833.696 2271.029,833.403 C 2272.085,832.212 2269.79,831.478 2268.998,831.281 C 2267.51,830.911 2266.185,831.374 2264.709,831.534 C 2262.902,831.731 2262.525,832.626 2263.339,834.368 C 2264.193,836.199 2266.254,836.897 2267.928,837.793" id="path2424"/>
    <path d="M 2268.068,771.84 C 2268.265,771.263 2268.047,770.685 2267.568,770.328 C 2267.711,770.84 2267.877,771.344 2268.068,771.84" id="path2426"/>
    <path d="M 2268.358,770.184 C 2269.144,769.371 2269.335,768.1 2269.186,767.021 C 2269.11,766.466 2268.625,764.822 2268.113,766.139 C 2267.757,767.056 2268.82,767.783 2268.472,768.478 C 2268.154,769.117 2267.512,769.591 2268.358,770.184" id="path2428"/>
    <path d="M 2272.898,806.973 C 2271.996,806.586 2271.28,808.614 2271.887,809.133 C 2272.938,810.031 2273.04,807.519 2272.898,806.973" id="path2432"/>
    <path d="M 2272.898,809.933 C 2272.031,810.384 2272.471,811.741 2273.478,811.443 C 2273.369,810.907 2273.176,810.403 2272.898,809.933" id="path2434"/>
    <path d="M 2273.618,797.253 C 2273.4,797.596 2273.528,797.729 2273.838,798.043 C 2273.765,797.78 2273.691,797.516 2273.618,797.253" id="path2438"/>
    <path d="M 2275.128,807.333 C 2274.891,807.648 2274.65,807.961 2274.408,808.272 C 2274.164,808.037 2273.924,807.796 2273.688,807.552 C 2273.026,809.55 2274.001,808.821 2274.627,810.212 C 2274.767,810.52 2273.842,811.842 2274.827,811.605 C 2275.361,811.477 2275.937,811.171 2276.132,811.874 C 2276.356,812.685 2279.229,811.325 2276.829,810.664 C 2275.849,810.394 2275.618,809.829 2276.147,808.98 C 2276.479,808.452 2275.423,807.627 2275.128,807.333" id="path2440"/>
    <path d="M 2274.118,801.433 C 2273.935,801.773 2273.788,801.684 2274.118,802.154 C 2274.118,801.913 2274.118,801.673 2274.118,801.433" id="path2442"/>
    <path d="M 2274.479,817.053 C 2273.366,817.918 2273.946,819.475 2275.269,819.654 C 2275.828,819.729 2276.322,819.294 2276.875,819.294 C 2277.476,819.294 2278.075,819.485 2278.677,819.519 C 2280.729,819.634 2278.715,817.665 2278.101,817.352 C 2276.756,816.665 2275.805,816.014 2274.479,817.053" id="path2444"/>
    <path d="M 2274.408,798.263 C 2274.241,798.24 2274.075,798.216 2273.908,798.193 C 2274.038,798.48 2274.205,798.744 2274.408,798.984 C 2274.408,798.743 2274.408,798.503 2274.408,798.263" id="path2446"/>
    <path d="M 2276.278,804.603 C 2276.339,805.497 2275.989,806.431 2276.619,807.206 C 2277.228,807.955 2277.876,808.316 2278.231,809.271 C 2278.62,810.314 2280.552,812.563 2281.467,812.013 C 2278.863,809.345 2278.559,806.126 2278.044,802.694 C 2277.82,801.207 2277.88,799.984 2279.732,800.928 C 2281.307,801.731 2283.592,802.218 2285.346,802.373 C 2284.244,801.834 2284.224,800.49 2283.33,799.784 C 2282.434,799.074 2279.9,799.19 2280.997,797.356 C 2281.729,796.132 2283.663,796.093 2284.029,794.569 C 2284.277,793.541 2283.55,791.816 2283.986,791.063 C 2282.141,790.594 2280.123,792.418 2279.376,793.944 C 2281.099,794.027 2279.417,794.921 2278.938,795.11 C 2277.896,795.521 2278.351,796.642 2277.859,797.407 C 2277.348,798.2 2276.267,797.386 2275.946,796.858 C 2275.373,795.919 2276.563,795.311 2277.201,794.891 C 2278.517,794.025 2279.377,791.357 2278.656,789.839 C 2278.347,789.189 2277.254,788.953 2277.519,788.082 C 2277.773,787.247 2278.91,786.731 2278.585,785.736 C 2276.825,785.749 2276.025,787.971 2275.605,789.343 C 2275.159,790.797 2274.004,793.368 2274.266,794.807 C 2274.394,795.514 2274.699,796.252 2275.096,796.849 C 2275.255,797.133 2275.482,797.341 2275.776,797.472 C 2275.297,798.072 2275.126,798.807 2275.201,799.561 C 2275.298,800.547 2275.639,801.111 2276.226,801.845 C 2276.817,802.582 2276.409,803.78 2276.278,804.603" id="path2452"/>
    <path d="M 2275.348,814.823 C 2275.085,815.016 2274.822,815.21 2274.559,815.403 C 2275.094,815.474 2275.598,815.378 2276.069,815.113 C 2275.828,815.017 2275.588,814.92 2275.348,814.823" id="path2454"/>
    <path d="M 2274.908,832.893 C 2274.474,833.495 2274.668,833.735 2275.488,833.614 C 2275.295,833.373 2275.102,833.133 2274.908,832.893" id="path2456"/>
    <path d="M 2275.988,831.963 C 2275.814,832.223 2275.744,832.51 2275.778,832.823 C 2276.05,832.826 2276.314,832.779 2276.568,832.682 C 2276.4,832.422 2276.207,832.183 2275.988,831.963" id="path2458"/>
    <path d="M 2277.868,829.873 C 2277.64,830.242 2277.401,830.602 2277.148,830.953 C 2277.717,830.834 2277.999,830.451 2277.868,829.873" id="path2462"/>
    <path d="M 2293.348,834.692 C 2294.956,835.622 2296.604,836.268 2298.077,837.419 C 2299.791,838.757 2300.031,836.751 2299.753,835.414 C 2299.581,834.591 2298.476,834.326 2298.313,833.543 C 2298.146,832.735 2298.146,831.833 2297.345,831.355 C 2296.605,830.913 2296.325,830.336 2295.358,830.523 C 2294.387,830.711 2293.456,830.576 2292.666,829.951 C 2291.897,829.342 2290.911,828.88 2289.911,828.911 C 2288.911,828.942 2288.444,829.755 2287.559,829.949 C 2286.778,830.119 2286.478,829.712 2286.063,829.224 C 2285.916,829.05 2283.467,829.352 2283.114,829.327 C 2279.749,829.083 2276.201,831.472 2277.579,835.204 C 2277.606,834.444 2278.653,831.356 2279.739,831.603 C 2279.394,832.619 2281.198,834.242 2282.063,834.403 C 2282.418,834.469 2286.466,831.105 2285.499,833.693 C 2286.132,833.554 2289.863,835.055 2290.075,833.768 C 2290.258,832.661 2293.155,833.58 2293.348,834.692" id="path2464"/>
    <path d="M 2278.298,835.703 C 2277.726,835.977 2277.289,836.546 2277.428,837.213 C 2278.278,837.106 2278.867,836.446 2279.668,836.213 C 2278.897,837.505 2280.701,836.268 2280.528,835.053 C 2279.717,835.109 2279.152,835.875 2278.298,835.703" id="path2466"/>
    <path d="M 2279.448,784.8 C 2279.666,784.472 2279.824,784.578 2279.518,784.08 C 2279.495,784.32 2279.472,784.56 2279.448,784.8" id="path2468"/>
    <path d="M 2280.598,787.176 C 2283.305,786.567 2284.303,784.268 2282.689,781.848 C 2282.371,782.629 2281.49,782.827 2280.958,783.406 C 2280.037,784.406 2280.015,786.029 2280.598,787.176" id="path2470"/>
    <path d="M 2281.468,835.053 C 2281.257,835.321 2281.23,835.611 2281.388,835.923 C 2281.691,835.779 2281.957,835.586 2282.189,835.343 C 2281.948,835.246 2281.708,835.149 2281.468,835.053" id="path2474"/>
    <path d="M 2282.548,834.843 C 2282.281,835.59 2282.869,836.226 2283.628,835.853 C 2283.54,835.285 2283.137,834.867 2282.548,834.843" id="path2478"/>
    <path d="M 2288.958,804.603 C 2289.334,805.249 2289.901,805.781 2290.608,806.043 C 2290.108,805.526 2289.611,804.936 2288.958,804.603" id="path2482"/>
    <path d="M 2293.778,822.093 C 2294.27,822.259 2294.773,822.379 2295.288,822.453 C 2295.841,822.44 2296.136,821.939 2296.728,822.022 C 2296.147,821.225 2297.36,820.837 2296.734,819.627 C 2296.064,818.33 2293.192,820.473 2292.047,820.653 C 2292.225,821.464 2292.95,822.063 2293.778,822.093" id="path2484"/>
    <path d="M 2292.988,814.533 C 2292.689,814.681 2292.423,814.875 2292.188,815.113 C 2292.878,815.495 2293.145,815.302 2292.988,814.533" id="path2486"/>
    <path d="M 2294.788,804.673 C 2294.908,804.913 2295.028,805.153 2295.148,805.393 C 2295.244,804.791 2295.059,804.98 2294.788,804.673" id="path2488"/>
    <path d="M 2299.468,808.413 C 2302.477,809.264 2299.471,806.37 2298.819,805.683 C 2299.668,805.563 2300.237,806.474 2300.694,807.021 C 2301.548,808.042 2301.777,807.163 2302.605,807.577 C 2304.172,808.36 2304.313,806.119 2303.172,805.494 C 2302.097,804.906 2296.485,804.064 2296.368,806.182 C 2297.801,805.217 2297.014,806.776 2296.148,806.472 C 2296.836,807.354 2298.403,806.752 2298.097,808.342 C 2298.181,807.798 2298.574,807.386 2299.108,807.262 C 2299.248,807.64 2299.367,808.023 2299.468,808.413" id="path2490"/>
    <path d="M 2297.378,811.803 C 2297.119,813.159 2300.766,812.681 2300.399,811.293 C 2299.627,812.155 2298.337,811.503 2297.378,811.803" id="path2494"/>
    <path d="M 2298.818,808.272 C 2298.388,808.45 2298.052,808.736 2297.808,809.132 C 2298.434,809.512 2299.836,808.695 2298.818,808.272" id="path2496"/>
    <path d="M 2300.838,812.373 C 2298.104,812.829 2299.18,814.737 2300.662,816.203 C 2301.523,817.054 2303.227,812.837 2300.838,812.373" id="path2500"/>
    <path d="M 2302.638,813.243 C 2302.413,813.563 2302.187,813.423 2302.348,813.963 C 2302.444,813.723 2302.541,813.483 2302.638,813.243" id="path2506"/>
    <path d="M 2303.068,839.883 C 2303.185,840.33 2303.172,840.516 2303.717,840.603 C 2303.501,840.363 2303.285,840.123 2303.068,839.883" id="path2508"/>
    <path d="M 2303.998,838.942 C 2303.374,838.848 2303.447,839.088 2304.218,839.663 C 2304.145,839.423 2304.071,839.183 2303.998,838.942" id="path2510"/>
    <path d="M 2318.548,840.173 C 2318.258,841.17 2320.228,841.382 2320.708,841.893 C 2320.148,841.11 2319.109,840.954 2318.548,840.173" id="path2522"/>
    <path class="landxx id" d="M 2240.928,891.143 C 2241.323,891.438 2241.27,891.747 2241.868,891.503 C 2242.066,892.57 2240.096,892.909 2239.918,894.314 C 2240.779,894.048 2241.651,894.281 2242.522,894.115 C 2243.657,893.898 2244.502,892.684 2245.654,892.651 C 2247.505,892.598 2248.138,892.068 2249.371,890.693 C 2250.166,889.806 2251.498,888.684 2251.832,887.571 C 2252.34,885.879 2251.3128,887.23785 2252.7978,886.74185 C 2252.7978,886.74185 2253.0456,886.78346 2253.3405,885.98555 C 2253.0173,885.94005 2252.9482,885.0359 2252.9565,884.31203 C 2254.5906,882.15607 2252.2426,881.77911 2252.2524,881.68626 C 2249.2734,882.89026 2251.595,883.52925 2248.09,884.20925 C 2247.2165,884.3785 2247.8947,886.85878 2247.2778,887.25075 C 2246.6609,887.64272 2246.499,886.32138 2246.0702,886.88738 C 2245.6414,887.45338 2244.1957,885.43797 2243.8866,886.12931 C 2243.5775,886.82066 2241.186,890.375 2240.928,891.143" id="path2392"/>
    <path d="M 2207.958,875.013 C 2207.454,875.211 2207.284,875.571 2207.448,876.093 C 2208.119,876.03 2208.291,875.564 2207.958,875.013" id="path2346"/>
    <path d="M 2103.338,861.192 C 2103.032,861.326 2102.769,861.519 2102.548,861.772 C 2103.241,862.156 2103.504,861.962 2103.338,861.192" id="path2278"/>
    <path d="M 2114.718,857.083 C 2114.079,856.587 2110.874,855.993 2110.248,856.583 C 2110.167,854.545 2108.274,857.212 2108.018,857.883 C 2107.757,858.562 2107.723,861.833 2106.289,860.543 C 2106.046,861.618 2105.382,862.48 2104.989,863.493 C 2104.563,862.483 2103.645,862.759 2102.909,863.213 C 2105.23,864.339 2107.97,862.989 2110.062,864.443 C 2110.714,864.896 2111.649,864.932 2112.364,865.275 C 2113.505,865.821 2111.909,866.656 2111.74,867.445 C 2111.284,869.575 2117.41,869.205 2118.547,869.409 C 2120.735,869.802 2122.555,871.4 2124.738,871.963 C 2125.427,872.141 2126.372,872.374 2127.051,872.025 C 2127.776,871.652 2127.605,871.016 2128.646,870.986 C 2131.3,870.908 2133.982,871.536 2136.452,872.357 C 2139.509,873.372 2142.461,875.007 2145.595,875.724 C 2146.897,876.022 2148.755,875.748 2149.921,876.317 C 2151.123,876.903 2152.15,876.366 2153.46,876.492 C 2156.14,876.75 2159.132,878.416 2161.588,876.733 C 2163.592,875.358 2165.91,877.865 2167.636,878.758 C 2168.219,879.06 2171.778,879.278 2171.704,879.689 C 2171.57,880.436 2172.687,880.885 2173.292,880.59 C 2173.739,880.371 2172.553,878.871 2172.415,878.659 C 2171.204,876.797 2172.673,874.952 2172.852,873.036 C 2172.964,871.833 2171.77,871.608 2170.901,871.335 C 2169.413,870.867 2168.952,871.516 2167.561,871.379 C 2166.246,871.251 2164.457,872.205 2163.319,871.644 C 2162.596,871.288 2160.524,870.593 2160.19,869.798 C 2159.904,869.119 2161.166,865.903 2159.501,867.315 C 2159.829,866.25 2158.908,865.217 2158.991,864.144 C 2158.019,864.641 2156.722,864.299 2155.68,864.399 C 2155.271,864.355 2154.936,864.172 2154.674,863.851 C 2153.955,863.228 2153.208,863.696 2152.441,863.285 C 2152.038,863.069 2151.362,861.394 2150.661,862.235 C 2149.968,863.066 2148.575,862.923 2148.039,861.983 C 2146.773,859.759 2145.286,859.487 2144.249,862.252 C 2143.899,863.183 2143.286,865.3 2141.866,864.833 C 2141.074,864.572 2140.909,864.158 2140.02,864.219 C 2139.301,864.268 2138.55,864.456 2137.831,864.296 C 2136.423,863.983 2135.831,864.061 2134.468,864.146 C 2133.679,864.196 2132.419,863.156 2131.716,863.538 C 2128.873,865.084 2128.262,861.094 2127.177,859.571 C 2126.177,858.166 2126.36,859.467 2125.456,859.467 C 2124.394,859.467 2123.475,857.93 2122.375,858.48 C 2121.331,859.002 2120.614,857.628 2119.971,857.087 C 2118.914,856.199 2118.119,856.561 2116.951,855.867 C 2116.738,856.877 2115.733,858.263 2114.718,857.083" id="path2280"/>
    <path d="M 2129.979,871.133 C 2130.053,871.89 2130.97,871.818 2131.489,871.853 C 2131.028,871.521 2130.525,871.281 2129.979,871.133" id="path2296"/>
    <path d="M 2161.729,864.293 C 2160.998,864.176 2159.125,865.627 2159.993,866.349 C 2160.81,867.03 2162.764,867.619 2162.879,866.453 C 2164.058,867.85 2165.571,867.041 2166.97,866.539 C 2167.546,866.332 2168.174,866.165 2168.789,866.302 C 2168.663,865.333 2169.737,865.355 2170.369,865.153 C 2170.122,863.713 2168.186,864.05 2167.176,864.19 C 2165.375,864.44 2163.494,863.813 2161.729,864.293" id="path2308"/>
    <path d="M 2160.358,854.213 C 2159.784,854.302 2159.667,854.715 2159.858,855.213 C 2160.54,855.203 2160.693,854.756 2160.358,854.213" id="path2310"/>
    <path d="M 2172.168,865.733 C 2172.234,866.266 2172.136,866.509 2172.818,866.522 C 2172.618,866.246 2172.401,865.982 2172.168,865.733" id="path2312"/>
    <path d="M 2172.818,874.653 C 2171.881,876.429 2174.253,877.291 2175.471,877.87 C 2176.238,878.235 2176.851,878.608 2177.37,879.293 C 2178.308,880.531 2177.138,880.201 2177.076,881.095 C 2177.036,881.68 2179.398,879.006 2179.502,878.97 C 2180.478,878.624 2182.929,877.655 2181.702,876.364 C 2180.815,875.43 2179.056,874.113 2177.728,874.843 C 2176.21,875.68 2174.312,875.704 2172.818,874.653" id="path2314"/>
    <path d="M 2173.608,866.373 C 2173.816,866.556 2174.055,866.556 2174.328,866.373 C 2174.088,866.373 2173.849,866.373 2173.608,866.373" id="path2316"/>
    <path d="M 2179.658,863.783 C 2179.053,864.107 2178.915,864.662 2179.438,865.153 C 2179.992,864.469 2180.909,864.365 2181.677,864.722 C 2181.343,863.855 2180.44,863.837 2179.658,863.783" id="path2318"/>
    <path d="M 2180.878,880.853 C 2181.107,880.468 2181.107,880.082 2180.878,879.693 C 2180.596,879.863 2180.356,880.08 2180.158,880.343 C 2180.398,880.513 2180.638,880.683 2180.878,880.853" id="path2320"/>
    <path d="M 2186.928,875.733 C 2185.89,876.253 2184.606,877.22 2184.733,878.543 C 2184.835,879.596 2184.469,881.299 2183.119,880.203 C 2182.004,881.923 2187.789,881.442 2188.009,881.713 C 2187.532,880.66 2189.052,878.64 2189.561,877.786 C 2190.342,876.476 2187.93,875.358 2186.928,875.733" id="path2322"/>
    <path d="M 2198.448,881.853 C 2199.996,881.054 2201.525,880.794 2202.408,879.124 C 2202.74,879.678 2201.984,880.826 2202.553,881.122 C 2203.164,881.439 2203.591,881.206 2204.155,880.979 C 2204.868,880.691 2205.45,881.111 2206.149,881.214 C 2205.79,880.735 2205.313,880.469 2204.718,880.414 C 2206.093,879.583 2208.067,881.141 2208.458,878.904 C 2207.211,880.451 2207.365,877.586 2207.064,876.987 C 2206.337,875.535 2204.817,877.025 2204.858,878.114 C 2204.644,877.03 2203.812,875.187 2202.577,876.557 C 2201.642,877.592 2200.862,875.554 2200.386,875.027 C 2199.69,874.256 2196.822,874.842 2197.65,876.132 C 2198.663,877.709 2199.993,878.084 2201.548,878.974 C 2201.059,879.63 2199.743,879.985 2198.991,880.154 C 2197.699,880.443 2197.546,879.251 2197.003,878.434 C 2195.736,876.532 2191.769,877.207 2190.449,878.715 C 2189.815,879.439 2190.47,880.569 2189.886,881.307 C 2189.104,882.296 2190.506,883.09 2191.302,883.3 C 2191.795,883.43 2192.216,883.454 2192.64,883.126 C 2193.425,882.519 2193.397,882.919 2194.414,882.879 C 2195.843,882.822 2197.01,881.386 2198.448,881.853" id="path2328"/>
    <path d="M 2195.928,877.253 C 2196.434,876.641 2196.724,875.897 2197.149,875.233 C 2195.645,874.545 2195.775,876.318 2195.928,877.253" id="path2330"/>
    <path d="M 2205.938,887.183 C 2206.619,888.97 2208.283,888.913 2209.898,889.093 C 2211.81,889.306 2213.33,891.076 2214.479,892.447 C 2216.636,895.019 2221.608,892.692 2219.01,889.817 C 2218.199,888.921 2217.589,888.176 2216.397,888.174 C 2216.453,888.174 2214.523,885.865 2214.279,885.646 C 2214.099,885.484 2210.9,885.535 2210.747,885.633 C 2209.259,886.589 2207.19,885.071 2205.938,887.183" id="path2342"/>
    <path d="M 2210.548,877.603 C 2210.143,878.274 2209.645,879.058 2210.328,879.773 C 2210.352,878.779 2212.315,878.297 2210.548,877.603" id="path2350"/>
    <path d="M 2211.769,879.043 C 2211.973,879.609 2211.903,880.139 2211.559,880.633 C 2212.523,880.871 2212.357,879.76 2212.559,879.193 C 2212.295,879.143 2212.032,879.093 2211.769,879.043" id="path2352"/>
    <path d="M 2214.428,880.923 C 2215.198,880.511 2216.68,880.975 2217.577,880.874 C 2218.404,880.781 2219.648,880.962 2220.461,881.128 C 2221.35,881.31 2221.822,882.304 2222.843,881.839 C 2223.548,881.518 2225.228,882.137 2224.948,880.772 C 2227.184,881.71 2228.14,881.094 2230.295,880.308 C 2231.735,879.783 2236.781,880.496 2236.038,877.602 C 2237.033,877.793 2236.916,876.559 2237.838,876.533 C 2237.429,875.343 2236.158,873.568 2235.818,875.883 C 2236.273,875.598 2236.754,875.549 2237.257,875.734 C 2236.255,876.885 2234.468,876.963 2233.711,878.374 C 2232.796,880.082 2231.257,878.946 2230.293,878.047 C 2229.379,877.195 2228.25,878.603 2227.275,878.215 C 2226.76,878.01 2226.797,878.418 2226.472,878.724 C 2225.963,879.205 2224.983,878.367 2224.608,878.096 C 2223.36,877.195 2221.387,876.633 2219.91,876.287 C 2217.583,875.743 2216.836,876.627 2214.903,877.59 C 2213.934,878.074 2213.303,878.061 2213.193,879.41 C 2213.137,880.108 2213.407,882.178 2214.428,880.923" id="path2354"/>
    <path d="M 2219.838,854.713 C 2218.606,855.125 2219.207,859.689 2219.479,860.613 C 2219.729,858.64 2220.168,856.709 2219.838,854.713" id="path2356"/>
    <path d="M 2220.479,865.373 C 2220.26,865.73 2220.26,866.09 2220.479,866.453 C 2220.756,866.307 2220.997,866.113 2221.199,865.873 C 2220.958,865.707 2220.719,865.54 2220.479,865.373" id="path2358"/>
    <path d="M 2221.708,867.533 C 2221.906,868.149 2222.788,867.831 2223.148,867.673 C 2222.661,867.702 2222.181,867.656 2221.708,867.533" id="path2360"/>
    <path d="M 2224.878,896.403 C 2225.131,896.509 2225.37,896.459 2225.598,896.253 C 2225.357,896.303 2225.118,896.353 2224.878,896.403" id="path2362"/>
    <path d="M 2226.388,895.893 C 2226.338,896.013 2226.288,896.133 2226.239,896.253 C 2226.813,896.472 2227.665,896.309 2228.128,895.909 C 2228.36,895.723 2228.416,895.495 2228.296,895.224 C 2227.809,894.308 2226.966,895.588 2226.388,895.893" id="path2364"/>
    <path d="M 2234.809,897.772 C 2234.501,898.238 2234.478,898.718 2234.739,899.212 C 2235.958,898.519 2237.804,897.314 2239.129,896.972 C 2238.878,896.508 2238.662,896.028 2238.48,895.533 C 2237.109,896.161 2236.337,897.227 2234.809,897.772" id="path2374"/>
    <path d="M 2236.828,878.333 C 2236.898,878.546 2236.968,878.759 2237.038,878.973 C 2237.341,878.441 2237.751,878.011 2238.268,877.683 C 2237.682,877.662 2237.201,877.878 2236.828,878.333" id="path2380"/>
    <path d="M 2238.338,876.313 C 2238.004,876.595 2237.838,876.955 2237.838,877.392 C 2238.449,877.315 2239.848,877.589 2240.068,876.743 C 2240.348,875.677 2238.827,876.03 2238.338,876.313" id="path2382"/>
    <path d="M 2239.559,892.513 C 2239.051,892.841 2238.72,893.417 2238.699,894.023 C 2239.635,894.148 2239.824,893.199 2239.559,892.513" id="path2384"/>
    <path d="M 2240.718,876.243 C 2240.914,876.463 2241.08,876.702 2241.218,876.963 C 2240.656,877.192 2239.765,877.606 2239.574,878.246 C 2239.238,879.378 2241.371,878.522 2241.798,878.833 C 2241.652,877.475 2243.732,876.382 2244.819,876.173 C 2243.415,874.627 2242.692,876.63 2241.718,877.103 C 2242.201,876.063 2241.465,876.127 2240.718,876.243" id="path2390"/>
    <path d="M 2245.108,876.813 C 2244.801,877.107 2244.658,877.467 2244.677,877.892 C 2245.294,877.746 2245.841,878.064 2246.117,878.613 C 2246.701,878.004 2247.903,876.535 2247.117,875.663 C 2246.729,876.464 2245.873,876.593 2245.108,876.813" id="path2402"/>
    <path d="M 2248.988,875.013 C 2248.566,875.481 2248,876.673 2249.347,875.953 C 2248.598,876.286 2247.952,876.968 2247.908,877.823 C 2249.684,877.654 2251.726,877.799 2253.387,877.051 C 2254.436,876.579 2254.08,875.475 2253.042,875.292 C 2252.428,875.184 2251.891,875.442 2251.278,875.383 C 2250.507,875.309 2249.781,874.907 2248.988,875.013" id="path2406"/>
    <path d="M 2257.918,875.163 C 2257.359,875.105 2256.939,876.063 2256.838,876.453 C 2257.377,876.214 2257.778,875.736 2257.918,875.163" id="path2410"/>
    <path d="M 2258.998,874.013 C 2260.504,872.756 2261.967,873.042 2263.748,873.352 C 2264.695,873.517 2265.65,871.68 2266.848,871.703 C 2265.284,869.772 2264.474,870.461 2262.215,871.34 C 2261.584,871.586 2261.143,871.012 2260.61,871.241 C 2259.635,871.659 2259.138,873.055 2258.998,874.013" id="path2420"/>
    <path d="M 2271.748,869.613 C 2271.439,869.991 2271.293,870.425 2271.309,870.913 C 2271.953,870.691 2272.259,870.186 2271.748,869.613" id="path2430"/>
    <path d="M 2273.258,875.303 C 2273.08,876.153 2273.321,876.202 2273.979,875.453 C 2273.738,875.403 2273.498,875.353 2273.258,875.303" id="path2436"/>
    <path d="M 2274.269,874.733 C 2273.948,875.988 2276.329,876.409 2276.569,875.012 C 2275.785,875.157 2275.014,874.949 2274.269,874.733" id="path2450"/>
    <path d="M 2276.638,875.663 C 2276.664,875.88 2276.692,876.097 2276.718,876.313 C 2276.989,876.134 2277.226,875.917 2277.428,875.663 C 2277.164,875.663 2276.901,875.663 2276.638,875.663" id="path2460"/>
    <path d="M 2281.178,865.803 C 2280.855,865.996 2280.688,866.283 2280.678,866.663 C 2281.464,866.752 2281.631,866.466 2281.178,865.803" id="path2472"/>
    <path d="M 2282.328,875.522 C 2282.557,876.011 2282.941,876.228 2283.478,876.172 C 2283.369,875.565 2282.906,875.297 2282.328,875.522" id="path2476"/>
    <path d="M 2288.448,872.133 C 2287.806,872.692 2288.14,874.062 2289.077,874.045 C 2290.862,874.013 2290.046,871.949 2288.448,872.133" id="path2480"/>
    <path d="M 2296.798,876.893 C 2297.979,876.497 2299.053,875.758 2300.038,875.013 C 2298.796,875.493 2297.561,875.375 2296.798,876.893" id="path2492"/>
    <path d="M 2298.098,869.543 C 2297.97,870.162 2298.21,870.185 2298.819,869.613 C 2298.578,869.59 2298.338,869.566 2298.098,869.543" id="path2498"/>
    <path d="M 2299.388,868.823 C 2299.272,869.016 2299.155,869.21 2299.038,869.403 C 2299.615,869.597 2299.999,869.334 2300.118,868.753 C 2299.875,868.776 2299.631,868.8 2299.388,868.823" id="path2502"/>
    <path d="M 2304.218,866.233 C 2303.035,866.305 2299.691,870.285 2300.838,871.342 C 2298.509,870.269 2300.132,876.124 2301.31,873.419 C 2301.831,872.221 2303.131,871.636 2303.774,870.496 C 2304.305,869.553 2303.676,867.423 2304.869,867.022 C 2304.688,866.729 2304.472,866.466 2304.218,866.233" id="path2504"/>
    <path d="M 2305.008,866.233 C 2305.393,866.871 2306.252,866.697 2306.738,867.243 C 2306.743,866.256 2305.749,866.176 2305.008,866.233" id="path2512"/>
    <path d="M 2313.288,853.772 C 2313.107,853.541 2312.867,853.471 2312.568,853.563 C 2312.877,854.084 2312.378,855.688 2313.072,855.902 C 2314.156,856.238 2313.479,854.158 2313.288,853.772" id="path2516"/>
    <path d="M 2313.868,852.913 C 2313.237,852.782 2313.214,853.022 2313.798,853.633 C 2313.821,853.393 2313.845,853.153 2313.868,852.913" id="path2518"/>
    <path d="M 2313.868,856.583 C 2315.164,855.623 2316.447,852.217 2316.888,850.683 C 2315.801,851.067 2315.915,852.134 2315.547,853.017 C 2315.028,854.261 2314.255,855.23 2313.868,856.583" id="path2520"/>
    <path d="M 2323.588,858.022 C 2323.387,858.91 2323.224,860.114 2323.948,860.833 C 2322.501,860.751 2322.022,865.087 2324.415,863.946 C 2327.525,862.461 2325.508,859.458 2323.588,858.022" id="path2524"/>
    <path d="M 2327.838,861.692 C 2327.321,862.019 2327.257,862.694 2327.479,863.213 C 2327.854,862.766 2327.974,862.259 2327.838,861.692" id="path2534"/>
    <path d="M 2327.838,860.473 C 2327.17,860.865 2327.264,861.105 2328.118,861.193 C 2328.024,860.953 2327.932,860.712 2327.838,860.473" id="path2536"/>
    <path d="M 2328.838,860.613 C 2329.694,858.006 2328.229,859.77 2328.838,860.613" id="path2538"/>
    <path d="M 2081.738,852.123 C 2081.943,851.149 2080.542,850.03 2079.798,850.973 C 2080.497,851.31 2080.789,852.485 2081.738,852.123" id="path2230"/>
    <path d="M 2027.448,754.488 C 2028.532,754.829 2027.974,753.45 2026.948,753.767 C 2027.115,754.008 2027.281,754.248 2027.448,754.488" id="path2202"/>
    <path d="M 2044.518,759.168 C 2043.883,759.538 2043.063,760.093 2042.273,759.874 C 2041.521,759.665 2041.311,758.984 2040.459,758.962 C 2039.035,758.925 2037.802,759.661 2036.338,759.52 C 2034.615,759.354 2033.487,758.79 2032.364,757.499 C 2031.526,756.536 2029.2,755.194 2027.926,756.086 C 2023.224,759.382 2034.63,768.606 2036.436,771.019 C 2037.124,771.939 2038.054,771.975 2039.105,772.167 C 2040.842,772.485 2041.397,774.424 2042.189,775.701 C 2042.798,776.682 2044.025,779.142 2045.075,779.585 C 2046.901,780.356 2046.193,782.129 2046.699,783.756 C 2047.175,785.288 2048.755,784.905 2049.931,785.722 C 2050.68,786.241 2051.412,786.799 2052.151,787.333 C 2053.058,787.99 2053.983,788.665 2054.766,789.472 C 2055.866,790.606 2054.814,790.73 2055.149,791.989 C 2055.551,793.498 2056.203,794.952 2056.614,796.477 C 2057.049,798.092 2058.205,799.778 2058.09,801.482 C 2057.968,803.279 2059.361,802.176 2060.044,802.983 C 2061.161,804.304 2062.803,803.996 2063.107,806.154 C 2063.503,808.967 2067.209,810.794 2067.839,813.744 C 2067.929,814.166 2067.226,814.214 2067.56,814.669 C 2068.013,815.286 2068.413,815.937 2068.757,816.62 C 2069.538,818.168 2071.158,820.202 2071.218,821.958 C 2071.276,823.656 2071.234,824.734 2072.279,826.207 C 2073.066,827.316 2074.271,828.097 2074.932,829.292 C 2075.72,830.716 2076.132,832.151 2077.386,833.283 C 2078.627,834.403 2080.365,835.154 2081.124,836.727 C 2081.874,838.279 2081.741,839.19 2083.177,840.372 C 2084.632,841.569 2085.923,842.934 2087.316,844.201 C 2088.521,845.296 2089.693,846.33 2091.14,847.084 C 2092.583,847.836 2092.775,848.394 2093.488,849.68 C 2093.866,850.362 2094.362,851.001 2094.84,851.616 C 2095.251,852.054 2095.695,852.456 2096.174,852.82 C 2096.469,853.086 2096.114,853.379 2096.487,853.644 C 2096.753,853.833 2099.148,856.952 2099.161,855.302 C 2099.168,854.335 2098.248,853.596 2098.149,852.641 C 2098.009,851.299 2100.118,853.032 2100.37,853.255 C 2102.163,854.846 2103.914,854.853 2103.7,851.836 C 2104.818,852.803 2105.572,854.768 2107.16,855.076 C 2108.111,855.261 2108.076,852.385 2108.098,851.899 C 2108.182,850.027 2108.461,848.167 2108.6,846.296 C 2108.673,845.299 2109.319,843.885 2108.874,842.929 C 2108.531,842.191 2108.292,841.77 2108.527,840.976 C 2108.814,840.011 2109.597,838.718 2109.473,837.688 C 2109.381,836.924 2108.523,836.891 2108.719,835.858 C 2108.889,834.961 2109.439,834.178 2109.89,833.406 C 2110.376,832.573 2110.758,832.218 2110.574,831.18 C 2110.397,830.193 2109.585,830.595 2109.03,830.015 C 2108.274,829.224 2107.068,826.585 2107.519,825.555 C 2106.648,825.413 2102.465,824.313 2102.12,825.195 C 2101.99,824.936 2101.87,824.674 2101.76,824.406 C 2100.846,825.542 2100.728,827.142 2099.38,827.935 C 2099.776,827.348 2101.609,825.35 2099.811,825.335 C 2100.783,824.94 2101.942,824.091 2101.781,822.898 C 2101.66,822 2100.716,822.313 2100.155,821.974 C 2099.585,821.629 2099.269,820.147 2099.14,819.572 C 2098.76,817.884 2098.355,816.281 2098.143,814.551 C 2098.012,813.487 2094.895,813.373 2093.98,813.585 C 2093.26,813.752 2091.885,812.331 2091.361,811.895 C 2090.473,811.158 2090.83,809.611 2090.451,808.565 C 2090.85,809.047 2091.446,809.14 2091.961,808.774 C 2091.757,807.941 2090.952,807.831 2090.232,807.774 C 2090.816,807.732 2091.248,807.319 2091.392,806.764 C 2090.905,806.734 2090.424,806.781 2089.952,806.905 C 2090.8,806.657 2091.304,805.905 2091.926,805.343 C 2092.405,804.91 2093.657,805.037 2093.948,804.551 C 2094.428,803.748 2092.664,802.006 2092.204,801.488 C 2090.927,800.052 2089.978,800.172 2088.505,801.219 C 2087.365,802.028 2084.48,804.074 2083.252,802.444 C 2085.803,803.907 2090.71,799.384 2086.063,798.335 C 2085.322,798.168 2084.558,798.489 2083.833,798.194 C 2083.011,797.861 2082.665,796.929 2082,796.402 C 2080.634,795.319 2081.401,793.094 2079.908,791.944 C 2079.357,791.52 2078.785,790.862 2078.189,790.519 C 2077.555,790.153 2076.711,790.466 2076.009,790.101 C 2074.867,789.506 2074.958,787.739 2074.223,786.769 C 2073.862,786.291 2071.289,783.78 2070.725,785.279 C 2070.305,786.395 2071.664,787.82 2071.802,788.906 C 2071.333,788.427 2071.198,787.783 2070.817,787.251 C 2070.186,786.371 2069.033,786.239 2068.272,785.522 C 2067.013,784.335 2066.883,782.437 2065.682,781.203 C 2065.338,781.669 2065.268,782.173 2065.472,782.715 C 2065.269,782.353 2064.022,779.902 2064.462,781.851 C 2063.653,780.921 2064.859,779.763 2063.634,778.963 C 2062.524,778.239 2061.663,776.976 2060.572,776.164 C 2059.317,775.229 2058.304,773.726 2056.848,773.092 C 2056.33,772.866 2055.775,772.747 2055.261,772.512 C 2054.373,772.106 2054.507,771.449 2053.931,770.839 C 2053.175,770.038 2051.835,769.986 2051.072,769.251 C 2050.112,768.327 2051.389,766.71 2050.364,765.873 C 2049.129,764.864 2048.893,764.854 2048.349,763.265 C 2047.885,761.916 2045.593,760.12 2044.518,759.168" id="path2204"/>
    <path d="M 2047.038,799.272 C 2048.532,800.869 2049.049,796.416 2048.386,795.513 C 2047.604,794.45 2046.356,792.442 2045.121,791.907 C 2044.208,791.511 2043.684,792.799 2042.718,792.215 C 2042.871,793.557 2044.06,794.548 2044.635,795.704 C 2044.992,796.421 2045.821,796.669 2046.239,797.332 C 2046.618,797.935 2046.687,798.658 2047.038,799.272" id="path2208"/>
    <path d="M 2044.229,786.888 C 2044.217,786.168 2043.686,784.981 2042.789,785.664 C 2043.331,785.998 2043.811,786.406 2044.229,786.888" id="path2210"/>
    <path d="M 2053.229,808.703 C 2053.42,808.065 2053.171,807.436 2052.649,807.043 C 2052.252,807.584 2051.96,808.418 2051.805,809.066 C 2051.598,809.934 2053.516,809.967 2053.229,808.703" id="path2212"/>
    <path d="M 2051.928,804.603 C 2052.305,805.671 2052.869,806.65 2053.439,807.624 C 2053.684,806.826 2053.053,804.474 2051.928,804.603" id="path2214"/>
    <path d="M 2053.729,790.56 C 2053.836,789.998 2053.596,789.998 2053.008,790.56 C 2053.248,790.56 2053.488,790.56 2053.729,790.56" id="path2216"/>
    <path d="M 2055.598,803.813 C 2055.402,802.861 2053.897,803.013 2053.369,803.522 C 2054.037,803.967 2054.838,804.021 2055.598,803.813" id="path2218"/>
    <path d="M 2057.838,820.223 C 2060.825,819.802 2056.17,813.976 2056.248,812.593 C 2055.757,812.862 2054.925,812.731 2054.577,813.218 C 2054.077,813.919 2053.758,815.268 2054.29,816.02 C 2055.299,817.442 2055.884,819.803 2057.838,820.223" id="path2220"/>
    <path d="M 2063.378,825.192 C 2063.191,824.421 2062.208,821.707 2061.149,822.43 C 2059.981,823.229 2062.846,824.875 2063.378,825.192" id="path2222"/>
    <path d="M 2064.389,828.653 C 2064.616,829.507 2065.611,829.112 2065.915,828.568 C 2066.369,827.758 2064.883,826.535 2064.309,826.203 C 2064.005,827.026 2064.543,827.82 2064.389,828.653" id="path2224"/>
    <path d="M 2067.768,833.473 C 2067.799,832.922 2067.288,832.508 2067.385,831.968 C 2067.511,831.269 2067.926,831.434 2067.717,830.739 C 2067.566,830.232 2066.944,828.663 2066.171,828.955 C 2065.016,829.392 2067.214,833.014 2067.768,833.473" id="path2226"/>
    <path d="M 2078.139,787.824 C 2077.84,786.4 2075.855,785.962 2075.343,787.501 C 2075.059,788.358 2075.603,789.579 2076.549,789.768 C 2077.691,789.996 2077.728,788.485 2078.139,787.824" id="path2228"/>
    <path d="M 2083.468,791.567 C 2082.396,791.113 2081.16,791.177 2080.149,790.559 C 2079.95,791.613 2080.843,792.092 2081.723,792.253 C 2082.587,792.411 2083.051,793.208 2083.678,793.727 C 2083.678,793.001 2083.745,792.261 2083.468,791.567" id="path2232"/>
    <path d="M 2083.608,794.231 C 2083.168,793.483 2082.522,792.642 2081.597,792.503 C 2081.356,793.818 2081.486,796.171 2083.177,796.462 C 2083.443,795.738 2083.312,794.949 2083.608,794.231" id="path2234"/>
    <path d="M 2087.998,797.833 C 2087.5,796.887 2086.531,796.24 2085.524,795.947 C 2084.964,795.783 2084.453,794.979 2084.089,794.909 C 2083.667,794.828 2083.289,797.22 2083.802,797.562 C 2084.983,798.354 2086.692,797.54 2087.858,798.552 C 2087.905,798.313 2087.951,798.073 2087.998,797.833" id="path2236"/>
    <path d="M 2088.868,797.253 C 2088.563,796.018 2084.647,793.633 2085.188,795.814 C 2086.689,795.142 2087.339,797.738 2088.868,797.253" id="path2238"/>
    <path d="M 2089.878,799.563 C 2089.773,799.05 2089.459,798.76 2088.939,798.692 C 2088.576,799.234 2088.646,799.714 2089.149,800.132 C 2089.415,799.973 2089.658,799.783 2089.878,799.563" id="path2240"/>
    <path d="M 2090.878,795.743 C 2090.676,795.453 2090.412,795.238 2090.088,795.096 C 2090.001,795.703 2090.3,795.946 2090.878,795.743" id="path2242"/>
    <path d="M 2091.458,798.263 C 2091.282,797.696 2090.842,797.32 2090.229,797.543 C 2090.25,798.144 2090.609,798.66 2091.098,798.983 C 2091.218,798.743 2091.338,798.503 2091.458,798.263" id="path2244"/>
    <path d="M 2093.118,806.973 C 2092.592,806.613 2090.639,806.82 2091.435,807.755 C 2091.923,808.327 2093.032,807.526 2093.118,806.973" id="path2248"/>
    <path d="M 2094.768,796.603 C 2094.974,795.85 2094.711,795.707 2093.979,796.173 C 2094.241,796.316 2094.505,796.459 2094.768,796.603" id="path2252"/>
    <path d="M 2096.348,794.808 C 2095.827,793.607 2094.838,794.971 2094.838,795.686 C 2094.838,796.541 2096.455,795.628 2096.348,794.808" id="path2256"/>
    <path d="M 2096.568,797.763 C 2096.963,797.045 2096.769,796.805 2095.988,797.043 C 2096.182,797.283 2096.375,797.523 2096.568,797.763" id="path2258"/>
    <path d="M 2100.309,795.743 C 2099.278,793.226 2096.398,795.591 2097.6,796.038 C 2098.403,796.336 2098.76,795.564 2099.018,796.893 C 2099.277,798.229 2100.337,796.302 2100.309,795.743" id="path2262"/>
    <path d="M 2097.358,798.553 C 2097.351,798.275 2097.282,798.012 2097.148,797.763 C 2097.219,798.026 2097.288,798.29 2097.358,798.553" id="path2264"/>
    <path d="M 2098.948,807.623 C 2098.604,807.898 2097.925,808.037 2097.509,808.172 C 2096.711,808.43 2098.055,810.01 2098.149,810.433 C 2098.202,809.713 2098.57,809.487 2099.159,809.933 C 2099.353,809.074 2099.979,808.228 2098.948,807.623" id="path2266"/>
    <path d="M 2098.518,803.233 C 2098.106,803.686 2098.272,803.949 2099.018,804.022 C 2098.851,803.759 2098.685,803.497 2098.518,803.233" id="path2268"/>
    <path d="M 2099.229,804.603 C 2099.333,805.117 2098.427,807.008 2099.514,806.869 C 2100.121,806.791 2100.489,806.46 2101.096,806.767 C 2101.718,807.082 2102.243,807.586 2102.979,807.193 C 2102.596,806.586 2099.938,804.531 2099.229,804.603" id="path2270"/>
    <path d="M 2099.948,797.973 C 2099.697,797.96 2099.458,798.01 2099.228,798.123 C 2099.469,798.073 2099.708,798.023 2099.948,797.973" id="path2272"/>
    <path d="M 2099.518,802.803 C 2099.194,803.506 2100.242,804.071 2100.678,804.314 C 2100.312,803.794 2099.925,803.291 2099.518,802.803" id="path2274"/>
    <path d="M 2105.498,819.503 C 2106.342,820.661 2104.202,821.185 2103.768,821.953 C 2104.652,822.669 2105.326,823.254 2106.513,822.818 C 2107.313,822.525 2108.638,823.208 2108.866,824.028 C 2109.069,824.754 2109.626,825.625 2109.744,826.264 C 2110.018,827.735 2109.151,828.352 2110.825,829.269 C 2111.544,829.663 2114.798,832.163 2115.579,831.453 C 2116.355,830.746 2115.165,830.418 2114.93,829.834 C 2114.558,828.91 2115.693,827.239 2116.589,827.063 C 2115.45,826.375 2113.887,826.709 2112.839,825.842 C 2111.874,825.043 2111.532,823.065 2111.638,821.887 C 2111.7,821.187 2111.227,818.925 2110.759,818.352 C 2109.439,816.727 2107.73,818.595 2108.949,820.082 C 2108.435,820.152 2105.6,816.236 2105.498,819.503" id="path2282"/>
    <path d="M 2116.518,829.733 C 2116.301,829.803 2116.085,829.872 2115.869,829.942 C 2116.017,830.539 2116.468,830.798 2117.018,830.453 C 2116.879,830.194 2116.712,829.954 2116.518,829.733" id="path2286"/>
    <path d="M 2121.559,829.083 C 2121.224,829.204 2120.934,829.394 2120.689,829.653 C 2121.252,829.922 2121.58,829.706 2121.559,829.083" id="path2290"/>
    <path d="M 2123.348,826.772 C 2122.398,827.739 2122.174,829.41 2122.278,830.686 C 2122.388,832.048 2124.404,832.765 2124.069,831.313 C 2124.662,831.352 2125.548,832.997 2126.135,832.418 C 2126.94,831.624 2127.763,830.016 2127.631,828.849 C 2127.438,827.142 2124.734,826.494 2123.348,826.772" id="path2292"/>
    <path d="M 2125.658,771.336 C 2126.278,771.438 2126.841,771.791 2127.238,772.273 C 2126.654,771.905 2126.268,772.211 2126.228,772.848 C 2128.016,773.105 2128.523,771.311 2127.828,769.957 C 2126.593,767.554 2125.066,769.461 2125.658,771.336" id="path2294"/>
    <path d="M 2130.838,819.503 C 2131.14,819.531 2131.43,819.485 2131.708,819.363 C 2131.405,819.335 2131.115,819.382 2130.838,819.503" id="path2298"/>
    <path d="M 2133.068,818.213 C 2132.663,817.893 2132.256,817.893 2131.847,818.213 C 2132.015,818.497 2132.232,818.736 2132.497,818.933 C 2132.717,818.715 2132.906,818.475 2133.068,818.213" id="path2302"/>
    <path d="M 2139.188,813.603 C 2138.059,812.002 2135.761,814.511 2137.177,815.904 C 2137.855,815.227 2139.642,814.919 2139.188,813.603" id="path2306"/>
    <path d="M 2188.588,833.043 C 2185.978,831.949 2186.815,838.579 2187.359,839.453 C 2188.047,838.319 2188.786,838.664 2188.687,837.111 C 2188.609,835.893 2188.112,834.206 2188.588,833.043" id="path2324"/>
    <path d="M 2189.738,834.123 C 2188.924,834.272 2188.992,835.14 2189.517,835.563 C 2189.634,835.09 2189.708,834.61 2189.738,834.123" id="path2326"/>
    <path d="M 2197.438,775.8 C 2197.282,776.457 2197.522,776.529 2198.158,776.016 C 2197.918,775.944 2197.679,775.872 2197.438,775.8" id="path2332"/>
    <path d="M 2198.309,774.936 C 2197.996,775.631 2198.662,776.241 2199.239,776.449 C 2199.286,775.8 2199.302,774.706 2198.309,774.936" id="path2334"/>
    <path d="M 2198.519,769.176 C 2198.503,769.797 2198.901,770.213 2199.529,769.896 C 2199.455,769.318 2199.095,769.038 2198.519,769.176" id="path2336"/>
    <path d="M 2198.809,807.843 C 2198.583,808.051 2198.487,808.315 2198.519,808.633 C 2198.615,808.37 2198.712,808.106 2198.809,807.843" id="path2338"/>
    <path d="M 2200.038,773.712 C 2200.064,774.229 2200.328,774.565 2200.828,774.72 C 2200.845,774.194 2200.526,773.851 2200.038,773.712" id="path2340"/>
    <path d="M 2206.298,784.584 C 2206.351,785.102 2206.591,785.509 2207.019,785.808 C 2206.801,785.387 2206.561,784.979 2206.298,784.584" id="path2344"/>
    <path class="landxx id" d="M 2132.639,797.113 C 2133.573,796.487 2132.938,795.831 2133.138,794.988 C 2133.399,793.893 2134.988,793.822 2135.159,792.575 C 2134.828,793.119 2134.422,793.598 2133.939,794.014 C 2133.061,792.722 2134.149,791.449 2134.977,790.492 C 2135.517,789.869 2135.678,789.284 2135.959,788.533 C 2136.323,787.56 2137.569,787.334 2138.258,786.671 C 2138.126,787.561 2136.95,788.652 2138.292,789.188 C 2138.483,789.265 2138.964,791.367 2139.341,791.778 C 2140.444,792.98 2142.424,793.921 2143.132,795.455 C 2143.406,796.048 2144.629,796.562 2145.217,796.867 C 2146.319,797.437 2147.872,796.31 2148.823,795.887 C 2150.713,795.044 2152.511,796.158 2154.524,795.962 C 2155.329,795.883 2156.06,795.265 2156.77,794.886 C 2157.832,794.319 2157.531,792.933 2158.335,792.058 C 2159.173,791.146 2163.429,790.222 2164.087,791.617 C 2164.714,792.944 2168.599,794.21 2169.841,793.444 C 2170.914,792.782 2171.145,791.727 2172.601,792.228 C 2173.317,792.475 2173.886,791.888 2174.501,791.643 C 2175.01,791.438 2175.605,792.099 2176.127,792.073 C 2175.846,790.687 2178.787,787.669 2177.979,786.774 C 2176.585,785.23 2179.711,783.26 2180.877,782.857 C 2180.193,782.012 2178.908,778.902 2181.039,778.646 C 2183.716,778.323 2181.711,776.666 2183.243,775.199 C 2183.834,774.634 2183.072,772.322 2183.163,771.349 C 2183.247,770.449 2183.479,769.309 2183.884,768.498 C 2184.234,767.8 2184.95,767.467 2185.127,766.656 C 2186.543,767.183 2186.896,767.464 2188.405,766.815 C 2188.919,766.594 2190.183,768.499 2190.307,766.511 C 2191.285,767.455 2193.132,767.218 2194.352,767.106 C 2195.915,766.963 2196.867,768.662 2198.376,768.528 C 2197.922,768.849 2197.419,769.041 2196.866,769.103 C 2197.138,769.34 2199.847,771.595 2198.016,770.688 C 2198.388,770.95 2199.349,771.125 2199.537,771.561 C 2199.725,771.998 2199.713,772.433 2200.246,772.631 C 2199.383,773.738 2197.975,773.046 2197.586,771.911 C 2197.764,772.4 2197.834,772.904 2197.796,773.423 C 2197.019,772.941 2195.658,772.99 2194.846,773.351 C 2195.875,773.749 2196.407,774.805 2197.517,775.08 C 2197.234,775.646 2197.104,776.895 2196.147,776.448 C 2196.852,777.365 2197.903,777.748 2198.957,778.103 C 2198.081,779.103 2199.776,779.201 2198.377,779.399 C 2198.86,779.418 2199.317,779.538 2199.747,779.759 C 2198.319,780.149 2200.111,780.962 2200.655,781.63 C 2201.276,782.395 2201.826,783.23 2202.337,784.072 C 2202.98,785.131 2201.626,787.214 2200.387,787.175 C 2201.195,787.459 2200.799,788.264 2201.429,788.787 C 2202.336,789.54 2203.242,790.317 2204.187,791.022 C 2204.949,791.59 2205.398,792.562 2206.288,792.882 C 2207.48,793.311 2208.318,794.525 2209.108,795.455 C 2209.441,795.847 2210.308,795.539 2209.703,796.486 C 2208.955,797.656 2207.735,797.075 2206.703,797.598 C 2204.991,798.464 2202.968,796.079 2201.618,795.532 C 2202.241,796.444 2202.732,797.595 2201.274,797.678 C 2199.95,797.752 2199.531,799.771 2199.194,800.763 C 2198.293,803.413 2197.86,806.449 2198.018,809.212 C 2198.399,808.774 2198.878,808.534 2199.457,808.492 C 2199.272,809.086 2198.319,809.796 2199.027,810.072 C 2198.345,810.494 2198.345,811.089 2199.027,811.511 C 2198.091,811.775 2197.051,812.893 2196.797,811.581 C 2196.508,813.438 2193.054,817.355 2192.547,813.601 C 2191.823,814.914 2193.051,815.942 2191.57,816.971 C 2190.887,817.445 2191.224,818.184 2190.747,818.782 C 2190.192,819.476 2189.166,819.453 2188.657,820.221 C 2190.921,819.362 2190.17,822.511 2189.087,823.322 C 2189.646,823.871 2190.791,823.11 2191.261,823.678 C 2191.717,824.231 2190.798,826.368 2190.457,826.851 C 2190.061,826.385 2189.582,826.266 2189.017,826.492 C 2190.434,827.718 2188.579,831.645 2187.578,829.652 C 2187.308,830.527 2188.041,831.28 2188.517,831.891 C 2186.537,832.254 2187.146,835.265 2185.479,835.939 C 2183.651,836.679 2181.806,837.909 2179.948,838.731 C 2179.459,838.947 2175.601,841.56 2175.764,840.087 C 2175.863,839.193 2175.757,838.212 2175.798,837.3 C 2175.857,836 2175.017,834.923 2175.337,833.611 C 2173.875,835.044 2172.875,834.504 2173.898,832.531 C 2173.235,833.653 2171.766,834.375 2172.097,832.39 C 2171.943,833.443 2169.163,835.89 2168.353,834.267 C 2168.073,833.706 2168.848,833.275 2168.311,832.685 C 2167.324,831.602 2166.272,833.564 2165.978,832.1 C 2166.303,833.766 2164.162,830.801 2163.957,830.08 C 2163.97,831.588 2162.346,833.267 2161.065,833.922 C 2159.732,834.604 2158.565,832.685 2157.272,833.618 C 2156.803,833.956 2154.987,836.565 2154.635,834.871 C 2154.207,832.809 2154.905,830.967 2153.807,828.93 C 2152.944,831.163 2152.644,830.123 2150.993,830.077 C 2150.298,830.058 2149.729,830.659 2149.191,830.999 C 2148.168,831.647 2147.498,831.068 2146.466,830.88 C 2146.8,830.516 2147.16,830.179 2147.546,829.87 C 2146.711,829.996 2146.438,831.078 2145.526,830.95 C 2145.908,828.759 2143.7,829.973 2142.796,830.66 C 2141.746,829.35 2142.585,827.178 2142.123,825.667 C 2141.81,824.64 2142.001,823.554 2141.812,822.51 C 2141.653,821.629 2140.275,820.921 2140.366,820.15 C 2140.387,819.969 2141.145,819.036 2141.236,818.767 C 2141.558,817.805 2141.734,816.971 2141.143,816.084 C 2140.739,815.478 2140.654,814.825 2140.31,814.243 C 2139.891,813.535 2138.995,813.273 2139.045,812.3 C 2138.839,812.55 2138.622,812.789 2138.396,813.02 C 2137.633,811.547 2136.822,812.902 2135.78,812.272 C 2135.19,811.916 2135.564,811.258 2135.824,810.855 C 2136.086,810.449 2137.053,810.917 2137.387,810.93 C 2136.824,810.616 2136.489,809.852 2135.866,809.696 C 2135.403,809.579 2134.805,809.666 2134.503,809.201 C 2134.078,808.549 2133.828,805.892 2135.156,806.69 C 2134.811,805.762 2134.238,804.639 2135.376,804.531 C 2134.415,803.262 2133.14,802.159 2132.799,800.546 C 2132.657,799.872 2132.032,797.685 2132.639,797.113" id="path2554"/>
    <path d="M 2195.788,774.144 C 2195.811,773.976 2195.835,773.808 2195.858,773.64 C 2196.627,773.75 2197.341,774.067 2198.018,774.432 C 2197.26,774.57 2196.501,774.392 2195.788,774.144" id="path2556"/>
    <path d="M 2199.098,768.6 C 2199.973,768.484 2201.116,768.319 2200.899,769.609 C 2200.063,769.783 2199.647,769.035 2199.098,768.6" id="path2558"/>
    <path d="M 2324.378,856.803 C 2324.428,857.397 2324.667,857.901 2325.098,858.314 C 2324.903,857.971 2325.247,855.878 2325.399,855.405 C 2325.785,854.198 2324.514,854.824 2324.878,853.773 C 2325.858,854.265 2326.489,852.656 2326.902,852.073 C 2327.61,851.072 2328.539,853.218 2328.526,853.761 C 2328.492,855.182 2328.4,856.708 2328.319,858.129 C 2328.19,860.358 2326.526,859.336 2325.387,858.534 C 2326.856,860.282 2321.572,856.643 2324.378,856.803" id="path2564"/>
    <path d="M 2326.108,859.033 C 2326.408,859.23 2327.347,860.451 2326.729,860.743 C 2325.723,861.219 2325.044,859.284 2326.108,859.033" id="path2566"/>
  </g>
  <g class="landxx coastxx ye" id="ye" transform="translate(-29.9017, -45.0745)">
    <path d="M 1625.188,684.288 C 1625.284,683.957 1625.306,683.621 1625.257,683.28 C 1624.155,683.457 1624.604,683.897 1625.188,684.288" id="path2190"/>
    <path d="M 1698.268,699.552 C 1698.226,698.774 1697.266,699.048 1696.829,699.12 C 1697.308,699.264 1697.788,699.408 1698.268,699.552" id="path2192"/>
    <path d="M 1708.918,698.112 C 1710.603,697.725 1713.527,697.91 1714.389,696.024 C 1713.377,695.417 1712.039,694.487 1710.791,694.768 C 1710.2,694.902 1709.69,695.289 1709.057,695.228 C 1708.387,695.165 1707.812,694.58 1707.116,694.706 C 1706.356,694.843 1705.68,696.323 1706.308,696.877 C 1706.963,697.458 1708.045,697.987 1708.918,698.112" id="path2194"/>
    <path class="landxx ye" d="M 1702.658,660.279 C 1700.232,660.189 1701.6044,655.21807 1699.1984,654.93507 C 1700.2778,653.2208 1696.2761,647.90196 1695.5873,646.026 L 1692.7797,639.53725 C 1685.6548,640.93734 1678.4965,642.47056 1671.4847,643.41794 C 1665.6343,645.22282 1662.6822,648.13037 1661.359,651.297 C 1660.138,653.121 1659.485,657.727 1656.558,657.55 C 1654.566,657.43 1654.783,654.443 1652.867,654.669 C 1650.452,654.954 1648.476,654.909 1646.028,654.565 C 1644.639,654.37 1643.215,653.381 1641.854,653.381 C 1640.347,653.381 1638.846,653.526 1637.348,653.668 C 1635.509,653.842 1633.956,654.668 1632.05,654.563 C 1631.002,654.505 1630.613,653.339 1629.542,653.206 C 1628.518,653.078 1627.73,653.769 1627.301,654.635 C 1626.832,655.58 1626.682,656.726 1626.859,657.764 C 1626.973,658.436 1627.915,659.103 1627.403,659.792 C 1626.919,660.445 1626.535,661.116 1626.037,661.796 C 1625.771,662.168 1625.436,662.461 1625.033,662.676 C 1624.073,663.156 1624.562,663.521 1624.76,664.574 C 1625.072,666.242 1624.444,667.015 1624.147,668.535 C 1623.872,669.942 1625.053,671.165 1624.919,672.564 C 1624.652,672.324 1623.218,672.287 1623.457,672.846 C 1623.406,672.729 1624.904,673.572 1624.901,673.568 C 1625.703,674.424 1626.003,675.31 1626.276,676.43 C 1626.814,678.626 1626.856,681.074 1627.546,683.21 C 1627.829,684.087 1628.533,684.743 1628.867,685.598 C 1629.268,686.629 1628.889,687.747 1628.869,688.802 C 1628.828,690.889 1631.352,692.645 1630.669,694.765 C 1631.59,694.836 1632.211,694.319 1633.143,694.691 C 1634.135,695.087 1634.817,695.414 1635.91,695.197 C 1638.165,694.75 1639.168,693.174 1641.519,694.407 C 1641.398,693.767 1642.08,693.117 1642.31,694.048 C 1642.699,693.62 1642.925,693.123 1643.239,692.647 C 1643.735,691.894 1644.54,691.911 1645.204,691.409 C 1646.684,690.291 1646.852,689.164 1648.89,688.696 C 1650.791,688.259 1652.806,688.176 1654.745,688.354 C 1655.887,688.458 1657.802,687.148 1658.946,686.824 C 1660.653,686.341 1661.498,684.864 1663,684.032 C 1663.461,683.777 1664.111,683.2 1664.61,683.051 C 1665.161,683.069 1665.686,683.2 1666.186,683.442 C 1666.75,683.469 1667.302,683.401 1667.84,683.239 C 1671.468,682.655 1672.284,678.901 1675.545,677.3 C 1677.06,676.556 1678.398,676.368 1680.017,676.085 C 1682.012,675.737 1683.666,674.399 1685.629,673.931 C 1686.994,673.605 1688.377,673.348 1689.701,672.87 C 1691.024,672.392 1691.494,671.338 1692.612,670.767 C 1693.54,670.293 1695.542,670.352 1695.827,669.092 C 1696.158,667.632 1695.144,666.304 1695.674,664.815 C 1696.742,661.806 1700.416,661.925 1702.658,660.279" id="path2592"/>
  </g>
  <g class="landxx coastxx mg" id="mg" transform="translate(-29.9017, -45.0745)">
    <path d="M 1625.398,995.613 C 1625.968,998.016 1624.754,1000.946 1626.46,1003.045 C 1627.293,1004.072 1627.279,1005.225 1627.708,1006.413 C 1628.102,1007.507 1629.197,1007.679 1628.114,1008.921 C 1627.052,1010.139 1627.218,1014.094 1627.912,1015.536 C 1628.543,1016.847 1629.178,1017.82 1629.528,1019.273 C 1629.982,1021.161 1631.208,1021.69 1632.597,1022.833 C 1630.846,1021.817 1632.361,1023.176 1632.798,1023.312 C 1633.439,1023.512 1634.122,1023.444 1634.77,1023.609 C 1636.342,1024.01 1637.196,1025.746 1638.761,1026.092 C 1641.472,1026.691 1644.295,1023.231 1646.79,1022.499 C 1647.707,1022.23 1648.671,1022.776 1649.513,1022.467 C 1650.181,1022.221 1650.761,1021.777 1651.472,1021.641 C 1652.47,1021.451 1653.068,1020.266 1653.45,1019.443 C 1653.965,1018.335 1654.318,1017.156 1654.636,1015.979 C 1655.075,1014.349 1656.08,1013.026 1656.584,1011.461 C 1657.513,1008.571 1658.514,1005.778 1659.309,1002.813 C 1660.44,998.592 1662.141,994.822 1663.601,990.732 C 1664.785,987.417 1665.459,984.027 1666.925,980.818 C 1667.353,979.881 1668.165,978.398 1667.738,977.332 C 1668.242,977.331 1668.813,975.303 1668.939,974.967 C 1669.783,972.723 1670.239,970.351 1671.156,968.139 C 1672.496,964.907 1673.709,962.156 1674.407,958.691 C 1674.928,956.106 1673.161,952.333 1676.304,950.636 C 1676.92,950.303 1677.599,948.351 1677.674,947.581 C 1677.744,946.862 1677.585,946.154 1677.713,945.435 C 1677.886,944.464 1676.691,944.576 1676.706,943.646 C 1676.719,942.94 1675.991,937.465 1677.745,938.211 C 1679.25,938.851 1678.733,941.97 1680.187,942.481 C 1681.814,943.052 1682.878,938.658 1683.059,937.694 C 1683.319,936.303 1682.389,935.609 1682.009,934.374 C 1681.315,932.107 1681.38,929.904 1681.296,927.569 C 1681.141,923.281 1680.116,918.736 1678.387,914.901 C 1678.142,915.06 1677.906,915.23 1677.676,915.411 C 1677.867,914.828 1677.62,914.712 1677.264,914.355 C 1676.684,913.774 1677.588,913.408 1677.069,912.773 C 1676.573,912.168 1676.256,909.89 1675.006,910.651 C 1674.931,910.392 1674.907,910.129 1674.937,909.861 C 1677.108,910.069 1673.946,906.785 1674.305,909.539 C 1674.446,910.622 1672.213,912.622 1671.187,912.6 C 1673.117,913.569 1671.866,916.423 1671.477,917.71 C 1670.906,919.607 1670.934,920.819 1668.596,920.88 C 1668.881,922.253 1668.446,921.167 1667.792,921.959 C 1667.317,922.534 1667.297,923.358 1667.086,924.04 C 1666.502,924.023 1664.353,919.638 1664.046,922.738 C 1663.935,923.846 1664.32,924.568 1664.716,925.561 C 1665.072,926.454 1664.437,928.899 1664.277,927.14 C 1664.116,927.999 1661.454,928.101 1662.504,930.078 C 1663.062,931.127 1665.191,930.716 1664.347,932.4 C 1664.637,931.09 1663.281,930.97 1662.472,931.595 C 1661.34,932.471 1661.437,934.439 1659.888,934.919 C 1659.683,933.81 1660.755,932.959 1660.537,931.819 C 1659.424,932.063 1659.683,932.828 1659.237,933.619 C 1658.804,934.386 1657.548,934.802 1657.367,935.709 C 1657.241,936.338 1657.901,937.043 1658.163,937.556 C 1658.674,938.558 1656.867,938.769 1656.287,939.24 C 1656.405,938.548 1657.67,936.538 1655.892,936.583 C 1654.871,936.609 1654.05,937.881 1653.274,938.412 C 1652.755,938.766 1651.36,939.501 1651.503,940.305 C 1651.575,940.713 1651.241,941.227 1651.388,941.611 C 1651.594,942.145 1652.315,942.047 1652.478,942.69 C 1651.682,942.639 1650.851,942.801 1650.707,941.677 C 1650.526,940.262 1649.14,940.286 1649.087,941.76 C 1648.753,941.208 1647.158,941.049 1646.601,941.322 C 1645.898,941.667 1646.166,942.849 1645.638,943.34 C 1645.638,941.846 1643.926,943.022 1643.827,943.92 C 1643.585,943.458 1643.298,943.024 1642.967,942.62 C 1642.106,943.217 1641.444,944.247 1640.443,944.62 C 1639.504,944.97 1637.838,944.21 1637.075,944.923 C 1636.702,945.272 1637.019,948.118 1636.708,947.81 C 1636.982,948.962 1636.158,950.29 1635.6,951.227 C 1634.666,952.792 1631.672,955.71 1632.821,957.599 C 1633.614,958.902 1632.923,961.253 1633.05,962.771 C 1633.196,964.496 1634.433,965.933 1634.426,967.629 C 1634.42,969.208 1634.32,970.209 1635.174,971.61 C 1636.092,973.114 1635.569,973.791 1635.117,975.256 C 1634.793,976.309 1636.119,976.635 1635.494,977.766 C 1634.828,978.971 1633.96,980.119 1633.236,981.298 C 1632.459,982.565 1631.498,983.831 1630.942,985.215 C 1630.513,986.286 1630.865,988.072 1629.506,988.547 C 1627.873,989.117 1627.487,989.149 1627.329,991.024 C 1627.198,992.584 1625.783,994.014 1625.398,995.613" id="path2622"/>
    <path d="M 1667.448,920.733 C 1667.804,920.107 1667.375,919.441 1666.948,919.003 C 1666.241,919.586 1666.404,920.749 1667.448,920.733" id="path2624"/>
    <path d="M 1678.758,949.603 C 1678.758,949.46 1678.758,949.316 1678.758,949.173 C 1678.322,949.975 1677.221,950.877 1677.678,951.913 C 1678.135,951.188 1678.336,950.345 1678.758,949.603" id="path2628"/>
  </g>
  <g class="landxx bo" id="bo">
    <path d="M 742.08629,854.27855 C 743.99329,854.40255 745.90929,854.56755 747.81129,854.71055 C 748.68329,854.77555 748.23629,854.92055 748.36629,855.44055 C 748.60129,856.38055 749.95729,855.41755 750.33829,855.25255 C 750.77829,855.06155 751.65329,854.87955 751.93629,854.43355 C 752.37129,853.74755 752.51029,852.77755 753.16229,852.24055 C 754.60629,851.05155 755.75629,852.96855 756.54129,851.05155 C 757.22429,849.38255 759.34629,849.30355 760.65729,848.37855 C 761.87829,847.51655 762.49729,845.73555 764.01329,845.29355 C 765.82229,844.76655 767.84329,843.90955 769.76329,844.17655 C 771.88829,844.47255 771.41829,844.66455 772.82729,843.33855 C 775.88229,845.45055 772.60629,849.47355 773.49429,852.18755 C 773.71629,852.86755 774.27829,853.25655 774.51429,853.85755 C 774.77829,854.53055 774.13029,855.22455 773.97329,855.83655 C 773.50629,857.66355 775.28929,858.64755 775.69329,860.00855 C 775.98429,860.98855 775.89229,860.87455 776.73629,861.59355 C 777.25929,862.03955 777.12829,863.15055 777.61129,863.43055 C 778.26129,863.80755 778.62159,863.47173 779.31529,864.04055 C 779.97829,864.5842 780.61729,864.89898 781.38529,865.51855 C 781.4885,865.60181 781.91208,866.52515 782.08508,866.71415 C 782.59908,867.27915 783.13029,867.43055 783.84729,867.59855 C 784.82129,867.82755 785.58508,868.22555 786.54208,867.69355 C 787.32708,867.25755 787.89129,867.46355 788.60429,867.96055 C 789.18329,868.36355 789.66529,868.89055 790.26329,869.26955 C 791.08929,869.79155 791.24829,868.70755 791.90829,868.89955 C 793.15329,869.26155 793.18929,871.59855 794.80429,871.77955 C 796.21529,871.93755 797.19629,873.05555 798.59529,873.13155 C 800.53529,873.23655 800.81929,876.11155 802.44029,876.52855 C 803.77329,876.87155 805.05429,876.64655 806.36729,876.44555 C 808.23229,876.16055 810.11829,877.64355 811.66529,878.51955 C 814.19829,879.95455 812.57929,881.31955 813.49329,883.39655 C 814.50029,885.68555 814.88329,887.70455 815.37829,890.13955 C 814.75329,890.12155 814.12929,890.14455 813.50829,890.20955 C 813.58629,891.98355 815.84729,893.00955 816.14529,894.91655 C 816.36829,896.34155 816.21329,898.01355 816.83729,899.34055 C 817.62829,901.02155 820.70729,900.44955 822.22129,900.59955 C 825.07029,900.88155 828.02729,901.29555 830.85829,900.64855 C 830.89929,901.91755 830.15729,902.69255 830.06129,903.83755 C 829.98229,904.78255 830.36629,905.97855 830.61129,906.89155 C 830.99929,908.33355 832.68329,910.57855 834.25129,910.87055 C 835.05029,911.01955 835.47029,910.48655 835.83129,911.49455 C 836.10529,912.25955 836.46929,912.96355 836.77429,913.71955 C 837.57729,915.70555 838.61829,916.81155 838.15829,918.96555 C 837.92729,920.04855 837.01929,921.91055 837.39329,923.01955 C 837.91929,924.58055 836.80529,925.90955 836.20129,927.41255 C 835.38629,929.43855 835.34429,930.36655 836.99229,931.75655 C 838.21729,932.78955 836.25229,933.43055 835.39829,934.05955 C 835.98929,932.10655 833.52931,929.90729 831.99631,928.92929 C 830.38731,927.90229 828.57172,927.09964 826.71839,926.70336 C 821.58358,925.60544 816.2301,928.47107 811.34075,928.97582 C 808.94689,929.22295 807.75402,929.45873 807.37002,932.03873 C 807.05002,934.18273 805.5011,935.36028 805.1521,937.44028 C 804.8241,939.39828 805.66529,941.30655 805.26529,943.32855 C 804.69829,946.19655 804.55456,949.31873 803.86056,952.15873 C 801.67254,950.75706 802.59106,949.80941 800.60429,949.89382 C 799.05854,949.9595 797.17838,949.58804 795.73338,950.07174 C 794.00943,950.6488 794.11038,949.31082 793.19338,951.64482 C 792.40838,953.64182 792.12729,955.35355 791.90829,957.46155 C 789.72029,954.41555 790.08129,951.46655 785.87529,950.79455 C 784.16229,950.52055 781.65629,951.33755 780.13129,950.59555 C 778.79629,949.94555 778.02329,948.32655 776.49829,948.02155 C 776.28729,950.18955 776.41029,950.87255 774.23129,951.66355 C 773.14029,952.06055 771.9476,954.1548 771.34229,955.19255 C 770.44286,956.73455 770.09413,957.40255 768.00329,957.69055 C 767.24126,957.79551 766.02987,957.41495 765.67248,956.96765 C 764.32641,953.46418 763.941,951.34718 762.42911,948.52637 C 761.12286,946.19902 761.31791,944.29636 760.30284,942.72291 C 759.70969,941.80349 758.64369,941.04607 758.10784,940.14928 C 757.40484,938.97277 758.48229,937.80029 758.02129,937.84155 C 757.13932,937.92049 756.55429,936.99355 756.15129,936.18955 C 755.70029,935.28955 756.76229,934.67755 755.54829,933.55255 C 757.68829,932.25955 757.51229,932.46155 756.00529,930.42855 C 755.78629,930.13255 756.66179,928.88449 756.76929,928.60755 C 757.21505,927.45917 756.68874,926.91632 755.71029,926.08155 C 753.76832,924.42475 752.87429,923.03655 752.22329,920.58855 C 751.92929,919.48055 750.22429,916.64755 751.29929,915.62255 C 747.68998,914.88724 747.65922,909.44195 746.18929,909.14355 C 746.31029,908.47755 746.80229,908.12155 747.31229,907.75755 C 748.09429,907.20055 748.34229,906.29055 748.88229,905.52855 C 749.35029,904.86755 749.68929,904.26355 750.32229,903.75455 C 750.41829,903.67755 750.22152,902.57947 750.40529,902.67554 C 750.9669,902.96914 752.35127,903.25641 750.95776,901.61895 C 751.40078,901.90606 753.06875,900.04034 753.03635,900.55502 C 752.92289,898.6821 750.00862,901.78581 751.14936,897.46301 C 749.33261,897.15752 750.86369,896.70276 746.92547,894.11519 C 747.12347,893.08519 748.39229,892.00255 747.87529,890.92955 C 747.06329,889.24155 745.79429,888.93055 746.84629,886.83855 C 747.52429,885.48855 748.22729,884.47555 748.68129,883.13855 C 749.22029,881.55255 749.54129,881.61655 748.60929,879.93755 C 747.75329,878.39455 748.38529,877.15455 748.23229,875.49955 C 748.10829,874.16555 747.29529,871.42955 748.30329,870.29655 C 749.30429,869.17055 750.22529,868.40455 749.36229,866.93355 C 746.91929,862.76755 744.32129,858.54955 742.08629,854.27855" id="path2375"/>
    <path d="M 750.24716,899.89622 C 750.26405,899.89762 749.7766,899.86692 749.13517,899.68022 C 748.83492,899.59283 749.36535,898.78795 749.64135,898.64095 C 749.96435,898.95595 750.57016,899.68222 750.24716,899.89622" id="path5225"/>
  </g>
  <g id="rs">
    <path class="landxx rs rs-" d="m 1412.8746,360.84017 c -1.1632,-0.49295 -2.1938,1.27374 -2.5954,1.43631 -0.7054,0.28554 -1.9268,0.0736 -3.3766,0.55435 0.408,1.008 0.7094,1.59184 0.9323,2.64583 0.1401,0.66501 0.3472,0.85695 1.0332,0.88194 -1.97,1.36902 1.7062,2.20823 2.3182,2.82222 -0.09,-0.09 -1.9916,0.42815 -2.2426,0.52916 0.6052,1.20267 -0.038,1.8186 -0.2181,2.16706 0.9093,-0.35381 1.7201,-0.58904 1.9315,-0.12599 0.4959,1.08595 0.6043,1.0026 -1.0583,3.52777 -0.5471,0.83102 1.218,2.06874 2.7214,3.12459 1.6291,1.51246 -1.1564,0.78697 -1.5875,1.13393 -0.2585,0.0647 2.2908,2.87288 1.7553,3.12845 -1.408,0.67198 -2.8812,0.80328 -3.9172,0.82883 0.4945,-0.0759 1.7541,1.45429 3.6321,2.95755 0.5973,0.47811 0.7326,0.97785 2.0804,1.43364 1.0681,0.63632 2.9998,0.0298 3.3183,1.27864 0.1368,0.088 -1.4794,0.89156 -1.3798,0.93196 0.1404,0.0569 0.9754,1.42975 0.6812,1.57768 1.9232,0.62106 -0.3816,1.80513 2.9945,2.44744 1.0199,0.19405 0.9895,4.74483 1.0994,3.64245 1.4581,-1.23144 1.347,-3.4604 4.7303,-2.23916 0.036,-1.0878 1.1719,-0.9067 1.7762,-1.33072 0.6862,0.70034 1.4447,-0.71777 2.5061,-0.39451 0.6618,0.20156 5.1305,-0.28197 4.0065,-1.94027 -0.6473,-0.95494 -0.8096,-2.07719 -0.252,-2.67103 0.5378,-0.57278 1.2127,0.055 1.8143,-0.70554 0.5905,-0.74644 1.0643,-1.47966 1.3859,-2.24266 -0.991,-0.31399 -2.2648,-1.60724 -3.0238,-2.31824 -2.441,-2.288 -1.2228,-4.13559 0.1512,-6.42558 -0.313,-0.287 -2.0137,-2.01423 -1.3607,-2.44424 0.547,-0.35898 1.2573,-0.13217 1.8143,-0.40317 -0.549,-0.67201 -1.9842,-1.57715 -2.8222,-0.90714 -0.265,0.213 -1.0353,1.5249 -1.3104,1.5119 -0.376,-0.018 -1.0919,-1.03654 -1.5119,-1.20952 -0.98,-0.40401 -2.0788,-0.72252 -3.0238,-1.20952 -0.7006,-0.36079 -0.272,-0.23339 -0.99,-0.49091 0.9067,-0.48648 1.2833,-1.2698 0.032,-1.42416 1.403,-1.31899 0.2651,-1.59192 -0.9179,-2.02893 -1.215,-0.44801 -2.7658,-1.54378 -3.8188,-2.25479 1.3149,-2.60901 -1.4946,-2.62458 -1.8648,-4.43491 -0.3305,-1.61668 -3.73,-1.64496 -5.4428,-1.36071 z" id="rs-"/>
    <g id="xk">
      <path class="limitxx xk rs" d="m 1416.3623,389.94751 c 0.3886,-0.28762 1.2014,-0.80838 1.6044,-0.62738 0.102,-0.687 0.1751,0.0958 0.446,-0.36136 0.4402,-0.74276 0.9736,-0.63922 1.4596,-0.84822 0.995,-0.429 0.9674,-0.884 0.4974,-1.977 0.473,-0.064 1.0663,-0.39132 1.4593,-0.66332 0.302,1.962 5.1327,1.2728 4.7927,3.2238 -0.172,0.986 1.7403,1.33666 1.9483,2.14666 0.305,1.186 -1.9177,2.39958 -1.0437,3.35758 -0.792,0.36608 -2.0137,0.41644 -1.7187,1.37906 -1.2018,-0.14952 -0.9944,-0.8207 -2.752,-0.11302 -1.4004,0.56382 -1.3094,2.19826 -2.0306,2.41717 -0.4785,-0.13207 0.2003,-2.39307 -0.7893,-3.16287 -0.2401,-0.86349 -1.0971,-0.55671 -1.9566,-1.21121 -0.5657,-0.91442 -0.1561,-1.37997 -1.2131,-1.71337 -0.2133,-0.49318 -1.1792,-1.45054 -0.7037,-1.84652" id="xk-"/>
      <circle class="unxx rs xk" cx="1423.1255" cy="389.55505" id="xk_" r="4.2338"/>
    </g>
  </g>
  <path class="landxx ci" d="M 1205.0463,703.22155 C 1205.4743,702.29655 1206.5863,702.09455 1206.9753,701.10055 C 1207.2323,700.44155 1207.4323,699.38955 1207.4763,698.68655 C 1207.5573,697.41155 1207.0503,695.09455 1206.1263,694.14955 C 1206.7173,693.52555 1207.5333,693.84655 1208.1363,694.22155 C 1208.2663,693.46555 1209.4793,692.64155 1209.0013,691.80955 C 1208.6143,691.13655 1209.2483,689.70355 1210.1563,690.18955 C 1209.4683,689.16755 1208.2673,688.53355 1207.9573,687.27255 C 1207.3733,684.89955 1211.8073,686.76555 1212.5263,687.09355 C 1212.0093,685.51755 1212.2033,683.75955 1210.1563,683.42155 C 1210.5293,682.36955 1210.5343,681.24855 1211.8863,681.04555 C 1210.7623,679.71455 1210.5503,680.00255 1210.9463,678.02155 C 1208.5033,679.59455 1208.607,674.18217 1208.9263,672.98155 C 1209.2336,671.82601 1210.0541,671.53216 1211.0739,671.23195 C 1211.6261,671.0694 1211.6653,670.03555 1212.0963,669.66955 C 1212.7313,669.13155 1213.4793,669.47955 1214.1863,669.16555 C 1214.0343,670.56955 1216.8723,671.97155 1218.0063,671.46955 C 1217.2413,670.20255 1218.5413,670.13055 1219.4673,670.14555 C 1220.7043,670.16455 1220.2673,668.11255 1220.3053,667.29355 C 1220.7923,667.56455 1221.2983,668.48255 1221.9653,668.30155 C 1222.6053,668.12655 1222.7323,666.92355 1223.4053,666.64555 C 1223.9563,667.64355 1223.1722,671.00242 1224.6113,671.35955 C 1225.912,671.68234 1224.8646,668.85207 1228.8023,668.97058 C 1229.1841,668.98207 1227.9675,670.10664 1231.1181,670.30557 C 1232.6007,670.39919 1232.9073,673.32892 1233.1083,673.56855 C 1234.1493,674.80953 1236.0825,676.11736 1238.0153,676.50955 C 1238.0333,674.76255 1240.6063,674.38555 1241.8353,673.91755 C 1242.9593,673.48955 1245.8643,674.77455 1246.5853,673.70155 C 1247.791,675.13272 1248.2783,676.69128 1249.8953,678.23655 C 1250.0783,678.00555 1250.2483,677.76555 1250.4053,677.51655 C 1250.8791,678.94563 1249.7717,680.31128 1249.9773,681.03255 C 1250.3197,682.23406 1251.0293,682.81355 1251.1913,684.20255 C 1251.3253,685.35255 1252.0373,687.29355 1251.6143,688.37655 C 1251.1373,689.59555 1250.5229,690.24855 1249.9269,691.35555 C 1249.1329,692.82955 1247.7838,693.68681 1247.8279,695.32062 C 1247.8852,697.44794 1246.8352,698.99807 1246.4023,700.57515 C 1245.9443,702.2437 1246.1578,704.93442 1247.1853,706.28755 C 1248.1975,707.62055 1248.6663,709.41955 1248.2783,710.85255 C 1250.8393,710.14355 1249.3393,714.03255 1249.7563,715.24555 C 1249.2413,715.23655 1248.7373,715.30855 1248.2463,715.46155 C 1248.3943,715.18655 1248.5613,714.92155 1248.7463,714.66955 C 1247.5423,716.47755 1246.5483,714.20255 1246.2963,713.30155 C 1245.6453,714.02155 1245.4003,715.01955 1244.2863,715.10155 C 1243.5653,715.15455 1242.6433,714.81355 1242.1263,714.30955 C 1241.7283,715.04255 1240.8153,714.34855 1240.4663,714.02155 C 1241.0933,713.88355 1241.7913,714.49655 1242.2663,713.80555 C 1242.0593,713.57155 1241.8203,713.37955 1241.5463,713.22955 C 1241.8443,714.66755 1241.2553,713.32855 1240.2563,713.58955 C 1240.4633,714.01655 1237.8873,713.77255 1237.2613,713.84155 C 1236.9413,713.87755 1233.7433,714.44855 1234.0013,714.58755 C 1235.6053,715.45455 1238.7493,712.68155 1240.0363,714.30955 C 1238.0423,714.41855 1236.1343,714.74655 1234.1813,715.09355 C 1233.7413,715.17155 1232.8283,715.35955 1232.5473,714.84055 C 1232.1493,714.10955 1230.2193,714.44255 1229.5273,715.10155 C 1230.6613,714.64755 1231.2893,714.76855 1232.4073,715.17355 C 1229.9553,715.45455 1227.3423,715.32055 1225.0313,716.31855 C 1223.0353,717.17955 1221.2433,718.79855 1219.0123,719.02555 C 1217.8763,719.14155 1215.4503,720.42555 1214.6153,721.22955 C 1212.7383,723.03655 1212.4093,719.69855 1212.7013,718.32355 C 1213.0483,716.68755 1213.6143,714.88155 1214.0673,713.52855 C 1214.6693,711.73455 1214.1553,709.77655 1212.7353,708.58455 C 1211.8733,707.86155 1210.6213,707.53655 1210.7623,706.02155 C 1210.9243,704.28355 1205.6223,705.79555 1206.6963,703.79755 C 1206.0613,703.94255 1205.4873,703.65955 1205.0463,703.22155" id="ci"/>
  <path class="landxx dz" d="M 1208.8563,523.43755 L 1209.3663,512.56555 C 1208.8952,509.64211 1213.3773,508.67755 1214.9913,507.20755 C 1215.7915,506.47876 1217.0995,504.98963 1218.3312,505.06637 C 1219.4027,505.13312 1219.6506,503.5187 1221.0593,503.24555 C 1223.3526,502.80087 1224.5286,505.02464 1225.9076,501.98564 C 1226.3356,501.04464 1227.7643,501.43864 1228.5853,501.33064 C 1230.0633,501.13464 1231.5523,500.93355 1232.9473,500.41755 C 1235.7493,499.38155 1235.8651,496.37237 1238.2541,495.11037 C 1239.2391,494.58937 1240.5283,494.53155 1241.5673,494.14755 C 1242.8463,493.67355 1243.6863,492.23955 1245.0053,491.97455 C 1245.4883,491.87755 1248.0874,491.66637 1247.0244,490.29637 C 1246.3374,489.41137 1245.2453,489.08855 1246.6553,487.79955 C 1244.6383,487.24555 1246.4593,485.13155 1247.9393,485.25555 C 1249.8973,485.41955 1251.7033,484.72155 1252.3973,482.88855 C 1253.7273,479.37755 1262.7903,482.48755 1265.3753,481.82355 C 1262.4113,480.48579 1267.5402,478.85219 1265.9293,477.61055 C 1265.1686,477.02419 1264.3173,476.50655 1263.6993,475.75655 C 1262.9833,474.88955 1263.3563,473.64655 1262.6533,472.58455 C 1261.9183,471.47455 1262.3257,470.56109 1262.5377,469.45609 C 1262.7367,468.42009 1261.9373,468.40755 1262.0493,467.40255 C 1262.1543,466.46355 1262.1399,465.33237 1262.1519,464.39737 C 1262.1689,463.14237 1261.3323,462.13455 1262.5763,461.16055 C 1261.2833,460.63255 1261.8064,459.91182 1261.6194,458.94482 C 1261.3704,457.65682 1259.1367,457.23882 1258.6477,455.91382 C 1261.0427,455.90782 1262.0663,455.32655 1264.1013,454.28655 C 1264.9873,453.83355 1265.5453,453.83255 1266.0803,452.89855 C 1266.5533,452.07155 1266.9103,451.37755 1267.7893,450.90755 C 1268.2853,450.64155 1269.2463,449.86955 1269.7703,450.28355 C 1270.4503,450.82055 1271.3083,449.88755 1271.7603,449.47355 C 1272.9343,448.40055 1274.5463,451.10155 1275.7463,448.63355 C 1277.3553,445.32655 1282.9003,443.89555 1286.1483,443.52155 C 1288.4603,443.25555 1290.7963,442.61955 1293.1263,442.77655 C 1294.8643,442.89355 1296.0613,440.67755 1297.5693,441.53055 C 1298.2253,441.90155 1298.0763,441.69355 1298.3263,441.28055 C 1298.3813,441.19055 1300.9873,441.22855 1301.3753,441.07455 C 1303.6273,440.18155 1305.9013,440.51855 1308.2693,440.45255 C 1310.2423,440.39755 1310.7113,440.80655 1312.2133,441.93955 C 1314.1323,443.38655 1314.7983,441.24655 1316.4623,441.04255 C 1316.9973,440.97755 1320.3173,440.83355 1320.1353,439.85055 C 1319.8073,438.08055 1322.2243,439.17055 1322.7993,439.55655 C 1324.0003,440.36155 1325.8073,440.76855 1327.0383,439.78455 C 1327.7143,439.24355 1327.2733,438.53955 1328.5603,438.86055 C 1329.4743,439.08855 1330.1913,439.76055 1331.1853,439.70755 C 1330.3433,441.09855 1332.4313,440.61155 1333.0913,440.43955 C 1334.4083,440.09655 1335.8313,440.41755 1337.0953,439.99455 C 1337.1903,441.78855 1334.9163,442.66755 1333.9253,443.81055 C 1336.6243,444.39255 1334.0353,448.18755 1334.5803,449.92655 C 1335.5273,452.94755 1335.1323,455.23055 1334.4953,458.21055 C 1333.9713,460.66255 1331.2423,462.57255 1329.4193,464.18155 C 1328.3563,465.11955 1328.8733,467.56955 1329.4943,468.60855 C 1330.4983,470.28955 1330.0413,472.12155 1332.1103,472.90755 C 1334.9353,473.97955 1334.1063,477.55855 1336.3833,479.26255 C 1337.4393,480.05355 1338.6853,480.62555 1339.5793,481.61855 C 1340.7873,482.96155 1340.9643,485.16255 1341.2453,486.86655 C 1341.8253,490.38655 1343.4953,494.29755 1343.4953,497.84655 C 1343.4953,498.80455 1341.4813,498.43809 1342.7803,500.17009 C 1344.0373,501.84909 1345.0779,503.8229 1345.4933,505.88855 C 1346.9232,512.99892 1344.8215,512.95496 1345.8933,516.60055 C 1346.4047,518.33994 1346.4318,519.61146 1345.7558,521.30446 C 1344.9927,523.21545 1345.6885,525.00182 1346.1035,526.90882 C 1346.7295,529.79182 1345.4473,529.83764 1343.3983,531.38564 C 1341.0633,533.15164 1344.9453,536.58155 1345.9583,538.13955 C 1347.8523,541.05155 1346.7133,544.50855 1349.2323,547.22755 C 1350.1993,548.27155 1351.3113,547.01955 1352.3433,547.09755 C 1353.6493,547.19655 1355.0663,548.18155 1356.2923,548.62655 C 1359.0903,549.64255 1360.0303,551.99055 1361.0623,554.64855 C 1361.5523,555.90755 1362.0463,556.05955 1360.6823,556.83355 C 1346.4891,565.88635 1332.1714,574.94385 1320.0763,586.71855 C 1319.3806,587.73138 1317.1303,590.26255 1315.3823,591.22855 C 1314.0559,591.96158 1312.1043,592.02855 1310.6423,592.33155 C 1308.6553,592.74255 1306.6633,593.13255 1304.6783,593.55355 C 1303.6823,593.76455 1294.9543,596.76355 1295.3293,593.67455 C 1295.4613,592.58555 1296.2393,592.16355 1296.1323,590.95755 C 1296.0913,590.49455 1296.2301,588.06264 1295.7481,587.82164 C 1294.2811,587.08764 1292.4489,586.75173 1290.8299,586.44073 C 1290.0319,586.28673 1288.9599,584.25714 1288.3505,584.40673 C 1284.8532,585.26519 1284.3029,581.77993 1283.6743,581.25855 C 1282.9316,580.64256 1281.6503,580.58655 1280.9703,580.06955 C 1280.2053,579.48855 1280.8639,578.04546 1280.4039,577.26546 C 1279.8259,576.28446 1278.8223,575.43055 1277.8813,574.87355 C 1276.2693,573.91955 1274.7593,572.78655 1273.2823,571.63955 C 1269.6123,568.79055 1266.0063,565.85655 1262.3533,562.98555 C 1248.2653,551.90955 1234.1963,540.87455 1219.4563,530.66955 C 1215.9393,528.22955 1212.4043,525.82455 1208.8563,523.43755" id="dz"/>
  <path class="landxx ch" d="M 1319.5263,361.14955 C 1319.7503,360.71455 1319.9933,360.29255 1320.2553,359.88055 C 1320.5913,359.23355 1320.0754,358.71442 1320.306,358.10637 C 1320.6509,357.19711 1322.2116,356.48975 1322.265,356.20037 C 1322.4806,355.03381 1323.0599,354.2935 1324.7186,353.3465 C 1325.6142,352.83511 1326.1854,351.77382 1327.0507,350.92655 C 1325.4382,351.12047 1326.4123,348.59372 1327.7383,349.79255 C 1329.1652,351.08269 1329.8043,348.96155 1331.1853,348.76655 C 1330.4653,349.60755 1334.1073,348.96155 1334.7763,348.99655 C 1336.0753,349.06355 1337.1028,349.10637 1337.3838,348.28037 C 1336.3898,348.65137 1336.1893,347.72466 1337.0906,347.41473 C 1337.4564,347.28894 1337.8257,347.42454 1338.3908,347.64664 C 1339.3357,348.0181 1340.5405,348.04137 1340.7319,348.11328 C 1341.845,348.53126 1343.2343,348.68755 1344.1373,349.47955 C 1345.3253,350.52055 1343.5603,352.21655 1343.3553,353.37455 C 1345.6523,353.08555 1345.8403,354.74855 1347.7153,355.11255 C 1348.5253,355.26955 1348.8463,353.73855 1349.7313,353.98955 C 1350.7663,354.28255 1349.8813,355.44355 1349.7923,355.97655 C 1349.7233,356.39055 1350.6323,357.68455 1349.7073,357.70955 C 1349.0433,357.72755 1348.5253,356.53455 1347.8963,357.09055 C 1347.6693,357.29055 1346.8903,358.46655 1347.4333,358.71655 C 1348.3943,359.15755 1347.8253,359.49255 1347.6753,360.42955 C 1346.7383,360.05355 1346.6993,359.04155 1345.5153,359.27755 C 1344.3593,359.50755 1344.2323,359.75255 1343.2853,358.84555 C 1343.1923,358.75655 1342.1623,357.00355 1342.0243,358.43655 C 1341.8973,359.75055 1341.9223,360.07055 1340.9713,361.06355 C 1339.7603,362.32955 1340.8493,365.29755 1338.9133,362.46355 C 1338.4813,361.83255 1337.9713,361.52355 1337.3753,361.07755 C 1336.4643,360.39755 1336.8273,359.10555 1335.9353,358.48455 C 1335.4243,359.12655 1334.8213,359.77555 1333.9953,359.99755 C 1334.8983,361.04455 1332.9193,362.88655 1331.8053,362.78355 C 1329.7053,362.58955 1327.9703,363.91955 1326.3663,362.22955 C 1325.8013,361.63355 1325.0163,361.00755 1325.2113,360.06955 C 1325.4843,358.75855 1324.4482,358.44362 1323.3272,358.55719 C 1321.4467,358.74772 1321.7668,361.09335 1319.5263,361.14955" id="ch"/>
  <path class="landxx cm" d="M 1336.1563,717.90855 C 1337.0993,717.10355 1337.6847,715.91618 1337.7263,714.72055 C 1337.7651,713.60364 1339.2271,711.443 1338.8146,710.67442 C 1338.3943,709.89129 1338.2233,709.57832 1339.3833,708.27555 C 1340.5339,706.98332 1341.6645,705.66668 1342.9355,704.48568 C 1343.6645,703.80668 1344.5848,703.51711 1344.9638,702.57611 C 1345.2908,701.76511 1346.5557,699.61449 1347.4587,699.35949 C 1348.4517,699.07849 1349.0293,700.39455 1350.2463,700.12855 C 1351.2103,699.91755 1351.2663,698.57755 1351.4863,697.82055 C 1351.6673,698.59055 1353.7392,700.69396 1354.4713,700.88955 C 1355.2036,701.0852 1355.4028,702.12113 1355.1169,702.90753 C 1354.9103,703.47582 1355.9449,704.32833 1357.0651,704.0648 C 1357.4036,703.48251 1359.2694,701.75891 1358.8103,700.71955 C 1358.4917,699.9984 1359.9983,699.03555 1360.5203,698.61755 C 1361.3883,697.92355 1360.5196,696.86821 1360.9886,695.93358 C 1361.908,694.10092 1363.2072,692.4248 1363.8344,690.12855 C 1364.1003,689.15513 1363.7473,688.12996 1364.0253,687.21055 C 1364.2405,686.49886 1365.3715,686.25372 1365.3143,685.07033 C 1364.9752,684.97124 1367.8258,685.35313 1368.4022,682.55655 C 1368.5322,681.92588 1369.6253,678.80555 1368.5573,678.59655 C 1369.7853,677.97055 1371.598,677.17455 1371.804,675.77155 C 1371.938,674.85655 1371.885,674.09908 1371.885,673.18308 C 1371.885,672.22908 1372.8446,672.31168 1373.275,671.65761 C 1373.7988,670.86154 1373.7387,669.73517 1374.0627,668.73598 C 1374.4006,667.69381 1374.7058,666.69283 1374.8403,666.06155 C 1375.1019,664.83419 1375.7967,662.53254 1377.0811,662.01402 C 1377.7513,661.74347 1381.7641,661.77313 1382.279,659.49921 C 1382.4239,658.85914 1381.7586,658.36075 1381.8543,657.61555 C 1382.0199,656.32573 1382.1037,655.58939 1382.3233,654.61055 C 1382.5802,653.46586 1380.4785,652.47938 1379.8033,652.69155 C 1379.2507,652.89674 1378.5298,651.71673 1378.2661,650.24109 C 1378.0548,647.89643 1378.181,647.01678 1377.8473,646.34155 C 1378.7173,646.34155 1380.0676,646.26515 1380.7433,647.01939 C 1381.7466,648.13938 1380.7116,649.2605 1382.9291,649.87268 C 1384.0726,650.18834 1384.2048,652.85254 1384.2758,653.77364 C 1384.3466,654.6936 1384.1682,653.92248 1385.1187,654.82642 C 1387.0288,656.64301 1384.9718,662.0083 1385.2363,664.38755 C 1385.6039,667.69369 1387.1713,671.46755 1390.1553,673.12555 C 1388.9517,674.16385 1387.4272,673.67437 1386.4423,673.18374 C 1385.4392,673.27825 1385.2141,673.43603 1384.1511,673.47649 C 1381.9241,673.56124 1379.6557,672.11794 1378.5273,673.95455 C 1377.1575,676.18398 1377.3265,676.56983 1379.1485,678.65583 C 1380.9735,680.74583 1382.9798,682.55983 1384.9878,684.46783 C 1385.5948,685.04383 1386.3938,685.92702 1386.7978,686.67968 C 1387.5843,688.14464 1388.2391,691.13677 1389.1024,692.56255 C 1389.9804,694.01266 1388.2197,695.03086 1387.7806,695.61852 C 1387.0809,696.55491 1386.1184,698.72204 1385.4872,700.23927 C 1385.278,700.74221 1384.779,701.52074 1384.573,702.03174 C 1384.193,702.97574 1383.6137,703.72757 1383.176,704.64836 C 1382.5691,705.92504 1379.4444,707.18057 1381.2413,707.90268 C 1382.5552,708.43067 1382.443,709.77255 1382.264,711.51755 C 1382.183,712.30655 1382.139,713.01614 1381.698,713.69314 C 1381.748,713.61614 1382.6551,714.86442 1382.6691,714.95442 C 1382.8921,716.42642 1382.5397,718.68029 1383.5187,719.86183 C 1384.124,720.59232 1384.9341,720.86621 1385.208,722.00136 C 1385.4121,722.84725 1385.7743,723.98355 1386.5553,724.53355 C 1384.0895,724.59655 1389.642,731.143 1390.1144,731.52055 C 1391.3152,732.48028 1392.9949,733.0474 1393.21,733.91455 C 1393.3716,734.56557 1393.4981,734.92336 1393.5271,735.55636 C 1393.6001,737.15836 1393.9683,738.82255 1394.3243,740.37255 C 1392.4464,741.25098 1394.5284,744.37642 1393.3243,745.26855 C 1392.2523,744.21555 1391.0673,743.26855 1389.6173,742.80455 C 1388.9883,742.60255 1387.1088,741.84034 1386.598,742.18214 C 1385.9401,742.62231 1384.6321,742.54567 1384.223,741.56314 C 1384.2026,741.51429 1382.0236,741.09852 1381.8106,741.01452 C 1381.1326,740.74852 1379.2123,740.95055 1378.3273,740.89855 C 1376.7143,740.80555 1375.0748,740.82552 1373.4668,740.59852 C 1373.1938,740.56052 1372.3983,740.81455 1372.0853,740.66155 C 1371.4803,740.36455 1372.3563,739.75455 1371.2313,739.86655 C 1369.7453,740.01355 1368.4463,740.25455 1366.9323,739.92955 C 1365.1553,739.54855 1363.7003,739.84755 1361.9153,739.79255 C 1360.7513,739.75655 1358.6653,739.09455 1357.5613,739.70355 C 1357.2463,739.87755 1357.3653,740.80655 1357.0263,740.80655 C 1355.9243,740.80655 1354.8233,740.80655 1353.7213,740.80655 C 1352.1853,740.80655 1350.6593,740.86355 1349.1253,740.89755 C 1345.8333,740.97155 1345.1273,739.16255 1345.8203,736.10855 C 1346.2153,734.36855 1346.8153,732.48155 1345.8753,730.79755 C 1345.4323,730.00355 1343.9093,728.65955 1345.5153,728.13355 C 1344.6103,728.65355 1342.8563,727.79755 1343.9083,726.74955 C 1344.4623,726.19755 1345.6953,725.23955 1344.0753,725.18155 C 1344.3903,724.83355 1344.5563,724.42555 1344.5753,723.95755 C 1344.0683,725.18955 1343.5303,724.20955 1342.7053,724.24555 C 1343.0103,724.65355 1343.2253,724.94555 1342.7053,724.67755 C 1342.7373,725.07155 1342.6903,725.45555 1342.5653,725.82955 C 1341.9033,725.80555 1341.5833,725.29355 1341.7753,724.67755 C 1341.2043,725.71855 1339.5693,724.41355 1339.3583,723.64755 C 1339.0253,722.43655 1338.9553,720.36555 1338.0943,719.42155 C 1337.1053,720.63955 1337.2963,719.20555 1336.6543,719.20555 C 1336.7703,719.67955 1336.8433,720.15855 1336.8743,720.64455 C 1334.8313,720.90055 1335.9433,718.98455 1336.1563,717.90855" id="cm"/>
  <path class="landxx mk" d="M 1420.0363,400.24555 C 1420.9343,400.11355 1420.3383,399.38355 1420.3283,398.84555 C 1420.3143,398.09855 1421.6313,397.72455 1421.8123,396.80855 C 1421.9703,396.00955 1422.6433,395.61855 1423.3123,395.29655 C 1424.7933,394.58655 1424.5243,395.52155 1425.8663,395.56555 C 1425.5813,394.48155 1426.3513,394.88555 1426.8763,394.34155 C 1427.1843,394.18655 1427.4913,394.19155 1427.8013,394.35755 C 1428.5383,394.57055 1428.6113,393.87455 1429.2463,393.76555 C 1429.9653,393.64255 1431.4583,393.92955 1432.0393,393.49055 C 1432.7703,392.93755 1433.5493,394.56655 1434.0073,394.98955 C 1435.7133,396.56455 1437.2024,397.03166 1437.1926,399.80399 C 1437.1906,400.38199 1437.5822,401.34009 1437.2493,401.99655 C 1437.0588,402.37225 1436.3558,401.95171 1436.4193,402.57841 C 1436.508,403.45487 1434.8288,404.06151 1433.8239,403.76115 C 1432.6963,403.42411 1431.5633,403.70955 1430.5953,404.41455 C 1428.6743,405.81155 1427.1523,405.81855 1424.7983,406.19655 C 1423.5323,406.39955 1424.0733,405.52955 1423.0063,405.80255 C 1421.6963,406.13855 1422.3373,405.00055 1421.6473,404.46255 C 1420.2583,403.38055 1420.5873,401.78555 1420.0363,400.24555" id="mk"/>
  <path class="landxx bw" d="M 1421.3263,973.79855 L 1422.1797,949.81883 L 1428.9946,949.82735 L 1429.9539,917.98357 C 1432.3705,917.39933 1438.5112,917.03887 1442.0099,915.8271 C 1443.0981,915.45021 1445.7628,914.88446 1446.8548,915.51046 C 1448.3338,916.35746 1448.6183,918.12055 1449.3373,919.50855 C 1451.2569,917.99129 1452.7693,914.88514 1455.0391,914.83246 C 1455.8883,914.81275 1455.8008,915.30946 1456.4218,915.39546 C 1457.2258,915.50646 1457.6974,914.40316 1458.299,914.04428 C 1460.2456,912.88296 1462.2787,913.04657 1461.9338,913.75082 C 1461.8364,913.94958 1461.9343,915.05955 1462.0283,915.29655 C 1462.5483,916.60855 1463.2789,917.86045 1464.1363,918.97755 C 1464.9763,920.07194 1465.6525,920.6584 1466.2803,922.54255 C 1466.7287,923.88848 1466.8972,924.17391 1467.2842,925.45191 C 1468.0972,928.13491 1470.0503,930.42726 1472.4912,931.68491 C 1475.0735,933.01542 1477.4406,932.74293 1476.5563,936.85655 C 1478.9139,936.30575 1480.7598,936.58177 1479.7403,939.71355 C 1479.3895,940.79129 1479.5083,941.61055 1480.1283,942.54855 C 1480.7993,943.56655 1481.2673,945.99085 1482.3431,946.32473 C 1484.5013,946.99454 1486.3707,946.88223 1488.6759,947.6681 C 1490.3876,948.25162 1488.9063,949.05611 1489.4343,949.935 C 1489.8415,950.61278 1491.3628,950.39715 1491.6035,951.36937 C 1488.9133,951.70255 1489.3432,952.27362 1488.3493,953.66355 C 1487.2389,955.21639 1484.2047,954.44366 1482.8905,955.85964 C 1481.7059,957.13599 1480.7132,959.27105 1479.1146,960.53 C 1478.9372,960.66965 1478.3174,961.24585 1477.6673,961.70255 C 1476.5851,962.4628 1475.3946,962.74022 1474.5713,963.67255 C 1473.4734,964.91581 1473.2606,965.27969 1473.0708,967.14791 C 1472.8376,969.44192 1471.4983,969.96216 1469.871,971.90136 C 1468.9148,973.04085 1467.3368,973.35619 1465.7339,973.61719 C 1464.578,973.80541 1465.2628,974.25255 1465.0418,975.37255 C 1464.7508,976.84555 1464.0358,978.18055 1463.5598,979.59255 C 1462.8508,981.68855 1461.7342,982.41991 1459.5922,982.23391 C 1457.6482,982.06591 1457.8754,983.46061 1455.3741,982.91117 C 1454.3813,982.6931 1452.004,981.58279 1451.1567,981.52564 C 1449.9159,981.44195 1449.1075,980.67008 1448.3939,979.77655 C 1447.3319,978.44655 1444.4355,977.47205 1442.9378,979.40373 C 1441.569,981.16922 1442.1732,983.13937 1440.9111,984.95501 C 1440.2885,985.85068 1439.4018,986.46582 1438.5588,987.04382 C 1437.5168,987.75682 1437.2136,988.90755 1436.3286,989.73355 C 1435.6696,990.34955 1435.1049,990.08219 1434.3679,990.38219 C 1434.1229,990.59219 1434.0992,990.83655 1434.1552,991.18555 C 1433.8682,991.98255 1432.6658,992.204 1431.9688,992.159 C 1430.8598,992.086 1430.239,991.81083 1429.1694,991.72646 C 1428.1985,991.64989 1427.332,991.2685 1426.0883,992.29855 C 1424.2761,988.84165 1428.0608,987.27701 1426.7373,982.57855 C 1426.2352,980.79603 1425.5243,978.81855 1424.6063,977.20855 C 1423.8043,975.80255 1422.2523,975.10255 1421.3263,973.79855" id="bw"/>
  <path class="landxx ke" d="M 1530.7577,768.59905 C 1532.2762,767.49265 1528.4746,762.88701 1533.7418,763.9683 C 1533.7531,763.6813 1534.0422,763.08727 1533.7219,762.543 L 1536.8505,762.13724 C 1534.7733,758.99154 1533.8806,761.45984 1532.8927,762.32309 C 1532.0485,763.06084 1529.9822,759.79571 1530.0893,757.80592 C 1530.8901,756.36229 1530.3595,755.70587 1531.8513,754.35255 C 1532.9584,753.34816 1533.838,750.03562 1535.6353,749.22655 C 1536.4881,748.84268 1536.3623,748.71155 1536.3153,747.95755 C 1536.2433,746.79355 1537.5993,746.23355 1537.8963,745.19755 C 1538.2493,743.96655 1538.3506,742.76055 1538.4726,741.52555 C 1538.5816,740.41455 1537.4643,739.62827 1537.2313,738.38255 C 1536.7085,735.58772 1534.9883,734.98055 1534.7313,733.74955 C 1534.5293,732.77255 1533.9853,731.94355 1533.6573,731.01355 C 1533.4383,730.60455 1533.3153,730.16955 1533.2873,729.70655 C 1533.2873,729.08355 1533.9853,728.72955 1533.7973,728.06055 C 1533.6303,727.47055 1532.9843,727.31555 1532.4863,727.12955 C 1531.3563,726.70855 1532.1833,725.85055 1530.9873,725.97255 C 1531.4762,724.85155 1530.0813,723.77355 1530.2152,722.99055 C 1530.3495,722.20529 1531.3063,721.59055 1531.8403,721.03055 C 1533.7033,719.07555 1535.9863,719.63655 1538.4303,719.63655 C 1539.8323,719.63655 1541.2373,719.52055 1542.6393,719.50555 C 1543.3263,719.49855 1544.0413,719.46455 1544.7233,719.57255 C 1545.3133,719.66555 1545.4943,720.69955 1546.2023,720.97355 C 1548.3833,721.82055 1550.8173,720.78655 1553.0433,721.81955 C 1555.3703,722.90055 1557.8603,724.89055 1559.6753,726.66855 C 1560.6003,727.57455 1561.3353,728.21055 1562.6693,728.35655 C 1563.5363,728.45155 1565.5113,727.91355 1566.1923,728.34855 C 1568.0963,729.56655 1571.0173,728.48455 1572.6753,730.07755 C 1572.9043,728.94455 1574.4633,727.29555 1575.1653,726.31955 C 1576.4133,724.58355 1579.0823,723.88855 1580.9263,723.02855 C 1582.8943,722.11155 1583.2393,724.27055 1584.7093,725.12955 C 1585.7183,725.71955 1586.8113,725.33455 1587.8603,725.14455 C 1588.7103,724.98955 1589.7253,725.55055 1590.5243,725.10955 C 1589.2993,727.21055 1588.1713,729.50655 1586.6793,731.43155 C 1585.1033,733.46355 1583.6903,734.59455 1583.6953,737.32455 C 1583.7063,743.77355 1583.7633,750.22155 1583.7573,756.67155 C 1583.7513,761.50455 1582.7893,766.08755 1586.0343,770.03655 C 1587.1343,771.37555 1588.7913,773.10455 1587.6103,774.85955 C 1586.9153,775.89455 1584.8443,777.78455 1583.5453,776.58955 C 1583.4413,776.84955 1583.3433,777.11255 1583.2553,777.37955 C 1583.0473,777.12955 1582.8573,776.86555 1582.6853,776.58955 C 1583.0253,777.37555 1583.8953,778.82455 1583.1153,779.61855 C 1582.9803,777.66055 1582.1713,779.90355 1581.9743,780.35455 C 1581.5713,781.28055 1580.7953,781.45855 1579.9113,781.58355 C 1577.9493,781.85955 1577.1583,783.59255 1577.7833,785.24555 C 1578.0693,786.00255 1576.6853,788.19255 1576.1153,788.65555 C 1575.1433,789.44455 1574.3273,794.75355 1573.2453,794.29955 C 1573.7173,796.35555 1571.6613,797.81655 1571.3053,799.69855 C 1570.4683,799.08955 1570.5643,800.15455 1569.8653,799.98955 C 1568.9833,799.78155 1567.8582,798.24284 1567.1773,797.72455 C 1564.1682,795.43408 1560.3932,792.94216 1558.5409,790.46893 C 1558.0514,789.81536 1557.1523,789.71055 1558.1773,788.57855 C 1559.6263,786.97855 1557.0907,785.05047 1555.7859,784.23442 L 1530.7126,768.55399" id="ke"/>
  <g class="landxx coastxx tw" id="tw" transform="translate(-29.9017, -45.0745)">
    <path d="M 2183.979,592.56 C 2183.72,592.776 2183.434,592.944 2183.119,593.064 C 2183.57,593.331 2184.05,593.403 2184.559,593.281 C 2184.388,593.021 2184.194,592.781 2183.979,592.56" id="path2738"/>
    <path d="M 2198.158,601.128 C 2198.349,601.845 2198.735,602.443 2198.5,603.204 C 2198.307,603.828 2198.103,604.373 2198.647,604.904 C 2199.649,605.88 2199.98,607.486 2200.614,608.752 C 2201.351,610.223 2202.959,610.648 2203.902,611.919 C 2204.205,612.327 2204.848,614.963 2205.6,614.57 C 2206.572,614.063 2206,612.396 2205.888,611.585 C 2205.708,610.277 2206.079,609.298 2206.819,608.231 C 2208.587,605.68 2208.905,600.938 2208.701,597.922 C 2208.659,597.298 2208.246,596.361 2208.451,595.747 C 2208.657,595.13 2208.983,594.775 2209.055,594.095 C 2209.22,592.512 2208.678,591.355 2208.678,589.824 C 2208.678,589.032 2209.462,588.391 2208.744,587.746 C 2208.306,587.352 2206.033,585.553 2205.462,586.123 C 2204.558,587.026 2202.758,587.25 2202.25,588.519 C 2201.727,589.828 2201.874,590.594 2200.941,591.716 C 2200.09,592.74 2199.899,594.207 2199.558,595.449 C 2199.058,597.281 2197.727,599.147 2198.158,601.128" id="path2746"/>
  </g>
  <path class="landxx jo" d="M 1530.4073,505.50955 C 1530.2693,503.09555 1531.2413,500.79655 1531.2643,498.39655 C 1531.2913,495.63755 1532.5223,493.59155 1532.8563,490.96555 C 1532.9963,489.87155 1532.6803,488.75555 1532.7663,487.64455 C 1532.8483,486.59455 1533.1978,485.56155 1533.2368,484.53855 C 1533.3168,482.39455 1533.0232,479.8195 1532.9442,477.6555 C 1532.8782,475.8185 1535.1703,476.38355 1536.2173,477.10755 C 1537.9423,478.29955 1539.5016,480.10541 1541.8206,479.64941 C 1544.1556,479.18941 1546.3043,476.99355 1548.2353,475.70655 C 1550.7033,474.06255 1553.1673,472.24455 1555.7573,470.80555 C 1556.5563,473.36555 1557.6043,475.83655 1558.4873,478.36655 C 1557.5973,478.46055 1558.1783,479.67555 1558.7513,479.82255 C 1559.6593,480.05555 1560.8049,479.91273 1559.3359,481.45373 C 1554.7274,484.27094 1550.1117,485.76424 1544.5722,487.146 L 1552.8316,495.81314 C 1552.0886,496.17814 1550.1241,496.13361 1549.5951,496.79961 C 1548.9871,497.56561 1549.6759,499.11214 1549.0259,499.87814 C 1547.9349,501.16314 1545.181,500.06891 1543.767,501.41891 C 1542.154,502.95891 1541.6265,505.85414 1539.3225,506.76114 C 1537.2375,507.58014 1532.5203,506.05655 1530.4073,505.50955" id="jo"/>
  <g class="landxx coastxx mx" id="mx" transform="translate(-29.9017, -45.0745)">
    <path d="M 497.737,588.312 C 497.841,588.85 498.033,589.354 498.313,589.825 C 498.325,588.92 497.832,588.095 497.161,587.52 C 497.353,587.783 497.545,588.048 497.737,588.312" id="path2614"/>
    <path d="M 433.296,552.024 C 432.382,552.483 432.439,553.509 432.936,554.255 C 433.428,553.526 432.805,552.754 433.296,552.024" id="path2594"/>
    <path d="M 453.672,559.08 C 453.178,559.659 451.655,561.564 452.952,562.103 C 453.383,561.205 454.058,560.12 453.672,559.08" id="path2596"/>
    <path d="M 467.137,589.752 C 467.005,590.321 467.31,590.559 467.857,590.472 C 467.536,591.045 467.716,591.661 468.216,592.056 C 466.812,590.274 469.024,587.824 468.936,586.008 C 468.871,587.383 467.926,588.69 467.137,589.752" id="path2600"/>
    <path d="M 470.592,553.536 C 470.777,552.671 470.201,548.99 469.073,548.934 C 467.352,548.851 470.305,553.287 470.592,553.536" id="path2602"/>
    <path d="M 468.577,592.272 C 469.148,592.843 469.642,593.559 470.521,593.639 C 470.25,592.809 469.485,592.166 468.577,592.272" id="path2604"/>
    <path d="M 476.137,555.84 C 477.292,554.618 477.083,553.034 477.721,551.592 C 475.699,551.88 475.782,553.717 474.625,554.976 C 475.118,555.282 475.622,555.57 476.137,555.84" id="path2606"/>
    <path d="M 478.729,579.024 C 477.73,579.158 477.362,580.07 477.361,580.967 C 477.673,580.215 478.768,579.945 478.729,579.024" id="path2608"/>
    <path d="M 479.305,587.304 C 479.159,588.06 479.335,588.816 479.737,589.464 C 480.685,588.799 480.08,587.761 479.305,587.304" id="path2610"/>
    <path d="M 484.417,595.872 C 484.54,595.291 484.302,594.721 483.841,594.36 C 483.833,594.94 484.025,595.444 484.417,595.872" id="path2612"/>
    <path d="M 498.384,587.88 C 498.107,588.456 498.227,588.936 498.744,589.32 C 498.89,588.774 498.77,588.294 498.384,587.88" id="path2616"/>
    <path d="M 503.641,616.968 C 503.571,617.586 503.956,617.993 504.577,617.976 C 504.585,617.399 504.23,616.982 503.641,616.968" id="path2618"/>
    <path class="landxx mx" d="M 452.088,523.152 C 458.105,522.557 464.111,521.759 470.16,521.567 C 469.706,522.174 469.066,522.632 468.792,523.368 L 492.43383,533.32074 L 512.655,533.43 L 512.71954,529.60302 L 525.671,529.58 C 527.442,529.602 527.652,530.371 528.063,531.779 C 528.407,532.959 529.891,533.222 530.472,534.403 C 530.976,535.428 531.902,537.167 532.845,537.827 C 533.735,538.45 535.033,538.933 535.275,540.131 C 535.625,541.864 535.597,543.354 535.375,545.063 C 534.882,548.851 538.225,550.525 540.867,552.311 C 541.789,552.934 543.218,554.258 544.375,553.547 C 545.209,553.034 546.202,552.153 546.648,551.261 C 547.222,550.112 548.375,548.286 549.404,547.492 C 550.052,546.992 551.435,546.806 552.196,546.45 C 554.24698,546.40402 556.34922,546.88858 558.416,547.139 C 559.005,547.257 559.74,548.806 560.31,549.336 C 561.244,550.203 562.45279,552.1964 562.69779,553.4154 C 563.02879,555.0664 563.443,556.751 563.537,558.439 C 563.636,560.213 564.771,561.28 565.348,562.894 C 565.613,563.632 565.868,564.374 566.244,565.065 C 566.539,565.607 567.069,565.588 567.342,566.049 C 568.013,567.179 567.064,569.001 567.057,570.203 C 567.049,571.715 568.42359,572.789 568.23459,574.435 C 568.83619,576.95738 570.7005,576.4252 569.927,576.507 C 570.9385,576.6064 571.58565,577.38669 572.69103,578.0644 C 573.40824,578.50412 573.66281,579.02263 574.439,579.047 C 576.23071,579.10324 577.61094,578.81925 578.849,580.358 C 579.87107,581.6283 580.628,579.66962 581.907,579.65662 C 578.28784,585.48011 576.17814,591.88415 573.75217,598.20936 C 572.8698,602.76731 574.35739,593.02857 577.89685,586.9293 C 578.5636,585.78034 578.80386,584.28153 578.00694,584.46698 C 578.09789,584.47652 577.4949,586.33378 576.54874,585.05311 C 576.52385,585.66628 575.78072,586.48643 575.79817,586.58519 C 576.35881,589.75884 575.72188,589.28989 575.70492,590.22957 C 575.70361,590.30234 574.69315,592.79569 574.65364,592.90291 C 574.61413,593.01012 573.95792,592.36403 573.88348,592.44237 C 573.94336,593.85055 574.35037,593.70461 574.31872,594.30663 C 574.28704,594.90929 573.65699,595.99256 573.53847,598.13901 C 573.51064,598.64294 573.51157,598.35896 573.051,598.823 C 574.107,600.46 572.681,602.808 572.511,604.511 C 572.3,606.63 570.952,608.344 570.844,610.52 C 570.733,612.75 570.98,614.767 572.303,616.62 C 573.726,618.613 573.076,618.529 572.259,620.567 C 572.169,619.763 573.171,618.63 572.676,617.964 C 571.739,616.704 571.532,615.234 570.819,613.871 C 571.142,615.125 570.625,616.459 571.107,617.687 C 571.542,618.793 572.096,618.907 572.181,620.25 C 572.358,623.054 573.482,625.962 574.891,628.313 C 576.512,631.016 577.079,633.195 577.791,636.203 C 578.089,637.463 579.018,638.399 579.434,639.602 C 579.793,640.642 579.919,641.594 581.043,642.094 C 580.441,642.048 580.017,641.321 579.603,641.735 C 580.897,642.081 579.593,643.73 579.171,644.255 C 580.386,644.198 579.301,642.901 580.351,642.702 C 580.729,642.725 580.96,642.547 581.043,642.167 C 581.717,642.546 582.508,642.594 583.26,642.625 C 585.698,642.725 587.731,643.986 588.854,646.074 C 589.809,647.848 590.808,647.704 592.808,647.204 C 594.065,646.889 596.227,646.464 597.172,645.479 C 596.405,645.664 595.641,645.904 594.94,646.272 C 595.945,645.67 603.769,643.871 604.012,644.975 C 603.559,642.959 609.495,641.871 610.277,643.463 C 608.696,643.179 610.284,644.554 610.061,645.263 C 610.595,644.808 611.025,645.026 611.645,644.975 C 612.025,645.056 612.409,645.104 612.797,645.12 C 612.894,644.837 613.028,644.572 613.199,644.325 C 614.079,643.974 616.096,642.034 613.877,641.88 C 614.365,641.475 614.869,641.091 615.389,640.728 C 614.68,641.154 613.982,641.604 613.229,641.953 C 614.109,640.472 616.086,639.911 617.491,639.088 C 619.449,637.942 619.425,636.273 620.316,634.437 C 620.779,633.481 621.618,633.033 622.246,632.229 C 622.879,631.418 622.718,630.308 622.823,629.354 C 623.074,627.072 624.277,625.419 624.822,623.306 C 624.662,623.594 624.47,623.858 624.246,624.098 C 625.402,620.873 629.982,620.257 632.889,619.924 C 634.986,619.684 636.662,619.122 638.639,618.394 C 641.221,617.443 643.422,618.527 645.991,618.77 C 645.553,618.59 645.096,618.495 644.623,618.482 C 645.789,618.183 646.928,618.902 648.007,619.203 C 646.181,618.828 649.332,617.653 650.586,618.224 C 652.458,619.077 651.878,621.482 651.282,623 C 650.492,625.014 648.823,626.12 647.374,627.618 C 646.615,628.404 645.713,629.39 645.343,630.435 C 645.135,631.021 645.528,633.399 644.911,632.091 C 645.203,633.26 643.18,634.296 643.039,635.475 C 646.623,635.032 642.018,637.287 642.823,638.427 C 643.044,637.887 643.513,637.486 644.119,637.491 C 643.128,638.894 642.84,640.625 642.081,642.141 C 641.461,643.38 640.069,645.819 640.014,647.21 C 638.332,645.342 639.609,643.922 639.222,641.882 C 638.106,642.659 638.173,642.688 637.598,643.743 C 637.134,644.594 636.408,644.563 635.656,644.982 C 634.059,645.871 632.918,648.399 632.094,649.946 C 631.599,649.405 630.5459,648.69572 630.1289,649.99772 C 629.8079,651.00072 629.248,650.45 627.838,650.45 C 624.741,650.45 621.645,650.45 618.548,650.45 C 617.845,650.45 615.866,650.15518 615.72,651.23018 C 615.601,652.10518 615.644,652.846 615.563,653.728 C 615.347,656.077 613.634,655.346 611.79,655.346 C 613.09668,656.671 614.05438,658.324 615.624,659.162 C 617.09914,659.94956 616.93759,661.64397 618.39491,662.35046 C 618.69632,662.49658 618.03201,665.75209 617.94518,665.53336 C 615.50314,666.46694 611.61147,665.36578 608.17664,665.46265 C 603.55562,674.07059 604.20389,672.80586 604.46551,673.83698 C 604.81651,674.40598 603.252,678.111 602.719,678.746 C 602.022,677.874 601.236,677.078 600.537,676.207 C 600.132,675.702 598.279,672.157 599.551,674.065 C 599.285,673.016 597.999,672.602 597.301,671.908 C 596.277,670.889 595.56,669.634 594.463,668.634 C 592.263,666.628 591.082,664.757 588.104,663.77 C 587.91,664.906 590.724,665.419 591.272,666.218 C 589.535,665.314 587.739,664.427 585.728,664.49 C 586.255,664.288 586.679,663.865 586.88,663.338 C 586.29,663.252 585.81,663.444 585.44,663.914 C 585.582,662.157 583.943,662.71 583.712,663.985 C 584.245,663.931 584.749,664.027 585.224,664.274 C 582.544,664.454 581.247,666.07 578.814,666.801 C 576.617,667.461 574.792,668.458 572.558,668.896 C 570.872,669.227 570.152,669.019 568.495,668.125 C 566.654,667.131 565.495,666.996 563.45,666.673 C 562.382,666.505 561.733,665.952 560.948,665.258 C 559.948,664.374 558.97,664.186 557.762,663.744 C 556.783,663.386 556.567,662.556 555.832,661.93 C 554.936,661.167 553.361,661.016 552.259,660.735 C 551.171,660.457 549.922,660.348 548.905,659.852 C 547.84,659.331 547.048,658.454 545.938,657.963 C 543.374,656.829 540.792,656.188 538.528,654.446 C 536.66,653.008 535.645,650.993 533.987,649.474 C 533.338,648.879 532.497,649.117 531.733,649.266 C 530.582,649.492 528.635,648.674 527.553,648.29 C 525.755,647.652 523.284,646.785 522.122,645.182 C 521.546,644.388 521.149,643.503 520.663,642.658 C 519.836,641.22 518.647,641.167 517.401,640.299 C 517.135,638.964 513.791,638.281 512.619,637.227 C 512.004,636.674 512.187,635.792 511.585,635.195 C 510.908,634.524 510.28,632.972 509.89,632.089 C 509.432,631.054 508.547,628.21 509.84,627.412 C 510.739,626.857 511.783,627.157 512.552,626.303 C 513.271,625.504 511.657,624.838 511.064,624.964 C 511.397,623.887 515.227,620.276 514.154,618.945 C 513.167,617.711 512.799,616.993 512.28,615.481 C 511.721,613.855 512.41,612.605 512.234,610.98 C 511.855,607.492 508.527,604.578 506.97,601.531 C 505.173,598.014 503.236,595.416 500.12,592.924 C 500.969,593.292 501.662,594.341 502.064,592.996 C 500.706,593.15 498.376,591.866 498.919,590.373 C 499.31,589.298 499.413,586.198 497.456,587.019 C 497.456,585.745 497.266,586.709 496.37,586.236 C 495.759,585.912 495.243,585.368 494.577,585.148 C 495.456,585.324 495.531,585.263 494.8,584.965 C 494.408,584.517 494.507,583.87 494.073,583.419 C 493.631,583.816 493.127,584.105 492.561,584.283 C 492.353,583.372 493.447,583.294 493.569,582.482 C 492.478,583.486 491.92,582.878 490.545,582.554 C 490.813,582.392 491.077,582.224 491.337,582.05 C 490.699,581.517 490.387,581.661 490.401,582.482 C 490.365,581.44 490.519,580.409 491.01,579.479 C 491.193,579.133 492.533,576.767 492.345,578.091 C 492.625,577.536 493.272,576.489 493.713,577.371 C 493.859,576.618 493.751,575.977 492.849,576.579 C 493.227,575.651 493.091,574.46 492.158,573.895 C 491.419,573.448 490.635,574.248 489.836,573.529 C 488.97,572.749 490,570.357 488.934,570.07 C 488.194,569.871 485.806,569.612 485.745,568.56 C 485.679,567.444 485.593,563.672 487.017,563.691 C 486.334,563.611 484.7,563.364 484.929,562.754 C 484.619,563.028 484.403,563.364 484.282,563.762 C 484.242,562.698 482.778,562.637 482.229,561.941 C 481.76,561.343 481.61,560.326 481.332,559.629 C 481.045,558.913 480.225,558.878 479.817,558.29 C 479.188,557.382 479.375,555.727 478.689,554.964 C 477.5,553.642 478.413,552.387 478.017,550.874 C 476.748,551.855 477.166,549.115 477.009,548.498 C 476.489,546.452 476.484,545.278 476.623,543.227 C 476.737,541.551 475.553,540.03 475.95,538.356 C 476.135,537.58 477.465,535.627 477.212,535.034 C 476.797,534.061 475.197,534.073 474.355,533.776 C 472.989,533.295 474.763,531.468 472.608,531.315 C 471.874,531.263 471.577,532.116 470.753,532.054 C 469.965,531.993 469.32,531.367 468.837,530.808 C 467.82,529.627 466.474,529.267 465.418,528.051 C 465.505,528.963 466.541,529.808 466.165,530.75 C 465.699,531.914 464.861,533.194 464.288,534.345 C 463.707,535.508 464.578,536.653 464.339,537.843 C 464.077,539.15 463.976,540.512 463.478,541.761 C 462.261,544.804 463.608,545.999 465.517,548.241 C 467.251,550.278 467.714,551.716 467.58,554.331 C 468.811,553.396 469.391,555.613 469.564,556.41 C 469.655,556.829 469.608,557.674 469.952,557.992 C 470.229,558.249 471.346,558.502 471.086,559.072 C 470.489,560.386 470.128,563.984 471.297,565.084 C 472.875,566.57 472.683,569.229 474.564,570.314 C 473.482,571.121 474.41,573.714 474.492,574.85 C 475.223,574.216 474.788,572.682 474.636,571.898 C 478.297,572.518 475.496,584.272 477.948,583.707 C 477.76,584.899 477.662,586.28 478.02,587.451 C 478.398,588.688 479.378,589.356 478.465,590.661 C 477.279,592.357 478.094,595.553 480.54,595.299 C 478.913,595.464 480.349,595.517 480.573,595.195 C 480.845,594.803 480.875,594.303 481.188,593.93 C 481.567,594.866 483.079,596.18 483.845,596.806 C 484.266,597.151 484.163,599.305 484.428,599.906 C 485.03,601.27 486.99,602.231 485.215,603.882 C 484.511,604.536 481.946,606.591 480.991,606.165 C 479.737,605.604 480.385,602.812 479.87,601.747 C 479.493,600.966 478.81,600.388 478.164,599.835 C 477.121,598.943 476.565,597.643 475.664,596.629 C 474.35,595.151 472.287,594.507 471.193,592.79 C 470.276,591.35 468.047,590.511 468.588,588.459 C 468.456,588.951 468.288,589.431 468.084,589.898 C 468.297,585.743 471.722,583.094 469.669,578.79 C 468.877,577.129 467.08,576.917 466.023,575.376 C 465.541,574.673 464.802,573.577 464.772,572.691 C 464.751,572.054 465.501,571.905 465.276,571.178 C 464.83,571.729 464.774,572.813 464.052,573.122 C 463.317,573.437 462.71,572.783 461.954,573.332 C 461.342,573.777 460.966,572.821 460.797,572.418 C 460.392,571.451 459.624,571.281 458.868,570.673 C 458.722,570.377 458.54,570.107 458.321,569.86 C 457.9,569.761 457.482,569.648 457.068,569.521 C 456.306,568.96 456.667,568.653 456.523,567.907 C 456.181,566.132 453.446,565.828 453.324,563.833 C 455.044,564.005 457.217,564.931 458.723,563.617 C 458.663,565.487 461.979,564.994 461.027,566.929 C 461.902,566.245 462.346,563.889 461.387,563.114 C 461.396,563.557 461.009,565.353 460.139,564.926 C 459.627,564.674 460.334,563.206 460.379,562.826 C 460.11,562.987 459.847,563.155 459.587,563.33 C 460.436,561.485 463.1,559.719 462.035,557.426 C 460.929,555.042 459.663,551.946 457.495,550.343 C 455.712,549.025 454.276,547.97 453.85,545.67 C 453.69,544.807 454.286,544.187 454.471,543.367 C 454.714,542.291 453.688,541.827 454.187,541.227 C 453.961,541.436 453.721,541.628 453.467,541.803 C 453,539.869 454.986,539.017 453.371,537.225 C 452.631,536.404 453.539,535.772 453.539,534.891 C 453.539,533.707 452.59,532.929 452.414,531.84 C 452.283,531.027 452.733,530.363 453.278,529.853 C 453.755,529.406 453.111,528.696 452.912,528.256 C 452.683,527.967 452.484,527.658 452.315,527.331 C 452.42,526.852 452.516,526.373 452.603,525.891 C 452.42,524.852 452.27095,525.20234 452.84395,524.08434" id="path2630"/>
    <path d="M 610.417,643.32 C 611.432,643.546 612.116,642.485 613.009,642.24 C 612.066,642.103 611.178,642.905 610.417,643.32" id="path2634"/>
    <path d="M 649.008,629.353 C 649.969,628.594 650.888,627.606 651.384,626.472 C 650.478,627.298 648.312,627.366 649.008,629.353" id="path2772"/>
  </g>
  <g class="landxx coastxx ae" id="ae" transform="translate(-29.9017, -45.0745)">
    <path d="M 1703.018,595.872 C 1703.518,595.905 1703.901,595.713 1704.168,595.296 C 1703.515,594.756 1702.689,594.883 1702.298,595.656 C 1702.538,595.729 1702.777,595.8 1703.018,595.872" id="path2704"/>
    <path class="landxx ae" d="M 1686.389,594.576 C 1686.878,594.253 1687.069,594.445 1686.958,595.152 C 1687.192,594.93 1687.433,594.714 1687.678,594.504 C 1688.145,595.186 1687.731,596.172 1688.258,596.808 C 1688.768,597.423 1689.948,597.281 1690.639,597.241 C 1691.537,597.188 1692.294,597.046 1692.985,596.425 C 1693.299,596.143 1693.662,595.318 1694.089,595.225 C 1694.552,595.122 1695.706,595.819 1696.275,595.831 C 1697.208,595.852 1698.137,595.96 1699.069,595.927 C 1701.602,595.837 1703.349,597.003 1705.83,595.513 C 1705.605,595.166 1705.365,594.83 1705.11,594.505 C 1705.891,594.21 1705.783,595.15 1706.4,595.081 C 1707.298,594.981 1708.06,593.898 1708.426,593.18 C 1709.422,591.222 1709.549,590.107 1711.292,588.652 C 1712.989,587.235 1713.681,584.68 1715.285,583.357 C 1717.061,581.893 1718.127,581.502 1718.639,579.097 C 1719.933,579.163 1719.375,581.035 1719.385,581.825 C 1719.403,583.319 1721.026,582.562 1721.22,584.222 C 1721.414,585.887 1722.5508,588.06395 1721.241,589.35238 C 1718.4456,592.10212 1719.3453,590.17771 1718.9098,588.99806 C 1718.4848,587.84664 1717.605,589.9561 1717.738,590.666 C 1717.8388,591.20367 1717.5921,591.20968 1717.675,591.787 C 1717.7638,592.405 1718.1646,593.32199 1718.1723,594.04706 C 1718.1937,596.0626 1720.2136,595.15403 1719.4013,596.49938 C 1718.7548,597.57 1716.643,596.629 1715.826,597.602 C 1717.7078,598.64969 1714.9144,603.56628 1714.553,606.589 C 1714.681,608.567 1713.672,609.019 1711.949,608.615 C 1709.829,608.118 1707.637,607.918 1705.485,607.599 C 1703.228,607.265 1700.972,606.934 1698.708,606.652 C 1697.045,606.444 1695.051,606.774 1693.936,605.294 C 1692.804,603.791 1691.725,602.267 1690.535,600.803 C 1689.704,599.78 1688.853,598.775 1688.015,597.758 C 1687.172,596.729 1687.038,595.666 1686.389,594.576" id="path2784"/>
  </g>
  <g class="landxx coastxx bz" id="bz" transform="translate(-29.9017, -45.0745)">
    <path d="M 638.569,654.192 C 639.138,654.02 639.515,653.582 639.289,652.967 C 639.042,653.372 638.802,653.78 638.569,654.192" id="path2808"/>
    <path class="landxx bz" d="M 626.977,667.152 C 627.552,663.422 628.535,659.9 628.893,656.13 C 629.038,654.609 629.344,646.946 632.089,649.944 C 632.849,648.516 634.8,644.247 636.913,644.688 C 636.472,645.095 636.255,645.599 636.265,646.2 C 639.45,644.586 636.668,650.803 636.36,651.855 C 635.986,653.132 635.759,654.73 635.537,656.075 C 635.377,657.043 635.749,657.763 635.418,658.752 C 635.091,659.73 634.587,660.636 634.249,661.607 C 634.804,659.615 633.751,662.156 633.523,662.441 C 632.883,663.241 632.056,663.538 631.287,664.165 C 630.635,664.696 630.046,665.376 629.654,666.123 C 629.468,666.441 629.368,666.784 629.353,667.151 C 628.567,666.957 627.766,667.071 626.977,667.152" id="path2810"/>
  </g>
  <g class="landxx coastxx br" id="br" transform="translate(-29.9017, -45.0745)">
    <path d="M 1005.837,922.033 C 1006.92,919.206 1003.98,921.508 1005.837,922.033" id="path2886"/>
    <path d="M 899.788,817.853 C 900.508,817.71 901.141,817.276 901.658,816.773 C 900.864,816.658 899.999,817.015 899.788,817.853" id="path2814"/>
    <path d="M 901.518,818.853 C 902.102,818.2 903.035,818.35 903.668,817.773 C 903,816.678 901.086,817.703 901.518,818.853" id="path2816"/>
    <path d="M 914.618,786.312 C 914.785,786.072 914.951,785.832 915.118,785.592 C 914.317,785.362 914.15,785.603 914.618,786.312" id="path2846"/>
    <path d="M 915.268,788.399 C 916.376,787.852 915.803,786.89 915.268,786.168 C 914.184,786.67 914.764,787.72 915.268,788.399" id="path2848"/>
    <path d="M 903.748,817.413 C 905.719,816.93 910.484,812.225 909.345,809.981 C 908.182,807.691 905.955,810.774 905.661,811.995 C 905.446,812.891 905.835,813.849 904.935,814.442 C 903.96,815.083 903.235,816.166 903.748,817.413" id="path2818"/>
    <path d="M 907.638,808.853 C 908.384,808.354 910.404,806.991 909.438,805.893 C 908.398,806.438 907.985,807.826 907.638,808.853" id="path2820"/>
    <path d="M 907.708,815.183 C 908.779,814.808 909.624,813.267 908.928,812.234 C 908.915,813.394 907.695,814.036 907.708,815.183" id="path2822"/>
    <path d="M 910.797,812.163 C 909.974,812.438 909.367,813.185 909.217,814.033 C 910.216,813.791 910.447,812.683 911.238,812.163 C 911.091,812.163 910.944,812.163 910.797,812.163" id="path2824"/>
    <path d="M 909.717,809.353 C 910.04,808.645 910.186,807.846 911.017,807.553 C 909.898,806.442 909.438,808.575 909.717,809.353" id="path2826"/>
    <path d="M 909.578,812.953 C 910.229,812.227 910.65,811.292 911.435,810.685 C 911.874,810.345 912.124,809.429 911.246,809.741 C 910.291,810.081 909.496,812.014 909.578,812.953" id="path2828"/>
    <path d="M 910.078,809.423 C 910.658,809.249 911.12,808.779 910.868,808.133 C 910.354,808.37 910.003,808.835 910.078,809.423" id="path2830"/>
    <path d="M 911.667,807.123 C 912.197,806.549 912.894,806.175 913.421,805.598 C 914.081,804.873 912.964,804.608 912.463,804.837 C 911.438,805.303 909.451,806.684 911.667,807.123" id="path2832"/>
    <path d="M 910.728,809.643 C 911.339,809.319 912.636,808.86 912.168,807.914 C 911.235,807.983 911.115,809.005 910.728,809.643" id="path2834"/>
    <path d="M 911.087,811.942 C 911.46,811.725 911.723,811.416 911.877,811.012 C 911.272,811.01 910.945,811.327 911.087,811.942" id="path2836"/>
    <path d="M 911.308,804.603 C 911.331,804.747 911.354,804.89 911.378,805.034 C 911.943,804.963 912.423,804.723 912.818,804.314 C 912.281,803.918 911.687,804.101 911.308,804.603" id="path2838"/>
    <path d="M 912.308,817.343 C 912.843,818.205 913.205,819.206 913.929,819.941 C 914.474,820.495 916.713,820.66 915.558,820.013 C 916.583,820.13 917.861,819.523 918.811,819.799 C 919.483,819.994 920.469,820.535 920.238,819.293 C 920.39,819.565 920.557,819.829 920.738,820.083 C 921.446,819.603 922.172,818.92 922.178,817.993 C 922.581,819.488 925.148,818.419 925.968,817.992 C 927.464,817.214 927.42,815.642 928.288,814.356 C 929.124,813.117 929.388,811.877 929.672,810.431 C 929.939,809.074 931.521,806.769 929.149,806.623 C 927.294,806.509 924.951,805.352 923.189,806.113 C 922.412,806.449 923.272,806.78 922.037,806.726 C 920.952,806.679 920.065,806.532 919.008,806.253 C 917.077,805.744 914.557,804.882 913.28,807.041 C 912.577,808.229 912.517,810.117 914.118,810.574 C 913.537,810.449 912.96,810.305 912.388,810.143 C 912.15,810.806 912.634,811.109 912.598,811.734 C 912.557,812.454 911.977,813.001 912.361,813.77 C 912.78,814.61 913.982,813.506 914.618,813.674 C 913.634,815.333 913.845,814.143 912.098,814.394 C 912.483,815.288 912.351,816.399 912.308,817.343" id="path2840"/>
    <path d="M 913.898,804.603 C 914.859,805.315 914.844,804.283 914.53,803.651 C 913.805,802.194 913.033,803.745 913.898,804.603" id="path2842"/>
    <path d="M 914.837,803.163 C 915.562,802.088 916.11,800.823 915.117,799.703 C 914.669,800.578 913.814,802.876 914.837,803.163" id="path2844"/>
    <path d="M 917.638,804.603 C 918.336,806.015 919.708,803.778 920.294,803.175 C 921.824,801.599 919.091,801.872 918.246,802.136 C 917.412,802.396 915.534,802.478 915.248,803.522 C 915.007,804.403 917.155,804.542 917.638,804.603" id="path2850"/>
    <path d="M 916.268,801.653 C 916.672,801.327 918.618,799.842 917.883,799.218 C 916.709,798.221 915.382,800.746 916.268,801.653" id="path2852"/>
    <path d="M 916.348,802.653 C 916.802,802.461 917.185,802.174 917.498,801.793 C 916.891,801.701 916.484,802.095 916.348,802.653" id="path2854"/>
    <path d="M 916.348,798.192 C 917.066,798.17 917.952,797.966 917.858,797.043 C 917.095,796.821 916.088,797.24 916.348,798.192" id="path2856"/>
    <path d="M 918.068,796.603 C 918.524,796.334 918.597,795.978 918.288,795.534 C 917.853,795.809 917.78,796.165 918.068,796.603" id="path2858"/>
    <path d="M 920.018,804.603 C 918.987,806.184 921.529,805.88 922.332,805.611 C 923.749,805.136 921.246,803.383 920.018,804.603" id="path2860"/>
    <path d="M 926.208,820.223 C 926.447,819.598 926.59,818.951 926.638,818.284 C 925.74,818.599 925.263,819.6 926.208,820.223" id="path2862"/>
    <path d="M 930.028,814.683 C 931.976,816.591 930.451,812.362 930.028,814.683" id="path2864"/>
    <path d="M 935.788,810.573 C 936.548,810.92 935.817,809.821 935.788,809.783 C 935.788,810.046 935.788,810.31 935.788,810.573" id="path2866"/>
    <path d="M 944.858,813.453 C 945.253,813.345 945.446,813.082 945.438,812.663 C 944.668,812.497 944.475,812.76 944.858,813.453" id="path2872"/>
    <path d="M 956.518,816.772 C 957.074,816.57 957.382,816.119 957.098,815.543 C 956.82,816.361 956.627,816.528 956.518,816.043 C 956.518,816.286 956.518,816.529 956.518,816.772" id="path2874"/>
    <path d="M 958.248,819.073 C 958.511,818.976 958.775,818.88 959.038,818.783 C 958.35,818.328 958.086,818.425 958.248,819.073" id="path2876"/>
    <path d="M 959.837,831.173 C 960.306,830.228 960.661,829.286 960.627,828.213 C 959.484,828.699 959.883,830.222 959.837,831.173" id="path2878"/>
    <path d="M 966.528,824.973 C 966.745,824.733 966.961,824.493 967.178,824.253 C 966.411,824.183 966.194,824.422 966.528,824.973" id="path2882"/>
    <path class="landxx br" d="M 736.705,869.763 C 737.21467,868.78423 736.99677,868.25129 738.126,868.048 C 739.25237,867.84522 737.9418,866.49605 738.26,864.987 C 738.77353,862.55166 740.6267,862.34388 742.23141,861.3016 C 743.37409,860.5594 742.538,858.928 742.259,857.735 C 742.004,856.641 742.572,855.655 743.184,854.793 C 744.413,853.061 743.108,850.545 744.362,849.023 C 745.008,848.238 746.384,848.187 747.145,847.442 C 747.82,846.781 747.8446,846.0228 748.7806,845.6988 C 749.9856,845.2808 750.666,844.099 751.826,843.623 C 752.9,843.183 754.561,843.213 755.724,842.971 C 757.106,842.683 758.25618,842.74917 759.275,841.668 C 759.43866,841.49432 759.69294,840.78975 761.23,840.429 C 761.80293,840.29453 761.93177,840.55269 762.711,840.723 C 763.37776,840.86873 764.115,841.662 764.498,842.182 C 764.692,841.942 764.885,841.703 765.078,841.462 C 766.121,843.181 766.64,839.752 766.73,839.144 C 767.143,836.37 767.55735,833.59454 767.834,830.803 C 768.20958,827.01315 768.6217,823.21167 769.388,819.502 C 769.71306,817.92839 770.36921,816.4994 769.99121,814.8634 C 769.62821,813.2904 768.51874,812.48178 768.46841,810.79879 C 768.41641,809.05979 767.575,808.7659 766.285,807.558 C 764.84431,806.209 764.82061,804.792 764.82061,803.079 C 764.82061,802.083 764.664,800.831 765.199,799.914 C 766.55819,798.43964 769.24916,797.6842 770.325,798.787 C 771.502,799.412 771.91371,799.05795 771.82,797.699 C 771.674,795.58183 768.418,794.44146 766.586,795.456 C 766.57597,793.92474 765.76256,791.2729 766.47902,789.94502 C 767.176,788.65323 769.709,789.697 770.627,789.697 C 772.822,789.697 774.87901,789.6586 777.07001,789.7486 C 777.45801,789.7646 778.69781,789.8926 778.81481,789.2866 C 778.87581,788.9726 778.69112,788.06369 779.10804,787.79218 C 780.64601,786.7906 780.27227,789.93171 781.53041,789.3136 C 782.60406,788.78613 783.19891,787.44696 784.118,786.711 C 785.80362,785.36124 786.64385,787.38872 787.0536,788.3518 C 787.86748,790.2648 787.86907,791.2 787.67779,793.065 C 787.63277,793.504 787.361,794.491 788.16,794.344 C 789.278,794.138 789.05402,793.50624 789.93402,794.39324 C 791.00902,795.47624 791.71064,796.86403 793.12864,797.51903 C 795.03364,798.39803 796.30359,797.00686 797.773,796.283 C 800.68803,794.847 799.12659,798.273 799.555,799.055 C 800.265,798.824 801.43221,796.979 801.86321,796.359 C 802.59721,795.305 802.91821,794.6676 804.20021,794.1716 C 805.27221,793.7566 805.85162,793.6074 806.66762,792.8074 C 807.24762,792.2384 808.764,791.161 808.704,792.722 C 809.56,791.3 810.516,790.78 810.934,789.122 C 811.264,787.818 811.75683,787.8655 812.91621,787.3746 C 813.41413,787.16378 815.2958,786.19447 815.52142,785.8552 C 816.90037,783.78164 815.86252,783.40343 814.24275,783.59859 C 813.44804,783.69434 812.11541,783.66889 811.47799,783.13658 C 811.10362,782.82395 811.54165,781.39593 811.39381,780.996 C 810.65118,778.987 809.87503,778.20919 809.924,774.865 C 809.94452,773.46365 808.8966,772.49699 807.81,771.38 C 807.18875,770.74137 805.165,768.474 806.114,767.521 C 806.764,767.806 807.1016,768.7224 807.8056,768.7914 C 808.7686,768.8854 810.84112,768.17709 811.08979,769.6268 C 811.26348,770.63939 811.65229,770.89668 812.768,770.74698 C 814.60501,770.50051 815.6643,771.40758 815.974,770.851 C 817.418,768.25587 818.53539,774.02415 820.54301,773.6576 C 821.85539,773.41799 820.64383,770.96176 821.305,769.933 C 821.70739,769.30689 822.785,769.625 823.371,768.924 C 824.224,767.903 825.03715,769.29335 825.91021,768.9914 C 827.28327,768.51653 830.37749,767.69299 830.9984,766.53181 C 831.04227,766.44976 832.66593,765.51166 832.856,765.527 C 833.65685,765.59162 834.62922,766.02936 835.1724,765.04659 C 836.05316,763.45301 839.30155,762.41271 836.994,759.42461 C 838.087,759.25961 839.01934,759.43272 840.122,759.11502 C 841.018,758.85686 841.68115,759.14445 842.27581,760.0524 C 843.53883,761.98087 841.92346,763.36737 841.61701,765.08 C 841.45958,765.9598 843.54252,765.50252 844.663,766.622 C 844.91211,766.87089 844.35,767.595 844.563,768.093 C 844.909,768.901 845.2928,769.52921 845.7408,770.26321 C 846.2368,771.07621 843.96018,773.2322 843.78018,773.9042 C 843.60018,774.5732 843.587,775.34 843.411,776.063 C 843.088,777.389 842.3514,778.41621 842.1914,779.77521 C 841.9554,781.79121 842.24314,783.63384 843.569,784.936 C 844.5178,785.86784 843.458,788.7879 844.702,788.692 C 844.607,788.932 844.51,789.172 844.412,789.412 C 845.91087,789.58107 847.04245,791.26127 847.641,792.384 C 848.13909,793.31831 849.14145,792.45471 849.73302,793.32662 C 850.26249,794.10699 851.16547,793.91435 851.629,794.009 C 852.15659,794.11673 852.6889,792.6787 853.28061,792.18783 C 855.38452,790.44245 856.82399,790.60996 858.50101,789.8856 C 859.46536,789.46907 859.94921,790.06041 860.76321,789.07841 C 861.84021,787.77941 862.96014,787.07785 864.839,787.65981 C 866.367,788.13309 867.16704,787.20577 868.76304,787.65602 C 869.80408,787.94971 872.6792,789.40796 873.0694,787.63301 C 873.4284,786.00001 871.41224,785.8726 872.16318,784.3626 C 873.5418,781.59044 873.81952,783.98161 875.468,783.564 C 876.51125,783.29971 877.24548,783.05437 878.19582,782.70321 C 880.21085,781.95866 880.11999,782.3592 881.61899,783.6032 C 881.91599,783.8492 881.697,783.011 882.056,783.567 C 882.502,784.258 883.2044,784.4954 883.8554,784.9884 C 884.6974,785.6264 885.483,785.709 886.358,786.186 C 886.73146,786.54159 887.28499,785.09938 889.228,784.403 C 889.718,784.131 890.334,784.782 890.855,784.829 C 891.497,784.887 893.9584,784.19181 893.4944,785.51481 C 894.2024,785.18081 895.3274,786.01481 895.9114,785.49081 C 896.7364,784.74981 897.47202,784.4384 898.09202,783.4844 C 898.96702,782.1384 900.61975,777.81229 901.50612,776.47456 C 904.30187,772.2552 906.9131,768.80364 905.64408,766.03177 C 905.73508,766.60877 907.239,769.684 907.634,770.115 C 907.305,769.179 906.519,768.021 907.344,767.091 C 909.53,768.445 909.813,770.623 909.574,772.995 C 909.698,772.297 909.765,771.536 910.364,771.051 C 910.281,774.186 910.606,777.458 911.713,780.407 C 912.165,781.613 914.021,785.271 912.664,786.386 C 913.712,786.04 913.887,787.172 914.217,787.859 C 914.86,789.197 916.379,788.66 917.494,789.122 C 920.038,790.177 919.548,793.177 917.344,794.162 C 917.917,794 918.56,794.102 919.004,794.521 C 918.035,795.49 916.353,796.36 915.962,797.769 C 915.76,798.494 914.935,798.822 914.545,799.415 C 913.954,800.313 913.693,801.312 912.973,802.141 C 911.997,803.263 910.743,804.079 909.65,805.073 C 908.3,806.302 907.596,807.953 906.46,809.355 C 905.407,810.654 905.02,811.656 905.168,813.328 C 905.25,814.265 903.522,814.431 903.461,815.718 C 903.379,817.453 902.111,816.518 901.223,816.489 C 900.41,816.462 897.854,817.715 897.555,818.496 C 898.274,818.329 902.157,817.591 901.075,819.366 C 901.81,818.557 902.956,818.631 903.846,818.107 C 904.885,817.496 905.683,816.832 906.806,816.35 C 908.268,815.722 910.428,813.901 911.236,812.525 C 913.128,813.623 911.322,816.464 912.554,817.983 C 913.049,818.592 912.572,820.223 913.388,820.464 C 914.069,820.664 915.248,820.268 914.835,821.456 C 916.433,821.769 919.293,818.856 919.585,822.176 C 919.442,821.153 920.452,820.963 921.217,820.816 C 921.599,820.828 921.871,820.657 922.033,820.302 C 922.33,819.511 923.178,819.805 923.834,819.656 C 922.982,822.33 922.188,824.077 922.314,826.855 C 923.534,825.28 922.813,821.853 924.861,820.887 C 926.224,820.245 926.449,820.054 927.256,818.738 C 928.588,816.563 928.952,818.173 930.304,819.006 C 930.276,818.549 930.159,818.115 929.954,817.706 C 930.707,817.654 931.444,817.483 932.184,817.346 C 931.424,816.964 930.571,817.28 929.804,817.416 C 929.61,816.365 930.354,816.321 930.943,815.757 C 931.836,814.901 930.872,813 931.972,811.914 C 933.268,810.634 932.902,811.713 933.764,811.376 C 934.37,811.14 935.639,809.612 935.784,811.156 C 936.978,809.991 937.045,811.618 938.154,811.225 C 937.677,810.815 937.566,810.198 937.864,809.645 C 938.545,809.844 940.17,810.241 939.814,811.295 C 940.05,811.167 940.29,811.047 940.534,810.935 C 940.631,811.202 940.727,811.469 940.824,811.736 C 941.446,811.242 941.544,811.238 941.544,812.086 C 942.12,811.896 942.524,811.41 942.554,810.796 C 942.192,811.728 943.465,811.902 943.124,812.817 C 943.213,812.231 943.59,811.853 944.204,812.087 C 944.195,812.706 943.788,813.174 943.194,813.317 C 944.222,812.72 945.158,814.181 946.144,813.247 C 946.012,814.601 947.012,813.8 947.304,813.097 C 946.891,814.737 948.267,814.058 948.164,815.118 C 948.55,814.964 948.84,814.701 949.034,814.328 C 948.901,815.256 950.058,814.917 950.254,815.767 C 950.5,815.533 950.74,815.293 950.974,815.047 C 951.021,815.586 951.215,816.066 951.554,816.487 C 951.792,816.295 952.029,816.102 952.264,815.907 C 952.387,816.627 952.671,816.97 953.344,816.266 C 952.819,816.641 952.681,817.238 953.064,817.776 C 953.213,817.222 953.64,816.796 954.213,816.696 C 953.4,817.428 953.689,818.989 954.434,819.657 C 954.759,819.048 954.368,818.022 955.19,817.704 C 956.684,817.124 956.015,818.197 956.734,818.427 C 956.692,817.826 957.05,817.355 957.594,817.137 C 957.344,817.609 957.224,818.113 957.234,818.647 C 958.943,817.622 957.01,820.681 958.824,819.578 C 958.574,819.807 958.334,820.048 958.104,820.298 C 958.687,820.5 959.22,820.214 959.543,819.728 C 959.052,820.792 960.782,820.21 960.033,821.637 C 959.364,822.912 959.679,823.974 958.033,824.478 C 958.625,824.106 959.088,824.438 959.253,825.048 C 959.295,824.133 960.302,822.727 961.146,823.914 C 962.048,825.185 960.468,825.582 959.971,826.537 C 959.547,827.351 958.793,829.861 959.084,830.744 C 959.406,831.721 957.856,832.532 958.534,833.259 C 958.624,831.886 960.253,831.47 960.834,830.379 C 961.411,829.295 961.032,827.479 961.66,826.587 C 961.981,826.129 963.43,825.088 963.899,825.878 C 964.669,827.174 962.231,827.865 962.065,828.939 C 962.775,828.482 963.015,828.652 962.785,829.449 C 963.624,829.043 964.803,825.757 965.235,827.429 C 965.306,826.275 967.531,825.649 968.395,826.139 C 968.303,823.867 973.012,826.383 973.847,826.864 C 974.439,827.205 977.7,828.532 977.755,829.159 C 978.465,828.696 980.348,829.867 979.995,828.369 C 980.938,828.578 981.576,829.283 982.435,829.659 C 983.425,830.091 984.487,830.025 985.315,830.819 C 985.299,829.434 986.154,829.581 987.23,829.542 C 989.387,829.465 991.293,828.6 993.49,829.069 C 997.512,829.927 1001.233,833.361 1004.368,835.821 C 1005.412,836.64 1006.609,836.778 1007.475,837.809 C 1008.304,838.797 1008.945,839.901 1009.73,840.915 C 1010.615,842.059 1011.8,842.569 1012.612,843.693 C 1013.479,844.893 1015.073,845.025 1016.027,846.266 C 1016.985,847.512 1018.381,847.008 1019.474,848.081 C 1020.739,849.324 1022.961,848.822 1024.513,848.628 C 1028.081,848.183 1030.717,849.517 1031.754,853.057 C 1032.156,854.428 1032.758,855.793 1032.782,857.245 C 1032.789,857.683 1032.417,858.416 1032.755,858.584 C 1033.142,858.775 1033.357,859.306 1033.511,859.674 C 1034.159,861.223 1035.258,864.673 1034.424,866.307 C 1034.654,865.956 1034.867,865.597 1035.064,865.227 C 1034.908,866.865 1036.105,868.838 1034.424,870.047 C 1034.69,870.111 1034.953,870.184 1035.214,870.267 C 1034.541,871.622 1035.607,873.254 1034.784,874.482 C 1033.932,875.754 1034.568,877.468 1033.901,878.804 C 1033.125,880.355 1032.856,882.097 1032.137,883.672 C 1031.347,885.405 1029.625,886.536 1028.954,888.337 C 1028.704,888.159 1028.464,887.969 1028.234,887.767 C 1028.287,888.497 1028.059,889.406 1027.364,888.197 C 1027.485,888.657 1027.775,888.898 1028.234,888.918 C 1027.184,890.989 1025.646,893.691 1023.907,895.258 C 1023.19,895.904 1022.222,896.162 1021.492,896.804 C 1020.731,897.473 1020.138,898.349 1019.234,898.848 C 1019.385,898.233 1018.991,897.8 1018.434,897.628 C 1018.365,898.297 1019.205,898.513 1019.084,899.138 C 1018.927,899.943 1018.232,901.017 1017.434,900.008 C 1017.829,900.524 1018.759,901.022 1018.197,901.8 C 1017.872,902.251 1016.515,903.745 1017.064,902.088 C 1016.613,902.805 1016.707,903.708 1016.064,904.328 C 1016.353,904.238 1016.64,904.141 1016.924,904.038 C 1015.34,906.299 1012.822,917.691 1008.714,916.998 C 1009.47,915.276 1007.723,914.955 1007.204,913.607 C 1007.17,914.388 1006.307,916.583 1005.624,914.908 C 1005.836,915.609 1006.92,915.897 1006.888,916.652 C 1006.83,917.979 1005.153,918.926 1005.624,920.378 C 1004.47,919.598 1004.707,921.586 1004.685,922.398 C 1004.768,922.026 1004.887,921.666 1005.044,921.319 C 1005.431,922.329 1005.44,923.204 1005.404,924.268 C 1005.124,924.046 1004.861,923.806 1004.614,923.548 C 1005.239,924.077 1005.798,925.787 1005.554,926.578 C 1005.75,926.074 1005.94,925.567 1006.124,925.057 C 1006.223,928.086 1005.258,930.831 1005.695,933.905 C 1006.07,936.542 1007.217,939.303 1007.348,941.932 C 1007.58,946.6 1004.65,950.948 1005.948,955.736 C 1006.673,958.406 1004.984,958.734 1003.84,960.741 C 1002.521,963.054 1002.319,965.928 1002.649,968.523 C 1002.884,970.375 1003.981,974.106 1001.789,975.069 C 999.722,975.978 1001.382,980.024 998.925,980.356 C 1000.67,981.9 997.715,983.834 996.718,984.784 C 995.864,985.598 996.111,986.212 995.809,987.304 C 995.364,988.915 994.631,990.367 994.89,992.084 C 995.025,992.977 995.672,994.372 995.002,995.187 C 993.897,996.532 992.207,996.49 990.818,997.319 C 989.876,997.882 988.905,998.931 988.482,999.947 C 988.208,1000.604 988.527,1001.213 988.718,1001.84 C 988.915,1002.483 988.088,1002.818 988.415,1003.537 C 987.432,1002.449 984.612,1002.994 983.375,1003.252 C 982.651,1003.403 981.488,1003.802 980.785,1003.397 C 979.742,1002.796 980.483,1001.783 980.425,1000.877 C 978.109,1001.148 980.01,1002.114 979.701,1003.279 C 979.395,1004.432 974.767,1003.985 973.724,1004.547 C 974.431,1003.793 975.652,1004.131 976.534,1003.827 C 975.586,1002.98 974.526,1002.519 973.351,1003.235 C 972.308,1003.87 971.84,1003.365 970.762,1003.487 C 969.98,1003.575 968.68,1003.665 968.407,1004.593 C 968.152,1005.458 969.486,1005.839 969.985,1006.207 C 968.848,1007.135 967.334,1006.159 966.59,1007.256 C 965.95,1008.2 964.923,1008.312 964.038,1008.927 C 963.256,1009.469 964.219,1010.304 963.435,1010.818 C 962.351,1011.529 960.504,1009.516 958.964,1010.818 C 958.802,1010.955 957.735,1012.915 957.352,1012.431 C 957.018,1012.011 957.281,1011.394 956.444,1011.537 C 956.885,1012.846 955.191,1012.996 954.251,1013.623 C 953.097,1014.393 952.564,1015.488 951.653,1016.477 C 950.66,1017.553 949.132,1017.853 948.056,1018.823 C 947.138,1019.653 946.614,1020.91 945.573,1021.607 C 945.812,1021.755 946.052,1021.902 946.293,1022.047 C 945.485,1022.883 944.679,1023.857 944.203,1024.927 C 944.129,1024.382 944.272,1023.903 944.633,1023.488 C 943.881,1023.81 943.102,1023.603 942.403,1023.267 C 942.728,1024.761 941.897,1025.625 940.463,1024.417 C 940.448,1025.278 942.483,1026.164 943.263,1026.077 C 942.725,1026.577 942.538,1027.253 942.403,1027.947 C 942.296,1028.501 941.176,1028.325 940.823,1028.597 C 941.305,1028.582 941.785,1028.605 942.263,1028.667 C 942.046,1029.348 942.327,1030.046 942.129,1030.724 C 941.896,1031.521 941.048,1031.658 940.823,1030.757 C 941.101,1031.346 940.973,1032.147 941.626,1032.427 C 942.48,1032.793 941.981,1034.604 942.183,1035.297 C 942.505,1036.401 943.874,1038.018 943.263,1039.397 C 944.319,1039.146 944.211,1040.392 943.343,1040.547 C 944.294,1041.273 943.551,1044.348 944.312,1045.62 C 945.025,1046.812 943.76,1049.774 943.703,1050.987 C 943.325,1050.71 943.012,1050.373 942.763,1049.976 C 943.022,1050.593 942.999,1051.326 943.773,1051.567 C 942.969,1052.701 941.691,1053.378 940.661,1054.274 C 939.564,1055.229 938.806,1056.507 938.127,1057.775 C 936.476,1060.86 935.994,1064.148 935.117,1067.488 C 934.294,1070.625 932.737,1073.473 930.929,1076.141 C 929.188,1078.71 926.656,1080.545 924.622,1082.886 C 924.984,1081.958 924.467,1080.826 924.262,1079.926 C 925.322,1080.217 925.74,1080.309 926.679,1079.656 C 927.514,1079.076 927.98,1077.742 928.735,1077.318 C 929.778,1076.732 929.954,1076.246 929.95,1075.041 C 929.947,1074.264 930.085,1072.902 931.033,1074.025 C 930.678,1072.882 931.214,1072.448 931.886,1071.651 C 932.437,1070.997 932.553,1070.246 932.432,1069.417 C 932.345,1068.815 931.706,1066.801 933.042,1068.046 C 933.075,1067.733 932.646,1065.802 932.149,1065.973 C 931.833,1066.081 930.104,1067.384 930.092,1067.766 C 929.617,1067.228 929.345,1066.587 928.64,1066.336 C 927.808,1066.04 927.968,1064.856 927.211,1064.456 C 927.145,1066.426 928.079,1066.348 928.861,1067.766 C 927.208,1068.17 928.41,1070.08 928.291,1071.146 C 928.176,1070.62 927.96,1070.141 927.642,1069.707 C 927.405,1070.388 928.049,1071.262 927.708,1071.832 C 927.557,1072.085 927.026,1072.038 927.1,1072.431 C 927.411,1074.07 926.016,1074.945 924.854,1075.608 C 923.937,1076.131 924.304,1077.275 924.041,1078.136 C 923.82,1078.858 922,1080.623 923.751,1080.796 C 922.821,1081.375 923.482,1082.651 924.401,1081.945 C 924.549,1082.862 923.873,1083.714 923.503,1084.487 C 922.898,1085.754 922.944,1087.282 922.751,1088.645 C 922.371,1091.33 920.732,1092.931 919.341,1095.107 C 918.7755,1095.9915 918.03739,1096.4853 917.57881,1096.3039 C 917.09284,1096.1117 916.44847,1094.9464 916.20964,1094.3955 C 915.76569,1093.3715 915.95144,1092.5106 915.88373,1091.6302 C 915.79534,1090.4808 916.92275,1089.6274 917.42591,1088.6805 C 918.21576,1087.194 916.507,1086.527 915.497,1086.059 C 914.321,1085.514 913.84006,1084.4884 913.129,1083.465 C 911.47049,1081.078 909.15361,1081.188 907.596,1079.512 C 906.30794,1078.1619 904.8013,1077.4147 903.231,1076.545 C 902.656,1076.27 902.437,1075.6949 901.861,1075.4319 C 900.22289,1076.4744 898.24154,1072.9068 896.541,1071.575 C 897.41472,1072.8161 893.90135,1074.9251 893.97873,1072.29 C 894.03997,1070.2046 891.458,1068.869 889.986,1067.759 C 888.893,1066.935 886.73536,1064.0674 885.08736,1065.2614 C 884.53836,1065.6594 884.51004,1066.3955 883.639,1066.624 C 882.8282,1066.8367 881.70786,1066.7343 880.97191,1065.814 C 882.51252,1065.1975 882.09995,1062.7652 883.86609,1062.05 C 885.44397,1061.411 886.0645,1059.2098 886.681,1057.576 C 887.37969,1055.7244 888.69719,1054.1563 889.61819,1052.4283 C 890.08519,1051.5533 891.21,1049.223 892.431,1050.483 C 892.723,1049.897 892.462,1049.385 891.861,1049.182 C 892.832,1048.007 893.82382,1046.4925 895.16182,1045.7425 C 896.53082,1044.9745 896.78309,1044.3596 897.72209,1043.3416 C 898.66909,1042.3136 899.87573,1042.1537 901.04773,1041.5557 C 901.90373,1041.1177 903.517,1039.241 904.602,1039.752 C 904.85,1037.975 904.788,1036.547 904.74,1034.773 C 904.682,1032.621 904.475,1031.49 903.745,1029.484 C 903.228,1028.066 902.903,1026.43 901.324,1025.87 C 900.412,1025.546 898.684,1025.417 898.482,1026.722 C 898.19725,1026.452 897.49843,1026.6469 896.98172,1026.3046 C 896.61139,1026.0593 896.81563,1025.083 896.77909,1024.446 C 896.68167,1022.748 897.2245,1021.5332 897.43275,1019.7862 C 897.57544,1018.5892 897.29402,1016.8496 897.83102,1015.7536 C 898.39102,1014.6096 896.832,1013.686 898.122,1012.753 C 897.239,1012.149 896.5794,1011.076 895.4324,1010.94 C 894.2264,1010.796 893.33802,1011.8074 892.292,1012.173 C 889.25585,1013.2341 888.60236,1010.2731 888.042,1008.003 C 887.70862,1006.6524 886.74431,1005.2575 886.49631,1003.8915 C 885.62271,1000.7207 884.69774,999.55112 883.6195,997.99529 C 882.4095,997.67729 881.22583,997.4376 880.21552,996.55421 C 879.12862,995.60386 879.13801,996.31427 878.16205,997.89559 C 876.17439,1001.1161 876.4713,997.85069 875.5193,997.72869 C 874.14925,997.55312 872.50461,997.58178 871.27561,996.91278 C 870.62161,996.55678 868.01904,996.29007 867.81944,995.45945 C 867.61537,994.61018 868.25508,993.81402 868.461,993.022 C 868.66239,992.24737 867.872,991.093 868.174,990.316 C 868.705,988.947 868.139,987.702 868.42,986.381 C 868.635,985.367 866.8218,984.9802 867.7858,983.8382 C 866.61123,983.36124 866.16424,981.3575 865.5616,980.0798 C 864.2466,978.2398 868.67421,978.1716 866.80421,976.7276 C 864.54221,974.9816 865.409,973.683 866.478,971.236 C 866.767,970.575 867.71321,969.20501 867.36321,968.57401 C 866.77021,967.50501 867.293,966.714 867.567,965.52 C 868.73541,962.31509 867.93836,961.86749 866.871,959.541 C 865.89421,958.08254 865.97672,955.71816 865.058,956.089 C 863.47226,956.72908 861.36,954.67216 860.798,952.945 C 860.42529,951.79958 859.959,950.427 859.924,949.216 C 859.889,948.006 860.792,946.99 860.751,945.72 C 858.318,946.276 855.7558,946.0886 853.3028,945.8376 C 851.7458,945.6786 848.2032,946.1506 847.0822,944.9006 C 846.0602,943.7596 846.275,941.406 846.033,939.987 C 845.711,938.101 843.476,936.994 843.401,935.28 C 844.023,935.215 844.646,935.192 845.271,935.21 C 844.8499,933.13493 844.17949,930.28477 843.47062,928.72562 C 843.07719,927.86027 843.03027,927.67444 843.0858,926.8518 C 843.1572,925.79416 843.20496,924.9786 842.32702,924.28481 C 840.15036,922.5647 837.82291,921.42687 836.6474,921.5601 C 832.69779,922.00775 833.7329,921.67512 832.51902,921.66481 C 831.53364,921.65644 830.239,919.482 829.772,918.855 C 829.261,918.171 828.897,918.494 828.22,918.44 C 827.61,918.391 826.27529,917.4006 825.7304,917.1322 C 825.3484,916.94404 824.79596,916.99596 824.296,916.76 C 823.1284,916.20895 823.10076,915.31177 822.61101,914.6518 C 821.97467,913.79431 821.837,913.979 820.971,914.365 C 819.952,914.82 819.41501,913.6536 818.72301,913.1996 C 816.92601,912.0166 816.398,913.394 814.574,912.965 C 812.484,912.475 812.28,912.313 811.143,910.716 C 810.607,909.963 808.6826,908.7362 807.7996,908.6442 C 806.9336,908.5542 807.248,907.297 806.73,906.722 C 806.353,906.305 805.49544,906.37901 805.501,905.727 C 805.5131,904.30752 803.85865,903.03394 803.76702,901.51622 C 803.71504,900.6554 804.35793,900.13336 804.462,899.257 C 804.54699,898.54135 803.52131,898.40637 803.374,897.594 C 803.24717,896.89456 803.1711,896.04713 803.154,895.604 C 803.11971,894.71521 803.57262,893.99481 803.71662,893.21781 C 803.99262,891.73081 804.094,889.36 802.719,888.409 C 802.385,888.723 802.101,889.528 801.548,889.484 C 800.164,889.373 799.08852,889.31303 797.704,889.547 C 796.18761,889.80325 793.62338,890.12463 792.639,891.249 C 790.97857,893.14557 790.73322,894.39299 788.45694,894.46022 C 787.93694,894.47558 786.50098,895.50924 786.37498,896.30162 C 786.19939,897.40588 784.98339,896.94999 784.34239,896.88799 C 783.32639,896.78799 782.52315,897.69763 782.386,898.5 C 782.20316,899.56969 781.20249,900.23053 780.034,900.532 C 779.65684,900.62931 777.90043,901.54578 778.24299,900.17002 C 778.33398,899.80459 777.18041,899.73645 776.3466,899.70919 C 774.14807,899.63731 770.9893,899.18984 769.3466,899.28679 C 768.308,899.34809 767.17061,900.6022 766.04161,900.2542 C 765.36861,900.0472 764.631,898.933 763.907,899.569 C 763.58427,896.93264 763.39726,894.01024 763.3258,891.50901 C 763.28922,890.2285 763.476,889.965 763.851,888.878 C 763.872,888.526 763.766,888.211 763.531,887.933 C 763.159,887.211 763.829,886.896 763.987,886.249 C 762.32697,886.41298 760.54855,889.41193 759.19499,890.4906 C 757.18201,892.09476 753.832,891.429 751.527,891.139 C 751.919,889.893 750.8766,887.96299 749.9516,887.16799 C 748.2946,885.74499 745.42,886.688 743.607,885.74 C 745.716,882.977 745.029,882.196 742.758,879.865 C 742.274,879.368 742.1676,879.01999 741.9686,878.38599 C 741.7566,877.71199 741.041,877.413 740.629,876.89 C 739.68,875.688 739.9,874.174 738.423,873.209 C 739.89775,872.3635 737.09203,870.67594 736.705,869.763" id="path2812"/>
    <path d="M 941.978,1032.623 C 942.196,1032.838 942.413,1033.054 942.628,1033.272 C 942.88,1032.569 943.156,1031.818 942.558,1031.182 C 942.151,1031.577 941.958,1032.058 941.978,1032.623" id="path2868"/>
    <path d="M 944.428,1045.373 C 945.079,1044.473 945.659,1042.672 944.858,1041.692 C 943.806,1042.323 944.319,1044.417 944.428,1045.373" id="path2870"/>
    <path d="M 965.158,1012.033 C 965.262,1011.366 964.712,1010.921 965.018,1010.233 C 964.103,1010.037 963.794,1010.982 963.648,1011.672 C 964.195,1011.609 964.698,1011.729 965.158,1012.033" id="path2880"/>
    <path d="M 971.428,1005.553 C 972.019,1005.467 972.536,1005.101 972.868,1004.614 C 972.172,1004.118 971.087,1004.619 971.428,1005.553" id="path2884"/>
    <path d="M 1006.848,918.293 C 1007.397,917.77 1008.744,916.901 1007.638,916.133 C 1007.82,917.019 1006.805,917.441 1006.848,918.293" id="path2888"/>
  </g>
  <g class="landxx coastxx sl" id="sl" transform="translate(-29.9017, -45.0745)">
    <path d="M 1202.397,738.72 C 1202.327,738.888 1202.257,739.056 1202.187,739.224 C 1203.163,739.528 1204.117,739.945 1205.137,740.088 C 1205.314,739.704 1205.446,739.172 1205.012,738.897 C 1204.271,738.424 1203.213,738.77 1202.397,738.72" id="path2896"/>
    <path class="landxx sl" d="M 1199.448,726.479 C 1200.064,725.914 1200.855,726.408 1201.538,725.975 C 1202.2,725.556 1202.022,724.412 1202.83,724.175 C 1204.162,723.784 1205.062,721.21 1205.479,720.048 C 1206.177,718.107 1208.233,719.536 1209.681,718.567 C 1210.92,717.737 1214.389,717.673 1215.628,718.589 C 1216.819,719.469 1217.399,720.972 1218.173,722.191 C 1218.577,722.827 1219.02,723.477 1219.289,724.186 C 1219.486,724.705 1218.904,725.336 1219.265,725.707 C 1219.71,726.165 1220.205,726.059 1220.251,726.831 C 1220.287,727.435 1220.059,728.483 1220.522,728.969 C 1221.171,729.652 1218.985,731.863 1219.108,733.032 C 1219.273,732.67 1220.887,731.82 1221.268,731.592 C 1222.749,730.702 1222.5069,731.92217 1222.148,733.038 C 1221.738,734.31275 1219.8678,733.96478 1219.7004,735.81438 C 1219.54,737.58697 1219.4805,737.50153 1218.503,738.536 C 1217.7249,739.35942 1216.9615,740.1776 1215.85,740.987 C 1214.3533,742.07692 1213.6805,744.84757 1213.021,744.64738 C 1212.3174,744.43381 1206.498,741.303 1205.569,740.52 C 1207.193,740.981 1205.917,740.214 1205.499,739.945 C 1206.193,739.461 1207.095,739.458 1207.729,738.865 C 1206.393,739.031 1204.448,739.236 1205.569,737.281 C 1204.69,739.296 1203.597,737.031 1202.844,737.135 C 1202.311,737.209 1201.847,733.674 1201.829,733.248 C 1200.728,733.504 1199.002,732.589 1199.809,731.16 C 1200.286,731.536 1200.789,731.872 1201.319,732.168 C 1200.059,731.535 1201.911,730.457 1202.619,730.512 C 1202.331,730.033 1202.464,730.156 1202.549,729.576 C 1201.899,730.883 1201.265,730.212 1200.459,729.72 C 1200.348,729.979 1200.251,730.242 1200.169,730.512 C 1199.229,728.926 1200.014,728.317 1201.389,727.992 C 1200.858,727.29 1199.808,727.352 1199.448,726.479" id="path2902"/>
  </g>
  <path class="landxx ml" d="M 1178.5463,631.79755 C 1179.8973,631.47255 1180.5003,632.20355 1181.5993,630.79455 C 1182.5263,629.60555 1182.1103,627.04555 1182.5633,625.56355 C 1183.5913,622.17855 1188.2933,627.78655 1188.7763,628.84555 C 1189.6743,626.38355 1191.1693,625.03455 1194.0003,626.18355 C 1195.8613,626.93855 1199.8063,626.80455 1200.2163,624.23655 C 1201.2763,626.80455 1206.5013,625.65655 1208.5463,625.50055 C 1213.5483,625.11855 1218.6803,625.38955 1223.6953,625.38955 C 1225.4613,625.38955 1227.6374,625.18382 1229.3838,625.36217 C 1230.6668,625.4932 1230.2948,623.3601 1230.4193,622.26355 C 1230.5603,621.02185 1231.7691,618.63002 1230.4713,617.80555 C 1228.2296,616.38137 1228.8296,615.12311 1228.8443,613.31355 L 1223.6153,543.23755 L 1236.8626,543.24413 C 1250.5584,553.50778 1263.4102,563.72648 1276.3665,573.90517 C 1277.9234,575.03099 1280.8498,576.15835 1280.6323,578.50223 C 1280.5642,579.23582 1280.4606,579.88946 1281.7742,580.41646 C 1282.4129,580.67268 1283.1226,580.85169 1283.6646,581.21369 C 1284.5696,581.81769 1284.4206,582.90369 1285.1436,583.62669 C 1285.8316,584.31469 1286.5499,584.40906 1287.4373,584.56255 C 1288.0818,584.67402 1287.6054,583.96987 1288.7594,584.54328 C 1289.587,584.95454 1289.9463,586.05055 1290.8923,586.34855 C 1292.3663,586.81455 1294.3411,586.93833 1295.7061,587.74533 C 1296.7021,588.33433 1296.4128,591.10269 1295.9888,592.06669 C 1295.3988,593.40769 1294.5388,594.67982 1296.6728,594.98482 C 1298.9298,595.30882 1301.4862,594.10696 1303.7162,593.82996 C 1303.4264,599.97296 1304.3028,606.49745 1303.5068,612.58209 C 1303.4153,613.28114 1303.631,613.75755 1303.1053,617.08755 C 1302.9335,618.17607 1303.3182,616.78771 1301.7928,619.98855 C 1301.3242,620.97165 1301.3513,622.16055 1300.9243,623.17455 C 1300.5123,624.15155 1299.5603,624.44955 1298.7813,625.05155 C 1297.8233,625.79055 1298.4763,626.76455 1297.1623,626.54055 C 1296.8623,626.48955 1294.2403,625.78955 1294.1203,626.24355 C 1293.8053,627.43155 1285.6115,627.02987 1284.3695,627.12887 C 1281.3285,627.37087 1281.3913,626.99755 1279.2393,629.35055 C 1278.5643,630.08955 1277.2363,630.35155 1276.2893,630.14255 C 1275.7413,630.04255 1274.4873,630.03041 1273.9313,630.04441 C 1272.9413,630.02541 1273.0053,629.26455 1273.0053,630.50355 C 1272.2873,630.23255 1271.6019,629.88846 1270.8933,629.59155 C 1269.189,628.87742 1268.7799,630.07448 1268.1338,629.59741 C 1267.505,629.13308 1266.4288,628.52475 1265.0153,629.78455 C 1263.5425,631.09723 1261.4052,632.19274 1260.564,632.75622 C 1259.1505,633.70293 1258.1129,634.44858 1256.6673,634.81255 C 1255.479,635.11174 1256.4463,637.62855 1254.6863,636.85755 C 1253.6033,636.38355 1252.6158,635.72528 1251.4678,636.52028 C 1250.4918,637.19728 1249.9643,638.44155 1249.6073,639.51055 C 1249.4033,640.12155 1249.1267,641.10205 1248.8397,641.68105 C 1248.5907,641.50989 1246.9973,641.12106 1246.3653,641.01655 C 1246.9313,642.02655 1246.3843,643.51955 1246.5153,644.61755 C 1245.5413,644.58355 1244.7613,644.42455 1245.0053,645.62555 C 1243.9713,645.52255 1240.9853,644.01055 1241.0453,642.74455 C 1239.1497,647.28505 1237.7443,645.14406 1238.7023,646.76255 C 1239.2324,647.65805 1239.2524,648.33676 1239.0389,649.10882 C 1238.9518,649.4239 1236.6752,649.0246 1237.2282,649.97846 C 1238.2474,651.73675 1237.7975,653.62981 1235.8237,655.2968 C 1234.9056,656.07218 1233.8113,655.75755 1232.7003,655.95955 C 1231.6623,656.14755 1231.0803,657.27455 1229.9553,657.21855 C 1231.9533,657.72029 1230.5053,658.76203 1231.1116,659.11442 C 1232.1402,659.71235 1231.2631,662.35798 1230.6954,663.3033 C 1230.5081,663.6153 1229.4368,663.74629 1229.5013,664.04655 C 1229.8267,665.56055 1230.0235,667.43155 1228.9803,668.79455 C 1228.5615,669.34179 1227.3393,669.32155 1226.7483,669.48555 C 1225.5113,669.82955 1226.1523,671.47255 1224.5893,671.32955 C 1222.8573,671.17055 1223.9943,667.71755 1223.4053,666.64955 C 1222.7323,666.92755 1222.6053,668.13055 1221.9653,668.30555 C 1221.2983,668.48755 1220.7923,667.56855 1220.3053,667.29755 C 1220.2663,668.11555 1220.7043,670.16755 1219.4673,670.14955 C 1218.5413,670.13455 1217.2403,670.20655 1218.0063,671.47355 C 1216.8723,671.97555 1214.0333,670.57455 1214.1863,669.16955 C 1213.6033,669.42855 1212.6603,669.12955 1212.2163,669.66355 C 1211.5203,670.50155 1211.6113,671.49955 1210.2263,671.61755 C 1211.2483,669.71455 1207.8733,669.43355 1207.8653,667.81455 C 1207.8603,666.77955 1208.1253,665.57155 1207.8823,664.56455 C 1207.5353,663.12855 1205.6293,664.80755 1204.9663,664.77655 C 1205.2193,663.85455 1205.9523,661.41955 1207.3463,661.89655 C 1207.8503,660.50755 1204.9923,659.99455 1204.8263,658.65655 C 1203.9183,659.17255 1204.0003,658.15155 1204.0833,657.57055 C 1204.2693,656.25855 1203.7423,655.75355 1203.1873,654.65455 C 1202.7503,653.78855 1203.3003,652.56155 1202.2833,652.05455 C 1201.8873,651.85655 1200.5053,651.09055 1200.0953,651.47655 C 1199.6443,651.90055 1201.0973,652.96755 1200.1203,653.42855 C 1199.0133,653.95055 1198.1413,654.83655 1197.0233,655.39455 C 1196.2483,655.78155 1195.3803,654.66655 1194.7463,654.40755 C 1193.3763,653.84855 1192.3203,653.94855 1191.3493,655.10555 C 1188.9083,658.01555 1190.2473,654.54955 1188.1273,653.83255 C 1186.7503,653.36755 1185.2953,657.77055 1183.9453,654.37355 C 1183.5133,653.28755 1185.3083,652.29355 1184.4393,651.06755 C 1183.9393,650.36155 1184.3313,648.67855 1184.6023,647.93955 C 1184.9213,647.06755 1182.2663,642.09855 1181.7173,644.25655 C 1181.0933,644.07955 1180.1003,642.14655 1179.8253,641.55955 C 1179.5783,641.03155 1180.3763,639.97355 1180.3463,638.92855 C 1180.3023,637.35855 1180.4813,636.69055 1179.3703,635.45555 C 1178.9843,635.02655 1178.5803,634.51455 1178.8373,633.89655 C 1179.2753,632.83755 1178.9703,632.83855 1178.5463,631.79755" id="ml"/>
  <path class="landxx cd" d="M 1363.8063,809.49855 C 1364.4923,808.97755 1365.3416,809.41874 1365.6106,808.48674 C 1365.8286,807.73174 1366.451,804.76555 1365.921,804.15855 C 1364.534,802.57055 1369.9153,800.17055 1370.9263,799.26755 C 1371.1993,800.41455 1372.8793,800.58155 1372.9463,801.78855 C 1373.4843,801.22955 1374.1953,800.89855 1374.7933,800.41755 C 1375.6093,799.76155 1375.2683,798.57255 1375.9763,797.82855 C 1377.1763,799.41955 1378.9733,796.37555 1380.3903,797.08455 C 1381.7403,797.76055 1380.4943,798.06155 1380.2863,798.90855 C 1380.0533,799.85955 1380.6563,800.90155 1380.6563,801.85755 C 1381.2553,801.08555 1382.5783,802.05955 1383.3863,801.49855 C 1384.5793,800.66955 1385.3203,799.09455 1386.2283,797.98255 C 1386.6993,797.40655 1388.6143,796.48155 1388.7153,795.95855 C 1388.9843,794.57655 1390.1343,794.24555 1391.2763,793.75855 C 1393.9333,792.62455 1394.4093,787.62055 1394.3453,785.23055 C 1394.2963,783.41155 1394.4623,781.76755 1394.2333,779.95155 C 1394.0143,778.20955 1396.207,777.76255 1396.9426,776.38455 C 1398.6756,773.13802 1398.5434,771.08368 1402.0653,768.69255 C 1403.5473,767.68636 1404.4503,766.50355 1405.4343,765.11555 C 1406.4143,763.73355 1405.7063,761.71555 1406.1043,760.13055 C 1406.4543,758.73855 1407.1353,752.76255 1407.1953,751.19555 C 1407.2633,749.42455 1408.2893,747.87155 1408.5803,746.14555 C 1408.8623,744.47455 1408.3743,742.70755 1408.7503,741.05355 C 1409.4573,737.93955 1411.3169,735.39202 1412.3819,732.43102 C 1413.0009,730.71102 1412.7033,728.89755 1412.6553,727.14355 C 1412.6333,726.33455 1413.0083,725.48655 1412.8793,724.69555 C 1412.7653,723.99055 1412.3453,723.40255 1412.1523,722.72455 C 1411.8463,721.64655 1413.7273,721.67555 1414.1583,720.69255 C 1414.7543,719.33555 1415.3823,718.14355 1416.4103,717.04555 C 1417.3943,715.99355 1418.6253,715.09355 1420.1323,715.10455 C 1421.4083,715.11355 1421.9183,716.21655 1422.7333,716.39755 C 1424.6273,716.81655 1426.3833,718.86355 1427.0903,720.60355 C 1427.6663,722.02455 1429.7083,720.96355 1430.8613,721.31055 C 1431.4863,721.49855 1431.7953,722.34355 1432.5143,722.30955 C 1433.1513,722.27855 1433.2823,722.16455 1433.9403,722.50055 C 1434.9313,723.00655 1435.1293,722.49855 1436.1023,722.43655 C 1436.7523,722.39455 1437.4673,722.73855 1438.0633,722.95855 C 1439.1383,723.35655 1440.6063,724.13155 1441.7793,723.71855 C 1443.8763,722.98055 1443.2323,717.96955 1446.0273,717.83855 C 1447.6163,717.76255 1448.0403,720.98055 1450.1153,719.01555 C 1450.9923,718.18455 1457.0533,717.38255 1456.6873,715.39055 C 1457.9423,715.33655 1458.3633,717.05955 1459.5683,716.93855 C 1461.0153,716.79255 1463.4953,716.57455 1464.0973,715.03055 C 1464.3223,714.45455 1463.7423,713.87955 1464.5033,713.49755 C 1465.4363,713.03155 1466.2023,713.30955 1466.9303,714.02555 C 1468.0963,715.17255 1469.1773,713.82255 1470.4663,714.43355 C 1471.1043,714.73555 1471.6323,715.21555 1472.2573,715.53855 C 1472.9853,715.91455 1473.5043,715.59455 1474.2753,715.71055 C 1476.1053,715.98555 1476.8153,713.95455 1478.7453,715.04855 C 1479.4793,715.46455 1482.0273,717.06955 1482.0273,718.01855 C 1482.0273,718.48355 1482.6303,719.63055 1483.0813,719.81455 C 1484.9773,720.58755 1486.7843,723.38155 1488.8093,721.47355 C 1489.9563,720.39155 1490.8033,720.17555 1492.3253,720.91855 C 1493.8893,721.68355 1494.5111,721.37409 1495.4343,719.55555 C 1496.6509,717.15905 1499.7571,723.16319 1499.9163,723.55555 C 1500.6284,725.31003 1505.367,725.5411 1503.9183,728.27855 C 1504.8393,728.06255 1505.9533,728.00555 1506.1833,729.17855 C 1506.2993,729.49155 1506.4583,729.78155 1506.6623,730.04555 C 1506.7913,730.64755 1505.97,731.77341 1506.0597,732.40068 C 1506.3476,734.41308 1506.006,736.41592 1505.7198,738.33424 C 1505.569,739.34482 1506.3192,739.21208 1507.453,739.0118 C 1507.9738,738.9198 1509.9611,740.3074 1509.5985,741.40849 C 1508.733,744.03698 1505.4703,746.64455 1503.5943,748.59255 C 1502.9738,749.23655 1501.7247,750.95036 1500.9734,751.58286 C 1500.2222,752.21536 1500.1491,751.67642 1499.7176,752.50142 C 1499.1776,753.53542 1499.8383,754.79755 1499.1833,755.85555 C 1498.4573,757.02955 1497.7353,758.79855 1497.5803,760.18055 C 1497.4413,761.41855 1497.4583,762.66955 1497.3723,763.91155 C 1497.2713,765.36455 1497.076,766.63355 1496.927,768.01355 C 1496.819,769.02255 1497.0773,770.43855 1496.6133,771.36255 C 1496.0863,772.41355 1495.2337,772.80824 1494.4647,773.65924 C 1493.2747,774.97624 1493.3742,775.40055 1493.3312,776.94555 C 1493.3032,777.96355 1492.2122,779.10936 1491.6612,779.83636 C 1490.4452,781.43936 1492.1385,783.11379 1493.0776,784.6413 C 1493.7849,785.79167 1494.3011,786.60032 1493.7753,788.18524 C 1491.8638,789.85939 1494.1815,795.11705 1492.3734,796.84412 C 1493.6905,797.60071 1493.4499,794.20669 1493.9067,794.78129 C 1494.3593,797.4537 1492.6868,799.39177 1492.6738,802.01104 C 1492.6569,805.4135 1495.5232,806.98042 1494.3172,809.33255 C 1493.6208,810.69078 1492.8882,810.98973 1493.4768,811.86514 C 1494.3774,813.20471 1494.7597,815.41345 1496.5385,818.89733 C 1497.7272,821.22547 1501.124,820.78883 1501.331,826.39662 C 1501.4123,828.59797 1503.13,827.39032 1503.7904,830.76699 C 1499.0834,831.83999 1495.6123,831.96255 1490.8923,832.97255 C 1492.8433,836.19655 1487.7803,837.53655 1486.8623,839.95155 C 1489.5783,840.78955 1488.6763,844.98655 1488.4933,847.03755 C 1488.3593,848.53055 1488.9253,849.94055 1488.8383,851.41855 C 1488.7923,852.19355 1488.6237,852.93005 1488.3147,853.59905 C 1487.8677,854.56705 1487.5543,855.29655 1487.3063,856.31855 C 1486.8173,858.32455 1486.1363,859.4723 1487.4103,861.4123 C 1488.2963,862.7623 1489.9285,864.46042 1491.0995,865.56742 C 1491.6245,866.06442 1494.4031,867.76756 1494.6203,866.73355 C 1495.0061,864.89719 1495.4884,865.24047 1497.2933,864.72155 C 1497.2933,867.61855 1497.1963,870.52655 1497.1183,873.42255 C 1497.1053,873.86955 1497.4393,875.77855 1496.6893,875.85355 C 1495.8113,875.94155 1495.6153,874.92555 1496.2223,874.44055 C 1494.9243,872.84755 1492.0133,877.81686 1490.6423,874.83286 C 1490.1853,873.83886 1489.757,872.57855 1489.087,871.67755 C 1488.706,871.16555 1487.6855,870.50655 1487.5675,869.90155 C 1487.3365,868.71855 1487.1023,867.34655 1485.6123,867.30455 C 1484.3643,867.26955 1483.8313,865.96455 1482.6323,865.82055 C 1481.6303,865.69955 1481.0163,866.4558 1480.3233,865.3548 C 1479.7873,864.5008 1479.7712,863.25655 1478.9882,862.55555 C 1477.8732,861.55555 1477.0804,858.07961 1476.1084,861.19761 C 1475.1054,864.41361 1473.35,862.72268 1470.841,862.81168 C 1469.15,862.87168 1468.7502,862.33474 1467.3232,861.68774 C 1467.2272,861.64474 1465.7272,861.28674 1465.6852,861.29374 C 1463.4092,861.69374 1462.7293,858.23155 1463.5323,856.65255 C 1462.1343,856.73155 1460.7289,857.02778 1459.4233,857.52255 C 1457.819,858.13051 1457.4797,858.92297 1455.6833,858.17255 C 1456.3754,856.93458 1456.416,855.30122 1455.8173,855.32955 C 1455.0379,855.36643 1454.7117,854.70298 1454.1709,854.19011 C 1452.9149,852.99914 1453.0879,854.40483 1452.7903,854.5078 C 1451.6777,854.89281 1451.1893,854.83555 1450.1853,854.57655 C 1448.9793,854.26355 1448.23,855.10731 1447.1613,855.4848 C 1445.7885,855.9697 1444.9152,855.04792 1444.026,855.20155 C 1443.7067,855.25671 1442.4373,855.11755 1442.3973,855.13855 C 1441.5383,855.60155 1441.0983,856.89955 1439.9153,856.87055 C 1440.1823,855.98155 1439.6493,855.28355 1439.4943,854.44855 C 1439.2803,853.30155 1440.072,853.09642 1440.324,852.21442 C 1440.907,850.16942 1440.1073,846.61955 1438.8653,844.97455 C 1437.3523,842.96755 1436.7043,841.71155 1436.8943,839.11355 C 1437.0903,836.42855 1438.0271,833.61155 1437.3291,830.93955 C 1437.0611,829.91855 1436.3351,829.03255 1436.3501,827.94955 C 1436.3681,826.72555 1437.0423,825.57955 1437.1353,824.36455 C 1437.2893,822.36255 1435.9933,822.5848 1434.3713,822.5698 C 1432.8273,822.5568 1431.2924,822.5293 1429.7523,822.59455 C 1426.5894,822.72855 1427.0718,822.03755 1427.9643,819.36055 C 1427.2553,819.36055 1426.5595,819.18068 1425.8695,819.45068 C 1425.4025,819.63268 1425.7463,819.83555 1425.4843,820.05955 C 1424.6473,820.77655 1420.8023,820.15055 1419.6843,820.15055 C 1420.0573,821.09655 1419.3646,822.20355 1419.5256,823.21455 C 1419.6566,824.03155 1419.6873,825.32855 1418.3843,825.05055 C 1418.8083,826.29655 1418.0543,827.53255 1418.3143,828.79055 C 1417.7853,828.78587 1414.3438,828.9053 1413.7853,828.58155 C 1412.1388,827.62715 1409.2379,829.40955 1408.6763,829.45955 C 1407.3669,829.24701 1406.1356,829.30887 1404.7043,829.31617 C 1402.8843,828.93617 1402.097,825.15255 1401.184,823.81855 C 1400.862,823.34855 1400.6123,823.01155 1400.2813,822.54455 C 1399.6693,821.68055 1399.949,821.06055 1399.821,820.13955 C 1399.594,818.50055 1397.8812,817.42444 1398.127,815.68374 C 1398.3479,814.11929 1398.122,812.81318 1397.3879,811.89855 C 1396.3636,810.62229 1396.3821,810.15213 1394.7893,810.22855 C 1390.4033,810.43899 1386.0553,810.61091 1381.6763,810.3613 C 1378.1393,810.15968 1373.9983,810.74455 1370.5253,810.07555 C 1367.7473,809.53955 1365.4453,813.67655 1363.8063,809.49855" id="cd"/>
  <g class="landxx coastxx it eu" id="it" transform="translate(-29.9017, -45.0745)">
    <path d="M 1391.118,486.792 C 1390.948,486.552 1390.778,486.313 1390.608,486.072 C 1391.372,485.978 1391.542,486.218 1391.118,486.792" id="path2916"/>
    <path class="landxx it eu" d="M 1353.5711,414.65164 C 1354.5811,413.57664 1357.2431,414.48464 1357.1191,412.24764 C 1357.0691,411.34564 1356.411,410.95 1355.838,410.332 C 1355.527,410.013 1355.1701,409.64432 1354.9011,409.28932 C 1355.7468,408.80713 1356.603,407.74092 1356.0773,407.16239 C 1356.9793,407.40239 1357.971,408.279 1358.939,407.942 C 1360.155,407.518 1360.7132,407.72227 1361.8742,407.80927 C 1362.6372,407.86627 1363.5707,406.94036 1363.9157,406.34836 C 1364.1957,405.89436 1364.237,405.492 1363.897,405.072 C 1364.723,404.85 1365.326,404.201 1365.837,403.559 C 1366.7328,404.11066 1366.3771,404.83194 1366.8665,405.61418 C 1367.3331,406.35991 1368.0421,406.55999 1368.6826,407.17827 C 1368.9307,407.41769 1368.9872,408.13197 1369.9254,408.70955 C 1370.2277,408.89569 1370.1022,406.85659 1370.7664,406.23418 C 1371.5221,405.52595 1371.7659,405.06534 1371.8639,403.91627 C 1372.031,401.95811 1372.6776,403.34771 1373.1147,403.81209 C 1373.8456,404.58865 1374.1087,404.70505 1375.0382,404.38736 C 1376.5411,403.87369 1376.5354,404.97701 1377.576,405.503 C 1377.682,404.844 1378.19,404.178 1377.351,403.796 C 1376.6,403.454 1377.596,402.10809 1378.076,402.01209 C 1378.551,401.91709 1379.0742,402.61727 1379.5152,402.73827 C 1380.4862,403.00527 1379.5847,401.41782 1379.6821,401.09736 C 1379.8709,400.47606 1379.9572,400.27088 1380.061,399.98291 C 1380.691,400.04569 1382.2718,400.52909 1382.6348,400.71509 C 1383.5878,401.20409 1384.238,399.946 1384.796,399.444 C 1385.541,398.775 1386.713,398.965 1387.617,399.018 C 1389.083,399.105 1390.488,398.006 1391.907,398.302 C 1390.335,399.245 1393.5281,401.70153 1394.4131,401.98253 C 1397.0751,402.82653 1398.9421,401.97352 1401.6001,402.74652 C 1401.4891,403.75352 1400.4052,404.4439 1399.897,405.071 C 1400.493,405.408 1401.089,405.784 1401.767,405.935 C 1401.191,406.611 1400.3093,407.586 1401.557,407.951 C 1400.8172,408.91389 1402.0239,409.27455 1402.4225,409.35427 C 1403.3678,409.5433 1403.6368,411.11004 1402.197,410.831 C 1403.204,408.754 1400.483,409.92 1399.967,410.111 C 1399.749,408.678 1398.009,409.983 1397.503,410.301 C 1396.948,410.65 1396.368,410.973 1395.783,411.267 C 1395.395,411.462 1392.102,412.526 1394.137,411.477 C 1393.591,411.59 1392.862,411.516 1392.407,411.91 C 1392.085,412.189 1392.237,413.34 1391.617,413.206 C 1391.611,413.539 1392.839,415.25 1392.958,415.283 C 1394.785,415.785 1393.629,418.438 1392.557,417.238 C 1390.47,420.924 1395.155,424.426 1397.68,426.037 C 1398.916,426.826 1400.917,427.381 1401.784,428.618 C 1403.048,430.419 1403.716,432.788 1404.457,434.843 C 1405.408,437.481 1408.874,440.18 1411.342,441.314 C 1413.169,442.154 1414.142,442.413 1416.174,442.141 C 1416.933,442.04 1420.283,441.547 1419.868,443.286 C 1419.67,444.116 1417.296,444.888 1418.165,445.903 C 1419.103,446.998 1421.587,447.621 1422.86,448.075 C 1424.89,448.8 1426.716,449.517 1428.415,450.871 C 1430.148,452.252 1432.276,452.662 1433.967,454.187 C 1435.104,455.212 1436.619,456.286 1436.335,457.994 C 1436.27,458.382 1436.189,460.128 1435.606,460.186 C 1434.812,460.265 1433.83,459.793 1433.409,459.103 C 1432.49,457.597 1432.551,455.853 1430.276,456.082 C 1428.449,456.266 1428.081,455.073 1427.826,454.39 C 1425.898,454.715 1425.475,454.696 1424.26,456.488 C 1423.508,457.599 1422.961,458.993 1422.686,460.308 C 1422.371,461.815 1424.292,462.182 1425.332,462.66 C 1427.002,463.427 1427.304,464.62 1427.468,466.308 C 1427.744,469.158 1422.814,467.109 1423.276,470.55 C 1423.502,472.233 1423.047,472.365 1421.973,473.508 C 1421.065,474.476 1420.361,475.67 1419.277,476.464 C 1418.258,477.211 1417.397,476.791 1416.886,475.702 C 1416.109,474.044 1418.516,473.275 1418.638,471.767 C 1418.679,471.255 1418.43,470.638 1418.784,470.181 C 1419.374,469.417 1420.532,469.988 1420.847,469.005 C 1421.121,468.149 1420.045,467.178 1419.834,466.396 C 1419.506,465.181 1419.401,464.113 1418.745,463.003 C 1417.959,461.673 1417.842,459.68 1416.797,458.556 C 1416.174,457.887 1415.74,458.046 1415.151,458.535 C 1414.448,459.12 1413.725,457.932 1413.264,457.562 C 1412.742,457.144 1411.893,457.254 1411.732,456.413 C 1411.6,455.72 1412.054,455.513 1411.686,454.661 C 1410.891,452.822 1408.822,452.779 1407.308,453.741 C 1408.085,452.562 1407.838,451.456 1406.238,451.617 C 1405.046,451.736 1404.372,449.977 1403.941,449.155 C 1403.176,447.694 1402.27,448.544 1401.021,448.176 C 1400.354,447.98 1399.732,447.666 1399.015,447.837 C 1397.513,448.194 1398.476,447.996 1397.441,447.151 C 1396.414,446.314 1395.116,446.084 1394.096,445.16 C 1392.963,444.135 1392.297,442.619 1391.057,441.754 C 1389.981,441.002 1389.563,441.193 1388.845,440.019 C 1388.096,438.793 1386.174,437.353 1384.709,438.261 C 1385.549,436.854 1382.122,433.222 1380.854,433.402 C 1381.004,433.381 1378.979,427.645 1378.902,427.272 C 1378.207,423.895 1375.907,424.272 1373.479,422.637 C 1370.967,420.946 1367.774,420.027 1365.666,422.756 C 1364.917,423.725 1364.296,424.843 1363.18,425.445 C 1362.785,425.659 1359.1707,426.80528 1359.4172,425.87946 C 1359.5982,425.19963 1362.298,422.20476 1359.085,423.34181 C 1357.1636,424.02178 1353.5426,420.98964 1355.9245,419.02164 C 1356.7792,418.31541 1356.6251,417.73137 1355.6125,417.22064 C 1353.9421,416.37814 1353.4559,415.09597 1353.678,415.008" id="path2906"/>
    <path d="M 1365.478,455.04 C 1365.228,454.465 1364.743,454.1 1364.399,453.595 C 1363.764,452.664 1364.262,451.792 1364.118,450.792 C 1366.017,452.938 1368.455,450.371 1369.948,449.088 C 1370.49,448.623 1371.247,447.823 1372.026,448.317 C 1372.535,448.64 1372.888,449.501 1373.618,449.135 C 1373.359,449.694 1373.576,450.041 1374.188,450.072 C 1373.953,450.266 1373.716,450.459 1373.478,450.649 C 1374.95,450.589 1375.582,454.014 1375.268,455.04 C 1374.995,455.934 1373.657,456.413 1374.486,457.459 C 1375.308,458.497 1374.537,461.137 1374.472,462.389 C 1374.424,463.327 1374.503,465.269 1373.57,465.86 C 1372.707,466.406 1371.269,465.322 1370.308,465.408 C 1369.806,466.172 1369.978,468.288 1368.49,468.039 C 1367.075,467.802 1367.05,468.178 1366.348,466.776 C 1366.157,467.017 1365.963,467.257 1365.768,467.496 C 1365.529,467.293 1365.335,467.053 1365.188,466.776 C 1366.443,466.526 1365.651,465.742 1365.609,464.67 C 1365.557,463.351 1365.746,462.04 1365.838,460.728 C 1366.079,460.918 1366.319,461.111 1366.558,461.304 C 1366.542,460.85 1366.746,460.372 1366.606,459.921 C 1366.454,459.431 1365.722,459.609 1365.617,459.025 C 1365.522,458.5 1366.167,457.795 1366.16,457.192 C 1366.15,456.405 1365.296,455.779 1365.478,455.04" id="path2910"/>
    <path d="M 1379.738,435.456 C 1379.029,435.022 1378.186,435.404 1377.508,434.88 C 1378.235,434.54 1379.09,434.619 1379.738,434.088 C 1379.814,434.544 1379.814,435 1379.738,435.456" id="path2912"/>
    <path d="M 1397.087,479.592 C 1396.207,479.564 1394.939,479.288 1394.399,478.52 C 1393.226,476.853 1395.606,475.605 1396.147,474.552 C 1397.222,476.115 1397.924,475.222 1399.16,474.534 C 1401.058,473.479 1402.408,476.388 1404.592,475.927 C 1405.843,475.662 1407.081,475.873 1408.335,475.647 C 1409.495,475.438 1410.581,474.255 1411.708,474.264 C 1412.708,474.273 1412.897,475.01 1413.852,474.339 C 1414.471,473.903 1416.159,473.121 1416.889,473.544 C 1415.968,474.689 1415.057,476.052 1414.474,477.404 C 1414.068,478.346 1412.859,480.233 1413.134,481.255 C 1413.374,482.148 1414.975,483.484 1414.482,484.493 C 1414.065,485.346 1412.936,485.988 1413.51,487.08 C 1412.371,487.346 1411,487.229 1409.89,486.888 C 1408.327,486.408 1408.41,484.574 1406.897,483.955 C 1405.301,483.301 1403.601,483.213 1402.051,482.353 C 1400.462,481.474 1399.032,479.635 1397.087,479.592" id="path2920"/>
  </g>
  <g class="landxx coastxx so" id="so">
    <path class="landxx so so-" d="M 1583.6963,739.50955 C 1583.6963,738.10255 1583.3333,736.15655 1583.9403,734.83955 C 1584.5183,733.58555 1585.698,732.61742 1586.491,731.48742 C 1588.079,729.22942 1589.4563,725.18755 1591.8633,723.73455 C 1594.0603,722.40755 1596.6313,723.26555 1598.5543,721.20055 C 1600.7383,718.85555 1602.8053,717.32355 1606.1103,716.84555 C 1608.5823,716.48855 1612.9273,718.24655 1614.6353,715.98255 L 1636.3253,690.33455 C 1633.7033,690.33455 1630.9833,690.78055 1628.4053,690.11855 C 1625.4833,689.36855 1622.6933,687.96055 1619.8693,686.90855 C 1616.9583,685.82455 1614.0623,684.69955 1611.1573,683.59955 C 1608.4683,682.58155 1606.1083,681.87055 1604.0443,679.80155 C 1603.3543,679.10855 1603.1303,678.74455 1602.1733,678.49055 C 1601.1993,678.23155 1600.8203,676.72155 1600.3653,675.92255 C 1599.5933,674.56655 1599.0793,673.46955 1597.9853,672.27355 C 1596.6813,670.84655 1595.1473,669.01655 1595.7413,666.89355 C 1596.3933,664.56355 1598.434,662.51842 1599.344,660.22142 C 1601.746,661.56142 1603.0693,664.76255 1604.7123,666.83555 C 1606.7643,669.42355 1609.8913,670.14155 1613.0643,669.09555 C 1614.2793,668.69555 1615.8523,667.61155 1616.8903,666.84555 C 1618.7233,665.49155 1619.2413,665.71955 1621.4973,666.09055 C 1623.8283,666.47455 1624.8773,666.96255 1627.0543,665.60455 C 1628.1533,664.91855 1630.6753,662.36055 1632.0913,662.86455 C 1633.6513,663.41855 1634.7273,663.48755 1636.4043,662.91055 C 1637.4153,662.56255 1638.3733,662.05055 1639.4233,661.81755 C 1640.6253,661.54955 1641.4293,662.07255 1642.5753,662.18655 C 1644.4533,662.37455 1645.8913,661.26055 1647.5153,660.59955 C 1649.2083,659.91055 1651.1343,660.21555 1652.8413,659.34855 C 1654.2623,658.62655 1654.4063,656.77055 1655.6143,656.17655 C 1656.7853,655.59955 1660.5673,656.60255 1660.1023,658.29255 C 1659.4993,660.48155 1659.0933,661.81655 1659.4963,664.13755 C 1659.7753,665.75455 1660.1973,668.07255 1658.7843,669.31255 C 1659.7243,669.66555 1660.5013,669.44655 1660.2943,668.44855 C 1660.7603,668.75955 1661.2673,668.97555 1661.8143,669.09655 C 1661.5023,670.64755 1657.9853,668.52255 1657.9723,671.24555 C 1657.9623,673.46955 1658.1493,675.74555 1657.5303,677.92555 C 1657.0023,679.78755 1655.8823,681.21255 1654.8293,682.77555 C 1653.6033,684.59955 1653.7463,687.30155 1652.2443,689.05855 C 1651.6613,689.74055 1650.8573,689.91455 1650.4533,690.81455 C 1650.0513,691.70955 1650.3993,692.71955 1650.0643,693.66655 C 1649.2983,695.82655 1647.7533,697.54155 1646.6773,699.53155 C 1645.2443,702.17955 1645.1283,705.09455 1644.0093,707.83255 C 1642.8723,710.61555 1641.1003,713.07455 1639.4683,715.57955 C 1637.8853,718.01055 1636.8363,720.82655 1635.0243,723.09655 C 1633.0123,725.61755 1630.9953,728.13155 1628.9443,730.62055 C 1624.8213,735.62455 1620.6293,739.69155 1615.1653,743.20755 C 1612.8763,744.68155 1610.5983,746.21255 1608.6863,748.16855 C 1606.5333,750.36955 1604.0713,752.54255 1602.1253,754.92055 C 1600.1993,757.27455 1598.1743,759.56555 1596.2713,761.92855 C 1595.3443,763.07955 1594.5333,764.31655 1593.6313,765.48755 C 1593.0733,766.21155 1591.4023,768.82455 1591.1053,767.30055 C 1591.1053,769.77655 1589.4882,772.15649 1587.9952,774.02549 C 1588.1233,772.08327 1586.2488,770.47715 1584.9162,768.55949 C 1583.3947,766.37006 1583.637,764.8623 1583.612,762.3263 C 1583.547,754.6593 1583.6963,747.17355 1583.6963,739.50955" id="so-"/>
    <path class="limitxx so xs" d="M 1636.3253,690.33455 C 1633.7033,690.33455 1630.9833,690.78055 1628.4053,690.11855 C 1625.4833,689.36855 1622.6933,687.96055 1619.8693,686.90855 C 1616.9583,685.82455 1614.0623,684.69955 1611.1573,683.59955 C 1608.4683,682.58155 1606.1083,681.87055 1604.0443,679.80155 C 1603.3543,679.10855 1603.1303,678.74455 1602.1733,678.49055 C 1601.1993,678.23155 1600.8203,676.72155 1600.3653,675.92255 C 1599.5933,674.56655 1599.0793,673.46955 1597.9853,672.27355 C 1596.6813,670.84655 1595.1473,669.01655 1595.7413,666.89355 C 1596.3933,664.56355 1598.434,662.51842 1599.344,660.22142 C 1601.746,661.56142 1603.0693,664.76255 1604.7123,666.83555 C 1606.7643,669.42355 1609.8913,670.14155 1613.0643,669.09555 C 1614.2793,668.69555 1615.8523,667.61155 1616.8903,666.84555 C 1618.7233,665.49155 1619.2413,665.71955 1621.4973,666.09055 C 1623.8283,666.47455 1624.8773,666.96255 1627.0543,665.60455 C 1628.1533,664.91855 1630.6753,662.36055 1632.0913,662.86455 C 1633.6513,663.41855 1634.7273,663.48755 1636.4043,662.91055 C 1637.4153,662.56255 1638.3734,662.05081 1639.4233,661.81755 C 1639.7729,661.73988 1640.0888,661.72857 1640.3892,661.75491 L 1640.4678,685.32204 L 1636.3253,690.33455 z" id="xs"/>
  </g>
  <path class="landxx af" d="M 1712.2163,464.82955 C 1711.9493,463.90155 1712.5733,463.19255 1713.3013,462.72755 C 1714.4363,462.00355 1713.8273,461.50955 1713.2163,460.65255 C 1715.1353,460.43455 1715.0583,457.98455 1715.1853,456.55955 C 1715.2493,455.82855 1714.4253,454.98255 1714.8483,454.27455 C 1715.2173,453.65655 1715.2253,452.70255 1715.3413,452.00755 C 1715.5963,450.48255 1717.6773,452.53055 1718.3583,452.87955 C 1719.3813,453.40355 1720.1573,452.72855 1721.1863,453.01155 C 1722.3103,453.32155 1723.1613,454.40055 1723.5163,455.46855 C 1724.2333,453.94955 1726.1253,454.78655 1727.2333,453.78855 C 1727.8263,453.25455 1728.4823,452.98255 1728.8463,452.22755 C 1729.1843,451.52555 1728.4693,451.22755 1729.4163,450.71555 C 1728.9093,450.36055 1728.4783,449.86355 1728.2663,449.27655 C 1731.3703,449.67655 1734.6833,447.56755 1737.0913,445.86055 C 1739.4573,444.18355 1736.4673,438.88755 1739.3833,437.69055 C 1740.4423,437.25555 1742.3053,437.87855 1743.1763,437.32355 C 1744.1003,436.73455 1743.0063,435.00855 1744.3393,434.79255 C 1745.3493,434.62955 1746.4743,435.67155 1747.2893,436.12055 C 1748.5123,436.79355 1749.5833,436.42655 1750.9033,436.25455 C 1752.4323,436.05455 1753.8023,436.01555 1754.9243,437.18855 C 1756.0533,438.37055 1757.3723,437.11155 1758.7313,437.48455 C 1759.2763,437.63355 1760.2853,438.95055 1760.7703,439.38855 C 1761.8353,440.34755 1761.9483,439.17755 1763.0463,439.41255 C 1762.8603,438.17255 1764.9533,438.11455 1765.5683,437.43555 C 1766.2003,436.73955 1766.1063,437.57055 1766.7153,437.32355 C 1767.0133,437.20255 1766.9863,436.52155 1767.3913,436.58455 C 1768.4733,436.75155 1768.8433,438.78955 1770.1843,438.48155 C 1771.7933,438.11255 1770.0283,436.08055 1770.6853,435.13455 C 1772.0773,433.13455 1775.0843,435.51655 1775.9603,434.56155 C 1777.0323,433.39155 1775.3973,432.96155 1775.2153,432.13955 C 1775.0053,431.18255 1776.3583,430.05555 1776.7073,429.10155 C 1777.6713,426.47355 1781.0953,426.24755 1782.5653,428.84055 C 1782.9083,429.44555 1782.4033,431.37855 1783.0673,431.59455 C 1783.7473,431.81655 1784.7233,430.91055 1785.1453,431.88755 C 1785.5873,432.91455 1785.3263,435.11655 1785.3753,436.23655 C 1785.4313,437.49755 1785.6353,438.78955 1786.0963,439.96755 C 1787.5233,443.61555 1789.6903,441.60855 1791.7503,439.88055 C 1792.5983,439.16955 1794.3093,439.96455 1794.7713,438.87155 C 1795.3913,437.40555 1796.2933,436.50255 1797.7783,435.76155 C 1798.8773,435.21255 1803.0703,435.32555 1801.2743,437.32355 C 1802.5263,437.84755 1804.5917,436.44919 1805.7103,435.89055 C 1806.6439,435.4243 1809.6782,436.28482 1810.2053,437.39555 C 1809.3633,437.77555 1808.5362,437.05609 1807.6902,437.46609 C 1806.6872,437.95209 1807.3504,438.81891 1808.4004,439.16091 C 1807.4984,439.46891 1806.3109,439.88428 1805.6173,440.52655 C 1804.4323,441.6238 1803.5739,440.00673 1802.4063,440.26555 C 1799.9265,440.81526 1797.381,440.52626 1795.0933,440.89855 C 1794.0077,441.07521 1793.0791,441.69952 1792.0093,441.99655 C 1790.7999,442.33231 1790.5565,443.89629 1789.7808,444.30455 C 1788.9293,444.7527 1788.6567,443.7923 1787.7533,445.06455 C 1787.3137,445.68368 1786.4288,446.56846 1786.2228,447.25746 C 1785.9908,448.03146 1787.1262,448.54701 1787.6112,448.91201 C 1788.6831,450.22172 1789.8245,451.39369 1790.38,453.34473 C 1789.3571,454.58083 1790.909,454.50419 1790.8814,455.25646 C 1790.2501,455.74527 1790.3678,456.24355 1790.0923,456.94583 C 1789.5544,458.31685 1788.3828,458.6342 1788.0013,459.33555 C 1787.804,459.69815 1787.7403,460.04955 1787.8853,460.43455 C 1787.6723,460.44555 1786.9983,460.48455 1787.1703,460.90855 C 1787.4123,461.50655 1788.1593,461.87655 1788.5913,462.32155 C 1790.1003,463.87555 1787.7325,465.51573 1786.6115,465.76173 C 1785.5195,466.00173 1784.2343,465.79155 1783.1413,465.57155 C 1782.5983,465.46255 1781.0654,464.89593 1780.5461,465.41464 C 1779.4276,466.53182 1782.789,467.5472 1783.1086,468.91855 C 1783.2713,469.61693 1784.4323,470.79355 1784.0883,471.48155 C 1783.7503,472.15855 1782.1692,473.12415 1781.2983,473.11355 C 1778.2534,473.07649 1779.1176,474.02008 1779.2496,475.22074 C 1779.3577,476.20413 1779.2363,477.36655 1778.5553,478.14655 C 1777.8163,478.99255 1778.5923,480.11255 1778.8293,481.03155 C 1779.2723,482.75155 1779.0173,484.90055 1777.2573,485.81655 C 1776.2133,486.36055 1775.7848,485.79955 1775.2688,484.97455 C 1775.0563,484.63555 1774.1255,484.33255 1773.7041,484.25842 C 1773.4365,484.20993 1773.9064,484.40457 1773.4853,484.91555 C 1769.479,483.44635 1771.8437,486.27774 1767.2853,486.85955 C 1767.5163,487.75155 1768.5273,487.78855 1769.0853,488.37155 C 1767.9903,488.95555 1766.7658,489.33184 1765.5043,489.45855 C 1764.2879,489.58073 1763.797,488.72267 1763.148,488.69146 C 1761.9851,488.63553 1761.4117,490.82525 1760.5623,491.42455 C 1759.546,492.14162 1759.3193,492.11155 1759.2363,493.66055 C 1759.1593,495.09255 1759.7343,496.21055 1759.8573,497.59755 C 1759.7323,498.14355 1759.7252,498.51437 1759.6222,499.06537 C 1759.6902,499.59037 1760.1565,499.88955 1760.3445,500.31755 C 1761.0255,501.86155 1755.0743,502.88255 1754.3183,503.06255 C 1753.0453,503.36555 1752.1021,503.932 1750.7711,503.877 C 1749.4341,503.822 1747.8171,503.32319 1746.5801,503.98719 C 1745.3731,504.63519 1744.61,505.31073 1743.208,504.91973 C 1741.901,504.55573 1740.3959,504.14228 1739.0469,504.32728 C 1736.0529,504.74128 1732.6219,505.36737 1729.6319,504.48937 C 1726.1229,503.45737 1723.8593,501.99455 1720.2743,501.18855 C 1721.8023,499.43455 1722.8893,497.36455 1724.1993,495.45455 C 1725.3263,493.81255 1726.4513,492.01355 1725.7663,489.93855 C 1724.5143,486.14355 1718.6723,489.14055 1717.7543,484.98855 C 1717.1733,482.35655 1717.0703,480.07055 1715.8733,477.59055 C 1714.8413,475.45255 1713.4653,473.38155 1714.9433,471.16555 C 1716.1353,469.37855 1715.8621,469.57928 1714.0113,469.07355 C 1712.3193,468.61121 1712.0469,465.77692 1712.2163,464.82955" id="af"/>
  <g class="landxx coastxx bd" id="bd" transform="translate(-29.9017, -45.0745)">
    <path d="M 1977.479,602.496 C 1977.265,602.472 1977.052,602.448 1976.838,602.424 C 1977.223,602.718 1977.652,602.91 1978.128,603 C 1977.911,602.832 1977.695,602.664 1977.479,602.496" id="path3240"/>
    <path d="M 1977.768,607.248 C 1977.795,607.696 1977.989,608.057 1978.348,608.328 C 1978.248,607.773 1979.269,607.317 1979.279,607.391 C 1979.158,606.422 1978.216,606.762 1977.768,607.248" id="path3242"/>
    <path d="M 1978.559,605.952 C 1978.752,606.096 1978.946,606.24 1979.139,606.384 C 1978.992,606.177 1978.799,606.033 1978.559,605.952" id="path3244"/>
    <path d="M 1978.639,608.688 C 1978.935,609.447 1979.64,609.931 1979.873,610.733 C 1980.004,611.277 1980.019,611.824 1979.916,612.374 C 1979.857,612.798 1979.18,614.973 1980.218,613.44 C 1980.01,613.716 1979.991,613.979 1980.162,614.229 C 1981.302,614.735 1981.423,611.015 1981.298,610.559 C 1981.045,609.64 1979.542,608.821 1979.521,607.985 C 1979.501,607.201 1978.469,607.88 1978.639,608.688" id="path3246"/>
    <path d="M 1978.918,613.872 C 1979.108,614.158 1979.439,613.472 1979.139,614.376 C 1979.463,613.985 1979.533,613.553 1979.348,613.08 C 1979.204,613.344 1979.062,613.608 1978.918,613.872" id="path3248"/>
    <path d="M 1979.568,610.704 C 1979.53,611.029 1979.58,611.342 1979.717,611.64 C 1979.783,611.31 1979.734,610.998 1979.568,610.704" id="path3250"/>
    <path d="M 1982.809,613.584 C 1984.492,612.719 1983.392,610.959 1982.668,609.84 C 1982.979,611.05 1982.955,612.352 1982.809,613.584" id="path3252"/>
    <path d="M 1986.188,611.28 C 1986.908,610.548 1985.771,609.467 1985.257,609.048 C 1985.026,609.927 1985.493,610.767 1986.188,611.28" id="path3254"/>
    <path d="M 1989.358,615.024 C 1989.18,615.518 1989.157,616.022 1989.288,616.536 C 1989.428,616.038 1989.451,615.534 1989.358,615.024" id="path3256"/>
    <path d="M 1989.938,618.84 C 1991.139,618.394 1990.252,616.903 1989.427,616.609 C 1989.592,617.354 1989.68,618.118 1989.938,618.84" id="path3258"/>
    <path class="landxx bd" d="M 1957.318,575.424 C 1957.345,575.101 1957.419,574.789 1957.538,574.488 C 1958.354,575.077 1959.725,575.457 1960.134,576.474 C 1960.502,577.393 1962.029,577.36 1962.798,577.656 C 1962.589,577.067 1962.252,576.563 1961.788,576.144 C 1962.994,575.922 1963.239,577.02 1963.669,577.863 C 1964.312,579.123 1965.976,579.399 1967.118,580.032 C 1967.36,579.426 1967.534,578.845 1967.188,578.231 C 1969.412,578.043 1969.915,582.416 1969.993,583.773 C 1970.044,584.67 1969.795,585.474 1970.832,585.828 C 1972.387,586.358 1974.252,586.799 1975.898,586.799 C 1979.38,586.799 1982.926,586.484 1986.426,586.82 C 1988.045,586.975 1992.549,589.754 1988.638,589.391 C 1989.805,591.611 1987.121,593.621 1986.908,595.44 C 1986.218,594.828 1985.523,595.981 1985.038,595.223 C 1985.176,596.421 1983.843,596.186 1983.098,596.087 C 1983.407,597.072 1982.226,598.726 1982.146,599.799 C 1982.034,601.306 1983.522,603.508 1984.109,604.871 C 1984.247,604.396 1984.27,603.916 1984.178,603.431 C 1984.898,603.751 1986.116,607.473 1987.441,605.287 C 1987.934,604.472 1986.959,603.099 1987.419,602.062 C 1987.872,601.041 1988.123,600.843 1987.779,599.687 C 1988.758,600.551 1989.784,598.373 1990.423,599.914 C 1991.181,601.739 1991.568,603.86 1992.116,605.76 C 1992.823,608.208 1993.678,610.545 1994.297,613.03 C 1994.894,615.424 1995.039,618.069 1995.989,620.352 C 1995.14,620.608 1994.115,619.105 1993.127,619.268 C 1991.644,619.512 1993.67,623.885 1993.679,624.744 C 1992.995,623.664 1991.768,622.347 1991.404,621.133 C 1991.017,619.848 1990.967,619.088 1990.8,617.832 C 1990.487,615.509 1987.301,613.026 1989.08,610.632 C 1988.62,611 1988.338,611.711 1988.35,612.288 C 1987.692,610.309 1985.161,608.684 1985.33,606.456 C 1985.264,607.381 1984.227,607.73 1983.707,608.345 C 1983.498,608.569 1983.439,608.827 1983.53,609.12 C 1982.903,609.33 1982.329,609.024 1982.019,608.472 C 1982.177,608.734 1982.188,608.985 1982.051,609.225 C 1981.031,609.816 1979.437,606.071 1979.134,605.664 C 1978.854,605.19 1978.75,604.682 1978.825,604.137 C 1978.85,603.227 1978.439,602.981 1978.188,602.206 C 1977.995,601.608 1979.037,601.375 1978.639,600.84 C 1978.501,601.396 1979.362,601.133 1977.989,601.273 C 1978.022,601.006 1978.068,600.742 1978.129,600.48 C 1977.773,600.769 1977.39,600.817 1976.98,600.625 C 1977.293,600.852 1978.554,602.469 1977.382,602.318 C 1976.796,602.243 1975.749,601.871 1975.39,601.777 C 1976.329,602.343 1977.73,602.805 1978.292,603.818 C 1978.675,604.507 1977.954,604.986 1977.409,605.232 C 1978.152,606.131 1977.744,606.946 1977.119,607.753 C 1978.472,607.782 1978.039,609.104 1977.989,610.057 C 1978.165,609.806 1978.355,609.566 1978.559,609.337 C 1979.682,610.336 1979.896,612.944 1978.059,613.297 C 1978.178,612.787 1978.105,612.308 1977.838,611.858 C 1978.004,613.046 1977.67,614.028 1977.451,615.171 C 1977.193,616.525 1975.591,615.974 1976.258,614.882 C 1976.049,615.106 1975.832,615.321 1975.609,615.53 C 1975.241,614.25 1976.692,613.788 1976.838,612.651 C 1976.435,613.311 1975.824,613.807 1975.248,614.305 C 1975.306,613.823 1975.449,613.367 1975.678,612.938 C 1975.374,613.398 1975.064,613.854 1974.748,614.305 C 1974.285,612.841 1974.091,611.595 1974.678,610.129 C 1973.094,610.773 1974.482,612.242 1973.925,613.315 C 1973.522,614.091 1974.479,614.141 1974.143,615.083 C 1973.842,615.928 1973.01,615.466 1972.659,614.953 C 1972.595,615.407 1972.571,615.863 1972.589,616.321 C 1971.463,616.334 1972.009,612.113 1972.009,611.425 C 1971.633,612.381 1972.101,613.337 1971.869,614.305 C 1971.501,613.504 1971.112,612.67 1970.999,611.785 C 1971.012,613.22 1972.441,615.366 1970.999,616.825 C 1970.577,616.113 1970.952,615.27 1970.429,614.593 C 1970.716,615.311 1970.242,616.875 1969.709,617.401 C 1969.784,616.028 1968.323,614.966 1968.151,613.457 C 1967.961,611.786 1967.03,610.51 1966.678,608.906 C 1966.508,608.133 1966.35,607.357 1966.109,606.601 C 1965.783,605.58 1966.07,605.093 1965.986,604.121 C 1965.938,603.57 1965.16,603.609 1964.783,603.517 C 1963.994,603.325 1964.471,602.409 1964.529,601.87 C 1964.656,600.693 1962.444,600.613 1962.532,599.041 C 1962.577,598.234 1963.471,597.951 1963.57,597.211 C 1963.67,596.462 1963.382,595.607 1963.097,594.928 C 1962.631,593.818 1960.381,593.702 1959.338,592.993 C 1958.725,592.577 1957.668,591.967 1957.609,591.121 C 1957.546,590.234 1958.35,589.602 1958.399,588.745 C 1959.616,590.884 1960.096,586.865 1960.419,586.801 C 1961.228,586.639 1962.908,586.48 1963.659,586.873 C 1964.586,585.38 1962.626,585.918 1962.469,584.497 C 1962.361,583.522 1961.066,584.325 1960.372,583.866 C 1959.427,583.242 1959.029,582.05 1957.93,581.614 C 1955.174,580.522 1956.64,577.294 1958.909,576.432 C 1958.704,575.741 1957.975,575.438 1957.318,575.424" id="path3340"/>
  </g>
  <g class="landxx coastxx do" id="do" transform="translate(-29.9017, -45.0745)">
    <path class="landxx do" d="M 760.898,648.576 C 761.22,647.914 761.575,646.647 761.152,645.954 C 760.787,645.357 760.223,644.145 759.745,643.68 C 760.713,643.732 762.945,642.176 761.853,641.099 C 760.8,640.06 763.912,639.437 762.646,638.035 C 762.232,637.577 762.771,636.532 762.901,636.063 C 763.149,635.171 762.338,634.228 762.768,633.385 C 763.765,631.431 765.765,633.295 766.972,632.812 C 767.966,632.415 768.963,632.189 769.957,632.72 C 770.946,633.248 771.938,633.324 772.882,633.796 C 773.75,634.229 774.21,634.658 775.228,634.681 C 776.264,634.705 776.42,634.504 776.539,635.678 C 776.753,637.781 777.675,638.034 779.577,637.544 C 780.264,637.367 781.163,637.121 781.779,637.633 C 781.488,638.594 780.635,638.6 779.804,638.461 C 778.982,638.324 777.169,638.911 778.736,639.355 C 779.394,639.542 782.05,640.561 782.644,640.286 C 784.362,639.489 785.412,641.671 786.485,642.649 C 788.612,644.588 785.248,645.531 784.839,646.323 C 784.137,647.683 783.857,646.359 783.25,645.751 C 782.67,645.17 782.022,645.547 781.306,645.398 C 780.804,645.224 780.288,645.108 779.76,645.049 C 779.138,645.049 778.559,645.592 777.956,645.216 C 775.396,643.619 774.724,646.605 772.768,646.962 C 772.297,647.048 770.744,647.357 770.33,647.022 C 769.754,646.556 770.39,645.522 769.54,645.122 C 769.176,645.937 768.235,645.842 767.688,646.447 C 767.23,646.954 766.556,646.154 766.329,646.824 C 765.584,649.028 764.639,650.501 763.129,652.322 C 762.831,651.652 761.99,651.34 761.834,650.593 C 761.681,649.864 761.618,649.008 760.898,648.576" id="path3372"/>
    <path d="M 785.018,648 C 784.501,647.938 784.021,647.771 783.578,647.496 C 784.061,647.379 784.86,647.358 785.018,648" id="path3374"/>
  </g>
  <g class="landxx coastxx gw" id="gw" transform="translate(-29.9017, -45.0745)">
    <path d="M 1176.908,705.312 C 1176.885,705.121 1176.861,704.928 1176.838,704.737 C 1176.568,704.891 1176.328,705.083 1176.118,705.312 C 1176.381,705.312 1176.645,705.312 1176.908,705.312" id="path3430"/>
    <path d="M 1177.348,702.216 C 1178.008,701.888 1177.911,701.648 1177.058,701.496 C 1177.154,701.736 1177.251,701.976 1177.348,702.216" id="path3432"/>
    <path d="M 1177.998,709.272 C 1178.262,709.065 1178.549,708.897 1178.858,708.767 C 1178.281,707.552 1176.819,708.361 1177.998,709.272" id="path3434"/>
    <path d="M 1179.218,701.784 C 1178.855,701.807 1178.495,701.855 1178.138,701.928 C 1178.204,702.982 1179.518,702.813 1179.218,701.784" id="path3436"/>
    <path d="M 1179.147,704.952 C 1179.05,705.192 1178.954,705.431 1178.857,705.672 C 1179.46,705.976 1179.841,705.623 1179.937,705.025 C 1179.673,705.011 1179.41,704.986 1179.147,704.952" id="path3438"/>
    <path d="M 1181.018,708.048 C 1181.417,707.728 1181.681,707.32 1181.808,706.824 C 1181.285,707.025 1180.935,707.466 1181.018,708.048" id="path3440"/>
    <path d="M 1184.547,708.192 C 1184.762,707.497 1184.522,707.305 1183.827,707.616 C 1184.067,707.809 1184.308,708 1184.547,708.192" id="path3442"/>
    <path class="landxx gw" d="M 1174.038,698.04 C 1174.735,697.556 1175.646,697.923 1176.395,697.586 C 1177.199,697.225 1177.946,696.768 1178.858,696.744 C 1180.786,696.692 1182.536,697.202 1184.148,695.808 C 1185.566,694.58 1187.351,694.944 1189.063,694.944 C 1191.662,694.944 1194.26,694.944 1196.859,694.944 C 1196.639,695.954 1197.761,697.126 1196.861,698.012 C 1196.372,698.493 1195.032,698.383 1194.838,699.119 C 1194.563,700.163 1198.283,700.974 1196.436,703.056 C 1195.484,704.128 1194.086,703.553 1192.888,703.756 C 1191.763,703.946 1190.421,704.837 1189.265,705.184 C 1187.277,705.78 1188.23,709.26 1186.199,709.992 C 1185.712,708.117 1185.448,707.871 1183.759,707.04 C 1183.91,706.766 1184.076,706.502 1184.259,706.248 C 1182.811,707.331 1184.19,702.897 1186.269,704.519 C 1185.768,703.108 1184.495,704.049 1183.759,704.664 C 1183.051,703.401 1182.926,702.551 1184.479,702 C 1185.802,701.531 1186.271,702.949 1187.499,702.864 C 1186.84,702.599 1186.452,701.758 1187.069,701.208 C 1186.588,701.044 1186.133,701.106 1185.704,701.392 C 1184.998,701.777 1184.409,701.303 1183.679,701.424 C 1182.97,701.542 1182.517,702.083 1181.989,702.504 C 1181.304,703.05 1180.379,702.965 1179.579,703.152 C 1179.629,701.845 1180.724,701.424 1181.599,700.704 C 1180.356,700.582 1179.371,701.876 1178.139,701.608 C 1175.738,701.086 1177.279,698.958 1178.43,698.183 C 1177.97,698.232 1177.513,698.304 1177.06,698.399 C 1176.821,698.688 1176.558,698.952 1176.27,699.192 C 1175.379,699.229 1174.883,698.177 1174.038,698.04" id="path3444"/>
  </g>
  <path class="landxx gh" d="M 1246.5153,705.45355 C 1246.0553,703.00055 1245.8913,701.03555 1246.9663,698.72155 C 1247.8903,696.73155 1247.3255,696.03558 1248.0665,694.04058 C 1248.6155,692.56358 1250.6004,690.51309 1251.1904,689.23009 C 1251.9924,687.48609 1251.358,684.89782 1250.829,683.10582 C 1250.618,682.39182 1250.3133,681.93555 1250.0223,681.26855 C 1249.7103,680.55355 1250.1728,679.80069 1250.2268,679.06769 C 1250.3508,677.38469 1249.5343,676.19955 1249.8803,674.57855 C 1250.2503,672.84555 1249.5343,670.98555 1249.2143,669.29955 C 1249.0633,668.49855 1248.2193,667.59855 1248.7113,666.75455 C 1249.0983,666.09055 1249.6413,665.14755 1249.3963,664.34255 C 1253.591,664.71774 1257.4521,664.07326 1261.9153,664.36836 C 1262.8159,664.42791 1265.48,664.1965 1265.6923,664.71655 C 1266.0861,665.68125 1267.6576,664.14497 1268.4763,663.55055 C 1269.309,662.94593 1271.6996,663.44983 1271.175,664.52101 C 1270.3751,666.15415 1270.1912,667.77488 1272.5369,669.53825 C 1274.478,670.99748 1273.3836,670.4008 1273.7138,671.91082 C 1273.8736,672.64189 1273.9654,672.5843 1273.8959,673.42355 C 1273.8256,674.2739 1273.8089,675.65004 1273.5863,676.72655 C 1272.5223,676.56255 1272.9723,677.32055 1272.5063,677.95155 C 1273.1523,678.06155 1273.6133,677.32855 1274.2163,677.49655 C 1275.0923,677.74155 1275.0811,679.00639 1274.8753,679.65055 C 1274.4794,680.89005 1275.3304,684.08593 1273.6563,683.78355 C 1274.0763,684.85055 1275.2053,685.50755 1275.8543,686.41955 C 1276.6313,687.51155 1275.9592,687.8425 1275.5092,688.8565 C 1274.9622,690.0885 1276.0099,691.70887 1275.316,692.86996 C 1274.4866,694.25796 1274.4043,694.94519 1275.5454,696.12819 C 1276.085,696.6875 1275.4847,698.08836 1275.2307,698.73036 C 1274.6167,700.28236 1274.4623,700.97618 1275.3473,702.52218 C 1275.9053,703.49918 1277.094,704.35955 1277.984,705.05055 C 1278.227,705.23955 1279.7493,706.13655 1279.7773,706.24655 C 1279.9913,707.06955 1278.6423,707.46755 1278.4303,708.16055 C 1278.1413,709.10355 1277.4903,709.90455 1276.3603,709.77655 C 1275.7293,709.70455 1275.5283,708.29255 1275.1853,707.87155 C 1274.4453,706.96455 1273.1953,707.31155 1272.2173,706.89355 C 1273.0033,707.52955 1274.2943,707.13755 1275.1353,707.75555 C 1276.7983,708.97655 1274.2533,709.75055 1273.4283,709.89555 C 1270.2893,710.44655 1268.3033,712.22855 1265.6983,713.78255 C 1264.2033,714.67355 1262.5443,714.65255 1260.9513,715.30355 C 1259.9873,715.69855 1258.8913,715.91255 1258.0653,716.57655 C 1257.3793,717.12855 1256.8123,717.86255 1256.0203,718.27655 C 1255.2033,718.70455 1254.5533,718.56055 1253.9453,717.92455 C 1253.1183,717.06155 1252.2163,716.94855 1251.1233,716.62155 C 1249.9793,716.27855 1249.0573,716.54828 1248.3483,715.38128 C 1249.2853,715.35428 1248.8193,715.21955 1249.7563,715.24655 C 1249.3393,714.03355 1250.7466,710.17641 1248.1856,710.88541 C 1249.0509,709.35241 1247.4327,706.72055 1246.5153,705.45355" id="gh"/>
  <path class="landxx at eu" d="M 1343.7863,351.57455 C 1344.4933,350.33655 1344.3483,350.68655 1344.0043,349.56855 C 1343.8853,349.18355 1349.4013,349.34655 1348.1063,351.57455 C 1349.6693,351.52855 1349.7113,350.06355 1350.0563,348.90855 C 1350.7213,349.50355 1352.2958,349.5101 1353.1298,349.4311 C 1352.4443,349.78018 1354.3232,350.37525 1354.5582,350.29764 C 1355.0477,350.53082 1355.7755,349.86042 1356.4563,349.62955 C 1357.8213,348.80555 1360.092,348.2051 1361.645,348.7121 C 1361.625,348.4431 1361.5958,348.16128 1361.5228,347.90228 C 1362.2978,347.99828 1363.5506,348.64207 1364.3177,348.27764 C 1365.9214,347.51573 1365.738,349.29636 1366.7233,349.66891 C 1367.5316,349.97455 1367.8591,348.01904 1367.1633,347.92355 C 1366.2664,347.80045 1366.8132,346.77228 1366.7088,346.30928 C 1366.5681,345.68562 1366.0143,345.50655 1365.7573,344.95155 C 1365.1013,343.53355 1368.7543,343.04655 1369.4843,342.55255 C 1370.1163,342.12555 1369.6278,340.27228 1370.8028,340.71728 C 1372.3858,341.31728 1371.4659,339.67529 1371.8389,338.24329 C 1372.967,338.54121 1374.2928,340.33835 1375.5729,340.5051 C 1377.9087,339.91902 1378.1152,340.47858 1378.3473,340.55855 C 1378.3513,339.86255 1378.7384,339.1809 1379.4283,338.95755 C 1380.3272,338.66655 1380.2198,338.03755 1380.2873,336.88655 C 1381.7729,336.75618 1383.5635,337.44765 1384.9638,337.91291 C 1385.4399,337.66432 1387.721,339.44765 1389.4172,339.04355 C 1390.7459,338.727 1393.3209,338.58897 1393.5197,340.16291 C 1393.5439,340.35464 1393.5509,341.10713 1393.7158,341.78216 C 1393.8971,342.5241 1393.7556,343.58702 1394.1441,344.13202 C 1394.6754,344.87747 1395.4171,345.00976 1395.3493,345.48733 C 1395.2815,345.9649 1394.4179,346.7778 1394.0603,347.71427 C 1393.3883,349.47427 1392.5628,347.23348 1390.9281,347.88964 C 1390.2162,348.17542 1393.5191,348.86129 1391.5602,349.85664 C 1390.0243,350.6371 1390.4283,352.69855 1390.8753,353.87755 C 1389.7749,353.64371 1386.9575,355.30298 1387.8553,356.75755 C 1386.2883,355.76655 1385.5563,356.90155 1384.0743,357.03655 C 1383.0073,357.13555 1382.2622,356.62843 1381.2598,356.83219 C 1380.4962,356.98743 1378.8494,358.58033 1378.2464,358.89433 C 1377.1164,359.48333 1374.2059,358.14096 1373.0803,357.97814 C 1370.1622,357.55602 1367.6085,357.91451 1364.6185,356.97851 C 1363.7095,356.69351 1360.2683,354.27155 1362.0063,353.22855 C 1361.0133,353.02155 1360.2053,353.74255 1359.2663,353.87655 C 1357.8923,354.07355 1356.7243,353.88955 1355.3473,354.27855 C 1354.5233,354.51055 1353.8673,355.99455 1353.0153,355.74655 C 1352.1553,355.49555 1351.3743,355.29955 1350.5073,355.08455 C 1349.7923,354.90655 1350.1736,354.0761 1349.7106,354.0001 C 1349.1526,353.9081 1348.7829,354.43553 1348.4539,354.84346 C 1347.8514,355.59053 1346.1183,354.52935 1345.6332,354.00909 C 1345.1074,353.44523 1344.3341,353.36119 1344.0381,353.35573 C 1344.3271,352.69673 1344.1603,352.09855 1343.7863,351.57455" id="at"/>
  <g class="landxx coastxx se eu" id="se" transform="translate(-29.9017, -45.0745)">
    <path d="M 1387.368,306 C 1386.888,306.24 1386.409,306.48 1385.928,306.72 C 1386.338,307.282 1387.969,307.55 1388.291,306.706 C 1388.491,306.181 1387.836,305.31 1387.368,306" id="path3388"/>
    <path d="M 1387.008,307.584 C 1386.864,307.824 1386.721,308.063 1386.578,308.304 C 1387.184,308.391 1387.703,308.059 1387.948,307.512 C 1387.634,307.536 1387.321,307.56 1387.008,307.584" id="path3390"/>
    <path d="M 1417.248,320.616 C 1417.367,321.064 1417.414,321.52 1417.388,321.983 C 1418.598,321.735 1419.108,319.452 1419.38,318.482 C 1419.895,316.648 1420.789,315.029 1421.498,313.272 C 1420.306,313.238 1419.973,315.429 1419.498,316.188 C 1418.637,317.562 1416.853,318.714 1417.248,320.616" id="path3392"/>
    <path d="M 1427.328,299.808 C 1427.088,299.928 1426.848,300.048 1426.608,300.168 C 1426.883,300.115 1427.123,299.995 1427.328,299.808" id="path3394"/>
    <path d="M 1429.348,316.008 C 1429.048,315.19 1431.826,312.923 1432.438,312.48 C 1429.698,312.217 1435.726,307.944 1431.282,308.869 C 1430.026,309.13 1428.865,310.182 1428.044,311.124 C 1427.31,311.968 1427.626,315.077 1428.908,315.289 C 1428.497,315.742 1428.164,316.246 1427.908,316.801 C 1428.544,316.932 1429.063,316.544 1429.348,316.008" id="path3396"/>
    <path d="M 1427.978,288.937 C 1428.286,289.28 1428.669,289.424 1429.128,289.369 C 1428.804,288.841 1428.326,288.401 1427.758,288.146 C 1427.831,288.409 1427.904,288.673 1427.978,288.937" id="path3398"/>
    <path d="M 1428.908,299.231 C 1428.741,299.471 1428.575,299.711 1428.408,299.952 C 1428.896,299.777 1428.924,299.672 1428.908,299.231" id="path3400"/>
    <path d="M 1429.638,296.28 C 1429.875,296.135 1430.111,295.992 1430.348,295.847 C 1429.762,295.614 1429.374,295.868 1429.348,296.495 C 1429.444,296.424 1429.541,296.353 1429.638,296.28" id="path3402"/>
    <path d="M 1433.228,308.808 C 1433.772,309.02 1434.255,308.924 1434.678,308.521 C 1434.153,307.882 1433.507,308.104 1433.228,308.808" id="path3404"/>
    <path d="M 1448.208,249.12 C 1447.968,249.096 1447.728,249.072 1447.488,249.048 C 1447.947,249.429 1447.854,249.283 1448.208,249.12" id="path3408"/>
    <path class="landxx se eu" d="M 1384.488,299.231 C 1385.551,299.046 1385.875,301.037 1386.635,300.832 C 1387.841,300.506 1388.161,297.751 1387.754,296.86 C 1387.029,295.272 1388.7105,294.91599 1388.468,293.564 C 1388.12,291.62392 1391.6099,292.90489 1392.066,290.89 C 1392.4014,289.40831 1392.8767,288.38106 1391.9007,287.04106 C 1391.3617,286.30106 1390.0061,284.45069 1391.5041,283.81469 C 1392.9981,283.18069 1393.4687,283.84687 1394.0117,281.80087 C 1394.3777,280.41787 1392.787,280.165 1391.863,279.82 C 1390.155,279.179 1389.6967,277.896 1390.1297,276.317 C 1390.6477,274.426 1390.3579,273.43393 1389.711,271.692 C 1389.4566,271.007 1389.0541,269.93344 1389.4399,269.24344 C 1389.7529,268.68377 1390.151,268.793 1389.96,268.235 C 1389.724,267.547 1389.187,267.14546 1389.07,266.381 C 1388.9286,265.4571 1389.7922,264.98191 1389.7558,264.275 C 1389.6888,262.97354 1391.8671,261.84068 1392.49,261.336 C 1394.3289,259.8461 1396.0553,260.05529 1397.966,260.329 C 1399.2397,260.51147 1401.4916,260.57272 1401.147,258.805 C 1400.8049,257.05026 1398.9366,256.87293 1398.168,256.319 C 1398.861,255.468 1399.644,254.697 1400.42,253.923 C 1401.13,253.215 1402.049,251.398 1402.861,250.959 C 1403.885,250.405 1403.206,245.364 1402.988,244.583 C 1404.562,244.643 1406.6455,244.52163 1408.018,243.653 C 1408.6745,243.23755 1408.0449,242.62241 1407.9659,242.15241 C 1407.8795,241.63831 1408.868,241.451 1409.15,241.151 C 1410.425,239.791 1411.7197,238.87919 1413.1427,237.71519 C 1414.3087,236.76219 1412.072,235.382 1411.557,234.935 C 1413.87,233.875 1414.518,231.66 1416.747,230.615 C 1418.176,229.945 1419.067,230.12887 1420.53,230.45787 C 1421.269,230.62387 1422.908,230.097 1422.811,229.118 C 1422.746,228.465 1422.1629,227.5311 1422.441,226.873 C 1422.792,226.04227 1425.4144,227.0585 1426.0065,226.94353 C 1427.51,226.65156 1431.9364,228.66548 1434.168,227.158 C 1433.682,226.939 1433.178,226.771 1432.658,226.654 C 1433.89,225.898 1435.594,224.215 1433.448,223.341 C 1436.715,221.876 1440.364,225.127 1443.168,226.365 C 1445.083,227.211 1447.4683,227.23076 1449.362,228.08127 C 1449.981,228.35927 1453.301,230.36056 1453.656,231.085 C 1453.9611,231.70759 1453.0611,231.68644 1452.9205,232.057 C 1452.8673,232.19715 1452.9822,232.3807 1453.0553,234.1019 C 1453.0735,234.53109 1453.8911,234.63595 1454.2661,234.73695 C 1455.7161,235.12895 1454.6767,235.78232 1454.1737,236.41632 C 1453.6207,237.11332 1455.3511,238.20047 1455.7741,238.53747 C 1456.7191,239.29247 1456.443,240.775 1455.647,241.657 C 1454.015,243.461 1458.0267,245.25294 1458.5407,246.98294 C 1456.9247,247.20294 1454.131,248.072 1452.748,246.811 C 1452.976,247.038 1453.19,247.278 1453.388,247.532 C 1452.229,247.532 1451.284,247.282 1450.298,246.668 C 1450.106,248.639 1448.321,245.768 1447.923,247.163 C 1447.734,247.826 1448.535,248.498 1448.638,249.116 C 1447.906,249.099 1447.216,248.834 1446.558,248.54 C 1446.814,248.697 1447.078,248.841 1447.348,248.972 C 1446.832,249.33 1446.279,249.402 1445.688,249.189 C 1448.019,249.877 1443.91,250.21 1443.458,250.269 C 1443.939,251.082 1445.116,251.401 1444.108,252.716 C 1443.76,253.171 1441.826,254.523 1441.228,254.228 C 1441.633,254.687 1442.137,254.975 1442.738,255.092 C 1440.693,255.523 1444.427,256.811 1444.828,257.468 C 1443.418,258.147 1441.442,258.933 1440.877,260.547 C 1440.552,261.474 1437.997,262.55 1438.198,263.372 C 1438.045,263.099 1437.879,262.835 1437.698,262.58 C 1437.325,263.515 1436.041,263.825 1435.165,263.983 C 1434.24,264.149 1433.954,266.029 1433.087,264.379 C 1433.324,265.593 1433.052,264.938 1432.222,264.967 C 1431.752,264.984 1431.363,266.069 1430.942,266.361 C 1429.422,267.416 1427.479,267.283 1425.887,268.484 C 1426.388,268.548 1426.891,268.573 1427.397,268.555 C 1427.424,269.359 1426.76,269.891 1426.004,269.513 C 1425.591,269.306 1425.356,270.44 1424.597,270.068 C 1426.431,269.879 1424.18,269.706 1423.877,269.78 C 1424.047,268.965 1423.009,268.61 1422.937,267.836 C 1422.68,268.991 1424.008,269.657 1424.178,270.711 C 1424.441,272.338 1422.22,273.451 1421.137,272.156 C 1420.577,273.444 1421.306,274.371 1422.647,274.46 C 1421.772,274.578 1421.305,275.982 1421.137,276.692 C 1420.88,277.781 1422.292,278.339 1421.927,279.284 C 1421.351,278.7 1420.543,278.264 1419.697,278.492 C 1422.43,279.109 1418.153,278.891 1420.487,279.356 C 1418.475,279.168 1419.864,279.594 1420.417,279.788 C 1418.319,279.409 1421.645,280.663 1419.627,280.868 C 1420.662,281.226 1420.078,282.612 1420.145,283.377 C 1420.245,284.504 1420.953,285.131 1421.497,286.052 C 1421.238,286.181 1420.974,286.301 1420.707,286.412 C 1421.079,286.631 1422.818,286.736 1422.96,287.049 C 1423.572,288.397 1424.474,287.4 1425.457,287.204 C 1424.913,289.185 1428.351,288.903 1429.127,290.012 C 1427.307,290.375 1429.768,291.049 1430.352,291.164 C 1430.771,291.247 1432.984,294.153 1431.217,292.748 C 1431.716,293.166 1432.136,293.683 1432.437,294.26 C 1431.33,294.296 1430.501,294.907 1429.654,295.545 C 1428.902,296.111 1426.716,296.44 1428.117,296.924 C 1427.571,297.07 1427.091,296.949 1426.677,296.564 C 1426.557,297.449 1429.309,297.976 1428.987,296.636 C 1429.489,296.912 1430.521,298.59 1429.057,297.788 C 1430.957,298.442 1428.266,297.758 1427.907,297.645 C 1428.024,298.237 1428.465,298.678 1429.057,298.795 C 1428.459,299.113 1427.784,299.16 1427.157,299.391 C 1426.285,299.712 1426.187,300.581 1425.478,301.023 C 1425.003,301.319 1424.659,299.403 1423.877,299.444 C 1424.343,299.815 1424.533,300.295 1424.447,300.883 C 1424.207,300.715 1423.967,300.547 1423.727,300.379 C 1423.953,301.045 1423.953,301.531 1423.007,301.1 C 1423.135,301.336 1423.255,301.576 1423.367,301.82 C 1422.43,302.341 1421.444,302.282 1420.417,302.179 C 1420.666,302.36 1420.906,302.552 1421.137,302.755 C 1419.536,303.605 1417.519,302.077 1416.027,303.188 C 1417.235,302.887 1418.976,303.022 1419.627,304.268 C 1418.663,304.799 1417.679,304.322 1416.677,304.268 C 1418.199,304.798 1420.834,307.239 1418.187,308.227 C 1418.831,308.078 1419.063,308.415 1418.907,309.02 C 1418.406,308.743 1417.876,308.67 1417.317,308.803 C 1417.66,309.244 1419.459,310.251 1418.224,310.745 C 1417.128,311.183 1419.171,311.831 1418.543,312.657 C 1417.102,314.549 1418.441,316.521 1416.879,318.494 C 1415.873,319.766 1415.354,323.448 1413.044,322.756 C 1411.662,322.342 1410.216,322.99 1408.816,322.9 C 1408.044,322.851 1406.035,322.68 1407.097,323.996 C 1405.743,324.526 1404.556,323.704 1403.765,325.459 C 1403.313,326.462 1404.403,327.239 1404.499,328.198 C 1404.681,330.001 1402.196,328.808 1401.472,328.873 C 1399.172,329.08 1397.301,329.159 1394.997,329.036 C 1395.741,329.007 1396.26,326.719 1395.566,325.986 C 1394.8,325.177 1392.907,322.984 1392.987,321.836 C 1395.449,322.698 1394.024,321.839 1393.847,320.468 C 1397.204,319.914 1393.921,318.802 1393.102,317.658 C 1392.313,316.556 1391.183,315.657 1390.684,314.369 C 1390.364,313.544 1390.002,312.462 1388.877,312.835 C 1388.933,312.284 1388.981,311.712 1388.86,311.165 C 1388.65,310.22 1387.094,310.091 1388.517,309.595 C 1387.141,310.087 1388.525,305.804 1388.597,305.276 C 1387.677,305.083 1387.207,306.114 1386.287,305.996 C 1386.695,305.511 1387.078,305.007 1387.437,304.484 C 1386.101,304.348 1387.167,305.249 1385.927,305.635 C 1385.8,304.959 1385.316,304.807 1384.777,305.203 C 1385.351,303.849 1383.261,300.076 1384.488,299.231" id="path3452"/>
    <path d="M 1420.988,272.592 C 1421.164,272.967 1421.404,272.823 1421.138,273.384 C 1421.088,273.12 1421.038,272.855 1420.988,272.592" id="path3454"/>
  </g>
  <g class="landxx tr" id="tr" transform="translate(-29.9017, -45.0745)">
    <path d="M 1489.389,452.376 C 1490.137,452.02 1491.167,450.999 1491.392,450.188 C 1491.742,448.924 1490.771,447.975 1492.524,447.184 C 1493.482,446.751 1493.133,446.745 1492.892,445.738 C 1492.75,445.147 1492.514,444.556 1491.878,444.348 C 1489.581,443.597 1492.414,443.235 1492.284,442.315 C 1492.248,442.059 1494.743,441.348 1495.148,441.216 C 1496.472,440.785 1497.25,440.341 1498.211,441.431 C 1499.108,442.448 1499.611,441.796 1500.748,441.683 C 1501.278,441.63 1501.901,441.686 1502.331,442.042 C 1503.063,442.648 1501.965,442.858 1502.482,443.523 C 1504.409,446.004 1506.66,447.082 1509.72,448 C 1510.901,448.354 1509.919,450.077 1509.188,450.348 C 1507.788,450.866 1505.756,449.51 1504.332,449.427 C 1503.627,449.386 1503.196,449.877 1502.614,450.175 C 1501.899,450.541 1501.151,449.906 1500.33,450.152 C 1499.337,450.45 1498.928,451.52 1498.244,452.265 C 1497.133,453.475 1495.636,453.817 1494.356,454.751 C 1493.27,455.543 1492.223,457.753 1491.046,458.208 C 1490.691,456.806 1491.284,455.348 1492.665,454.857 C 1492.913,454.769 1495.561,453.968 1494.688,453.25 C 1493.273,452.083 1490.078,455.21 1489.389,452.376" id="path3456"/>
    <path d="M 1490.328,463.248 C 1490.428,462.503 1490.52,458.816 1491.071,458.605 C 1492.437,458.084 1492.875,456.104 1494.288,455.399 C 1495.041,455.024 1496.006,455.686 1496.704,455.197 C 1497.579,454.585 1497.957,455.051 1498.725,455.511 C 1499.826,456.172 1503.387,455.891 1501.128,454.535 C 1501.845,454.243 1502.622,454.281 1503.358,454.463 C 1502.993,454.595 1502.814,454.829 1502.822,455.164 C 1503.244,455.754 1505.385,455.406 1505.93,455.356 C 1507.607,455.203 1510.024,456.181 1511.427,454.967 C 1510.757,454.312 1509.893,454.886 1509.188,454.32 C 1510.092,452.016 1514.236,452.884 1516.108,452.448 C 1514.492,451.653 1512.185,452.321 1510.83,450.931 C 1509.309,449.371 1510.84,448.077 1512.471,448.231 C 1514.261,448.4 1516.047,449.233 1517.868,448.836 C 1520.298,448.307 1522.271,449.632 1524.748,449.387 C 1525.892,449.274 1526.061,448.687 1526.485,447.793 C 1526.953,446.808 1528.7,446.377 1529.589,445.856 C 1532.155,444.352 1534.198,443.03 1537.1,442.245 C 1540.126,441.426 1542.446,441.863 1545.518,441.863 C 1546.757,441.863 1548.071,442.373 1549.298,442.007 C 1549.865,441.837 1550.053,441.16 1550.598,440.999 C 1551.154,440.836 1551.636,441.166 1552.178,441.215 C 1551.26,442.322 1552.833,443.827 1553.834,444.132 C 1554.594,444.363 1555.379,444.554 1556.179,444.449 C 1557.054,444.335 1557.798,443.35 1557.798,444.599 C 1557.999,442.743 1559.192,445.001 1559.366,445.532 C 1559.596,446.234 1561.513,448.609 1562.265,447.802 C 1563.257,446.736 1564.319,446.971 1565.386,447.825 C 1566.686,448.866 1568.285,449.054 1569.739,449.781 C 1570.557,450.193 1570.698,448.502 1571.822,449.736 C 1572.725,450.728 1574.48,450.704 1575.718,450.828 C 1577.439,451.001 1579.201,449.811 1580.928,449.641 C 1582.404,449.495 1583.214,449.301 1584.626,449.807 C 1586.605,450.516 1588.074,451 1590.111,450.046 C 1592.555,448.901 1594.792,447.596 1596.88,445.915 C 1597.716,445.242 1598.492,446.381 1599.337,446.399 C 1599.854,446.411 1599.812,445.702 1600.417,445.823 C 1600.81,445.901 1601.2,445.99 1601.592,446.07 C 1602.16,446.187 1603.778,446.825 1604.124,446.059 C 1604.548,445.121 1606.59,444.339 1605.957,445.751 C 1606.602,445.63 1607.161,446.043 1607.62,446.427 C 1607.95,446.727 1608.295,447.01 1608.654,447.274 C 1609.233,447.346 1609.366,447.614 1609.05,448.08 C 1609.317,448.886 1610.408,448.101 1610.852,448.76 C 1611.141,449.189 1611.3277,450.46127 1611.755,450.80664 C 1613.1181,451.90827 1614.232,453.008 1613.451,454.133 C 1612.68,455.244 1614.76,456.894 1613.666,457.631 C 1615.42,458.516 1616.317,458.314 1618.153,458.382 C 1619.722,458.44 1621.924,460.524 1622.515,461.879 C 1621.661,461.175 1620.36,460.154 1620,461.879 C 1619.889,462.411 1620.027,462.97 1619.864,463.5 C 1619.652,464.186 1617.836,463.823 1617.264,464.039 C 1617.685,464.537 1618.528,465.821 1618.691,466.446 C 1618.962,467.489 1619.423,468.032 1619.793,468.998 C 1620.336,470.419 1619.841,472.496 1621.655,473.111 C 1621.04,473.692 1620.68,474.857 1620.487,475.653 C 1620.099,477.251 1621.754,477.017 1622.594,477.791 C 1623.44,478.568 1623.271,479.593 1623.539,480.609 C 1623.674,481.117 1624.573,481.342 1624.925,481.655 C 1625.315,482.001 1624.971,482.822 1625.185,483.262 C 1623.508,482.079 1622.815,485.019 1621.853,484.701 C 1620.608,484.289 1621.627,482.649 1620.714,482.038 C 1620.258,481.733 1619.501,481.761 1619.088,482.134 C 1618.343,482.806 1617.248,482.649 1616.369,482.373 C 1614.936,481.923 1613.832,481.45 1612.317,481.472 C 1611.242,481.488 1610.569,481.189 1610.006,482.134 C 1609.325,483.278 1607.964,484.306 1607.498,482.6 C 1607.093,481.113 1605.216,482.902 1604.394,483.168 C 1602.283,483.852 1600.513,483.832 1598.37,483.549 C 1594.521,483.041 1591.196,486.915 1587.3,487.186 C 1585.745,487.294 1582.938,487.678 1581.632,486.69 C 1580.207,485.612 1579.315,484.993 1577.556,485.847 C 1575.677,486.759 1571.589,488.91 1569.972,486.906 C 1569.161,485.901 1567.438,485.552 1567.191,487.134 C 1566.948,488.69 1567.491,489.779 1568.155,491.111 C 1567.449,491.151 1566.7,491.352 1565.995,491.182 C 1565.88,491.798 1566.292,492.449 1566.075,493.054 C 1565.819,493.772 1564.866,493.923 1564.636,494.71 C 1564.095,494.347 1563.55,493.937 1562.906,493.774 C 1563.525,492.967 1562.672,491.918 1562.184,491.314 C 1561.275,490.188 1562.259,489.525 1563.197,488.71 C 1564.161,487.873 1564.786,487.283 1563.847,486.016 C 1563.472,485.51 1562.686,484.885 1562.151,485.578 C 1561.41,486.536 1560.661,486.385 1559.736,487.078 C 1561.269,486.922 1559.696,488.465 1558.559,488.257 C 1557.148,487.999 1555.892,487.169 1554.629,486.536 C 1551.622,485.029 1550.06,489.451 1548.646,491.254 C 1548.431,491.013 1548.214,490.773 1547.996,490.534 C 1547.454,491.103 1547.097,492.115 1546.217,491.782 C 1545.453,491.493 1543.293,492.024 1542.573,492.384 C 1541.265,493.038 1539.732,492.94 1538.456,492.377 C 1536.498,491.512 1535.795,489.341 1533.931,488.429 C 1531.883,487.427 1529.814,486.282 1527.558,485.881 C 1526.371,485.671 1523.947,485.191 1523.947,487.078 C 1523.947,488.161 1524.148,490.13 1523.356,490.995 C 1522.924,491.466 1522.351,490.593 1521.943,490.537 C 1520.899,490.394 1519.822,491.63 1518.79,491.803 C 1517.6,492.003 1515.789,491.235 1514.808,490.606 C 1514.062,490.128 1514.028,488.565 1513.007,488.517 C 1513.801,487.397 1511.951,486.181 1511.717,487.654 C 1510.944,486.98 1508.995,486.727 1508.757,485.565 C 1508.575,486.405 1506.548,487.789 1505.877,488.301 C 1506.114,487.798 1505.947,487.487 1505.377,487.365 C 1505.944,487.243 1506.305,486.791 1506.308,486.213 C 1505.576,486.826 1502.141,487.56 1501.198,487.293 C 1501.947,486.377 1503.344,486.322 1504.423,486.35 C 1505.514,486.378 1505.399,486.667 1505.844,485.823 C 1506.155,485.236 1506.827,484.97 1507.248,484.485 C 1505.92,483.892 1501.358,483.815 1500.268,484.917 C 1498.622,482.984 1501.752,484.005 1501.987,483.506 C 1502.182,483.091 1502.157,482.511 1501.923,482.117 C 1501.372,481.194 1500.76,482.475 1501.06,481.029 C 1499.473,483.058 1499.589,478.876 1498.11,478.941 C 1499.136,478.384 1500.305,477.847 1499.426,476.455 C 1498.792,475.451 1496.984,475.971 1496.548,475.225 C 1496.097,474.453 1495.639,473.372 1494.963,474.723 C 1494.663,475.322 1492.529,473.711 1492.489,473.181 C 1494.635,473.695 1492.778,471.422 1492.856,470.536 C 1492.919,469.803 1494.075,470.338 1494.294,470.698 C 1494.749,471.444 1494.595,472.667 1495.299,473.253 C 1495.144,471.657 1497.629,473.654 1498.32,472.029 C 1496.888,471.852 1496.305,471.157 1495.44,470.085 C 1496.188,469.671 1496.574,468.764 1497.46,468.573 C 1497.067,467.405 1495.857,468.617 1495.969,467.295 C 1496.08,465.981 1495.385,465.812 1494.43,464.973 C 1495.067,464.24 1495.801,463.596 1496.38,462.813 C 1494.328,461.768 1492.391,463.923 1490.328,463.248" id="path3458"/>
  </g>
  <path class="landxx ug" d="M 1496.4963,769.60855 C 1496.9863,768.25455 1496.8293,766.43155 1497.0303,764.99555 C 1497.2833,763.18855 1497.4519,761.39647 1497.6873,759.57855 C 1497.974,757.36445 1499.255,756.09853 1499.4713,754.94455 C 1499.6031,754.24149 1499.3303,753.52655 1499.4663,752.81655 C 1499.6803,751.69955 1500.7353,751.98655 1501.2453,751.17855 C 1502.7983,748.72055 1505.2839,746.9696 1507.3859,744.7966 C 1508.2009,743.9526 1509.2973,742.48555 1509.6403,741.34755 C 1509.9223,740.41155 1508.1823,739.43055 1507.5083,739.07055 C 1506.9683,738.78155 1506.1543,739.66055 1505.7533,738.77355 C 1505.4363,738.07055 1505.6933,737.10755 1505.8393,736.39555 C 1506.2843,734.21355 1505.4673,732.45055 1506.5713,730.44755 C 1506.8943,729.86355 1506.1004,729.70928 1506.2994,729.09328 C 1506.5094,728.44128 1507.1043,727.89355 1507.6103,727.47555 C 1508.1083,727.06555 1508.6973,726.69155 1509.3783,726.81455 C 1509.9353,726.91555 1510.9823,728.07355 1511.5323,727.82455 C 1512.2363,727.50655 1513.2683,726.20955 1514.0183,727.19855 C 1514.8223,728.25655 1515.0393,728.63555 1516.4443,729.14155 C 1515.6561,727.83235 1517.8313,727.33255 1518.8523,727.18955 C 1520.1173,727.01238 1521.5606,725.99596 1522.7986,726.13996 C 1524.4936,726.33696 1525.0163,727.83355 1526.8663,726.55055 C 1528.1933,725.63155 1529.1153,724.25055 1530.1253,723.02055 C 1530.7143,723.81555 1531.3413,724.94755 1530.9853,725.97255 C 1532.1783,725.85155 1531.3862,726.72122 1532.512,727.05982 C 1533.6737,727.40919 1534.1176,727.6335 1533.6323,729.12255 C 1533.0006,731.06063 1534.9372,733.14369 1535.9453,735.44155 C 1536.143,735.89226 1537.3073,737.40855 1537.2853,737.80055 C 1537.2373,738.62955 1538.2164,739.8596 1538.4184,740.6666 C 1538.7734,742.0996 1538.4197,744.05233 1537.7987,745.48333 C 1537.2697,746.70433 1536.2533,746.88355 1536.5123,748.30455 C 1536.6463,749.04155 1535.4533,749.50355 1534.9083,749.77755 C 1533.8403,750.31655 1533.7953,751.75455 1533.1563,752.81855 C 1532.6783,753.61355 1531.8856,754.04228 1531.5546,754.86428 C 1531.1056,755.97728 1530.6939,756.79323 1530.1779,757.87323 C 1529.681,757.83998 1527.8496,758.44886 1528.1467,756.85068 C 1526.9842,757.29046 1527.3493,757.62287 1526.8708,757.6967 C 1525.4527,757.9155 1525.4326,757.0628 1526.496,757.1296 C 1526.223,756.55872 1525.2691,755.84572 1525.5912,755.43728 C 1524.7991,755.72725 1525.3224,755.51943 1524.2362,755.92533 L 1524.5807,756.5 C 1523.6333,756.75313 1524.3601,757.67973 1523.1661,757.6167 C 1523.3193,759.4781 1520.5187,758.51931 1520.1988,758.58367 C 1520.1421,758.61654 1520.3487,756.91304 1520.1403,757.19671 C 1518.7857,760.35232 1518.2917,758.67669 1518.0373,758.84269 C 1518.0868,759.51258 1517.5821,759.55546 1516.4156,759.67061 C 1515.8667,760.44944 1514.5708,760.43028 1514.6781,760.7047 C 1515.6991,763.31605 1513.5521,763.74792 1513.2157,765.58201 C 1512.9248,767.16845 1513.5053,767.25337 1513.6858,768.30028 C 1506.4968,768.30028 1515.2465,768.61237 1508.0773,768.07655 C 1505.602,767.89155 1502.693,768.39827 1501.2443,770.19209 C 1500.741,770.81525 1500.0355,772.56249 1499.1559,771.98487 C 1498.9229,771.83184 1498.7871,770.99141 1498.3783,771.02355 C 1497.6783,771.07858 1496.5659,771.85331 1496.2353,772.08824 C 1496.4002,771.436 1497.3175,770.30524 1496.4963,769.60855" id="ug"/>
  <g class="landxx coastxx mz" id="mz" transform="translate(-28.7126, 62.0078)">
    <path d="M 1566.1089,886.0907 C 1566.3639,885.3737 1566.5079,884.6137 1566.4689,883.8507 C 1566.1569,884.5367 1565.8149,885.3267 1566.1089,886.0907" id="path3382"/>
    <path d="M 1573.6689,860.9607 C 1574.2729,860.3867 1574.1299,860.1467 1573.2379,860.2407 C 1573.3819,860.4807 1573.5259,860.7207 1573.6689,860.9607" id="path3384"/>
    <path class="landxx mz" d="M 1528.5989,827.3307 C 1534.1359,825.2497 1539.6929,823.2167 1545.2629,821.2267 C 1546.5079,820.7817 1547.6806,820.18916 1548.9146,819.71216 C 1549.3326,819.55116 1550.3912,818.86415 1550.8562,819.01015 L 1555.0189,823.6617 C 1555.0599,822.7287 1557.0629,823.0547 1558.0049,822.7887 C 1558.8149,822.5607 1560.0349,821.8017 1560.6999,822.6637 C 1561.9289,824.2577 1561.9359,828.1197 1561.2969,829.8867 C 1560.4859,832.1257 1558.5999,834.1387 1559.7969,836.4427 C 1560.0819,836.9897 1559.9179,837.6417 1560.2489,838.1617 C 1560.6229,838.7497 1561.2869,839.0697 1561.6589,839.6567 C 1562.3879,840.8057 1563.5109,841.8807 1564.5449,842.7637 C 1565.5309,843.6057 1565.4779,843.7597 1564.8019,844.8657 C 1564.0799,846.0467 1566.4199,846.4917 1566.5739,844.9017 C 1566.7219,843.3557 1566.2759,841.9887 1565.3879,840.7327 C 1566.7939,839.7087 1566.0629,837.5857 1568.1289,837.1567 C 1570.9289,836.5757 1571.1769,835.3537 1571.1859,832.5307 C 1571.1889,831.5037 1571.0079,830.4557 1571.0689,829.4307 C 1571.1689,827.7607 1572.0459,825.9167 1571.3289,824.2787 C 1570.6109,822.6377 1569.1119,821.0917 1568.1479,819.5637 C 1567.2519,818.1427 1566.3759,816.6137 1565.2409,815.3627 C 1564.4069,814.4447 1564.7212,815.28164 1563.9712,814.44164 C 1563.7487,813.723 1564.1309,812.3132 1563.6725,810.91987 C 1563.0719,809.09456 1564.1448,807.9911 1563.9059,807.18212 C 1562.5206,802.50467 1563.358,803.64973 1563.292,803.06147 C 1563.8334,801.83029 1565.6378,802.06121 1565.1994,797.6999 C 1566.6998,797.84069 1568.0625,798.17195 1569.8179,797.6637 C 1570.8105,797.37631 1571.7449,796.1537 1572.8309,796.8567 C 1573.5819,797.3427 1573.9459,798.2717 1574.7439,798.6967 C 1575.3549,799.0227 1577.3839,799.1347 1578.0569,798.9617 C 1579.1759,798.6737 1579.9819,796.7937 1581.2449,798.1647 C 1583.5889,800.7087 1586.4039,797.6957 1587.5389,795.8267 C 1588.1649,794.7947 1589.2809,794.7667 1590.2579,795.3857 C 1591.8499,796.3947 1592.2349,796.3857 1593.7449,795.3257 C 1595.0069,794.4407 1595.9969,793.8177 1597.6419,794.1407 C 1598.5879,794.3277 1600.4149,792.5127 1601.2979,792.1107 C 1603.0399,791.3187 1604.4949,790.2047 1605.9969,789.0457 C 1607.8309,787.6287 1609.0259,789.9997 1607.4349,790.7607 C 1607.6779,790.8747 1607.9169,790.9947 1608.1549,791.1207 C 1607.4269,792.0587 1607.8399,793.1147 1607.4349,794.1407 C 1607.2179,794.6927 1605.8269,795.4077 1606.6469,795.9907 C 1607.3269,796.4727 1607.1679,799.0517 1607.2649,800.0497 C 1607.3369,800.7977 1607.3959,801.5597 1607.3289,802.3107 C 1607.2289,803.4497 1606.8739,803.1637 1607.1919,804.1727 C 1607.2649,804.5487 1607.2389,804.9187 1607.1129,805.2797 C 1606.9649,805.9777 1607.4039,806.1897 1607.5839,806.7937 C 1607.8479,807.6717 1607.5159,808.9317 1606.9249,809.6207 C 1608.9289,809.9057 1606.3209,814.3047 1607.1059,814.5557 C 1608.4769,814.9947 1607.1379,819.5717 1607.5749,820.6407 C 1609.0849,819.7457 1607.3559,822.6647 1607.7959,823.3707 C 1607.9169,822.0977 1609.1999,822.3167 1608.9799,823.5787 C 1608.7419,824.9467 1609.0699,825.9167 1607.6459,826.6907 C 1607.9049,826.8927 1608.1709,827.0827 1608.4469,827.2607 C 1606.7989,826.8677 1608.2199,827.4687 1607.7469,828.0357 C 1607.4479,828.3937 1606.9829,828.5287 1606.7869,828.9907 C 1608.3779,828.4247 1607.0039,830.9487 1606.7669,831.4477 C 1605.9699,833.1327 1604.2729,834.0667 1603.4539,835.7407 C 1602.6409,837.4027 1601.3129,838.3687 1600.4089,839.9437 C 1599.7169,841.1517 1595.0249,842.7827 1595.1939,844.0967 C 1595.2719,844.7047 1591.6509,845.3267 1590.9459,845.7607 C 1590.3219,846.1447 1589.6149,846.4517 1588.9129,846.6567 C 1587.7799,846.9877 1587.9749,846.5547 1587.1519,847.3157 C 1585.9679,848.4107 1584.3089,848.9287 1582.9009,849.6607 C 1581.4999,850.3897 1579.2679,853.7977 1577.6959,852.4607 C 1580.6319,855.6817 1573.5019,856.9187 1574.1659,860.1707 C 1573.5599,860.2537 1573.1529,859.8577 1573.0159,859.3007 C 1573.1579,860.4437 1572.0719,861.8927 1572.2259,860.3107 C 1570.8549,861.4997 1569.1959,862.3157 1568.0809,863.7837 C 1567.0939,865.0827 1566.1959,866.3157 1565.0079,867.4497 C 1564.5269,867.9087 1562.6149,869.9257 1561.9279,869.0907 C 1561.4849,868.5527 1560.9189,866.7077 1559.9159,867.2917 C 1561.6449,867.1877 1561.0409,869.3637 1561.3819,869.8917 C 1561.7019,870.3887 1561.6849,871.1957 1561.4449,871.7177 C 1561.3359,871.9927 1561.1359,872.1517 1560.8459,872.1917 C 1561.0679,872.6057 1561.1419,873.0457 1561.0679,873.5117 C 1560.8799,875.7847 1561.6269,875.1567 1562.7969,876.6527 C 1563.4529,877.4917 1563.6419,878.4847 1563.6279,879.5247 C 1563.6219,879.9277 1563.3009,883.0627 1563.8769,881.1227 C 1563.4389,882.8157 1564.5649,884.6457 1564.7359,886.3727 C 1564.8399,887.4077 1565.2099,888.4907 1565.0389,889.5357 C 1564.8939,890.4167 1564.5129,891.3957 1565.4569,891.9937 C 1565.6259,890.8847 1565.5779,889.8857 1566.3259,888.9637 C 1566.3079,889.3027 1566.3289,891.7317 1566.6859,889.6927 C 1566.4849,890.8637 1566.1919,892.0217 1566.0879,893.2087 C 1565.9939,894.2737 1566.7489,895.2207 1566.6129,896.2567 C 1566.2409,899.0837 1564.2079,901.8097 1564.6659,904.8127 C 1564.4039,903.1707 1565.6079,904.8627 1565.6759,903.4427 C 1566.7589,905.2687 1564.6579,907.9987 1563.5339,909.2867 C 1561.6759,911.4157 1558.8569,912.3887 1556.2619,913.3037 C 1553.3249,914.3397 1550.3659,915.4177 1547.7459,917.1227 C 1546.5629,917.8927 1545.3269,918.5697 1544.6689,919.8857 C 1544.1039,921.0147 1543.8179,922.3277 1542.3459,922.5217 C 1543.3279,922.5107 1544.8979,926.0117 1545.7259,923.8917 C 1545.5499,925.6867 1546.4889,930.33598 1544.6349,930.20843 L 1540.1599,930.0352 C 1540.2266,928.56341 1539.3125,926.89789 1539.5467,925.7577 C 1539.7435,924.79933 1539.2777,923.87246 1538.628,923.01516 C 1538.1498,922.38408 1538.258,920.8575 1538.7593,919.6531 C 1539.0443,918.96836 1539.0059,916.5137 1539.0359,915.8097 C 1539.1079,914.0857 1539.0951,912.3627 1539.2251,910.6427 C 1539.3421,909.0907 1538.9859,908.0557 1538.7359,906.5737 C 1538.3599,904.3537 1536.8479,902.5387 1536.3869,900.3777 C 1536.1649,899.3347 1536.3313,898.2347 1536.1483,897.1587 C 1535.8943,895.6637 1535.5929,894.2267 1535.1859,892.7707 C 1534.9219,891.8267 1534.4991,892.10678 1535.0295,891.40978 C 1535.5598,890.99096 1537.0434,889.05832 1537.6689,888.4197 C 1538.6011,887.46796 1542.9349,881.3737 1544.0749,882.2027 C 1542.9259,880.7197 1544.0759,879.8607 1544.4349,878.3827 C 1544.6559,877.4737 1543.7749,876.2047 1544.7739,875.6087 C 1546.1639,874.7797 1546.5269,873.6507 1547.2939,872.2217 C 1547.6219,871.6107 1547.8005,871.09349 1548.0375,870.43349 C 1548.4135,869.39249 1548.1249,868.5927 1548.1759,867.5827 C 1547.9359,867.6787 1547.6969,867.7757 1547.4559,867.8727 C 1546.929,866.82476 1546.5563,865.8829 1546.7621,865.30161 C 1547.0118,864.59676 1547.9073,864.01372 1547.5701,863.0691 C 1547.281,862.25901 1546.1136,862.54746 1546.4425,861.24569 C 1546.6881,860.27369 1548.3908,860.1154 1548.2529,858.9317 C 1548.0158,856.89659 1549.282,857.65959 1549.1579,856.3257 C 1549.0553,855.22355 1548.4359,854.17509 1548.6459,853.06409 C 1548.7929,852.28309 1549.3055,851.41595 1549.1121,850.6367 C 1548.727,849.08497 1548.8562,848.03396 1548.8907,846.37028 C 1548.9019,845.82987 1548.6117,845.43768 1548.4617,844.91168 C 1548.2437,844.14868 1548.9879,842.8517 1549.1159,842.0927 C 1548.0606,842.38989 1547.5251,842.37992 1546.9639,841.3417 C 1546.4332,840.35981 1543.4029,840.2769 1542.6139,839.8227 C 1540.1839,838.4238 1537.861,836.96575 1535.248,836.2267 C 1534.4114,835.99008 1531.7953,836.06623 1530.8815,835.84851 C 1529.4021,835.49604 1529.9969,833.0947 1529.8599,831.9927 C 1529.6649,830.3907 1528.4559,828.9637 1528.5989,827.3307" id="path3462"/>
    <path d="M 1600.8089,838.8507 C 1601.0489,838.6837 1601.2879,838.5177 1601.5289,838.3507 C 1601.5329,838.7707 1601.4639,839.1777 1601.3189,839.5717 C 1601.1489,839.3307 1600.9789,839.0907 1600.8089,838.8507" id="path3466"/>
  </g>
  <g class="landxx coastxx jp" id="jp" transform="translate(-29.9017, -45.0745)">
    <path d="M 2224.078,593.496 C 2223.849,593.849 2223.705,594.233 2223.648,594.649 C 2224.146,594.694 2224.626,594.621 2225.087,594.431 C 2225.002,593.876 2224.641,593.536 2224.078,593.496" id="path3482"/>
    <path d="M 2226.238,593.208 C 2226.279,593.746 2226.568,594.01 2227.107,594 C 2227.783,593.864 2227.467,592.9 2227.467,592.488 C 2227.313,593.075 2226.829,593.373 2226.238,593.208" id="path3484"/>
    <path d="M 2234.448,589.464 C 2233.881,590.34 2235.052,590.991 2235.748,590.472 C 2235.152,590.412 2234.677,590.003 2234.448,589.464" id="path3486"/>
    <path d="M 2242.148,576.36 C 2241.742,576.717 2241.912,576.958 2242.658,577.08 C 2242.488,576.84 2242.318,576.601 2242.148,576.36" id="path3496"/>
    <path d="M 2249.718,579.024 C 2250.694,578.692 2250.34,576.501 2250.93,575.753 C 2251.647,574.848 2253.493,573.962 2252.809,572.543 C 2251.874,572.718 2251.159,574.949 2250.149,573.695 C 2250.284,574.758 2250.627,575.036 2249.896,575.917 C 2249.267,576.677 2249.017,578.219 2249.718,579.024" id="path3504"/>
    <path d="M 2253.238,567.432 C 2253.335,567.6 2253.431,567.769 2253.528,567.937 C 2253.738,567.725 2253.908,567.484 2254.038,567.216 C 2253.771,567.288 2253.505,567.359 2253.238,567.432" id="path3514"/>
    <path d="M 2254.828,563.256 C 2254.285,563.891 2255.061,564.992 2255.838,564.768 C 2255.574,564.215 2255.237,563.711 2254.828,563.256" id="path3516"/>
    <path d="M 2256.118,560.736 C 2256.237,561.067 2256.477,561.258 2256.838,561.311 C 2256.598,561.12 2256.358,560.928 2256.118,560.736" id="path3518"/>
    <path d="M 2258.998,557.928 C 2258.27,558.512 2254.554,560.175 2256.908,561.096 C 2258.361,561.664 2257.446,560.581 2257.847,559.872 C 2258.277,559.113 2259.955,559.081 2258.998,557.928" id="path3520"/>
    <path d="M 2238.769,504.504 C 2237.169,504.838 2238.274,507.227 2239.348,507.528 C 2238.797,506.671 2238.993,505.459 2238.769,504.504" id="path3488"/>
    <path d="M 2239.269,508.464 C 2239.05,508.037 2238.717,507.917 2238.269,508.104 C 2238.491,509.775 2239.866,509.626 2239.269,508.464" id="path3490"/>
    <path d="M 2239.418,521.712 C 2239.51,522.035 2239.63,522.348 2239.778,522.65 C 2240.242,522.34 2240.745,522.124 2241.288,522.001 C 2240.832,521.14 2239.929,521.397 2239.198,520.993 C 2239.271,521.232 2239.345,521.472 2239.418,521.712" id="path3492"/>
    <path d="M 2241.648,518.544 C 2241.551,518.424 2241.455,518.304 2241.358,518.184 C 2241.38,518.858 2240.996,519.86 2242.007,519.984 C 2241.773,519.532 2241.653,519.053 2241.648,518.544" id="path3494"/>
    <path d="M 2243.878,516.456 C 2243.711,516.36 2243.545,516.264 2243.378,516.168 C 2243.177,516.653 2243.104,517.157 2243.158,517.68 C 2243.689,517.482 2244.027,517.039 2243.878,516.456" id="path3498"/>
    <path d="M 2243.878,511.992 C 2243.661,512.015 2243.445,512.04 2243.229,512.063 C 2243.391,512.615 2243.811,512.934 2244.388,512.711 C 2244.218,512.472 2244.048,512.232 2243.878,511.992" id="path3500"/>
    <path d="M 2245.828,516.312 C 2245.51,516.059 2244.615,516.058 2244.306,516.322 C 2243.894,516.673 2244.998,517.754 2245.194,517.956 C 2245.835,518.615 2246.772,518.65 2247.583,518.974 C 2248.212,519.225 2248.306,520.242 2248.849,520.633 C 2248.395,520.402 2247.757,520.634 2247.339,520.345 C 2246.869,520.021 2246.786,519.308 2246.4,518.906 C 2245.008,519.988 2248,521.097 2248.06,522.218 C 2248.279,521.814 2250.04,520.325 2250.374,521.261 C 2250.554,521.768 2250.31,522.646 2251.242,522.441 C 2252.275,522.212 2251.37,519.503 2249.719,520.562 C 2250.253,519.575 2249.407,518.699 2248.709,518.115 C 2249.835,516.503 2252.159,519.742 2252.827,520.684 C 2253.826,522.094 2254.765,524.005 2253.773,525.774 C 2253.183,526.827 2252.376,526.558 2252.852,527.97 C 2253.1,528.707 2253.467,529.263 2253.886,529.905 C 2254.204,530.394 2256.256,533.566 2254.319,532.875 C 2255.072,533.673 2255.55,534.235 2256.699,534.351 C 2257.274,534.409 2257.518,535.088 2258.139,534.963 C 2259.037,534.782 2258.377,534.067 2257.984,533.851 C 2257.101,533.369 2256.374,531.602 2257.041,530.694 C 2257.808,529.648 2259.326,531.591 2257.639,531.075 C 2257.941,531.93 2258.667,532.51 2259.07,533.306 C 2259.56,534.277 2259.355,535.228 2259.29,536.258 C 2260.403,535.809 2260.916,534.738 2261.88,534.098 C 2259.892,532.781 2261.259,532.275 2262.89,533.09 C 2264.037,530.858 2262.053,527.607 2262.216,525.166 C 2262.245,524.729 2261.862,522.889 2262.071,522.536 C 2262.727,521.436 2263.975,520.239 2262.24,519.411 C 2262.402,519.165 2262.572,518.926 2262.75,518.691 C 2261.69,518.74 2261.61,518.005 2260.97,517.42 C 2260.187,516.705 2258.924,517.236 2258.14,516.459 C 2259.272,515.914 2259.574,514.022 2258.304,513.438 C 2256.535,512.624 2255.026,515.399 2253.33,512.961 C 2252.356,511.559 2253.035,510.898 2251.158,511.128 C 2249.834,511.29 2248.04,512.334 2248.99,514.011 C 2247.223,513.946 2247.994,514.058 2246.84,515.307 C 2246.088,513.579 2245.588,515.338 2245.828,516.312" id="path3502"/>
    <path d="M 2250.068,524.016 C 2250.168,524.766 2250.632,525.399 2250.868,526.104 C 2253.766,524.515 2249.349,522.456 2250.068,524.016" id="path3506"/>
    <path d="M 2250.068,530.64 C 2250.687,530.806 2250.807,530.567 2250.427,529.92 C 2250.308,530.16 2250.188,530.399 2250.068,530.64" id="path3508"/>
    <path d="M 2251.878,511.128 C 2251.99,509.723 2253.918,511.233 2255.065,510.83 C 2257.414,510.006 2258.961,510.576 2261.088,511.848 C 2260.316,511.246 2259.796,507.487 2261.229,507.384 C 2262.029,507.327 2262.412,508.364 2263.127,508.574 C 2263.641,508.724 2265.986,507.53 2266.348,507.168 C 2266.354,507.537 2266.512,507.758 2266.823,507.83 C 2267.029,507.574 2267.061,507.281 2266.919,506.953 C 2267.509,507.373 2268.214,507.229 2268.149,506.377 C 2268.964,506.866 2269.649,505.92 2270.514,506.236 C 2271.768,506.694 2272.634,504.97 2271.239,505.442 C 2272.585,505.092 2273.773,504.234 2275.148,504.001 C 2276.672,503.742 2277.835,505.17 2279.21,504.903 C 2280.234,504.704 2281.376,503.733 2282.132,505.065 C 2282.758,506.169 2281.512,507.507 2280.599,507.961 C 2281.383,508.55 2282.035,509.368 2281.935,510.409 C 2281.822,511.582 2283.22,512.116 2284.053,512.532 C 2284.954,512.983 2285.242,514.081 2286.224,514.439 C 2286.967,514.71 2287.646,515.158 2288.417,515.356 C 2289.325,515.589 2289.844,511.571 2290.009,510.985 C 2290.167,510.427 2290.066,509.107 2290.547,508.695 C 2291.489,507.89 2292.836,508.557 2293.849,508.033 C 2292.939,506.87 2292.259,506.169 2290.897,505.536 C 2289.236,504.764 2289.841,504.012 2289.819,502.534 C 2289.808,501.871 2290.433,500.611 2291.027,501.872 C 2291.373,502.609 2291.183,504.334 2292.48,504.073 C 2292.025,503.676 2291.904,503.196 2292.12,502.633 C 2292.548,504.557 2294.882,502.861 2295.29,504.362 C 2294.737,504.507 2294.331,504.914 2294.35,505.513 C 2296.793,504.075 2299.678,504.981 2302.271,505.369 C 2300.578,503.759 2303.38,499.096 2305.521,501.697 C 2305.252,501.787 2304.985,501.883 2304.72,501.984 C 2305.311,502.842 2305.15,504.666 2306.295,505.083 C 2307.538,505.537 2307.77,503.607 2307.693,502.831 C 2307.596,501.844 2306.63,501.225 2306.532,500.235 C 2306.326,498.172 2309.824,499.784 2310.41,500.4 C 2310.981,499.484 2308.027,497.789 2309.98,497.377 C 2308.898,496.075 2312.936,495.535 2311.818,497.992 C 2311.532,498.621 2311.418,502.402 2312.529,502.689 C 2312.96,502.8 2315.431,500.429 2315.442,499.889 C 2315.46,499.046 2314.829,498.479 2314.799,497.692 C 2314.752,496.449 2316.021,495.998 2316.889,495.504 C 2314.55,493.988 2312.904,491.738 2312.124,489.066 C 2311.839,488.091 2311.716,486.996 2311.738,485.982 C 2311.751,485.362 2311.856,485.401 2312.363,485.167 C 2312.821,484.955 2312.366,484.531 2312.348,484.2 C 2312.191,481.291 2309.729,478.996 2308.31,476.677 C 2306.646,473.959 2307.947,471.518 2310.989,473.472 C 2309.869,472.269 2307.707,467.246 2309.979,466.777 C 2309.182,466.134 2309.809,464.886 2309.241,464.083 C 2308.78,463.433 2308.823,462.659 2308.261,461.965 C 2307.626,461.18 2307.339,460.148 2306.679,459.336 C 2306.087,458.607 2305.48,458.289 2305.007,457.48 C 2303.991,455.743 2301.692,454.9 2300.212,453.628 C 2298.281,451.968 2296.873,449.192 2296.23,446.761 C 2295.142,447.752 2292.554,445.166 2291.883,446.277 C 2290.893,447.914 2294.307,450.147 2294.789,447.841 C 2295.904,448.368 2297.094,450.079 2296.299,451.297 C 2295.594,450.844 2294.836,450.481 2294.07,450.146 C 2294.384,450.65 2294.384,451.154 2294.07,451.658 C 2292.867,451.36 2292.476,450.055 2291.759,449.21 C 2291.006,448.323 2289.942,448.666 2289.099,447.985 C 2289.076,449.031 2291.227,451.224 2290.286,452.038 C 2289.803,452.456 2288.063,452.898 2289.063,453.901 C 2289.83,454.669 2291.3,455.694 2291.629,456.696 C 2291.923,457.59 2291.962,459.738 2290.469,458.713 C 2290.759,459.726 2291.281,459.577 2292.12,459.575 C 2293.237,459.572 2294.065,460.834 2294.569,461.665 C 2295.866,463.802 2296.261,467.519 2295.917,469.949 C 2295.65,471.833 2297.508,475.126 2295.495,476.237 C 2293.19,477.508 2294.349,479.721 2293.186,481.872 C 2292.228,483.645 2290.11,483.737 2288.57,484.761 C 2287.609,485.401 2286.4,488.367 2284.978,485.875 C 2284.592,485.199 2284.733,484.271 2283.764,484.093 C 2283.184,483.987 2282.507,482.844 2283.284,482.983 C 2284.91,483.272 2284.284,480.652 2284.559,480.169 C 2283.426,479.853 2281.496,480.864 2281.133,481.983 C 2280.634,483.528 2283.364,485.36 2283.012,487.358 C 2282.73,488.957 2282.02,489.941 2281.069,491.186 C 2279.484,493.262 2282.8,494.469 2282.689,496.224 C 2281.362,495.127 2282.418,496.978 2281.109,496.512 C 2281.608,498.855 2277.014,496.173 2278.519,497.808 C 2278.083,497.529 2277.077,497.499 2276.963,496.87 C 2276.838,496.171 2277.335,495.099 2276.127,495.438 C 2275.607,495.584 2275.162,495.911 2274.651,496.064 C 2273.391,496.442 2272.041,496.051 2270.845,496.658 C 2270.005,497.085 2269.236,497.363 2268.298,497.491 C 2267.409,497.613 2265.562,497.066 2264.885,497.521 C 2263.928,498.165 2263.548,497.706 2262.671,497.21 C 2261.685,496.652 2259.976,497.738 2259.067,498.095 C 2260.669,499.89 2254.3,504.797 2253.828,506.163 C 2253.332,507.598 2250.376,506.337 2250.415,507.747 C 2250.447,508.871 2251.351,510.164 2251.878,511.128" id="path3510"/>
    <path d="M 2251.729,523.8 C 2251.911,524.709 2253.208,523.957 2253.239,523.296 C 2252.662,523.106 2252.086,523.324 2251.729,523.8" id="path3512"/>
    <path d="M 2259.578,492.12 C 2258.939,492.039 2258.843,492.304 2259.288,492.912 C 2259.385,492.648 2259.481,492.384 2259.578,492.12" id="path3522"/>
    <path d="M 2259.868,541.224 C 2258.59,541.45 2259.634,542.913 2260.396,543.062 C 2261.896,543.353 2260.84,541.385 2259.868,541.224" id="path3524"/>
    <path d="M 2260.298,490.392 C 2259.712,490.686 2259.635,491.522 2260.368,491.689 C 2261.56,491.959 2260.913,490.629 2260.298,490.392" id="path3526"/>
    <path d="M 2262.818,511.344 C 2262.584,510.886 2261.591,511.544 2261.378,511.632 C 2261.873,511.612 2262.354,511.516 2262.818,511.344" id="path3528"/>
    <path d="M 2262.098,508.248 C 2261.259,508.547 2261.329,508.787 2262.309,508.968 C 2262.238,508.728 2262.168,508.488 2262.098,508.248" id="path3530"/>
    <path d="M 2264.398,513.72 C 2263.882,514.908 2262.503,515.207 2261.738,516.168 C 2262.642,516.23 2263.203,515.4 2263.967,515.088 C 2264.135,515.504 2264.467,516.767 2265.089,516.719 C 2265.901,516.657 2266.216,517.245 2265.117,517.464 C 2265.935,517.735 2265.934,519.961 2267.034,519.815 C 2268.003,519.685 2268.155,520.838 2268.937,521.136 C 2269.709,521.43 2270.514,520.969 2271.166,521.64 C 2269.664,520.598 2270.568,517.737 2271.179,516.444 C 2272.744,513.128 2275.483,515.532 2277.506,516.816 C 2277.715,516.056 2277.486,515.269 2277.716,514.512 C 2277.989,513.609 2278.844,513.033 2279.32,512.243 C 2279.772,511.492 2277.851,509.496 2277.797,508.464 C 2276.119,508.812 2275.016,507.661 2273.458,507.381 C 2272.158,507.148 2271.031,508.882 2270.087,508.32 C 2271.41,509.497 2270.752,510.299 2269.297,510.696 C 2267.21,511.266 2267.468,509.863 2265.558,509.544 C 2265.388,510.5 2265.432,513.397 2264.398,513.72" id="path3532"/>
    <path d="M 2262.818,509.04 C 2262.132,509.44 2262.607,509.925 2263.248,509.76 C 2263.104,509.52 2262.962,509.28 2262.818,509.04" id="path3534"/>
    <path d="M 2262.818,537.912 C 2262.006,538.33 2262.099,542.02 2263.318,541.584 C 2262.811,540.4 2263.443,539.073 2262.818,537.912" id="path3536"/>
    <path d="M 2265.908,508.032 C 2265.357,507.928 2265.357,508.167 2265.908,508.752 C 2265.908,508.512 2265.908,508.272 2265.908,508.032" id="path3538"/>
    <path d="M 2271.678,412.128 C 2271.223,412.046 2272.576,413.036 2272.828,413.136 C 2272.65,412.616 2272.223,412.227 2271.678,412.128" id="path3540"/>
    <path d="M 2274.698,506.016 C 2274.158,505.719 2273.697,505.962 2273.398,506.449 C 2273.885,506.907 2275.191,507.088 2274.698,506.016" id="path3542"/>
    <path d="M 2274.048,413.856 C 2273.985,414.739 2274.913,415.158 2275.559,414.576 C 2275.127,414.186 2274.623,413.945 2274.048,413.856" id="path3544"/>
    <path d="M 2276.278,412.128 C 2276.904,413.149 2277.055,414.34 2277.984,415.176 C 2279.369,416.422 2280.996,417.722 2281.832,419.425 C 2282.566,420.918 2283.422,422.595 2283.988,424.153 C 2284.53,425.638 2282.987,426.239 2284.05,427.521 C 2285.142,428.839 2287.69,431.957 2284.156,431.546 C 2282.564,431.361 2280.765,430.025 2279.242,430.067 C 2278.299,430.093 2279.371,431.54 2279.676,431.746 C 2280.543,432.334 2281.405,432.636 2281.219,433.9 C 2280.881,436.173 2278.351,434.877 2279.401,437.874 C 2280.198,440.149 2282.559,440.036 2283.969,441.676 C 2284.862,442.713 2284.389,444.251 2285.175,445.414 C 2286.425,447.263 2286.856,446.034 2288.309,445.464 C 2287.32,444.687 2288.311,443.736 2289.194,443.568 C 2289.432,443.523 2292.475,444.351 2291.715,443.08 C 2291.173,442.174 2289.146,441.668 2288.276,441.059 C 2287.004,440.169 2285.295,440.311 2283.978,439.399 C 2283.241,438.888 2282.734,437.784 2283.22,436.918 C 2283.699,436.064 2285.108,436.85 2285.684,437.147 C 2286.437,437.535 2287.081,438.249 2287.846,438.555 C 2288.656,438.879 2289.665,436.998 2290.415,436.569 C 2292.888,435.154 2296.228,438.332 2298.446,439.132 C 2300.051,439.71 2303.375,441.854 2305.079,441.651 C 2306.12,441.527 2305.57,440.445 2305.278,439.888 C 2304.578,438.554 2304.422,437.554 2305.005,436.087 C 2306.232,432.999 2308.14,432.906 2311.129,433.657 C 2309.796,431.973 2314.255,433.877 2312.789,431.928 C 2313.49,431.629 2314.251,431.42 2315.019,431.568 C 2314.925,430.934 2315.298,430.42 2315.878,430.201 C 2314.925,429.104 2314.974,430.546 2314.292,430.876 C 2313.877,431.076 2313.134,430.188 2312.992,430.062 C 2312.665,429.771 2310.021,427.292 2311.988,428.185 C 2310.592,427.554 2308.991,427.248 2308.699,425.477 C 2308.509,424.32 2308.829,422.123 2307.458,421.56 C 2307.254,422.696 2306.945,424.67 2305.641,425.086 C 2304.074,425.586 2301.647,423.15 2300.758,424.153 C 2300.484,423.281 2299.842,423.341 2299.101,423.47 C 2298.069,423.649 2297.871,422.88 2297.015,422.551 C 2294.35,421.525 2291.92,420.637 2289.438,419.123 C 2286.576,417.378 2283.912,415.341 2281.058,413.581 C 2280.126,413.006 2278.968,411.988 2277.869,411.768 C 2277.078,411.61 2277.142,412.791 2276.278,412.128" id="path3546"/>
    <path d="M 2279.158,505.8 C 2278.073,505.742 2277.378,507.948 2278.17,508.604 C 2279.638,509.82 2279.1,506.188 2279.158,505.8" id="path3548"/>
    <path d="M 2278.078,439.704 C 2278.053,440.283 2278.371,440.579 2278.948,440.496 C 2278.994,439.888 2278.699,439.559 2278.078,439.704" id="path3550"/>
    <path d="M 2289.598,473.472 C 2288.825,473.74 2287.915,475.393 2288.983,475.876 C 2289.932,476.306 2289.751,476.592 2289.598,477.649 C 2290.78,477.615 2291.312,476.521 2290.72,475.53 C 2290.516,475.335 2290.277,475.205 2290.002,475.141 C 2289.626,474.938 2289.744,473.842 2289.598,473.472" id="path3552"/>
    <path d="M 2310.118,503.712 C 2309.829,504.363 2310.046,504.603 2310.768,504.432 C 2310.569,504.176 2310.353,503.936 2310.118,503.712" id="path3556"/>
  </g>
  <g class="landxx coastxx nz" id="nz" transform="translate(-29.9017, -45.0745)">
    <path d="M 2409.408,1240.343 C 2408.907,1239.498 2405.224,1241.716 2405.378,1242.644 C 2405.905,1242.393 2406.822,1242.78 2407.149,1242.337 C 2407.852,1241.388 2408.334,1240.925 2409.408,1240.343" id="path3560"/>
    <path d="M 2446.418,1209.383 C 2448.303,1210.328 2447.556,1205.935 2445.688,1208.179 C 2444.351,1209.785 2443.266,1211.232 2441.159,1211.973 C 2441.823,1212.433 2444.132,1211.596 2444.914,1211.388 C 2445.482,1211.237 2446.518,1211.417 2446.909,1210.857 C 2447.548,1209.937 2446.345,1209.597 2445.699,1209.963 C 2445.938,1209.77 2446.178,1209.576 2446.418,1209.383" id="path3564"/>
    <path d="M 2442.168,1201.823 C 2442.986,1201.613 2443.755,1201.243 2444.479,1200.813 C 2444.37,1201.764 2443.2,1201.571 2442.888,1202.403 C 2443.764,1202.392 2444.225,1201.455 2445.118,1201.533 C 2444.18,1201.803 2442.683,1202.518 2442.388,1203.553 C 2443.6,1203.674 2445.537,1204.561 2446.644,1203.756 C 2447.915,1202.831 2449.364,1203.452 2448.428,1205.133 C 2449.151,1205.181 2449.924,1204.613 2450.635,1204.83 C 2451.787,1205.183 2450.565,1206.239 2450.598,1206.934 C 2451.186,1206.619 2452.762,1206.538 2453.42,1206.636 C 2453.964,1206.717 2453.539,1207.496 2454.414,1207.506 C 2456.879,1207.532 2459.077,1206.386 2461.403,1205.768 C 2461.844,1205.651 2462.613,1204.716 2463.099,1204.482 C 2464.183,1203.959 2465.424,1203.731 2466.427,1203.044 C 2467.351,1202.411 2468.049,1201.557 2469.181,1201.268 C 2470.192,1201.009 2471.473,1201.194 2472.118,1200.174 C 2471.621,1200.361 2471.118,1200.528 2470.608,1200.674 C 2473.359,1199.421 2475.287,1196.805 2477.655,1195.013 C 2478.767,1194.17 2480.068,1193.61 2481.129,1192.695 C 2482.336,1191.655 2483.121,1190.234 2484.288,1189.155 C 2486.432,1187.172 2489.171,1186.318 2491.813,1185.229 C 2493.147,1184.68 2494.759,1184.398 2496.014,1183.727 C 2496.855,1183.277 2498.23,1181.789 2499.118,1182.964 C 2498.622,1183.173 2498.119,1183.363 2497.608,1183.534 C 2498.827,1182.995 2500.268,1184.096 2501.347,1183.684 C 2503.769,1182.768 2502.01,1181.968 2500.707,1181.454 C 2501.953,1181.147 2502.168,1179.289 2503.072,1178.546 C 2504.63,1177.265 2506.848,1176.666 2508.618,1175.707 C 2510.204,1174.848 2511.376,1173.626 2512.752,1172.496 C 2513.919,1171.537 2515.274,1171.076 2516.589,1170.377 C 2517.599,1169.84 2518.179,1169.378 2518.847,1168.508 C 2519.714,1167.378 2521.262,1167.057 2522.377,1166.254 C 2523.671,1165.323 2523.148,1163.428 2524.262,1162.41 C 2525.412,1161.362 2527.342,1161.404 2528.207,1159.994 C 2527.737,1160.56 2525.071,1161.973 2524.467,1161.074 C 2525.525,1160.874 2527.803,1160.506 2527.637,1158.984 C 2526.482,1159.019 2526.504,1159.794 2525.646,1160.04 C 2525.157,1160.181 2523.704,1161.067 2525.398,1160.714 C 2524.627,1160.798 2523.924,1161.146 2523.168,1161.294 C 2523.827,1160.904 2526.157,1159.386 2524.538,1159.344 C 2525.176,1159.203 2525.754,1158.813 2526.188,1158.334 C 2524.759,1158.579 2523.587,1159.484 2522.228,1159.924 C 2520.94,1160.34 2519.681,1161.903 2518.198,1161.364 C 2518.76,1160.231 2520.704,1158.558 2520.427,1157.184 C 2520.052,1157.381 2518.284,1157.642 2518.618,1156.705 C 2519.095,1155.366 2520.774,1154.795 2522.017,1155.093 C 2520.14,1154.136 2515.936,1156.418 2514.461,1157.689 C 2513.538,1158.485 2512.863,1159.505 2512.12,1160.46 C 2511.099,1161.772 2509.805,1162.599 2508.513,1163.614 C 2507.203,1164.642 2505.978,1164.904 2504.43,1165.37 C 2502.992,1165.802 2502.034,1167.111 2501.065,1168.175 C 2498.764,1170.698 2496.237,1172.85 2493.296,1174.603 C 2492.02,1175.365 2490.753,1175.874 2489.334,1176.296 C 2487.746,1176.77 2486.402,1177.809 2484.909,1178.513 C 2481.705,1180.026 2478.625,1181.427 2475.333,1182.752 C 2472.217,1184.007 2468.901,1184.691 2465.926,1186.274 C 2464.712,1186.92 2460.05,1188.74 2460.676,1190.593 C 2460.197,1188.853 2457.985,1191.239 2457.201,1191.522 C 2456.361,1191.827 2453.522,1192.838 2453.976,1193.763 C 2453.262,1193.488 2452.355,1193.971 2451.784,1194.355 C 2451.164,1194.772 2449.997,1195.564 2451.386,1195.564 C 2450.244,1195.589 2450.553,1196.904 2449.796,1197.434 C 2450.496,1196.675 2450.376,1196.482 2449.437,1196.854 C 2449.524,1196.435 2449.218,1195.409 2448.859,1196.197 C 2448.501,1196.982 2447.286,1197.475 2446.487,1198.013 C 2447.3,1198.216 2448.02,1197.758 2448.797,1197.644 C 2448.038,1197.963 2447.244,1198.189 2446.487,1198.513 C 2446.99,1198.667 2447.494,1198.667 2447.997,1198.513 C 2447.491,1198.746 2446.965,1198.913 2446.417,1199.013 C 2448.474,1199.381 2445.032,1199.944 2444.52,1200.033 C 2443.457,1200.215 2442.301,1200.575 2442.168,1201.823" id="path3566"/>
    <path d="M 2527.708,1156.394 C 2526.891,1156.792 2525.859,1157.157 2525.688,1158.194 C 2526.631,1157.996 2527.276,1157.196 2527.708,1156.394" id="path3568"/>
    <path d="M 2530.078,1161.363 C 2529.996,1161.868 2529.902,1162.372 2529.798,1162.873 C 2530.404,1162.526 2531.122,1162.609 2531.738,1162.293 C 2531.883,1162.904 2531.713,1163.411 2531.228,1163.814 C 2532.582,1164.935 2535.352,1162.853 2536.487,1162.185 C 2539.029,1160.69 2541.15,1159.086 2543.466,1157.251 C 2544.47,1156.455 2545.658,1155.305 2546.922,1154.969 C 2547.978,1154.689 2548.464,1153.563 2549.295,1152.959 C 2550.335,1152.205 2551.464,1151.827 2552.438,1150.918 C 2553.623,1149.812 2554.632,1148.525 2555.858,1147.463 C 2553.695,1147.038 2557.582,1144.056 2558.158,1143.673 C 2559.486,1142.791 2564.752,1140.962 2563.778,1144.083 C 2565.554,1142.97 2565.543,1140.427 2567.48,1139.358 C 2568.172,1138.976 2568.972,1138.855 2569.675,1138.504 C 2570.921,1137.882 2571.48,1136.894 2572.332,1135.862 C 2573.611,1134.314 2574.71,1132.41 2576.294,1131.215 C 2577.096,1130.61 2576.972,1129.935 2576.087,1129.462 C 2574.92,1128.838 2573.178,1129.983 2572.187,1130.525 C 2569.72,1131.876 2567.447,1134.099 2564.474,1132.814 C 2563.736,1132.495 2559.695,1130.539 2559.602,1129.758 C 2559.535,1129.191 2562.978,1122.862 2561.328,1123.272 C 2561.814,1122.822 2562.381,1122.457 2562.988,1122.192 C 2562.421,1122.209 2561.867,1122.306 2561.328,1122.483 C 2561.502,1121.708 2561.673,1120.889 2561.188,1120.172 C 2560.494,1120.452 2560.596,1120.598 2560.773,1121.14 C 2560.985,1121.786 2560,1122.551 2559.748,1123.052 C 2559.16,1124.22 2559.185,1126.248 2557.533,1126.505 C 2556.68,1126.637 2557.451,1124.851 2557.479,1124.489 C 2557.56,1123.453 2554.667,1123.289 2553.988,1123.412 C 2553.988,1123.321 2555.098,1122.631 2555.375,1122.134 C 2555.809,1121.354 2556.028,1120.78 2556.557,1120.043 C 2556.948,1119.496 2558.188,1119.196 2557.817,1118.381 C 2557.424,1117.517 2556.765,1115.807 2557.588,1114.992 C 2557.119,1114.958 2556.709,1114.817 2556.9,1114.328 C 2557.2,1113.561 2558.187,1114.471 2558.597,1114.771 C 2559.243,1113.066 2559.572,1111.276 2559.528,1109.452 C 2559.055,1109.574 2558.575,1109.621 2558.088,1109.592 C 2558.211,1109.854 2558.332,1110.117 2558.448,1110.382 C 2558.189,1110.457 2557.926,1110.481 2557.658,1110.452 C 2557.801,1109.859 2557.511,1108.721 2558.518,1108.872 C 2557.536,1108.736 2557.367,1106.439 2556.288,1108.012 C 2557.018,1106.884 2553.113,1106.613 2555.278,1106.212 C 2554.293,1105.834 2553.936,1107.088 2553.268,1107.502 C 2553.929,1106.608 2553.544,1103.984 2553.118,1103.042 C 2553.7,1103.124 2554.142,1102.773 2554.347,1102.252 C 2553.804,1102.41 2551.546,1102.368 2551.544,1102.951 C 2551.541,1103.647 2551.92,1104.308 2552.065,1104.979 C 2552.311,1106.111 2552.848,1109.06 2550.886,1108.941 C 2551.149,1109.899 2551.18,1110.907 2551.246,1111.891 C 2551.779,1110.867 2553.501,1111.077 2553.696,1109.812 C 2553.745,1111.384 2551.492,1110.944 2551.351,1112.626 C 2551.159,1114.915 2552.478,1116.983 2551.896,1119.381 C 2553.336,1119.652 2552.962,1117.898 2552.755,1117.081 C 2553.318,1117.233 2553.744,1117.676 2553.905,1118.231 C 2553.813,1116.71 2554.725,1118.204 2555.495,1117.371 C 2554.82,1117.661 2554.22,1118.013 2555.495,1118.301 C 2555.094,1118.578 2554.59,1118.512 2554.181,1118.75 C 2553.756,1118.998 2553.753,1119.982 2554.776,1119.241 C 2554.171,1119.894 2554.102,1120.831 2553.476,1121.47 C 2553.119,1121.028 2552.975,1120.525 2553.045,1119.96 C 2551.8,1120.27 2552.196,1123.761 2551.825,1124.71 C 2552.559,1124.816 2553.878,1123.645 2554.269,1124.715 C 2554.542,1125.462 2552.767,1126.404 2552.544,1125.14 C 2551.977,1125.483 2550.929,1127.469 2552.185,1127.59 C 2551.418,1127.841 2551.103,1128.739 2550.903,1129.432 C 2550.791,1129.843 2550.683,1130.256 2550.581,1130.67 C 2550.335,1131.518 2549.777,1131.462 2549.386,1132.133 C 2549.044,1132.721 2548.939,1133.799 2548.516,1134.29 C 2548.249,1134.297 2547.982,1134.297 2547.715,1134.29 C 2547.17,1134.773 2546.675,1135.316 2546.199,1135.867 C 2544.754,1137.541 2543.77,1139.834 2541.892,1141.086 C 2540.083,1142.291 2537.558,1142.046 2535.617,1143.211 C 2534.785,1143.71 2534.069,1144.461 2533.973,1145.479 C 2533.864,1146.626 2535.25,1146.689 2535.87,1147.288 C 2536.799,1148.186 2536.904,1148.952 2538.269,1149.421 C 2539.613,1149.883 2539.836,1151.127 2539.373,1152.355 C 2538.428,1154.852 2536.012,1156.506 2533.964,1158.042 C 2533.127,1158.669 2532.446,1159.803 2531.407,1160.119 C 2530.399,1160.428 2529.167,1161.052 2528.634,1162.012 C 2529.275,1162.304 2529.826,1161.969 2530.078,1161.363" id="path3570"/>
    <path d="M 2558.158,1122.483 C 2557.708,1122.597 2557.277,1122.764 2556.868,1122.983 C 2557.441,1123.336 2557.955,1123.105 2558.158,1122.483" id="path3572"/>
    <path d="M 2563.269,1116.503 C 2562.309,1116.974 2561.991,1119.263 2562.629,1118.919 C 2563.247,1118.586 2563.057,1117.076 2563.269,1116.503" id="path3574"/>
    <path d="M 2571.269,1184.903 C 2571.231,1185.24 2571.231,1185.576 2571.269,1185.913 C 2571.991,1185.622 2572.701,1185.272 2573.498,1185.263 C 2572.795,1184.534 2573.321,1182.242 2574.508,1183.394 C 2574.293,1183.66 2574.054,1183.899 2573.788,1184.114 C 2574.472,1183.53 2575.209,1183.001 2576.018,1182.604 C 2574.705,1182.915 2572.328,1181.745 2571.627,1183.463 C 2572.14,1183.286 2572.643,1183.31 2573.137,1183.533 C 2572.835,1184.323 2572.061,1184.753 2571.269,1184.903" id="path3576"/>
    <path d="M 2572.128,1187.493 C 2572.593,1187.348 2572.809,1187.335 2572.918,1186.773 C 2572.646,1187.003 2572.383,1187.243 2572.128,1187.493" id="path3578"/>
  </g>
  <g class="landxx coastxx cu" id="cu" transform="translate(-29.9017, -45.0745)">
    <path d="M 666.865,615.312 C 668.251,614.811 669.718,613.513 671.257,614.304 C 671.762,613.168 670.351,612.208 671.353,611.094 C 672.166,610.19 673.462,608.83 674.461,608.164 C 675.644,607.376 677.175,607.08 678.508,606.643 C 679.655,606.267 680.693,605.512 681.932,605.539 C 684.508,605.596 686.657,604.283 689.205,604.094 C 690.387,604.006 691.812,603.921 692.931,604.395 C 693.188,604.503 693.499,604.694 693.505,605.015 C 694.436,604.575 695.475,604.428 696.385,603.936 C 694.891,604.329 696.522,605.168 697.332,604.874 C 698.079,604.603 700.33,604.578 700.592,605.004 C 701.003,605.671 702.105,606.663 702.98,606.4 C 704.002,606.093 705.364,607.276 706.25,606.528 C 706.042,606.78 705.85,607.044 705.674,607.32 C 707.324,607.752 707.003,608.527 708.144,609.651 C 709.639,611.124 710.318,610.759 712.288,610.939 C 714.169,611.111 714.899,611.957 716.403,612.865 C 716.973,613.209 717.784,613.127 718.171,613.771 C 718.528,614.366 718.995,615.434 719.787,614.592 C 719.491,615.474 720.208,616.083 721.022,616.157 C 721.973,616.244 722.27,617.083 723.099,617.399 C 722.754,616.71 722.516,615.817 721.659,615.599 C 722.434,615.77 724.945,616.746 724.539,617.904 C 724.023,617.834 723.519,617.714 723.027,617.544 C 723.38,617.999 723.517,618.505 723.819,618.983 C 724.921,617.094 725.257,619.253 726.555,619.272 C 726.377,619.523 726.185,619.763 725.979,619.992 C 726.454,619.713 726.958,619.641 727.491,619.776 C 727.392,620.039 727.296,620.302 727.203,620.567 C 727.749,620.674 728.229,620.915 728.643,621.288 C 728.54,619.332 731.308,621.6 731.871,621.999 C 732.608,622.521 733.282,621.895 734.07,621.822 C 735.35,621.703 735.806,623.477 734.403,623.664 C 734.667,623.808 734.931,623.951 735.195,624.096 C 734.593,624.284 734.061,624.674 733.683,625.176 C 735.738,625.614 739.219,624.871 740.854,626.281 C 742.237,627.475 742.61,628.727 744.536,628.996 C 746.815,629.313 743.807,630.903 742.871,631.012 C 742.587,631.045 737.698,632.349 737.598,632.188 C 737.632,631.933 737.766,631.731 738.003,631.584 C 737.098,631.584 737.148,632.737 735.605,632.587 C 733.957,632.426 732.416,631.838 730.744,631.765 C 726.555,631.583 722.333,633.161 718.131,633.025 C 718.192,630.803 720.948,630.271 722.396,629.192 C 723.249,628.556 723.771,627.956 722.937,627.27 C 722.313,626.756 723.19,625.117 721.683,625.487 C 718.86,626.182 716.014,625.662 714.4,623.056 C 713.821,622.121 713.924,620.882 713.76,619.835 C 713.684,619.356 713.566,618.838 713.247,618.454 C 712.868,617.997 712.034,618.183 712.875,617.76 C 711.586,617.144 710.307,618.105 709.059,618.228 C 707.762,618.356 706.659,617.146 705.469,616.931 C 704.305,616.72 703.59,616.91 702.959,616.147 C 701.955,614.934 700.623,615.002 700.419,613.08 C 700.197,614.688 695.319,614.21 695.235,612.072 C 694.95,612.536 694.806,613.04 694.802,613.584 C 694.367,613.569 694.028,613.387 693.785,613.036 C 693.167,612.413 692.187,612.649 691.419,612.649 C 690.154,612.649 689.06,611.9 688.179,611.063 C 688.849,610.326 690.139,610.709 691,610.565 C 692.281,610.35 692.197,609.052 691.082,608.678 C 688.885,607.94 684.747,607.869 682.766,609.172 C 681.539,609.98 680.142,610.875 679.156,611.971 C 678.234,612.994 677.58,612.228 676.49,612.681 C 675.866,612.94 675.199,612.546 674.601,612.904 C 673.969,613.283 674.184,613.898 673.707,614.376 C 672.706,615.38 671.081,615.286 670.035,616.248 C 670.342,613.33 666.921,617.002 666.579,615.096 C 666.673,615.168 666.769,615.239 666.865,615.312" id="path3580"/>
    <path d="M 681.193,619.2 C 680.422,619.021 679.569,618.716 679.393,617.832 C 680.939,619.488 680.289,616.549 680.302,616.081 C 680.329,615.07 681.872,614.664 682.633,615.024 C 683.429,615.402 684.032,616.853 684.013,617.682 C 683.988,618.808 681.977,618.974 681.193,619.2" id="path3582"/>
    <path d="M 703.081,606.024 C 702.817,605.976 702.553,605.927 702.289,605.879 C 702.534,605.716 702.798,605.596 703.081,605.519 C 703.081,605.688 703.081,605.856 703.081,606.024" id="path3586"/>
    <path d="M 714.313,627.192 C 714.037,626.965 713.772,626.725 713.521,626.471 C 714.175,626.454 714.142,626.688 714.313,627.192" id="path3588"/>
    <path d="M 715.897,610.704 C 715.401,610.523 714.921,610.308 714.457,610.056 C 715.273,609.687 716.748,609.106 717.049,610.416 C 716.661,610.499 716.277,610.595 715.897,610.704" id="path3590"/>
    <path d="M 718.705,612 C 718.06,611.619 717.094,611.597 716.905,610.704 C 718.042,610.046 718.305,611.226 718.705,612" id="path3592"/>
    <path d="M 719.712,613.44 C 718.88,613.314 718.393,612.431 718.92,611.711 C 719.157,612.137 719.998,612.856 719.712,613.44" id="path3594"/>
    <path d="M 720.649,615.096 C 720.169,614.712 719.689,614.328 719.209,613.945 C 720.234,613.011 722.261,614.035 720.649,615.096" id="path3596"/>
  </g>
  <g class="landxx coastxx ve" id="ve" transform="translate(-29.9017, -45.0745)">
    <path class="landxx ve" d="M 742.105,725.256 C 744.075,722.932 745.061,720.438 745.682,717.508 C 746.278,714.699 746.93191,712.51846 748.39591,710.06246 C 749.58891,708.06246 752.29336,706.90228 753.43736,704.80428 C 754.74136,702.41428 756.96573,703.03164 759.09673,701.71664 C 758.22873,703.83464 755.95436,703.00709 754.70036,704.51709 C 753.63236,705.80309 754.772,708.581 756.144,709.128 C 754.903,709.066 756.1,711.819 756.22,712.339 C 756.588,713.938 755.051,715.557 753.986,716.565 C 752.599,717.877 751.923,719.472 752.782,721.194 C 753.115,721.86 753.226,722.702 753.865,723.175 C 754.573,723.699 754.687,723.468 754.566,724.42 C 754.311,726.421 755.846,726.872 757.454,725.958 C 759.114,725.014 759.771,723.618 759.985,721.775 C 760.26,719.411 759.198,718.088 758.011,716.159 C 757.539,715.393 757.124,714.531 756.844,713.674 C 756.592,712.902 757.249,711.848 756.648,711.217 C 757.55,711.323 758.332,709.959 759.024,709.561 C 760.29,708.832 761.475,707.875 762.95,707.616 C 764.226,707.391 765.246,707.217 766.366,706.542 C 766.788,706.288 768.044,705.273 768.597,705.53 C 768.523,705.128 768.333,704.792 768.027,704.521 C 768.577,704.745 770.255,706.149 770.827,705.486 C 771.215,705.036 770.813,703.93 770.313,703.722 C 769.458,703.367 768.312,704.175 767.447,704.233 C 767.616,703.241 766.782,697.982 769.722,699.191 C 771.06,699.741 770.785,702.645 770.992,703.782 C 771.451,706.305 774.163,704.63 775.959,705.256 C 777.969,705.957 780.474,706.499 781.353,708.833 C 781.891,710.262 780.493,713.468 782.873,713.973 C 785.704,714.574 789.289,713.263 792.094,712.89 C 794.499,712.57 797,712.057 798.473,714.206 C 799.083,715.096 799.937,715.823 800.957,716.182 C 801.788,716.475 802.65,716.669 803.5,716.894 C 803.41,716.87 806.29,717.482 804.956,717.482 C 807.22,718.709 808.297,716.055 810.087,715.889 C 811.175,715.788 811.424,714.655 812.327,714.233 C 813.572,713.653 814.992,714.761 816.265,714.027 C 815.169,712.87 812.593,714.109 811.865,712.659 C 812.494,712.659 813.124,712.659 813.754,712.659 C 814.64,712.659 814.9,712.113 815.545,712.083 C 815.863,712.193 816.171,712.325 816.47,712.48 C 817.352,712.64 818.268,712.531 819.142,712.383 C 822.876,711.753 826.701,712.457 830.375,711.722 C 829.801,712.886 827.962,712.667 826.905,713.223 C 825.247,714.096 823.294,712.71 821.735,714.026 C 822.237,713.525 822.758,713.592 822.955,714.313 C 822.339,714.262 821.896,714.649 821.805,715.25 C 823.17,714.445 824.217,715.263 824.187,716.864 C 824.163,718.132 822.037,717.283 821.375,717.266 C 821.852,719.045 824.78,717.793 824.475,716.186 C 825.264,716.298 825.6,717.203 825.664,717.884 C 825.749,718.79 826.589,719.81 826.485,720.65 C 827.046,720.052 827.328,719.223 827.065,718.419 C 827.789,719.127 827.898,720.495 827.275,721.299 C 828.04,720.753 827.724,719.682 827.635,718.923 C 828.027,719.015 828.41,719.135 828.785,719.283 C 828.711,719.202 826.334,717.934 827.733,718.11 C 829.061,718.277 830.791,719.671 830.085,721.155 C 830.401,720.581 830.855,720.632 830.515,720.075 C 831.05,719.808 831.603,719.761 832.175,719.93 C 830.956,718.578 832.929,719.797 833.374,720.517 C 834.191,721.839 835.632,721.297 836.69,722.118 C 839.515,724.309 836.2,724.319 835.485,726.339 C 835.713,726.087 835.953,725.847 836.205,725.619 C 836.404,726.654 835.281,728.449 834.792,729.31 C 834.49,729.842 834.577,730.298 833.89,730.298 C 833.282,730.298 831.421,729.754 831.025,730.298 C 831.908,730.798 833.334,730.946 833.835,731.954 C 834.406,731.21 835.211,730.682 835.345,731.882 C 836.172,730.972 836.792,730.542 838.005,730.298 C 839.138,730.07 839.578,731.276 840.633,730.259 C 842.456,728.501 845.738,732.865 844.712,733.383 C 843.691,733.899 843.104,734.644 842.273,735.435 C 841.496,736.176 840.15,736.532 839.535,737.183 C 839.094,737.65 838.735,738.293 838.338,738.805 C 837.534,739.843 838.706,740.498 838.619,741.625 C 838.564,742.326 838.826,743.105 839.639,742.66 C 840.611,742.129 841.291,743.408 840.556,744.116 C 839.747,744.895 838.544,745.17 837.761,745.952 C 837.386,746.327 836.734,745.783 836.348,745.745 C 836.004,745.711 835.808,746.383 835.475,746.513 C 832.445,747.697 835.526,749.575 833.922,751.352 C 833.514,751.804 831.887,752.827 832.464,753.616 C 833.445,754.957 834.452,756.314 835.507,757.596 C 836.758,759.117 839.309,761.525 836.918,763.269 C 836.11,763.858 835.5384,764.95039 834.7264,765.53539 C 834.2764,765.86039 832.724,765.485 832.624,765.65 C 832.186,766.369 830.877,766.207 830.853,766.953 C 830.825,767.839 829.259,767.849 828.643,768.144 C 827.466,768.708 826.019,769.448 824.717,768.797 C 823.834,768.356 823.019,769.461 822.1,769.641 C 820.981,769.862 820.962,771.036 821.224,771.954 C 821.602,773.28 820.495,774.301 819.382,773.292 C 818.629,772.608 818.257,771.698 817.567,770.975 C 816.651,770.014 816.063,771.307 815.166,770.981 C 813.934,770.532 813.474,770.896 812.233,770.763 C 811.267,770.659 811.483,771.034 811.107,769.846 C 810.752,768.724 809.306,768.627 808.343,768.819 C 807.068,769.073 807.105,767.957 806.113,767.523 C 805.284,768.356 806.816,770.233 807.321,770.849 C 808.774,772.621 809.973,772.939 810.015,775.465 C 810.029,776.285 810.056,777.098 810.269,777.896 C 810.539,778.904 811.599,779.941 811.457,781.002 C 811.216,782.805 811.21,783.598 813.402,783.653 C 814.037,783.669 816.39164,783.17891 816.18901,784.3288 C 815.81401,786.4568 814.71964,786.2871 813.68121,787.01559 C 812.899,787.56433 811.51939,787.68779 811.20439,788.42279 C 810.84739,789.25679 810.929,789.862 810.355,790.67 C 809.843,791.39 809.164,791.959 808.703,792.722 C 808.72782,790.78012 805.72263,793.76191 805.161,793.872 C 803.80991,794.13683 802.94,794.912 802.131,796.037 C 801.589,796.791 800.459,798.759 799.553,799.055 C 798.957,798.334 799.725,797.428 799.533,796.645 C 799.209,795.321 797.264,796.606 796.752,796.975 C 793.82,799.085 791.58542,796.78642 789.67242,794.27342 C 788.77842,793.09942 788.868,791.729 788.456,790.354 C 787.984,788.775 786.829,786.58 786.901,784.927 C 786.957,783.626 783.48719,779.37327 781.91919,780.55227 C 781.86919,779.09227 784.224,777.538 785.239,776.739 C 787.01,775.344 785.829,773.86 784.58,772.431 C 781.821,769.278 782.61946,763.9019 782.62846,760.1159 C 782.63146,758.7619 783.06701,758.11431 783.89064,757.19554 C 784.41686,756.60854 784.16335,755.40094 784.583,754.634 C 785.03998,753.79884 785.77354,753.20332 786.004,752.469 C 786.41603,751.15611 784.61537,749.8428 783.87528,750.18246 C 782.09747,750.99837 780.65611,751.18278 779.1119,751.07846 C 778.40784,751.0309 776.89738,751.78122 775.28527,751.34481 C 774.14437,751.03596 774.21046,751.48277 772.76936,751.68491 C 772.50812,751.72156 770.41891,751.0551 770.61791,751.1831 C 769.82191,750.6741 769.456,749.544 768.954,748.779 C 768.245,747.697 767.574,746.631 766.926,745.514 C 766.594,744.941 766.128,744.24381 765.307,744.49181 C 763.776,744.95481 762.79891,743.47427 761.35291,743.32327 C 760.42191,743.22527 760.012,743.979 759.163,744.195 C 757.91,744.515 756.734,743.64 755.481,743.87 C 754.678,744.016 754.201,743.498 753.363,743.921 C 752.537,744.338 751.434,743.842 750.97,743.086 C 750.612,742.503 750.43173,742.07781 750.27073,741.43281 C 750.04773,740.53781 748.798,740.63 748.35,739.99 C 747.49,738.761 748.16437,736.153 748.89937,735.14 C 750.29337,733.219 747.909,731.331 747.283,729.939 C 746.623,728.472 747.342,725.065 745.195,725.62 C 745.234,725.1 745.162,724.596 744.979,724.107 C 744.138,724.842 743.252,725.289 742.105,725.256" id="path3410"/>
    <path d="M 834.118,731.304 C 833.387,731.158 832.639,731.052 831.958,730.728 C 832.613,730.354 834.078,730.197 834.118,731.304" id="path3412"/>
    <path d="M 834.837,729.432 C 835.271,728.776 835.961,727.949 836.858,728.352 C 836.561,729.323 835.6,729.15 834.837,729.432" id="path3414"/>
    <path d="M 835.058,731.088 C 834.865,730.872 834.671,730.656 834.478,730.441 C 834.998,730.413 835.504,730.317 835.998,730.153 C 835.963,730.717 835.622,731.056 835.058,731.088" id="path3416"/>
    <path d="M 835.708,729.864 C 835.468,729.768 835.228,729.672 834.988,729.576 C 835.445,729.292 835.925,729.244 836.428,729.431 C 836.188,729.576 835.948,729.72 835.708,729.864" id="path3418"/>
    <path d="M 836.638,730.08 C 836.518,729.561 836.785,729.465 837.438,729.792 C 837.171,729.888 836.904,729.984 836.638,730.08" id="path3420"/>
    <path d="M 836.858,727.704 C 836.618,727.704 836.378,727.704 836.138,727.704 C 836.333,727.172 836.775,726.834 837.358,726.983 C 837.191,727.224 837.024,727.464 836.858,727.704" id="path3422"/>
    <path d="M 837.217,725.832 C 836.476,725.599 836.55,725.359 837.438,725.112 C 837.364,725.353 837.291,725.592 837.217,725.832" id="path3424"/>
    <path d="M 837.358,726.192 C 836.636,726.534 836.111,727.154 835.488,727.631 C 835.745,726.818 836.511,726.244 837.358,726.192" id="path3426"/>
    <path d="M 837.507,725.76 C 837.627,725.429 837.866,725.237 838.227,725.184 C 837.987,725.376 837.748,725.567 837.507,725.76" id="path3428"/>
    <path d="M 804.528,710.496 C 804.288,710.424 804.048,710.352 803.808,710.28 C 804.012,710.043 804.252,709.851 804.528,709.704 C 804.851,710.019 804.851,710.282 804.528,710.496" id="path3598"/>
    <path d="M 813.818,710.712 C 813.632,710.172 810.233,709.423 811.997,708.862 C 812.934,708.564 812.979,709.563 813.528,709.488 C 814.439,709.364 814.592,708.287 815.398,707.975 C 815.887,709.905 815.667,709.708 813.818,710.712" id="path3600"/>
    <path d="M 823.388,713.952 C 823.776,714.41 824.089,714.914 824.328,715.464 C 823.867,715.137 823.395,714.55 823.388,713.952" id="path3602"/>
  </g>
  <g class="landxx coastxx pt eu" id="pt" transform="translate(-29.9017, -45.0745)">
    <path class="landxx pt eu" d="M 1240.127,464.76 C 1240.054,464.52 1239.981,464.281 1239.907,464.04 C 1242.228,463.864 1243.026,460.098 1243.694,458.386 C 1244.28,456.884 1244.607,455.221 1245.04,453.661 C 1245.158,453.239 1245.559,453.167 1245.815,452.874 C 1246.029,452.629 1246.049,451.034 1245.457,452.736 C 1245.718,450.893 1246.177,449.793 1245.781,447.932 C 1245.651,447.323 1244.718,444.123 1245.957,444.095 C 1244.157,444.643 1245.599,441.985 1246.561,441.575 C 1247.649,441.111 1249.181,441.051 1250.067,440.207 C 1249.962,440.808 1250.23,441.156 1250.857,441.071 C 1250.463,441.355 1249.096,443.177 1250.381,443.143 C 1251.434,443.115 1252.515,442.295 1253.558,442.468 C 1254.931,442.695 1255.446,443.239 1256.827,442.511 C 1257.276,441.037 1259.348,442.166 1260.577,442.043 C 1262.012,441.899 1261.204,442.935 1261.3,443.707 C 1261.362,444.207 1263.061,444.509 1263.442,444.856 C 1264.273,445.614 1259.05,449.746 1258.371,449.912 C 1258.955,449.769 1258.698,453.618 1258.707,453.901 C 1258.725,454.432 1258.962,455.41 1258.707,455.903 C 1258.369,456.557 1257.385,456.295 1257.173,457.137 C 1257.047,457.637 1258.834,459.664 1257.227,460.186 C 1256.618,460.384 1256.974,461.171 1256.469,461.468 C 1255.733,461.9 1253.866,461.736 1253.087,461.518 C 1253.353,462.1 1253.903,462.443 1254.352,462.861 C 1254.929,463.398 1254.396,463.987 1254.785,464.676 C 1255.186,465.387 1256.753,466.509 1256.484,467.349 C 1256.023,468.791 1254.439,469.516 1254.033,470.975 C 1253.677,472.257 1254.889,474.643 1256.477,474.119 C 1255.787,475.773 1254.663,475.364 1253.831,477.028 C 1253.42,477.85 1252.634,478.399 1252.251,479.242 C 1251.734,480.38 1252.878,481.502 1252.288,482.642 C 1251.4,484.355 1248.832,484.559 1247.327,483.946 C 1245.516,483.209 1243.106,483.13 1241.498,484.487 C 1241.27,483.748 1241.809,482.756 1242.108,482.116 C 1242.588,481.091 1242.518,479.78 1242.986,478.88 C 1243.541,477.817 1242.801,477.064 1242.862,476.003 C 1242.952,474.44 1243.907,473.379 1242.937,471.814 C 1243.459,473.462 1244.082,471.655 1243.947,471.023 C 1243.632,471.787 1241.242,472.031 1240.417,472.319 C 1240.829,471.623 1240.574,470.828 1240.777,470.087 C 1241.067,470.2 1241.353,470.32 1241.637,470.447 C 1241.697,469.893 1242.125,469.787 1242.516,469.536 C 1243.081,469.171 1242.436,468.202 1242.427,467.711 C 1241.915,468.386 1240.204,470.83 1239.14,470.028 C 1237.78,469.003 1240.122,466.056 1240.127,464.76" id="path3446"/>
    <path d="M 1102.678,471.528 C 1102.067,471.65 1100.786,470.779 1101.924,470.5 C 1102.634,470.326 1103.188,470.871 1102.678,471.528" id="path3614"/>
    <path d="M 1105.268,472.68 C 1104.648,472.32 1103.893,472.371 1103.328,471.888 C 1103.915,470.297 1106.047,471.861 1106.788,472.464 C 1106.265,472.425 1105.759,472.497 1105.268,472.68" id="path3616"/>
    <path d="M 1108.587,471.24 C 1107.591,470.99 1106.092,470.735 1105.637,469.656 C 1106.721,469.945 1107.813,470.397 1108.587,471.24" id="path3620"/>
    <path d="M 1113.478,470.52 C 1112.742,470.394 1111.075,469.541 1112.618,469.008 C 1113.685,468.64 1114.915,470 1113.478,470.52" id="path3622"/>
    <path d="M 1124.358,478.44 C 1123.637,478.335 1120.909,477.926 1122.188,476.711 C 1123.362,478.074 1125.257,476.924 1126.728,477.287 C 1126.784,478.382 1125.076,478.4 1124.358,478.44" id="path3624"/>
    <path d="M 1126.587,485.136 C 1126.22,484.967 1125.907,484.727 1125.648,484.416 C 1126.241,484.259 1126.608,484.506 1126.587,485.136" id="path3626"/>
    <path d="M 1180.868,522.216 C 1180.141,522.07 1179.563,521.696 1179.045,521.179 C 1178.634,520.769 1179.401,520.147 1179.829,520.598 C 1180.387,521.187 1181.19,520.24 1181.834,520.963 C 1182.541,521.759 1181.687,522.413 1180.868,522.216" id="path3628"/>
  </g>
  <path class="landxx co" d="M 667.27529,745.84555 C 667.01129,745.77355 666.74729,745.70155 666.48329,745.62955 C 666.76429,743.90955 668.61029,743.36655 670.01129,744.18955 C 670.77029,742.72155 668.88229,741.33755 669.63029,739.84055 C 670.58829,737.91955 672.74129,738.01755 674.54729,737.71055 C 674.89029,737.65255 674.77029,736.82155 675.23729,737.10255 C 675.58029,737.30855 676.35929,737.48955 676.36429,736.88455 C 676.37429,735.67755 677.95329,734.17655 676.99529,733.39055 C 678.71229,733.34455 679.02929,731.13455 679.53529,729.96255 C 679.93529,729.03655 680.71629,728.39555 681.53129,727.84655 C 680.51729,727.01855 681.72029,726.89355 682.03529,726.04655 C 681.32829,726.07455 680.42529,726.65755 680.16329,725.61455 C 680.71729,725.47055 681.12329,725.06355 681.09929,724.46255 C 680.55429,724.27155 679.44829,725.72955 679.33529,724.60655 C 679.21729,723.42855 679.88929,723.19455 680.81129,722.73355 C 679.36429,722.74755 680.02529,720.82155 680.22329,720.02955 C 680.61029,718.47455 680.27029,716.39055 680.12029,714.80655 C 680.00629,713.60755 679.47429,712.46355 679.39129,711.26855 C 679.34629,710.61455 681.43829,710.35755 681.08329,708.83055 C 680.87129,707.91655 678.76829,705.81755 680.16429,705.28855 C 681.41929,704.81355 680.34229,702.15055 679.79729,701.54255 C 678.69829,700.31555 678.56329,698.48055 677.37629,697.44655 C 676.59529,696.76655 677.37529,695.90955 677.82929,695.28755 C 678.37729,694.53855 677.85029,693.56155 678.50629,692.92555 C 678.86729,693.35255 679.13129,693.83255 679.29829,694.36555 C 680.83429,693.29055 683.01929,691.19055 682.30929,689.10455 C 681.58129,686.96755 680.22329,686.51455 681.53029,684.50055 C 682.27329,685.46155 683.08129,686.77355 683.72229,687.84155 C 684.38329,688.94355 686.00629,689.27955 684.55429,690.62055 C 686.50429,692.85855 685.54729,686.44055 685.27129,685.75355 C 684.87529,684.76355 689.43829,682.48155 690.02129,681.68155 C 690.75429,680.67455 691.27829,678.98455 692.50329,678.45355 C 693.19729,678.15355 694.84129,678.32455 695.23129,677.54555 C 695.78629,676.43855 694.71129,676.03855 694.93729,675.35655 C 695.33829,674.14355 695.50529,672.61955 696.21829,671.54155 C 695.15929,671.43355 695.44629,671.08255 696.16529,670.71455 C 696.82229,670.37855 696.56829,669.47455 696.55029,668.91055 C 696.50829,667.58655 698.73829,667.21555 698.73829,666.75255 C 698.73829,666.05955 701.45429,663.38155 702.08729,663.39855 C 703.34129,663.43055 704.40029,664.54355 705.65029,664.55755 C 704.59429,664.41755 704.74429,665.56255 703.77829,665.56555 C 704.51229,668.28655 706.11829,664.66555 706.48629,663.68455 C 707.40329,661.23555 708.48529,661.41155 711.06429,661.84155 C 714.58329,662.42855 715.70329,660.14055 718.47829,658.47755 C 720.00429,657.56255 724.44329,656.47355 723.07429,653.83055 C 723.65729,653.65655 725.30629,653.13055 724.37029,654.33455 C 725.45029,654.16955 724.77029,653.09655 725.95429,652.55855 C 727.38029,651.91155 728.63329,651.91055 729.85129,652.92855 C 734.00629,656.40155 725.06129,657.80855 723.72129,659.51855 C 723.00529,660.43155 722.73429,661.76455 721.88629,662.55755 C 721.29729,663.10855 720.33929,663.00255 719.81429,663.59955 C 718.82529,664.72355 717.78829,666.43955 717.15229,667.78755 C 716.20329,669.79655 716.16329,671.85855 715.62529,673.97555 C 715.09529,676.05455 713.58429,678.55355 712.20229,680.18355 C 713.34929,680.21655 714.23529,679.76955 715.08229,679.03155 C 715.26529,679.51955 715.33729,680.02455 715.29829,680.54455 C 717.47829,679.98155 716.74729,683.84855 717.85329,685.25455 C 719.43229,687.26255 719.86629,688.44255 718.94229,690.90955 C 718.47829,692.14955 717.73329,694.80255 719.41029,695.43155 C 721.31329,696.14455 720.30529,698.41055 722.26329,698.87255 C 723.05529,699.05955 723.81329,698.55555 724.58729,698.47255 C 725.85029,698.33655 727.00129,699.18355 728.26929,699.06755 C 729.39829,698.96455 730.34429,698.16255 731.49629,698.18455 C 732.52029,698.20455 733.28229,699.19355 734.25029,699.42155 C 734.96129,699.58855 735.62429,698.99155 736.27329,699.56055 C 737.12729,700.30855 737.70529,701.67655 738.33529,702.60355 C 738.91929,703.46355 739.45529,704.33655 739.98129,705.23055 C 740.66129,706.38455 740.94229,706.06155 741.96929,706.35855 C 743.46429,706.79155 744.83929,705.53628 746.78829,706.31828 C 747.38429,706.55728 748.69729,705.97455 749.34129,705.96055 C 750.75229,705.92855 751.95229,706.04055 753.19029,705.28955 C 754.59229,704.43955 756.90829,706.11155 756.17229,707.58655 C 755.75129,708.42955 754.72629,709.07655 754.54229,710.01955 C 754.31529,711.18055 754.78329,711.78955 753.68029,712.54055 C 752.21129,713.54155 752.79329,717.56555 752.64829,719.19755 C 752.47529,721.13655 753.27829,723.19855 753.64229,725.24555 C 753.91629,726.78755 756.87229,728.82455 756.39329,730.39755 C 755.80329,732.33255 752.15629,733.08755 752.23629,735.40755 C 753.55829,734.41455 756.78829,738.04855 757.20629,739.15155 C 757.11429,739.45055 757.13808,739.75255 757.07408,740.05855 C 757.24508,740.61955 757.67829,741.17455 757.67929,741.77555 C 757.68129,743.02455 758.40729,744.34455 758.71629,745.58155 C 758.83829,746.07105 759.48662,747.96568 759.62024,748.72381 C 759.75387,749.48193 759.34891,748.99212 758.74629,749.17955 C 757.73973,749.49262 757.42986,749.01647 757.67731,748.42648 C 758.08286,747.45949 757.50521,741.12944 754.98529,741.22555 C 754.00041,741.26311 751.32829,746.35555 750.57829,743.54755 C 750.29829,742.50255 748.95429,742.02655 749.10229,743.54355 C 749.26929,745.24855 747.10329,744.62455 745.94329,744.62455 C 744.20529,744.62455 742.46629,744.62455 740.72829,744.62455 C 739.57829,744.62455 738.33629,743.94755 737.20529,744.44655 C 735.51529,745.19055 736.86029,748.98455 736.68729,750.38455 C 738.05629,749.45955 741.26729,750.26155 741.88329,751.77255 C 742.11129,752.33155 742.24029,754.21355 741.17429,754.11655 C 740.51529,754.05655 740.11629,753.18055 739.40629,753.30755 C 737.83929,753.58755 735.09929,754.01855 734.98429,755.84755 C 734.84429,758.07555 734.64829,761.19355 736.57229,762.65055 C 737.54329,763.38555 738.67629,764.02055 738.62629,765.41055 C 738.57829,766.74455 739.25729,767.25555 739.77829,768.38155 C 740.96129,770.93655 739.42929,774.43155 739.06829,777.04455 C 738.20029,783.33555 738.34129,790.17855 736.39829,796.25155 C 735.93129,795.09455 735.01429,794.25836 734.18729,793.34136 C 733.24529,792.29736 732.05302,793.33936 730.96302,792.71936 C 729.95202,792.14436 732.49875,787.88728 732.85075,787.28728 C 733.53775,786.11928 735.59793,783.82455 734.55093,782.70355 C 734.09793,782.21955 733.63527,780.94386 732.90329,781.20055 C 731.60059,781.65737 730.12029,779.57792 729.79329,779.30755 C 728.54329,778.27402 727.48629,779.31338 726.16657,779.95709 C 724.90885,780.57057 724.07683,778.91732 722.92166,778.72355 C 722.21698,778.60535 720.21733,781.70749 717.87249,780.99682 C 717.0516,780.74803 716.48929,780.75455 715.47029,780.88655 C 714.06429,781.07055 713.32202,780.92609 712.21602,780.06709 C 711.35302,779.39609 712.26029,777.61155 711.95029,776.59155 C 711.52729,775.19955 709.32129,775.95455 709.07429,774.55755 C 708.89829,773.56255 708.4022,772.42782 708.0022,771.49982 C 707.5822,770.52482 706.31502,770.94264 705.71402,770.08364 C 705.06302,769.15264 704.64529,768.88955 703.77429,768.59655 C 702.66629,768.22455 702.0152,766.72836 701.9312,765.72836 C 701.7722,763.82236 698.28429,760.77155 696.50929,760.25155 C 696.16729,760.15055 695.80929,759.75436 695.50629,760.23636 C 695.17829,760.75736 695.18629,760.76455 694.48729,760.44655 C 693.44629,759.97255 691.84529,759.41155 690.96829,758.72555 C 690.28729,758.19255 687.13138,754.21109 686.75738,757.04609 C 686.56338,758.51909 681.00929,756.64155 679.85629,756.43555 C 677.98629,756.10155 679.36229,754.51955 677.98429,753.93055 C 676.77529,753.41355 677.31029,752.38355 675.75329,752.36455 C 674.52929,752.35055 672.11048,750.99264 671.20548,750.24464 C 669.68248,748.98864 668.44729,747.44655 667.27529,745.84555" id="co"/>
  <g class="landxx coastxx mr" id="mr" transform="translate(-29.9017, -45.0745)">
    <path d="M 1178.278,635.04 C 1178.786,634.399 1178.887,633.593 1178.928,632.808 C 1178.347,633.361 1177.851,634.229 1178.278,635.04" id="path3878"/>
    <path class="landxx mr" d="M 1174.038,624.96 C 1173.834,623.583 1173.9521,622.63231 1174.5022,621.30676 C 1174.656,620.93615 1175.1676,620.04229 1176.1362,619.992 L 1204.0196,619.96014 C 1205.4377,619.21307 1204.4635,613.05087 1204.369,610.679 C 1204.196,608.295 1203.955,606.74215 1205.8662,604.93336 C 1207.4667,603.41862 1207.4139,603.8047 1212.4988,601.72564 L 1213.6158,579.89337 L 1238.4949,579.60173 L 1238.8293,568.53082 C 1248.3013,574.90382 1257.5966,581.468 1266.7336,588.313 C 1262.3396,588.313 1257.911,588.313 1253.518,588.313 L 1258.9415,661.22393 C 1259.3879,662.41646 1260.6158,662.9305 1260.725,663.20913 C 1261.4513,665.06355 1260.4953,665.62421 1260.4268,667.05316 C 1260.3464,668.72766 1260.1813,670.23584 1259.6934,670.4156 L 1243.8056,670.4346 C 1234.7811,670.74688 1230.9156,671.98298 1230.118,669.314 C 1229.733,671.73 1226.1541,672.12502 1224.389,671.424 C 1221.6014,670.31687 1219.8215,670.78355 1218.679,673.922 C 1218.0302,672.85061 1217.0856,671.93698 1216.118,671.004 C 1214.9715,669.89849 1213.8663,669.30561 1213.2574,669.64322 C 1210.9685,670.9123 1214.1631,677.29358 1208.5413,676.89061 C 1208.5708,676.99275 1207.0251,675.79759 1206.6158,675.01507 C 1206.1101,674.04811 1205.5263,674.35622 1204.556,673.528 C 1203.505,672.63089 1203.5789,671.16707 1202.7168,670.59399 C 1200.4084,669.05948 1201.3234,669.04033 1201.109,667.99975 C 1200.8449,666.71837 1199.5407,665.08639 1198.888,665.077 C 1195.4916,665.02812 1195.2482,662.27194 1193.0458,661.21079 C 1191.9347,660.67544 1190.9663,660.39256 1189.4684,660.34362 C 1187.1356,660.26739 1186.3333,662.54721 1180.9789,662.14739 C 1179.8502,662.0631 1180.0008,661.30922 1179.0395,661.58224 C 1178.1062,661.92386 1177.0088,664.92766 1176.6368,665.715 C 1176.6368,660.513 1180.061,656.27 1180.555,651.171 C 1180.852,648.101 1180.428,645.505 1180.043,642.499 C 1179.904,641.415 1179.939,640.226 1179.25,639.307 C 1178.649,638.505 1177.601,637.935 1177.49,636.843 C 1178.292,637.093 1179.175,636.498 1179.08,635.619 C 1178.538,635.764 1178.105,636.171 1177.92,636.699 C 1178.117,635.655 1178.959,634.922 1179.417,634.005 C 1179.713,633.412 1179.284,632.464 1179.631,632.067 C 1181.196,630.275 1178.8,627.097 1178.21,625.323 C 1177.966,625.751 1177.799,626.207 1177.71,626.691 C 1176.122,625.489 1176.501,623.003 1174.9,621.795 C 1174.645,622.857 1174.261,623.887 1174.038,624.96" id="path3882"/>
  </g>
  <g class="landxx ao" id="ao" transform="translate(-29.9017, -45.0745)">
    <path d="M 1390.038,953.923 C 1389.929,950.414 1390.879,946.873 1390.482,943.353 C 1390.42,942.809 1390.019,942.285 1390.245,941.728 C 1390.568,940.932 1391.445,940.469 1391.795,939.641 C 1392.076,938.977 1392.108,938.26 1392.201,937.556 C 1392.285,936.922 1393.121,936.146 1392.912,935.609 C 1392.465,934.463 1394.028,931.381 1394.278,930.094 C 1394.635,928.257 1394.243,926.262 1395.272,924.65 C 1395.957,923.576 1395.523,921.33 1396.314,920.636 C 1397.127,919.923 1398.443,918.376 1398.954,917.415 C 1399.709,915.996 1399.245,913.75 1401.543,913.54 C 1403.16,913.393 1403.302,912.153 1404.101,910.757 C 1405.997,907.443 1406.533,902.306 1405.658,898.635 C 1405.254,896.94 1404.588,895.733 1403.634,894.3 C 1402.64,892.808 1402.608,891.649 1402.061,890.088 C 1401.812,889.376 1401.178,888.839 1401.082,888.078 C 1400.964,887.147 1401.102,886.285 1400.689,885.409 C 1400.077,884.109 1398.864,882.428 1400.688,881.496 C 1400.201,881.849 1399.852,882.393 1399.898,883.016 C 1400.286,882.285 1400.713,881.537 1401.308,880.952 C 1401.784,880.484 1402.553,880.513 1402.636,879.764 C 1402.851,877.812 1402.454,876.021 1401.52,874.315 C 1400.469,872.396 1399.918,870.385 1399.103,868.364 C 1398.742,867.468 1398.77,866.471 1398.509,865.549 C 1398.121,864.178 1396.853,863.252 1396.148,862.058 C 1395.571,861.081 1393.56,857.674 1395.313,857.132 C 1396.122,856.882 1396.957,856.877 1397.787,856.769 C 1398.941,856.618 1399.51,855.581 1400.786,855.458 C 1406.678,854.892 1412.631,855.564 1418.554,855.43 C 1420.802,855.379 1423.105,855.2 1425.351,855.281 C 1427.461,855.358 1427.01,856.797 1427.989,858.151 C 1428.397,858.715 1428.021,860.422 1428.209,861.174 C 1428.546,862.528 1429.209,863.762 1429.711,865.055 C 1430.101,866.057 1429.498,866.934 1430.342,867.832 C 1431.243,868.79 1431.475,869.487 1432.015,870.653 C 1432.348,871.37 1433.899,874.744 1434.92,874.462 C 1435.533,874.293 1438.642,874.532 1438.639,874.525 C 1438.479,874.132 1441.989,873.117 1442.628,873.259 C 1444.426,873.658 1446.076,873.869 1448.208,873.869 C 1447.948,872.611 1448.703,871.375 1448.278,870.129 C 1449.342,870.357 1449.439,869.339 1449.304,868.564 C 1449.117,867.491 1449.948,866.17 1449.578,865.229 C 1451.429,865.229 1453.285,865.265 1455.135,865.249 C 1455.688,865.244 1455.364,864.6 1455.815,864.447 C 1456.458,864.228 1457.201,864.44 1457.859,864.44 C 1457.447,866.503 1456.32,867.68 1458.943,867.68 C 1461.235,867.68 1463.548,867.559 1465.837,867.61 C 1467.737,867.652 1467.39,869.553 1466.948,870.834 C 1466.701,871.553 1466.376,872.287 1466.311,873.053 C 1466.21,874.267 1467.262,875.493 1467.479,876.706 C 1467.934,879.254 1466.988,882.073 1466.79,884.634 C 1466.583,887.305 1467.526,888.355 1469.059,890.355 C 1470.141,891.767 1470.683,895.422 1470.357,897.15 C 1470.243,897.759 1469.668,898.146 1469.455,898.725 C 1469.059,899.801 1470.145,900.834 1469.809,901.95 C 1470.739,901.973 1471.567,900.646 1472.325,900.238 C 1472.976,899.888 1474.943,900.414 1475.664,900.613 C 1476.942,900.966 1478.454,899.528 1479.82,899.581 C 1481.124,899.632 1482.579,900.441 1483.13,898.711 C 1483.7401,899.59164 1483.814,902.39163 1483.6307,903.388 C 1483.3772,904.76655 1483.4004,906.14151 1483.062,907.506 C 1482.6546,909.14843 1483.697,910.647 1483.269,912.251 C 1482.837,913.869 1481.679,915.188 1483.059,916.711 C 1480.222,917.90792 1476.2469,917.33165 1473.2206,917.31472 C 1471.7037,917.30624 1469.9253,916.94057 1468.4168,917.14236 C 1467.0908,917.31975 1467.7529,922.25589 1467.6925,923.836 C 1467.5319,928.03864 1467.7621,931.00498 1467.6014,934.221 C 1467.4413,937.42459 1467.6108,940.69 1467.523,943.901 C 1467.4332,947.18315 1469.714,948.06 1471.604,950.402 C 1473.62,952.9 1475.652,955.407 1478.379,957.18 C 1473.922,958.047 1469.438,958.805 1465.039,959.945 C 1463.286,960.4 1462.611,959.956 1460.949,959.77 C 1460.207,959.687 1459.576,960.043 1458.9,960.267 C 1457.802,960.631 1456.755,960.096 1455.706,959.808 C 1451.974,958.787 1447.978,959.198 1444.182,958.605 C 1442.554,958.351 1441.7244,956.94157 1440.949,955.675 C 1440.2591,954.54808 1438.0538,955.01276 1436.895,955.067 C 1427.5879,955.50266 1418.021,955.09 1408.701,955.09 C 1406.781,955.09 1406.574,954.971 1404.918,953.672 C 1403.253,952.366 1401.9373,950.74182 1399.695,951.736 C 1398.041,952.46933 1396.3251,954.46714 1394.436,953.565 C 1392.1208,952.45938 1391.7903,953.60265 1390.038,953.923" id="path3888"/>
    <path d="M 1392.268,848.013 C 1393.561,847 1395.016,844.403 1396.715,844.062 C 1397.471,843.911 1397.403,842.689 1398.238,842.693 C 1399.175,842.697 1399.776,844.186 1400.398,844.704 C 1398.884,845.741 1397.518,846.663 1396.21,847.971 C 1395.57,848.61 1395.754,848.611 1396.045,849.348 C 1396.455,850.387 1396.225,851.615 1396.13,852.689 C 1396.001,854.147 1393.262,855.332 1393.243,853.423 C 1393.233,852.37 1393.799,852.123 1393.475,850.959 C 1393.237,850.103 1391.935,848.663 1392.988,847.874 C 1392.748,847.919 1392.508,847.966 1392.268,848.013" id="path3890"/>
  </g>
  <g class="landxx coastxx de eu" id="de" transform="translate(-29.9017, -45.0745)">
    <path d="M 1367.638,334.944 C 1367.803,334.181 1367.563,334.133 1366.918,334.799 C 1367.158,334.848 1367.397,334.896 1367.638,334.944" id="path3832"/>
    <path d="M 1383.908,337.104 C 1384.341,337.153 1384.775,337.153 1385.208,337.104 C 1384.911,336.344 1384.141,336.077 1383.408,336.383 C 1383.574,336.624 1383.741,336.864 1383.908,337.104" id="path3834"/>
    <path class="landxx de eu" d="M 1366.5765,332.3355 C 1365.9045,332.6125 1365.8348,333.57077 1365.9718,334.14978 C 1365.9718,333.75778 1366.3315,333.41727 1366.7781,333.24264 C 1366.4563,333.13649 1366.2923,332.82357 1366.5765,332.3355 z M 1367.4837,333.14185 C 1367.2681,333.14041 1367.0772,333.17707 1366.8789,333.24264 C 1367.2555,333.33451 1367.7879,333.27464 1368.0884,333.24264 C 1367.8913,333.15951 1367.6992,333.14328 1367.4837,333.14185 z M 1369.2312,333.03494 C 1368.6027,332.97021 1368.8234,333.17943 1368.1892,333.24264 C 1368.6052,333.45464 1369.06,334.75471 1369.298,335.15771 C 1369.509,335.51271 1370.4901,336.20362 1370.2051,336.66961 C 1369.7361,337.43661 1368.4208,337.01193 1367.8869,337.97993 C 1368.5929,338.27793 1369.2895,338.14693 1370.0035,337.97993 C 1368.4035,338.71593 1370.1297,339.22058 1370.4067,340.09659 C 1369.7297,340.06259 1369.4785,340.48872 1370.0035,341.00373 C 1370.5765,341.56473 1371.4132,341.39149 1372.1202,341.60849 C 1372.8712,341.83949 1373.2528,342.43959 1373.7329,343.01959 C 1374.3089,343.71459 1375.0797,343.79791 1375.7487,344.32991 C 1374.0127,344.14091 1373.552,342.69587 1372.221,341.91087 C 1371.348,341.39687 1370.3348,342.00387 1369.3988,341.91087 C 1368.3088,341.80287 1367.2505,341.4498 1367.0805,342.9188 C 1367.0125,343.5048 1367.4947,343.9665 1367.4837,344.53149 C 1367.3247,344.95249 1367.2623,345.3988 1367.1813,345.84181 C 1367.1113,345.11381 1367.2888,344.41653 1366.4757,344.02753 C 1365.5217,343.57153 1365.1782,344.47426 1365.7702,345.13625 C 1365.2442,345.26925 1364.7931,345.15887 1364.3591,344.83387 C 1366.7281,343.23387 1360.3128,342.98239 1359.6218,343.12039 C 1358.8278,343.27839 1357.4902,344.54764 1357.5051,345.43863 C 1357.5181,346.17764 1360.2801,346.56941 1359.8234,346.34577 C 1358.7044,345.79777 1357.9386,350.03046 1357.7067,350.98227 C 1357.4077,352.20932 1356.373,351.29697 1355.4893,351.88941 C 1354.1137,352.81162 1356.6798,353.24357 1357.0246,353.81671 C 1358.0119,355.45789 1354.592,355.94809 1355.6553,357.17241 C 1356.6936,358.36803 1352.0498,358.08896 1351.4158,357.76495 C 1351.8348,358.91794 1351.069,357.90997 1350.248,358.44098 C 1350.3304,358.70825 1350.0408,358.7749 1350.248,359.14653 C 1351.167,359.46253 1352.2863,361.62283 1351.7833,362.59184 C 1351.3323,363.45984 1349.5692,364.73247 1350.4196,365.9175 C 1350.8753,366.55258 1350.5193,367.43276 1350.6095,367.6661 C 1350.8839,368.37676 1352.5419,368.41245 1351.7599,369.42745 C 1352.7489,369.26044 1352.6606,370.39852 1352.9516,371.0045 C 1349.9792,370.97655 1349.478,373.32542 1352.3162,374.7094 C 1354.3119,375.68254 1352.3171,377.56012 1353.0413,378.41186 C 1353.1517,378.54172 1355.0752,380.91604 1355.5565,380.83702 C 1356.2536,380.72255 1355.7018,380.29811 1356.7166,380.43174 C 1357.2579,380.50302 1356.7187,381.39185 1357.6415,381.15512 C 1357.985,381.067 1358.4747,381.16459 1358.8277,381.12559 C 1360.1107,380.68159 1360.2424,381.37616 1361.4244,381.74817 C 1362.4904,382.08316 1363.8203,382.22696 1364.9282,382.41197 C 1364.0272,383.20696 1363.176,383.99262 1362.444,384.94963 C 1361.605,386.04561 1361.3173,387.71431 1360.7595,388.92843 C 1360.3865,389.74042 1358.9898,395.60014 1361.0802,393.83726 C 1360.3582,394.67625 1363.9945,394.08682 1364.6615,394.12182 C 1365.9605,394.18981 1367.1029,394.14149 1367.3829,393.31547 C 1365.8239,393.89646 1366.3462,391.7977 1368.1892,392.71071 C 1369.4892,393.35573 1372.6354,393.75046 1373.9604,394.52046 C 1374.7734,394.99246 1379.0952,394.70068 1377.9662,396.64165 C 1379.5302,396.59566 1379.6371,395.176 1379.9821,394.02103 C 1380.6471,394.61601 1382.1718,394.60398 1383.0059,394.52499 C 1382.2819,394.88298 1384.1508,395.42313 1384.5178,395.43213 C 1385.3787,395.45213 1385.9242,394.70678 1386.7352,394.62579 C 1388.3062,394.46777 1389.8313,393.25145 1391.5733,393.81944 C 1391.5533,393.55044 1391.5456,393.27209 1391.4725,393.01309 C 1392.2815,393.11308 1393.0685,393.51706 1393.8915,393.51706 C 1394.2905,393.38104 1394.6871,393.29467 1395.1011,393.21468 C 1395.5501,393.26567 1395.737,394.1392 1396.109,394.4242 C 1397.231,395.28518 1397.2201,394.84843 1397.5201,393.81944 C 1397.8191,392.79344 1396.5419,393.18855 1396.613,392.30754 C 1396.6999,391.23255 1396.7384,391.27848 1395.9074,390.39247 C 1395.2814,389.72446 1395.7054,389.20559 1396.4114,388.88057 C 1397.2184,388.50958 1399.8383,388.0617 1399.8383,386.8647 C 1399.8383,384.75771 1401.2828,386.59119 1402.0558,385.65518 C 1403.4608,383.95517 1401.0142,382.77139 1399.9391,382.12741 C 1398.6831,381.37441 1397.6155,380.33158 1396.3106,379.60758 C 1394.6716,378.69657 1394.3428,376.62445 1392.7828,375.77743 C 1394.0128,374.79943 1393.4398,373.9954 1392.2788,373.35839 C 1391.3806,372.86484 1391.1803,371.97289 1390.7669,371.24173 C 1391.4203,371.14636 1391.9268,371.40658 1392.178,372.35046 C 1392.431,369.69545 1395.8437,370.72902 1397.2177,369.62904 C 1399.4107,367.87304 1403.0744,366.90369 1405.6844,365.89968 C 1405.4504,365.67768 1405.2248,365.50492 1404.9788,365.29492 C 1405.9788,365.05892 1406.2212,364.7718 1407.0838,365.9058 C 1409.2044,368.63966 1410.5465,363.57782 1409.4967,362.26554 C 1409.3647,361.44954 1408.0639,361.40922 1407.9909,360.75922 C 1407.8909,359.87622 1407.9591,359.57316 1407.2971,358.84415 C 1406.853,358.35615 1407.5187,357.96347 1407.7002,357.49873 C 1408.7413,354.83348 1406.077,354.11829 1406.7753,353.31834 C 1408.1753,351.71445 1404.3134,351.15559 1403.781,349.64802 C 1404.4923,349.08972 1406.739,345.6415 1405.2812,345.43863 C 1404.9842,344.32963 1404.6499,343.15522 1403.4669,342.71721 C 1402.7389,342.44821 1401.5541,342.2511 1401.451,341.30611 C 1401.398,340.81411 1401.1761,339.29023 1400.4431,339.29024 C 1399.9961,339.29024 1399.0913,339.29921 1399.3344,339.79421 C 1398.5254,339.32321 1396.7772,337.24259 1396.0082,337.17358 C 1394.3932,337.02858 1393.2853,337.16009 1392.0773,338.3831 C 1391.8812,337.0281 1394.1493,336.93179 1395.0003,337.07279 C 1393.3463,336.45679 1393.1077,336.93214 1391.8757,337.87913 C 1391.2947,338.32713 1390.0178,338.82621 1390.3638,339.79421 C 1390.5248,338.16021 1387.2836,339.61558 1386.5336,340.09659 C 1388.0046,339.38159 1386.3744,340.5811 1386.1304,341.30611 C 1385.5315,340.74411 1382.0273,339.9631 1382.8043,341.30611 C 1382.6613,340.78911 1382.3034,340.50997 1381.7963,340.39897 C 1382.7713,339.39797 1383.651,338.73438 1383.913,337.27437 C 1382.586,337.53737 1381.8,338.06797 1380.486,337.47596 C 1379.668,337.10696 1378.3672,336.60735 1377.9662,337.77834 C 1377.8602,337.25834 1377.8358,337.2284 1378.1678,336.77041 C 1377.4958,336.47141 1376.7101,336.50041 1376.0511,336.77041 C 1376.5531,336.53441 1377.6963,335.02593 1376.4543,335.05692 C 1378.4503,334.14992 1371.6583,333.54417 1371.9303,333.69316 C 1371.3258,333.36216 1369.8927,333.10306 1369.2312,333.03494 z " id="path3884"/>
    <path d="M 1396.797,337.752 C 1397.879,337.616 1397.367,336.588 1397.017,336.024 C 1397.778,335.956 1398.496,336.231 1399.177,336.528 C 1398.859,335.641 1397.003,335.9 1398.457,334.944 C 1398.399,336.404 1399.947,335.124 1400.187,335.987 C 1400.304,336.41 1399.567,336.95 1400.083,337.276 C 1400.462,337.516 1400.769,337.548 1400.758,338.111 C 1400.045,337.99 1399.423,337.562 1398.814,338.086 C 1398.28,338.545 1397.106,338.414 1396.797,337.752" id="path3892"/>
    <path d="M 1401.268,340.128 C 1401.275,339.784 1401.226,339.448 1401.118,339.12 C 1401.806,339.995 1403.247,339.981 1403.869,340.941 C 1404.476,341.878 1401.937,341.883 1401.478,341.64 C 1402.502,341.456 1401.543,340.413 1402.708,340.344 C 1402.21,340.388 1401.73,340.316 1401.268,340.128" id="path3894"/>
  </g>
  <g class="landxx coastxx th" id="th" transform="translate(-29.9017, -45.0745)">
    <path d="M 2049.559,733.824 C 2049.188,735.032 2049.397,736.339 2049.838,737.496 C 2050.91,736.634 2050.791,734.499 2049.559,733.824" id="path3902"/>
    <path d="M 2051.348,734.472 C 2051.565,734.975 2051.825,735.462 2051.998,735.984 C 2052.039,735.369 2051.822,734.865 2051.348,734.472" id="path3904"/>
    <path d="M 2055.958,740.088 C 2055.875,739.51 2055.635,739.006 2055.238,738.576 C 2055.3,739.165 2055.54,739.669 2055.958,740.088" id="path3906"/>
    <path d="M 2060.568,746.64 C 2060.059,747.15 2060.006,747.905 2060.858,748.08 C 2060.767,747.599 2060.67,747.119 2060.568,746.64" id="path3908"/>
    <path d="M 2061.798,721.8 C 2061.301,722.253 2061.324,722.685 2061.868,723.096 C 2062.352,722.648 2062.328,722.216 2061.798,721.8" id="path3910"/>
    <path d="M 2061.648,719.928 C 2061.426,720.441 2061.843,720.654 2062.297,720.648 C 2062.102,720.389 2061.885,720.149 2061.648,719.928" id="path3912"/>
    <path d="M 2077.639,700.776 C 2077.875,700.799 2078.112,700.824 2078.348,700.847 C 2078.357,700.264 2077.994,699.8 2077.489,699.551 C 2077.493,699.966 2077.544,700.374 2077.639,700.776" id="path3914"/>
    <path d="M 2079.868,702.936 C 2079.639,703.497 2079.76,703.977 2080.228,704.376 C 2080.37,703.831 2080.25,703.351 2079.868,702.936" id="path3916"/>
    <path class="landxx th" d="M 2034.288,643.824 C 2034.733,644.681 2036.586,644.647 2037.197,644.019 C 2037.934,643.263 2036.477,641.212 2036.668,640.223 C 2036.774,639.672 2037.212,639.292 2037.238,638.711 C 2037.279,637.811 2036.55,637.228 2036.794,636.269 C 2036.948,635.658 2038.06,633.589 2038.476,633.665 C 2039.696,633.891 2040.89,634.419 2042.143,634.145 C 2042.738,634.016 2043.308,633.337 2043.899,633.35 C 2044.5,633.363 2044.736,633.949 2045.244,633.172 C 2045.743,632.407 2044.813,630.951 2045.991,630.648 C 2046.837,630.431 2050.058,631.087 2048.329,628.847 C 2049.563,628.785 2052.881,626.598 2052.869,629.28 C 2053.546,629.092 2053.761,628.123 2054.553,628.36 C 2054.906,628.465 2056.101,630.056 2056.385,630.381 C 2057.081,631.173 2056.103,632.361 2055.898,633.154 C 2055.61,634.272 2056.2904,635.00161 2057.0994,635.66461 C 2057.3832,635.89611 2058.2384,635.66218 2058.5453,635.8035 C 2059.2413,636.12394 2059.0151,635.33064 2059.4523,635.20949 C 2059.8282,635.1053 2059.7468,635.19351 2060.1029,635.08742 C 2060.459,634.98133 2060.9716,634.89045 2061.3059,634.93736 C 2062.8415,635.15285 2062.6014,637.63867 2063.531,639.942 C 2063.8304,640.68375 2063.0331,641.57861 2063.262,642.361 C 2063.7151,643.90992 2062.9223,643.65097 2062.5506,644.67636 C 2062.3963,645.10185 2063.05,646.01371 2063.137,646.59871 C 2063.282,647.57271 2063.5328,648.11529 2063.212,649.054 C 2062.3855,651.47237 2061.9581,653.42131 2064.169,653.30402 C 2064.2764,653.29832 2067.6099,650.2433 2068.129,648.359 C 2069.35,649.09 2069.444,647.103 2070.379,647.085 C 2071.025,647.073 2071.894,648.332 2072.483,648.625 C 2072.944,648.855 2074.009,648.936 2074.291,649.409 C 2075.184,650.903 2076.225,648.711 2077.35,648.935 C 2077.958,649.056 2078.176,646.623 2079.151,646.487 C 2077.466,644.631 2082.6151,645.65875 2083.1731,645.81375 C 2086.3111,646.68575 2087.4231,651.05775 2090.3301,652.58175 C 2092.7321,653.84175 2091.3071,658.52541 2092.02,660.674 C 2092.4794,662.05847 2092.7337,662.56086 2093.6754,663.45736 C 2095.3082,665.01174 2096.1856,665.41423 2097.94,666.215 C 2096.315,667.324 2098.7099,668.11902 2099.472,668.668 C 2101.2498,669.94865 2097.7551,672.45831 2099.81,671.975 C 2099.497,672.688 2098.78,673.38 2099.274,674.189 C 2099.554,674.647 2100.119,674.852 2100.059,675.476 C 2099.894,677.193 2100.568,678.889 2098.855,679.953 C 2097.711,680.663 2097.092,682.149 2096.206,680.55 C 2095.292,678.904 2092.693,680.533 2091.344,680.128 C 2090.408,679.848 2089.51,680.186 2088.588,680.315 C 2087.599,680.453 2087.141,679.844 2086.27,679.773 C 2085.069,679.674 2083.7387,680.222 2082.6367,680.655 C 2080.8967,681.339 2080.502,683.25692 2079.9678,684.81553 C 2079.7344,685.49665 2078.9844,686.18445 2078.332,686.786 C 2077.8871,687.19622 2077.1847,687.22845 2076.769,687.312 C 2077.367,688.7438 2076.709,689.76791 2078.099,691.856 C 2078.8331,692.95877 2079.0282,694.30632 2079.976,695.42575 C 2080.8969,696.51345 2081.2222,697.53287 2081.16,698.95641 C 2081.0611,701.22315 2082.7788,702.19694 2082.599,703.945 C 2082.581,703.064 2081.847,702.584 2081.459,701.88 C 2080.874,700.817 2080.952,699.481 2079.509,699.048 C 2079.542,699.48 2079.566,699.911 2079.579,700.344 C 2079.181,699.389 2075.958,698.94 2077.849,697.68 C 2077.328,697.991 2076.963,697.659 2076.769,697.176 C 2076.769,698.727 2074.934,696.328 2074.61,695.592 C 2074.61,696.956 2073.58,694.192 2072.296,694.779 C 2071.67,695.065 2071.037,695.596 2070.29,695.447 C 2069.375,695.264 2068.318,694.761 2067.371,694.995 C 2065.519,695.453 2066.558,693.533 2066.434,692.392 C 2066.362,691.724 2066.34,691.047 2066.247,690.381 C 2066.12,689.46 2066.797,688.742 2066.619,687.887 C 2065.835,688.515 2063.74,687.935 2063.449,686.951 C 2063.79,688.925 2057.801,687.181 2059.514,690.248 C 2060.368,691.777 2059.825,692.749 2059.52,694.347 C 2059.198,696.043 2060.459,697.628 2060.139,699.335 C 2060.012,700.012 2059.509,700.523 2059.17,701.093 C 2058.753,701.791 2059.017,702.636 2058.538,703.324 C 2057.637,704.622 2057.496,706.617 2057.167,708.131 C 2056.766,709.977 2056.496,711.309 2055.725,713.004 C 2055.45,713.609 2056.03,715.272 2054.878,715.103 C 2056.181,716.677 2054.799,719.59 2055.623,721.493 C 2056.016,722.398 2056.598,723.001 2056.144,723.921 C 2055.631,724.963 2057.526,725.195 2058.072,724.906 C 2059.995,723.888 2061.177,723.867 2061.468,726.48 C 2061.564,727.342 2062.781,733.518 2064.029,731.52 C 2064.791,733.276 2064.917,735.228 2065.497,737.041 C 2066.017,738.662 2066.821,740.373 2066.839,742.104 C 2065.644,741.886 2065.409,736.646 2064.117,737.551 C 2063.188,738.202 2064.573,740.253 2064.937,740.777 C 2065.372,741.4 2065.885,741.88 2066.438,742.387 C 2066.899,742.81 2067.574,742.649 2068.024,743.071 C 2069.067,744.051 2069.268,745.074 2070.991,745.124 C 2072.257,745.16 2073.669,744.658 2074.782,745.509 C 2076.007,746.445 2076.266,748.156 2077.452,749.148 C 2078.706,750.197 2079.515,750.702 2078.888,752.364 C 2078.6,753.126 2078.097,754.674 2077.149,754.875 C 2076.42,755.03 2076.276,753.838 2075.719,753.618 C 2074.259,753.04 2073.448,755.617 2072.514,755.612 C 2070.711,755.603 2071.0953,753.84593 2071.7253,752.85693 C 2072.4113,751.78293 2071.1833,750.79993 2070.3963,750.60593 C 2069.2793,750.33093 2069.789,748.723 2068.36,748.531 C 2067.045,748.355 2066.281,748.3 2065.438,747.198 C 2064.257,745.653 2064.677,748.322 2064.168,749.017 C 2064.058,746.977 2058.961,745.063 2060.998,742.897 C 2059.282,743.927 2059.97,741.254 2059.128,740.738 C 2059.216,741.787 2058.331,741.564 2057.949,740.925 C 2057.457,740.1 2057.567,738.848 2056.853,738.161 C 2056.38,737.706 2055.435,738.299 2055.342,737.552 C 2055.258,736.873 2055.398,736.623 2054.997,736.056 C 2054.393,735.202 2054.465,735.267 2053.569,735.096 C 2053.008,734.99 2053.114,734.069 2052.937,733.682 C 2052.651,733.055 2052.326,732.089 2051.426,732.314 C 2050.466,732.554 2050.863,733.431 2050.557,734.115 C 2048.15,733.619 2048.631,730.017 2048.906,728.279 C 2049.46,724.779 2050.201,721.223 2050.942,717.756 C 2051.229,716.41 2052.338,714.622 2052.322,713.298 C 2052.302,711.742 2052.908,711.653 2053.503,710.267 C 2054.252,708.522 2058.833,703.841 2057.283,701.92 C 2056.62,701.102 2056.067,698.876 2055.829,697.873 C 2055.467,696.345 2054.904,695.826 2054.149,694.534 C 2053.714,693.79 2052.736,691.703 2052.882,690.891 C 2053.236,688.929 2053.152,687.501 2052.352,685.614 C 2051.049,682.543 2048.016,680.687 2046.049,678.061 C 2045.188,676.91 2044.347,675.777 2044.163,674.29 C 2044.059,673.454 2044.305,671.532 2045.516,672.626 C 2045.255,671.608 2046.459,671.826 2046.706,671.112 C 2047.058,670.099 2046.417,667.729 2046.259,666.641 C 2045.997,664.843 2047.405,665.763 2048.04,664.756 C 2048.447,664.11 2048.467,662.713 2047.826,662.185 C 2047.645,662.752 2047.355,663.256 2046.956,663.698 C 2045.742,662.195 2045.393,660.438 2044.876,658.658 C 2044.548,657.525 2042.873,656.404 2042.082,655.522 C 2040.863,654.164 2039.255,652.725 2038.228,651.244 C 2037.66,650.424 2037.924,649.651 2037.654,648.765 C 2037.409,647.966 2036.561,647.332 2036.737,646.418 C 2035.246,646.855 2034.592,644.89 2034.288,643.824" id="path3918"/>
  </g>
  <g class="landxx coastxx au" id="au" transform="translate(-29.9017, -45.0745)">
    <path d="M 2402.638,964.653 C 2403.59,964.072 2402.799,963.148 2402.569,962.423 C 2401.738,963.06 2402.134,963.968 2402.638,964.653" id="path3366"/>
    <path d="M 2419.488,980.573 C 2420.168,980.296 2420.117,979.571 2419.778,979.052 C 2419.678,979.559 2419.581,980.065 2419.488,980.573" id="path3368"/>
    <path d="M 2308.388,902.813 C 2308.871,901.855 2308.733,900.825 2308.388,899.852 C 2307.603,900.91 2308.061,901.697 2308.388,902.813" id="path3294"/>
    <path d="M 2314.078,905.613 C 2314.224,905.469 2314.372,905.326 2314.518,905.182 C 2314.108,904.891 2313.962,905.035 2314.078,905.613" id="path3298"/>
    <path d="M 2334.958,909.213 C 2335.238,908.894 2335.527,908.584 2335.828,908.283 C 2335.26,908.306 2334.939,908.643 2334.958,909.213" id="path3310"/>
    <path d="M 2335.248,905.763 C 2336.042,905.454 2336.808,904.996 2337.269,904.253 C 2336.715,904.957 2335.671,904.917 2335.248,905.763" id="path3312"/>
    <path d="M 2337.908,903.743 C 2339.122,903.042 2340.069,901.547 2340.148,900.142 C 2339.908,901.552 2338.624,902.578 2337.908,903.743" id="path3318"/>
    <path d="M 2291.908,907.053 C 2292.291,907.001 2292.65,906.881 2292.988,906.693 C 2292.786,905.943 2291.838,905.847 2291.838,905.029 C 2291.706,904.699 2291.536,904.391 2291.328,904.103 C 2291.383,903.596 2291.359,903.093 2291.258,902.593 C 2290.141,903.009 2289.958,903.732 2289.854,904.823 C 2289.776,905.633 2288.52,906.131 2288.958,906.984 C 2290.022,906.715 2290.869,906.617 2291.908,907.053" id="path3288"/>
    <path d="M 2297.958,903.173 C 2297.731,902.204 2296.654,902.187 2296.438,903.173 C 2295.705,902.194 2294.614,903.212 2293.674,902.938 C 2293.278,902.823 2292.056,901.305 2291.765,902.204 C 2291.214,903.908 2293.404,908.101 2295.528,907.623 C 2296.735,907.351 2299.166,905.769 2299.55,904.51 C 2299.716,903.97 2300.687,902.293 2299.247,902.013 C 2298.237,901.815 2297.292,901.895 2297.958,903.173" id="path3290"/>
    <path d="M 2347.198,1155.673 C 2347.562,1155.462 2347.853,1155.176 2348.068,1154.814 C 2347.496,1154.784 2347.175,1155.1 2347.198,1155.673" id="path3334"/>
    <path d="M 2348.788,1154.883 C 2349.193,1154.814 2350.462,1153.714 2349.623,1153.379 C 2348.903,1153.092 2347.019,1153.477 2347.269,1154.524 C 2347.86,1154.282 2348.366,1154.402 2348.788,1154.883" id="path3336"/>
    <path d="M 2290.968,1116.723 C 2291.734,1116.513 2292.308,1115.944 2292.94,1115.499 C 2293.954,1114.786 2294.483,1115.264 2295.559,1115.191 C 2296.434,1115.131 2297.158,1114.094 2295.869,1113.918 C 2295.698,1113.974 2295.575,1114.085 2295.501,1114.253 C 2295.004,1114.651 2294.089,1113.904 2293.56,1113.913 C 2295.441,1112.137 2287.747,1113.577 2287.395,1113.676 C 2286.648,1113.885 2284.494,1115.105 2285.648,1116.145 C 2287.169,1117.517 2289.508,1115.08 2290.968,1116.723" id="path3286"/>
    <path d="M 2329.559,1174.394 C 2329.611,1175.178 2330.145,1176.071 2329.128,1176.484 C 2330.385,1176.784 2330.066,1175.264 2330.975,1174.987 C 2332.175,1174.621 2332.828,1176.322 2331.579,1176.774 C 2331.888,1175.17 2330.433,1176.506 2330.517,1177.295 C 2330.618,1178.225 2331.401,1178.07 2331.798,1177.414 C 2331.44,1179.435 2333.523,1175.949 2333.809,1175.474 C 2332.505,1175.347 2333.434,1174.702 2333.997,1174.03 C 2334.69,1173.203 2335.518,1172.507 2336.233,1171.701 C 2336.818,1171.042 2337.173,1170.208 2337.873,1169.645 C 2338.472,1169.163 2339.561,1168.616 2339.719,1167.775 C 2340.077,1168.526 2339.859,1169.388 2339.349,1170.004 C 2340.441,1169.589 2340.665,1168.274 2341.079,1167.334 C 2341.669,1165.998 2342.642,1164.832 2343.402,1163.591 C 2344.424,1161.924 2347.26,1159.072 2345.441,1157.419 C 2344.522,1156.585 2344.237,1158.343 2343.253,1158.155 C 2341.935,1157.904 2341.352,1158.516 2340.244,1159.077 C 2339.73,1159.338 2334.165,1158.48 2335.678,1160.574 C 2334.881,1160.836 2334.642,1160.593 2334.958,1159.844 C 2333.087,1159.74 2331.316,1161.331 2329.561,1159.932 C 2328.939,1159.436 2325.798,1157.932 2326.178,1157.115 C 2325.088,1157.616 2323.057,1157.586 2322.649,1156.175 C 2320.644,1158.18 2320.03,1160.216 2319.149,1162.795 C 2318.451,1164.841 2320.452,1167.985 2318.329,1169.495 C 2318.581,1169.332 2318.845,1169.188 2319.119,1169.065 C 2319.047,1170.102 2319.69,1171.331 2318.549,1172.014 C 2318.795,1171.251 2318.599,1170.472 2318.11,1169.855 C 2315.629,1172.292 2316.996,1175.771 2316.82,1178.645 C 2317.577,1178.2 2318.267,1178.94 2319.049,1178.714 C 2319.462,1179.789 2318.153,1180.063 2317.61,1179.294 C 2317.331,1179.789 2317.067,1180.292 2316.82,1180.804 C 2317.861,1180.61 2319.726,1181.372 2320.56,1180.514 C 2320.675,1183.341 2323.479,1180.409 2324.046,1179.945 C 2324.927,1179.223 2325.103,1178.865 2325.307,1177.908 C 2325.471,1177.143 2326.347,1177.815 2326.11,1178.495 C 2327.543,1178.667 2329.548,1175.634 2329.559,1174.394" id="path3300"/>
    <path d="M 2319.908,1150.563 C 2319.701,1150.915 2319.511,1151.274 2319.338,1151.643 C 2320.992,1151.545 2322.916,1149.342 2322.918,1147.658 C 2322.919,1146.256 2321.712,1147.757 2321.528,1147.903 C 2320.796,1148.486 2319.843,1149.549 2319.908,1150.563" id="path3306"/>
    <path d="M 2348.208,1150.563 C 2348.154,1151.008 2347.701,1152.822 2348.539,1152.848 C 2350.242,1152.904 2350.25,1151.142 2351.158,1150.273 C 2350.173,1150.111 2350.138,1148.973 2349.508,1148.403 C 2348.697,1148.884 2348.554,1149.771 2348.208,1150.563" id="path3338"/>
    <path d="M 2139.048,1095.843 C 2140.988,1095.83 2143.001,1093.706 2143.798,1092.093 C 2144.706,1090.256 2143.619,1087.554 2145.528,1086.193 C 2145.071,1086.555 2144.835,1087.125 2144.958,1087.704 C 2144.976,1087.118 2145.312,1086.841 2145.889,1086.984 C 2145.403,1085.392 2146.66,1083.963 2147.17,1082.651 C 2148.281,1079.792 2147.781,1077.405 2147.299,1074.485 C 2146.852,1071.777 2145.839,1068.797 2146.595,1066.046 C 2147.314,1063.431 2148.565,1059.795 2148.135,1057.098 C 2147.957,1055.98 2147.34,1054.976 2147.304,1053.834 C 2147.267,1052.644 2147.302,1051.597 2146.847,1050.467 C 2146.411,1049.384 2145.407,1048.414 2145.348,1047.219 C 2145.28,1045.847 2145.936,1044.544 2145.993,1043.174 C 2146.104,1040.524 2145.26,1037.741 2143.827,1035.528 C 2143.223,1034.595 2142.499,1033.605 2142.276,1032.493 C 2142.071,1031.473 2142.95,1030.973 2142.869,1030.033 C 2143.101,1030.758 2142.87,1031.484 2142.719,1032.192 C 2143.283,1031.586 2143.496,1030.77 2143.519,1029.963 C 2143.377,1030.973 2143.986,1031.899 2143.799,1032.913 C 2144.152,1030.95 2144.025,1034.032 2143.949,1034.423 C 2144.951,1032.79 2144.412,1034.472 2144.518,1035.143 C 2146.424,1034.116 2147.313,1033.201 2145.723,1031.354 C 2144.636,1030.091 2143.847,1027.018 2145.239,1025.713 C 2145.891,1026.293 2146.343,1027.267 2146.542,1028.102 C 2146.813,1029.239 2145.813,1030.011 2145.89,1031.114 C 2146.802,1031.167 2147.269,1030.308 2147.4,1029.534 C 2147.552,1030.678 2147.17,1031.964 2147.798,1033.017 C 2148.148,1033.601 2149.14,1032.646 2149.266,1032.356 C 2150.101,1030.444 2150.786,1028.73 2149.767,1026.806 C 2148.9,1025.166 2148.978,1023.269 2147.882,1021.776 C 2147.464,1021.207 2147.69,1020.733 2147.69,1020.093 C 2147.691,1018.7 2146.992,1017.454 2146.81,1016.093 C 2146.526,1013.956 2147.795,1011.493 2148.984,1009.794 C 2150.372,1007.81 2151.354,1006.353 2151.87,1003.893 C 2152.107,1002.76 2151.29,1001.899 2151.29,1000.808 C 2151.29,999.658 2152.286,998.727 2152.761,997.751 C 2153.461,996.312 2154.104,993.861 2155.83,993.314 C 2156.029,994.195 2155.627,995.008 2155.318,995.8 C 2154.883,996.914 2155.171,998.149 2154.89,999.293 C 2157.243,998.481 2157.282,997.255 2158.419,995.333 C 2159.589,993.351 2161.537,992.485 2163.634,991.815 C 2166.596,990.87 2168.706,988.836 2171.081,986.908 C 2171.699,986.406 2172.427,985.099 2173.208,984.998 C 2174.956,984.773 2176.323,984.089 2177.499,982.722 C 2177.183,984.701 2179.312,982.956 2180.113,983.065 C 2180.872,983.168 2181.185,983.953 2182.044,983.953 C 2183.141,983.953 2184.179,983.574 2185.113,983.023 C 2187.137,981.83 2188.109,980.478 2190.604,980.483 C 2191.625,980.485 2192.731,980.464 2193.559,979.772 C 2194.497,978.988 2194.846,977.568 2196.294,977.568 C 2198.496,977.568 2199.848,977.936 2202.04,977.356 C 2204.576,976.685 2206.88,976.087 2209.201,974.837 C 2211.072,973.83 2217.983,969.667 2216.597,966.532 C 2217.382,966.613 2217.598,966.212 2217.796,965.554 C 2218.106,964.532 2219.052,964.181 2219.89,963.656 C 2221.112,962.89 2223.84,961.683 2222.574,960.154 C 2221.495,958.852 2222.175,955.185 2222.881,953.84 C 2223.283,953.072 2225.215,950.19 2226.309,951.483 C 2225.629,950.143 2226.641,950.293 2227.505,949.835 C 2228.565,949.274 2228.602,947.2 2230.059,946.512 C 2230.661,947.32 2228.754,949.168 2230.708,949.102 C 2229.34,951.003 2233.099,954.76 2232.868,956.813 C 2233.984,955.369 2233.028,953.038 2233.948,951.623 C 2234.768,952.032 2235.102,952.921 2235.677,953.573 C 2235.568,952.307 2235.376,951.395 2236.608,950.613 C 2235.638,950.944 2233.613,949.008 2234.448,948.092 C 2233.879,948.071 2233.561,947.734 2233.588,947.162 C 2234.645,947.995 2233.963,946.967 2234.909,946.79 C 2236.046,946.577 2234.823,945.309 2234.597,944.852 C 2235.59,944.228 2237.422,944.971 2235.887,945.292 C 2236.459,945.5 2236.829,945.968 2236.537,946.582 C 2237.201,946.404 2237.311,945.461 2237.977,945.397 C 2238.85,945.313 2238.877,945.907 2239.421,946.351 C 2240.102,946.906 2240.513,945.907 2241.238,946.334 C 2242.051,946.813 2242.987,946.208 2243.806,946.722 C 2243.113,945.86 2242.464,946.31 2241.576,946.002 C 2240.342,945.575 2241.255,944.666 2241.356,943.772 C 2241.517,944.019 2241.687,944.259 2241.866,944.493 C 2242.896,943.791 2242.437,942.238 2243.666,941.692 C 2243.049,941.459 2242.609,941.841 2242.437,942.413 C 2241.55,941.477 2240.926,939.506 2242.422,938.804 C 2243.107,938.483 2245.204,937.84 2244.096,936.793 C 2244.884,936.831 2245.463,937.391 2246.07,937.813 C 2246.257,938.094 2246.438,938.377 2246.617,938.664 C 2246.999,938.768 2247.382,938.865 2247.767,938.954 C 2246.651,938.293 2247.483,937.021 2245.827,937.004 C 2246.126,936.487 2246.556,936.128 2247.117,935.924 C 2246.572,936.034 2246.092,936.274 2245.677,936.644 C 2245.616,935.677 2247.972,933.807 2247.837,935.424 C 2247.916,933.78 2249.046,935.856 2250.067,935.644 C 2249.685,934.966 2248.831,934.252 2249.426,933.415 C 2247.256,933.299 2249.809,931.373 2249.926,930.604 C 2250.754,931.481 2251.049,929.971 2251.947,930.535 C 2252.18,929.708 2252.118,928.847 2252.227,928.005 C 2253.303,928.228 2252.864,930.208 2252.227,931.105 C 2253.571,929.568 2252.988,929.46 2254.176,931.255 C 2254.206,930.477 2255.053,930.501 2255.186,929.815 C 2255.268,929.394 2255.586,928.293 2255.756,927.935 C 2256.031,927.355 2257.071,927.397 2256.482,926.535 C 2256.28,926.356 2256.052,926.31 2255.797,926.4 C 2255.54,926.415 2254.815,925.003 2256.187,926.066 C 2255.876,925.191 2257.451,924.531 2256.477,925.925 C 2258.07,924.47 2256.103,927.834 2257.987,927.365 C 2257.603,926.485 2258.457,925.719 2259.287,925.705 C 2258.808,926.351 2259.325,926.944 2259.717,927.434 C 2260.404,926.662 2261.828,926.352 2262.238,925.344 C 2262.561,924.55 2261.285,924.19 2262.741,923.589 C 2263.566,923.249 2263.954,925.329 2265.254,925.365 C 2266.365,925.396 2266.49,925.956 2267.259,926.763 C 2268.235,927.789 2268.768,929.135 2269.729,930.173 C 2270.304,930.795 2271.178,931.557 2271.098,932.5 C 2271.009,933.544 2270.378,934.487 2270.202,935.528 C 2270.038,936.5 2270.029,937.947 2269.229,938.663 C 2270.139,938.213 2269.516,937.11 2270.159,936.642 C 2271.024,936.012 2271.286,937.389 2271.388,937.942 C 2271.799,936.939 2270.561,935.972 2271.029,934.993 C 2271.604,933.791 2272.23,935.078 2273.119,934.703 C 2272.265,931.241 2279.11,932.463 2278.369,935.852 C 2279.53,935.409 2279.064,934.043 2279.089,933.123 C 2280.699,933.261 2281.21,936.458 2282.549,936.143 C 2282.244,935.386 2281.439,933.984 2282.38,933.309 C 2283.039,932.836 2283.921,933.226 2284.56,932.614 C 2283.301,932.784 2282.54,932.402 2282.12,931.174 C 2282.683,931.144 2283.187,930.953 2283.63,930.604 C 2282.514,930.947 2280.02,930.065 2280.932,928.538 C 2281.303,927.915 2281.477,926.982 2282.04,926.495 C 2282.536,926.066 2283.649,926.361 2283.915,925.64 C 2284.344,924.469 2284.55,923.01 2285.138,921.924 C 2285.682,920.919 2286.12,922.048 2286.83,921.476 C 2287.29,921.105 2288.45,919.78 2289.1,920.165 C 2288.504,919.542 2287.767,918.41 2287.941,917.498 C 2288.234,915.954 2288.856,917.186 2289.539,916.078 C 2289.995,915.339 2289.809,914.596 2290.688,914.084 C 2291.321,913.716 2291.739,914.784 2292.481,914.615 C 2291.696,914.302 2291.447,912.128 2292.34,911.815 C 2293.284,911.484 2293.932,913.69 2294.79,913.894 C 2293.878,913.213 2293.729,910.724 2295.43,911.524 C 2295.64,910.915 2295.196,910.179 2295.964,909.87 C 2296.75,909.554 2297.033,910.708 2297.521,909.505 C 2298.714,912.272 2305.947,909.406 2306.09,911.085 C 2306.086,909.894 2307.331,909.033 2308.39,909.725 C 2307.273,908.541 2308.558,906.753 2308.25,905.406 C 2307.781,903.349 2305.802,903.64 2304.29,904.326 C 2304.335,903.346 2303.106,902.904 2302.556,902.454 C 2302.2,902.162 2303.758,901.202 2303.861,900.935 C 2304.573,901.491 2304.721,902.363 2304.941,903.175 C 2305.393,902.492 2305.481,901.666 2305.081,900.935 C 2305.502,901.774 2306.047,901.838 2306.521,901.005 C 2306.772,902.089 2308.435,903.33 2309.041,904.325 C 2309.494,903.74 2310.888,903.678 2310.481,902.665 C 2311.397,903.303 2311.777,906.083 2312.861,906.074 C 2313.821,906.065 2315.13,905.949 2315.021,907.054 C 2316.097,907.095 2316.874,905.754 2317.97,906.124 C 2316.105,907.518 2320.298,907.208 2319.7,909.144 C 2320.408,907.842 2320.574,908.778 2321.556,908.963 C 2322.672,909.172 2323.284,908.294 2324.31,908.134 C 2324.568,910.425 2326.773,910.103 2328.27,910.584 C 2327.127,908.772 2332.111,909.656 2330.499,908.134 C 2331.429,907.901 2332.881,907.661 2332.879,906.414 C 2333.54,907.336 2332.262,907.956 2331.65,908.353 C 2333.02,908.353 2331.96,908.809 2331.457,909.379 C 2330.976,909.925 2330.807,910.5 2331.22,911.163 C 2331.718,910.461 2332.36,909.654 2333.311,909.642 C 2332.354,910.657 2333.541,910.083 2333.503,911.243 C 2333.491,911.603 2333.547,912.148 2333.979,912.268 C 2334.885,912.522 2335.797,911.241 2336.111,910.582 C 2333.522,910.54 2337.657,907.966 2338.131,907.492 C 2336.882,908.148 2337.876,910.461 2338.921,910.802 C 2338.269,909.623 2339.689,909.82 2340.052,910.32 C 2340.361,910.743 2340.082,911.403 2339.891,911.806 C 2339.22,913.22 2337.88,915.186 2336.402,915.772 C 2336.866,916.202 2337.225,916.705 2337.482,917.283 C 2337.228,917.098 2336.964,916.928 2336.692,916.772 C 2336.436,917.891 2336.357,918.91 2335.181,919.442 C 2335.165,918.961 2335.189,918.481 2335.251,918.003 C 2334.707,918.334 2334.228,918.741 2333.812,919.223 C 2333.723,917.494 2332.662,919.174 2332.662,919.803 C 2332.446,918.244 2330.832,920.119 2331.802,921.023 C 2329.645,920.612 2331.19,924.73 2332.372,922.753 C 2331.304,924.983 2330.262,926.982 2328.832,928.996 C 2328.224,929.854 2327.609,930.748 2326.836,931.468 C 2325.949,932.295 2325.459,931.98 2326.197,933.431 C 2327.199,935.401 2329.502,936.087 2330.799,937.789 C 2331.559,938.785 2332.051,939.707 2333.137,940.392 C 2333.898,940.872 2334.639,941.477 2334.313,942.484 C 2335.136,942.167 2335.977,942.84 2336.552,941.904 C 2337.341,943.139 2338.706,943.742 2339.942,944.421 C 2341.522,945.291 2342.243,946.514 2343.326,947.886 C 2345.197,950.255 2347.923,949.412 2350.222,951.066 C 2351.448,951.948 2350.52,953.753 2351.634,954.649 C 2352.242,955.14 2353.058,955.1 2353.632,955.687 C 2354.234,956.301 2354.96,956.703 2355.657,957.199 C 2357.787,958.713 2362.425,957.258 2363.684,954.973 C 2365.491,951.695 2367.821,948.447 2369.15,944.92 C 2369.915,942.894 2370.489,940.803 2371.241,938.773 C 2372.042,936.614 2372.974,934.977 2372.819,932.6 C 2372.697,930.741 2372.813,929.624 2373.481,927.938 C 2374.157,926.231 2372.602,924.726 2373.497,922.855 C 2374.223,921.338 2375.704,920.127 2375.242,918.331 C 2374.713,916.271 2376.469,916.172 2376.873,914.475 C 2377.309,914.885 2377.525,915.388 2377.523,915.986 C 2377.502,915.409 2377.866,914.972 2378.382,914.765 C 2378.094,914.728 2376.58,913.086 2376.432,912.745 C 2376.19,912.955 2375.953,913.171 2375.722,913.394 C 2375.558,912.575 2376.12,911.468 2376.666,910.896 C 2376.869,910.649 2377.128,910.521 2377.442,910.514 C 2377.041,909.721 2377.528,908.945 2378.022,908.354 C 2378.183,908.914 2378.588,909.199 2379.171,909.075 C 2379.066,906.05 2380.898,903.778 2381.261,900.865 C 2381.378,899.921 2381.376,899.298 2382.407,898.881 C 2383.377,898.489 2383.84,896.774 2385.071,897.625 C 2384.713,898.099 2384.402,898.602 2384.14,899.135 C 2384.393,898.974 2384.657,898.83 2384.93,898.705 C 2385.23,899.62 2386.141,900.057 2385.97,901.09 C 2385.895,901.513 2385.939,901.925 2386.102,902.327 C 2386.391,903.266 2385.941,903.261 2385.941,904.099 C 2385.941,904.989 2385.757,905.905 2385.91,906.79 C 2386.108,907.938 2387.891,907.809 2388.681,908.214 C 2387.243,909.276 2386.781,910.388 2387.91,911.641 C 2389.113,912.974 2388.912,914.095 2388.532,915.774 C 2390.134,915.46 2389.337,922.228 2388.995,923.037 C 2388.579,924.02 2389.171,924.768 2389.475,925.682 C 2389.832,926.751 2389.278,927.891 2389.761,928.945 C 2390.618,930.812 2392.967,929.64 2393.211,928.006 C 2394.669,929.823 2395.36,925.662 2396.026,929.583 C 2396.179,930.488 2398.024,931.576 2398.629,932.429 C 2398.705,932.537 2400.801,933.896 2400.991,933.916 C 2400.525,934.66 2399.147,936.115 2400.631,936.645 C 2398.652,939.046 2401.046,941.668 2400.118,944.27 C 2399.548,945.88 2399.22,946.702 2400.06,948.175 C 2400.485,948.918 2401.75,950.332 2401.201,951.265 C 2401.641,950.892 2402.147,950.676 2402.721,950.616 C 2402.338,952.136 2402.093,952.748 2402.495,954.333 C 2402.796,955.515 2402.782,956.758 2402.602,957.96 C 2402.405,959.292 2401.707,960.154 2401.182,961.341 C 2400.546,962.777 2402.172,963.097 2402.065,964.246 C 2401.98,965.152 2403.254,964.464 2402.854,965.617 C 2402.505,966.62 2401.426,967.417 2402.051,968.563 C 2402.419,969.236 2403.628,970.672 2404.508,970.591 C 2405.326,970.516 2406.039,972.542 2406.89,970.925 C 2406.795,972.814 2409.877,973.116 2409.561,971.855 C 2410.052,972.919 2409.754,975.837 2411.14,976.255 C 2411.167,975.69 2411.506,975.349 2412.071,975.316 C 2411.641,976.807 2413.299,976.918 2414.339,977.302 C 2415.193,977.617 2414.181,979.477 2415.852,978.973 C 2416.217,978.863 2415.933,978.232 2416.279,978.09 C 2417.008,977.791 2416.92,978.938 2416.996,979.244 C 2417.161,979.901 2417.91,979.533 2418.212,980.13 C 2418.555,980.809 2418.511,981.613 2418.842,982.296 C 2418.467,981.546 2417.464,981.415 2417.098,982.254 C 2416.634,983.318 2418.043,984.467 2417.332,985.465 C 2421.23,984.065 2418.393,991.058 2421.072,991.005 C 2420.15,992.074 2420.19,993.799 2420.316,995.115 C 2420.423,996.22 2421.499,998.141 2420.641,999.145 C 2422.055,998.568 2422.241,998.993 2422.881,1000.225 C 2423.525,998.948 2422.752,997.528 2423.741,996.335 C 2424.919,997.388 2425.501,998.982 2426.911,999.795 C 2427.116,999.394 2426.941,998.599 2427.271,998.285 C 2427.895,997.69 2428.375,998.147 2428.439,998.849 C 2428.604,1000.717 2428.409,1001.883 2427.799,1003.633 C 2427.438,1004.673 2427.723,1005.733 2427.384,1006.81 C 2426.703,1008.974 2428.194,1007.589 2428.85,1008.935 C 2428.655,1008.414 2428.728,1007.934 2429.071,1007.496 C 2430.667,1008.008 2430.559,1011.841 2428.991,1009.225 C 2429.165,1010.374 2432.451,1014.185 2432.451,1012.175 C 2434.089,1013.496 2433.69,1015.427 2434.541,1017.145 C 2434.942,1017.956 2435.817,1018.129 2436.341,1018.805 C 2436.957,1019.601 2436.541,1020.566 2436.682,1021.479 C 2436.988,1023.466 2438.964,1022.709 2438.721,1024.635 C 2438.628,1025.362 2437.925,1025.867 2437.779,1026.583 C 2437.582,1027.551 2438.52,1028.112 2438.072,1029.246 C 2438.295,1028.989 2438.535,1028.748 2438.792,1028.525 C 2439.883,1029.421 2438.154,1030.643 2437.852,1031.475 C 2437.311,1032.962 2436.84,1034.767 2436.583,1036.33 C 2436.389,1037.516 2436.699,1038.575 2435.687,1039.374 C 2434.591,1040.24 2434.679,1041.066 2435.445,1042.14 C 2435.965,1042.869 2435.851,1044.896 2435.668,1045.765 C 2435.545,1046.347 2435.303,1046.955 2435.327,1047.56 C 2435.352,1048.22 2435.853,1048.618 2435.692,1049.332 C 2434.909,1052.784 2433.293,1054.989 2431.31,1057.837 C 2429.177,1060.9 2428.001,1064.416 2425.646,1067.397 C 2423.662,1069.908 2423.52,1072.913 2421.58,1075.409 C 2420.638,1076.621 2419.843,1077.974 2418.842,1079.135 C 2417.851,1080.284 2416.291,1080.895 2415.508,1082.222 C 2415.006,1083.074 2414.769,1084.647 2414.092,1085.335 C 2413.688,1085.745 2412.995,1085.711 2412.535,1086.038 C 2411.686,1086.639 2410.849,1087.536 2409.772,1087.705 C 2411.632,1087.883 2408.619,1088.601 2408.136,1088.753 C 2406.859,1089.153 2405.7,1090.322 2404.882,1091.338 C 2404.301,1092.058 2403.318,1092.349 2402.714,1093.056 C 2401.875,1094.04 2402.073,1095.098 2400.341,1094.906 C 2400.218,1095.552 2400.385,1095.769 2400.841,1095.555 C 2400.302,1096.329 2399.889,1097.214 2399.349,1097.972 C 2398.922,1098.575 2398.288,1098.374 2397.855,1099.09 C 2396.968,1100.559 2395.247,1101.034 2394.465,1102.601 C 2394.069,1103.394 2394.204,1103.141 2393.564,1103.27 C 2393.543,1103.275 2393.132,1104.668 2392.921,1104.9 C 2392.556,1105.302 2390.433,1107.177 2391.631,1107.645 C 2391.278,1108.216 2390.812,1108.869 2390.692,1107.645 C 2390.187,1108.101 2390.4,1109.063 2389.625,1109.177 C 2388.664,1109.319 2388.178,1109.6 2387.512,1110.354 C 2385.84,1112.248 2384.016,1114.068 2382.483,1116.076 C 2381.172,1117.794 2380.251,1119.856 2378.924,1121.589 C 2378.169,1122.574 2376.671,1125.247 2375.572,1125.726 C 2376.61,1125.956 2375.656,1127.535 2375.369,1127.984 C 2375.103,1128.396 2374.801,1128.807 2374.439,1129.142 C 2373.877,1129.663 2373.284,1129.213 2372.739,1129.496 C 2371.061,1130.368 2370.23,1131.446 2368.113,1131.699 C 2365.864,1131.968 2363.592,1131.747 2361.341,1131.965 C 2356.788,1132.403 2352.984,1135.168 2349.337,1137.69 C 2347.672,1138.841 2345.93,1139.413 2343.903,1139.581 C 2341.922,1139.745 2342.446,1139.364 2342.091,1141.277 C 2342.535,1140.901 2343.038,1140.638 2343.601,1140.487 C 2343.349,1141.459 2342.456,1142.787 2341.522,1143.216 C 2341.474,1142.577 2341.718,1141.545 2341.276,1141.001 C 2340.59,1140.155 2340.17,1141.09 2339.351,1141.136 C 2339.552,1140.404 2339.477,1139.642 2339.572,1138.897 C 2338.643,1139.098 2337.991,1139.608 2337.824,1138.389 C 2337.643,1137.069 2338.798,1137.038 2339.227,1135.957 C 2339.844,1134.404 2336.298,1136.661 2336.116,1136.816 C 2334.662,1138.051 2334.088,1137.565 2333.312,1136.166 C 2334.284,1137.245 2335.496,1136.028 2336.261,1135.376 C 2337.514,1134.309 2336.916,1133.615 2336.982,1132.206 C 2335.093,1132.527 2333.416,1133.848 2331.801,1134.796 C 2332.534,1135.033 2333.294,1134.643 2333.961,1135.156 C 2333.304,1136.09 2331.944,1135.825 2331.001,1136.166 C 2329.646,1136.656 2328.103,1137.457 2326.972,1138.354 C 2325.838,1139.253 2324.643,1139.715 2323.368,1140.36 C 2322.313,1140.893 2322.1,1140.005 2320.994,1139.444 C 2319.08,1138.474 2317.873,1136.393 2315.608,1136.706 C 2314.346,1136.88 2313.789,1135.438 2312.497,1135.687 C 2311.775,1135.826 2311.608,1136.346 2311.007,1136.628 C 2310.564,1136.836 2309.955,1136.57 2309.482,1136.666 C 2311.421,1134.439 2305.695,1133.877 2305.013,1132.907 C 2304.328,1131.934 2303.979,1130.639 2303.974,1129.458 C 2303.97,1128.711 2302.843,1128.241 2303.029,1126.909 C 2303.355,1124.555 2305.522,1123.458 2306.022,1121.336 C 2306.822,1117.939 2305.599,1115.046 2303.501,1112.406 C 2305.302,1113.621 2306.237,1115.834 2306.381,1117.945 C 2306.7,1116.518 2306.235,1115.301 2305.415,1114.136 C 2304.873,1113.364 2303.8,1111.842 2305.591,1111.975 C 2305.551,1112.48 2305.501,1112.983 2305.442,1113.486 C 2306.321,1113.328 2306.182,1112.833 2306.25,1112.14 C 2306.332,1111.306 2307.719,1111.514 2307.313,1110.675 C 2306.854,1109.726 2303.782,1110.841 2304.512,1111.975 C 2304.149,1111.867 2301.732,1111.593 2303.792,1112.255 C 2302.068,1112.081 2298.857,1114.388 2297.382,1112.835 C 2298.121,1111.812 2299.468,1111.262 2300.519,1110.634 C 2301.464,1110.07 2302.041,1108.857 2302.515,1107.913 C 2303.774,1105.407 2303.597,1102.674 2302.993,1099.944 C 2302.343,1100.342 2302.312,1101.065 2301.806,1101.571 C 2301.243,1102.135 2300.599,1102.621 2300.122,1103.269 C 2299.148,1104.596 2298.287,1106.067 2297.428,1107.471 C 2296.487,1109.012 2296.271,1108.731 2294.637,1108.571 C 2293.087,1108.42 2291.46,1110.016 2289.752,1109.955 C 2290.016,1109.006 2291.037,1107.928 2291.762,1107.295 C 2292.982,1106.228 2294.561,1107.617 2295.748,1106.5 C 2297.819,1104.552 2297.691,1101.093 2299.619,1098.929 C 2300.689,1097.728 2301.327,1096.908 2302.739,1096.098 C 2304.395,1095.149 2303.78,1093.999 2304.302,1092.488 C 2304.595,1091.637 2305.534,1091.497 2306.176,1091.026 C 2306.382,1090.876 2306.199,1089.736 2306.194,1089.545 C 2306.179,1089.046 2306.994,1086.716 2306.228,1086.372 C 2305.914,1086.231 2305.467,1088.588 2305.35,1088.819 C 2304.729,1090.043 2302.849,1090.832 2301.901,1091.873 C 2300.974,1092.891 2300.148,1093.995 2299.181,1094.976 C 2298.763,1095.4 2298.345,1096.088 2297.742,1096.276 C 2297.221,1096.437 2296.734,1095.981 2296.232,1096.136 C 2295.298,1096.427 2294.265,1097.61 2293.197,1098.001 C 2291.644,1098.571 2290.467,1099.807 2289.348,1100.972 C 2288.938,1101.399 2287.408,1103.443 2286.871,1103.331 C 2285.345,1103.014 2285.07,1105.25 2284.201,1105.926 C 2284.702,1105.717 2285.208,1105.524 2285.721,1105.346 C 2285.097,1105.992 2285.01,1106.956 2284.351,1107.576 C 2283.867,1106.099 2282.515,1107.071 2281.925,1105.892 C 2281.506,1105.057 2281.057,1103.214 2279.741,1103.975 C 2280.075,1103.417 2280.52,1102.925 2281.041,1102.536 C 2280.863,1103.487 2282.056,1104.278 2282.245,1102.992 C 2282.461,1101.515 2282.841,1100.039 2282.841,1098.537 C 2282.841,1097.131 2280.919,1096.236 2281.53,1094.77 C 2281.77,1094.194 2282.163,1093.583 2281.839,1092.95 C 2281.581,1092.449 2281.267,1092.159 2281.681,1091.596 C 2280.989,1092.174 2280.484,1091.468 2279.718,1091.67 C 2278.624,1091.959 2279.089,1091.189 2278.881,1090.726 C 2278.774,1090.488 2278.216,1089.235 2278.379,1089.03 C 2278.603,1088.748 2280.035,1086.635 2279.601,1088.426 C 2280.285,1087.932 2280.978,1086.838 2280.402,1086.009 C 2279.729,1085.04 2278.63,1085.869 2277.801,1086.046 C 2277.957,1085.558 2278.713,1084.895 2278.664,1084.373 C 2278.6,1083.682 2277.858,1083.324 2277.792,1082.596 C 2277.758,1082.232 2275.406,1083.26 2275.323,1083.25 C 2273.843,1083.075 2272.696,1081.796 2271.38,1081.21 C 2269.911,1080.557 2269.807,1082.148 2268.679,1081.948 C 2265.81,1081.437 2264.637,1077.484 2261.606,1077.151 C 2260.589,1077.039 2259.808,1077.634 2258.906,1077.956 C 2257.808,1078.347 2256.568,1077.868 2255.441,1077.858 C 2252.606,1077.833 2249.274,1077.969 2246.502,1078.573 C 2241.371,1079.692 2236.826,1082.641 2231.609,1083.687 C 2229.098,1084.19 2226.842,1084.1 2224.318,1083.868 C 2221.241,1083.585 2219.613,1084.877 2216.998,1086.205 C 2214.859,1087.291 2212.377,1088.731 2210.062,1089.351 C 2207.58,1090.015 2205.929,1090.167 2204.331,1092.391 C 2203.109,1094.091 2201.898,1095.711 2200.215,1096.996 C 2199.913,1097.226 2198.57,1098.404 2198.165,1098.208 C 2197.06,1097.674 2196.479,1098.742 2195.352,1098.936 C 2196.041,1097.162 2190.629,1097.826 2189.85,1098.135 C 2188.549,1098.653 2188.326,1099.465 2187.78,1097.683 C 2187.766,1097.637 2184.156,1097.405 2183.704,1097.35 C 2181.727,1097.111 2179.385,1097.685 2177.443,1098.055 C 2175.981,1098.334 2174.713,1098.204 2173.271,1098.432 C 2171.431,1098.722 2167.285,1099.605 2167.783,1102.256 C 2167.359,1102.038 2166.059,1102.827 2165.594,1103.108 C 2164.919,1103.513 2164.253,1102.608 2163.512,1102.717 C 2162.772,1102.826 2162.235,1103.477 2161.714,1103.948 C 2161.151,1104.458 2160.635,1104.627 2159.916,1104.898 C 2158.869,1105.292 2158.672,1105.893 2157.862,1106.52 C 2156.778,1107.359 2155.192,1107.772 2153.813,1107.725 C 2155.582,1108.495 2151.823,1108.149 2151.137,1108.08 C 2148.727,1107.84 2146.365,1107.986 2143.952,1107.505 C 2144.043,1107.264 2144.137,1107.025 2144.232,1106.785 C 2142.136,1107.137 2141.172,1105.354 2140.305,1103.942 C 2139.593,1102.784 2137.698,1100.425 2136.242,1101.964 C 2134.979,1100.62 2136.591,1094.337 2139.048,1095.843" id="path3348"/>
    <path d="M 2307.748,906.043 C 2307.651,905.806 2307.555,905.57 2307.458,905.333 C 2308.299,905.415 2308.337,905.503 2307.748,906.043" id="path3350"/>
    <path d="M 2381.758,897.843 C 2382.406,897.498 2382.594,896.913 2381.979,896.404 C 2381.388,896.757 2381.225,897.337 2381.758,897.843" id="path3956"/>
    <path d="M 2381.979,892.803 C 2382.265,892.583 2382.432,892.293 2382.479,891.933 C 2381.681,891.85 2381.514,892.14 2381.979,892.803" id="path3958"/>
    <path d="M 2382.988,893.453 C 2383.196,893.123 2383.29,892.763 2383.268,892.373 C 2382.517,892.52 2382.445,892.931 2382.988,893.453" id="path3960"/>
  </g>
  <g class="landxx coastxx pg" id="pg" transform="translate(-29.9017, -45.0745)">
    <path d="M 2394.508,880.203 C 2394.857,878.745 2392.539,878.133 2391.559,877.823 C 2392.621,878.473 2393.686,879.261 2394.508,880.203" id="path3962"/>
    <path d="M 2394.218,878.113 C 2395.179,877.129 2393.289,876.963 2392.708,877.533 C 2393.179,877.812 2393.682,878.005 2394.218,878.113" id="path3964"/>
    <path d="M 2395.148,877.683 C 2395.257,877.102 2394.876,876.908 2394.358,876.963 C 2394.606,877.219 2394.87,877.459 2395.148,877.683" id="path3966"/>
    <path d="M 2395.658,874.733 C 2395.776,874.226 2395.536,874.152 2394.938,874.512 C 2395.179,874.586 2395.418,874.66 2395.658,874.733" id="path3968"/>
    <path d="M 2397.888,872.353 C 2397.828,871.943 2397.588,871.704 2397.168,871.633 C 2397.408,871.873 2397.647,872.113 2397.888,872.353" id="path3970"/>
    <path class="landxx pg" d="M 2374.198,883.653 L 2375.638,865.373 C 2375.531,864.123 2374.424,863.498 2374.702,862.129 C 2374.935,860.98 2375.58,859.59228 2376.015,858.62228 L 2377.7094,827.14746 C 2378.8114,827.59546 2381.144,828.331 2382.201,828.871 C 2385.147,830.376 2387.845,832.51 2391.089,833.364 C 2392.689,833.785 2394.35,833.551 2395.892,834.217 C 2397.672,834.986 2398.822,836.462 2400.421,837.457 C 2401.686,838.244 2401.923,837.351 2402.922,837.541 C 2404.183,837.781 2404.221,839.173 2405.412,839.447 C 2406.796,839.765 2407.063,841.269 2407.921,842.118 C 2408.809,842.997 2410.112,842.712 2411.091,843.66 C 2412.157,844.694 2413.011,846.072 2413.265,847.55 C 2413.487,848.84 2412.454,849.886 2412.607,850.967 C 2412.83,852.548 2416.666,852.862 2417.825,853.288 C 2418.994,853.718 2420.342,854.602 2421.428,855.213 C 2421.956,855.51 2422.172,856.064 2422.798,856.223 C 2423.531,856.409 2424.227,856.005 2424.958,856.223 C 2426.733,856.751 2429.426,860.681 2427.921,862.435 C 2426.733,863.821 2423.716,862.657 2422.178,862.738 C 2419.748,862.866 2422.09,867.852 2422.824,868.944 C 2423.683,870.224 2425.072,871.13 2425.991,872.41 C 2426.576,873.223 2428.173,874.702 2429.348,874.223 C 2429.246,875.411 2429.463,877.392 2429.82,878.529 C 2430.009,879.135 2430.401,879.365 2430.897,879.662 C 2431.602,880.082 2431.339,881.091 2431.467,881.773 C 2431.85,883.823 2433.514,883.491 2435.022,882.901 C 2436.094,882.481 2437.942,882.359 2437.191,884.021 C 2435.973,886.715 2437.053,887.54 2439.794,887.678 C 2440.314,887.704 2442.338,887.521 2442.417,888.326 C 2442.504,889.21 2440.447,889.316 2439.939,889.633 C 2440.396,890.172 2440.594,890.909 2441.159,891.363 C 2441.87,891.935 2443.032,892.019 2443.845,892.529 C 2445.083,893.306 2446.944,893.722 2448.36,893.233 C 2447.635,893.55 2446.664,894.275 2445.831,894.126 C 2445.174,894.009 2444.456,893.595 2444.33,894.603 C 2445.908,895.013 2447.691,895.803 2445.326,896.534 C 2443.754,897.019 2441.595,897.606 2440.73,895.534 C 2441.256,895.68 2441.712,895.56 2442.099,895.174 C 2441.634,894.911 2441.227,894.985 2440.879,895.394 C 2440.775,893.906 2438.542,894.026 2437.509,893.944 C 2435.719,893.803 2433.927,892.855 2432.089,893.164 C 2432.155,892.899 2432.23,892.636 2432.31,892.374 C 2431.556,892.717 2430.726,892.942 2429.892,892.808 C 2429.299,892.713 2428.14,891.682 2427.591,892.038 C 2426.231,892.924 2425.32,891.118 2424.459,891.934 C 2424.228,891.012 2423.02,890.648 2423.02,889.634 C 2423.02,888.852 2422.435,887.858 2421.982,887.255 C 2421.468,886.57 2420.478,887.092 2419.954,886.331 C 2419.527,885.711 2418.518,884.184 2419.2,883.443 C 2416.748,883.127 2417.13,882.005 2416.481,880.343 C 2415.869,878.774 2414.505,875.134 2412.862,874.455 C 2412.325,874.233 2411.894,873.791 2411.452,873.43 C 2410.853,872.942 2409.916,873.424 2409.219,873.334 C 2408.177,873.2 2407.113,872.493 2406.03,872.283 C 2404.106,871.909 2404.917,871.553 2404.591,870.193 C 2404.6,871.388 2403.38,870.767 2403.071,870.124 C 2402.925,871.62 2402.113,870.313 2402.211,869.693 C 2401.847,870.346 2401.613,870.982 2401.272,869.693 C 2401.279,870.389 2401.328,871.083 2401.421,871.773 C 2401.07,871.294 2400.613,870.961 2400.051,870.773 C 2399.806,874.016 2395.962,869.242 2395.522,869.044 C 2396.048,870.67 2397.168,872.064 2397.391,873.794 C 2397.097,873.709 2396.808,873.612 2396.522,873.504 C 2396.91,875.37 2393.895,873.115 2392.992,873.004 C 2393.382,873.632 2395.583,875.293 2394.242,875.853 C 2393.351,876.225 2392.36,876.186 2391.427,876.351 C 2390.649,876.489 2389.918,876.84 2389.127,876.544 C 2388.506,876.312 2388.006,876.081 2387.329,876.245 C 2386.275,876.501 2385.615,877.037 2385.152,875.664 C 2385.201,876.676 2385.617,877.031 2386.574,876.81 C 2387.906,876.502 2388.859,876.863 2390.01,877.593 C 2392.219,878.994 2393.944,881.633 2390.928,882.917 C 2389.581,883.491 2388.133,885.038 2386.651,885.158 C 2385.245,885.272 2384.053,883.976 2382.484,884.118 C 2380.737,884.277 2378.89,884.718 2377.203,883.95 C 2376.065,883.433 2375.104,885.284 2374.198,883.653" id="path3954"/>
    <path d="M 2414.598,845.853 C 2415.477,845.491 2415.63,844.539 2415.029,843.843 C 2414.255,844.243 2413.68,845.222 2414.598,845.853" id="path3972"/>
    <path d="M 2423.229,851.833 C 2424.625,851.754 2423.974,849.632 2422.884,849.653 C 2422.078,849.669 2422.949,851.58 2423.229,851.833" id="path3974"/>
    <path d="M 2429.568,855.293 C 2430.657,854.454 2430.334,852.257 2428.778,852.263 C 2426.876,852.27 2429.013,854.813 2429.568,855.293" id="path3976"/>
    <path d="M 2429.348,824.973 C 2429.965,825.14 2430.089,824.9 2429.718,824.253 C 2429.595,824.493 2429.471,824.733 2429.348,824.973" id="path3978"/>
    <path d="M 2436.269,856.153 C 2436.742,856.776 2437.551,857.043 2437.848,857.813 C 2438.776,857.144 2440.166,856.855 2441.044,857.824 C 2441.676,858.521 2441.633,859.253 2442.818,859.137 C 2444.365,858.986 2445.923,859.324 2447.461,859.02 C 2449.481,858.62 2450.558,856.687 2452.288,856.721 C 2454.068,856.757 2456.463,854.259 2456.208,852.483 C 2457.213,852.602 2462.023,853.278 2461.317,850.794 C 2461.197,850.368 2459.525,847.431 2460.958,847.728 C 2463.315,848.217 2463.919,845.679 2463.92,843.882 C 2463.92,843.218 2464.109,842.459 2463.407,842.138 C 2462.734,841.83 2461.965,841.514 2462.758,840.743 C 2462.102,840.995 2460.634,842.596 2459.968,841.636 C 2459.264,840.622 2457.411,840.664 2457.589,842.203 C 2457.758,843.668 2458.563,844.912 2458.329,846.473 C 2458.096,848.03 2455.9,847.208 2454.814,848.083 C 2453.638,849.031 2453.447,850.518 2452.538,851.646 C 2452.157,852.119 2451.434,852.087 2450.909,852.229 C 2449.713,852.553 2449.912,852.069 2448.783,851.869 C 2447.922,851.717 2446.908,853.326 2446.234,852.401 C 2445.337,851.171 2445.384,849.208 2446.847,848.453 C 2445.602,847.112 2444.518,850.843 2444.867,851.426 C 2445.508,852.498 2440.413,853.016 2439.868,852.983 C 2439.16,852.941 2438.795,852.595 2438.164,852.337 C 2437.316,851.991 2436.962,852.486 2436.184,852.431 C 2435.495,852.382 2433.755,852.738 2433.393,852.182 C 2432.808,851.287 2431.809,853.457 2432.731,854.406 C 2433.381,855.074 2434.056,854.877 2434.829,855.212 C 2435.4,855.461 2435.602,856.055 2436.269,856.153" id="path3982"/>
    <path d="M 2442.098,845.493 C 2442.252,844.839 2442.013,844.769 2441.378,845.283 C 2441.618,845.353 2441.857,845.423 2442.098,845.493" id="path3984"/>
    <path d="M 2444.618,818.493 C 2445.404,817.767 2443.964,816.062 2443.222,816.717 C 2442.764,817.121 2444.32,818.343 2444.618,818.493" id="path3986"/>
    <path d="M 2444.908,887.043 C 2445.182,886.513 2445.246,883.674 2443.948,884.407 C 2442.389,885.287 2444.156,886.564 2444.908,887.043" id="path3988"/>
    <path d="M 2448.648,888.692 C 2450.181,887.092 2448.019,886.307 2446.818,885.855 C 2444.668,885.046 2446.85,887.307 2445.628,888.043 C 2446.483,887.729 2447.996,888.06 2448.648,888.692" id="path3990"/>
    <path d="M 2446.488,826.343 C 2447.107,826.913 2447.518,827.652 2448.427,827.783 C 2449.37,827.919 2450.49,827.33 2449.785,826.246 C 2449.175,825.305 2446.649,824.753 2446.488,826.343" id="path3992"/>
    <path d="M 2447.998,896.833 C 2449.166,894.916 2445.887,896.734 2447.998,896.833" id="path3994"/>
    <path d="M 2448.798,891.143 C 2449.012,891.143 2449.225,891.143 2449.439,891.143 C 2448.512,891.753 2450.093,892.296 2450.514,892.446 C 2451.562,892.818 2451.989,891.153 2451.309,890.573 C 2450.936,890.781 2450.7,891.092 2450.598,891.503 C 2449.382,891.069 2448.92,889.641 2447.859,888.983 C 2448.281,889.646 2448.695,890.346 2448.798,891.143" id="path3996"/>
    <path d="M 2448.358,897.043 C 2448.554,896.706 2448.844,896.493 2449.227,896.403 C 2448.66,896.202 2448.336,896.442 2448.358,897.043" id="path3998"/>
    <path d="M 2451.598,881.063 C 2451.133,880.129 2451.405,878.375 2451.819,877.462 C 2451.006,877.596 2450.378,878.752 2451.598,878.832 C 2451.303,879.595 2450.622,880.46 2451.598,881.063" id="path4000"/>
    <path d="M 2452.248,830.453 C 2452.737,830.365 2453.218,830.242 2453.688,830.083 C 2453.15,829.767 2452.589,829.963 2452.248,830.453" id="path4002"/>
    <path d="M 2466.358,838.013 C 2465.307,837.865 2464.888,836.1 2464.261,835.422 C 2463.284,834.365 2461.487,832.719 2460.144,832.178 C 2458.729,831.608 2457.955,830.092 2456.534,829.518 C 2456.013,829.309 2455.646,829.878 2455.428,829.116 C 2455.211,828.359 2453.67,827.853 2453.109,827.284 C 2453.38,827.848 2453.157,828.192 2452.539,828.214 C 2453.243,828.83 2454.257,828.76 2454.956,829.392 C 2455.815,830.169 2456.673,830.951 2457.579,831.674 C 2458.571,832.465 2459.498,833.489 2460.63,834.083 C 2461.476,834.526 2462.469,834.646 2463.226,835.263 C 2464.699,836.464 2465.205,838.509 2466.023,840.14 C 2466.588,841.269 2466.058,842.447 2466.15,843.623 C 2466.24,844.767 2467.047,845.586 2467.52,846.573 C 2468.412,845.843 2470.019,842.426 2469.56,841.329 C 2469.185,840.435 2468.365,839.864 2467.972,839.027 C 2467.481,837.986 2467.361,838.373 2466.358,838.013" id="path4004"/>
    <path d="M 2462.108,897.333 C 2462.351,897.143 2462.594,896.953 2462.837,896.763 C 2462.113,896.766 2461.394,896.717 2460.677,896.614 C 2461.153,896.855 2461.631,897.095 2462.108,897.333" id="path4006"/>
    <path d="M 2464.128,884.522 C 2466.467,884.433 2463.908,880.999 2462.399,882.722 C 2463.194,883.135 2463.293,884.159 2464.128,884.522" id="path4012"/>
    <path d="M 2465.068,902.303 C 2464.957,901.786 2465.01,901.536 2464.347,901.654 C 2464.588,901.87 2464.828,902.086 2465.068,902.303" id="path4014"/>
    <path d="M 2467.368,905.403 C 2467.88,905.267 2468.337,905.027 2468.738,904.682 C 2468.064,904.057 2466.038,902.671 2465.068,903.243 C 2465.56,903.645 2465.537,904.344 2465.998,904.753 C 2466.592,905.28 2466.985,904.26 2467.368,905.403" id="path4016"/>
    <path d="M 2466.428,832.603 C 2466.77,832.026 2466.502,831.449 2465.998,831.093 C 2465.785,831.695 2465.929,832.199 2466.428,832.603" id="path4018"/>
    <path d="M 2471.758,903.603 C 2472.188,903.292 2472.668,903.149 2473.198,903.173 C 2472.547,902.22 2471.786,903.125 2470.968,903.173 C 2471.231,903.316 2471.494,903.459 2471.758,903.603" id="path4020"/>
    <path d="M 2480.479,851.833 C 2481.516,850.938 2481.581,849.037 2480.479,848.163 C 2479.863,849.267 2480.092,850.697 2480.479,851.833" id="path4024"/>
    <path d="M 2482.848,858.453 C 2483.395,858.96 2484.189,859.174 2484.303,859.988 C 2484.446,861.002 2484.278,862.036 2485.06,862.836 C 2486.202,864.003 2490.241,864.338 2489.811,862.099 C 2489.63,861.158 2489.37,860.503 2488.768,859.709 C 2488.411,859.236 2488.016,858.356 2487.438,858.12 C 2486.906,857.902 2486.396,857.891 2486.131,857.307 C 2485.027,854.872 2483.584,852.581 2480.908,851.693 C 2481.625,852.487 2481.149,853.715 2481.084,854.631 C 2480.975,856.148 2482.155,857.235 2482.848,858.453" id="path4026"/>
    <path d="M 2488.108,866.303 C 2488.348,866.23 2488.587,866.157 2488.828,866.083 C 2488.675,865.135 2487.596,865.019 2487.388,866.013 C 2487.628,866.109 2487.868,866.206 2488.108,866.303" id="path4028"/>
    <path d="M 2490.628,865.083 C 2490.782,864.603 2490.782,864.123 2490.628,863.643 C 2490.628,864.123 2490.628,864.603 2490.628,865.083" id="path4030"/>
    <path d="M 2420.348,823.973 C 2419.378,822.849 2421.03,821.845 2421.96,821.599 C 2423.461,821.202 2425.164,822.029 2426.618,822.314 C 2425.343,822.596 2425.249,823.603 2423.668,823.603 C 2423.029,823.603 2422.429,823.021 2421.798,823.103 C 2421.25,823.173 2420.832,823.752 2420.348,823.973" id="path4032"/>
  </g>
  <path class="landxx iq" d="M 1555.7563,470.80555 C 1560.119,468.66597 1563.9512,466.57756 1567.8319,463.91191 C 1569.0655,463.06453 1570.3962,462.01591 1571.1053,460.93355 C 1572.467,458.85508 1571.3814,456.10758 1571.6532,455.32982 C 1571.682,452.09858 1572.5534,451.69887 1572.3679,450.14627 C 1572.1607,448.41071 1569.691,445.36392 1571.5973,443.87855 C 1572.688,443.02868 1574.0423,443.06426 1575.1681,442.12476 C 1576.1134,441.33584 1578.6559,439.27041 1577.9017,438.43657 C 1577.5569,438.05539 1578.2481,438.47235 1578.7651,438.31535 C 1579.6191,438.05735 1579.8976,437.25442 1580.2693,436.57355 C 1580.6342,435.90528 1584.4376,436.52324 1585.2713,436.84355 C 1585.9313,437.09713 1586.5973,437.33655 1587.2923,437.47955 C 1588.4353,437.71555 1588.7773,436.97591 1589.7713,436.79155 C 1592.2779,436.32664 1590.5669,439.61116 1591.9883,439.48255 C 1593.0239,439.38885 1593.4349,437.82076 1594.5749,437.86376 C 1596.0219,437.91776 1595.9705,440.37655 1596.6055,441.22755 C 1596.9065,441.63155 1597.2993,441.56055 1597.5503,442.06655 C 1597.8563,442.68455 1597.4503,443.56378 1597.9707,444.07234 C 1599.0673,445.14407 1601.478,446.20003 1600.4935,448.05095 C 1602.1207,447.68695 1602.789,449.14794 1603.5145,449.42255 C 1604.9018,449.94771 1604.9225,448.84736 1607.9135,449.63855 C 1607.6525,450.76255 1606.3469,450.50195 1605.8779,451.37095 C 1605.0839,452.83895 1607.3273,453.93055 1607.4563,455.26455 C 1607.6253,457.02555 1605.4733,455.08955 1605.5353,456.87455 C 1605.5623,457.67855 1604.6923,457.54315 1604.4703,458.30015 C 1604.2193,459.15615 1604.6413,459.79355 1604.7883,460.58255 C 1604.2443,460.57955 1603.7403,460.43555 1603.2773,460.14955 C 1603.1554,460.87366 1602.6522,461.42448 1603.2235,462.02355 C 1603.8461,462.67646 1604.3134,462.95537 1603.9697,464.02575 C 1603.6418,465.04679 1602.7273,465.27855 1603.6063,466.07355 C 1604.3593,466.75355 1605.021,467.56694 1605.8673,468.13555 C 1606.7127,468.70355 1611.1167,472.47755 1608.6073,473.18255 C 1609.5857,473.42119 1609.3374,474.55998 1610.1443,474.37235 C 1611.3039,474.10269 1611.9534,474.84054 1612.8825,475.36515 C 1614.5894,476.32894 1615.6802,478.51865 1618.0427,478.70274 C 1619.1628,478.79002 1621.2273,482.43919 1622.2165,483.31335 C 1623.7128,484.63563 1622.7953,486.00155 1622.4193,487.82055 C 1622.2393,488.68955 1622.2845,491.15676 1623.5725,491.24876 C 1624.0245,491.28176 1624.7805,491.18774 1625.1345,491.63874 C 1625.5915,492.21774 1625.5894,493.38503 1625.4853,494.09255 C 1625.2539,495.66442 1626.0742,496.12502 1627.8313,497.39115 C 1628.7173,498.02962 1628.517,500.0108 1629.7727,500.31417 C 1629.1237,500.36317 1625.2543,499.81155 1625.1683,499.10355 C 1625.7013,500.36155 1623.4885,499.35234 1622.6235,499.31434 C 1621.3725,499.26034 1619.7549,499.55582 1619.1883,500.75955 C 1617.9603,503.36855 1618.1805,508.56626 1614.0133,508.08955 L 1603.4959,506.88638 C 1599.9756,506.3828 1599.8435,504.53246 1596.6233,502.50855 C 1587.7178,495.95848 1579.1874,488.10925 1569.216,483.74119 C 1565.633,482.1716 1562.4511,482.30511 1559.2135,481.591 C 1561.243,479.39173 1558.7986,480.36094 1558.1901,479.42536 C 1558.039,479.19298 1557.913,478.72991 1558.044,478.49591 C 1558.553,477.91191 1558.049,477.31773 1557.731,476.46573 C 1557.024,474.57973 1556.3553,472.72555 1555.7563,470.80555" id="iq"/>
  <g class="landxx coastxx hr" id="hr" transform="translate(-29.9017, -45.0745)">
    <path class="landxx hr" d="M 1401.337,411.84 C 1401.831,411.955 1402.3193,412.08445 1402.817,412.191 C 1403.4639,412.3295 1403.9702,411.70835 1404.3154,411.58745 C 1404.4887,411.52676 1405.9437,412.92305 1408.0448,410.23818 C 1409.3489,411.51588 1410.4094,411.05393 1411.824,411.84 C 1413.6727,412.86731 1412.7556,410.59955 1413.577,410.4 C 1413.219,410.319 1412.918,410.012 1412.567,409.896 C 1415.4212,407.84864 1416.0964,410.00736 1415.0612,406.11118 C 1414.9294,405.61502 1417.4336,405.16889 1417.8427,404.77737 C 1418.4714,404.1757 1417.8202,404.38636 1418.4742,403.95736 C 1418.4802,403.95336 1419.461,404.204 1419.628,404.209 C 1417.7315,402.04246 1421.2982,402.90731 1422.0113,403.62174 C 1422.5151,404.12647 1425.2338,405.75581 1425.5945,406.24818 C 1427.0418,407.8758 1429.1484,408.67346 1431.4092,409.40936 C 1436.7919,409.70923 1434.352,408.2964 1436.7348,407.96846 C 1436.6794,407.4488 1437.1135,408.55045 1437.4217,409.199 C 1437.8667,410.1353 1437.4608,411.353 1438.847,411.409 C 1436.555,413.002 1441.5985,413.9125 1441.0656,414.24327 L 1438.847,414.793 C 1439.574,415.984 1438.6439,417.94173 1437.1279,416.76773 C 1436.132,414.99792 1436.1613,415.93186 1435.659,415.45064 C 1434.723,415.10764 1433.8626,414.91087 1432.9167,415.15945 C 1431.8897,415.42935 1431.9561,414.63975 1430.9541,415.19182 C 1430.2138,415.59969 1429.7078,415.17627 1429.0048,415.00827 C 1427.3101,414.97182 1425.8213,414.42699 1424.2335,413.97336 C 1423.0619,414.07909 1424.6853,414.57291 1422.8695,414.07918 C 1421.5399,413.71762 1421.4344,414.05822 1420.9779,414.88163 C 1419.9959,416.65301 1419.8539,415.87355 1418.1566,414.61738 C 1415.9727,413.00113 1416.2733,415.9617 1416.4067,417.06491 C 1416.4892,417.74685 1416.2825,417.87138 1416.9878,418.17536 C 1417.6198,418.44776 1418.6854,419.12323 1419.1329,419.74162 C 1420.0647,421.02937 1419.0936,422.82908 1421.0798,423.99009 C 1422.9791,425.10028 1424.0529,426.95099 1425.8113,428.26481 C 1427.2273,429.32281 1427.8765,430.6344 1429.0999,431.70082 C 1430.5765,432.98781 1429.8367,434.47024 1428.165,432.52 C 1426.436,430.50287 1422.129,427.547 1419.34,428.586 C 1418.609,428.858 1418.225,428.577 1417.757,427.969 C 1417.175,427.214 1417.536,427.012 1416.434,426.465 C 1414.826,425.666 1412.62,423.855 1411.987,422.137 C 1412.987,421.965 1413.891,422.109 1414.877,422.282 C 1413.581,421.33 1411.983,420.617 1411.079,419.22 C 1410.324,418.053 1410.823,416.558 1410.089,415.401 C 1409.443,414.384 1408.001,413.412 1406.841,413.095 C 1405.987,412.862 1405.205,415.319 1405.368,415.946 C 1405.001,415.794 1404.865,415.903 1404.578,415.586 C 1404.468,416.41 1404.27,417.292 1403.638,417.89 C 1403.126,417.495 1401.474,415.451 1402.268,414.794 C 1401.357,415.005 1400.916,412.507 1400.688,411.77 C 1400.904,411.792 1401.121,411.815 1401.337,411.84" id="path4066"/>
    <path class="landxx hr" d="M 1429.917,433.944 C 1430.6635,433.28714 1431.5783,434.28448 1432.6004,435.18391 C 1433.8772,436.30744 1436.7268,436.54185 1435.827,437.616 C 1434.672,437.455 1432.664,435.822 1431.75,435.091 C 1430.95,434.451 1429.908,434.548 1428.987,434.232 C 1427.99,433.89 1427.163,433.005 1426.027,433.153 C 1427.206,432.247 1429.485,433.651 1430.427,434.377 C 1430.258,434.232 1430.087,434.088 1429.917,433.944" id="path4068"/>
    <path d="M 1406.667,415.944 C 1406.26,415.507 1406.141,415.004 1406.307,414.431 C 1407.139,415.803 1407.285,417.391 1407.747,418.895 C 1406.794,418.752 1405.864,416.649 1406.667,415.944" id="path4070"/>
    <path d="M 1407.888,420.408 C 1407.502,419.91 1407.095,419.43 1406.668,418.968 C 1407.184,419.336 1407.617,419.833 1407.888,420.408" id="path4072"/>
    <path d="M 1408.538,415.944 C 1407.892,415.875 1406.739,414.55 1407.818,414.215 C 1408.569,413.982 1409.146,415.658 1409.688,415.944 C 1409.298,416.342 1408.914,416.342 1408.538,415.944" id="path4074"/>
    <path d="M 1409.038,417.384 C 1409.278,417.528 1409.518,417.672 1409.758,417.816 C 1409.11,418.122 1408.871,417.978 1409.038,417.384" id="path4076"/>
    <path d="M 1410.478,422.784 C 1410.238,422.592 1409.998,422.4 1409.758,422.208 C 1410.075,422.308 1410.315,422.5 1410.478,422.784" id="path4078"/>
    <path d="M 1411.198,420.696 C 1410.498,420.024 1409.796,419.326 1409.328,418.464 C 1410.208,419.684 1411.758,420.126 1412.638,421.344 C 1411.938,421.881 1411.523,421.228 1411.198,420.696" id="path4080"/>
    <path d="M 1411.268,424.08 C 1411.031,423.864 1410.795,423.647 1410.558,423.432 C 1411.492,423.502 1411.712,424.597 1412.428,425.016 C 1411.879,424.904 1411.493,424.592 1411.268,424.08" id="path4082"/>
    <path d="M 1412.928,425.52 C 1413.487,425.832 1413.99,426.215 1414.438,426.672 C 1413.872,426.41 1413.276,426.057 1412.928,425.52" id="path4086"/>
    <path d="M 1412.928,424.296 C 1412.688,424.128 1412.448,423.96 1412.208,423.792 C 1412.569,423.804 1412.809,423.972 1412.928,424.296" id="path4088"/>
    <path d="M 1415.667,427.248 C 1415.93,427.368 1416.194,427.488 1416.457,427.608 C 1416.18,427.52 1415.917,427.399 1415.667,427.248" id="path4090"/>
    <path d="M 1419.768,428.688 C 1420.008,428.664 1420.248,428.64 1420.488,428.616 C 1420.252,428.712 1420.012,428.736 1419.768,428.688" id="path4092"/>
    <path d="M 1420.917,430.2 C 1420.677,430.08 1420.437,429.96 1420.197,429.84 C 1420.505,429.839 1420.745,429.959 1420.917,430.2" id="path4094"/>
    <path d="M 1423.078,430.776 C 1422.336,430.769 1421.402,430.59 1421.278,429.696 C 1422.275,429.665 1423.284,429.795 1424.228,430.128 C 1423.928,430.493 1423.544,430.709 1423.078,430.776" id="path4096"/>
    <path d="M 1424.448,433.872 C 1423.951,433.755 1423.448,433.684 1422.938,433.656 C 1423.829,432.881 1425.723,433.007 1426.678,433.584 C 1425.932,433.819 1425.149,433.419 1424.448,433.872" id="path4098"/>
    <path d="M 1425.748,432 C 1424.267,432.052 1422.774,431.916 1421.348,431.496 C 1423.039,430.94 1424.76,431.842 1426.468,431.784 C 1426.228,431.856 1425.988,431.928 1425.748,432" id="path4100"/>
    <path d="M 1430.568,435.6 C 1429.845,435.349 1429.084,435.25 1428.338,435.096 C 1429.099,434.929 1429.925,435.206 1430.568,435.6" id="path4102"/>
  </g>
  <g class="landxx coastxx gl" id="gl">
    <path d="M 991.85532,170.99755 C 992.45732,171.13755 993.03032,171.35255 993.57532,171.64455 C 992.37432,171.43355 990.49032,170.76655 990.26532,172.58155 C 990.86732,172.58255 991.42232,172.89455 991.77532,173.37355 C 990.86532,173.01755 989.59332,172.38955 988.82532,173.37355 C 990.23732,174.56955 992.37132,174.36955 994.08532,174.30955 C 992.17932,175.03355 994.55632,174.60955 994.87532,174.52555 C 992.94032,175.45955 996.60432,174.02355 997.10432,174.09355 C 996.44432,174.11055 994.88732,174.80255 996.38432,175.10155 C 994.84832,175.10155 993.30532,174.99955 991.77432,175.17355 C 991.69532,177.72355 998.65832,175.82355 999.75032,175.66555 C 1000.9933,175.48555 1005.7493,174.87455 1005.7023,173.02555 C 1005.6713,171.82655 1003.5483,171.74655 1002.7483,171.49755 C 1001.0513,170.97055 1000.2323,169.38955 998.31832,169.21655 C 996.52732,169.05455 994.91832,168.49255 993.14932,169.12955 C 992.47732,169.37155 990.22332,170.32155 991.85532,170.99755" id="path4242"/>
    <path d="M 992.13632,167.54155 C 991.84432,167.78455 991.87032,168.04855 992.21632,168.33355 C 992.49232,168.18655 992.73232,167.99455 992.93632,167.75755 C 992.66932,167.68555 992.40232,167.61355 992.13632,167.54155" id="path4252"/>
    <path d="M 993.93532,157.53355 C 994.34332,158.41955 995.34032,157.96255 995.59532,157.24455 C 995.03632,157.30855 994.48332,157.40455 993.93532,157.53355" id="path4262"/>
    <path d="M 1003.2253,161.70955 C 1002.1843,161.74755 1000.9653,162.28955 1000.3453,163.14855 C 1001.4563,163.63355 1004.4963,163.65555 1003.2253,161.70955" id="path4322"/>
    <path d="M 1006.1763,160.12555 C 1007.1973,159.97955 1008.2443,159.64155 1009.1363,159.11755 C 1008.1093,159.28555 1006.8033,158.79155 1005.8453,159.14855 C 1005.0273,159.45255 1005.4553,160.15855 1006.1763,160.12555" id="path4330"/>
    <path d="M 1005.7463,161.42155 C 1004.4153,163.05855 1008.8523,162.23255 1009.5653,161.34955 C 1008.2813,161.43855 1007.0043,160.95655 1005.7463,161.42155" id="path4332"/>
    <path d="M 1007.9753,160.19755 C 1007.4313,160.12955 1006.9283,160.24855 1006.4653,160.55655 C 1006.9763,160.46655 1007.4793,160.34655 1007.9753,160.19755" id="path4336"/>
    <path d="M 1010.5763,164.01355 C 1009.9933,163.92755 1009.6963,164.22255 1009.7763,164.80555 C 1010.5703,164.90655 1011.4163,164.97855 1012.1563,164.58955 C 1011.6293,164.39755 1011.1023,164.20555 1010.5763,164.01355" id="path4342"/>
    <path d="M 1011.6563,166.31755 C 1011.4483,165.71155 1010.6343,165.67655 1010.1363,165.81355 C 1010.6253,166.03355 1011.1313,166.20155 1011.6563,166.31755" id="path4344"/>
    <path d="M 1010.4263,167.10955 C 1010.6443,167.24855 1010.8843,167.32055 1011.1463,167.32555 C 1010.9063,167.25255 1010.6653,167.18155 1010.4263,167.10955" id="path4346"/>
    <path d="M 1012.5863,167.46955 C 1012.1223,167.07455 1011.5953,166.95455 1011.0063,167.10955 C 1011.5233,167.26955 1012.0503,167.38955 1012.5863,167.46955" id="path4348"/>
    <g id="g27049" transform="translate(-29.9017, -45.0745)">
      <path d="M 1015.638,240.048 C 1016.188,240.07 1016.692,239.927 1017.148,239.616 C 1016.619,239.671 1016.116,239.814 1015.638,240.048" id="path4228"/>
      <path d="M 1019.087,238.896 C 1017.811,238.909 1016.584,239.361 1015.347,239.617 C 1016.346,238.744 1017.861,238.856 1019.087,238.896" id="path4528"/>
      <path d="M 1020.388,230.184 C 1020.773,230.845 1021.036,230.748 1021.178,229.896 C 1020.915,229.991 1020.651,230.088 1020.388,230.184" id="path4248"/>
      <path d="M 1024.778,227.088 C 1024.278,227.276 1023.821,227.54 1023.408,227.88 C 1024.019,228.211 1025.267,228.064 1024.778,227.088" id="path4260"/>
      <path d="M 1025.428,226.584 C 1025.177,226.682 1024.937,226.802 1024.708,226.944 C 1024.948,226.824 1025.188,226.704 1025.428,226.584" id="path4268"/>
      <path d="M 1026.217,225.647 C 1025.946,225.692 1025.683,225.764 1025.427,225.864 C 1025.691,225.792 1025.954,225.72 1026.217,225.647" id="path4272"/>
      <path d="M 1027.148,225.432 C 1026.914,225.496 1026.662,225.551 1026.438,225.648 C 1026.675,225.575 1026.911,225.504 1027.148,225.432" id="path4282"/>
      <path d="M 1027.438,229.392 C 1028.009,229.703 1030.167,229.059 1028.238,229.896 C 1029.455,229.805 1031.023,229.671 1031.978,228.816 C 1030.483,229.133 1028.957,229.259 1027.438,229.392" id="path4294"/>
      <path d="M 1029.028,230.04 C 1029.47,230.705 1032.268,230.098 1032.768,229.68 C 1031.568,229.394 1030.195,229.765 1029.028,230.04" id="path4304"/>
      <path d="M 1022.038,219.456 C 1022.231,219.659 1022.471,219.684 1022.758,219.528 C 1022.518,219.504 1022.277,219.48 1022.038,219.456" id="path4256"/>
      <path d="M 1030.108,225.576 C 1028.301,225.29 1026.539,225.838 1024.778,226.151 C 1026.146,226.585 1028.931,226.486 1030.108,225.576" id="path4270"/>
      <path d="M 1041.478,216.647 C 1040.919,216.569 1040.393,216.665 1039.898,216.936 C 1041.966,216.82 1037.411,218.088 1038.844,219.353 C 1039.553,219.979 1040.121,219.412 1040.454,218.803 C 1040.968,217.863 1041.99,217.981 1042.778,217.44 C 1042.412,217.065 1041.979,216.801 1041.478,216.647" id="path4338"/>
      <path d="M 1028.598,196.416 C 1027.61,196.256 1026.343,196.572 1025.568,197.208 C 1026.526,197.161 1027.872,197.161 1028.598,196.416" id="path4274"/>
      <path d="M 1026.578,198 C 1026.865,197.88 1027.152,197.76 1027.438,197.64 C 1026.861,197.581 1026.311,197.677 1025.788,197.928 C 1026.051,197.952 1026.314,197.976 1026.578,198" id="path4276"/>
      <path d="M 1028.448,190.08 C 1027.684,190.179 1026.851,190.012 1026.218,190.584 C 1027.213,190.452 1028.225,190.483 1029.168,190.08 C 1028.928,190.08 1028.688,190.08 1028.448,190.08" id="path4278"/>
      <path d="M 1030.178,199.151 C 1028.964,199.164 1027.35,199.493 1026.358,200.231 C 1027.413,201.328 1029.336,199.837 1030.178,199.151" id="path4280"/>
      <path d="M 1026.938,198.216 C 1027.15,198.371 1027.39,198.347 1027.658,198.144 C 1027.418,198.168 1027.178,198.191 1026.938,198.216" id="path4284"/>
      <path d="M 1027.148,191.16 C 1027.412,191.233 1027.679,191.233 1027.948,191.16 C 1027.681,191.16 1027.415,191.16 1027.148,191.16" id="path4286"/>
      <path d="M 1027.948,189.288 C 1027.782,189.274 1027.466,189.138 1027.148,189.288 C 1027.414,189.288 1027.682,189.299 1027.948,189.288" id="path4288"/>
      <path d="M 1027.948,184.392 C 1027.681,184.488 1027.415,184.584 1027.148,184.68 C 1028.657,184.438 1030.566,185.243 1031.908,184.248 C 1030.585,184.033 1029.271,184.432 1027.948,184.392" id="path4290"/>
      <path d="M 1031.328,195.912 C 1030.422,196.315 1028.33,197.175 1027.368,197.208 C 1029.114,197.127 1030.547,197.585 1032.048,196.272 C 1031.808,196.152 1031.567,196.032 1031.328,195.912" id="path4292"/>
      <path d="M 1029.818,193.104 C 1028.908,193.058 1028.184,193.745 1027.518,194.255 C 1028.359,194.121 1029.155,193.611 1029.818,193.104" id="path4296"/>
      <path d="M 1028.158,191.088 C 1028.639,190.82 1029.142,190.605 1029.668,190.441 C 1029.083,190.468 1028.58,190.684 1028.158,191.088" id="path4298"/>
      <path d="M 1029.748,189.432 C 1030.33,189.332 1030.88,189.14 1031.398,188.857 C 1030.418,188.77 1029.02,188.198 1028.308,189.216 C 1028.817,189.093 1029.297,189.165 1029.748,189.432" id="path4300"/>
      <path d="M 1029.888,192.024 C 1029.485,192.205 1029.101,192.42 1028.738,192.671 C 1029.313,192.895 1029.731,192.578 1029.888,192.024" id="path4302"/>
      <path d="M 1029.308,189.504 C 1029.537,189.698 1029.804,189.746 1030.108,189.648 C 1029.841,189.6 1029.574,189.552 1029.308,189.504" id="path4306"/>
      <path d="M 1030.967,197.784 C 1030.449,197.679 1029.946,197.726 1029.457,197.928 C 1029.983,198.116 1030.486,198.068 1030.967,197.784" id="path4308"/>
      <path d="M 1029.598,195.624 C 1029.464,196.224 1029.704,196.271 1030.318,195.768 C 1030.078,195.72 1029.838,195.672 1029.598,195.624" id="path4310"/>
      <path d="M 1030.388,183.744 C 1030.125,183.84 1029.861,183.936 1029.598,184.032 C 1030.244,184.355 1030.507,184.26 1030.388,183.744" id="path4312"/>
      <path d="M 1031.118,195.408 C 1031.358,195.216 1031.598,195.024 1031.838,194.832 C 1031.088,194.554 1030.273,194.415 1029.598,194.976 C 1030.101,195.133 1030.607,195.277 1031.118,195.408" id="path4314"/>
      <path d="M 1030.757,190.872 C 1030.517,190.872 1030.277,190.872 1030.037,190.872 C 1030.84,191.153 1031.556,191.622 1032.337,191.952 C 1031.932,191.424 1031.296,191.216 1030.757,190.872" id="path4316"/>
      <path d="M 1030.248,191.521 C 1030.57,191.734 1030.445,191.845 1030.968,191.521 C 1030.728,191.521 1030.487,191.521 1030.248,191.521" id="path4318"/>
      <path d="M 1031.688,196.992 C 1031.088,196.981 1030.584,197.197 1030.178,197.639 C 1030.777,197.648 1031.281,197.433 1031.688,196.992" id="path4320"/>
      <path d="M 1031.618,192.672 C 1032.177,192.502 1032.703,192.262 1033.198,191.952 C 1032.639,192.122 1032.112,192.362 1031.618,192.672" id="path4324"/>
      <path d="M 997.558,164.808 C 998.051,164.877 998.531,164.829 998.998,164.664 C 998.587,164.632 996.32,165.171 998.428,164.88 C 998.138,164.855 997.848,164.832 997.558,164.808" id="path4216"/>
      <path d="M 969.188,165.024 C 968.64,164.989 968.186,165.181 967.828,165.599 C 969.61,165.807 971.322,165.916 973.078,165.455 C 971.764,165.544 970.485,165.139 969.188,165.024" id="path4204"/>
      <path d="M 973.658,171.937 C 973.921,171.84 974.185,171.745 974.448,171.648 C 974.075,171.512 973.692,171.464 973.298,171.504 C 973.417,171.648 973.538,171.792 973.658,171.937" id="path4208"/>
      <path d="M 974.158,165.024 C 974.43,166.348 978.696,165.478 979.558,165.311 C 977.749,165.375 975.967,164.738 974.158,165.024" id="path4210"/>
      <path d="M 974.598,170.856 C 975.425,171.051 976.291,171.049 977.118,170.856 C 976.375,170.435 975.335,170.408 974.598,170.856" id="path4212"/>
      <path d="M 996.478,164.16 C 997.02,164.129 997.55,164.033 998.068,163.872 C 997.526,163.903 996.996,163.999 996.478,164.16" id="path4214"/>
      <path d="M 997.558,164.808 C 998.029,164.674 998.509,164.626 998.998,164.664 C 998.531,164.829 998.051,164.877 997.558,164.808" id="path4522"/>
      <path d="M 997.558,164.808 C 997.848,164.832 998.138,164.856 998.428,164.88 C 998.13,164.929 997.84,164.905 997.558,164.808" id="path4524"/>
      <path d="M 1015.488,145.512 C 1015.976,145.724 1016.479,145.773 1016.998,145.657 C 1016.525,145.283 1016.022,145.235 1015.488,145.512" id="path4226"/>
      <path d="M 1095.268,138.24 C 1094.133,137.108 1092.247,138.037 1091.052,137.082 C 1089.873,136.141 1087.829,136.133 1086.698,137.16 C 1088.513,138.959 1092.967,138.516 1095.268,138.24" id="path4380"/>
      <path d="M 1092.028,137.016 C 1092.582,137.146 1093.109,137.074 1093.608,136.8 C 1093.043,136.593 1092.517,136.665 1092.028,137.016" id="path4384"/>
      <path d="M 1104.268,134.136 C 1104.795,134.219 1105.321,134.219 1105.848,134.136 C 1105.321,134.051 1104.794,134.051 1104.268,134.136" id="path4388"/>
      <path d="M 1110.667,136.008 C 1111.056,135.986 1111.369,135.818 1111.607,135.504 C 1110.807,135.328 1109.946,135.178 1109.157,135.504 C 1109.704,135.543 1110.207,135.711 1110.667,136.008" id="path4396"/>
      <path d="M 1112.328,134.136 C 1113.439,134.136 1114.683,134.408 1115.718,133.848 C 1114.628,133.642 1113.238,133.239 1112.328,134.136" id="path4398"/>
      <path d="M 1115.998,135.504 C 1116.721,136.642 1118.636,136.713 1119.82,136.809 C 1121.66,136.958 1123.6,137.969 1125.428,137.735 C 1125.159,137.526 1124.896,137.31 1124.638,137.088 C 1125.577,136.641 1126.638,136.481 1127.668,136.441 C 1124.986,133.656 1117.996,134.487 1114.638,134.857 C 1115.067,135.122 1115.521,135.338 1115.998,135.504" id="path4400"/>
      <path d="M 1119.388,133.848 C 1119.965,133.848 1120.541,133.848 1121.118,133.848 C 1119.992,133.844 1118.929,133.374 1117.798,133.416 C 1118.328,133.559 1118.858,133.702 1119.388,133.848" id="path4402"/>
      <path d="M 1144.368,131.976 C 1145.135,131.976 1145.901,131.976 1146.668,131.976 C 1145.918,131.812 1145.111,131.727 1144.368,131.976" id="path4406"/>
      <path d="M 1151.278,133.128 C 1149.442,132.087 1146.841,132.029 1144.798,132.264 C 1146.951,132.613 1149.084,133.128 1151.278,133.128" id="path4408"/>
      <path d="M 1152.507,133.056 C 1153.093,133.076 1153.62,132.908 1154.087,132.552 C 1152.492,132.351 1150.691,131.82 1149.117,132.408 C 1150.19,132.782 1151.367,133.046 1152.507,133.056" id="path4410"/>
      <path d="M 1155.958,132.479 C 1156.416,132.481 1156.873,132.457 1157.328,132.408 C 1155.578,131.797 1150.967,130.706 1149.338,131.76 C 1151.566,131.76 1153.743,132.31 1155.958,132.479" id="path4412"/>
      <path d="M 1153.948,131.328 C 1154.729,131.551 1155.669,131.78 1156.318,131.112 C 1155.524,130.983 1154.706,131.063 1153.948,131.328" id="path4416"/>
      <path d="M 1233.868,140.256 C 1234.875,140.113 1235.91,140.238 1236.888,139.896 C 1236.384,139.831 1235.88,139.758 1235.378,139.68 C 1234.865,139.847 1234.361,140.039 1233.868,140.256" id="path4470"/>
      <path d="M 1237.828,138.744 C 1238.452,138.883 1238.978,138.715 1239.408,138.24 C 1238.331,137.89 1237.451,137.036 1236.238,137.303 C 1236.124,138.254 1237.18,138.458 1237.828,138.744" id="path4478"/>
      <path d="M 1244.877,140.04 C 1245.144,139.944 1245.411,139.848 1245.677,139.752 C 1244.695,139.233 1243.532,138.925 1242.507,139.536 C 1243.312,139.637 1244.082,139.897 1244.877,140.04" id="path4488"/>
      <path d="M 1243.728,138.168 C 1243.97,138.168 1244.21,138.144 1244.448,138.096 C 1244.012,137.715 1243.509,137.572 1242.938,137.664 C 1243.201,137.832 1243.464,138 1243.728,138.168" id="path4490"/>
      <path d="M 1223.928,161.567 C 1223.347,161.447 1222.744,161.844 1222.418,162.288 C 1222.946,162.102 1223.45,161.861 1223.928,161.567" id="path4450"/>
      <path d="M 1228.178,160.272 C 1227.605,160.077 1227.051,160.125 1226.518,160.416 C 1227.092,160.615 1227.646,160.567 1228.178,160.272" id="path4456"/>
      <path d="M 1226.667,161.28 C 1226.926,161.385 1227.189,161.361 1227.457,161.208 C 1227.194,161.232 1226.931,161.256 1226.667,161.28" id="path4458"/>
      <path d="M 1229.118,162.504 C 1230.18,162.736 1231.269,162.777 1232.348,162.864 C 1231.409,161.596 1228.486,161.565 1227.098,161.784 C 1227.734,162.117 1228.423,162.332 1229.118,162.504" id="path4460"/>
      <path d="M 1234.518,156.672 C 1234.751,156.706 1235.668,156.448 1235.154,156.022 C 1234.527,155.502 1233.473,156.328 1232.859,156.457 C 1233.536,157.085 1234.482,157.142 1235.309,156.816 C 1235.044,156.768 1234.781,156.72 1234.518,156.672" id="path4466"/>
      <path d="M 1234.438,158.976 C 1234.175,158.976 1233.911,158.976 1233.648,158.976 C 1233.886,159.184 1234.149,159.351 1234.438,159.48 C 1234.438,159.312 1234.438,159.144 1234.438,158.976" id="path4468"/>
      <path d="M 1235.598,160.56 C 1235.104,160.42 1234.419,160.772 1234.008,160.993 C 1234.58,161.001 1235.109,160.857 1235.598,160.56" id="path4472"/>
      <path d="M 1234.158,160.056 C 1234.673,159.986 1235.177,159.867 1235.668,159.697 C 1235.121,159.451 1234.529,159.576 1234.158,160.056" id="path4474"/>
      <path d="M 1239.268,148.319 C 1238.523,147.01 1235.503,147.833 1234.658,148.608 C 1236.165,149.046 1237.763,148.457 1239.268,148.319" id="path4476"/>
      <path d="M 1240.917,162.216 C 1239.797,161.937 1238.397,162.985 1237.607,163.656 C 1238.69,163.826 1240.943,163.797 1240.917,162.216" id="path4480"/>
      <path d="M 1239.478,157.392 C 1238.865,157.148 1238.246,157.449 1237.898,157.968 C 1238.424,157.775 1238.951,157.584 1239.478,157.392" id="path4482"/>
      <path d="M 1240.708,156.168 C 1240.189,156.293 1239.92,156.221 1239.908,156.888 C 1240.189,156.664 1240.456,156.424 1240.708,156.168" id="path4484"/>
      <path d="M 1243.658,153.937 C 1242.872,153.894 1241.64,154.452 1241.638,155.376 C 1242.357,154.911 1243.226,154.906 1244.018,154.656 C 1243.897,154.416 1243.778,154.177 1243.658,153.937" id="path4486"/>
      <path d="M 1224.578,172.296 C 1224.066,171.965 1222.973,172.002 1222.268,172.296 C 1223.063,172.692 1223.749,172.202 1224.578,172.296" id="path4448"/>
      <path d="M 1192.678,195.552 C 1191.695,195.355 1190.656,195.696 1189.728,195.984 C 1190.818,196.133 1191.867,196.606 1192.678,195.552" id="path4424"/>
      <path d="M 1192.538,192.672 C 1195.525,192.152 1198.52,194.063 1201.468,193.032 C 1198.729,192.639 1193.812,190.118 1191.458,192.599 C 1191.816,192.65 1192.176,192.674 1192.538,192.672" id="path4426"/>
      <path d="M 1207.948,195.479 C 1205.581,194.339 1196.479,192.994 1194.408,195.551 C 1196.387,195.368 1199.964,195.134 1201.81,195.897 C 1202.29,196.096 1203.225,197.052 1203.763,196.954 C 1204.949,196.737 1206.186,196.907 1207.368,197.062 C 1207.297,196.485 1206.863,196.125 1206.288,196.126 C 1206.878,196.006 1207.431,195.79 1207.948,195.479" id="path4432"/>
      <path d="M 1216.297,184.896 C 1215.704,184.819 1209.931,185.48 1211.321,186.488 C 1213.394,187.991 1217.687,187.198 1219.967,186.697 C 1219.192,186.464 1218.872,185.693 1218.214,185.297 C 1217.631,184.945 1216.924,185.078 1216.297,184.896" id="path4434"/>
      <path d="M 1220.118,170.856 C 1220.355,170.664 1220.591,170.472 1220.828,170.28 C 1220.044,170.28 1219.223,170.17 1218.528,170.639 C 1219.052,170.755 1219.582,170.827 1220.118,170.856" id="path4436"/>
      <path d="M 1221.908,182.952 C 1222.441,182.768 1222.948,182.528 1223.428,182.231 C 1222.946,181.745 1222.209,181.518 1221.558,181.799 C 1221.933,181.643 1222.244,181.403 1222.488,181.079 C 1221.819,180.846 1220.886,180.566 1220.188,180.863 C 1219.454,181.176 1219.378,182.053 1218.598,182.302 C 1219.41,183.05 1220.875,183.462 1221.908,182.952" id="path4438"/>
      <path d="M 1222.268,171.144 C 1221.472,171.045 1220.838,171.637 1220.038,171.504 C 1220.735,171.966 1221.943,172.162 1222.268,171.144" id="path4442"/>
      <path d="M 1222.778,170.712 C 1222.228,170.565 1221.725,170.66 1221.268,171 C 1221.77,170.897 1222.273,170.801 1222.778,170.712" id="path4446"/>
      <path d="M 1224.788,171.72 C 1224.129,171.451 1223.116,171.796 1222.488,172.008 C 1223.255,171.909 1224.049,172.007 1224.788,171.72" id="path4452"/>
      <path d="M 1228.388,183.815 C 1227.663,183.186 1226.878,183.479 1226.088,183.815 C 1226.778,184.363 1227.837,184.45 1228.388,183.815" id="path4454"/>
      <path d="M 1234.008,178.488 C 1233.294,178.357 1233.133,179.027 1232.498,179.063 C 1231.892,179.098 1231.377,178.561 1230.781,178.668 C 1229.503,178.899 1228.484,179.81 1228.388,181.15 C 1230.024,180.897 1231.997,180.743 1233.642,180.976 C 1234.878,181.152 1235.98,181.62 1236.748,180.287 C 1236.071,180.004 1235.675,180.464 1235.031,180.521 C 1234.214,180.595 1233.561,180.153 1232.928,179.71 C 1233.457,179.516 1233.987,179.324 1234.518,179.134 C 1234.348,178.92 1234.178,178.704 1234.008,178.488" id="path4462"/>
      <path d="M 1232.138,173.448 C 1232.373,173.022 1233.057,170.635 1232.84,170.203 C 1232.191,168.913 1230.471,170.554 1232.138,170.424 C 1231.707,170.568 1230.649,170.591 1230.585,171.223 C 1230.507,172.009 1231.332,173.891 1231.778,174.529 C 1232.053,174.219 1232.173,173.859 1232.138,173.448" id="path4464"/>
      <path d="M 1152.428,229.464 C 1152.935,229.709 1153.442,229.709 1153.948,229.464 C 1153.517,228.968 1152.806,228.781 1152.428,229.464" id="path4414"/>
      <path d="M 1183.388,208.44 C 1182.389,208.419 1181.381,208.746 1180.449,209.065 C 1178.876,209.602 1177.424,209.381 1175.828,209.664 C 1174.313,209.932 1173.065,210.853 1171.607,211.26 C 1171.147,211.368 1170.684,211.461 1170.218,211.536 C 1169.529,211.675 1169.394,212.499 1168.708,212.688 C 1170.761,213.861 1173.749,212.567 1175.906,212.361 C 1178.852,212.079 1181.762,212.305 1184.618,211.248 C 1184.376,211.033 1184.137,210.817 1183.898,210.6 C 1185.415,210.454 1184.074,208.953 1183.108,209.088 C 1183.201,208.872 1183.294,208.656 1183.388,208.44" id="path4418"/>
      <path d="M 1174.248,209.808 C 1173.416,209.59 1172.37,209.527 1171.588,209.953 C 1171.978,211.158 1173.632,210.283 1174.248,209.808" id="path4420"/>
      <path d="M 1179.868,212.399 C 1179.397,212.083 1178.691,212.342 1178.278,212.616 C 1178.856,212.902 1179.386,212.83 1179.868,212.399" id="path4422"/>
      <path d="M 1193.547,218.088 C 1192.999,217.866 1192.354,218.137 1191.957,218.521 C 1192.544,218.774 1193.156,218.572 1193.547,218.088" id="path4428"/>
      <path d="M 1203.627,199.512 C 1202.984,199.582 1202.644,199.207 1202.827,198.576 C 1204.039,198.853 1205.323,199.175 1206.577,199.08 C 1206.542,198.252 1204.798,197.674 1204.197,197.424 C 1203.889,197.478 1203.581,197.482 1203.272,197.437 C 1202.641,197.035 1202.044,196.68 1201.437,196.248 C 1200.451,195.546 1198.224,195.799 1197.063,195.788 C 1195.493,195.772 1194.1,195.301 1194.159,197.141 C 1194.207,198.621 1199.001,199.151 1200.059,199.613 C 1201.367,200.184 1204.541,202.267 1205.427,200.089 C 1204.816,199.932 1204.216,199.739 1203.627,199.512" id="path4430"/>
      <path d="M 1077.268,258.84 C 1077.091,258.741 1077.125,258.426 1076.548,258.84 C 1076.788,258.84 1077.027,258.84 1077.268,258.84" id="path4374"/>
      <path d="M 1078.058,259.344 C 1078.848,259.247 1079.645,259.273 1080.438,259.273 C 1080.366,257.916 1078.479,258.581 1078.058,259.344" id="path4376"/>
      <path d="M 1083.667,257.544 C 1083.727,256.896 1083.217,256.482 1082.597,256.536 C 1083.204,255.947 1083.308,254.721 1082.57,254.165 C 1082.389,254.029 1081.045,253.658 1081.39,254.347 C 1082.176,255.916 1081.863,257.004 1083.667,257.544" id="path4378"/>
      <path d="M 1089.648,251.136 C 1089.888,250.992 1090.128,250.848 1090.368,250.704 C 1089.878,250.529 1089.398,250.554 1088.928,250.776 C 1089.168,250.896 1089.408,251.016 1089.648,251.136" id="path4382"/>
      <path d="M 1101.238,248.903 C 1101.56,248.363 1102.604,248.416 1103.131,248.339 C 1104.657,248.115 1103.058,247.254 1104.408,247.392 C 1103.247,246.651 1102.192,246.059 1101.047,247.269 C 1100.518,247.827 1099.865,249.087 1101.238,248.903" id="path4386"/>
      <path d="M 1105.488,248.76 C 1105.754,248.546 1105.994,248.306 1106.208,248.04 C 1105.687,247.7 1105.184,247.724 1104.698,248.112 C 1104.938,248.357 1105.201,248.573 1105.488,248.76" id="path4390"/>
      <path d="M 1106.928,247.608 C 1107.357,247.408 1107.596,247.072 1107.648,246.599 C 1107.07,246.671 1106.79,247.032 1106.928,247.608" id="path4392"/>
      <path d="M 1107.938,247.176 C 1108.386,246.709 1108.916,246.277 1109.518,246.025 C 1108.766,245.919 1107.894,246.292 1107.938,247.176" id="path4394"/>
      <path d="M 975.958,169.488 C 978.975,170.219 982.104,170.226 985.178,169.92 C 985.029,171.345 982.416,170.701 981.542,170.684 C 979.885,170.652 977.017,171.213 975.528,171.937 C 976.549,172.933 978.608,173.471 979.998,173.521 C 978.392,173.837 980.075,174.07 980.611,174.155 C 981.388,174.28 982.177,174.251 982.961,174.272 C 984.652,174.315 986.62,175.277 988.269,174.6 C 987.442,174.287 986.97,173.395 986.039,173.304 C 987.811,173.096 987.947,172.614 989.411,173.698 C 990.333,174.38 992.823,172.098 994.469,173.016 C 993.895,173.214 993.392,173.527 992.959,173.953 C 993.926,174.669 995.03,174.33 995.646,173.515 C 995.819,173.286 997.247,173.714 997.761,173.431 C 998.108,173.33 998.456,173.331 998.807,173.436 C 999.332,173.494 999.331,172.949 999.771,172.852 C 1000.765,172.632 1002.074,172.344 1001.959,173.521 C 1002.453,173.536 1009.779,171.001 1008.939,173.232 C 1010.043,172.324 1011.752,172.815 1012.999,173.054 C 1014.773,173.395 1016.603,173.207 1018.369,173.592 C 1018.229,173.834 1018.086,174.073 1017.939,174.312 C 1018.694,174.467 1021.951,173.787 1021.539,174.96 C 1022.071,174.93 1022.598,174.857 1023.119,174.744 C 1022.872,174.952 1022.632,175.168 1022.399,175.391 C 1023.424,175.716 1024.469,175.162 1025.499,175.463 C 1023.951,175.52 1025.311,176.409 1025.91,176.438 C 1026.667,176.473 1027.405,176.23 1028.159,176.255 C 1026.166,176.824 1028.781,176.364 1028.169,177.576 C 1027.63,178.644 1026.221,178.386 1025.279,178.703 C 1026.104,179.047 1026.776,178.375 1027.589,178.415 C 1026.901,178.739 1026.349,179.955 1027.589,179.855 C 1027.344,180.041 1027.105,180.233 1026.869,180.43 C 1028.338,181.311 1031.008,180.81 1032.119,182.23 C 1031.546,182.625 1030.426,182.396 1030.469,183.382 C 1031.887,182.44 1032.826,184.514 1033.78,184.174 C 1033.063,184.913 1031.709,184.994 1030.759,184.822 C 1031.454,185.285 1032.28,185.211 1033.059,185.11 C 1032.362,185.574 1031.538,185.44 1030.759,185.542 C 1031.243,185.875 1031.769,185.971 1032.339,185.83 C 1030.35,186.494 1028.191,186.398 1026.219,187.126 C 1028.335,187.472 1030.466,186.741 1032.479,186.19 C 1032.081,186.689 1031.558,187.094 1030.969,187.341 C 1031.498,187.572 1032.052,187.692 1032.629,187.701 C 1030.866,187.885 1032.916,188.638 1033.349,188.709 C 1032.463,188.787 1031.993,189.667 1033.199,189.789 C 1033.018,189.862 1029.527,189.864 1031.619,190.077 C 1029.883,190.399 1033.996,191.638 1034.209,192.381 C 1033.642,192.064 1030.82,192.812 1032.629,193.173 C 1031.769,193.38 1031.155,194.384 1030.179,193.965 C 1031.305,194.558 1032.267,194.03 1033.269,194.397 C 1034.785,194.95 1033.931,196.243 1032.699,196.341 C 1033.215,196.599 1033.719,196.575 1034.209,196.27 C 1033.122,196.647 1032.291,197.505 1031.189,197.854 C 1032.201,198.133 1033.247,198.189 1034.279,198.358 C 1031.966,198.904 1029.455,197.571 1027.229,198.645 C 1028.32,199.723 1030.699,198.374 1031.839,199.221 C 1030.689,199.302 1029.902,200.212 1028.849,200.554 C 1027.854,200.877 1026.459,201.157 1025.719,201.958 C 1028.291,203.174 1030.161,201.229 1032.479,200.446 C 1030.645,201.676 1029.294,202.386 1027.087,202.634 C 1025.673,202.793 1023.665,203.166 1022.548,204.118 C 1023.297,204.274 1023.375,205.002 1023.652,205.579 C 1023.969,206.238 1025.839,206.731 1024.918,205.413 C 1026.146,206.385 1027.956,206.555 1029.459,206.4 C 1030.151,206.329 1030.843,206.175 1031.486,205.904 C 1033.243,205.161 1031.609,205.117 1032.189,203.829 C 1032.593,204.31 1033.099,204.478 1033.709,204.333 C 1032.792,203.03 1036.801,204.212 1036.403,201.968 C 1036.262,201.171 1035.553,200.963 1035.869,200.085 C 1036.212,199.132 1036.917,199.365 1037.739,199.365 C 1035.004,199.461 1037.26,201.587 1036.659,203.396 C 1037.742,202.875 1039.365,202.12 1040.549,201.957 C 1039.534,202.407 1037.616,203.23 1036.799,203.973 C 1038.27,204.159 1039.827,204.398 1041.275,204.531 C 1042.615,204.653 1043.985,203.493 1045.299,203.901 C 1043.015,205.88 1039.495,204.572 1036.949,205.99 C 1039.887,206.188 1043.216,204.765 1046.019,205.486 C 1044.888,206.139 1043.423,205.684 1042.199,205.918 C 1040.867,206.172 1039.866,207.191 1038.529,207.43 C 1040.089,208.436 1042.305,206.581 1043.859,206.351 C 1044.179,207.157 1043.867,207.408 1042.923,207.101 C 1041.949,207.197 1041.021,207.501 1040.118,207.863 C 1041.224,208.786 1043.511,208.006 1044.788,207.935 C 1042.763,208.46 1045.824,208.789 1046.308,208.726 C 1044.52,209.933 1042.582,207.664 1040.838,208.726 C 1041.877,209.039 1042.705,209.657 1043.823,209.719 C 1044.567,209.76 1045.399,209.77 1046.132,209.947 C 1046.927,210.139 1045.932,210.968 1045.381,210.721 C 1044.422,210.29 1043.621,210.657 1042.639,210.671 C 1043.183,211.247 1045.456,211.176 1046.309,211.463 C 1045.779,211.378 1045.249,211.378 1044.719,211.463 C 1044.951,211.711 1045.191,211.951 1045.44,212.183 C 1044.631,211.776 1043.368,211.797 1042.49,211.895 C 1042.898,212.201 1045.098,212.847 1043.21,212.688 C 1044.244,213.377 1045.415,212.931 1046.31,212.255 C 1045.811,212.593 1045.726,213.075 1045.332,213.473 C 1044.768,214.044 1043.336,213.396 1042.711,213.257 C 1040.488,212.763 1039.131,212.287 1037.09,211.242 C 1035.083,210.216 1032.907,211.068 1030.816,210.543 C 1028.88,210.057 1026.591,209.983 1024.919,211.247 C 1025.999,212.007 1026.905,213.006 1028.237,213.316 C 1030.066,213.742 1032.306,213.34 1033.799,214.743 C 1035.169,216.031 1036.209,215.731 1037.954,216.049 C 1038.238,216.101 1038.955,216.391 1039.244,216.285 C 1040.019,216.002 1040.974,215.188 1041.769,215.927 C 1041.259,216.019 1040.755,216.139 1040.259,216.287 C 1042.207,216.542 1044.077,215.714 1046.019,215.783 C 1045.695,216.356 1044.836,216.405 1044.359,216.791 C 1044.965,216.85 1045.249,217.292 1045.079,217.871 C 1043.912,217.807 1042.925,217.661 1041.989,218.519 C 1042.581,218.352 1043.084,218.496 1043.499,218.951 C 1042.668,218.779 1041.02,218.294 1040.979,219.671 C 1041.855,219.671 1043.132,219.101 1043.929,219.671 C 1041.889,219.323 1039.647,220.585 1038.239,221.902 C 1039.604,222.313 1040.369,221.243 1041.604,221.243 C 1042.154,221.243 1042.31,222.134 1042.918,222.262 C 1042.2,222.578 1041.389,222.265 1040.758,222.838 C 1041.275,222.9 1041.756,223.067 1042.198,223.342 C 1041.655,223.329 1041.175,223.497 1040.758,223.846 C 1040.676,222.515 1039.243,222.995 1038.388,223.054 C 1038.632,222.867 1038.872,222.675 1039.108,222.478 C 1038.178,222.228 1037.471,222.929 1037.156,223.781 C 1036.869,224.555 1035.867,224.459 1035.438,225.215 C 1036.475,225.465 1037.44,224.836 1038.458,224.783 C 1037.802,225.002 1036.475,225.22 1037.088,226.222 C 1035.986,225.764 1034.903,226.806 1033.791,226.707 C 1032.681,226.607 1031.428,226.253 1030.319,226.51 C 1030.793,226.389 1031.25,226.221 1031.689,226.006 C 1030.71,226.493 1029.637,225.833 1028.668,226.439 C 1029.197,226.418 1029.723,226.442 1030.248,226.51 C 1029.224,226.455 1028.085,226.566 1027.148,227.014 C 1029.049,227.014 1026.023,227.742 1025.626,227.823 C 1024.571,228.04 1023.471,228.156 1022.478,228.598 C 1023.749,228.638 1025.018,228.015 1026.288,228.31 C 1025.663,228.296 1025.077,228.602 1024.708,229.103 C 1025.591,228.235 1028.018,229.181 1029.204,229.015 C 1030.505,228.832 1032.784,229.297 1033.708,228.094 C 1033.112,228.196 1032.605,228.028 1032.188,227.59 C 1033.749,228.026 1035.248,227.192 1036.797,227.158 C 1036.128,227.471 1035.374,227.478 1034.702,227.784 C 1034.081,228.067 1034.032,228.793 1033.267,228.885 C 1033.763,229.016 1034.246,229.184 1034.717,229.389 C 1033.973,229.331 1033.203,229.035 1032.477,229.389 C 1034.51,229.897 1036.338,231.077 1038.457,231.261 C 1036.198,231.702 1034.218,230.065 1031.977,230.289 C 1030.835,230.403 1029.652,230.467 1028.517,230.253 C 1027.299,230.024 1026.142,229.033 1024.847,229.461 C 1025.394,229.643 1025.897,229.906 1026.357,230.253 C 1024.707,229.76 1022.925,229.05 1021.177,229.461 C 1022.203,229.461 1022.448,229.749 1023.174,230.335 C 1024.369,231.298 1026.007,231.084 1027.437,231.045 C 1025.364,231.641 1023.449,230.108 1021.387,230.541 C 1021.878,230.724 1022.385,230.795 1022.907,230.757 C 1022.153,230.912 1021.369,230.987 1020.667,231.333 C 1022.49,231.279 1020.546,231.882 1019.957,231.477 C 1021.483,233.392 1017.445,231.088 1018.297,232.486 C 1017.476,232.156 1017.26,232.372 1017.648,233.133 C 1017.859,232.911 1018.075,232.695 1018.297,232.486 C 1018.818,233.507 1016.667,233.003 1016.927,234.358 C 1019.035,234.399 1020.639,232.72 1022.644,232.343 C 1025.273,231.848 1027.96,231.368 1030.607,230.901 C 1029.216,231.74 1027.536,231.276 1026.067,231.838 C 1026.904,232.67 1028.057,232.622 1029.097,232.271 C 1027.279,232.758 1030.481,232.904 1031.327,232.199 C 1031.144,231.929 1030.954,231.666 1030.757,231.407 C 1031.783,231.397 1032.796,230.556 1033.778,231.262 C 1033.04,231.419 1032.283,231.45 1031.548,231.622 C 1031.729,232.985 1034.085,232.345 1034.859,232.054 C 1033.051,231.857 1034.798,231.45 1035.244,231.447 C 1035.9,231.441 1036.569,231.635 1037.159,231.911 C 1035.222,232.428 1033.198,232.775 1031.189,232.775 C 1033.072,233.946 1034.976,232.894 1036.949,232.703 C 1035.732,233.234 1034.391,233.332 1033.129,233.711 C 1034.177,234.489 1035.558,234.47 1036.799,234.431 C 1034.636,235.045 1032.814,233.636 1030.748,233.333 C 1028.315,232.976 1025.65,232.686 1023.189,232.559 C 1025.277,232.571 1021.389,232.936 1020.959,233.135 C 1022.788,232.865 1021.008,233.302 1020.641,233.402 C 1019.748,233.644 1018.851,233.875 1017.974,234.173 C 1017.022,234.495 1014.79,235.197 1015.059,236.592 C 1016.11,236.405 1017.098,235.97 1018.142,235.755 C 1019.373,235.502 1020.538,235.958 1021.804,235.669 C 1024.829,234.978 1027.909,235.706 1030.9,235.008 C 1030.179,235.249 1029.429,235.399 1028.67,235.441 C 1029.509,238.143 1032.106,236.118 1034,236.665 C 1033.039,236.785 1029.907,237.766 1029.159,236.971 C 1028.062,235.805 1028.313,235.45 1026.447,235.624 C 1024.702,235.787 1022.938,235.807 1021.18,235.981 C 1020.193,236.079 1019.201,236.114 1018.214,236.211 C 1018.003,236.232 1014.835,236.526 1017.07,236.593 C 1016.092,236.835 1015.007,236.58 1014.12,237.169 C 1015.053,237.648 1016.091,237.521 1017.07,237.313 C 1016.352,237.557 1015.593,237.626 1014.84,237.601 C 1015.48,238.157 1016.304,238.011 1017.07,238.106 C 1016.066,238.193 1015.058,238.25 1014.049,238.25 C 1015.663,238.918 1017.408,238.615 1019.089,238.897 C 1017.862,238.857 1016.348,238.746 1015.349,239.618 C 1016.856,239.307 1021.073,237.511 1022.399,238.681 C 1021.412,238.875 1021.438,239.092 1022.479,239.329 C 1021.762,239.706 1021.09,239.269 1020.374,239.203 C 1019.551,239.128 1018.763,239.525 1017.94,239.329 C 1018.439,239.434 1018.942,239.506 1019.45,239.545 C 1018.814,240.117 1017.974,239.922 1017.22,240.121 C 1017.963,240.485 1018.751,240.14 1019.52,240.193 C 1018.018,240.659 1016.403,240.255 1014.91,240.768 C 1015.781,242.027 1018.963,241.72 1020.24,241.489 C 1019.279,241.875 1016.035,242.08 1014.99,241.777 C 1016.817,241.777 1014.705,241.597 1014.363,241.682 C 1013.592,241.873 1013.032,242.528 1012.385,242.942 C 1011.01,243.821 1015.182,243.522 1015.35,243.505 C 1014.133,243.727 1012.763,243.726 1011.82,244.657 C 1013.491,245.376 1015.596,244.219 1017.07,243.505 C 1018.829,242.652 1020.985,242.521 1022.534,241.249 C 1023.934,240.1 1025.872,239.74 1027.555,239.205 C 1029.853,238.474 1032.082,237.814 1034.5,237.601 C 1034.013,237.784 1033.509,237.856 1032.99,237.817 C 1033.696,238.195 1034.513,238.205 1035.29,238.177 C 1034.728,238.46 1033.963,238.372 1033.368,238.301 C 1032.099,238.147 1031.185,238.365 1029.96,238.681 C 1030.633,239.505 1032.139,239.306 1033.06,239.257 C 1031.374,239.785 1029.626,238.955 1027.935,239.417 C 1026.131,239.91 1024.206,240.48 1022.664,241.574 C 1021.198,242.614 1019.173,242.231 1017.753,243.366 C 1016.27,244.553 1014.243,244.549 1012.54,245.232 C 1013.037,245.349 1013.54,245.421 1014.05,245.449 C 1013.563,245.632 1013.059,245.704 1012.54,245.665 C 1013.385,246.46 1014.54,245.713 1015.42,245.449 C 1014.691,245.749 1013.952,246.031 1013.191,246.241 C 1014.457,247.323 1016.274,245.525 1017.651,245.665 C 1018.649,245.767 1017.804,246.548 1019.09,245.929 C 1020.018,245.482 1021.138,245.379 1022.121,245.089 C 1022.132,246.319 1021.289,245.581 1020.463,245.694 C 1019.54,245.82 1019.154,246.266 1018.425,246.716 C 1018.237,246.832 1016.855,246.503 1016.676,246.501 C 1015.364,246.489 1014.163,246.775 1012.971,247.319 C 1013.516,247.435 1014.046,247.387 1014.561,247.175 C 1014.09,247.529 1013.56,247.697 1012.971,247.679 C 1013.92,248.471 1015.217,247.944 1015.921,247.104 C 1015.94,247.673 1015.621,248.034 1015.061,248.112 C 1016.972,247.919 1014.728,248.704 1014.341,248.975 C 1015.211,249.481 1016.024,248.585 1016.641,248.112 C 1016.178,248.612 1015.674,249.068 1015.131,249.479 C 1016.098,249.111 1016.194,249.303 1015.421,250.055 C 1015.94,249.961 1016.46,249.878 1016.982,249.806 C 1018.053,249.63 1018.8,249.228 1019.669,248.597 C 1020.638,247.893 1021.891,247.579 1023.044,247.328 C 1024.515,247.008 1025.829,247.817 1027.232,247.536 C 1026.734,247.936 1026.878,248.104 1027.662,248.04 C 1025.169,248.46 1022.754,246.448 1020.323,248.45 C 1019.439,249.179 1018.553,249.207 1017.588,249.736 C 1016.988,249.977 1016.375,250.179 1015.749,250.341 C 1015.135,250.587 1015.068,251.463 1014.482,251.784 C 1015.209,251.498 1015.827,250.401 1016.722,250.92 C 1016.566,251.17 1016.396,251.409 1016.212,251.64 C 1016.709,251.46 1017.189,251.244 1017.652,250.993 C 1016.732,251.604 1015.686,252.335 1016.282,253.585 C 1015.909,253.536 1015.65,253.673 1015.504,253.996 C 1015.247,255.119 1017.479,254.062 1015.492,254.809 C 1016.63,254.755 1018.115,252.531 1019.242,253.37 C 1019.017,253.602 1018.777,253.818 1018.522,254.017 C 1019.279,253.751 1020.012,253.381 1020.822,253.297 C 1019.185,253.761 1016.537,254.368 1015.164,255.367 C 1014.718,255.692 1013.977,258.76 1014.051,259.273 C 1016.874,259.459 1018.544,253.966 1021.391,255.096 C 1020.811,255.242 1020.304,255.619 1019.961,256.105 C 1021.293,256.307 1023.731,255.271 1024.092,253.87 C 1024.235,253.317 1024.253,252.956 1023.731,253.058 C 1023.186,253.164 1023.536,251.744 1023.561,251.496 C 1024.015,251.919 1024.519,251.991 1025.071,251.712 C 1024,251.858 1023.957,253.256 1024.248,254.033 C 1024.957,255.92 1026.76,254.568 1027.951,254.16 C 1027.499,254.374 1025.523,256.049 1027.371,255.456 C 1025.007,256.825 1028.385,256.879 1028.241,258.336 C 1027.396,257.996 1026.234,257.803 1025.641,257.04 C 1024.961,256.165 1025.222,254.965 1023.701,255.528 C 1025.693,255.925 1022.519,255.9 1022.191,255.96 C 1022.686,256.24 1023.189,256.504 1023.701,256.752 C 1021.924,257.106 1024.532,257.533 1025.281,257.184 C 1024.582,258.088 1023.396,257.86 1022.41,257.804 C 1021.768,257.768 1018.865,258.289 1020.67,258.84 C 1019.141,259.993 1017.181,258.901 1015.49,259.344 C 1016.017,259.392 1016.543,259.441 1017.07,259.489 C 1016.527,259.494 1016.024,259.638 1015.56,259.921 C 1017.832,261.151 1021.846,259.52 1023.77,258.193 C 1023.31,258.58 1022.83,258.94 1022.33,259.273 C 1023.051,259.412 1024.726,258.543 1024.85,259.777 C 1024.366,259.305 1021.511,259.535 1020.832,259.779 C 1019.334,260.319 1017.374,260.867 1015.78,260.496 C 1016.557,260.729 1016.708,261.186 1016.07,261.72 C 1016.943,262.304 1018.139,261.609 1019.02,261.432 C 1018.263,261.662 1017.613,262.131 1016.86,262.369 C 1016.576,262.459 1013.189,263.353 1015.35,262.945 C 1013.792,263.222 1015.543,263.952 1016,264.529 C 1016.349,263.019 1019.022,263.524 1020.1,263.449 C 1019.125,263.809 1017.003,263.152 1016.57,264.457 C 1017.293,264.457 1018.013,264.409 1018.73,264.312 C 1017.762,264.637 1016.707,264.647 1015.78,265.105 C 1016.543,265.053 1017.054,265.68 1017.805,265.68 C 1018.654,265.68 1019.481,265.471 1020.31,265.32 C 1019.073,266.066 1016.924,265.198 1015.942,266.079 C 1015.224,266.724 1016.75,267.016 1017.22,266.689 C 1015.342,267.641 1017.848,266.89 1018.276,266.854 C 1019.169,266.779 1020.065,266.771 1020.961,266.76 C 1020.314,266.984 1018.549,267.357 1018.143,267.931 C 1017.61,268.685 1019.141,268.973 1019.601,268.489 C 1019.13,268.943 1018.72,269.447 1018.371,270.001 C 1019.102,269.577 1019.73,268.756 1020.671,268.849 C 1019.824,269.304 1019.006,269.845 1018.371,270.578 C 1019.442,270.56 1020.454,270 1021.321,269.426 C 1020.192,270.393 1018.429,270.925 1017.624,272.239 C 1016.675,273.788 1021.466,274.375 1022.121,274.538 C 1021.52,274.423 1020.962,274.719 1020.601,275.185 C 1021.115,275.159 1021.622,275.206 1022.121,275.33 C 1021.36,275.804 1020.559,275.185 1019.811,275.69 C 1020.832,275.893 1020.505,276.061 1019.811,276.481 C 1020.526,276.11 1021.299,276.482 1022.041,276.265 C 1022.847,276.029 1023.179,275.19 1023.991,274.969 C 1023.847,275.547 1024.12,275.865 1024.711,275.833 C 1023.915,275.822 1023.077,276.077 1022.352,276.384 C 1021.748,276.641 1019.748,278.556 1021.471,277.561 C 1021.3,277.824 1021.13,278.088 1020.961,278.354 C 1022.309,278.237 1022.96,276.024 1024.491,276.698 C 1023.659,276.893 1022.104,277.537 1021.901,278.498 C 1022.47,278.509 1022.996,278.364 1023.481,278.065 C 1022.922,278.329 1022.442,278.761 1022.121,279.29 C 1022.863,279.407 1023.607,279.269 1024.351,279.29 C 1023.239,279.513 1021.587,279.106 1020.671,279.938 C 1022.257,281.212 1024.201,279.702 1025.931,279.793 C 1024.971,279.981 1024.971,280.077 1025.931,280.081 C 1024.508,280.221 1022.454,280.112 1021.471,281.377 C 1022.475,280.63 1024.015,281.094 1025.072,281.449 C 1024.648,281.856 1024.145,282.024 1023.562,281.953 C 1023.415,283.03 1024.575,282.684 1025.142,282.457 C 1024.923,282.694 1024.706,282.935 1024.492,283.177 C 1025.678,283.27 1025.929,281.323 1027.732,281.665 C 1026.84,281.677 1026.274,282.43 1025.502,282.745 C 1025.908,283.703 1026.777,283.349 1027.522,283.105 C 1027.439,284.471 1025.028,283.888 1024.202,284.329 C 1025.426,284.297 1026.647,284.184 1027.872,284.184 C 1026.643,284.693 1025.31,284.916 1024.132,285.552 C 1025.337,285.66 1028.463,286.005 1029.392,285.048 C 1028.65,285.125 1027.869,285.305 1027.152,284.976 C 1028.901,284.794 1030.653,284.533 1032.412,284.76 C 1032.142,284.968 1031.879,285.185 1031.622,285.408 C 1032.252,285.921 1033.473,285.729 1033.062,284.688 C 1034.008,284.812 1036.501,282.718 1036.802,284.399 C 1037.461,284.179 1037.948,283.92 1036.802,283.536 C 1037.858,283.966 1040.051,282.518 1038.462,282.24 C 1039.285,282.615 1039.879,281.738 1040.692,281.807 C 1040.415,282.033 1040.152,282.273 1039.902,282.528 C 1040.142,282.695 1040.382,282.863 1040.622,283.032 C 1039.951,283.904 1038.243,283.551 1038.032,284.832 C 1039.47,284.16 1041.502,284.245 1042.202,282.528 C 1042.317,283.704 1042.916,283.148 1043.722,282.672 C 1042.113,284.908 1038.926,283.934 1036.952,285.552 C 1037.428,285.31 1037.931,285.214 1038.462,285.264 C 1037.712,285.558 1036.893,285.707 1036.232,286.201 C 1037.264,286.816 1038.032,285.097 1039.182,285.553 C 1038.925,285.753 1038.661,285.946 1038.392,286.129 C 1039.807,286.176 1041.052,285.013 1042.132,284.257 C 1042.795,285.391 1041.507,285.035 1040.771,285.379 C 1040.238,285.629 1037.616,286.705 1039.322,286.705 C 1038.55,286.912 1037.811,287.225 1037.092,287.568 C 1037.955,287.852 1038.474,286.909 1039.322,286.993 C 1037.858,287.765 1039.373,286.97 1039.252,288.217 C 1040.067,287.446 1040.923,286.67 1042.132,286.778 C 1041.27,286.953 1040.522,287.474 1039.972,288.147 C 1040.765,287.897 1041.479,287.459 1042.272,287.209 C 1041.713,287.473 1041.233,287.904 1040.912,288.433 C 1042.707,288.49 1043.562,286.694 1045.232,286.49 C 1044.339,286.984 1043.956,287.956 1043.227,288.62 C 1042.489,289.293 1041.051,289.691 1040.842,290.809 C 1042.426,289.999 1044.281,288.449 1045.952,288.001 C 1044.323,289.352 1042.367,289.99 1040.762,291.241 C 1041.164,291.82 1041.75,291.874 1042.202,291.312 C 1042.374,292.295 1043.128,292.048 1043.432,291.312 C 1042.566,293.169 1044.766,291.007 1045.082,290.808 C 1045.855,290.321 1046.779,290.622 1046.742,289.369 C 1047.243,289.983 1047.704,289.898 1047.962,289.152 C 1047.863,289.722 1047.599,290.202 1047.172,290.592 C 1047.605,290.669 1048.038,290.741 1048.472,290.808 C 1047.904,290.657 1047.424,290.801 1047.033,291.241 C 1047.471,291.605 1047.974,291.773 1048.543,291.745 C 1046.981,292.117 1048.353,292.356 1048.989,292.508 C 1049.821,292.706 1050.431,292.404 1050.273,291.457 C 1050.951,291.704 1051.62,292.05 1052.363,292.033 C 1052.284,290.973 1049.337,290.954 1048.613,290.808 C 1050.107,290.501 1051.637,291.952 1053.073,291.096 C 1052.96,290.527 1052.58,290.243 1052.003,290.376 C 1052.474,290.113 1052.978,290.017 1053.513,290.088 C 1053.176,289.259 1051.561,289.229 1053.003,288.864 C 1052.229,288.973 1051.531,289.341 1050.773,289.511 C 1051.333,289.244 1051.815,288.814 1052.143,288.287 C 1050.904,287.869 1049.523,287.943 1048.393,287.207 C 1049.244,287.518 1050.599,287.964 1051.273,287.062 C 1051.593,288.491 1054.904,288.378 1053.073,287.71 C 1054.06,287.991 1055.15,287.817 1056.103,287.494 C 1055.593,287.466 1055.09,287.394 1054.593,287.278 C 1055.416,287.408 1056.295,287.384 1056.823,286.63 C 1055.826,287.065 1054.906,287.133 1053.873,286.775 C 1054.729,286.86 1056.183,287.314 1056.393,286.054 C 1055.454,285.658 1054.401,285.805 1053.443,285.478 C 1054.627,285.638 1056.015,286.103 1057.183,285.623 C 1055.964,285.173 1054.599,285.259 1053.443,284.615 C 1054.335,284.916 1055.315,285.079 1056.258,284.993 C 1057.248,284.902 1057.467,283.987 1058.333,283.679 C 1056.769,283.984 1054.773,283.648 1053.293,283.104 C 1054.967,283.015 1056.909,283.93 1058.483,283.032 C 1057.726,282.807 1056.949,282.665 1056.173,282.528 C 1057.062,282.528 1058.55,283.08 1059.194,282.168 C 1058.719,282.232 1058.242,282.256 1057.763,282.24 C 1059.61,281.768 1056.724,281.608 1056.243,281.448 C 1057.535,281.582 1061.021,282.338 1060.713,280.008 C 1059.768,280.008 1058.569,280.311 1057.833,279.504 C 1058.741,279.693 1060.219,280.067 1061.13,279.778 C 1061.828,279.556 1062.098,278.76 1062.793,278.496 C 1061.824,279.069 1060.739,278.552 1059.913,277.992 C 1060.601,277.987 1064.725,278.762 1063.373,276.983 C 1064.033,277.107 1064.331,276.746 1064.093,276.12 C 1063.511,276.242 1062.935,276.387 1062.364,276.552 C 1063.038,276.099 1064.951,275.092 1064.407,274.002 C 1063.979,273.147 1062.037,272.862 1061.284,272.304 C 1062.75,272.451 1064.149,273.804 1065.674,273.312 C 1065.181,273.079 1064.704,272.815 1064.244,272.521 C 1065.698,272.811 1064.619,271.753 1063.719,271.828 C 1062.847,271.9 1061.634,271.04 1061.074,270.432 C 1061.836,270.795 1062.809,270.981 1063.647,270.985 C 1063.963,270.986 1064.455,270.966 1064.649,270.652 C 1064.757,270.409 1064.742,270.167 1064.603,269.928 C 1065.633,269.956 1065.656,269.836 1064.673,269.568 C 1065.569,269.742 1066.4,269.59 1067.194,270.146 C 1066.423,269.999 1065.524,270.049 1065.244,270.937 C 1065.865,270.931 1066.468,270.821 1067.051,270.608 C 1069.073,269.932 1066.57,270.193 1068.238,269.862 C 1068.751,269.76 1069.967,269.086 1070.364,269.281 C 1070.657,267.713 1067.31,268.956 1068.194,267.121 C 1068.565,267.939 1070.173,268.539 1071.004,268.273 C 1070.901,267.69 1070.474,267.24 1069.924,267.048 C 1070.412,267.048 1072.109,268.37 1072.361,267.486 C 1072.836,265.825 1070.34,266.724 1070.864,265.248 C 1071.532,265.555 1073.669,267.544 1074.483,266.923 C 1074.788,266.691 1072.685,265.183 1072.444,265.032 C 1073.226,265.29 1073.904,265.764 1074.604,266.183 C 1074.185,265.742 1073.705,265.382 1073.164,265.104 C 1073.873,265.566 1074.114,265.398 1073.884,264.599 C 1074.18,264.834 1076.27,266.4 1074.824,264.744 C 1075.34,265.251 1075.628,265.154 1075.991,264.589 C 1076.219,264.235 1077.229,264.681 1077.563,264.6 C 1077.288,263.164 1074.513,262.475 1073.313,262.728 C 1074.072,261.854 1075.327,262.714 1076.146,262.96 C 1077.2,263.275 1078.327,262.847 1079.363,263.231 C 1079.177,262.507 1079.925,261.634 1078.853,261.36 C 1079.877,261.201 1080.927,259.801 1079.278,259.65 C 1077.964,259.53 1076.382,260.109 1075.393,258.912 C 1076.357,257.844 1077.91,258.693 1079.104,258.233 C 1080.274,257.784 1081.088,258.563 1082.163,258.263 C 1082.136,257.698 1081.797,257.358 1081.233,257.327 C 1082.692,255.698 1080.584,255.071 1079.863,253.799 C 1080.622,253.705 1080.348,253.382 1080.634,252.909 C 1081.058,252.206 1082.616,252.613 1083.277,252.415 C 1083.755,252.272 1084.237,252.961 1084.614,253.125 C 1085.086,253.329 1087.731,252.823 1086.194,252.143 C 1087.19,252.309 1087.57,251.021 1088.644,251.352 C 1088.444,251.12 1088.254,250.879 1088.074,250.631 C 1088.332,250.478 1088.595,250.333 1088.864,250.199 C 1088.297,249.998 1087.72,249.83 1087.134,249.695 C 1087.506,248.41 1088.2,249.369 1089.01,249.245 C 1089.656,249.146 1089.829,248.037 1090.536,248.09 C 1091.092,248.132 1090.994,249.031 1091.454,249.118 C 1092.158,249.251 1093.227,248.625 1092.604,247.822 C 1093.866,248.289 1094.966,248.843 1096.354,248.758 C 1096.137,248.518 1095.92,248.279 1095.704,248.038 C 1096.652,248.069 1097.241,247.472 1097.004,248.686 C 1097.9,247.812 1098.892,248.676 1099.664,247.03 C 1099.015,246.728 1097.766,246.342 1098.654,245.446 C 1098.698,246.904 1100.45,246.284 1101.104,245.662 C 1101.491,245.294 1100.49,244.87 1101.424,244.723 C 1102.176,244.604 1102.242,243.642 1102.974,243.358 C 1102.421,243.295 1101.918,243.104 1101.464,242.782 C 1103.237,242.733 1104.956,243.267 1106.724,243.286 C 1105.265,243.386 1102.28,244.773 1101.964,246.382 C 1102.731,245.83 1104.205,246.849 1105.087,246.949 C 1106.284,247.086 1105.333,245.26 1106.724,245.374 C 1106.213,245.725 1106.044,246.293 1106.434,246.813 C 1107.162,245.727 1109.439,246.598 1109.604,245.013 C 1110.2,245.216 1110.455,245.727 1110.174,246.309 C 1111.013,245.77 1112.053,246.187 1112.849,245.498 C 1113.9,244.589 1114.109,244.893 1115.365,244.653 C 1114.966,243.868 1115.675,243.006 1114.565,242.636 C 1115.524,241.948 1115.401,243.089 1116.255,243.459 C 1116.937,243.754 1117.717,243.849 1118.455,243.86 C 1117.854,243.363 1118.012,242.807 1118.675,242.492 C 1118.26,244.569 1122.576,241.659 1123.205,241.555 C 1122.711,240.998 1122.968,240.429 1123.565,240.116 C 1123.404,241.763 1124.944,239.737 1123.855,241.268 C 1124.44,240.959 1125.682,239.382 1126.445,239.971 C 1126.261,239.279 1126.448,238.13 1127.455,238.532 C 1126.24,237.515 1129.696,237.954 1129.866,236.975 C 1130,236.205 1131.07,236.213 1130.554,235.364 C 1131.42,235.775 1132.077,234.858 1131.914,234.068 C 1132.909,234.376 1133.082,233.508 1133.769,233.096 C 1134.605,232.595 1136.316,232.278 1137.303,231.906 C 1137.967,231.656 1139.57,232.159 1139.92,231.509 C 1140.199,230.992 1140.025,230.115 1140.914,230.324 C 1140.853,229.303 1139.769,229.449 1139.124,229.028 C 1139.657,229.163 1140.16,229.091 1140.634,228.811 C 1140.319,228.373 1139.261,228.154 1139.155,227.598 C 1138.997,226.767 1139.942,226.935 1140.053,226.147 C 1140.257,226.887 1140.245,228.043 1141.273,227.299 C 1141.435,229.065 1142.471,228.563 1143.793,229.028 C 1142.264,229.158 1143.453,230.172 1144.432,230.168 C 1146.241,230.159 1147.98,230.334 1149.773,229.964 C 1149.507,229.43 1149.101,229.046 1148.553,228.812 C 1149.304,228.845 1149.636,229.463 1150.25,229.77 C 1150.926,230.108 1151.593,229.874 1152.293,229.821 C 1152.195,229.226 1151.752,228.821 1151.143,228.812 C 1151.861,228.677 1152.6,229.142 1152.886,228.208 C 1153.132,227.407 1154.23,228.567 1154.578,228.697 C 1155.271,228.956 1158.099,229.256 1157.762,227.804 C 1158.923,228.523 1160.393,228.084 1161.567,227.658 C 1162.201,227.428 1162.7,226.928 1163.413,227.279 C 1164.046,227.592 1164.623,227.21 1164.602,226.508 C 1165.153,227.331 1167.056,226.836 1167.773,226.58 C 1168.935,226.166 1170.019,226.431 1171.235,226.038 C 1172.117,225.753 1173.008,226.044 1173.887,225.804 C 1174.547,225.623 1174.846,224.924 1175.577,224.924 C 1177.121,224.924 1178.14,223.656 1179.582,223.34 C 1177.8,223.4 1180.241,222.835 1181.022,223.124 C 1180.752,221.825 1181.147,222.698 1182.029,222.527 C 1182.613,222.414 1183.379,221.481 1181.962,221.685 C 1182.814,220.735 1183.969,221.811 1184.982,221.468 C 1183.148,221.147 1185.848,221.636 1184.692,220.316 C 1185.387,220.5 1186.153,220.539 1186.867,220.474 C 1187.43,220.422 1189.608,219.597 1187.792,219.237 C 1188.597,219.288 1190.031,219.791 1190.714,219.412 C 1191.746,218.839 1191.468,217.929 1190.242,217.941 C 1190.865,217.995 1193.582,218.459 1193.262,217.148 C 1193.957,217.414 1194.71,218.018 1195.492,217.581 C 1195.077,217.218 1194.597,217.074 1194.052,217.148 C 1195.64,217.029 1198.857,216.637 1200.172,215.709 C 1199.445,214.859 1197.875,215.335 1196.956,215.449 C 1195.375,215.644 1193.468,215.577 1191.939,215.117 C 1190.606,214.716 1189.086,215.028 1187.816,214.392 C 1186.296,213.631 1185.077,213.731 1183.473,214.125 C 1183.698,213.868 1183.915,213.604 1184.123,213.333 C 1182.438,213.003 1180.829,213.949 1179.223,214.269 C 1177.121,214.687 1174.893,213.773 1173.127,215.346 C 1171.756,216.566 1171.333,215.56 1169.956,215.808 C 1168.752,216.025 1166.414,216.012 1165.323,215.421 C 1166.905,215.652 1170.056,215.486 1171.507,214.823 C 1172.399,214.417 1173.333,213.939 1174.323,213.837 C 1175.657,213.699 1176.913,214.088 1178.073,213.19 C 1176.403,211.469 1172.109,212.947 1170.078,213.154 C 1169.334,213.23 1168.595,213.373 1167.843,213.298 C 1167.137,213.227 1167.487,212.304 1166.754,212.558 C 1165.892,212.856 1163.419,213.407 1162.663,212.758 C 1164.105,212.867 1166.664,212.8 1167.807,211.775 C 1168.947,210.751 1171.609,210.028 1168.713,209.014 C 1169.817,209.364 1171.413,209.446 1172.523,209.086 C 1171.999,208.915 1171.611,208.483 1171.523,207.935 C 1172.319,207.723 1172.702,208.459 1173.169,208.929 C 1173.908,209.673 1175.257,209.397 1176.193,209.376 C 1177.963,209.335 1182.899,209.022 1183.698,207.3 C 1183.933,206.794 1185.368,207.863 1185.233,206.639 C 1185.157,205.958 1184.216,205.604 1183.669,205.462 C 1182.734,205.219 1181.76,205.532 1180.809,205.518 C 1179.218,205.495 1177.644,205.2 1176.053,205.2 C 1178.064,204.669 1175.004,204.688 1174.504,204.67 C 1173.003,204.613 1171.497,205.332 1170.004,204.912 C 1171.602,204.715 1174.942,205.312 1176.054,203.832 C 1175.507,203.709 1171.811,202.277 1173.825,202.536 C 1172.819,202.277 1171.755,202.284 1170.725,202.319 C 1171.271,201.774 1171.107,201.192 1170.435,200.88 C 1172.008,201.692 1173.537,201.918 1175.042,202.734 C 1176.166,203.343 1177.682,205.23 1179.037,205.026 C 1179.599,204.942 1180.02,204.528 1180.595,204.48 C 1181.369,204.414 1182.131,204.849 1182.895,204.913 C 1185.002,205.088 1186.328,206.726 1188.533,206.536 C 1189.799,206.427 1191.299,208.186 1191.097,209.398 C 1190.723,211.642 1192.547,212.022 1194.335,212.582 C 1195.333,212.895 1197.817,213.523 1198.743,212.66 C 1199.331,212.113 1198.895,209.557 1200.315,210.456 C 1198.449,213.477 1201.851,211.916 1202.19,212.83 C 1202.788,214.44 1205.792,211.449 1203.706,211.752 C 1203.843,211.161 1204.307,210.773 1204.926,210.815 C 1204.446,210.563 1203.943,210.371 1203.416,210.24 C 1204.313,210.679 1204.668,209.443 1205.356,209.16 C 1204.835,209.009 1204.309,208.984 1203.776,209.088 C 1204.016,208.848 1204.256,208.609 1204.496,208.368 C 1203.754,208.433 1203.013,208.535 1202.266,208.512 C 1203.029,208.334 1204.43,207.324 1205.216,207.792 C 1205.242,207.167 1204.874,206.896 1204.276,207 C 1204.802,206.924 1205.309,206.78 1205.796,206.567 C 1204.743,206.376 1204.743,206.184 1205.796,205.992 C 1203.853,205.452 1202.816,205.882 1201.256,207.072 C 1201.527,204.975 1203.027,204.294 1205.076,203.903 C 1203.7,203.878 1202.446,203.943 1201.256,204.84 C 1201.157,204.229 1201.557,203.852 1202.126,203.76 C 1201.071,203.374 1199.98,204.061 1199.026,204.408 C 1199.366,203.291 1201.417,202.799 1202.406,202.679 C 1201.449,201.799 1199.978,202.384 1199.181,201.71 C 1198.225,200.901 1196.467,201.276 1196.576,200.159 C 1195.281,200.697 1193.38,199.411 1192.193,199.008 C 1190.189,198.325 1188.992,200.146 1187.146,200.375 C 1187.672,200.086 1188.071,199.588 1188.226,199.007 C 1187.456,198.979 1186.655,199.198 1185.916,198.863 C 1186.615,198.888 1192.689,199.154 1192.221,197.532 C 1191.359,194.546 1185.094,196.88 1183.476,197.495 C 1183.724,197.238 1183.964,196.974 1184.196,196.703 C 1183.406,196.804 1182.61,196.775 1181.816,196.775 C 1182.364,196.71 1182.891,196.565 1183.396,196.342 C 1181.901,195.838 1180.236,196.576 1178.796,195.838 C 1180,195.921 1181.198,195.766 1182.401,195.766 C 1183.478,195.766 1184.409,196.46 1185.502,196.361 C 1186.597,196.261 1187.563,195.662 1188.656,195.55 C 1189.773,195.436 1190.986,195.534 1191.756,194.542 C 1190.83,194.01 1189.718,194.296 1188.726,194.038 C 1187.131,193.624 1185.832,193.296 1184.191,193.729 C 1183.055,194.029 1181.568,193.605 1180.565,194.092 C 1179.836,194.446 1179.345,195.287 1178.437,195.19 C 1179.39,194.924 1179.358,193.722 1178.287,193.75 C 1179.361,193.788 1180.454,193.298 1181.522,193.59 C 1182.77,193.931 1183.491,193.726 1184.407,192.814 C 1182.809,192.864 1181.441,192.085 1179.947,191.663 C 1181.114,191.022 1182.086,192.455 1183.193,192.546 C 1184.648,192.665 1186.087,193.267 1187.554,193.184 C 1188.904,193.108 1189.655,192.487 1190.801,191.931 C 1191.965,191.365 1193.318,191.498 1194.568,191.376 C 1193.319,190.56 1191.361,189.783 1190.388,188.64 C 1191.619,188.083 1193.045,189.933 1193.863,190.578 C 1195.149,191.59 1196.139,190.389 1197.191,189.722 C 1198.544,188.865 1200.342,189.502 1201.746,189.91 C 1203.84,190.519 1205.439,190.556 1207.598,190.513 C 1205.617,191.107 1204.052,190.773 1202.132,190.193 C 1201.081,189.876 1199.044,189.288 1198.448,190.657 C 1202.596,192.434 1205.738,193.934 1210.165,192.13 C 1211.981,191.39 1213.55,191.956 1215.476,191.812 C 1217.055,191.693 1216.308,191.576 1216.852,190.558 C 1217.226,189.861 1218.199,189.565 1218.318,188.713 C 1216.025,188.652 1213.753,187.521 1211.533,187.543 C 1209.055,187.567 1211.79,189.95 1209.968,190.297 C 1210.265,188.023 1209.645,188.064 1207.448,187.417 C 1208.265,187.211 1209.226,187.47 1209.688,186.553 C 1209.268,186.13 1208.761,185.89 1208.168,185.833 C 1209.396,186.054 1210.843,185.829 1211.918,185.185 C 1211.697,184.628 1211.241,184.186 1210.688,183.961 C 1211.614,183.853 1212.145,184.866 1213.185,184.844 C 1215.147,184.802 1217.09,184.897 1219.053,184.897 C 1219.646,184.897 1220.049,185.828 1220.557,186.049 C 1221.136,186.3 1221.804,186.229 1222.415,186.283 C 1224.169,186.442 1225.422,185.357 1226.887,184.609 C 1226.139,184.673 1225.142,184.236 1225.377,183.312 C 1223.875,183.372 1222.574,184.141 1221.104,183.406 C 1219.64,182.673 1217.907,183.767 1216.377,183.529 C 1217.171,183.484 1217.999,183.394 1218.687,182.953 C 1217.756,182.156 1218.758,181.307 1219.327,180.721 C 1217.573,180.497 1216.088,180.971 1214.4,181.332 C 1212.689,181.698 1212.231,180.295 1210.767,180.144 C 1212.283,180.223 1212.898,181.49 1214.589,180.963 C 1216.382,180.405 1218.29,180.79 1220.047,180.216 C 1218.043,178.234 1215.089,178.579 1212.567,177.912 C 1213.595,178.038 1214.642,177.865 1215.657,178.128 C 1214.904,177.212 1213.71,177.125 1212.637,176.903 C 1214.558,176.202 1216.473,177.899 1218.107,178.632 C 1218.921,178.997 1223.864,178.526 1222.857,179.928 C 1223.833,180.257 1225.119,180.364 1225.957,179.64 C 1226.762,178.945 1227.216,177.336 1226.097,176.689 C 1226.337,176.544 1226.577,176.4 1226.817,176.256 C 1223.973,173.486 1218.755,174.744 1215.227,174.744 C 1216.972,174.558 1218.734,174.747 1220.482,174.76 C 1222.192,174.772 1223.81,174.099 1225.456,174.121 C 1227.206,174.145 1226.452,172.739 1225.017,172.837 C 1223.319,172.954 1221.584,172.17 1219.908,172.511 C 1220.97,172.715 1221.877,173.355 1222.928,173.591 C 1221.838,173.713 1221.002,172.909 1219.978,172.727 C 1218.9,172.536 1217.883,173.178 1216.808,172.944 C 1217.033,172.686 1217.249,172.422 1217.458,172.151 C 1216.44,171.319 1214.761,171.687 1213.568,171.719 C 1214.704,170.99 1216.126,171.977 1217.318,171.36 C 1216.78,171.325 1216.251,171.373 1215.728,171.504 C 1216.228,171.355 1216.734,171.235 1217.248,171.144 C 1216.765,170.85 1216.258,170.802 1215.728,171 C 1217.187,170.757 1214.905,170.095 1214.218,171.144 C 1214.037,170.596 1213.587,170.185 1212.998,170.136 C 1213.304,169.837 1215.007,168.912 1215.422,168.873 C 1216.382,168.783 1216.764,169.877 1217.547,170.098 C 1218.949,170.493 1220.818,169.535 1222.137,169.128 C 1220.959,168.505 1219.574,168.928 1218.317,168.768 C 1219.981,168.289 1222.033,169.509 1223.577,168.552 C 1223.101,168.286 1222.598,168.215 1222.067,168.336 C 1224.531,167.55 1227.414,168.19 1229.873,168.707 C 1231.418,169.031 1234.536,169.992 1235.905,168.798 C 1238.587,166.46 1234.104,165.158 1232.279,166.28 C 1231.378,166.834 1230.61,166.265 1230.046,165.727 C 1229.061,164.788 1226.121,165.365 1224.878,165.311 C 1225.89,164.992 1226.946,165.428 1227.968,165.167 C 1226.432,164.794 1224.841,164.786 1223.288,164.52 C 1224.307,164.466 1225.297,164.822 1226.318,164.664 C 1224.372,164.292 1226.898,164.321 1227.178,164.376 C 1226.352,163.809 1225.067,163.616 1224.088,163.728 C 1227.163,162.774 1230.198,164.253 1233.298,164.016 C 1232.893,162.266 1228.783,162.449 1227.471,162.246 C 1226.223,162.054 1225.218,161.481 1223.939,161.935 C 1222.458,162.461 1221.679,164.095 1220.198,164.448 C 1222.259,163.731 1219.046,163.584 1218.608,163.584 C 1219.795,162.41 1221.289,161.617 1222.788,160.92 C 1222.285,160.875 1221.781,160.827 1221.278,160.776 C 1222.114,160.699 1223.588,160.87 1223.708,159.685 C 1223.784,158.929 1225.276,158.209 1225.878,157.968 C 1225.436,157.551 1224.933,157.454 1224.368,157.68 C 1224.765,157.491 1227.684,156.726 1225.668,156.529 C 1226.69,156.304 1227.851,155.815 1228.914,155.99 C 1229.871,156.147 1231.265,156.522 1231.894,155.491 C 1232.477,154.535 1233.132,154.87 1234.088,154.585 C 1234.693,154.404 1235.069,153.834 1235.678,153.648 C 1236.412,153.425 1236.508,154.207 1237.188,153.432 C 1235.591,152.969 1234.674,154.072 1233.368,153.937 C 1233.924,153.648 1234.404,153.264 1234.807,152.785 C 1235.075,152.484 1234.443,152.211 1234.924,151.971 C 1236.311,151.28 1236.635,150.399 1238.404,150.445 C 1241.605,150.529 1243.935,150.156 1246.907,148.896 C 1244.857,147.322 1241.56,148.711 1239.235,148.675 C 1238.001,148.656 1236.747,148.627 1235.53,148.871 C 1234.906,148.996 1232.16,150.476 1233.947,150.12 C 1232.434,150.673 1232.117,149.916 1230.777,149.975 C 1231.459,149.423 1233.674,148.826 1232.217,148.536 C 1233.836,148.374 1235.331,147.619 1236.961,147.532 C 1239.332,147.404 1241.763,147.969 1244.155,147.945 C 1247.744,147.908 1251.008,147.622 1254.247,146.232 C 1250.011,145.245 1245.641,146.323 1241.358,145.863 C 1239.573,145.671 1237.767,145.35 1235.967,145.512 C 1233.942,145.694 1231.963,146.214 1229.917,145.945 C 1233.223,145.544 1236.379,145.257 1239.706,145.559 C 1241.005,145.677 1242.336,145.762 1243.629,145.54 C 1244.667,145.362 1245.509,144.605 1246.555,144.621 C 1249.207,144.659 1251.844,145.32 1254.516,144.936 C 1255.794,144.753 1256.993,144.323 1258.301,144.351 C 1259.373,144.374 1260.889,144.59 1261.667,143.639 C 1261.104,143.656 1260.624,143.464 1260.227,143.063 C 1261.749,142.5 1263.36,143.021 1264.919,142.784 C 1266.147,142.597 1267.289,142.07 1268.52,141.881 C 1270.662,141.553 1273.596,141.661 1275.347,140.184 C 1271.583,139.883 1267.845,139.017 1264.06,138.681 C 1259.464,138.274 1254.87,138.637 1250.287,139.033 C 1250.454,139.275 1250.623,139.515 1250.797,139.753 C 1250.03,139.71 1247.354,140.4 1249.357,140.257 C 1247.873,141.026 1245.516,141.233 1243.86,140.976 C 1242.916,140.83 1242.058,140.459 1241.083,140.495 C 1239.559,140.55 1238.123,141.15 1236.607,141.265 C 1236.911,140.437 1237.766,140.121 1238.557,139.969 C 1234.261,140.532 1230.187,141.567 1226.032,142.781 C 1224.173,143.324 1221.847,143.536 1220.102,144.23 C 1218.232,144.974 1216.717,145.603 1214.727,146.089 C 1215.153,145.895 1217.545,145.233 1215.437,145.297 C 1217.613,144.824 1219.613,143.798 1221.656,142.951 C 1223.305,142.267 1225.075,141.985 1226.695,141.251 C 1227.641,140.823 1228.995,140.232 1229.501,139.254 C 1230.3,137.709 1227.103,137.629 1226.357,137.675 C 1224.785,137.772 1223.068,137.722 1221.523,137.991 C 1219.744,138.3 1219.065,139.803 1217.093,139.829 C 1214.713,139.861 1212.433,140.394 1210.056,140.597 C 1207.713,140.797 1204.954,141.794 1202.627,141.41 C 1203.132,141.424 1203.635,141.4 1204.137,141.338 C 1203.633,141.117 1203.106,140.997 1202.557,140.978 C 1204.396,140.016 1207.017,140.289 1209.04,140.077 C 1210.645,139.909 1213.604,140.26 1214.578,138.602 C 1213.022,137.33 1209.052,138.087 1207.207,138.052 C 1203.868,137.988 1200.464,137.847 1197.137,138.183 C 1190.759,138.827 1184.376,139.097 1178.008,139.97 C 1178.577,138.229 1182.4,138.413 1183.783,138.391 C 1186.877,138.342 1190.012,138.453 1193.048,137.738 C 1191.346,136.426 1188.532,137.163 1186.568,137.163 C 1190.307,136.717 1194.035,137.688 1197.777,137.397 C 1202.157,137.057 1206.527,137.405 1210.91,137.387 C 1215.15,137.369 1219.356,136.649 1223.593,136.635 C 1225.656,136.628 1227.659,136.713 1229.665,136.135 C 1231.117,135.717 1232.974,135.784 1234.308,135.074 C 1230.778,133.373 1226.331,133.852 1222.568,134.282 C 1222.865,134.075 1223.155,133.859 1223.438,133.634 C 1220.396,132.681 1217.22,134.307 1214.148,134.21 C 1216.02,134.104 1217.836,133.613 1219.618,133.058 C 1215.958,131.285 1211.5,132.699 1207.639,132.699 C 1202.843,132.699 1198.135,132.214 1193.34,132.668 C 1189.173,133.062 1185.287,133.171 1181.078,133.263 C 1180.048,133.285 1179.326,133.262 1178.444,133.846 C 1178.206,134.004 1176.535,133.488 1176.227,133.484 C 1174.118,133.456 1172.126,134.228 1170.089,134.212 C 1170.494,134.128 1172.744,133.746 1170.949,133.564 C 1175.435,132.999 1179.972,132.917 1184.489,132.917 C 1183.483,132.362 1182.346,132.479 1181.249,132.341 C 1183.986,132.341 1186.71,132.899 1189.449,132.629 C 1192.215,132.357 1194.992,132.197 1197.772,132.197 C 1203.906,132.197 1210.07,132.518 1216.169,131.693 C 1212.601,130.001 1208.364,130.884 1204.569,130.756 C 1206.732,131.019 1202.713,130.859 1202.269,130.756 C 1202.793,130.687 1203.32,130.662 1203.849,130.684 C 1202.953,130.317 1202.099,130.739 1201.189,130.684 C 1200.185,130.624 1199.252,130.18 1198.239,130.18 C 1196.176,130.18 1194.113,130.314 1192.049,130.252 C 1187.922,130.129 1183.783,129.796 1179.659,130.18 C 1179.95,130.323 1180.24,130.467 1180.529,130.613 C 1179.529,130.984 1179.045,130.261 1178.219,130.109 C 1177.51,129.979 1175.941,130.322 1175.11,130.358 C 1172.402,130.474 1169.384,129.709 1166.769,130.613 C 1167.322,130.675 1167.825,130.866 1168.279,131.188 C 1166.514,131.422 1164.857,130.412 1163.093,130.559 C 1161.242,130.713 1159.425,131.117 1157.558,130.9 C 1157.103,132.638 1163.934,132.381 1164.898,132.268 C 1162.24,132.919 1159.465,132.454 1156.768,132.701 C 1158.699,133.231 1160.71,132.688 1162.668,132.917 C 1159.977,132.917 1157.174,132.454 1154.528,133.133 C 1155.343,133.237 1155.892,134.099 1156.768,133.925 C 1156.026,134.56 1154.868,133.907 1154.149,133.596 C 1153.041,133.117 1151.896,133.154 1150.713,133.194 C 1148.004,133.284 1145.265,131.972 1142.578,132.557 C 1144.769,132.219 1141.324,131.65 1140.783,131.605 C 1139.488,131.497 1137.913,131.418 1136.678,131.909 C 1137.28,131.892 1137.88,131.916 1138.478,131.981 C 1137.046,132.201 1135.596,131.981 1134.159,131.981 C 1135.466,132.48 1136.909,132.273 1138.259,132.557 C 1136.32,132.451 1134.377,132.423 1132.442,132.25 C 1130.441,132.072 1128.447,132.654 1126.45,132.341 C 1128.221,132.95 1130.107,132.998 1131.92,133.421 C 1130.273,133.241 1128.615,133.192 1126.965,133.042 C 1125.008,132.862 1122.867,132.178 1120.979,133.061 C 1121.79,133.019 1122.625,133.13 1123.359,133.494 C 1122.256,133.431 1121.152,133.055 1120.049,133.349 C 1121.291,133.929 1122.744,133.863 1124.079,133.853 C 1124.911,133.847 1125.742,133.857 1126.574,133.831 C 1126.866,133.822 1129.47,133.913 1127.529,133.494 C 1131.941,133.72 1136.357,133.627 1140.767,133.804 C 1142.543,133.875 1144.306,134.122 1146.082,134.175 C 1146.684,134.193 1151.262,134.671 1149.199,135.366 C 1150.728,135.573 1149.194,136.383 1148.86,136.038 C 1148.295,135.455 1147.407,135.166 1146.653,134.938 C 1145.151,134.484 1142.815,133.867 1141.279,134.501 C 1141.529,134.634 1141.772,134.777 1142.009,134.933 C 1141.414,136.736 1140.666,134.5 1140.149,134.35 C 1138.642,133.911 1136.854,134.038 1135.31,134.093 C 1133.408,134.161 1131.502,134.179 1129.6,134.206 C 1127.882,134.231 1125.847,133.717 1124.219,134.43 C 1125.413,134.812 1126.633,135.109 1127.858,135.37 C 1128.479,135.502 1129.102,135.626 1129.727,135.744 C 1130.164,135.827 1132.825,135.891 1130.769,136.374 C 1132.885,136.787 1135.065,136.589 1137.179,137.021 C 1135.006,137.158 1132.745,136.937 1130.592,136.632 C 1129.698,136.506 1128.764,136.37 1127.864,136.53 C 1127.317,136.628 1125.258,137.527 1127.099,137.669 C 1126.304,137.827 1125.43,137.756 1124.729,138.245 C 1125.232,138.297 1125.735,138.345 1126.239,138.389 C 1124.459,138.959 1126.572,139.014 1127.029,139.037 C 1124.374,139.656 1121.942,139.163 1119.43,138.252 C 1117.351,137.498 1115.1,137.76 1113.046,137.023 C 1111.122,136.334 1108.603,136.05 1106.579,135.652 C 1103.872,135.119 1101.022,135.256 1098.299,135.581 C 1098.986,136.058 1098.667,136.944 1099.32,137.354 C 1100.385,138.021 1102.12,138.23 1103.339,138.461 C 1101.074,139.164 1098.488,138.867 1096.209,138.389 C 1097.649,139.211 1099.376,139.233 1100.819,140.045 C 1099.152,140.471 1097.766,139.483 1096.209,139.109 C 1094.27,138.643 1092.154,138.522 1090.178,138.237 C 1089.55,138.146 1088.631,137.865 1088,138.013 C 1087.199,138.201 1086.742,139.034 1085.887,139.242 C 1084.684,139.535 1082.803,140.758 1081.599,140.548 C 1082.293,139.336 1084.372,139.177 1084.852,137.75 C 1085.602,135.522 1078.055,136.586 1077.199,136.66 C 1079.412,136.54 1074.77,137.026 1074.158,137.088 C 1072.658,137.239 1071.259,137.042 1069.757,137.143 C 1065.726,137.414 1061.664,137.347 1057.689,138.172 C 1058.46,138.457 1059.29,138.296 1060.069,138.532 C 1060.786,138.75 1061.257,139.426 1061.892,139.782 C 1062.586,140.172 1063.569,140.034 1064.33,140.25 C 1065.496,140.58 1066.591,141.119 1067.768,141.412 C 1065.591,141.487 1063.499,140.359 1061.297,140.243 C 1060.326,140.192 1060.038,140.027 1059.533,139.285 C 1058.99,138.486 1057.602,138.761 1056.775,138.707 C 1054.729,138.574 1052.665,138.294 1050.615,138.53 C 1049.344,138.676 1045.635,140.387 1048.838,140.837 C 1047.867,141.316 1046.631,141.115 1045.808,141.917 C 1045.996,142.161 1046.19,142.401 1046.388,142.637 C 1044.081,143.463 1041.329,142.547 1038.96,142.391 C 1037.736,142.31 1036.27,142.073 1035.166,142.774 C 1034.356,143.288 1035.323,144.319 1035.589,144.87 C 1034.944,144.379 1034.586,143.616 1034.001,143.067 C 1033.171,142.289 1031.2,143.226 1030.197,143.348 C 1027.329,143.698 1024.654,144.713 1021.803,145.115 C 1019.41,145.451 1017.067,145.869 1014.696,146.318 C 1013.417,146.56 1010.801,146.789 1010.169,148.181 C 1011.252,148.09 1012.242,148.86 1013.503,148.74 C 1014.825,148.614 1015.974,148.971 1017.276,149.006 C 1020.373,149.088 1023.651,147.902 1026.728,148.398 C 1024.532,148.319 1021.466,148.338 1019.688,149.83 C 1018.938,150.459 1018.995,151.472 1018.23,152.085 C 1017.188,152.92 1015.713,153.188 1014.443,153.459 C 1011.371,154.113 1009.003,154.765 1005.798,154.478 C 1004.011,154.318 1002.501,154.763 1000.749,154.963 C 998.442,155.228 996.078,155.073 993.829,155.742 C 995.953,156.067 993.12,156.305 992.781,156.341 C 991.547,156.474 990.3,156.436 989.073,156.642 C 988.07,156.81 987.046,156.807 986.041,156.96 C 985.157,157.095 984.343,157.749 983.504,157.819 C 981.793,157.963 980.295,157.683 978.527,157.993 C 977.369,158.196 976.21,158.173 975.043,158.28 C 973.968,158.379 973.236,159.332 972.158,159.414 C 974.118,159.83 970.553,160.667 969.918,160.494 C 971.147,160.731 971.963,161.633 973.088,162.078 C 974.171,162.506 975.696,162.505 976.838,162.366 C 976.326,162.545 975.82,162.737 975.318,162.942 C 977.796,163.567 980.249,162.376 982.738,162.582 C 981.455,162.99 980.111,163.263 978.778,163.446 C 980.276,164.155 982.61,163.883 984.108,163.301 C 982.868,164.197 981.15,164.048 979.718,164.309 C 981.682,164.974 983.905,165.14 985.965,164.986 C 986.948,164.912 987.204,164.828 987.698,164.078 C 988.168,163.364 989.27,164.646 989.902,164.732 C 991.986,165.014 993.855,164.443 995.82,163.874 C 997.114,163.499 999.657,162.948 1000.088,164.812 C 999.399,165.243 998.137,165.22 997.345,165.192 C 996.487,165.161 996.728,166.295 997.648,166.18 C 995.925,166.533 993.645,165.408 991.795,165.463 C 989.306,165.537 986.694,165.627 984.248,166.109 C 985.689,167.205 987.983,166.76 989.649,166.699 C 992.317,166.601 994.906,167.308 997.567,166.684 C 994.911,167.925 970.789,165.078 971.217,168.268 C 972.003,168.108 972.742,168.492 973.527,168.413 C 973.003,168.526 972.473,168.598 971.937,168.629 C 974.229,171.007 976.984,167.094 979.497,168.053 C 978.371,168.059 976.666,168.563 975.958,169.488" id="path4206"/>
      <path d="M 1017.648,267.983 C 1017.413,267.833 1017.173,267.857 1016.928,268.055 C 1017.168,268.031 1017.408,268.008 1017.648,267.983" id="path4232"/>
      <path d="M 1017.358,267.192 C 1017.115,267.499 1016.891,267.381 1016.998,267.912 C 1017.118,267.672 1017.238,267.433 1017.358,267.192" id="path4234"/>
      <path d="M 1020.528,274.608 C 1020.267,274.682 1020.027,274.802 1019.808,274.967 C 1020.048,274.848 1020.288,274.729 1020.528,274.608" id="path4244"/>
      <path d="M 1020.818,276.983 C 1020.568,277.055 1020.328,277.151 1020.098,277.272 C 1020.338,277.176 1020.578,277.08 1020.818,276.983" id="path4246"/>
      <path d="M 1021.898,278.928 C 1021.652,278.948 1021.412,278.995 1021.178,279.073 C 1021.418,279.024 1021.658,278.976 1021.898,278.928" id="path4250"/>
      <path d="M 1022.688,281.952 C 1022.415,281.934 1022.152,281.982 1021.898,282.096 C 1022.161,282.048 1022.425,282 1022.688,281.952" id="path4254"/>
      <path d="M 1042.848,288.72 C 1042.609,288.504 1041.567,289.555 1041.408,289.657 C 1041.983,289.49 1042.463,289.178 1042.848,288.72" id="path4350"/>
      <path d="M 1046.518,290.808 C 1046.505,291.079 1046.602,291.32 1046.808,291.529 C 1046.711,291.288 1046.614,291.048 1046.518,290.808" id="path4360"/>
      <path d="M 1050.408,292.032 C 1050.408,292.224 1050.408,292.416 1050.408,292.608 C 1050.699,292.454 1050.962,292.261 1051.198,292.032 C 1050.938,291.894 1050.675,291.894 1050.408,292.032" id="path4362"/>
      <path d="M 1056.458,285.12 C 1056.991,285.4 1057.495,285.352 1057.968,284.975 C 1057.457,284.948 1056.954,284.997 1056.458,285.12" id="path4366"/>
      <path d="M 1012.178,248.832 C 1012.807,249.121 1013.603,248.999 1013.688,248.184 C 1013.104,248.212 1012.601,248.428 1012.178,248.832" id="path4218"/>
      <path d="M 1013.118,249.983 C 1013.37,249.336 1013.175,248.976 1012.388,249.624 C 1012.631,249.744 1012.875,249.863 1013.118,249.983" id="path4220"/>
      <path d="M 1015.848,251.928 C 1015.522,251.865 1015.259,251.985 1015.058,252.288 C 1015.321,252.168 1015.584,252.048 1015.848,251.928" id="path4224"/>
      <path d="M 1018.148,258.408 C 1017.68,257.979 1016.981,258.422 1016.717,258.84 C 1017.238,258.843 1017.715,258.698 1018.148,258.408" id="path4230"/>
      <path d="M 1020.308,256.608 C 1019.364,256.204 1018.074,257.459 1017.358,257.903 C 1018.559,257.941 1019.539,257.545 1020.308,256.608" id="path4236"/>
      <path d="M 1021.967,256.824 C 1020.533,255.663 1019.168,258.574 1018.227,259.2 C 1019.64,258.771 1020.82,257.704 1021.967,256.824" id="path4240"/>
      <path d="M 1024.198,281.376 C 1023.691,281.25 1023.21,281.322 1022.758,281.592 C 1023.267,281.714 1023.747,281.643 1024.198,281.376" id="path4258"/>
      <path d="M 1026.938,285.768 C 1026.07,285.334 1025.135,285.576 1024.268,285.84 C 1024.747,287.101 1026.41,286.833 1026.938,285.768" id="path4264"/>
      <path d="M 1025.278,283.464 C 1025.015,283.56 1024.751,283.656 1024.488,283.752 C 1024.991,283.7 1025.494,283.652 1025.998,283.608 C 1025.758,283.561 1025.518,283.512 1025.278,283.464" id="path4266"/>
      <path d="M 1036.658,285.048 C 1036.681,284.904 1036.704,284.76 1036.728,284.616 C 1035.492,284.742 1033.762,285.144 1032.908,286.128 C 1034.028,285.395 1035.46,285.579 1036.658,285.048" id="path4326"/>
      <path d="M 1035.868,285.983 C 1035.315,286.046 1034.812,286.237 1034.358,286.559 C 1034.91,286.494 1035.413,286.303 1035.868,285.983" id="path4328"/>
      <path d="M 1037.738,286.488 C 1037.149,286.469 1036.619,286.637 1036.148,286.992 C 1036.714,286.939 1037.245,286.771 1037.738,286.488" id="path4334"/>
      <path d="M 1041.408,289.08 C 1040.124,288.802 1039.896,289.646 1039.388,290.592 C 1040.177,290.279 1041.017,289.891 1041.408,289.08" id="path4340"/>
      <path d="M 1042.417,288.649 C 1042.171,288.77 1041.891,288.873 1041.697,289.08 C 1041.938,288.937 1042.178,288.793 1042.417,288.649" id="path4352"/>
      <path d="M 1044.217,292.032 C 1044.879,292.112 1045.587,292.055 1046.157,291.672 C 1045.606,290.634 1044.533,291.125 1044.217,292.032" id="path4354"/>
      <path d="M 1045.587,292.032 C 1045.347,292.224 1045.107,292.416 1044.867,292.608 C 1045.483,293.13 1046.477,292.261 1047.237,292.032 C 1046.688,291.862 1046.137,291.862 1045.587,292.032" id="path4356"/>
      <path d="M 1047.167,293.76 C 1047.792,293.687 1048.322,293.423 1048.757,292.968 C 1047.563,292.141 1046.801,292.473 1045.947,293.544 C 1046.537,293.579 1047.04,293.387 1047.457,292.968 C 1047.361,293.231 1047.264,293.496 1047.167,293.76" id="path4358"/>
      <path d="M 1051.348,292.032 C 1051.202,292.272 1051.055,292.511 1050.908,292.752 C 1051.164,292.565 1051.311,292.325 1051.348,292.032" id="path4364"/>
      <path d="M 1065.888,272.592 C 1066.503,271.898 1066.384,271.657 1065.528,271.872 C 1065.648,272.112 1065.768,272.352 1065.888,272.592" id="path4368"/>
      <path d="M 1070.717,269.567 C 1071.169,269.286 1071.502,268.902 1071.717,268.416 C 1071.162,268.554 1070.75,268.982 1070.717,269.567" id="path4370"/>
      <path d="M 1073.598,267.624 C 1073.727,267.001 1073.484,266.978 1072.868,267.552 C 1073.111,267.576 1073.354,267.6 1073.598,267.624" id="path4372"/>
      <path d="M 1047.748,292.032 C 1048.011,291.936 1048.275,291.84 1048.538,291.744 C 1047.969,291.772 1047.466,291.604 1047.028,291.24 C 1047.738,290.467 1052.264,290.76 1052.358,292.032 C 1051.614,292.049 1050.945,291.703 1050.268,291.456 C 1050.553,293.167 1048.589,292.458 1047.748,292.032" id="path4534"/>
      <path d="M 1065.238,270.937 C 1065.611,269.755 1067.016,270.071 1067.908,270.289 C 1067.06,270.622 1066.166,270.958 1065.238,270.937" id="path4536"/>
      <path d="M 1070.858,265.176 C 1071.183,264.371 1072.017,264.155 1072.438,265.032 C 1071.928,264.694 1071.285,264.701 1070.858,265.176" id="path4538"/>
      <path d="M 1220.978,172.151 C 1221.707,172.196 1222.368,172.537 1222.998,172.872 C 1222.307,172.686 1221.631,172.443 1220.978,172.151" id="path4544"/>
    </g>
  </g>
  <path class="landxx ne" d="M 1272.9353,634.74955 C 1271.8763,633.49755 1273.1133,631.21555 1273.0053,629.78155 C 1274.2803,630.22855 1275.8523,630.10155 1277.1903,630.11655 C 1278.7673,630.13455 1279.1673,629.32655 1280.1023,628.24355 C 1281.2273,626.93955 1282.5363,627.26455 1284.1433,627.13955 C 1286.1913,626.97955 1288.2333,626.79255 1290.2893,626.75155 C 1291.2593,626.73255 1292.2093,626.73255 1293.1733,626.84555 C 1295.0723,627.06755 1293.2833,626.26355 1294.1753,626.03655 C 1295.5993,625.67155 1297.4643,627.59255 1298.0853,625.96555 C 1298.6303,624.53755 1300.3043,624.33955 1300.8673,623.04755 C 1301.4433,621.72255 1301.4243,620.30255 1302.1563,619.01055 C 1303.6843,616.31555 1303.3433,613.33155 1303.4963,610.25155 C 1303.7163,605.86155 1303.6463,601.46455 1303.6223,597.07055 C 1303.6073,594.33955 1303.8463,593.62855 1306.5593,592.99755 C 1308.9613,592.43855 1311.4453,592.30655 1313.8403,591.71755 C 1316.6663,591.02255 1318.5273,588.24455 1320.5303,586.29755 C 1323.4403,583.46855 1326.2263,580.38755 1329.4333,577.89155 C 1335.9693,572.80455 1343.2193,568.19755 1350.2003,563.74455 C 1353.4893,561.64655 1356.6473,559.34955 1359.9353,557.25055 C 1363.0133,555.28555 1367.1423,557.30255 1370.4193,558.19655 C 1373.2833,558.97955 1375.5903,560.84355 1377.5453,563.02255 C 1378.3893,563.96155 1378.6493,564.00555 1379.6813,563.29655 C 1381.1443,562.29255 1382.7163,561.48655 1384.2443,560.58855 C 1384.9693,564.40655 1385.5653,568.14355 1385.8633,572.00955 C 1385.9823,573.54655 1386.8843,574.60555 1387.6783,575.85855 C 1388.1313,576.57355 1388.5443,577.32255 1388.8513,578.11355 C 1389.1403,578.85455 1388.3263,579.45455 1388.9653,580.18755 C 1389.7883,581.13055 1391.1703,581.93355 1391.5183,583.20855 C 1391.9443,584.76855 1390.1693,586.37955 1390.0243,587.94055 C 1389.5923,592.59655 1388.8533,597.23655 1388.8533,601.92355 C 1388.8533,603.94355 1388.6473,605.95455 1388.5883,607.97255 C 1388.5413,609.59555 1388.7633,611.57955 1388.2213,613.13555 C 1387.7533,614.47655 1386.4153,615.61955 1385.5503,616.71455 C 1384.1473,618.49055 1382.6193,620.16255 1381.1963,621.92255 C 1378.8583,624.81655 1376.8733,628.13155 1375.5733,631.60655 C 1375.2603,632.44455 1374.8853,633.26055 1374.3123,633.95855 C 1374.0383,634.29155 1373.4673,634.19855 1373.2603,634.60455 C 1372.6113,635.87455 1374.1783,639.47355 1374.3823,640.87155 C 1373.6933,640.87155 1372.8073,640.69755 1372.1883,641.10155 C 1371.5313,641.53055 1371.6413,641.99155 1370.8933,642.32155 C 1369.2273,643.05555 1368.0213,643.73355 1366.7573,645.06455 C 1364.8113,647.11455 1363.6533,646.22255 1361.2903,645.26255 C 1358.5243,644.13955 1356.0393,643.98955 1353.0863,643.91155 C 1350.5613,643.84555 1348.3833,644.30655 1346.4623,646.04255 C 1345.8083,646.63355 1345.3283,647.31155 1344.8303,648.02755 C 1343.9703,649.26655 1343.4513,648.47955 1342.2083,648.59355 C 1339.2983,648.86055 1338.0493,647.38955 1335.6193,646.41855 C 1333.8423,645.70755 1331.8103,643.20355 1329.6683,644.33655 C 1328.8583,644.76455 1328.2463,645.47455 1327.4563,645.93355 C 1326.8483,646.28655 1325.8893,645.75355 1325.6163,646.34255 C 1324.4923,648.76355 1322.5383,645.76355 1322.0813,644.75555 C 1321.0033,642.37855 1319.3473,641.46855 1316.9353,641.03355 C 1315.5623,640.78655 1314.5253,639.62555 1313.1233,639.49755 C 1311.8813,639.38355 1311.3883,640.67255 1310.0663,640.64555 C 1307.6803,640.59855 1305.5003,640.39855 1303.7573,642.38155 C 1303.1843,643.03255 1302.5833,643.35055 1302.5043,644.24455 C 1302.3733,645.72655 1302.3283,646.97255 1301.4413,648.22055 C 1300.2453,649.90555 1298.7843,651.01855 1298.8663,653.25755 C 1298.8843,653.72855 1298.6263,658.50955 1298.3343,658.34555 C 1296.9143,657.54455 1293.8563,652.14455 1292.3683,652.32555 C 1291.9003,652.38255 1291.7063,652.95255 1291.2923,653.11055 C 1290.7423,653.32055 1289.5143,653.16655 1289.1453,653.72955 C 1288.9553,654.01855 1289.7833,656.41155 1289.2183,656.39355 C 1288.7123,656.37755 1287.5503,654.24855 1287.3113,653.85655 C 1286.5323,652.57755 1287.1353,652.31455 1288.2023,651.88755 C 1287.8613,651.17555 1287.7943,650.18555 1287.1243,649.67155 C 1285.9133,648.74455 1285.1663,650.67455 1283.9103,650.45255 C 1281.5723,650.03855 1277.7573,646.34555 1278.8423,643.89555 C 1279.4963,644.37055 1280.1383,644.38855 1280.8523,644.04055 C 1280.5033,643.74655 1275.9663,640.56155 1275.8163,640.86955 C 1276.1473,640.19255 1274.4683,638.92255 1274.2483,638.16555 C 1273.8723,636.86355 1274.1743,635.65255 1272.9353,634.74955" id="ne"/>
  <g class="landxx coastxx dk eu" id="dk" transform="translate(-29.9017, -45.0745)">
    <path d="M 1374.908,331.992 C 1375.408,332.32 1375.886,334.086 1376.858,333.431 C 1377.868,332.752 1375.624,331.58 1374.908,331.992" id="path4494"/>
    <path class="landxx dk eu" d="M 1365.118,319.464 C 1365.215,319.224 1365.311,318.985 1365.408,318.744 C 1365.825,319.206 1367.075,319.953 1367.718,319.536 C 1367.994,320.747 1368.797,320.307 1368.998,319.432 C 1369.484,317.323 1372.453,318.272 1370.877,319.033 C 1371.526,319.097 1372.52,318.911 1372.467,319.896 C 1372.697,318.99 1372.707,318.575 1371.747,318.672 C 1371.703,317.934 1371.504,316.836 1372.159,316.278 C 1372.586,316.002 1373.056,315.892 1373.568,315.947 C 1374.38,315.995 1374.837,315.649 1375.547,315.649 C 1376.411,315.649 1376.953,315.688 1377.763,316.01 C 1378.75,316.404 1378.449,317.635 1378.947,318.383 C 1377.941,318.289 1376.987,318.597 1376.067,318.959 C 1376.998,318.977 1377.647,318.265 1378.605,318.675 C 1379.173,318.918 1378.044,319.991 1378.007,320.399 C 1378.255,319.045 1381.864,319.757 1382.57,320.322 C 1383.17,320.802 1381.972,322.663 1381.362,322.661 C 1380.738,322.659 1380.48,322.6 1380.236,323.352 C 1379.494,323.182 1379.345,322.672 1379.876,322.127 C 1378.5,321.67 1378.559,322.835 1378.497,323.809 C 1378.397,325.383 1377.197,325.077 1376.067,325.368 C 1376.488,325.408 1377.317,325.874 1376.833,326.423 C 1376.221,327.117 1374.951,326.527 1374.187,326.664 C 1374.718,326.854 1375.248,327.046 1375.777,327.24 C 1374.524,328.526 1373.96,327.784 1374.812,329.709 C 1375.328,330.875 1373.219,330.842 1373.257,331.992 C 1374.25,331.881 1375.452,332.476 1375.197,333.649 C 1374.472,333.604 1374.545,332.863 1373.857,333.399 C 1373.118,333.974 1372.314,333.812 1371.513,333.562 C 1370.414,333.218 1367.683,333.45 1368.287,331.56 C 1367.706,331.779 1367.262,331.456 1367.067,330.913 C 1370.435,330.916 1366.334,328.124 1366.127,327.6 C 1365.973,329.177 1365.142,327.671 1365.062,326.968 C 1364.96,326.083 1365.149,325.165 1364.978,324.289 C 1365.062,324.95 1366.263,326.144 1366.638,325.009 C 1366.816,324.471 1364.744,322.178 1365.048,324.073 C 1364.44,322.885 1364.737,320.688 1365.118,319.464" id="path4548"/>
    <path d="M 1368.358,318.744 C 1369.372,318.51 1370.038,317.457 1369.797,316.44 C 1369.415,316.712 1368.997,316.91 1368.544,317.032 C 1368.091,317.168 1368.197,317.971 1367.637,318.168 C 1367.855,318.388 1368.096,318.58 1368.358,318.744" id="path4492"/>
    <path d="M 1374.988,328.176 C 1375.011,329.388 1376.61,330.152 1376.858,331.344 C 1376.858,330.295 1377.55,330.818 1377.733,331.446 C 1377.884,331.967 1380.058,332.032 1380.421,331.967 C 1382.302,331.631 1382.577,328.963 1380.597,328.536 C 1380.841,328.398 1381.081,328.255 1381.317,328.104 C 1381.15,327.801 1380.938,327.538 1380.677,327.311 C 1380.734,328.032 1380.23,328.715 1379.447,328.608 C 1380.865,327.029 1375.573,328.142 1374.988,328.176" id="path4496"/>
    <path d="M 1379.448,333.792 C 1379.543,333.021 1378.512,332.762 1378.008,332.496 C 1378.445,332.975 1378.925,333.407 1379.448,333.792" id="path4498"/>
    <path d="M 1380.028,332.353 C 1380.315,332.424 1380.601,332.497 1380.888,332.568 C 1380.924,331.731 1380.477,331.962 1380.028,332.353" id="path4500"/>
    <path d="M 1381.038,325.08 C 1380.614,325.009 1380.324,324.769 1380.168,324.36 C 1380.585,324.861 1380.562,325.34 1380.098,325.799 C 1380.674,325.942 1381.013,325.665 1381.038,325.08" id="path4502"/>
    <path d="M 1381.388,332.856 C 1380.837,333.358 1380.831,334.074 1381.468,334.511 C 1381.794,333.622 1382.851,332.583 1382.398,331.559 C 1382.195,332.097 1381.859,332.529 1381.388,332.856" id="path4504"/>
    <path d="M 1382.978,314.28 C 1383.378,314.166 1383.618,313.902 1383.698,313.488 C 1383.135,313.47 1382.655,313.662 1382.258,314.063 C 1382.498,314.136 1382.738,314.208 1382.978,314.28" id="path4506"/>
    <path d="M 1383.768,329.472 C 1384.433,329.707 1384.516,330.407 1385.026,330.667 C 1385.49,330.904 1386.291,330.705 1386.788,330.665 C 1388.21,330.549 1388.528,332.136 1387.298,331.705 C 1387.612,331.768 1388.637,332.077 1388.683,332.493 C 1388.504,332.711 1388.259,332.76 1387.948,332.64 C 1387.925,333.229 1388.857,335.326 1389.168,335.808 C 1389.369,335.263 1390.367,334.201 1390.355,333.696 C 1390.34,333.073 1389.25,332.904 1389.028,332.353 C 1389.578,332.576 1390.128,332.576 1390.678,332.353 C 1389.829,331.16 1390.802,330.707 1391.823,330.334 C 1393.062,329.882 1391.691,329.236 1391.242,328.789 C 1390.258,327.807 1392.149,327.352 1392.692,326.976 C 1393.942,326.111 1392.213,324.752 1393.348,323.929 C 1391.966,322.56 1390.407,323.636 1388.958,324.217 C 1390.023,324.644 1390.622,326.95 1389.028,327.026 C 1389.442,326.431 1389.843,325.743 1389.458,325.009 C 1389.161,325.208 1388.359,327.081 1387.954,326.822 C 1387.29,326.399 1387.487,324.921 1387.869,324.865 C 1387.283,324.386 1386.258,324.497 1385.569,324.577 C 1385.94,324.489 1386.181,324.593 1386.292,324.887 C 1385.981,326.174 1383.563,326.427 1382.619,326.377 C 1383.08,326.898 1385.803,329.344 1383.768,329.472" id="path4508"/>
    <path d="M 1383.118,334.008 C 1384.083,334.688 1387.557,336.154 1388.521,334.953 C 1389.605,333.601 1387.449,333.55 1386.761,333.643 C 1385.419,333.827 1384.768,332.075 1383.408,333.072 C 1383.572,333.314 1383.739,333.553 1383.908,333.792 C 1383.645,333.864 1383.381,333.936 1383.118,334.008" id="path4510"/>
    <path d="M 1390.758,332.424 C 1390.685,332.712 1390.611,333 1390.538,333.288 C 1390.827,333.197 1393.163,332.719 1392.925,332.28 C 1392.67,331.809 1391.152,332.393 1390.758,332.424" id="path4512"/>
    <path d="M 1392.837,327.24 C 1393.176,328.053 1393.442,328.004 1393.637,327.095 C 1393.371,327.144 1393.104,327.192 1392.837,327.24" id="path4514"/>
    <path d="M 1408.678,330.768 C 1407.714,330.004 1406.061,330.867 1407.24,331.751 C 1409.125,333.167 1410.299,330.948 1408.678,330.768" id="path4516"/>
    <path d="M 1366.848,318.744 C 1365.385,317.829 1366.635,316.218 1367.597,315.517 C 1368.948,314.533 1371.175,315.035 1372.743,314.879 C 1374.137,314.74 1374.86,313.262 1375.659,312.312 C 1376.549,311.254 1379.174,310.551 1380.528,310.105 C 1378.905,311.546 1380.821,312.52 1379.975,314.168 C 1379.039,315.99 1378.038,316.07 1376.193,315.505 C 1375.305,315.234 1374.516,315.427 1373.628,315.661 C 1372.642,315.92 1371.835,315.516 1370.893,315.703 C 1370.053,315.869 1368.881,316.121 1368.298,316.809 C 1367.923,317.252 1367.585,318.633 1366.848,318.601 C 1367.467,318.497 1367.818,318.839 1367.718,319.465 C 1367.408,319.248 1367.118,319.008 1366.848,318.744" id="path4550"/>
  </g>
  <path class="landxx lv eu" d="M 1416.2863,274.96555 C 1416.3833,275.20555 1416.4793,275.44455 1416.5763,275.68555 C 1416.4783,274.43555 1416.1833,273.11855 1417.0603,272.05755 C 1417.4363,271.60255 1418.1043,271.28055 1418.2623,270.66555 C 1418.4603,269.89355 1418.2453,269.22555 1418.6513,268.46355 C 1419.4353,266.99255 1420.4023,266.31655 1421.9893,265.92555 C 1422.7523,265.73755 1424.8073,264.51355 1425.4083,265.24455 C 1426.3613,266.40455 1427.3993,266.90955 1428.4783,267.90355 C 1429.7043,269.03255 1430.2113,270.52955 1431.9333,271.08055 C 1433.2913,271.51555 1434.6133,270.94755 1435.7243,270.16155 C 1437.3313,269.02355 1437.0993,267.65955 1436.4183,266.02555 C 1436.1483,265.37955 1435.8693,264.66755 1436.0163,263.94955 C 1436.6233,264.12355 1437.2453,263.67955 1437.7293,263.39655 C 1438.6923,262.83455 1440.5703,262.02755 1441.5853,262.67355 C 1442.2583,263.10155 1443.5603,262.97855 1444.4413,263.46755 C 1445.7593,264.19655 1446.8703,264.39755 1448.0323,265.45455 C 1448.9823,266.31855 1449.8753,266.83355 1451.0903,266.21555 C 1452.4443,265.52655 1453.9582,266.56075 1455.3863,266.61055 C 1456.3896,266.64554 1459.0357,268.77481 1458.8327,269.57365 C 1458.721,270.01318 1458.1219,271.73255 1457.7653,272.08455 C 1459.9073,271.82555 1459.0616,272.70891 1460.0016,273.06591 C 1461.1076,273.48591 1462.1311,277.33348 1462.3301,278.37648 C 1461.8914,278.57398 1461.1837,277.87314 1460.4138,278.26828 C 1459.8305,278.56761 1459.3488,279.31673 1458.9738,279.83473 C 1458.7618,280.12673 1458.8463,280.59855 1458.4863,280.79555 C 1457.3333,281.42655 1455.8948,280.07001 1454.6098,280.71701 C 1453.1248,281.46401 1452.4363,282.20855 1450.7093,281.25255 C 1449.2893,280.46655 1448.0023,279.44755 1446.6693,278.53155 C 1445.3853,277.64755 1443.7563,278.09455 1442.4913,277.29755 C 1441.3843,276.60055 1441.1613,275.10455 1439.6563,275.99055 C 1438.3473,276.76055 1437.8923,276.73055 1436.4523,276.87055 C 1435.1633,276.99555 1433.6233,276.15342 1432.2183,276.17355 C 1430.7524,276.19455 1430.5347,276.44155 1429.1114,275.94855 C 1428.2517,275.65074 1425.9852,275.88928 1425.0532,275.88928 C 1423.5962,275.88928 1422.3112,276.12228 1420.9042,276.46328 C 1420.0102,276.67928 1418.5872,276.88455 1418.1992,277.88455 C 1417.9442,278.54155 1417.2056,278.86987 1416.7536,278.31387 C 1415.9656,277.34987 1416.2053,276.11555 1416.2863,274.96555" id="lv"/>
  <path class="landxx ro eu" d="M 1416.7253,361.36555 C 1417.5063,360.82955 1418.1053,360.87455 1418.8933,361.07855 C 1419.6603,361.27655 1419.7073,360.28555 1420.3253,360.06955 C 1421.0043,359.83255 1421.7403,360.00455 1422.4213,359.79055 C 1423.4083,359.48055 1423.2173,358.28555 1423.6823,357.54555 C 1424.1243,356.84155 1424.7213,356.25055 1425.2183,355.58755 C 1425.8583,354.73455 1425.7773,353.97155 1426.3493,353.20055 C 1426.8533,352.52055 1427.4419,351.72177 1427.7172,350.90855 C 1428.0373,349.96273 1427.6623,349.52255 1428.3363,349.16455 C 1429.0003,348.81155 1429.1963,348.57255 1429.6563,347.98555 C 1430.3773,347.06755 1431.6913,347.72155 1432.5543,346.99855 C 1433.2653,346.40155 1433.7603,345.56355 1434.5443,345.04955 C 1435.6283,344.33755 1436.8543,345.23755 1437.9653,345.38255 C 1439.0233,345.52055 1440.0693,345.74255 1441.1303,345.85055 C 1442.3943,345.97955 1443.9953,345.42055 1445.1373,346.15655 C 1447.2983,347.55055 1447.4973,348.39155 1449.5073,346.47355 C 1450.7653,345.27255 1453.3833,345.88355 1454.9753,345.61655 C 1456.0233,345.44055 1455.8414,344.67864 1456.4414,344.01564 C 1457.0874,343.30164 1458.3304,343.25628 1459.2164,343.35728 C 1461.2014,343.58228 1462.3243,345.66855 1463.1563,347.24555 C 1464.0793,348.99355 1465.3803,350.51755 1466.7903,351.87055 C 1468.8493,353.84855 1469.8033,355.30655 1470.6323,358.02555 C 1470.9363,359.02055 1470.3423,360.02955 1470.1193,360.96955 C 1469.8633,362.05055 1470.3413,363.19855 1470.4223,364.28955 C 1470.5563,366.10255 1471.0013,366.84055 1472.4313,367.84955 C 1472.9803,368.23655 1474.291,369.18167 1475.1125,368.70545 C 1476.026,368.17586 1480.6929,365.59413 1481.3413,368.31355 C 1482.1646,371.76656 1480.3143,373.05555 1477.1263,373.02955 C 1477.5163,372.71555 1477.8523,370.82055 1477.0233,370.84855 C 1476.0343,370.88255 1476.5643,373.16755 1475.8353,373.74955 C 1476.3093,373.63855 1476.7893,373.56655 1477.2743,373.53355 C 1476.5483,373.84955 1475.2533,373.94655 1476.1253,374.90055 C 1476.3833,374.67855 1476.6233,374.43855 1476.8453,374.18055 C 1476.5363,375.31955 1475.3183,375.94455 1475.1153,377.13255 C 1474.8993,378.39955 1475.8559,380.47493 1475.0279,381.65793 C 1474.5469,381.25022 1473.4716,381.69932 1472.9927,381.45073 C 1472.6119,381.25305 1471.2258,380.23555 1470.4763,379.55155 C 1469.4532,379.2364 1468.4856,379.72217 1467.1156,379.0637 C 1466.0105,378.53256 1464.1067,377.84044 1461.9809,378.70073 C 1457.3067,379.16101 1456.2992,381.78836 1454.1591,382.27295 C 1452.8445,382.57063 1451.0259,381.93104 1449.1263,381.68756 C 1446.8438,381.39501 1445.629,382.04168 1444.3693,382.01234 C 1443.3167,381.98782 1442.0294,380.92434 1440.1854,380.85234 C 1439.2484,380.81575 1436.2312,381.61146 1435.5863,380.59055 C 1434.6423,379.09629 1438.3941,379.03192 1434.9483,377.78455 C 1434.0983,377.47686 1433.805,377.52483 1432.6767,375.93953 C 1431.6956,374.56104 1433.9935,374.89995 1434.4745,374.66495 C 1433.9325,374.00095 1432.7893,373.29555 1431.9163,373.60555 C 1431.3163,373.81855 1431.1603,375.40234 1430.4433,375.24734 C 1429.7243,375.09234 1428.6933,373.90534 1427.7753,373.64134 C 1426.7793,373.35434 1425.6133,373.09255 1424.8553,372.30955 C 1425.6903,372.06655 1425.8843,370.99655 1424.8553,370.87055 C 1425.8133,369.96955 1425.2963,369.34155 1424.1623,368.94455 C 1422.6343,368.40955 1421.4783,367.51255 1420.1603,366.62255 C 1421.3393,364.28455 1418.0713,362.86755 1416.7253,361.36555" id="ro"/>
  <path class="landxx zm" d="M 1437.6063,899.49855 C 1437.6063,891.31455 1437.6348,883.10491 1437.7398,874.92191 C 1437.7508,874.04491 1437.4343,872.23955 1438.6543,871.97255 C 1439.7223,871.73955 1441.0403,872.04155 1442.1273,872.07055 C 1445.7623,872.16555 1449.6243,872.79555 1453.1553,871.62855 C 1452.0103,870.36555 1452.5923,869.23455 1453.1313,867.86855 C 1453.7843,866.20855 1452.9073,864.91055 1452.9943,863.25455 C 1453.1563,860.18955 1454.4713,856.67755 1453.2253,853.62855 C 1454.3733,853.64755 1454.2605,854.89751 1455.4543,855.14355 C 1456.7752,855.41579 1456.1873,857.37681 1455.6753,858.16855 C 1457.2633,858.51755 1457.1173,858.47755 1458.6643,857.80455 C 1460.1683,857.15055 1461.8893,856.74055 1463.5243,856.64855 C 1462.9363,857.80455 1463.2253,859.90955 1464.1423,860.82855 C 1464.8623,861.54955 1466.0393,861.10555 1466.9313,861.51455 C 1468.2273,862.10955 1469.1689,862.67755 1470.6459,862.69755 C 1473.0279,862.72955 1475.2383,864.30155 1476.0813,861.26955 C 1476.9993,857.96555 1477.9753,861.65855 1479.1223,862.66155 C 1479.9073,863.34655 1479.8013,864.44655 1480.3703,865.27955 C 1481.2783,866.60955 1482.2762,865.3494 1483.0677,865.86973 C 1484.2651,866.65697 1484.7279,867.37455 1486.0261,867.35291 C 1487.5542,867.32745 1487.1963,869.82655 1488.2153,870.76955 C 1489.7103,872.15355 1489.9923,873.31555 1490.8413,875.08655 C 1491.9123,877.32155 1495.0343,872.99055 1496.2153,874.43955 C 1495.7003,874.84955 1495.6793,875.53355 1496.3363,875.79555 C 1497.4893,876.25655 1497.0973,873.92055 1497.1103,873.42055 C 1497.1883,870.52455 1497.2853,867.61655 1497.2853,864.71955 C 1496.7633,864.91655 1496.1883,864.96955 1495.6553,865.13555 C 1494.7403,865.41955 1495.0078,866.20802 1494.6129,866.83046 C 1493.9283,867.90962 1490.8973,865.43376 1490.2013,864.70755 C 1488.0995,862.51443 1485.6895,860.71798 1486.9278,857.07819 C 1487.5978,855.10869 1488.1663,853.48069 1488.7163,851.80655 C 1489.0958,850.65135 1488.3297,848.72555 1488.4367,847.42355 C 1488.6087,845.30155 1489.6577,840.82455 1486.8197,839.94855 C 1487.7377,837.53255 1492.8363,836.19355 1490.8853,832.96955 C 1495.0916,832.07068 1498.8314,831.80849 1503.6253,830.74055 C 1503.6235,831.55151 1502.6237,832.58447 1502.6543,832.78874 C 1502.7191,833.22177 1504.0801,832.61213 1503.3973,833.74271 C 1503.5397,834.01326 1504.0588,832.56592 1505.8406,834.34062 C 1507.6818,836.17451 1508.3601,834.96057 1508.1726,834.04828 C 1510.2247,834.06898 1510.7459,834.79412 1511.5953,836.29855 C 1511.9667,836.95627 1512.3849,836.68791 1512.9479,836.48591 C 1513.6769,836.22391 1513.933,837.43672 1514.5309,837.68791 C 1515.7399,838.19582 1517.9685,838.0274 1518.3613,839.45655 C 1518.4788,839.88391 1519.2883,839.72562 1519.7874,839.88155 C 1520.6174,840.14087 1521.3447,840.40892 1521.6487,840.7623 C 1521.9527,841.11569 1521.9404,841.55441 1521.7903,842.01255 C 1521.395,843.2192 1524.1673,842.76155 1524.6553,844.56055 C 1524.8183,845.16155 1524.3423,845.51255 1524.4343,846.07055 C 1524.5753,846.93455 1525.2313,847.24455 1525.8283,847.77455 C 1526.3463,848.23655 1526.5403,849.79855 1526.7613,850.44355 C 1527.1743,851.65055 1525.7613,852.79755 1524.5963,853.15355 C 1523.3273,853.54055 1524.0603,854.51855 1524.3723,855.48855 C 1524.8593,857.00555 1523.9663,857.81155 1523.9163,859.27355 C 1523.8463,861.37355 1522.8823,865.86855 1525.5153,866.66155 C 1524.4433,867.81055 1521.1653,868.35355 1521.1863,870.28255 C 1521.1993,871.54355 1521.5187,872.01121 1521.2623,873.43655 C 1520.889,875.51163 1519.5471,876.73083 1519.0545,878.18155 C 1518.7706,879.01767 1519.7124,879.68884 1520.3505,880.13936 C 1521.2725,880.79035 1521.7039,880.31128 1522.0703,880.94873 C 1514.9444,884.27009 1507.3206,886.2875 1500.0703,889.24255 C 1499.1408,889.89362 1501.6546,893.19648 1501.1871,894.85701 C 1500.4931,894.93801 1498.5723,895.22655 1497.8983,894.95555 C 1496.5783,894.42655 1495.0133,895.24355 1493.7723,895.66155 C 1492.0013,896.25755 1489.7173,897.30655 1489.3343,899.36355 C 1489.1703,900.23955 1489.3913,901.19455 1488.9913,902.02455 C 1488.4253,903.20355 1486.8623,903.34655 1485.7843,903.77155 C 1483.9113,904.50855 1481.3813,906.03955 1480.3453,907.80755 C 1479.2943,909.60055 1475.1663,917.44955 1472.0623,915.59355 C 1471.3853,915.18855 1470.6643,914.80955 1469.9103,914.57055 C 1468.8103,914.22055 1467.7163,915.67555 1466.8493,915.19555 C 1466.1673,914.81655 1465.8998,913.54641 1465.0363,913.75282 C 1464.8429,913.79905 1462.6379,914.35169 1461.5413,913.31782 C 1460.2988,912.14636 1459.7351,911.33585 1458.7517,911.07147 C 1457.6998,910.85097 1456.9913,911.95343 1455.0757,910.81791 C 1453.5567,910.58391 1452.5031,911.38882 1451.0093,911.50455 C 1450.2388,911.56424 1448.7634,912.15604 1448.2713,911.96755 C 1445.6753,910.97323 1443.9406,907.99564 1442.1076,905.95364 C 1441.1026,904.83464 1440.3298,903.92873 1439.2463,902.89255 C 1438.2293,901.92003 1437.9422,900.94783 1437.6063,899.49855" id="zm"/>
  <g class="landxx coastxx mm" id="mm" transform="translate(-29.9017, -45.0745)">
    <path d="M 1997.858,627.768 C 1997.808,628.139 1997.974,628.404 1998.358,628.56 C 1998.191,628.296 1998.025,628.031 1997.858,627.768" id="path4568"/>
    <path d="M 1999.438,631.584 C 1999.771,632.063 2000.061,632.567 2000.308,633.096 C 2000.098,632.277 1999.749,631.476 1999.148,630.865 C 1999.245,631.104 1999.342,631.344 1999.438,631.584" id="path4570"/>
    <path d="M 2003.118,632.016 C 2002.865,632.594 2003.61,633.246 2003.978,633.529 C 2003.955,632.913 2003.599,632.376 2003.118,632.016" id="path4572"/>
    <path d="M 2005.918,641.231 C 2005.524,641.252 2005.165,641.372 2004.838,641.591 C 2007.176,644.282 2007.203,641.181 2005.918,641.231" id="path4574"/>
    <path d="M 2007.868,637.056 C 2007.447,636.209 2006.403,636.179 2005.847,635.472 C 2004.802,636.537 2006.897,638.522 2007.868,637.056" id="path4576"/>
    <path d="M 2016.218,664.056 C 2015.734,664.587 2013.96,666.254 2014.989,667.008 C 2015.727,666.229 2016.801,665.246 2016.218,664.056" id="path4578"/>
    <path d="M 2017.878,666.864 C 2017.684,667.206 2017.53,667.101 2017.878,667.584 C 2017.878,667.344 2017.878,667.104 2017.878,666.864" id="path4580"/>
    <path d="M 2037.678,661.824 C 2037.289,661.996 2037.165,663.552 2037.663,663.552 C 2038.638,663.552 2038.607,661.945 2037.678,661.824" id="path4582"/>
    <path d="M 2044.729,698.256 C 2044.936,698.032 2045.056,697.769 2045.088,697.464 C 2044.233,697.323 2044.113,697.587 2044.729,698.256" id="path4584"/>
    <path d="M 2045.668,703.728 C 2045.691,703.185 2045.522,702.706 2045.158,702.289 C 2045.363,702.755 2045.533,703.235 2045.668,703.728" id="path4586"/>
    <path d="M 2045.448,699.336 C 2045.592,699.336 2045.734,699.336 2045.878,699.336 C 2045.696,699.044 2045.48,698.779 2045.229,698.544 C 2045.302,698.808 2045.375,699.072 2045.448,699.336" id="path4588"/>
    <path d="M 2046.458,692.352 C 2046.914,691.777 2046.66,691.184 2046.028,690.913 C 2046.167,691.394 2046.31,691.873 2046.458,692.352" id="path4590"/>
    <path d="M 2048.038,711.864 C 2047.944,711.021 2047.533,710.251 2046.818,709.776 C 2046.631,709.99 2046.489,710.23 2046.388,710.496 C 2047.374,710.019 2047.74,711.21 2048.038,711.864" id="path4592"/>
    <path d="M 2046.678,705.528 C 2047.889,705.445 2047.773,703.251 2047.248,702.576 C 2047.001,703.547 2046.811,704.536 2046.678,705.528" id="path4594"/>
    <path d="M 2047.178,697.608 C 2047.546,698.133 2049.175,694.92 2046.889,695.376 C 2047.029,696.113 2047.331,696.842 2047.178,697.608" id="path4596"/>
    <path d="M 2047.538,718.128 C 2047.585,718.368 2047.631,718.607 2047.678,718.848 C 2048.127,718.562 2048.344,718.154 2048.328,717.624 C 2048.048,717.766 2047.785,717.934 2047.538,718.128" id="path4598"/>
    <path d="M 2048.479,704.16 C 2048.647,704.576 2048.958,704.792 2049.409,704.807 C 2049.768,702.468 2047.163,702.023 2048.479,704.16" id="path4600"/>
    <path d="M 2048.398,700.776 C 2048.595,701.11 2048.789,701.446 2048.978,701.784 C 2049.261,701.616 2049.5,701.399 2049.698,701.136 C 2049.118,700.91 2048.637,700.55 2048.258,700.056 C 2048.305,700.296 2048.352,700.536 2048.398,700.776" id="path4602"/>
    <path d="M 2048.328,697.176 C 2048.473,696.79 2048.717,696.933 2048.398,696.384 C 2048.375,696.647 2048.352,696.912 2048.328,697.176" id="path4604"/>
    <path class="landxx mm" d="M 1992.528,621.288 C 1992.599,620.734 1992.3057,619.24286 1993.1247,619.15087 C 1994.1397,619.03687 1994.6681,620.04824 1995.988,620.352 C 1995.2189,618.49764 1994.9525,616.49481 1994.6037,614.53841 C 1994.5588,614.51377 1995.475,613.78205 1995.267,612.936 C 1996,613.198 1996.405,614.751 1997.313,614.466 C 1997.963,614.262 1997.665,612.986 1997.788,612.503 C 1999.298,613.384 1998.652,611.165 1998.375,610.504 C 1997.596,608.644 1997.753,607.199 1997.498,605.231 C 1998.984,606.334 1999.041,604.064 1999.014,603.128 C 1998.946,600.821 1998.512,598.836 1997.718,596.663 C 1998.064,596.638 1998.369,596.738 1998.635,596.962 C 1999.318,597.42 1999.275,597.085 2000.092,596.94 C 2000.906,596.794 2003.344,597.816 2003.979,598.32 C 2003.951,595.016 2008.5,589.127 2005.937,586.554 C 2004.577,585.189 2007.459,583.286 2007.769,581.922 C 2008.129,580.333 2008.609,579.414 2007.817,578.007 C 2007.166,576.849 2006.966,574.624 2008.229,573.767 C 2008.62,573.5 2009.808,573.853 2009.879,573.161 C 2009.944,572.515 2011.235,572.086 2011.733,571.612 C 2012.362,571.012 2012.705,570.402 2013.126,569.651 C 2013.84,568.374 2015.695,568.56 2016.939,568.368 C 2017.537,568.276 2018.267,567.857 2018.795,568.362 C 2019.508,569.043 2020.242,570.633 2021.548,570.095 C 2021.345,568.989 2020.259,568.454 2019.59,567.666 C 2018.366,566.225 2019.185,565.115 2020.281,564.018 C 2021.857,562.444 2021.0279,562.69341 2021.0279,560.99241 C 2021.0279,560.38941 2021.5135,560.54044 2021.4565,559.98244 C 2021.3865,559.28944 2022.025,558.325 2022.047,557.64 C 2023.393,559.136 2026.095,559.378 2027.185,561.289 C 2027.581,561.982 2027.457,562.837 2027.808,563.544 C 2028.284,564.502 2028.991,565.322 2029.467,566.28 C 2030.027,566.081 2030.331,565.631 2030.037,565.055 C 2033.023,566.443 2033.2761,569.47969 2034.0892,572.43394 C 2034.4902,573.89094 2034.6247,575.09457 2034.6684,576.667 C 2034.6858,577.29118 2035.377,579.355 2033.787,578.735 C 2034.1728,579.33537 2036.4935,581.71474 2034.1116,581.07687 C 2033.7436,580.97834 2033.0515,582.20944 2033.1135,582.50444 C 2033.3115,583.44444 2032.05,583.235 2031.558,583.2 C 2031.408,583.729 2031.294,587.679 2029.608,586.152 C 2029.679,587.484 2028.979,588.48 2029.757,589.752 C 2028.416,589.857 2028.5572,590.69081 2028.6132,591.74281 C 2028.6702,592.83281 2029.4926,592.93594 2029.8716,593.77994 C 2030.0366,594.14794 2030.5925,595.30313 2030.4255,595.71713 C 2030.1355,596.43713 2029.01,597.374 2029.928,598.099 C 2030.357,598.437 2032.816,596.49 2033.235,596.255 C 2035.049,595.235 2037.219,596.874 2038.967,595.585 C 2039.245,596.71 2037.779,597.153 2038.152,598.271 C 2038.363,598.905 2038.8791,599.221 2039.3101,599.689 C 2039.8561,600.28 2039.0954,600.93152 2039.469,601.267 C 2040.7297,602.39916 2040.372,603.98697 2041.4801,604.48353 C 2042.9919,605.161 2044.3904,603.76403 2045.4931,606.385 C 2043.8295,606.2609 2044.5127,608.83818 2044.6439,609.84991 C 2044.7869,610.95306 2042.9345,613.69693 2044.6171,613.50225 C 2045.5828,613.39052 2048.6613,612.96052 2049.7881,613.98566 C 2050.9897,615.07876 2049.221,617.081 2051.638,617.113 C 2051.309,617.615 2051.569,618.539 2051.913,618.966 C 2052.472,619.659 2052.999,618.151 2053.778,618.653 C 2055.385,619.689 2057.478,617.155 2058.697,616.25 C 2058.784,617.505 2059.743,618.207 2058.578,619.252 C 2058.041,619.734 2056.8275,620.10013 2056.5125,620.65513 C 2056.2655,621.09113 2054.702,624.757 2056.177,623.882 C 2056.063,624.951 2054.356,624.412 2053.728,624.89 C 2052.632,625.724 2053.004,627.395 2052.717,628.562 C 2052.0713,627.128 2050.105,628.762 2048.327,628.85 C 2048.796,629.456 2049.2857,630.07797 2048.4997,630.43097 C 2047.7217,630.78097 2046.6834,630.64091 2045.9604,630.69991 C 2045.0604,630.77491 2045.5878,632.39594 2045.4238,632.98594 C 2045.2398,633.64694 2044.6083,633.74777 2044.133,633.42719 C 2043.666,633.1122 2042.636,634.17626 2041.6869,634.33206 C 2041.3571,634.3862 2038.4345,633.74024 2038.4426,633.7316 C 2038.7395,633.4161 2036.7249,635.75829 2036.8596,636.87113 C 2036.9288,637.44298 2037.2444,637.87928 2037.3372,638.41328 C 2037.4634,639.13947 2036.6337,640.26075 2036.7943,640.96378 C 2037.0075,641.89678 2037.9887,642.91119 2037.3437,643.87819 C 2036.7613,644.75128 2034.83,644.872 2034.287,643.827 C 2034.591,644.892 2035.245,646.858 2036.737,646.419 C 2036.607,647.098 2037.0862,647.6471 2037.4476,648.15547 C 2038.0163,648.95547 2037.744,649.79935 2038.0007,650.595 C 2038.4803,652.08159 2040.5296,653.70868 2041.453,654.832 C 2042.5569,656.17488 2044.0355,656.60235 2044.877,658.659 C 2045.5613,660.33132 2045.743,662.196 2046.957,663.699 C 2047.356,663.258 2047.646,662.754 2047.827,662.186 C 2048.736,662.935 2048.5848,664.47136 2047.7647,665.138 C 2046.9419,665.80684 2045.9231,665.07356 2046.318,666.986 C 2046.5184,667.95645 2046.9734,669.82152 2046.8548,670.89859 C 2046.7609,671.75118 2045.258,671.619 2045.516,672.627 C 2044.819,671.998 2044.3992,672.51153 2044.2452,673.27853 C 2044.0252,674.37753 2044.5986,675.77988 2045.1422,676.70453 C 2046.6144,679.20894 2049.1245,681.00599 2050.7223,683.021 C 2052.4321,685.1772 2053.3656,687.40428 2053.0367,689.89328 C 2052.8853,691.03857 2052.6664,691.06384 2053.215,692.36 C 2053.7352,693.58892 2053.9,694.18183 2054.6578,695.26 C 2055.4113,696.33199 2055.8939,697.2145 2056.0291,698.35353 C 2056.1732,699.56704 2057.2779,701.63627 2057.5878,702.32494 C 2058.1306,703.53116 2056.5953,705.71049 2055.6597,707.157 C 2054.8434,708.41897 2053.9031,709.63924 2052.6404,711.84819 C 2052.055,712.87216 2052.7304,713.45119 2052.2884,714.33519 C 2051.6364,715.63419 2051.5654,716.98 2050.8004,718.275 C 2049.9394,716.921 2050.063,714.931 2049.705,713.412 C 2049.559,712.791 2049.226,712.256 2049.841,711.874 C 2050.593,711.409 2050.019,711.13 2050.38,710.552 C 2051.157,709.307 2051.327,707.816 2051.03,706.355 C 2050.889,705.663 2050.464,703.366 2051.715,703.37 C 2051.516,703.037 2051.419,702.677 2051.425,702.29 C 2051.076,702.773 2050.567,703.149 2049.986,703.298 C 2049.807,702.27 2051.223,701.71 2049.765,701.859 C 2051.432,701.065 2048.99,699.809 2050.345,699.05 C 2049.748,699.208 2049.245,698.899 2049.045,698.33 C 2049.44,698.271 2049.8,698.127 2050.125,697.897 C 2049.473,697.545 2049.022,696.764 2050.056,696.53 C 2049.167,696.235 2049.704,695.371 2049.57,694.571 C 2049.346,693.241 2048.842,691.843 2048.33,690.598 C 2047.791,689.289 2046.997,688.036 2046.362,686.767 C 2045.758,685.558 2045.701,684.098 2044.876,682.994 C 2045.258,684.023 2045.732,686.376 2044.946,687.313 C 2045.223,686.46 2044.49,685.697 2044.418,684.876 C 2044.349,684.102 2044.326,682.834 2044.147,682.129 C 2043.977,681.46 2041.229,676.732 2042.926,677.665 C 2042.856,677.379 2042.736,677.115 2042.567,676.873 C 2042.327,677.017 2042.087,677.161 2041.846,677.305 C 2040.607,674.21 2040.884,670.746 2039.75,667.662 C 2039.53,667.065 2038.996,666.741 2038.733,666.189 C 2038.499,665.697 2038.767,665.118 2038.739,664.602 C 2038.689,663.658 2038.019,661.814 2039.256,661.322 C 2038.46,661.449 2037.061,662 2036.467,661.223 C 2035.385,659.806 2034.528,658.248 2034.426,656.426 C 2033.716,656.9 2033.854,655.861 2033.249,655.305 C 2032.727,654.826 2032.054,654.411 2031.836,653.69 C 2031.69,653.873 2031.949,655.426 2031.943,655.78 C 2031.925,656.999 2032.665,658.246 2032.015,659.521 C 2031.546,660.445 2030.948,661.727 2029.755,661.825 C 2028.53,661.927 2027.125,660.27 2028.026,659.161 C 2026.603,659.518 2027.834,660.565 2028.214,661.44 C 2028.74,662.651 2027.463,664.036 2026.365,662.905 C 2027.539,663.731 2025.208,665.198 2024.344,664.344 C 2024.637,665.448 2023.761,669.195 2021.949,668.508 C 2021.014,668.154 2021.883,665.665 2021.684,664.92 C 2021.639,664.988 2020.701,668.416 2021.184,666.144 C 2020.888,666.759 2020.768,667.408 2020.824,668.088 C 2019.925,667.645 2019.677,665.685 2020.385,665.136 C 2019.321,664.441 2019.741,669.46 2018.309,668.21 C 2017.627,667.615 2018.221,664.363 2018.805,664.056 C 2017.755,664.469 2017.467,666.619 2016.935,667.584 C 2016.622,667.398 2016.359,667.158 2016.145,666.864 C 2017.026,666.495 2017.145,665.585 2017.365,664.776 C 2016.952,665.483 2016.784,666.541 2015.785,666.647 C 2016.869,665.186 2016.375,663.349 2016.575,661.679 C 2016.431,661.693 2016.289,661.693 2016.145,661.679 C 2016.705,662.881 2015.073,665.571 2013.954,666.021 C 2012.336,666.672 2013.647,659.824 2013.855,659.065 C 2014.064,658.303 2013.833,656.054 2014.255,655.654 C 2015.327,654.64 2014.157,651.788 2013.652,650.67 C 2013.003,649.232 2013.221,647.047 2012.155,645.868 C 2011.703,645.368 2011.441,644.717 2011.149,644.12 C 2010.711,643.225 2009.526,642.697 2010.744,642.455 C 2009.894,641.868 2008.874,641.67 2008.874,640.428 C 2008.874,639.503 2008.928,638.016 2008.435,637.199 C 2008.553,637.872 2008.851,639.422 2007.645,638.566 C 2008.201,639.248 2008.65,640.605 2008.225,641.447 C 2006.483,640.738 2005.418,638.936 2004.394,637.458 C 2004.113,637.054 2004.999,636.643 2005.298,637.192 C 2005.522,637.605 2006.112,637.749 2006.527,637.869 C 2007.463,638.14 2008.663,637.642 2007.799,636.5 C 2007.312,635.855 2005.128,635.023 2006.424,634.103 C 2005.84,634.09 2005.264,634.018 2004.695,633.887 C 2004.873,633.398 2004.943,632.894 2004.905,632.375 C 2004.244,632.698 2003.334,631.92 2002.967,631.459 C 2002.281,630.598 2001.696,631.204 2000.735,630.791 C 2001.069,631.56 2001.491,632.291 2001.745,633.095 C 2001.438,632.6 2001.148,632.096 2000.875,631.583 C 2000.745,632.108 2000.842,632.589 2001.165,633.023 C 2000.306,631.843 1999.841,631.103 2000.595,629.71 C 2000.261,629.773 1999.925,629.796 1999.585,629.782 C 1999.783,629.271 1999.996,628.767 2000.226,628.27 C 2000.086,628.301 1999.969,628.374 1999.875,628.486 C 1999.729,627.934 1999.729,627.382 1999.875,626.83 C 1999.375,628.039 1999.454,629.391 1998.865,630.574 C 1997.902,630.152 1998.345,629.626 1998.453,628.869 C 1998.579,627.987 1997.566,627.259 1997.136,626.614 C 1997.261,627.348 1997.692,628.13 1997.205,628.845 C 1996.844,627.512 1995.8455,626.43443 1994.702,625.809 C 1993.3981,625.09585 1993.1708,622.79932 1992.528,621.288" id="path4608"/>
  </g>
  <path class="landxx et" d="M 1522.1263,691.34155 C 1522.2693,690.17555 1523.3803,689.55855 1523.3843,688.37355 C 1523.3923,686.34555 1524.8193,686.15455 1526.5263,686.44455 C 1528.0423,686.70255 1529.3563,687.18455 1530.4813,685.50855 C 1531.2593,684.35155 1530.7483,682.34055 1530.5943,681.06955 C 1530.3403,678.94855 1530.0273,676.66155 1530.8223,674.61755 C 1531.1343,673.81755 1531.5483,673.06155 1531.8403,672.25255 C 1532.2703,671.06455 1531.7973,670.26855 1531.7773,669.09255 C 1531.7573,667.91655 1532.3923,666.73355 1533.1533,665.85055 C 1534.0293,664.83455 1535.3353,666.04755 1535.3073,667.14755 C 1535.8133,666.59155 1537.8263,663.50955 1536.8223,662.54055 C 1536.2543,661.99055 1537.1713,660.68955 1537.3413,660.13555 C 1537.6743,659.04455 1536.9403,657.66555 1537.9103,656.77655 C 1539.4653,655.35155 1540.0863,653.47555 1541.0493,651.55555 C 1541.4553,650.74355 1541.7053,649.89655 1542.6973,649.84955 C 1543.1093,649.82955 1545.0563,650.04155 1545.2343,649.55055 C 1545.7763,648.05555 1545.6503,646.19555 1545.8833,644.62955 C 1546.0173,643.73855 1546.3803,643.04455 1546.7513,642.22355 C 1547.4143,640.75255 1547.2503,639.37755 1547.5463,637.84555 C 1548.1093,634.95255 1549.3253,636.01255 1551.5163,635.98955 C 1552.3623,635.98055 1552.5373,634.25755 1553.7263,634.94555 C 1554.6533,635.48155 1555.1663,636.71755 1555.7573,637.55755 C 1556.6493,635.34155 1557.5303,633.10755 1558.1273,630.78955 C 1558.4693,631.31755 1558.8553,631.98655 1559.4273,632.30155 C 1560.0513,632.64455 1560.7563,632.36855 1561.1333,633.12255 C 1561.8753,634.60455 1562.7833,634.77955 1564.3323,634.49855 C 1564.9043,634.39455 1565.5063,634.23955 1565.9583,633.84855 C 1566.2283,633.61455 1566.4623,632.72855 1566.7453,632.64855 C 1567.2673,632.50055 1568.1153,634.28855 1568.2173,634.67755 C 1569.6933,632.82555 1572.1823,634.97655 1573.6963,634.57355 C 1575.7743,634.02055 1577.9083,636.23655 1579.6803,636.98655 C 1582.0523,637.99055 1582.7413,641.05955 1584.6733,642.60355 C 1585.8343,643.53155 1587.3613,644.27455 1588.0813,645.64155 C 1588.7383,646.89155 1589.0483,647.93955 1590.2173,648.83455 C 1592.1813,650.33955 1593.0303,651.49555 1591.7313,653.77755 C 1590.1963,656.47455 1588.0963,658.00955 1588.5843,661.30355 C 1588.8263,662.93955 1588.1843,664.93255 1590.5263,664.77255 C 1591.8963,664.67855 1593.2553,664.02255 1594.5813,663.68555 C 1595.6563,663.41255 1596.068,664.35811 1597.072,664.02511 C 1596.964,665.73611 1594.9913,667.17644 1595.9402,669.16511 C 1597.0803,671.5542 1598.1225,672.16395 1599.6056,674.29205 C 1600.099,674.99992 1600.9763,677.35755 1601.5363,678.00755 C 1602.0973,678.65655 1602.5807,678.45399 1603.2317,678.98399 C 1604.2327,679.80099 1605.7799,681.53968 1607.0509,681.99668 C 1612.1559,683.83568 1616.7483,685.75255 1621.8393,687.63055 C 1624.2513,688.52055 1626.2045,689.57924 1628.7025,690.16824 C 1631.0185,690.71424 1633.9873,690.33355 1636.3273,690.33355 L 1614.9443,715.71555 C 1613.6363,717.24455 1612.6008,717.38993 1610.7465,717.13986 C 1607.6995,716.72895 1603.4348,716.5551 1599.4624,720.2078 C 1598.4114,721.17427 1598.0493,722.37455 1596.5183,722.67855 C 1595.1083,722.95955 1593.6176,722.91005 1592.3116,723.62505 C 1590.9006,724.39805 1590.9956,725.7208 1589.1416,725.2958 C 1587.5876,724.9388 1585.407,726.22042 1584.035,724.70142 C 1583.453,724.05542 1582.9844,723.05424 1582.0124,722.94024 C 1581.0754,722.83024 1579.8193,723.51255 1579.0073,723.88455 C 1577.7323,724.47055 1575.6066,725.59813 1574.793,726.79905 C 1573.9122,728.09934 1573.2831,729.08175 1572.6773,730.07655 C 1571.7123,729.14855 1569.5303,729.00655 1568.2723,729.12155 C 1567.3033,729.20955 1566.62,728.43461 1565.752,728.24961 C 1564.849,728.05861 1563.84,728.47823 1562.9329,728.39361 C 1559.2526,728.05035 1562.3546,727.57742 1552.6634,721.74599 C 1551.0116,720.75203 1548.608,721.88493 1546.5786,721.16768 C 1545.6385,720.83545 1544.2163,718.98955 1543.7963,718.13955 C 1543.2103,716.95455 1544.2123,715.38655 1543.8303,714.08555 C 1543.4863,712.91355 1541.0793,712.20255 1540.2763,713.22855 C 1539.7873,712.75455 1539.1043,711.34955 1538.7013,710.72355 C 1537.7693,709.27855 1537.4383,707.33755 1536.8103,705.73655 C 1536.3383,704.53455 1536.1033,703.17855 1535.4543,702.05155 C 1535.1553,701.53155 1534.7993,701.34655 1534.2933,701.09255 C 1533.7323,700.81255 1533.716,700.25561 1533.302,699.88561 C 1533.044,699.72061 1532.9433,699.57555 1532.6393,699.53855 C 1531.8463,699.31755 1531.7673,698.46555 1531.3463,697.89255 C 1530.4053,696.61455 1529.7683,695.05455 1528.5133,694.02955 C 1527.5943,693.27955 1526.4963,692.76655 1525.3183,692.62355 C 1524.2233,692.49255 1522.7093,692.48355 1522.1263,691.34155" id="et"/>
  <path class="landxx gt" d="M 572.81029,633.66955 C 573.62429,632.70055 573.48629,631.67755 573.77829,630.51555 C 573.92029,629.94955 574.71429,629.55755 574.54429,628.93255 C 574.22829,627.77155 574.24,627.57825 574.919,626.44725 C 575.939,624.74925 576.80306,622.56531 577.50736,621.12431 C 578.22511,619.65581 580.97329,620.49355 582.42829,620.49355 C 583.26329,620.49355 587.64729,621.01655 588.07829,620.31355 C 588.22129,620.08155 588.59729,617.37655 588.48029,617.27555 C 587.97729,616.83755 587.29329,617.03555 587.13729,616.22555 C 586.95729,615.29955 586.74829,614.79755 585.88529,614.27455 C 584.23129,613.27355 583.20229,611.63255 581.88429,610.26955 C 582.88429,610.26955 584.66829,610.73855 585.27029,609.62455 C 585.89129,608.47755 585.13329,606.72755 585.98829,605.72855 C 586.90429,604.65855 591.00829,605.37355 592.32129,605.37355 C 594.84229,605.37355 597.36329,605.37355 599.88429,605.37355 C 599.24629,608.09655 599.18229,610.91055 598.66029,613.65255 C 598.12629,616.45955 597.51229,619.25255 597.07629,622.07755 C 598.41329,621.93955 601.25629,621.87755 601.46829,623.73255 C 601.68629,623.25155 601.90929,622.67655 602.47629,622.50855 C 602.29829,622.25755 602.10629,622.01755 601.90029,621.78855 C 602.90229,622.15155 603.58329,623.00155 604.49229,623.51655 C 602.06229,624.53255 600.55829,626.97855 598.31929,628.29955 C 597.27729,628.91455 595.92229,629.64855 596.22129,631.08655 C 596.43229,632.10455 596.66029,632.79955 595.82429,633.56955 C 595.03329,634.29755 592.79129,634.74255 593.00829,635.89955 C 593.13729,636.58355 592.62929,636.44155 592.14329,636.86755 C 592.04729,636.95155 591.21129,638.06655 591.33329,638.01155 C 590.34829,638.45655 587.86329,638.61155 588.36429,640.65255 C 587.19029,640.14155 586.13429,639.28655 584.84429,639.06455 C 583.52929,638.83855 582.13029,639.26055 580.78629,639.10055 C 577.46029,638.70455 575.03629,635.89555 572.81029,633.66955" id="gt"/>
  <path class="landxx sr" d="M 829.77629,728.70955 C 828.95329,728.22055 827.55705,725.1951 827.56805,724.2151 C 827.58305,722.7801 827.74114,719.2075 827.74313,717.90538 C 827.74689,715.45966 830.91011,716.13546 832.51665,716.13146 C 833.27466,716.12958 834.67983,714.0646 833.09629,713.60737 C 834.72029,711.66037 834.07729,707.37955 837.60629,707.86755 C 838.71729,708.02155 844.66729,708.53955 844.10629,710.49255 C 844.88329,708.95855 843.67429,707.89055 846.14729,707.90055 C 847.06229,707.90455 849.94829,707.78055 850.00629,709.19755 C 850.48329,708.59455 851.40929,708.93255 852.02629,708.90855 C 851.87429,708.80355 849.88329,708.06855 850.17829,707.86655 C 851.49429,706.96555 854.93429,708.35055 856.35529,708.58955 C 858.84029,709.00655 858.72929,710.74255 857.64629,712.58155 C 856.46329,714.58755 854.83529,715.72855 854.93029,718.27655 C 855.01329,720.50455 854.77229,723.15755 855.85329,725.18755 C 856.71129,726.79955 858.83429,728.15855 857.69029,730.23455 C 857.29829,730.94555 856.88788,731.74653 856.68788,732.56153 C 856.49288,733.35753 856.63574,734.46446 856.40625,735.29463 C 855.84826,737.31318 854.28977,739.56055 853.62467,739.59365 C 853.05756,739.62187 851.07529,738.17755 850.79729,737.82855 C 849.83029,736.61155 847.75329,738.08755 846.36229,738.34155 C 845.24029,738.54655 845.04629,738.79455 844.04129,738.24555 C 842.98429,737.66855 843.06511,738.13461 842.59529,738.86155 C 841.55798,740.46655 842.96689,740.38355 843.27829,742.15155 C 843.67139,744.38342 839.79066,743.14255 838.91966,742.59055 C 838.28466,742.18855 837.17823,741.62034 836.64611,741.1141 C 835.9411,740.44337 835.9037,739.91177 835.91284,739.3141 C 835.93657,737.76198 833.43929,735.73055 833.43729,734.20155 C 833.43729,733.62455 833.18847,730.63373 832.44547,730.53673 C 831.91447,730.46773 831.02417,731.02728 830.57929,730.46955 C 830.13168,729.9084 830.12684,729.09944 829.77629,728.70955" id="sr"/>
  <path class="landxx eh" d="M 1144.1363,579.88555 C 1142.9523,577.79955 1144.7083,574.26055 1145.0683,572.09755 C 1145.3153,570.61855 1145.6963,569.23155 1146.4883,567.94155 C 1147.4473,566.37755 1149.2023,566.37455 1149.7913,564.41955 C 1150.0933,563.41755 1150.0743,561.87455 1151.0703,561.27655 C 1151.6353,560.93855 1151.3013,559.80355 1151.7423,559.15655 C 1152.4473,558.12155 1155.2863,554.58755 1154.4963,553.24455 C 1153.9653,553.58655 1153.5183,554.06355 1153.2063,554.61255 C 1154.4473,551.88255 1157.5893,549.87355 1159.6533,547.76655 C 1162.4023,544.95955 1161.8713,542.05855 1163.2093,538.61555 C 1164.3043,535.80055 1165.1533,532.69355 1167.8903,531.13455 C 1170.9063,529.41655 1172.5464,528.01573 1173.7953,524.69355 C 1174.5891,522.58178 1176.0918,519.58438 1175.5707,520.19919 L 1209.0699,519.92848 L 1208.5928,534.52396 L 1183.7363,534.81055 L 1182.5923,556.64355 C 1181.1133,557.71055 1179.0113,557.82755 1177.4663,558.78855 C 1175.1413,560.23355 1174.2023,561.77955 1174.3863,564.52555 C 1174.4933,566.12155 1175.7183,574.43655 1174.0233,574.91655 L 1146.3743,574.91655 C 1144.2523,574.91755 1143.9133,578.38355 1144.1363,579.88555" id="eh"/>
  <path class="landxx cz eu" d="M 1360.7763,325.94155 C 1361.5213,325.94355 1362.1053,326.55955 1362.2863,327.23755 C 1362.546,324.64589 1366.2836,325.31261 1367.007,324.76096 C 1369.5931,322.78903 1372.9686,321.79997 1375.7563,320.83055 C 1375.5223,320.60755 1375.2813,320.39255 1375.0363,320.18255 C 1375.6643,320.03655 1376.3963,319.80855 1376.8863,320.36455 C 1377.3213,320.85855 1377.7068,321.789 1378.4953,321.51455 C 1379.0331,321.32735 1379.7166,321.43887 1379.8233,320.80355 C 1379.9507,320.0448 1380.2661,320.19649 1380.7661,320.32449 C 1381.6051,320.53849 1381.8413,320.66555 1382.0813,321.45655 C 1382.7823,323.35845 1386.6038,321.27911 1386.6963,323.56655 C 1387.0713,323.16855 1388.0001,323.0302 1388.5433,323.13755 C 1390.3401,323.49264 1388.5512,324.55808 1387.9263,325.07955 C 1389.1753,325.51355 1389.6023,326.17955 1390.3213,327.17655 C 1391.2003,328.39655 1392.6153,326.96855 1393.3963,326.44755 C 1392.9233,326.05255 1392.6603,325.54955 1392.6063,324.93555 C 1394.4653,325.42255 1396.2003,326.83855 1398.2163,326.23155 C 1398.2376,327.16161 1398.1031,328.71 1399.1785,328.62728 C 1400.3196,328.53951 1400.7293,328.32855 1401.9343,328.89555 C 1403.2253,329.50255 1403.5246,329.06455 1404.1886,330.46755 C 1403.0611,329.69959 1405.5341,330.70987 1406.5541,332.9706 C 1405.2471,333.4566 1403.276,333.6926 1402.115,334.5616 C 1401.28,335.1866 1401.2809,336.56657 1400.4019,337.0016 C 1399.5064,337.44476 1398.3984,338.58671 1396.4373,338.55201 C 1395.1502,338.52923 1394.2182,338.5579 1393.4663,340.19955 C 1392.8083,337.79855 1389.4783,339.51655 1387.9783,338.99955 C 1386.7423,338.57355 1385.5313,338.08155 1384.2833,337.67755 C 1383.2713,337.35055 1381.3383,336.81355 1380.2863,336.88755 C 1380.3643,337.57255 1380.0803,338.22055 1380.1463,338.90355 C 1379.1273,338.76255 1378.3533,339.57155 1378.3463,340.55955 C 1376.8323,339.95655 1376.8023,340.60555 1375.3183,340.46955 C 1374.3313,340.37855 1373.5093,339.31755 1372.7323,338.79555 C 1371.7353,338.12555 1370.6732,337.56365 1369.6363,336.96055 C 1368.6683,336.39755 1367.5679,335.00072 1366.5355,334.59614 C 1364.8143,333.92166 1364.4133,331.53655 1362.8663,330.69555 C 1365.0434,328.77649 1360.9755,328.65104 1360.7763,325.94155" id="cz"/>
  <path class="landxx td" d="M 1373.1663,634.82155 C 1373.4333,634.14755 1374.1013,634.29955 1374.4463,633.75355 C 1374.7513,633.27055 1374.8403,632.59155 1375.3263,632.23055 C 1375.9363,631.77655 1376.0423,630.05455 1376.3033,629.37055 C 1377.7113,625.67255 1380.8723,622.29255 1383.3903,619.31655 C 1385.6963,616.59055 1388.3183,614.28455 1388.4943,610.60055 C 1388.7143,605.98455 1388.8053,601.35855 1389.1313,596.74755 C 1389.3823,593.18555 1389.2023,588.49855 1390.9713,585.34155 C 1391.9893,583.52355 1391.3193,582.67455 1390.0873,581.25455 C 1389.7373,580.85155 1389.2963,580.53455 1388.9413,580.13555 C 1388.3253,579.44555 1389.1313,578.82655 1388.8543,578.11555 C 1388.2143,576.47155 1386.8731,575.19855 1386.1131,573.63455 C 1385.4701,572.31155 1385.8833,570.70355 1385.7393,569.28455 C 1385.5013,566.93855 1384.8013,564.67455 1384.4323,562.34855 C 1384.3893,562.07855 1383.9412,561.12124 1384.3142,560.93524 L 1392.5459,557.02761 L 1452.1161,590.74993 L 1452.7273,623.73455 C 1450.7583,623.12355 1450.5799,623.3685 1448.5179,623.69324 C 1447.1534,623.90813 1445.3873,623.6926 1444.9501,624.9702 C 1444.6312,625.90178 1445.5302,627.67402 1444.8486,628.57549 C 1444.2072,629.42364 1443.2334,632.32965 1441.8823,632.79955 C 1440.3475,633.33335 1440.9716,636.00169 1442.0683,637.05555 C 1439.5773,639.05155 1437.9544,640.30311 1439.8887,643.08593 C 1441.1968,644.96798 1438.1474,645.76374 1437.3084,647.34674 C 1436.7324,648.43374 1436.507,648.81005 1437.382,649.68505 C 1438.127,650.43005 1439.4045,649.63126 1440.2066,649.63842 C 1441.2143,649.64742 1442.057,650.11393 1441.4096,650.90142 C 1440.468,652.04661 1442.3794,653.40871 1441.6393,655.41755 C 1441.8983,655.28655 1442.1603,655.16655 1442.4293,655.05755 C 1443.0073,655.98655 1442.3513,657.03755 1442.3593,658.00955 C 1442.3673,659.12055 1442.5011,659.13536 1443.2027,659.90592 C 1443.8391,660.60479 1445.3686,660.22417 1445.4723,662.12455 C 1445.5185,662.97079 1444.6987,664.50701 1444.9445,665.47112 C 1444.5118,664.73878 1442.437,664.03682 1441.5233,664.51155 C 1440.9808,664.7934 1440.2273,665.03155 1439.8703,665.56155 C 1439.4303,666.21355 1439.338,665.79949 1438.844,666.34649 C 1438.053,667.22349 1436.1523,667.00655 1435.8803,668.37855 C 1435.7693,668.94355 1436.2171,669.42083 1436.0353,670.00455 C 1435.4272,671.95708 1433.6661,672.76315 1432.2487,674.05155 C 1431.119,675.07849 1430.5433,676.52712 1429.5293,677.65255 C 1427.7325,679.64674 1426.4854,680.09575 1423.9479,681.03955 C 1422.5049,681.57622 1418.9083,681.80955 1417.3893,681.66455 C 1416.8713,681.61555 1413.6873,682.36355 1414.8233,683.21955 C 1415.3173,683.59255 1416.6323,683.84255 1416.3363,684.72655 C 1415.9963,685.74355 1414.8103,687.17855 1414.0223,687.90755 C 1413.4033,688.47855 1412.8603,689.90655 1412.0783,690.09955 C 1410.5993,690.46655 1408.8053,690.18355 1407.2773,690.31955 C 1405.2633,690.49855 1403.7643,690.99755 1402.0393,692.06455 C 1401.0913,692.65055 1399.3133,693.16455 1398.8693,694.29655 C 1398.1873,693.31655 1397.2413,692.62555 1396.9293,691.41655 C 1396.2983,691.64355 1396.0003,692.47255 1395.6993,693.00055 C 1395.4853,693.37455 1394.2543,693.47555 1393.8373,693.65855 C 1391.7693,694.56755 1391.0603,694.40655 1388.8593,694.43955 C 1390.0163,692.98355 1388.7193,692.45055 1388.3023,691.08555 C 1387.7343,689.23055 1387.2443,687.15055 1386.0203,685.59755 C 1384.9773,684.27355 1383.5393,683.34955 1382.2443,682.29955 C 1380.9583,681.25455 1380.2433,679.98255 1379.2043,678.74255 C 1378.4973,677.89855 1376.7023,676.30255 1377.5543,675.07655 C 1378.4173,673.83455 1378.8343,672.95255 1380.4473,673.02755 C 1382.2493,673.11155 1384.0163,673.84355 1385.8043,673.23555 C 1386.5903,672.96855 1387.2863,673.36255 1387.9893,673.64455 C 1388.6413,673.90455 1389.5533,673.30455 1390.1593,673.12755 C 1386.5823,671.14055 1384.9123,666.34855 1385.1013,662.43655 C 1385.1443,661.54255 1385.5953,660.82755 1385.6863,659.97755 C 1385.7803,659.09555 1385.5543,658.20755 1385.5313,657.32755 C 1385.5143,656.66255 1385.6503,655.57155 1385.2603,654.98355 C 1384.9533,654.52155 1384.1063,654.86455 1384.1593,654.08455 C 1384.2293,653.05555 1383.8153,651.67255 1383.6053,650.66355 C 1383.4533,649.93555 1382.4198,649.8237 1381.8993,649.50155 C 1380.2383,648.47355 1382.35,646.91943 1379.0673,646.42655 C 1378.2822,646.30867 1378.5126,646.92391 1377.6671,646.27548 C 1376.4938,644.89209 1374.9346,642.19262 1374.529,641.44961 C 1373.4386,639.45152 1373.5983,636.98755 1373.1663,634.82155" id="td"/>
  <path class="landxx al" d="M 1412.2563,408.09355 C 1412.6463,407.42755 1412.9813,406.64355 1412.6963,405.86255 C 1413.7513,406.03155 1413.2113,403.61155 1413.4293,402.82755 C 1413.5973,402.22255 1412.6573,402.19255 1412.9963,401.42155 C 1413.2513,400.84355 1413.4613,400.23555 1413.5883,399.61555 C 1413.8033,398.55955 1413.6253,397.38655 1412.2563,397.72555 C 1411.7413,396.12755 1411.7536,393.98367 1412.7713,392.63596 C 1413.598,391.5412 1413.7344,389.79667 1414.8921,391.83776 C 1415.6257,393.13126 1416.7482,390.31175 1417.8371,392.39655 C 1418.0637,392.83047 1418.03,393.14542 1418.3875,393.49096 C 1419.0775,394.15796 1419.5402,393.46604 1420.5833,394.93436 C 1420.7586,395.18113 1421.0439,396.72914 1420.9729,397.93592 C 1419.8339,398.66046 1421.2368,399.81108 1420.0615,400.37154 C 1420.1155,400.80654 1420.4833,401.14055 1420.6743,401.50855 C 1420.7083,401.82555 1420.8077,401.99256 1420.6197,402.26056 C 1420.4787,402.95556 1421.0502,403.87607 1421.5183,404.24855 C 1422.069,404.68676 1421.9093,405.77665 1422.9223,405.65855 C 1423.9105,405.54334 1423.8522,406.14049 1424.1613,406.83555 C 1425.3634,409.53836 1423.4381,408.90691 1422.7505,410.61715 C 1422.5192,411.19229 1421.9299,412.52215 1421.2819,412.86556 C 1419.6075,413.75288 1419.6679,413.68368 1420.3263,415.29255 C 1418.8298,415.11407 1420.7835,416.19153 1418.8083,416.51776 C 1417.661,416.70725 1417.3383,416.07455 1417.1933,415.15655 C 1416.8163,412.76655 1413.6413,411.91055 1412.2563,410.03655 C 1412.7413,410.11155 1413.0783,410.37555 1413.2663,410.82855 C 1413.9853,409.99555 1412.5943,409.46855 1413.3363,409.17255 C 1412.8963,408.89355 1412.5363,408.53255 1412.2563,408.09355" id="al"/>
  <g class="landxx coastxx fi eu" id="fi" transform="translate(-29.9017, -45.0745)">
    <path d="M 1435.388,289.656 C 1434.819,290.29 1434.823,290.421 1435.678,290.592 C 1435.668,290.253 1435.571,289.941 1435.388,289.656" id="path4630"/>
    <path d="M 1437.118,291.168 C 1438.086,291.327 1438.145,290.395 1437.698,289.8 C 1438.167,289.67 1438.722,290.645 1439.23,290.4 C 1440.259,289.903 1437.14,288.842 1436.618,288.937 C 1437.142,289.365 1437.16,289.972 1436.688,290.449 C 1436.639,289.858 1436.288,289.491 1435.678,289.728 C 1435.919,290.388 1436.15,291.446 1437.118,291.168" id="path4632"/>
    <path d="M 1437.837,291.528 C 1440.511,292.551 1438.753,290.717 1437.837,291.528" id="path4634"/>
    <path d="M 1439.858,292.032 C 1440.124,292.159 1440.388,292.159 1440.648,292.032 C 1440.384,292.032 1440.121,292.032 1439.858,292.032" id="path4636"/>
    <path d="M 1443.528,267.264 C 1445.008,267.091 1443.937,265.671 1442.808,266.617 C 1443.023,266.859 1443.264,267.075 1443.528,267.264" id="path4638"/>
    <path d="M 1445.908,287.784 C 1446.171,287.76 1446.435,287.736 1446.698,287.712 C 1446.153,287.221 1445.89,287.245 1445.908,287.784" id="path4642"/>
    <path d="M 1448.278,265.608 C 1448.015,265.704 1447.751,265.799 1447.488,265.895 C 1447.799,265.922 1448.063,265.826 1448.278,265.608" id="path4644"/>
    <path d="M 1449.218,265.896 C 1448.978,265.92 1448.738,265.944 1448.498,265.968 C 1449.214,266.634 1449.454,266.61 1449.218,265.896" id="path4646"/>
    <path d="M 1449.797,288.864 C 1449.554,288.618 1449.288,288.402 1448.997,288.216 C 1448.405,289.192 1449.586,289.503 1450.297,289.296 C 1450.147,289.25 1449.782,288.853 1449.797,288.864" id="path4648"/>
    <path d="M 1449.578,290.664 C 1449.376,290.551 1449.166,290.408 1448.928,290.376 C 1449.145,290.472 1449.361,290.568 1449.578,290.664" id="path4650"/>
    <path d="M 1449.868,290.592 C 1450.349,290.802 1450.589,290.706 1450.588,290.304 C 1450.348,290.399 1450.107,290.496 1449.868,290.592" id="path4652"/>
    <path d="M 1452.818,262.08 C 1452.555,262.103 1452.291,262.128 1452.028,262.151 C 1452.268,262.339 1452.531,262.483 1452.818,262.584 C 1452.894,262.269 1453.155,261.948 1452.818,262.08" id="path4660"/>
    <path d="M 1452.167,289.512 C 1452.407,289.536 1452.647,289.56 1452.887,289.584 C 1452.556,288.987 1452.548,289.241 1452.167,289.512" id="path4662"/>
    <path d="M 1452.598,292.032 C 1453.169,291.973 1453.752,292.03 1454.326,291.992 C 1455.45,291.919 1455.052,290.281 1455.909,289.799 C 1455.079,290.317 1453.781,290.589 1452.819,290.375 C 1454.478,291.006 1452.229,290.753 1452.598,292.032" id="path4664"/>
    <path d="M 1455.118,292.032 C 1455.506,292.701 1456.796,293.244 1456.847,292.032 C 1456.271,292.189 1455.694,292.189 1455.118,292.032" id="path4666"/>
    <path d="M 1461.598,253.224 C 1462.991,253.937 1462.907,252.633 1463.909,253.008 C 1463.467,252.347 1461.845,252.382 1461.598,253.224" id="path4670"/>
    <path class="landxx fi eu" d="M 1435.897,223.056 C 1436.727,222.144 1437.879,223.164 1438.847,222.624 C 1438.602,222.413 1438.362,222.197 1438.127,221.976 C 1440.787,220.382 1443.305,222.333 1445.269,224.267 C 1446.649,225.625 1449.337,226.517 1451.31,225.938 C 1452.378,225.625 1453.069,225.472 1453.935,224.831 C 1454.732,224.241 1456.931,225.157 1457.782,225.4 C 1458.995,225.747 1459.0824,226.42472 1460.3164,226.20172 C 1461.3834,226.00772 1461.6067,224.18069 1463.0577,224.08069 C 1463.7897,224.03069 1464.46,223.948 1464.295,223.049 C 1464.011,221.505 1464.615,220.072 1464.988,218.593 C 1465.8718,217.42382 1467.5658,216.1817 1469.2977,216.58153 C 1470.8957,216.95044 1471.8923,216.33987 1473.3413,215.72687 C 1474.9303,215.05587 1476.091,216.443 1477.493,216.961 C 1478.958,217.501 1480.4185,217.81281 1481.912,218.178 C 1486.4484,219.28724 1480.4785,221.5517 1483.0018,223.14654 C 1481.8462,223.77474 1480.3875,223.872 1479.386,224.21 C 1479.935,224.282 1482.2844,224.39714 1481.1974,225.48014 C 1480.1214,226.55314 1479.7419,227.34687 1481.0959,228.46487 C 1482.3799,229.52387 1484.043,229.584 1485.46,230.347 C 1486.707,231.019 1487.9763,231.8412 1489.1571,232.63868 C 1491.9865,234.54971 1484.2759,237.11079 1485.7749,239.20668 C 1487.9544,242.25416 1491.6267,244.084 1492.9257,247.897 C 1492.1637,247.937 1491.3939,248.18007 1490.6489,248.31407 C 1491.6229,248.58907 1491.1699,250.10664 1490.658,250.808 C 1490.2718,251.33721 1491.8227,251.66245 1492.0507,252.18614 C 1490.6857,252.47914 1490.6221,253.54338 1491.5322,254.36227 C 1492.4212,255.16217 1493.881,254.17282 1494.507,255.602 C 1491.743,256.849 1495.6719,258.20427 1496.7769,258.75727 C 1499.2219,259.98227 1495.5969,262.26814 1494.4589,262.84314 C 1497.8496,264.40225 1501.0492,266.21625 1504.0828,268.35027 C 1505.0778,269.22127 1504.5789,269.64385 1504.147,270.792 C 1503.4182,272.72936 1502.1742,273.55499 1500.649,274.748 C 1497.1262,278.66118 1493.9803,281.79463 1488.7057,284.72893 C 1487.7787,285.38793 1486.826,286.072 1485.967,286.818 C 1485.547,287.116 1485.1532,287.46707 1484.8832,287.90407 C 1484.3882,288.62107 1483.353,288.068 1482.681,288.13 C 1481.715,288.22 1481.655,287.36 1480.584,287.866 C 1479.766,288.253 1478.998,288.571 1478.072,288.574 C 1477.683,288.599 1477.328,288.505 1477.006,288.291 C 1476.919,288.626 1476.712,288.846 1476.385,288.95 C 1475.592,289.198 1474.571,288.783 1473.906,288.363 C 1474.53,289.069 1474.181,289.313 1473.475,289.371 C 1473.567,289.613 1473.663,289.852 1473.765,290.091 C 1473.019,290.282 1472.169,289.974 1472.185,289.083 C 1471.469,290.283 1469.41,290.288 1468.201,290.536 C 1467.521,290.675 1466.922,290.942 1466.268,291.156 C 1465.78,291.315 1465.727,291.92 1465.226,292.056 C 1464.305,292.305 1464.04,291.486 1463.07,291.736 C 1461.598,292.115 1459.966,292.9 1458.425,292.612 C 1458.727,292.438 1459.017,292.246 1459.295,292.036 C 1458.101,292.485 1457.238,293.432 1455.905,293.548 C 1456.368,293.282 1456.974,293.008 1457.248,292.511 C 1457.855,291.408 1456.022,292.414 1457.715,291.676 C 1457.04,291.775 1456.509,291.47 1455.986,291.094 C 1455.091,290.45 1455.971,289.835 1456.634,289.228 C 1456.021,289.417 1451.839,290.792 1453.244,289.012 C 1451.909,288.732 1450.537,288.745 1449.383,287.937 C 1448.897,287.596 1447.496,286.535 1447.774,287.932 C 1447.36,287.612 1446.235,287.236 1446.393,286.826 C 1446.678,286.085 1445.784,284.205 1445.393,283.541 C 1447.838,283.573 1444.916,278.661 1447.343,279.941 C 1446.29,279.228 1444.988,277.356 1444.637,276.139 C 1444.352,275.152 1445.084,274.504 1444.427,273.431 C 1443.947,272.645 1443.449,271.544 1442.943,271.445 C 1443.497,270.767 1444.833,267.716 1446.043,268.277 C 1445.476,267.839 1444.988,267.056 1445.393,266.333 C 1446.469,267.557 1449.053,266.797 1450.223,266.117 C 1449.904,265.851 1448.682,264.15 1450.223,265.253 C 1449.985,265.011 1449.745,264.772 1449.503,264.533 C 1450.391,265.476 1450.944,262.87 1452.673,263.669 C 1452.33,262.171 1454.702,261.7 1455.694,261.437 C 1455.624,261.172 1455.55,260.908 1455.473,260.645 C 1456.441,260.607 1456.9,260.078 1457.459,259.374 C 1458.363,258.235 1459.75,257.786 1460.733,256.757 C 1461.373,256.088 1461.653,255.197 1462.411,254.548 C 1463.569,253.553 1464.949,254.268 1466.203,254.526 C 1466.023,254.295 1465.857,254.054 1465.703,253.805 C 1466.094,253.672 1466.43,253.455 1466.713,253.157 C 1466.32,252.686 1465.165,252.335 1465.299,251.86 C 1465.519,251.087 1465.928,250.14 1465.222,249.531 C 1464.134,248.591 1460.965,248.785 1460.734,247.181 C 1459.668,247.845 1458.688,247.25 1458.003,246.375 C 1457.185,245.329 1456.403,244.762 1455.475,243.854 C 1453.984,242.395 1457.624,240.203 1455.48,238.376 C 1455.095,238.047 1453.768,237.093 1454.035,236.535 C 1454.211,236.167 1455.329,235.179 1454.532,234.874 C 1453.444,234.459 1453.028,234.892 1452.928,233.39 C 1452.897,232.924 1452.932,232.438 1452.85,231.977 C 1452.769,231.515 1453.715,231.59 1453.674,231.34 C 1453.526,230.42 1451.546,229.956 1450.902,229.581 C 1450.278,229.218 1450.219,228.476 1449.424,228.166 C 1448.626,227.855 1447.753,227.837 1446.923,227.669 C 1445.453,227.372 1443.879,226.648 1442.484,226.093 C 1440.361,225.244 1438.181,223.32 1435.897,223.056" id="path4682"/>
    <path d="M 1451.448,289.944 C 1451.193,289.758 1450.906,289.686 1450.588,289.727 C 1451.054,289.521 1451.756,289.376 1452.168,289.799 C 1451.928,289.848 1451.688,289.896 1451.448,289.944" id="path4688"/>
    <path d="M 1452.308,289.151 C 1452.381,288.992 1452.505,288.921 1452.678,288.936 C 1452.554,289.008 1452.431,289.08 1452.308,289.151" id="path4690"/>
  </g>
  <path class="landxx sy" d="M 1532.7863,471.66955 C 1533.4134,470.8751 1534.6211,469.22195 1535.7363,469.00555 C 1534.0343,467.41755 1536.4663,466.41855 1537.8953,466.84555 C 1536.6393,465.31955 1538.511,465.13415 1538.9435,463.67575 C 1539.1832,462.8676 1538.1789,461.23303 1537.2527,461.06135 C 1536.1965,460.86557 1538.2076,459.34994 1536.5709,459.80895 C 1533.8435,460.5738 1533.6426,458.83473 1533.5163,457.42355 C 1533.4015,456.14111 1534.2313,453.79855 1533.0793,452.85455 C 1532.1563,452.09755 1532.8735,450.08517 1532.8385,448.97817 C 1532.8125,448.14717 1534.3573,449.45315 1534.6543,449.60015 C 1534.678,449.70998 1535.0325,448.86227 1535.6887,448.46555 C 1536.4985,447.97601 1535.9513,446.87455 1536.0933,446.10955 C 1536.7983,446.27955 1537.5473,446.07755 1538.2533,446.03855 C 1537.6895,444.88155 1537.3211,444.58155 1537.1593,443.18955 C 1537.1004,442.68297 1537.1463,441.25195 1537.8563,441.07495 C 1538.3903,440.94095 1538.9267,440.93599 1539.3517,441.15999 C 1540.4977,441.76299 1540.7227,442.78674 1541.9037,442.60274 C 1543.3447,442.37774 1544.3003,442.20695 1545.6753,441.61295 C 1547.0853,441.00395 1548.1183,440.15795 1549.7013,440.38495 C 1550.8703,440.55195 1551.5573,441.38255 1552.5463,441.84555 C 1553.6723,442.37355 1555.8771,442.15374 1557.0971,442.04876 C 1560.2108,441.78082 1562.4348,440.00214 1565.2543,438.87456 C 1568.1047,437.73463 1571.2187,439.33375 1574.1437,438.16616 C 1575.6369,437.57012 1577.8288,435.28853 1578.0463,438.85655 C 1578.097,439.68755 1576.0703,441.37255 1575.4863,441.95055 C 1574.4273,442.99955 1573.2463,442.92555 1572.0533,443.62555 C 1569.4933,445.12855 1572.1753,448.38255 1572.3763,450.17855 C 1572.4953,451.24055 1571.8533,452.54355 1571.7763,453.64155 C 1571.6633,455.27655 1571.7766,457.9215 1571.7682,459.54047 C 1571.7484,463.32802 1555.8015,470.57263 1547.7083,476.11255 C 1545.5045,477.62109 1541.9393,481.24955 1539.0733,479.24355 C 1538.0303,478.51455 1537.2791,477.95408 1536.2033,477.21455 C 1534.337,475.93153 1533.4982,476.89194 1533.1887,476.84734 C 1533.1238,476.61234 1533.8631,472.13955 1532.7863,471.66955" id="sy"/>
  <path class="landxx kg" d="M 1766.3196,413.06009 C 1767.4497,411.08726 1769.6798,411.62706 1770.8299,412.1361 C 1771.3194,412.35272 1774.0263,413.21555 1772.9763,414.14055 C 1774.1105,413.88186 1775.2614,412.15619 1775.8647,411.63074 C 1777.0258,410.61967 1777.2703,410.39055 1778.8593,411.07955 C 1779.2713,411.25855 1779.6463,411.69555 1780.0373,411.83655 C 1780.6653,412.06255 1780.5733,410.92355 1781.1573,411.85855 C 1781.5003,412.40755 1782.6553,411.45355 1783.3463,411.47555 C 1781.1843,410.32655 1785.3673,410.07855 1785.7963,410.10755 C 1785.5153,409.56355 1785.2193,409.09355 1785.2863,408.45255 C 1786.4783,408.96455 1787.3663,409.26955 1788.0033,407.86555 C 1788.1593,407.52155 1790.1623,406.88155 1789.4083,406.42455 C 1789.0153,406.18655 1787.6623,406.06555 1787.1693,406.07555 C 1785.5313,406.10655 1784.5443,403.93655 1782.7753,404.99655 C 1782.7103,403.19055 1781.6413,403.98555 1780.6243,403.08655 C 1779.5733,402.15955 1778.2253,401.27755 1777.6563,400.24455 C 1777.7643,400.84155 1778.0863,401.63355 1777.7753,402.22255 C 1777.5653,402.54255 1777.2613,402.69955 1776.8653,402.69155 C 1777.0553,403.10055 1777.0553,403.50855 1776.8653,403.91655 C 1775.6003,402.72455 1772.9143,404.21355 1772.1103,402.21755 C 1771.6853,401.16355 1771.2093,401.04455 1770.1703,401.36055 C 1769.3253,401.61755 1768.3673,400.55255 1767.5063,400.53255 C 1767.8653,398.08755 1771.9873,396.17955 1773.5563,394.70055 C 1771.4363,393.60255 1770.0703,394.76355 1771.2823,392.05455 C 1771.6023,391.33855 1771.2923,390.79455 1772.1123,390.32555 C 1772.9633,389.83855 1774.0993,390.11255 1774.8283,389.62855 C 1776.3553,388.61455 1780.1993,390.05155 1781.7403,390.53455 C 1782.2823,390.65355 1782.7743,390.87955 1783.2183,391.21155 C 1784.1233,391.87855 1784.5543,391.56855 1785.5773,391.67655 C 1786.5933,391.78255 1787.5443,393.03955 1788.6063,392.68455 C 1787.5593,391.69855 1786.6013,388.07155 1788.1823,387.20355 C 1788.5643,386.99355 1791.6233,386.26055 1791.2663,385.62955 C 1793.3643,386.31755 1795.4473,387.42855 1797.4193,388.41555 C 1798.1823,388.79755 1801.7073,390.04855 1802.6463,389.58955 C 1803.3643,389.23555 1802.7403,388.36555 1803.7263,388.36555 C 1805.1623,388.36555 1806.6233,388.90855 1808.0533,388.72255 C 1809.0973,388.58655 1810.0713,388.07655 1811.1363,388.07655 C 1812.2533,388.07655 1813.2013,388.90555 1814.3393,388.56355 C 1815.3553,388.25955 1816.5283,388.57755 1817.5363,388.75455 C 1818.8503,388.98555 1820.1143,388.72755 1821.4313,388.81155 C 1823.6213,388.95155 1826.8543,389.76155 1828.5663,391.17155 C 1829.3033,391.77655 1829.5993,392.59155 1830.7033,392.55055 C 1831.6463,392.51455 1832.2313,392.53355 1833.0433,393.00055 C 1834.0603,393.58355 1835.6843,394.48655 1836.1653,395.62755 C 1836.3063,395.96155 1833.4963,396.91255 1832.9503,397.36555 C 1831.1943,398.82155 1828.4663,399.46555 1826.5003,400.76555 C 1825.5883,401.36955 1825.0704,401.7153 1825.088,402.79028 C 1825.1031,403.71128 1824.343,404.77573 1823.0935,404.66873 C 1822.0261,404.57733 1821.1524,404.87617 1820.0785,404.99891 C 1818.8667,405.13741 1817.7529,404.76934 1816.5648,404.73182 C 1814.9885,404.68204 1815.0512,406.61654 1814.8538,407.96491 C 1814.344,411.44767 1812.5548,410.19949 1810.5273,410.781 C 1808.1996,411.44861 1808.5708,409.19545 1807.536,408.35982 C 1806.9423,407.88037 1805.943,409.37427 1805.488,409.64327 C 1804.419,410.27927 1803.6553,409.55455 1802.4953,409.31355 C 1802.3873,409.81055 1802.4363,410.29055 1802.6443,410.75355 C 1801.7083,410.77755 1801.1362,411.6181 1800.6412,412.28919 C 1799.9135,413.27568 1798.4195,412.6786 1797.6668,413.72264 C 1796.5969,415.20683 1799.329,417.04339 1797.4848,418.18419 C 1796.7245,418.65444 1795.7676,417.79145 1794.9611,418.44455 C 1794.0061,419.21792 1792.7933,419.07039 1791.6288,418.92901 C 1790.3033,418.76807 1787.4554,418.49262 1787.1643,420.54655 C 1785.4762,418.90189 1784.2329,420.02865 1783.2136,419.13438 C 1783.1146,419.38033 1783.1452,418.5972 1782.7567,418.22546 C 1782.3707,418.25946 1781.8085,418.11474 1781.4295,418.04174 C 1780.9905,417.76174 1781.245,417.29802 1780.818,417.05382 C 1780.5072,416.87606 1777.823,419.05698 1776.6982,418.88628 C 1775.8528,418.75798 1775.7283,418.53072 1775.1883,417.85255 C 1774.4346,416.90606 1773.0742,417.30292 1772.1143,417.66655 C 1772.0883,417.40255 1772.0465,417.13955 1772.0275,416.87455 C 1771.4497,417.29766 1770.6026,417.40633 1769.6554,417.28664 C 1767.4199,417.00419 1766.8292,418.1606 1765.9107,417.56537 C 1764.8978,416.90902 1764.6017,412.61305 1765.9963,414.35455 C 1766.0034,413.83746 1766.2176,413.32142 1766.2839,413.11355" id="kg"/>
  <g class="landxx coastxx sb" id="sb" transform="translate(-29.9017, -45.0745)">
    <path d="M 2494.658,873.433 C 2494.691,872.896 2495.73,872.058 2495.357,871.571 C 2495.183,871.345 2494.439,870.182 2494.148,870.193 C 2492.488,870.254 2493.934,872.979 2494.658,873.433" id="path4726"/>
    <path d="M 2493.648,875.663 C 2493.854,874.909 2493.937,874.062 2493.358,873.433 C 2493.358,874.189 2493.286,874.962 2493.648,875.663" id="path4728"/>
    <path d="M 2496.309,866.233 C 2497.194,867.661 2499.446,869.382 2501.208,868.533 C 2500.663,867.893 2499.822,867.668 2499.287,867.018 C 2498.599,866.183 2498.628,865.032 2497.838,864.249 C 2496.772,863.19 2495.274,862.358 2493.868,861.843 C 2493.03,863.35 2495.741,865.074 2496.309,866.233" id="path4730"/>
    <path d="M 2497.898,875.163 C 2498.451,874.644 2498.648,872.708 2497.577,872.624 C 2495.832,872.489 2496.517,875.088 2497.898,875.163" id="path4732"/>
    <path d="M 2499.188,880.203 C 2498.435,879.196 2499.921,878.46 2499.337,877.463 C 2497.934,878.462 2498.268,878.98 2499.188,880.203" id="path4734"/>
    <path d="M 2502.428,878.833 C 2503.293,879.139 2503.42,878.314 2503.293,877.683 C 2503.049,876.473 2502.045,876.101 2501.428,875.163 C 2501.09,874.65 2501.206,872.944 2500.043,873.929 C 2499.433,874.445 2498.169,875.638 2498.328,876.533 C 2499.275,876.783 2500.26,875.764 2500.987,876.758 C 2501.101,877.011 2501.095,877.26 2500.969,877.505 C 2500.893,878.04 2502.088,878.641 2502.428,878.833" id="path4736"/>
    <path d="M 2500.918,880.772 C 2501.094,880.063 2500.854,879.97 2500.198,880.493 C 2500.438,880.586 2500.678,880.679 2500.918,880.772" id="path4738"/>
    <path d="M 2501.718,869.183 C 2501.903,868.486 2501.66,868.367 2500.989,868.824 C 2501.231,868.943 2501.475,869.063 2501.718,869.183" id="path4740"/>
    <path d="M 2502.718,869.333 C 2503.451,869.473 2503.451,869.229 2502.718,868.603 C 2502.718,868.846 2502.718,869.089 2502.718,869.333" id="path4742"/>
    <path d="M 2503.798,880.563 C 2505.063,880.188 2504.285,879.315 2504.519,878.403 C 2503.113,878.11 2502.672,879.886 2503.798,880.563" id="path4744"/>
    <path d="M 2507.979,870.043 C 2507.649,869.755 2507.898,869.501 2507.258,869.692 C 2507.498,869.809 2507.738,869.926 2507.979,870.043" id="path4748"/>
    <path d="M 2508.769,870.913 C 2508.475,870.313 2507.858,870.061 2507.258,870.413 C 2507.771,870.549 2508.275,870.716 2508.769,870.913" id="path4750"/>
    <path d="M 2512.798,874.873 C 2514.206,876.324 2516.335,876.967 2517.838,878.403 C 2518.596,876.583 2516.079,873.945 2514.39,873.624 C 2513.266,873.409 2512.082,872.106 2511.258,871.372 C 2510.813,870.976 2510.574,870.486 2509.984,870.268 C 2509.409,870.057 2508.839,870.194 2508.338,869.763 C 2509.123,871.749 2510.659,874.166 2512.798,874.873" id="path4752"/>
    <path d="M 2511.938,883.513 C 2512.377,883.306 2512.58,883.206 2512.518,882.654 C 2511.751,882.553 2511.558,882.839 2511.938,883.513" id="path4754"/>
    <path d="M 2518.198,889.633 C 2519.508,889.814 2520.369,889.469 2521.612,890.105 C 2522.322,890.467 2523.836,890.98 2524.173,889.839 C 2524.577,888.471 2522.253,886.321 2521.003,886.194 C 2519.759,886.069 2518.155,886.784 2517.313,885.553 C 2516.766,884.753 2516.061,884.45 2515.425,885.463 C 2514.447,887.02 2516.783,889.316 2518.198,889.633" id="path4756"/>
    <path d="M 2516.838,878.613 C 2516.647,878.151 2516.573,877.671 2516.618,877.173 C 2515.291,877.173 2516.08,878.305 2516.838,878.613" id="path4758"/>
    <path d="M 2516.328,904.753 C 2516.82,904.994 2517.91,905.074 2518.198,905.253 C 2518.946,905.717 2518.908,906.833 2519.858,907.053 C 2520.87,905.689 2516.036,903.152 2516.328,904.753" id="path4760"/>
    <path d="M 2520.798,883.653 C 2521.146,883.005 2520.733,882.404 2519.998,882.503 C 2519.889,883.092 2520.213,883.546 2520.798,883.653" id="path4764"/>
    <path d="M 2520.578,884.233 C 2520.993,883.98 2521.427,883.763 2521.878,883.583 C 2521.282,883.361 2520.774,883.645 2520.578,884.233" id="path4766"/>
    <path d="M 2525.328,883.803 C 2526.794,884.879 2527.521,886.584 2528.718,887.904 C 2529.156,887.033 2528.346,883.776 2527.929,882.857 C 2527.678,882.306 2527.078,881.192 2526.527,880.901 C 2525.674,880.45 2527.11,879.655 2526.709,879.077 C 2526.262,878.433 2526.105,877.371 2525.433,876.904 C 2525.115,876.683 2523.708,876.139 2524.017,877.211 C 2524.612,879.287 2524.319,881.79 2525.328,883.803" id="path4768"/>
    <path d="M 2531.158,897.692 C 2532.394,898.115 2533.716,898.107 2534.978,898.413 C 2534.218,896.019 2533.576,895.773 2531.599,894.725 C 2530.951,894.381 2527.908,892.171 2527.488,893.662 C 2528.022,893.95 2528.578,894.118 2528.802,894.727 C 2529.326,896.146 2530.015,896.748 2531.158,897.692" id="path4770"/>
    <path d="M 2530.008,889.413 C 2530.479,888.205 2529.85,886.891 2528.998,886.033 C 2529.065,886.71 2529.049,887.385 2528.946,888.057 C 2528.902,888.552 2529.875,888.928 2530.008,889.413" id="path4772"/>
    <path d="M 2561.838,898.703 C 2562.402,897.953 2563.991,898.225 2563.988,897.043 C 2562.846,896.73 2561.665,897.114 2561.108,898.203 C 2561.352,898.37 2561.595,898.537 2561.838,898.703" id="path4774"/>
    <path d="M 2568.168,905.833 C 2568.385,905.736 2568.602,905.64 2568.818,905.543 C 2568.58,904.946 2568.07,904.691 2567.518,905.113 C 2567.708,905.378 2567.924,905.618 2568.168,905.833" id="path4778"/>
  </g>
  <g class="landxx coastxx om" id="om" transform="translate(-29.9017, -45.0745)">
    <path d="M 1741.968,630.145 C 1742.639,629.55 1742.871,628.756 1743.193,627.955 C 1743.376,627.714 1743.59,627.509 1743.839,627.337 C 1744.112,627.016 1743.415,626.015 1743.338,625.68 C 1742.967,627.242 1741.335,628.403 1741.968,630.145" id="path4692"/>
    <path class="landxx om" d="M 1718.639,579.096 C 1718.846,578.517 1719.109,577.966 1719.428,577.441 C 1720.108,578.201 1720.762,577.167 1720.439,576.504 C 1721.023,576.82 1722.196,578.807 1720.729,578.448 C 1721.672,579.103 1720.983,583.117 1719.942,582.932 C 1718.528,582.682 1720.443,579.188 1718.639,579.096" id="path4696"/>
    <path class="landxx om" d="M 1692.715,639.39554 C 1703.2427,636.97556 1709.833,634.43977 1714.601,631.85039 C 1716.551,628.13128 1714.4532,633.98886 1715.386,629.847 C 1716.148,626.46378 1716.6753,623.08694 1717.443,619.863 C 1717.9742,617.63248 1719.219,614.828 1717.929,612.704 C 1717,611.176 1715.116,609.87 1714.53,608.182 C 1713.793,606.06 1717.8,598.757 1715.829,597.601 C 1716.714,596.547 1718.451,597.459 1719.219,596.45 C 1720.062,595.343 1718.706,595.712 1718.429,595.154 C 1717.693,593.668 1717.105,590.405 1718.096,588.898 C 1718.558,588.194 1719.499,589.276 1719.353,589.833 C 1718.894,591.57 1720.648,589.687 1720.926,589.423 C 1721.784,588.605 1721.776,588.743 1722.31,589.897 C 1723.198,591.815 1724.36,593.628 1725.775,595.2 C 1728.012,597.685 1731.142,598.578 1734.328,599.218 C 1735.819,599.517 1737.086,600.104 1738.604,600.274 C 1740.634,600.502 1741.744,602.974 1742.906,604.354 C 1743.844,605.467 1744.481,606.811 1745.416,607.876 C 1746.389,608.985 1746.903,609.031 1748.188,609.544 C 1751.201,610.748 1747.793,615.024 1747.193,616.791 C 1746.059,620.129 1742.483,621.871 1741.514,625.447 C 1741.285,626.291 1740.974,628.732 1739.636,628.33 C 1738.577,628.013 1738.783,625.993 1737.429,626.834 C 1735.814,627.812 1735.902,631.327 1735.486,632.772 C 1734.941,634.661 1735.557,636.125 1736.056,637.955 C 1736.463,639.451 1736.719,640.582 1734.902,640.637 C 1731.982,640.724 1728.77,641.585 1728.051,644.733 C 1727.769,645.97 1727.544,647.33 1726.767,648.375 C 1725.8,649.676 1723.873,649.605 1722.344,649.72 C 1720.492,649.858 1719.75,650.159 1718.663,651.618 C 1717.61,653.03 1718.294,653.299 1718.206,654.912 C 1718.136,656.195 1716.825,657.126 1715.792,657.629 C 1715.294,657.87 1714.793,658.194 1714.284,657.793 C 1713.405,657.1 1712.68,657.289 1711.589,657.289 C 1708.43,657.289 1705.6655,660.35419 1702.4805,660.26419 C 1700.4175,660.20519 1701.344,655.155 1699.129,654.913 C 1700.068,653.933 1697.776,650.487 1697.31,649.624 C 1695.68,646.612 1693.7511,642.61967 1692.6831,639.36367" id="path4786"/>
  </g>
  <g class="landxx coastxx pa" id="pa" transform="translate(-29.9017, -45.0745)">
    <path class="landxx pa" d="M 668.592,733.032 C 667.901,732.365 669.655,731.536 669.78,730.907 C 670.017,729.727 668.798,728.438 670.434,727.731 C 671.768,727.154 669.312,726.326 669.312,725.832 C 669.312,724.809 670.1134,722.3954 669.8834,721.9944 C 670.8374,721.04471 671.751,723.23574 672.407,721.872 C 672.905,722.314 673.632,722.869 673.847,723.528 C 673.846,724.009 673.822,724.489 673.775,724.967 C 673.862,725.344 674.527,724.931 674.472,725.48 C 674.237,727.834 676.552,726.985 678.023,727.272 C 677.741,726.772 677.429,726.293 677.086,725.832 C 678.331,725.452 679.545,728.481 680.986,728.68 C 684.02,729.098 685.723,726.736 688.283,725.78 C 690.038,725.125 692.088,724.603 693.418,723.259 C 694.145,722.524 694.553,721.583 695.724,721.573 C 697.035,721.562 698.307,722.144 699.621,722.016 C 699.385,722.314 699.169,722.626 698.973,722.953 C 701.534,722.667 705.022,723.265 706.901,725.23 C 707.795,726.165 709.801,730.07 711.429,729.577 C 710.22,731.439 711.5964,732.2704 712.2284,734.1954 C 712.9094,736.2674 710.745,738.357 709.197,739.441 C 709.03,738.908 708.766,738.427 708.405,738.001 C 707.878,738.76885 708.17859,738.89272 708.127,739.49 C 708.07376,740.10643 707.37671,740.65737 706.92179,742.0284 C 704.96979,740.6204 703.173,737.215 703.077,734.833 C 704.714,735.442 704.149,733.472 704.847,732.728 C 705.621,731.901 707.661,733.948 708.333,734.329 C 707.883,733.779 707.252,733.463 706.707,733.027 C 705.932,732.407 706.073,731.373 705.597,730.585 C 705.798,731.192 705.467,731.726 704.949,732.025 C 705.272,730.575 704.046,732.084 703.509,730.728 C 703.677,731.313 703.653,731.889 703.437,732.457 C 702.604,730.924 702.554,729.374 701.196,728.589 C 700.583,728.235 700.002,727.818 699.452,727.373 C 698.473,726.58 698.96,726.619 697.831,726.685 C 696.274,726.775 693.629,727.108 693.184,729.086 C 692.845,730.595 690.675,733.181 688.819,732.88 C 685.893,732.406 689.042,736.735 689.699,737.455 C 690.203,738.007 691.279,739.139 690.665,739.959 C 690.123,740.684 688.87,740.205 688.197,740.804 C 687.317,741.587 685.288,742.805 684.04,742.116 C 683.744,741.953 683.85,739.959 683.657,739.476 C 683.278,738.527 683.563,737.074 682.99,736.274 C 682.601,736.727 682.193,736.751 681.766,736.345 C 683.002,738.733 680.298,739.33 679.174,737.321 C 678.22,735.617 678.253,734.366 676.232,733.645 C 675.572,733.41 674.894,733.992 674.444,733.274 C 673.882,732.38 673.134,733.088 672.694,732.385 C 671.983,733.399 670.937,732.331 670.03,732.817 C 669.022,733.356 669.74,734.535 669.022,735.194 C 669.107,734.30901 668.92367,733.54034 668.592,733.032" id="path2682"/>
    <path d="M 672.913,733.248 C 672.914,733.774 673.082,733.751 673.417,733.176 C 673.223,733.049 673.055,733.073 672.913,733.248" id="path4792"/>
    <path d="M 677.665,738.504 C 677.212,738.966 676.432,739.313 676.673,740.117 C 676.872,740.781 677.539,741.201 678.169,740.736 C 677.698,740.067 677.849,739.251 677.665,738.504" id="path4794"/>
    <path d="M 681.336,739.944 C 681.89,739.779 682.37,739.491 682.776,739.08 C 682.223,739.245 681.743,739.533 681.336,739.944" id="path4796"/>
    <path d="M 699.913,731.448 C 699.193,731.565 699.112,732.408 699.265,732.96 C 700.274,733.109 700.094,732.052 699.913,731.448" id="path4802"/>
  </g>
  <g class="landxx coastxx ar" id="ar" transform="translate(-29.9017, -45.0745)">
    <path d="M 867.308,1142.573 C 867.426,1142.068 867.189,1141.995 866.598,1142.353 C 866.834,1142.427 867.071,1142.5 867.308,1142.573" id="path4200"/>
    <path d="M 868.538,1144.083 C 868.601,1143.229 867.724,1142.811 867.028,1143.213 C 867.483,1143.587 867.986,1143.877 868.538,1144.083" id="path4202"/>
    <path class="landxx ar" d="M 788.547,1057.823 C 788.684,1057.101 789.512,1056.925 789.677,1056.254 C 789.901,1055.339 789.365,1053.985 789.407,1053.003 C 789.526,1050.21 791.079,1049.058 792.094,1046.687 C 792.754,1045.146 792.744,1043.563 793.094,1041.956 C 793.265,1041.169 793.453,1040.283 793.904,1039.601 C 794.355,1038.919 795.195,1039.468 795.896,1039.145 C 798.876,1037.773 795.804,1035.961 794.806,1034.396 C 793.67,1032.616 795.679,1031.504 795.047,1029.773 C 794.597,1028.541 791.061,1022.217 794.088,1022.113 C 793.782,1021.332 792.16,1019.497 792.398,1018.728 C 792.959,1016.917 794.131,1015.645 795.846,1014.883 C 797.362,1014.21 799.806,1013.609 800.53,1011.933 C 801.257,1010.248 801.134,1007.987 801.424,1006.194 C 801.539,1005.485 801.31528,1003.4717 801.01228,1002.7627 C 800.80528,1002.2777 799.67983,1002.6028 799.79783,1002.0898 C 800.41196,1002.1397 802.19647,997.65493 804.032,996.732 C 806.05,995.723 806.164,995.515 806.4,993.093 C 808.026,993.419 808.852,995.073 810.308,995.713 C 811.983,996.449 813.95,995.508 815.699,995.776 C 819.861,996.412 819.688,999.579 821.81,1002.534 C 821.949,1001.198 821.919,999.931 822.456,998.67 C 822.745,997.99 823.217,995.566 823.913,995.289 C 824.443,995.078 825.438,995.186 826.06,995.032 C 827.205,994.749 828.52,994.973 829.688,994.973 C 830.379,994.973 831.906,994.6 832.125,995.382 C 832.421,996.438 834.542,997.696 835.316,998.47 C 836.209,999.363 836.836,999.485 837.393,1000.647 C 837.831,1001.562 838.402,1002.474 839.027,1003.271 C 840.473,1005.114 842.895,1006.414 844.781,1007.763 C 845.862,1008.536 846.382,1009.633 847.37,1010.454 C 848.287,1011.216 849.555,1011.169 850.617,1011.59 C 852.049,1012.158 852.986,1012.534 854.576,1012.534 C 855.782,1012.534 856.641,1013.628 857.637,1014.139 C 860.219,1015.467 862.237,1017.975 865.09,1018.893 C 865.929,1019.164 866.929,1019.249 867.602,1019.873 C 868.581,1020.783 869.005,1020.251 870.007,1020.85 C 870.974,1021.428 872.256,1021.517 872.981,1022.434 C 873.493,1023.082 874.061,1023.67 874.567,1024.326 C 875.566,1025.619 873.672,1026.984 873.418,1028.481 C 873.038,1030.71 871.043,1031.298 871.504,1033.92 C 871.926,1036.32 870.142,1037.061 869.621,1039.102 C 869.362,1040.119 869.003,1039.757 869.551,1040.972 C 871.019,1040.401 872.589,1040.391 874.118,1040.687 C 876.823,1041.21 879.362,1042.798 882.011,1041.94 C 882.842,1041.671 883.237,1042.945 884.094,1042.333 C 884.918,1041.745 885.798,1043.094 886.471,1043.352 C 886.94,1042.372 887.013,1041.019 888.341,1040.832 C 889.358,1040.689 889.88,1041.85 890.761,1041.732 C 892.158,1041.546 891.938,1039.861 892.28,1038.929 C 892.673,1037.859 894.939,1037.969 894.826,1037.348 C 894.79,1037.151 896.129,1035.218 896.286,1035.244 C 897.28,1035.405 897.087,1033.077 897.205,1032.257 C 897.44,1030.621 897.417,1028.841 896.996,1027.234 C 896.566,1025.596 897.596,1026.155 898.491,1026.722 C 898.694,1025.417 900.421,1025.547 901.333,1025.87 C 902.912,1026.43 903.237,1028.065 903.754,1029.484 C 904.484,1031.49 904.691,1032.621 904.749,1034.773 C 904.779,1035.895 905.061,1036.907 904.891,1038.04 C 904.793,1038.61 904.7,1039.181 904.611,1039.752 C 904.254,1039.615 903.914,1039.651 903.592,1039.862 C 902.399,1040.538 901.628,1041.438 900.329,1041.968 C 899.68,1042.234 897.538,1042.979 897.462,1043.846 C 897.345,1045.165 895.889,1045.285 894.997,1045.859 C 894.478,1046.192 894.386,1046.914 893.939,1047.354 C 893.287,1047.997 892.456,1048.473 891.87,1049.182 C 892.471,1049.385 892.732,1049.897 892.44,1050.483 C 891.116,1049.115 888.504,1054.56 887.761,1055.621 C 886.466,1057.472 886.126,1061.244 883.706,1062.218 C 882.617,1062.656 882.697,1064.128 882.166,1064.995 C 881.885,1065.453 881.448,1065.724 880.919,1065.814 C 881.217,1066.585 880.935,1067.6089 880.363,1068.1179 C 878.737,1069.5639 880.69581,1069.7468 880.85281,1071.2878 C 880.93081,1072.0528 880.45136,1072.1899 880.41536,1072.7929 C 880.37336,1073.5039 880.469,1073.679 880.537,1074.379 C 880.606,1075.082 880.54027,1075.7135 880.22527,1076.3455 C 879.58727,1077.6245 880.8,1076.956 880.691,1078.189 C 880.553,1079.751 879.659,1080.044 879.952,1081.746 C 880.203,1083.205 881.077,1083.752 880.603,1085.231 C 880.175,1086.565 881.082,1088.205 881.748,1089.318 C 882.908,1091.259 880.298,1090.55 880.4,1091.676 C 880.534,1093.148 879.661,1094.642 880.352,1096.125 C 880.739,1096.955 883.865,1101.446 881.349,1101.314 C 882.49,1102.241 882.613,1103.44 883.978,1104.293 C 885.462,1105.221 886.939,1105.937 888.525,1106.671 C 891.113,1107.87 894.745,1109.9 893.341,1113.099 C 892.33,1115.404 895.383,1120.049 898.134,1118.904 C 899.219,1118.451 899.698,1121.714 899.926,1122.298 C 900.607,1124.047 899.435,1126.015 898.72,1127.584 C 898.062,1129.027 896.376,1131.076 896.326,1132.67 C 896.273,1134.36 896.437,1134.859 894.877,1135.973 C 892.051,1137.988 888.808,1139.021 885.443,1139.7 C 881.382,1140.519 877.447,1141.71 873.281,1141.899 C 871.673,1141.972 870.324,1141.94 868.745,1141.797 C 866.944,1141.634 865.725,1140.488 863.998,1140.264 C 864.238,1140.731 864.561,1141.131 864.964,1141.466 C 865.409,1141.873 864.918,1142.755 865.171,1143.261 C 865.313,1143.543 868.595,1145.972 866.518,1145.164 C 867.475,1145.863 867.931,1146.357 868.142,1147.549 C 868.294,1148.409 868.425,1149.834 867.098,1149.124 C 867.295,1149.992 867.543,1150.877 867.511,1151.775 C 867.486,1152.502 866.74,1153.133 867.28,1153.782 C 867.674,1154.256 868.221,1154.585 868.676,1154.994 C 869.369,1155.616 869.01,1156.623 869.978,1156.104 C 869.388,1159.638 865.793,1160.174 862.817,1160.442 C 858.87,1160.797 855.64,1158.131 851.758,1157.544 C 853.365,1157.231 851.68,1156.647 851.165,1156.913 C 850.487,1157.263 850.08,1158.132 850.051,1158.865 C 849.97,1160.981 852.551,1162.516 852.946,1164.599 C 853.267,1166.288 853.306,1168.38 855.183,1169.098 C 856.141,1169.464 858.773,1169.669 858.029,1171.293 C 858.429,1171.392 862.126,1171.805 861.877,1170.845 C 861.674,1170.061 859.967,1170.03 859.389,1169.713 C 860.619,1169.901 861.277,1168.633 862.419,1168.564 C 863.596,1168.493 864.811,1170.132 865.009,1171.154 C 865.193,1172.1 866.389,1173.491 865.729,1174.464 C 865.272,1175.137 863.589,1175.347 862.863,1175.258 C 861.654,1175.111 861.85,1173.737 861.25,1173.139 C 860.267,1172.16 858.919,1171.764 857.716,1172.57 C 854.53,1174.703 859.787,1176.44 861.629,1175.974 C 861.475,1176.979 860.403,1177.308 859.596,1177.64 C 858.002,1178.297 858.068,1179.02 857.451,1180.439 C 856.81,1181.916 856.708,1182.418 857.726,1183.799 C 858.079,1184.278 858.573,1184.862 858.634,1185.485 C 858.515,1185.906 858.507,1186.328 858.609,1186.752 C 859.035,1187.146 859.26,1187.632 859.283,1188.21 C 859.235,1188.481 859.371,1189.463 858.974,1189.569 C 858.411,1189.717 857.638,1189.959 857.559,1190.665 C 857.427,1191.851 859.226,1192.037 859.034,1192.97 C 858.824,1193.989 851.665,1192.538 853.199,1195.135 C 850.108,1194.482 849.289,1198.191 848.862,1200.445 C 848.145,1204.22 850.623,1205.718 853.465,1207.639 C 854.861,1208.583 856.067,1209.832 857.631,1210.518 C 859.624,1211.392 862.036,1210.212 863.926,1211.331 C 866.567,1212.894 866.97,1218.125 862.85,1217.595 C 863.609,1217.9 864.154,1217.225 864.87,1217.165 C 865.958,1217.074 866.385,1217.818 867.03,1218.535 C 866.4,1218.61 866.196,1218.978 866.52,1219.535 C 865.332,1219.294 862.941,1222.362 862.205,1223.091 C 861.241,1224.042 860.498,1224.967 859.785,1226.133 C 859.304,1226.92 858.836,1227.386 858.879,1228.357 C 858.913,1229.13 858.992,1229.864 858.31,1230.416 C 858.897,1230.52 859.358,1230.201 859.47,1229.616 C 859.545,1231.757 861.898,1235.472 858.698,1236.494 C 856.341,1237.245 855.585,1233.661 853.56,1233.296 C 854.081,1233.492 855.455,1234.542 855,1235.236 C 854.663,1235.752 853.279,1235.307 852.77,1235.736 C 854.743,1234.928 855.579,1236.238 857.31,1236.606 C 857.096,1237.93 855.007,1237.861 854.526,1239.33 C 853.879,1241.307 855.27,1242.587 854.14,1244.596 C 855.6,1242.784 857.715,1246.76 858.246,1247.8 C 859.062,1249.398 855.748,1248.462 854.93,1249.136 C 855.788,1248.854 857.807,1249.662 858.174,1249.447 C 859.355,1248.753 860.269,1250.339 861.169,1251.052 C 862.915,1252.436 864.783,1253.444 865.59,1255.466 C 865.166,1254.743 864.194,1254.777 863.478,1254.7 C 862.195,1254.564 861.023,1253.868 859.797,1253.511 C 856.708,1252.611 853.382,1252.226 850.18,1252.226 C 848.761,1252.226 847.341,1252.226 845.922,1252.226 C 844.728,1252.226 842.213,1252.712 841.182,1252.068 C 839.575,1251.064 838.224,1248.816 836.14,1248.556 C 836.796,1247.59 836.192,1246.02 835.359,1245.414 C 834.257,1244.612 835.111,1243.562 834.518,1242.575 C 834.152,1241.967 833.018,1240.387 832.116,1240.8 C 831.41,1241.121 830.797,1240.667 830.09,1240.846 C 829.566,1240.978 829.292,1241.606 828.843,1241.865 C 827.895,1242.413 826.554,1239.896 826.257,1239.301 C 825.666,1238.117 823.287,1237.306 823.097,1236.102 C 822.997,1235.465 822.662,1235.447 822.378,1234.915 C 822.153,1234.495 822.336,1233.958 822.17,1233.507 C 821.935,1232.872 821.183,1232.64 820.95,1231.997 C 820.688,1231.272 820.913,1230.508 820.8,1229.767 C 822.002,1229.961 822.092,1229.26 822.39,1228.328 C 822.56,1227.797 822.241,1227.363 822.67,1226.888 C 823.051,1226.467 823.798,1226.764 824.191,1226.308 C 824.966,1225.409 824.135,1223.944 823.535,1223.149 C 822.844,1222.235 824.662,1222.165 824.761,1221.419 C 824.87,1220.591 823.655,1219.545 823.065,1219.127 C 821.866,1218.278 821.691,1217.736 821.699,1216.292 C 821.704,1215.3 821.944,1211.782 823.681,1212.049 C 823.33,1211.259 822.362,1211.183 822.403,1210.352 C 822.429,1209.819 821.882,1209.391 822.117,1208.834 C 822.284,1208.439 822.71,1208.241 822.901,1207.854 C 823.685,1206.263 820.987,1203.792 819.871,1202.979 C 820.849,1202.933 821.435,1201.069 820.361,1200.764 C 819.594,1200.546 818.349,1198.78 819.169,1198.025 C 819.866,1197.382 820.083,1197.812 820.572,1196.772 C 820.986,1195.892 821.128,1195.872 820.343,1195.111 C 819.653,1194.441 818.89,1193.822 818.106,1193.265 C 816.995,1192.474 815.425,1193.306 814.413,1192.198 C 813.554,1191.257 819.131,1191.66 819.581,1191.03 C 819.332,1191.378 819.606,1189.197 819.721,1189.519 C 818.876,1187.146 815.044,1189.567 814.106,1187.243 C 813.871,1186.865 813.641,1186.483 813.418,1186.097 C 813.258,1185.476 814.215,1184.923 813.801,1184.332 C 813.274,1183.579 811.958,1183.148 812.696,1182.315 C 813.618,1181.274 809.052,1180.455 810.703,1178.975 C 811.586,1178.183 810.476,1177.543 809.669,1177.405 C 808.132,1177.142 807.537,1176.565 807.059,1175.079 C 806.694,1173.944 806.655,1172.592 806.26,1171.518 C 805.96,1170.704 804.739,1170.011 805.902,1169.189 C 806.623,1168.68 807.883,1169.091 806.973,1167.867 C 806.197,1166.822 804.508,1165.372 804.656,1163.95 C 804.911,1161.489 802.384,1159.205 801.959,1156.978 C 801.787,1156.075 801.85,1154.876 802.26,1154.066 C 802.914,1152.773 801.068,1152.454 800.862,1151.359 C 802.577,1151.936 801.47,1149.449 800.977,1148.88 C 799.983,1147.734 800.269,1147.158 801.652,1147.109 C 800.774,1146.253 801.225,1143.76 800.819,1142.572 C 800.195,1140.751 802.021,1140.398 803.127,1139.394 C 804.331,1138.301 801.719,1136.549 801.353,1135.204 C 800.89,1133.504 798.255,1131.425 798.523,1129.747 C 798.703,1128.62 797.573,1125.885 797.047,1124.806 C 796.484,1123.648 796.192,1121.098 797.109,1120.094 C 797.604,1119.552 798.05,1120.133 798.346,1119.476 C 798.568,1118.981 798.517,1118.263 798.893,1117.846 C 799.19,1117.516 799.978,1117.312 800.019,1116.819 C 800.117,1115.644 799.696,1114.399 799.243,1113.338 C 798.86,1112.442 798.394,1111.584 797.958,1110.713 C 797.851,1110.498 796.972,1109.916 796.971,1109.82 C 796.964,1109.033 797.94,1109.38 798.038,1108.488 C 798.135,1107.612 797.961,1106.728 797.876,1105.859 C 797.793,1105.012 798,1101.054 799.287,1101.284 C 800.533,1101.506 798.769,1098.378 798.664,1098.127 C 798.177,1096.962 798.8,1092.947 797.428,1092.503 C 796.649,1092.251 797.121,1093.519 796.051,1092.188 C 795.266,1091.211 795.489,1090.74 795.656,1089.654 C 795.787,1088.802 795.17,1088.491 794.592,1088.07 C 793.844,1087.525 794.088,1087.009 793.85,1086.24 C 793.328,1084.554 791.628,1083.507 791.141,1081.808 C 792.751,1082.549 790.963,1080.347 790.347,1080.033 C 788.875,1079.283 788.382,1076.866 788.319,1075.416 C 788.242,1073.635 789.599,1074.295 789.211,1072.399 C 788.927,1071.008 790.336,1068.469 789.341,1067.338 C 791.775,1068.964 790.856,1064.761 790.527,1063.864 C 790.045,1062.554 787.771,1059.083 788.547,1057.823" id="path4518"/>
    <path class="landxx ar" d="M 865.438,1257.553 C 866.895,1258.868 868.727,1259.769 869.837,1261.444 C 868.834,1259.596 866.527,1262.818 870.344,1262.855 C 871.773,1262.869 872.207,1264.459 873.12,1265.276 C 874.24,1266.279 875.886,1266.528 876.935,1267.463 C 879.711,1269.937 883.738,1271.094 887.102,1272.628 C 888.634,1273.326 890.861,1274.202 892.561,1274.152 C 893.398,1274.128 894.227,1273.956 895.067,1273.994 C 896.25,1274.047 895.984,1275.286 895.677,1276.274 C 894.728,1275.75 893.613,1276.658 892.493,1276.226 C 891.271,1275.754 889.587,1277.324 888.054,1276.996 C 886.333,1276.627 884.833,1276.21 883.047,1276.156 C 881.056,1276.096 878.507,1276.19 876.637,1275.409 C 875.434,1274.908 874.708,1276.143 873.677,1274.621 C 872.73,1273.223 872.195,1271.473 871.436,1269.965 C 869.386,1265.886 867.168,1261.777 865.438,1257.553" id="path4520"/>
    <path d="M 899.858,1276.203 C 900.839,1274.985 902.983,1275.323 904.318,1274.833 C 902.505,1274.371 900.815,1274.98 899.138,1275.623 C 899.357,1275.842 899.598,1276.035 899.858,1276.203" id="path4874"/>
  </g>
  <g class="landxx coastxx gb eu" id="gb" transform="translate(-29.9017, -45.0745)">
    <path class="landxx gb eu" d="M 1260.508,337.032 C 1261.121,336.062 1262.494,336.081 1263.388,335.52 C 1263.057,335.277 1262.743,335.012 1262.448,334.727 C 1264.3,335.304 1265.011,333.238 1265.893,332.227 C 1266.732,331.266 1268.474,332.53 1268.718,330.767 C 1270.35,331.522 1272.386,329.639 1274.016,330.766 C 1274.677,331.222 1275.068,332.288 1275.375,332.975 C 1275.319,332.851 1276.594,334.325 1276.56,334.063 C 1276.597,334.341 1275.171,335.037 1274.978,335.231 C 1275.478,335.181 1276.972,334.543 1277.301,335.202 C 1277.635,335.87 1278.28,337.23 1277.138,337.536 C 1277.316,336.828 1277.055,335.884 1276.128,336.096 C 1276.687,336.427 1276.615,337.009 1276.268,337.464 C 1276.561,337.31 1276.811,337.331 1277.018,337.527 C 1277.091,338.02 1275.965,338.658 1275.586,338.698 C 1274.608,338.801 1274.61,340.437 1273.42,340.084 C 1272.419,339.787 1272.235,339.475 1271.382,340.035 C 1270.657,340.511 1269.467,340.023 1269.788,339.049 C 1269.388,339.07 1266.639,336.003 1266.454,337.971 C 1266.126,341.45 1261.861,338.159 1260.508,337.032" id="path2690"/>
    <path d="M 1268.788,315.288 C 1267.835,314.948 1268.392,313.982 1269.218,313.704 C 1268.001,311.325 1266.78,315.744 1268.788,315.288" id="path2640"/>
    <path d="M 1269.147,312.048 C 1269.157,311.829 1270.483,310.945 1269.891,310.769 C 1269.23,310.571 1267.646,310.447 1267.707,311.544 C 1268.25,311.531 1268.73,311.699 1269.147,312.048" id="path2642"/>
    <path d="M 1268.788,312.84 C 1268.989,312.07 1268.748,311.95 1268.068,312.48 C 1268.308,312.6 1268.548,312.72 1268.788,312.84" id="path2644"/>
    <path d="M 1270.297,320.616 C 1270.624,320.326 1270.984,320.086 1271.377,319.895 C 1270.799,319.765 1270.416,320.048 1270.297,320.616" id="path2646"/>
    <path d="M 1272.958,308.16 C 1274.131,309.783 1275.458,306.743 1274.118,307.223 C 1274.82,307.136 1277.147,306.159 1275.628,305.856 C 1276.965,305.247 1276.71,303.467 1275.297,304.444 C 1274.603,304.924 1273.775,305.036 1273.061,305.464 C 1272.285,305.929 1272.959,306.739 1271.957,307.08 C 1271.938,306.815 1271.912,306.551 1271.877,306.288 C 1271.331,306.394 1270.85,306.635 1270.437,307.008 C 1271.049,307.443 1271.405,307.917 1270.367,308.16 C 1270.894,308.271 1271.374,308.487 1271.807,308.807 C 1271.309,309.038 1270.829,309.302 1270.367,309.599 C 1271.164,310.914 1272.775,309.026 1272.958,308.16" id="path2648"/>
    <path d="M 1271.877,319.608 C 1272.429,319.424 1272.862,319.088 1273.177,318.599 C 1272.628,318.742 1272.149,319.109 1271.877,319.608" id="path2650"/>
    <path d="M 1273.468,327.672 C 1274.66,327.143 1275.684,325.99 1274.058,325.085 C 1273.016,324.505 1272.451,326.249 1272.098,326.88 C 1272.576,326.587 1273.079,326.347 1273.608,326.16 C 1273.71,326.678 1273.664,327.182 1273.468,327.672" id="path2652"/>
    <path d="M 1276.417,315.937 C 1277.248,315.708 1278.57,314.968 1278.427,313.92 C 1275.265,315.035 1276.614,311.215 1274.907,310.608 C 1274.731,311.072 1274.634,311.551 1274.617,312.047 C 1273.189,311.712 1273.376,312.618 1272.307,312.407 C 1272.134,313.854 1273.893,312.956 1274.547,313.631 C 1272.913,313.749 1274.582,314.774 1275.207,314.768 C 1275.896,314.762 1276.676,314.411 1277.347,314.711 C 1276.757,314.852 1276.371,315.318 1276.417,315.937" id="path2654"/>
    <path d="M 1275.627,373.68 C 1276.508,373.88 1277.77,372.163 1278.446,371.623 C 1279.691,370.629 1282.132,371.454 1282.967,369.936 C 1282.971,370.867 1284.826,371.768 1285.591,371.827 C 1287.019,371.938 1287.945,369.119 1287.937,368.063 C 1288.386,368.903 1289.957,368.056 1290.534,367.852 C 1292.049,367.317 1293.601,368.014 1294.707,369.072 C 1294.801,368.142 1295.751,368.423 1296.374,368.536 C 1297.054,368.66 1298.993,368.06 1297.157,367.704 C 1297.99,367.704 1303.232,367.679 1301.757,366.264 C 1302.635,367.159 1303.75,366.459 1304.787,366.697 C 1305.058,366.749 1305.179,366.9 1305.152,367.149 C 1305.185,367.576 1306.892,367.108 1306.876,367.11 C 1308.409,366.916 1309.871,366.846 1311.368,367.289 C 1312.625,367.661 1314.399,366.693 1315.637,366.368 C 1316.728,366.081 1316.969,365.721 1317.698,365.05 C 1318.485,364.328 1320.016,364.488 1320.489,363.4 C 1321.014,362.193 1315.38,362.026 1314.788,361.945 C 1315.188,361.838 1315.575,361.693 1315.948,361.512 C 1315.221,361.343 1314.448,361.164 1313.788,361.657 C 1314.438,360.964 1315.319,361.24 1316.137,361.082 C 1317.512,360.815 1317.464,359.43 1316.088,359.281 C 1316.621,359.391 1316.955,358.948 1317.376,358.743 C 1318.213,358.336 1318.839,359.167 1319.759,358.344 C 1319.53,357.85 1319.146,357.562 1318.609,357.48 C 1319.184,357.373 1320.56,357.459 1319.039,356.905 C 1321.139,357.535 1321.436,356.177 1322.333,354.682 C 1323.496,352.744 1323.289,350.871 1321.043,349.757 C 1319.741,349.111 1316.763,348.209 1315.475,349.314 C 1315.032,349.693 1314.878,350.514 1314.289,350.712 C 1313.393,351.013 1312.845,349.567 1311.909,349.776 C 1313.075,348.011 1315.489,347.901 1313.469,345.322 C 1311.796,343.187 1309.959,343.312 1307.599,343.007 C 1309.166,342.328 1311.948,342.919 1313.209,343.944 C 1312.918,342.939 1309.705,340.183 1311.989,339.551 C 1310.514,339.173 1310.231,337.687 1309.249,336.743 C 1308.332,335.861 1306.921,335.889 1305.767,335.709 C 1303.835,335.407 1303.313,330.81 1303.194,329.319 C 1302.994,326.838 1299.884,324.688 1297.504,324.252 C 1296.876,324.138 1296.352,323.269 1295.673,323.879 C 1295.237,324.273 1294.828,324.462 1294.242,324.453 C 1292.687,324.429 1290.539,324.523 1289.38,323.279 C 1291.79,325.071 1293.43,322.558 1295.86,322.558 C 1297.242,322.558 1296.745,321.491 1296.019,320.918 C 1294.994,320.108 1293.95,321.395 1292.83,321.262 C 1295.673,320.709 1297.989,319.387 1299.818,317.134 C 1300.641,316.121 1300.886,314.924 1301.585,313.887 C 1302.07,313.166 1303.398,312.697 1303.025,311.631 C 1302.353,309.707 1298.549,310.442 1296.97,310.618 C 1295.332,310.801 1293.749,310.155 1292.152,310.633 C 1290.805,311.036 1289.275,311.873 1287.94,312.118 C 1288.143,311.434 1288.933,311.373 1289.16,310.678 C 1288.371,310.603 1287.664,310.979 1286.93,311.182 C 1287.797,310.288 1289.99,310.909 1290.39,309.527 C 1289.354,309.284 1288.11,309.521 1287.22,308.806 C 1288.93,309.46 1288.6,309.03 1289.719,308.28 C 1290.643,307.661 1291.753,307.4 1292.685,306.804 C 1293.069,306.559 1297.961,303.369 1295.414,302.856 C 1293.861,302.543 1291.926,303.291 1290.414,303.586 C 1289.571,303.751 1288.67,303.958 1287.806,303.888 C 1287.59,303.871 1287.302,303.831 1287.155,303.645 C 1286.919,303.346 1285.908,304.102 1285.561,304.271 C 1286.33,303.019 1285.652,303.883 1285.103,303.548 C 1284.883,303.414 1284.499,302.929 1284.191,303.001 C 1283.561,303.147 1283.8,304.98 1282.9,305.423 C 1283.175,305.573 1283.76,305.778 1283.76,306.071 C 1281.577,305.07 1282.241,307.449 1281.17,307.799 C 1281.718,308.027 1282.17,308.469 1282.39,309.024 C 1281.686,309.297 1279.181,308.362 1279.37,309.815 C 1279.152,308.48 1277.532,310.743 1279.3,311.832 C 1278.822,311.769 1278.342,311.744 1277.861,311.76 C 1277.747,313.154 1278.901,313.161 1279.951,313.056 C 1279.418,313.223 1278.938,313.487 1278.511,313.848 C 1279.152,313.61 1279.86,313.744 1280.241,314.353 C 1279.301,313.265 1278.037,315.359 1280.161,315.216 C 1279.936,315.3 1276.493,315.446 1278.871,316.296 C 1277.838,315.375 1276.683,317.099 1278.081,317.376 C 1277.503,317.292 1277.161,317.587 1277.141,318.168 C 1276.363,317.796 1275.354,317.891 1274.691,318.456 C 1275.831,318.722 1277.413,317.959 1278.431,318.672 C 1277.618,318.3 1277.086,319.26 1276.271,318.888 C 1276.945,321.854 1280.63,318.257 1281.821,317.592 C 1281.351,318.113 1281.072,318.938 1282.251,318.456 C 1281.672,318.573 1278.648,319.948 1280.601,319.752 C 1280.355,319.989 1280.112,320.23 1279.871,320.472 C 1280.385,320.499 1280.892,320.451 1281.391,320.328 C 1281.094,320.489 1279.733,320.545 1279.646,320.653 C 1279.401,320.959 1279.031,321.22 1278.704,321.429 C 1278.32,321.674 1278.588,322.372 1278.543,322.761 C 1278.494,323.182 1276.522,325.93 1278.221,323.928 C 1277.37,324.52 1277.523,326.759 1278.721,325.728 C 1277.4,326.472 1277.361,327.908 1276.498,328.998 C 1275.894,329.761 1277.785,330.326 1277.98,328.802 C 1278.119,327.711 1279.568,326.447 1279.202,325.407 C 1278.681,323.928 1280.898,323.023 1281.821,322.489 C 1281.21,323.023 1279.587,323.835 1279.757,324.861 C 1279.881,325.346 1280.162,325.467 1280.601,325.224 C 1280.838,325.687 1281.148,326.096 1281.531,326.449 C 1281.704,325.836 1281.343,325.225 1280.811,324.937 C 1281.375,324.171 1281.683,324.677 1281.821,325.296 C 1282.825,324.399 1282.134,322.681 1283.331,322.92 C 1282.829,323.339 1282.758,323.819 1283.121,324.36 C 1282.969,322.999 1283.799,324.439 1284.631,324.792 C 1284.066,324.763 1282.781,324.282 1282.54,325.109 C 1282.079,326.691 1283.138,326.663 1283.635,327.781 C 1284.482,329.686 1280.188,330.795 1281.032,333.072 C 1280.94,332.911 1279.989,332.04 1279.859,332.535 C 1279.611,333.484 1281.147,334.782 1281.682,335.376 C 1281.119,333.719 1281.782,333.391 1283.331,334.094 C 1284.609,334.675 1284.857,334.633 1284.912,333.073 C 1286.405,334.553 1287.344,333.96 1289.02,333.473 C 1289.54,333.322 1290.351,333.154 1290.242,332.425 C 1291.391,332.566 1292.55,332.484 1293.702,332.569 C 1292.405,333.429 1290.853,333.541 1290.102,335.09 C 1289.626,336.07 1290.516,339.348 1291.972,338.546 C 1291.763,339.087 1291.86,339.567 1292.262,339.985 C 1292.566,339.578 1292.829,339.147 1293.052,338.69 C 1293.264,339.576 1294.015,339.138 1294.492,338.834 C 1294.023,339.476 1294.029,340.295 1293.482,340.994 C 1293.165,341.4 1291.893,343.058 1293.632,342.794 C 1291.881,343.667 1292.43,346.66 1294.562,345.962 C 1293.592,346.672 1290.643,343.958 1292.122,346.898 C 1290.13,345.2 1287.293,346.954 1285.132,346.898 C 1285.93,345.917 1283.418,345.001 1282.779,345.54 C 1281.487,346.63 1283.591,348.156 1284.842,347.043 C 1283.668,348.36 1282.004,349.13 1280.882,350.499 C 1282.34,350.613 1283.673,348.963 1285.132,349.707 C 1284.767,350.308 1285.096,350.899 1285.642,351.219 C 1284.777,351.617 1285.006,352.723 1286.002,352.587 C 1285.185,352.703 1285.127,353.478 1284.816,354.075 C 1284.409,354.858 1283.378,355.326 1282.596,355.616 C 1280.859,356.26 1278.347,356.701 1277.212,358.131 C 1277.833,358.379 1278.055,358.895 1277.572,359.427 C 1278.346,359.591 1279.127,359.501 1279.662,358.851 C 1279.287,359.545 1278.528,359.766 1277.792,359.715 C 1278.768,360.955 1281.757,359.544 1282.762,358.851 C 1282.708,359.894 1283.912,359.931 1284.632,359.715 C 1284.328,360.158 1283.918,360.278 1283.402,360.075 C 1283.601,361.326 1284.529,360.716 1285.27,360.37 C 1286.41,359.839 1287.038,361.592 1288.034,362.027 C 1289.85,362.819 1290.148,361.177 1291.63,360.962 C 1293.249,360.727 1294.345,359.884 1295.642,358.922 C 1294.407,360.402 1292.668,361.425 1291.322,362.81 C 1290.97,363.171 1291.489,363.606 1290.743,363.702 C 1289.895,363.81 1289.039,363.862 1288.197,363.688 C 1287.008,363.442 1285.007,363.029 1283.865,363.714 C 1283.18,364.126 1283.137,364.943 1282.439,365.362 C 1281.477,365.938 1281.292,364.506 1280.948,366.213 C 1280.81,366.899 1279.119,369.495 1278.245,369.297 C 1277.705,369.175 1277.569,369.197 1277.365,369.797 C 1277.04,370.752 1275.688,371.186 1274.845,371.531 C 1274.172,371.806 1273.427,372.105 1272.854,372.567 C 1272.617,372.757 1272.915,373.416 1273.175,373.36 C 1274.079,373.163 1275.176,372.384 1275.627,373.68" id="path2656"/>
    <path d="M 1274.397,316.584 C 1274.541,316.512 1274.684,316.44 1274.827,316.368 C 1274.642,315.683 1274.163,315.565 1273.677,316.08 C 1273.917,316.248 1274.158,316.416 1274.397,316.584" id="path2658"/>
    <path d="M 1274.328,321.912 C 1275.594,321.824 1276.766,321.302 1277.998,321.048 C 1277.754,319.691 1274.888,318.362 1274.038,319.824 C 1274.616,319.869 1275.095,320.109 1275.478,320.544 C 1274.977,320.93 1274.286,321.487 1275.628,321.264 C 1275.113,321.318 1274.68,321.534 1274.328,321.912" id="path2660"/>
    <path d="M 1275.558,325.872 C 1276.173,324.931 1277.227,324.162 1277.428,322.992 C 1276.47,323.567 1275.94,324.688 1274.908,325.151 C 1275.095,325.418 1275.312,325.658 1275.558,325.872" id="path2662"/>
    <path d="M 1276.417,313.488 C 1276.637,313.006 1276.781,312.502 1276.847,311.975 C 1276.396,312.393 1276.253,312.897 1276.417,313.488" id="path2664"/>
    <path d="M 1280.308,328.824 C 1281.268,328.435 1280.965,326.377 1279.868,326.448 C 1278.587,326.532 1279.183,328.896 1280.308,328.824" id="path2666"/>
    <path d="M 1295.138,301.824 C 1295.282,301.243 1295.097,300.665 1294.418,300.599 C 1294.269,301.184 1294.606,301.626 1295.138,301.824" id="path2670"/>
    <path d="M 1295.138,300.24 C 1295.891,300.919 1297.885,301.261 1298.518,300.24 C 1297.737,300.186 1296.414,300.239 1296.132,299.274 C 1295.903,298.492 1293.934,299.25 1295.138,300.24" id="path2672"/>
    <path d="M 1296.868,302.184 C 1297.11,301.948 1297.229,301.66 1297.228,301.32 C 1296.459,301.266 1296.339,301.554 1296.868,302.184" id="path2674"/>
    <path d="M 1297.658,298.224 C 1297.764,297.728 1297.571,297.392 1297.078,297.216 C 1296.816,297.783 1297.051,298.148 1297.658,298.224" id="path2676"/>
    <path d="M 1298.808,298.584 C 1299.23,298.241 1299.687,297.953 1300.178,297.72 C 1299.543,297.629 1299.028,298.005 1298.808,298.584" id="path2678"/>
    <path d="M 1307.667,292.032 C 1307.547,292.296 1307.427,292.56 1307.307,292.824 C 1308.876,292.827 1308.96,290.137 1309.537,289.151 C 1308.937,289.102 1308.61,288.757 1308.747,288.145 C 1308.507,288.313 1308.267,288.481 1308.027,288.649 C 1307.971,288.14 1308.065,287.659 1308.307,287.208 C 1307.411,287.085 1307.571,288.226 1306.657,288 C 1307.054,288.487 1307.534,288.871 1308.097,289.151 C 1307.415,289.532 1306.646,289.704 1305.867,289.656 C 1306.115,290.925 1306.209,289.987 1307.177,290.315 C 1307.967,290.584 1308.37,291.396 1307.667,292.032" id="path2776"/>
    <path d="M 1309.318,288.216 C 1309.842,287.813 1310.617,286.607 1309.388,286.56 C 1309.456,287.095 1308.986,287.716 1309.318,288.216" id="path2778"/>
    <path d="M 1311.118,286.776 C 1312.169,285.023 1309.064,286.511 1311.118,286.776" id="path2780"/>
    <path d="M 1310.828,287.64 C 1311.071,287.544 1311.315,287.449 1311.558,287.353 C 1310.838,286.963 1310.594,287.059 1310.828,287.64" id="path2782"/>
    <path d="M 1302.268,368.856 C 1302.889,368.799 1303.434,368.427 1303.778,367.919 C 1302.602,367.264 1301.934,367.472 1300.828,368.135 C 1301.342,368.308 1301.821,368.548 1302.268,368.856" id="path5354"/>
  </g>
  <path class="landxx cr" d="M 617.88329,669.88555 C 618.31029,669.53855 619.46229,667.92555 619.54129,667.38955 C 619.66929,666.52255 617.72629,665.46455 618.46229,665.02955 C 619.33729,664.51355 619.56329,663.77255 620.15729,662.78255 C 620.39129,662.39155 623.26129,663.78655 623.67529,663.99155 C 625.68929,664.99055 625.42222,664.33894 627.29038,663.82128 C 628.11178,663.59367 628.47195,664.86948 629.83283,664.62736 C 630.44583,665.42636 630.65177,666.19016 631.71229,666.38955 C 633.04352,666.63984 634.97217,666.69027 634.91211,665.27536 C 635.50011,666.27436 634.92329,666.44555 635.49629,667.31755 C 635.97529,668.04855 636.26829,668.90455 636.58829,669.71255 C 637.26429,671.41955 637.93829,672.01655 639.07429,673.15055 C 640.10329,674.17655 640.63729,675.82055 641.95029,676.49855 C 642.67229,676.87055 642.56429,677.66255 641.66629,677.39255 C 640.98129,677.18655 640.8135,676.36134 639.9575,676.97234 C 640.3425,677.64534 639.5062,679.27324 639.47208,680.38174 C 639.43695,681.52287 642.19477,681.89902 640.31629,682.87655 C 638.79995,683.66562 640.83435,685.36321 639.51829,686.51955 C 638.0654,687.79612 639.42729,688.55055 639.12229,690.12055 C 639.21029,689.32755 638.86529,688.71555 638.35329,688.14655 C 638.02829,687.92555 637.73229,687.67155 637.46529,687.38355 C 637.55429,687.08455 637.63229,686.78255 637.69729,686.47755 C 637.41229,685.52355 636.31829,683.41955 635.01729,684.21555 C 635.26629,685.37555 636.76529,685.80455 636.31329,687.16755 C 635.24129,686.83255 633.56329,686.45555 633.30129,685.12455 C 633.13029,684.25855 634.29129,683.59155 634.37029,682.70355 C 634.56529,680.51055 632.17229,679.67655 630.75829,678.57455 C 629.89729,677.90355 629.68429,677.42955 628.49129,677.39855 C 627.31629,677.36855 626.62529,676.79455 626.67529,675.56655 C 626.79429,672.65355 623.32529,672.66055 622.20229,670.60755 C 622.42529,671.16455 622.47329,671.76055 622.67929,672.32055 C 623.00529,673.20755 623.79929,673.09855 624.41329,673.63055 C 625.26829,674.37155 624.25229,675.87755 623.45529,676.18955 C 622.48929,676.56755 621.66829,674.52955 620.62229,674.29655 C 618.86229,673.90055 617.52229,671.70755 617.88329,669.88555" id="cr"/>
  <path class="landxx py" d="M 803.78629,951.97855 C 804.48529,949.11755 804.88029,946.13255 805.22129,943.20955 C 805.40929,941.59055 805.17529,940.09055 805.06729,938.48355 C 804.96329,936.92855 805.80629,935.5795 806.44229,934.22055 C 807.33946,932.30355 806.84896,929.55455 809.74629,929.19255 C 812.78069,928.81342 815.85329,928.11455 818.77529,927.21055 C 823.52829,925.74055 826.87702,926.02028 831.17602,928.41628 C 832.88602,929.36928 835.28729,930.69255 835.42029,932.90955 C 835.50029,934.23955 836.52629,938.31155 837.91729,938.73955 C 837.02429,939.79755 838.46529,940.18155 838.70729,940.96855 C 838.89029,941.56655 838.70529,943.45355 838.60729,944.13455 C 838.44929,944.61255 838.35929,945.10255 838.33629,945.60555 C 838.42029,946.27155 838.89729,946.92955 838.67129,947.62455 C 838.40329,948.45255 837.93962,949.51938 838.03162,950.37838 C 838.08662,950.88838 843.39296,952.44172 844.16996,952.45672 C 845.31096,952.47772 846.67229,952.94772 847.78429,952.89272 C 848.86029,952.83772 848.76679,950.38288 849.84279,951.14988 C 850.89679,951.89988 852.63629,952.78288 853.93929,952.96288 C 854.91129,953.09688 855.02312,953.37322 855.20912,954.43822 C 855.39812,955.52425 856.17012,956.45205 856.32412,957.73405 C 856.52212,959.37805 857.97829,960.24155 858.16829,961.87655 C 858.38429,963.74155 858.83529,967.52455 861.38729,967.27655 C 862.80829,967.13855 863.77029,966.07255 865.12629,965.80955 C 866.38129,965.56555 867.33429,967.06855 868.22629,967.67955 C 867.67329,968.07955 867.62029,968.62255 867.91329,969.19755 C 868.48629,970.32255 867.55,970.93355 867.846,972.01155 C 868.157,973.14355 867.47802,975.15544 867.27202,976.26044 C 867.07402,977.32644 866.76348,979.57164 866.92892,980.68368 L 867.96698,987.6613 C 868.06013,988.28745 867.55229,989.90555 866.68829,990.06955 C 866.26429,990.14955 865.07965,991.84801 865.25665,992.25601 C 865.43565,992.66901 863.55856,992.97536 863.32756,993.09236 C 862.86956,993.32436 862.20759,993.82659 862.34929,994.27455 C 862.55427,994.92253 861.3976,997.53509 860.46129,996.72255 C 858.50405,995.02403 857.43829,996.45655 856.56529,998.27855 C 855.99629,998.06155 855.26215,997.07235 854.6362,997.20082 C 853.8082,997.37077 853.64829,998.11019 852.84929,997.08755 C 852.28936,996.3709 851.57629,997.39655 850.80629,997.41955 C 850.06829,997.44155 849.19975,997.16909 848.50375,996.98209 C 846.17075,996.35509 843.89802,995.48564 841.43002,995.73764 C 840.26102,995.85764 839.55302,996.67764 839.14602,995.26764 C 839.06802,994.99964 839.94675,992.47674 840.20975,992.30374 C 842.69975,990.66374 840.14829,986.44355 842.49629,985.28955 C 843.58429,984.75455 843.56289,982.2316 844.1592,981.13246 C 845.2262,979.16574 844.01156,978.21962 842.35956,976.97009 C 841.06885,975.99383 839.14429,976.01755 838.00529,975.05855 C 837.32929,974.48855 836.46829,974.28355 835.62229,974.12155 C 834.58329,973.92355 833.65348,973.23309 832.77348,972.72809 C 830.80348,971.59809 829.32229,970.47755 827.51829,969.03255 C 825.54229,967.45055 823.71229,967.81955 821.49629,966.93355 C 820.12229,966.38455 818.2258,966.17072 816.97938,964.87472 C 815.17442,962.99798 812.33881,961.27511 811.05529,960.12055 C 809.51326,958.73346 807.87014,956.94389 807.46566,955.79628 C 806.78451,953.86369 804.97029,953.59355 803.78629,951.97855" id="py"/>
  <g class="landxx coastxx gn" id="gn" transform="translate(-29.9017, -45.0745)">
    <path d="M 1186.638,709.848 C 1186.206,710.396 1186.247,711.298 1187.068,711.432 C 1186.848,710.849 1186.968,710.345 1187.428,709.92 C 1187.165,709.896 1186.901,709.872 1186.638,709.848" id="path5380"/>
    <path class="landxx gn" d="M 1186.778,709.704 C 1187.775,708.462 1187.676,706.427 1188.939,705.368 C 1189.795,704.651 1191.454,704.505 1192.477,703.971 C 1193.497,703.438 1194.553,704.084 1195.599,703.528 C 1196.287,703.162 1196.8688,702.50362 1196.77,701.687 C 1196.6227,700.46957 1194.4683,699.98693 1194.8198,699.04382 C 1195.1351,698.1978 1196.8497,698.44847 1197.0806,697.59396 C 1197.3396,696.6356 1196.6179,695.25931 1196.8791,694.96206 L 1200.4947,694.9524 C 1201.0161,695.0874 1201.6016,695.48534 1201.9798,695.51402 C 1202.2618,695.59951 1201.6252,697.23245 1202.7298,696.56159 C 1203.4878,696.10125 1205.0583,697.15871 1205.4756,697.4904 C 1205.5334,697.53634 1207.1488,698.0306 1207.1078,698.0346 C 1208.1608,697.9336 1208.9082,697.1309 1210.0512,697.25562 C 1211.7148,697.43715 1213.3671,696.71738 1214.569,697.249 C 1213.9883,698.41193 1213.1248,699.27114 1214.689,700.58581 C 1215.5773,701.33237 1216.9048,698.47989 1217.9402,698.8736 C 1220.4365,699.82284 1218.8232,702.81661 1221.0142,700.3234 C 1223.8215,697.12881 1225.5352,700.78813 1226.599,700.51899 C 1227.2343,700.35827 1227.6502,699.75845 1228.228,699.276 C 1228.6998,698.88206 1229.8531,698.77044 1230.0769,698.395 C 1230.5096,697.66901 1228.7449,695.95267 1231.19,696.582 C 1231.7133,696.71669 1232.974,697.31424 1232.875,698.3 C 1232.8422,698.62652 1233.2901,700.29999 1233.521,700.518 C 1234.6843,701.6166 1233.298,704.50263 1234.73,703.729 C 1234.9146,705.16546 1237.5802,705.23254 1237.299,706.94399 C 1236.6631,706.97419 1236.4072,706.93321 1235.999,707.427 C 1235.435,708.10928 1235.102,709.009 1234.871,709.849 C 1235.616,709.72699 1236.0489,709.11224 1236.904,709.02 C 1238.2922,708.87026 1237.8763,710.64834 1237.8442,711.63627 C 1237.8206,712.36325 1237.4978,713.55618 1238.839,714.30164 C 1239.0917,714.44211 1240.2122,715.17671 1240.3863,715.83427 C 1240.6205,716.71924 1239.7726,717.14555 1239.09,717.65191 C 1238.6488,717.97917 1238.6443,719.47168 1238.6505,720.15018 C 1238.6622,721.4085 1239.1019,724.2147 1240.852,723.097 C 1240.456,725.078 1240.668,724.79 1241.792,726.121 C 1240.44,726.324 1240.435,727.445 1240.062,728.497 C 1242.109,728.834 1241.916,730.593 1242.432,732.169 C 1241.3763,731.80337 1238.2866,730.59516 1237.9021,731.62227 C 1237.3977,732.96965 1239.421,734.311 1240.062,735.265 C 1239.2954,734.87146 1238.7552,736.336 1238.7661,736.38819 C 1238.9323,737.18543 1239.207,736.92515 1239.0522,737.41667 C 1238.8948,737.91605 1238.0205,739.19087 1238.0327,739.29662 C 1237.8212,739.11274 1236.7267,738.57952 1236.1259,739.10886 C 1235.8046,739.39193 1236.2077,738.54532 1235.2418,738.06428 C 1234.3033,737.73271 1233.9961,739.73701 1233.7662,740.19573 C 1233.0996,741.52574 1232.606,741.803 1231.132,742.393 C 1231.05,740.714 1229.678,740.447 1228.322,740.954 C 1228.697,739.855 1228.9327,738.72606 1228.9719,737.559 C 1228.9868,737.11405 1228.8382,734.68129 1228.7024,734.90819 C 1228.6109,735.0753 1228.4303,735.51798 1228.4109,734.06527 C 1228.4079,733.84248 1227.8881,733.04061 1228.3696,732.40951 C 1226.0567,729.7583 1225.8893,731.18498 1225.5227,730.99673 C 1224.7435,731.62108 1224.4976,731.22191 1223.4254,731.11218 C 1222.6865,731.03656 1221.889,731.223 1221.272,731.593 C 1220.892,731.822 1219.277,732.671 1219.112,733.033 C 1218.9551,731.58039 1220.9416,729.70426 1220.619,729.267 C 1219.3525,727.55016 1221.2914,727.02999 1219.3169,725.76518 C 1218.7223,725.38431 1219.5244,724.73275 1219.2315,724.19436 C 1217.1268,720.32675 1216.3273,717.32647 1211.5912,718.03991 C 1210.3157,718.23205 1210.0046,718.63056 1208.881,718.88682 C 1206.9054,719.33737 1206.2879,718.32602 1205.442,720.26 C 1204.7646,721.80885 1204.2106,723.60418 1203.234,724.04109 C 1202.3529,724.43527 1202.2917,724.75654 1201.8846,725.62946 C 1201.4009,726.66682 1200.163,725.829 1199.452,726.481 C 1199.369,725.81 1199.801,725.322 1200.462,725.257 C 1200.316,724.979 1200.123,724.739 1199.882,724.537 C 1199.704,724.769 1199.537,725.009 1199.382,725.257 C 1199.32,724.778 1199.297,724.299 1199.312,723.817 C 1199.049,723.841 1198.785,723.865 1198.522,723.889 C 1198.748,722.998 1197.449,722.829 1198.012,721.946 C 1196.69,723.279 1197.336,719.829 1196.862,719.498 C 1196.932,720.076 1196.609,720.287 1196.072,720.145 C 1196.302,719.637 1196.469,719.109 1196.572,718.561 C 1196.262,719.621 1195.564,719.253 1194.912,718.778 C 1193.985,718.102 1194.468,717.623 1194.122,716.69 C 1193.629,716.867 1193.296,717.203 1193.122,717.698 C 1193.012,715.884 1189.176,716.038 1190.242,713.738 C 1189.915,713.856 1189.578,713.928 1189.232,713.954 C 1189.801,712.897 1190.19,711.719 1190.522,710.569 C 1190.086,711.006 1189.752,711.509 1189.522,712.082 C 1189.403,710.911 1188.938,711.512 1188.582,712.082 C 1188.413,711.292 1188.301,709.619 1189.302,709.274 C 1188.498,709.718 1187.659,709.696 1186.778,709.704" id="path5400"/>
  </g>
  <g class="landxx coastxx ie eu" id="ie" transform="translate(-29.9017, -45.0745)">
    <path d="M 1258.488,332.353 C 1258.625,332.463 1258.413,332.829 1258.668,332.497 C 1258.925,332.161 1258.628,332.465 1258.488,332.353" id="path5352"/>
    <path d="M 1248.047,341.353 C 1249.205,340.855 1247.088,340.483 1246.537,340.777 C 1247.117,340.768 1247.621,340.96 1248.047,341.353" id="path5348"/>
    <path class="landxx ie eu" d="M 1245.678,360.36 C 1245.438,360.383 1245.198,360.408 1244.958,360.431 C 1245.911,359.572 1246.921,358.511 1248.268,358.343 C 1247.005,358.131 1244.925,359.911 1243.779,358.936 C 1242.491,357.84 1246.697,355.757 1247.548,356.039 C 1246.286,355.477 1244.569,355.959 1243.298,356.255 C 1242.955,355.239 1244.034,355.082 1244.769,354.985 C 1245.156,354.934 1245.6,354.907 1245.952,354.718 C 1246.283,354.54 1246.99,355.178 1247.759,355.031 C 1247.585,354.48 1247.321,353.976 1246.969,353.518 C 1248.343,353.524 1248.746,352.267 1250.175,352.287 C 1251.837,352.31 1253.326,351.653 1254.959,351.503 C 1254.406,351.486 1253.934,351.159 1253.739,350.639 C 1251.649,351.98 1250.047,351.931 1247.619,352.368 C 1248.28,351.904 1251.565,350.28 1251.161,349.188 C 1250.584,347.63 1253.782,347.669 1254.458,346.967 C 1253.207,346.888 1252.026,346.844 1250.775,346.854 C 1250.165,346.859 1249.123,346.2 1248.338,346.103 C 1248.48,345.862 1248.623,345.623 1248.768,345.383 C 1248.044,345.164 1247.466,345.863 1246.758,345.455 C 1247.002,345.292 1247.241,345.124 1247.478,344.951 C 1247.24,344.781 1246.999,344.613 1246.758,344.447 C 1247.609,344.309 1247.283,344.208 1247.882,343.826 C 1248.365,343.517 1249.114,343.859 1249.638,343.799 C 1248.888,343.894 1248.052,343.236 1248.871,342.58 C 1249.207,342.31 1249.926,342.363 1250.329,342.233 C 1252.349,341.58 1248.396,341.322 1248.573,341.583 C 1248.298,341.178 1249.187,341.157 1249.347,341.28 C 1248.893,340.54 1249.654,339.313 1248.407,339.983 C 1248.747,339.552 1248.724,339.144 1248.337,338.759 C 1248.126,338.984 1247.932,339.224 1247.757,339.479 C 1247.871,338.953 1248.088,338.473 1248.407,338.04 C 1248.518,339.174 1250.102,337.987 1251.243,337.915 C 1252.31,337.848 1253.684,338.118 1253.806,339.408 C 1254.347,337.055 1256.82,339.458 1258.056,338.256 C 1257.789,338.119 1257.526,337.975 1257.266,337.824 C 1258.055,336.787 1260.241,336.892 1260.426,335.664 C 1259.211,335.427 1257.886,335.809 1256.756,335.16 C 1257.017,334.306 1257.943,334.083 1258.706,334.368 C 1258.416,333.744 1258.4,333.318 1259.346,333.649 C 1258.772,332.89 1259.368,331.653 1260.112,331.254 C 1261.438,330.542 1262.622,331.145 1263.886,330.983 C 1263.306,329.979 1264.265,329.896 1264.59,330.469 C 1265.168,331.489 1264.583,331.485 1264.317,332.352 C 1265.406,332.369 1265.844,331.197 1265.037,330.479 C 1266.605,329.389 1267.186,329.982 1268.717,330.263 C 1267.893,331.189 1266.0617,331.39194 1265.652,332.675 C 1265.2914,333.80437 1263.9426,335.36258 1262.447,334.727 C 1262.742,335.012 1263.056,335.277 1263.387,335.52 C 1262.066,336.349 1259.8282,336.15209 1261.5624,337.74381 C 1262.32,338.43918 1263.42,338.85479 1264.3114,339.13862 C 1265.1566,339.40775 1265.8247,339.73298 1266.2188,338.38281 C 1266.4273,337.66871 1266.1379,336.98776 1267.682,337.379 C 1268.7164,337.64109 1269.2099,338.88268 1269.787,339.046 C 1269.7792,340.4483 1270.7384,340.05283 1271.447,339.982 C 1272.0803,339.9187 1272.061,339.424 1272.957,340.271 C 1272.458,340.526 1271.826,340.549 1271.393,340.94 C 1271.251,341.069 1272.038,342.075 1271.968,342.63 C 1271.871,343.41 1273.854,345.433 1271.877,345.814 C 1273.196,347.842 1272.626,349.051 1271.812,351.081 C 1271.39,352.134 1270.793,354.267 1269.287,354.094 C 1270.929,354.317 1269.853,355.717 1268.653,355.497 C 1267.697,355.321 1266.229,356.084 1265.977,354.67 C 1266.597,356.518 1263.188,355.827 1262.239,356.163 C 1261.355,356.476 1261.936,357.32 1260.561,357.32 C 1259.486,357.32 1258.947,358.797 1257.557,358.702 C 1258.385,357.482 1256.881,357.819 1256.257,358.054 C 1257.368,358.371 1256.985,358.901 1256.257,359.323 C 1255.933,359.453 1255.668,359.658 1255.459,359.939 C 1255.207,360.431 1254.789,360.002 1254.34,360.186 C 1253.281,360.619 1252.148,360.601 1251.047,360.957 C 1249.819,361.354 1248.406,360.966 1247.257,361.51 C 1247.568,360.988 1248.001,360.604 1248.557,360.359 C 1248.031,360.435 1247.524,360.579 1247.037,360.791 C 1247.667,360.223 1248.464,359.934 1249.267,359.711 C 1248.326,359.213 1246.475,359.767 1245.678,360.36" id="path5402"/>
  </g>
  <g class="landxx coastxx ng" id="ng" transform="translate(-29.9017, -45.0745)">
    <path d="M 1355.258,766.728 C 1356.172,766.667 1356.499,766.104 1355.758,765.504 C 1355.445,765.853 1355.279,766.26 1355.258,766.728" id="path5382"/>
    <path d="M 1357.198,766.296 C 1357.438,766.152 1357.678,766.008 1357.918,765.864 C 1357.41,765.871 1356.93,765.991 1356.478,766.223 C 1356.718,766.248 1356.958,766.271 1357.198,766.296" id="path5386"/>
    <path class="landxx ng" d="M 1321.058,736.344 C 1321.957,733.224 1320.661,729.511 1322.034,726.563 C 1322.369,725.842 1323.637,726.48 1324.239,725.787 C 1325.268,724.6 1324.332,723.162 1325.388,721.974 C 1326.079,721.196 1326.04,719.965 1326.883,719.474 C 1327.767,718.958 1329.087,717.858 1328.655,716.602 C 1328.32,715.627 1327.875,714.819 1329.176,714.552 C 1331.002,714.178 1329.528,711.567 1329.339,710.784 C 1329.145,709.986 1329.422,709.284 1329.004,708.512 C 1328.645,707.849 1328.088,707.322 1327.743,706.65 C 1327.036,705.272 1328.063,704.106 1328.471,702.825 C 1328.963,701.282 1328.617,699.637 1328.728,698.036 C 1328.862,696.102 1329.381,695.703 1330.735,694.372 C 1331.9,693.226 1332.405,691.497 1332.364,689.903 C 1332.33,688.558 1333.259,687.757 1334.223,686.955 C 1335.359,686.011 1336.755,685.751 1338.191,685.694 C 1340.07,685.619 1341.082,684.941 1342.805,684.505 C 1343.814,684.249 1345.13,685.407 1345.964,685.825 C 1347.4,686.546 1348.895,686.125 1350.142,687.332 C 1351.242,688.398 1351.765,689.67 1352.512,690.959 C 1352.795,691.448 1353.179,691.967 1353.744,692.151 C 1354.931,692.538 1355.099,691.205 1356.046,691.101 C 1359.203,690.752 1360.239,687.922 1363.539,690.265 C 1363.995,690.588 1364.502,690.969 1364.799,691.453 C 1364.943,691.688 1365.808,691.57 1366.059,691.705 C 1367.435,692.442 1367.983,693.469 1369.763,693.536 C 1370.804,693.576 1372.974,694.146 1373.878,693.744 C 1375.157,693.175 1375.595,691.517 1376.691,690.69 C 1379.329,688.7 1383.316,688.529 1386.47,688.828 C 1388.124,688.985 1389.603,689.429 1391.069,690.202 C 1392.297,690.85 1393.843,691.804 1395.286,691.274 C 1396.672,690.765 1397.39,689.18 1398.53,688.321 C 1399.155,687.85 1399.764,687.41 1400.572,687.366 C 1401.803,687.3 1400.9492,687.04615 1401.63,686.592 C 1404.2521,684.84297 1405.0904,687.32338 1406.046,689.10237 C 1406.8582,690.61432 1408.4892,690.73688 1408.0997,694.9279 C 1408.0567,695.3913 1408.7761,697.23827 1409.1861,697.56827 C 1409.6921,697.97527 1410.0755,697.59823 1410.7,697.824 C 1413.945,698.99706 1410.9927,702.19777 1411.976,703.721 C 1413.2297,705.66308 1409.2084,707.15551 1407.7515,707.00309 C 1405.5556,706.77334 1404.7704,711.561 1404.1984,713.293 C 1403.8734,714.277 1403.6882,715.317 1403.3902,716.299 C 1403.0572,717.396 1401.8254,717.22854 1401.7974,718.46754 C 1401.7724,719.57654 1402.043,720.98591 1401.189,721.96991 C 1400.467,722.80191 1399.43,723.175 1398.459,723.671 C 1399.539,723.883 1398.466,726.647 1398.389,727.272 C 1398.166,729.063 1397.01,730.085 1395.219,730.152 C 1395.386,731.089 1394.664,731.296 1394.165,731.891 C 1393.542,732.633 1394.0506,733.732 1393.8776,734.616 C 1393.4766,736.667 1392.4554,738.82429 1391.168,740.514 C 1390.0602,741.96792 1392.1075,742.85885 1389.551,744.525 C 1388.4733,745.22741 1389.118,745.991 1388.739,747 C 1388.489,747.665 1387.6159,748.29728 1387.2763,748.92191 C 1386.9473,749.52691 1385.8593,749.16891 1385.4013,748.98591 C 1384.8391,748.7613 1384.968,747.095 1384.93,746.596 C 1384.887,746.018 1382.9068,745.49382 1382.6348,744.82782 C 1382.2018,743.76782 1381.676,744.124 1381.388,742.895 C 1381.168,743.652 1381.113,744.992 1380.148,745.203 C 1378.931,745.469 1378.4099,744.16136 1377.428,744.479 C 1375.6591,745.05122 1375.283,747.81097 1374.1335,748.5669 C 1372.6403,749.54879 1367.8683,754.42655 1368.5039,755.07372 C 1369.522,756.11024 1367.6911,758.77531 1367.686,759.522 C 1367.6774,760.79303 1367.1981,761.84111 1365.6556,764.30237 C 1365.0426,764.14537 1364.474,763.274 1364.757,762.696 C 1364.101,763.121 1363.596,762.472 1363.317,761.975 C 1363.953,764.342 1364.546,764.878 1361.798,765.277 C 1360.711,765.435 1357.134,766.15 1358.136,763.919 C 1357.668,764.825 1357.646,765.067 1356.556,764.999 C 1355.349,764.924 1355.557,764.612 1354.386,763.847 C 1354.935,764.248 1355.598,766.316 1354.293,765.794 C 1353.419,765.445 1354.713,763.346 1353.166,764.135 C 1353.598,765.003 1353.97,765.827 1353.596,766.799 C 1352.88,766.496 1352.056,763.915 1352.306,763.127 C 1351.482,763.84 1352.849,765.921 1352.421,766.895 C 1351.825,768.253 1351.83,765.066 1351.866,764.783 C 1351.73,765.18 1351.514,767.362 1351.436,765.43 C 1351.417,766 1351.338,766.641 1350.936,767.086 C 1350.915,766.569 1350.772,766.089 1350.506,765.646 C 1350.814,766.265 1350.789,767.262 1349.841,767.177 C 1348.131,767.024 1348.601,767.416 1348.126,765.79 C 1348.144,767.327 1346.947,768.662 1346.686,766.438 C 1346.704,767.881 1346.328,766.935 1345.277,766.421 C 1344.24,765.914 1343.542,764.903 1343.016,763.918 C 1342.521,762.993 1341.902,761.984 1342.164,760.907 C 1342.339,760.187 1342.434,759.527 1341.426,759.813 C 1341.24,759.031 1341.323,758.592 1342.123,758.431 C 1343.674,758.119 1342.224,757.561 1343.375,756.862 C 1342.554,756.926 1341.828,758.041 1341.02,757.858 C 1338.442,757.274 1341.969,756.373 1342.585,756.286 C 1342.168,755.781 1341.688,755.709 1341.145,756.07 C 1341.167,755.757 1341.19,755.445 1341.215,755.133 C 1341.058,756.032 1339.922,756.681 1339.485,755.565 C 1339.089,754.555 1340.303,754.109 1340.855,753.549 C 1340.201,753.815 1339.612,754.27 1339.125,754.774 C 1338.138,752.527 1335.954,750.041 1333.499,749.416 C 1332.306,749.112 1330.877,748.956 1329.646,749.031 C 1328.685,749.091 1327.283,749.508 1326.456,748.798 C 1327.502,749.003 1329.039,748.307 1329.626,747.43 C 1328.556,747.38 1327.797,748.64 1326.746,747.79 C 1326.371,748.242 1326.274,748.746 1326.456,749.302 C 1324.732,749.528 1322.974,749.198 1321.266,749.59 C 1321.209,748.51 1321.817,747.559 1321.846,746.494 C 1321.876,745.392 1321.383,744.583 1321.657,743.444 C 1322.106,741.589 1322.146,738.024 1321.058,736.344" id="path5406"/>
  </g>
  <g class="landxx coastxx tn" id="tn">
    <path class="landxx tn" d="M 1328.9563,464.90155 C 1329.1283,464.05355 1329.9223,463.73055 1330.5383,463.26155 C 1331.4973,462.53155 1331.5772,461.87778 1332.6383,461.07255 C 1333.4855,460.42964 1334.1237,459.22675 1334.4512,458.16049 C 1335.1087,456.02041 1334.9252,454.35467 1335.0312,452.30155 C 1335.098,451.0087 1334.2833,449.88455 1334.4563,448.58755 C 1334.6523,447.11455 1336.2873,444.31455 1333.9263,443.80555 C 1334.7753,442.82655 1336.1573,442.32355 1336.8123,441.16255 C 1337.0543,440.73355 1336.8993,439.99855 1337.4883,439.82955 C 1338.1353,439.64455 1338.5403,439.82755 1339.0543,439.28355 C 1340.1723,438.09955 1341.3373,437.57455 1342.9073,437.12555 C 1343.6363,436.91755 1344.6373,436.36955 1345.4133,436.67955 C 1346.3343,437.04855 1345.4153,437.49455 1345.5163,438.18855 C 1346.2833,438.18855 1346.4653,437.80655 1346.0163,437.18055 C 1347.4233,437.48655 1347.8263,437.27255 1348.3963,438.54855 C 1348.6473,439.10955 1347.8663,439.16355 1348.6243,439.69755 C 1349.2043,440.10655 1349.2983,440.81755 1348.7573,441.28455 C 1348.8013,441.23855 1349.4043,441.94755 1350.0193,441.69455 C 1350.8553,441.34955 1351.5613,440.66755 1352.2433,440.09255 C 1352.5093,439.86855 1353.4313,438.87655 1353.8623,439.07255 C 1354.8933,439.54255 1354.7563,440.55855 1354.2893,441.40455 C 1353.3933,443.02755 1350.3293,444.58455 1350.2263,446.43055 C 1350.1503,447.81655 1350.9693,449.48655 1352.2213,450.16255 C 1354.1163,451.18355 1354.0533,451.23155 1354.5043,453.40555 C 1355.0723,456.14355 1353.1593,458.98955 1351.0563,460.65055 C 1348.9653,462.30155 1345.7673,463.23155 1347.8403,466.43755 C 1348.2953,467.14255 1348.9033,467.80655 1349.6813,468.16055 C 1350.5473,468.55455 1350.8993,468.01255 1351.6963,468.12355 C 1351.9583,468.15955 1352.1243,469.44255 1352.2843,469.71355 C 1352.6753,470.37455 1353.7703,468.42555 1354.3673,468.64255 C 1355.1833,468.93955 1354.6323,469.60255 1354.8773,470.15455 C 1355.2183,470.92155 1356.3933,471.61655 1355.1673,471.73855 C 1355.8123,472.42755 1357.2183,471.66955 1357.8203,472.46155 C 1358.3233,473.12455 1357.4923,475.70155 1357.6733,476.66955 C 1357.9443,478.12055 1359.0043,478.73941 1357.4137,479.69149 C 1355.8405,480.63317 1352.6212,481.67757 1351.6853,482.97355 C 1351.1503,483.39955 1351.3375,484.62678 1350.5902,484.92986 C 1349.0012,485.57433 1347.4916,486.09878 1348.1203,488.52055 C 1348.6229,490.45657 1349.3683,491.97855 1348.2473,493.87755 C 1346.9403,496.09255 1345.7913,496.93955 1343.4963,497.94655 C 1342.9473,494.33255 1341.9123,490.85248 1341.2013,487.27655 C 1340.4653,483.57488 1340.0922,481.44005 1336.9692,479.70455 C 1335.5099,478.89361 1334.855,477.91342 1334.488,476.29342 C 1334.166,474.86842 1333.521,473.85874 1332.226,473.09774 C 1331.579,472.71674 1330.9553,472.40755 1330.5253,471.72755 C 1330.0853,471.03055 1330.2365,470.08637 1329.796,469.38149 C 1328.881,467.91708 1328.3773,466.36413 1328.9563,464.90155" id="path5410"/>
    <path d="M 1353.2163,468.50155 C 1353.0983,467.54355 1352.4363,468.32155 1352.2163,467.85355 C 1351.6783,466.70855 1352.2073,466.34155 1353.1723,466.35455 C 1355.2883,466.38255 1353.9653,467.64555 1353.2163,468.50155" id="path5412"/>
  </g>
  <path class="landxx pl eu" d="M 1373.8863,304.84555 C 1373.8383,303.12855 1376.3973,302.67955 1375.4553,300.67855 C 1375.1293,299.98555 1374.3693,298.95255 1374.6543,298.11455 C 1374.7863,297.72655 1376.1273,298.22255 1376.4063,298.22155 C 1376.2623,297.65555 1376.3823,297.15155 1376.7663,296.70955 C 1376.6913,296.82855 1376.6183,296.94855 1376.5463,297.06955 C 1376.1113,296.69455 1375.6083,296.47855 1375.0363,296.42155 C 1375.3403,297.18555 1374.1553,296.75255 1374.0963,296.06155 C 1375.7533,296.68455 1378.4093,295.24855 1379.9143,294.64355 C 1381.7203,293.91655 1386.1153,294.11455 1387.2493,292.60955 C 1388.6353,290.77155 1391.8993,289.92455 1394.0843,289.34755 C 1396.8133,288.62555 1400.5743,288.04555 1402.8953,290.01355 C 1402.2703,289.58255 1401.5273,289.14155 1400.7353,289.43755 C 1401.6853,290.33055 1401.9283,292.04655 1403.3103,292.44655 C 1404.9743,292.92755 1407.2073,292.20555 1408.8053,291.74155 C 1408.1033,292.16555 1407.2433,292.27655 1406.5553,292.74255 C 1405.5683,293.41255 1407.6692,293.02873 1408.0473,292.83555 C 1410.621,291.52073 1412.8754,292.5777 1415.9403,292.53455 C 1419.9779,292.47771 1425.6702,293.47605 1429.6792,292.88505 C 1432.6732,292.44505 1433.3903,293.84755 1434.1373,296.59055 C 1434.9063,299.41055 1438.3723,302.72455 1437.3773,305.82455 C 1436.8743,307.39355 1434.897,306.83101 1434.064,308.03401 C 1433.231,309.23801 1432.5029,310.14969 1434.5339,310.52569 C 1437.7889,311.12769 1435.0787,314.31121 1436.0436,316.06624 C 1437.0589,317.91287 1437.4196,320.12723 1440.1963,321.47755 C 1439.7963,321.68155 1439.3883,321.87355 1438.9763,322.05355 C 1440.2053,323.14155 1440.5313,324.37655 1438.7743,325.02655 C 1437.0063,325.67955 1435.4267,327.39037 1434.2173,328.77414 C 1433.1728,329.96914 1430.9431,331.52455 1431.6363,333.37155 C 1431.9266,334.14508 1431.8343,334.96555 1432.1263,335.73255 C 1432.3583,336.34155 1433.0423,336.13255 1432.9263,336.95755 C 1431.7303,336.11255 1429.9283,336.12755 1428.5653,335.64455 C 1427.2253,335.16855 1426.5733,334.14455 1425.1983,333.77655 C 1423.8463,333.41355 1422.3373,333.20155 1421.0103,333.75455 C 1419.2943,334.46955 1419.0933,333.81855 1417.4363,333.61455 C 1416.1603,333.45755 1415.6893,333.70955 1414.7543,334.49955 C 1414.3493,334.84155 1413.8303,335.27855 1413.2713,335.32555 C 1412.3133,335.40555 1412.6303,334.39455 1412.2563,333.93355 C 1411.8553,333.43755 1410.9383,333.02655 1410.4283,332.41455 C 1409.6063,331.42855 1407.6113,333.59255 1406.7163,333.71755 C 1406.9483,332.94355 1403.6413,329.46855 1402.5313,329.16455 C 1401.2853,328.82455 1400.6773,328.31155 1399.4653,328.69655 C 1398.1973,329.09755 1398.0743,326.97855 1398.2163,326.22855 C 1396.2003,326.83555 1394.4663,325.41955 1392.6063,324.93255 C 1392.6593,325.54555 1392.9223,326.04955 1393.3963,326.44455 C 1392.6153,326.96655 1391.2013,328.39355 1390.3213,327.17355 C 1389.6023,326.17655 1389.1753,325.51055 1387.9263,325.07655 C 1388.2563,324.79055 1389.4953,324.12855 1389.2263,323.53955 C 1388.8733,322.76855 1387.1363,323.09655 1386.6963,323.56355 C 1386.6313,321.59155 1383.0081,323.00601 1382.2473,321.86555 C 1381.5235,320.78055 1382.2386,320.96183 1380.6172,320.30249 C 1379.3647,319.79316 1380.39,321.7157 1378.7763,321.40355 C 1379.9002,318.60991 1380.6353,317.71355 1378.4933,316.25755 C 1377.5713,315.63055 1378.3013,314.92555 1377.5753,314.28155 C 1376.6303,313.44255 1377.6753,313.05055 1377.8333,312.15555 C 1377.9303,311.60555 1377.7183,310.61355 1377.6463,310.05455 C 1377.5743,309.49155 1376.4193,309.63255 1376.6403,308.69255 C 1376.7853,308.07655 1377.5543,307.46555 1376.7203,306.96355 C 1375.7093,306.35955 1374.7433,305.65455 1373.8863,304.84555" id="pl"/>
  <path class="landxx na" d="M 1360.1363,908.84853 C 1361.1943,908.68353 1362.0303,907.75153 1363.1613,907.95253 C 1364.1243,908.12353 1364.9953,908.85453 1366.0153,908.65953 C 1368.1403,908.25353 1369.5963,905.77553 1372.0163,906.43553 C 1373.9433,906.96153 1374.9684,909.12811 1376.9027,909.77035 C 1378.0001,910.13473 1381.3796,909.99317 1383.1753,910.00853 C 1391.3905,910.07882 1399.6176,910.24291 1407.8223,909.91253 C 1409.9936,909.8251 1410.7293,909.66644 1411.7542,911.58253 C 1413.0588,914.02156 1415.0112,913.48291 1417.3553,913.74053 C 1420.4283,914.07826 1423.6993,913.93781 1426.5777,914.94762 C 1427.6842,915.33584 1428.6798,915.47634 1429.6576,914.88362 C 1430.5218,914.35974 1431.3459,914.54768 1432.5945,914.8529 C 1433.8568,915.16146 1434.6049,914.95216 1435.4448,914.75726 C 1440.5717,913.56756 1446.3682,912.36451 1451.8023,911.33653 C 1452.8553,911.13734 1453.9233,910.65453 1455.0163,910.82153 C 1455.6623,910.92053 1456.6403,911.54253 1457.2813,911.22253 C 1459.3243,910.20253 1460.2686,912.36081 1461.6166,913.28681 C 1460.6326,913.03081 1459.4903,913.60453 1458.5573,913.96753 C 1457.3973,914.41853 1457.0563,916.14953 1455.9443,915.12653 C 1454.1543,913.48053 1450.6563,918.87353 1449.3383,919.50753 C 1448.6963,918.27153 1448.0043,915.51953 1446.1743,915.34653 C 1443.8173,915.12353 1440.8813,916.31353 1438.5843,916.80053 C 1435.8813,917.37353 1432.686,917.48362 1429.994,917.98562 L 1429.0001,949.88516 L 1422.2006,949.78653 L 1420.7239,1004.1675 C 1420.8928,1006.6524 1418.7693,1005.6063 1417.5855,1006.7327 C 1417.001,1007.2888 1417.0435,1007.8306 1415.8767,1008.2329 C 1415.0797,1008.5077 1414.7093,1008.8425 1415.3593,1009.5785 C 1413.5643,1010.1095 1413.6502,1009.4356 1411.9809,1009.1891 C 1410.2268,1008.93 1408.729,1009.8778 1406.6533,1009.4315 C 1403.3679,1008.7251 1401.4194,1007.5138 1401.2686,1006.9054 C 1400.8532,1005.2296 1401.2378,1005.9224 1401.3057,1004.5901 C 1401.2437,1004.0637 1400.6697,1004.2012 1400.2701,1003.7859 C 1399.6998,1003.1932 1399.2028,1001.9288 1398.5993,1002.3115 C 1396.6977,1003.5172 1397.2231,1005.3075 1396.5593,1005.8265 C 1395.6159,1006.5642 1394.4323,1007.5309 1393.4473,1006.2915 C 1392.1459,1004.654 1390.4063,1003.3465 1389.2453,1001.5985 C 1388.2133,1000.0435 1387.3903,998.36653 1386.5583,996.69953 C 1386.1163,995.81453 1386.0753,994.86253 1385.9213,993.90753 C 1385.7623,992.92153 1385.0913,992.40753 1384.8983,991.50953 C 1384.5433,989.86253 1385.2783,989.25753 1384.0593,987.73953 C 1383.5723,987.13353 1383.7353,986.74253 1383.7523,985.96853 C 1383.7813,984.71753 1383.3473,983.63453 1383.2123,982.41453 C 1383.1163,981.54553 1383.1913,980.66553 1382.9423,979.81553 C 1382.5913,978.61353 1383.1923,977.45053 1382.9583,976.24853 C 1382.5553,974.17653 1381.5703,972.28553 1380.8183,970.33053 C 1379.9693,968.12353 1380.6043,966.26053 1380.3823,964.02653 C 1380.2323,962.51053 1380.0114,960.61259 1380.1483,959.82953 C 1380.2124,959.46259 1381.1323,955.78053 1380.7293,955.86953 C 1380.6333,953.88653 1379.6323,951.94453 1378.5073,950.35353 C 1377.8743,949.45853 1377.1513,948.62453 1376.6193,947.66053 C 1376.1313,946.77453 1376.0303,945.69053 1375.3913,944.88353 C 1372.4463,941.16453 1371.8143,936.64953 1369.6393,932.61953 C 1367.6443,928.92553 1366.9743,923.85153 1363.7323,920.94553 C 1360.2433,917.82053 1359.8353,913.25753 1360.1363,908.84853" id="na"/>
  <path class="landxx za" d="M 1471.8763,1016.0586 c 0.955,0.943 0.87,2.302 1.643,3.276 0.667,0.842 0.439,1.957 1.141,2.651 0.88,0.872 3.176,3.81 4.698,2.576 0.666,-0.541 1.229,-2.344 1.697,-3.126 0.739,-1.237 1.994,-0.988 3.196,-1.324 0.875,-0.245 2.747,-0.978 3.12,-1.901 0.475,-1.178 0.528,-1.852 1.365,-2.892 0.316,-0.393 1.323,-1.458 1.154,-2.036 -0.526,-1.819 -1.285,-1.935 -2.491,-3.196 -0.943,-0.987 -1.948,-1.908 -2.88,-2.901 -0.807,-0.858 -2.188,0.498 -2.924,0.814 -0.813,0.349 -1.202,0.728 -1.999,1.167 -0.934,0.514 -1.905,0.329 -2.649,1.286 -0.802,1.031 -1.201,2.307 -1.917,3.393 -0.822,1.244 -2.02,1.389 -3.154,2.213 m -77.111,-9.21 c 0.75,-0.707 1.891,-0.563 2.036,-1.677 0.123,-0.942 0.783,-2.577 1.782,-2.896 1.316,-0.421 1.151,1.419 2.051,1.678 1.55,0.445 -0.032,1.993 0.722,2.938 1.322,1.657 3.362,1.809 5.309,2.428 1.773,0.564 3.197,-0.23 4.931,-0.187 1.555,0.04 2.157,0.921 3.759,0.447 -0.962,-1.091 0.232,-1.264 0.877,-1.517 0.89,-0.348 0.956,-1.5 2.082,-1.828 0.726,-0.212 2.23,-0.315 2.319,-1.321 l 0.691,-31.11505 c 0.852,1.2 2.32,1.768 3.113,3.018 0.904,1.427 1.4604,3.3248 2.0494,4.9058 0.594,1.596 1.1206,3.3302 0.3456,4.9752 -0.992,2.105 -1.718,3.256 -0.748,5.602 1.715,-1.98 5.964,1.029 7.828,-0.782 0.309,-0.3 0.072,-1.068 0.526,-1.199 0.508,-0.147 0.951,-0.042 1.446,-0.325 1.293,-0.738 1.563,-2.433 2.906,-3.232 1.477,-0.878 2.422,-1.844 2.761,-3.606 0.295,-1.534 0.245,-2.809 1.509,-3.955 1.101,-0.998 2.526,-1.199 3.87,-0.593 1.107,0.5 2.704,2.852 3.758,2.823 2.367,-0.063 5.147,2.79 7.389,1.017 1.004,-0.794 2.505,0.31 3.756,-0.621 1.336,-0.995 1.822,-2.657 2.395,-4.158 0.304,-0.795 0.574,-1.614 0.706,-2.458 0.216,-1.372 -0.443,-1.271 0.949,-1.666 0.808,-0.229 3.171,-0.631 3.615,-1.389 0.793,-1.355 2.364,-1.963 3.025,-3.396 0.721,-1.561 0.413,-3.174 1.48,-4.645 0.913,-1.259 1.938,-1.371 3.12,-2.195 1.211,-0.845 2.756,-2.146 3.68,-3.311 0.787,-0.993 1.31,-2.516 2.494,-3.118 1.185,-0.602 2.585,-0.657 3.825,-1.109 1.188,-0.433 1.154,-0.958 1.735,-1.877 0.492,-0.78 1.834,-1.033 2.653,-1.204 3.341,-0.7 5.381,1.499 8.631,1.522 2.893,0.021 5.523,-1.095 6.474,2.456 0.53,1.981 0.985,4.001 1.038,6.06 0.049,1.874 1.195,3.483 1.873,5.166 1.453,3.604 1.0984,7.02677 0.9078,10.831 -0.073,1.46319 0.01,3.25521 -0.5178,4.801 -0.2569,0.7547 -0.3711,0.70813 -0.1002,2.88032 0.1223,0.056 -0.8771,-1.07593 -1.8758,-1.41825 -0.9468,-0.32452 -1.7891,-0.20229 -2.5594,-0.19371 -0.6363,1.24344 -1.6053,0.9878 -2.3317,2.56341 -0.8752,1.89823 -1.6607,4.81678 -1.9324,4.60599 1.4102,1.27135 1.8891,2.28858 2.7624,3.28933 1.4221,1.62951 3.4697,2.49532 5.0243,0.95308 0.4247,-0.42127 -0.212,-0.6492 1.3318,-2.80702 l 1.2916,0.0343 4.5335,0.15683 c -0.857,3.791 -1.9411,7.1347 -2.9541,10.88975 -0.524,1.943 -1.549,5.89 -4.224,5.699 0.423,0.951 -0.82,0.917 -1.392,1.032 -0.717,0.145 -1.326,1.046 -1.781,1.552 -1.609,1.792 -3.254,3.922 -3.993,6.205 -0.415,1.282 -1.549,2.092 -2.194,3.24 -0.993,1.767 -2.032,3.518 -3.194,5.179 -2.092,2.992 -4.332,5.01 -7.119,7.307 -2.326,1.918 -3.698,4.703 -6.021,6.613 -2.956,2.43 -5.849,4.938 -8.896,7.254 -2.395,1.819 -5.14,3.997 -8.224,4.413 -1.482,0.201 -2.95,-0.614 -4.415,-0.166 -1.061,0.324 -2.539,1.604 -1.08,2.52 -1.735,0.771 -3.66,-1.44 -5.286,-0.325 -1.897,1.301 -2.3,2.068 -4.903,1.076 -1.954,-0.744 -4.458,-1.538 -6.521,-0.751 -0.924,0.353 -0.459,0.948 -1.859,0.447 -0.824,-0.295 -2.262,0.215 -2.681,-0.737 -0.027,0.12 -0.027,0.24 0,0.36 -2.114,-0.343 -5.499,0.631 -6.583,2.599 -0.945,1.718 -5.67,-0.056 -7.173,0.592 -0.902,0.39 -1.853,0.13 -2.699,0.53 -1.025,0.484 -2.162,1.467 -3.063,2.167 -1.137,0.886 -0.687,0.71 -2.002,0.372 -0.571,-0.146 -1.215,0.181 -1.725,-0.238 -0.581,-0.479 -1.13,-1.048 -1.875,-1.271 1.309,-1.796 -2.712,-1.865 -3.6,-2.02 0.546,-1.399 -0.024,-2.649 -1.689,-2.342 -1.13,0.209 -0.718,1.547 -0.691,2.342 -2.054,-0.799 -0.429,-3.24 -0.23,-4.486 0.173,-1.085 -0.419,-2.204 -0.962,-3.095 -0.313,-0.513 -2.872,-4.058 -1.178,-2.419 -0.724,-0.727 -0.409,-1.529 -1.488,-1.521 -0.913,0.01 0.09,-2.272 0.188,-2.589 2.329,1.367 2.927,-1.308 2.969,-3.101 0.073,-3.069 -0.773,-5.659 -2.361,-8.29 -2.363,-3.915 -4.455,-7.792 -5.811,-12.19 -0.633,-2.054 -1.04,-4.173 -1.801,-6.188 -0.76,-2.018 -2.592,-3.813 -2.784,-6.012" id="za"/>
  <path class="landxx eg" d="M 1455.3863,499.96555 C 1453.9613,497.68455 1456.8963,495.29484 1456.5353,492.95555 C 1456.1477,490.44427 1453.8596,488.6908 1457.4753,485.70955 C 1457.5513,487.53955 1459.6843,486.88455 1460.7903,486.55555 C 1462.8113,485.95455 1464.4673,486.62455 1466.4753,486.82655 C 1468.6723,487.04655 1470.7803,487.53555 1472.9073,488.10255 C 1473.8063,488.34255 1473.9493,489.32055 1474.8623,489.60255 C 1475.6733,489.85255 1476.2903,489.75155 1477.0943,489.66655 C 1477.6343,489.60855 1477.8843,490.45255 1478.4933,490.57855 C 1479.1603,490.71655 1479.8413,490.71955 1480.5143,490.78655 C 1482.6773,491.00055 1484.1213,492.35755 1486.1503,492.69655 C 1488.1123,493.02455 1489.9883,491.70355 1491.3953,490.50355 C 1492.1253,489.88155 1492.8053,489.20555 1493.4743,488.51855 C 1494.0473,488.96555 1494.4793,489.42555 1495.2043,489.31055 C 1494.7903,489.10255 1494.8803,488.98355 1494.3343,489.02255 C 1495.0433,488.64655 1495.7513,487.96355 1495.9243,487.15055 C 1497.1493,487.88255 1498.3013,486.56755 1499.5243,486.57655 C 1498.7023,486.95455 1497.9133,487.42455 1497.2853,488.08755 C 1497.5023,487.95955 1502.4643,486.45955 1500.0953,486.28755 C 1501.8483,485.70755 1502.9863,487.71455 1504.6703,487.47555 C 1505.4743,487.36155 1506.2323,486.56655 1507.0853,486.86455 C 1507.8783,487.14055 1508.2523,488.28955 1508.9553,488.73555 C 1508.2433,488.23755 1507.5143,487.14355 1506.5753,487.08055 C 1506.8533,487.82655 1505.8253,489.23755 1507.0393,489.52755 C 1508.3593,489.84255 1507.7413,490.16955 1508.5953,490.96755 C 1508.5953,490.67355 1510.9503,489.67955 1509.2453,488.87955 C 1511.0593,489.30155 1511.6623,490.97555 1513.7903,490.78955 C 1514.4803,490.72955 1516.5173,489.45555 1516.1553,490.96755 C 1517.4743,490.12955 1518.9543,490.61855 1520.3713,490.19155 C 1521.6193,489.81655 1522.8684,489.32794 1523.8694,488.45894 L 1529.9489,504.50824 C 1528.4119,506.46324 1528.7553,510.07855 1527.7773,512.39455 C 1527.2053,513.74855 1527.2293,514.59255 1527.2443,516.02355 C 1527.2573,517.24755 1525.7013,518.43155 1526.0843,519.62455 C 1525.3413,518.93755 1524.3473,518.62755 1523.5273,518.05355 C 1522.4863,517.32455 1521.7583,516.26255 1521.0453,515.23555 C 1520.1423,513.93355 1517.5013,513.08355 1517.8843,511.12855 C 1518.2703,509.15855 1516.3703,507.79255 1515.3993,506.40455 C 1514.0563,504.48755 1513.0983,502.25455 1512.5543,499.96855 C 1511.6523,500.93755 1510.6923,503.05155 1511.6333,504.15755 C 1513.2423,506.05255 1513.0753,508.37855 1514.4243,510.33655 C 1515.5923,512.03255 1516.7763,514.10855 1518.2363,515.55555 C 1518.8823,516.19555 1519.6193,516.74655 1520.2083,517.44355 C 1520.5163,517.80855 1521.7853,519.90055 1520.2553,518.75955 C 1521.4933,519.72555 1520.4693,520.95555 1521.2543,521.75155 C 1522.3083,522.81955 1522.8683,523.95155 1523.4203,525.29755 C 1523.6913,525.95955 1524.1383,526.52655 1524.5233,527.12155 C 1524.8173,527.57655 1524.2333,528.15655 1524.2073,528.60155 C 1524.1713,529.19655 1525.1433,530.07955 1525.4083,530.59255 C 1526.0173,531.76655 1526.6813,532.91255 1527.2963,534.08455 C 1529.3153,537.93455 1531.3533,541.71655 1533.2643,545.62555 C 1534.0123,547.15455 1535.1113,548.24955 1536.0503,549.62355 C 1536.9243,550.90255 1538.4563,551.29855 1539.2663,552.59955 C 1538.5003,552.64055 1537.7273,552.50755 1537.0373,552.16755 C 1536.5323,554.25655 1536.7761,557.44607 1537.8423,558.67755 C 1538.8574,559.84993 1536.5144,562.20415 1535.0853,562.27355 C 1534.3773,562.30793 1534.1084,561.79253 1533.3948,562.58849 C 1532.8327,563.21549 1532.5363,564.29055 1532.2803,565.09455 C 1531.9863,566.02355 1531.5493,566.84455 1530.5233,567.08355 C 1529.6933,567.27655 1528.1997,567.10674 1527.7587,568.05874 C 1526.9527,569.80174 1527.4445,571.83582 1524.9875,571.64074 C 1522.125,571.41345 1520.6696,569.09252 1519.1615,569.15568 C 1514.193,569.36376 1511.3412,569.81686 1507.4463,569.15955 C 1507.5583,568.76855 1508.0103,567.94155 1507.7313,567.53955 C 1507.0283,566.52255 1506.4893,568.36655 1506.2563,568.69955 C 1505.3233,570.03355 1500.6613,569.23155 1499.1673,569.23155 L 1459.0783,569.14482 L 1457.6493,514.95955 C 1457.4993,512.42655 1457.5613,509.76955 1457.2113,507.26055 C 1456.9113,505.10255 1456.3573,501.91055 1455.3863,499.96555" id="eg"/>
  <g class="landxx coastxx tz" id="tz" transform="translate(-29.9017, -45.0745)">
    <path d="M 1599.559,856.733 C 1599.841,858.532 1600.098,859.174 1601.521,860.41 C 1602.508,861.267 1602.794,856.922 1601.208,858.243 C 1601.467,857.138 1601.132,855.326 1600.559,854.352 C 1600.322,855.128 1599.321,855.84 1599.559,856.733" id="path4894"/>
    <path d="M 1602.798,873.793 C 1603.571,873.096 1604.855,872.641 1604.378,871.343 C 1603.939,872.387 1602.627,872.439 1602.798,873.793" id="path4896"/>
    <path d="M 1603.298,847.873 C 1603.416,848.79 1604.205,849.559 1603.149,849.673 C 1603.4,849.829 1603.643,849.995 1603.878,850.173 C 1602.391,851.018 1603.725,852.309 1604.348,850.947 C 1605.086,849.332 1604.879,847.762 1603.369,847.154 C 1603.345,847.393 1603.321,847.633 1603.298,847.873" id="path4898"/>
    <path class="landxx tz" d="M 1527.5614,851.34618 C 1528.1381,847.11097 1528.2989,850.05803 1526.5102,846.25711 C 1526.1694,845.53294 1526.9575,844.23754 1526.5747,843.02741 C 1528.3247,842.8424 1528.613,842.69458 1529.747,841.396 C 1530.6545,840.35681 1532.1309,838.93631 1533.0489,837.81167 C 1533.7138,836.99717 1532.3062,836.203 1532.5922,835.606 C 1532.9362,834.888 1534.3711,835.46779 1534.1191,834.61679 C 1533.9531,834.05379 1535.377,833.243 1535.677,832.954 C 1536.535,832.128 1535.3,831.157 1536.048,830.373 C 1533.75,831.02 1532.308,829.637 1533.598,827.573 C 1532.348,827.693 1533.227,826.776 1533.685,826.049 C 1534.267,825.126 1535.3945,825.49114 1535.8825,824.54914 C 1536.7915,822.79314 1535.3936,819.64315 1534.6045,817.95442 C 1534.0526,816.77319 1532.1285,814.06959 1533.9582,813.47095 C 1536.2118,812.73364 1541.2349,813.30505 1543.6566,813.37673 C 1544.664,817.16004 1542.9116,817.23924 1543.0437,817.67837 C 1543.4995,819.19333 1542.1803,820.99343 1542.2311,821.36489 C 1543.0203,822.42515 1542.1567,823.43955 1542.3202,823.98735 C 1542.8677,825.82113 1543.3222,824.36656 1543.5437,828.84001 C 1543.6605,827.89241 1544.3564,827.38397 1544.9232,828.41522 C 1544.6985,826.57864 1544.9449,826.40241 1546.0858,826.54891 C 1545.2247,826.08025 1545.7904,825.02378 1547.1367,824.273 C 1547.399,824.53652 1548.7118,827.34555 1550.0928,825.55087 C 1549.9134,825.98084 1550.9221,826.13038 1551.0937,826.70104 C 1551.432,827.82635 1551.4211,829.85462 1550.4364,829.40857 L 1551.016,830.1651 C 1551.5058,829.5954 1550.6281,829.39105 1552.5695,829.23361 C 1551.4021,829.44532 1549.64,822.84925 1555.7065,826.62525 C 1555.9721,825.00764 1560.0764,823.17324 1558.8333,823.04709 C 1557.328,822.89432 1557.5146,823.52682 1556.6835,823.41576 C 1554.1767,823.08072 1554.2926,822.5921 1554.1546,822.70169 C 1554.7486,822.02201 1554.1943,822.02318 1556.8084,821.8394 C 1556.5779,820.88412 1554.2242,821.40415 1554.7824,821.24553 C 1555.0386,821.01317 1553.982,820.61273 1556.8693,820.22522 C 1557.1866,820.18263 1557.6259,819.66978 1557.4919,819.08951 C 1557.4106,818.73737 1557.8399,819.02753 1557.979,818.84609 C 1556.8411,817.64653 1557.6429,817.50334 1557.914,817.34872 C 1558.1802,817.19696 1558.4662,817.50093 1559.7397,817.75318 C 1559.4884,817.78198 1558.7712,816.67548 1559.3505,816.03016 C 1559.7036,815.56585 1559.3538,816.53792 1559.6821,816.10643 C 1559.2872,815.81961 1559.4235,815.1528 1559.7181,814.80333 C 1560.0152,814.42597 1560.4362,814.01302 1560.7311,813.63433 L 1583.5715,827.95514 C 1585.5055,829.18214 1587.9149,830.06473 1588.4909,832.41973 C 1588.7369,833.42973 1587.939,833.394 1587.729,834.26 C 1587.574,834.899 1588.679,835.621 1589.042,836.153 C 1590.291,837.986 1592.771,839.211 1594.404,840.685 C 1595.376,841.563 1599.9819,844.53118 1599.8099,845.81118 C 1599.5349,847.86518 1598.789,850.174 1597.938,852.066 C 1597.505,853.029 1597.041,853.983 1596.731,854.995 C 1596.578,855.473 1596.481,855.961 1596.442,856.461 C 1596.399,857.145 1597.03,857.77 1596.922,858.377 C 1596.461,860.968 1600.649,861.168 1600.128,863.932 C 1602.391,863.215 1602.089,865.894 1601.24,866.968 C 1599.888,868.676 1600.68,870.695 1599.767,872.352 C 1601.336,870.924 1601.237,873.886 1600.871,874.644 C 1600.333,875.757 1599.905,877.103 1600.067,878.354 C 1600.217,879.512 1601.91,881.462 1600.637,881.712 C 1601.603,881.896 1601.038,883.691 1601.786,883.617 C 1603.005,883.498 1602.38,884.845 1602.115,885.432 C 1601.627,886.518 1602.895,886.917 1603.116,887.909 C 1603.239,888.461 1602.82,888.788 1603.116,889.344 C 1603.504,890.072 1603.68,890.513 1602.936,891.142 C 1604.202,890.907 1604.7,891.985 1604.946,893.012 C 1604.8355,891.21797 1609.2312,893.80474 1607.8262,895.78857 C 1604.8009,898.12534 1602.1376,899.49645 1599.3431,901.21778 C 1598.3281,901.57278 1598.1841,900.93168 1597.1601,901.28268 C 1596.0841,901.65068 1594.457,902.943 1593.364,903.302 C 1592.284,903.656 1590.9782,901.54564 1589.756,902.158 C 1588.344,902.86549 1587.7395,905.00999 1586.2352,905.67518 C 1583.7237,906.78572 1583.2334,905.93609 1582.4377,905.25346 C 1580.8797,903.91677 1580.3113,906.37259 1578.5967,906.17346 C 1576.6259,905.94458 1575.9564,906.21276 1574.873,904.84698 C 1574.0299,903.7841 1573.0606,903.29252 1571.8296,904.431 C 1570.9602,905.235 1567.9465,905.0917 1566.397,904.77053 C 1566.2526,904.63048 1566.1768,903.82786 1565.9778,903.53186 C 1565.431,902.71865 1564.8764,901.90049 1564.4234,900.99949 C 1563.7924,899.74649 1564.2245,898.38435 1563.9625,897.05535 C 1563.6115,895.28135 1563.913,893.64 1563.469,891.895 C 1563.006,890.076 1561.8719,888.39112 1560.4272,887.267 C 1559.02,886.17211 1558.7451,888.29627 1558.875,888.622 C 1557.439,887.076 1556.089,888.52 1554.741,887.044 C 1554.318,886.58 1552.7222,886.46037 1552.105,886.102 C 1551.058,885.49408 1550.117,884.99603 1548.5472,884.766 C 1548.2985,884.72956 1548.0836,884.34441 1547.9666,884.12341 C 1547.4686,883.50841 1547.0625,883.65595 1546.4255,883.33695 C 1545.8285,883.03895 1545.035,883.14 1544.445,882.827 C 1543.901,882.538 1543.773,881.86 1543.274,881.614 C 1542.865,881.412 1542.3899,881.90164 1541.9579,881.77564 C 1541.2539,881.56964 1541.1559,880.82096 1540.7829,880.30064 C 1540.0501,879.27857 1539.1197,879.09031 1538.1224,879.16005 C 1537.6532,878.10591 1535.1747,873.04683 1534.0296,869.99844 C 1532.9729,867.18539 1533.7632,861.95892 1529.8243,860.46824 C 1529.23,860.24332 1529.7142,860.92014 1529.031,860.60473 C 1527.5865,859.93789 1526.1389,858.31106 1528.7019,855.78936 C 1529.6904,854.81687 1528.0131,852.96619 1527.5614,851.31432" id="path5432"/>
  </g>
  <g class="landxx coastxx sa" id="sa" transform="translate(-29.9017, -45.0745)">
    <path d="M 1619.718,661.32 C 1619.677,660.509 1619.342,659.753 1618.639,659.303 C 1618.623,659.568 1618.596,659.832 1618.559,660.095 C 1617.621,660.132 1617.284,659.141 1616.479,658.871 C 1617.259,660.169 1619.268,660.143 1619.718,661.32" id="path3234"/>
    <path class="landxx sa" d="M 1624.548,662.653 C 1624.645,661.745 1624.091,659.593 1623.274,659.073 C 1622.858,658.808 1620.843,654.574 1620.959,656.617 C 1620.772,652.958 1618.236,652.017 1616.218,649.54 C 1615.454,648.602 1614.376,647.248 1614.018,646.101 C 1613.555,644.621 1612.72,644.199 1611.962,642.996 C 1611.518,642.29 1611.896,641.674 1611.384,640.949 C 1611.019,640.434 1610.409,638.795 1610.168,638.134 C 1609.942,637.515 1609.805,636.861 1609.515,636.266 C 1609.248,635.719 1608.691,635.642 1608.423,635.161 C 1608.167,634.699 1608.418,634.069 1608.064,633.664 C 1607.705,633.255 1607.119,633.198 1606.73,632.836 C 1605.753,631.927 1604.622,631.008 1603.759,629.995 C 1602.712,628.765 1601.485,629.28 1600.304,628.41 C 1598.345,626.964 1597.042,624.279 1595.811,622.223 C 1595.094,621.026 1595.3,620.341 1595.24,619.078 C 1595.174,617.71 1594.184,616.578 1594.082,615.195 C 1593.948,613.38 1594.699,611.575 1594.338,609.767 C 1594.173,608.943 1593.723,608.246 1593.281,607.551 C 1593.049,607.185 1591.642,605.37 1593.23,606.611 C 1592.881,605.699 1592.088,605.689 1591.882,604.69 C 1591.697,603.797 1591.259,602.82 1590.91,601.977 C 1589.944,599.642 1588.059,598.051 1586.292,596.371 C 1585.003,595.147 1583.726,594.954 1582.154,594.524 C 1581.484,594.341 1581.164,593.066 1580.967,592.522 C 1580.545,591.356 1579.729,590.556 1578.998,589.584 C 1581.528,588.662 1578.004,584.509 1577.178,583.504 C 1576.621,582.829 1575.058,581.599 1575.125,580.708 C 1575.242,579.173 1573.265,578.286 1572.6,576.905 C 1570.584,572.715 1567.402,569.234 1565.168,565.123 C 1564.487,563.871 1563.727,561.94 1562.226,561.517 C 1561.544,561.325 1558.862,560.879 1558.809,562.138 C 1557.912,561.318 1559.362,559.163 1559.614,558.302 C 1560.142,556.497 1559.72,554.61929 1560.015,552.797 C 1560.3887,550.48848 1560.0286,551.9254 1560.3535,550.54963 C 1562.0765,551.0622 1566.817,552.33479 1568.536,551.96 C 1571.951,551.21544 1571.897,546.25 1575.106,545.744 C 1576.309,545.554 1579.2619,545.85807 1579.133,544.236 C 1578.8389,540.53658 1580.4483,541.64719 1582.6876,540.89194 C 1579.765,538.09245 1576.9846,535.17277 1574.3111,532.13794 C 1579.5036,531.05316 1582.9504,529.88325 1589.0711,526.61336 C 1591.6638,527.11059 1596.2881,527.39321 1599.1306,528.70914 C 1608.682,533.13084 1616.549,539.891 1624.84,546.2 C 1626.596,547.537 1628.305,548.864 1629.944,550.337 C 1631.811,552.014 1633.1087,551.93027 1635.5687,552.16827 C 1638.4047,552.44227 1641.393,552.843 1644.23,553.105 C 1645.271,553.202 1645.785,552.671 1646.915,552.915 C 1648.442,553.244 1650.026,553.43 1651.581,553.571 C 1653.229,553.719 1653.359,555.635 1653.885,556.848 C 1654.664,558.65 1657.289,557.512 1658.779,557.643 C 1660.32,557.779 1660.521,558.614 1661.025,559.875 C 1661.57,561.24 1664.28,562.979 1663.859,564.51 C 1663.72,564.269 1663.55,564.053 1663.348,563.863 C 1662.887,564.96 1663.993,564.835 1663.859,565.803 C 1663.878,565.818 1668.491,566.245 1666.729,567.6 C 1667.405,567.814 1668.84,571.275 1669.529,569.325 C 1669.069,571.061 1673.235,573.011 1674.199,574.066 C 1673.685,573.912 1673.302,573.6 1673.049,573.132 C 1673.073,575.046 1674.233,575.074 1674.645,576.61 C 1674.886,577.51 1674.79,579.409 1673.48,578.234 C 1673.599,579.255 1673.269,579.458 1674.197,579.646 C 1674.34,579.674 1675.556,582.311 1675.64,582.617 C 1675.37,582.384 1675.107,582.144 1674.849,581.898 C 1675.646,583.067 1677.069,583.724 1677.699,585.018 C 1678.338,586.326 1678.532,587.526 1679.3,588.797 C 1679.89,589.775 1680.743,591.451 1681.965,591.751 C 1682.524,591.888 1684.552,591.358 1684.26,592.461 C 1684.474,591.901 1684.959,591.696 1685.481,592.029 C 1684.798,592.863 1684.1808,595.13714 1686.4878,594.54814 C 1686.3358,594.58714 1687.668,597.18459 1687.796,597.36759 C 1690.4559,600.38582 1694.0439,605.69524 1695.4539,606.26755 C 1704.2175,607.24656 1708.2698,607.72271 1713.0378,608.78434 C 1713.6593,608.63552 1714.2039,608.26213 1714.4029,607.85982 C 1715.467,610.31957 1717.884,611.60687 1718.2902,613.20382 C 1718.6572,614.64673 1718.5549,615.60485 1718.1767,617.08865 C 1716.9105,622.05712 1716.2131,631.00532 1714.378,631.99405 C 1701.8839,638.72558 1686.2472,641.0812 1671.399,643.541 C 1669.473,644.223 1667.6204,645.09378 1665.8625,646.11109 C 1663.2191,647.64081 1660.6488,652.23604 1659.7912,654.97914 C 1659.3949,656.2468 1657.9032,657.63486 1656.5082,657.64086 C 1655.0502,657.64686 1654.7807,655.27666 1653.5925,654.87681 C 1652.2423,654.42243 1651.2454,655.11243 1650.0813,654.96995 C 1648.2762,654.749 1645.8635,655.03267 1644.241,654.19782 C 1642.292,653.19496 1640.8559,653.6552 1638.6969,653.65946 C 1636.8059,653.66318 1634.8091,654.82094 1632.9264,654.797 C 1630.5887,654.76728 1630.247,652.98814 1629.0304,653.308 C 1627.3999,653.73668 1626.9674,655.87719 1626.8766,657.23027 C 1626.787,658.56544 1628.4577,659.34178 1627.0426,660.30627 C 1626.7712,660.49129 1626.4881,662.38272 1624.4535,663.05554 C 1624.4405,662.87654 1624.558,662.828 1624.548,662.653" id="path3380"/>
    <path d="M 1617.268,657.504 C 1617.799,657.636 1617.918,657.804 1617.628,658.008 C 1617.508,657.84 1617.388,657.672 1617.268,657.504" id="path5480"/>
    <path d="M 1617.628,658.008 C 1617.847,658.318 1617.893,658.654 1617.768,659.016 C 1617.504,658.943 1617.265,658.823 1617.048,658.657 C 1617.241,658.44 1617.435,658.224 1617.628,658.008" id="path5482"/>
    <path d="M 1668.889,568.368 C 1669.244,567.829 1669.799,567.973 1670.189,568.368 C 1669.755,568.451 1669.322,568.451 1668.889,568.368" id="path5488"/>
  </g>
  <g class="landxx coastxx vn" id="vn" transform="translate(-29.9017, -45.0745)">
    <path d="M 2108.088,720.288 C 2108.326,721.055 2108.776,721.841 2109.598,722.088 C 2109.018,721.561 2108.49,720.965 2108.088,720.288" id="path3474"/>
    <path d="M 2112.698,712.368 C 2112.866,713.156 2112.315,713.948 2113.128,714.528 C 2113.494,713.768 2113.244,712.96 2112.698,712.368" id="path3476"/>
    <path d="M 2113.918,713.664 C 2113.859,713.902 2113.859,714.142 2113.918,714.384 C 2113.918,714.144 2113.918,713.904 2113.918,713.664" id="path3478"/>
    <path d="M 2130.559,695.304 C 2130.676,695.643 2130.913,695.859 2131.268,695.952 C 2131.031,695.736 2130.795,695.52 2130.559,695.304" id="path3480"/>
    <path class="landxx vn" d="M 2065.538,610.848 C 2066.7346,610.7925 2066.8311,608.65107 2067.5781,608.06547 C 2068.8142,607.09645 2070.188,608.876 2070.821,609.51 C 2071.9526,610.64339 2072.6366,609.77373 2073.3346,608.57973 C 2074.2946,606.93773 2074.88,608.415 2075.618,609.264 C 2075.892,608.629 2075.878,607.077 2076.872,608.07 C 2077.486,608.683 2078.054,609.438 2078.795,609.905 C 2079.258,610.197 2079.577,607.156 2080.878,607.104 C 2080.506,609.123 2084.2939,607.32796 2084.8311,606.65114 C 2085.4063,605.92651 2084.6707,605.39548 2084.886,604.741 C 2085.1698,603.87855 2087.478,603.61614 2087.997,602.353 C 2089.063,603.828 2090.412,604.991 2092.044,605.778 C 2093.321,606.395 2094.497,605.215 2095.421,606.345 C 2096.348,607.479 2097.785,605.719 2098.913,606.847 C 2099.485,607.42 2100.181,609.743 2098.587,609.193 C 2098.579,611.16 2099.624,612.605 2100.237,614.52 C 2100.806,614.031 2102.907,614.696 2102.907,615.528 C 2102.907,616.775 2105.307,616.496 2105.637,617.689 C 2105.877,617.545 2106.117,617.401 2106.357,617.256 C 2106.745,618.275 2107.3368,617.59167 2108.1923,617.36952 C 2109.4731,617.03696 2113.0056,618.21907 2111.8104,618.88718 C 2110.9464,619.64418 2109.909,618.012 2109.024,618.747 C 2108.222,619.413 2107.965,620.362 2106.884,620.53 C 2105.726,620.71 2106.638,621.765 2106.214,622.53 C 2105.801,623.275 2103.463,623.996 2104.637,622.656 C 2103.883,622.473 2102.875,623.62 2102.836,624.024 C 2102.201,623.669 2101.953,622.486 2101.036,622.799 C 2101.32,623.107 2101.653,623.346 2102.036,623.519 C 2101.4,624.263 2101.88,624.631 2102.362,625.421 C 2102.764,626.079 2101.311,626.06 2101.036,625.895 C 2101.363,626.576 2101.546,628.483 2100.816,628.992 C 2100.913,629.024 2101.009,629.024 2101.106,628.992 C 2101.028,629.537 2098.781,631.987 2098.295,631.368 C 2097.083,631.973 2096.694,633.789 2096.577,634.997 C 2096.415,636.664 2097.118,637.793 2096.256,639.361 C 2095.278,641.139 2097.176,641.545 2097.285,643.175 C 2097.714,642.05 2097.97,643.746 2098.435,644.359 C 2099.061,645.183 2099.789,645.822 2100.635,646.412 C 2101.44,646.974 2102.254,647.558 2102.973,648.229 C 2104.006,649.193 2103.553,650.23 2103.905,651.456 C 2103.461,651.11 2102.958,650.918 2102.395,650.88 C 2104.193,651.185 2105.042,653.428 2106.448,654.554 C 2107.474,655.376 2110.726,656.977 2109.886,658.728 C 2110.986,658.093 2112.735,660.295 2113.345,660.96 C 2113.039,660.82 2112.726,660.7 2112.406,660.6 C 2113.751,661.438 2114.979,662.359 2115.866,663.697 C 2117.064,662.261 2119.484,665.482 2118.816,666.217 C 2119.45,665.773 2118.468,664.385 2119.755,665.065 C 2117.825,666.728 2123.747,670.575 2123.856,672.049 C 2124.347,670.798 2124.574,671.842 2124.968,672.417 C 2125.584,673.316 2124.639,672.947 2124.795,673.633 C 2125.085,674.909 2126.121,676.045 2126.628,677.236 C 2127.315,678.85 2127.673,680.601 2128.4,682.223 C 2128.693,682.876 2129.008,683.536 2129.168,684.24 C 2129.246,684.584 2129.278,686.398 2128.825,684.649 C 2128.799,685.624 2129.947,688.455 2128.825,686.738 C 2129.08,687.317 2129.439,687.821 2129.905,688.25 C 2128.565,688.434 2129.644,689.104 2129.856,689.864 C 2130.163,690.964 2130.512,691.707 2131.096,692.684 C 2131.334,693.082 2131.349,695.06 2131.264,695.665 C 2130.858,695.256 2130.571,694.776 2130.405,694.225 C 2129.045,695.022 2130.321,696.672 2130.764,697.61 C 2130.314,697.344 2129.833,697.152 2129.325,697.034 C 2129.969,697.619 2130.622,700.696 2130.264,701.569 C 2130.258,701.048 2130.115,700.569 2129.834,700.13 C 2130.285,700.846 2129.902,701.587 2129.685,702.29 C 2130.497,702.268 2130.539,702.86 2130.326,703.476 C 2130.098,704.133 2129.137,704.447 2129.075,704.666 C 2128.698,706.017 2129.227,706.056 2127.755,706.811 C 2127.341,707.166 2126.903,707.488 2126.442,707.776 C 2125.801,708.038 2125.599,708.966 2125.109,709.114 C 2124.297,709.359 2123.461,709.706 2122.75,710.172 C 2122.417,710.39 2121.915,712.004 2121.659,711.917 C 2120.928,711.668 2119.7,712.575 2119.147,712.992 C 2118.397,713.558 2116.915,714.056 2116.366,714.747 C 2115.857,713.759 2114.52,713.791 2114.426,712.515 C 2114.331,713.02 2114.236,713.524 2114.137,714.028 C 2114.022,713.622 2113.855,713.238 2113.637,712.876 C 2114.01,713.709 2113.886,713.661 2113.266,712.731 C 2114.077,712.731 2112.46,712.223 2112.477,712.227 C 2112.725,713.101 2112.371,714.221 2111.256,713.956 C 2111.519,714.761 2112.678,713.539 2112.944,714.676 C 2113.284,716.127 2110.696,715.416 2110.176,715.396 C 2111.073,716.124 2112.547,715.983 2113.126,717.125 C 2112.257,717.257 2112.208,718.628 2111.21,717.93 C 2110.548,717.467 2110.109,716.074 2109.236,715.973 C 2109.795,716.628 2112.574,718.927 2112.116,719.645 C 2110.344,718.924 2109.414,717.165 2107.946,716.045 C 2108.262,717.198 2113.537,720.74 2111.171,722.032 C 2109.617,722.881 2106.984,718.749 2105.857,718.06 C 2106.625,719.323 2109.464,721.272 2108.979,722.953 C 2108.603,724.253 2106.395,724.608 2105.35,725.131 C 2103.586,726.014 2103.526,727.333 2102.446,728.742 C 2101.711,729.703 2099.798,731.297 2098.587,730.156 C 2098.958,729.686 2099.439,729.398 2100.027,729.292 C 2099.892,728.992 2099.796,728.68 2099.737,728.355 C 2099.432,728.497 2099.119,728.617 2098.797,728.715 C 2098.845,726.9 2098.469,725.098 2098.456,723.282 C 2098.45,722.387 2098.514,721.47 2098.797,720.615 C 2098.987,720.039 2100.261,717.593 2100.527,719.354 C 2100.521,716.527 2095.131,716.939 2095.637,714.531 C 2095.533,714.957 2094.9576,714.66216 2094.6746,714.18816 C 2095.1566,714.02216 2095.8739,713.48076 2096.4119,713.42641 C 2100.6066,713.00261 2098.4385,711.30617 2099.666,709.851 C 2101.032,711.165 2102.944,709.741 2104.419,709.451 C 2104.932,709.35 2105.2718,709.797 2105.5038,710.176 C 2106.0658,711.091 2107.119,711.004 2108.016,711.292 C 2108.037,710.504 2107.213,709.846 2108.157,709.708 C 2107.974,708.378 2106.312,708.328 2105.768,707.406 C 2105.302,706.618 2104.99,705.606 2104.8,704.714 C 2104.619,703.868 2105.508,703.771 2105.929,703.117 C 2106.571,702.121 2108.708,703.572 2109.527,703.731 C 2109.07,702.895 2109.662,701.908 2109.027,701.139 C 2109.722,700.964 2110.5888,701.2242 2111.172,701.026 C 2111.7678,700.82353 2112.8154,699.94527 2113.3734,699.55627 C 2114.0204,699.10327 2114.255,698.521 2115.076,698.286 C 2115.711,698.105 2116.564,698.176 2117.076,697.693 C 2118.455,696.392 2116.347,692.822 2116.522,691.157 C 2116.67,689.73 2117.201,688.707 2116.495,687.327 C 2116.263,686.904 2116.028,686.482 2115.791,686.061 C 2115.391,685.308 2115.641,684.772 2115.357,684.076 C 2114.956,683.092 2114.404,683.182 2114.306,681.871 C 2114.188,680.285 2115.198,679.357 2115.29,677.953 C 2115.312,677.615 2115.2214,676.03918 2115.0284,675.87818 C 2113.8284,674.87918 2115.067,674.935 2115.521,673.891 C 2116.221,672.282 2114.311,670.695 2113.154,669.949 C 2112.624,669.607 2111.258,668.881 2111.326,668.093 C 2111.407,667.152 2112.978,666.767 2113.346,665.933 C 2112.34,665.551 2111.034,664.768 2110.391,663.882 C 2109.912,663.224 2109.463,663.771 2108.933,662.869 C 2108.218,661.654 2107.822,662.785 2106.937,662.261 C 2106.422,661.955 2105.832,660.654 2105.734,660.076 C 2105.426,658.257 2104.67,656.704 2103.182,655.422 C 2102.762,655.06 2101.806,654.819 2101.218,654.202 C 2100.111,653.037 2098.782,651.856 2097.847,650.553 C 2097.245,649.715 2096.987,648.687 2096.356,647.869 C 2096.105,647.545 2095.686,647.158 2095.266,647.5 C 2094.825,646.822 2094.005,646.585 2093.588,645.887 C 2093.281,645.373 2092.411,644.16 2092.99,643.586 C 2093.714,642.867 2090.318,642.169 2090.093,642.117 C 2087.83,641.6 2086.284,639.005 2084.141,638.417 C 2083.522,638.247 2081.85,637.461 2083.194,636.829 C 2085.104,635.931 2082.7,635.401 2083.037,634.397 C 2083.217,633.857 2085.814,634.651 2086.071,634.69 C 2087.127,634.85 2089.987,633.094 2088.507,632.669 C 2088.83,632.18 2089.797,631.922 2089.797,631.229 C 2089.797,630.354 2088.826,629.569 2087.976,629.68 C 2087.398,629.755 2087.328,629.381 2087.16,628.898 C 2086.789,627.831 2085.685,628.119 2084.838,627.773 C 2085.035,627.014 2086.139,627.163 2086.199,626.532 C 2086.303,625.461 2085.119,625.056 2084.423,624.552 C 2083.592,623.95 2083.032,623.227 2081.868,623.439 C 2080.055,623.768 2080.459,624.737 2079.648,625.83 C 2078.842,625.336 2077.654,624.402 2076.77,624.573 C 2075.762,624.768 2073.88,622.959 2073.388,622.229 C 2072.827,621.4 2072.318,619.416 2072.603,618.424 C 2073.107,616.667 2071.652,617.051 2071.297,615.677 C 2071.518,616.241 2071.421,616.745 2071.007,617.189 C 2070.062,616.903 2069.849,615.227 2069.032,614.581 C 2067.928,613.7 2066.11,612.149 2065.538,610.848" id="path3562"/>
    <path d="M 2102.618,624.312 C 2102.711,623.967 2102.568,623.727 2102.188,623.592 C 2102.332,623.832 2102.475,624.071 2102.618,624.312" id="path5494"/>
    <path d="M 2103.268,624.384 C 2103.582,624.832 2103.989,625.168 2104.489,625.392 C 2105.087,624.469 2103.916,624.218 2103.268,624.384" id="path5496"/>
    <path d="M 2106.868,620.567 C 2106.666,621.2 2106.546,621.848 2106.508,622.511 C 2107.262,622.065 2108.095,621.073 2106.868,620.567" id="path5498"/>
    <path d="M 2107.878,622.656 C 2107.509,622.98 2107.219,623.364 2107.008,623.807 C 2107.533,623.64 2107.898,623.228 2107.878,622.656" id="path5500"/>
    <path d="M 2109.748,619.632 C 2109.998,619.75 2110.238,619.702 2110.468,619.488 C 2110.228,619.536 2109.988,619.584 2109.748,619.632" id="path5502"/>
  </g>
  <g class="landxx coastxx ru" id="ru" transform="translate(-29.9017, -45.0745)">
    <path d="M 1526.188,253.224 C 1526.449,253.342 1526.711,253.342 1526.978,253.224 C 1526.732,251.929 1525.915,251.754 1524.748,252 C 1525.165,252.482 1525.646,252.89 1526.188,253.224" id="path2926"/>
    <path d="M 1527.908,252.792 C 1527.663,252.812 1527.423,252.859 1527.188,252.937 C 1527.429,252.889 1527.668,252.84 1527.908,252.792" id="path2928"/>
    <path d="M 1551.598,255.744 C 1552.592,255.973 1553.6,256.179 1554.559,256.536 C 1554.261,256.113 1554.054,255.563 1553.626,255.243 C 1553.08,254.834 1552.199,255.06 1553.118,255.96 C 1552.638,255.634 1552.131,255.562 1551.598,255.744" id="path2938"/>
    <path d="M 1552.389,253.368 C 1552.465,253.668 1552.612,253.932 1552.828,254.16 C 1552.682,253.896 1552.535,253.632 1552.389,253.368" id="path2940"/>
    <path d="M 2185.988,332.353 C 2185.941,332.593 2185.894,332.832 2185.847,333.073 C 2186.584,332.778 2187.392,332.807 2188.148,333.001 C 2187.725,332.585 2187.631,332.105 2187.868,331.561 C 2187.056,331.293 2186.545,331.904 2185.988,332.353" id="path3274"/>
    <path d="M 2189.738,332.353 C 2189.74,332.897 2189.884,333.401 2190.168,333.865 C 2191.803,332.687 2192.849,335.649 2194.289,334.996 C 2195.706,334.354 2194.131,330.966 2192.978,332.353 C 2193.421,330.757 2190.792,331.266 2190.028,330.697 C 2189.763,331.179 2190.275,332.032 2189.738,332.353" id="path3276"/>
    <path d="M 2195.568,337.319 C 2195.18,336.798 2194.747,336.318 2194.268,335.88 C 2194.385,336.565 2194.889,337.145 2195.568,337.319" id="path3278"/>
    <path d="M 2196.148,337.392 C 2196.432,337.228 2196.695,337.037 2196.938,336.816 C 2196.7,336.626 2196.437,336.482 2196.148,336.384 C 2196.217,336.72 2196.217,337.056 2196.148,337.392" id="path3280"/>
    <path d="M 2230.348,340.56 C 2230.988,340.74 2231.649,341.234 2231.843,341.899 C 2232.075,342.699 2233.1,342.245 2233.519,343.008 C 2233.003,343.19 2232.473,343.262 2231.928,343.224 C 2232.198,344.162 2233.256,344.052 2233.949,344.449 C 2233.821,344.942 2233.386,345.752 2232.799,345.816 C 2231.915,345.914 2232.147,345.029 2231.499,344.808 C 2228.566,343.807 2230.212,346.684 2231.179,347.636 C 2232.582,349.016 2234.132,350.427 2234.879,352.296 C 2235.628,354.172 2236.239,355.405 2237.469,357.06 C 2239.604,359.935 2243.837,361.004 2246.286,363.604 C 2247.454,364.842 2247.907,366.522 2248.659,368.006 C 2249.62,369.903 2251.576,371.128 2252.82,372.862 C 2255.006,375.91 2256.902,378.6 2258.148,382.101 C 2259.458,385.782 2263.311,386.873 2265.237,390.041 C 2266.084,391.434 2265.655,393.048 2266.604,394.488 C 2267.569,395.953 2269.262,396.985 2269.982,398.589 C 2270.682,400.15 2270.631,402.009 2271.669,403.426 C 2273.062,405.326 2274.765,407.059 2276.719,408.383 C 2276.174,407.648 2276.448,406.501 2276.068,405.656 C 2275.714,404.869 2275.592,404.102 2275.418,403.269 C 2275.166,402.065 2275.92,400.429 2277.186,402.048 C 2277.959,403.039 2280.682,402.332 2281.822,403.066 C 2282.969,403.805 2284.016,405.465 2284.64,406.655 C 2284.475,405.208 2283.94,404.024 2283.049,402.883 C 2282.588,402.293 2282.141,401.633 2281.622,401.092 C 2280.775,400.209 2279.902,400.98 2279.02,400.39 C 2279.452,400.781 2279.955,401.021 2280.53,401.111 C 2279.005,402.395 2276.705,398.56 2276.012,397.659 C 2274.244,395.355 2271.207,394.633 2269.268,392.499 C 2266.81,389.794 2266.772,386.041 2265.243,382.892 C 2264.836,382.054 2263.691,380.753 2264.873,380.038 C 2265.64,379.574 2266.693,379.482 2267.57,379.511 C 2266.643,379.341 2265.627,379.62 2264.759,379.152 C 2265.914,378.11 2271,380.125 2272.044,380.695 C 2273.347,381.406 2274.798,382.052 2275.962,382.982 C 2276.879,383.714 2277.705,384.793 2278.879,385.127 C 2277.998,384.684 2277.462,383.821 2276.739,383.188 C 2275.849,382.408 2274.683,381.927 2273.716,381.245 C 2271.349,379.575 2269.468,377.347 2267.401,375.342 C 2263.048,371.121 2258.478,367.179 2254.179,362.88 C 2255.738,364.684 2253.532,362.774 2253.396,362.556 C 2252.744,361.507 2251.991,360.883 2250.94,360.145 C 2251.149,360.373 2251.342,360.613 2251.52,360.865 C 2250.731,361.16 2250.357,360.366 2250.094,359.79 C 2249.668,358.858 2249.162,358.551 2248.257,358.094 C 2246.695,357.305 2244.763,354.45 2244.31,352.801 C 2244.595,353.318 2244.931,353.798 2245.32,354.241 C 2245.354,349.596 2238.652,349.843 2237.84,345.816 C 2239.468,345.923 2241.555,348.885 2242.66,349.92 C 2241.434,348.518 2240.279,346.913 2238.661,345.931 C 2236.767,344.781 2235.153,343.136 2233.23,342.073 C 2233.801,342.275 2234.304,342.587 2234.74,343.009 C 2234.237,342.534 2233.741,342.007 2233.113,341.692 C 2232.3,341.285 2232.626,340.57 2232.219,339.914 C 2231.367,338.537 2229.486,338.119 2228.259,337.178 C 2228.244,338.104 2227.323,338.49 2226.53,338.258 C 2227.833,338.931 2229.247,339.564 2230.348,340.56" id="path3282"/>
    <path d="M 2242.078,299.88 C 2244.519,300.822 2241.352,297.282 2242.078,299.88" id="path3284"/>
    <path d="M 2312.278,420.12 C 2311.705,421.56 2312.238,423.149 2311.614,424.599 C 2311.21,425.537 2310.63,427.179 2312.347,426.745 C 2311.655,425.13 2313.373,420.936 2315.518,421.417 C 2315.548,421.148 2315.525,420.884 2315.448,420.625 C 2314.288,421.089 2313.344,420.505 2312.278,420.12" id="path3296"/>
    <path d="M 2324.598,411.552 C 2323.418,411.288 2323.219,412.715 2322.606,413.357 C 2321.413,414.607 2320.141,412.945 2319.189,412.344 C 2319.075,412.899 2319.413,413.059 2319.7,413.43 C 2320.204,414.078 2319.116,414.567 2319.339,415.296 C 2319.076,415.246 2318.813,415.198 2318.549,415.152 C 2318.778,416.114 2316.673,418.254 2318.19,418.679 C 2317.052,418.845 2316.778,420.65 2317.9,421.055 C 2318.177,419.443 2320.825,418.055 2319.77,416.159 C 2321.443,416.981 2321.917,414.778 2322.94,413.999 C 2323.331,413.7 2327.388,412.68 2324.598,411.552" id="path3302"/>
    <path d="M 2320.848,425.592 C 2320.25,425.551 2319.705,425.86 2319.338,426.312 C 2319.978,427.265 2321.113,426.646 2320.848,425.592" id="path3304"/>
    <path d="M 2329.708,407.016 C 2329.296,408.363 2327.981,409.206 2328.674,410.819 C 2329.248,412.157 2332.18,406.306 2332.439,405.865 C 2331.174,405.014 2330.704,406.614 2329.708,407.016" id="path3308"/>
    <path d="M 2335.538,366.336 C 2335.992,366.73 2336.518,366.945 2337.118,366.983 C 2337.012,366.231 2336.135,365.581 2335.538,366.336" id="path3314"/>
    <path d="M 2338.198,397.872 C 2337.938,398.467 2337.707,400.366 2336.688,400.103 C 2336.856,400.407 2337.072,400.671 2337.337,400.895 C 2338.036,399.984 2338.272,398.933 2338.637,397.872 C 2338.491,397.872 2338.345,397.872 2338.198,397.872" id="path3316"/>
    <path d="M 2341.368,373.68 C 2340.369,372.293 2342.351,372.421 2342.738,371.449 C 2343.21,370.263 2341.058,366.228 2339.935,367.788 C 2339.078,368.981 2341.595,369.873 2339.169,370.678 C 2336.655,371.512 2340.756,373.116 2341.368,373.68" id="path3320"/>
    <path d="M 2339.428,390.096 C 2340.163,390.293 2340.043,390.101 2339.069,389.52 C 2339.188,389.712 2339.308,389.903 2339.428,390.096" id="path3322"/>
    <path d="M 2339.718,392.832 C 2339.982,392.856 2340.244,392.88 2340.508,392.904 C 2340.354,392.632 2340.16,392.392 2339.928,392.183 C 2339.857,392.399 2339.788,392.616 2339.718,392.832" id="path3324"/>
    <path d="M 2340.578,379.656 C 2342.281,380.703 2340.808,376.814 2339.788,376.632 C 2340.028,377.596 2339.909,378.846 2340.578,379.656" id="path3326"/>
    <path d="M 2340.068,384.336 C 2340.637,384.006 2340.938,383.416 2340.507,382.824 C 2340.273,383.303 2340.127,383.807 2340.068,384.336" id="path3328"/>
    <path d="M 2340.508,380.664 C 2340.766,380.934 2341.029,381.198 2341.298,381.456 C 2341.16,381.062 2340.896,380.798 2340.508,380.664" id="path3330"/>
    <path d="M 2342.448,366.479 C 2341.882,366.4 2341.279,366.667 2341.439,367.337 C 2341.676,368.329 2342.812,368.306 2343.597,368.495 C 2343.475,367.705 2343.052,366.998 2342.448,366.479" id="path3332"/>
    <path d="M 2325.748,303.552 C 2326.612,304.613 2327.036,303.227 2327.561,302.537 C 2328.187,301.716 2329.066,300.812 2330.207,301.032 C 2330.173,300.237 2328.073,297.856 2327.249,298.317 C 2326.401,298.792 2325.255,299.858 2324.228,299.807 C 2325.414,299.742 2326.761,302.782 2325.748,303.552" id="path3354"/>
    <path d="M 2371.248,329.616 C 2370.477,329.67 2369.668,329.295 2368.948,329.76 C 2371.913,331.989 2375.399,333.75 2378.958,334.8 C 2377.942,333.536 2376.438,333.131 2375.073,332.384 C 2374.354,331.99 2373.562,331.706 2372.908,331.2 C 2372.303,330.733 2371.979,329.934 2371.248,329.616" id="path3356"/>
    <path d="M 2382.698,333.432 C 2384.582,335.065 2387.352,335.617 2389.677,336.312 C 2387.75,335.009 2385.088,333.477 2382.698,333.432" id="path3358"/>
    <path d="M 2058.338,188.64 C 2058.987,188.334 2059.663,188.08 2060.278,187.704 C 2058.387,186.438 2056.308,185.466 2054.017,186.264 C 2054.179,188.017 2057.046,188.325 2058.338,188.64" id="path3220"/>
    <path d="M 2018.809,178.632 C 2019.531,178.632 2020.262,178.679 2020.968,178.488 C 2019.969,178 2020.697,176.993 2019.488,176.803 C 2018.114,176.588 2016.979,175.831 2015.639,175.535 C 2016.382,176.792 2017.656,177.773 2018.809,178.632" id="path3212"/>
    <path d="M 2024.428,177.12 C 2025.28,176.513 2025.981,177.455 2026.542,177.96 C 2027.23,178.579 2028.258,178.167 2028.958,178.776 C 2028.159,179.048 2027.368,178.526 2026.588,178.92 C 2028.283,179.991 2030.321,180.488 2032.268,180.864 C 2034.284,181.253 2036.133,182.265 2038.159,182.586 C 2040.136,182.899 2042.739,183.64 2044.723,183.475 C 2046.189,183.354 2044.194,181.725 2043.869,181.369 C 2045.167,181.33 2046.159,182.154 2047.39,182.377 C 2048.608,182.598 2049.896,182.037 2051.07,181.801 C 2053.223,181.368 2056.137,180.426 2058.19,181.657 C 2057.669,181.507 2057.142,181.483 2056.61,181.585 C 2057.714,182.079 2059.457,182.628 2060.455,181.939 C 2061.511,181.209 2063.992,182.247 2065.391,181.513 C 2062.86,180.523 2060.034,180.996 2057.486,180.071 C 2056.206,179.606 2054.84,179.396 2053.595,178.828 C 2052.386,178.276 2051.324,177 2049.91,176.977 C 2051.001,176.966 2050.977,176.871 2049.84,176.69 C 2050.117,176.741 2052.657,176.789 2050.71,176.401 C 2051.629,176.449 2056.416,176.255 2054.09,178.057 C 2057.535,179.451 2061.069,180.724 2064.817,180.902 C 2065.775,180.947 2067.416,181.069 2068.11,180.203 C 2069.141,178.919 2066.165,178.888 2065.831,178.201 C 2066.589,178.029 2067.287,177.662 2068.061,177.553 C 2065.745,176.899 2063.33,176.801 2060.996,176.252 C 2059.182,175.826 2056.828,174.789 2054.966,175.468 C 2050.562,177.074 2045.857,173.859 2041.491,173.522 C 2042.259,173.838 2043.053,174.093 2043.801,174.458 C 2042.908,174.248 2041.451,173.741 2040.771,174.674 C 2041.811,175.786 2043.362,176.142 2044.591,176.978 C 2043.811,176.978 2042.976,177.153 2042.211,176.906 C 2041.41,176.647 2040.864,175.984 2040.051,175.755 C 2039.269,175.534 2038.47,175.816 2037.681,175.683 C 2036.896,175.551 2036.331,174.985 2035.591,174.747 C 2033.848,174.189 2031.966,174.352 2030.228,173.805 C 2029.722,173.645 2025.464,172.409 2027.091,174.028 C 2026.817,173.829 2025.175,173.13 2025.515,174.057 C 2025.716,174.605 2024.207,174.356 2023.939,174.592 C 2023.574,174.913 2026.112,175.988 2026.372,176.115 C 2025.572,175.909 2024.683,175.844 2023.992,176.402 C 2024.135,176.64 2024.281,176.88 2024.428,177.12" id="path3214"/>
    <path d="M 2028.958,186.264 C 2030.893,187.931 2033.763,187.63 2035.948,188.785 C 2035.782,186.831 2030.373,186.422 2028.958,186.264" id="path3216"/>
    <path d="M 2043.148,176.76 C 2042.601,176.609 2042.048,176.489 2041.488,176.4 C 2041.993,176.742 2042.547,176.862 2043.148,176.76" id="path3218"/>
    <path d="M 2060.139,191.735 C 2059.627,191.701 2059.124,191.749 2058.628,191.88 C 2060.032,193.008 2061.725,192.043 2063.24,191.793 C 2065.096,191.487 2067.062,192.261 2068.874,192.563 C 2072.686,193.198 2076.874,193.583 2080.728,193.249 C 2076.425,190.668 2071.566,189.047 2066.561,188.573 C 2065.799,188.501 2062.44,188.388 2061.771,189.111 C 2061.356,189.559 2062.25,190.432 2062.167,190.974 C 2062.057,191.715 2060.651,191.647 2060.139,191.735" id="path3222"/>
    <path d="M 2066.688,205.272 C 2065.374,204.426 2063.455,204.53 2062.007,204.912 C 2062.543,205.616 2063.367,205.687 2064.167,205.848 C 2062.366,205.332 2064.304,205.578 2064.616,205.654 C 2065.336,205.826 2066.257,206.114 2066.688,205.272" id="path3224"/>
    <path d="M 2068.128,204.264 C 2068.392,204.216 2068.654,204.168 2068.918,204.12 C 2068.4,203.896 2067.874,203.872 2067.338,204.048 C 2067.602,204.12 2067.864,204.191 2068.128,204.264" id="path3226"/>
    <path d="M 2071.658,169.992 C 2072.012,170.866 2074.679,170.094 2075.398,170.208 C 2075.497,168.875 2072.226,169.569 2071.658,169.992" id="path3228"/>
    <path d="M 2095.348,182.735 C 2097.215,182.735 2099.756,182.756 2101.248,181.44 C 2100.209,181.127 2100.949,179.896 2099.595,180.153 C 2098.121,180.432 2096.802,179.919 2095.342,179.69 C 2093.95,179.472 2092.44,179.246 2091.029,179.28 C 2089.405,179.319 2088.116,179.956 2086.529,179.224 C 2084.887,178.467 2081.368,177.296 2079.67,178.414 C 2077.177,180.055 2075.308,176.715 2072.879,177.479 C 2073.856,177.776 2074.239,179.27 2075.437,179.734 C 2077.29,180.453 2079.674,180.581 2081.629,180.999 C 2085.968,181.927 2090.905,182.735 2095.348,182.735" id="path3230"/>
    <path d="M 2098.298,173.664 C 2098.562,173.712 2098.824,173.76 2099.088,173.808 C 2098.725,173.238 2098.09,172.946 2097.428,173.161 C 2097.718,173.328 2098.008,173.496 2098.298,173.664" id="path3232"/>
    <path d="M 1695.748,151.92 C 1696.86,152.941 1699.014,152.28 1700.348,152.28 C 1698.054,151.169 1695.115,152.493 1692.939,151.416 C 1693.711,152.016 1694.822,152.08 1695.748,151.92" id="path3068"/>
    <path d="M 1700.428,143.784 C 1700.308,144.024 1700.189,144.263 1700.069,144.504 C 1702.072,144.767 1704.326,144.93 1706.258,144.216 C 1704.625,143.035 1702.262,143.92 1700.428,143.784" id="path3072"/>
    <path d="M 1749.598,142.416 C 1750.813,143.889 1755.195,143.722 1756.729,142.92 C 1755.161,141.306 1751.54,142.187 1749.598,142.416" id="path3106"/>
    <path d="M 1975.608,189.072 C 1975.171,188.64 1974.567,188.382 1973.948,188.424 C 1974.386,188.853 1974.99,189.111 1975.608,189.072" id="path3202"/>
    <path d="M 1985.038,191.304 C 1984.645,190.979 1984.262,190.643 1983.889,190.296 C 1983.986,190.892 1984.429,191.296 1985.038,191.304" id="path3204"/>
    <path d="M 1993.389,191.231 C 1994.47,191.328 1995.55,191.489 1996.639,191.448 C 1995.573,191.263 1994.469,190.917 1993.389,191.231" id="path3206"/>
    <path d="M 2006.639,195.479 C 2006.879,195.479 2007.118,195.479 2007.359,195.479 C 2006.86,195.206 2006.333,195.134 2005.779,195.263 C 2006.065,195.336 2006.352,195.407 2006.639,195.479" id="path3208"/>
    <path d="M 2009.309,195.84 C 2008.86,195.591 2008.379,195.495 2007.869,195.552 C 2008.336,195.709 2008.816,195.805 2009.309,195.84" id="path3210"/>
    <path d="M 1957.248,193.68 C 1956.752,193.635 1956.319,193.78 1955.948,194.113 C 1956.707,194.923 1958.158,194.783 1958.908,194.041 C 1958.365,193.87 1957.813,193.75 1957.248,193.68" id="path3200"/>
    <path d="M 1889.208,170.496 C 1890.251,171.262 1891.918,171.447 1893.168,171.576 C 1891.855,170.766 1890.785,170.202 1889.208,170.496" id="path3190"/>
    <path d="M 1896.628,172.08 C 1897.176,172.144 1897.727,172.168 1898.278,172.151 C 1897.731,172.047 1897.204,171.879 1896.698,171.647 C 1896.675,171.792 1896.651,171.936 1896.628,172.08" id="path3194"/>
    <path d="M 1912.468,187.272 C 1912.477,186.395 1914.907,185.355 1912.35,184.71 C 1910.129,184.149 1907.612,184.408 1905.338,184.175 C 1906.127,185.053 1904.897,185.523 1904.189,185.399 C 1905.059,187.242 1910.834,187.32 1912.468,187.272" id="path3196"/>
    <path d="M 1927.148,185.903 C 1927.415,185.927 1927.681,185.951 1927.948,185.975 C 1927.684,185.744 1927.394,185.552 1927.078,185.399 C 1927.102,185.567 1927.125,185.735 1927.148,185.903" id="path3198"/>
    <path d="M 1797.768,175.392 C 1797.505,175.44 1797.242,175.489 1796.979,175.537 C 1797.281,175.636 1797.544,175.588 1797.768,175.392" id="path3152"/>
    <path d="M 1807.418,166.68 C 1805.95,167.287 1804.207,166.958 1802.809,167.76 C 1803.583,168.197 1805.037,167.907 1805.937,168.061 C 1806.849,168.216 1809.15,167.533 1807.418,166.68" id="path3154"/>
    <path d="M 1803.748,169.848 C 1803.481,169.92 1803.215,169.992 1802.948,170.064 C 1803.463,170.247 1803.993,170.319 1804.538,170.281 C 1804.274,170.136 1804.012,169.992 1803.748,169.848" id="path3156"/>
    <path d="M 1804.828,172.8 C 1804.588,172.872 1804.349,172.944 1804.108,173.016 C 1804.644,173.187 1805.195,173.284 1805.757,173.304 C 1805.476,173.085 1805.165,172.917 1804.828,172.8" id="path3158"/>
    <path d="M 1806.628,170.063 C 1806.388,169.991 1806.149,169.919 1805.908,169.847 C 1806.359,170.282 1806.863,170.353 1807.418,170.063 C 1807.154,170.063 1806.892,170.063 1806.628,170.063" id="path3160"/>
    <path d="M 1808.498,168.479 C 1807.974,168.335 1807.443,168.312 1806.908,168.408 C 1807.427,168.708 1807.956,168.731 1808.498,168.479" id="path3164"/>
    <path d="M 1810.518,169.848 C 1809.709,169.857 1808.956,170.231 1808.139,170.136 C 1808.67,170.415 1810.451,170.9 1810.518,169.848" id="path3166"/>
    <path d="M 1813.968,173.808 C 1813.663,173.643 1813.377,173.451 1813.109,173.232 C 1813.684,172.843 1814.331,173.012 1814.758,173.521 C 1814.207,170.999 1809.628,173.626 1808.139,172.657 C 1809.551,174.17 1812.214,173.134 1813.968,173.808" id="path3168"/>
    <path d="M 1815.188,169.704 C 1814.712,169.438 1814.209,169.367 1813.677,169.488 C 1814.175,169.604 1814.679,169.676 1815.188,169.704" id="path3172"/>
    <path d="M 1815.559,173.376 C 1815.822,173.232 1816.085,173.088 1816.348,172.944 C 1814.241,173.765 1816.986,172.097 1815.399,172.256 C 1814.629,172.334 1815.179,173.215 1815.559,173.376" id="path3174"/>
    <path d="M 1816.639,170.712 C 1816.183,170.274 1815.653,169.962 1815.048,169.776 C 1815.229,170.577 1815.943,170.707 1816.639,170.712" id="path3176"/>
    <path d="M 1818.938,173.808 C 1818.158,173.915 1817.419,174.205 1816.638,174.312 C 1817.424,174.737 1818.436,174.589 1818.938,173.808" id="path3180"/>
    <path d="M 1762.348,149.184 C 1763.173,149.584 1763.949,150.117 1764.87,150.274 C 1766.416,150.539 1767.954,150.166 1769.478,150.697 C 1768.976,150.801 1768.469,150.873 1767.957,150.913 C 1770.226,151.668 1772.881,151.234 1775.017,150.265 C 1773.966,148.669 1771.177,149.121 1769.573,149.094 C 1766.773,149.047 1763.847,148.945 1761.048,148.754 C 1761.521,148.778 1761.953,148.922 1762.348,149.184" id="path3112"/>
    <path d="M 1762.559,147.528 C 1765.001,147.287 1768.26,146.464 1770.559,147.672 C 1768.774,147.245 1766.883,147.382 1765.079,147.6 C 1766.368,147.858 1767.705,147.858 1768.969,148.248 C 1767.944,148.157 1766.899,147.988 1765.879,148.248 C 1767.98,148.189 1770.039,148.735 1772.107,149.02 C 1774.605,149.365 1776.403,148.682 1778.792,148.281 C 1780.689,147.963 1782.72,147.879 1784.641,148.007 C 1786.049,148.1 1788.448,148.505 1789.491,147.24 C 1788.553,147.32 1786.286,146.444 1786.111,145.368 C 1787.147,145.529 1788.5,145.617 1789.421,145.008 C 1786.611,143.988 1783.462,144.511 1780.716,143.259 C 1778.336,142.174 1775.067,141.278 1772.501,142.2 C 1773.314,142.304 1774.137,142.25 1774.951,142.344 C 1771.174,142.344 1767.607,143.778 1763.93,144.144 C 1764.183,145.372 1765.921,144.691 1766.741,145.008 C 1766.075,145.234 1765.922,145.734 1766.451,146.232 C 1765.449,146.232 1764.314,146.056 1763.43,146.665 C 1764.508,146.777 1765.611,146.684 1766.67,146.953 C 1765.182,146.953 1763.006,146.317 1761.77,147.457 C 1762.031,147.48 1762.295,147.504 1762.559,147.528" id="path3114"/>
    <path d="M 1766.809,151.272 C 1767.072,151.272 1767.335,151.272 1767.598,151.272 C 1766.684,150.66 1765.46,150.846 1764.428,150.767 C 1765.153,151.199 1765.996,151.209 1766.809,151.272" id="path3122"/>
    <path d="M 1769.479,152.784 C 1770.284,152.862 1771.139,153.055 1771.918,152.712 C 1771.105,152.422 1770.296,152.819 1769.479,152.784" id="path3136"/>
    <path d="M 1773.218,152.856 C 1773.722,152.935 1774.225,152.935 1774.729,152.856 C 1774.216,152.605 1773.712,152.605 1773.218,152.856" id="path3144"/>
    <path d="M 1779.048,149.184 C 1778.478,149.256 1777.951,149.449 1777.468,149.76 C 1778.269,149.901 1779.076,150.041 1779.838,150.336 C 1777.744,150.793 1775.824,151.842 1773.649,151.92 C 1774.409,152.114 1775.188,152.312 1775.965,152.413 C 1776.051,152.424 1780.584,152.06 1778.399,152.496 C 1779.027,152.724 1779.582,152.499 1780.204,152.433 C 1781.166,152.33 1781.773,153.185 1782.574,153.555 C 1784.512,154.45 1786.218,154.924 1788.335,154.696 C 1789.474,154.574 1790.306,154.965 1791.381,155.209 C 1792.663,155.5 1794.183,155.376 1795.491,155.376 C 1796.61,155.376 1797.72,155.615 1798.808,155.846 C 1799.288,155.948 1802.518,156.212 1800.508,156.456 C 1802.555,156.456 1804.53,156.596 1806.576,156.504 C 1808.094,156.435 1810.556,157.046 1811.517,155.448 C 1809.519,154.439 1807.148,154.534 1805.188,153.431 C 1806.01,153.458 1806.812,153.856 1807.638,153.576 C 1807.173,152.388 1806.075,151.743 1805.327,150.767 C 1805.64,150.691 1805.952,150.619 1806.266,150.551 C 1803.748,149.112 1799.632,148.107 1796.827,149.183 C 1797.422,149.114 1797.601,149.444 1797.407,149.975 C 1796.605,149.816 1795.763,149.669 1794.957,149.903 C 1795.462,150.883 1794.206,150.884 1793.597,150.84 C 1794.766,150.784 1794.121,149.408 1794.747,148.824 C 1793.49,148.804 1792.341,148.176 1791.08,148.176 C 1789.372,148.176 1787.692,148.574 1785.985,148.553 C 1784.064,148.528 1780.779,148.039 1779.048,149.184" id="path3146"/>
    <path d="M 1806.479,150.983 C 1806.89,151.79 1807.294,151.741 1807.918,151.2 C 1807.429,151.189 1806.949,151.117 1806.479,150.983" id="path3162"/>
    <path d="M 1816.198,159.12 C 1816.318,159.24 1816.438,159.359 1816.558,159.479 C 1815.937,159.962 1815.505,160.643 1815.188,161.352 C 1817.888,162.096 1820.026,161.537 1822.642,160.841 C 1825.298,160.136 1827.888,160.413 1830.58,160.396 C 1832.027,160.386 1833.702,160.013 1835.208,159.912 C 1837.626,159.75 1839.97,159.419 1842.118,158.328 C 1840.976,157.256 1839.389,156.618 1837.873,156.312 C 1837.187,156.173 1836.483,156.355 1835.793,156.293 C 1834.765,156.2 1833.687,155.461 1832.677,155.193 C 1831.285,154.823 1829.164,153.892 1827.752,154.505 C 1826.87,154.888 1825.462,156.628 1824.408,156.239 C 1824.879,155.176 1823.472,154.286 1824.408,153.287 C 1823.549,153.281 1822.717,152.936 1821.879,152.785 C 1821.252,152.673 1818.363,153.135 1820.309,153.791 C 1819.263,153.818 1818.424,153.108 1817.416,153.047 C 1816.727,153.005 1816.59,154.087 1815.908,154.295 C 1816.247,155.219 1817.227,155.063 1817.998,155.231 C 1817.531,155.161 1816.538,155.259 1816.197,155.628 C 1815.837,156.019 1818.329,156.81 1818.578,156.887 C 1818.176,156.799 1815.588,156.227 1815.504,157.005 C 1815.424,157.733 1816.039,158.438 1816.198,159.12" id="path3178"/>
    <path d="M 1847.518,160.416 C 1848.047,160.283 1848.554,160.092 1849.039,159.84 C 1848.226,159.156 1847.538,160.192 1846.659,159.912 C 1846.944,160.08 1847.231,160.248 1847.518,160.416" id="path3182"/>
    <path d="M 1854.868,161.28 C 1855.108,161.135 1855.347,160.992 1855.588,160.847 C 1853.759,159.711 1851.243,160.627 1849.248,160.559 C 1851.021,161.341 1853.173,160.283 1854.868,161.28" id="path3184"/>
    <path d="M 1855.368,165.528 C 1855.889,165.582 1856.392,165.509 1856.878,165.311 C 1856.356,164.945 1855.806,164.897 1855.229,165.167 C 1855.275,165.288 1855.321,165.408 1855.368,165.528" id="path3186"/>
    <path d="M 1860.768,166.536 C 1861.294,166.44 1861.822,166.344 1862.348,166.248 C 1861.632,165.796 1860.785,165.817 1859.979,165.744 C 1861.012,166.066 1861.035,166.282 1860.048,166.391 C 1860.288,166.439 1860.527,166.488 1860.768,166.536" id="path3188"/>
    <path d="M 1605.238,227.664 C 1606.109,228.081 1607.641,227.442 1608.478,227.16 C 1607.354,227.027 1606.254,227.128 1605.238,227.664" id="path3016"/>
    <path d="M 1624.468,175.032 C 1624.977,175.236 1625.692,174.921 1626.118,174.672 C 1625.579,174.841 1625.028,174.961 1624.468,175.032" id="path3042"/>
    <path d="M 1625.908,175.104 C 1626.181,175.121 1626.444,175.073 1626.698,174.959 C 1626.435,175.008 1626.172,175.057 1625.908,175.104" id="path3044"/>
    <path d="M 1627.778,174.168 C 1628.072,174.166 1628.359,174.118 1628.638,174.024 C 1628.352,174.072 1628.065,174.12 1627.778,174.168" id="path3046"/>
    <path d="M 1628.428,223.92 C 1628.93,223.945 1629.44,223.912 1629.939,223.992 C 1629.445,223.762 1628.941,223.738 1628.428,223.92" id="path3048"/>
    <path d="M 1629.798,173.521 C 1630.371,174.191 1631.493,173.416 1632.098,173.232 C 1631.324,173.273 1630.571,173.479 1629.798,173.521" id="path3050"/>
    <path d="M 1633.678,212.472 C 1633.902,212.273 1634.169,212.201 1634.479,212.256 C 1633.949,212.091 1633.419,212.091 1632.889,212.256 C 1633.151,212.328 1633.415,212.399 1633.678,212.472" id="path3054"/>
    <path d="M 1563.559,148.896 C 1564.062,149.055 1564.566,149.055 1565.069,148.896 C 1564.565,148.651 1564.062,148.651 1563.559,148.896" id="path2944"/>
    <path d="M 1576.008,147.096 C 1576.523,147.299 1577.053,147.347 1577.598,147.241 C 1577.075,147.108 1576.545,147.061 1576.008,147.096" id="path2952"/>
    <path d="M 1586.448,146.808 C 1586.904,147.029 1587.647,146.773 1588.028,146.521 C 1587.445,146.302 1586.918,146.397 1586.448,146.808" id="path2966"/>
    <path d="M 1589.548,146.735 C 1589.798,146.755 1590.038,146.707 1590.268,146.591 C 1590.027,146.64 1589.788,146.688 1589.548,146.735" id="path2976"/>
    <path d="M 1592.858,146.808 C 1593.671,146.996 1594.558,147.121 1595.378,146.88 C 1594.542,146.801 1593.697,146.682 1592.858,146.808" id="path2982"/>
    <path d="M 1595.809,146.231 C 1596.07,146.291 1596.332,146.291 1596.598,146.231 C 1596.335,146.231 1596.071,146.231 1595.809,146.231" id="path2988"/>
    <path d="M 1596.168,142.56 C 1597.248,143.196 1599.129,142.968 1600.128,142.273 C 1598.802,142.272 1597.488,142.461 1596.168,142.56" id="path2990"/>
    <path d="M 1600.198,143.28 C 1601.382,143.67 1603.124,144.248 1604.228,143.352 C 1602.976,143.123 1601.393,142.517 1600.198,143.28" id="path3000"/>
    <path d="M 1600.268,140.328 C 1600.868,140.546 1601.444,140.498 1601.998,140.183 C 1601.414,140.147 1600.838,140.196 1600.268,140.328" id="path3002"/>
    <path d="M 1601.568,142.776 C 1602.102,142.956 1602.629,142.932 1603.148,142.704 C 1602.62,142.69 1602.094,142.714 1601.568,142.776" id="path3006"/>
    <path d="M 1605.028,145.656 C 1605.297,145.637 1605.56,145.589 1605.818,145.511 C 1605.555,145.56 1605.292,145.608 1605.028,145.656" id="path3012"/>
    <path d="M 1613.158,148.104 C 1613.688,148.188 1614.218,148.188 1614.748,148.104 C 1614.218,147.938 1613.688,147.938 1613.158,148.104" id="path3032"/>
    <path d="M 1616.328,140.256 C 1616.867,140.581 1617.421,140.606 1617.988,140.328 C 1617.437,140.26 1616.884,140.236 1616.328,140.256" id="path3034"/>
    <path d="M 1617.768,139.824 C 1619.663,140.121 1623.289,141.402 1624.899,139.824 C 1622.637,139.271 1620.031,139.297 1617.768,139.824" id="path3036"/>
    <path class="landxx ru" d="M 1439.558,337.01842 C 1440.6671,336.23877 1441.5027,335.29331 1441.6367,333.90202 C 1441.7114,333.11468 1441.6378,332.79623 1442.5284,332.68752 C 1443.722,332.54148 1445.0035,332.66446 1446.0488,331.95839 C 1446.8955,331.38628 1447.4204,329.71497 1448.5745,329.66556 C 1447.5807,330.59565 1446.342,331.46863 1445.6535,332.66995 C 1448.1846,332.66995 1450.0745,334.05904 1449.9383,331.03818 C 1449.8175,328.36541 1453.3896,330.72742 1454.334,331.15128 C 1456.4401,332.09675 1459.3688,330.74169 1461.0708,332.59089 C 1461.7615,333.34089 1462.3951,333.49902 1461.5177,334.47193 C 1460.7853,335.28452 1461.0115,337.24902 1461.8614,337.88811 C 1454.8325,338.61615 1447.741,338.0649 1440.745,337.2556 C 1441.3841,336.99316 1441.5774,336.28818 1442.1726,335.9906 C 1443.039,335.55685 1443.957,335.82039 1444.6213,334.96278 C 1443.7637,334.89909 1442.86,334.775 1442.0013,334.86834 C 1441.1063,334.96607 1440.7407,336.84711 1439.558,337.01842" id="path2922"/>
    <path class="landxx ru" transform="translate(29.9017, 45.0745)" d="M 1507.4485,238.42619 c -0.3637,0.0788 -1.2135,0.80137 -1.3619,1.09831 -0.171,0.34189 -0.1623,0.47642 -0.4394,0.61505 -0.4269,0.21349 -0.4841,0.26358 -1.0104,0.26358 -0.2732,0 -0.5453,0 -0.8347,0 -0.2439,0 -0.581,0.052 -0.7908,-0.0879 -0.1842,-0.12289 -0.4878,-0.18027 -0.6151,-0.30754 -0.089,-0.0886 -0.08,-0.57895 0,-0.65899 0.219,-0.21897 0.5645,-0.30089 0.7469,-0.48325 0.1322,-0.13228 0.1335,-0.52891 0.1758,-0.57113 0.053,-0.053 0.1696,-0.36383 0.088,-0.52719 -0.1845,-0.36871 -0.6844,-0.60921 -0.9226,-0.96651 -0.2393,-0.35892 -0.5663,-0.91779 -0.8787,-1.2301 -0.2315,-0.23175 -2.0259,-0.17824 -2.3723,-0.35146 -0.1055,-0.0527 -0.2825,0.0376 -0.3954,0 -0.2735,-0.0911 -0.5659,-0.14147 -0.8786,-0.21965 -0.1003,-0.0252 -0.26,0.0457 -0.3515,0 -0.4323,-0.21613 -0.603,-0.38338 -0.9226,-0.70292 -0.082,-0.0821 -0.2254,-0.0497 -0.3074,-0.13179 -0.019,-0.0206 0.019,-0.0671 0,-0.0879 -0.076,-0.0758 -0.2886,-0.0248 -0.3514,-0.0879 -0.072,-0.0715 -0.2477,-0.11601 -0.3515,-0.21965 -0.068,-0.0678 -0.071,-0.15927 -0.1758,-0.26361 -0.2541,-0.62835 -0.2048,-1.25928 -0.615,-1.66943 -0.098,-0.0978 -0.013,-0.31402 -0.1758,-0.39539 -0.1896,-0.0948 -0.1887,-0.3774 -0.2635,-0.52719 -0.016,-0.0303 -0.1477,-0.17572 -0.1319,-0.17572 0.5148,0 0.7724,0.73969 1.1862,-0.0879 0.1412,-0.28251 -0.2242,-0.53164 -0.3074,-0.61505 -0.029,-0.0303 -0.047,-0.96357 -0.044,-0.96651 0.1319,-0.13179 0.3734,0.29654 0.3954,0.30754 0.2325,0.11631 0.5576,0.25909 0.8347,0.35146 0.2429,0.081 0.4782,-0.13518 0.7468,0.0439 0.1578,0.10521 0.4638,0.41976 0.5712,0.52718 0.053,0.0532 2.5363,0.0997 2.548,0.0879 0.026,-0.0255 0.4067,-0.17571 0.4833,-0.17571 0.3355,-0.41806 0.2197,-1.01054 -0.044,-1.40587 -0.1819,-0.27287 -0.2235,-0.61888 -0.4393,-0.83472 -0.092,-0.0918 -0.8013,0 -0.9226,0 -0.3051,0 -0.6835,-0.0375 -0.9665,-0.13179 -0.4423,-0.14746 -0.8133,0.1425 -1.274,-0.0879 -0.07,-0.0352 -0.1932,0.0352 -0.2635,0 -0.2503,-0.12514 -0.9142,-0.82628 -1.1423,-1.05437 -0.2061,-0.20604 -0.5509,-0.1116 -0.7468,-0.30754 -0.3227,-0.3226 -0.9226,-0.86787 -0.9226,-1.44977 0,-0.16308 -0.4491,-0.6492 -0.2636,-0.83472 0.067,-0.0671 1.2838,0.61506 1.5377,0.61506 0.1287,0 0.7263,0.39128 0.9665,0.43932 0.4991,0.0998 2.6637,0.81855 2.8995,1.05438 0.092,0.0921 0.8425,0.17768 1.0105,0.21965 0.3816,0.0954 0.6741,0.28725 1.0544,0.43932 0.1467,0.0586 0.3872,0.0177 0.5271,0.0879 0.08,0.0402 0.2239,0.15582 0.2636,0.17572 0.2802,0.14008 0.6208,0.0918 0.8786,0.26358 0.1377,0.0918 0.1716,0.2111 0.2197,0.30754 0.044,0.0881 0.3603,0.40423 0.4393,0.48325 0.1,0.0998 -0.1887,1.33864 -0.2197,1.4937 -0.042,0.2072 0.1013,0.80778 0.044,0.92258 -0.1638,0.32748 -0.4556,0.74184 -0.3074,1.18618 0.091,0.27387 0.2371,0.38633 0.3515,0.61505 0.023,0.0469 -0.023,0.12885 0,0.17572 0.1235,0.247 0.4248,0.60056 0.659,0.83472 0.3116,0.3117 0.7538,0.37691 1.1422,0.57112 0.5746,0.28731 1.2886,1.06889 1.7134,1.4937 0.088,0.0882 0.1374,0.26906 0.2196,0.35146 0.1874,0.18736 0.5812,0.3715 0.7469,0.70291 0.1058,0.21191 0.1022,1.03984 -0.044,1.18618 -0.1651,0.16527 0.04,0.48536 -0.3954,0.70292 z m -24.6792,-0.93419 c -0.2897,-0.18982 -1.6975,-0.13424 -1.9261,-0.24852 -0.3821,-0.19107 -0.7013,-0.34265 -1.2425,-0.55916 -0.5362,-0.21446 -1.0777,-0.45635 -1.4291,-0.80769 -0.052,-0.0518 -0.1967,-0.0103 -0.2486,-0.0621 -0.239,-0.23913 -0.5325,-0.67007 -0.8077,-0.80769 -0.4187,-0.20933 -0.8159,-0.50525 -1.1805,-0.86982 -0.013,-0.0135 -1.3766,-0.27248 -1.4911,-0.31063 -0.2264,-0.0754 -1.0834,-0.45239 -1.2426,-0.37278 -0.1538,0.0769 -0.067,0.32054 -0.1242,0.43491 -0.2654,0.5306 -0.7143,1.00933 -1.1183,1.61537 -0.088,0.13231 -0.1493,0.0582 -0.2487,0.12428 -0.2274,0.15165 -0.4012,0.38699 -0.6213,0.49703 -0.4636,0.23182 -1.244,0.24994 -1.5532,0.55917 -0.041,0.0407 -0.032,0.21739 -0.062,0.24852 -0.3029,0.30289 -0.083,0.35263 -0.2487,0.68343 -0.3357,0.67145 0.5123,1.0562 1.1184,1.0562 0.3219,0.23842 1.8898,0.9826 1.9266,1.05617 0.1186,0.23713 0.3238,0.39921 0.4349,0.6213 0.037,0.0741 -0.046,0.17959 0,0.24852 0.1658,0.24851 0.1035,-0.14499 0.1864,0.18639 0.2177,0.87136 0.1155,1.35794 0.6834,1.92602 0.2271,0.22704 0.527,0.26352 0.8698,0.43491 0.1629,0.0814 0.4479,-0.0578 0.6213,0 0.1578,0.0526 0.4544,0.39219 0.6214,0.55917 0.1957,0.19568 0.3687,1.2346 0.497,1.49111 0.1974,0.39511 0.6221,0.68503 0.8077,1.05621 0.1154,0.23103 0.1903,1.06394 0.2486,1.18047 0.086,0.17268 0.4467,0.57094 0.5592,0.68342 0.052,0.0517 0.4142,0.0621 0.497,0.0621 0.4729,0 0.9261,-0.30479 1.2426,-0.62129 0.077,-0.0771 0.068,-0.25461 0.1242,-0.31064 0.071,-0.0712 0.3833,-0.26951 0.4349,-0.37278 0.049,-0.0988 0.053,-0.488 0.1242,-0.55916 0.2448,-0.24481 0.4579,-0.69492 0.6834,-0.80769 0.199,-0.0995 0.5284,-0.0621 0.8077,-0.0621 0.2067,0 0.8234,-0.17069 0.9941,0 0.3356,0.33557 0.7276,0.97406 1.3047,1.11833 0.2799,0.07 0.5867,0.03 0.8698,0.12428 0.3084,0.10272 0.7178,0.076 0.9941,-0.0621 0.3795,-0.18975 0.6579,-1.37791 0.8698,-1.80176 0.1123,-0.22462 0.023,-1.03784 0.1865,-1.36685 0.2122,-0.42418 0.5911,-0.26452 0.9319,-0.43491 0.099,-0.0494 0.1335,-0.19572 0.1864,-0.24852 0.3432,-0.34314 -0.066,-1.27994 -0.2483,-1.55324 -0.1974,-0.29593 -0.8554,-0.7025 -1.1184,-0.80769 -0.1171,-0.0468 -0.2599,-0.006 -0.3727,-0.0621 -0.1452,-0.0726 -0.3643,-0.30216 -0.4971,-0.43491 -0.067,-0.067 -0.9179,-0.52107 -0.9941,-0.55917 -0.3464,-0.17323 -0.7793,-1.2622 -0.9319,-1.49111 -0.1358,-0.20381 -0.4742,-0.16356 -0.5592,-0.24852 -0.1954,-0.19555 -0.1738,-0.20133 -0.5591,-0.12427 z m 328.0392,-119.05989 c -1.3319,0.005 -2.5441,0.27671 -3.9309,0.50396 -1.526,0.251 -2.8139,1.7879 -4.4349,1.5119 0.439,0.073 0.8812,0.18338 1.3103,0.30238 -0.665,0.493 -1.3559,0.9121 -2.0159,1.41111 0.23,0.25 0.4556,0.47655 0.7056,0.70555 -1.461,-0.421 -1.0884,1.1261 0.2016,1.41111 0.722,0.15999 1.6999,0.24152 1.5119,1.20952 -0.045,0.23079 1.9187,0.62313 3.9309,0.90713 -0.4148,-0.0383 -0.7928,-0.0227 -1.2095,0 -0.956,0.052 -2.4065,-0.62617 -3.2254,-0.40317 -2.966,0.812 -6.1632,0.40279 -9.1722,0.1008 2.098,1.946 5.4964,0.9818 7.5595,3.02379 -3.727,-0.254 -6.1137,-1.69855 -9.8777,-0.70555 2.438,0.436 -1.2239,0.39996 -1.5119,0.50397 0.519,0.071 1.0169,0.23517 1.5119,0.40317 -0.798,0.306 -1.8764,0.47138 -2.7215,0.30238 -0.982,-0.197 -1.6575,0.57596 -2.6206,0.50397 -0.941,-0.071 -1.5228,-0.86597 -2.5198,-0.50397 -0.777,0.281 -1.599,0.50976 -2.4191,0.60476 0.4441,-0.501 0.9489,-0.94831 1.5119,-1.31031 -4.178,-1.304 -8.5557,0.22662 -12.8007,-0.30238 0.5,0.239 1.0587,0.30138 1.6127,0.30238 -1.455,0.583 -3.0801,-0.45821 -4.3341,0.10079 -1.159,0.517 0.436,1.19472 1.0079,1.10873 -2.047,-0.313 1.3022,-0.35038 1.8143,-0.30238 1.047,0.099 2.0855,0.10421 3.1246,-0.1008 -2.618,1.275 -5.8183,1.34891 -8.6682,1.5119 -3.306,0.19 -6.6685,0.57414 -9.9786,0.90714 0.771,0.708 -1.4457,0.92593 -2.1166,1.00793 -0.316,0.04 -3.7175,-0.0942 -1.7135,0.1008 -1.594,-0.048 -2.3503,1.55886 -3.7294,2.01586 -1.764,0.584 -3.6297,-0.72759 -5.4428,-0.20158 0.747,0.692 4.4041,0.0771 4.3341,1.4111 -0.818,-0.074 -1.566,-0.63417 -2.419,-0.40317 1.466,0.203 -0.3087,0.96027 0.8063,1.81428 -1.29,-0.455 -0.7294,0.17376 -0.3024,0.60476 -1.832,0.933 -3.4929,-1.55152 -5.342,-1.20952 1.997,0.617 -1.1688,0.41752 0.4032,1.20952 0.475,0.239 1.0927,0.32618 1.6127,0.40317 1.0769,0.159 2.1423,0.11559 3.2253,0.20159 -2.0609,0.087 0.8376,0.55376 1.2096,0.60476 0.5769,0.08 1.2784,0.15776 1.7134,0.60476 -0.447,-0.015 -1.2994,0.74317 -1.7134,0.40317 -0.295,-0.262 -0.6291,-0.49676 -1.008,-0.60476 -0.983,-0.359 -1.9189,0.0162 -2.923,-0.10079 0.602,0.073 1.0619,0.29856 1.5119,0.70555 -0.54,-0.17 -1.0467,-0.28138 -1.6127,-0.30238 0.984,1.116 2.637,0.69214 3.931,0.90714 1.629,0.27 3.0472,1.05352 4.7373,1.20952 -0.503,0 -1.009,0 -1.5119,0 0.3379,0.423 0.8292,0.75894 1.3103,1.00793 -1.135,0.854 -4.4051,0.0926 -5.846,0.20159 -1.246,0.094 -2.0017,1.13352 -3.1246,1.20952 -1.412,0.097 -2.6906,-0.56638 -4.1326,-0.30238 -3.537,0.645 -7.4892,0.60476 -11.0872,0.60476 -1.99,0 -7.0277,-0.56648 -8.4667,1.20952 1.6521,-0.329 -0.2215,0.83372 -0.7055,1.10872 0.519,0.008 2.6813,0.98394 0.8063,1.00794 1.922,-0.109 3.923,1.0802 4.4349,2.82221 0.565,1.923 2.8785,2.59321 4.6365,2.82221 2.269,0.297 4.7442,-0.0587 6.7532,1.31031 0.768,0.523 1.1976,0.89031 2.1166,1.31032 0.783,0.358 1.5739,0.73489 2.0159,1.5119 -2.286,0.50199 -4.6816,0.654 -6.9547,0 -1.82,-0.523 -3.4871,-1.40329 -5.3421,-1.81428 2.04,0.40599 -0.1818,-0.68234 -0.6047,-0.80635 -0.909,-0.267 -1.8853,-0.28717 -2.8222,-0.40317 -1.782,-0.222 -3.4714,-0.94752 -5.2413,-1.20952 -1.732,-0.256 -3.7967,-0.2378 -5.0397,-0.1008 -1.188,0.132 -2.8467,1.87353 -4.0317,1.20952 -0.524,1.368 1.2072,0.93535 1.9151,0.80635 1.187,-0.216 2.5483,0.29531 3.2254,1.31031 -1.955,1.524 -4.1525,0.069 -6.1484,-0.50396 -1.936,-0.55601 -3.6546,0.15396 -5.5437,0.50396 1.182,0.937 2.2375,2.26143 3.7294,2.72142 1.034,0.32 2.0825,0.42156 3.1246,0.70555 1.433,0.39 4.9234,2.01897 6.1484,0.50397 0.212,1.182 1.4126,0.41314 2.1166,0.90714 0.826,0.578 1.1971,1.69451 2.4191,1.20952 -10e-4,0.264 -10e-4,0.54235 0,0.80635 1.05,-0.21401 2.0418,0.084 3.0238,-0.50397 -1.22,3.646 -11.564,-3.46318 -11.692,-0.40317 -0.982,-1.69501 -3.787,-0.83894 -5.3421,-1.00794 -1.678,-0.183 -4.3096,-0.1519 -5.5436,-1.5119 1.696,0.178 0.4454,-0.9251 -0.2016,-1.4111 -1.388,-1.043 -1.4935,-1.32638 -1.2095,-3.22539 0.415,-2.78 -4.5861,-6.49445 -7.2571,-5.14045 1.157,0.79499 6.1694,3.88804 3.2253,5.34204 -1.472,0.727 -3.4613,0.3789 -4.7372,1.5119 -0.3811,0.34 -0.2342,1.0669 -0.4032,1.5119 -0.246,0.647 -0.8123,1.0689 -1.3103,1.5119 3.094,0.831 5.6497,2.7091 8.3658,4.33411 0.654,0.392 2.4209,1.11787 2.5198,2.01586 0.062,0.562 -0.9438,1.50967 -1.1087,2.11666 -0.194,0.709 0.026,1.40966 0.1008,2.11666 0.205,1.937 1.2966,2.17318 2.6206,3.32618 0.8721,0.758 1.0103,2.72121 2.3183,2.82221 1.797,0.139 3.9833,-0.0112 5.7452,-0.40317 1.803,-0.401 2.9411,-1.08476 4.8381,-0.60476 1.618,0.408 3.6455,0.25335 5.1404,0.80635 1.698,0.62799 3.1878,1.05389 4.9389,1.51189 1.806,0.472 2.0999,1.84201 3.427,2.92301 0.694,0.565 1.5527,0.88669 2.1166,1.61269 0.483,0.622 -1.6841,0.48331 -1.8142,1.31031 -0.1491,0.95501 0.8242,2.59642 1.8142,2.72142 1.736,0.219 3.3824,0.59555 5.1405,0.70556 0.581,0.036 0.5776,0.65359 1.2095,0.20158 0.6,-0.429 1.4111,0.19397 1.9151,0.50397 -3.37,-0.681 -8.9992,1.11534 -11.5912,-2.11666 1.423,0.413 0.3818,-0.76211 0.1008,-1.41111 -0.342,-0.794 -0.6797,-1.56624 -1.1088,-2.31824 -0.921,-1.613 -1.9168,-1.58545 -3.5277,-2.21745 -1.325,-0.519 -2.4773,-1.97525 -3.7294,-2.31825 -2.062,-0.566 -4.7491,0.44214 -6.7531,0.90714 -1,0.232 -3.7509,0.11548 -3.0238,1.71349 0.532,1.169 0.1092,1.59724 1.4111,2.31824 1.713,0.949 6.0476,3.79083 2.6206,5.44284 -2.132,1.027 -0.8316,2.26773 -1.2095,4.03173 -0.38,1.77 -2.6562,1.53205 -3.8302,2.41904 -1.117,0.843 -2.6886,0.20787 -2.1166,2.01586 0.21,0.66401 -0.85,1.4889 -1.5119,1.5119 -1.079,0.038 -2.0857,-0.61834 -3.1246,-0.80634 -2.061,-0.375 -3.9836,-0.0246 -6.0476,-0.20159 -1.952,-0.166 -3.9051,-0.78093 -5.846,-1.00793 -2.024,-0.236 -2.5334,-1.77224 -4.2333,-2.31825 1.5389,-0.666 3.3367,-0.05 4.9388,0 1.493,0.047 2.4517,1.19634 4.0318,0.80635 1.5609,-0.385 3.0594,-1.42369 4.6364,-1.61269 0.55,-0.066 3.9833,-1.15369 1.8143,-1.6127 0.489,-0.202 0.9939,-0.30658 1.5119,-0.20158 -0.276,-0.226 -0.5563,-0.45055 -0.8063,-0.70556 1.74,-0.72299 1.0895,-3.15975 3.6285,-3.52776 -0.669,-0.653 -0.3137,-1.83842 -1.1087,-2.72142 -0.514,-0.572 -0.5325,-0.7509 -0.2016,-1.5119 0.415,-0.955 1.3256,-1.22966 0.2016,-2.11666 -0.651,-0.514 -1.5602,-0.62814 -2.3182,-0.90714 -0.954,-0.352 -6.4824,-1.84635 -5.6445,-3.72935 0.464,-1.042 -0.618,-0.74807 -0.5039,-1.91507 -0.583,-0.864 -1.0527,-0.97066 -1.1088,-2.11666 -0.066,-1.338 -2.557,-2.41056 -2.923,-3.62856 -0.261,-0.869 1.261,-0.91449 0.504,-1.71348 -0.645,-0.681 -0.4654,-0.96928 -0.8063,-1.81428 -1.439,-3.575 -6.4986,-2.33205 -8.4667,-5.34205 0.472,-0.20599 1.2876,-0.23755 1.6127,-0.70555 0.384,-0.554 -0.3709,-1.0959 0,-1.5119 0.545,-0.613 0.6347,-1.95241 0.6048,-2.72142 -0.077,-1.931 -2.8377,-3.81052 -4.5357,-4.13252 -1.161,-0.22 -2.4158,0.11162 -3.5278,-0.30238 -1.339,-0.498 -2.7125,-0.48618 -4.1325,-0.40317 -2.671,0.15599 -5.1699,0.64159 -7.8619,0.20158 1.676,-0.375 -0.4409,-0.62258 -1.0079,-0.20158 -0.664,0.493 -1.0407,1.30466 -1.1087,2.11665 -0.1621,1.92201 1.5406,4.82956 0.2015,6.55157 -1.013,1.304 -2.883,2.04704 -4.4349,2.41904 -1.215,0.29 -2.5505,2.73624 -0.7055,2.31824 -1.65,1.032 0.7227,1.65311 1.6127,1.41111 2.089,-0.57 1.9048,1.11803 2.5198,2.41904 0.168,0.355 0.2118,1.48048 0.6048,1.71348 0.459,0.273 0.8954,0.41614 1.2095,0.90714 0.581,0.906 -1.5482,0.84635 -1.9151,0.80635 0.25,1.079 0.2508,2.61901 1.6127,2.923 0.139,-1.424 2.4768,-0.10841 3.5278,0.20159 0.892,0.262 2.3844,0.78079 3.2254,0.10079 0.5859,0.788 0.4284,1.70025 1.3103,2.31825 0.653,0.457 1.3269,0.81252 2.0158,1.20952 0.773,0.445 1.5808,1.08872 2.5199,1.10872 0.793,0.017 1.4745,-0.607 2.2174,0 -1.235,0.122 0.3532,0.76188 -0.1008,2.01587 -0.216,0.597 -0.784,1.45666 -0.5039,2.11666 1.066,2.514 -3.5443,-0.55876 -3.7294,-0.60476 -3.02,-0.753 -5.7599,-3.02397 -8.769,-3.42697 -1.335,-0.179 -2.6089,-0.44455 -3.9309,-0.70556 -0.506,-0.1 -4.3553,-1.04593 -2.3183,-1.00793 -0.941,-0.103 -4.4461,-1.45696 -4.8381,-0.50396 -0.416,1.01 -1.8384,-0.45135 -0.3023,-0.80635 -2.956,-0.612 -5.7103,-1.83404 -8.6683,-2.41904 -2.832,-0.562 -5.7902,-0.62455 -8.6682,-0.70555 -1.618,-0.045 -3.261,-0.10097 -4.8381,-0.50397 -0.948,-0.243 -1.9702,-0.5962 -2.8222,0.10079 -0.123,0.28701 -0.2423,0.60114 -0.3024,0.90714 -0.264,0.022 -0.5423,0.0758 -0.8063,0.1008 -0.64,0.371 -1.0569,-0.22025 -1.5119,0.60476 1.217,0.72 2.6788,1.33182 3.5278,2.51983 0.802,1.122 1.801,0.12972 2.419,1.10873 2.04,3.23499 -4.234,2.28118 -4.9389,3.32617 0.445,0.166 1.794,0.97869 1.5119,1.6127 -0.4,0.903 -2.3728,0.30958 -3.0238,0.20158 -1.634,-0.271 -3.1523,-2.12341 -0.8063,-2.72142 -0.553,-0.365 -4.0547,-1.64227 -2.1167,-1.81428 -1.548,0.254 -3.3655,-0.16106 -4.6365,1.00794 0.174,0.256 0.3918,0.48055 0.6048,0.70555 -0.119,-0.047 -2.4823,-1.32618 -2.3183,-0.40317 0.114,0.647 -0.6855,0.69954 -1.2095,0.70555 -0.83,0.01 -1.2449,0.86272 -2.0158,1.10873 -0.751,0.24 -1.5913,-0.0116 -2.3183,-0.20159 -2.235,-0.583 -10.408,-0.8512 -9.6761,3.0238 -0.931,0.39 -1.7321,-1.14935 -2.4191,-0.80635 -0.582,0.29 -1.6706,1.1822 -1.6127,-0.10079 -0.984,1.217 -4.0259,0.74697 -5.342,0.50397 0.014,-1.70001 2.5491,-0.54211 3.3262,-1.41111 -0.282,-0.465 -2.6938,-2.8328 -0.6048,-3.0238 -0.775,-0.267 -1.646,-0.23397 -2.419,-0.50397 1.477,-0.433 3.1736,0.28725 4.5357,-0.60476 -3.047,-0.10999 -8.5552,0.4148 -10.6841,3.0238 0.934,-0.047 1.6831,0.50073 2.3182,1.10873 -0.492,0.097 -2.3714,1.46356 -2.7214,0.70555 -0.32,-0.691 0.5688,-1.0811 -0.4032,-1.4111 -1.03,-0.35 -2.3533,0.14854 -3.2254,0.70555 -0.916,0.586 -2.2033,0.33017 -3.2253,0.40317 -1.928,0.137 -3.5215,1.98804 -5.1405,2.41904 -2.289,0.609 -3.9257,1.74052 -6.4508,1.20952 1.059,0.389 2.085,1.17907 2.923,1.91507 -0.919,-0.537 -3.8302,-0.76879 -4.7372,-0.10079 -0.863,0.636 -1.6473,-0.14603 -2.3183,0.50397 1.338,0.65099 0.5734,1.59161 0.3024,2.62062 -0.253,0.954 1.2933,1.65004 -0.1008,2.41904 -0.543,0.3 -2.0544,0.0896 -2.7214,0.20159 -1.053,0.176 -2.5599,0.0973 -3.427,0.80634 0.227,-0.23 0.3998,-0.45555 0.6048,-0.70555 -1.395,-0.087 -3.176,0.43283 -4.4349,-0.40317 -0.8901,-0.592 -1.3362,-1.64166 -2.3183,-2.11666 -1.049,-0.506 -2.5899,-0.15872 -3.427,-1.10873 -1.027,-1.164 0.5869,-1.78686 1.5119,-2.01586 -0.097,-0.264 -0.2054,-0.54235 -0.3023,-0.80635 2.445,-0.851 5.0955,-0.0863 7.5595,-0.80635 -0.809,-1.898 -2.9388,-2.40697 -4.5357,-3.42697 -1.747,-1.116 -2.8587,-1.81586 -5.0397,-2.01586 -2.28,-0.208 -4.7588,0.67779 -6.9547,0.10079 -1.067,-0.28 -3.2813,-1.47218 -4.2334,-0.40317 1.724,0.53 3.8889,1.17466 5.4429,2.11666 0.934,0.566 -0.016,1.96566 1.2095,2.11665 -1.08,0.176 -0.6878,0.61594 -0.1008,1.00794 -1.229,0.059 -0.3292,0.48352 -0.4032,1.20952 -0.059,0.588 -0.289,1.16948 -0.5039,1.71348 -0.277,0.702 -0.8871,1.91225 0.1008,2.31825 1.417,0.582 2.743,0.33769 3.9309,1.61269 0.669,0.718 0.2501,2.11603 1.4111,2.41904 -2.582,0.815 -1.5452,4.5088 -0.4032,5.9468 -0.947,-0.749 -1.4054,-2.09424 -2.7214,-2.31824 -0.699,-0.119 -1.991,-10e-4 -1.9151,1.00793 -0.296,-0.117 -0.5752,-0.28497 -0.8063,-0.50397 0.426,-0.411 0.9318,-0.68934 1.5119,-0.80634 -0.263,-1.245 -2.417,-1.5687 -3.427,-1.6127 -1.584,-0.07 -2.2981,-0.056 -3.8301,-0.50396 -1.505,-0.442 -2.4132,0.61369 -3.3262,1.61269 -1.12,1.227 -2.362,1.55807 -3.9309,1.91507 -1.595,0.363 -2.5022,1.36945 -3.8302,2.21746 -0.727,0.464 -2.8644,1.03145 -2.7214,2.21745 0.137,1.14 2.7087,2.3008 3.5278,3.0238 1.221,1.075 2.288,2.32713 2.923,3.83014 -0.937,-0.283 -3.0997,-0.92256 -4.0318,-0.70555 -1.415,0.33 -2.031,0.37942 -3.4269,-0.20159 -1.468,-0.61 -3.1272,-1.05114 -4.7373,-0.90714 -1.018,0.091 -2.2059,1.39 -2.5199,0 0.599,0.048 1.2127,-0.26756 1.6127,-0.70555 -2.392,0.702 -3.9943,-1.68666 -6.1484,-2.11666 -0.8169,-0.163 -2.3779,-0.9872 -2.923,0.10079 -0.5869,1.172 -0.6522,1.59691 -2.3182,1.5119 1.253,1.694 3.2271,2.64295 4.8381,3.93094 0.755,0.603 1.7606,0.4586 2.6206,0.20159 1.112,-0.332 1.8312,0.40456 2.8222,0.70555 -0.803,1.041 2.3145,3.37779 -1.2095,3.0238 -0.963,-0.097 -1.2447,0.92096 -2.6206,0.50396 -1.446,-0.438 -3.0039,-0.55292 -4.4349,-1.00793 -1.173,-0.373 -1.7842,-1.42786 -2.8222,-2.01586 -0.553,-0.315 -0.4978,0.7282 -1.1088,-0.1008 -0.257,-0.338 -0.5909,-0.59255 -1.0079,-0.70555 -1.192,-0.581 -1.8828,0.4522 -3.0238,-0.10079 -0.922,-0.448 -1.8472,-1.0911 -2.8222,-1.41111 0.219,-0.238 0.3918,-0.46255 0.6047,-0.70555 -0.261,-0.172 -0.5403,-0.33997 -0.8063,-0.50397 0.279,-0.181 0.6132,-0.34898 0.9071,-0.50396 -1.024,-0.071 -1.0566,-1.46971 -2.1166,-1.6127 1.043,-1.156 -1.5307,-2.10742 -2.1167,-2.72142 -0.591,-0.619 2.3424,-1.3882 1.3103,-2.82221 2.048,0.489 -1.1066,-2.40542 -1.6127,-2.72142 -2.161,-1.349 -5.0764,-1.68628 -7.5595,-1.81428 0.729,-0.251 1.9407,-0.1113 1.6127,-1.31031 -0.909,0.179 -3.0109,0.004 -3.9309,-0.20158 0.852,-0.443 1.9232,-0.44958 2.8222,-0.20159 -0.925,-0.946 -2.4147,-0.12818 -3.5278,-0.40317 0.85,0.146 1.5691,-0.65638 2.4191,-0.30238 -0.87,-0.70501 -2.1246,-0.47335 -3.1246,-0.80635 1.885,0.198 -1.4715,-1.23373 -2.2175,-1.10873 0.25,-0.255 0.4756,-0.43255 0.7056,-0.70555 -1.267,-0.885 -2.7626,-0.73331 -4.1325,-1.31031 0.978,0.152 1.972,-0.379 2.923,0 1.008,0.402 2.2521,0.54597 3.3262,0.50396 -2.1561,0.08 2.1634,2.22705 2.7214,2.41904 0.665,0.229 1.954,-0.2712 2.419,0.1008 0.685,0.548 1.4808,0.62741 2.0159,-0.20159 -0.267,1.29 1.5494,0.54649 2.2174,1.71349 0.075,0.12999 2.5526,-0.265 3.1246,0 2.67,1.23899 5.2192,1.78607 8.1643,1.91507 3.483,0.152 6.6433,1.26269 10.0793,1.61269 3.088,0.315 5.7861,-0.10034 8.769,-0.80635 2.727,-0.645 4.4007,-1.66276 6.4508,-3.52776 1.358,-1.237 2.8812,-3.70211 0.4032,-4.33411 -0.506,-0.129 -1.1374,-1.71645 -1.3104,-2.21745 -0.436,-1.263 -0.894,-0.99891 -1.915,-1.5119 -1.131,-0.566 -2.3412,-0.41232 -3.3262,-1.31032 -0.873,-0.795 -1.9997,-1.54107 -3.1246,-1.91507 0.202,0.255 0.326,0.53434 0.504,0.80635 -2.491,0.302 -5.2619,-1.14425 -7.3579,-2.31825 0.269,0.209 0.5493,0.38076 0.8063,0.60476 -2.684,-0.477 -4.7799,-2.45238 -7.3579,-3.22538 -3.003,-0.901 -6.0566,-1.77001 -8.9706,-2.92301 -2.604,-1.029 -4.8837,-0.28555 -7.4587,-0.70555 -1.105,-0.18 -2.2042,-0.67656 -3.3262,-0.70555 -1.016,-0.026 -2.1326,0.354 -3.1246,0 -0.026,0.66937 -0.33,0.95897 -0.7055,1.20952 -0.6832,0.45575 -1.6792,0.60207 -2.1167,1.51189 -0.591,-1.188 1.5267,-1.10289 2.1167,-1.51189 -0.542,-0.16601 -1.4528,-0.50956 -0.1008,-0.70556 -1.018,-0.203 -0.693,-0.32655 0,-0.70555 -0.831,-0.982 -2.2446,-0.4822 -3.1246,0.10079 0.182,-0.254 0.3519,-0.43255 0.504,-0.70555 -0.845,-0.02 -1.6808,-0.32818 -2.5199,-0.40317 -0.374,-0.034 -0.6688,0.0868 -1.0079,0.20158 0.1421,-0.0438 0.2595,-0.16182 0.4032,-0.20158 -0.915,-0.48 -1.8702,-0.72473 -2.8223,-1.10873 0.096,-0.269 0.2122,-0.49455 0.4032,-0.70555 0.825,0.996 5.3774,2.16759 5.6444,0.20158 -1.245,-0.532 -2.5791,-0.26814 -3.8301,-0.90714 -0.846,-0.432 -2.1466,-1.08054 -3.1246,-0.70555 0.24,0.216 0.4646,0.38977 0.7056,0.60476 -0.42,0.054 -1.4887,-0.13904 -1.6127,0.50397 -0.079,0.405 -1.1361,0.96435 -1.4111,0.80634 -0.919,-0.526 -2.2932,-0.61434 -3.3262,-0.80634 0.734,0.524 1.146,1.41749 0,1.71348 -1.172,0.304 -2.2852,-0.25555 -3.3262,-0.70555 0.914,2.062 -2.2009,2.07024 -3.427,2.31825 -1.415,0.286 -0.4045,0.98849 -1.2095,1.71348 -1.084,0.977 -3.0031,1.0329 -4.3341,1.5119 0.784,0.103 2.9557,0.2789 1.6127,1.5119 -1.161,1.092 -0.9808,2.1158 0.4032,3.0238 1.338,0.878 3.0111,0.90828 4.3341,1.81428 1.039,0.711 2.9334,1.48104 3.7293,2.41904 1.764,2.08 -5.7599,3.73536 -3.4269,6.65235 2.255,2.82 5.6139,4.62405 6.8539,8.26505 -0.761,0.04 -1.4724,0.26917 -2.2174,0.40318 0.974,0.27599 0.4459,1.77482 0,2.51983 -0.4211,0.703 1.0792,0.90211 1.3103,1.4111 -1.559,0.335 -1.2552,2.07983 0.1008,2.51984 0.898,0.291 1.894,-0.24965 2.419,0.80634 -2.764,1.247 1.2142,2.67238 2.3182,3.22539 2.445,1.225 -1.281,3.55751 -2.419,4.13252 2.967,1.652 5.9848,3.20986 8.8698,4.93887 1.203,0.721 1.6888,1.00124 1.1087,2.31825 -0.676,1.536 -1.1358,2.58155 -2.5198,3.62856 -2.978,2.25299 -5.2882,5.16448 -8.1642,7.55949 -2.78,2.314 -6.274,3.79878 -8.7691,6.45077 1.416,0.016 2.1162,-0.006 3.3262,-0.80634 0.455,-0.301 1.5857,-0.96517 2.1167,-0.40318 -0.72,0.422 -0.183,1.26628 0,1.81428 -0.512,-0.247 -1.0179,-0.52535 -1.5119,-0.80635 0.863,0.95701 2.9285,2.59381 4.1325,3.0238 1.719,0.613 3.1481,-0.55324 4.8381,0.60476 0.079,0.054 3.5508,1.91507 2.0158,1.91508 -1.675,0 -3.2249,-1.06074 -4.9388,-1.10873 -1.489,-0.042 -1.7996,1.62447 -3.1246,1.71349 -0.757,0.051 -1.7105,-1.013 -2.2175,0 -0.151,0.30199 -0.1878,0.80434 -0.6047,0.80634 -0.709,0.005 -2.0623,-1.393 -1.8143,0 0.117,0.658 0.7095,0.9327 0.2016,1.61269 0.498,0.194 1.0175,0.46494 1.2095,1.00794 -1.719,-0.024 -1.6114,1.20303 -2.2175,2.41904 -0.436,0.87199 -1.7364,1.25124 -1.7134,2.31824 0.024,1.068 0.9085,1.95079 0.7055,3.0238 -0.117,0.617 -0.2626,1.23448 0.3024,1.71349 0.688,0.584 1.1949,2.20925 2.0159,2.31824 -2.1381,1.692 -3.2829,2.72253 -0.1008,4.13253 1.379,0.612 0.9118,2.97872 0.1008,4.03173 1.5899,-0.193 2.9065,1.96638 3.6285,3.22538 0.275,0.479 0.59,3.72117 1.5119,3.32618 1.592,-0.683 2.3887,0.85995 4.0317,0.50397 0.769,-0.167 1.6891,-0.23621 2.4191,0.10079 0.78,0.36 0.395,0.86491 0.5039,1.5119 0.27,1.602 4.2901,-0.82296 5.3421,-0.50397 0.652,0.198 4.6923,1.97983 4.7373,2.51983 0.093,1.117 -0.4267,1.63341 0.3024,2.72142 0.3619,0.539 0.8398,1.62367 0.1007,2.11666 -1.2839,0.856 1.9469,3.06073 2.5199,4.03173 0.593,1.005 1.2282,2.09963 2.3182,2.62063 0.799,0.381 2.1641,0.22931 2.4191,1.31031 0.153,0.645 -0.9363,1.93369 0.6047,1.61269 1.092,-0.228 1.4478,-0.0634 2.5199,0.20159 1.8419,0.457 0.2275,0.6033 1.2095,1.31031 0.627,0.451 1.8876,0.88849 2.1166,1.71349 0.157,0.563 -2.9685,2.36204 -3.6285,2.41904 -1.847,0.16 -3.3235,-1.70059 -5.1405,-0.20159 -0.87,0.718 0.48,1.68766 1.008,2.11666 1.0479,0.851 0.6811,0.89566 0.9071,2.11666 0.341,1.831 0.8642,3.7699 2.8222,4.4349 1.58,0.536 2.9412,-0.30986 2.8222,-2.01586 1.828,0.127 2.8735,0.80679 4.6365,0.10079 1.61,-0.644 2.7416,0.48421 4.1325,-0.10079 1.467,-0.618 2.476,1.54383 2.923,2.51983 0.48,1.046 2.0529,1.17446 2.5199,2.21745 -0.638,0.396 -1.6109,0.14014 -2.0159,0.90714 -0.199,0.376 2.2267,2.56739 1.1087,3.22539 0.953,0.316 3.2874,1.07676 4.2333,0.60476 1.488,-0.743 2.1137,1.28552 3.5278,1.20951 -0.421,0.791 0.7957,3.10995 1.1087,3.93094 0.527,1.38102 1.581,2.05332 2.923,1.31032 1.222,-0.67601 1.9041,0.16355 2.923,0.70555 0.1461,0.078 2.0576,0.91014 2.1167,0.90714 1.727,-0.083 3.2883,-1.90949 5.2413,-1.71349 1.125,0.11399 1.6643,1.79483 2.3182,2.51983 0.72,0.799 1.961,0.83509 2.419,1.91508 1.68,-0.96701 0.8871,-1.66979 2.923,-0.60476 1.054,0.55199 3.9641,1.72855 4.8381,0.70555 0.754,1.504 1.8654,0.5947 2.7214,1.61269 0.603,0.71701 1.5969,1.06693 2.5199,1.00794 0.633,-0.04 0.7808,-0.55458 1.5119,-0.20159 -1.106,1.05599 1.536,3.40291 -1.008,4.4349 -0.656,0.267 -1.2186,0.33251 -0.7055,1.20952 0.658,1.12702 1.6652,-0.42227 1.8143,1.10873 -0.856,-0.256 -2.2953,0.5337 -1.8143,1.61269 0.165,0.36899 2.8221,1.29025 1.4111,2.31825 0.299,0.242 0.6338,0.30057 1.0079,0.20158 0.185,0.90901 -0.4718,3.83795 -1.5119,3.93094 -1.097,0.098 -2.351,-0.3148 -3.4269,-0.10079 -0.944,0.18801 -1.9868,-0.61405 -2.6207,0.50396 -0.316,0.559 -0.6633,1.02352 -1.3103,1.20952 -1.276,0.36699 -2.7157,2.60302 -1.1087,2.92301 -1.042,0.91301 -0.1638,1.79031 0.9071,1.31031 0.2699,-0.12095 0.4727,-0.26122 0.7056,-0.40317 -0.1327,0.13166 -0.3207,0.32247 -0.4032,0.50396 1.367,0.15502 2.623,-0.77893 3.931,-1.00793 1.653,-0.28899 0.9074,1.50874 2.2174,1.10873 -1.008,0.88801 -2.8271,1.91465 -4.3341,2.11666 -3.02,0.40501 -0.7568,0.49606 -1.1087,1.91507 -0.921,0.065 -1.5839,-0.48258 -2.5199,-0.20159 -0.846,0.253 -1.537,0.56161 -2.419,0.20159 0.648,2.07699 1.8221,1.71261 3.3262,2.62062 0.594,0.35799 2.8587,1.60983 2.6206,2.51984 -0.731,0.048 -1.5195,-1.10734 -2.2175,-0.80635 -0.5969,0.258 -0.533,1.63266 -0.9071,2.11666 -0.102,-0.479 -0.2104,-0.93011 -0.3024,-1.41111 -0.898,0.56299 -0.946,4.34895 -2.419,3.42697 -0.067,0.48201 0.3333,2.46155 0.8063,0.70556 0.099,0.739 0.6994,1.44845 0.3024,2.21745 -0.5162,-0.15488 -3.8293,-0.50861 -3.5278,0.10079 -0.9988,-0.81679 -3.0415,-2.14372 -3.5277,-0.40317 0.228,-0.18601 0.4524,-0.35399 0.7055,-0.50397 0.048,0.60399 0.3989,1.00794 1.008,1.00794 -0.652,0.38501 -1.4884,0.50832 -1.8143,1.31031 2.034,0.89701 4.1519,0.6004 5.4428,2.72142 0.641,1.05199 3.9641,1.91978 2.923,0.60476 1.489,0.41701 2.2218,2.10121 3.5278,2.82221 1.419,0.78399 3.0867,1.00925 4.5357,1.81428 3.315,1.83898 5.4898,5.16094 8.8698,6.85394 -0.157,-3.457 7.5504,-0.27683 9.0714,0.40317 1.214,0.542 2.5171,0.90772 3.8301,1.10873 1.228,0.18801 2.1703,-0.24858 3.3262,-0.20159 0.827,0.034 1.4563,0.57377 2.3183,0.60476 0.785,0.028 1.6292,-0.0668 2.3182,0.40318 0.842,0.57399 6.673,3.02592 5.846,3.93093 2.215,1.64502 5.8669,-2.7826 7.3579,0.30238 0.114,-1.60098 2.4527,-1.02757 3.1246,-0.20158 1.14,1.40101 1.8494,0.27493 3.2254,1.00793 0.883,0.471 0.2164,1.64804 0.3024,2.41904 0.086,0.768 1.819,1.37269 2.419,1.61269 2.274,0.90801 4.9304,1.09321 6.6524,2.82221 0.741,0.742 1.3595,1.0559 2.2174,1.5119 0.915,0.487 0.9132,1.76445 1.9151,2.21746 0.827,0.37401 2.527,1.11756 3.427,0.70555 0.502,-0.229 0.9083,-1.47407 1.3103,-1.91507 0.814,-0.89502 1.794,-1.54861 2.419,-2.62063 0.662,-1.13601 -0.3076,-1.38943 -1.1087,-2.21745 -0.93,-0.962 -1.5826,-2.12379 -2.6206,-3.0238 -0.904,-0.78499 -2.0714,-1.49284 -2.7214,-2.51983 -0.696,-1.10499 -1.0394,-2.204 -2.2175,-2.92301 -0.84,-0.51199 -0.5782,-2.28521 0.1008,-2.82221 -1.837,-0.32799 -0.6648,-3.87507 -0.1008,-4.83808 -1.351,-0.84298 -1.2218,2.08322 -1.1087,2.82222 -1.31,-0.19401 -0.3572,-0.84547 -0.4032,-1.71349 -0.049,-0.91301 -0.7084,-1.71203 -1.2095,-2.41904 -0.834,-1.17701 -1.4463,-1.74603 -2.7214,-2.41904 -0.953,-0.503 -2.6459,-0.80744 -2.5199,-2.21745 0.087,-0.985 2.2072,-1.78122 1.4111,-2.82221 0.1761,0.254 0.3937,0.47955 0.6048,0.70555 0.797,-2.75099 3.1802,-5.29089 0.9071,-7.86187 0.9211,0.009 1.3904,0.93368 1.8143,1.61269 -0.197,-0.61699 -0.2553,-1.37331 -0.8063,-1.81428 0.836,0.236 1.5591,0.81154 2.3182,1.20952 -0.312,-1.59799 1.1707,-0.50294 2.1167,-1.00793 0.465,-0.248 0.6922,-0.76955 1.3103,-0.70556 -1.241,-1.54199 1.9912,-0.27577 2.3182,-0.10079 0.079,-0.61199 -0.2093,-0.90414 -0.8063,-0.90714 0.36,-0.507 0.2609,-0.97371 -0.4032,-1.10873 0.517,-0.26099 1.0229,-0.2092 1.5119,0.1008 -0.042,-0.982 1.1335,-0.98578 1.7135,-0.60476 -0.101,-0.5 -0.1546,-1.0039 -0.2016,-1.5119 0.27,0.14001 0.5423,0.25416 0.8064,0.40317 -0.233,-1.478 -6.0494,-2.83656 -5.6445,-3.62856 0.515,-1.00799 3.5127,0.67143 3.1246,-0.70555 -0.278,-0.983 -1.4627,-1.79783 -2.1166,-2.51983 -2.206,-2.43201 -3.7243,-5.30283 -7.1563,-5.94681 -0.84,-0.15801 -2.5974,-0.54263 -3.2254,0.30238 -0.739,0.992 -1.7635,-1.78444 -2.2175,-2.21745 0.239,-0.17001 0.4646,-0.33795 0.7056,-0.50396 -1.042,-0.94501 -0.871,-1.61789 -2.4191,-2.01587 -0.854,-0.22 -1.93,-0.47829 -2.419,-1.31031 -0.911,-1.54999 0.2968,-3.83728 1.5119,-4.73729 1.685,-1.248 -1.3987,-2.30595 -1.6127,-3.42697 -0.467,-2.436 0.053,-3.36749 1.8143,-4.63649 1.076,-0.776 -1.0523,-3.75935 1.6127,-3.72935 2.31,0.026 3.8849,3.81327 5.846,4.73728 1.025,0.483 2.5663,-0.14914 3.3262,-0.90714 0.648,-0.64601 -0.4016,-1.26328 -0.7056,-1.81428 -0.553,-1.00299 -0.9715,-3.43516 -2.2174,-3.83014 1.748,0.459 3.3472,-0.38692 4.7372,-1.5119 0.5,-0.405 -0.2986,-1.21348 -0.2015,-1.71349 0.15,-0.77499 1.2772,-0.50397 1.8142,-0.50396 0.961,0 1.2647,-0.96752 2.1167,-1.20952 0.783,-0.223 1.7705,-0.17494 2.2174,-1.00793 0.691,-1.287 0.1569,-2.69129 2.0159,-1.81428 -0.193,-0.479 -0.311,-1.03289 -0.504,-1.5119 1.1201,0.239 3.8476,0.45268 4.1326,1.61269 0.428,1.741 1.7348,-0.39785 2.5198,0.90714 0.39,-0.481 0.3819,-1.43247 1.0079,-1.71349 0.883,-0.397 1.6826,-0.38414 2.6207,-0.90714 1.114,3.57001 4.3116,2.08205 7.0555,2.41904 1.598,0.196 1.7908,1.68745 3.0238,2.21746 1.11,0.47599 2.047,0.17452 2.923,1.20952 0.635,0.75099 3.1138,1.71445 3.0238,2.21745 -0.126,0.706 0.033,1.72945 0.6047,2.21745 0.978,0.83599 1.2233,-1.01867 0.9072,-1.61269 -0.406,-0.76 -1.2359,-1.39846 -1.5119,-2.21746 1.285,-0.883 3.1107,0.84191 4.0317,1.5119 1.006,0.73201 2.5052,0.8807 3.3262,1.6127 0.982,0.87501 2.9101,0.50706 3.3262,-1.00794 0.297,-1.083 1.8943,-1.34484 1.8143,-2.51983 0.5639,0.035 1.0687,0.25576 1.5119,0.60476 0.052,-0.974 3.0851,-1.22075 3.3261,-0.60476 0.235,0.6 1.6284,1.3349 2.2175,1.5119 0.662,0.198 0.9934,-0.63238 1.7135,-0.30238 -0.093,-3.308 0.7756,-0.36947 2.6206,-1.71348 1.455,-1.061 2.6714,1.16287 3.2254,2.01586 0.778,1.199 2.614,1.48871 3.9309,1.61269 0.552,0.052 3.5513,0.2301 2.8222,1.41111 0.829,-0.056 2.1215,-0.23732 2.2175,-1.31031 0.045,-0.509 -0.1072,-1.4437 0.6047,-1.6127 0.6591,-0.15699 1.2805,0.77572 1.7135,1.10873 0.637,0.491 1.65,0.56576 2.4191,0.60476 1.073,0.054 5.1244,-0.43407 5.1404,-1.91507 0.01,-0.877 -1.1268,-2.94459 0.4032,-3.12459 -1.43,-0.473 -1.713,-1.4457 -3.427,-1.6127 -1.336,-0.129 -1.805,-1.1401 -2.923,-1.4111 -1.332,-0.322 -2.1277,0.15628 -3.0238,-1.10873 -0.542,-0.766 -3.3201,-0.51768 -3.3262,-1.61269 0,0.004 2.5255,-1.43589 2.7215,-1.5119 0.749,-0.292 1.274,-0.0308 1.915,-0.60476 0.383,-0.343 0.7475,-0.86472 1.2096,-1.10873 -0.4381,-0.28199 -3.1599,-2.97137 -3.0238,-3.22538 0.3279,-0.618 1.7418,-1.26466 1.0079,-2.11666 0.802,-0.09 1.9274,-0.24358 2.7214,-0.20159 0.627,0.034 1.1633,0.40317 1.8143,0.40318 0.365,0 4.0545,-0.14252 2.2174,-1.20952 -1.818,-1.056 -3.8183,-0.97191 -5.7452,-1.5119 -2.79,-0.781 1.0143,-1.38429 1.3103,-1.81428 -0.547,-0.815 -1.1349,-0.50016 -1.915,-0.40317 -0.837,0.10399 -1.813,-0.42235 -2.5199,-0.80635 2.448,-0.202 0.5024,-1.84104 -0.2016,-2.41904 0.485,-0.095 0.9371,-0.0422 1.4111,0.10079 -0.455,-1.647 3.319,-0.22178 3.931,-0.10079 0.933,0.184 1.6946,-0.60476 2.6206,-0.60476 0.953,0 1.2238,0.94494 2.1167,1.00793 -0.44,-2.729 3.1193,-0.64507 3.2254,-1.91507 0.1019,-1.225 4.2724,-0.55835 5.1404,-0.80634 0.099,-0.23801 0.1146,-0.46256 0.2016,-0.70556 0.416,0.055 0.9143,0.0302 1.3103,-0.10079 1.065,-0.144 1.7333,-0.3996 2.8222,-0.20159 0.722,0.131 1.27,-0.619 1.9151,0 0.832,0.799 0.8154,-0.42813 0.8063,-0.90714 -0.038,-1.946 6.4995,-1.66187 8.0635,-2.01586 1.745,-0.394 3.3836,-0.66394 5.1405,-1.00794 1.3289,-0.261 3.973,-0.87306 4.838,-1.91507 -1.544,-0.277 -0.2592,-0.81294 0.1008,-1.00793 0.705,-0.383 1.935,-0.1791 1.5119,-1.41111 0.591,0.168 1.2053,0.1728 1.8143,0.1008 -0.211,-0.244 -0.4845,-0.46955 -0.7055,-0.70556 0.943,0.046 2.0613,0.95517 2.7214,0.40318 0.952,0.136 1.9431,0.16875 2.8222,0.60476 0.89,0.442 3.2853,1.96593 4.2333,1.00793 1.26,-1.273 2.1293,-1.20321 3.7294,0.10079 0.915,0.746 0.9584,1.89342 1.7134,2.72142 0.59,0.646 1.3646,0.91432 2.1167,1.31031 -0.955,0.993 1.5591,3.16919 -0.504,3.32618 1.271,1.567 2.3846,0.57752 3.6286,1.20952 0.746,0.379 1.3612,-0.35935 1.8143,-0.80634 0.462,-0.456 1.6843,0.29358 2.3182,0.20158 -0.55,-0.276 -0.8515,-0.79411 -0.7055,-1.4111 1.052,0.467 1.8039,1.84108 3.0238,1.91507 -1.01,0.75 0.065,1.05586 0.5039,2.01586 0.406,-0.441 0.912,-0.71555 1.5119,-0.70555 -0.325,-0.498 -0.9348,-0.77196 -1.5119,-0.50397 1.055,-2.982 5.9551,3.21162 7.2571,-0.30238 0.533,0.42 1.0099,1.01048 1.008,1.71349 -0.989,-0.137 -2.7023,0.54228 -2.3183,1.81428 0.287,0.953 1.6011,0.85627 1.9151,1.81428 0.789,-0.73 2.1809,-2.4331 3.427,-1.41111 0.646,0.53 1.0945,-0.25462 1.7135,0.30238 0.385,0.346 0.5247,0.70255 1.1087,0.70556 -0.126,-1.975 0.9708,-2.16941 2.5198,-2.72142 0.732,-0.261 2.958,-1.12345 2.0159,-2.21746 1.212,-0.10599 2.3209,-0.51592 3.427,-1.00793 0.124,-0.056 1.3201,-1.12173 1.4111,-1.10872 1.161,0.168 2.2271,-0.59694 3.3261,-1.00794 -0.207,0.252 -0.4316,0.53035 -0.6047,0.80635 1.017,0.263 0.6846,1.11449 0.2016,1.71348 -0.317,0.39301 -1.1778,0.0234 -1.6127,0.30238 3.354,2.544 7.3028,3.90759 10.8857,6.0476 3.3249,1.985 5.959,4.63911 8.769,7.25712 2.911,2.712 5.9867,5.31946 8.8698,8.06346 1.808,1.721 4.4348,5.79998 6.9547,6.34998 -0.944,-1.002 0.7642,-0.49248 1.4111,-1.71349 0.455,-0.86 -0.5285,-1.15187 -0.2016,-2.01586 0.11,-0.293 1.5505,-0.61077 1.7135,-0.60476 0.787,0.024 1.2522,0.87234 1.9151,0.80634 1.378,-0.139 0.5794,0.8147 0.8063,1.61269 0.185,0.653 1.2565,0.27938 1.7135,0.30238 0.834,0.045 1.3322,1.25348 1.9151,1.71349 0.8,0.63101 2.2035,-0.23379 3.1246,-0.10079 0.764,0.11 1.1081,0.68639 1.9151,0.30238 0.376,-0.191 0.7875,-0.2316 1.2095,-0.20159 1.156,-0.035 1.3845,-1.06353 2.2174,-1.20952 1.125,-0.196 0.7058,0.0297 1.5119,-0.80635 0.271,-0.28099 0.7817,-0.1558 1.1088,-0.10079 3.178,0.533 6.4372,2.64327 8.1642,5.24125 1.333,2.005 4.7524,0.96262 6.1484,2.62063 0.955,1.13401 1.6612,2.47596 2.8222,3.42697 1.129,0.92401 3.938,0.62234 5.3421,0.80635 0.675,0.089 1.8802,1.23641 2.3182,-0.20159 0.147,-0.478 0.7547,-0.98832 1.1087,-1.31031 0.656,-0.59699 1.4108,-0.37525 1.1088,0.60476 -0.3911,1.26599 0.9955,0.9779 1.7134,1.51189 0.9071,0.67601 1.4965,1.97165 2.7215,2.11666 1.249,0.14902 1.0617,0.95433 2.0158,1.31032 1.28,0.478 4.3389,-0.55187 4.4349,-2.01587 0.076,-1.14901 0.9998,-1.58811 2.1167,-1.41111 0.948,0.15002 2.1509,1.07664 2.0159,-0.30237 0.708,0.30199 1.6172,0.64578 2.3182,0.10079 -0.139,-0.243 -0.3561,-0.46855 -0.504,-0.70555 1.154,-0.18301 2.0416,-0.50413 3.1246,-0.90714 -0.575,-0.34799 -0.8966,-0.99268 -1.1087,-1.6127 0.858,-0.60398 2.1819,-0.0473 2.0159,-1.31031 -0.032,-0.232 1.8099,-0.93391 2.0158,-1.00793 0.864,-0.30799 1.4626,-0.0953 2.1167,-0.80635 0.581,-0.63201 1.0711,-1.14011 1.9151,-1.4111 1.276,-0.41002 1.5462,-0.10231 2.3182,-1.31032 0.365,-0.56999 0.8889,-0.65656 1.5119,-0.70555 0.85,-0.067 2.8864,0.0119 2.2175,-1.41111 1.508,-0.03 1.9437,2.11792 3.5277,1.00794 1.391,-0.976 0.8641,0.62095 1.9151,1.00793 2.368,0.87101 5.4238,-0.46904 7.8619,0.50397 1.264,0.50399 1.9759,1.71861 2.923,2.62062 1.25,1.19201 2.8607,1.75487 4.5357,2.01587 0.589,0.092 3.9222,1.09534 4.3341,0.80634 0.973,-0.681 1.7509,-0.94532 2.923,-0.80634 1.803,0.222 3.5111,0.79992 5.342,1.00793 1.315,0.14901 2.9464,1.27893 4.2334,1.00793 1.148,-0.242 1.2998,-1.45868 2.5198,-1.61269 1.287,-0.16301 1.5671,-1.89901 1.4111,-2.923 -0.24,-1.57099 -0.8287,-1.77326 -2.1167,-2.31825 -1.262,-0.535 -1.3998,-2.05163 -2.5198,-2.62063 -1.201,-0.611 -2.7798,-5.46228 -0.6048,-4.73728 -0.6129,-1.299 -0.4429,-2.18362 1.008,-2.62063 1.047,-0.315 0.6919,-2.2 0.5039,-2.923 2.694,0.842 5.2537,1.97622 7.9627,2.82221 2.841,0.887 5.7334,0.82307 8.5674,1.91507 2.23,0.858 4.517,0.88631 6.854,1.31032 2.023,0.369 2.7689,2.99789 3.9309,4.4349 0.599,0.74 1.3961,1.20187 1.9151,2.01587 0.529,0.82998 1.5004,0.80412 2.2174,1.4111 0.702,0.59499 2.2207,1.02371 3.1246,1.10873 1.248,0.11798 1.6355,0.62492 2.7215,1.00793 3.893,1.37901 7.5757,-0.96026 11.2888,-1.81428 1.922,-0.442 4.4789,-0.22381 6.35,0.40318 1.543,0.51799 2.5057,0.004 4.0317,0.20158 1.388,0.17901 2.5106,1.26787 3.6286,2.01587 1.422,0.95 2.7638,1.06573 4.4349,1.10872 1.922,0.05 2.29,0.50207 3.4269,1.91508 0.86,1.06699 2.0276,1.37066 3.1246,2.11665 2.976,2.025 5.7616,0.74829 8.9706,1.81428 2.069,0.68701 4.0163,0.56177 6.1484,0.60476 1.219,0.025 2.5653,0.66382 3.7294,0.1008 0.686,-0.33099 1.1422,-0.8725 1.8143,-1.20952 0.74,-0.37102 1.6389,-0.10996 2.419,-0.50397 1.075,-0.54199 3.0253,-0.50932 4.2333,-0.80635 1.34,-0.32898 3.0493,0.54429 3.7294,-1.10872 0.474,-1.15301 0.5747,-1.98923 1.6127,-2.82221 2.707,-2.17001 5.5228,-3.48948 8.8698,-1.71349 1.627,0.86402 2.8843,1.95426 4.7373,2.31825 1.5939,0.31299 2.4947,-0.71814 4.0317,-0.90714 1.937,-0.238 3.0816,0.92126 4.6365,1.81428 1.407,0.80699 3.0714,0.93852 4.6365,1.20952 1.191,0.207 3.6365,1.61198 4.6365,0.50396 0.8239,-0.91101 1.2899,-2.3672 2.5198,-2.82221 0.841,-0.31099 4.4526,-0.62847 3.6285,-2.21745 -0.414,-0.79899 -2.1852,-1.00271 -2.3182,-1.61269 -0.228,-1.051 -0.1362,-1.80422 0.1008,-2.82222 0.168,-0.718 -0.6504,-1.22006 -0.3024,-1.91507 0.314,-0.628 0.2596,-1.44766 0.2016,-2.11666 -0.106,-1.223 -0.7704,-2.84372 -0.3024,-4.03173 0.635,-1.609 2.1879,-1.57977 1.5119,-3.52776 -0.392,-1.128 -1.9209,-1.9296 -2.5198,-3.1246 -0.507,-1.016 -2.1517,-1.17071 -3.1246,-1.10872 -1.423,0.09 -1.8643,-0.14751 -2.8222,-1.20952 2.406,-0.812 1.3893,-4.08412 4.2333,-4.33411 2.766,-0.244 5.2795,-1.7939 8.0635,-1.5119 3.159,0.321 6.5025,-1.22184 9.4745,0.40317 0.699,0.381 1.5175,0.25476 2.2175,0.60476 0.631,0.315 1.1832,0.79473 1.8143,1.10873 1.233,0.616 2.6998,0.37993 3.9309,1.00793 0.016,-1.281 1.4017,-0.78997 2.1167,-0.50397 0.793,0.317 3.0469,0.68631 3.4269,1.31032 0.595,0.977 2.2044,1.26628 3.2254,1.81428 0.867,0.465 0.8526,0.41752 1.2095,1.20952 0.333,0.739 1.3521,0.65972 1.9151,1.10872 1.159,0.926 2.4416,1.62483 3.6286,2.51983 1.153,0.868 1.3496,2.17279 2.6206,3.0238 0.887,0.593 2.0868,2.07804 3.0238,2.41904 0.69,0.251 2.0947,1.52565 2.5198,2.11666 1.828,2.545 5.3597,3.47839 6.9548,6.14839 0.917,1.53599 1.5507,1.76564 3.0238,2.62062 0.8219,0.47701 1.0883,2.10521 1.8142,2.82222 1.153,1.13901 2.4377,2.14285 4.0318,2.51983 1.71,0.40501 2.5542,0.21781 4.2333,0.10079 1.214,-0.085 1.856,0.10534 2.923,0.80635 1.204,0.79099 2.641,0.67952 3.9309,1.20952 -1.078,-1.91901 1.5535,0.45597 1.7135,0.50396 1.018,0.307 1.5561,0.33773 2.4191,1.10873 1.5139,1.35398 3.3798,2.38946 5.4428,2.21745 2.887,-0.24 0.9135,1.64363 2.7214,2.62063 1.637,0.88601 3.0878,1.68755 3.0238,3.62856 -0.035,1.08098 4.3978,4.01015 5.4428,3.83014 0.6331,-0.10899 1.8211,-0.66159 2.4191,-0.20159 0.704,0.542 0.9999,0.51716 1.9151,0.40318 1.175,-0.14702 4.9388,0.21268 4.9388,-1.31032 0,-2.14501 2.6921,-2.77363 4.3341,-2.11665 -0.184,-0.073 1.6962,-1.57551 1.9151,-1.71349 1.271,-0.79999 2.8694,-1.12318 4.2333,-0.40317 0.981,0.51699 1.1854,0.16033 1.3103,1.31031 0.076,0.704 0.3523,1.28028 0.8064,1.81428 0.824,0.969 2.0101,1.30863 2.3182,2.62062 0.428,1.81799 -0.7871,1.13002 -1.4111,2.41904 -0.386,0.79799 0.4346,1.29606 0.7056,1.91507 0.25,0.57099 0.1443,1.2193 0.3024,1.81428 0.226,0.84999 0.782,1.46826 1.0079,2.31825 0.129,0.486 -0.084,1.16669 0.2016,1.61269 0.463,0.723 1.8734,1.83663 0.3024,2.62063 1.47,1.07399 -0.2686,1.67 0.3023,2.923 0.317,0.69101 1.2986,1.56767 0.2016,2.11666 -1.375,0.68901 -0.043,2.27361 0.4032,3.12459 1.111,2.11602 -4.191,0.48239 -4.8381,0.30238 -0.988,-0.27399 -3.2522,-1.50532 -4.2333,-1.31031 -0.484,0.097 -0.8588,1.81044 -1.1087,2.21745 -0.63,1.026 -1.5021,0.78474 -2.4191,1.10873 -0.708,0.25 1.5813,2.15704 1.8143,2.41904 2.021,2.26999 3.6148,4.62591 4.9389,7.35791 1.123,2.31898 3.0248,3.2676 3.0238,6.04759 0,0.486 0.4219,1.5537 -0.4032,1.6127 -1.251,0.09 -2.3956,-0.0147 -3.1246,1.31031 1.796,0.496 2.9048,2.18955 4.0318,3.62856 0.192,-1.59599 -0.3265,-2.3018 -1.7135,-3.0238 0.813,-0.84599 2.8774,1.6e-4 3.7293,0.40317 0.183,-0.766 0.3657,-2.55499 -0.3024,-2.923 1.938,0.32799 -0.1944,-3.37137 1.7135,-3.22539 1.445,0.10999 0.11,2.13066 1.5119,2.11666 0.568,-0.005 1.3085,-1.64266 1.2095,-2.11666 0.346,0.40701 1.9663,3.64598 2.3183,3.42697 0.712,-0.443 1.9362,0.54496 2.8222,0.50397 0.243,-0.012 2.5964,0.62634 2.7214,0.80635 0.694,0.99799 1.4533,0.43497 2.3183,0 2.8219,-1.41901 4.9862,-2.84947 7.2571,-5.14046 0.812,-0.81799 2.8722,-4.20987 1.4111,-4.93887 0.913,-0.553 0.4458,-2.07002 0.6047,-2.92301 0.2051,-1.09398 0.1461,-1.00648 1.008,-1.71348 0.707,-0.57899 -0.7001,-2.30303 0.5039,-2.41904 0.4191,-0.041 0.3881,-1.65988 0.504,-2.01587 0.33,-1.01299 0.087,-1.67562 0.7056,-2.62062 0.468,-0.71601 0.7297,-1.36766 1.1087,-2.11666 0.896,-1.766 1.7594,-4.68236 1.2095,-6.65236 -0.242,-0.86601 0.1167,-1.07228 0.1008,-1.81428 -0.181,-0.39001 -0.342,-0.81154 -0.504,-1.20952 -0.259,-1.08298 -0.7148,-2.80515 -0.1008,-3.83014 1.323,-2.212 -0.603,-4.58095 0.504,-6.85395 0.94,-1.932 2.6454,-3.5684 1.2095,-5.64442 -0.563,-0.81499 -0.8409,-3.5576 -2.0158,-3.12459 0.729,-0.28999 -0.2162,-2.03844 -0.9072,-2.21745 0.345,-0.89302 -0.034,-1.95742 -0.5039,-2.72142 -0.989,-1.60602 -2.9922,-2.01154 -1.9151,-4.13253 -1.648,0.542 -4.135,-3.45271 -4.8381,-4.53569 -0.66,-1.018 -0.3686,-2.50897 -1.1087,-3.42698 -0.544,-0.67499 -2.2271,-2.0012 -0.9072,-2.82221 -0.798,-0.169 -1.4115,-1.65245 -0.7055,-2.21745 0.772,-0.618 -0.2644,-1.77303 0.7055,-2.41904 -1.8169,-0.483 -0.4447,-2.15182 -2.1166,-2.51983 -0.964,-0.211 -3.0317,-0.69206 -3.1246,-1.91507 -0.254,-3.338 -5.0001,-3.72405 -7.2571,-5.34205 1.258,-0.236 2.6224,0.65073 3.7293,1.10873 -0.186,-1.897 0.4605,0.24445 0.7056,-0.70555 0.457,-1.774 -3.9346,-2.38842 -4.6365,-2.72142 -1.327,-0.628 -2.3932,-1.51288 -3.8302,-2.01587 -1.175,-0.412 -2.3729,-0.8279 -3.4269,-1.5119 -1.792,-1.163 -4.4501,-3.97636 -6.854,-3.72935 -1.347,0.138 -2.0957,0.90896 -3.5277,0.50397 -1.02,-0.288 -2.344,-0.88575 -3.427,-0.60476 1.119,0.187 0.9737,1.13928 1.6127,1.81428 0.688,0.727 0.9231,0.67769 1.4111,1.61269 0.912,1.749 1.68,2.66059 -0.504,3.12459 0.163,-0.803 0.5264,-3.54235 -0.7055,-3.72935 -1.211,-0.184 -0.1095,1.86262 -1.2095,2.62063 -1.043,0.719 -3.4615,1.40435 -4.6365,0.80634 -0.329,-0.167 0.7754,-2.99017 1.2095,-3.32618 -1.438,-0.35399 -3.4797,-0.17509 -4.5357,-1.4111 1.005,0.085 1.2642,-0.96949 1.4111,-1.71349 -0.901,0.264 -3.2946,-0.0643 -2.6206,1.61269 0.332,0.824 1.1216,0.30734 1.6127,0.80635 0.3549,0.361 0.3958,1.0489 0.5039,1.5119 -1.276,0.249 -3.0965,1.15545 -3.6285,-0.70555 -0.263,-0.917 -0.6415,-1.64403 -1.2096,-2.41904 -0.253,-0.344 -1.2693,-0.9891 -1.3103,-1.41111 -0.207,-2.173 -3.2603,-1.93927 -4.7373,-1.81428 -2.137,0.18 -3.8917,-0.0895 -5.9468,-0.70555 -1.435,-0.431 -4.258,-1.61398 -1.915,-3.42697 0.738,-0.571 1.6267,-0.86149 2.1166,-1.71349 0.349,-0.607 -0.053,-1.47285 0.4032,-2.01586 0.433,-0.518 1.2339,-0.41472 1.5119,-1.10873 0.272,-0.676 0.9779,-1.33527 1.5119,-1.81428 0.994,-0.894 0.4036,-3.94673 2.1167,-4.03173 -0.755,-0.332 -1.3027,-0.7197 -1.1088,-1.61269 0.15,-0.69 -0.2005,-2.16264 0.3024,-2.62063 0.733,-0.668 0.661,-1.76704 1.4111,-2.41904 0.733,-0.636 1.2767,-0.90207 1.6127,-1.91507 0.265,-0.797 -0.194,-1.91585 1.0079,-2.01586 1.1611,-0.097 1.0839,-0.71291 0.6048,-1.5119 -0.49,-0.814 -1.3897,-2.1346 -1.1087,-3.12459 0.319,-1.12 1.9259,-1.37825 2.5198,-2.31825 0.699,-1.109 -0.1993,-2.30597 0.2016,-3.42697 0.515,-1.442 2.0262,-1.82446 3.3262,-2.21745 1.185,-0.35801 1.9276,-0.0756 3.1246,-0.20159 1.32,-0.139 2.2555,-0.56617 3.6285,-0.40317 3.108,0.37 6.182,-0.3698 9.273,-0.1008 1.221,0.107 1.4142,0.8659 2.3183,1.5119 0.971,0.693 2.0197,-0.11661 3.0238,0.30238 -2.72,-3.012 0.5372,-1.24511 2.3182,-1.4111 1.148,-0.107 2.4869,0.11576 3.5278,0.60476 0.892,0.419 1.6181,-0.34041 2.419,0.20158 -0.185,-0.254 -0.349,-0.53335 -0.504,-0.80634 1.702,-0.576 3.2146,-0.49325 4.6365,0.60476 0.813,0.627 2.0638,-0.23162 3.0238,0.30238 -0.462,-1.263 -3.0392,-0.64946 -2.8222,-2.21746 0.717,0.39975 1.5671,0.0966 2.3183,0.20159 -0.9392,-0.39276 -2.2728,-0.81258 -1.008,-1.71349 1.525,-1.085 4.1971,0.0372 5.846,0.40318 0.5,0.111 6.7788,2.53755 5.9468,0.70555 2.253,-0.413 4.2704,0.63449 6.1484,1.71349 1.415,0.813 3.2089,0.23351 4.4349,1.20952 -0.86,0.71 -2.1146,0.42818 -3.1246,0.40317 -1.047,-0.026 -1.9897,0.44096 -3.0238,0.50397 2.3991,1.649 4.8493,1.76089 7.6603,1.51189 -0.17,-0.239 -0.3399,-0.46355 -0.5039,-0.70555 1.221,-0.458 4.3975,0.0736 5.5436,0.70555 -1.106,-0.693 -0.4147,-1.32507 0.3024,-1.91507 0.876,-0.721 0.5008,-0.72616 2.0158,-0.40317 0.568,0.121 5.6863,1.788 4.2334,0 1.753,-0.443 2.9142,1.03782 4.7372,-0.40317 -1.432,-0.403 -5.0502,-3.84208 -6.7531,-1.91508 0.25,0.189 0.5304,0.35698 0.8063,0.50397 -0.704,-0.303 -1.4834,-0.11337 -2.2174,-0.30238 -0.801,-0.207 -1.4227,-1.93851 -0.1008,-1.20952 -0.556,-0.253 -3.8716,-2.67624 -1.7135,-2.31824 -0.668,-0.75901 -1.0538,-1.80222 -1.0079,-2.82222 0.057,-1.279 0.8259,-1.29765 1.5119,-2.11666 0.378,-0.45399 0.9934,-1.37406 0.8063,-1.91507 -0.281,-0.814 -1.5226,-2.70996 -0.2016,-3.42697 1.077,-0.583 -0.6927,-1.72145 -1.1087,-2.21745 1.152,-0.274 0.6965,-2.21848 2.2175,-1.71349 -0.048,-0.264 -0.051,-0.54233 -0.1008,-0.80634 1.198,0.137 2.4545,0.23596 3.6285,0.50396 0.66,0.15 1.8366,-0.51197 2.6207,-0.50396 1.0889,0.011 1.9648,-0.68134 3.0238,-0.80635 1.0669,-0.125 1.9519,0.74092 2.923,1.00793 1.015,0.279 1.4473,1.5988 2.3182,0.1008 0.529,-0.909 1.1565,-1.66635 2.2175,-0.80635 0.676,0.548 1.2965,1.54926 1.7134,2.31825 0.3751,0.689 -0.602,0.96911 0.504,1.4111 -0.873,0 1.8299,1.25555 -0.4032,0.70555 1.008,0.594 1.9697,1.21645 2.6207,2.21746 0.897,1.376 1.3734,-0.5758 2.7214,-0.1008 1.148,0.405 0.6283,1.76186 1.3103,2.01587 0.736,0.273 0.6731,1.13791 1.4111,1.5119 0.45,-1.062 1.5266,-1.59072 2.6206,-1.10873 -1.381,-1.446 0.7056,-3.38087 0.7056,-4.93887 0,-1.891 1.4511,-2.41586 2.923,-2.01586 -1.826,-0.54 0.1863,-0.79398 0.8063,-0.50397 0.749,0.35 1.3327,0.93553 2.1167,1.20952 -0.158,-0.232 -0.2732,-0.45755 -0.4032,-0.70555 0.237,-0.127 0.4626,-0.18939 0.7056,-0.30238 -1.167,-1.191 -2.8598,-0.83424 -4.0318,-2.31825 -0.582,-0.734 -3.2039,-2.25839 -2.0158,-3.22538 -0.861,-0.079 -1.7886,-0.43131 -2.1167,-1.31032 2.292,-0.711 5.0997,-1.88335 7.4587,-0.80634 2.397,1.093 5.0065,1.0081 7.5595,1.4111 -1.022,0.243 -1.9998,-0.10079 -3.0238,-0.10079 0.676,0.508 1.4806,0.71072 2.2175,1.10873 -1.201,-0.521 -4.2019,-1.87694 -5.4429,-1.00794 -2.137,1.495 0.7251,3.43312 1.9151,4.33411 1.019,0.771 1.0659,1.43783 1.5119,2.51984 0.575,1.39399 1.8572,0.84068 2.8222,1.61269 1.222,0.979 0.092,0.99865 0.3024,2.11666 0.197,1.041 1.3774,1.2051 2.2174,1.4111 -0.28,0.114 -1.1441,1.24172 -1.4111,1.10873 -0.599,-0.299 -1.5356,-0.39501 -2.1166,0 -0.351,0.238 -0.2557,0.92653 -0.6048,1.20952 -0.783,0.633 -1.6799,1.0637 -2.5198,1.61269 0.439,0.66 1.3276,0.93549 1.6127,1.71349 0.205,0.559 -0.7992,1.11669 -0.9072,1.61269 -0.217,0.991 0.2827,1.8732 -0.3023,2.82221 -0.568,0.92 -1.4181,0.0201 -1.008,1.41111 0.473,1.606 -1.5243,2.9959 0.1008,4.4349 1.196,1.06 -0.3309,2.55176 0.9072,3.52777 -1.1421,-0.689 -0.6121,0.008 -1.008,0.70555 -0.247,0.434 -0.5293,0.89531 -0.8063,1.31031 -0.452,0.673 -0.8944,1.52127 0.2016,1.81428 -1.23,-0.597 -2.5862,0.42942 -3.8302,-0.20159 -0.287,0.494 0.4285,1.75187 -0.2016,2.01587 -1.019,0.427 -2.326,-0.56738 -3.4269,-0.30238 0.742,1.288 2.3659,1.76642 3.4269,2.72142 0.922,0.831 0.8437,2.19159 1.6127,3.12459 -1.039,-0.378 -1.2574,1.98801 -1.2095,2.92301 -0.432,-1.685 -0.3715,0.17875 -0.2016,0.60476 0.303,0.76 0.8293,1.36087 1.3103,2.01586 1.002,1.364 1.9698,2.80954 3.0238,4.13253 2.784,3.494 5.4618,6.55057 8.8698,9.47456 3.981,3.415 7.9966,6.87171 11.8936,10.38171 2.455,2.211 6.2227,7.72908 9.8778,7.76108 -1.463,0.149 0.029,0.87211 0.6047,1.41111 0.6571,0.616 1.334,1.1245 2.0159,1.71349 1.368,1.183 2.6702,2.43515 3.8302,3.83014 1.028,1.235 2.036,2.56697 3.4269,3.42697 1.199,0.74 2.7453,1.24143 3.2254,2.72142 0.919,-0.957 1.1263,-2.15237 1.8143,-3.22538 0.473,-0.737 -0.3535,-0.98549 0.3024,-1.71349 1.604,-1.776 0.6914,-4.45878 -0.2016,-6.45077 -0.38,-0.85 -1.3738,-1.26207 -2.0159,-1.91507 -0.468,-0.475 -0.2745,-0.96412 -0.7055,-1.41111 -0.428,-0.444 -0.938,-1.71729 -1.5119,-1.81428 -0.594,-0.101 -1.2353,0.003 -1.3103,-0.80634 0.87,-0.379 1.6344,0.39613 2.2174,0.90714 0.754,-0.912 2.1357,-2.62063 3.5278,-2.62063 0.62,0 1.1128,-0.52641 1.5119,0.20159 0.332,0.604 1.4058,0.81713 2.0158,0.90714 -0.893,-1.637 -2.8343,-2.15639 -4.2333,-3.22539 1.713,0.838 -1.7947,-1.98304 -2.1166,-2.41904 -0.829,-1.124 -2.1378,-2.48014 -1.1088,-3.83014 0.92,-1.206 0.965,-2.84441 2.923,-2.72142 0.933,0.058 1.6826,0.89134 2.6207,0.80635 0.702,-0.064 1.4157,-0.32079 2.1166,-0.1008 -0.174,-0.749 0.6459,-1.74304 0.1008,-2.41904 -0.699,-0.867 -1.973,-1.37707 -2.923,-1.91507 -1.822,-1.033 -3.5916,-1.97776 -5.0397,-3.52776 -1.254,-1.342 -1.6668,-2.87191 -2.5198,-4.43491 -0.653,-1.198 2.139,-2.43101 -0.504,-2.923 0.273,-0.894 1.2749,-1.06977 2.0159,-0.60476 -0.163,-0.664 0.7214,-0.90638 0.8064,-0.30238 0.1179,0.838 -0.9124,0.83928 -0.3024,1.81428 0.654,1.044 3.3424,2.60651 4.2333,1.20952 0.88,-1.379 -3.9621,-4.78966 -5.342,-5.03967 -1.553,-0.282 -2.054,-5.1e-4 -3.427,-1.20952 -0.91,-0.801 -1.6757,-1.65904 -2.6206,-2.41904 -0.475,-0.38099 -1.4593,-0.95247 -1.3104,-1.71348 0.218,-1.121 0.8015,-1.63662 -0.2015,-2.62063 -1.1501,-1.128 -3.1535,-1.25048 -4.6365,-1.71348 -2.3497,-0.73317 -1.1629,0.56458 -0.8064,1.61269 -0.6333,-0.82756 -2.4221,-0.8534 -3.2254,-1.20952 -1.578,-0.701 -2.541,-1.53279 -3.4269,-3.0238 -0.228,-0.383 -2.5105,-6.56442 -0.2016,-5.64442 -0.453,-0.466 -1.0905,-0.82092 -1.7135,-1.00793 0.526,-0.245 1.0867,-0.245 1.6127,0 -0.19,-0.21601 -3.2958,-1.9577 -1.1087,-1.6127 -0.307,-0.526 -3.0705,-1.74944 -2.7215,-2.21745 0.778,-1.048 -1.4303,-1.24282 -1.3103,-2.51983 0.592,0.176 0.976,-0.19634 1.008,-0.80635 -0.361,-0.04 -0.7497,-0.0448 -1.1088,-0.10079 1.481,-1.103 3.226,0.87499 4.4349,0 -1.647,-0.57 0.7548,-0.77159 1.6127,-0.20159 0.761,0.506 1.3725,1.2427 2.2175,1.6127 0.986,0.433 2.0044,-0.43012 1.3103,-1.41111 -0.438,-0.62 -1.6583,-0.67614 -2.3182,-0.90714 0.903,-0.068 1.3515,0.056 2.1166,-0.50397 0.853,-0.625 0.8349,-1.85625 1.5119,-2.31824 0.86,-0.587 2.2862,1.22448 2.8222,1.71348 0.66,0.601 2.4124,4.57939 3.7294,3.22539 0.887,-0.912 0.2389,-3.05854 0,-4.13253 0.51,0.097 1.0139,0.14501 1.5119,0 -0.203,-0.25 -0.3798,-0.47554 -0.6048,-0.70555 1.631,0.924 3.5335,-0.79094 5.1405,-1.00793 1.718,-0.233 4.0471,-0.2362 5.7452,0.10079 4.151,0.823 8.0466,3.54125 11.8936,5.24125 0.765,-1.053 -1.9632,-3.77753 -2.8222,-4.13252 1.007,0.201 0.7924,0.0102 0.3024,-0.60476 0.955,0.229 3.8039,-0.64748 2.0158,-1.71349 0.781,0.3 1.546,-1.0561 1.5119,-1.4111 0.943,0.79 1.3149,-0.26457 0.6048,-0.70556 -0.432,-0.268 -0.9601,-0.38577 -1.4111,-0.60476 0.79,0.089 1.5133,0.49597 2.3182,0.50397 -0.36,-0.48 -0.8979,-0.78035 -1.5119,-0.80635 0.99,-0.037 1.036,-0.25776 0,-0.60476 0.322,0 1.2699,0.273 1.5119,0 0.483,-0.547 -0.2726,-0.97193 -0.7055,-1.00793 0.526,0.096 0.986,0.20637 1.5119,0.30238 -0.63,-0.726 -1.552,-0.86372 -2.4191,-1.10872 1.008,-0.65 2.0808,0.70719 3.0238,-0.1008 -0.523,-0.466 -0.8336,-0.95513 0.3024,-0.90714 -0.54,-0.359 -1.0502,-0.77231 -1.4111,-1.31031 0.658,-0.052 1.269,0.27638 1.9151,0.30238 0.775,0.031 1.3786,-0.65694 0.2016,-1.00793 1.164,-0.229 2.4537,0.41762 3.5277,-0.30238 -0.406,-0.459 -0.9108,-0.79014 -1.5119,-0.90714 1.073,0.059 1.9439,0.12482 2.923,-0.40318 -0.156,-0.271 -0.327,-0.54834 -0.5039,-0.80634 0.96,0.296 2.9145,-0.72532 3.6285,-1.31031 1.468,-1.20401 1.6522,-2.00566 3.8302,-2.11666 -1.087,-0.23601 -2.1649,-0.18793 -3.0238,-1.00794 -0.49,-0.466 -0.8709,-1.0273 -1.5119,-1.31031 1.103,0.265 1.9575,1.17651 3.1246,1.20952 -1.469,-1.684 1.4016,0.38977 1.7135,0.60476 1.0859,0.75 2.9743,0.43676 4.2333,0.60476 2.847,0.381 6.0222,0.64228 8.6682,1.81428 -0.112,-0.564 -0.5457,-0.96674 -1.1087,-1.10873 0.951,-0.059 0.9104,-1.61466 0.3023,-2.11666 -1.239,-1.02299 -3.0688,-0.99529 -4.4349,-1.81428 -1.134,-0.679 -1.2335,-1.71866 -2.7214,-2.11665 -1.286,-0.344 -2.6068,-0.73415 -3.9309,-0.90714 0.2675,-0.0365 0.5546,-0.048 0.8063,0 -1.1948,-0.5784 -2.4228,-1.16732 -3.5277,-1.91508 0.488,0.397 1.0476,0.73294 1.6127,1.00794 -1.246,0.585 -2.2813,-1.39098 -3.7294,-0.50397 0.034,-1.09 1.2333,-0.67618 1.8143,-0.40317 -1.056,-0.376 -1.056,-0.48639 0,-0.30238 -0.492,-0.382 -0.9999,-0.75173 -1.5119,-1.10873 -0.646,-0.446 -1.3476,-0.89373 -2.1167,-1.10872 -0.572,-0.159 -1.1584,-0.28097 -1.7135,-0.50397 -0.3529,-0.142 -1.6468,-1.07017 -0.1008,-0.40317 -1.491,-0.985 -3.1582,-1.59705 -4.7372,-2.41904 -0.352,0.542 -0.1081,0.89992 0.5039,1.00793 -2.276,0.159 -6.5676,-0.71341 -7.9626,-2.72142 -0.832,-1.196 -2.6165,-2.26576 -3.6286,-0.60476 -0.392,0.642 -1.4292,0.0322 -1.9151,-0.10079 -0.813,-0.223 -1.3456,0.42897 -2.1166,0.50396 0.25,-0.22699 0.4815,-0.45255 0.7055,-0.70555 -1.512,-0.747 -3.0763,-1.39948 -4.7373,-1.71348 1.3811,-0.229 2.8056,1.10196 4.1326,0.50396 1.378,-0.622 2.6458,0.99898 4.0317,0.50397 -0.617,-1.168 -2.5143,-1.79006 -3.7294,-1.91507 -0.725,-0.075 -1.4057,0.23737 -2.1166,0.30238 -0.615,0.055 -1.0855,-0.49417 -1.7135,-0.40318 2.711,-0.917 6.7324,0.30949 9.0714,1.71349 2.573,1.545 5.8173,1.16393 8.6682,1.00793 -0.165,0.241 -0.336,0.46655 -0.5039,0.70555 0.5259,0.056 2.555,-1.18669 2.923,-1.61269 0.3679,-0.428 0.2351,-0.91412 0.4031,-1.4111 0.212,-0.622 0.8803,-0.47134 1.3103,-0.80635 0.722,-0.564 -0.6834,-2.05345 -1.2095,-2.21745 -0.4681,-0.14656 -2.4604,-0.0819 -2.0158,0.40317 -2.1172,-0.97968 2.0018,0.009 1.7134,-1.31031 -0.256,-1.174 -4.2972,-1.65086 -5.2412,-2.01587 -1.446,-0.559 -3.3541,-1.49859 -3.8302,-3.12459 0.611,0.811 1.6552,0.616 2.3183,0 -0.787,-0.298 -1.5512,-0.66493 -2.3183,-1.00793 0.5,0.124 0.9979,0.12679 1.5119,0.10079 -0.255,-0.23 -0.4275,-0.40276 -0.7055,-0.60476 1.335,0.456 2.2851,1.84669 3.8301,1.61269 0.014,-0.542 -0.1559,-0.9941 -0.5039,-1.4111 1.394,-0.071 3.6886,2.88141 2.1166,2.72142 1.512,0.595 2.9147,1.51828 4.5357,1.81428 1.399,0.25599 2.097,1.72265 3.427,2.11665 1.415,0.419 3.8212,0.48779 5.2413,0.1008 0.722,-0.197 1.2248,-1.00793 2.0158,-1.00793 0.816,0 1.61,0.22837 2.4191,0.30237 3.5409,0.322 7.2658,1.75656 10.2809,3.62856 1.412,0.877 2.852,1.33127 4.4349,1.81428 0.556,0.17 4.425,1.191 2.419,0 2.876,1.61 6.5852,1.57922 9.6762,2.82221 2.176,0.875 6.0325,1.34708 2.2174,-1.00793 2.093,0.446 3.6406,3.63684 6.0476,2.51983 -0.337,-0.78 -1.026,-1.3991 -1.9151,-1.4111 1.232,-0.684 2.4941,0.97116 3.8302,0.40317 -1.098,-0.893 -4.7913,-2.42128 -6.2492,-1.81428 0.972,-1.13 -1.2271,-1.70292 -1.9151,-1.00793 0,-0.995 1.1975,-0.72414 2.2175,-0.90714 0.594,-0.106 0.4896,-0.52614 0.2016,-0.90714 -1.127,-1.492 -2.6691,-1.3049 -4.3341,-1.5119 0.734,-0.368 1.5613,-0.38479 2.3182,-0.10079 -0.767,-0.445 -1.4209,-0.95769 -2.0159,-1.6127 -0.3319,-0.365 -1.422,0.15441 -1.915,-0.20158 -1.364,-0.982 -2.9467,-1.22169 -4.5357,-1.61269 1.606,-0.26101 3.1244,0.24955 4.6365,0.70555 0.8389,0.253 4.1205,0.31635 2.2174,0.80634 1.029,0.065 1.9969,-0.44816 3.0238,-0.40317 0.812,0.036 1.6051,0.48397 2.4191,0.50397 -1.005,-1.946 -4.3367,-1.70863 -6.0476,-2.62063 2.418,0.356 4.5835,1.55269 7.0555,1.6127 -0.147,-1.32401 -3.2791,-2.18919 -0.9072,-3.32618 -1.0799,-0.745 -2.2169,-0.66793 -3.4269,-1.00794 1.869,0.23701 -0.022,-0.57655 -0.4032,-0.70555 -0.651,-0.222 -1.4958,-0.23161 -2.0159,0.30238 0.314,-1.575 -7.0365,-2.7728 -8.0634,-3.0238 -1.623,-0.397 -3.0027,-1.15727 -4.5357,-1.81428 -2.1463,-0.92131 -4.7057,-0.75308 -7.0556,-0.90714 -0.2012,-0.0855 -2.2483,-0.31587 -2.5198,-0.30238 1.303,0.45901 2.7198,0.28555 4.0317,0.70556 -1.3449,0.215 -2.9014,0.46662 -4.1325,-0.30238 0.203,0.254 0.4277,0.53335 0.6048,0.80634 -0.797,0.25 -3.3288,-1.31031 -1.5119,-1.31031 -0.752,-0.178 -1.5663,0.099 -2.3183,0 -0.561,-0.073 -0.9469,-0.38317 -1.5119,-0.40317 -1.435,-0.052 -3.4988,0.0166 -4.9388,0.20158 0.51,0.069 9.153,2.07357 7.2571,3.62856 1.1,0.735 2.6311,0.44694 3.8301,1.00794 0.89,0.416 1.3082,1.11807 1.8143,1.91507 -0.454,-0.132 -4.2615,-2.93052 -3.6286,-1.20952 -1.625,0.059 -2.9246,-1.51249 -4.5357,-1.71349 -0.726,-0.091 -3.5685,-1.46093 -1.7134,-1.00793 -1.707,-1.196 -3.755,-1.7968 -5.4429,-3.0238 -1.383,-1.004 -2.6559,-2.20404 -4.4349,-2.41904 0.252,0.177 0.4776,0.29697 0.7056,0.50397 -0.696,0.631 -4.8505,-2.25783 -5.6444,-2.51983 -2.53,-0.833 -5.2679,-1.1445 -7.8619,-1.71349 -1.318,-0.289 -2.5191,-0.69593 -3.8302,-1.00793 -0.883,-0.21 -1.8934,-0.0334 -2.7214,-0.30238 -0.921,-0.3 -1.8132,-0.90476 -2.8222,-0.60476 -0.444,0.132 -2.2399,-0.74917 -0.504,-0.40317 -0.5383,-0.14705 -1.0754,-0.24868 -1.6127,-0.40318 -1.0989,0.764 -3.1884,-0.55613 -4.2333,-0.90714 -1.767,-0.594 -3.5532,-0.48511 -5.2412,-1.4111 0.708,0.12 1.4306,0.28797 2.1166,0.50396 -4.4837,-1.69498 -9.2458,-2.36697 -13.9094,-3.32618 -1.6654,-0.69831 -3.4423,-0.90054 -5.1405,-1.51189 -1.603,-0.577 -3.8205,0.35806 -5.1405,-1.00794 0.808,0.038 1.625,0.37797 2.4191,0.50397 -2.909,-1.627 -6.4513,-0.7929 -9.5754,-1.5119 -3.294,-0.758 -6.7016,-1.10207 -9.9785,-1.91507 -3.347,-0.831 -6.0815,-0.42097 -9.4746,-0.50397 -1.56,-0.038 -3.1772,0.046 -4.7373,0 -1.56,-0.046 -2.9577,-0.80576 -4.5357,-0.60476 0.769,0.209 1.4556,0.47256 2.2175,0.70555 -0.732,0.936 -2.9265,-0.20296 -3.7294,-0.50396 -1.284,-0.481 -2.4561,-0.56176 -3.8301,-0.60476 -4,-0.125 -8.1243,-1.27493 -12.0952,-1.00793 1.653,0.423 2.9742,1.54983 4.3341,2.51983 -1.11,0.064 -1.1721,1.3411 0,1.4111 1.228,0.074 2.1622,-0.047 3.3262,0.50397 2.299,1.088 4.6971,1.86997 6.7531,3.42697 -0.245,0.26 -0.4714,0.53635 -0.7055,0.80635 0.506,0.297 0.9581,0.63293 1.4111,1.00793 -0.461,-0.297 -0.913,-0.57534 -1.4111,-0.80635 0.256,0.591 0.7921,0.96473 1.4111,1.10873 -2.113,-0.667 -4.271,0.30742 -6.35,-0.20159 -1.494,-0.366 -2.5029,-1.60365 -3.9309,-2.11665 -3.089,-1.10901 -6.8324,-0.22725 -9.5754,-2.31825 -0.859,-0.654 -6.115,-3.91767 -6.8539,-2.11666 -0.889,2.169 -4.5929,1.54952 -6.35,1.20952 -3.839,-0.744 -7.7084,-0.0395 -11.4904,-1.20952 -1.93,-0.597 -4.0586,0.1308 -6.0476,0.10079 -1.964,-0.03 -3.884,0.0858 -5.846,0.1008 -2.019,0.017 -3.6494,1.17697 -5.6445,0.50396 -1.627,-0.548 -3.3366,-0.79093 -5.0396,-1.00793 -1.592,-0.202 -2.981,-0.32272 -4.4349,-1.10873 -0.444,-0.241 -0.3476,-1.0541 -0.7056,-1.4111 -0.204,-0.204 -0.5008,-0.27201 -0.8063,-0.30238 0.2639,0.012 0.5433,-0.0334 0.8063,0 -1.972,-1.181 -3.3455,-2.66497 -5.6444,-3.42697 -2.124,-0.703 -4.2438,-1.51208 -6.4508,-1.91507 -4.834,-0.88401 -9.8319,-1.1297 -14.7158,-1.6127 -4.757,-0.472 -9.4811,0.38211 -14.1111,1.41111 -4.207,0.935 -8.8823,0.45145 -13.0023,-0.70555 0.767,0.07 1.5583,0.221 2.3182,0 -1.646,-1.49801 -4.3267,-2.05004 -6.4507,-2.41904 -1.361,-0.237 -3.0212,-0.655 -4.3341,0 -0.732,0.366 -1.7996,0.66056 -2.6207,0.70555 1.512,-0.968 1.7417,-1.65107 -0.3023,-1.91507 -1.838,-0.236 -2.3316,0.57524 -3.6286,-0.60476 -1.792,-1.632 -6.2467,1.09669 -7.4587,-1.31031 0.963,-0.031 1.5118,0.47044 1.5119,-0.70556 1.024,0.009 1.9728,0.68676 3.0238,0.60476 1.671,-0.131 -0.9232,-1.59548 -1.3103,-1.71348 -3.408,-1.036 -6.7329,-1.59508 -10.2809,-1.91508 -1.952,-0.175 -3.9869,-0.27559 -5.9468,-0.20158 -1.942,0.073 -1.6803,1.17241 -1.8143,2.72142 -0.177,2.03199 -5.3765,2.08051 -6.5516,1.20952 1.738,-0.177 -0.902,-0.93132 -1.4111,-1.31032 1.547,0.285 3.0774,0.14538 4.6365,0.30238 -0.424,-0.807 -1.2918,-0.67893 -2.0159,-1.00793 -0.648,-0.296 -0.4792,-1.11952 -1.3103,-1.20952 0.956,-0.201 2.4737,0.52451 3.1246,1.20952 0.04,-0.265 0.048,-0.54335 0.1008,-0.80635 1.225,-0.032 1.0298,1.77246 2.0159,2.21746 1.2465,-2.69776 -0.4631,-3.22567 -2.5199,-3.22539 -0.6855,9e-5 -1.4551,0.0575 -2.1166,0.10079 -1.717,0.11301 -4.0255,0.0458 -5.6444,0.60476 -1.12,0.387 -2.1654,0.32284 -3.2254,-0.40317 0.579,-0.125 1.1865,0.0334 1.7135,0.30238 -0.413,-2.329 3.2297,-1.22894 4.5357,-1.00793 1.5218,0.25844 3.121,0.40921 4.7372,0.40317 0.7915,-0.003 1.5639,-0.0752 2.3183,-0.10079 -5.675,-2.431 -13.1068,-1.81424 -19.1507,-2.31825 -3.309,-0.276 -6.6575,-0.40317 -9.9786,-0.40317 -2.872,0 -5.9908,-0.89652 -8.8698,-1.20952 0.714,0.638 1.7366,0.74193 2.6207,1.00793 0.908,0.272 1.7194,1.29752 2.7214,1.20952 -1.37,-0.018 -2.6327,0.72956 -4.0318,0.70555 -1.004,-0.017 -2.1456,-0.33199 -3.1246,0 -1.5456,0.5248 0.045,1.59391 1.1088,2.11666 0.059,-0.19392 1.83,-0.11159 2.3182,-0.20158 0.865,-0.159 1.7696,-0.16921 2.6206,0.10079 -0.5409,0.298 -1.1484,0.3508 -1.7134,0.10079 -0.195,0.622 0.1975,1.1769 0.7055,1.5119 -0.501,0.061 -1.007,0.1148 -1.5119,0.1008 2.638,0.57899 5.3344,1.86217 7.6603,3.32617 -1.892,-0.591 -3.5248,0.72141 -5.4428,-0.20158 -0.619,-0.298 -3.2621,-1.74759 -3.3262,-0.20159 -0.639,-0.792 -1.5791,-0.44918 -2.4191,-0.40317 1.107,2.062 1.5984,1.36607 3.7294,1.91507 -0.547,-0.051 -1.0986,0.007 -1.6127,0.20159 0.284,0.178 0.6111,0.34597 0.9071,0.50396 -0.5299,-0.085 -1.0826,-0.085 -1.6126,0 0.766,0.286 1.6362,0.0668 2.3182,0.60476 -4.449,-0.714 -8.6511,-2.42138 -13.1031,-3.22538 -1.103,-0.199 -2.304,-0.41339 -3.427,-0.30238 -1.007,0.099 -2.0628,1.08593 -3.0238,1.00793 -1.205,-0.098 -0.4438,0.9769 -0.1008,1.5119 -0.914,-1.578 -4.1674,-0.88394 -5.6444,-1.00793 -2.731,-0.229 -4.8259,-1.25266 -7.3579,-2.11666 -0.866,-0.296 -4.2015,-2.5858 -3.6286,-0.1008 0.289,1.25401 1.8114,3.46956 3.2254,3.62856 -1.611,0.428 0.9535,4.59319 -2.2174,3.32618 -2.1041,-0.84 -4.306,-1.01707 -6.35,-1.91507 -2.533,-1.114 -5.3611,-1.24241 -7.7611,-2.72142 -1.027,-0.633 -1.8995,-1.24031 -3.1246,-1.31031 -0.613,-0.035 -1.5287,-0.009 -1.6127,-0.80635 0.842,0.007 1.6809,-0.61059 2.5198,-0.20159 -1.366,-0.857 -2.9537,-1.56507 -4.5357,-1.91507 0.804,0.457 1.7159,0.85331 2.5199,1.31032 -1.328,1.624 -7.7891,-1.63401 -8.769,-2.92301 2.39,0.325 5.1067,2.17555 7.4587,0.70555 -1.3861,0.089 -0.6219,-0.41455 -0.1008,-0.70555 -1.54,-0.275 -2.6744,-1.45748 -4.2333,-1.71349 -1.568,-0.25799 -3.3033,0.30024 -4.7373,-0.60476 0.96,0.023 1.9843,0.48442 2.8222,-0.20158 -0.306,-0.141 -0.6332,-0.30897 -0.9072,-0.50397 0.478,0.091 1.166,0.15162 1.5119,-0.30238 -1.055,-0.106 -2.0645,-0.33917 -3.1246,-0.40317 0.6601,-0.249 1.3909,-0.0212 2.0159,-0.40317 -1.609,-0.33801 -3.3669,-0.18156 -4.9389,-0.70556 0.823,-0.059 1.7636,-0.27172 2.1167,-1.10872 -0.714,-0.502 -1.2375,0.40379 -1.7135,0.10079 -0.269,-0.171 -0.05,-0.66835 -0.4032,-0.80635 -1.512,-0.589 -3.0677,-0.24014 -4.5357,-0.90714 -1.245,-0.565 -2.3783,-0.98593 -3.2253,-1.00793 -1.517,-0.039 -3.9561,-0.92338 -5.3421,-0.30238 0.263,0.197 0.5353,0.31696 0.8064,0.50397 -0.928,0.012 -1.7925,0.37438 -2.7215,0.30238 -0.5839,-0.046 -2.5036,-0.34835 -0.7055,-0.80635 -1.387,-0.615 -2.3743,0.59297 -3.7294,0.50397 -0.41,-0.027 -0.5399,-0.65856 -1.0079,-0.70556 -0.536,-0.039 -1.0058,-0.12938 -1.5119,-0.30238 -1.627,-0.66 -3.232,-0.68493 -4.9389,-1.00793 -1.297,-0.245 -2.6748,0.23562 -3.9309,-0.30238 0.301,0.222 0.6361,0.44755 0.9071,0.70556 -0.729,0.71 -1.5634,0.14761 -2.2174,-0.30238 -0.267,1.847 2.7135,3.15096 4.1325,3.42697 -1.619,0.433 1.4645,1.43027 -0.7055,1.81428 -1.1521,0.203 -2.4565,-0.26859 -3.6286,-0.20159 -2.707,0.158 -5.554,0.34759 -8.2651,0.20159 -3.2439,-0.176 -6.428,-0.60335 -9.6761,-0.80635 -2.244,-0.14 -5.1322,-0.47245 -6.7532,-2.21745 0.6281,-0.706 1.4336,-0.35018 2.2175,-0.40318 -1.911,-2.35299 -8.4002,-0.84273 -11.0873,-1.10872 -5.907,-0.585 -11.8229,-0.64045 -17.6388,0.70555 0.951,0.695 2.4406,1.27093 3.6286,1.00793 -1.554,0.515 -1.881,0.18942 -3.427,-0.20158 -1.583,-0.401 -0.6224,-0.99469 -1.3103,-1.61269 -1.337,-1.20201 -3.6721,-1.83404 -5.3421,-2.41904 0.461,0.484 1.0278,0.88651 1.6127,1.20952 -1.055,1.336 -3.929,0.92055 -5.342,0.70555 -0.768,-0.117 -6.0774,-1.97145 -3.7294,-2.21745 -0.559,-0.32 -1.5526,-0.55763 -1.6127,0.30238 -0.066,0.933 -2.4262,-0.0278 -2.8222,-0.1008 -0.81,-0.151 -2.7179,-0.5072 -3.4269,0.1008 -0.499,0.429 -0.1669,0.80692 -1.008,1.00793 1.627,3.066 5.7129,-1.08886 7.8619,0.90714 -2.527,0.875 -4.8384,1.86052 -7.5595,1.20952 0.481,0.252 0.9859,0.47276 1.5119,0.60476 -0.942,-0.31 -2.004,0.62155 -2.923,0.70555 -0.842,0.077 -1.6777,0.23038 -2.5198,0.30238 1.892,0.377 -1.6287,0.75234 -2.1167,0.80635 -1.297,0.143 -2.6429,-0.11 -3.9309,0 -0.98,0.084 -1.8713,-0.13079 -2.8222,-0.1008 -0.978,0.031 -0.4633,1.11031 0.1008,1.31032 -1.326,1.304 -2.8973,1.8409 -4.7373,1.5119 0.802,-0.019 2.7516,-0.5995 2.6206,-1.71349 -0.085,-0.715 -0.5345,-1.75207 0.7056,-1.91507 0.471,-0.062 2.0158,0.34904 2.0158,-0.50397 0,-0.802 0.9737,-1.3887 1.6127,-1.61269 1.061,-0.373 2.4078,-0.29196 3.5278,-0.50397 1.069,-0.203 0.7026,-0.72231 1.2095,-1.31031 1.041,-1.209 3.8463,-0.83863 4.2333,-2.62063 -0.714,-0.438 -1.5563,-0.18117 -2.3182,-0.40317 1.079,-0.286 2.2124,-0.0518 3.2253,-0.60476 -0.2589,-0.175 -0.5382,-0.34297 -0.8063,-0.50397 1.781,-0.291 2.6793,-0.43531 4.2333,-1.31031 0.654,-0.369 1.472,0.332 2.0159,0 0.403,-0.245 0.2421,-0.84151 0.4032,-1.20952 0.21,-0.48 1.4033,-0.38097 1.8142,-0.50396 1.016,-0.305 1.5015,-0.99748 2.2175,-1.71349 1.48,-1.48 -2.4621,-2.60721 -3.3262,-2.82221 -1.13,-0.279 -2.0618,0.58262 -3.0238,-0.30238 -1.002,-0.922 -2.4226,-0.75052 -3.6285,-1.20952 2.855,-0.523 5.7944,1.61669 8.5674,1.61269 -0.595,-0.653 -0.7885,-1.69287 -1.7135,-2.01586 -0.975,-0.339 -1.3463,0.18661 -2.3182,-0.30238 -1.482,-0.748 -2.6557,-2.62866 -4.5357,-2.11666 0.693,0.416 1.4884,0.57713 2.2174,0.90714 -0.833,-0.176 -1.7024,-0.38603 -2.2174,0.50396 -0.197,-0.651 -2.0347,-1.59707 -2.6207,-1.91507 -0.891,-0.481 -2.7836,-1.29538 -3.6285,-0.30238 -0.08,-2.143 -8.5583,-2.48204 -10.0794,-2.41904 -2.0449,0.085 -4.1194,-0.234 -6.1483,0 -2.237,0.258 -4.4385,-0.12321 -6.6524,0.1008 1.602,1.125 -0.7966,1.3821 -1.6127,1.4111 -1.683,0.06 -3.6172,0.55 -5.2412,0 1.203,0.126 3.1396,-2.40679 1.1087,-3.0238 -2.861,-0.869 -6.373,-0.1512 -9.273,0.1008 0.59,-0.07 1.0717,-0.39093 1.1087,-1.00794 -2.888,0.45701 -5.862,0.21639 -8.769,0.30238 1.645,-0.616 3.3896,-0.32334 5.0397,-0.80634 1.152,-0.338 2.1619,-1.07214 3.4269,-0.90714 -1.93,-1.195 -4.1231,-0.89332 -6.2491,-1.31031 -1.8611,-0.36501 -3.6059,-0.53635 -5.4429,-0.80635 -0.4627,-0.068 -0.9671,-0.10253 -1.4111,-0.10079 z m 500.741,64.70928 c -1.6294,-0.46857 -3.2625,-1.0486 -4.9388,-1.31031 1.627,0.444 3.3148,0.85331 4.9388,1.31031 z m -332.7186,97.87028 c -0.045,-0.0779 1.3595,0.0829 1.7135,0.40317 0.617,0.557 0.9266,1.46345 1.2095,2.21745 0.719,1.919 1.5022,3.81143 2.4191,5.64443 0.469,0.936 0.3761,1.95001 0.9071,2.923 0.395,0.722 1.1159,1.22609 1.5119,1.91507 0.775,1.345 -0.8678,2.72873 0.4032,4.03174 -1.198,1.19499 -1.4508,-1.68269 -2.5199,-1.6127 -0.879,0.058 -0.695,2.16443 -0.9071,2.72142 1.123,-0.132 3.2719,-0.74307 3.427,1.00794 0.155,1.75499 -2.8736,0.59461 -2.6207,2.62062 0.4781,3.824 -4.4379,2.55765 -5.846,5.03966 -0.298,0.525 -0.037,1.55031 -1.0079,1.31032 -0.7,-0.173 -1.166,-0.51939 -1.9151,-0.30238 -1.179,0.342 0.022,2.11421 0.1008,2.82221 0.355,3.191 -3.4617,3.17873 -5.5436,4.03173 -2.742,1.437 -6.4794,0.0623 -9.0714,-1.10873 0.99,-0.832 2.5942,-0.95894 3.8301,-1.00793 1.078,-0.042 1.8343,-0.66376 2.8223,-0.60476 1.921,0.115 3.4831,-1.68676 3.8301,-3.52776 0.221,-1.169 0.5945,-1.92305 1.7135,-2.41904 0.622,-0.276 1.3361,-0.6341 1.4111,-1.41111 0.086,-0.887 1.1695,-1.39908 0.2016,-1.91507 -0.763,-0.407 0.4963,-2.03626 0.8063,-2.31825 1.492,-1.361 1.4443,-2.78012 2.8222,-4.33411 1.456,-1.641 0.01,-3.81742 0.3024,-5.64442 0.294,-1.836 0.4169,-4.84936 -0.4032,-6.65236 -0.38,-0.835 -1.6481,-1.60762 -0.9071,-2.62062 0.4095,-0.55875 0.737,-1.14422 1.3103,-1.20952 z" id="path2924"/>
    <path d="M 1528.988,251.784 C 1528.445,251.653 1527.919,251.701 1527.408,251.928 C 1527.935,251.88 1528.462,251.832 1528.988,251.784" id="path2930"/>
    <path d="M 1541.088,145.584 C 1541.518,146.28 1543.892,145.97 1544.578,145.993 C 1546.264,146.05 1547.221,146.885 1548.878,146.203 C 1550.19,145.664 1551.986,145.311 1553.377,144.937 C 1555.212,144.443 1556.978,145.989 1558.878,145.44 C 1558.357,145.28 1557.827,145.16 1557.288,145.08 C 1559.27,144.725 1555.748,144.405 1555.354,144.388 C 1553.829,144.32 1552.304,144.524 1550.804,144.784 C 1547.57,145.342 1544.344,145.283 1541.088,145.584" id="path2932"/>
    <path d="M 1555.268,148.608 C 1556.357,148.931 1557.475,148.536 1558.574,148.536 C 1559.019,148.424 1559.454,148.28 1559.879,148.104 C 1560.381,148.189 1560.884,148.261 1561.39,148.319 C 1561.059,147.454 1560.217,147.504 1559.45,147.528 C 1560.486,146.303 1563.337,147.115 1564.789,146.555 C 1567.122,145.656 1569.721,146.052 1572.12,145.44 C 1570.254,144.004 1567.99,145.43 1565.93,145.007 C 1566.725,144.754 1567.539,144.522 1568.38,144.503 C 1567.301,143.553 1565.598,144.481 1564.35,144.287 C 1564.838,144.104 1565.341,144.032 1565.861,144.071 C 1563.992,143.525 1562.003,144.423 1560.171,144.718 C 1560.923,144.983 1561.735,145.378 1562.551,145.078 C 1560.431,145.954 1559.759,146.248 1557.507,145.891 C 1556.024,145.656 1554.473,146.297 1553.041,146.59 C 1554.413,146.684 1555.814,146.23 1557.151,146.734 C 1556.408,146.827 1555.658,146.806 1554.911,146.806 C 1555.414,147.044 1555.944,147.164 1556.501,147.166 C 1554.512,148.267 1552.225,146.665 1550.161,147.382 C 1552.186,148.436 1554.305,147.473 1556.431,147.814 C 1555.974,147.98 1555.587,148.245 1555.268,148.608" id="path2934"/>
    <path d="M 1562.188,239.688 C 1561.947,240.381 1562.187,240.429 1562.908,239.832 C 1562.668,239.784 1562.429,239.736 1562.188,239.688" id="path2942"/>
    <path d="M 1565.998,148.824 C 1566.625,148.829 1567.186,148.481 1567.518,147.96 C 1566.497,147.709 1565.284,147.696 1564.418,148.392 C 1564.944,148.536 1565.472,148.68 1565.998,148.824" id="path2946"/>
    <path d="M 1565.498,143.28 C 1565.771,143.701 1567.391,142.928 1567.728,142.847 C 1566.928,142.522 1566.065,142.546 1565.498,143.28" id="path2948"/>
    <path d="M 1569.598,149.184 C 1568.785,149.354 1567.956,149.095 1567.149,149.329 C 1569.389,149.552 1571.846,150.191 1574.069,149.545 C 1572.511,149.13 1571.188,148.429 1569.598,149.184" id="path2950"/>
    <path d="M 1577.088,147.888 C 1579.295,148.517 1582.087,148.505 1584.218,147.6 C 1582.813,147.496 1581.513,146.808 1580.094,146.808 C 1579.629,146.838 1579.197,146.971 1578.795,147.206 C 1577.943,147.612 1577.164,147.352 1576.298,147.6 C 1576.562,147.695 1576.824,147.792 1577.088,147.888" id="path2954"/>
    <path d="M 1580.328,146.231 C 1580.841,146.403 1581.344,146.379 1581.838,146.16 C 1581.154,145.75 1580.344,145.473 1579.538,145.656 C 1579.787,145.868 1580.05,146.06 1580.328,146.231" id="path2956"/>
    <path d="M 1582.639,141.912 C 1582.902,141.935 1583.165,141.96 1583.428,141.983 C 1582.948,141.754 1582.441,141.634 1581.908,141.624 C 1582.151,141.72 1582.396,141.816 1582.639,141.912" id="path2958"/>
    <path d="M 1582.988,144.504 C 1583.507,145.167 1585.739,144.963 1586.476,145.003 C 1588.483,145.112 1590.355,144.804 1592.347,144.648 C 1589.256,144.39 1586.059,143.722 1582.988,144.504" id="path2960"/>
    <path d="M 1583.278,146.304 C 1584,146.836 1585.129,146.794 1585.728,146.088 C 1584.947,145.687 1583.932,145.613 1583.278,146.304" id="path2962"/>
    <path d="M 1584.288,143.424 C 1586.954,144.164 1590.231,144.179 1592.986,144.469 C 1595.093,144.691 1598.228,146.103 1600.128,144.576 C 1597.359,144.053 1594.213,143.309 1591.396,143.307 C 1590.049,143.307 1585.263,142.188 1584.288,143.424" id="path2964"/>
    <path d="M 1587.168,146.231 C 1587.898,146.381 1588.695,146.686 1589.398,146.231 C 1588.678,145.943 1587.889,145.943 1587.168,146.231" id="path2968"/>
    <path d="M 1588.898,142.344 C 1590.218,142.86 1592,142.715 1593.379,142.585 C 1595.361,142.397 1597.559,142.469 1599.478,141.912 C 1597.069,141.264 1594.387,141.197 1591.917,141.479 C 1592.415,141.774 1592.944,141.991 1593.507,142.127 C 1591.73,142.188 1589.838,141.332 1588.108,142.055 C 1588.372,142.151 1588.635,142.248 1588.898,142.344" id="path2970"/>
    <path d="M 1598.328,221.112 C 1596.14,220.43 1593.819,219.257 1591.473,219.823 C 1590.704,220.008 1589.885,220.358 1589.378,220.992 C 1588.566,222.006 1589.437,222.601 1589.451,223.621 C 1589.477,225.447 1593.389,225.316 1594.585,225.097 C 1595.661,224.901 1596.847,224.813 1597.758,224.135 C 1598.274,223.752 1599.984,221.161 1600.198,222.983 C 1601.007,221.938 1599.021,221.253 1598.328,221.112" id="path2972"/>
    <path d="M 1589.118,147.672 C 1589.806,147.997 1590.885,147.882 1591.418,147.312 C 1590.608,147.095 1589.898,147.593 1589.118,147.672" id="path2974"/>
    <path d="M 1590.338,145.08 C 1591.85,145.828 1595.38,146.358 1596.748,145.08 C 1594.618,144.714 1592.478,145.002 1590.338,145.08" id="path2978"/>
    <path d="M 1591.848,142.992 C 1595.244,143.417 1598.784,144.897 1602.218,143.928 C 1598.762,143.613 1595.319,143.12 1591.848,142.992" id="path2980"/>
    <path d="M 1593.868,148.392 C 1594.183,149.508 1599.34,148.823 1600.198,148.609 C 1598.882,146.858 1596.138,146.598 1594.078,147.384 C 1594.273,147.622 1594.467,147.863 1594.658,148.105 C 1594.395,148.2 1594.132,148.296 1593.868,148.392" id="path2984"/>
    <path d="M 1600.128,141.552 C 1600.666,141.53 1601.17,141.385 1601.639,141.12 C 1599.078,140.558 1596.237,140.157 1593.718,141.12 C 1595.864,141.052 1597.995,141.371 1600.128,141.552" id="path2986"/>
    <path d="M 1598.828,139.824 C 1600.364,139.762 1603.46,140.379 1604.598,139.032 C 1602.411,138.78 1600.126,138.773 1597.968,139.248 C 1598.998,139.272 1599.022,139.44 1598.038,139.752 C 1598.302,139.776 1598.564,139.8 1598.828,139.824" id="path2992"/>
    <path d="M 1602.648,148.536 C 1603.297,148.686 1606.784,147.907 1604.668,147.528 C 1606.173,146.999 1607.795,147.306 1609.348,147.168 C 1605.812,145.927 1601.713,146.026 1598.038,146.448 C 1599.839,146.535 1600.792,148.513 1602.648,148.536" id="path2994"/>
    <path d="M 1599.118,145.656 C 1599.821,146.369 1601.606,146.12 1602.288,145.511 C 1601.231,145.512 1600.16,145.424 1599.118,145.656" id="path2996"/>
    <path d="M 1603.298,197.136 C 1604.306,197.004 1605.326,197.07 1606.319,197.281 C 1605.492,197.176 1604.545,197.027 1604.089,197.928 C 1604.552,197.74 1605.032,197.668 1605.529,197.712 C 1605.129,198.169 1604.625,198.408 1604.018,198.432 C 1604.156,198.988 1604.583,199.402 1605.168,199.441 C 1604.684,199.624 1604.048,199.716 1603.658,200.088 C 1603.395,200.339 1603.99,201.042 1603.424,201.226 C 1602.211,201.619 1600.807,200.177 1599.574,201.271 C 1598.122,202.559 1599.951,203.986 1600.749,204.87 C 1601.224,205.395 1601.93,205.539 1602.603,205.558 C 1603.171,205.574 1605.695,205.859 1603.948,205.056 C 1605.499,205.581 1607.15,204.807 1608.478,206.064 C 1608.599,205.825 1608.718,205.585 1608.837,205.344 C 1609.078,205.535 1609.318,205.727 1609.558,205.92 C 1609.63,205.343 1609.992,205.063 1610.568,205.2 C 1609.709,207.684 1612.884,205.911 1613.448,205.992 C 1612.869,206.222 1612.362,206.631 1612.008,207.143 C 1613.224,207.797 1614.666,207.614 1615.899,208.223 C 1615.485,207.994 1612.96,207.757 1615.039,208.511 C 1614.325,208.323 1613.573,208.3 1612.879,208.583 C 1613.595,208.698 1614.562,209.197 1614.239,210.095 C 1613.753,209.71 1613.249,209.686 1612.729,210.024 C 1613.958,209.9 1615.269,210.252 1616.414,210.669 C 1617.053,210.902 1617.703,210.516 1618.334,210.746 C 1618.99,210.986 1619.633,211.329 1620.359,211.248 C 1620.11,210.992 1619.847,210.752 1619.569,210.528 C 1620.362,211.201 1622.673,212.494 1623.519,211.277 C 1624.031,210.541 1624.938,210.913 1625.593,211.174 C 1626.048,211.355 1628.75,211.963 1626.559,211.175 C 1627.507,211.565 1630.241,212.587 1631.159,211.823 C 1630.41,211.327 1629.396,211.656 1628.789,210.886 C 1630.936,210.858 1632.885,212.116 1635.049,211.966 C 1634.039,211.653 1632.843,211.745 1631.949,211.103 C 1632.635,211.472 1633.511,211.603 1634.279,211.584 C 1634.792,211.571 1636.842,211.028 1634.979,210.742 C 1636.602,210.577 1634.62,210.117 1634.097,209.96 C 1633.038,209.644 1631.948,209.464 1630.883,209.173 C 1629.14,208.696 1626.826,208.03 1625.571,206.648 C 1624.042,204.963 1621.969,203.682 1620.313,202.105 C 1619.801,201.618 1620.177,200.317 1619.94,200.189 C 1619.078,199.725 1618.882,198.814 1617.837,198.573 C 1618.323,198.667 1618.803,198.642 1619.277,198.501 C 1618.917,198.019 1618.436,197.755 1617.837,197.709 C 1618.296,197.711 1618.801,197.928 1619.244,198.028 C 1620.129,198.228 1619.64,197.051 1618.987,196.917 C 1619.511,197.065 1620.042,197.09 1620.577,196.989 C 1619.932,196.436 1619.063,196.579 1618.347,196.197 C 1618.771,196.398 1622.181,196.627 1621.654,195.566 C 1621.271,194.796 1619.248,195.197 1618.558,195.045 C 1619.549,195.113 1620.858,195.081 1621.658,194.397 C 1620.865,194.397 1620.071,194.397 1619.278,194.397 C 1620.325,194.18 1621.545,194.374 1622.448,193.677 C 1621.225,192.767 1619.035,192.597 1617.562,192.597 C 1615.018,192.597 1613.253,191.479 1610.752,192.551 C 1609.153,193.237 1606.47,192.173 1605.099,193.491 C 1604.685,193.888 1605.744,195.559 1606.318,195.693 C 1604.749,195.406 1602.778,195.214 1601.788,196.773 C 1602.298,196.867 1602.802,196.987 1603.298,197.136" id="path2998"/>
    <path d="M 1601.428,144.432 C 1601.779,145.701 1605.458,144.72 1606.178,144.288 C 1604.638,143.922 1602.953,144.062 1601.428,144.432" id="path3004"/>
    <path d="M 1601.998,141.696 C 1603.26,142.026 1604.818,142.207 1606.038,141.624 C 1604.68,141.379 1603.337,141.285 1601.998,141.696" id="path3008"/>
    <path d="M 1603.298,143.063 C 1603.595,143.075 1603.881,143.027 1604.158,142.919 C 1603.871,142.968 1603.585,143.016 1603.298,143.063" id="path3010"/>
    <path d="M 1605.028,142.344 C 1606.238,142.727 1607.596,142.634 1608.838,142.489 C 1607.595,142.157 1606.294,142.344 1605.028,142.344" id="path3014"/>
    <path d="M 1611.508,190.296 C 1610.888,190.566 1608.626,190.921 1609.639,192.025 C 1609.356,192.163 1609.066,192.283 1608.768,192.384 C 1610.134,193.051 1611.841,191.978 1613.23,191.882 C 1614.126,191.82 1615.063,192.115 1615.942,192.247 C 1616.629,192.351 1619.684,193.297 1618.559,191.88 C 1618.991,192.581 1619.893,192.87 1620.653,193.011 C 1621.333,193.137 1623.819,193.61 1622.879,192.312 C 1623.75,193.025 1624.675,192.192 1625.119,191.449 C 1624.092,191.042 1622.813,191.021 1622.018,190.152 C 1623.123,190.68 1624.954,191.505 1626.092,190.504 C 1626.861,189.828 1621.159,188.876 1620.718,188.712 C 1621.887,189.014 1623.13,188.672 1624.276,189.064 C 1625.051,189.33 1625.943,189.963 1626.799,189.629 C 1628.758,188.865 1624.202,187.391 1623.598,187.344 C 1625.154,187.445 1624.24,186.986 1623.678,186.697 C 1625.341,186.552 1626.474,188.452 1628.208,187.705 C 1628.055,187.455 1627.888,187.215 1627.708,186.984 C 1628.262,187.365 1630.276,186.905 1628.428,186.697 C 1628.915,186.513 1629.419,186.441 1629.939,186.48 C 1629.322,184.907 1627.719,184.701 1626.409,183.96 C 1627.499,184.298 1628.957,184.974 1630.117,184.702 C 1630.714,184.562 1632.229,183.651 1630.729,183.456 C 1631.789,183.598 1631.789,183.43 1630.729,182.952 C 1631.823,183.254 1633.623,184.557 1634.548,183.24 C 1633.929,183.137 1633.353,182.921 1632.819,182.592 C 1633.542,182.93 1634.366,183.078 1635.161,182.985 C 1635.704,182.921 1637.709,182.537 1635.839,182.089 C 1636.342,182.089 1636.846,182.089 1637.349,182.089 C 1637.388,180.627 1634.845,181.706 1634.12,181.081 C 1634.807,181.096 1636.073,181.375 1636.2,180.361 C 1636.65,180.772 1637.18,180.965 1637.79,180.937 C 1637.326,179.74 1639.5,180.044 1639.59,178.92 C 1640.299,179.892 1641.197,179.141 1641.912,178.636 C 1642.969,177.89 1644.341,177.871 1645.577,177.645 C 1646.344,177.505 1649.047,177.399 1649.493,176.813 C 1649.74,176.489 1652.423,176.349 1653.115,176.213 C 1659.963,174.873 1666.937,174.368 1673.07,170.784 C 1672.309,170.725 1671.871,169.518 1671.145,169.13 C 1669.664,168.338 1667.905,168.044 1666.245,167.933 C 1663.285,167.736 1659.487,168.215 1657.15,170.207 C 1657.6,170.513 1658.084,170.753 1658.599,170.927 C 1657.556,171.131 1656.543,170.621 1655.499,170.783 C 1654.855,170.883 1654.258,171.377 1653.593,171.296 C 1652.888,171.21 1652.314,172.076 1651.503,172.184 C 1649.956,172.39 1648.38,172.176 1646.842,172.48 C 1644.282,172.985 1642.355,173.015 1639.742,172.78 C 1638.426,172.661 1637.127,172.534 1635.803,172.61 C 1634.859,172.664 1633.488,173.532 1635.119,174.024 C 1634.315,173.981 1633.596,174.801 1632.819,174.24 C 1633.078,174.064 1633.341,173.896 1633.609,173.735 C 1632.565,173.573 1631.251,173.559 1630.369,174.24 C 1630.925,174.176 1631.452,174.271 1631.949,174.528 C 1630.358,174.472 1628.41,174.501 1626.955,175.226 C 1625.955,175.725 1624.544,175.969 1623.464,176.306 C 1622.996,176.452 1620.083,176.835 1621.949,177.264 C 1621.166,177.427 1620.347,177.501 1619.639,177.912 C 1620.209,178.047 1620.7,178.427 1621.008,178.92 C 1619.501,179.113 1617.622,177.772 1616.399,179.208 C 1616.638,179.376 1616.878,179.544 1617.119,179.712 C 1615.967,181.477 1613.206,178.702 1612.35,179.954 C 1611.055,181.849 1617.567,180.715 1615.178,181.368 C 1615.664,181.402 1616.144,181.473 1616.618,181.584 C 1615.617,181.784 1614.38,181.744 1613.597,182.521 C 1614.549,182.834 1615.522,182.393 1616.465,182.787 C 1617.284,183.129 1618.179,183.306 1619.068,183.241 C 1617.891,183.42 1616.793,182.9 1615.596,183.12 C 1615.049,183.221 1613.163,182.987 1612.831,183.495 C 1612.201,184.463 1615.993,184.601 1616.547,184.68 C 1615.052,184.865 1613.188,184.379 1611.938,185.472 C 1612.793,185.483 1613.636,185.717 1614.388,186.12 C 1613.495,185.859 1611.962,185.575 1611.148,186.192 C 1612.697,186.276 1614.359,186.482 1615.757,187.2 C 1614.561,187.262 1613.389,186.778 1612.198,186.712 C 1611.358,186.665 1608.838,188.048 1610.637,188.136 C 1608.841,188.597 1607.063,189.121 1605.307,189.72 C 1606.696,189.821 1607.993,191.28 1609.402,190.381 C 1610.699,189.553 1612.37,189.937 1613.807,189.937 C 1613.027,189.976 1612.292,190.277 1611.508,190.296" id="path3018"/>
    <path d="M 1610.208,209.088 C 1611.145,207.109 1608.854,207.253 1607.828,206.28 C 1608.341,206.455 1608.848,206.647 1609.348,206.856 C 1608.546,206.128 1605.694,205.419 1605.458,207.072 C 1607.168,207.504 1608.91,207.775 1610.208,209.088" id="path3020"/>
    <path d="M 1606.758,149.544 C 1606.635,149.568 1606.512,149.592 1606.389,149.616 C 1607.491,149.576 1608.618,150.113 1609.708,149.76 C 1608.711,149.722 1607.763,149.259 1606.758,149.544" id="path3022"/>
    <path d="M 1608.628,141.912 C 1608.435,141.935 1608.241,141.96 1608.048,141.983 C 1609.247,142.14 1610.726,142.494 1611.868,141.912 C 1610.787,141.949 1609.707,141.777 1608.628,141.912" id="path3024"/>
    <path d="M 1608.908,146.088 C 1611.082,147.159 1614.298,146.68 1616.681,146.666 C 1618.26,146.657 1621.16,146.559 1621.797,144.72 C 1618.68,142.984 1614.687,144.718 1611.358,144.504 C 1613.245,144.313 1610.92,144.354 1610.639,144.378 C 1609.697,144.458 1608.873,144.87 1608.188,145.512 C 1608.406,145.731 1608.646,145.923 1608.908,146.088" id="path3026"/>
    <path d="M 1608.768,149.184 C 1610.45,149.397 1612.179,149.97 1613.809,149.184 C 1612.299,148.5 1610.159,147.939 1608.768,149.184" id="path3028"/>
    <path d="M 1610.858,143.496 C 1612.824,144.309 1616.3,144.643 1617.908,142.92 C 1615.571,142.646 1613.112,142.798 1610.858,143.496" id="path3030"/>
    <path d="M 1623.028,144.36 C 1624.632,145.641 1627.072,145.258 1628.959,145.122 C 1631.373,144.947 1633.731,144.812 1636.059,144.072 C 1635.392,142.039 1631.425,142.152 1629.855,142.698 C 1628.853,143.047 1629.893,143.639 1628.668,143.639 C 1626.68,143.64 1624.944,143.762 1623.028,144.36" id="path3040"/>
    <path d="M 1640.309,225.144 C 1640.88,225.59 1642.809,225.185 1643.329,224.785 C 1642.336,225.038 1641.304,224.912 1640.309,225.144" id="path3056"/>
    <path d="M 1644.268,216.072 C 1645.048,216.456 1645.897,216.687 1646.639,217.151 C 1644.446,215.925 1648.256,216.872 1648.762,217.024 C 1649.693,217.303 1650.168,218.166 1651.178,218.267 C 1651.936,218.343 1656.51,218.442 1654.348,216.963 C 1652.583,215.755 1650.471,215.133 1648.484,214.415 C 1647.302,213.987 1645.571,212.691 1644.281,212.877 C 1643.481,212.992 1642.992,213.933 1642.109,213.624 C 1642.654,213.829 1643.204,214.021 1643.758,214.2 C 1643.226,214.302 1642.699,214.278 1642.178,214.128 C 1642.834,214.818 1643.354,215.683 1644.268,216.072" id="path3058"/>
    <path d="M 1650.028,222.336 C 1649.347,221.33 1648.024,221.302 1646.998,220.896 C 1647.919,221.509 1648.97,222.012 1650.028,222.336" id="path3060"/>
    <path d="M 1687.389,222.624 C 1688.689,222.919 1689.903,223.501 1691.208,223.776 C 1689.982,223.239 1688.779,222.53 1687.389,222.624" id="path3062"/>
    <path d="M 1691.928,219.744 C 1692.43,220.05 1692.96,220.098 1693.518,219.888 C 1693.57,218.738 1692.369,219.202 1691.928,219.744" id="path3064"/>
    <path d="M 1692.358,191.952 C 1692.87,192.986 1693.302,193.022 1693.177,194.248 C 1693.045,195.546 1694.174,194.176 1694.228,194.184 C 1696.932,194.537 1699.412,194.167 1702.148,193.464 C 1700.835,192.575 1700.901,193.106 1699.57,193.101 C 1698.832,193.099 1698.084,192.32 1698.767,191.664 C 1696.633,191.374 1694.406,191.076 1692.358,191.952" id="path3066"/>
    <path d="M 1698.698,192.456 C 1699.367,193.07 1699.631,192.95 1699.488,192.096 C 1699.225,192.216 1698.962,192.336 1698.698,192.456" id="path3070"/>
    <path d="M 1718.858,240.336 C 1718.033,240.186 1717.16,240.45 1716.478,240.336 C 1715.559,240.182 1715.173,239.264 1714.177,239.903 C 1715.758,239.582 1716.652,241.166 1717.988,241.632 C 1719.396,242.122 1720.771,241.131 1718.858,240.336" id="path3076"/>
    <path d="M 1715.618,194.616 C 1716.688,195.457 1718.192,195.404 1719.428,195.84 C 1718.96,195.343 1717.914,194.048 1719.428,194.184 C 1718.196,194.077 1716.562,193.452 1715.618,194.616" id="path3078"/>
    <path d="M 1720.229,191.952 C 1720.918,191.329 1723.381,192.392 1723.178,190.872 C 1722.082,191.016 1720.896,190.863 1720.229,191.952" id="path3082"/>
    <path d="M 1722.389,240.479 C 1721.869,240.489 1721.366,240.584 1720.878,240.767 C 1721.764,241.738 1723.474,241.988 1724.689,241.631 C 1724.03,241.063 1723.229,240.688 1722.389,240.479" id="path3084"/>
    <path d="M 1723.828,191.16 C 1725.083,191.614 1726.481,191.608 1727.788,191.807 C 1726.632,191.24 1725.108,190.919 1723.828,191.16" id="path3086"/>
    <path d="M 1725.988,193.68 C 1726.791,193.806 1727.64,194.159 1728.358,193.536 C 1727.564,193.563 1726.758,193.383 1725.988,193.68" id="path3088"/>
    <path d="M 1733.479,199.44 C 1735.154,200.988 1739.458,199.804 1740.889,198.36 C 1739.737,197.756 1738.371,197.494 1737.078,197.496 C 1735.559,197.498 1734.688,198.743 1733.479,199.44" id="path3090"/>
    <path d="M 1734.268,164.664 C 1734.755,164.846 1735.258,164.942 1735.779,164.952 C 1735.303,164.71 1734.799,164.614 1734.268,164.664" id="path3092"/>
    <path d="M 1734.698,183.815 C 1735.433,184.105 1736.195,184.381 1736.998,184.391 C 1736.4,183.634 1735.488,183.691 1734.628,183.599 C 1735.659,183.807 1735.683,183.879 1734.698,183.815" id="path3094"/>
    <path d="M 1740.389,174.96 C 1741.377,175.325 1742.448,175.281 1743.479,175.392 C 1742.484,175.044 1741.418,175.1 1740.389,174.96" id="path3096"/>
    <path d="M 1740.458,195.552 C 1741.103,197.181 1744.068,196.799 1745.428,196.92 C 1745.766,195.861 1744.86,195.38 1744.065,194.981 C 1742.318,194.102 1741.937,194.465 1740.458,195.552" id="path3098"/>
    <path d="M 1745.348,177.768 C 1745.085,177.768 1744.822,177.768 1744.559,177.768 C 1745.662,179.128 1743.83,178.48 1743.409,178.128 C 1743.075,179.018 1743.974,179.342 1744.662,179.067 C 1745.145,178.874 1745.923,179.868 1746.429,179.928 C 1746.251,179.696 1746.084,179.456 1745.929,179.208 C 1746.19,179.105 1746.453,179.009 1746.719,178.92 C 1746.293,178.458 1745.767,178.242 1745.14,178.273 C 1745.208,178.104 1745.278,177.936 1745.348,177.768" id="path3100"/>
    <path d="M 1746.868,175.104 C 1747.389,174.954 1747.916,174.93 1748.448,175.032 C 1747.089,174.396 1745.283,174.318 1743.838,174.672 C 1744.836,174.92 1745.925,174.601 1746.868,175.104" id="path3102"/>
    <path d="M 1746.288,191.376 C 1745.855,190.938 1745.231,190.765 1744.639,190.944 C 1745.165,191.178 1745.715,191.321 1746.288,191.376" id="path3104"/>
    <path d="M 1754.568,186.84 C 1754.328,186.864 1754.088,186.888 1753.847,186.912 C 1754.314,187.273 1754.841,187.489 1755.427,187.559 C 1755.171,187.284 1754.885,187.044 1754.568,186.84" id="path3108"/>
    <path d="M 1756.368,187.344 C 1757.596,187.311 1758.925,187.94 1760.118,187.416 C 1759.265,186.83 1757.183,186.52 1756.368,187.344" id="path3110"/>
    <path d="M 1762.639,187.776 C 1763.432,188.096 1764.292,188.261 1765.149,188.208 C 1764.498,187.448 1763.472,187.902 1762.639,187.776" id="path3116"/>
    <path d="M 1763.858,184.464 C 1763.594,184.536 1763.332,184.608 1763.068,184.68 C 1763.506,185.114 1764.033,185.306 1764.648,185.256 C 1763.593,184.956 1763.593,184.788 1764.648,184.752 C 1764.385,184.656 1764.122,184.56 1763.858,184.464" id="path3118"/>
    <path d="M 1764.938,183.888 C 1764.698,183.96 1764.458,184.033 1764.217,184.105 C 1764.486,184.11 1764.727,184.038 1764.938,183.888" id="path3120"/>
    <path d="M 1766.158,182.376 C 1765.685,182.624 1765.181,182.744 1764.648,182.735 C 1765.593,182.97 1766.703,183.256 1767.677,183.024 C 1767.267,182.584 1766.76,182.368 1766.158,182.376" id="path3124"/>
    <path d="M 1768.538,184.248 C 1767.825,183.984 1767.059,183.913 1766.309,184.032 C 1766.675,184.548 1768.424,185.423 1768.538,184.248" id="path3126"/>
    <path d="M 1766.878,185.04 C 1767.407,185.096 1767.938,185.144 1768.468,185.184 C 1767.952,184.981 1767.422,184.933 1766.878,185.04" id="path3128"/>
    <path d="M 1770.048,167.04 C 1769.145,166.765 1768.929,167.005 1769.399,167.76 C 1769.764,167.64 1769.979,167.399 1770.048,167.04" id="path3130"/>
    <path d="M 1770.188,184.896 C 1770.428,184.752 1770.667,184.608 1770.908,184.464 C 1770.395,184.019 1769.279,183.674 1769.328,184.752 C 1769.615,184.8 1769.901,184.848 1770.188,184.896" id="path3132"/>
    <path d="M 1771.059,186.768 C 1770.533,186.72 1770.005,186.672 1769.479,186.624 C 1769.987,186.923 1770.514,186.971 1771.059,186.768" id="path3134"/>
    <path d="M 1773.148,181.224 C 1772.172,181.298 1770.757,181.039 1769.978,181.8 C 1770.935,182.4 1772.089,182.17 1773.148,182.304 C 1771.164,181.827 1773.393,181.848 1773.797,181.872 C 1773.581,181.656 1773.365,181.439 1773.148,181.224" id="path3138"/>
    <path d="M 1771.059,166.896 C 1771.761,166.838 1772.53,166.982 1772.859,166.176 C 1772.12,165.88 1771.299,165.888 1770.559,166.176 C 1770.698,166.435 1770.865,166.674 1771.059,166.896" id="path3140"/>
    <path d="M 1771.268,182.376 C 1771.732,182.678 1772.239,182.798 1772.789,182.735 C 1772.307,182.51 1771.8,182.39 1771.268,182.376" id="path3142"/>
    <path d="M 1775.378,167.04 C 1775.091,167.04 1774.805,167.04 1774.518,167.04 C 1775.126,167.246 1775.75,167.318 1776.389,167.256 C 1776.052,167.184 1775.715,167.112 1775.378,167.04" id="path3148"/>
    <path d="M 1776.309,188.567 C 1775.823,188.346 1775.32,188.178 1774.798,188.063 C 1775.27,188.328 1775.772,188.496 1776.309,188.567" id="path3150"/>
    <path d="M 1722.889,239.976 C 1724.438,238.972 1727.285,238.365 1727.859,240.768 C 1726.431,240.93 1723.425,242.12 1722.889,239.976" id="path3264"/>
    <path d="M 2196.358,210.168 C 2195.566,209.974 2194.859,209.543 2194.058,209.376 C 2194.582,210.075 2195.515,210.402 2196.358,210.168" id="path3342"/>
    <path d="M 2245.388,216.072 C 2243.438,217.624 2254.992,219.187 2255.328,219.24 C 2255.434,217.562 2253.385,216.886 2252.17,216.756 C 2250.081,216.53 2247.453,215.423 2245.388,216.072" id="path3344"/>
    <path d="M 2292.988,205.128 C 2291.519,206.092 2290.019,207.768 2292.131,208.985 C 2292.838,209.393 2294.111,210.196 2294.963,210.168 C 2295.75,210.142 2298.214,209.76 2298.772,209.202 C 2299.784,208.191 2302.666,209.342 2303.86,209.489 C 2305.339,209.672 2309.619,209.06 2310.699,208.008 C 2309.255,205.798 2305.294,205.92 2303.012,205.419 C 2301.423,205.069 2299.986,204.606 2298.326,204.604 C 2297.092,204.602 2293.944,204.453 2292.988,205.128" id="path3346"/>
    <path d="M 2409.338,255.24 C 2410.13,255.604 2411.004,255.359 2411.718,254.952 C 2410.858,254.535 2408.992,253.479 2409.338,255.24" id="path3360"/>
    <path d="M 2409.559,255.672 C 2410.029,255.94 2410.532,256.108 2411.069,256.176 C 2410.621,255.84 2410.117,255.672 2409.559,255.672" id="path3362"/>
    <path d="M 2318.038,292.392 C 2318.292,292.619 2318.532,292.859 2318.758,293.113 C 2318.083,293.126 2318.143,292.923 2318.038,292.392" id="path3364"/>
    <path d="M 2221.918,433.008 C 2221.63,432.8 2221.32,432.632 2220.988,432.504 C 2220.959,433.296 2221.269,433.464 2221.918,433.008" id="path5504"/>
  </g>
  <g class="landxx coastxx ht" id="ht" transform="translate(-29.9017, -45.0745)">
    <path class="landxx ht" d="M 743.905,643.392 C 745.623,644.105 746.204,643.845 747.865,643.896 C 745.882,644.139 748.759,644.637 749.232,644.695 C 750.349,644.831 751.433,644.863 752.552,645.049 C 754.079,645.301 753.786,644.686 755.01,644.276 C 755.788,644.016 757.032,644.589 757.153,643.609 C 757.367,641.869 754.092,641.316 754.542,639.063 C 754.986,636.841 755.325,635.405 752.473,634.825 C 751.354,634.598 748.678,634.714 750.923,633.135 C 753.519,631.31 755.495,632.517 758.077,633.271 C 759.781,633.769 762.83152,633.03923 762.9704,635.47501 C 763.01582,636.27159 762.26209,637.39613 762.8252,638.3024 C 763.43618,639.28574 761.8327,639.95412 761.66677,640.5382 C 761.5502,640.94852 762.23739,641.0228 762.13939,641.8438 C 762.00739,642.9568 760.846,643.739 759.744,643.681 C 761.01658,644.79412 761.98885,646.94104 760.897,648.577 C 759.366,646.424 756.672,646.709 754.416,647.281 C 752.25,647.83 747.205,645.217 745.857,647.705 C 744.83,649.601 743.585,646.448 742.03,646.184 C 739.53,645.759 742.112,642.216 743.905,643.392" id="path5512"/>
    <path d="M 747.216,648.432 C 746.952,648.36 746.688,648.288 746.424,648.216 C 746.98,648.002 746.854,648.234 747.216,648.432" id="path5514"/>
    <path d="M 753.625,642.888 C 753.43,642.743 749.244,640.963 750.72,640.66 C 751.661,640.467 754.383,641.563 753.625,642.888" id="path5516"/>
    <path d="M 755.569,631.584 C 755.329,631.512 755.089,631.44 754.849,631.368 C 755.403,630.881 756.136,631.05 756.577,631.584 C 756.241,631.584 755.905,631.584 755.569,631.584" id="path5518"/>
  </g>
  <path class="landxx ba" d="M 1386.4063,370.22155 C 1386.3253,367.87755 1388.2853,369.54255 1389.1893,370.27755 C 1390.1673,371.07355 1390.8473,370.40255 1391.2363,369.39455 C 1391.6103,368.42655 1393.0883,369.07755 1393.7563,369.21355 C 1393.8743,368.27155 1397.8303,369.86155 1398.7263,369.86155 C 1399.3453,369.86155 1400.4753,370.49155 1400.9763,370.13855 C 1402.1333,369.32555 1401.8333,370.30355 1402.8613,370.07655 C 1404.1143,369.79955 1405.2083,370.11255 1406.3803,370.58355 C 1407.3593,370.97855 1406.7443,371.92555 1408.0043,371.99455 C 1408.7893,372.03655 1409.8613,371.19455 1410.5483,371.53755 C 1411.6513,372.09055 1410.7283,373.61155 1410.2713,374.22555 C 1408.7513,376.26755 1410.4313,376.96355 1411.9203,378.09655 C 1414.4883,380.05155 1411.1783,379.08455 1410.8013,379.62655 C 1410.7533,379.69555 1412.5623,381.96155 1412.5743,382.51855 C 1412.5853,383.00455 1409.1613,383.65855 1408.5863,383.46855 C 1408.4463,384.37655 1409.5733,384.74355 1409.1663,385.70055 C 1408.6073,383.85855 1405.9423,386.19455 1406.7863,387.71655 C 1403.9699,387.09372 1407.5684,391.65654 1405.8228,391.62508 C 1404.7154,391.60513 1402.7713,390.26455 1401.9203,389.58255 C 1401.4693,389.22255 1401.4072,388.78974 1400.7774,388.70768 C 1400.195,388.63179 1400.0413,389.20324 1399.4473,388.36455 C 1401.3263,388.33955 1397.7617,385.35074 1397.4314,384.94079 C 1395.7397,382.84079 1394.4843,382.13355 1392.9483,380.40555 C 1392.1603,379.51855 1391.1173,379.05555 1390.3113,378.22055 C 1389.3633,377.23855 1389.9741,375.65755 1389.0751,374.56255 C 1388.3921,373.73055 1387.5163,373.51555 1386.6803,372.92855 C 1386.2583,372.63555 1386.4803,370.85955 1386.4063,370.22155" id="ba"/>
  <g class="landxx coastxx in" id="in">
    <path d="M 1977.8163,654.18955 C 1977.6223,654.53155 1977.4683,654.42655 1977.8163,654.90955 C 1977.8163,654.66855 1977.8163,654.42955 1977.8163,654.18955" id="path5460"/>
    <path d="M 1983.7263,685.65355 C 1983.7363,685.89455 1983.7593,686.13455 1983.7963,686.37455 C 1983.7733,686.13355 1983.7493,685.89355 1983.7263,685.65355" id="path5466"/>
    <path d="M 1974.5773,666.28553 C 1974.4063,665.93753 1974.2173,665.60153 1974.0073,665.27653 C 1973.3693,665.80253 1972.9953,666.83453 1973.1713,667.64453 C 1973.6163,669.69153 1975.2653,667.52353 1974.5773,666.28553" id="path5450"/>
    <path d="M 1975.1573,647.99753 C 1973.7323,649.18853 1975.1703,652.75953 1975.2973,654.18853 C 1973.5113,653.85753 1973.1593,659.80553 1975.5173,659.58853 C 1974.4713,658.85353 1975.5873,658.77553 1975.5873,657.94253 C 1975.5873,656.94853 1975.0013,656.01353 1975.3363,655.01453 C 1975.5913,654.25653 1976.2543,653.60253 1976.0973,652.74053 C 1975.8523,651.39353 1976.4613,652.29553 1976.7103,651.06553 C 1976.9603,649.82653 1975.5593,648.56653 1975.8573,647.61053 C 1976.1493,646.67653 1977.9153,644.73653 1976.0863,644.32353 C 1976.8393,643.74053 1976.7133,642.83553 1976.1663,642.16453 C 1974.3373,643.57153 1975.5653,646.11853 1975.1573,647.99753" id="path5452"/>
    <path d="M 1974.5063,647.42153 C 1974.3863,647.42153 1974.2663,647.42153 1974.1463,647.42153 C 1974.2273,647.70253 1974.3473,647.96753 1974.5063,648.21353 C 1974.5063,647.94953 1974.5063,647.68553 1974.5063,647.42153" id="path5454"/>
    <path d="M 1977.0963,679.60553 C 1976.7613,679.75853 1976.5953,680.02253 1976.5963,680.39753 C 1977.4143,680.62853 1977.5803,680.36453 1977.0963,679.60553" id="path5456"/>
    <path d="M 1976.8163,655.70153 C 1976.9813,655.95453 1977.2173,656.09753 1977.5263,656.13353 C 1977.2893,655.98953 1977.0523,655.84553 1976.8163,655.70153" id="path5458"/>
    <path d="M 1982.2863,690.26153 C 1980.2313,691.01553 1984.1613,691.81253 1982.2863,690.26153" id="path5462"/>
    <path d="M 1983.2863,688.53353 C 1982.2623,688.58953 1982.9413,689.73953 1983.3653,690.04553 C 1983.1443,689.55153 1983.1183,689.04853 1983.2863,688.53353" id="path5464"/>
    <path d="M 1984.8763,695.44553 C 1984.2753,695.67553 1984.1423,696.36453 1984.4453,696.88553 C 1985.0223,696.58353 1985.3033,696.02053 1984.8763,695.44553" id="path5468"/>
    <path d="M 1986.2463,697.38953 C 1983.5843,696.06453 1985.7363,700.20253 1986.2463,701.06153 C 1987.0533,699.27853 1986.2763,698.95153 1986.2463,697.38953" id="path5470"/>
    <path class="landxx in" d="M 1780.7563,553.89353 C 1780.6653,552.19553 1782.5303,552.16553 1783.7543,552.16553 C 1785.1983,552.16553 1784.4863,549.93753 1784.4263,549.14153 C 1785.3023,549.65103 1787.7702,549.3279 1788.5603,549.41353 C 1790.5295,549.62695 1791.8758,550.69919 1794.1749,550.27721 C 1794.3179,550.25097 1794.2214,549.13094 1794.863,549.11555 C 1795.6146,549.09753 1796.6664,547.85553 1797.7173,548.39153 C 1798.9838,550.4023 1800.0829,549.77003 1801.8563,548.34853 C 1799.4438,546.61911 1801.4049,546.36264 1801.2496,545.81572 C 1800.4299,542.92827 1798.4517,541.32503 1797.8063,539.75053 C 1797.3777,538.70484 1798.0378,536.91034 1796.1323,537.27653 C 1793.7126,537.74153 1792.8494,535.96479 1792.9273,534.41658 C 1792.9933,533.10524 1793.4903,530.83415 1792.4483,530.17958 C 1790.7643,529.12165 1788.0303,529.68538 1787.2907,527.13153 C 1786.7615,525.30399 1788.347,523.18275 1789.336,521.84675 C 1790.047,520.88775 1790.3015,520.06353 1790.8495,519.00953 C 1791.2345,518.26653 1792.3303,516.72299 1793.4243,516.99199 C 1794.8453,517.34199 1794.7183,519.57867 1796.2713,519.72067 C 1797.1563,519.80167 1797.8501,519.13867 1798.6151,518.79867 C 1799.5581,518.37967 1800.715,518.53764 1801.6993,518.32253 C 1803.8031,517.86274 1803.5573,516.1165 1804.8923,514.76553 C 1806.5044,513.13423 1805.5383,510.94753 1807.8773,509.61953 C 1810.0163,508.40553 1810.8913,506.89453 1811.5523,504.51153 C 1812.0363,502.76953 1811.5683,500.46153 1813.6803,499.74253 C 1814.4013,499.49753 1815.2783,499.33653 1815.6753,498.59653 C 1816.2333,497.55553 1814.8823,497.32153 1814.8853,496.79653 C 1814.8893,496.16453 1816.5523,493.93853 1816.9293,493.36053 C 1817.6353,492.28353 1819.3553,491.70253 1819.8553,490.53253 C 1818.3193,490.50253 1818.7073,489.05853 1818.6953,488.01153 C 1818.6803,486.74053 1817.5153,485.78453 1817.5473,484.69353 C 1817.6013,482.78453 1821.2813,482.42353 1822.3393,481.19253 C 1824.1533,479.07853 1819.1116,478.87353 1818.2106,478.54353 C 1817.2846,478.20453 1817.5253,476.83153 1816.5453,476.30053 C 1815.8763,475.93753 1815.0703,475.94353 1814.3763,475.62853 C 1814.6703,474.86353 1814.3453,474.42653 1813.6883,474.07653 C 1812.9883,473.70253 1810.9803,472.53453 1811.5203,471.62053 C 1812.0333,470.75453 1812.4013,469.83153 1811.4003,469.13453 C 1810.7863,468.70753 1809.7663,467.80153 1810.6503,467.04453 C 1811.0023,466.74353 1813.0693,465.35053 1811.3193,465.19453 C 1811.0333,465.16853 1808.3233,465.14653 1809.1043,464.16053 C 1810.0833,462.92453 1808.7823,463.01653 1808.0123,462.41753 C 1807.0523,461.67153 1808.6543,459.71753 1809.2813,459.33153 C 1810.8943,458.33853 1812.5783,459.09853 1814.2483,459.47053 C 1816.1403,459.89153 1818.0143,460.36453 1819.8993,460.80253 C 1821.5063,461.17553 1822.4603,460.17453 1823.8853,459.67753 C 1825.1283,459.24453 1826.4383,459.30053 1827.6683,458.90653 C 1828.0523,458.78353 1830.2443,456.55353 1830.0763,456.26453 C 1829.6203,455.48153 1831.9273,454.28453 1832.5473,453.78153 C 1833.0453,453.37853 1833.5853,452.93153 1834.2003,452.71653 C 1834.7163,452.53553 1834.7099,452.7688 1835.0737,453.03071 C 1836.5117,454.06605 1836.5429,454.49147 1837.0473,455.11653 C 1837.8842,456.15373 1838,458.35054 1839.3958,459.34953 C 1841.1404,460.59805 1844.1605,462.04167 1845.7305,463.16704 C 1846.2291,463.52445 1844.9672,464.07139 1844.6982,464.39908 C 1844.2229,464.97795 1844.4167,464.98692 1844.8943,466.80453 C 1845.295,468.32979 1846.031,469.63455 1847.1146,470.86653 C 1848.0908,471.97641 1849.9105,471.89762 1850.6567,472.65739 C 1851.0379,473.04558 1850.7462,473.69953 1850.9732,474.17253 C 1851.2292,474.70653 1851.4595,475.07756 1851.8043,475.54553 C 1853.0291,477.20784 1851.2964,477.8014 1850.575,478.41271 C 1849.6031,479.23623 1848.8638,480.49841 1847.6694,478.65912 C 1846.7164,477.19154 1845.9487,476.90253 1844.4733,478.00553 C 1844.9933,478.59753 1845.1863,479.27153 1845.3903,480.01053 C 1845.6193,480.83753 1846.385,481.29648 1846.9653,481.85153 C 1848.6161,483.43061 1847.7903,484.57553 1848.8403,485.68853 C 1849.4621,486.34768 1848.6466,489.16164 1850.1314,488.67807 C 1850.66,488.50594 1850.9168,488.01706 1851.4782,487.82439 C 1852.3588,487.5222 1852.7479,488.52339 1853.0849,489.06239 C 1853.5649,489.83239 1854.9533,491.44353 1855.8463,491.62153 C 1856.9133,491.83453 1857.8662,491.11567 1858.7612,492.13367 C 1859.3042,492.75167 1859.8465,492.70477 1860.5233,493.05353 C 1861.1772,493.39053 1861.203,494.00053 1861.0683,494.46153 C 1860.8451,495.22538 1862.1001,495.34335 1862.7813,495.52753 C 1864.9031,496.10123 1866.4192,496.65177 1867.7343,498.23753 C 1865.7393,498.35553 1865.5431,502.10059 1864.1671,503.06259 C 1864.0351,503.15459 1863.7384,504.22008 1863.5814,504.55308 C 1863.0184,505.74008 1864.1516,505.53782 1863.9639,506.66407 C 1863.7888,507.71493 1862.3012,509.04338 1863.1889,509.98107 C 1863.8789,510.70991 1866.5043,512.86953 1866.5043,511.41353 C 1867.7153,512.16673 1869.4601,512.83281 1870.5043,513.48362 C 1872.3068,514.60709 1872.7773,515.46607 1874.2253,516.33007 C 1874.6531,516.58536 1877.2451,518.34048 1877.8481,518.09948 C 1879.2601,517.53448 1880.3543,518.91453 1881.5553,519.53153 C 1882.0673,519.79453 1883.8113,519.82153 1884.0893,520.16253 C 1885.0573,521.34953 1885.1913,521.67153 1886.8833,522.01453 C 1887.7733,522.19553 1888.6455,522.83275 1889.5285,522.17875 C 1890.7065,521.30575 1891.6893,522.28253 1893.0043,522.86153 C 1892.1523,520.60053 1897.8093,522.89753 1898.1973,523.01553 C 1899.7553,523.49053 1899.2233,525.58339 1900.6573,525.91539 C 1901.8863,526.19939 1903.1468,526.7228 1903.9048,527.7948 C 1904.5118,528.6528 1905.8933,527.48453 1906.6183,527.42353 C 1908.3933,527.27453 1907.5223,529.94953 1909.8653,528.99053 C 1910.5553,528.70753 1913.5793,529.75153 1914.2433,530.15053 C 1915.9533,531.17553 1916.3423,530.15553 1917.9143,529.62953 C 1918.0471,530.75182 1919.3818,531.59224 1920.7473,531.04053 C 1921.1689,530.87021 1923.4551,531.04553 1924.0161,531.05753 C 1925.2171,531.08453 1925.49,530.77748 1925.667,529.48648 C 1925.97,527.28648 1924.8384,526.81285 1923.9964,524.92985 C 1923.5344,523.89685 1923.9233,522.44153 1924.0203,521.34253 C 1924.1363,520.00953 1923.8183,518.97753 1923.7453,517.67753 C 1925.0803,517.58453 1926.6053,516.45453 1927.9393,516.84253 C 1930.5583,517.60353 1928.2663,520.84853 1929.3853,522.21253 C 1929.5863,522.45753 1930.5713,523.03553 1930.2653,523.46253 C 1929.7933,524.12053 1929.3513,524.03753 1929.8753,524.97953 C 1931.0293,527.05153 1932.5703,526.98853 1934.5803,527.25053 C 1935.9323,527.42653 1936.7703,528.62353 1938.1113,528.48153 C 1938.6733,528.42253 1939.4478,528.3408 1939.9358,528.0098 C 1940.4428,527.6658 1941.5623,526.44453 1942.2903,526.95353 C 1944.1143,528.22753 1946.3873,527.80953 1948.4773,527.72953 C 1949.1153,527.70553 1950.0807,526.80194 1950.4387,527.22994 C 1950.9887,527.88894 1952.9283,527.30726 1953.7313,527.23053 C 1955.6891,527.04346 1953.4062,525.88398 1953.6813,524.33253 C 1954.2767,520.97535 1950.7049,522.95655 1950.3053,521.31053 C 1950.0713,520.34681 1949.3633,519.07453 1950.8063,519.42453 C 1951.9623,519.70553 1953.1169,519.28256 1954.1309,519.03099 C 1955.5819,518.67099 1956.6914,519.21007 1957.0403,516.57107 C 1957.19,515.43876 1959.9773,514.34407 1960.1073,512.82707 C 1960.3153,510.43307 1963.248,511.73953 1964.637,510.37753 C 1965.751,509.28653 1966.399,508.02339 1967.645,507.03139 C 1968.104,506.66539 1968.7303,506.23153 1969.2413,505.93953 C 1969.6813,505.68753 1971.8223,507.26253 1972.2733,507.52753 C 1971.7423,506.99553 1972,506.9158 1972.92,507.4188 C 1973.635,507.7198 1974.4043,508.28653 1975.2123,508.25353 C 1976.0383,508.21953 1975.9083,507.23253 1976.2073,506.70653 C 1976.7403,505.76653 1978.5294,505.04707 1979.4744,504.66807 C 1979.6834,504.58507 1981.9142,506.58153 1982.1652,506.80853 C 1980.4699,508.26187 1980.6832,508.15002 1981.2002,508.71553 C 1981.7466,509.31328 1983.0643,508.13653 1983.4333,508.03153 C 1983.1263,509.08753 1985.06,509.82026 1984.723,510.65826 C 1984.207,511.93226 1983.6233,512.30153 1983.3593,513.57653 C 1983.1703,514.48453 1984.8003,513.40775 1985.1163,513.31975 C 1986.4623,512.94675 1987.8386,514.70012 1989.2076,514.28412 C 1990.3816,513.92712 1991.2133,515.41053 1991.2133,516.40853 C 1991.2133,516.78153 1991.7633,517.08753 1991.7023,517.36553 C 1991.5563,518.03453 1990.4601,518.82128 1990.0293,519.29353 C 1987.5913,521.9661 1991.2183,522.99013 1991.6423,525.02353 C 1989.9202,525.75421 1989.2452,522.92262 1988.2283,523.13053 C 1984.6931,523.85331 1983.8448,523.68762 1983.1313,524.89453 C 1982.5313,525.90939 1982.12,526.7003 1981.2503,527.02053 C 1980.5543,527.22927 1980.8007,527.55584 1980.4247,527.54353 C 1979.9707,527.91153 1980.4349,528.21075 1979.6957,528.59953 C 1979.3349,528.78924 1978.1201,528.36779 1977.5633,529.65753 C 1977.2235,530.44475 1977.4673,531.42953 1977.6943,532.24253 C 1977.8673,532.86253 1978.3441,533.28891 1978.5303,533.86853 C 1978.7794,534.64406 1978.2592,535.38904 1978.1983,536.14858 C 1978.0281,538.27404 1975.1525,539.45385 1975.9415,541.15985 C 1976.2245,541.77185 1977.0023,542.52955 1976.9523,543.2534 C 1976.7053,546.82541 1974.7933,549.55477 1974.0723,553.24853 C 1973.3483,552.67353 1971.4833,552.11853 1970.5773,551.88453 C 1970.1223,551.91653 1969.6723,552.07612 1969.2263,552.17212 C 1968.6493,552.05512 1968.5929,551.54812 1967.9069,551.68812 C 1968.6509,553.72412 1968.9703,555.44153 1969.0873,557.60053 C 1969.1363,558.51053 1969.2423,561.38453 1967.5913,560.16053 C 1967.8463,562.12953 1967.6893,563.57353 1968.4683,565.43353 C 1968.7453,566.09453 1969.3913,568.31353 1967.8813,567.43253 C 1968.0268,571.06315 1966.6861,569.39685 1965.3603,567.86553 C 1965.4843,568.44753 1965.1487,569.14499 1964.6857,569.50499 C 1963.4879,564.56404 1961.9949,560.07563 1960.4162,554.77599 C 1960.1742,553.47799 1958.6434,555.45399 1957.8394,554.74499 C 1958.2897,555.3203 1958.0219,556.16794 1957.6336,556.88826 C 1957.1439,557.79656 1957.6383,558.68628 1957.6748,559.58058 C 1957.7117,560.48587 1957.1544,561.43706 1956.0485,560.62853 C 1955.2214,560.02381 1954.935,558.89179 1954.2713,558.36053 C 1954.3633,558.84553 1954.3393,559.32653 1954.2023,559.80053 C 1953.5545,558.32553 1952.5136,556.88639 1952.1998,555.39739 C 1951.8661,553.81438 1953.6153,552.36953 1953.1913,551.01553 C 1953.9363,551.11453 1955.2693,551.34953 1955.1313,550.15153 C 1955.6163,550.90953 1956.3113,549.75653 1957.0013,550.36853 C 1957.2243,548.47853 1959.9163,546.57253 1958.7313,544.31953 C 1959.1823,544.36153 1960.7755,544.43985 1959.8905,543.53585 C 1958.8545,542.47685 1957.229,541.7621 1955.7813,541.72853 C 1954.5561,541.70012 1953.6175,541.56343 1952.009,541.62021 C 1950.3066,541.6803 1948.8025,541.90928 1947.1173,541.78453 C 1944.6792,541.60405 1939.6548,542.09191 1939.9653,538.89253 C 1940.0971,537.53451 1939.4393,532.97653 1937.2813,533.15953 C 1937.6273,533.77353 1937.4533,534.35353 1937.2113,534.96053 C 1936.2083,534.40553 1934.715,534.14294 1933.999,533.15994 C 1933.4,532.33694 1933.2213,530.82653 1931.8813,531.07253 C 1932.3453,531.49153 1932.6823,531.99553 1932.8913,532.58453 C 1932.2574,532.38187 1931.5181,532.44087 1930.5733,531.91453 C 1930.1232,531.6638 1930.0398,530.97153 1929.6098,530.62753 C 1929.0318,530.16553 1928.2303,529.84853 1927.6313,529.41653 L 1927.4386,530.37231 C 1927.7774,530.3839 1928.7159,530.48867 1929.0013,531.35953 C 1926.7313,532.22053 1925.2663,535.44953 1928.0223,536.54153 C 1929.1223,536.97753 1929.5193,538.16953 1930.4643,538.79353 C 1931.1573,539.25153 1932.4523,538.44953 1932.5613,539.42453 C 1932.7183,540.84553 1934.6783,540.30753 1933.7513,541.80053 C 1933.1253,541.47253 1932.1143,541.57553 1931.4423,541.63953 C 1930.5323,541.72653 1930.3203,541.78631 1930.2313,542.62553 C 1930.1363,543.52155 1929.2663,545.39138 1928.4913,543.67253 C 1928.3951,545.15056 1926.7014,545.92275 1928.8069,547.52867 C 1929.9345,548.38873 1931.5944,548.3167 1932.5533,549.12953 C 1934.3705,550.66992 1933.4967,551.29616 1933.7403,552.06623 C 1933.9902,552.85643 1931.8092,553.62539 1933.1092,554.8726 C 1934.2576,555.97432 1934.2183,555.78098 1934.5464,556.17253 C 1934.8624,556.54958 1934.3937,557.42753 1934.6077,558.16453 C 1934.6888,558.4438 1935.652,558.42407 1935.789,558.46507 C 1936.556,558.69407 1936.1305,559.91953 1936.1855,560.37653 C 1936.2945,561.30153 1936.5709,561.89776 1936.7043,562.82853 C 1936.885,564.08953 1937.2671,564.81939 1937.7148,565.95339 C 1938.2021,567.18767 1938.6853,568.67553 1938.0723,569.88053 C 1937.7923,569.70153 1937.5053,569.53353 1937.2123,569.37653 C 1937.4333,569.85853 1939.1243,571.75453 1938.5403,572.23953 C 1937.9193,572.75753 1937.0783,571.83253 1937.0613,571.24853 C 1936.9283,572.03053 1936.3373,572.71553 1935.8413,571.68053 C 1935.3193,570.59053 1935.9463,569.21753 1935.9113,568.08053 C 1935.2313,568.41653 1935.9703,569.10353 1935.1903,569.73553 C 1935.3723,568.97353 1934.8843,568.24553 1935.2603,567.50453 C 1933.9003,568.54153 1935.1363,571.89753 1934.9013,573.33653 C 1934.5903,572.38453 1934.9603,570.84953 1934.0413,570.16853 C 1934.0523,570.27253 1934.2793,572.84253 1933.8503,572.55853 C 1933.7143,572.37653 1933.5923,572.18653 1933.4853,571.98753 C 1932.6613,570.82453 1933.3863,573.71453 1932.6723,571.53653 C 1932.9973,572.05153 1933.0213,572.57953 1932.7423,573.12053 C 1931.7553,572.22753 1931.7793,570.30353 1931.6623,569.08853 C 1931.4793,567.21853 1930.0053,567.57753 1929.4323,566.06453 C 1929.6723,567.58053 1932.1393,567.28753 1930.9793,569.22853 C 1929.6703,571.41553 1928.4493,572.30653 1925.9723,573.04953 C 1924.6593,573.44453 1922.8613,574.13253 1922.5693,575.66653 C 1922.3703,576.70953 1923.3523,580.12053 1924.1733,580.89653 C 1923.6673,581.28753 1922.8813,581.68853 1922.6383,582.32153 C 1922.2683,583.28653 1923.2093,583.36153 1922.0563,584.27853 C 1920.6813,585.37153 1920.1613,587.47853 1918.4843,585.28953 C 1918.7223,586.11253 1919.6843,586.50853 1920.3633,586.87353 C 1918.2623,587.92453 1916.0123,588.98953 1913.6643,589.32053 C 1914.4043,588.91653 1914.6113,587.12053 1913.4373,587.52553 C 1912.2343,587.94053 1911.1663,589.67653 1911.5743,590.90453 C 1911.7243,590.44053 1912.8293,588.90153 1913.2343,589.68053 C 1912.5003,589.89053 1911.3733,591.80753 1910.6713,592.49253 C 1909.0943,594.02953 1908.2943,595.90453 1907.1973,597.79453 C 1906.0883,599.70653 1904.7843,601.60353 1902.7083,602.54753 C 1900.9273,603.35753 1900.7253,605.72953 1898.9733,606.52853 C 1900.1353,607.22753 1894.4973,609.85653 1893.8703,610.42153 C 1893.2583,610.97353 1892.6073,611.77753 1892.2863,612.53753 C 1891.7683,613.76453 1893.4063,614.07653 1892.8583,615.47753 C 1891.7743,618.25153 1888.4883,618.01153 1886.0863,618.13053 C 1885.1083,618.17953 1884.9973,618.88353 1884.7773,619.71553 C 1884.5373,620.62353 1883.5743,622.23853 1883.6423,623.08853 C 1882.8683,622.70453 1882.8513,621.92253 1882.5623,621.21653 C 1882.8803,622.02953 1882.3793,622.78253 1882.1323,623.52053 C 1882.6663,621.08453 1879.1403,622.46553 1878.5133,623.64653 C 1877.5933,625.37753 1876.4063,628.02553 1877.0893,629.95653 C 1877.6513,631.54453 1878.3013,633.26853 1878.3243,634.89453 C 1878.3643,637.68053 1878.9093,639.89153 1879.6813,642.60153 C 1879.4463,642.13353 1878.3343,639.89253 1878.4523,641.73753 C 1877.8073,640.41553 1877.8293,642.12953 1878.3183,642.42953 C 1878.9463,642.81353 1879.5553,642.85953 1879.9843,643.54453 C 1880.7713,644.80053 1880.0933,648.06853 1879.8923,649.42753 C 1879.5353,651.85553 1878.1023,653.70153 1877.3833,655.98653 C 1876.8223,657.76553 1877.6613,660.34253 1876.3613,661.82553 C 1876.5833,661.60953 1876.8233,661.41853 1877.0823,661.25053 C 1877.8903,662.19553 1877.8983,663.54653 1877.8883,664.71853 C 1877.8703,666.64353 1878.3073,668.54553 1878.2423,670.46653 C 1877.7343,670.42753 1877.2303,670.35653 1876.7323,670.25053 C 1878.6793,670.87953 1876.6893,670.56853 1876.3423,670.47553 C 1875.6183,670.28053 1874.5633,669.90753 1874.1173,670.79653 C 1873.8863,671.25853 1873.5233,671.81853 1873.7983,672.34353 C 1874.2083,673.12853 1873.0553,673.89453 1872.6913,674.49753 C 1871.9833,675.66753 1870.6003,678.23153 1872.6133,679.08553 C 1873.1353,679.30753 1873.7913,679.24253 1874.3053,679.03753 C 1874.6663,678.89253 1875.4213,680.19853 1875.6523,680.40253 C 1873.6723,678.35553 1870.3743,679.63453 1868.2323,680.69053 C 1865.4453,682.06353 1866.7193,684.67453 1865.3783,686.78853 C 1864.6993,687.85553 1863.3673,688.58953 1862.2463,689.08953 C 1860.4803,689.87753 1859.4753,689.08353 1858.1523,687.89153 C 1857.5423,687.34053 1852.8283,682.37053 1854.3323,681.98753 C 1853.0063,682.46253 1852.4933,679.80953 1852.0233,678.69353 C 1851.3563,677.10753 1850.6423,675.23253 1851.0913,673.49153 C 1851.4793,674.11653 1851.6363,674.85353 1851.6713,675.57953 C 1851.8403,673.84453 1851.5723,677.25053 1852.6103,677.38053 C 1852.2533,676.43353 1852.5263,675.35453 1852.1003,674.42853 C 1851.7363,673.63653 1850.9443,673.20853 1850.8803,672.26853 C 1850.9143,674.10953 1850.3013,672.32553 1850.3413,671.92953 C 1850.4343,670.99753 1849.8973,670.70753 1849.4663,669.70253 C 1848.6983,667.91353 1848.0843,666.04653 1847.3993,664.22453 C 1847.1413,663.53853 1842.9343,656.69353 1843.1703,656.35653 C 1840.5203,653.09453 1838.6873,649.17053 1837.7743,645.08253 C 1837.4503,643.62853 1837.2653,642.30353 1836.7433,640.93753 C 1836.3963,640.03153 1836.0213,639.18853 1835.6523,638.29853 C 1835.3743,637.62753 1834.3373,635.91253 1835.5403,636.34053 C 1834.2093,635.89553 1834.5163,634.04753 1833.8103,633.10053 C 1833.3623,632.49853 1831.1373,631.29953 1833.0203,630.94053 C 1831.9073,631.44853 1831.3473,630.12053 1831.0273,629.30653 C 1830.7763,628.66753 1830.5713,628.00353 1830.2533,627.39353 C 1829.9273,626.76853 1828.9283,626.47953 1830.2113,626.33153 C 1829.5793,626.17153 1829.0433,625.27053 1829.0633,624.64953 C 1829.0843,624.00553 1828.3753,623.80753 1827.8793,623.10953 C 1826.6813,621.42353 1825.8933,619.41353 1825.2623,617.45653 C 1824.8233,616.09553 1824.4513,614.66753 1824.4513,613.22753 C 1824.4513,611.72753 1823.6393,610.32653 1823.4013,608.85253 C 1823.1783,607.47153 1822.3993,606.28853 1821.9803,604.95953 C 1821.7063,604.09053 1820.1333,601.21753 1821.5713,601.77853 C 1821.3093,601.44653 1819.3203,598.87653 1821.0713,599.83553 C 1820.7863,598.78153 1819.5143,598.10753 1819.9913,596.88353 C 1820.2343,597.07153 1820.4743,597.26353 1820.7113,597.45953 C 1820.7533,596.84453 1820.5363,596.34053 1820.0613,595.94653 C 1821.1033,595.53653 1820.5503,594.53653 1820.3513,593.78753 C 1820.2903,594.39353 1820.0753,594.94153 1819.6313,595.37153 C 1818.8523,594.11653 1819.0903,591.29153 1820.8513,593.28353 C 1820.5623,592.57753 1817.9473,592.13253 1818.7703,591.13953 C 1819.0573,590.79353 1817.8513,589.36853 1817.7173,588.92753 C 1817.3593,587.74653 1817.3183,586.58053 1817.5623,585.37053 C 1817.9453,583.47853 1819.0753,581.49953 1818.3183,579.56253 C 1818.2203,579.31053 1817.7133,575.99853 1816.7523,577.22553 C 1817.0823,576.35753 1816.8893,576.33253 1816.1723,577.15353 C 1817.0623,576.09553 1815.5263,575.77453 1815.7863,574.99453 C 1815.9543,574.48853 1816.0923,573.94653 1816.1683,573.41953 C 1816.2803,572.65153 1818.8653,572.16753 1819.3423,571.25053 C 1817.4533,572.66553 1813.1043,571.56653 1816.0323,569.44953 C 1815.0373,569.89253 1813.8703,569.19453 1814.5923,568.08253 C 1815.2103,567.13053 1816.1933,568.09853 1816.8223,567.14553 C 1815.8373,567.39853 1813.7743,567.03853 1813.2213,566.06553 C 1813.1583,566.55653 1812.1933,566.74653 1812.2113,566.64153 C 1812.1243,567.11353 1812.8313,567.18953 1812.8493,567.91553 C 1812.8743,568.89353 1812.1793,569.18853 1811.3513,569.44853 C 1811.5373,569.69453 1811.7303,569.93453 1811.9313,570.16953 C 1811.3533,570.01553 1810.9333,570.34953 1810.7713,570.88953 C 1812.8033,571.51853 1813.6803,572.84753 1812.3433,574.73553 C 1811.8683,575.40453 1812.1283,576.33953 1811.2063,576.68653 C 1810.0873,577.10753 1809.0793,577.85253 1808.0603,578.46553 C 1806.2583,579.54853 1804.2073,580.88253 1802.0283,580.17053 C 1799.8313,579.45353 1797.2233,577.39253 1795.9203,575.49853 C 1793.9803,572.68153 1791.0893,570.91153 1788.8213,568.43853 C 1788.2243,567.78753 1786.5293,565.57253 1788.4873,565.29953 C 1789.0213,565.22553 1789.0473,566.24853 1789.4493,566.46653 C 1790.2173,566.88153 1791.6323,566.55253 1791.7713,565.56253 C 1792.5263,566.62753 1796.0243,564.62453 1796.7323,563.90653 C 1797.3463,563.28253 1797.2303,562.31553 1797.6713,561.60253 C 1798.1363,560.85353 1799.3393,560.35053 1798.3213,559.44253 C 1797.8083,559.95053 1798.3493,561.15353 1797.3543,561.09853 C 1796.2803,561.03853 1795.4053,561.24653 1794.3663,561.59853 C 1793.4823,561.89853 1793.1353,562.72253 1792.1283,562.67453 C 1790.9013,562.61553 1789.6813,562.02653 1788.6113,561.47053 C 1787.5443,560.91553 1786.3353,560.43253 1785.3663,559.72553 C 1784.5933,559.16153 1784.5383,557.75953 1783.6183,557.42953 C 1780.5073,556.30553 1784.1493,553.89653 1785.0713,553.17953 C 1783.4123,552.99153 1782.3253,556.16453 1780.6803,555.41053 C 1780.9983,554.91453 1781.0223,554.41053 1780.7563,553.89353" id="path5490"/>
  </g>
  <g id="cn">
    <g class="landxx coastxx cn cnx" id="cnx">
      <path d="M 2134.1363,419.10953 C 2134.2693,418.52153 2133.9503,418.32053 2133.4073,418.46253 C 2133.6343,418.69553 2133.8773,418.91053 2134.1363,419.10953" id="path2726"/>
      <path d="M 2142.9163,420.04553 C 2142.7363,419.81853 2142.4963,419.72353 2142.1953,419.75753 C 2142.4363,419.85353 2142.6763,419.94953 2142.9163,420.04553" id="path2730"/>
      <path d="M 2157.7463,484.48553 C 2159.1383,485.10353 2162.1893,488.55353 2163.7963,486.86253 C 2162.8283,486.05753 2159.0013,483.25353 2157.7463,484.48553" id="path2740"/>
      <path d="M 2166.8963,516.45353 C 2166.9433,516.57353 2166.9903,516.69253 2167.0373,516.81253 C 2168.2853,516.46053 2167.1713,514.90353 2166.8963,516.45353" id="path2744"/>
      <path d="M 2168.7663,499.96553 C 2169.5493,499.85953 2170.3043,500.35353 2171.0663,499.96553 C 2170.5923,499.18553 2169.1053,498.48453 2168.3273,499.24553 C 2168.4733,499.48653 2168.6193,499.72553 2168.7663,499.96553" id="path2748"/>
      <path d="M 2168.5463,497.37353 C 2168.7923,497.72653 2169.1293,497.86953 2169.5563,497.80553 C 2169.4863,497.15453 2169.0663,496.97353 2168.5463,497.37353" id="path2750"/>
      <path d="M 2170.4163,502.70153 C 2170.5553,502.12053 2170.3143,502.04853 2169.6953,502.48453 C 2169.9363,502.55753 2170.1763,502.62953 2170.4163,502.70153" id="path2752"/>
      <path d="M 2147.2373,554.54153 C 2147.4303,554.70953 2147.6233,554.87753 2147.8163,555.04553 C 2148.7723,554.03653 2147.2973,553.60553 2147.2373,554.54153" id="path2734"/>
      <path d="M 2161.5663,539.06153 C 2162.2943,540.37453 2162.5403,538.55353 2162.3563,537.98153 C 2162.0803,538.00553 2161.8163,538.07853 2161.5663,538.19853 C 2161.8303,538.41453 2162.0923,538.63053 2162.3563,538.84553 C 2162.0923,538.91753 2161.8293,538.98853 2161.5663,539.06153" id="path2742"/>
      <path d="M 2157.0263,533.94953 C 2157.5123,534.20553 2158.0163,534.42153 2158.5373,534.59653 C 2157.9043,534.94453 2157.2983,534.52553 2157.0263,533.94953" id="path2766"/>
      <path d="M 2151.6263,547.26953 C 2151.4163,547.85953 2151.6203,548.21753 2152.2753,548.13353 C 2152.4043,547.64153 2152.1853,547.19953 2151.6263,547.26953" id="path2736"/>
      <path d="M 2088.4873,592.41353 C 2088.1233,593.00053 2088.8483,595.35853 2088.9693,596.01553 C 2089.1683,597.09353 2089.1133,598.69553 2089.9263,599.54153 C 2090.9193,600.57653 2092.5963,600.37953 2093.7703,601.08553 C 2094.8853,601.75653 2096.6453,601.87953 2097.9163,601.84553 C 2097.4243,600.43153 2099.2843,599.59553 2099.8423,600.54653 C 2099.4593,599.89453 2101.8683,597.93553 2102.2583,597.56753 C 2102.8153,597.04453 2103.5243,594.12653 2102.6663,593.78153 C 2103.5973,593.28853 2104.5323,590.82053 2104.3263,589.82053 C 2106.1953,591.21153 2105.0913,587.60553 2104.8033,586.90553 C 2104.3233,585.73953 2100.9693,584.50453 2102.3763,587.15553 C 2101.9413,586.61453 2099.7433,585.72953 2100.8653,587.22753 C 2100.2183,585.33053 2098.2173,586.51553 2097.6263,587.37253 C 2097.0823,586.14753 2094.9713,586.30653 2094.0963,587.01253 C 2094.2603,587.28053 2094.4303,587.54353 2094.6063,587.80453 C 2093.7503,588.09653 2093.2123,586.99953 2092.4463,587.22853 C 2091.4513,587.52553 2091.9363,588.85153 2092.7373,589.02853 C 2090.9523,589.52253 2089.7963,591.23453 2088.4873,592.41353" id="path2710"/>
      <path d="M 2098.0563,578.15653 C 2098.8603,577.99653 2099.6663,577.66353 2100.2163,578.51653 C 2100.7543,577.01753 2098.5303,577.00553 2098.0563,578.15653" id="path2712"/>
      <path d="M 2100.0063,576.14153 C 2099.4243,576.49553 2099.5443,576.73553 2100.3653,576.86253 C 2100.2453,576.62153 2100.1263,576.38153 2100.0063,576.14153" id="path2714"/>
      <path d="M 2109.1463,573.04553 C 2109.5383,572.85753 2109.7543,572.54553 2109.7963,572.10853 C 2109.1993,572.19453 2109.2013,572.54353 2109.1463,573.04553" id="path2716"/>
      <path d="M 2115.1263,571.74953 C 2114.6993,571.88753 2114.4333,572.17553 2114.3263,572.61353 C 2114.9093,572.64653 2115.2113,572.33153 2115.1263,571.74953" id="path2718"/>
      <path d="M 2116.4163,572.90153 C 2116.4833,572.34053 2116.3403,571.83653 2115.9863,571.38953 C 2116.0483,571.91653 2116.1913,572.42053 2116.4163,572.90153" id="path2720"/>
      <path d="M 2119.2263,561.38153 C 2119.3133,561.61353 2120.3753,563.02853 2120.8163,562.60553 C 2120.4003,562.08453 2119.9553,561.46153 2119.2263,561.38153" id="path2722"/>
      <path d="M 2120.3063,562.82153 C 2120.3773,563.25753 2120.3803,563.41453 2120.8863,563.54253 C 2120.6933,563.30153 2120.4993,563.06153 2120.3063,562.82153" id="path2724"/>
      <path class="landxx cn cnx" d="M 1796.5263,418.31753 C 1797.5413,418.32253 1798.2553,417.81153 1798.0373,416.73353 C 1797.7353,415.25353 1797.0183,415.53053 1797.4723,413.96853 C 1797.8403,412.70453 1798.9023,413.15053 1799.8373,412.77353 C 1800.9263,412.33453 1801.3603,410.79153 1802.6473,410.75753 C 1802.4393,410.29353 1802.3893,409.81353 1802.4983,409.31753 C 1803.2393,409.47053 1804.7403,410.19853 1805.4253,409.58253 C 1805.9393,409.12053 1806.5103,408.35153 1807.2433,408.24953 C 1808.0083,408.14353 1808.3713,410.13053 1808.8903,410.58553 C 1809.7723,411.35853 1811.4843,410.30953 1812.6063,410.40753 C 1814.6693,410.58653 1814.8213,407.65853 1815.0553,406.22353 C 1815.2473,405.04053 1815.7893,404.59653 1817.0763,404.78753 C 1818.7093,405.02953 1820.2233,405.06053 1821.7983,404.71053 C 1822.6203,404.52753 1823.7613,404.83353 1824.4353,404.19153 C 1825.1793,403.48253 1825.0533,402.08253 1825.7583,401.25453 C 1827.5293,399.17153 1830.9983,398.96453 1833.0633,397.27853 C 1833.8933,396.60053 1834.3173,396.23453 1835.3723,396.20853 C 1836.7273,396.17453 1836.3583,395.68353 1835.9453,394.69953 C 1835.4193,393.44653 1834.9613,392.34853 1834.3443,391.14453 C 1833.5353,389.56853 1835.2513,389.21753 1836.3383,388.94153 C 1835.9153,388.54153 1835.4353,388.22953 1834.8983,388.00553 C 1835.3413,387.34853 1836.0903,386.36053 1836.9893,386.99753 C 1837.5703,385.68053 1835.9733,385.55053 1835.9073,384.45553 C 1835.8193,383.02153 1834.4003,382.01753 1833.6083,380.94953 C 1832.5223,379.48353 1831.3733,378.00953 1830.7503,376.27253 C 1830.3403,375.12753 1831.0203,373.26453 1829.9353,372.76753 C 1829.0193,372.34853 1825.2883,372.78453 1826.3333,371.22853 C 1826.9263,370.34753 1829.1403,370.09853 1830.1243,369.79053 C 1831.9163,369.22953 1833.6893,369.08553 1835.4933,368.43653 C 1837.2893,367.79053 1837.5643,368.69653 1839.0453,369.28853 C 1840.2343,369.76353 1841.2383,368.54253 1842.4223,369.37153 C 1843.7853,370.32653 1843.8103,368.74753 1843.5263,367.81653 C 1843.1013,366.42153 1841.3643,367.23053 1840.6603,366.18953 C 1840.0583,365.29953 1840.6923,363.59753 1840.7693,362.62653 C 1840.8853,361.16253 1840.5313,359.62453 1840.5363,358.14753 C 1840.5403,356.75053 1841.1363,355.21553 1840.8553,353.85653 C 1840.5173,352.21753 1841.3653,351.75553 1842.8173,352.45253 C 1844.0443,353.04053 1845.1073,353.59953 1846.4513,353.91553 C 1848.2033,354.32653 1849.8773,353.66953 1851.6323,353.84453 C 1852.9773,353.97853 1853.4453,355.74853 1854.6853,354.70353 C 1855.6403,353.89953 1856.9503,353.79153 1857.8533,352.98553 C 1859.4993,351.51653 1855.3883,347.47053 1854.9203,345.88553 C 1854.6663,345.02553 1854.4793,342.67653 1855.3403,342.07353 C 1856.1663,341.49353 1857.3763,342.08753 1858.2913,341.67753 C 1860.6083,340.64053 1860.4333,339.56353 1859.6993,337.41253 C 1859.2203,336.00753 1860.8259,336.02059 1861.7483,335.99953 C 1862.7627,335.97637 1864.2748,336.69285 1866.1762,335.78093 C 1866.7641,336.32669 1867.5575,337.27362 1867.0949,337.39972 C 1865.8401,337.74176 1868.0979,338.73703 1868.4323,338.88014 C 1870.0855,339.58761 1868.5693,340.18474 1869.5823,340.66274 C 1870.9913,341.32574 1872.9043,341.35553 1873.9723,342.65653 C 1875.1703,344.11653 1878.4455,345.82517 1880.2839,345.30253 C 1882.1796,344.76361 1884.046,347.76072 1884.9139,346.29233 C 1885.6746,346.52953 1887.7186,347.7078 1888.5761,348.53233 C 1889.7877,349.69741 1890.3901,351.23865 1891.6505,352.33613 C 1892.7471,353.291 1894.1845,354.16462 1895.3145,355.57949 C 1896.9345,357.60787 1896.7736,356.70396 1896.9029,357.87065 C 1897.2933,361.39265 1898.8772,361.61194 1897.9759,362.85554 C 1896.9626,364.25377 1897.0578,365.68977 1898.3063,366.87853 C 1899.165,367.69612 1900.2779,368.84515 1901.5813,369.17553 C 1903.1877,369.58271 1904.7756,369.80346 1906.4889,370.02354 C 1908.0926,370.22953 1909.8693,370.54737 1911.4943,370.66653 C 1913.0474,370.78042 1914.5414,370.53805 1916.1033,370.60453 C 1919.0245,370.72887 1921.6649,372.3335 1923.9341,373.38834 C 1924.7229,373.75502 1925.4978,374.06705 1925.8087,374.25754 C 1927.7884,375.47065 1928.6641,375.85893 1930.2101,376.44593 C 1931.333,376.87228 1932.2257,376.69441 1933.3971,376.85214 C 1934.9122,377.05614 1934.2652,378.26953 1935.0233,378.90553 C 1936.328,380.00006 1937.3193,380.09453 1938.3233,381.62253 C 1939.1153,382.82853 1940.1949,383.98593 1941.1449,385.07293 C 1942.2529,386.34193 1943.9185,387.03513 1945.1965,388.10013 C 1946.0905,388.84513 1946.059,389.90168 1947.2423,390.06674 C 1948.6698,390.26585 1950.3167,389.68577 1951.9593,389.58274 C 1954.424,389.42813 1959.0478,390.88207 1962.7419,391.04574 C 1964.5583,391.12621 1966.3413,391.52853 1968.1593,391.48253 C 1969.9013,391.43853 1971.1137,390.36955 1972.8857,390.49355 C 1974.7797,390.62555 1976.8645,391.17337 1978.7435,391.42374 C 1980.361,391.63926 1981.8491,391.0003 1983.8823,392.05453 C 1985.2707,392.77441 1986.2059,394.07408 1987.6413,394.61253 C 1989.6443,395.36391 1992.0394,395.68475 1994.2065,396.19152 C 1997.4252,396.94419 2000.1713,398.23889 2004.1983,397.90553 C 2005.6231,397.78758 2005.7312,398.25794 2006.8831,398.99552 C 2008.0499,399.74265 2011.2717,399.52913 2010.4753,399.65853 C 2009.1772,399.86944 2017.135,395.18445 2018.8593,394.70453 C 2020.0797,394.36485 2022.1489,393.47885 2023.9843,393.36453 C 2025.9591,393.24153 2027.7661,392.68672 2030.1985,392.70492 C 2033.3315,392.72836 2037.1129,393.34214 2039.2823,391.92153 C 2040.5063,391.12003 2042.5195,390.69474 2043.4545,389.58774 C 2044.3505,388.52774 2044.5218,386.91216 2045.2287,385.73974 C 2045.8487,384.71146 2046.8235,384.95467 2047.8713,383.71393 C 2048.3115,383.19276 2048.8346,382.90314 2049.3575,382.33113 C 2049.6308,382.03226 2050.0791,382.19514 2050.1591,381.85514 C 2050.6381,379.79214 2045.4481,377.58907 2044.1979,376.24314 C 2043.3044,375.28117 2044.4013,374.74052 2043.8726,373.54457 C 2043.1911,372.00309 2044.1405,370.43214 2045.5627,369.96835 C 2047.6045,369.3025 2049.0151,371.39525 2055.0033,372.66753 C 2055.8033,372.8375 2057.8711,373.25511 2058.0023,372.64754 C 2058.292,371.30639 2060.0113,371.25441 2060.6023,370.28553 C 2061.1433,369.39865 2061.1007,368.32832 2061.7657,367.50932 C 2062.4027,366.72532 2064.1335,367.45952 2064.9735,367.54072 C 2067.2087,367.75677 2069.6743,366.35853 2070.9741,364.41034 C 2071.5437,363.55667 2069.8485,362.77813 2070.6009,361.17691 C 2071.2119,359.87676 2070.7915,358.74421 2073.9579,359.05634 C 2074.9768,359.15678 2075.6261,359.40275 2075.7597,357.91394 C 2075.8709,356.67421 2077.6065,357.62912 2078.1667,357.59255 C 2078.5704,357.56619 2080.1204,356.44733 2080.9257,356.26734 C 2081.5768,356.1218 2082.1283,356.20294 2082.7693,356.27894 C 2084.0523,356.83095 2084.7595,355.73077 2084.6039,355.99892 C 2085.0979,355.99592 2086.1959,356.60392 2086.7809,356.71692 C 2087.8419,356.91992 2088.8373,357.1248 2089.9123,357.01653 C 2093.4651,356.6587 2090.0734,353.67478 2088.7433,352.75653 C 2084.7755,350.01736 2080.8649,347.85505 2076.9253,345.95053 C 2075.0861,345.06143 2072.2466,344.40728 2071.1243,346.57853 C 2069.6023,349.52301 2067.3725,346.14665 2065.0753,346.57353 C 2064.244,346.72802 2062.7028,346.55495 2061.9189,346.70213 C 2060.9048,346.89251 2060.7663,348.27153 2059.4083,347.50753 C 2058.5453,347.02153 2055.4193,345.43753 2056.2163,344.09053 C 2056.6643,343.33153 2057.2663,343.35753 2056.6713,342.41153 C 2056.0803,341.47253 2056.0673,340.68753 2056.0893,339.62553 C 2056.1523,336.42853 2055.8253,333.23253 2055.5793,330.04953 C 2057.9523,331.18653 2060.5633,331.15653 2062.8523,332.17353 C 2065.4593,333.33053 2065.6501,331.02233 2067.2417,329.56513 C 2068.3465,328.55361 2071.1612,328.75712 2071.4387,327.57112 C 2071.6619,326.61757 2069.9195,326.31951 2069.2899,325.65974 C 2068.8302,325.17792 2069.0483,324.61853 2068.9963,324.06053 C 2068.9143,323.17153 2069.5203,322.77053 2069.1113,321.85753 C 2068.8633,321.44253 2068.8263,321.00653 2069.0023,320.54853 C 2069.3683,319.60953 2068.8283,319.08353 2068.8283,318.16053 C 2068.8283,316.20453 2067.8803,314.54734 2069.7383,313.16334 C 2071.8153,311.61734 2068.9603,309.93753 2068.2083,308.57453 C 2067.5413,307.36553 2065.9583,306.37553 2064.5413,306.94153 C 2063.2733,307.44753 2062.6663,306.47153 2061.9183,305.64253 C 2063.9633,304.95353 2063.3237,301.69234 2065.7357,301.22434 C 2068.6097,300.66734 2071.3739,299.56694 2074.3299,299.72994 C 2077.2679,299.89094 2079.7185,298.43577 2082.5817,299.61294 C 2083.6145,300.03756 2085.6734,300.30264 2086.6655,301.06655 C 2088.0927,302.16553 2090.0003,302.10053 2091.5773,302.90553 C 2091.1034,300.77045 2097.8206,302.98754 2097.5413,303.84513 C 2097.3114,304.55074 2100.2983,305.30113 2101.0343,305.70413 C 2101.6043,306.01713 2101.2301,306.42455 2101.7341,306.72655 C 2102.5231,307.19855 2103.189,307.61677 2103.9603,308.17353 C 2105.4575,309.25428 2107.5636,310.10599 2108.2923,311.42953 C 2108.8984,312.53039 2109.9166,313.48761 2110.9353,314.15553 C 2111.8733,314.77053 2112.4534,315.84749 2113.8733,316.33196 C 2115.3354,316.83082 2116.0463,318.70075 2117.2143,319.57575 C 2118.5243,320.55575 2119.9165,321.45834 2121.1915,322.50034 C 2122.2735,323.38334 2122.5037,324.66255 2123.5227,325.58055 C 2124.6587,326.60355 2125.5747,326.24255 2126.1097,327.81355 C 2126.9867,330.38655 2131.0013,332.61655 2133.6113,332.12755 C 2134.7153,331.92055 2136.3113,331.81353 2137.3123,332.36853 C 2138.9473,333.27653 2140.5033,333.45853 2142.1973,334.15453 C 2140.9066,331.8937 2144.1883,334.77747 2144.7183,334.80253 C 2145.9519,334.86085 2147.4553,336.65453 2148.5143,337.26453 C 2150.2233,338.24953 2151.7503,337.45853 2153.3993,338.11453 C 2153.8903,338.31053 2153.6479,339.54594 2154.0229,339.99594 C 2154.4489,340.50694 2155.3239,340.87494 2155.8859,341.19894 C 2157.5529,342.15794 2157.2267,343.1958 2157.8813,344.67253 C 2158.3463,345.72153 2161.6678,347.61169 2162.7003,348.06653 C 2163.3109,348.33549 2163.9583,347.68953 2164.6373,347.68953 C 2165.6563,347.68953 2166.4993,348.31953 2167.5383,348.19353 C 2168.7073,348.05053 2172.6531,348.21162 2172.2403,346.84753 C 2171.8027,345.40153 2174.032,344.31953 2175.4623,344.47253 C 2175.5504,344.48195 2176.7933,345.01953 2176.8743,344.66253 C 2177.0623,343.83553 2176.9963,343.60253 2177.9233,343.35953 C 2178.7743,343.13553 2180.2256,342.00024 2181.2583,342.21753 C 2181.8396,342.33985 2183.6367,342.81176 2184.0283,343.22553 C 2183.9323,344.20053 2184.2113,344.99653 2184.8903,345.69253 C 2185.8833,346.71053 2187.9915,347.92875 2187.0275,349.63975 C 2186.8225,350.00275 2186.1103,350.43053 2185.8033,350.74253 C 2185.0353,351.51953 2186.1763,352.09653 2186.4743,352.72753 C 2186.7163,353.23953 2186.5853,353.84553 2186.6983,354.38553 C 2186.9053,355.37953 2187.811,356.21632 2187.7433,357.26653 C 2187.6635,358.50435 2190.3454,360.50358 2188.2783,361.15353 C 2189.5653,362.09453 2188.4807,362.45232 2188.5387,363.55932 C 2188.5807,364.37632 2189.6525,364.97513 2189.3633,365.79553 C 2189.0484,366.68876 2187.5583,366.0396 2188.5843,368.19953 C 2188.8628,368.78587 2189.3441,369.46232 2189.4471,370.11632 C 2189.6121,371.17032 2183.9095,369.42593 2183.0135,369.22293 C 2182.6195,369.13393 2180.1563,368.06653 2179.9843,368.40653 C 2179.6023,369.15753 2179.6743,370.50853 2178.8903,370.96253 C 2178.2633,371.32653 2177.2133,371.47832 2176.7813,371.97732 C 2176.3653,372.45632 2178.0693,373.59653 2178.2873,373.82153 C 2180.3823,375.98953 2181.9463,378.52153 2183.2863,381.19953 C 2184.4103,383.44453 2186.6787,384.88915 2186.4815,387.54513 C 2186.3955,388.70323 2187.0303,389.38891 2185.6893,389.19132 C 2184.5584,389.02469 2183.5273,389.20853 2182.8793,390.38653 C 2185.0224,390.88053 2185.9594,392.81107 2186.7784,393.79207 C 2187.9159,395.15445 2182.3373,391.32153 2181.3073,389.77853 C 2180.4543,388.50053 2179.3303,388.14053 2177.8383,388.22553 C 2178.1063,389.14453 2179.5743,391.76853 2178.9823,392.66553 C 2178.5223,393.36253 2177.1733,392.64153 2176.5433,393.17653 C 2175.6213,393.95953 2177.3013,395.57153 2175.2813,396.08653 C 2173.4733,396.54753 2171.5753,396.11953 2169.7783,396.57853 C 2170.7233,397.86053 2174.1861,399.35113 2173.7821,401.11613 C 2173.5671,402.05913 2169.4323,401.42353 2168.4673,401.27553 C 2167.5713,401.13953 2166.6353,400.94353 2165.8443,400.48153 C 2164.9163,399.93953 2164.0243,398.35553 2162.7963,398.49253 C 2162.1583,398.56453 2162.0463,399.29653 2161.2773,399.31253 C 2161.3353,400.94953 2161.4723,401.83253 2160.9293,403.40953 C 2160.6903,404.10553 2160.5763,405.28353 2159.9423,405.75753 C 2159.6173,405.99953 2158.0873,406.05953 2158.1833,406.60353 C 2158.3073,407.31053 2157.6733,407.21453 2157.3473,407.73753 C 2156.9843,408.07653 2156.5563,408.28553 2156.0643,408.36453 C 2155.6203,408.58753 2155.1903,409.62953 2154.9373,409.70653 C 2153.3743,410.18153 2152.1583,411.89753 2152.1773,413.65453 C 2152.2003,415.72353 2149.0393,415.79353 2147.8873,414.57953 C 2148.0503,415.67253 2146.5253,415.54853 2145.8663,415.51453 C 2146.1443,416.31453 2145.4193,416.46453 2144.8193,416.56553 C 2143.5743,416.77353 2142.6703,417.72753 2141.6963,418.46653 C 2140.7333,419.19753 2140.0693,419.50553 2139.9503,420.82753 C 2139.8643,421.79053 2138.7523,421.69953 2139.0263,422.64153 C 2138.5503,422.31053 2138.0243,422.16653 2137.4463,422.20953 C 2138.2193,423.24953 2134.8633,425.79053 2134.1133,423.33153 C 2133.7223,422.04853 2136.2013,422.07353 2136.5433,421.70053 C 2137.0393,421.15953 2136.3973,420.71153 2136.3433,420.17153 C 2136.2753,419.49753 2137.3323,419.14453 2137.6573,418.68153 C 2135.4173,419.67353 2134.2573,418.14853 2132.2573,417.81753 C 2132.6173,417.30953 2133.1623,416.93253 2133.7773,416.80953 C 2132.5243,414.91953 2133.4123,415.22553 2134.8113,413.82853 C 2135.4523,413.18753 2135.7113,411.27553 2135.7883,410.40153 C 2136.0623,407.25453 2131.1973,407.92153 2130.8173,405.14553 C 2131.0463,406.97353 2129.1733,406.17553 2128.2043,405.82853 C 2126.9793,405.38953 2124.3533,406.77153 2126.2183,407.52153 C 2124.7993,407.79953 2123.9573,409.98753 2124.0763,411.32753 C 2124.1753,412.43153 2120.9163,413.64353 2120.1683,414.14553 C 2119.4453,414.63053 2118.7753,414.98653 2118.3673,415.80153 C 2117.7013,417.13153 2119.0003,417.78953 2118.6803,418.80953 C 2117.9223,421.22853 2115.8813,420.41753 2114.0743,421.52053 C 2112.8383,422.27553 2111.5523,420.29553 2110.1933,420.55153 C 2108.2203,420.92353 2109.2073,424.14953 2109.6603,425.37053 C 2110.4543,427.50753 2112.8613,428.29053 2114.5813,429.49453 C 2115.1233,429.87453 2116.4133,430.01553 2117.0283,429.77153 C 2118.0503,429.36853 2118.2863,430.84753 2119.1583,430.99253 C 2119.3953,430.48853 2119.3953,429.98453 2119.1583,429.48053 C 2120.6173,429.48053 2122.6193,430.94453 2123.3043,432.23453 C 2124.2483,434.01253 2122.9073,435.72953 2124.8503,437.22253 C 2126.1553,438.22453 2127.7463,438.49553 2129.3333,438.24153 C 2130.4493,438.06153 2131.7973,437.44653 2130.5273,436.32053 C 2131.1413,435.96053 2134.0083,433.58453 2131.8973,433.58553 C 2133.1093,433.81153 2134.2203,432.74653 2135.3113,432.41753 C 2136.3413,432.10553 2137.9983,433.18153 2138.6163,433.93453 C 2139.1543,434.59053 2139.6063,434.51253 2140.3513,434.28853 C 2140.7803,434.16053 2141.3003,435.01953 2141.5183,435.24053 C 2142.5173,436.25353 2144.2023,434.76253 2145.2173,435.81653 C 2145.3343,435.51053 2145.4313,435.19953 2145.5073,434.88053 C 2147.0303,435.95353 2149.1473,435.15353 2150.6163,436.24853 C 2150.3503,436.38653 2150.0873,436.53053 2149.8273,436.68053 C 2151.4183,437.97053 2149.0083,438.33253 2150.5153,439.40653 C 2151.1193,439.83753 2151.0423,440.49253 2150.2453,440.80853 C 2148.9303,441.33053 2148.7953,440.12753 2148.2473,439.34453 C 2148.0813,439.71953 2147.8883,440.07953 2147.6673,440.42453 C 2147.5073,439.34153 2146.8303,439.65653 2146.2223,440.20553 C 2145.8703,440.52253 2143.3583,442.31653 2144.7873,440.78553 C 2143.9703,441.31353 2142.4943,442.71053 2141.5723,442.87753 C 2140.9203,442.99553 2139.3623,442.12653 2139.7473,443.44853 C 2140.8063,442.51853 2141.7543,443.90853 2141.1163,444.88853 C 2140.8063,444.12953 2139.9173,444.07853 2139.7473,444.96053 C 2139.6013,445.70853 2140.4323,446.14453 2140.2863,446.91553 C 2140.2203,447.26853 2139.1403,447.77253 2138.8163,447.69653 C 2137.7953,447.45453 2138.2253,446.26253 2137.3663,445.82453 C 2138.3433,447.21853 2136.9803,446.05853 2136.2173,446.18453 C 2136.2373,447.12253 2137.3963,448.43953 2138.3773,448.41653 C 2136.3353,448.72453 2137.5213,452.32053 2135.0663,451.72853 C 2135.0983,452.69853 2134.8773,454.74753 2134.3763,455.58653 C 2133.5423,456.98253 2134.2413,457.90153 2134.6373,459.28853 C 2135.6683,458.30453 2137.3343,459.70853 2138.1593,460.31853 C 2139.7863,461.52253 2141.6853,461.96653 2143.5023,462.79453 C 2144.9253,463.44353 2145.4543,465.58753 2146.2973,466.77553 C 2147.5753,468.57553 2148.7913,470.65853 2150.4393,472.13853 C 2151.5743,473.15853 2152.6783,474.68853 2152.7793,476.22253 C 2152.8993,478.04453 2155.6603,478.30253 2156.9763,478.88753 C 2158.1303,479.40053 2157.6153,480.08353 2158.1773,480.95953 C 2158.6113,481.63253 2159.3193,481.63353 2159.9863,481.96853 C 2161.0843,482.52053 2163.5313,483.73553 2162.7883,485.35153 C 2161.2283,484.92053 2159.5053,483.36553 2157.8033,483.85153 C 2155.5183,484.50453 2155.6013,482.87353 2153.6483,482.14753 C 2151.6873,481.41853 2150.8443,483.91153 2149.1933,483.00753 C 2148.3763,482.56053 2147.9173,481.79753 2147.3963,481.06153 C 2146.7503,480.14853 2145.8163,480.20053 2144.9383,479.66453 C 2145.3833,480.00653 2145.5503,480.59753 2145.2893,481.10353 C 2145.6923,480.63653 2146.1263,480.61253 2146.5893,481.03153 C 2146.3183,480.82553 2146.0543,480.60953 2145.7993,480.38453 C 2146.6723,480.05753 2148.4363,482.35853 2146.7393,481.60853 C 2147.9093,482.39053 2149.2673,484.13053 2150.9033,483.38753 C 2151.5603,483.08853 2151.4843,482.45153 2152.3213,482.49553 C 2153.0423,482.53253 2153.9673,482.85253 2154.4313,483.43853 C 2154.7193,483.80253 2154.3803,484.24653 2154.8693,484.48853 C 2155.6653,484.88453 2156.6013,485.00253 2157.4293,485.32553 C 2159.2173,486.02453 2160.7613,487.63653 2162.4863,488.54253 C 2163.4373,489.04253 2165.5093,491.02953 2165.4483,492.23153 C 2165.4203,492.78953 2164.2503,492.57653 2163.9343,492.60553 C 2162.7113,492.71653 2162.0483,493.71953 2161.1443,494.39253 C 2160.5413,494.84153 2159.8003,495.07353 2159.6683,495.86253 C 2159.6093,496.21353 2159.5403,497.06553 2159.0313,497.09653 C 2158.1623,497.14853 2157.6643,496.30653 2156.6773,496.68653 C 2155.8483,497.00553 2155.2393,497.75253 2154.5783,498.31153 C 2155.0433,498.08453 2156.7803,496.64953 2157.2843,497.00253 C 2158.2423,497.67353 2158.2603,499.49653 2159.6973,499.46353 C 2159.3963,499.27353 2159.0833,499.10553 2158.7583,498.95953 C 2160.4653,499.21853 2161.5273,497.18953 2163.1733,497.66053 C 2164.3073,497.98453 2164.8693,498.90953 2165.6443,499.69453 C 2166.5653,500.62653 2168.6083,500.82853 2169.8483,501.11953 C 2169.0003,501.51753 2168.6253,502.16053 2168.0483,502.84753 C 2167.8503,503.21253 2167.6103,503.54853 2167.3283,503.85553 C 2166.8923,504.19653 2166.3073,504.01553 2165.8883,504.43153 C 2166.0293,504.71053 2166.2193,504.95053 2166.4583,505.15153 C 2166.9043,504.52153 2167.7503,503.64153 2168.5483,504.35953 C 2168.1313,503.93753 2168.2143,502.79553 2169.0433,503.24753 C 2169.9633,503.75053 2169.1573,504.52853 2169.5993,505.26653 C 2169.6083,505.28153 2170.4793,507.70953 2169.5473,507.10553 C 2169.0293,506.76953 2169.0833,505.90453 2168.6183,505.51153 C 2168.7343,505.98553 2168.8063,506.46553 2168.8383,506.95153 C 2168.2273,507.11153 2167.8203,506.72553 2167.6883,506.15853 C 2167.9493,507.49353 2167.5663,506.64653 2167.2493,506.15853 C 2167.1233,506.73853 2166.8833,507.26653 2166.5283,507.74253 C 2167.4563,507.55453 2168.8433,508.43653 2168.9783,509.39853 C 2168.5513,508.94653 2168.2183,508.99453 2167.9783,509.54353 C 2168.7353,509.48953 2169.6013,510.54453 2168.7373,511.07053 C 2167.6653,511.72353 2166.1553,510.44253 2165.2383,510.04753 C 2165.7253,510.43353 2166.0163,511.24653 2166.6083,511.48653 C 2167.4443,511.82553 2168.0133,511.14053 2168.5253,512.16453 C 2168.7703,512.65353 2170.1233,513.89053 2169.4513,514.49053 C 2168.8413,515.03553 2168.0693,514.99653 2168.4093,516.09453 C 2167.4703,515.66253 2167.4193,514.40553 2166.4593,514.00653 C 2166.1883,514.48253 2166.1583,514.58053 2165.6693,514.79853 C 2167.0973,514.99053 2164.9833,519.23753 2162.6483,516.38253 C 2162.9603,517.43253 2165.2833,517.22953 2164.9773,518.44253 C 2164.7143,519.48653 2163.6463,521.38053 2164.9483,522.07153 C 2164.2163,522.41453 2164.2323,522.88453 2164.3353,523.57453 C 2164.5573,525.06353 2163.3063,523.50053 2163.7293,524.51853 C 2163.2433,524.09353 2162.7633,523.66153 2162.2903,523.22253 C 2162.2693,523.74253 2164.8083,525.53553 2162.7203,525.09453 C 2163.0923,525.94853 2162.4333,526.72853 2161.6403,526.96653 C 2162.5093,528.18153 2163.1583,530.34153 2160.6303,529.70253 C 2161.3933,529.38853 2161.7503,528.60053 2161.4203,527.83053 C 2161.0763,528.10353 2160.8143,528.43953 2160.6303,528.83853 C 2160.1473,528.48853 2159.7703,527.97953 2159.6193,527.39853 C 2159.3213,527.93053 2158.7823,528.25253 2158.1803,527.97453 C 2158.5523,528.38653 2158.6993,528.86653 2158.6193,529.41453 C 2159.8013,528.88153 2161.1683,530.94753 2159.4793,530.78253 C 2159.5463,531.67153 2160.6513,532.14853 2160.9193,531.07053 C 2161.8103,531.99153 2160.8203,531.94953 2160.2143,532.54653 C 2159.5893,533.16053 2159.2713,534.00853 2158.9793,534.81453 C 2158.2033,534.08953 2157.0163,533.67953 2156.0193,533.37553 C 2156.8003,533.86453 2157.2823,535.23853 2158.3913,535.05353 C 2159.4883,534.86953 2159.8713,533.72253 2160.7793,535.10353 C 2160.0413,535.40853 2160.3053,536.61453 2160.0733,537.37753 C 2159.6883,538.63753 2161.6373,538.67753 2161.2103,539.99953 C 2160.9833,539.03953 2160.7173,539.03953 2160.4103,539.99953 C 2159.9593,539.30653 2159.3193,538.71653 2158.5403,538.41553 C 2158.4873,539.02053 2158.1353,539.55353 2157.6093,539.85453 C 2158.1323,540.13853 2159.4473,540.32053 2159.2333,541.22253 C 2158.9243,542.52353 2157.8993,540.65053 2157.8993,542.30353 C 2157.7353,541.55853 2157.0243,540.27053 2156.2403,541.36753 C 2157.2333,541.71653 2156.3673,543.18653 2157.5293,543.23853 C 2157.2713,544.03553 2155.5903,545.25953 2155.5903,543.88653 C 2154.7963,544.52053 2154.9793,544.69053 2155.6413,545.11153 C 2156.4643,545.63553 2155.7563,546.88453 2155.3703,547.41453 C 2154.9483,546.99753 2154.5413,546.56453 2154.1503,546.11853 C 2153.8993,547.06653 2152.6563,547.66753 2152.1403,546.55053 C 2151.5903,547.34253 2150.2473,546.84253 2151.0603,547.98953 C 2150.5123,547.82953 2149.9593,547.80453 2149.4003,547.91853 C 2149.9043,548.46653 2150.6313,548.67453 2151.3403,548.42253 C 2151.1973,548.99853 2151.4753,549.33753 2152.0603,549.35853 C 2151.8573,549.89053 2151.3713,549.88853 2150.9983,550.21053 C 2150.4163,550.71453 2150.8033,551.67553 2149.8993,552.23853 C 2150.2193,551.53953 2150.0033,551.27553 2149.2503,551.44653 C 2149.5413,552.22253 2149.0493,552.91253 2149.0403,553.67853 C 2148.9623,552.22553 2147.5053,553.75053 2146.7403,553.75053 C 2146.5813,553.93153 2146.0073,556.44153 2145.2263,555.67853 C 2144.5403,555.00753 2144.0803,554.65853 2143.9493,555.91653 C 2143.8033,557.29953 2143.1513,557.62553 2141.7693,557.42253 C 2142.5973,557.74153 2143.6313,557.72153 2143.8593,558.79053 C 2142.9343,557.65653 2142.9953,559.42253 2142.3393,558.71853 C 2142.0503,559.59753 2142.5663,560.71853 2141.4083,561.07053 C 2140.6513,561.30053 2139.3383,561.59453 2138.5983,561.16653 C 2139.6783,562.31153 2135.5123,562.89153 2136.4383,561.09453 C 2135.8123,561.79753 2134.9883,561.50253 2134.2783,561.16653 C 2134.4193,561.78053 2136.0653,562.59753 2135.6303,563.34653 C 2135.3623,563.80553 2134.7513,563.41753 2134.4833,563.23453 C 2133.6503,562.66753 2132.8803,562.75653 2132.9793,561.74253 C 2132.5333,562.35553 2130.8773,563.35853 2130.3193,562.96753 C 2131.5893,563.41153 2130.7643,564.38453 2129.9983,564.17353 C 2129.0983,563.92653 2129.1323,562.84553 2129.7403,562.31953 C 2128.8863,562.31653 2128.1233,562.76153 2127.6603,563.47153 C 2128.6243,563.64653 2127.5933,564.63253 2128.6603,564.91053 C 2128.4483,565.10153 2128.2083,565.24553 2127.9393,565.34353 C 2127.4843,563.79753 2123.9273,563.93553 2123.1193,565.19953 C 2123.0153,563.09153 2120.3803,562.54253 2120.6703,560.30353 C 2120.3843,561.73853 2120.4233,560.88153 2119.8103,560.37553 C 2120.2873,559.56953 2121.1143,559.43353 2121.9703,559.58353 C 2120.8563,558.80753 2119.9213,560.37353 2118.8003,559.87153 C 2120.3993,560.51453 2118.7243,561.75553 2117.6503,561.38353 C 2118.0533,561.54553 2119.5623,562.10653 2119.6413,562.59253 C 2119.4833,562.71953 2119.2993,562.72453 2119.0903,562.60753 C 2119.9873,563.20353 2122.1533,566.08453 2121.2503,567.35953 C 2120.2853,568.71153 2118.2593,564.80153 2117.7903,564.26353 C 2117.7263,565.55253 2119.2783,566.70653 2119.8803,567.71953 C 2119.3463,567.84353 2118.9533,568.25553 2118.8693,568.79953 C 2118.3393,568.53653 2117.9403,568.04553 2117.9393,567.43153 C 2116.6303,568.03253 2117.9943,569.51953 2117.1393,570.38353 C 2116.1543,569.10953 2115.4343,570.13453 2114.7693,571.04253 C 2114.3193,571.65653 2112.7393,570.34653 2113.6193,569.51953 C 2112.9983,569.76153 2112.6093,569.46253 2112.6803,568.79953 C 2112.5603,569.63553 2113.1733,570.49853 2112.9123,571.33453 C 2112.5843,572.38753 2109.8513,570.47653 2109.2903,570.31153 C 2110.5883,571.95453 2108.8653,571.19953 2107.9303,571.24853 C 2109.5563,571.89053 2107.1923,574.34353 2106.1303,572.83253 C 2105.9593,573.37553 2105.6023,573.91253 2105.0503,574.12853 C 2105.5853,573.02953 2104.2483,573.69753 2103.5133,573.87053 C 2102.6023,574.08553 2101.3053,574.38153 2100.7303,575.20853 C 2100.8933,574.79653 2101.1063,574.41253 2101.3703,574.05653 C 2100.7723,574.47553 2100.3033,575.19953 2100.7303,575.92853 C 2099.1703,576.39853 2099.7163,575.28653 2098.7803,574.56053 C 2098.4103,575.38153 2099.5203,575.91853 2098.7923,576.65853 C 2098.2783,577.17953 2097.1633,577.71253 2097.6193,578.65753 C 2097.8803,579.19853 2098.6463,579.22653 2099.1393,579.39653 C 2099.8253,579.63253 2099.6173,581.17153 2099.0713,579.52853 C 2099.1753,580.79253 2100.9353,581.45053 2100.8443,582.70753 C 2100.6913,584.82553 2097.8143,583.78753 2096.6203,584.35353 C 2096.6833,583.62553 2095.8373,582.46253 2097.1313,583.12953 C 2096.8663,581.90653 2094.6663,582.15353 2095.5413,580.46553 C 2094.5273,580.60053 2095.1633,579.15353 2094.3153,579.06253 C 2093.7763,579.00453 2093.6253,576.98853 2093.9833,576.61653 C 2094.6883,575.88353 2093.8113,575.37353 2094.7833,574.78153 C 2096.2763,573.87153 2094.1713,573.34853 2093.6003,572.83453 C 2093.7313,573.09353 2093.8513,573.35753 2093.9603,573.62653 C 2092.8483,573.86553 2092.2933,572.21553 2091.8003,571.53853 C 2091.4463,572.13553 2091.7043,572.69753 2092.3003,572.97753 C 2091.8813,573.44753 2090.4683,574.67853 2089.9303,573.69753 C 2089.0493,575.71853 2088.8563,572.66653 2088.4103,572.32953 C 2087.8873,571.93353 2086.0173,572.76753 2087.1193,571.25053 C 2086.8613,571.15853 2086.5983,571.08653 2086.3303,571.03353 C 2086.2643,571.55953 2086.3833,572.04053 2086.6893,572.47353 C 2085.8143,572.34053 2085.4043,570.67553 2084.5293,570.88953 C 2084.9253,570.38353 2084.7533,569.89153 2084.1703,569.66553 C 2083.9403,570.17953 2083.7153,570.90653 2083.5903,569.88153 C 2083.4023,570.94253 2085.0203,571.60153 2084.4193,572.68353 C 2084.1303,573.20353 2083.1363,571.99453 2082.7303,572.04153 C 2083.0063,572.66253 2082.7163,573.24353 2082.1503,573.55353 C 2082.1243,572.90753 2080.4665,572.59605 2079.6183,572.34353 C 2077.803,571.8031 2076.8338,573.60982 2076.4613,572.18553 C 2076.2223,572.33053 2075.9823,572.47453 2075.7413,572.61853 C 2075.0452,571.57021 2074.0368,571.71545 2073.2343,571.05651 C 2072.8363,570.72964 2073.0388,570.50065 2072.8084,570.10626 C 2072.5846,569.72338 2071.3314,568.98406 2070.3413,569.44953 C 2069.7283,567.53453 2068.6833,566.08953 2068.6913,564.12253 C 2070.3963,564.72532 2069.6207,560.41937 2067.0293,561.60126 C 2065.9851,562.07745 2065.7272,561.31485 2064.9342,560.91885 C 2063.9732,560.43985 2063.5502,561.3384 2062.3205,560.82474 C 2060.1816,559.93126 2059.3333,558.98453 2058.1023,557.28353 C 2057.5773,558.49553 2056.1223,558.63353 2055.2003,559.39853 C 2054.6113,559.88753 2055.7063,560.86753 2054.9683,561.61353 C 2054.3613,562.22853 2050.6113,564.05253 2050.9833,562.03453 C 2049.6823,562.08653 2049.3633,565.12753 2048.9003,564.83553 C 2048.2403,564.41953 2047.7003,563.75553 2047.1663,563.19553 C 2046.0293,562.00453 2046.1123,563.29553 2045.7243,564.19453 C 2044.3253,562.58553 2044.1423,562.23753 2043.0523,564.25053 C 2042.0493,566.10353 2040.6343,564.13153 2039.6783,563.33853 C 2037.6033,561.61753 2037.2984,564.08449 2036.4779,565.46239 C 2035.9716,566.31268 2035.3697,565.3793 2034.8149,565.39739 C 2034.4328,565.40986 2033.9663,565.46853 2033.6013,565.57953 C 2033.1103,565.72653 2033.2303,564.91453 2032.7993,564.91453 C 2030.9183,564.91453 2031.8333,568.55153 2032.2993,569.24653 C 2032.8273,570.03453 2033.5033,570.78853 2033.7863,571.71253 C 2033.9893,572.00453 2034.1753,572.30653 2034.3443,572.61953 C 2034.4133,573.14553 2033.9653,573.60353 2033.9843,574.13153 C 2034.0033,574.69053 2035.1653,575.86653 2034.6343,576.36253 C 2033.9183,577.03353 2033.0933,575.51353 2032.2013,575.78253 C 2030.8413,576.19453 2030.6843,576.29253 2030.1273,574.88853 C 2029.6473,573.68053 2028.8963,572.50753 2028.8043,571.17853 C 2027.5853,572.08353 2025.4923,574.61753 2023.8853,573.58153 C 2023.1173,573.08653 2022.9463,574.23253 2022.3033,574.09853 C 2021.6453,573.95953 2021.4383,572.51053 2021.7443,572.04253 C 2019.2293,572.00953 2021.0763,569.54053 2019.7383,568.87853 C 2018.2723,568.15253 2016.4693,568.24653 2014.9333,568.46253 C 2012.9533,568.74153 2014.5223,565.88353 2014.6443,565.18653 C 2014.8253,564.15953 2013.6403,561.23653 2015.5533,561.31353 C 2015.0413,559.79753 2014.0733,559.65753 2012.6033,559.65753 C 2010.6653,559.65753 2010.8383,558.08253 2010.0393,556.84753 C 2009.9663,556.73453 2009.2923,555.98853 2009.4243,555.79953 C 2010.0973,554.83053 2008.7313,554.16653 2008.3193,553.44753 C 2007.6283,552.23953 2009.3733,551.71953 2009.0743,550.51453 C 2007.3263,551.80453 2005.1563,550.16453 2003.3423,551.18453 C 2002.9233,551.41953 2000.4643,553.36653 2000.0353,553.02853 C 1999.1593,552.33653 2000.0843,551.61353 2000.3703,550.90353 C 2000.6543,550.19953 2000.1693,549.07853 1999.7843,548.49753 C 1999.1873,547.59553 1998.8263,547.71553 1998.5983,546.42153 C 1998.4283,545.45953 1998.8413,544.76153 1999.8643,544.68253 C 1999.0863,543.41153 1999.7863,542.41453 1999.7153,541.08253 C 2001.4023,542.60853 2001.5153,538.65953 2001.6653,538.13053 C 2004.1443,538.30853 2002.3563,537.25153 2003.9653,536.04253 C 2004.2183,535.85253 2005.1133,536.35553 2005.1703,535.91753 C 2005.2843,535.04053 2004.1813,534.39753 2003.8953,533.66653 C 2005.3763,534.24353 2004.6153,531.57753 2004.6153,530.84753 C 2004.6153,529.25953 2004.2484,527.41353 2003.7124,525.92353 C 2003.2854,524.73753 2003.2773,523.42269 2002.6068,522.35866 C 2002.0857,521.53178 2001.4618,520.62415 2000.1463,519.98653 C 2000.4403,520.56153 2000.1353,521.01153 1999.5763,521.21153 C 1999.0603,520.17353 1998.0903,519.12853 1997.7573,518.03453 C 1997.4633,517.06853 1997.4973,516.23953 1996.6943,515.51753 C 1995.3663,514.32253 1993.3073,513.85053 1992.1563,512.57053 C 1992.0581,513.33737 1991.6155,514.71764 1991.5488,515.20733 C 1991.5184,515.43006 1991.2442,515.58464 1991.157,515.73391 C 1990.9388,515.91039 1991.2227,515.44881 1990.4222,514.66481 C 1989.5518,513.81228 1989.0813,514.74353 1988.2123,514.34853 C 1987.4173,513.98853 1986.4403,513.28153 1985.5363,513.29053 C 1984.7003,513.29953 1984.3763,514.37053 1983.4833,514.01253 C 1982.8433,513.75653 1983.3663,512.83453 1983.5673,512.47853 C 1983.8843,511.91753 1984.4893,511.33853 1984.6483,510.72153 C 1984.9033,509.73053 1983.0843,509.27753 1983.4463,508.03353 C 1982.8053,508.21653 1982.1333,509.09653 1981.4053,508.92953 C 1979.7823,508.55753 1981.5873,507.05553 1982.1463,506.81053 C 1981.4953,506.22053 1980.6213,505.42653 1979.9033,504.91253 C 1979.1923,504.40253 1977.0883,505.98253 1976.5033,506.39153 C 1975.9013,506.81253 1976.1953,507.74553 1975.6213,508.13853 C 1975.1083,508.49053 1974.1693,508.04253 1973.7203,507.85353 C 1973.5483,507.78053 1971.4083,506.47253 1972.2863,507.52953 C 1971.6283,507.14353 1970.6113,506.35153 1969.9063,506.16153 C 1969.0943,505.94253 1968.4583,506.62453 1967.6653,507.09353 C 1966.5233,507.76753 1965.9083,509.03153 1965.0693,510.00653 C 1964.0063,511.24353 1962.9773,511.21453 1961.4883,511.47453 C 1959.8463,511.76153 1960.4863,512.87253 1959.7713,513.88753 C 1959.1143,514.81953 1957.5013,515.23853 1957.2133,516.34453 C 1956.8353,517.78853 1956.8193,518.84553 1954.8693,518.91553 C 1953.6143,518.95953 1952.6893,519.73553 1951.4563,519.57053 C 1951.0028,519.51003 1950.247,519.68209 1949.961,519.51934 C 1949.675,519.35659 1949.8588,518.85903 1949.8423,518.36053 C 1949.8143,517.47353 1947.5612,516.96483 1946.7854,516.98097 C 1945.6001,517.00563 1945.9788,518.41052 1944.2453,517.24953 C 1942.132,517.1775 1940.9654,517.14496 1940.2367,516.77766 C 1939.1114,516.21049 1940.2082,515.35766 1939.0862,515.02066 C 1937.6562,514.59266 1936.4433,514.81684 1935.1723,515.66084 C 1933.8373,516.54784 1933.0376,518.63078 1932.3496,520.05978 C 1931.8276,521.14578 1930.5002,522.18884 1930.4092,523.37984 C 1927.6909,521.28703 1929.6863,519.65372 1929.1543,518.11453 C 1928.3096,515.67078 1925.6328,517.55593 1923.7553,517.68253 C 1923.8013,517.94753 1923.8513,518.21153 1923.9043,518.47453 C 1922.2763,518.00353 1919.9813,518.60853 1918.2583,518.57353 C 1916.7033,518.54153 1915.7073,517.94453 1914.2663,517.21553 C 1913.7633,516.96153 1912.9083,516.41753 1912.3083,516.67753 C 1911.7743,516.90853 1912.1203,517.76953 1911.5483,517.87353 C 1910.6783,518.03153 1908.9663,516.79053 1908.9243,515.88353 C 1908.7173,516.05353 1908.6743,519.01153 1907.9383,517.78353 C 1907.1883,516.53353 1905.5753,515.79653 1905.1043,514.37153 C 1904.8503,515.53453 1900.9433,515.04853 1900.6583,514.02153 C 1900.5063,513.47253 1901.0703,512.50553 1900.6283,512.05153 C 1900.1703,511.58153 1898.7873,512.63753 1897.9913,512.16053 C 1896.7253,511.40153 1893.8783,510.44753 1893.3763,508.96253 C 1892.7593,507.13453 1890.8753,505.13353 1888.9513,507.05653 C 1888.2583,507.74953 1886.0293,504.77553 1885.6723,504.34753 C 1884.6883,503.16953 1882.7423,502.92253 1881.2813,502.03353 C 1879.0063,500.64853 1876.9773,499.10953 1875.0243,497.35953 C 1874.2783,496.69053 1873.2523,496.78953 1872.3543,496.50053 C 1871.3853,496.18853 1871.3753,497.41853 1870.3343,497.01953 C 1870.7303,499.63953 1869.2543,499.97053 1867.7263,498.29553 C 1866.3593,496.79653 1864.6403,496.09653 1862.7423,495.57853 C 1862.3413,495.46853 1861.4003,495.35053 1861.1143,495.00453 C 1860.6733,494.46953 1861.4893,493.60753 1860.5343,493.22753 C 1859.8113,492.93953 1859.4503,492.85553 1858.8203,492.27753 C 1857.7403,491.28653 1857.0283,491.99953 1855.8753,491.65953 C 1854.1353,491.14753 1853.4333,489.27253 1852.2703,488.12953 C 1851.0973,486.97553 1849.8283,490.24853 1849.0933,487.94453 C 1848.8333,487.13353 1849.1473,486.31153 1848.6623,485.55353 C 1848.1723,484.78653 1847.9803,484.15853 1847.8323,483.27353 C 1847.6723,482.32153 1846.7193,481.74153 1846.0613,481.15453 C 1845.0883,480.28653 1845.3123,478.95353 1844.4853,478.01153 C 1846.4483,476.54953 1846.8333,477.61553 1847.9643,478.96653 C 1848.5223,479.63253 1849.3593,479.74953 1849.8073,479.00253 C 1850.3653,478.06953 1851.3327,478.14374 1851.9287,477.33274 C 1852.6627,476.33274 1851.6743,475.35553 1851.0873,474.66353 C 1850.6213,474.11353 1850.8583,472.60453 1850.2853,472.43253 C 1847.9433,471.72453 1848.0881,471.88894 1846.4179,470.19422 C 1844.533,468.28168 1844.3004,464.84447 1844.3674,464.37354 C 1844.4008,464.13909 1846.9,463.40126 1844.7865,462.72189 C 1842.9593,462.13454 1841.5361,460.7173 1839.8215,459.68165 C 1837.4586,458.2544 1838.2135,454.95331 1834.2883,452.65553 C 1831.9693,452.75753 1824.9723,452.27653 1824.8253,448.92453 C 1824.1373,449.10153 1823.1463,450.21153 1822.4093,449.54853 C 1821.8373,449.03353 1820.3053,448.12653 1820.0043,447.45853 C 1819.7283,446.84553 1819.7973,445.62353 1819.6093,444.91053 C 1819.3513,443.92853 1818.8673,443.13953 1818.1233,442.44453 C 1817.0743,441.46653 1815.2713,442.01353 1814.6203,441.01153 C 1813.9143,439.92753 1811.6053,438.61553 1810.7163,440.21253 C 1809.9993,438.47453 1808.0023,439.75453 1807.3193,438.48853 C 1806.4173,436.81653 1809.5553,437.52253 1810.0093,437.44053 C 1811.8903,437.10153 1811.1453,435.35753 1810.0783,434.70153 C 1809.1763,434.14453 1809.6023,433.85253 1809.5723,432.97753 C 1809.5593,432.58653 1808.8863,431.41253 1808.7053,430.99353 C 1808.4493,430.40153 1807.7303,430.33753 1807.4973,429.67753 C 1807.2243,428.90353 1807.4793,428.07253 1807.3183,427.28253 C 1807.0203,425.82553 1804.5773,425.42653 1803.3753,425.16353 C 1801.7803,424.81253 1800.6523,426.94353 1799.4363,425.41253 C 1799.0423,424.91753 1797.3823,422.65453 1798.9863,422.64453 C 1798.4723,421.92053 1797.7753,421.36553 1797.1533,420.74353 C 1796.5383,420.13153 1796.7903,419.05853 1796.5263,418.31753" id="path2708"/>
    </g>
    <g id="hk">
      <g class="landxx coastxx cn hk" id="hk-">
        <path d="M 2124.3363,564.83753 C 2125.0943,564.57853 2125.8133,564.21253 2126.4263,564.98153 C 2126.2043,565.19653 2125.9643,565.38753 2125.7063,565.55753 C 2126.0063,565.29453 2126.5713,565.10953 2126.9303,565.38753 C 2128.0473,566.25453 2125.6753,565.46853 2126.6463,566.78253 C 2125.7983,566.60353 2122.6303,565.94853 2124.3363,564.83753" id="path2764"/>
        <path d="M 2123.3273,567.50153 C 2123.2953,566.74853 2124.2253,566.57853 2124.6963,566.27753 C 2124.7963,567.06553 2124.0213,567.53353 2123.3273,567.50153" id="path2762"/>
        <path d="M 2125.7763,566.78153 C 2125.7773,567.15653 2125.9703,567.39653 2126.3563,567.50153 C 2126.2653,567.17553 2126.0723,566.93553 2125.7763,566.78153" id="path2760"/>
      </g>
      <circle class="circlexx subxx cn hk" cx="2126.8741" cy="566.867665" id="hk_" r="4.2338"/>
    </g>
    <g id="mo">
      <path class="landxx coastxx cn mo" d="M 2121.0963,567.50153 C 2121.2273,567.42253 2121.3423,567.92253 2121.2463,567.35653 C 2121.3683,567.63453 2121.3923,567.92253 2121.3153,568.22053 C 2121.2433,567.98153 2121.1693,567.74153 2121.0963,567.50153" id="mo-"/>
      <circle class="circlexx subxx cn mo" cx="2118.7195" cy="569.462015" id="mo_" r="4.2338"/>
    </g>
  </g>
  <g class="landxx coastxx ca" id="ca" transform="translate(-29.9017, -45.0745)">
    <path d="M 480.241,334.296 C 479.628,334.229 479.1,334.421 478.657,334.872 C 479.212,334.755 479.74,334.563 480.241,334.296" id="path3776"/>
    <path d="M 475.705,335.448 C 474.688,335.145 473.959,335.964 473.401,336.672 C 474.236,336.41 475.031,336.013 475.705,335.448" id="path4128"/>
    <path d="M 466.777,373.68 C 466.556,373.897 466.364,374.137 466.201,374.4 C 466.889,375.106 467.862,374.906 468.289,374.041 C 467.599,375.427 468.783,374.166 469.585,374.617 C 469.318,375.099 469.102,375.603 468.937,376.129 C 469.836,375.853 470.746,376.573 471.889,376.345 C 469.958,377.367 468.66,376.757 467.209,378.721 C 468.465,378.643 469.744,378.814 470.953,378.362 C 470.733,378.623 470.517,378.887 470.305,379.154 C 470.89,379.173 471.37,378.957 471.745,378.506 C 471.347,378.993 470.867,379.377 470.305,379.658 C 470.846,379.87 471.326,379.774 471.745,379.37 C 471.532,379.957 471.745,380.33 472.393,380.306 C 471.747,380.8 470.979,381.43 470.089,381.17 C 471.574,382.967 471.548,382.265 473.617,381.89 C 471.531,382.667 474.791,382.324 475.296,382.223 C 476.372,382.01 477.41,381.084 477.865,380.09 C 477.534,381.945 475.187,382.25 473.905,383.186 C 473.017,383.834 472.784,383.898 473.617,384.699 C 473.996,385.063 475.609,386.265 476.152,385.957 C 476.636,385.682 479.359,388.006 480.744,387.65 C 481.404,387.481 482.361,387.281 482.973,387.055 C 483.722,386.779 483.601,385.572 483.913,384.987 C 483.338,385.154 482.858,385.466 482.473,385.924 C 483.072,384.955 483.998,383.303 483.049,382.252 C 484.972,381.669 482.985,380.062 482.132,379.541 C 480.86,378.766 478.694,378.612 479.98,376.632 C 480.467,375.882 479.897,374.901 479.826,374.095 C 479.75,373.231 480.427,371.894 480.21,371.163 C 480.004,370.472 475.83,369.723 475.269,369.702 C 473.802,369.648 471.524,369.058 470.314,368.161 C 468.65,366.928 467.593,366.161 465.415,366.692 C 464.381,366.944 462.088,368.595 463.866,369.627 C 464.434,369.957 465.195,369.863 465.8,369.713 C 466.288,369.592 468.565,368.838 466.851,368.643 C 467.611,368.585 468.323,368.917 469.083,368.86 C 468.246,368.902 467.457,369.509 467.355,370.372 C 467.948,367.987 462.228,371.601 464.835,371.019 C 464.488,371.867 463.622,372.264 462.963,372.819 C 463.784,373.075 466.074,371.758 465.987,373.467 C 466.626,372.921 467.477,373.021 468.218,372.747 C 467.679,372.967 467.199,373.278 466.777,373.68" id="path4804"/>
    <path d="M 473.329,351.576 C 475,351.572 475.664,349.517 477.145,349.055 C 476.317,350.011 475.143,350.602 474.409,351.649 C 475.774,352.431 478.934,348.824 478.729,347.544 C 478.28,344.744 474.485,348.089 473.689,349.056 C 474.416,349.569 476.327,348.814 475.92,347.76 C 476.128,347.999 476.295,348.263 476.424,348.552 C 475.263,348.585 472.25,350.025 473.329,351.576" id="path4806"/>
    <path d="M 477.721,350.496 C 476.503,350.745 475.192,352.741 474.985,353.88 C 476.326,353.258 477.138,351.779 477.721,350.496" id="path4808"/>
    <path d="M 478.369,355.752 C 479.741,355.184 481.803,355.109 482.545,353.592 C 479.773,353.625 477.024,355.441 475.129,357.336 C 476.512,357.46 477.218,356.24 478.369,355.752" id="path4810"/>
    <path d="M 479.017,344.448 C 478.14,344.828 477.197,345.372 476.641,346.176 C 477.778,345.721 478.465,345.664 479.017,344.448" id="path4812"/>
    <path d="M 479.233,350.28 C 478.897,350.417 478.55,350.52 478.193,350.589 C 477.62,350.778 477.396,351.664 477,352.08 C 477.784,351.532 478.536,350.936 479.233,350.28" id="path4814"/>
    <path d="M 479.521,370.512 C 479.785,370.368 480.049,370.224 480.313,370.08 C 479.815,370.228 479.311,370.349 478.8,370.44 C 479.041,370.464 479.281,370.488 479.521,370.512" id="path4816"/>
    <path d="M 480.025,377.136 C 479.881,377.402 479.905,377.665 480.097,377.928 C 480.073,377.664 480.049,377.399 480.025,377.136" id="path4818"/>
    <path d="M 480.097,373.68 C 480.667,373.416 481.696,371.878 481.393,371.232 C 479.993,371.472 480.646,372.792 480.097,373.68" id="path4820"/>
    <path d="M 481.752,369.864 C 481.209,369.988 480.705,370.204 480.24,370.511 C 480.813,370.767 481.642,370.59 481.752,369.864" id="path4822"/>
    <path d="M 482.256,370.08 C 481.896,370.561 481.416,370.825 480.817,370.872 C 481.837,370.843 482.003,371.934 481.249,372.456 C 482.116,372.181 483.114,370.904 482.256,370.08" id="path4824"/>
    <path d="M 482.689,372.024 C 482.137,372.235 481.751,372.7 481.897,373.32 C 482.375,373.061 482.733,372.594 482.689,372.024" id="path4826"/>
    <path d="M 484.057,377.928 C 483.971,376.659 483.318,376.091 482.329,375.408 C 481.817,376.517 483.34,377.424 484.057,377.928" id="path4828"/>
    <path d="M 482.689,377.784 C 482.533,378.441 482.773,378.513 483.409,378 C 483.169,377.928 482.929,377.856 482.689,377.784" id="path4830"/>
    <path d="M 483.84,371.231 C 483.211,371.467 482.898,372.083 483.336,372.671 C 483.686,372.255 483.854,371.775 483.84,371.231" id="path4832"/>
    <path d="M 484.345,382.68 C 483.793,383.128 483.807,383.843 483.841,384.48 C 484.882,384.382 484.91,383.332 484.345,382.68" id="path4834"/>
    <path d="M 485.065,371.231 C 484.708,371.353 484.42,371.569 484.201,371.879 C 484.775,372.084 485.094,371.835 485.065,371.231" id="path4836"/>
    <path d="M 484.632,382.176 C 484.8,382.701 485.213,383.064 485.784,383.04 C 485.381,382.777 484.997,382.489 484.632,382.176" id="path4838"/>
    <path d="M 486.361,375.624 C 485.672,375.58 485.068,375.974 484.849,376.632 C 485.326,376.256 485.83,375.92 486.361,375.624" id="path4842"/>
    <path d="M 488.88,378.72 C 488.634,378.847 488.33,379.079 488.16,379.296 C 488.4,379.104 488.641,378.912 488.88,378.72" id="path4848"/>
    <path d="M 489.384,377.856 C 488.982,377.712 488.867,377.83 488.592,378.215 C 488.856,378.096 489.12,377.977 489.384,377.856" id="path4850"/>
    <path d="M 889.488,299.016 C 888.438,298.32 888.309,300.013 887.688,300.456 C 888.598,300.632 889.29,299.809 889.488,299.016" id="path5230"/>
    <path d="M 758.592,277.704 C 758.974,277.628 759.31,277.46 759.6,277.2 C 759.056,276.866 758.607,277.022 758.592,277.704" id="path5010"/>
    <path d="M 786.818,347.256 C 785.553,346.808 784.047,346.798 782.739,347.032 C 781.23,347.303 778.778,348.531 781.291,349.433 C 782.81,349.979 785.225,351.311 786.845,350.894 C 788.377,350.501 787.181,348.087 786.818,347.256" id="path5042"/>
    <path d="M 793.518,356.256 C 792.728,356.16 790.611,356.703 790.638,357.768 C 791.741,357.315 792.81,357.375 793.518,356.256" id="path5048"/>
    <path d="M 794.667,347.399 C 794.311,347.514 794.07,347.754 793.947,348.12 C 794.188,347.88 794.428,347.64 794.667,347.399" id="path5052"/>
    <path d="M 794.238,346.608 C 794.917,346.714 795.06,346.426 794.668,345.744 C 794.072,345.795 794.162,346.155 794.238,346.608" id="path5054"/>
    <path d="M 806.757,325.008 C 807.882,325.848 809.687,323.845 810.507,323.28 C 810.119,324.474 808.78,324.909 807.907,325.656 C 808.542,325.213 809.297,325.298 810.001,325.08 C 811.148,324.725 812.211,323.47 813.097,322.704 C 813.187,323.679 812.175,324.278 811.587,324.864 C 812.897,323.867 813.772,322.449 814.967,321.336 C 815.165,322.807 812.759,324.011 811.947,324.937 C 813.045,323.953 814.045,322.77 815.298,321.982 C 816.205,321.412 815.459,320.512 815.366,319.713 C 815.34,319.484 811.078,322.758 810.647,322.992 C 811.69,322.234 812.704,321.266 813.307,320.112 C 811.347,322.031 808.508,322.94 806.757,325.008" id="path5074"/>
    <path d="M 812.448,317.231 C 811.967,317.365 811.747,317.337 811.798,317.952 C 812.014,317.712 812.231,317.472 812.448,317.231" id="path5090"/>
    <path d="M 813.818,323.64 C 813.325,324.47 812.511,325.027 811.948,325.8 C 812.724,325.236 813.431,324.532 813.818,323.64" id="path5092"/>
    <path d="M 813.238,316.584 C 812.874,316.649 812.564,316.817 812.308,317.088 C 813.022,317.493 813.085,317.364 813.238,316.584" id="path5094"/>
    <path d="M 817.708,320.832 C 816.826,320.294 815.116,322.664 814.678,323.208 C 816.13,323.001 817.113,322.148 817.708,320.832" id="path5098"/>
    <path d="M 875.808,391.176 C 875.546,391.737 875.116,392.22 874.588,392.544 C 873.755,391.63 875.196,391.36 875.808,391.176" id="path5602"/>
    <path d="M 876.098,390.384 C 875.954,390.648 875.811,390.912 875.668,391.176 C 875.603,390.636 875.836,390.707 876.098,390.384" id="path5606"/>
    <path d="M 850.388,417.6 C 850.416,418.201 850.012,418.609 849.458,418.752 C 849.44,418.213 849.811,417.64 850.388,417.6" id="path5598"/>
    <path d="M 852.768,421.344 C 852.505,421.56 852.241,421.777 851.978,421.992 C 852.181,421.701 852.445,421.485 852.768,421.344" id="path5600"/>
    <path d="M 826.348,293.832 C 826.548,293.563 826.715,293.275 826.848,292.968 C 826.238,293.045 825.76,293.478 825.558,294.048 C 825.821,293.976 826.084,293.904 826.348,293.832" id="path5126"/>
    <path d="M 842.398,285.696 C 841.386,285.51 840.265,285.777 839.377,286.272 C 840.371,286.46 841.545,286.243 842.398,285.696" id="path5150"/>
    <path d="M 816.268,311.832 C 816.712,311.683 816.857,311.679 816.838,311.112 C 816.647,311.353 816.458,311.592 816.268,311.832" id="path5104"/>
    <path d="M 808.198,322.56 C 809.902,322.086 811.46,321.282 812.668,319.968 C 811.901,320.615 807.921,321.51 808.198,322.56" id="path5078"/>
    <path d="M 852.837,149.832 C 853.591,149.951 854.85,149.566 855.525,149.937 C 856.369,150.401 855.568,150.769 856.917,150.825 C 857.828,150.863 859.886,150.055 860.246,149.123 C 860.455,148.581 858.133,148.404 857.873,148.38 C 856.059,148.209 854.231,148.575 852.837,149.832" id="path5172"/>
    <path d="M 865.658,159.84 C 865.288,159.818 864.952,159.914 864.648,160.128 C 864.86,160.317 865.101,160.461 865.368,160.56 C 865.464,160.319 865.561,160.08 865.658,159.84" id="path5194"/>
    <path d="M 870.408,149.184 C 870.878,149.452 871.381,149.62 871.918,149.689 C 871.13,149.778 870.335,149.735 869.548,149.833 C 871.159,149.708 869.533,150.446 870.457,150.967 C 871.068,151.311 871.892,151.322 872.572,151.341 C 874.865,151.404 877.28,151.111 879.478,150.481 C 876.947,153.039 872.734,150.675 869.838,152.354 C 870.237,152.884 870.876,153.122 871.488,152.786 C 871.262,153.017 871.045,153.258 870.838,153.506 C 872.687,154.089 876.161,153.211 878.038,152.859 C 877.248,153.104 876.398,153.155 875.668,153.579 C 877.456,154.098 879.492,153.288 881.138,152.642 C 880.087,153.139 880.075,153.287 881.105,153.086 C 881.887,153.027 882.058,152.428 882.648,152.355 C 883.159,152.292 883.497,152.792 884.224,152.688 C 884.994,152.578 885.759,152.47 886.537,152.57 C 885.821,152.738 885.073,152.745 884.377,153.003 C 886.509,153.205 888.66,153.178 890.787,152.931 C 888.27,153.81 885.498,152.671 883.007,153.651 C 884.626,154.371 886.376,153.745 888.06,153.886 C 889.899,154.041 891.768,153.941 893.597,153.723 C 891.975,154.245 890.232,154.084 888.557,154.155 C 886.754,154.231 884.95,154.556 883.142,154.39 C 879.647,154.071 876.213,154.624 872.787,155.379 C 873.518,156.705 875.329,156.26 876.527,156.675 C 875.494,157.149 874.357,156.582 873.287,156.747 C 874.609,157.673 874.78,157.885 876.453,157.659 C 878.253,157.416 880.051,158.106 881.858,158.043 C 879.54,158.605 877.094,157.855 874.797,158.547 C 875.529,159.79 878.335,160.247 879.557,160.419 C 880.66,160.573 881.751,160.799 882.868,160.842 C 883.542,160.868 887.076,159.943 885.027,159.771 C 886.606,159.186 887.932,159.789 889.267,160.635 C 889.876,159.723 888.862,157.859 890.427,157.755 C 889.799,161.026 893.466,160.652 895.107,159.411 C 895.59,159.046 896.091,157.625 896.885,157.953 C 898.251,158.517 899.047,158.69 900.437,157.971 C 900.166,157.784 899.9,157.592 899.637,157.395 C 901.175,155.948 903.009,155.259 905.037,154.803 C 903.82,155.493 902.123,155.727 901.437,157.108 C 903.305,158.777 907.224,154.945 909.164,155.407 C 910.398,155.701 912.642,154.584 914.025,154.495 C 916.147,154.358 918.3,154.113 920.377,153.651 C 919.255,151.071 918.619,151.828 916.487,152.571 C 916.745,152.349 916.985,152.109 917.207,151.851 C 915.119,150.088 912.512,152.659 910.227,152.067 C 911.94,151.077 913.965,150.79 915.627,149.692 C 914.598,149.88 913.577,150.139 912.527,150.196 C 913.544,150.013 914.509,149.558 915.557,149.548 C 915.066,148.364 913.357,148.707 912.387,148.828 C 912.711,148.526 916.311,147.972 915.287,146.932 C 914.35,145.98 911.445,146.63 910.586,147.387 C 910.848,147.581 911.111,147.773 911.376,147.963 C 910.059,148.591 908.661,148.776 907.636,147.531 C 907.924,147.479 910.499,146.955 908.566,146.955 C 910.906,146.494 906.192,146.241 906.116,146.163 C 905.561,145.596 903.658,145.81 902.956,145.948 C 904.556,145.636 903.411,145.39 902.947,144.745 C 902.506,144.132 902.112,143.317 901.296,143.14 C 903.318,142.737 898.838,141.64 898.096,141.592 C 896.262,141.473 893.332,141.128 891.646,141.916 C 892.917,143.158 894.978,141.983 896.476,142.563 C 894.662,143.607 892.169,143.176 890.206,142.923 C 890.679,143.17 891.182,143.29 891.716,143.283 C 890.941,143.195 890.21,143.471 889.456,143.58 C 888.628,143.698 888.102,143.294 887.32,143.204 C 886.609,143.122 882.853,143.817 884.517,144.146 C 884.02,144.263 883.517,144.335 883.007,144.363 C 883.725,144.738 884.536,144.679 885.307,144.579 C 883.105,144.709 887.666,144.973 888.047,145.083 C 887.268,145.283 886.462,145.227 885.667,145.227 C 886.335,145.838 887.229,145.768 888.047,145.948 C 886.148,146.177 884.271,145.576 882.373,145.75 C 880.722,145.901 878.646,146.458 877.038,146.38 C 877.591,146.443 878.094,146.634 878.548,146.956 C 876.74,147.242 874.892,147.172 873.069,147.172 C 874.029,147.246 874.029,147.366 873.069,147.532 C 874.891,147.774 876.709,148.111 878.548,147.819 C 878.074,147.93 877.594,148.003 877.109,148.036 C 878.491,149.21 881.9,148.005 883.588,148.108 C 882.526,148.374 881.421,148.344 880.348,148.54 C 880.561,148.76 880.778,148.976 880.998,149.187 C 877.709,149.52 873.985,147.693 870.838,148.467 C 872.496,148.962 871.037,148.991 870.408,149.184" id="path5200"/>
    <path d="M 883.368,145.296 C 881.815,145.094 880.104,145.108 878.548,145.296 C 879.438,145.639 883.093,146.009 883.368,145.296" id="path5218"/>
    <path d="M 898.558,158.76 C 897.595,158.752 896.173,159.319 895.538,160.056 C 896.652,159.898 897.585,159.273 898.558,158.76" id="path5242"/>
    <path d="M 913.757,154.944 C 912.419,155.215 911.15,155.794 909.787,155.952 C 911.804,156.283 915.201,156.296 916.917,155.015 C 915.868,154.893 914.811,154.944 913.757,154.944" id="path5262"/>
    <path d="M 790.348,159.552 C 791.627,159.856 793.365,159.799 794.519,159.121 C 796.976,157.679 791.892,156.789 790.778,156.888 C 788.284,157.109 785.888,157.911 783.395,158.155 C 780.515,158.436 777.552,158.355 774.788,159.337 C 775.652,160.113 777.142,159.598 778.124,159.423 C 779.698,159.142 781.028,159.901 782.568,159.913 C 783.864,159.924 784.911,158.81 786.201,159.585 C 787.389,160.298 789.088,159.7 790.348,159.552" id="path5032"/>
    <path d="M 656.857,198.36 C 655.567,198.94 654.284,199.974 652.835,200.106 C 651.173,200.258 650.187,202.022 648.504,202.104 C 650.038,202.354 651.588,201.784 653.112,202.175 C 651.128,202.485 653.81,203.158 654.3,203.317 C 655.025,203.552 656.198,204.018 655.995,205.005 C 655.807,205.918 653.98,207.834 655.817,208.228 C 658.509,208.804 662.446,206.956 664.909,206.061 C 666.597,205.447 668.958,206.069 670.801,205.781 C 672.993,205.439 674.606,204.728 676.318,203.343 C 680.193,200.209 684.937,200.397 689.183,198.143 C 691.276,197.032 693.751,197.194 695.89,196.221 C 697.954,195.282 700.304,195.078 702.513,194.69 C 705.096,194.235 707.645,193.607 710.224,193.134 C 712.253,192.762 714.437,192.702 716.396,192.032 C 716.802,191.893 719.058,191.208 718.792,190.521 C 718.622,190.083 717.563,189.877 717.216,189.508 C 716.632,188.888 716.278,188.095 715.709,187.463 C 714.307,185.906 711.578,185.915 709.668,186.109 C 707.298,186.348 705.212,187.745 702.935,187.92 C 703.6,187.451 704.589,187.368 704.879,186.48 C 703.313,186.535 701.374,186.565 700.199,187.777 C 700.677,186.776 702.095,186.668 703.007,186.337 C 700.656,186.505 699.311,185.415 697.261,184.563 C 694.615,183.463 690.894,184.79 688.115,184.834 C 685.122,184.882 681.726,184.756 678.797,185.42 C 677.615,185.688 674.152,188.672 676.583,189.578 C 672.807,191.763 667.731,191.714 664.271,194.546 C 666.012,194.432 664.351,195.547 663.701,195.599 C 662.923,195.662 662.058,195.443 661.3,195.678 C 660.605,195.893 660.429,197.006 659.737,197.42 C 658.863,197.941 657.829,198.104 656.857,198.36" id="path4926"/>
    <path d="M 716.185,175.536 C 716.871,175.415 719.992,175.179 720.185,174.357 C 720.345,173.676 722.648,172.288 723.385,172.295 C 723.266,172.857 723.026,173.36 722.665,173.807 C 723.7,173.879 724.689,173.599 725.503,172.949 C 726.42,172.216 727.763,172.576 728.713,171.864 C 728.451,171.693 728.187,171.525 727.921,171.36 C 729.485,170.591 731.081,171.278 732.602,170.193 C 733.868,169.291 735.001,169.867 736.274,169.2 C 735.171,170.54 733.191,171.264 731.593,171.792 C 734.727,174.052 736.966,170.885 740.112,170.764 C 741.345,170.716 742.598,170.763 743.814,170.518 C 744.587,170.362 746.858,169.317 745.129,168.697 C 746.08,168.101 747.275,168.742 748.297,168.48 C 747.759,168.239 747.339,167.768 747.217,167.184 C 749.28,166.544 751.525,166.812 753.553,166.033 C 752.902,163.252 748.519,165.173 747.001,164.953 C 747.273,165.107 747.538,165.275 747.793,165.457 C 747.282,165.939 744.565,166.074 743.873,165.982 C 742.085,165.743 740.391,165.598 738.58,165.696 C 734.212,165.931 730.406,167.816 726.152,168.53 C 722.465,169.149 718.981,170.338 715.388,171.329 C 713.462,171.86 711.487,171.594 709.533,171.878 C 707.318,172.2 705.291,173.338 703.154,173.953 C 704.227,173.852 705.248,173.472 706.322,173.377 C 705.361,173.658 705.361,173.754 706.322,173.665 C 705.557,174.043 704.746,174.31 703.946,174.601 C 705.487,175.724 707.782,174.032 709.49,174.385 C 710.587,174.613 711.22,175.02 712.514,174.53 C 713.565,174.131 714.288,173.074 715.538,173.378 C 715.29,173.585 715.05,173.801 714.818,174.026 C 715.057,174.172 715.297,174.316 715.538,174.458 C 715.045,174.765 714.541,175.052 714.026,175.321 C 714.76,175.689 715.501,175.018 716.185,175.536" id="path4960"/>
    <path d="M 704.017,174.096 C 703.801,174.192 703.585,174.288 703.369,174.384 C 703.864,174.364 704.344,174.268 704.809,174.096 C 704.544,174.096 704.281,174.096 704.017,174.096" id="path4962"/>
    <path d="M 703.584,175.032 C 703.82,175.088 704.06,175.088 704.304,175.032 C 704.064,175.032 703.825,175.032 703.584,175.032" id="path4964"/>
    <path d="M 712.441,175.032 C 711.503,174.742 710.009,175.757 709.345,176.328 C 710.427,176.045 711.559,175.763 712.441,175.032" id="path4990"/>
    <path d="M 730.873,173.735 C 728.453,173.626 726.072,174.671 723.718,175.094 C 721.681,175.46 718.821,175.74 717.265,177.263 C 718.783,177.281 719.869,177.862 721.422,177.449 C 722.681,177.114 723.903,176.66 725.164,176.326 C 726.985,175.844 729.694,175.383 730.873,173.735" id="path4994"/>
    <path d="M 728.641,180.432 C 728.881,180.384 729.121,180.336 729.361,180.288 C 728.909,180.52 728.429,180.641 727.921,180.647 C 728.627,181.04 730.924,181.835 731.269,180.571 C 731.401,180.085 733.227,180.36 733.826,180.071 C 733.543,180.264 733.28,180.48 733.034,180.718 C 733.686,181.61 735.568,181.176 736.42,180.951 C 736.977,180.804 740.15,179.091 738.146,179.35 C 738.734,178.852 738.894,179.163 739.403,179.45 C 740.116,179.85 741.326,178.952 741.962,178.702 C 742.523,178.481 742.4,177.573 743.11,178.257 C 743.761,178.884 745.041,178.285 745.85,178.342 C 744.896,178.484 743.947,178.675 743.028,178.972 C 742.512,179.138 740.226,179.508 741.962,179.855 C 741.417,180.007 740.865,180.126 740.306,180.214 C 741.311,181.241 743.011,180.545 744.214,180.425 C 745.483,180.298 747.801,178.982 748.946,179.566 C 748.682,179.734 748.418,179.903 748.154,180.071 C 750.939,181.252 753.755,178.929 756.578,179.423 C 752.977,181.144 749.008,180.948 745.13,181.223 C 743.109,181.367 741.201,182.231 739.162,182.231 C 737.247,182.231 735.353,182.748 733.538,183.311 C 735.794,187.654 745.695,184.396 749.063,183.952 C 752.146,183.544 754.776,182.262 757.946,182.16 C 760.726,182.071 763.737,180.368 766.439,180.72 C 764.651,181.58 767.386,181.479 768.012,181.523 C 769.314,181.614 770.717,181.461 771.699,180.504 C 771.481,180.724 771.288,180.964 771.119,181.224 C 773.057,182.643 776.797,181.243 778.949,180.907 C 781.552,180.502 784.04,179.253 786.324,177.975 C 786.991,177.601 790.386,175.974 789.997,174.893 C 789.697,174.059 787.199,174.101 786.526,174.13 C 785.803,174.161 781.55,175.42 783.65,175.753 C 782.834,175.737 782.075,176.092 781.42,176.545 C 781.542,175.984 781.926,175.514 782.43,175.249 C 780.572,174.815 778.82,175.499 777.03,175.896 C 777.798,175.061 779.021,174.954 780.05,174.672 C 779.282,173.751 777.824,174.168 776.81,174.168 C 777.809,173.449 778.811,173.474 779.864,172.996 C 780.653,172.638 780.962,171.693 779.76,171.792 C 780.453,171.475 782.586,170.188 782.5,169.272 C 779.333,168.527 776.219,170.673 773.21,171.36 C 772.135,171.605 770.68,171.265 769.83,172.151 C 770.258,173.392 771.993,172.995 772.85,173.735 C 771.902,174.068 770.98,174.47 770.017,174.759 C 769.14,175.022 768.086,174.725 767.31,175.319 C 768.637,175.538 770.069,175.578 771.12,176.544 C 769.801,178.225 767.57,177.712 765.736,177.58 C 763.114,177.391 760.54,177.761 757.947,177.624 C 757.826,176.839 758.974,175.547 759.459,175.68 C 758.201,175.522 756.808,175.394 755.571,175.752 C 756.547,175.188 757.681,175.346 758.739,175.105 C 756.455,172.439 753.429,172.819 750.027,173.232 C 750.249,172.678 750.702,172.236 751.251,172.008 C 750.27,169.665 742.743,171.501 741.243,172.584 C 742.36,173.408 744.736,173.528 746.067,173.376 C 744.206,173.92 742.027,173.478 740.145,173.264 C 737.444,172.958 736.01,174.27 733.611,174.96 C 734.658,175.697 736.135,175.427 737.31,175.276 C 739.171,175.037 741.037,175.274 742.899,175.105 C 740.774,176.125 738.186,175.544 735.923,175.734 C 733.202,175.962 730.515,176.105 727.923,177.048 C 729.212,177.709 731.022,177.621 732.416,177.473 C 734.806,177.221 737.374,176.935 739.731,176.472 C 734.011,178.032 728.073,177.487 722.451,179.353 C 723.886,181.013 726.838,180.177 728.643,179.785 C 728.373,179.97 728.11,180.162 727.851,180.361 C 728.113,180.384 728.376,180.408 728.641,180.432" id="path4996"/>
    <path d="M 754.561,169.92 C 755.935,169.778 757.324,169.877 758.521,169.056 C 756.796,167.895 753.034,168.456 751.393,169.56 C 752.45,169.656 753.523,169.668 754.561,169.92" id="path5002"/>
    <path d="M 759.168,175.319 C 758.725,175.213 758.073,175.407 757.656,175.536 C 758.167,175.509 758.671,175.437 759.168,175.319" id="path5006"/>
    <path d="M 786.458,162.072 C 787.543,162.002 788.631,161.801 789.628,161.352 C 788.415,160.365 786.37,160.818 784.961,160.948 C 782.832,161.145 780.674,161.312 778.539,161.43 C 776.909,161.521 766.873,161.295 767.483,164.746 C 767.505,164.871 769.714,165.866 769.999,165.915 C 771.223,166.125 772.555,165.714 773.757,165.519 C 777.067,164.98 781.15,165.943 783.869,163.44 C 783.187,162.816 782.297,163.101 781.489,163.08 C 782.36,161.591 785.07,162.3 786.458,162.072" id="path5024"/>
    <path d="M 791.788,180.647 C 792.696,180.419 795.067,178.746 792.931,178.336 C 790.765,177.921 789.161,179.544 787.178,179.999 C 787.728,181.769 790.55,180.825 791.788,180.647" id="path5044"/>
    <path d="M 797.547,173.952 C 800.039,174.249 806.797,175.665 808.127,172.584 C 806.452,172.105 804.63,172.595 802.947,172.764 C 801.149,172.944 799.007,172.639 797.547,173.952" id="path5056"/>
    <path d="M 807.837,174.024 C 806.37,173.529 804.705,174.115 803.237,174.311 C 801.715,174.514 799.904,174.292 798.557,175.175 C 801.327,175.176 805.479,175.842 807.837,174.024" id="path5058"/>
    <path d="M 801.148,175.319 C 800.349,175.198 799.569,175.34 798.768,175.319 C 799.488,175.786 800.381,175.476 801.148,175.319" id="path5060"/>
    <path d="M 799.778,172.368 C 802.437,172.368 805.196,172.705 807.698,171.576 C 806.999,171.055 804.844,170.443 804.028,170.639 C 806.133,170.002 802.086,170.349 801.473,170.537 C 800.283,170.903 800.747,171.927 799.778,172.368" id="path5062"/>
    <path d="M 808.488,174.888 C 806.456,173.247 801.986,175.377 799.918,176.113 C 802.848,176.473 805.659,175.472 808.488,174.888" id="path5064"/>
    <path d="M 804.667,167.256 C 805.602,167.044 807.147,167.293 807.799,166.39 C 808.405,165.55 807.104,165.214 806.547,165.384 C 808.732,164.5 806.269,162.869 804.937,162.977 C 804.349,163.024 803.374,164.334 803.327,164.838 C 803.283,165.301 802.612,165.921 802.901,166.36 C 803.255,166.895 804.174,166.893 804.667,167.256" id="path5068"/>
    <path d="M 808.778,166.896 C 808.139,166.775 807.494,167.009 807.048,167.472 C 807.69,167.478 808.267,167.286 808.778,166.896" id="path5076"/>
    <path d="M 816.118,158.472 C 815.524,161.322 825.11,159.049 826.488,159.264 C 824.285,159.657 829.041,159.874 829.588,159.912 C 830.603,159.983 832.847,159.886 833.328,161.063 C 832.258,161.227 832.285,161.858 833.111,162.36 C 834.286,163.074 835.848,162.882 837.151,162.773 C 838.389,162.669 839.744,162.568 840.813,161.861 C 841.658,161.301 841.911,159.472 840.389,159.767 C 841.054,158.734 842.348,158.464 843.409,158.039 C 842.939,157.667 842.412,157.547 841.829,157.679 C 842.713,157.453 843.092,156.904 842.04,156.591 C 840.781,156.216 839.411,157.109 838.228,156.528 C 838.674,156.168 839.177,155.927 839.738,155.807 C 838.679,155.615 837.767,154.923 836.672,154.908 C 835.021,154.886 833.208,155.561 831.668,156.096 C 832.203,155.276 833.196,155.397 832.932,154.274 C 832.62,152.946 830.303,153.284 829.253,153.212 C 826.003,152.992 822.206,152.927 819.288,154.584 C 820.179,155.487 821.311,154.885 822.388,155.088 C 821.521,155.886 820.32,155.989 819.288,156.456 C 821.425,156.872 823.273,155.027 825.408,155.448 C 824.922,156.284 823.967,156.392 823.108,156.528 C 823.646,157.15 825.882,156.83 826.558,156.599 C 825.405,157.502 823.995,157.203 822.668,157.536 C 823.424,157.729 824.207,157.68 824.978,157.68 C 824.016,158.723 822.512,158.445 821.279,158.233 C 819.545,157.933 817.8,157.882 816.118,158.472" id="path5100"/>
    <path d="M 823.757,169.776 C 823.93,169.479 820.584,169.755 820.282,169.81 C 818.969,170.048 817.275,169.951 816.188,170.856 C 818.754,170.949 821.249,170.188 823.757,169.776" id="path5102"/>
    <path d="M 827.208,162.288 C 826.008,162.134 822.54,161.546 822.168,163.296 C 824.301,164.084 826.897,163.641 829.078,163.296 C 828.868,162.387 827.937,162.416 827.208,162.288" id="path5122"/>
    <path d="M 825.268,169.776 C 826.136,169.732 827.031,170.385 827.858,169.847 C 827.102,169.532 826.031,169.417 825.268,169.776" id="path5124"/>
    <path d="M 841.678,167.76 C 841.183,167.78 840.703,167.876 840.238,168.048 C 840.43,168.114 842.011,168.42 841.678,167.76" id="path5152"/>
    <path d="M 846.868,174.672 C 845.93,174.614 844.447,174.554 843.988,175.609 C 845.145,177.145 846.238,175.525 846.868,174.672" id="path5164"/>
    <path d="M 846.288,173.592 C 845.721,173.67 845.194,173.862 844.708,174.168 C 845.313,174.193 845.84,174.001 846.288,173.592" id="path5166"/>
    <path d="M 849.888,162.937 C 850.652,162.8 851.396,162.497 852.188,162.577 C 849.942,162.523 852.648,162.319 852.956,162.297 C 854.119,162.214 855.286,162.233 856.45,162.209 C 858.426,162.168 860.528,162.034 862.268,160.993 C 862.026,160.448 861.533,160.061 860.908,160.201 C 862.155,159.811 863.48,159.765 864.718,159.337 C 863.214,157.783 861.155,158.994 859.388,158.257 C 861.249,157.784 857.477,156.994 856.881,156.812 C 855.906,156.515 854.921,156.451 853.915,156.498 C 853.266,156.528 849.256,156.797 850.758,158.186 C 849.983,158.311 849.135,158.015 848.448,158.546 C 848.713,158.711 848.977,158.879 849.238,159.05 C 847.585,159.89 850.295,159.676 848.518,160.13 C 849.843,160.233 851.196,160.235 852.478,160.634 C 851.34,161.919 849.251,160.587 847.868,161.282 C 848.628,161.514 849.305,161.977 850.108,162.074 C 849.227,162.227 849.154,162.515 849.888,162.937" id="path5168"/>
    <path d="M 810.717,181.224 C 812.334,181.167 813.771,180.96 815.257,180.288 C 813.33,181.104 816.053,181.077 816.632,180.813 C 817.157,180.573 818.005,181.108 818.572,181.061 C 819.338,180.998 820.453,180.785 821.203,180.901 C 821.6,180.963 822.195,179.871 822.596,179.639 C 822.525,180.718 823.778,180.372 824.396,180.215 C 824.315,179.637 823.882,179.233 823.316,179.135 C 823.994,178.686 824.766,178.215 825.626,178.343 C 823.517,178.112 826.8,177.681 827.136,177.335 C 827.298,178.831 829.137,177.048 829.656,176.759 C 829.13,176.578 828.6,176.41 828.066,176.255 C 828.805,175.865 829.767,175.84 830.391,175.244 C 830.825,174.83 831.163,174.48 831.718,174.229 C 832.12,174.047 832.732,173.863 833.036,173.52 C 833.617,172.866 833.241,172.001 834.336,171.791 C 833.891,171.03 833.068,170.816 832.246,170.783 C 834.048,170.599 831.248,169.957 830.596,170.062 C 831.085,170.264 831.588,170.312 832.106,170.207 C 831.439,170.553 829.316,170.172 828.975,170.749 C 828.52,171.519 827.845,171.35 827.136,171.863 C 827.753,171.177 827.63,170.926 826.768,171.111 C 825.96,170.944 825.225,170.502 824.396,170.423 C 823.084,170.298 821.831,170.545 820.543,170.743 C 820.131,170.806 818.206,171.039 818.715,171.863 C 819.183,172.622 821.021,171.947 821.736,172.439 C 821.003,172.819 819.858,172.727 819.066,172.655 C 819.75,173.446 821.497,173.535 822.456,173.519 C 821.34,174.065 819.786,172.907 819.216,174.455 C 819.706,174.827 820.365,174.893 820.946,174.743 C 819.886,175.055 818.808,175.179 817.706,175.103 C 818.599,173.909 816.001,174.121 817.916,173.734 C 817.413,173.778 816.909,173.827 816.406,173.879 C 816.971,173.112 816.539,172.927 815.836,172.943 C 816.852,172.03 815.577,171.667 814.78,171.583 C 813.072,171.405 811.631,172.004 810.216,172.943 C 811.533,173.111 812.862,172.992 814.176,172.871 C 813.949,172.939 809.674,173.549 810.37,174.323 C 810.652,174.636 811.602,174.518 811.946,174.527 C 811.439,174.657 810.936,174.801 810.436,174.959 C 810.623,175.882 811.627,176.015 812.375,175.751 C 809.992,176.63 807.974,174.624 805.385,175.751 C 807.144,176.097 804.435,176.338 803.912,176.396 C 803.16,176.479 801.337,176.547 801.425,177.695 C 804.542,177.914 807.673,177.458 810.766,177.135 C 814.436,176.753 818.144,176.865 821.805,176.398 C 820.792,176.767 819.688,176.635 818.635,176.687 C 818.076,176.714 814.145,177.515 816.115,177.478 C 815.275,177.545 814.435,177.618 813.595,177.695 C 814.05,177.748 814.507,177.772 814.965,177.766 C 814.372,177.894 813.772,177.966 813.165,177.982 C 815.149,177.901 811.422,178.218 810.865,178.199 C 812.864,178.638 810.534,178.732 810.145,178.703 C 810.637,179.088 811.187,179.207 811.794,179.062 C 809.656,179.437 812.497,179.495 812.804,179.495 C 811.953,179.678 810.37,179.135 809.722,179.908 C 809.201,180.534 810.2,181.192 810.717,181.224" id="path5066"/>
    <path d="M 815.908,187.272 C 814.27,186.793 812.687,187.704 811.055,187.704 C 809.234,187.704 807.482,188.265 805.678,188.424 C 806.107,188.812 806.61,189.004 807.188,189 C 805.095,189.061 807.513,189.237 807.913,189.251 C 808.813,189.281 809.716,189.229 810.61,189.122 C 812.341,188.914 814.607,188.564 815.908,187.272" id="path5072"/>
    <path d="M 813.667,177.336 C 813.934,177.216 814.2,177.096 814.467,176.976 C 813.125,177.133 811.763,177.117 810.438,177.409 C 811.518,177.447 812.588,177.165 813.667,177.336" id="path5084"/>
    <path d="M 812.877,185.616 C 812.342,185.546 811.839,185.641 811.367,185.903 C 811.902,185.974 812.406,185.878 812.877,185.616" id="path5088"/>
    <path d="M 821.448,183.672 C 820.328,183.73 819.303,184.201 818.348,184.752 C 819.379,184.845 820.802,184.571 821.448,183.672" id="path5110"/>
    <path d="M 842.188,176.903 C 839.31,177.124 836.587,177.893 833.837,178.416 C 834.34,178.468 834.843,178.516 835.347,178.56 C 834.591,178.789 833.814,178.93 833.037,179.064 C 834.823,179.246 832.175,179.506 831.733,179.604 C 830.622,179.85 829.398,180.161 828.577,181.008 C 829.334,181.005 830.027,181.716 830.737,181.08 C 829.697,182.059 832.771,182.708 833.25,182.827 C 835.483,183.381 837.422,183.663 839.752,183.555 C 843.91,183.363 848.749,177.5 842.188,176.903" id="path5132"/>
    <path d="M 830.598,183.528 C 830.023,183.583 829.516,183.799 829.078,184.175 C 829.825,184.399 830.601,184.612 831.388,184.463 C 831.462,183.896 831.167,183.556 830.598,183.528" id="path5134"/>
    <path d="M 836.068,177.12 C 834.942,176.918 829.566,176.842 829.797,178.776 C 830.924,178.544 832.255,177.422 833.402,177.551 C 834.222,177.643 835.786,178.406 836.068,177.12" id="path5138"/>
    <path d="M 852.118,172.8 C 853.317,173.182 855.648,171.357 856.785,172.159 C 857.717,172.817 856.027,175.121 857.958,175.248 C 856.751,175.935 856.884,177.181 855.3,177.488 C 853.866,177.766 851.955,178.484 851.328,179.928 C 851.815,180.111 852.319,180.183 852.838,180.144 C 851.737,180.888 849.259,182.181 851.512,183.14 C 853.439,183.96 855.102,182.765 857.088,181.945 C 856.684,182.405 856.119,182.73 855.508,182.808 C 856.659,182.714 855.716,183.444 856.298,183.672 C 857.184,184.02 858.219,183.649 859.169,183.93 C 862.069,184.787 863.681,183.223 866.239,182.521 C 866.067,182.783 865.897,183.047 865.729,183.312 C 866.951,183.138 868.22,181.383 869.399,182.665 C 868.332,182.599 867.041,183.563 866.378,184.32 C 867.733,185.091 871.036,184.833 872.433,184.323 C 872.98,184.124 876.034,185.682 875.808,184.032 C 876.772,184.806 878.043,184.66 878.908,183.815 C 878.595,185.313 881.995,184.823 882.648,183.744 C 882.317,185.659 885.27,183.601 885.818,183.24 C 885.577,183.484 885.36,183.748 885.168,184.032 C 886.827,185.065 890.416,184.488 892.238,184.041 C 893.552,183.719 895.065,181.419 896.188,182.304 C 894.381,184.94 899.993,184.826 901.563,184.673 C 903.26,184.507 904.839,183.895 906.572,184.054 C 908.442,184.225 909.972,183.683 911.518,182.664 C 911.33,182.399 911.139,182.135 910.948,181.872 C 911.998,181.059 912.411,182.093 913.388,182.304 C 914.363,182.514 915.58,182.136 916.488,181.8 C 916.073,180.001 913.275,181.428 912.168,181.441 C 912.624,179.343 917.619,180.302 918.868,178.632 C 918.099,177.531 916.576,177.946 915.478,178.056 C 916.087,177.854 916.789,177.823 917.348,177.48 C 916.008,176.078 913.676,177.167 912.028,176.76 C 912.273,176.599 912.513,176.431 912.748,176.256 C 909.697,173.753 904.59,176.792 901.229,175.975 C 898.741,175.371 896.364,176.456 893.903,176.708 C 891.5,176.955 888.732,177.072 886.388,177.624 C 886.834,178.06 887.413,178.353 888.048,178.344 C 886.313,178.631 884.541,178.635 882.788,178.704 C 884.767,178.553 881.824,177.403 881.288,177.226 C 880.231,176.877 878.993,177.288 878.258,178.127 C 879.177,176.552 875.368,177.874 874.658,177.911 C 876.085,177.643 874.688,176.958 874.069,176.728 C 872.743,176.236 872.674,177.606 872.022,177.933 C 871.218,178.335 871.175,177.986 870.818,177.418 C 870.335,176.649 868.839,177.448 868.178,177.479 C 870.576,177.001 866.329,177.542 868.857,176.986 C 869.925,176.751 871.06,176.953 871.998,176.256 C 871.668,175.396 870.676,175.188 869.978,175.752 C 871.193,174.56 869.346,174.606 868.534,174.696 C 866.902,174.878 864.342,174.491 863.068,175.536 C 864.76,173.954 867.208,174.611 869.257,174.096 C 868.201,173.873 867.149,174.052 866.087,174.096 C 868.039,173.658 864.386,173.451 863.858,173.521 C 864.889,173.443 864.912,173.227 863.928,172.873 C 865.71,172.575 868.209,173.524 870.048,173.665 C 867.907,173.25 871.389,173.429 871.774,173.442 C 872.883,173.481 874.035,173.493 875.088,173.088 C 873.827,170.933 868.464,171.504 866.378,171.504 C 867.816,170.672 869.468,171.672 870.988,171.432 C 870.021,169.06 866.38,169.984 864.499,170.287 C 862.951,170.537 861.455,170.285 859.903,170.335 C 858.206,170.39 856.662,171.232 855.068,171.719 C 856.221,171.064 857.589,170.86 858.714,170.114 C 860.722,168.782 855.538,168.6 855,168.491 C 852.745,168.039 850.875,167.556 848.518,167.615 C 847.182,167.648 845.827,167.672 844.513,167.946 C 844.027,168.038 843.562,168.191 843.118,168.407 C 843.193,168.973 843.554,169.274 844.128,169.199 C 843.53,169.203 841.29,169.243 840.888,169.631 C 842.449,170.387 844.639,171.13 846.358,170.495 C 845.382,171.08 844.274,171.017 843.188,171.142 C 843.265,172.543 847.119,172.183 848.018,172.294 C 846.946,172.186 845.917,172.62 844.848,172.582 C 845.882,173.314 847,172.591 848.088,172.51 C 849.443,172.41 850.761,172.896 852.118,172.8" id="path5156"/>
    <path d="M 867.458,163.44 C 866.405,162.949 865.229,163.059 864.108,163.122 C 862.414,163.218 860.731,162.911 859.038,163.008 C 856.419,163.157 853.407,162.516 851.118,164.16 C 851.984,166.147 856.76,164.976 858.35,164.897 C 861.648,164.732 864.521,165.703 867.458,163.44" id="path5170"/>
    <path d="M 873.438,171.432 C 874.691,170.896 876.006,170.46 877.108,169.632 C 875.867,168 870.991,169.424 873.438,171.432" id="path5202"/>
    <path d="M 874.228,166.319 C 873.674,166.327 873.195,166.519 872.788,166.895 C 873.65,167.078 874.532,167.27 875.018,166.319 C 874.754,166.319 874.491,166.319 874.228,166.319" id="path5204"/>
    <path d="M 879.268,166.608 C 879.951,166.421 882.618,166.138 882.156,164.91 C 881.665,163.602 879.331,163.659 878.224,163.785 C 877.274,163.893 876.192,164.441 875.431,164.99 C 873.608,166.306 878.645,166.587 879.268,166.608" id="path5208"/>
    <path d="M 880.917,170.567 C 880.667,171.167 880.165,171.639 879.557,171.864 C 880.168,171.537 880.914,171.572 881.021,170.866 C 881.149,170.022 882.695,169.507 883.367,169.416 C 882.197,169.894 882.118,171.407 880.997,171.937 C 882.057,171.996 883.485,172.356 884.513,172.213 C 885.501,172.075 885.838,170.71 886.897,170.784 C 886.507,172.347 888.721,172.081 889.593,171.872 C 890.693,171.608 892.16,170.083 892.867,171.072 C 892.222,171.139 891.63,171.525 891.217,172.008 C 894.313,172.363 897.552,173.141 900.648,172.368 C 899.899,172.166 899.198,171.8 898.418,171.72 C 898.635,171.205 899.085,170.819 899.638,170.712 C 899.798,173.32 903.121,170.337 903.711,170.332 C 904.752,170.324 904.156,171.725 905.354,171.738 C 906.442,171.749 908.359,171.66 908.208,170.064 C 908.881,170.479 908.903,171.099 908.278,171.577 C 909.679,172.296 912.621,172.324 913.318,170.568 C 912.37,170.534 912.517,170.13 912.958,169.632 C 913.408,170.911 915.048,170.147 915.988,170.208 C 915.319,170.329 914.196,170.229 914.188,171.216 C 915.246,171.465 918.843,170.406 919.148,172.008 C 918.014,172.364 917.14,173.203 916.058,173.664 C 919.022,173.04 922.384,172.664 925.401,172.533 C 926.449,172.488 930.216,170.052 930.942,171.194 C 931.692,172.376 936.866,169.881 937.383,168.96 C 938.422,167.107 932.127,169.112 931.758,168.912 C 932.017,168.737 932.28,168.569 932.548,168.408 C 931.777,168.408 930.994,168.459 930.238,168.263 C 931.339,167.98 932.457,167.654 933.338,166.895 C 931.1,165.401 928.831,167.515 926.498,167.255 C 927.04,167.075 927.566,166.859 928.078,166.608 C 926.548,166.468 925.022,166.156 923.478,166.248 C 921.434,166.369 920.656,166.694 918.718,166.032 C 920.312,165.745 922.072,166.521 923.548,165.599 C 922.955,165.7 922.452,165.532 922.038,165.095 C 922.544,164.791 922.884,164.264 922.758,163.656 C 923.578,164.081 924.112,164.855 924.884,165.344 C 926.023,166.065 927.512,166.086 928.811,166.195 C 930.079,166.301 931.351,166.092 932.601,165.906 C 933.185,165.819 937.637,165.378 935.568,165.96 C 936.84,165.89 938.117,165.733 939.352,165.411 C 940.077,165.222 943.729,163.932 941.978,163.584 C 943.113,163.429 942.771,162.214 941.838,162.072 C 943.009,161.951 943.79,161.927 944.969,161.927 C 946.472,161.927 947.969,162.299 949.473,162.161 C 950.661,162.053 951.839,161.312 953.053,161.532 C 954.32,161.762 955.648,161.8 956.598,160.775 C 955.113,160.829 952.95,161.141 951.698,160.127 C 953.915,160.287 959.297,161.078 961.058,159.191 C 959.035,158.486 956.849,158.768 954.788,158.254 C 957.025,158.082 959.275,158.395 961.522,158.269 C 962.289,158.227 963.335,158.293 964.007,157.837 C 964.699,157.366 965.107,156.475 966.028,156.382 C 963.666,155.547 961.051,155.943 958.608,156.023 C 959.703,155.742 960.882,155.989 961.918,155.447 C 957.6,154.42 953.501,156.212 949.248,156.383 C 950.477,155.859 951.793,155.608 953.068,155.231 C 951.499,154.635 949.791,155.056 948.168,154.872 C 950.702,154.953 953.196,155.022 955.73,154.988 C 957.593,154.964 959.787,155.409 961.558,154.656 C 958.433,153.968 955.213,154.529 952.058,154.151 C 955.14,154.261 958.454,153.776 961.492,154.104 C 963.943,154.369 966.776,156.198 969.268,154.872 C 969.033,154.626 968.793,154.386 968.548,154.151 C 969.082,154.131 969.609,154.059 970.128,153.936 C 965.366,153.564 960.565,153.397 955.798,153.791 C 957.87,153.478 954.67,153.309 954.218,153.071 C 955.271,153.207 956.401,153.284 957.463,153.288 C 958.248,153.291 958.079,152.334 958.513,152.409 C 959.406,152.563 960.103,153.21 961.117,153.289 C 962.111,153.367 963.184,153.343 964.15,153.071 C 963.032,151.646 960.866,152.633 959.4,152.062 C 962.929,152.498 966.588,153.256 970.106,152.674 C 972.341,152.304 974.61,152.154 976.68,152.232 C 977.835,152.275 980.643,151.356 978.178,150.703 C 976.53,150.266 974.766,150.407 973.081,150.407 C 975.64,148.965 978.523,150.627 981.221,150.191 C 981.12,150.453 981.023,150.716 980.931,150.982 C 984.245,151.833 987.925,150.965 991.151,150.119 C 990.644,149.778 990.093,149.682 989.501,149.831 C 991.011,149.157 992.927,149.885 994.321,148.894 C 993.479,148.072 991.937,148.493 990.933,148.65 C 989.278,148.908 987.57,148.823 985.901,148.823 C 986.704,148.627 987.533,148.615 988.351,148.535 C 987.856,148.341 987.353,148.173 986.841,148.031 C 988.672,148.241 990.472,148.406 992.311,148.175 C 994.095,147.95 996.18,148.695 997.851,147.815 C 996.776,147.521 997.585,146.386 996.481,146.16 C 997.529,145.974 998.045,146.655 998.919,147.014 C 999.895,147.415 1001.297,147.057 1002.252,146.778 C 1005.426,145.848 1008.495,145.041 1011.769,144.497 C 1015.167,143.934 1018.509,143.083 1021.906,142.522 C 1025.037,142.005 1028.327,142.21 1031.33,141.047 C 1030.102,140.076 1027.976,140.518 1026.575,140.723 C 1024.239,141.065 1021.899,141.181 1019.546,141.345 C 1014.193,141.719 1008.905,142.749 1003.541,142.991 C 1004.043,142.887 1004.546,142.791 1005.051,142.703 C 1004.476,142.639 1003.9,142.615 1003.322,142.631 C 1009.51,142.19 1015.661,141.374 1021.832,140.759 C 1018.575,139.661 1015.165,141.544 1011.892,140.615 C 1012.469,140.615 1013.045,140.615 1013.622,140.615 C 1012.745,139.98 1011.563,139.823 1010.522,139.678 C 1011.605,139.557 1012.618,140.048 1013.636,140.318 C 1015.107,140.708 1016.626,140.403 1018.111,140.292 C 1021.468,140.041 1024.876,140.063 1028.242,139.894 C 1027.475,139.806 1026.702,139.823 1025.932,139.823 C 1032.376,139.365 1038.765,139.17 1045.152,138.023 C 1045.971,137.876 1052.076,137.098 1051.597,136.292 C 1050.856,135.043 1048.04,135.356 1046.882,135.646 C 1049.264,135.193 1046.128,135.522 1045.837,135.554 C 1044.583,135.693 1043.329,135.898 1042.062,135.863 C 1042.867,135.863 1043.634,135.604 1044.362,135.287 C 1043.511,135.147 1042.851,134.536 1041.992,134.423 C 1042.519,134.375 1043.045,134.327 1043.572,134.279 C 1042.325,133.14 1040.493,134.107 1039.056,134.139 C 1038.181,134.159 1037.447,133.464 1036.586,133.635 C 1035.702,133.81 1035.098,134.215 1034.145,134.128 C 1029.254,133.681 1023.974,134.318 1019.091,134.926 C 1022.927,133.737 1027.294,135.34 1030.971,133.487 C 1029.026,133.093 1026.984,133.047 1025.008,133.159 C 1022.347,133.31 1019.641,132.724 1017.002,133.054 C 1018.03,132.816 1017.959,132.75 1016.789,132.857 C 1015.812,132.804 1014.854,132.581 1013.877,132.523 C 1012.067,132.415 1009.996,132.62 1008.222,132.982 C 1008.981,133.111 1009.71,133.363 1010.452,133.558 C 1009.4,133.595 1008.253,133.596 1007.282,133.126 C 1007.541,132.951 1007.804,132.783 1008.072,132.622 C 1004.631,132.237 1001.154,132.728 997.782,133.414 C 999.132,133.476 1000.325,134.246 1001.672,134.351 C 1000.444,134.961 998.526,134.412 997.335,133.959 C 995.414,133.229 994.161,132.597 991.991,132.849 C 990.023,133.077 988.066,132.883 986.089,132.877 C 984.602,132.872 982.368,132.288 981.073,133.271 C 982.678,133.604 984.246,134.134 985.903,134.134 C 985.35,134.155 984.82,134.275 984.313,134.494 C 985.371,134.655 986.413,134.916 987.483,134.998 C 985.954,135.134 984.429,135.866 982.873,135.574 C 983.457,135.531 983.987,135.339 984.463,134.998 C 983.153,134.713 982.368,134.352 981.146,133.821 C 979.759,133.219 977.573,134.289 976.473,133.63 C 977.009,133.551 977.536,133.432 978.053,133.271 C 976.665,132.914 974.775,133.956 973.494,133.399 C 971.744,132.64 969.089,133.485 967.253,133.558 C 967.987,133.974 968.838,133.818 969.633,133.918 C 967.546,134.426 970.331,134.456 970.725,134.496 C 971.987,134.624 973.256,134.638 974.523,134.638 C 973.458,134.774 972.37,134.598 971.31,134.502 C 970.216,134.403 968.828,134.907 967.835,134.703 C 965.904,134.306 963.67,133.183 961.713,134.278 C 962.567,134.34 963.335,134.78 963.943,135.358 C 961.911,135.358 958.628,133.523 956.882,135.213 C 957.698,135.338 958.504,135.537 959.332,135.573 C 957.478,135.631 955.202,134.927 953.642,136.293 C 957.522,137.477 961.946,136.952 965.522,139.102 C 964.45,139.199 963.53,138.63 962.541,138.338 C 961.191,137.939 959.644,137.945 958.252,137.733 C 956.62,137.484 952.593,135.839 951.192,137.086 C 951.983,137.963 953.289,137.598 954.292,137.95 C 952.739,137.584 951.12,137.733 949.542,137.733 C 949.79,137.577 950.03,137.409 950.262,137.229 C 948.726,135.5 945.985,136.397 944.002,135.934 C 946.151,135.732 943.285,135.531 942.843,135.524 C 941.592,135.505 940.322,135.663 939.102,135.934 C 939.7,135.892 940.245,136.201 940.612,136.654 C 938.711,136.755 934.955,135.885 933.482,137.446 C 935.846,137.703 938.639,137.099 940.832,138.238 C 940.139,139.092 936.095,137.45 937.442,138.885 C 936.157,138.323 934.928,137.774 933.487,137.869 C 932.823,137.913 929.376,138.154 931.112,138.741 C 930.328,138.71 930.07,137.987 929.404,137.745 C 928.579,137.445 927.278,137.745 926.406,137.737 C 925.077,137.725 923.786,137.82 922.477,138.058 C 921.715,138.197 920.104,138.368 919.442,138.668 C 921.633,138.683 918.689,138.828 918.18,138.738 C 916.886,138.508 915.431,138.653 914.125,138.745 C 911.196,138.952 908.345,139.893 905.402,140.108 C 907.269,140.224 908.442,140.468 910.372,140.108 C 912.345,139.74 914.293,140.223 916.272,140.179 C 914.064,140.471 911.517,140.089 909.512,141.259 C 911.22,142.212 914.021,141.388 915.796,141.038 C 919.085,140.39 922.24,140.294 925.563,140.827 C 921.212,140.415 916.588,140.492 912.393,141.835 C 913.664,141.948 914.942,141.995 916.203,142.195 C 913.939,142.076 911.474,141.87 909.293,142.627 C 909.923,143.987 912.224,143.617 913.379,143.486 C 915.895,143.202 918.415,143.269 920.944,143.166 C 923.793,143.051 926.655,142.884 929.465,142.376 C 931.577,141.994 934.028,141.365 936.153,141.979 C 932.425,142.852 927.971,143.492 924.136,143.492 C 919.955,143.492 915.606,142.888 911.523,144.068 C 916.297,147.131 922.005,144.879 926.905,143.745 C 930.132,142.998 933.465,143.302 936.739,143.007 C 939.7,142.74 942.93,141.925 945.873,142.7 C 941.673,142.439 937.468,143.204 933.279,143.395 C 931.111,143.494 929.098,143.902 926.974,144.268 C 925.503,144.521 923.527,144.656 922.473,145.869 C 923.419,146.05 924.358,145.916 925.3,145.798 C 925.738,145.743 928.552,145.31 926.284,146.157 C 929.113,146.059 932.364,146.798 935.127,146.008 C 936.108,145.728 936.502,144.889 937.567,144.693 C 938.438,144.533 939.442,144.422 940.323,144.574 C 939.565,144.729 938.764,144.791 938.093,145.221 C 940.946,145.296 943.985,143.653 946.803,144.574 C 944.355,145.003 940.473,144.19 938.523,146.086 C 942.374,146.086 946.213,145.731 950.043,145.366 C 954.013,144.989 957.45,143.915 961.273,142.99 C 960.94,142.844 960.607,142.7 960.273,142.557 C 962.893,143.374 965.701,142.195 968.263,141.694 C 967.856,141.691 967.45,141.691 967.043,141.694 C 968.987,141.51 970.983,140.923 972.943,141.261 C 970.836,141.597 968.702,141.738 966.603,142.125 C 968.791,141.991 967.118,142.258 966.665,142.338 C 965.823,142.486 964.974,142.592 964.132,142.745 C 963.031,142.945 961.19,143.203 960.633,144.357 C 963.837,144.097 967.281,143.426 970.423,144.428 C 965.103,144.428 959.612,144.249 954.444,145.724 C 957.188,146.15 959.969,145.621 962.723,145.869 C 958.323,146.418 953.874,146.17 949.461,146.536 C 945.446,146.869 940.568,146.127 936.723,147.453 C 938.594,148.655 940.115,148.864 942.123,149.613 C 941.233,149.449 940.589,150.234 939.893,150.621 C 942.367,151.918 945.676,151.324 948.313,150.98 C 946.58,151.502 944.72,151.736 942.913,151.773 C 942.102,151.791 941.349,151.439 940.543,151.413 C 939.775,151.388 939.01,151.668 938.243,151.485 C 939.622,151.001 938.134,150.577 937.529,149.988 C 936.682,149.164 935.392,148.708 934.341,148.195 C 931.744,146.926 928.969,147.825 926.228,147.578 C 923.896,147.368 921.075,146.698 919.113,148.432 C 918.223,149.218 922.625,149.319 923.263,149.757 C 921.854,149.272 917.774,148.486 917.353,150.694 C 918.501,150.75 920.12,150.524 921.159,151.112 C 921.892,151.526 922.339,152.593 923.263,152.637 C 923.106,153.18 922.867,153.684 922.543,154.149 C 923.67,153.986 924.486,154.986 925.563,155.086 C 923.495,155.424 921.281,153.44 919.443,155.158 C 921.996,155.495 924.576,155.503 927.14,155.715 C 928.392,155.819 929.623,156.068 930.88,155.841 C 932.182,155.605 933.423,154.913 934.783,155.158 C 933.304,155.293 932.293,156.277 930.665,156.292 C 928.89,156.309 926.565,155.731 924.913,156.526 C 926.186,156.854 927.55,156.609 928.803,157.03 C 928.233,156.888 927.707,156.984 927.223,157.317 C 927.458,157.516 927.698,157.708 927.944,157.893 C 926.802,158.249 924.266,156.807 923.133,156.448 C 921.075,155.796 918.278,155.801 916.133,155.878 C 913.971,155.955 911.802,156.003 909.659,156.32 C 908.58,156.48 907.506,156.707 906.474,157.063 C 906.057,157.207 903.168,158.193 905.193,158.11 C 904.201,158.074 903.177,158.15 902.219,158.422 C 901.517,158.621 898.82,160.591 901.082,160.342 C 900.273,160.318 899.425,160.329 898.712,160.775 C 902.164,161.125 905.629,160.355 908.862,159.191 C 907.595,159.857 905.98,160.119 904.971,161.207 C 906.887,161.031 908.838,160.947 910.661,160.271 C 912.476,159.598 914.084,158.239 916.061,158.039 C 915.536,158.238 912.866,159.407 914.691,159.407 C 913.831,159.564 912.918,159.616 912.311,160.343 C 913.62,160.024 914.891,160.62 916.201,160.488 C 914.571,160.795 911.903,159.428 910.731,161.135 C 911.834,161.32 912.937,160.97 914.041,161.135 C 911.628,161.046 908.492,160.98 906.271,162.072 C 908.121,163.141 910.545,161.865 912.531,162.288 C 910.968,162.088 908.363,161.726 907.061,162.864 C 907.559,162.722 908.062,162.698 908.571,162.792 C 907.668,162.774 906.838,163.198 906.191,163.8 C 907.501,164.085 908.792,163.846 910.081,163.584 C 909.259,163.769 908.459,164.082 907.781,164.592 C 910.552,165.045 913.245,164.399 915.906,163.672 C 918.27,163.026 921.12,160.799 923.621,161.208 C 921.352,161.742 919.403,163.019 917.213,163.743 C 916.187,164.082 915.116,164.25 914.069,164.506 C 913.45,164.657 910.74,165.131 912.671,165.744 C 910.988,165.596 909.037,164.928 907.583,165.976 C 906.595,166.689 903.04,165.156 901.731,165.24 C 904.537,163.657 901.327,162.689 899.689,162.417 C 897.165,161.996 893.666,162.012 891.72,163.944 C 892.76,164.099 893.005,165.208 892.01,165.672 C 893.323,166.031 894.661,165.36 895.97,165.744 C 895.46,165.764 894.952,165.807 894.447,165.871 C 892.027,166.402 896.363,165.907 893.601,166.679 C 894.128,166.727 894.654,166.776 895.181,166.824 C 894.456,167.297 893.714,166.824 892.956,166.824 C 892.075,166.824 891.305,167.288 890.431,167.328 C 888.43,167.42 886.488,167.381 884.509,167.765 C 883.03,168.052 880.829,168.079 879.481,168.697 C 878.886,168.969 879.01,169.358 878.551,169.705 C 877.84,170.242 876.862,170.246 876.241,170.929 C 876.986,170.744 877.317,171.669 877.985,171.803 C 879.14,172.032 879.888,170.814 880.917,170.567" id="path5214"/>
    <path d="M 905.618,164.231 C 905.052,164.252 904.498,164.349 903.958,164.52 C 905.112,164.297 906.033,165.163 907.198,164.664 C 906.739,164.271 906.212,164.127 905.618,164.231" id="path5258"/>
    <path d="M 914.618,173.304 C 914.883,173.448 915.146,173.424 915.408,173.232 C 915.145,173.256 914.881,173.279 914.618,173.304" id="path5264"/>
    <path d="M 926.928,173.735 C 925.234,173.394 922.487,174.459 921.308,175.679 C 922.035,175.993 922.356,175.434 923.033,175.372 C 923.768,175.305 924.471,175.648 925.199,175.391 C 925.006,175.154 924.816,174.913 924.629,174.671 C 925.422,174.407 926.277,174.325 926.928,173.735" id="path5270"/>
    <path d="M 945.068,162.432 C 944.462,162.407 943.936,162.599 943.488,163.008 C 944.068,162.963 944.595,162.771 945.068,162.432" id="path5318"/>
    <path d="M 843.188,216.072 C 844,216.009 844.847,216.289 845.662,216.314 C 845.975,216.324 847.699,215.939 846.735,215.513 C 846.055,215.214 842.757,214.435 843.188,216.072" id="path5162"/>
    <path d="M 861.558,217.656 C 861.081,217.25 860.394,217.543 860.038,217.944 C 860.542,217.84 861.049,217.744 861.558,217.656" id="path5184"/>
    <path d="M 866.018,217.512 C 866.345,218.056 866.912,218.279 867.458,217.872 C 867.006,217.64 866.525,217.52 866.018,217.512" id="path5196"/>
    <path d="M 867.238,217.008 C 867.152,217.622 867.419,217.742 868.038,217.368 C 867.771,217.248 867.504,217.128 867.238,217.008" id="path5198"/>
    <path d="M 879.558,219.744 C 880.582,218.644 884.398,219.327 884.448,217.584 C 883.577,217.446 882.738,217.72 881.906,217.922 C 880.661,218.225 880.207,217.729 879.118,217.62 C 878.338,217.541 878.982,218.24 878.478,218.304 C 877.624,218.41 877.061,217.693 876.238,218.304 C 877.725,218.857 878.509,218.788 879.558,219.744" id="path5212"/>
    <path d="M 878.038,228.168 C 877.371,228.284 876.912,228.824 877.388,229.464 C 877.838,229.28 878.293,229.113 878.755,228.963 C 879.503,228.711 878.346,228.092 878.038,228.168" id="path5216"/>
    <path d="M 887.547,220.536 C 885.961,220.505 884.914,221.861 883.541,222.411 C 882.095,222.99 880.275,222.896 879.478,224.496 C 881.056,224.889 882.838,223.965 884.158,223.2 C 885.503,222.421 887.559,222.123 888.408,220.679 C 888.121,220.632 887.834,220.584 887.547,220.536" id="path5220"/>
    <path d="M 880.268,227.448 C 881.259,227.456 881.788,226.498 882.578,226.08 C 882.061,225.851 881.534,225.827 880.998,226.008 C 882.675,226.647 880.198,226.848 879.558,227.232 C 879.794,227.304 880.031,227.376 880.268,227.448" id="path5222"/>
    <path d="M 883.728,235.008 C 882.441,237.184 890.756,236.197 891.458,236.037 C 892.992,235.687 895.617,235.015 896.625,233.741 C 897.253,232.948 897.438,231.73 898.107,231.037 C 898.562,230.565 899.814,230.245 899.647,229.373 C 899.301,227.567 893.403,228.472 892.244,228.629 C 890.478,228.868 882.517,232.451 883.728,235.008" id="path5224"/>
    <path d="M 891.288,217.944 C 889.33,217.988 887.738,219.289 885.888,219.744 C 886.65,219.891 892.306,219.159 891.288,217.944" id="path5228"/>
    <path d="M 895.248,220.104 C 893.876,220.051 892.25,222.631 893.292,222.442 C 894.298,222.259 896.867,221.806 896.837,220.391 C 896.265,220.534 895.734,220.438 895.248,220.104" id="path5234"/>
    <path d="M 900.938,228.24 C 902.206,228.532 903.737,226 902.243,225.592 C 901.578,225.41 900.575,225.657 900.148,226.224 C 899.382,227.241 900.444,227.468 900.938,228.24" id="path5250"/>
    <path d="M 906.698,230.688 C 905.846,230.892 903.226,230.571 903.958,229.536 C 903.506,229.703 903.071,229.905 902.653,230.144 C 902.099,230.447 901.506,229.885 900.939,230.184 C 899.472,230.956 900.649,232.231 901.798,232.416 C 903.919,232.757 906.781,232.966 907.848,230.832 C 907.48,230.654 907.097,230.606 906.698,230.688" id="path5252"/>
    <path d="M 904.038,225.72 C 903.652,225.896 903.385,226.184 903.238,226.584 C 903.752,226.61 904.258,226.563 904.758,226.44 C 904.541,226.177 904.301,225.937 904.038,225.72" id="path5256"/>
    <path d="M 905.328,227.304 C 904.66,227.695 904.768,228.389 905.328,228.816 C 905.992,228.365 906.03,227.742 905.328,227.304" id="path5260"/>
    <path d="M 854.998,187.992 C 855.191,187.824 855.385,187.656 855.578,187.488 C 855.112,187.534 854.679,187.677 854.278,187.92 C 854.518,187.944 854.758,187.968 854.998,187.992" id="path5178"/>
    <path d="M 900.068,195.768 C 901.77,196.293 902.345,196.766 904.265,196.467 C 906.298,196.151 908.303,195.698 910.368,195.624 C 913.225,195.522 918.432,197.407 920.958,195.552 C 920.127,195.427 920.016,195.161 920.624,194.756 C 920.835,194.251 919.987,194.137 920.013,193.655 C 920.056,192.849 918.832,191.984 918.245,191.46 C 916.389,189.8 912.168,190.406 909.887,190.462 C 908.64,190.492 907.38,190.468 906.144,190.28 C 905.064,190.115 904.029,189.449 902.938,189.473 C 902.324,189.487 900.288,190.109 900.133,190.807 C 899.92,191.765 899.128,191.516 898.559,192.24 C 899.084,193.192 900.267,192.469 900.789,193.176 C 901.232,193.777 900.18,195.153 900.068,195.768" id="path5244"/>
    <path d="M 899.858,198.647 C 899.525,198.765 899.332,199.005 899.278,199.368 C 899.471,199.128 899.665,198.888 899.858,198.647" id="path5246"/>
    <path d="M 903.028,198.792 C 902.73,198.941 902.49,199.157 902.308,199.44 C 902.548,199.225 902.788,199.008 903.028,198.792" id="path5254"/>
    <path d="M 922.538,203.688 C 922.787,203.836 923.051,203.836 923.328,203.688 C 923.064,203.688 922.801,203.688 922.538,203.688" id="path5274"/>
    <path d="M 924.188,200.376 C 923.897,200.389 923.634,200.484 923.398,200.664 C 923.661,200.568 923.925,200.472 924.188,200.376" id="path5276"/>
    <path d="M 927.788,206.784 C 928.688,206.742 928.985,205.682 929.668,205.272 C 929.197,205.429 928.717,205.549 928.228,205.631 C 928.468,205.391 928.708,205.152 928.948,204.911 C 928.193,204.957 926.168,206.467 927.788,206.784" id="path5284"/>
    <path d="M 929.308,203.976 C 928.688,204.041 928.136,204.403 927.788,204.913 C 928.266,204.553 928.772,204.241 929.308,203.976" id="path5286"/>
    <path d="M 930.528,204.336 C 930.269,204.53 930.029,204.746 929.808,204.983 C 930.503,205.213 931.472,205.193 931.388,204.192 C 931.101,204.24 930.814,204.288 930.528,204.336" id="path5292"/>
    <path d="M 933.337,208.512 C 932.132,208.307 931.42,209.624 930.237,209.521 C 930.704,210.69 934.743,210.033 934.847,208.8 C 934.346,208.697 933.842,208.601 933.337,208.512" id="path5296"/>
    <path d="M 937.868,207.864 C 937.293,207.861 936.79,208.053 936.358,208.44 C 936.921,208.406 937.424,208.214 937.868,207.864" id="path5302"/>
    <path d="M 947.667,218.231 C 947.056,218.889 945.817,218.714 945.067,219.168 C 946.097,219.695 947.368,219.496 947.667,218.231" id="path5320"/>
    <path d="M 949.467,219.096 C 948.874,219.115 948.344,219.307 947.877,219.672 C 948.46,219.626 948.99,219.434 949.467,219.096" id="path5326"/>
    <path d="M 918.358,240.12 C 918.118,240.312 917.878,240.504 917.638,240.696 C 918.353,241.084 918.593,240.892 918.358,240.12" id="path5266"/>
    <path d="M 922.757,277.704 C 922.186,277.528 921.679,277.647 921.237,278.063 C 921.822,278.276 922.328,278.157 922.757,277.704" id="path5268"/>
    <path d="M 923.328,289.584 C 925.084,289.108 922.593,288.299 921.888,288.432 C 921.978,289.232 922.766,289.287 923.328,289.584" id="path5272"/>
    <path d="M 927.068,276.912 C 925.475,276.323 925.89,278.383 927.358,278.424 C 927.333,277.906 927.236,277.402 927.068,276.912" id="path5282"/>
    <path d="M 931.038,247.608 C 930.531,247.566 930.051,247.662 929.598,247.895 C 930.092,247.874 930.572,247.778 931.038,247.608" id="path5288"/>
    <path d="M 931.108,271.728 C 930.546,271.66 930.043,271.804 929.598,272.161 C 930.189,272.326 930.693,272.182 931.108,271.728" id="path5290"/>
    <path d="M 933.768,272.231 C 932.427,271.522 931.154,272.251 930.098,273.095 C 931.667,273.565 932.385,272.744 933.768,272.231" id="path5294"/>
    <path d="M 934.998,264.312 C 934.805,264.312 934.611,264.312 934.418,264.312 C 934.601,264.583 934.817,264.823 935.068,265.033 C 935.044,264.792 935.021,264.552 934.998,264.312" id="path5298"/>
    <path d="M 938.377,255.528 C 937.698,254.681 935.785,256.335 935.427,256.895 C 936.422,256.621 937.755,256.475 938.377,255.528" id="path5300"/>
    <path d="M 937.368,242.712 C 937.128,242.688 936.888,242.664 936.648,242.64 C 937.019,243.152 937.499,243.296 938.087,243.073 C 937.848,242.952 937.607,242.832 937.368,242.712" id="path5304"/>
    <path d="M 938.377,258.12 C 937.807,258.097 937.467,258.416 937.438,258.983 C 938.129,259.23 938.42,258.739 938.377,258.12" id="path5306"/>
    <path d="M 939.098,263.664 C 937.715,263.222 937.675,265.366 938.158,266.112 C 939.174,265.981 938.783,264.32 939.098,263.664" id="path5308"/>
    <path d="M 939.238,257.688 C 938.716,257.667 938.669,257.931 939.098,258.479 C 939.145,258.216 939.191,257.952 939.238,257.688" id="path5310"/>
    <path d="M 940.108,224.712 C 940.557,225.403 942.521,225.655 943.198,225.288 C 942.167,225.103 941.137,224.911 940.108,224.712" id="path5312"/>
    <path d="M 941.118,262.872 C 940.875,262.872 940.631,262.872 940.388,262.872 C 940.968,263.284 940.941,262.969 941.118,262.872" id="path5314"/>
    <path d="M 941.688,226.08 C 941.918,226.271 942.181,226.319 942.478,226.224 C 942.214,226.176 941.951,226.128 941.688,226.08" id="path5316"/>
    <path d="M 945.938,228.456 C 946.163,228.57 946.402,228.618 946.658,228.6 C 946.418,228.552 946.178,228.504 945.938,228.456" id="path5322"/>
    <path d="M 948.598,229.608 C 948.886,229.433 949.103,229.193 949.248,228.887 C 948.506,228.938 947.765,229.072 947.088,229.391 C 947.596,229.43 948.099,229.502 948.598,229.608" id="path5324"/>
    <path d="M 952.488,229.464 C 952.259,230.078 952.499,230.126 953.208,229.609 C 952.968,229.561 952.728,229.512 952.488,229.464" id="path5328"/>
    <path d="M 959.618,233.424 C 959.171,233.636 958.811,233.948 958.538,234.361 C 959.059,234.415 959.562,234.342 960.048,234.144 C 959.904,233.903 959.761,233.664 959.618,233.424" id="path5330"/>
    <path d="M 960.627,236.16 C 960.84,236.088 961.054,236.016 961.267,235.944 C 960.499,235.672 959.628,235.385 958.967,236.088 C 959.522,236.067 960.076,236.091 960.627,236.16" id="path5332"/>
    <path d="M 960.478,247.68 C 960.023,247.671 959.57,247.695 959.118,247.752 C 959.399,248.531 960.651,248.85 960.478,247.68" id="path5334"/>
    <path d="M 961.348,237.312 C 961.584,237.375 961.825,237.375 962.068,237.312 C 961.828,237.312 961.588,237.312 961.348,237.312" id="path5336"/>
    <path d="M 962.428,237.744 C 962.188,237.672 961.948,237.6 961.708,237.528 C 961.981,237.773 962.16,238.023 962.428,237.744" id="path5338"/>
    <path d="M 963.068,243.576 C 962.782,243.677 962.518,243.821 962.278,244.008 C 962.541,243.864 962.805,243.72 963.068,243.576" id="path5340"/>
    <path d="M 962.858,237.672 C 963.836,238.142 965.169,237.373 965.878,236.735 C 964.818,236.703 963.798,237.262 962.858,237.672" id="path5342"/>
    <path d="M 965.958,236.521 C 965.682,236.503 965.416,236.551 965.158,236.665 C 965.424,236.616 965.691,236.568 965.958,236.521" id="path5344"/>
    <path d="M 884.667,257.04 C 884.4,257.136 884.134,257.232 883.867,257.328 C 884.117,257.469 884.384,257.565 884.667,257.616 C 884.667,257.424 884.667,257.232 884.667,257.04" id="path5226"/>
    <path d="M 889.638,256.608 C 888.719,256.472 888.745,257.598 887.978,257.832 C 888.802,257.841 889.127,257.088 889.638,256.608" id="path5232"/>
    <path d="M 898.058,270.288 C 897.34,269.793 895.783,269.318 895.108,270.145 C 895.697,270.061 896.224,270.204 896.688,270.576 C 895.299,270.229 896.118,272.527 897.698,272.231 C 898.939,271.999 899.618,270.722 898.058,270.288" id="path5236"/>
    <path d="M 895.388,269.208 C 895.639,269.332 895.906,269.38 896.188,269.353 C 895.921,269.305 895.654,269.256 895.388,269.208" id="path5238"/>
    <path d="M 895.538,268.92 C 895.699,269.166 895.939,269.262 896.258,269.208 C 896.018,269.112 895.777,269.016 895.538,268.92" id="path5240"/>
    <path d="M 903.598,287.64 C 902.168,287.12 900.012,289.07 899.568,290.304 C 900.492,290.292 904.998,289.204 903.598,287.64" id="path5248"/>
    <path d="M 928.007,279.576 C 927.247,278.687 924.173,278.305 923.334,279.451 C 923.06,279.826 925.047,280.79 925.346,280.922 C 926.767,281.548 927.174,280.536 928.007,279.576" id="path5278"/>
    <path d="M 736.489,226.512 C 735.681,226.333 734.758,226.329 734.113,226.945 C 734.913,227.843 736.199,228.069 737.137,227.232 C 736.95,226.966 736.733,226.726 736.489,226.512" id="path4998"/>
    <path d="M 754.561,225.072 C 753.502,224.237 751.608,225.094 750.889,226.008 C 752.176,226.353 753.818,226.423 754.561,225.072" id="path5000"/>
    <path d="M 758.592,221.903 C 758.045,222.13 757.517,222.394 757.008,222.696 C 757.626,222.684 758.202,222.373 758.592,221.903" id="path5004"/>
    <path d="M 763.778,223.416 C 762.961,222.519 761.166,224.017 761.104,224.942 C 761.076,225.349 764.823,225.114 763.778,223.416" id="path5012"/>
    <path d="M 763.348,219.672 C 762.933,218.585 761.734,219.559 761.838,220.392 C 762.444,220.368 762.947,220.128 763.348,219.672" id="path5016"/>
    <path d="M 765.217,223.776 C 765.733,223.467 766.282,222.413 765.197,222.64 C 764.49,222.788 764.27,223.816 765.217,223.776" id="path5020"/>
    <path d="M 781.127,227.448 C 782.852,227.043 784.678,226.764 786.252,225.904 C 787.769,225.075 789.641,225.57 791.067,224.496 C 790.113,223.678 789.03,225.057 788.037,224.28 C 788.543,223.64 788.555,222.801 788.977,222.12 C 788.323,221.9 787.752,222.256 787.467,222.84 C 787.757,221.657 789.153,221.325 787.757,220.248 C 787.283,219.882 786.7,219.768 786.221,219.423 C 785.547,218.936 785.705,217.83 784.727,218.664 C 786.493,215.943 776.817,217.488 779.327,219.744 C 778.41,218.973 775.419,219.685 776.877,220.895 C 774.44,221.934 770.925,221.41 768.753,222.799 C 768.137,223.193 768.147,224.475 769.05,224.394 C 769.861,224.321 770.736,223.693 771.557,223.847 C 771.31,224.031 771.07,224.222 770.837,224.423 C 771.926,225.145 772.737,225.181 773.858,224.495 C 772.46,225.817 775.648,225.547 775.978,226.147 C 776.557,227.199 779.912,226.861 781.127,227.448" id="path5026"/>
    <path d="M 775.007,211.319 C 774.305,211.231 773.525,211.475 773.347,212.256 C 773.828,212.483 774.335,212.603 774.867,212.616 C 774.742,211.972 775.094,211.642 775.727,211.824 C 775.487,211.656 775.248,211.487 775.007,211.319" id="path5030"/>
    <path d="M 790.778,219.384 C 789.761,219.487 788.685,219.291 787.678,219.168 C 788.262,220.8 789.968,220.907 790.778,219.384" id="path5046"/>
    <path d="M 794.518,219.24 C 793.577,218.368 790.594,219.09 790.998,220.679 C 791.784,220.398 792.167,219.173 793.148,219.528 C 792.737,219.981 792.404,220.485 792.148,221.04 C 793.119,220.785 794.513,220.519 794.518,219.24" id="path5050"/>
    <path d="M 759.457,190.872 C 759.046,190.148 758.242,190.679 757.873,191.088 C 758.406,191.058 758.934,190.985 759.457,190.872" id="path5008"/>
    <path d="M 762.047,191.592 C 763.156,191.426 764.253,192.177 765.077,192.816 C 764.867,193.062 764.65,193.302 764.427,193.537 C 765.848,193.387 764.891,195.487 765.852,195.662 C 767.24,195.915 768.613,194.612 769.767,194.066 C 770.514,193.713 776.98,191.736 775.055,190.306 C 772.612,188.491 764.047,189.23 762.047,191.592" id="path5018"/>
    <path d="M 780.188,199.296 C 780.695,199.711 784.585,199.385 783.648,200.592 C 785.448,199.817 785.127,201.422 785.932,202.414 C 787.163,203.932 787.6,204.39 786.748,206.352 C 789.476,207.004 791.583,206.647 794.238,205.272 C 792.723,203.839 794.918,203.113 795.958,202.536 C 795.431,202.92 794.905,203.305 794.378,203.688 C 795.357,205.006 797.975,204.585 799.326,204.3 C 800.529,204.046 801.633,203.369 802.861,203.245 C 803.283,203.203 806.484,202.765 804.388,202.536 C 804.83,202.178 805.334,202.034 805.898,202.104 C 805.63,201.942 805.367,201.774 805.108,201.599 C 805.686,201.437 806.466,201.502 806.944,201.072 C 807.561,200.516 807.639,199.673 806.618,199.584 C 808.829,199.738 810.758,198.755 811.438,196.56 C 810.237,197.197 808.809,197.081 807.558,197.568 C 808.21,197.216 809.188,196.026 807.838,195.84 C 808.436,195.919 808.907,195.611 808.918,194.976 C 806.763,194.307 804.772,194.883 802.728,195.624 C 803.064,194.258 805.476,193.986 806.573,193.723 C 808.74,193.202 810.82,192.555 813.029,192.168 C 812.748,191.183 811.8,191.512 811.079,191.664 C 812.31,190.16 814.977,191.432 816.336,190.142 C 817.301,189.225 813.83,188.676 813.239,188.928 C 815.431,188.131 812.504,188.744 812.209,188.812 C 811.289,189.02 810.376,189.296 809.429,189.361 C 807.888,189.466 806.055,189.255 804.669,190.081 C 804.605,187.782 801.488,188.282 800.279,188.857 C 800.99,189.156 801.74,189.044 802.439,188.785 C 801.428,189.465 800.62,189.129 799.535,188.973 C 798.139,188.772 796.266,189.13 795.029,189.793 C 795.574,190.073 796.101,190.385 796.609,190.729 C 795.986,191.626 794.96,190.998 794.182,190.767 C 792.936,190.397 791.674,191.172 790.489,191.45 C 791.707,192.372 792.802,193.068 794.391,192.783 C 794.899,192.692 795.36,192.417 795.878,192.363 C 796.602,192.288 796.715,193.455 797.689,193.321 C 797.076,193.642 796.85,193.479 796.322,193.193 C 795.455,192.723 794.36,193.181 793.588,193.61 C 795.724,193.727 793.861,195.846 792.508,195.698 C 793.1,195.58 793.541,195.137 793.658,194.546 C 791.838,194.718 792.785,195.795 791.57,196.374 C 790.249,197.003 787.284,197.137 785.878,196.778 C 786.736,196.481 786.067,196.085 785.777,195.062 C 785.308,193.411 782.225,194.618 781.268,195.05 C 780.527,195.384 778.453,195.974 778.538,197.066 C 778.638,198.35 780.792,197.807 780.988,199.083 C 780.722,199.154 780.455,199.226 780.188,199.296" id="path5034"/>
    <path d="M 813.888,262.656 C 814.175,264.213 811.691,265.38 811.438,266.84 C 811.098,268.805 817.064,266.566 817.652,266.273 C 820.193,265.007 822.439,263.449 825.222,262.62 C 826.752,262.164 826.658,261.184 827.951,260.432 C 829.15,259.736 830.489,260.029 831.668,259.415 C 831.758,259.992 831.411,260.406 830.878,260.566 C 831.645,261.197 832.853,261.049 833.725,261.462 C 834.76,261.952 832.166,262.57 832.264,262.909 C 832.372,263.283 833.528,263.271 833.821,263.292 C 835.162,263.386 835.879,263.59 837.079,264.151 C 840.283,265.65 842.867,263.581 846.069,262.797 C 845.889,261.897 844.933,261.816 844.199,261.717 C 844.792,261.822 845.272,261.5 845.429,260.925 C 844.742,260.943 844.717,260.438 844.191,260.217 C 843.629,259.981 843.11,260.585 842.568,260.584 C 840.809,260.581 839.21,260.83 837.439,260.925 C 838.245,259.568 840.303,260.648 840.072,258.851 C 839.985,258.168 841.002,257.826 840.944,257.083 C 840.881,256.283 840.129,255.706 839.512,255.315 C 838.93,254.947 837.947,255.336 837.576,254.894 C 836.958,254.157 836.438,254.113 835.631,253.652 C 834.914,253.243 835.39,252.256 834.91,252.018 C 834.103,251.617 832.441,251.69 831.528,251.564 C 831.946,250.885 831.03,249.825 830.387,249.697 C 829.57,249.534 828.098,251.322 827.208,251.564 C 826.677,250.663 828.087,250.035 826.778,249.476 C 827.19,249.079 827.921,249.122 828.288,248.685 C 828.642,248.264 828.415,247.611 828.788,247.172 C 828.285,246.848 827.781,246.848 827.278,247.172 C 828.474,245.731 825.795,246.484 825.249,246.742 C 823.985,247.34 822.92,248.309 821.903,249.25 C 819.99,251.021 818.305,253.029 816.45,254.882 C 815.548,255.782 814.414,256.445 813.709,257.538 C 813.105,258.474 813.567,259.947 812.284,260.346 C 809.5,261.21 806.347,261.367 804.238,263.66 C 805.252,264.95 806.929,263.878 808.131,263.555 C 810.056,263.041 812.038,263.691 813.888,262.656" id="path5070"/>
    <path d="M 811.797,195.624 C 810.976,195.768 810.087,195.811 809.717,196.704 C 810.581,196.787 811.434,196.461 811.797,195.624" id="path5080"/>
    <path d="M 813.598,193.608 C 812.979,193.481 810.14,194.12 810.486,195.101 C 810.935,196.375 814.339,194.758 813.598,193.608" id="path5086"/>
    <path d="M 815.908,200.808 C 815.127,200.868 814.307,200.943 813.668,201.456 C 814.499,202.089 816.173,201.979 817.145,201.972 C 817.719,201.968 818.299,201.939 818.865,201.842 C 819.277,201.771 820.71,201.085 819.149,201.672 C 821.006,200.626 822.928,199.991 824.847,199.118 C 827.219,198.039 825.186,196.602 823.539,196.704 C 827.088,195.34 830.451,196.932 834.002,196.941 C 835.476,196.945 836.885,196.231 838.148,195.546 C 838.83,195.177 839.497,194.784 840.177,194.412 C 840.751,194.098 842.924,193.605 841.319,193.247 C 843.42,193.156 845.136,191.015 847.014,191.015 C 849.104,191.015 850.827,188.432 852.909,188.856 C 850.961,187.688 847.691,187.351 845.49,187.87 C 845.044,187.974 843.018,188.803 842.763,188.101 C 842.34,186.935 841.726,187.418 840.621,187.075 C 839.588,186.754 838.288,186.501 837.217,186.749 C 835.78,187.082 834.649,186.974 833.173,187.109 C 831.235,187.286 828.143,187.005 826.639,188.496 C 826.93,188.633 827.216,188.776 827.499,188.928 C 826.972,189.426 827.124,189.925 827.789,190.152 C 827.039,190.542 826.336,190.207 825.743,189.726 C 825.083,189.191 824.093,189.671 823.435,189.934 C 821.507,190.706 821.01,192.923 819.422,194.013 C 818.749,194.474 815.716,195.806 816.541,197.022 C 817.542,198.5 815.397,199.639 814.469,200.809 C 815.214,200.83 815.954,200.716 816.699,200.737 C 816.435,200.76 816.171,200.783 815.908,200.808" id="path5096"/>
    <path d="M 817.058,272.735 C 816.583,273.527 817.611,274.218 816.768,274.895 C 818.012,275.229 819.067,273.893 820.179,274.822 C 820.964,275.477 822.9,274.101 823.58,273.732 C 826.342,272.232 829.005,271.307 831.529,269.424 C 829.238,267.701 826.569,269.956 824.104,269.731 C 822.77,269.609 822.191,269.292 821.017,270.376 C 819.772,271.523 818.505,271.931 817.058,272.735" id="path5106"/>
    <path d="M 817.348,218.088 C 817.903,217.941 818.328,217.517 818.568,217.008 C 817.965,217.082 817.502,217.505 817.348,218.088" id="path5108"/>
    <path d="M 819.217,221.903 C 820.805,221.761 819.92,219.991 818.637,221.328 C 819.498,220.928 819.691,221.12 819.217,221.903" id="path5112"/>
    <path d="M 819.788,218.808 C 819.525,218.856 819.261,218.905 818.998,218.953 C 819.3,219.052 819.563,219.004 819.788,218.808" id="path5114"/>
    <path d="M 819.438,225.36 C 820.04,225.579 820.564,225.284 820.797,224.712 C 820.217,224.567 819.709,224.846 819.438,225.36" id="path5116"/>
    <path d="M 822.748,220.319 C 822.315,219.781 820.805,220.7 820.438,220.967 C 821.247,222.02 822.107,220.851 822.748,220.319" id="path5118"/>
    <path d="M 831.598,245.664 C 830.048,245.118 828.517,247.001 829.798,248.256 C 831.444,249.868 831.478,246.6 831.598,245.664" id="path5136"/>
    <path d="M 835.848,229.536 C 836.55,228.432 835.311,228.619 834.636,228.831 C 833.474,229.198 832.711,229.936 832.003,230.859 C 831.622,231.356 830.841,232.63 832.2,232.606 C 834.035,232.573 834.884,230.765 835.848,229.536" id="path5140"/>
    <path d="M 837.288,245.521 C 836.859,245.151 834.475,243.962 834.527,245.36 C 834.54,245.702 835.054,245.752 835.272,245.862 C 835.738,246.1 835.028,247.231 834.909,247.536 C 835.656,247.536 836.403,247.536 837.149,247.536 C 836.881,247.726 836.614,247.918 836.349,248.112 C 837.17,248.182 838.759,248.904 839.089,247.679 C 838.285,247.756 837.54,247.409 836.86,247.032 C 837.712,246.941 838.417,245.896 837.288,245.521" id="path5142"/>
    <path d="M 836.348,278.784 C 837.318,280.745 840.077,277.531 841.096,276.875 C 841.884,276.367 845.132,273.34 842.348,272.953 C 840.551,272.703 833.188,277.463 836.348,278.784" id="path5144"/>
    <path d="M 837.358,205.704 C 836.551,207.438 847.352,207.009 847.797,207.072 C 846.562,208.24 849.64,208.837 850.318,208.872 C 848.058,208.81 845.812,209.937 843.558,209.304 C 841.724,208.789 839.38,207.971 837.438,208.296 C 837.66,208.528 837.873,208.768 838.078,209.016 C 837.683,209.207 836.619,209.538 836.498,210.025 C 836.332,210.689 837.249,211.215 837.183,211.897 C 837.06,213.168 840.218,213.181 840.958,213.625 C 840.455,213.705 839.952,213.705 839.448,213.625 C 840.085,213.848 843.501,214.26 841.758,213.913 C 843.238,213.565 844.751,214.287 846.218,213.841 C 845.973,213.655 845.733,213.463 845.498,213.265 C 846.544,212.637 848.428,213.33 847.008,213.481 C 848.081,214.095 849.758,212.953 850.758,212.618 C 850.173,212.94 847.99,213.699 848.416,214.708 C 849.12,216.373 852.234,216.283 853.639,216.074 C 852.956,215.64 852.123,215.485 851.329,215.641 C 852.282,215.148 853.372,215.388 854.376,215.514 C 855.512,215.657 856.161,215.148 856.828,215.781 C 857.901,216.799 860.395,216.402 861.706,216.313 C 863.316,216.205 864.882,215.883 866.463,216.363 C 866.995,216.538 867.527,216.713 868.059,216.889 C 868.172,216.924 869.45,217.659 869.547,217.658 C 870.24,217.65 871.986,216.921 872.637,216.578 C 871.958,216.443 871.469,215.931 870.945,215.528 C 870.048,214.839 868.881,214.766 867.957,214.13 C 870.738,214.742 873.043,216.05 875.241,217.757 C 875.694,218.109 877.191,217.776 876.462,217.111 C 876.012,216.701 875.894,216.09 875.222,215.966 C 874.628,215.858 874.289,216.144 873.787,215.641 C 876.201,213.718 880.687,217.885 882.937,216.217 C 882.897,217.844 887.851,217.025 888.702,216.81 C 889.756,216.544 890.484,215.263 890.277,214.201 C 890.159,213.591 887.691,213.289 887.188,213.265 C 887.611,213.012 888.052,212.797 888.511,212.62 C 889,212.425 889.233,213.281 889.692,212.863 C 890.213,212.388 891.55,210.884 892.367,211.61 C 891.877,211.902 891.35,212.094 890.787,212.185 C 890.77,212.796 891,212.977 891.582,212.923 C 892.272,212.86 891.708,213.962 892.727,213.481 C 892.261,214.868 894.285,214.282 894.967,214.219 C 897.156,214.017 894.591,216.974 894.237,217.586 C 895.701,218.039 897.303,216.523 898.777,216.506 C 898.015,216.743 897.201,216.793 896.477,217.153 C 897.245,217.556 898.022,216.967 898.777,217.442 C 898.589,218.665 897.122,218.343 896.256,218.45 C 897.404,219.931 900.144,218.444 901.586,218.522 C 900.732,218.896 899.396,218.646 898.689,219.333 C 898.153,219.854 902.403,220.789 902.21,222.027 C 901.978,223.516 898.317,222.985 897.321,222.985 C 896.409,222.985 895.619,223.191 894.956,223.849 C 894.877,224.054 894.877,224.26 894.957,224.467 C 894.954,224.85 893.972,225.141 893.806,225.649 C 895.931,226.041 899.818,224.982 901.407,223.548 C 902.306,222.736 902.5,224.092 903.386,224.065 C 904.559,224.028 905.015,222.462 906.336,222.985 C 906.005,223.247 902.831,223.966 904.756,223.778 C 902.844,224.271 905.269,223.878 905.546,223.849 C 905.125,224.28 904.621,224.568 904.036,224.713 C 904.589,224.992 905.092,224.92 905.546,224.497 C 904.133,225.006 905.455,226.475 906.266,226.801 C 907.223,227.186 908.956,226.476 907.556,225.505 C 908.349,225.38 909.162,225.446 909.866,225.865 C 909.447,225.864 908.373,226.907 908.072,227.286 C 907.051,228.574 908.521,227.965 908.346,228.745 C 909.144,228.412 910.036,228.391 910.796,227.953 C 910.35,228.261 909.87,228.501 909.357,228.673 C 910.196,228.402 910.751,229.305 911.597,229.033 C 911.092,229.741 911.101,230.705 910.367,231.264 C 912.154,232.028 909.85,234.842 911.087,236.16 C 909.986,236.587 909.25,237.337 908.046,237.577 C 906.942,237.798 906.026,238.107 905.123,238.811 C 903.389,240.165 906.361,240.137 907.202,240.23 C 909.041,240.435 911.474,241.698 913.155,240.353 C 915.092,238.802 915.169,237.448 917.997,238.318 C 916.161,238.706 918.056,238.429 917.846,239.111 C 917.672,239.677 917.686,239.864 918.512,240.01 C 919.269,240.144 924.185,241.326 922.318,241.846 C 922.842,241.841 923.348,241.745 923.838,241.558 C 923.066,241.895 922.197,241.923 921.458,242.351 C 922.178,242.653 922.942,241.794 923.548,242.422 C 923.098,242.538 922.642,242.61 922.178,242.638 C 922.75,242.865 923.275,242.52 923.838,242.71 C 922.876,242.936 921.978,243.548 921.071,243.931 C 920.106,244.337 918.763,243.701 917.565,243.987 C 916.297,244.29 915.459,243.392 914.14,243.923 C 912.638,244.527 911.285,245.985 909.578,245.951 C 910.06,245.38 910.854,245.462 911.378,244.943 C 910.073,244.682 911.699,244.224 912.049,243.198 C 912.292,242.487 912.337,241.643 911.509,241.347 C 909.398,240.592 907.232,240.447 905.011,240.375 C 903.274,240.319 892.037,243.185 892.402,245.086 C 892.614,246.191 894.215,246.632 894.595,247.714 C 895.57,250.482 892.376,248.998 891.13,249.375 C 889.968,249.726 888.859,250.178 887.734,250.627 C 887.407,250.758 885.412,250.341 884.918,250.341 C 884.076,250.341 884.101,251.331 882.913,251.241 C 881.846,251.161 880.6,250.534 879.628,251.205 C 880.511,251.787 881.318,253.45 882.149,252.861 C 881.574,253.689 880.488,253.442 879.838,254.157 C 880.294,254.509 880.161,254.637 880.198,255.165 C 879.295,255.293 879.233,254.492 879.423,253.84 C 879.599,253.236 880.792,253.766 880.998,253.005 C 881.379,251.595 878.82,251.163 877.944,250.901 C 877.267,250.698 871.142,249.084 871.638,250.486 C 870.833,250.202 870.537,251.258 870.169,251.719 C 869.64,252.383 868.576,252.361 867.819,252.493 C 866.94,252.646 865.957,252.928 865.211,253.431 C 864.205,254.111 863.869,255.386 862.487,255.453 C 864.036,255.253 862.675,256.014 862.852,256.848 C 862.996,257.528 863.575,258.069 864.147,258.406 C 865.525,259.217 867.365,258.675 868.87,259.273 C 869.629,259.575 869.282,258.788 869.758,258.546 C 870.446,258.196 871.376,258.719 872.099,258.658 C 872.795,258.599 873.44,258.299 874.106,258.116 C 874.593,257.97 875.09,257.922 875.596,257.973 C 875.28,257.159 875.516,256.991 876.306,257.469 C 874.854,255.539 879.347,257.249 879.694,257.376 C 880.569,257.698 882.122,258.271 882.884,257.427 C 884.012,256.177 883.265,255.834 882.286,254.877 C 883.399,254.877 884.229,254.132 885.306,254.013 C 884.766,254.514 883.189,255.096 884.272,255.962 C 884.939,256.495 886.494,255.75 886.965,255.237 C 886.912,256.186 885.294,256.357 886.755,256.749 C 885.854,256.525 885.821,257.721 885.165,258.045 C 885.949,258.208 887.516,257.214 887.395,256.316 C 888.006,256.935 889.077,256.767 889.195,255.812 C 889.815,255.971 890.04,256.43 889.635,256.964 C 890.531,257.001 890.845,255.831 891.785,256.172 C 891.169,256.672 890.339,256.844 889.743,257.39 C 889.292,257.803 890.006,259.152 890.359,259.437 C 890.519,259.436 890.664,259.387 890.796,259.292 C 891.154,259.211 891.184,260.746 891.286,260.923 C 891.535,260.716 891.774,260.501 892.006,260.276 C 892.048,260.891 891.832,261.395 891.357,261.788 C 892.111,262.707 893.43,261.796 894.307,261.645 C 893.442,262.005 892.724,262.645 892.147,263.372 C 893.106,262.88 894.201,263.283 895.177,262.868 C 894.028,262.751 894.554,262.606 895.177,262.292 C 895.504,263.532 897.258,263.462 898.197,263.876 C 896.52,265.286 894.3,265.373 892.227,265.101 C 893.743,265.281 893.292,266.345 893.985,267.303 C 894.94,268.623 896.939,268.223 897.987,267.26 C 897.504,267.998 896.681,268.385 895.827,268.485 C 897.337,268.593 898.473,269.747 899.824,270.287 C 901.476,270.948 903.4,270.205 905.117,270.428 C 903.141,270.534 903.926,272.018 905.014,272.811 C 906.251,273.712 908.091,274.447 909.581,274.612 C 911.849,274.864 913.69,275.021 915.808,275.956 C 917.636,276.763 919.552,277.627 921.597,277.197 C 921.446,276.724 921.326,276.244 921.237,275.757 C 921.855,275.84 922.398,275.507 922.677,274.965 C 922.086,274.828 921.729,274.416 922.037,273.813 C 921.536,273.669 921.029,273.645 920.517,273.742 C 921.431,273.662 921.555,273.457 920.889,273.127 C 920.698,272.574 921.31,272.26 920.877,271.654 C 920.504,271.133 919.223,271.062 918.648,270.934 C 918.718,270.693 918.792,270.453 918.868,270.213 C 917.725,270.258 918.106,268.137 916.638,268.989 C 918.265,268.347 916.29,268.253 915.768,268.917 C 915.975,268.363 916.472,268.018 917.068,268.053 C 914.694,267.727 913.221,266.162 912.646,263.838 C 912.128,261.746 915.712,263.61 916.127,263.301 C 915.986,264.199 917.429,265.388 918.148,265.677 C 918.398,264.792 918.655,263.639 918.438,262.725 C 918.817,263.604 920.891,268.096 921.958,265.677 C 921.821,266.198 921.775,266.725 921.818,267.261 C 922.056,267.019 922.296,266.78 922.538,266.541 C 922.637,267.346 921.895,268.198 922.968,268.629 C 922.07,267.439 924.168,267.882 923.548,266.397 C 924.416,267.602 923.614,268.13 925.128,269.061 C 925.282,268.582 925.282,268.101 925.128,267.622 C 925.576,268.307 926.5,269.031 927.288,268.342 C 927.015,268.521 926.752,268.713 926.498,268.918 C 927.101,269.05 927.672,268.772 928.008,268.271 C 926.818,269.81 929.099,270.253 929.958,269.134 C 930.117,269.603 927.63,272.395 930.251,271.865 C 931.584,271.596 930.44,269.419 930.528,268.63 C 931.484,268.528 931.902,269.784 932.838,269.638 C 934.051,269.448 933.421,267.901 933.558,267.118 C 932.994,267.382 932.666,267.164 932.688,266.542 C 933.274,266.663 933.8,266.364 934.127,265.894 C 933.865,265.748 933.602,265.603 933.337,265.462 C 934.143,265.206 934.001,264.136 934.917,264.094 C 934.77,263.546 934.53,263.042 934.197,262.582 C 934.777,263.086 935.251,263.813 935.332,264.593 C 935.383,265.084 935.036,265.699 935.342,266.16 C 935.873,266.961 936.606,266.244 936.217,265.462 C 937.61,265.712 937.187,263.771 937.947,263.158 C 937.486,262.865 936.982,262.697 936.437,262.654 C 936.958,261.579 937.915,263.067 938.957,263.229 C 938.674,263.007 938.408,262.767 938.157,262.509 C 938.401,262.432 939.014,262.111 939.011,261.795 C 939.007,261.29 937.017,262.176 936.717,262.077 C 937.091,261.56 937.718,261.308 938.307,261.142 C 937.627,260.672 936.783,260.681 935.997,260.708 C 937.023,260.334 935.967,259.577 935.347,259.7 C 936.649,259.183 935.695,258.904 934.997,258.979 C 935.588,258.151 935.977,258.885 936.693,258.526 C 937.41,258.168 938.056,257.681 938.447,256.963 C 937.773,257.111 937.034,257.054 936.437,257.467 C 938.062,256.244 935.695,257.013 934.916,257.108 C 934.986,256.171 936.069,256.058 936.576,255.452 C 934.805,255.452 937.538,254.137 935.834,254.224 C 935.089,254.262 934.998,255.599 934.126,255.452 C 934.265,255.185 934.409,254.922 934.556,254.66 C 933.876,254.728 933.42,255.213 932.906,255.596 C 933.236,255.094 933.596,254.614 933.986,254.157 C 933.188,254.286 932.332,253.851 932.186,253.005 C 931.658,253.438 931.178,253.918 930.746,254.445 C 930.739,253.406 931.747,253.118 930.456,252.861 C 931.055,252.814 931.536,252.55 931.896,252.069 C 931.081,252.115 930.521,252.888 929.666,252.716 C 930.247,252.56 930.745,252.161 931.036,251.636 C 929.981,251.53 930.524,250.133 929.016,250.7 C 929.814,250.25 931.053,250.761 931.466,249.692 C 930.956,249.665 930.452,249.713 929.956,249.836 C 929.957,249.257 930.319,248.806 930.886,248.685 C 929.842,247.296 927.51,248.728 926.426,249.332 C 927.213,248.71 927.963,248.04 928.656,247.315 C 928.16,247.185 927.657,247.137 927.146,247.171 C 927.607,246.479 927.414,246.191 926.566,246.307 C 927.47,246.275 927.614,245.939 926.996,245.299 C 927.502,246.031 928.641,245.908 928.516,244.867 C 928.073,244.848 927.673,244.71 927.314,244.452 C 926.541,244.066 925.655,244.458 924.835,244.219 C 926.804,243.957 927.496,244.538 929.305,244.94 C 928.749,245.178 928.17,246.456 929.158,246.54 C 930.061,246.616 932.078,246.36 932.909,245.977 C 934.18,245.392 932.745,244.678 932.179,244.258 C 931.524,243.772 931.89,243.048 931.606,242.349 C 932.998,242.063 933.559,244.076 934.916,243.357 C 934.68,243.113 934.44,242.873 934.196,242.636 C 934.719,242.629 935.226,242.533 935.716,242.349 C 935.256,241.513 934.379,241.556 933.556,241.485 C 934.691,240.908 937.465,242.375 937.146,240.837 C 937.633,241.041 938.139,241.161 938.666,241.197 C 937.208,241.507 938.032,243.264 939.236,242.564 C 939.013,243.1 938.703,243.581 938.306,244.004 C 939.097,244.194 939.797,243.738 940.536,243.572 C 938.839,243.933 940.836,244.949 941.777,244.651 C 942.961,244.275 944.048,243.494 944.996,242.708 C 944.326,243.432 940.317,244.815 940.676,246.021 C 942.721,245.507 944.75,245.705 946.797,244.936 C 948.323,244.363 949.469,243.07 951.186,242.997 C 950.096,243.662 946.982,243.908 947.375,245.805 C 945.599,245.586 942.322,245.848 941.835,248.037 C 943.126,248.356 944.293,247.579 945.575,247.677 C 944.558,247.817 943.614,248.546 942.555,248.397 C 943.414,249.65 944.529,248.625 945.505,248.181 C 945.226,248.44 942.447,249.769 942.55,250.017 C 942.815,250.655 945.784,249.801 946.365,249.837 C 945.35,249.848 944.228,250.24 943.415,250.845 C 944.822,251.011 943.02,252.507 944.735,252.049 C 945.493,251.846 946.41,250.661 947.155,250.198 C 946.639,250.545 946.209,250.976 945.865,251.494 C 946.339,251.773 946.842,251.845 947.375,251.71 C 946.809,251.805 946.416,252.184 946.515,252.79 C 947.132,252.093 948.128,252.552 948.745,251.854 C 947.372,252.901 949.864,252.643 948.165,253.078 C 948.734,253.13 949.166,253.513 949.245,254.086 C 949.799,253.365 950.696,252.935 951.36,252.326 C 951.725,251.992 952.113,251.565 952.168,251.045 C 952.242,250.335 951.154,249.769 952.636,249.982 C 952.421,249.716 952.205,249.453 951.986,249.191 C 952.521,249.262 953.025,249.166 953.496,248.902 C 953.092,248.517 952.612,248.301 952.056,248.255 C 952.975,247.887 954.478,247.991 953.856,246.527 C 954.427,247.298 956.488,250.11 956.666,247.607 C 956.314,248.648 957.518,248.512 958.106,248.327 C 957.837,248.162 957.571,247.994 957.306,247.823 C 957.979,247.833 958.352,247.627 957.886,246.886 C 958.337,247.039 961.655,246.879 960.427,245.957 C 959.961,245.607 958.374,245.593 958.386,244.798 C 959.529,246.185 961.439,245.307 962.926,245.519 C 962.85,243.573 959.611,244.929 959.546,243.214 C 960.223,244.263 961.646,243.749 962.567,243.43 C 962.475,242.846 962.096,242.476 961.487,242.71 C 962.518,242.114 963.135,242.858 964.027,243.264 C 965.207,243.799 966.425,243.016 967.608,242.926 C 966.929,242.516 966.138,242.41 965.378,242.638 C 966.041,242.183 966.849,242.212 967.608,242.134 C 967.413,240.771 966.069,240.809 965.018,241.054 C 965.758,240.733 966.429,240.352 967.192,240.848 C 967.842,241.271 968.334,241.74 969.182,241.491 C 970.627,241.067 969.736,239.33 968.478,239.47 C 970.562,238.963 967.475,238.983 967.038,239.111 C 967.523,238.913 968.026,238.84 968.548,238.894 C 968.102,238.343 967.491,238.371 967.038,238.894 C 967.805,238.034 966.733,237.761 966.142,238.241 C 965.41,238.836 964.565,238.959 964.728,240.119 C 964.501,239.752 964.253,238.408 963.575,238.619 C 962.519,238.95 962.201,240.635 960.988,240.622 C 961.711,240.041 962.43,239.332 962.638,238.39 C 960.868,238.269 959.704,239.249 958.388,240.262 C 959.582,238.968 957.345,239.66 956.598,239.542 C 957.563,239.465 959.557,238.983 960.305,238.385 C 960.826,237.968 960.936,237.272 961.643,237.078 C 962.298,236.899 962.889,236.675 963.288,236.086 C 962.01,235.021 959.205,237.624 957.958,237.958 C 958.736,237.591 959.572,237.33 960.268,236.806 C 958.456,235.234 954.251,237.477 952.418,238.103 C 953.223,236.285 955.933,236.888 957.458,236.374 C 956.18,235.731 954.123,236.34 952.848,236.733 C 953.822,236.191 954.917,236.456 955.948,236.158 C 955.253,235.817 954.48,235.71 953.718,235.654 C 954.845,235.475 955.956,236.075 957.087,235.996 C 958.92,235.867 958.225,234.591 956.668,234.79 C 957.419,234.611 958.351,234.46 958.678,233.638 C 957.904,233.638 957.773,233.583 958.108,232.846 C 957.327,232.887 956.566,233.074 955.797,233.206 C 956.276,232.984 956.78,232.864 957.307,232.846 C 957.389,231.268 955.478,232.338 955.007,232.991 C 955.007,231.505 953.35,232.726 952.707,233.351 C 953.231,232.266 955.039,231.101 956.237,231.046 C 955.394,229.237 951.643,231.692 951.697,233.422 C 950.856,232.753 951.444,231.8 952.197,231.406 C 951.124,229.765 948.761,232.615 948.027,233.35 C 948.412,231.861 950.825,230.836 950.907,229.534 C 950.188,230.683 946.639,232.135 945.357,231.765 C 946.524,231.802 947.29,231.112 948.307,230.686 C 947.715,230.413 947.093,230.079 948.387,229.965 C 947.617,229.395 946.584,229.62 946.227,230.541 C 947.151,229.149 945.484,230.013 944.997,230.253 C 945.476,229.789 945.983,229.357 946.517,228.957 C 945.705,228.819 944.881,228.412 944.073,228.853 C 943.414,229.213 942.811,229.845 941.977,229.677 C 943.12,229.807 943.889,228.566 944.997,228.453 C 944.1,227.759 942.732,228.191 941.977,228.885 C 943.729,227.821 941.746,228.396 941.187,228.453 C 943.291,227.47 945.666,228.031 947.876,227.589 C 946.775,227.102 945.654,227.932 944.63,227.58 C 943.753,227.279 943.227,226.658 942.696,227.877 C 942.697,227.297 943.025,226.893 943.636,226.941 C 942.441,226.296 941.614,227.182 940.676,227.804 C 940.941,227.562 941.18,227.298 941.396,227.012 C 940.118,226.526 938.935,226.477 937.585,226.58 C 939.416,226.215 937.135,226.18 936.785,226.364 C 938.085,225.49 939.93,225.868 941.395,225.788 C 940.468,225.208 939.343,225.322 938.305,225.284 C 937.36,225.249 936.196,224.651 935.345,224.708 C 936.329,224.696 936.306,224.6 935.275,224.42 C 936.536,224.183 937.789,224.72 939.03,224.834 C 940.321,224.952 941.619,224.771 942.905,224.996 C 942.215,224.599 941.439,224.401 940.675,224.204 C 941.412,224.04 942.099,224.414 942.835,224.349 C 942.299,223.659 941.452,223.384 940.606,223.556 C 942.029,222.971 943.664,223.46 945.146,223.269 C 943.422,221.777 939.925,221.583 938.306,223.34 C 938.994,222.625 938.898,222.409 938.016,222.693 C 939.043,222.541 940.168,222.52 941.116,222.045 C 940.715,221.924 940.304,221.853 939.886,221.829 C 941.168,221.502 942.49,221.788 943.776,221.9 C 942.703,220.619 939.858,220.764 938.446,221.253 C 940.312,220.05 943.365,220.937 945.229,221.715 C 946.551,222.266 947.567,222.146 948.971,222.048 C 949.406,222.018 953.042,222.784 951.722,221.027 C 950.225,219.037 946.558,220.157 944.415,219.458 C 943.037,219.008 941.899,219.065 940.499,219.374 C 938.675,219.775 936.839,219.415 934.996,219.596 C 936.667,218.911 938.733,219.512 940.499,219.269 C 942.511,218.993 944.459,218.781 946.361,218.051 C 947.276,217.699 952.392,218.579 952.312,217.098 C 952.253,216.015 951.287,214.393 950.208,214.068 C 948.366,213.512 946.706,215.522 945.207,216.221 C 943.703,216.923 942.065,216.603 940.506,217.34 C 939.778,217.684 939.189,217.578 938.423,217.533 C 937.312,217.468 936.391,218.273 935.495,218.804 C 937.9,216.116 942.491,217.545 945.285,215.132 C 943.958,213.901 941.245,214.67 939.765,215.168 C 938.987,215.431 938.631,216.503 937.759,216.377 C 936.559,216.204 935.382,217.086 934.195,217.219 C 935.548,216.609 937.012,216.4 938.445,216.068 C 937.98,215.617 937.477,215.568 936.935,215.923 C 939.387,214.618 942.304,214.18 945.033,213.885 C 946.04,213.776 947.162,212.584 948.04,212.713 C 948.389,212.764 950.01,212.612 949.156,212.009 C 947.99,211.186 945.085,210.25 943.775,210.524 C 942.419,210.807 941.3,211.559 939.881,211.641 C 939.376,211.566 938.907,211.648 938.474,211.886 C 938.129,212.304 938.45,212.653 937.654,212.612 C 938.131,210.859 941.455,210.962 942.764,210.236 C 941.292,209.209 938.121,210.626 936.67,211.152 C 933.859,212.169 932.062,214.604 929.374,215.78 C 930.971,214.8 932.28,213.435 933.834,212.395 C 931.821,211.232 930.759,213.656 928.944,213.979 C 930.075,213.686 929.578,213.377 928.944,213.116 C 930.211,212.83 931.225,211.566 932.59,211.697 C 933.818,211.814 934.321,211.556 935.454,211.164 C 936.635,210.755 939.092,210.128 939.675,208.884 C 940.526,207.068 935.336,209.478 935.051,209.548 C 933.663,209.89 932.188,210.175 930.777,210.389 C 929.362,210.603 927.913,211.608 926.495,211.533 C 927.681,211.539 928.538,210.695 929.595,210.308 C 928.854,209.885 928.068,210.216 927.285,210.164 C 929.31,209.75 931.861,207.877 933.92,208.255 C 935.051,208.462 937.441,207.923 937.85,206.654 C 938.533,204.532 933.398,204.452 932.375,204.68 C 929.468,205.328 927.492,207.968 924.555,208.579 C 925.672,208.292 926.416,207.001 927.645,207.067 C 925.514,205.349 924.361,208.57 922.315,208.435 C 923.729,208.206 924.855,206.946 926.065,206.275 C 924.282,206.786 926.049,205.281 926.495,205.05 C 924.875,204 922.432,207.002 921.025,207.498 C 921.878,207.08 922.657,206.518 923.325,205.842 C 922.79,205.806 922.264,205.854 921.745,205.987 C 923.576,205.439 925.19,204.314 927.065,203.898 C 925.842,202.929 923.999,203.978 923.325,205.123 C 923.313,204.559 923.63,204.197 924.185,204.115 C 923.257,203.465 921.523,204.296 920.701,204.781 C 920.146,205.109 919.966,205.569 919.588,206.058 C 919.018,206.797 917.742,207.099 916.915,207.426 C 917.544,207.023 918.221,206.701 918.889,206.368 C 919.673,205.976 918.599,204.941 917.856,205.483 C 918.826,204.877 920.048,205.067 921.026,204.474 C 919.688,203.783 917.89,205.011 916.556,205.195 C 917.628,204.935 918.663,204.549 919.726,204.258 C 919.002,203.85 918.196,204.035 917.426,204.114 C 919.155,203.807 920.896,203.575 922.621,203.268 C 924.127,203 924.718,202.501 926.066,201.954 C 923.916,198.247 916.689,204.179 914.186,203.897 C 915.404,203.905 916.308,203.149 917.276,202.53 C 918.231,201.919 919.486,201.961 920.376,201.234 C 917.918,200.406 915.773,202.078 913.466,202.602 C 914.921,201.837 916.564,201.05 918.228,200.93 C 919.634,200.828 921.316,201.185 922.522,200.248 C 924.63,198.61 919.932,196.563 918.506,198.498 C 919.352,197.121 914.078,196.554 913.054,196.554 C 911.233,196.554 909.369,196.91 907.642,197.476 C 907.268,197.598 904.94,198.276 905.328,198.983 C 906.068,200.333 909.944,200.102 911.156,200.73 C 910.139,200.408 908.564,200.256 907.776,201.163 C 908.016,200.944 908.233,200.705 908.426,200.443 C 907.48,199.628 904.725,199.103 904.176,200.659 C 904.665,199.35 902.946,199.378 902.731,200.262 C 902.625,200.7 905.173,202.924 905.616,203.108 C 904.71,203.301 904.55,202.237 903.746,202.1 C 903.318,202.687 903.564,203.176 904.246,203.324 C 902.308,203.471 903.561,199.079 900.506,201.74 C 900.762,201.896 901.026,202.04 901.296,202.172 C 900.513,202.36 899.773,201.886 898.996,202.172 C 899.915,201.32 901.189,200.904 901.946,199.868 C 900.717,198.536 898.7,199.637 897.476,200.372 C 898.334,199.985 898.501,199.057 898.996,198.356 C 898.233,198.627 894.422,200.059 896.836,200.731 C 895.444,201.064 895.542,198.868 893.787,200.661 C 892.58,201.893 890.885,202.331 889.197,202.604 C 889.477,202.457 889.743,202.289 889.997,202.1 C 889.732,201.905 889.466,201.713 889.197,201.524 C 890.143,201.02 891.153,201.428 892.147,201.236 C 889.918,200.65 894.354,199.099 894.747,198.715 C 893.196,197.636 890.853,199.77 889.347,200.083 C 892.243,199.122 894.851,197.413 897.837,196.699 C 896.414,195.71 897.804,194.868 898.637,194.178 C 898.118,193.377 897.207,193.378 896.397,193.099 C 898.593,190.464 897.204,189.476 894.11,189.789 C 892.406,189.961 890.734,189.767 889.025,190.051 C 887.146,190.364 885.31,190.884 883.437,191.226 C 884.033,191.376 884.292,191.828 883.947,192.378 C 882.867,191.138 880.603,191.454 879.267,192.019 C 878.735,192.244 878.766,193.184 877.933,192.498 C 877.241,191.928 875.858,192.593 875.307,193.1 C 876.999,194.534 879.404,193.577 881.207,194.684 C 879.488,194.472 877.444,194.717 875.802,194.156 C 874.965,193.87 872.768,193.738 872.137,194.468 C 873.887,195.127 876.312,196.284 878.187,196.412 C 876.081,196.568 874.309,195.105 872.265,195.054 C 871.223,195.028 867.078,197.168 869.036,197.78 C 868.351,197.745 868.017,198.131 868.57,198.641 C 869.57,199.565 870.94,198.851 872.067,198.643 C 870.088,199.201 872.387,199.28 872.857,199.291 C 868.371,199.878 873.371,200.605 871.917,202.243 C 871.795,200.618 869.581,200.198 868.31,200.065 C 867.456,199.976 866.881,199.996 866.22,200.62 C 865.196,201.586 864.001,201.665 862.637,201.74 C 862.935,201.908 864.407,202.781 864.191,203.234 C 863.966,203.398 863.712,203.427 863.427,203.324 C 864.064,204.442 866.042,204.691 867.097,204.116 C 866.976,205.847 865.506,205.947 864.57,207.132 C 863.987,207.87 862.217,209.62 861.187,209.517 C 861.637,209.17 861.926,208.633 861.267,208.292 C 861.747,208.244 862.227,208.195 862.707,208.147 C 860.987,206.011 854.108,209.16 851.977,209.084 C 852.855,208.531 853.942,208.587 854.925,208.412 C 856.133,208.197 857.202,207.486 858.448,207.423 C 859.684,207.36 860.573,207.505 861.813,207.108 C 862.213,206.98 864.06,206.888 864.247,206.524 C 864.723,205.599 858.304,203.803 861.165,200.634 C 861.919,199.798 863.061,199.662 863.835,198.887 C 864.683,198.038 862.729,197.552 863.658,196.575 C 864.614,195.571 866.282,195.186 867.479,194.579 C 869.258,193.676 870.972,192.697 872.859,192.028 C 874.749,191.358 876.695,190.869 878.63,190.347 C 880.511,189.84 878.797,189.139 877.833,189.139 C 873.741,189.139 869.875,189.139 865.789,189.702 C 861.696,190.265 857.92,192.059 854.074,193.462 C 852.736,193.95 851.498,194.599 850.709,195.809 C 849.986,196.916 849.299,196.134 848.33,196.695 C 847.253,197.32 846.27,198.146 845.15,198.684 C 844.554,198.97 841.909,200.587 843.838,200.587 C 842.67,201.434 840.991,201.436 840.168,202.747 C 839.369,204.024 838.496,204.541 837.358,205.704" id="path5146"/>
    <path d="M 839.598,246.888 C 839.252,246.675 839.115,246.798 838.798,247.105 C 839.064,247.032 839.331,246.96 839.598,246.888" id="path5148"/>
    <path d="M 844.348,243.72 C 843.942,243.839 843.125,243.383 842.908,243.073 C 842.52,244.085 843.777,244.315 844.348,243.72" id="path5160"/>
    <path d="M 852.908,271.728 C 852.8,272.292 853.044,272.292 853.638,271.728 C 853.395,271.728 853.151,271.728 852.908,271.728" id="path5174"/>
    <path d="M 854.208,271.728 C 854.079,272.349 854.319,272.374 854.928,271.8 C 854.688,271.775 854.448,271.752 854.208,271.728" id="path5176"/>
    <path d="M 856.587,264.744 C 855.797,264.751 855.067,265.095 854.277,265.104 C 854.402,267.261 856.545,268.306 858.387,267.263 C 861.287,265.623 857.868,265.056 856.587,264.744" id="path5180"/>
    <path d="M 861.698,228.24 C 861.31,227.796 860.632,227.869 860.188,228.168 C 860.686,228.312 861.189,228.336 861.698,228.24" id="path5186"/>
    <path d="M 862.708,260.856 C 862.16,260.322 861.01,260.752 860.478,261.072 C 861.23,261.381 862.044,261.375 862.708,260.856" id="path5188"/>
    <path d="M 865.518,264.815 C 866.284,263.279 862.696,263.058 861.908,263.52 C 862.307,265.259 864.472,265.971 865.948,265.104 C 865.804,265.008 865.661,264.912 865.518,264.815" id="path5190"/>
    <path d="M 874.728,270.576 C 874.511,270.767 874.295,270.96 874.078,271.151 C 874.999,271.44 876.223,272.03 877.038,271.151 C 876.241,270.929 875.388,271.23 874.728,270.576" id="path5206"/>
    <path d="M 876.028,258.192 C 876.215,258.403 876.455,258.475 876.748,258.408 C 876.508,258.336 876.268,258.265 876.028,258.192" id="path5210"/>
    <path d="M 581.617,218.952 C 581.858,218.942 582.098,218.918 582.337,218.88 C 582.097,218.904 581.856,218.928 581.617,218.952" id="path4862"/>
    <path d="M 582.048,219.096 C 582.326,219.084 582.59,219.012 582.84,218.88 C 582.576,218.952 582.313,219.023 582.048,219.096" id="path4864"/>
    <path d="M 553.249,219.528 C 553.794,219.298 554.322,219.034 554.833,218.735 C 554.035,218.693 553.228,218.836 552.529,219.24 C 552.769,219.336 553.009,219.433 553.249,219.528" id="path4854"/>
    <path d="M 572.113,220.104 C 572.257,219.538 571.992,219.417 571.321,219.744 C 571.585,219.864 571.849,219.984 572.113,220.104" id="path4856"/>
    <path d="M 572.329,221.112 C 572.541,220.373 572.278,220.205 571.537,220.608 C 571.801,220.776 572.064,220.944 572.329,221.112" id="path4858"/>
    <path d="M 573.336,219.168 C 573.887,219.102 574.391,218.909 574.848,218.592 C 574.295,218.654 573.791,218.846 573.336,219.168" id="path4860"/>
    <path class="landxx ca" d="M 814.43303,201.80805 C 813.53315,201.8873 812.6193,202.16561 812.11478,202.51361 C 812.95377,203.0656 814.32843,203.10819 815.23938,202.71519 C 814.33137,203.22619 813.30237,203.11415 812.31637,203.21916 C 811.51338,203.30516 811.02951,203.7715 810.3005,204.0255 C 808.96352,204.49051 807.39292,204.11965 806.16798,204.93264 C 806.86399,205.45064 807.66327,205.45982 808.48623,205.33582 C 807.61421,205.51582 806.73362,207.01171 805.8656,206.84772 C 805.00662,206.68572 805.05748,205.86037 803.95053,206.04137 C 802.41654,206.29337 801.44395,206.90665 800.32197,207.85565 C 799.48098,208.56765 797.22537,209.42307 797.19738,210.57707 C 797.16338,211.96307 799.43241,211.19324 800.12038,210.98024 C 799.63838,211.24624 799.13345,211.419 798.60848,211.585 C 798.86448,211.765 799.14387,211.93297 799.41483,212.08897 C 797.57984,211.89997 795.63055,212.58966 794.17358,213.70166 C 793.41558,214.28066 791.58582,216.02928 793.56882,216.32229 C 793.32582,216.53329 793.1003,216.80783 792.86327,217.02784 C 793.40026,217.72584 794.2149,217.6326 794.97993,217.6326 C 795.47293,217.6326 795.71844,218.39874 796.18945,218.53974 C 797.32046,218.87874 798.58462,219.62294 799.51562,218.43894 C 800.01262,219.08994 800.21134,220.01343 801.12832,220.15243 C 802.42231,220.34843 803.40605,219.59967 804.65608,219.54767 C 804.69208,220.14867 804.30997,220.55539 803.74894,220.6564 C 804.13196,220.6954 804.47462,220.69539 804.85767,220.6564 C 803.68866,221.2684 802.43508,221.5435 801.22911,222.0675 C 801.67212,221.0155 803.09329,220.67201 804.05132,220.35402 C 802.71034,220.16202 799.5611,220.38713 798.81007,221.76512 C 799.09407,221.94412 799.32039,222.11209 799.61642,222.26909 C 798.52243,222.49709 797.39119,223.48761 796.29024,223.47861 C 795.87023,223.47561 793.95922,224.7205 794.37517,224.99051 C 795.73215,225.86851 798.14803,224.63817 799.31404,224.18416 C 798.81504,224.34316 798.31017,224.45834 797.80214,224.58733 C 798.15212,224.13234 799.97456,222.45202 800.52356,223.27702 C 801.29055,224.43002 798.70947,225.49344 798.10452,225.99844 C 798.60152,226.22844 799.09942,226.28223 799.61642,226.09923 C 796.94741,226.58423 794.60472,227.92123 792.15771,229.02224 C 789.74171,230.10924 787.86476,230.26535 785.30377,230.43334 C 784.58277,230.48134 781.36209,232.05638 781.27204,232.85238 C 781.16205,233.83338 782.14602,233.77087 781.27204,234.56587 C 780.79204,235.00187 780.48866,235.50577 780.96966,236.07777 C 779.11664,236.37877 778.74708,238.70818 776.83714,238.79919 C 776.59959,238.81027 775.91477,238.67807 775.32524,238.5976 C 774.42561,238.4748 773.56353,238.43352 773.51096,239.00077 C 773.45496,239.59677 774.0047,240.2035 774.61968,240.1095 C 774.08369,240.6195 773.37362,241.02102 772.70461,241.31902 C 774.38362,240.48802 772.71661,238.98883 772.20064,237.99284 C 773.06244,237.25467 773.74438,238.69176 774.61968,238.09363 C 774.54702,238.42083 774.87223,238.56563 775.32524,238.5976 C 775.92922,238.64023 776.7595,238.57281 777.13952,238.49681 C 778.17351,238.28881 778.35897,237.59033 778.853,236.78332 C 778.109,236.56732 777.32259,236.61891 776.63555,236.98491 C 777.32156,236.46891 778.15377,236.15718 778.95379,235.87618 C 777.9018,234.85718 776.22967,236.34495 775.12365,236.48094 C 775.12465,235.50994 776.21412,235.40104 776.83714,234.96904 C 776.24415,235.06904 775.73925,234.90208 775.32524,234.46508 C 776.66022,233.47508 777.63297,233.29783 778.853,232.14683 C 780.15399,230.92083 781.65573,229.28886 783.3887,228.71986 C 782.1717,228.93286 780.66851,229.16479 779.55855,229.72779 C 778.90554,230.05979 778.22886,230.58114 777.4419,230.53414 C 778.27788,229.62914 779.56268,229.43207 780.46569,228.61907 C 779.22269,228.76907 778.09213,229.23566 776.83714,228.82065 C 778.31114,228.63765 776.96422,227.54096 776.33317,227.20796 C 775.91616,226.98796 774.68048,226.727 774.21651,226.70399 C 773.43652,226.66599 772.88127,228.16768 771.89826,228.31669 C 770.79928,228.48369 769.64811,228.05751 768.57209,227.91351 C 768.38308,228.84951 769.67526,229.47797 768.47129,230.13096 C 768.9353,228.59597 767.58776,229.67035 767.76574,230.43334 C 767.88073,230.92334 768.16607,231.42324 768.06812,231.94524 C 769.07412,231.18824 771.37004,230.2831 772.50302,231.0381 C 774.02102,232.0501 772.45519,232.37317 772.9062,232.95318 C 771.10321,233.04718 768.63759,234.46725 767.86653,231.94524 C 767.03455,229.22124 764.77361,230.77662 766.35463,232.24762 C 764.84664,232.20762 763.24994,233.26117 761.81894,232.95318 C 760.56594,232.68318 759.38454,232.11225 758.08959,231.94524 C 755.0406,231.55224 752.40433,232.5618 749.42136,232.6508 C 748.07838,232.6898 745.93211,232.2548 744.68408,232.6508 C 743.72907,232.9548 743.4226,232.22684 742.46663,232.14683 C 741.66563,232.07983 739.41857,230.80938 740.55155,229.92938 C 740.62854,229.97538 731.19091,231.09844 734.50396,228.92145 C 733.99396,228.82444 733.49103,228.87624 732.99206,229.02224 C 733.23406,228.80924 733.45964,228.63647 733.69761,228.41748 C 732.12362,228.39648 732.14385,228.79993 730.67381,227.71193 C 731.59482,226.84293 731.88649,227.65782 732.28651,226.30082 C 732.51751,225.51582 733.16575,224.72916 732.18571,224.18416 C 730.5137,223.25316 727.61557,224.22913 725.93653,224.68813 C 720.94254,226.05213 715.7955,225.66206 711.3215,228.61907 C 711.8255,228.86307 712.32937,228.86307 712.8334,228.61907 C 712.22842,229.43007 712.39159,229.54424 713.23657,229.02224 C 711.26158,229.97424 714.2895,229.60721 714.74847,229.5262 C 715.47448,229.39821 716.24196,229.18041 716.96593,229.42541 C 716.71093,229.20041 716.48538,228.97485 716.26037,228.71986 C 717.59436,227.41686 718.89109,229.12806 720.2921,228.61907 C 721.01311,228.35706 722.01591,227.90531 722.81194,228.01431 C 723.20295,228.06831 724.22946,228.43552 724.52542,227.91351 C 725.50042,226.19251 729.62321,226.53644 731.48016,225.99844 C 729.97616,226.71044 729.64659,227.67872 727.8516,227.81272 C 726.3586,227.92472 724.36128,228.11524 723.11432,229.02224 C 721.83132,229.95624 720.4048,229.50138 718.98179,229.92938 C 717.58978,230.34938 716.07369,230.49372 714.64768,230.73572 C 716.35568,230.23473 714.75689,230.22914 714.54689,230.53414 C 714.1819,231.10114 714.36665,231.3269 715.15165,231.1389 C 714.14665,231.8419 712.29867,231.4216 711.72468,232.75159 C 711.28167,233.77859 712.30073,234.11863 711.22071,235.17063 C 710.44271,235.92763 711.39553,236.99808 710.81754,237.38808 C 710.15853,237.83108 710.38698,238.42457 710.11198,239.10157 C 709.29299,237.20957 708.14414,240.75619 707.59215,238.79919 C 708.19114,238.75219 708.74303,238.47384 709.10405,237.99284 C 708.67104,238.19684 707.82138,237.89322 707.49136,238.29522 C 707.15837,238.70322 706.84283,239.53933 706.28184,239.70633 C 705.22085,240.02233 706.0638,242.20192 706.28184,242.83092 C 705.31884,242.50392 705.25691,240.8007 705.27391,240.00871 C 705.02491,240.18971 704.80035,240.41047 704.56835,240.61347 C 705.13234,239.86747 704.36899,239.50715 703.76201,239.30315 C 704.38602,239.41015 704.93934,238.99601 705.07232,238.39601 C 704.41431,238.34701 703.17979,237.9687 703.8628,237.0857 C 704.68179,237.9317 706.87495,238.31129 707.69294,237.28729 C 708.65795,236.07829 707.37924,234.24701 706.08025,235.47301 C 706.41624,234.99401 706.7522,234.5409 707.08818,234.0619 C 706.13418,233.3649 704.93418,235.14301 704.16518,235.47301 C 703.48617,234.45101 705.28171,233.75459 705.3747,232.75159 C 704.3607,232.66359 703.3349,232.59018 702.3509,232.95318 C 702.65289,232.31618 702.11096,231.28565 701.34297,231.74366 C 701.75298,231.52866 702.03614,231.24552 702.25011,230.83652 C 701.5441,230.51652 699.26176,231.27628 698.52076,231.44128 C 697.04576,231.77028 695.38304,231.83321 693.98506,232.44921 C 695.89407,231.24021 692.30683,232.60079 691.66681,232.6508 C 690.72181,232.7248 690.34818,232.47917 689.55015,232.95318 C 689.18017,233.17218 687.63711,232.6258 687.13112,232.6508 C 686.09112,232.7018 685.03852,232.93396 684.00652,233.05397 C 681.78052,233.31297 679.27179,233.11938 677.05179,232.85238 C 675.03779,232.60938 672.59301,231.42307 670.60102,231.54207 C 671.42201,230.95107 672.67404,231.27573 673.52402,230.73572 C 674.74302,229.96273 674.23828,229.69599 675.84227,229.627 C 677.34826,229.562 678.32878,228.56965 679.97479,228.82065 C 681.06378,228.98665 682.16557,228.68452 682.99859,227.91351 C 684.06058,226.93151 682.05375,224.75833 680.98272,224.58733 C 679.72673,224.38633 678.55159,222.93761 677.15258,223.47861 C 676.26257,223.82361 675.43393,223.3972 674.53195,223.68019 C 674.65494,223.9192 674.81811,224.14375 674.93513,224.38575 C 673.91113,225.14375 670.2429,224.17258 669.18991,223.98257 C 667.70891,223.71357 665.94521,223.62506 664.65421,222.77306 C 662.96521,221.65806 661.90312,221.19657 659.81614,221.05957 C 658.11113,220.94857 657.0046,220.51488 655.68361,219.44688 C 654.0256,218.10588 652.55338,217.68859 650.44236,217.6326 C 648.45636,217.5796 646.68011,216.9758 644.69714,217.5318 C 642.78714,218.06781 641.14932,219.48943 639.25431,220.15243 C 637.27631,220.84343 635.06314,221.17239 633.00512,220.6564 C 634.08511,219.4204 636.08762,218.85574 637.64161,218.53974 C 637.14062,218.29974 636.58391,218.13657 636.02892,218.13656 C 636.71993,217.21457 637.73512,216.20932 638.85113,215.81832 C 637.18413,216.51332 635.15768,214.85867 633.71068,216.62466 C 635.30966,215.90167 636.06621,216.2987 637.33923,216.1207 C 635.99225,216.5877 632.95219,216.65219 631.99719,217.83418 C 632.57018,217.75718 633.12688,217.82057 633.60988,218.13656 C 632.58889,217.63656 630.53277,218.10691 629.77974,218.94291 C 631.82773,219.35191 627.63312,220.00522 627.15911,220.25322 C 627.93311,220.09122 628.76439,219.86701 629.47736,220.35402 C 627.83235,220.82002 625.52368,221.30668 624.43769,219.54767 C 623.3947,217.85867 625.07742,216.96394 625.54642,215.51594 C 625.92744,214.33994 625.48311,212.67576 624.23611,212.18976 C 623.17712,211.77676 620.70183,213.50607 622.42183,213.50007 C 622.18183,213.69207 621.95628,213.91283 621.71628,214.10483 C 622.23227,214.17483 622.73617,214.23721 623.22817,214.40721 C 621.67818,214.78921 620.06065,214.48256 618.59168,215.21356 C 617.62368,215.69556 616.66471,216.29246 615.66868,216.72546 C 614.68068,217.15346 611.30864,218.98798 610.32663,217.93498 C 611.05464,216.86998 612.54522,217.15229 613.45123,216.32229 C 610.16024,217.81729 606.4405,217.33895 603.06952,218.43894 C 600.27853,219.34894 597.67648,221.64229 594.80447,222.1683 C 595.11046,221.9683 598.22738,220.11698 596.31637,220.85798 C 596.88836,220.65498 597.39229,220.27205 597.82827,219.85005 C 597.34927,220.14305 596.8454,220.4684 596.31637,220.6564 C 597.13636,220.3384 597.66503,219.10091 598.43303,218.94291 C 598.92903,218.84091 599.47392,218.95532 599.94493,218.74132 C 601.35594,218.09932 602.96463,217.74402 604.48062,217.43101 C 606.86664,216.93801 610.50971,217.02697 612.24171,215.01197 C 610.88672,215.12397 611.60792,214.49584 612.14091,214.10483 C 611.08992,214.64883 609.74536,215.42535 608.51236,215.31435 C 607.89034,215.25835 605.23369,215.35839 606.89966,214.81039 C 605.37467,215.33838 603.77917,215.5397 602.26317,216.1207 C 601.39816,216.4517 599.66496,217.82808 599.94493,216.42308 C 598.63994,217.23608 596.89223,217.61797 595.40923,217.93498 C 594.69223,218.08898 594.0286,218.25836 593.29257,218.23736 C 593.07857,218.23086 592.59482,218.1309 592.38543,218.03577 C 591.39137,218.24826 589.75758,218.59235 591.47829,218.64053 C 590.11528,218.90453 588.95617,218.2907 587.64815,219.0437 C 586.60215,219.6467 585.83035,220.20381 584.62435,220.45481 C 582.24034,220.94981 579.83704,221.17712 577.46803,221.76512 C 576.27002,222.06112 574.98622,222.39644 573.94026,223.07543 C 573.13827,223.59543 573.30334,224.53789 572.93233,225.29289 C 572.04131,227.10889 571.43923,224.76857 571.52122,223.98257 C 570.13521,224.08657 568.07274,223.41675 566.88473,224.38575 C 568.57973,224.41375 566.5391,224.76886 567.18711,225.79685 C 566.14012,225.94785 565.39651,224.92454 564.56649,224.48654 C 563.2375,223.78654 561.55459,224.36817 560.13158,224.18416 C 559.16958,224.06016 554.28534,222.89016 554.89033,221.26116 C 554.02735,222.18315 553.06728,220.86585 552.97526,219.95084 C 552.92026,219.40884 550.86118,219.19749 550.55622,219.1445 C 548.72523,218.8275 545.87367,218.61953 544.00466,218.64053 L 458.22957,289.80059 C 460.18251,290.7084 461.15661,289.58321 462.86606,290.50614 C 465.22208,290.19814 467.38083,288.82542 469.8208,289.39741 C 469.00681,290.01741 467.66137,290.50428 467.09938,291.41328 C 466.52139,292.34628 468.11354,292.50093 468.00652,293.52994 C 467.86451,294.90194 468.55648,296.20089 469.01445,297.46088 C 469.45446,298.66988 468.36493,299.76227 467.80494,300.68626 C 469.30093,300.16426 470.84067,299.88695 472.34063,299.37595 C 473.48964,298.98395 474.89317,298.80923 475.96919,298.26722 C 477.18019,297.65722 477.73261,296.45614 479.19458,296.35215 C 480.44857,296.26315 479.96486,295.23784 480.90806,295.04184 C 481.86287,294.84343 482.67748,295.11234 484.53662,294.33628 C 485.13261,294.08749 486.23753,293.49817 486.85487,293.93311 C 488.96946,295.42296 486.36686,295.95836 486.25011,297.25929 C 486.17224,298.12656 487.87318,298.45226 487.45962,299.37595 C 487.1057,300.16641 486.63254,300.74278 487.15725,301.39181 C 487.82722,302.22057 488.55039,303.40788 487.8628,304.91958 C 487.75059,305.16625 487.27723,307.65796 487.15725,308.14496 C 486.86625,309.32696 486.76826,310.66231 486.65328,311.87432 C 486.56929,312.76332 486.50991,313.70315 485.84693,314.39415 C 485.34193,314.92015 484.60916,315.61984 485.74614,316.00684 C 484.50314,316.54984 484.49324,318.21301 484.23424,319.33302 C 485.17525,319.19102 488.6596,320.94209 488.76994,321.85285 C 488.86902,322.67069 489.57416,322.32778 489.57628,322.96158 C 489.57819,323.53473 490.82319,323.49655 491.08818,323.46554 C 490.28219,325.53054 488.05484,325.64725 486.85487,328.00124 C 486.30987,329.06924 485.851,329.85908 484.839,330.52107 C 484.16699,330.96007 483.16931,331.12277 483.22631,332.13377 C 483.58513,331.83299 484.03488,331.57111 484.43583,331.32742 C 484.07265,331.675 483.71257,332.05355 483.52869,332.33535 C 484.27069,332.21235 485.00314,332.35157 485.74614,332.23456 C 484.29714,333.01156 482.95286,332.56465 481.51282,333.64566 C 480.72083,334.23866 479.37446,335.33595 481.21044,335.45994 C 480.15145,335.72694 479.7208,334.54276 478.58982,335.05677 C 477.75383,335.43677 478.80656,336.93528 479.69854,336.26629 C 479.04653,336.68229 478.31466,337.14076 478.18664,337.97978 C 477.86565,337.13978 478.12349,336.19514 477.78347,335.35915 C 477.16445,335.83615 474.93075,337.95619 476.67474,337.77819 C 476.33976,338.11919 476.0576,338.40613 475.7676,338.78612 C 476.18836,339.14368 477.37204,339.14927 478.58982,338.98771 C 477.83895,339.17258 477.30679,339.61649 477.3803,340.80199 C 477.3613,339.07999 476.07643,339.2966 475.36443,340.49961 C 474.58144,341.82361 475.28499,344.45607 476.06998,345.64006 C 477.29499,344.80706 478.13894,343.56364 479.49696,342.91865 C 480.23096,342.56965 481.3014,342.29155 481.71441,341.50754 C 480.83839,343.04954 484.01862,340.9436 484.53662,340.49961 C 484.39861,341.35861 482.91364,341.56572 484.53662,341.91071 C 483.28362,341.41571 480.13887,343.03814 481.00885,344.63213 C 481.33636,345.23276 483.01921,344.80748 483.83107,345.03531 C 482.83678,345.01021 481.73166,345.04137 481.00885,345.84165 C 480.83884,345.47365 480.21854,344.16016 479.69854,344.12817 C 479.33456,344.10617 479.34756,344.86331 479.19458,345.03531 C 478.75157,345.5343 477.84098,348.00159 478.99299,346.84958 C 480.08098,348.11358 477.52666,348.49579 477.68268,349.6718 C 477.88268,351.1738 479.5783,349.91178 480.3033,349.6718 C 479.0683,350.3428 478.35669,351.54243 477.17871,352.29242 C 476.2987,352.85242 475.0077,353.99547 474.25571,354.71146 C 475.35237,354.76704 476.32298,354.13545 477.2795,353.50194 C 477.11146,353.70086 476.99886,353.96204 476.97712,354.30829 C 477.26989,354.24624 477.57079,354.08177 477.78347,353.90511 C 477.38684,354.36474 477.06592,354.87572 477.07792,355.31622 C 478.57979,354.33916 481.07827,353.77997 483.22631,352.89718 C 482.9976,353.25999 482.89225,353.8412 483.12551,354.1067 C 483.27024,354.27102 483.89957,354.2133 484.43583,354.1067 C 483.11042,354.60786 484.17515,355.67646 483.42789,356.72733 C 483.26288,355.89832 483.49576,354.46087 482.52075,354.50987 C 481.86674,354.54187 481.34926,354.77762 480.80727,355.11463 C 480.38926,355.37418 478.51854,356.17043 478.18664,356.22336 C 477.62236,356.52468 477.04673,356.90688 476.57395,357.33209 C 476.05395,357.79909 475.28767,357.55863 474.75967,358.03764 C 473.84266,358.87164 472.75342,360.23539 472.94539,361.5654 C 473.43765,361.46715 473.90301,361.32306 474.25571,361.06144 C 475.31382,360.27656 475.89859,359.00044 477.48109,358.74319 C 476.9381,358.90619 476.50557,359.23712 476.27157,359.75112 C 477.91958,358.68412 480.92477,361.07299 482.01679,358.84398 C 481.71979,360.59199 478.95348,359.90892 477.78347,359.85192 C 476.49851,359.78964 475.33714,360.35904 474.25571,361.06144 C 473.49655,361.55452 472.7998,362.13918 472.13905,362.67413 C 473.38704,362.47313 474.6224,362.82234 475.8684,362.57334 C 475.6344,362.68433 469.34731,363.31461 471.03032,364.38762 C 471.22694,364.51311 471.38269,364.6203 471.63508,364.69 C 468.87105,366.11978 475.73303,366.72464 472.94539,366.00031 C 474.81941,366.44031 476.77086,366.64338 478.08585,364.99237 C 477.74835,365.58787 477.87601,365.90256 478.18664,366.00031 C 477.55705,365.99762 476.95255,366.10885 476.77554,366.80665 C 477.20455,366.64866 478.60598,366.08428 478.99299,366.50427 C 479.70199,367.27527 478.07166,367.80098 477.68268,368.11697 C 480.53069,367.53497 482.4,367.31459 484.33503,364.89158 C 483.53804,367.87258 479.39613,368.23834 476.97712,368.41935 C 477.35214,368.98134 478.17761,368.49792 478.69061,368.62093 C 477.85362,368.77395 477.02615,368.76907 476.47316,369.52807 C 477.39617,369.45908 479.66758,369.01804 479.69854,370.03204 C 480.52953,369.41905 481.54633,367.88016 482.72234,368.01617 C 481.76634,368.17019 481.29527,369.12345 480.80727,369.83045 C 481.73728,369.68244 482.64767,370.44024 483.52869,369.93125 C 484.23669,369.52223 484.83658,369.03094 485.54455,368.62093 C 486.40654,368.12093 486.41666,367.61345 486.95566,366.90745 C 487.30565,366.45045 487.83396,366.25952 488.26597,365.89951 C 488.52697,366.29351 486.05474,368.82091 485.44376,369.1249 C 484.95576,369.3679 484.47263,369.49546 484.03265,369.83045 C 483.83016,369.98514 483.46989,370.64054 483.42789,370.83839 C 483.53698,370.78272 483.7732,370.59738 484.13345,370.23363 C 483.10445,371.69163 485.09155,370.842 485.54455,370.53601 C 486.35954,369.98601 487.32796,369.95747 488.26597,369.83045 C 487.81397,370.01146 484.95715,370.41657 485.24217,371.24156 C 485.45018,371.84454 483.72953,373.73517 483.12551,373.86218 C 482.4675,374.0002 483.58368,375.49365 484.03265,375.57567 C 484.74366,375.70568 485.5033,375.59131 486.14931,375.27329 C 486.78633,374.9593 487.48781,373.61051 487.35883,375.1725 C 487.47382,374.56951 487.65876,373.65102 488.36676,373.45901 C 489.05977,373.27 490.13189,372.77213 489.2739,371.94711 C 491.23091,372.5011 488.26716,373.50055 488.16518,374.16456 C 488.03216,375.03058 486.82442,375.5628 487.56042,376.48281 C 487.74491,376.71281 488.0419,376.68049 488.36676,376.5836 C 487.61315,376.87905 486.83606,377.1453 486.3509,377.89391 C 487.27491,375.72091 485.09255,376.44252 485.54455,377.59154 C 485.95156,378.62653 487.27736,379.12208 488.06438,378.29709 C 488.9534,377.36408 490.30958,377.14382 491.39056,376.48281 C 489.7552,377.71481 488.96736,380.17891 490.7858,379.40581 C 489.68269,379.82266 488.34683,380.01109 488.46756,381.22009 C 488.73755,381.1411 489.01591,381.0307 489.2739,380.91771 C 488.71991,381.16371 488.23756,381.58624 487.96359,382.12723 C 488.72559,382.12323 489.82861,381.62498 490.38263,382.73199 L 675.74147,382.53041 L 676.04385,378.70026 C 676.65303,378.04854 678.13007,379.18429 677.85813,380.51454 C 677.64929,381.53615 676.79198,382.88349 676.95099,383.94151 C 677.13,385.13252 682.1039,384.42238 681.3859,385.95738 C 683.0999,386.49337 684.32661,385.07423 685.9216,385.05024 C 687.1206,385.03324 688.21055,385.80317 689.34857,386.05817 C 688.98158,386.64216 689.31731,387.02611 689.95333,387.0661 C 689.45533,387.6091 690.00132,388.94063 690.45729,388.27562 C 690.9073,387.61861 691.16417,387.44721 691.96919,387.97324 C 692.58821,388.37725 692.98948,388.91618 693.78347,388.98118 C 693.64846,389.32316 693.68484,389.7543 694.08585,389.88832 C 696.02986,390.53933 698.13664,388.598 700.03266,388.578 C 699.26866,389.97001 701.67389,389.73972 702.3509,389.68673 C 703.97691,389.55971 704.89641,390.5811 706.48343,390.49308 C 710.17143,390.28707 711.41265,385.67538 715.15165,385.95738 C 714.35265,386.47537 713.93481,387.35445 713.13578,387.87245 C 715.32579,387.38645 715.93745,385.05953 717.67148,383.94151 C 719.04249,383.0585 719.33749,384.35323 718.17545,385.05024 C 717.41845,385.50424 716.43361,385.93151 716.15958,386.86452 C 716.90658,386.60352 717.27363,385.80098 718.07465,385.655 C 718.94467,385.49699 719.8401,385.1937 720.2921,384.34468 C 720.76711,383.45167 720.05478,381.81224 721.9048,382.12723 C 722.73579,382.26925 725.18995,383.2577 725.73494,383.84072 C 726.08493,384.21473 727.13221,383.90274 727.54922,383.84072 C 728.29322,383.73173 728.67453,384.03411 729.3635,384.1431 C 731.0685,384.41609 732.45568,383.85756 732.18571,386.36055 C 732.0577,387.55456 731.18675,389.47208 732.18571,390.49308 C 733.7727,392.11609 736.91903,390.49546 738.83807,390.79545 C 738.29507,391.76146 735.938,393.45959 736.923,394.6256 C 737.77698,395.6376 738.38597,395.93543 737.42696,397.14543 C 737.11597,397.53944 735.28266,398.72129 736.11665,399.1613 C 736.54866,399.3903 737.84036,399.56202 737.72934,400.27002 C 736.52434,400.03802 735.26263,402.28237 736.62062,401.07637 C 736.37162,401.90636 735.45948,402.16142 735.20951,402.99144 C 736.0445,403.20945 739.61174,402.67274 737.52776,404.30175 C 738.51376,405.02276 739.80391,404.58211 740.85393,405.1081 C 742.08993,405.72612 743.51565,405.91445 744.88567,405.91445 C 746.53768,405.91445 748.18256,405.61923 749.82454,406.01524 C 750.70155,406.22625 753.22477,406.06678 752.14278,407.22476 C 752.60679,406.89977 753.11066,406.67422 753.65468,406.51921 C 753.38468,406.72821 753.1063,406.89996 752.84833,407.12397 C 754.59433,406.77398 753.05576,407.67151 752.64675,407.82952 C 753.80276,407.19051 755.24625,408.29955 756.27531,407.32555 C 755.3653,408.55755 756.96625,407.67231 757.78721,407.93031 C 758.25821,408.07833 758.73011,408.40725 758.79514,408.93825 C 758.87913,409.62725 758.54907,411.42473 759.80307,410.65173 C 759.81007,411.21772 758.98692,412.65699 759.90386,412.96998 C 760.35387,413.12399 760.97624,412.814 761.21418,412.46601 C 761.48217,413.18801 760.41041,413.38412 760.60942,413.87712 C 760.86342,414.50913 761.02949,414.92424 760.10545,414.78426 C 760.47647,415.67627 761.20221,416.37989 761.21418,417.40488 C 761.48217,417.26787 761.76156,417.15272 762.02052,417.00171 C 761.34152,418.72771 759.53049,416.38747 759.09752,417.60647 C 758.7105,418.69545 759.11258,420.01886 757.58562,420.32789 C 755.61662,420.7269 754.64771,417.19213 752.64675,419.21916 C 752.86975,418.70716 753.19613,418.20326 753.45309,417.70726 C 752.9321,417.50825 752.38017,417.56586 751.94119,417.90885 C 752.45719,417.57886 752.85072,417.13253 753.15071,416.59853 C 751.46471,417.05954 752.24399,415.41724 752.04199,414.78426 C 751.76699,413.91924 750.35592,413.75155 749.62295,413.67553 C 751.77396,415.88253 750.45158,419.10205 748.31264,420.73106 C 746.16863,422.36307 744.54732,424.46325 743.57535,426.98024 C 742.35535,430.14125 740.43938,431.45366 737.22538,432.62467 C 735.29537,433.32767 734.97609,436.60421 732.99206,436.75719 C 735.88307,439.31418 731.67223,438.47026 729.96826,438.67226 C 728.85724,438.80428 727.3735,440.70168 728.85953,441.39368 C 729.84653,441.85369 731.12153,440.64965 731.78254,441.89765 C 732.88452,440.51664 734.75486,439.45103 736.51982,439.27702 C 738.07881,439.12201 739.19173,437.63498 740.45076,436.85798 C 743.78978,434.796 747.59886,437.31518 751.13485,436.75719 C 750.35985,436.6892 749.56057,436.60502 748.8166,436.35402 C 750.47961,434.85802 752.51167,434.44056 754.66261,434.13657 C 756.5076,433.87557 758.45128,434.55938 760.20624,433.73339 C 758.98224,432.77539 760.29959,431.69679 760.50862,430.6088 C 759.29362,430.78181 754.0455,432.10809 756.67848,429.50007 C 758.33149,427.86306 760.57487,426.38671 762.82687,425.77072 C 765.71588,424.98173 768.76664,424.91461 771.69668,424.35962 C 772.43468,424.2206 776.12665,424.55195 775.62762,425.66993 C 776.36962,425.37894 777.15387,425.17416 777.94586,425.06517 C 776.11488,425.05217 778.18838,423.90629 778.65141,423.55327 C 778.15142,423.70928 777.64651,423.92422 777.13952,424.05724 C 777.61152,422.70925 774.35236,423.44329 773.81334,423.55327 C 774.67835,423.29527 775.53998,422.9937 776.43396,422.84772 C 776.91496,422.76973 779.41964,422.61446 777.54269,423.45248 C 778.62968,423.60849 779.47066,422.78934 780.46569,422.54534 C 781.4147,422.31234 782.38493,422.2508 783.28791,421.83979 C 785.19991,420.97077 787.03954,420.24338 788.83154,419.11837 C 790.63453,417.98535 794.48764,416.26219 796.59262,415.79219 L 817.86,415.99378 C 819.64699,415.99378 818.76101,416.13189 819.77507,414.48188 C 820.55907,413.20388 821.98611,413.68798 823.20204,412.96998 C 823.32403,413.20898 823.38641,413.43453 823.50442,413.67553 C 823.68043,413.59954 824.49336,413.10019 824.51236,412.86918 C 824.57035,412.12818 825.03368,412.78523 825.3187,412.36522 C 825.98771,411.37822 826.21659,410.81577 827.33457,410.14776 C 829.09357,409.09677 829.72626,408.25775 830.55995,406.41841 C 831.10295,405.22047 832.17274,404.87316 833.07979,403.99937 C 834.23093,402.89039 834.86523,401.29347 836.10358,400.16923 C 837.2106,399.16423 838.28333,398.31399 839.53056,397.24622 C 839.89928,396.93052 842.42137,394.38679 842.85673,395.12957 C 843.17966,395.68056 842.84409,396.4762 843.3607,397.04464 C 844.40744,398.19646 847.68881,395.49245 849.10592,396.43988 C 851.98622,398.36553 849.23761,402.32072 848.29957,404.80572 C 847.80357,406.12071 845.4761,409.31036 847.09005,410.34935 C 847.90004,410.87135 848.49445,410.80242 847.8964,411.65966 C 847.36756,412.41769 846.68649,413.79158 847.59402,414.68346 C 848.35102,414.19946 849.33076,413.51726 849.20671,414.78426 C 850.37972,414.02426 850.03822,415.2818 850.9202,415.48981 C 851.8792,415.71681 852.74364,414.36705 853.64162,414.88505 C 854.10962,415.15505 855.04253,414.15769 855.65748,414.0787 C 856.42248,413.98072 856.51382,413.62976 855.95986,413.07077 C 856.63887,412.86776 857.22712,412.39305 857.77414,411.96204 C 857.51914,412.54103 857.13045,413.04493 856.66541,413.47394 C 857.92441,415.03393 861.38925,412.57703 862.61222,411.96204 C 863.69121,411.41905 865.81593,411.29594 866.64395,410.55094 C 868.15095,409.19595 868.32225,408.96598 868.45823,407.12397 C 868.56421,407.90496 868.86279,408.72121 869.16378,409.44221 C 869.55979,409.0412 870.11272,408.82045 870.67568,408.83745 C 869.11069,409.54246 867.8426,410.77088 866.34157,411.55887 C 865.98658,411.74488 865.1963,412.043 865.03126,412.46601 C 864.76926,413.13902 865.3774,413.57577 865.9384,413.07077 C 867.04238,412.07677 868.94255,412.45479 870.27251,412.5668 C 871.98951,412.71182 873.71693,412.4742 875.41296,412.86918 C 873.86997,413.85818 868.41878,412.80877 869.16378,415.99378 C 868.60879,415.99378 868.95571,414.18362 867.75268,415.08664 C 868.11066,414.62163 868.17227,414.11873 867.95426,413.57474 C 864.68227,415.50374 859.97972,415.3496 857.47176,418.51361 C 857.95176,418.36859 858.50369,418.25542 858.98366,418.11043 C 858.14267,418.47542 856.85368,419.94539 856.56462,418.6144 C 855.32262,419.43839 853.94706,419.99625 852.93606,421.13423 C 853.87607,420.59924 854.73686,419.39333 855.95986,419.62233 C 854.22586,420.90833 849.96987,424.46106 852.02892,427.08104 C 852.46193,426.80204 852.80463,426.46608 853.13765,426.0731 C 853.30466,426.42109 853.33462,428.16857 853.64162,428.29056 C 854.26663,428.54056 855.04653,427.99911 855.15351,428.99611 C 855.41351,427.88509 856.69838,428.01985 856.66541,426.67786 C 856.90342,426.82688 857.12793,426.94205 857.37097,427.08104 C 857.52398,426.82204 857.64016,426.54369 857.77414,426.27469 C 858.21115,428.32568 860.4963,425.46138 861.10032,424.96438 C 862.03333,424.19738 865.02822,422.8832 865.03126,421.6382 C 865.29525,421.73319 865.57457,421.84356 865.8376,421.94058 C 865.94859,421.01657 867.83186,418.58366 867.65188,420.42868 C 868.35789,420.36069 868.19884,420.12633 868.15585,419.62233 C 868.10286,419.01035 869.27873,418.74238 869.66775,418.6144 C 868.13575,419.97542 872.61259,421.89462 871.68361,418.51361 C 872.45161,420.14562 874.77995,418.44381 875.91693,418.41281 C 877.78395,418.36182 879.92038,417.31973 881.56135,416.49774 C 883.24236,415.65675 885.52842,414.68288 887.40736,414.48188 C 888.47535,414.36789 890.89149,414.44278 891.23751,413.07077 C 890.39252,412.80577 889.49822,413.38702 888.91926,412.46601 C 890.60727,412.521 890.1343,410.22475 888.91926,410.14776 C 888.29425,410.10877 887.87882,411.17092 887.20578,410.55094 C 886.69978,410.08493 886.6126,409.21985 886.29864,408.63587 C 884.80264,409.85087 882.28984,410.74172 880.35184,410.65173 C 880.58584,410.21572 880.97459,409.93837 881.46056,409.84539 C 880.46656,409.20537 879.32723,409.48521 878.23518,409.44221 C 877.35516,409.40722 876.90751,408.76988 876.11852,408.63587 C 874.56753,408.37187 873.80092,408.45595 872.48996,407.12397 C 873.30595,406.95596 874.18822,406.96063 874.8082,406.31762 C 873.6012,405.78063 872.27912,405.64229 870.97806,405.51127 C 869.25706,405.33726 870.8264,404.25415 869.36537,404.60413 C 870.85437,403.98815 869.59911,400.86046 869.97013,399.56447 C 870.04612,399.29747 871.33514,398.18036 869.97013,398.15336 C 868.95313,398.13337 868.17791,398.57215 867.14792,398.25416 C 870.30692,397.11315 871.31811,394.43453 873.90107,392.71053 C 872.60707,393.29951 873.47893,392.19198 872.48996,392.00497 C 871.92697,391.89899 871.31847,392.06158 870.77647,392.20656 C 869.52248,392.54055 868.2489,393.01646 867.14792,393.71846 C 867.08292,391.42447 866.7121,391.45366 864.62808,390.69466 C 863.87009,390.41967 863.35044,390.08771 862.51142,390.1907 C 861.71743,390.28768 861.13293,390.79107 860.29397,390.49308 C 861.56597,390.33106 862.49675,389.44312 863.82174,389.48514 C 864.79074,389.51614 865.99955,389.08774 866.84554,389.18276 C 868.51754,389.37077 869.81586,391.07571 871.58282,390.1907 C 873.97483,388.99469 876.0106,387.72953 878.53756,386.86452 C 879.25856,386.61752 880.15564,385.58188 879.94866,384.74786 C 879.73266,384.33785 879.3987,384.0125 878.94073,383.94151 C 880.07974,383.71851 877.74569,383.32453 879.4447,383.43755 C 879.8777,383.51654 880.34097,383.58791 880.75501,383.73993 C 881.67602,381.63691 876.81578,380.20358 875.51376,380.01057 C 873.22277,379.67059 870.81502,379.96274 868.55902,380.41375 C 866.12102,380.90075 863.9383,381.81121 861.60428,382.6312 C 859.32329,383.43419 857.01034,383.49147 854.75034,384.44548 C 849.91333,386.48747 844.22691,389.14451 840.3369,392.71053 C 838.27791,394.59854 835.81894,396.41778 833.48296,397.95178 C 831.14198,399.48877 828.42665,400.13536 825.82267,401.07637 C 827.31367,400.16736 828.8246,399.35595 830.25757,398.35495 C 831.46158,397.51296 832.77473,395.65656 834.08772,395.12957 C 836.97973,393.96956 839.34626,392.14115 841.14325,389.48514 C 840.81426,389.41464 840.64583,389.20857 840.4377,388.98118 C 840.70086,389.13522 841.02684,389.36416 841.14325,389.38435 C 841.95824,389.52736 843.17285,388.37925 843.76387,387.97324 C 845.78987,386.58023 847.10531,384.46654 849.4083,383.43755 C 850.52231,382.93955 853.50667,381.89875 853.64162,380.41375 C 853.87962,380.87575 853.77124,381.31727 853.33924,381.62327 C 853.79924,381.32127 854.30411,381.1571 854.85114,381.22009 C 853.88313,379.85608 858.03662,379.5926 858.98366,379.50661 C 860.46866,379.37259 861.65518,378.98813 862.61222,377.79312 C 863.89821,376.18311 867.64575,371.37867 870.17171,372.65266 C 869.32973,370.95866 871.9369,372.17311 872.48996,371.94711 C 874.18696,371.2561 875.82036,371.30578 877.63042,371.14077 C 880.41441,370.88677 883.86999,372.08435 886.60102,371.34235 C 888.25603,370.89435 889.65046,372.29352 891.23751,371.74552 C 892.73151,371.23053 894.0714,371.14757 895.67241,371.24156 C 897.21641,371.33155 898.76188,371.6645 900.3089,371.74552 C 900.91889,371.77652 900.8986,372.55587 901.41763,372.55187 C 901.84264,372.47688 902.22714,372.30971 902.62715,372.1487 C 902.19614,372.43669 901.79881,372.66328 901.31684,372.85425 C 903.78284,372.60425 905.74458,371.86212 908.27157,371.94711 C 909.25158,371.97911 910.11482,371.61354 911.09379,371.74552 C 911.85678,371.84851 912.56527,371.93973 913.31124,371.64473 C 914.22025,371.28575 914.57572,370.24004 915.52869,370.03204 C 916.01969,369.92505 916.6976,369.97908 917.04059,369.52807 C 918.1736,368.04107 920.19034,367.67328 921.57629,366.50427 C 921.44627,367.83426 922.03181,367.01648 922.28184,366.40348 C 922.74784,365.26248 922.13783,365.57797 923.28977,365.19396 C 923.97778,364.96496 924.77145,364.43224 925.40643,364.08524 C 925.89843,363.81624 925.11547,363.50188 925.40643,363.27889 C 926.48242,362.45589 928.11137,363.27452 929.33737,362.97651 C 930.74138,362.63651 931.95612,361.78061 933.3691,361.46461 C 934.81211,361.14261 936.06165,362.11919 937.50163,361.6662 C 939.02462,361.1882 940.19414,359.99495 941.63415,359.34795 C 943.14715,358.66695 944.63724,358.13949 945.96826,357.1305 C 946.51125,356.7185 947.0756,356.20056 947.78254,356.12257 C 947.50454,355.58956 947.56009,355.13845 947.98413,354.71146 C 947.19213,354.39046 946.49971,353.93294 945.76667,353.50194 C 946.74068,353.89794 947.82853,353.67969 948.79047,354.1067 C 949.15446,352.3997 945.10502,352.44662 944.15398,352.19163 C 945.24697,352.24663 947.43078,352.90266 947.17778,351.68766 C 947.87479,351.83766 948.81402,352.56828 948.99206,351.38528 C 948.25806,351.45228 947.41456,351.36215 947.27857,350.47814 C 949.55057,350.59814 949.97158,349.36727 947.78254,348.46228 C 948.90855,348.42328 950.19939,348.51655 950.80634,347.35355 C 951.20535,346.58855 950.90845,345.28507 949.79841,345.64006 C 950.24741,344.65007 948.54976,344.03158 950.10079,343.92658 C 948.98677,344.07158 947.36624,343.83724 946.47223,343.12023 C 947.07721,343.34123 947.54757,342.99268 947.78254,342.41468 C 946.87253,343.27668 945.64592,342.32665 944.65795,342.91865 C 943.75594,343.45865 943.13354,344.71314 941.93653,344.63213 C 942.52052,343.60313 941.55277,343.80479 940.8278,343.82579 C 941.99881,343.07379 943.37215,342.7395 944.55716,342.01151 C 943.66314,341.36451 944.20743,339.5783 943.44843,339.18929 C 942.82441,338.86929 941.99381,339.10172 941.33177,338.98771 C 940.36277,338.82071 937.54285,337.89946 937.40083,339.59247 C 938.39783,339.51647 939.3509,339.33127 940.32384,339.69326 C 939