package Template::Plugin::GantryAuthCookie;
use strict;

use base 'Template::Plugin';

use Gantry::Plugins::AuthCookie (); # no imports, we don't want auth_secret
use Gantry::Conf;

sub new {
    my $class         = shift;
                        shift; # discard context;
    my $arg_hash      = shift;

    my $cookie        = delete $arg_hash->{ cookie };
    my $self          = $arg_hash;

    bless $self, $class;

    # If the cookie has anything in it, get user name, but not password.
    if ( $cookie ) {
        ( $self->{ user_name }, undef ) =
                Gantry::Plugins::AuthCookie::decrypt_cookie( $self, $cookie );
    }

    return $self;
}

sub user_name {
    my $self = shift;

    return $self->{ user_name };
}

sub auth_secret {
    my $self = shift;

    # use value passed to constructor first
    return $self->{ auth_secret } if defined $self->{ auth_secret };

    # use value in conf second
    return $self->_conf()->{ auth_secret };
}

sub _conf {
    my $self = shift;

    return $self->{ conf } if defined $self->{ conf };

    my $conf          = Gantry::Conf->retrieve( {
        instance    => $self->{ conf_instance },
        config_file => $self->{ conf_file     },
    } );

    $self->{ conf } = $conf;

    return $conf;
}

1;

__END__

=head1 NAME

Template::Plugin::GantryAuthCookie - decrypter for AuthCookie plugin

=head1 SYNOPSIS

In a Template Toolkit:

    [% USE decrypter = GantryAuthCookie( {
            cookie        = cookies.cookie_name,
            conf_instance = 'gantry_conf_instance',
            conf_file     = '/etc/strange.gantry.conf',
       } ) %]

    [% user_name = decrypter.user_name %]

conf_file is optional and defaults to /etc/gantry.conf

Alternatively, you could omit conf_instance and conf and directly supply
the auth_secret key:

    [% USE decrypter = GantryAuthCookie( {
            cookie        = cookies.cookie_name,
            auth_secret   = env.auth_secret
       } ) %]

In httpd.conf (or something it includes):

    PerlModule Apache::Template
    PerlSetEnv auth_secret '$ecr3t'
    TT2Params  cookies env

    <Files *.something>
        SetHandler  perl-script
        PerlHandler Apache::Template
    </Files>

=head1 DESCRIPTION

You don't need this module if your page is generated by a Gantry controller.
In that case, you can use Gantry::Plugins::AuthCookie and call its
validate_user function directly.  Use this when you need to display user
information on a page which is generated by Apache::Template.

This module is a Template Toolkit plugin for decrypting
Gantry::Plugins::AuthCookie cookies.  In the USE statement in
your template, pass at least the contents of the cookie and either the
name of your Gantry::Conf instance or your auth_secret.  If your
master Gantry::Conf file is not /etc/gantry.conf, it as well.

You must define auth_secret in Gantry::Conf, or pass it directly.
If you pass it directly, you probably want to use PerlSetEnv to keep
it out of the template.  That's what I did in the SYNOPSIS above.

=head1 METHODS

=over 4

=item new

Called for you by TT when you USE the plugin.  There are four parameters:

    cookie        - the auth cookie content
    conf_instance - your Gantry::Conf instance
    conf_file     - your master Gantry::Conf file (optional)
    auth_secret   - the compression secret key

Usually you get the cookie contents from the Apache::Template module's
TT2Params cookies parameter.  If you pass the secret directly, you
should get it by these commands in httpd.conf:

    PerlSetEnv  auth_secret '4our$ecr3t1ssaf3'
    TT2Params   cookie env

Then fish it out of the env hash in your template.  See the SYNOPSIS above.

=item user_name

Returns the name of the user from the cookie, if there is one.

=item auth_secret

Used by Gantry::Plugins::CRUD to retrieve the decryption key.  This
is taken from the Gantry::Conf variable or constructor argument of
the same name.  Precendence is given to the constructor argument if
both are suppied.

=back

=head1 SEE ALSO

Gantry::Plutings::AuthCookie, Gantry::Conf

=head1 AUTHOR

Phil Crow E<lt>philcrow2000@yahoo.comE<gt>

=head1 COPYRIGHT and LICENSE

Copyright (c) 2006, Phil Crow

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut
