/* 
 * Copyright (C) 1995 Andrew Scherpbier <Andrew@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <rplay.h>
#include <sys/signal.h>
#include "forms.h"
#include "xrplay.h"
#include "state.h"
#include "io.h"
#include "gui.h"
#include "sound_list.h"
#include "vu.h"
#include "getopt.h"

#include "icons/next.h"
#include "icons/prev.h"
#include "icons/stop.h"
#include "icons/pause.h"
#include "icons/play.h"
#include "icons/speaker.h"
#include "icons/headphone.h"
#include "icons/lineout.h"

/*
 * Global program state.
 */
State		state;
FD_xrplay	*fd_xrplay;

static struct option longopts[] =
{
	{ "debug",		no_argument,		NULL,	'x' },
	{ "help",		no_argument,		NULL,	1 },
	{ "host",		required_argument,	NULL,	'h' },
	{ "port",		required_argument,	NULL,	'p' },
	{ "version",	no_argument,		NULL,	'v' },
	{ "fork",		no_argument,		NULL,	'f' },
	{ "quit",		no_argument,		NULL,	'q' },
	{ NULL,			0,					NULL,	0 }
};


static void customize_form(FD_xrplay *);
static void parse_arguments(int argc, char **argv);
static void set_defaults();
static void xforms_callback(int fd);
static void interrupt();
static void usage();

/********************************************************************
 * Main program for xrplay.
 */
int main(int argc, char *argv[])
{
	FL_IOPT cntl;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT, interrupt);
		   
	/*
	 * Override the default 3d look.  We like to use only 2 pixels of ramps and no
	 * black box around widgets.  (hence -s)
	 */	
	cntl.borderWidth = -2;
	fl_set_defaults(FL_PDBorderWidth, &cntl);

	/*
	 * Give the XForms library a chance to set stuff up.  This will deal with standard
	 * options for X programs.  Options that are actually used will be taken out of argv.
	 * argc will be updated accordingly.
	 */
	fl_set_icm_color(FL_COL1, 170, 170, 170);
	fl_set_icm_color(FL_MCOL, 151, 151, 151);
  	fl_initialize("XRPlay", "XRPlay", 0, 0, &argc, argv);
	fd_xrplay = create_form_xrplay();
	fl_set_form_minsize(fd_xrplay->xrplay, (FL_Coord) 240, (FL_Coord) (190.0 * 240.0 / 280.0));

	/*
	 * Deal with the xrplay specific arguments
	 */
	set_defaults();
	parse_arguments(argc, argv);

	/*
	 * Make any changes to widgets that were not done by the form creation process
	 */
	customize_form(fd_xrplay);

	/*
	 * Show the main xrplay window
	 */
	fl_show_form(fd_xrplay->xrplay, FL_PLACE_ASPECT, FL_FULLBORDER,"XRPlay");

	/*
	 * Before we go on with regular things, we need to initiate the connection to the rplay server
	 */
	start_connect_machine();

	/*
	 * Perform the program's main loop...
	 */
	register_read(ConnectionNumber(fl_display), xforms_callback);
	fl_check_forms();

	io_loop();
	while (1)
		fl_do_forms();
}


/********************************************************************
 * An X I/O operation needs to be done.  We'll call fl_check_forms()
 * to deal with stuff.
 */
static void xforms_callback(int fd)
{
	fl_check_forms();
}


/********************************************************************
 * Modify the widgets in the form to conform to what we want.
 */
static void customize_form(FD_xrplay *fd_xrplay)
{
	/*
	 * Set the bitmaps for the buttom row of buttons
	 */
	fl_set_bitmapbutton_data(fd_xrplay->action_skip_back, prev_width, prev_height, prev_bits);
	fl_set_bitmapbutton_data(fd_xrplay->action_skip_forward, next_width, next_height, next_bits);
	fl_set_bitmapbutton_data(fd_xrplay->action_pause_play, pause_width, pause_height, pause_bits);
	fl_set_bitmapbutton_data(fd_xrplay->action_stop, stop_width, stop_height, stop_bits);
	fl_set_bitmapbutton_data(fd_xrplay->action_start, play_width, play_height, play_bits);

	/*
	 * Set the bitmaps for the headphone, speaker, and lineout buttons
	 */
	fl_set_bitmapbutton_data(fd_xrplay->headphone, headphone_width, headphone_height, headphone_bits);
	fl_set_bitmapbutton_data(fd_xrplay->speaker, speaker_width, speaker_height, speaker_bits);
	fl_set_bitmapbutton_data(fd_xrplay->lineout, lineout_width, lineout_height, lineout_bits);

	/*
	 * Set the menus
	 */
#if ALLOW_RESIZE
	fl_set_menu(fd_xrplay->file_menu, "Bigger|Smaller|Quit");
#else
	fl_set_menu(fd_xrplay->file_menu, "Quit");
#endif

	/*
	 * Configure the volume slider.  We will change the range, size and step.
	 */
	fl_set_slider_return(fd_xrplay->slider, 0);
	fl_set_slider_bounds(fd_xrplay->slider, 0.0, 255.0);
	fl_set_slider_step(fd_xrplay->slider, 1.0);
	fl_set_slider_size(fd_xrplay->slider, 0.04);

	/*
	 * We always want to disable the start button at this point.
	 */
	fl_deactivate_object(fd_xrplay->action_start);
	fl_set_object_color(fd_xrplay->action_start, FL_COL1, FL_INACTIVE);
	
	/*
	 * Furthermore, we want to make the start button invisible if there are no
	 * sounds to be played.
	 */
	if (!state.playing_sounds)
	{
		fl_hide_object(fd_xrplay->action_start);
	}
	set_gray_state(0, 0);
}


/********************************************************************
 * Initialize the state structure with reasonable default values.
 * For the rptp specific fields we will assign the correct values according
 * to the enviroment.
 */
static void set_defaults()
{
	if (getenv("RPLAY_HOST"))
		state.rptp_host = getenv("RPLAY_HOST");
	else
		state.rptp_host = "localhost";

	state.rptp_port = RPTP_PORT;
	state.debug = 0;
	state.command_state = Not_connected;
	state.data_state = Not_connected;
	state.command = NULL;
	state.data = NULL;
	state.version = "XRPlay 3.1";
	state.is_playing = 1;
	state.is_paused = 0;
	state.playing_sounds = 0;
	state.currently_playing = 0;
	strcpy(state.current_id, "");
	state.volume = 128;
	state.quit = 0;
	state.audio_port[0] = '\0';
}


/********************************************************************
 * Extract xrplay specific options/arguments from the command line
 */
static void parse_arguments(int argc, char **argv)
{
	int			c;
	int			do_fork = 0;
	extern int	optind;
	extern char	*optarg;

	state.program_name = argv[0];

	while ((c = getopt_long(argc, argv, "fh:p:qvx", longopts, 0)) != -1)
	{
		switch (c)
		{
			case 1:				/* --help */
				usage();
				exit(0);

			case 'f':
				do_fork++;
				break;
				
			case 'h':
				state.rptp_host = strdup(optarg);
				break;

			case 'p':
				state.rptp_port = atoi(optarg);
				break;

			case 'q':
				state.quit++;
				break;
				
			case 'v':
				printf("%s\n", state.version);
				exit(0);

			case 'x':
				state.debug++;
				break;

			default:
				fprintf(stderr, "Try `xrplay --help' for more information\n");
				exit(1);
		}
	}

	if (do_fork)
	{
		if (fork())
			exit(0);
	}
	
	if (optind != argc)
		state.playing_sounds = 1;

	for (; optind < argc; optind++)
	{
		sound_list_append(argv[optind]);
	}
}


static void interrupt()
{
	char line[256];
	if (state.current_id[0])
	{
		sprintf(line, "stop id=%s\r\n", state.current_id);
		io_write(state.command->fd, line, strlen(line), really_quit_cb);
	}
	else
	{
		really_quit_cb();
	}
}

static void usage()
{
	printf("%s\n\n", state.version);
	printf("usage: xrplay [options] [soundfile] ...\n\n");
	
	printf("-x, --debug\n");
	printf("\tEnable debugging information.\n");
	printf("\n");

	printf("-f, --fork\n");
	printf("\tCauses xrplay to fork itself after starting.\n\n");

	printf("--help\n");
	printf("\tDisplay helpful information.\n");
	printf("\n");

	printf("-h HOST, --host=HOST\n");
	printf("\tUse HOST as the RPlay server.\n");
	printf("\n");

	printf("-p PORT, --port=PORT\n");
	printf("\tUse PORT as the RPlay port.\n");
	printf("\n");

	printf("-q, --quit\n");
	printf("\tQuit after playing all sounds.\n\n");
	
	printf("-v, --version\n");
	printf("\tDisplay the xrplay version and exit.\n");
	printf("\n");

	printf("And the following XForms options:\n\n");
	
	printf("-bw WIDTH\n");
	printf("\tChange the border width.  The default is 3.\n\n");

	printf("-display HOST:DPY\n");
	printf("\tX Windows display.  The default is $DISPLAY.\n\n");

	printf("-visual CLASS, -depth DEPTH, -private, -shared, -stdcmap,\n");
	printf("-double, -sync, -name, etc.\n");
	printf ("\n");

	printf ("Keyboard shortcuts:\n");
	printf ("       b B : skip backwards\n");
	printf ("       f F : skip forwards\n");
	printf ("       p P : pause\n");
	printf ("       s S : stop\n");
	printf ("       q Q : quit\n");
	printf ("    return : play\n");
	printf (" 0 2 5 7 1 : set volume to 0%, 25%, 50%, 75%, 100%\n");
	printf ("   + = > . : increase volume\n");
	printf ("   - _ , < : decrease volume\n");
	printf ("\n");
}


/*
 * Local variables:
 * tab-width: 4
 * End:
 */
