/*
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <sys/types.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <stdio.h>
#include <unistd.h>
#include "rplayd.h"
#include "timer.h"

int	timer_enabled = 0;
int	timer_blocked = 0;
int	timer_count = 0;
int	timer_rate = AUDIO_RATE;

void	timer_update()
{
	timer_count++;
}

void	timer_block()
{
	sigset_t	sigset;

	timer_blocked++;
	
	sigemptyset(&sigset);
	sigaddset(&sigset, SIGALRM);
	
	if (sigprocmask(SIG_BLOCK, &sigset, NULL) < 0)
	{
		report(REPORT_ERROR, "timer_block: sigprocmask: %s\n", sys_errlist[errno]);
		done(1);
	}
}

void	timer_unblock()
{
	sigset_t	sigset;

	timer_blocked--;
	
	sigemptyset(&sigset);
	sigaddset(&sigset, SIGALRM);
	
	if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0)
	{
		report(REPORT_ERROR, "timer_unblock: sigprocmask: %s\n", sys_errlist[errno]);
		done(1);
	}
}

void	timer_init()
{
	struct sigaction	sa;

	sa.sa_handler = timer_update;
#ifdef SA_RESTART
	sa.sa_flags |= SA_RESTART;
#else
	sa.sa_flags = 0;
#endif
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGALRM);
	if (sigaction(SIGALRM, &sa, (struct sigaction *)NULL) < 0)
	{
		report(REPORT_ERROR, "timer_init: sigaction: %s\n", sys_errlist[errno]);
		done(1);
	}
}

#ifdef __STDC__
void	timer_start(int new_rate)
#else
void	timer_start(new_rate)
int	new_rate;
#endif
{
	struct itimerval	it;

	timer_block();

	timer_enabled = 1;
	timer_count = 0;
	
	if (new_rate <= 0)
	{
		report(REPORT_ERROR, "timer_start: new_rate=%d, must be >= 1\n", new_rate);
		done(1);
	}
	else if (new_rate == 1)
	{
		it.it_interval.tv_sec = 1;
		it.it_interval.tv_usec = 0;
	}
	else
	{
		it.it_interval.tv_sec = 0;
		it.it_interval.tv_usec = 1000000 / new_rate;
	}
	it.it_value = it.it_interval;
	if (setitimer(ITIMER_REAL, &it, NULL) < 0)
	{
		report(REPORT_ERROR, "timer_start: setitimer: %s\n", sys_errlist[errno]);
		done(1);
	}

	timer_unblock();
}

void	timer_stop()
{
	struct itimerval	it;
	
	timer_block();

	timer_enabled = 0;
	
	it.it_interval.tv_sec = 0;
	it.it_interval.tv_usec = 0;
	it.it_value = it.it_interval;
	if (setitimer(ITIMER_REAL, &it, NULL) < 0)
	{
		report(REPORT_ERROR, "timer_stop: setitimer: %s\n", sys_errlist[errno]);
		done(1);
	}

	timer_unblock();
}

void	timer_wait()
{
	while (!timer_count)
	{
		pause();
	}
}
