/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include <stdio.h>
#include <string.h>
#include "rplayd.h"
#include "rplay.h"
#include "spool.h"
#include "sound.h"
#include "buffer.h"
#include "audio.h"

int	spool_size = 0;
int	spool_prio = 0;
SPOOL	spool[SPOOL_SIZE];

/*
 * initialize the sound spool
 */
void	spool_init()
{
	int	i;

	memset((char *)spool, 0, sizeof(spool));

	for(i = 0; i < SPOOL_SIZE; i++)
	{ 
		 spool_clear(&spool[i]);
	}
}

/*
 * return the next available spool entry
 * 
 * 'priority' is the priority of a new sound and it is used to replace
 * lower priority spool entries if the spool is full.
 */
#ifdef __STDC__
SPOOL	*spool_next(int priority)
#else
SPOOL	*spool_next(priority)
int	priority;
#endif
{
	int	i;
	int	min_priority = RPLAY_MAX_PRIORITY, index = -1;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == SPOOL_NULL)
		{
			spool[i].id = spool_id();
			if (spool[i].id < 0)
			{
				return NULL;
			}
			else
			{
				return &spool[i];
			}
		}
		else if (spool[i].rp->priority < min_priority)
		{
			min_priority = spool[i].rp->priority;
			index = i;
		}
	}

	/*
	 * replace a spool entry with a lower priority if possible
	 */
	if (priority > min_priority && index >= 0)
	{
		report(REPORT_DEBUG, "spool[%d] priority %d replaced with priority %d\n", 
			index, min_priority, priority);
		spool_clear(&spool[index]);
		spool_size--;
		spool[i].id = spool_id();
		if (spool[i].id < 0)
		{
			return NULL;
		}
		else
		{
			return &spool[index];
		}
	}
	else
	{
		return NULL;
	}
}

#ifdef __STDC__
void	spool_clear(SPOOL *sp)
#else
void	spool_clear(sp)
SPOOL	*sp;
#endif
{
	sp->id = -1;
	sp->curr_sound = 0;
	sp->curr_count = 0;
	sp->list_count = 0;
	if (sp->rp)
	{
		rplay_destroy(sp->rp);
	}
	sp->rp = NULL;
	sp->ptr = NULL;
	sp->end = NULL;
	sp->state = SPOOL_NULL;
	spool_setprio();
}

#ifdef __STDC__
void	spool_setprio(void)
#else
void	spool_setprio()
#endif
{
	int	i;

	spool_prio = 0;
	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == SPOOL_PLAY)
		{
			spool_prio = MAX(spool_prio, spool[i].rp->priority);
		}
	}
}

/*
 * find the given RPLAY object in the spool
 */
#ifdef __STDC__
int	spool_match(RPLAY *match,  void (*action)(SPOOL *), struct sockaddr_in sin)
#else
int	spool_match(match, action, sin)
RPLAY			*match;
void			(*action)();
struct sockaddr_in	sin;
#endif
{
	int		i, id, nmatch = 0;
	RPLAY		*rp;
	RPLAY_ATTRS	*a1, *a2;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == SPOOL_NULL)
		{
			continue;
		}

		a2 = match->attrs; 
		if (a2->sound[0] == '#')
		{
			for (; a2; a2 = a2->next)
			{
				id = atoi(a2->sound+1);
				if (id == 0 || id == spool[i].id)
				{
					spool[i].sin = sin;
					(*action)(&spool[i]);
					nmatch++;
					break;
				}
			}
			continue;
		}

		rp = spool[i].rp;
		for (a1 = rp->attrs, a2 = match->attrs; a1 && a2; a1 = a1->next, a2 = a2->next)
		{
			if (strcmp(a1->sound, a2->sound))
			{
				break;
			}
		}
		if (!a1 && !a2)
		{
			spool[i].sin = sin;
			(*action)(&spool[i]);
			nmatch++;
		}
	}

	return nmatch;
}

#ifdef __STDC__
void	spool_remove(SOUND *sound)
#else
void	spool_remove(sound)
SOUND	*sound;
#endif
{
	int	i, j, n;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == SPOOL_NULL)
		{
			continue;
		}
		n = 0;
		for (j = 0; j < spool[i].rp->nsounds; j++)
		{
			if (spool[i].sound[j] == sound)
			{
				n++;
			}
		}
		if (n)
		{
			if (spool[i].state == SPOOL_PLAY)
			{
				spool_size--;
			}
			spool_clear(&spool[i]);
		}
	}
}

/*
 * make all spool entries waiting for this sound ready
 */
#ifdef __STDC__
void	spool_ready(SOUND *sound)
#else
void	spool_ready(sound)
SOUND	*sound;
#endif
{
	int	i, j, n;

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		if (spool[i].state == SPOOL_WAIT)
		{
			n = 0;
			for (j = 0; j < spool[i].rp->nsounds; j++)
			{
				if (spool[i].sound[j]->status != SOUND_READY)
				{
					n++;
					if (spool[i].sound[j] == sound)
					{
						n--;
					}
				}
			}
			if (n == 0)
			{
				for (j = 0; j < spool[i].rp->nsounds; j++)
				{
					sound_load(spool[i].sound[j]);
				}
				spool[i].ptr = spool[i].sound[0]->start;
				spool[i].end = spool[i].sound[0]->stop;
				spool[i].state = SPOOL_PLAY;
				spool_size++;
				spool_setprio();
			}
		}
	}
}

BUFFER	*spool_list_create()
{
	BUFFER	*spool_list, *b;
	int	i, n;
	char	buf[RPTP_MAX_LINE];
	SOUND	*s;
	
	b = buffer_create();
	spool_list = b;
	sprintf(b->buf, "+spool\r\nSID HOST            STATE  VOL PRI COUNT SECONDS REMAIN SOUND\r\n");
	b->nbytes += strlen(b->buf);

	for (i = 0; i < SPOOL_SIZE; i++)
	{
		sprintf(buf, "%3d %-15s ", spool[i].id, inet_ntoa(spool[i].sin.sin_addr));
		switch (spool[i].state)
		{
		case SPOOL_PLAY:
			strcat(buf, "play   ");
			break;
			
		case SPOOL_PAUSE:
			strcat(buf, "pause  ");
			break;
			
		case SPOOL_WAIT:
			strcat(buf, "(wait) ");
			break;
			
		default:
			continue;
		}

		s = spool[i].sound[spool[i].curr_sound];
		
		sprintf(buf+27, "%3d %3d %5d %7d %6d %s\r\n",
			spool[i].curr_attrs->volume,
			spool[i].rp->priority,
			spool[i].curr_attrs->count,
			s->size / 8000,
			(spool[i].end - spool[i].ptr) / 8000,
			spool[i].curr_attrs->sound);

		n = strlen(buf);
		if (b->nbytes + n > BUFFER_SIZE)
		{
			b->next = buffer_create();
			b = b->next;
		}
		strcat(b->buf, buf);
		b->nbytes += n;
	}

	if (b->nbytes + 3 > BUFFER_SIZE)
	{
		b->next = buffer_create();
		b = b->next;
	}
	strcat(b->buf, ".\r\n");
	b->nbytes += 3;
	
	return spool_list;
}

/*
 * Return a unique spool id or -1. 
 */
int	spool_id()
{
	static int	id = 0;
	int		i;

	if (spool_size == SPOOL_SIZE)
	{
		return -1;
	}

	for (;;)
	{
		id++;
		if (id > SPOOL_MAX_ID)
		{
			id = SPOOL_MIN_ID;
		}
		for (i = 0; i < SPOOL_SIZE; i++)
		{
			if (spool[i].state != SPOOL_NULL && spool[i].id == id)
			{
				break;
			}
		}
		if (i == SPOOL_SIZE)
		{
			return id;
		}
	}
}

#ifdef __STDC__
void	spool_stop(SPOOL *sp)
#else
void	spool_stop(sp)
SPOOL	*sp;
#endif
{
	if (sp->state == SPOOL_PLAY)
	{
		spool_size--;
	}
	spool_clear(sp);
}

#ifdef __STDC__
void	spool_pause(SPOOL *sp)
#else
void	spool_pause(sp)
SPOOL	*sp;
#endif
{
	if (sp->state == SPOOL_PLAY)
	{
		spool_size--;
	}
	sp->state = SPOOL_PAUSE;
}

#ifdef __STDC__
void	spool_continue(SPOOL *sp)
#else
void	spool_continue(sp)
SPOOL	*sp;
#endif
{
	if (sp->state == SPOOL_PAUSE)
	{
		spool_size++;
	}
	sp->state = SPOOL_PLAY;
}
