/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include "version.h"
#include "rplay.h"
#include <sys/param.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <sys/file.h>
#ifdef ultrix
#include <fcntl.h>
#else
#include <sys/fcntl.h>
#endif
#include <sys/uio.h>
#include <netdb.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/errno.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "rplayd.h"
#include "connection.h"
#include "audio.h"
#include "spool.h"
#include "sound.h"
#ifdef AUTH
#include "host.h"
#endif /* AUTH */
#include "server.h"
#include "hash.h"
#include "misc.h"
#include "cache.h"
#include "timer.h"

fd_set		read_mask;
fd_set		write_mask;

int		debug = 0;
int		inetd = 1;
int		rplayd_timeout = RPLAYD_TIMEOUT;
int		rptp_timeout = RPTP_CONNECTION_TIMEOUT;
int	       	sound_cleanup_timeout = SOUND_CLEANUP_TIMEOUT;

static char	*progname;
static char	*sounds_file = RPLAY_CONF;
#ifdef AUTH
static char	*hosts_file = RPLAY_HOSTS;
#endif /* AUTH */
static char	*servers_file = RPLAY_SERVERS;
static char	*cache_dir = RPLAY_CACHE;
static char	*log_file = RPLAY_LOG;
static int	logging = 0;
static int	report_level;
static int	log_fd = -1;
static int	rplayd_pid;
static int	forward_fd = -1;
static char	*forward = NULL;

extern char	*sys_errlist[];
extern char	*optarg;
extern int	optind;

char		hostname[MAXHOSTNAMELEN];
char		*hostaddr;
int		rplay_fd, rptp_fd;

#ifdef sun
#ifdef __STDC__
extern int	gethostname(char *name, int namelen);
#else
extern int	gethostname(/* char *name, int namelen */);
#endif
#endif

extern char	*ctime();

#ifdef __STDC__
main(int argc, char **argv)
#else
main(argc, argv)
int	argc;
char	**argv;
#endif
{
	int		nfds, tcount = 0, icount = 0, c;
	int		nmissed = 0;
	struct timeval	select_timeout;
	fd_set		rfds, wfds;
	int		curr_rate, curr_bufsize;
	int		idle = 0;
	struct hostent	*hp;
	struct in_addr	addr;

	progname = argv[0];

#ifdef RPLAY_LOG_LEVEL
	report_level = RPLAY_LOG_LEVEL;
	if (report_level != REPORT_NONE)
	{
		logging++;
	}
#else
	report_level = 0;
#endif

#ifdef AUTH
	while ((c = getopt(argc, argv, "A:C:D:H:S:b:c:df:l:L:nr:s:t:T:v")) != -1)
#else /* AUTH */
	while ((c = getopt(argc, argv, "A:C:D:S:b:c:df:l:L:nr:s:t:T:v")) != -1)
#endif /* AUTH */
	{
		switch (c)
		{
		case 'A':
			audio_device = optarg;
			break;

		case 'C':
			sounds_file = optarg;
			break;
		
		case 'D':
			cache_dir = optarg;
			break;

#ifdef AUTH
		case 'H':
			hosts_file = optarg;
			break;
#endif /* AUTH */
		
		case 'S':
			servers_file = optarg;
			break;

		case 'b':
			audio_bufsize = atoi(optarg);
			break;

		case 'c':
			audio_timeout = atoi(optarg);
			break;

		case 'd':
			debug++;
			report_level = REPORT_DEBUG;
			break;

		case 'f':
			forward = optarg;
			break;
			
		case 'L':
			log_file = optarg;
			break;

		case 'l':
			logging++;
			report_level = atoi(optarg);
			break;

		case 'n':
			inetd = 0;
			break;

		case 'r':
			timer_rate = atoi(optarg);
			break;

		case 's':
			cache_max_size = atoi(optarg);
			break;

		case 't':
			rplayd_timeout = atoi(optarg);
			break;

		case 'T':
			rptp_timeout = atoi(optarg);
			break;

		case 'v':
			printf("%s\n", rplay_version);
			done(0);
			break;

		case '?':
			usage();
			done(1);
		}
	}

	if (gethostname(hostname, sizeof(hostname)) < 0)
	{
		report(REPORT_ERROR, "gethostname: %s\n", sys_errlist[errno]);
		done(1);
	}

	hp = gethostbyname(hostname);
	if (hp == NULL)
	{
		report(REPORT_ERROR, "gethostbyname: cannot resolve hostname: %s\n", hostname);
		done(1);
	}
	memcpy((char *)&addr, (char *)hp->h_addr, hp->h_length);
	hostaddr = strdup(inet_ntoa(addr));

	rplayd_pid = getpid();

	/*
	 * ingore SIGPIPE since read will handle closed TCP connections
	 */
	signal(SIGPIPE, SIG_IGN);

	/*
	 * Call all the initialization routines.
	 */
	rplayd_init();
	hash_init(MAX_SOUNDS);
	sound_read(sounds_file);
	cache_init(cache_dir);
	cache_read();
#ifdef AUTH
	host_read(hosts_file);
#endif /* AUTH */
	server_read(servers_file);
	spool_init();
	audio_init();
	timer_init();

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	FD_ZERO(&read_mask);
	FD_ZERO(&write_mask);
	FD_SET(rplay_fd, &read_mask);
	FD_SET(rptp_fd, &read_mask);

	curr_rate = timer_rate;
	curr_bufsize = audio_bufsize;

	report(REPORT_DEBUG, "rate=%d bufsize=%d\n", curr_rate, curr_bufsize);

	/*
	 * Start the timer.
	 */
	timer_start(curr_rate);

	if (forward)
	{
		report(REPORT_DEBUG, "forwarding sounds to %s\n", forward);
	}
	
	report(REPORT_DEBUG, "%s rplayd ready.\n", hostname);
	
	for (;;)
	{
		/*
		 * Wait for the next timer.
		 */
		timer_wait();

		/*
		 * Check if any timers were missed.
		 */
		if (timer_count > 1)
		{
			report(REPORT_DEBUG, "missed %d\n", timer_count - 1);
			nmissed += timer_count - 1;
		}
		else if (nmissed)
		{
			nmissed--;
		}

		/*
		 * Write audio_bufsize bytes to the audio device.
		 */
		if (spool_size)
		{
			icount = 0;
			audio_write(audio_bufsize);
		}

		tcount += timer_count;

		/*
		 * Reset the timer_count after audio data has been written.
		 */
		timer_count = 0;

		/*
		 * See if a second has passed.
		 */
		if (tcount >= curr_rate)
		{
			/*
			 * Timeouts that occur when no sounds are playing.
			 */
			if (!spool_size)
			{
				icount++;
				if (audio_timeout)
				{
					audio_open_count++;
#ifndef sgi
					if (audio_open_count == audio_timeout)
					{
						if (audio_isopen())
						{
							audio_close();
						}
					}
#endif
				}
				if (rplayd_timeout && icount == rplayd_timeout)
				{
					done(0);
				}
				if (sound_cleanup_timeout && icount == sound_cleanup_timeout)
				{
					sound_cleanup();
				}
			}

			/*
			 * Timeouts that occur all the time.
			 */
			if (rptp_timeout && connections)
			{
				connection_check_timeout();
			}

			/*
			 * See if rplayd should enter idle mode.  Idle mode should only be entered
			 * when nothing is happening.
			 */
			if (!spool_size && audio_open_count >= audio_timeout && icount >= sound_cleanup_timeout && !connections)
			{
				idle = 1;
			}
			    
			tcount = 0;
		}

		rfds = read_mask;
		wfds = write_mask;

		if (idle)
		{
			report(REPORT_DEBUG, "entering idle mode\n");
			/*
			 * Stop the timer and block until a socket is ready.
			 */
			timer_stop();
#ifdef __hpux
			nfds = select(FD_SETSIZE, (int *)&rfds, (int *)&wfds, 0, 0);
#else
			nfds = select(FD_SETSIZE, &rfds, &wfds, 0, 0);
#endif
			idle = 0;

			/*
			 * Start the timer again.
			 */
			timer_start(curr_rate);
		}
		else
		{
			/*
			 * Poll all of the socket descriptors to check for RPLAY packets
			 * and RPTP transfers.
			 */
			select_timeout.tv_sec = 0;
			select_timeout.tv_usec = 0;
#ifdef __hpux
			nfds = select(FD_SETSIZE, (int *)&rfds, (int *)&wfds, 0, &select_timeout);
#else
			nfds = select(FD_SETSIZE, &rfds, &wfds, 0, &select_timeout);
#endif
		}
		if (nfds < 0)
		{
			if (errno != EINTR)
			{
				report(REPORT_ERROR, "select: %s\n", sys_errlist[errno]);
				done(1);
			}
		}
		else if (nfds > 0)
		{
			if (FD_ISSET(rplay_fd, &rfds))
			{
				rplayd_read();
			}
			connection_update(&rfds, &wfds);
			tcount = 0;
		}
	}
}

/*
 * set up the sockets to receive RPLAY packets and RPTP connections
 */
void	rplayd_init()
{
	struct sockaddr_in	s;
	int			on = 1;
	
	if (inetd)
	{
		rplay_fd = 0;
	}
	else
	{
		rplay_fd = socket(AF_INET, SOCK_DGRAM, 0);
		if (rplay_fd < 0)
		{
			report(REPORT_ERROR, "socket: %s\n", sys_errlist[errno]);
			done(1);
		}

		s.sin_family = AF_INET;
		s.sin_port = htons(RPLAY_PORT);
		s.sin_addr.s_addr = INADDR_ANY;

		if (bind(rplay_fd, (struct sockaddr *)&s, sizeof(s)) < 0)
		{
			report(REPORT_ERROR, "bind: %s\n", sys_errlist[errno]);
			done(1);
		}
	}

	fd_nonblock(rplay_fd);

	if (forward)
	{
		unsigned long addr = inet_addr(forward);

		memset((char *)&s, 0, sizeof(s));
		s.sin_family = AF_INET;
		s.sin_port = htons(RPLAY_PORT);

		if (addr == 0xffffffff)
		{
			struct hostent	*hp = gethostbyname(forward);
			if (hp == NULL)
			{
				report(REPORT_ERROR, "gethostbyname: %s unknown host\n", forward);
				done(1);
			}
			memcpy((char *)&s.sin_addr.s_addr, (char *)hp->h_addr, hp->h_length);
		}
		else
		{
			memcpy((char *)&s.sin_addr.s_addr, (char *)&addr, sizeof(addr));
		}
		
		forward_fd = socket(AF_INET, SOCK_DGRAM, 0);
		if (forward_fd < 0)
		{
			report(REPORT_ERROR, "socket: %s\n", sys_errlist[errno]);
			done(1);
		}
		
		if (connect(forward_fd, (struct sockaddr *)&s, sizeof(s)) < 0)
		{
			report(REPORT_ERROR, "connect: %s\n", sys_errlist[errno]);
			done(1);
		}
	}
	
	rptp_fd = socket(AF_INET, SOCK_STREAM, 0);
	if (rptp_fd < 0)
	{
		report(REPORT_ERROR, "socket: %s\n", sys_errlist[errno]);
		done(1);
	}

	if (setsockopt(rptp_fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) < 0)
	{
		report(REPORT_ERROR, "setsockopt: SO_REUSEADDR: %s\n", sys_errlist[errno]);
		done(1);
	}

	s.sin_family = AF_INET;
	s.sin_port = htons(RPTP_PORT);
	s.sin_addr.s_addr = INADDR_ANY;

	if (bind(rptp_fd, (struct sockaddr *)&s, sizeof(s)) < 0)
	{
		report(REPORT_ERROR, "bind: %s\n", sys_errlist[errno]);
		done(1);
	}

	if (listen(rptp_fd, 5) < 0)
	{
		report(REPORT_ERROR, "listen: %s\n", sys_errlist[errno]);
		done(1);
	}
}

/*
 * read RPLAY packets from the rplay_fd UDP socket
 */
#ifdef __STDC__
void	rplayd_read(void)
#else
void	rplayd_read()
#endif
{
	static char		recv_buf[MAX_PACKET];
	char			*packet;
	struct sockaddr_in	f;
	int			j, n, flen;
	RPLAY			*rp;

	for (j = 0; j < SPOOL_SIZE; j++)
	{
		flen = sizeof(f);
		
		n = recvfrom(rplay_fd, recv_buf, sizeof(recv_buf), 0, (struct sockaddr *)&f, &flen);
		if (n <= 0)
		{
			continue;
		}

		if (forward)
		{
			write(forward_fd, recv_buf, n);
			continue;
		}
		
#ifdef AUTH
		if (!host_access(f, HOST_EXECUTE))
		{
			if (!host_access(f, HOST_READ) && !host_access(f, HOST_WRITE))
			{
				report(REPORT_NOTICE, "%s permission denied\n", inet_ntoa(f.sin_addr));
			}
			continue;
		}
#endif /* AUTH */

		packet = recv_buf;

		switch (*packet)
		{
#ifdef OLD_RPLAY
		case OLD_RPLAY_PLAY:
		case OLD_RPLAY_STOP:
		case OLD_RPLAY_PAUSE:
		case OLD_RPLAY_CONTINUE:
			packet = rplay_convert(recv_buf);
#endif /* OLD_RPLAY */
				
		case RPLAY_VERSION:
			rp = rplay_unpack(packet);
			if (rp == NULL)
			{
				report(REPORT_ERROR, "rplay_unpack: %s\n", rplay_errlist[rplay_errno]);
				break;
			}

			switch (rp->command)
			{
			case RPLAY_PING:
				report(REPORT_DEBUG, "received a ping packet\n");
				rplay_destroy(rp);
				break;

			case RPLAY_PLAY:
				report(REPORT_INFO, "%s play %s volume=%d\n",
					inet_ntoa(f.sin_addr),
					(char *)rplay_get(rp, RPLAY_SOUND, 0),
					rplay_get(rp, RPLAY_VOLUME, 0));
				rplayd_play(rp, f);
				break;

			case RPLAY_STOP:
				report(REPORT_INFO, "%s stop %s volume=%d\n",
					inet_ntoa(f.sin_addr),
					(char *)rplay_get(rp, RPLAY_SOUND, 0),
					rplay_get(rp, RPLAY_VOLUME, 0));
				rplayd_stop(rp, f);
				break;

			case RPLAY_PAUSE:
				report(REPORT_INFO, "%s pause %s volume=%d\n",
					inet_ntoa(f.sin_addr),
					(char *)rplay_get(rp, RPLAY_SOUND, 0),
					rplay_get(rp, RPLAY_VOLUME, 0));
				rplayd_pause(rp, f);
				break;

			case RPLAY_CONTINUE:
				report(REPORT_INFO, "%s continue %s volume=%d\n",
					inet_ntoa(f.sin_addr),
					(char *)rplay_get(rp, RPLAY_SOUND, 0),
					rplay_get(rp, RPLAY_VOLUME, 0));
				rplayd_continue(rp, f);
				break;
			}
			break;
				
		default:
			report(REPORT_ERROR, "unknown RPLAY packet received from %s\n", inet_ntoa(f.sin_addr));
			break;
		}
	}
}

#ifdef __STDC__
int	rplayd_play(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_play(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	SPOOL		*sp, *s;
	int		i, n;
	RPLAY_ATTRS	*attrs;

	sp = spool_next(rp->priority);
	if (!sp)
	{
		report(REPORT_DEBUG, "spool full\n");
		rplay_destroy(rp);
		return -1;
	}

	n = 0;
	for (i = 0, attrs = rp->attrs; i < rp->nsounds && attrs; i++, attrs = attrs->next)
	{
		if (attrs->rptp_server)
		{
			SERVER		*s;

			s = (SERVER *)malloc(sizeof(SERVER));
			s->next = NULL;
			s->sin.sin_family = AF_INET;
			s->sin.sin_port = htons(attrs->rptp_server_port);
			s->sin.sin_addr.s_addr = inet_addr(attrs->rptp_server);
			sp->sound[i] = sound_lookup(attrs->sound, attrs->rptp_search ? SOUND_FIND : SOUND_DONT_FIND, s);
		}
		else
		{
			sp->sound[i] = sound_lookup(attrs->sound, attrs->rptp_search ? SOUND_FIND : SOUND_DONT_FIND, NULL);
		}
		if (sp->sound[i] == NULL)
		{
			rplay_destroy(rp);
			return -1;
		}
		if (sp->sound[i]->status != SOUND_READY)
		{
			n++;
		}
	}

	sp->rp = rp;
	sp->curr_attrs = rp->attrs;
	sp->curr_sound = 0;
	sp->curr_count = sp->curr_attrs->count;
	sp->list_count = rp->count;
	sp->sin = sin;

	if (n)
	{
		sp->state = SPOOL_WAIT;
	}
	else
	{
		sp->ptr = sp->sound[0]->start;
		sp->end = sp->sound[0]->stop;
		sp->state = SPOOL_PLAY;
		spool_size++;
		spool_setprio();
	}

	return sp->id;
}

#ifdef __STDC__
int	rplayd_stop(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_stop(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	int	n;

	n = spool_match(rp, spool_stop, sin);
	rplay_destroy(rp);

	return n > 0 ? 0 : -1;
}

#ifdef __STDC__
int	rplayd_pause(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_pause(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	int	n;

	n = spool_match(rp, spool_pause, sin);
	rplay_destroy(rp);

	return n > 0 ? 0 : -1;
}

#ifdef __STDC__
int	rplayd_continue(RPLAY *rp, struct sockaddr_in sin)
#else
int	rplayd_continue(rp, sin)
RPLAY			*rp;
struct sockaddr_in	sin;
#endif
{
	int	n;

	n = spool_match(rp, spool_continue, sin);
	rplay_destroy(rp);

	return n > 0 ? 0 : -1;
}

#ifdef __STDC__
void	done(int exit_value)
#else
void	done(exit_value)
int	exit_value;
#endif
{
	connection_cleanup();
	audio_close();
	close(rplay_fd);
	close(rptp_fd);
	report(REPORT_DEBUG, "exit(%d)\n", exit_value);
	if (logging && log_fd >= 0)
	{
		close(log_fd);
	}
	exit(exit_value);
}

#ifdef __STDC__
void	report(int level, char *fmt, ...)
#else
void	report(va_alist)
va_dcl
#endif
{
	va_list			args;
	char			*p;
	time_t			now;
	static struct iovec	iov[2];
	static char		header[256];
	static char		message[256];

#ifdef __STDC__
	va_start(args, fmt);
#else
	char	*fmt;
	int	level;
	va_start(args);
	level = va_arg(args, int);
	fmt = va_arg(args, char *);
#endif

	if (report_level >= level)
	{

		vsprintf(message, fmt, args);
		iov[1].iov_base = message;
		iov[1].iov_len = strlen(message);

		if (logging)
		{
			now = time(0);
			p = ctime(&now);
			p += 4;
			p[15] = '\0';

			if (log_fd < 0)
			{
				log_fd = open(log_file, O_WRONLY|O_CREAT|O_APPEND|O_SYNC, 0644);
				if (log_fd < 0)
				{
					fprintf(stderr, "%s: cannot open logfile '%s': %s\n",
						progname, log_file, sys_errlist[errno]);
					/*
					 * Turn off logging to avoid further problems.
					 */
					logging = 0;
					log_fd = -1;
					return;
				}
			}

			sprintf(header, "%s %s rplayd[%d]: ", p, hostname, rplayd_pid);
			iov[0].iov_base = header;
			iov[0].iov_len = strlen(header);

			if (writev(log_fd, iov, 2) < 0)
			{
				fprintf(stderr, "%s: cannot write to log file '%s': \n",
					progname, log_file, sys_errlist[errno]);
				logging = 0;
				log_fd = -1;
				fprintf(stderr, "%s: logging disabled\n", progname);
			}
		}

		if (debug)
		{
			sprintf(header, "%s: ", progname);
			iov[0].iov_base = header;
			iov[0].iov_len = strlen(header);
#ifndef STDERR_FILENO
#define STDERR_FILENO	3
#endif
			writev(STDERR_FILENO, iov, 2);
		}

	}

	va_end(args);
}

void	usage()
{
	printf("\n");
	printf("%s\n", rplay_version);
	printf("\n");
	printf("usage: %s [options]\n", progname);
	printf("\t-A device  Use device for the audio device (%s).\n", audio_device);
	printf("\t-b n       Audio buffer size (%d).\n", audio_bufsize);
	printf("\t-c n       Close %s after n idle seconds, disabled with 0 (%d).\n", AUDIO_DEVICE, audio_timeout);
	printf("\t-C file    Use file for rplay.conf (%s).\n", RPLAY_CONF);
	printf("\t-d         Enable debug mode.\n");
	printf("\t-D dir     Use dir for rplay.cache (%s).\n", RPLAY_CACHE);
	printf("\t-f host    Forward all RPLAY packets to host.\n");
#ifdef AUTH
	printf("\t-H file    Use file for rplay.hosts (%s).\n", RPLAY_HOSTS);
#endif /* AUTH */
	printf("\t-l n       Use logging level n where %d <= n <= %d.\n", REPORT_MIN, REPORT_MAX);
	printf("\t-L file    Use file for rplay.log (%s).\n", RPLAY_LOG);
	printf("\t-n         Disable inetd mode (%s).\n", inetd ? "enabled" : "disabled");
	printf("\t-r n       Write the audio buffer n times per second (%d).\n", timer_rate); 
	printf("\t-s n       Maximum size in bytes of the rplay cache, disabled with 0 (%d).\n", RPLAY_CACHE_SIZE);
	printf("\t-S file    Use file for rplay.servers (%s).\n", RPLAY_SERVERS);
	printf("\t-t n       Exit after n idle seconds, disabled with 0 (%d).\n", rplayd_timeout);
	printf("\t-T n       Close idle RPTP connections after n seconds, disabled with 0 (%d).\n", rptp_timeout);
	printf("\t-v         Print rplay version and exit.\n");
	printf("\n");
}
