/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <klocalizedstring.h>

#include "skgbankincludes.h"
#include "skgimportexportmanager.h"
#include "skgtestmacro.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        // Test import GSK
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)
            SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("missingfile.gsb")));
            SKGTESTERROR(QLatin1String("imp1.importFile"), impmissing.importFile(), false)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/test-obfuscated.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Account 0")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1029"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Account 4")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("7.5"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Account 1")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("757.5"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Account 3")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("43.5"))
        }
    }

    {
        // Test import GSK
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/version_0.5.9.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), false)
        }
    }

    {
        // Double import
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/essai.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/essai.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Split
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/split.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Compte banque A")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("350"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("GSB.setName"), account.setName(QLatin1String("Compte banque B")), true)
            SKGTESTERROR(QLatin1String("GSB.load"), account.load(), true)
            SKGTEST(QLatin1String("GSB:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-9400"))
        }
    }

    {
        // Budget
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_GSB"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportgsb/imputations_budgetaires.gsb"));
            SKGTESTERROR(QLatin1String("GSB.importFile"), imp1.importFile(), true)
        }
        {
            SKGObjectBase::SKGListSKGObjectBase transactions;
            SKGTESTERROR(QLatin1String("GSB.getObjects"), document1.getObjects(QLatin1String("v_operation_display"), QLatin1String("d_date='2013-08-11'"), transactions), true)
            SKGTEST(QLatin1String("GSB:count"), transactions.count(), 1)
            SKGOperationObject op(transactions[0]);
            SKGTEST(QLatin1String("GSB:Budgetary allocation"), op.getProperty(i18nc("Noun", "Budgetary allocation")), QLatin1String("imputation1"))
            SKGTEST(QLatin1String("GSB:Fiscal year"), op.getProperty(i18nc("Noun", "Fiscal year")), QLatin1String("2013"))
            SKGTEST(QLatin1String("GSB:category"), op.getAttribute(QLatin1String("t_CATEGORY")), QLatin1String("categorie1"))
        }
        {
            SKGObjectBase::SKGListSKGObjectBase transactions;
            SKGTESTERROR(QLatin1String("GSB.getObjects"), document1.getObjects(QLatin1String("v_operation_display"), QLatin1String("d_date='2013-08-12'"), transactions), true)
            SKGTEST(QLatin1String("GSB:count"), transactions.count(), 1)
            SKGOperationObject op(transactions[0]);
            SKGTEST(QLatin1String("GSB:Budgetary allocation"), op.getProperty(i18nc("Noun", "Budgetary allocation")), "imputation1" % OBJECTSEPARATOR % "subimputation1")
            SKGTEST(QLatin1String("GSB:Fiscal year"), op.getProperty(i18nc("Noun", "Fiscal year")), QLatin1String("2013"))
            SKGTEST(QLatin1String("GSB:category"), op.getAttribute(QLatin1String("t_CATEGORY")), "categorie1" % OBJECTSEPARATOR % "subcategorie1")
        }
    }
    // End test
    SKGENDTEST()
}
