#
# Module for administration tasks on displays.
#


from config.ConfigManager import ConfigManager
from utils import singleton
from utils import vfs

import random, time
import gtk



#
# Returns a hashtable "id -> path" of all open displays.
#
def get_displays():

    config = singleton.get(ConfigManager)
    displays = config.get("profiles", get_profile(), "main", "displays")

    ret = {}

    parts = displays.split(",")
    for p in parts:
        if (not p): continue

        if (len(p.split("::")) > 1):
            id, path = p.split("::")
        else:
            # keep backwards compatibility
            id, path = p.split(":")
            
        if (vfs.exists(path) and not vfs.isdir(path)):
            ret[id] = path
    #end for

    return ret




#
# Saves the given hashtable of displays.
#
def _set_displays(displays):

    config = singleton.get(ConfigManager)
    parts = []
    for id, path in displays.items():
        parts.append(id + "::" + path)
    config.set("profiles", get_profile(), "main", "displays", ",".join(parts))



#
# Returns a unique ID.
#
def new_id():

    return (str(time.time()) + \
            str(random.random())).replace(".", "").replace(",", "")



#
# Adds the given display file.
#
def add_display(file, id = ""):

    displays = get_displays()
    if (not id): id = new_id()
    if (vfs.exists(file) and not vfs.isdir(file)):
        displays[id] = file
        
    _set_displays(displays)



#
# Removes the given display.
#
def remove_display(id):

    displays = get_displays()
    del displays[id]
    _set_displays(displays)



#
# Restarts the given display.
#
def restart_display(id):

    displays = get_displays()
    path = displays[id]
    
    remove_display(id)
    gtk.timeout_add(250, add_display, path, id)



#
# Returns the geometry of a display. If for_nw is True, then the coordinates
# of the topleft corner of the window are returned. Otherwise it will be the
# coordinates of the anchor.
#
def get_geometry(id, for_nw = 0):

    key = id + "_default_"
    
    config = singleton.get(ConfigManager)
    if (not for_nw):
        x = int(config.get("profiles", get_profile(), key, "x"))
        y = int(config.get("profiles", get_profile(), key, "y"))
    else:
        x = int(config.get("profiles", get_profile(), key, "real-x") or -1)
        y = int(config.get("profiles", get_profile(), key, "real-y") or -1)

    w = int(config.get("profiles", get_profile(), key, "width"))
    h = int(config.get("profiles", get_profile(), key, "height"))
        
    return (x, y, w, h)



#
# Sets the position of a display.
#
def set_position(id, x, y):

    key = id + "_default_"

    config = singleton.get(ConfigManager)
    config.set("profiles", get_profile(), key, "x", str(x))
    config.set("profiles", get_profile(), key, "y", str(y))



#
# Returns the current profile.
#
def get_profile():

    config = singleton.get(ConfigManager)
    profile = config.get("profile")
    return profile



#
# Returns a list of the available profiles.
#
def get_profiles():

    config = singleton.get(ConfigManager)
    profiles = config.list("profiles")

    return profiles



#
# Sets the profile.
#
def set_profile(profile):

    config = singleton.get(ConfigManager)
    config.set("profile", profile)
