/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gnome.h>
#include <glade/glade.h>

#include <libxklavier/xklavier_config.h>

#include "xkb_capplet_private.h"

static void ProcessLayoutTreeNode( GtkTreeIter * iter,
                                   const XklConfigItemPtr configItem,
                                   const gchar * layoutName,
                                   GSwitchItCapplet * gswic )
{
  GtkWidget *layoutsTree =
    GTK_WIDGET( g_object_get_data
                ( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY ) );

  if( gswic->config.xkbLayouts != NULL )
  {
    int groupNo =
      GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gswic->capplet ),
                                          CURRENT_GROUP_NO_PROPERTY ) );
    const char *btnId = XkbCappletLayout2Id( groupNo, "btn" );
    GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
    const gchar *fullLayoutName =
      g_object_get_data( G_OBJECT( btn ), FULL_LAYOUT_NAME_BTN_PROPERTY );

    if( fullLayoutName != NULL
        && !g_strcasecmp( layoutName, fullLayoutName ) )
    {
      GtkTreeSelection *selection =
        gtk_tree_view_get_selection( GTK_TREE_VIEW( layoutsTree ) );
      GtkTreePath *path = gtk_tree_model_get_path( gtk_tree_view_get_model
                                                   ( GTK_TREE_VIEW
                                                     ( layoutsTree ) ),
                                                   iter );
      g_object_set_data( G_OBJECT( gswic->capplet ),
                         NODE_TO_SHOW_PROPERTY, path );
    }
  }
}

static void LayoutTreeVariantFiller( const XklConfigItemPtr configItem,
                                     GSwitchItCapplet * gswic )
{
  GtkWidget *layoutsTree =
    GTK_WIDGET( g_object_get_data
                ( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY ) );
  GtkTreeIter iter;
  GtkTreeStore *treeStore =
    GTK_TREE_STORE( gtk_tree_view_get_model( GTK_TREE_VIEW( layoutsTree ) ) );
  GtkTreeIter *superIter = g_object_get_data( G_OBJECT( gswic->capplet ),
                                              CURRENT_NODE_PROPERTY );
  gchar *pname = g_object_get_data( G_OBJECT( gswic->capplet ),
                                    CURRENT_LAYOUT_NAME_PROPERTY );
  const gchar *fullLayoutName = GSwitchItConfigXkbMergeItems( pname,
                                                              configItem->
                                                              name );

  gtk_tree_store_append( treeStore, &iter, superIter );
  gtk_tree_store_set( treeStore, &iter, 0,
                      XkbCappletConfigItemDesc( configItem ),
                      1, fullLayoutName, -1 );

  ProcessLayoutTreeNode( &iter, configItem, fullLayoutName, gswic );
}

static void LayoutTreeFiller( const XklConfigItemPtr configItem,
                              GSwitchItCapplet * gswic )
{
  GtkWidget *layoutsTree =
    GTK_WIDGET( g_object_get_data
                ( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY ) );
  GtkTreeIter iter;
  GtkTreeStore *treeStore =
    GTK_TREE_STORE( gtk_tree_view_get_model( GTK_TREE_VIEW( layoutsTree ) ) );

  gtk_tree_store_append( treeStore, &iter, NULL );
  gtk_tree_store_set( treeStore, &iter, 0,
                      XkbCappletConfigItemDesc( configItem ),
                      1, configItem->name, -1 );

  ProcessLayoutTreeNode( &iter, configItem, configItem->name, gswic );

  g_object_set_data( G_OBJECT( gswic->capplet ),
                     CURRENT_LAYOUT_NAME_PROPERTY, configItem->name );
  g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_NODE_PROPERTY,
                     &iter );

  XklConfigEnumLayoutVariants( configItem->name,
                               ( ConfigItemProcessFunc )
                               LayoutTreeVariantFiller, gswic );

  g_object_set_data( G_OBJECT( gswic->capplet ),
                     CURRENT_LAYOUT_NAME_PROPERTY, NULL );
  g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_NODE_PROPERTY,
                     NULL );
}

static void FillLayoutsTree( GSwitchItCapplet * gswic )
{
  GtkWidget *layoutsTree =
    GTK_WIDGET( g_object_get_data
                ( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY ) );

  XklConfigEnumLayouts( ( ConfigItemProcessFunc ) LayoutTreeFiller, gswic );
}

static gint LayoutTreeKeyPress( GtkWidget * widget,
                                GdkEventKey * event,
                                GSwitchItCapplet * gswic )
{
  if( event->keyval == GDK_Escape )
  {
    GtkWidget *popup = GTK_WIDGET
      ( g_object_get_data( G_OBJECT( gswic->capplet ),
                           LAYOUTS_POPUP_PROPERTY ) );
    gtk_widget_hide_all( popup );
    return TRUE;
  }
  return FALSE;
}

const char *XkbCappletLayout2Id( const int groupNumber, const char *ctlType )
{
  static char id[32];
  g_snprintf( id, sizeof( id ), "%sLayout%d", ctlType, groupNumber );
  return id;
}

static void XkbCappletLayoutTreeProcessSelectedItem( GtkTreeModel * model,
                                                     GtkTreePath * path,
                                                     GtkTreeIter * iter,
                                                     GSwitchItCapplet *
                                                     gswic )
{
  GtkWidget *layoutsTree =
    GTK_WIDGET( g_object_get_data
                ( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY ) );
  const int groupNumber =
    GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gswic->capplet ),
                                        CURRENT_GROUP_NO_PROPERTY ) );
  const char *btnId = XkbCappletLayout2Id( groupNumber, "btn" );
  GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
  const char *btnClearId = XkbCappletLayout2Id( groupNumber, "btnClear" );
  GtkWidget *btnClear = CappletGetGladeWidget( gswic, btnClearId );
  const char *lblId = XkbCappletLayout2Id( groupNumber, "lbl" );
  GtkWidget *lbl = CappletGetGladeWidget( gswic, lblId );
  GtkTreePath *path2 = gtk_tree_path_copy( path );
  gchar *id, *descr;

  gtk_tree_model_get( model, iter, 0, &descr, 1, &id, -1 );
  if( gtk_tree_path_up( path2 ) && ( gtk_tree_path_get_depth( path2 ) != 0 ) )
  {
    gchar *parentDescr;
    GtkTreeIter parentIter;
    gtk_tree_model_get_iter( model, &parentIter, path2 );
    gtk_tree_model_get( model, &parentIter, 0, &parentDescr, -1 );
    gtk_label_set_text( GTK_LABEL( lbl ),
                        GSwitchItConfigFormatFullLayout( parentDescr,
                                                         descr ) );
    g_free( parentDescr );
  } else
  {
    gtk_label_set_text( GTK_LABEL( lbl ), descr );
  }

  g_free( descr );

  g_object_set_data_full( G_OBJECT( btn ), FULL_LAYOUT_NAME_BTN_PROPERTY,
                          id, ( GtkDestroyNotify ) g_free );
  gtk_tree_path_free( path2 );

  gtk_widget_set_sensitive( GTK_WIDGET( btnClear ), True );
}

void XkbCappletLayoutTreeItemSelected( GtkTreeSelection * selection,
                                       GSwitchItCapplet * gswic )
{
  GtkWidget *popup =
    GTK_WIDGET( g_object_get_data( G_OBJECT( gswic->capplet ),
                                   LAYOUTS_POPUP_PROPERTY ) );

  gtk_tree_selection_selected_foreach( selection,
                                       ( GtkTreeSelectionForeachFunc )
                                       XkbCappletLayoutTreeProcessSelectedItem,
                                       gswic );

  g_object_set_data( G_OBJECT( gswic->capplet ),
                     CURRENT_GROUP_NO_PROPERTY, NULL );
  g_object_set_data( G_OBJECT( gswic->capplet ), LAYOUTS_POPUP_PROPERTY,
                     NULL );
  g_object_set_data( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY,
                     NULL );

  XkbCappletSetStateToChanged( G_OBJECT( popup ), gswic );
  gtk_widget_hide_all( popup );
}

void XkbCappletLayoutButtonClicked( GtkButton * layoutButton,
                                    GSwitchItCapplet * gswic )
{
  int groupNumber =
    GPOINTER_TO_INT( g_object_get_data
                     ( G_OBJECT( layoutButton ), GROUP_NO_PROPERTY ) );

  XkbCappletPopupLayoutTree( gswic, groupNumber );
}

void XkbCappletClearLayoutButtonClicked( GtkButton * clearLayoutButton,
                                         GSwitchItCapplet * gswic )
{
  const int groupNumber =
    GPOINTER_TO_INT( g_object_get_data
                     ( G_OBJECT( clearLayoutButton ), GROUP_NO_PROPERTY ) );
  const char *btnId = XkbCappletLayout2Id( groupNumber, "btn" );
  GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
  const char *lblId = XkbCappletLayout2Id( groupNumber, "lbl" );
  GtkWidget *lbl = CappletGetGladeWidget( gswic, lblId );

  g_object_set_data( G_OBJECT( btn ), FULL_LAYOUT_NAME_BTN_PROPERTY, NULL );
  gtk_label_set_text( GTK_LABEL( lbl ), "" );

  XkbCappletSetStateToChanged( G_OBJECT( clearLayoutButton ), gswic );
  gtk_widget_set_sensitive( GTK_WIDGET( clearLayoutButton ), False );
}

void XkbCappletPopupLayoutTree( GSwitchItCapplet * gswic, int groupNumber )
{
  GtkWidget *popup = gtk_window_new( GTK_WINDOW_TOPLEVEL );
  GtkWidget *scroll = gtk_scrolled_window_new( NULL, NULL );
  GtkTreeStore *treeStore =
    gtk_tree_store_new( 2, G_TYPE_STRING, G_TYPE_STRING );
  GtkWidget *treeView =
    gtk_tree_view_new_with_model( GTK_TREE_MODEL( treeStore ) );
  GtkTreeSelection *selection =
    gtk_tree_view_get_selection( GTK_TREE_VIEW( treeView ) );
  GtkCellRenderer *renderer =
    GTK_CELL_RENDERER( gtk_cell_renderer_text_new(  ) );
  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes( NULL,
                                                                        renderer,
                                                                        "text",
                                                                        0,
                                                                        NULL );
  GtkTreePath *path;

  gtk_window_set_title( GTK_WINDOW( popup ), _( "..." ) );
  gtk_window_set_policy( GTK_WINDOW( popup ), FALSE, FALSE, TRUE );
  gtk_window_set_decorated( GTK_WINDOW( popup ), FALSE );
  gtk_window_set_transient_for( GTK_WINDOW( popup ),
                                GTK_WINDOW( gswic->capplet ) );
  gtk_window_set_position( GTK_WINDOW( popup ), GTK_WIN_POS_MOUSE );
  gtk_window_set_modal( GTK_WINDOW( popup ), TRUE );

  gtk_tree_view_append_column( GTK_TREE_VIEW( treeView ), column );
  gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( treeView ), FALSE );
  gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

  g_object_set_data( G_OBJECT( gswic->capplet ), LAYOUTS_POPUP_PROPERTY,
                     popup );
  g_object_set_data( G_OBJECT( gswic->capplet ), LAYOUTS_TREE_VIEW_PROPERTY,
                     treeView );
  g_object_set_data( G_OBJECT( gswic->capplet ),
                     CURRENT_GROUP_NO_PROPERTY,
                     GINT_TO_POINTER( groupNumber ) );

  g_signal_connect( G_OBJECT( popup ), "unrealize",
                    G_CALLBACK( gtk_widget_destroy ), NULL );
  g_signal_connect( G_OBJECT( treeView ), "key_press_event",
                    G_CALLBACK( LayoutTreeKeyPress ), gswic );

  FillLayoutsTree( gswic );

  gtk_container_add( GTK_CONTAINER( scroll ), treeView );
  gtk_container_add( GTK_CONTAINER( popup ), scroll );

  gtk_widget_set_usize( popup, 400, 500 );
  gtk_widget_show_all( popup );
  gtk_widget_realize( popup );

  path = ( GtkTreePath * ) g_object_get_data( G_OBJECT( gswic->capplet ),
                                              NODE_TO_SHOW_PROPERTY );
  if( path != NULL )
  {
    GtkTreePath *path2 = gtk_tree_path_copy( path );
    gtk_tree_selection_unselect_all( selection );

    if( gtk_tree_path_up( path ) )      // variant
      gtk_tree_view_expand_row( GTK_TREE_VIEW( treeView ), path, TRUE );

    gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( treeView ), path2, NULL,
                                  TRUE, 0.5, 0 );
    gtk_tree_selection_select_path( selection, path2 );
    gtk_tree_view_row_activated( GTK_TREE_VIEW( treeView ), path2, column );
    gtk_tree_view_set_cursor( GTK_TREE_VIEW( treeView ), path2, column,
                              FALSE );

    gtk_tree_path_free( path );
    gtk_tree_path_free( path2 );
    g_object_set_data( G_OBJECT( gswic->capplet ),
                       NODE_TO_SHOW_PROPERTY, NULL );
  }
  gtk_widget_grab_focus( treeView );

  g_signal_connect( G_OBJECT( selection ), "changed",
                    G_CALLBACK( XkbCappletLayoutTreeItemSelected ), gswic );

}

void XkbCappletUpdateLayoutButtonsFromConfig( GSwitchItCapplet * gswic )
{
  int i, j = 0;

  const GSList *groupItem = gswic->config.xkbLayouts;

  for( i = XkbNumKbdGroups; --i >= 0; j++ )
  {
    const char *btnId = XkbCappletLayout2Id( j, "btn" );
    GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
    const char *btnClearId = XkbCappletLayout2Id( j, "btnClear" );
    GtkWidget *btnClear = CappletGetGladeWidget( gswic, btnClearId );
    Bool isClearPossible = False;
    const char *lblId = XkbCappletLayout2Id( j, "lbl" );
    GtkWidget *lbl = CappletGetGladeWidget( gswic, lblId );
    const char *labelText = "";
    gpointer buttonData = NULL;

    if( groupItem != NULL )
    {
      if( groupItem->data != NULL && *( ( char * ) groupItem->data ) != '\0' )
      {
        char *ln, *vn;
        buttonData = g_strdup( groupItem->data );

        if( GSwitchItConfigGetDescriptions( groupItem->data, &ln, &vn ) )
        {
          labelText = GSwitchItConfigFormatFullLayout( ln, vn );
        }

        isClearPossible = True;
      }
      groupItem = groupItem->next;
    }
    g_object_set_data_full( G_OBJECT( btn ),
                            FULL_LAYOUT_NAME_BTN_PROPERTY,
                            buttonData, ( GtkDestroyNotify ) g_free );
    gtk_label_set_text( GTK_LABEL( lbl ), labelText );

    if( gswic->config.xkbOverrideSettings )
      gtk_widget_set_sensitive( btnClear, isClearPossible );
  }
}

void XkbCappletInitLayoutButtons( GSwitchItCapplet * gswic )
{
  int i, j = 0;
  GladeXML *data =
    GLADE_XML( g_object_get_data( G_OBJECT( gswic->capplet ), "gladeData" ) );

  for( i = XkbNumKbdGroups; --i >= 0; j++ )
  {
    char handlerName[32];
    const char *btnId = XkbCappletLayout2Id( j, "btn" );
    GtkWidget *btn = CappletGetGladeWidget( gswic, btnId );
    const char *btnClearId = XkbCappletLayout2Id( j, "btnClear" );
    GtkWidget *btnClear = CappletGetGladeWidget( gswic, btnClearId );

    g_snprintf( handlerName, sizeof( handlerName ),
                "on_btnLayout%d_clicked", j );
    glade_xml_signal_connect_data( data,
                                   handlerName,
                                   G_CALLBACK
                                   ( XkbCappletLayoutButtonClicked ), gswic );

    g_snprintf( handlerName, sizeof( handlerName ),
                "on_btnClearLayout%d_clicked", j );
    glade_xml_signal_connect_data( data,
                                   handlerName,
                                   G_CALLBACK
                                   ( XkbCappletClearLayoutButtonClicked ),
                                   gswic );

    g_object_set_data( G_OBJECT( btn ), GROUP_NO_PROPERTY,
                       GINT_TO_POINTER( j ) );
    g_object_set_data( G_OBJECT( btnClear ), GROUP_NO_PROPERTY,
                       GINT_TO_POINTER( j ) );
  }
}
