/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "bookmarks-tree-view.h"
#include "bookmarks-context-menu.h"
#include "bookmarks-context-menu-several.h"
#include "bookmarks-single-editor.h"
#include "bookmarks-util.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-string.h"
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GbTreeViewPrivate 
{
	GSList *selected;
	GbTreeViewAction double_click_action;
	GbLocationSource *location_source;
};

/**
 * Private functions, only availble from this file
 */
static void		gb_tree_view_class_init			(GbTreeViewClass *klass);
static void		gb_tree_view_init			(GbTreeView *e);
static void		gb_tree_view_finalize_impl		(GObject *o);
static void		gb_tree_view_selection_changed_cb	(GtkTreeSelection *treeselection, GbTreeView *e);
static void		gb_tree_view_selection_add_selected	(GtkTreeModel *model, GtkTreePath *path, 
								 GtkTreeIter *iter, gpointer data);

static gboolean 	gb_tree_view_treeview_button_press_event_cb (GtkWidget *widget, 
								     GdkEventButton *event, 
								     GbTreeView *e);
static void 		gb_tree_view_ensure_selected		(GtkTreeSelection *sel, 
								 GtkTreePath *path);


enum
{
	TARGET_GTK_TREE_MODEL_ROW
};
static GtkTargetEntry tree_view_row_targets[] = {
	{ "GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_APP, TARGET_GTK_TREE_MODEL_ROW }
};

static gpointer gtk_tree_view_class;

/* signals enums and ids */
enum GbTreeviewSignalsEnum {
	GB_TREE_VIEW_BOOKMARK_ACTIVATED,
	GB_TREE_VIEW_LAST_SIGNAL
};
static gint GbTreeViewSignals[GB_TREE_VIEW_LAST_SIGNAL];

/**
 * TreeView object
 */

MAKE_GET_TYPE (gb_tree_view, "GbTreeView", GbTreeView, gb_tree_view_class_init, 
	       gb_tree_view_init, GTK_TYPE_TREE_VIEW);

static void
gb_tree_view_class_init (GbTreeViewClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_tree_view_finalize_impl;

	GbTreeViewSignals[GB_TREE_VIEW_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbTreeViewClass, bookmark_activated), 
		NULL, NULL, 
		g_cclosure_marshal_VOID__POINTER,
		G_TYPE_NONE, 1, G_TYPE_POINTER);

	gtk_tree_view_class = g_type_class_peek_parent (klass);
}

static void 
gb_tree_view_init (GbTreeView *tv)
{
	GbTreeViewPrivate *p = g_new0 (GbTreeViewPrivate, 1);
	GtkTreeSelection *s;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	
	tv->priv = p;
	
	s = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));
	gtk_tree_selection_set_mode (s, GTK_SELECTION_MULTIPLE);
	g_signal_connect (s, "changed", G_CALLBACK (gb_tree_view_selection_changed_cb), tv);
	
	g_signal_connect (tv, "button_press_event",
			  G_CALLBACK (gb_tree_view_treeview_button_press_event_cb), tv);

	column = gtk_tree_view_column_new ();
	renderer = gtk_cell_renderer_pixbuf_new ();
	
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "pixbuf", GB_TREE_MODEL_COL_ICON, 
					     NULL);
	
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "text", GB_TREE_MODEL_COL_TITLE, 
					     NULL);
	gtk_tree_view_column_set_title (column,  _("Title"));
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tv), column);
	
	column = gtk_tree_view_column_new ();
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "text", GB_TREE_MODEL_COL_URL, 
					     NULL);
	gtk_tree_view_column_set_title (column,  _("Location"));
        gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tv), column);
	
	/* dnd */
	gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (tv),
						GDK_BUTTON1_MASK,
						tree_view_row_targets,
						G_N_ELEMENTS (tree_view_row_targets),
						GDK_ACTION_MOVE | GDK_ACTION_COPY);
	
	gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (tv),
					      tree_view_row_targets,
					      G_N_ELEMENTS (tree_view_row_targets),
					      GDK_ACTION_MOVE | GDK_ACTION_COPY);
	
	p->double_click_action = GB_TV_EDIT;

}

static void
gb_tree_view_finalize_impl (GObject *o)
{
	GbTreeView *tv = GB_TREE_VIEW (o);
	GbTreeViewPrivate *p = tv->priv;
	
	gb_tree_view_set_location_source (tv, NULL);
	
	g_slist_free (p->selected);
	g_free (p);
	
	G_OBJECT_CLASS (gtk_tree_view_class)->finalize (o);
}

GbTreeView *
gb_tree_view_new (void)
{
	GbTreeView *ret = g_object_new (GB_TYPE_TREE_VIEW, NULL);
	return ret;
}

void
gb_tree_view_set_model (GbTreeView *tv, GbTreeModel *m)
{
	GbTreeViewPrivate *p = tv->priv;
	GtkTreePath *path;
	
	g_slist_free (p->selected);
	p->selected = NULL;
	
	gtk_tree_view_set_model (GTK_TREE_VIEW (tv),
				 GTK_TREE_MODEL (m));
	
	/* make sure the root is expanded */
	path = gtk_tree_path_new_from_string ("0");
	gtk_tree_view_expand_row (GTK_TREE_VIEW (tv), path, FALSE);
	gtk_tree_path_free (path);
}

void
gb_tree_view_set_double_click_action (GbTreeView *tv, GbTreeViewAction a)
{
	GbTreeViewPrivate *p = tv->priv;
	p->double_click_action = a;
}

static void 
gb_tree_view_ensure_selected (GtkTreeSelection *sel, GtkTreePath *path)
{
	if (!gtk_tree_selection_path_is_selected (sel, path))
	{
		gtk_tree_selection_unselect_all (sel);
		gtk_tree_selection_select_path (sel, path);
	}
}

static gboolean
gb_tree_view_treeview_button_press_event_cb (GtkWidget *widget, 
					     GdkEventButton *event, 
					     GbTreeView *tv)
{
	GbTreeViewPrivate *p = tv->priv;
	GtkTreeView *tree_view;
	GtkTreePath *path;
	GtkTreeSelection *sel;
	
	tree_view = GTK_TREE_VIEW (widget);
	
	if (event->window != gtk_tree_view_get_bin_window (tree_view)) 
	{
		return FALSE;
	}

	sel = gtk_tree_view_get_selection (tree_view);
	
	if (gtk_tree_view_get_path_at_pos (tree_view, event->x, event->y,
					   &path, NULL, NULL, NULL)) 
	{
		if (event->button == 3)
		{
			GbBookmark *b;

			gb_tree_view_ensure_selected (sel, path);
			gtk_tree_path_free (path);

			b = p->selected && !p->selected->next ? p->selected->data : NULL;
			if (b)
			{
				gb_context_menu_quick (b, (GdkEventButton *) event, 
						       p->location_source, G_OBJECT (tv));
				return TRUE;

			}
			else if (p->selected)
			{
				/* a context menu for several bookmarks */
				gb_context_menu_several_quick (p->selected,
							       (GdkEventButton *) event, 
							       p->location_source, G_OBJECT (tv));
				return TRUE;
			}
			else
			{
				DEBUG_MSG (("Context menu not shown because nothing was selected.\n"));
			}
		}
		else if (event->type == GDK_2BUTTON_PRESS)
		{
			GSList *li;

			gb_tree_view_ensure_selected (sel, path);
			gtk_tree_path_free (path);

			for (li = p->selected; li; li = li->next)
			{
				GbBookmark *b = li->data;
				if (p->double_click_action == GB_TV_EDIT)
				{
					GbSingleEditor *e = gb_single_editor_new ();
					gb_single_editor_set_bookmark (e, b);
					gb_single_editor_show (e);
				}
				else if (p->double_click_action == GB_TV_ACTIVATE)
				{
					const gchar *url = GB_IS_SITE (b) ? GB_SITE (b)->url : NULL;
					gb_activated_event (tv, b, url,
							    GB_BAF_DEFAULT,
							    (GdkEvent *) event);
				}
			}

			return TRUE;
		}
		else if (event->button == 2)
		{
			GSList *li;

			gb_tree_view_ensure_selected (sel, path);
			gtk_tree_path_free (path);

			for (li = p->selected; li; li = li->next)
			{
				GbBookmark *b = li->data;
				const gchar *url = GB_IS_SITE (b) ? GB_SITE (b)->url : NULL;
				gb_activated_event (tv, b, url,
						    GB_BAF_NEW_TAB_OR_WINDOW,
						    (GdkEvent *) event);
			}
			return TRUE;

		}
	} 
	else
	{
		/* Deselect if people click outside any row. It's OK to
		   let default code run; it won't reselect anything. */
		gtk_tree_selection_unselect_all (sel);
	}
	
	return FALSE;
}

void
gb_tree_view_set_location_source (GbTreeView *tv, GbLocationSource *s)
{
	GbTreeViewPrivate *p = tv->priv;
	
	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}
	
	p->location_source = s;
	
	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}
}

static void
gb_tree_view_selection_changed_cb (GtkTreeSelection *treeselection, GbTreeView *tv)
{
	GbTreeViewPrivate *p = tv->priv;
	
	g_slist_free (p->selected);
	p->selected = NULL;
	gtk_tree_selection_selected_foreach (treeselection, gb_tree_view_selection_add_selected, tv);

	p->selected = g_slist_reverse (p->selected);
}

static void
gb_tree_view_selection_add_selected (GtkTreeModel *model, GtkTreePath *path, 
				     GtkTreeIter *iter, gpointer data)
{
	GbTreeView *tv = data;
	GbTreeViewPrivate *p = tv->priv;
	GbBookmark *b = gb_tree_model_bookmark_from_iter ((GbTreeModel *) model, iter);
	p->selected = g_slist_prepend (p->selected, b);
}

void
gb_tree_view_select (GbTreeView *tv, GbBookmark *b)
{
	GtkTreeIter it;
	GtkTreeSelection *s = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));
	GtkTreePath *path;
	GtkTreeModel *tm = gtk_tree_view_get_model (GTK_TREE_VIEW (tv));
	
	g_return_if_fail (GB_IS_TREE_MODEL (tm));
	
	gtk_tree_selection_unselect_all (s);
	gb_tree_model_iter_from_bookmark (GB_TREE_MODEL (tm), b, &it);
	path = gtk_tree_model_get_path (tm, &it);
	while (gtk_tree_path_up (path))
	{
		gtk_tree_view_expand_row (GTK_TREE_VIEW (tv), path, FALSE);
	}
	gtk_tree_path_free (path);
	gtk_tree_selection_select_iter (s, &it);
}

const GSList *
gb_tree_view_get_selected_list (GbTreeView *tv)
{
	GbTreeViewPrivate *p = tv->priv;
	return p->selected;
}

GSList *
gb_tree_view_get_selected_list_prepared (GbTreeView *tv)
{
	GbTreeViewPrivate *p = tv->priv;
	return gb_util_remove_descendants_from_list (p->selected);
}

