/*
 *  Copyright (C) 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ephy-toolbars-model.h"
#include "ephy-dnd.h"
#include "ephy-new-bookmark.h"
#include "ephy-shell.h"
#include "ephy-debug.h"

#include <string.h>

static void ephy_toolbars_model_class_init (EphyToolbarsModelClass *klass);
static void ephy_toolbars_model_init       (EphyToolbarsModel *t);
static void ephy_toolbars_model_finalize   (GObject *object);

enum
{
  ACTION_ADDED,
  LAST_SIGNAL
};

static GObjectClass *parent_class = NULL;

struct EphyToolbarsModelPrivate
{
	gpointer dummy;
};

GType
ephy_toolbars_model_get_type (void)
{
	static GType ephy_toolbars_model_type = 0;

	if (ephy_toolbars_model_type == 0)
	{
		static const GTypeInfo our_info = {
		sizeof (EphyToolbarsModelClass),
		NULL,			/* base_init */
		NULL,			/* base_finalize */
		(GClassInitFunc) ephy_toolbars_model_class_init,
		NULL,
		NULL,			/* class_data */
		sizeof (EphyToolbarsModel),
		0,			/* n_preallocs */
		(GInstanceInitFunc) ephy_toolbars_model_init
	};

	ephy_toolbars_model_type = g_type_register_static (EGG_TOOLBARS_MODEL_TYPE,
							   "EphyToolbarsModel",
							   &our_info, 0);
	}

	return ephy_toolbars_model_type;
}

static const char *
impl_add_item (EggToolbarsModel *t,
	       int toolbar_position,
	       int position,
	       GdkAtom type,
	       const char *name)
{
	EphyBookmarks *bookmarks;
	char *action_name = NULL;
	const char *res;
	gboolean topic = FALSE, normal_item = FALSE;
	int id = -1;

	LOG ("Add item %s", name)

	bookmarks = ephy_shell_get_bookmarks (ephy_shell);

	if (gdk_atom_intern (EPHY_DND_TOPIC_TYPE, FALSE) == type)
	{
		GList *nodes;

		topic = TRUE;
		nodes = ephy_dnd_node_list_extract_nodes (name);
		id = ephy_node_get_id (EPHY_NODE (nodes->data));
		action_name = g_strdup_printf ("GoTopicId%d", id);
		g_list_free (nodes);
	}
	else if (gdk_atom_intern (EPHY_DND_BOOKMARK_TYPE, FALSE) == type)
	{
		GList *nodes;

		nodes = ephy_dnd_node_list_extract_nodes (name);
		id = ephy_node_get_id (EPHY_NODE (nodes->data));
		action_name = g_strdup_printf ("GoBookmarkId%d", id);
		g_list_free (nodes);
	}
	else
	{
		normal_item = TRUE;
	}

	res = action_name ? action_name : name;

	if (normal_item ||
	    !ephy_toolbars_model_has_bookmark (EPHY_TOOLBARS_MODEL (t), topic, id))
	{
		EGG_TOOLBARS_MODEL_CLASS (parent_class)->add_item
			(t, toolbar_position, position, type, res);
	}

	return res;
}

static void
ephy_toolbars_model_class_init (EphyToolbarsModelClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	EggToolbarsModelClass *etm_class;

	etm_class = EGG_TOOLBARS_MODEL_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = ephy_toolbars_model_finalize;

	etm_class->add_item = impl_add_item;
}

static void
ephy_toolbars_model_init (EphyToolbarsModel *t)
{
	t->priv = g_new0 (EphyToolbarsModelPrivate, 1);
}

static void
ephy_toolbars_model_finalize (GObject *object)
{
	EphyToolbarsModel *t = EPHY_TOOLBARS_MODEL (object);

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_EPHY_TOOLBARS_MODEL (object));

	g_free (t->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

EphyToolbarsModel *
ephy_toolbars_model_new (void)
{
	EphyToolbarsModel *t;

	t = EPHY_TOOLBARS_MODEL (g_object_new (EPHY_TOOLBARS_MODEL_TYPE, NULL));

	g_return_val_if_fail (t->priv != NULL, NULL);

	return t;
}

static int
get_item_pos (EphyToolbarsModel *model,
	      int toolbar_pos,
	      const char *name)
{
	int i, n_items;

	n_items = egg_toolbars_model_n_items
		(EGG_TOOLBARS_MODEL (model), toolbar_pos);

	for (i = 0; i < n_items; i++)
	{
		const char *i_name;
		gboolean is_separator;

		i_name = egg_toolbars_model_item_nth
			(EGG_TOOLBARS_MODEL (model), toolbar_pos, i,
			 &is_separator);
		if (!is_separator && strcmp (name, i_name) == 0)
		{
			return i;
		}
	}

	return -1;
}

static char *
get_action_name (gboolean topic, gulong id)
{
	char *action_name;

	if (topic)
	{
		action_name = g_strdup_printf ("GoTopicId%ld", id);
	}
	else
	{
		action_name = g_strdup_printf ("GoBookmarkId%ld", id);
	}

	return action_name;
}

static int
get_toolbar_pos (EphyToolbarsModel *model,
		 const char *name)
{
	int i, n_toolbars;

	n_toolbars = egg_toolbars_model_n_toolbars
		(EGG_TOOLBARS_MODEL (model));

	for (i = 0; i < n_toolbars; i++)
	{
		const char *t_name;

		t_name = egg_toolbars_model_toolbar_nth
			(EGG_TOOLBARS_MODEL (model), i);
		if (strcmp (name, t_name) == 0)
		{
			return i;
		}
	}

	return -1;
}

void
ephy_toolbars_model_remove_bookmark (EphyToolbarsModel *model,
				     gboolean topic,
				     gulong id)
{
	char *action_name;
	int toolbar_position, position;

	action_name = get_action_name (topic, id);

	toolbar_position = get_toolbar_pos (model, "BookmarksBar");
	g_return_if_fail (toolbar_position != -1);

	position = get_item_pos (model, toolbar_position, action_name);

	egg_toolbars_model_remove_item (EGG_TOOLBARS_MODEL (model),
				        toolbar_position, position);

	g_free (action_name);
}

void
ephy_toolbars_model_add_bookmark (EphyToolbarsModel *model,
				  gboolean topic,
				  gulong id)
{
	char *action_name;
	int toolbar_position;

	action_name = get_action_name (topic, id);

	toolbar_position = get_toolbar_pos (model, "BookmarksBar");
	g_return_if_fail (toolbar_position != -1);

	egg_toolbars_model_add_item (EGG_TOOLBARS_MODEL (model),
				     toolbar_position, -1,
				     0, action_name);

	g_free (action_name);
}

gboolean
ephy_toolbars_model_has_bookmark (EphyToolbarsModel *model,
				  gboolean topic,
				  gulong id)
{
	char *action_name;
	int toolbar_position, position;

	action_name = get_action_name (topic, id);

	toolbar_position = get_toolbar_pos (model, "BookmarksBar");
	g_return_val_if_fail (toolbar_position != -1, FALSE);
	position = get_item_pos (model, toolbar_position, action_name);

	g_free (action_name);

	return (position != -1);
}
