/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Copyright (C) 2005 Vincent Noel <vnoel@cox.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "gdict-entry.h"
#include "gdict-pref.h"
#include "gdict-app.h"
#include "gdict-utils.h"

enum {
	WORD_LOOKUP_START_SIGNAL,
	WORD_LOOKUP_DONE_SIGNAL,
	WORD_NOT_FOUND_SIGNAL,
	SOCKET_ERROR_SIGNAL,
	LAST_SIGNAL
};

#define GDICT_ENTRY_RESPONSE 100
#define GDICT_MAX_COMPLETIONS 25

static gint signals[LAST_SIGNAL] = { 0, 0, 0 };

static GtkEntryClass *parent_class;

static void gdict_entry_init (GDictEntry *entry);
static void gdict_entry_class_init (GDictEntryClass *class);
static void gdict_entry_word_status_cb (dict_command_t *command,
					DictStatusCode code, int num_found,
					gpointer data);

GType
gdict_entry_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GDictEntryClass),
			NULL, NULL,
			(GClassInitFunc) gdict_entry_class_init,
			NULL, NULL,
			sizeof (GDictEntry),
			0,
			(GInstanceInitFunc) gdict_entry_init
		};

		type = g_type_register_static (GTK_TYPE_ENTRY,
					       "GDictEntry", &info, 0);
	}

	return type;
}

/* Word search is over */

static void
gdict_entry_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
	GDictWindow *gdict = GDICT_WINDOW (data);
	GDictEntry *entry = GDICT_ENTRY (widget);
	GtkEntryCompletion *completion;
	
	completion = gtk_entry_get_completion (GTK_ENTRY (entry));
	gtk_entry_completion_complete (completion);

	gtk_widget_hide (gdict->statusbar);
	gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
}

/* Word search just began */

static void
gdict_entry_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
	GDictWindow *gdict = data;
	gtk_widget_show (gdict->statusbar);

	gtk_statusbar_pop (GTK_STATUSBAR(gdict->statusbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR(gdict->statusbar), 0, _("Looking up entry..."));
}

/* Word not found */

static void
gdict_entry_not_found_cb (GtkWidget *widget, gpointer data) 
{
	GDictWindow *gdict = data;
	gtk_widget_show (gdict->statusbar);

	gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("No matches found"));
	gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
}

static void
gdict_entry_error_cb (dict_command_t *command, DictStatusCode code,
                gchar *message, gpointer data)
{
	GDictEntry *entry = data;
	GDictWindow *gdict = entry->gdict;
	
	if (code != DICT_SOCKET_ERROR) {
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new_with_markup (NULL,
							     GTK_DIALOG_DESTROY_WITH_PARENT,
							     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
							     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
							     _("Error invoking query"), message);
		
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		
		if (command->cmd == C_MATCH)
			g_signal_emit (G_OBJECT (entry), signals[WORD_LOOKUP_DONE_SIGNAL], 0);
	}
	else {
		g_signal_emit (G_OBJECT (entry),
			       signals[SOCKET_ERROR_SIGNAL], 0, message);
	}
}

/* We just received a new word, append it to the completion list */

static void 
gdict_entry_word_data_cb (dict_command_t *command,
			  dict_res_t *res,
			  gpointer data)
{
	GDictEntry *entry = data;
	GtkListStore *model;
	GtkTreeIter iter;
	GtkEntryCompletion *completion;

	completion = gtk_entry_get_completion (GTK_ENTRY (entry));
	model = GTK_LIST_STORE (gtk_entry_completion_get_model (completion));
	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, 0, res->desc, -1);
	entry->completions++;
	if (entry->completions > GDICT_MAX_COMPLETIONS) {
		dict_command_destroy (entry->spell_cmd);
		entry->spell_cmd = NULL;
		gdict_entry_lookup_done_cb (GTK_WIDGET (entry), entry->gdict);
	}
}

/* Send the command to fetch new words */

static void 
gdict_entry_lookup (GDictEntry *entry)
{
	GDictWindow *gdict;
	char *text;

	g_return_if_fail (GDICT_IS_ENTRY (entry));

	gdict = entry->gdict;
	if (entry->context == NULL) {		
		gdict_init_context (gdict);
		entry->context = gdict->context;
	}

	text = entry->current_text;
	while (isspace (*text))
		text++;

	if (*text == '\0')
		return;

	g_signal_emit (G_OBJECT (entry), signals[WORD_LOOKUP_START_SIGNAL], 0);

	g_free (entry->database);
	entry->database = g_strdup (gdict->pref->database);
    
	entry->spell_cmd = dict_match_command_new (entry->database, "prefix", text);
	entry->spell_cmd->error_notify_cb = gdict_entry_error_cb;
	entry->spell_cmd->status_notify_cb = gdict_entry_word_status_cb;
	entry->spell_cmd->data_notify_cb = gdict_entry_word_data_cb;
	entry->spell_cmd->user_data = entry;

	dict_command_invoke (entry->spell_cmd, entry->context);
}

/* The entry just changed */

void
gdict_entry_changed (GtkEditable *editable, gpointer data)
{
	GDictEntry *entry = GDICT_ENTRY (editable);
	GtkEntryCompletion *completion;
	GtkTreeModel *model;
	GtkTreeIter iter;
	char *new_search;

	new_search = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

	/* If the new word has the same prefix as the old, we already have
	   valid completions in the model, so do nothing */
	if ((entry->current_text) && 
	    (entry->completions > 0) &&
	    (g_str_has_prefix (new_search, entry->current_text))) {
		g_free (entry->current_text);
		entry->current_text = new_search;
	} else {
		if ((entry->current_text == NULL) ||
		    g_ascii_strcasecmp (new_search, entry->current_text) != 0) {
			
			g_free (entry->current_text);
			entry->current_text = new_search;
			
			if (entry->spell_cmd) {
				dict_command_destroy (entry->spell_cmd);
				entry->spell_cmd = NULL;
			}
			
			completion = gtk_entry_get_completion (GTK_ENTRY(entry));
			model = gtk_entry_completion_get_model (completion);
			if (GTK_IS_LIST_STORE (model)) {
				gtk_list_store_clear (GTK_LIST_STORE (model));
			}
			
			entry->completions = 0;
			gdict_entry_lookup (GDICT_ENTRY(entry));
		}
	}
}

/* A new status code just arrived */

static void 
gdict_entry_word_status_cb (dict_command_t *command,
		      DictStatusCode code, 
		      int num_found,
		      gpointer data)
{
	GDictEntry *entry;

	g_return_if_fail (GDICT_IS_ENTRY (data));

	entry = GDICT_ENTRY (data);

	if (code == DICT_STATUS_OK)
		g_signal_emit (G_OBJECT (entry), 
                	       signals[WORD_LOOKUP_DONE_SIGNAL], 0);
	else if (code == DICT_STATUS_NO_MATCH)
		g_signal_emit (G_OBJECT (entry), 
                	       signals[WORD_NOT_FOUND_SIGNAL], 0);
}

GtkWidget *
gdict_entry_new (GDictWindow *gdict)
{
	GtkWidget *widget;
	GDictEntry *entry;
	
	widget = g_object_new (GDICT_TYPE_ENTRY, NULL);
	entry = GDICT_ENTRY (widget);
	entry->gdict = gdict;
	g_signal_connect (entry, "word_lookup_start",
			  G_CALLBACK (gdict_entry_lookup_start_cb), gdict);
	g_signal_connect (entry, "word_lookup_done",
			  G_CALLBACK (gdict_entry_lookup_done_cb), gdict);
	g_signal_connect (entry, "word_not_found",
			  G_CALLBACK (gdict_entry_not_found_cb), gdict);
	g_signal_connect (entry, "socket_error",
			  G_CALLBACK (gdict_socket_error_cb), gdict);
	g_signal_connect (entry, "changed",
			  G_CALLBACK (gdict_entry_changed), gdict);
	return widget;
}

static void 
gdict_entry_init (GDictEntry *entry)
{
	GtkEntryCompletion *completion;
	GtkTreeModel *model;

	entry->context = NULL;
	entry->spell_cmd = NULL;
	model = GTK_TREE_MODEL (gtk_list_store_new (1, G_TYPE_STRING));

	completion = gtk_entry_completion_new ();
	gtk_entry_set_completion (GTK_ENTRY (entry), completion);
	gtk_entry_completion_set_model (completion, model);
	g_object_unref (completion);
	g_object_unref (model);

	gtk_entry_completion_set_text_column (completion, 0);
	gtk_entry_completion_set_popup_completion (completion, TRUE);
	gtk_entry_completion_set_minimum_key_length (completion, 5);
	gtk_entry_completion_set_inline_completion (completion, TRUE);
}

static void
gdict_entry_finalize (GObject *object)
{
	GDictEntry *entry;

	g_return_if_fail (GDICT_IS_ENTRY (object));

	entry = GDICT_ENTRY (object);

	g_free (entry->database);
	dict_command_destroy (entry->spell_cmd);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void 
gdict_entry_class_init (GDictEntryClass *class)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (class);

	signals[WORD_LOOKUP_START_SIGNAL] =
		g_signal_new ("word_lookup_start",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictEntryClass, word_lookup_start),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[WORD_LOOKUP_DONE_SIGNAL] =
		g_signal_new ("word_lookup_done",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictEntryClass, word_lookup_done),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[WORD_NOT_FOUND_SIGNAL] =
		g_signal_new ("word_not_found",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictEntryClass, word_not_found),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[SOCKET_ERROR_SIGNAL] =
        	g_signal_new ("socket_error",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictEntryClass, socket_error),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);

	parent_class = g_type_class_peek_parent (class);
	class->word_lookup_done = NULL;
	class->word_not_found = NULL;
	
	object_class->finalize = gdict_entry_finalize;
}
