/*
 *  Copyright (C) 2004 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: EphyUtils.cpp,v 1.13 2005/07/24 21:37:01 chpe Exp $
 */

#include "mozilla-config.h"

#include "config.h"

#include "EphyUtils.h"
#include "ephy-embed-shell.h"
#include "ephy-embed-single.h"
#include "print-dialog.h"
#include "ephy-file-helpers.h"

#include <nsIIOService.h>
#include <nsIURI.h>
#include <nsIDOMWindow.h>
#include <nsIServiceManager.h>
#undef MOZILLA_INTERNAL_API
#include <nsEmbedString.h>
#define MOZILLA_INTERNAL_API 1
#include <nsIWindowWatcher.h>
#include <nsIEmbeddingSiteWindow.h>
#include <nsIWebBrowserChrome.h>
#include <gtkmozembed.h>
#include <nsIPrintSettings.h>

nsresult
EphyUtils::GetIOService (nsIIOService **ioService)
{
	nsresult rv;

	nsCOMPtr<nsIServiceManager> mgr; 
	NS_GetServiceManager (getter_AddRefs (mgr));
	if (!mgr) return NS_ERROR_FAILURE;

	rv = mgr->GetServiceByContractID ("@mozilla.org/network/io-service;1",
					  NS_GET_IID (nsIIOService),
					  (void **)ioService);
	return rv;
}

nsresult
EphyUtils::NewURI (nsIURI **result,
		   const nsAString &spec,
		   const char *charset,
		   nsIURI *baseURI)
{
	nsEmbedCString cSpec;
	NS_UTF16ToCString (spec, NS_CSTRING_ENCODING_UTF8, cSpec);

	return NewURI (result, cSpec, charset, baseURI);
}

nsresult
EphyUtils::NewURI (nsIURI **result,
		   const nsACString &spec,
		   const char *charset,
		   nsIURI *baseURI)
{
	nsresult rv;

	nsCOMPtr<nsIIOService> ioService;
	rv = EphyUtils::GetIOService (getter_AddRefs (ioService));
	NS_ENSURE_SUCCESS (rv, rv);

	rv = ioService->NewURI (spec, charset, baseURI, result);

	return rv;
}

GtkWidget *
EphyUtils::FindEmbed (nsIDOMWindow *aDOMWindow)
{
	if (!aDOMWindow) return nsnull;

	nsCOMPtr<nsIWindowWatcher> wwatch
		(do_GetService("@mozilla.org/embedcomp/window-watcher;1"));
	NS_ENSURE_TRUE (wwatch, nsnull);

	/* this DOM window may belong to some inner frame, we need
	 * to get the topmost DOM window to get the embed
	 */
	nsCOMPtr<nsIDOMWindow> topWindow;
	aDOMWindow->GetTop (getter_AddRefs (topWindow));
	if (!topWindow) return nsnull;
	
	nsCOMPtr<nsIWebBrowserChrome> windowChrome;
	wwatch->GetChromeForWindow (topWindow, getter_AddRefs(windowChrome));
	NS_ENSURE_TRUE (windowChrome, nsnull);

	nsCOMPtr<nsIEmbeddingSiteWindow> window (do_QueryInterface(windowChrome));
	NS_ENSURE_TRUE (window, nsnull);

	nsresult rv;
	GtkWidget *mozembed;
	rv = window->GetSiteWindow ((void **)&mozembed);
	NS_ENSURE_SUCCESS (rv, nsnull);

	return mozembed;
}

GtkWidget *
EphyUtils::FindGtkParent (nsIDOMWindow *aDOMWindow)
{
	GtkWidget *embed = FindEmbed (aDOMWindow);
	NS_ENSURE_TRUE (embed, nsnull);

	return gtk_widget_get_toplevel (GTK_WIDGET (embed));
}

nsresult
EphyUtils::CollatePrintSettings (EmbedPrintInfo *info,
				 nsIPrintSettings *options,
				 gboolean preview)
{
	const static int frame_types[] = {
		nsIPrintSettings::kFramesAsIs,
		nsIPrintSettings::kSelectedFrame,
		nsIPrintSettings::kEachFrameSep
	};

	switch (info->range)
	{
	case GNOME_PRINT_RANGE_CURRENT:
	case GNOME_PRINT_RANGE_SELECTION_UNSENSITIVE:
	case GNOME_PRINT_RANGE_ALL:
		options->SetPrintRange (nsIPrintSettings::kRangeAllPages);
		break;
	case GNOME_PRINT_RANGE_RANGE:
		options->SetPrintRange (nsIPrintSettings::kRangeSpecifiedPageRange);
		options->SetStartPageRange (info->from_page);
		options->SetEndPageRange (info->to_page);
		break;
	case GNOME_PRINT_RANGE_SELECTION:
		options->SetPrintRange (nsIPrintSettings::kRangeSelection);
		break;
	}
	
	const GnomePrintUnit *unit, *inch, *mm;
	double value;
	
	mm = gnome_print_unit_get_by_abbreviation ((const guchar *) "mm");
	inch = gnome_print_unit_get_by_abbreviation ((const guchar *) "in");
	g_assert (mm != NULL && inch != NULL);

	/* top margin */
	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAGE_MARGIN_TOP,
					   &value, &unit)
	    && gnome_print_convert_distance (&value, unit, inch))
	{
		options->SetMarginTop (value);
	}

	/* bottom margin */
	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAGE_MARGIN_BOTTOM,
					   &value, &unit)
	    && gnome_print_convert_distance (&value, unit, inch))
	{
		options->SetMarginBottom (value);
	}

	/* left margin */
	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAGE_MARGIN_LEFT,
					   &value, &unit)
	    && gnome_print_convert_distance (&value, unit, inch))
	{
		options->SetMarginLeft (value);
	}

	/* right margin */
	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAGE_MARGIN_RIGHT,
					   &value, &unit)
	    && gnome_print_convert_distance (&value, unit, inch))
	{
		options->SetMarginRight (value);
	}


	nsEmbedString tmp;

	NS_CStringToUTF16 (nsEmbedCString(info->header_left_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetHeaderStrLeft (tmp.get());

	NS_CStringToUTF16 (nsEmbedCString(info->header_center_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetHeaderStrCenter (tmp.get());

	NS_CStringToUTF16 (nsEmbedCString(info->header_right_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetHeaderStrRight (tmp.get());

	NS_CStringToUTF16 (nsEmbedCString(info->footer_left_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetFooterStrLeft (tmp.get());

	NS_CStringToUTF16 (nsEmbedCString(info->footer_center_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetFooterStrCenter(tmp.get());

	NS_CStringToUTF16 (nsEmbedCString(info->footer_right_string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetFooterStrRight(tmp.get());

	options->SetPrintToFile (PR_FALSE);
	
	if (!preview)
	{
		char *cmd, *base;
		const char *temp_dir;
		
		temp_dir = ephy_file_tmp_dir ();
		base = g_build_filename (temp_dir, "printXXXXXX", NULL);
		info->tempfile = ephy_file_tmp_filename (base, "ps");
		g_free (base);
		
		/* use cat instead of print to file to avoid fflush to ensure
		 * the file has been written completely and we don't need to
		 * select a printer (i.e. should be printing backend independent)
		 */
		
		cmd = g_strconcat ("cat > ", info->tempfile, NULL);
		NS_CStringToUTF16 (nsEmbedCString(cmd),
				   NS_CSTRING_ENCODING_UTF8, tmp);
		options->SetPrintCommand (tmp.get());
		g_free (cmd);
	}

	/* paper size */
	options->SetPaperSize (nsIPrintSettings::kPaperSizeDefined);
	options->SetPaperSizeUnit (nsIPrintSettings::kPaperSizeMillimeters);

	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAPER_WIDTH,
					   &value, &unit)
	   && gnome_print_convert_distance (&value, unit, mm))
	{
		options->SetPaperWidth (value);	
	}

	if (gnome_print_config_get_length (info->config,
					   (const guchar *) GNOME_PRINT_KEY_PAPER_HEIGHT,
					   &value, &unit)
	   && gnome_print_convert_distance (&value, unit, mm))
	{
		options->SetPaperHeight (value);	
	}
	
	char *string;

	/* paper name */
	string = (char *) gnome_print_config_get (info->config,
						  (const guchar *) GNOME_PRINT_KEY_PAPER_SIZE);
	NS_CStringToUTF16 (nsEmbedCString(string),
			   NS_CSTRING_ENCODING_UTF8, tmp);
	options->SetPaperName (tmp.get());
	g_free (string);

	/* paper orientation */
	string = (char *) gnome_print_config_get (info->config,
						  (const guchar *) GNOME_PRINT_KEY_ORIENTATION);
	if (string == NULL) string = g_strdup ("R0");

	if (strncmp (string, "R90", 3) == 0 || strncmp (string, "R270", 4) == 0)
	{
		options->SetOrientation (nsIPrintSettings::kLandscapeOrientation);
	}
	else
	{
		options->SetOrientation (nsIPrintSettings::kPortraitOrientation);
	}
	g_free (string);

	options->SetPrintInColor (info->print_color);
	options->SetPrintFrameType (frame_types[info->frame_type]);

	return NS_OK;
}

PRBool
EphyUtils::StringEquals (nsAString &s1, nsAString &s2)
{
	if (s1.Length() != s2.Length())
	{
		return FALSE;
	}

	return (memcmp (s1.BeginReading(),
		        s2.BeginReading(),
		        s2.Length() * sizeof (PRUnichar)) == 0);
}
