/* $Id: gdict-speller.c,v 1.41 2005/01/18 20:56:55 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict Spell Dialog
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "gdict-speller.h"
#include "gdict-pref.h"
#include "gdict-app.h"
#include "gdict-utils.h"

enum {
	WORD_LOOKUP_START_SIGNAL,
	WORD_LOOKUP_DONE_SIGNAL,
	WORD_NOT_FOUND_SIGNAL,
	SOCKET_ERROR_SIGNAL,
	LAST_SIGNAL
};

#define GDICT_SPELL_RESPONSE 100

static gint signals[LAST_SIGNAL] = { 0, 0, 0 };

static GtkVBoxClass *parent_class;

static void gdict_speller_init (GDictSpeller *speller);
static void gdict_speller_class_init (GDictSpellerClass *class);

GType
gdict_speller_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GDictSpellerClass),
			NULL, NULL,
			(GClassInitFunc) gdict_speller_class_init,
			NULL, NULL,
			sizeof (GDictSpeller),
			0,
			(GInstanceInitFunc) gdict_speller_init
		};

		type = g_type_register_static (GTK_TYPE_VBOX,
					       "GDictSpeller", &info, 0);
	}

	return type;
}

static void
gdict_speller_row_selected_cb (GtkTreeSelection *selection, GDictSpeller *speller)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *word;

	g_return_if_fail (GDICT_IS_SPELLER (speller));

	if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
		return;
    
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (speller->word_list));
	gtk_tree_model_get (model, &iter, 0, &word, -1);
	speller->current_word = g_strdup (word);

	gdict_lookup_definition (speller->gdict, word);
}

static void
gdict_speller_hide (GtkButton *button, gpointer data)
{
	GDictSpeller *speller = data;
	GDictWindow *gdict = speller->gdict;
	GtkWidget *check;
	check = gtk_ui_manager_get_widget (gdict->ui, "/MenuBar/ViewMenu/ViewSpellings");
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (check), FALSE);
}

static void 
gdict_speller_init (GDictSpeller *speller)
{
	GtkWidget *label, *button, *scrolled_win;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkListStore *model;
	GtkTreeViewColumn *column;
	GtkTreeSelection *selection;
	GtkCellRenderer *cell;

	speller->context = NULL;
	speller->spell_cmd = NULL;

	vbox = GTK_WIDGET(&(speller->vbox));
	gtk_box_set_spacing (GTK_BOX (vbox), 3);
	
	hbox = gtk_hbox_new (FALSE, 0);
	label = gtk_label_new ("Spellings");
	gtk_misc_set_alignment (GTK_MISC(label), 0., 0.5);
	gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
	button = gdict_button_new_with_stock_image (NULL, GTK_STOCK_CLOSE);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (gdict_speller_hide), speller);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	scrolled_win = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_win),
					     GTK_SHADOW_ETCHED_IN);

	model = gtk_list_store_new (1, G_TYPE_STRING);

	speller->word_list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (model));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (speller->word_list), FALSE);
	g_object_unref (G_OBJECT (model));
    
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (speller->word_list));
	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (gdict_speller_row_selected_cb), speller);

	cell = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("words", cell, "text", 0, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 0);

	gtk_tree_view_append_column (GTK_TREE_VIEW (speller->word_list), column);

	gtk_container_add (GTK_CONTAINER (scrolled_win), speller->word_list);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_win, TRUE, TRUE, 0);

	gtk_widget_show (GTK_WIDGET (speller));
}

static void
gdict_speller_finalize (GObject *object)
{
	GDictSpeller *speller;

	g_return_if_fail (GDICT_IS_SPELLER (object));

	speller = GDICT_SPELLER (object);

	g_free (speller->database);
	dict_command_destroy (speller->spell_cmd);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void 
gdict_speller_class_init (GDictSpellerClass *class)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (class);

	signals[WORD_LOOKUP_START_SIGNAL] =
		g_signal_new ("word_lookup_start",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictSpellerClass, word_lookup_start),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[WORD_LOOKUP_DONE_SIGNAL] =
		g_signal_new ("word_lookup_done",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictSpellerClass, word_lookup_done),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[WORD_NOT_FOUND_SIGNAL] =
		g_signal_new ("word_not_found",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictSpellerClass, word_not_found),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[SOCKET_ERROR_SIGNAL] =
        	g_signal_new ("socket_error",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GDictSpellerClass, socket_error),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);

	parent_class = g_type_class_peek_parent (class);
	class->word_lookup_done = NULL;
	class->word_not_found = NULL;
	
	object_class->finalize = gdict_speller_finalize;
}

static void
gdict_speller_set_sensitive (GDictWindow *gdict, gboolean sensitive)
{
  GtkAction *action;
  gtk_widget_set_sensitive (gdict->speller, sensitive);
  action = gtk_ui_manager_get_action (gdict->ui, "/MenuBar/ViewMenu/ViewSpellings");
  gtk_action_set_sensitive (action, sensitive);
}

static void
gdict_speller_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
  gtk_statusbar_pop (GTK_STATUSBAR(gdict->statusbar), 0);
  gtk_statusbar_push (GTK_STATUSBAR(gdict->statusbar), 0, _("Spell-checking..."));
  gdict_speller_set_sensitive (gdict, FALSE);
}

static void
gdict_speller_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("Spell check done"));
  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
  gdict_speller_set_sensitive (gdict, TRUE);
}

static void
gdict_speller_not_found_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("No matches found"));
  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
  gdict_speller_set_sensitive (gdict, TRUE);
}

GtkWidget *
gdict_speller_new (GDictWindow *gdict)
{
	GtkWidget *widget;
  GDictSpeller *speller;

	widget = g_object_new (GDICT_TYPE_SPELLER, NULL);
	speller = GDICT_SPELLER (widget);
  speller->gdict = gdict;
  g_signal_connect (speller, "word_lookup_start",
        G_CALLBACK (gdict_speller_lookup_start_cb), gdict);
  g_signal_connect (speller, "word_lookup_done",
        G_CALLBACK (gdict_speller_lookup_done_cb), gdict);
  g_signal_connect (speller, "word_not_found",
        G_CALLBACK (gdict_speller_not_found_cb), gdict);
  g_signal_connect (speller, "socket_error",
        G_CALLBACK (gdict_socket_error_cb), gdict);
	return widget;
}

static void
gdict_speller_error_cb (dict_command_t *command, DictStatusCode code,
                gchar *message, gpointer data)
{
	GtkWindow *speller;

	speller = GTK_WINDOW (data);
    
	if (code != DICT_SOCKET_ERROR) {
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new_with_markup (NULL,
							     GTK_DIALOG_DESTROY_WITH_PARENT,
							     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
							     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
							     _("Error invoking query"), message);

		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		if (command->cmd == C_MATCH)
			g_signal_emit (G_OBJECT (speller), signals[WORD_LOOKUP_DONE_SIGNAL], 0);
	}
	else {
		g_signal_emit (G_OBJECT (speller),
                	       signals[SOCKET_ERROR_SIGNAL], 0, message);
	}
}

void
gdict_speller_change_db (GDictSpeller *speller, gchar *new_db)
{
	GDictWindow *gdict = speller->gdict;
	gchar *text;

	g_free (speller->database);
	speller->database = g_strdup_printf ("%s", new_db);
	text = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
	gtk_editable_select_region (GTK_EDITABLE (gdict->word_entry), 0,
				    strlen (text));
	gdict_speller_lookup (speller->gdict, speller, text);
    	g_free (text);
}

void
gdict_speller_change_strategy (GDictSpeller *speller, gchar *new_strategy)
{
	GDictWindow *gdict = speller->gdict;
	gchar *text;

	g_free (speller->strat);   
	speller->strat = g_strdup_printf ("%s", new_strategy);
	text = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
	gtk_editable_select_region (GTK_EDITABLE (gdict->word_entry), 0,
				    strlen (text));
	gdict_speller_lookup (speller->gdict, speller, text);
    	g_free (text);
}
	
/* gdict_speller_word_status_cb
 *
 * Callback used when a status code has arrived over the link
 */
static void 
gdict_speller_word_status_cb (dict_command_t *command,
		      DictStatusCode code, 
		      int num_found,
		      gpointer data)
{
	GDictSpeller *speller;

	g_return_if_fail (GDICT_IS_SPELLER (data));

	speller = GDICT_SPELLER (data);

	if (!GTK_WIDGET_VISIBLE (GTK_WIDGET (speller)))
		return;

	if (code == DICT_STATUS_OK)
		g_signal_emit (G_OBJECT (speller), 
                	       signals[WORD_LOOKUP_DONE_SIGNAL], 0);
	else if (code == DICT_STATUS_NO_MATCH)
		g_signal_emit (G_OBJECT (speller), 
                	       signals[WORD_NOT_FOUND_SIGNAL], 0);
}

/* speller_set_strat_cb
 *
 * Sets the current search strategy to the one indicated
 */
static void
gdict_speller_set_strat_cb (GtkWidget *widget, GDictSpeller *speller)
{
	gchar *text;
	gchar *strat;
	GDictWindow *gdict = speller->gdict;

	g_return_if_fail (GDICT_IS_SPELLER (speller));

	strat = g_object_get_data (G_OBJECT (widget), "strat_name");
	speller->strat = strat;

	text = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
	gtk_editable_select_region (GTK_EDITABLE (gdict->word_entry), 0,
				    strlen (text));
	gdict_speller_lookup (speller->gdict, speller, text);
    	g_free (text);
}

/* gdict_speller_clear
 *
 * Clears the text in a speller and eliminates the current command structure
 */
void 
gdict_speller_clear (GDictSpeller *speller)
{
	GtkTreeModel *model;
	GDictWindow *gdict = speller->gdict;

	g_return_if_fail (GDICT_IS_SPELLER (speller));
    
	gtk_entry_set_text (GTK_ENTRY(gdict->word_entry), "");
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (speller->word_list));
	gtk_list_store_clear (GTK_LIST_STORE (model));
	speller->current_word = NULL;
    
	if (speller->spell_cmd) {
		dict_command_destroy (speller->spell_cmd);
		speller->spell_cmd = NULL;
	}
}

static void 
gdict_speller_add_word (GDictSpeller *speller, gchar *word)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (speller->word_list));
    
	gtk_list_store_insert (GTK_LIST_STORE (model), &iter, 0);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, word, -1);
    
	if (!speller->current_word)
		speller->current_word = word;
}

/* gdict_speller_word_data_cb
 *
 * Callback used when a new word has arrived over the link
 */
static void 
gdict_speller_word_data_cb (dict_command_t *command,
		    dict_res_t *res,
		    gpointer data)
{
	GDictSpeller *speller;

	g_return_if_fail (GDICT_IS_SPELLER (data));

	speller = GDICT_SPELLER (data);

	if (!GTK_WIDGET_VISIBLE (GTK_WIDGET (speller)))
		return;

	gdict_speller_add_word (speller, res->desc);
}

/* gdict_speller_lookup
 *
 * Sends the command to the server to commence looking up matches for a word
 * of a word and sets the callbacks so that the definition will be displayed
 * in this speller
 *
 * Returns 0 on success and -1 on command invocation error
 */
gint 
gdict_speller_lookup (GDictWindow *gdict, GDictSpeller *speller, gchar *text)
{
	g_return_val_if_fail (speller != NULL, -1);
	g_return_val_if_fail (GDICT_IS_SPELLER (speller), -1);
	g_return_val_if_fail (text != NULL, -1);

	if (speller->context == NULL) {		
		gdict_init_context (gdict);
		speller->context = gdict->context;
	}

	while (isspace (*text))
		text++;

	if (*text == '\0')
		return 0;
    
	g_signal_emit (G_OBJECT (speller), 
		       signals[WORD_LOOKUP_START_SIGNAL], 0);

	gdict_speller_clear (speller);
    
	gtk_entry_set_text (GTK_ENTRY(gdict->word_entry), text);
	gtk_editable_select_region (GTK_EDITABLE (gdict->word_entry), 
				    0, strlen (text));

	g_free (speller->database);
	speller->database = g_strdup (gdict->pref->database);
    
	speller->spell_cmd = dict_match_command_new (speller->database, speller->strat, text);
	speller->spell_cmd->error_notify_cb = gdict_speller_error_cb;
	speller->spell_cmd->status_notify_cb = gdict_speller_word_status_cb;
	speller->spell_cmd->data_notify_cb = gdict_speller_word_data_cb;
	speller->spell_cmd->user_data = speller;

	if (dict_command_invoke (speller->spell_cmd, speller->context) < 0)
		return -1;
    
	return 0;
}

/* gdict_speller_reset
 *
 * Reinvokes the search with a new context and preferences
 */
void
gdict_speller_reset (GDictSpeller *speller, dict_context_t *context)
{
	gchar *word;
        GDictWindow *gdict = speller->gdict;
    
	/* If we have a new context, we cannot trust that the search strategy
	 * we were using will still be present */
	if (gdict->context != speller->context) {
		speller->context = gdict->context;
		speller->strat = gdict->pref->dfl_strat;
	}

	/* Re-invoke current query only if there is a new database specified */
	if (context != speller->context ||
	    strcmp (speller->database, gdict->pref->database)) {
		if (speller->spell_cmd) {
			word = g_strdup (speller->spell_cmd->search_term);
			dict_command_destroy (speller->spell_cmd);
			speller->spell_cmd = NULL;
			gdict_speller_lookup (speller->gdict, speller, word);
			g_free (word);
		}
	}
}
