/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-printer.c: 
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Jose M. Celorio <chema@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */

#include "config.h"

#include <string.h>
#include <sys/types.h>
#include <dirent.h>

#include <gmodule.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>

#include "gpa-utils.h"
#include "gpa-value.h"
#include "gpa-reference.h"
#include "gpa-settings.h"
#include "gpa-model.h"
#include "gpa-printer.h"
#include "gpa-root.h"

/* GPAPrinter */

static void gpa_printer_class_init (GPAPrinterClass *klass);
static void gpa_printer_init (GPAPrinter *printer);

static void gpa_printer_finalize (GObject *object);

static gboolean  gpa_printer_verify    (GPANode *node);
static guchar *  gpa_printer_get_value (GPANode *node);
static GPANode * gpa_printer_get_child (GPANode *node, GPANode *previous_child);
static GPANode * gpa_printer_lookup    (GPANode *node, const guchar *path);
static void      gpa_printer_modified  (GPANode *node, guint flags);

static GPANode * gpa_printer_new_from_file (const gchar *filename);

static GPANodeClass *parent_class = NULL;

GType
gpa_printer_get_type (void) {
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPAPrinterClass),
			NULL, NULL,
			(GClassInitFunc) gpa_printer_class_init,
			NULL, NULL,
			sizeof (GPAPrinter),
			0,
			(GInstanceInitFunc) gpa_printer_init
		};
		type = g_type_register_static (GPA_TYPE_NODE, "GPAPrinter", &info, 0);
	}
	return type;
}

static void
gpa_printer_class_init (GPAPrinterClass *klass)
{
	GObjectClass *object_class;
	GPANodeClass *node_class;

	object_class = (GObjectClass *) klass;
	node_class = (GPANodeClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gpa_printer_finalize;

	node_class->verify    = gpa_printer_verify;
	node_class->get_value = gpa_printer_get_value;
	node_class->get_child = gpa_printer_get_child;
	node_class->lookup    = gpa_printer_lookup;
	node_class->modified  = gpa_printer_modified;
}

static void
gpa_printer_init (GPAPrinter *printer)
{
	printer->name     = NULL;
	printer->model    = NULL;
	printer->settings = NULL;
}

static void
gpa_printer_finalize (GObject *object)
{
	GPAPrinter *printer;

	printer = GPA_PRINTER (object);

	printer->name     = gpa_node_detach_unref (GPA_NODE (printer), GPA_NODE (printer->name));
	printer->settings = gpa_node_detach_unref (GPA_NODE (printer), GPA_NODE (printer->settings));
	printer->model    = gpa_node_detach_unref (GPA_NODE (printer), GPA_NODE (printer->model));

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gpa_printer_verify (GPANode *node)
{
	GPAPrinter *printer;

	printer = GPA_PRINTER (node);

	gpa_return_false_if_fail (GPA_NODE_ID_EXISTS (node));
	gpa_return_false_if_fail (printer->name);
	gpa_return_false_if_fail (gpa_node_verify (printer->name));
	gpa_return_false_if_fail (printer->settings);
	gpa_return_false_if_fail (gpa_node_verify (printer->settings));
	gpa_return_false_if_fail (printer->model);
	gpa_return_false_if_fail (gpa_node_verify (printer->model));
	
	return TRUE;
}

static guchar *
gpa_printer_get_value (GPANode *node)
{
	GPAPrinter *printer;

	printer = GPA_PRINTER (node);

	if (GPA_NODE_ID_EXISTS (node))
		return g_strdup (gpa_node_id (node));

	/* Should never happen (Chema) */
	g_warning ("Printer without an ID.\n");
	
	return NULL;
}

static GPANode *
gpa_printer_get_child (GPANode *node, GPANode *previous_child)
{
	GPAPrinter *printer;
	GPANode *child = NULL;

	printer = GPA_PRINTER (node);

	g_return_val_if_fail (printer->settings != NULL, NULL);
	g_return_val_if_fail (printer->model    != NULL, NULL);
	
	if (previous_child == NULL) {
		child = printer->name;
	} else if (previous_child == printer->name) {
		child = printer->settings;
	} else if (previous_child == printer->settings) {
		child = printer->model;
	}

	if (child)
		gpa_node_ref (child);

	return child;
}

static GPANode *
gpa_printer_lookup (GPANode *node, const guchar *path)
{
	GPAPrinter *printer;
	GPANode *child;

	printer = GPA_PRINTER (node);

	child = NULL;

	if (gpa_node_lookup_helper (GPA_NODE (printer->name),     path, "Name",     &child))
		return child;
	if (gpa_node_lookup_helper (GPA_NODE (printer->settings), path, "Settings", &child))
		return child;
	if (gpa_node_lookup_helper (GPA_NODE (printer->model),    path, "Model",    &child))
		return child;

	return NULL;
}

static void
gpa_printer_modified (GPANode *node, guint flags)
{
	GPAPrinter *printer;

	printer = GPA_PRINTER (node);

	if (printer->name && (GPA_NODE_FLAGS (printer->name) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (printer->name, 0);
	}
	if (printer->model && (GPA_NODE_FLAGS (printer->model) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (printer->model, 0);
	}
	if (printer->settings && (GPA_NODE_FLAGS (printer->settings) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (GPA_NODE (printer->settings), 0);
	}
}

/* Public methods */

/**
 * gpa_printer_new_from_tree:
 * @tree: The xml tree where to create the printer from
 * 
 * Create a GPAPrinter form an xml tree.
 * 
 * Return Value: a newly created GPAPrinter or NULL on error
 **/
static GPANode *
gpa_printer_new_from_tree (xmlNodePtr tree)
{
	GPAPrinter *printer = NULL;
	GPANode *name = NULL;
	GPANode *model = NULL;
	xmlChar *printer_id, *xml_version;
	xmlNodePtr xml_node;
	GSList *list = NULL;

	g_return_val_if_fail (tree != NULL, NULL);

	/* Check that tree is <Printer> */
	if (strcmp (tree->name, "Printer")) {
		g_warning ("file %s: line %d: Base node is <%s>, should be <Printer>",
			   __FILE__, __LINE__, tree->name);
		return NULL;
	}
	
	/* Check that printer has an Id */
	printer_id = xmlGetProp (tree, "Id");
	if (!printer_id) {
		g_warning ("file %s: line %d: Printer node does not have Id",
			   __FILE__, __LINE__);
		return NULL;
	}

	/* Check that the Printer definition version is 1.0 */
	xml_version = xmlGetProp (tree, "Version");
	if (!xml_version || strcmp (xml_version, "1.0")) {
		g_warning ("file %s: line %d: Wrong printer version %s, "
			   "should be 1.0", __FILE__, __LINE__, xml_version);
		xmlFree (printer_id);
		if (xml_version)
			xmlFree (xml_version);
		return NULL;
	}
	xmlFree (xml_version);

	/* Scan the XML for the child nodes that we care about */
	xml_node = tree->xmlChildrenNode;
	for (; xml_node != NULL; xml_node = xml_node->next) {
		
		/* <Name> */
		if (strcmp (xml_node->name, "Name") == 0) {
			xmlChar *printer_name;
			printer_name = xmlNodeGetContent (xml_node);
			if (!printer_name || !*printer_name) {
				g_warning ("Invalid Name node in printer tree (ID:%s)\n",
					   printer_id);
				xmlFree (printer_name);
				goto gpa_printer_new_from_tree_error;
			}
			name = gpa_value_new ("Name", printer_name);
			xmlFree (printer_name);
			continue;
		}

		/* <Model> */
		if (strcmp (xml_node->name, "Model") == 0) {
			xmlChar *model_id;
			model_id = xmlNodeGetContent (xml_node);
			if (model_id && *model_id) {
				model = gpa_model_get_by_id (model_id, FALSE);
				if (!model) {
					g_warning ("Could not find model by id:%s for printer:%s\n",
						   model_id, printer_id);
					goto gpa_printer_new_from_tree_error;
				}
				/* FIXME: Right now we are not loading the models on demand (Chema) */
				gpa_model_load (GPA_MODEL (model));
				xmlFree (model_id);
			}
			continue;
		}
		
		/* <Settings> */
		if (strcmp (xml_node->name, "Settings") == 0) {
			GPANode *settings;
			if (!model) {
				g_warning ("Settings without model in printer definition. "
					   "Model should come before any settings.");
				goto gpa_printer_new_from_tree_error;
			}
			settings = gpa_settings_new_from_model_and_tree (model, xml_node);
			if (!settings)
				g_warning ("Could not create settings for printer. (ID:%s)\n",
					   printer_id);
			else
				list = g_slist_prepend (list, settings);
			continue;
		}

	}

	if (!name || !model || !list) {
		g_warning ("Could not load printer (ID:%s). Model, Name or Settings are missing\n",
			   printer_id);
		goto gpa_printer_new_from_tree_error;
	}
	    
	printer = (GPAPrinter *) gpa_node_new (GPA_TYPE_PRINTER, printer_id);
	xmlFree (printer_id);
	
	/* Attach the name */
	printer->name = name;
	gpa_node_attach (GPA_NODE (printer), name);

	/* Attach the settings */
	printer->settings = (GPANode *) gpa_list_new (GPA_TYPE_SETTINGS, "Settings", TRUE);
	gpa_node_attach (GPA_NODE (printer), GPA_NODE (printer->settings));
		
	while (list) {
		gpa_list_prepend (GPA_LIST (printer->settings), GPA_NODE (list->data));
		list = g_slist_remove (list, list->data); /* Equivalent to list = list->next */
	}
	/* FIXME: Set the default Settings, for now use the first one (Chema) */
	gpa_list_set_default (GPA_LIST (printer->settings), GPA_LIST (printer->settings)->children);
	
        /* Attach the Model */
	printer->model = gpa_reference_new (model);
	printer->model->parent = GPA_NODE (printer);
	gpa_node_unref (GPA_NODE (model));
	
	return (GPANode *) printer;

gpa_printer_new_from_tree_error:
	if (name)
		gpa_node_unref (name);
	if (model)
		gpa_node_unref (model);
	while (list) {
		gpa_node_unref (GPA_NODE (list->data));
		list = g_slist_remove (list, list->data);
	}
	xmlFree (printer_id);

	return NULL;
}

static GPANode *
gpa_printer_new_from_file (const gchar *filename)
{
	GPANode *printer;
	xmlDocPtr doc;
	xmlNodePtr root;

	doc = xmlParseFile (filename);
	if (!doc)
		return NULL;
	root = doc->xmlRootNode;
	printer = NULL;
	if (!strcmp (root->name, "Printer")) {
		printer = gpa_printer_new_from_tree (root);
	}
	xmlFreeDoc (doc);
	return printer;
}

/* GPAPrinterList */
static void
gpa_printers_gone (gpointer data, GObject *gone)
{
	GPARoot *root;

	root = GPA_ROOT (gpa_root_get ());
	root->printers = NULL;
}

typedef struct _GpaModuleInfo GpaModuleInfo;
struct _GpaModuleInfo {
	GPAList *  (*printer_list_append) (GPAList *printers);
};

/**
 * gpa_printer_list_load_from_module:
 * @path: 
 * 
 * Load printers from a module
 **/
static gboolean
gpa_printer_list_load_from_module (GPAList *printers, const gchar *path)
{
	GpaModuleInfo info;
	GModule *handle;
	gboolean (*init) (GpaModuleInfo *info);
	gint ret = FALSE;

	handle = g_module_open (path, G_MODULE_BIND_LAZY);
	if (handle == NULL) {
		g_warning ("Can't g_module_open %s\n", path);
		return FALSE;
	}

	if (!g_module_symbol (handle, "gpa_module_init", (gpointer*) &init)) {
		g_warning ("Error. Module %s does not contains an init function\n", path);
		goto module_error;
	}
	
	if (((init) (&info)) == FALSE) {
		g_warning ("Could not initialize module %s\n", path);
		goto module_error;
	}

	(info.printer_list_append) (printers);
	ret = TRUE;
	
module_error:
	g_module_close (handle);

	return ret;
}

/**
 * gpa_printer_list_load_from_module_dir:
 * @list: 
 * @dir_path: 
 * 
 * Loads the printers from the gnome-print-modules. We load the module and find
 * for the gpa_printer_append symbol. If found, we asume this module is ok and we
 * call the function so that we get the printers from it.
 **/
static gboolean
gpa_printer_list_load_from_module_dir (GPAList *printers, const gchar *dir_path)
{
	DIR *d;
	struct dirent *entry;
	gint extension_len = strlen (LTDL_SHLIB_EXT);

	g_assert (extension_len > 0);

	if (!g_module_supported ()) {
		g_warning ("g_module is not supported on this platform an thus we can't "
			   "load dynamic printers\n");
		return FALSE;
	}
	
	d = opendir (dir_path);
	if (!d) {
		/* Not an error. since modules are optional */
		g_print ("Modules dir could not be opened [%s]\n", dir_path);
		return FALSE;
	}

	while ((entry = readdir (d)) != NULL) {
		if (strncmp (entry->d_name + strlen (entry->d_name) - extension_len,
			     LTDL_SHLIB_EXT, extension_len) == 0) {
			gchar *module_path = g_strconcat (dir_path, "/", entry->d_name, NULL);
			gpa_printer_list_load_from_module (printers, module_path);
			g_free (module_path);
		}
	}
	closedir (d);
	
	return TRUE;
}

/**
 * gpa_printer_list_load_from_dir:
 * @printers: 
 * @dirname: The path where to load printers from
 * 
 * Loads printers from a @dirname
 * 
 * Return Value: 
 **/
static gboolean
gpa_printer_list_load_from_dir (GPAList *printers, const gchar *dir_name)
{
	struct dirent *dent;
	DIR *dir;
	GHashTable *iddict;
	GSList *list = NULL;

	dir = opendir (dir_name);
	if (!dir)
		return FALSE;

	iddict = g_hash_table_new (g_str_hash, g_str_equal);
	while ((dent = readdir (dir))) {
		GPANode *printer;
		gchar *filename;
		gint len;

		/* Check if this directory entry interests us */
		len = strlen (dent->d_name);
		if (len < 9)
			continue;
		if (strcmp (dent->d_name + len - 8, ".printer"))
			continue;

		/* Load the printer */
		filename = g_strdup_printf ("%s/%s", dir_name, dent->d_name);
		printer = gpa_printer_new_from_file (filename);
		g_free (filename);

		if (!printer)
			continue;

		/* Verify that the printer we created is ok */
		if (!gpa_node_verify (GPA_NODE (printer))) {
			g_warning ("Not adding printer because it could not be verified\n");
			gpa_node_unref (GPA_NODE (printer));
			continue;
		}
		
		/* This checks that we don't load two printers with the same id. But..
		 *   
		 * FIXME: this does not check when we have a name clash from printers loaded
		 * from different sources. It only check for a clash when loading from this
		 * directory, we need a global check (Chema);
		 */
		if (g_hash_table_lookup (iddict, gpa_node_id (printer))) {
			gpa_node_unref (printer);
		} else {
			g_hash_table_insert (iddict, (gpointer) GPA_NODE_ID (printer), printer);
			list = g_slist_prepend (list, printer);
		}
	}
	g_hash_table_destroy (iddict);
	closedir (dir);

	while (list) {
		GPANode *printer;
		
		printer = GPA_NODE (list->data);
		printer->next = printers->children;
		printers->children = printer;
		printer->parent = GPA_NODE (printers);
		
		list = g_slist_remove (list, printer); /* Equiv. to list = list->next */
	}

	return TRUE;
}

/**
 * gpa_printer_list_load:
 * @void: 
 * 
 * Loads all of the printers available
 * 
 * Return Value: 
 **/
GPAList *
gpa_printer_list_load (void)
{
	GPARoot *root = GPA_ROOT (gpa_root_get ());
	GPAList *printers;

	g_return_val_if_fail (root != NULL,        NULL);
	g_return_val_if_fail (root->media != NULL, NULL);
	
	if (root->printers) {
		return (GPAList *) gpa_node_ref (GPA_NODE (root->printers));
	}

	printers = gpa_list_new (GPA_TYPE_PRINTER, "Printers", TRUE);
	g_object_weak_ref (G_OBJECT (printers), gpa_printers_gone, &printers);

	gpa_printer_list_load_from_dir (printers, GNOME_PRINT_DATA_DIR "/printers");
	gpa_printer_list_load_from_module_dir (printers, GPA_MODULES_DIR);

	/* We should always have printers, at least one loaded from code. (Chema) */
	g_assert (printers->children);

        /* FIXME: During parsing, please (Lauris) */
	/* FIXME: We are not setting the real default, just the first (Chema) */
	gpa_list_set_default (printers, printers->children);

	root->printers = gpa_node_attach (GPA_NODE (root), GPA_NODE (printers));
	
	/* We need a count ref of 2 when we leave this funtion. root->printers has one
	 * and one for the calling function to keep. (Chema)
	 */
	gpa_node_ref (GPA_NODE (printers));
	gpa_node_ref (GPA_NODE (printers));

	return printers;
}

/* Deprecated */

GPANode *
gpa_printer_get_default (void)
{
	GPAList *printers;
	GPANode *def;

	printers = gpa_printer_list_load ();

	if (printers->def) {
		def = GPA_REFERENCE_REFERENCE (printers->def);
	} else {
		def = printers->children;
	}

	if (def)
		gpa_node_ref (def);

	gpa_node_unref (GPA_NODE (printers));
	
	return def;
}

/**
 * gpa_printer_get_by_id:
 * @id: 
 * 
 * Get a printer from the globals printer list by printer ID
 * 
 * Return Value: 
 **/
GPANode *
gpa_printer_get_by_id (const guchar *id)
{
	GPAList *printers;
	GPANode *child = NULL;

	g_return_val_if_fail (id != NULL, NULL);
	g_return_val_if_fail (*id != '\0', NULL);

	printers = gpa_printer_list_load ();

	if (!printers)
		return NULL;
	
	for (child = printers->children; child != NULL; child = child->next) {
		g_assert (GPA_IS_PRINTER (child));
		if (GPA_NODE_ID_COMPARE (child, id))
			break;
	}

	if (child)
		gpa_node_ref (child);

	return child;
}

/**
 * gpa_printer_get_default_settings:
 * @printer: 
 * 
 * Returns a refcounted pointer to the default settings
 * 
 * Return Value: 
 **/
GPANode *
gpa_printer_get_default_settings (GPAPrinter *printer)
{
	GPANode *child;
	
	g_return_val_if_fail (printer != NULL, NULL);
	g_return_val_if_fail (GPA_IS_PRINTER (printer), NULL);

	/* FIXME: Are we getting default? or just the first? (Chema)
	 * FIXME: Also, set the default settings if none is marked as such (Chema)
	 */
	if (!printer->settings)
		return NULL;

	child = ((GPAList *)printer->settings)->children;

	if (!child)
		return NULL;

	g_assert (GPA_IS_SETTINGS (child));
	
	gpa_node_ref (child);

	return child;
}

/* FIXME: Thorough check (Lauris) */
#if 0
/* FIXME: Not yet used. (Chema) */
GPANode *
gpa_printer_new_from_model (GPAModel *model, const guchar *name)
{
	GPAList *printers;
	GPAPrinter *printer;
	GPANode *settings;
	guchar *id;

	g_return_val_if_fail (model != NULL, NULL);
	g_return_val_if_fail (GPA_IS_MODEL (model), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (*name != '\0', NULL);

	printers = gpa_printer_list_load ();

	id = gpa_id_new (GPA_NODE_ID (model));
	printer = (GPAPrinter *) gpa_node_new (GPA_TYPE_PRINTER, id);
	g_free (id);

	/* Name */
	printer->name = gpa_node_attach (GPA_NODE (printer), gpa_value_new ("Name", name));

	/* Settings */
	/* fixme: Implement helper */
	/* fixme: defaults again */
	/* FIXME: (Chema) */
	printer->settings = GPA_LIST (gpa_node_attach (GPA_NODE (printer), gpa_list_new (GPA_TYPE_SETTINGS, TRUE)));
	settings = gpa_settings_new_from_model (GPA_NODE (model), "Default");
	gpa_list_add_child (printer->settings, settings, NULL);
	gpa_node_unref (settings);
	/* fixme: */
	gpa_list_set_default (printer->settings, settings);

	/* Model */
	printer->model = gpa_node_attach (GPA_NODE (printer), gpa_reference_new (GPA_NODE (model)));

	gpa_list_add_child (printers, GPA_NODE (printer), NULL);

	gpa_node_unref (gpa_node_cache (GPA_NODE (printers)));

	return (GPANode *) printer;
}
#endif

#if 0
/* Not beeing used yet. We are not configuring printers so we don't need
 * it right now. Chema
 */
gboolean
gpa_printer_save (GPAPrinter *printer)
{
	xmlDocPtr doc;
	xmlNodePtr root, xmln;
	GPANode *child;
	guchar *filename;

	g_return_val_if_fail (printer != NULL, FALSE);
	g_return_val_if_fail (GPA_IS_PRINTER (printer), FALSE);

	g_return_val_if_fail (gpa_node_verify (GPA_NODE (printer)), FALSE);

	doc = xmlNewDoc ("1.0");
	root = xmlNewDocNode (doc, NULL, "Printer", NULL);
	xmlSetProp (root, "Version", "1.0");
	xmlSetProp (root, "Id", GPA_NODE_ID (printer));
	xmlDocSetRootElement (doc, root);

	xmln = xmlNewChild (root, NULL, "Name", GPA_VALUE (printer->name)->value);

	xmln = xmlNewChild (root, NULL, "Model", GPA_NODE_ID (GPA_REFERENCE (printer->model)->ref));

	for (child = printer->settings->children; child != NULL; child = child->next) {
		xmln = gpa_settings_write (doc, child);
		if (xmln)
			xmlAddChild (root, xmln);
	}

	filename = g_strdup_printf ("%s/.gnome/printers/%s.printer", g_get_home_dir (), GPA_NODE_ID (printer));
	xmlSaveFile (filename, doc);
	g_free (filename);

	xmlFreeDoc (doc);

	return TRUE;
}
#endif

/* Convenience functions */
GPAPrinter *
gpa_printer_new_full (const gchar *printer_id, const gchar *printer_name)
{
	GPAPrinter *printer;

	/* Setup the Printer */
	printer = (GPAPrinter *) gpa_node_new (GPA_TYPE_PRINTER, printer_id);
	printer->name = gpa_value_new ("Name", printer_name);

	return printer;
}


