#define __GPA_ROOT_C__

/*
 * This file is part of libgnomeprint 2
 *
 * Libgnomeprint is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * Libgnomeprint is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the libgnomeprint; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors :
 *   Jose M. Celorio <chema@ximian.com>
 *   Lauris Kaplinski <lauris@ximian.com>
 *
 * Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */

#include <string.h>
#include "gpa-utils.h"
#include "gpa-vendor.h"
#include "gpa-printer.h"
#include "gpa-media.h"
#include "gpa-root.h"

/* GPARoot */

static void gpa_root_class_init (GPARootClass *klass);
static void gpa_root_init (GPARoot *root);

static void gpa_root_finalize (GObject *object);

static GPANode *gpa_root_get_child (GPANode *node, GPANode *ref);
static GPANode *gpa_root_lookup (GPANode *node, const guchar *path);
static void gpa_root_modified (GPANode *node, guint flags);

static GPANode *root_instance = NULL;

/* Helpers */

static GPANodeClass *parent_class;

GType
gpa_root_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPARootClass),
			NULL, NULL,
			(GClassInitFunc) gpa_root_class_init,
			NULL, NULL,
			sizeof (GPARoot),
			0,
			(GInstanceInitFunc) gpa_root_init
		};
		type = g_type_register_static (GPA_TYPE_NODE, "GPARoot", &info, 0);
	}
	return type;
}

static void
gpa_root_class_init (GPARootClass *klass)
{
	GObjectClass *object_class;
	GPANodeClass *node_class;

	object_class = (GObjectClass*) klass;
	node_class = (GPANodeClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gpa_root_finalize;

	node_class->get_child = gpa_root_get_child;
	node_class->lookup = gpa_root_lookup;

	node_class->modified = gpa_root_modified;
}

static void
gpa_root_init (GPARoot *root)
{
	root_instance = GPA_NODE (root);

	root->media = gpa_node_attach (GPA_NODE (root), gpa_media_load ());
	root->vendors = gpa_node_attach (GPA_NODE (root), GPA_NODE (gpa_vendor_list_load ()));
	root->printers = gpa_node_attach (GPA_NODE (root), GPA_NODE (gpa_printer_list_load ()));
}

static void
gpa_root_finalize (GObject *object)
{
	GPARoot *root;

	root = (GPARoot *) object;

	if (root->vendors) {
		root->vendors = gpa_node_detach_unref (GPA_NODE (root), GPA_NODE (root->vendors));
	}

	if (root->printers) {
		root->printers = gpa_node_detach_unref (GPA_NODE (root), GPA_NODE (root->printers));
	}

	if (root->media) {
		root->media = gpa_node_detach_unref (GPA_NODE (root), root->media);
	}

	root_instance = NULL;

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static GPANode *
gpa_root_get_child (GPANode *node, GPANode *ref)
{
	GPARoot *root;

	root = GPA_ROOT (node);

	g_return_val_if_fail (root->vendors != NULL, NULL);
	g_return_val_if_fail (root->printers != NULL, NULL);
	g_return_val_if_fail (root->media != NULL, NULL);
	g_return_val_if_fail (!ref || (ref->parent == node), NULL);

	if (!ref) {
		gpa_node_ref (root->vendors);
		return root->vendors;
	} else if (ref == root->vendors) {
		gpa_node_ref (root->printers);
		return root->printers;
	} else if (ref == root->printers) {
		gpa_node_ref (root->media);
		return root->media;
	}

	return NULL;
}

static GPANode *
gpa_root_lookup (GPANode *node, const guchar *path)
{
	GPARoot *root;
	GPANode *child;

	root = GPA_ROOT (node);

	child = NULL;

	if (gpa_node_lookup_ref (&child, GPA_NODE (root->media), path, "Media")) return child;
	if (gpa_node_lookup_ref (&child, GPA_NODE (root->vendors), path, "Vendors")) return child;
	if (gpa_node_lookup_ref (&child, GPA_NODE (root->printers), path, "Printers")) return child;

	return NULL;
}

static void
gpa_root_modified (GPANode *node, guint flags)
{
	GPARoot *root;

	root = GPA_ROOT (node);

	if (root->vendors && (GPA_NODE_FLAGS (root->vendors) & GPA_MODIFIED_FLAG)) {
		gpa_node_emit_modified (GPA_NODE (root->vendors), 0);
	}
	if (root->printers && (GPA_NODE_FLAGS (root->printers) & GPA_MODIFIED_FLAG)) {
		gpa_node_emit_modified (GPA_NODE (root->printers), 0);
	}
}

GPARoot *
gpa_root_get (void)
{
	GPARoot *root;

	root = gpa_root ();

	gpa_node_ref (GPA_NODE (root));

	return root;
}

GPARoot *
gpa_root (void)
{
	if (!root_instance) {
		gpa_node_new (GPA_TYPE_ROOT, "Globals");
	}

	return (GPARoot *) root_instance;
}

