/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright(C) 2000, 2001 Edgar Denny(e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "food.h"
#include "recipe.h"
#include "plan.h"
#include "recipe_win.h"
#include "plan_win.h"
#include "edit_recipe_dlg.h"

typedef struct Dialog_s {
	GtkButton *ok_button;
	GtkButton *cancel_button;

	GtkEntry *recipe_entry;
	GtkEntry *no_serv_entry;

	GtkWidget *win;
} Dialog_t;

static GladeXML *xml = NULL;
static void load_xml( void);
static void connect_signals( Dialog_t *);
static Dialog_t *create_dialog( void);
static Dialog_t *get_dialog( void);

/* callbacks. */
static void on_ok_button_released( GtkButton *, gpointer);
static void on_cancel_button_released( GtkButton *, gpointer);

/* create the dialog struct. */
static Dialog_t *
create_dialog()
{
	static Dialog_t *dialog = NULL;

	if ( !xml) load_xml();

	if ( !dialog) {
		dialog = g_new( Dialog_t, 1);

		dialog->ok_button = GTK_BUTTON( 
			glade_xml_get_widget( xml, "ok_button"));
		dialog->cancel_button = GTK_BUTTON(
			glade_xml_get_widget( xml, "cancel_button"));

		dialog->recipe_entry = GTK_ENTRY( 
			glade_xml_get_widget( xml, "recipe_entry"));
		dialog->no_serv_entry = GTK_ENTRY( 
			glade_xml_get_widget( xml, "no_serv_entry"));

		dialog->win = glade_xml_get_widget( xml, 
			"gnutr_edit_recipe_dlg");
	}

	return dialog;	
}

static Dialog_t *
get_dialog()
{
	return create_dialog();
}

/* load the glade xml if not already loaded. */
static void
load_xml()
{
	static gboolean loaded_xml = FALSE;

	/* load the glade interface. */
	if ( !loaded_xml) {
		xml = glade_xml_new( 
			GNUTRITION_GLADEDIR "/edit_recipe_dlg.glade", NULL);
		loaded_xml = TRUE;

		if ( xml) {
			Dialog_t *dialog;

			dialog = create_dialog();
			connect_signals( dialog);
		} else {
			g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
				"edit_recipe_dlg: cannot load glade file\n");
			return;
		}
	}
}

/* connect the signals. */
static void
connect_signals( Dialog_t *dialog)
{
	gtk_signal_connect( GTK_OBJECT( dialog->ok_button), "released", 
		GTK_SIGNAL_FUNC( on_ok_button_released), (gpointer)dialog);

	gtk_signal_connect( GTK_OBJECT( dialog->cancel_button), "released", 
		GTK_SIGNAL_FUNC( on_cancel_button_released), (gpointer)dialog);
}

/* reopens the search dialog window, so an ingredient can be edited,
 * when the edit button is released in the AppBar on the recipe window. */
void
gnutr_show_edit_recipe_dlg( char *recipe_name,
                            char *no_serv)
{
	Dialog_t *dialog;

	if (!xml) load_xml();

	g_return_if_fail( recipe_name != NULL);
	g_return_if_fail( no_serv != NULL);

	dialog = get_dialog();

	gtk_entry_set_text( dialog->recipe_entry, recipe_name);
	gtk_entry_set_text( dialog->no_serv_entry, no_serv);

	gnome_dialog_close_hides( GNOME_DIALOG( dialog->win), TRUE);
	gtk_widget_show( dialog->win);
}

/* callback when the "OK" button is released. */
static void 
on_ok_button_released( GtkButton *button, 
                       gpointer   data)
{
	char *date;
	char *time_of_day;
	char *no_serv;
	GtkCList *clist;
	enum { FOOD_ROW, RECIPE_ROW } type = RECIPE_ROW;
	int selected_row;
	char *hidden_data;
	Dialog_t *dialog;

	dialog = (Dialog_t *)data;

	no_serv = gtk_entry_get_text( dialog->no_serv_entry);

	clist = gnutr_get_plan_win_clist();
	selected_row = gnutr_get_plan_selected_row();
	hidden_data = (char *)gtk_clist_get_row_data( clist, selected_row);

	date = gnutr_get_date();
	time_of_day = gnutr_get_time_of_day();

	/* modify temporary plan database table. */
	gnutr_edit_plan_temp_db( date, time_of_day, type, 
		NULL, NULL, no_serv, hidden_data);

	/* modify the plan window clist. */
	gtk_clist_set_text( clist, selected_row, 1, no_serv);

	gtk_widget_hide( dialog->win);
}

/* callback when the "Cancel" button is released. */
static void 
on_cancel_button_released( GtkButton *button, 
                           gpointer   data)
{
	Dialog_t *dialog;

	dialog = (Dialog_t *)data;
	gtk_widget_hide( dialog->win);
}
