use strict;
use warnings;

package Footprintless::Database::DatabasePlugin::Command::db::restore;
$Footprintless::Database::DatabasePlugin::Command::db::restore::VERSION = '0.01';
# ABSTRACT: restore a database from a backup
# PODNAME: Footprintless::Database::DatabasePlugin::Command::db::restore

use parent qw(Footprintless::App::Action);

use Carp;
use Footprintless::App -ignore;
use Log::Any;

my $logger = Log::Any->get_logger();

sub execute {
    my ( $self, $footprintless, $coordinate, $opts, $args ) = @_;

    $logger->debugf( 'options=%s', $opts );
    eval {
        $self->{db}->connect();
        $self->{db}->restore( ( delete( $opts->{file} ) || \*STDIN ), %$opts );
    };
    my $error = $@;
    $self->{db}->disconnect();
    die($error) if ($error);

    $logger->info('Done...');
}

sub opt_spec {
    return (
        [ 'clean',          'drop all data on the target before restoring' ],
        [ 'file=s',         'the input file' ],
        [ 'ignore-deny',    'will allow running on denied coordinates' ],
        [ 'post-restore=s', 'a sql script to run after the restore' ],
    );
}

sub validate_args {
    my ( $self, $footprintless, $coordinate, $opts, $args ) = @_;

    my $command_helper = $footprintless->db_command_helper();
    croak("destination [$coordinate] not allowed")
        unless $opts->{ignore_deny}
        || $command_helper->allowed_destination($coordinate);

    eval { $self->{db} = $footprintless->db($coordinate); };
    croak("invalid coordinate [$coordinate]: $@") if ($@);
}

1;

__END__

=pod

=head1 NAME

Footprintless::Database::DatabasePlugin::Command::db::restore - restore a database from a backup

=head1 VERSION

version 0.01

=head1 AUTHOR

Lucas Theisen <lucastheisen@pastdev.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Lucas Theisen.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<Footprintless::Database::DatabasePlugin|Footprintless::Database::DatabasePlugin>

=back

=for Pod::Coverage execute opt_spec usage_desc validate_args

=cut
