
use strict;
use warnings;
use Test::More;


use Crypt::NaCl::Sodium qw(:utils);

my $crypto_stream = Crypt::NaCl::Sodium->stream();

my @tests = (
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "0000000000000000",
        expected => "76b8e0ada0f13d90405d6ae55386bd28bdd219b8a08ded1aa836efcc8b770dc7da41597c5157488d7724e03fb8d84a376a43b8f41518a11cc387b669b2ee65869f07e7be5551387a98ba977c732d080dcb0f29a048e3656912c6533e32ee7aed29b721769ce64e43d57133b074d839d531ed1f28510afb45ace10a1f4b794d6f2d09a0e663266ce1ae7ed1081968a0758e718e997bd362c6b0c34634a9a0b35d",
    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000001",
        nonce_hex => "0000000000000000",
        expected => "4540f05a9f1fb296d7736e7b208e3c96eb4fe1834688d2604f450952ed432d41bbe2a0b6ea7566d2a5d1e7e20d42af2c53d792b1c43fea817e9ad275ae5469633aeb5224ecf849929b9d828db1ced4dd832025e8018b8160b82284f3c949aa5a8eca00bbb4a73bdad192b5c42f73f2fd4e273644c8b36125a64addeb006c13a096d68b9ff7b57e7090f880392effd5b297a83bbaf2fbe8cf5d4618965e3dc776",

    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "0000000000000001",
        expected => "de9cba7bf3d69ef5e786dc63973f653a0b49e015adbff7134fcb7df137821031e85a050278a7084527214f73efc7fa5b5277062eb7a0433e445f41e31afab757283547e3d3d30ee0371c1e6025ff4c91b794a291cf7568d48ff84b37329e2730b12738a072a2b2c7169e326fe4893a7b2421bb910b79599a7ce4fbaee86be427c5ee0e8225eb6f48231fd504939d59eac8bd106cc138779b893c54da8758f62a",

    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "0100000000000000",
        expected => "ef3fdfd6c61578fbf5cf35bd3dd33b8009631634d21e42ac33960bd138e50d32111e4caf237ee53ca8ad6426194a88545ddc497a0b466e7d6bbdb0041b2f586b5305e5e44aff19b235936144675efbe4409eb7e8e5f1430f5f5836aeb49bb5328b017c4b9dc11f8a03863fa803dc71d5726b2b6b31aa32708afe5af1d6b690584d58792b271e5fdb92c486051c48b79a4d48a109bb2d0477956e74c25e93c3c2",

    },
    {
        key_hex => "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f",
        nonce_hex => "0001020304050607",
        expected => "f798a189f195e66982105ffb640bb7757f579da31602fc93ec01ac56f85ac3c134a4547b733b46413042c9440049176905d3be59ea1c53f15916155c2be8241a38008b9a26bc35941e2444177c8ade6689de95264986d95889fb60e84629c9bd9a5acb1cc118be563eb9b3a4a472f82e09a7e778492b562ef7130e88dfe031c79db9d4f7c7a899151b9a475032b63fc385245fe054e3dd5a97a5f576fe064025",

    },

);

my @tests_ietf = (
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "000000000000000000000000",
        expected => "76b8e0ada0f13d90405d6ae55386bd28bdd219b8a08ded1aa836efcc8b770dc7da41597c5157488d7724e03fb8d84a376a43b8f41518a11cc387b669b2ee65869f07e7be5551387a98ba977c732d080dcb0f29a048e3656912c6533e32ee7aed29b721769ce64e43d57133b074d839d531ed1f28510afb45ace10a1f4b794d6f2d09a0e663266ce1ae7ed1081968a0758e718e997bd362c6b0c34634a9a0b35d",
        ic => 0,
    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "000000000000000000000000",
        expected => "9f07e7be5551387a98ba977c732d080dcb0f29a048e3656912c6533e32ee7aed29b721769ce64e43d57133b074d839d531ed1f28510afb45ace10a1f4b794d6f2d09a0e663266ce1ae7ed1081968a0758e718e997bd362c6b0c34634a9a0b35d012737681f7b5d0f281e3afde458bc1e73d2d313c9cf94c05ff3716240a248f21320a058d7b3566bd520daaa3ed2bf0ac5b8b120fb852773c3639734b45c91a4",
        ic => 1,
    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000001",
        nonce_hex => "000000000000000000000000",
        expected => "3aeb5224ecf849929b9d828db1ced4dd832025e8018b8160b82284f3c949aa5a8eca00bbb4a73bdad192b5c42f73f2fd4e273644c8b36125a64addeb006c13a096d68b9ff7b57e7090f880392effd5b297a83bbaf2fbe8cf5d4618965e3dc776cd430d9b4e7eda8a767fb0e860319aadb5fd96a855de1fbfc92cb0489190cfdd87da6dbf1f736a2d499941ca097e5170bd685578611323120cebf296181ed4f5",
        ic => 1,
    },
    {
        key_hex => "00ff000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "000000000000000000000000",
        expected => "72d54dfbf12ec44b362692df94137f328fea8da73990265ec1bbbea1ae9af0ca13b25aa26cb4a648cb9b9d1be65b2c0924a66c54d545ec1b7374f4872e99f096bf74dbd52cc4fc95ceb6097fe5e65358c9dbc0a5ecbf7894a132a9a54ae3e951f2e9f209aa9c3d9a877ac9dab62433d2961a17d103e455dfb7337c90f6857aad233065955a212b5c7a8eab4dc8a629e5b6b8ba914afd06de7177054b33d21c96",
        ic => 2,
    },
    {
        key_hex => "0000000000000000000000000000000000000000000000000000000000000000",
        nonce_hex => "000000000000000000000002",
        expected => "c2c64d378cd536374ae204b9ef933fcd1a8b2288b3dfa49672ab765b54ee27c78a970e0e955c14f3a88e741b97c286f75f8fc299e8148362fa198a39531bed6d1a91288c874ec254f322c2a197340c55bb3e9b3998f7de2309486a0bb494abd20c9c5ef99c1370d61e77f408ac5514f49202bcc6828d45409d2d1416f8ae106b06ebd2541256264fa415bd54cb12e1d4449ed85299a1b7a249b75ff6c89b2e3f",
        ic => 0,
    },
    {
        key_hex => "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f",
        nonce_hex => "000000090000004a00000000",
        expected => "10f1e7e4d13b5915500fdd1fa32071c4c7d1f4c733c068030422aa9ac3d46c4ed2826446079faa0914c2d705d98b02a2b5129cd1de164eb9cbd083e8a2503c4e0a88837739d7bf4ef8ccacb0ea2bb9d69d56c394aa351dfda5bf459f0a2e9fe8e721f89255f9c486bf21679c683d4f9c5cf2fa27865526005b06ca374c86af3bdcbfbdcb83be65862ed5c20eae5a43241d6a92da6dca9a156be25297f51c2718",
        ic => 1,
    },
);

my ($key, $nonce, $ic, $bytes, $bytes_hex, $msg, $xored_hex);
for my $test ( @tests ) {
    $key = hex2bin($test->{key_hex});
    $nonce = hex2bin($test->{nonce_hex});

    $bytes = $crypto_stream->chacha20_bytes(160, $nonce, $key);
    $bytes_hex = bin2hex($bytes);

    is($bytes_hex, $test->{expected}, "Got expected bytes: $bytes_hex");
}

is($crypto_stream->chacha20_bytes(0, $nonce, $key), '');
is($crypto_stream->chacha20_xor('', $nonce, $key), '');
is($crypto_stream->chacha20_xor('', $nonce, $key), '');
is($crypto_stream->chacha20_xor_ic('', $nonce, 1, $key), '');


$msg = "\x42" x 160;

$msg = $crypto_stream->chacha20_xor($msg, $nonce, $key);
$xored_hex = bin2hex($msg);
is($xored_hex, "b5dae3cbb3d7a42bc0521db92649f5373d15dfe15440bed1ae43ee14ba18818376e616393179040372008b06420b552b4791fc1ba85e11b31b54571e69aa66587a42c9d864fe77d65c6606553ec89c24cb9cd7640bc49b1acbb922aa046b8bffd818895e835afc147cfbf1e6e630ba6c4be5a53a0b69146cb5514cca9da27385dffb96b585eadb5759d8051270f47d81c7661da216a19f18d5e7b734bc440267", "chacha20_xor returns expected: $xored_hex");

$msg = $crypto_stream->chacha20_xor_ic($msg, $nonce, 0, $key);
$xored_hex = bin2hex($msg);
is($xored_hex,
    "42424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242",
    "chacha20_xor_ic returns expected: $xored_hex");


$msg = $crypto_stream->chacha20_xor_ic($msg, $nonce, 1, $key);
$xored_hex = bin2hex($msg);
is($xored_hex,
    "7a42c9d864fe77d65c6606553ec89c24cb9cd7640bc49b1acbb922aa046b8bffd818895e835afc147cfbf1e6e630ba6c4be5a53a0b69146cb5514cca9da27385dffb96b585eadb5759d8051270f47d81c7661da216a19f18d5e7b734bc440267918c466e1428f08745f37a99c77c7f2b1b244bd4162e8b86e4a8bf85358202954ced04b52fef7b3ba787744e715554285ecb0ed6e133c528d69d346abc0ce8b0",
    "chacha20_xor_ic returns expected: $xored_hex");

for my $test ( @tests_ietf ) {
    $key = hex2bin($test->{key_hex});
    $nonce = hex2bin($test->{nonce_hex});
    $ic = $test->{ic};

    $msg = "\0" x 160;

    $bytes = $crypto_stream->chacha20_ietf_xor_ic($msg, $nonce, $ic, $key);
    $bytes_hex = bin2hex($bytes);

    is($bytes_hex, $test->{expected}, "Got expected bytes: $bytes_hex");
}

is($crypto_stream->chacha20_ietf_bytes(0, $nonce, $key), '');
is($crypto_stream->chacha20_ietf_xor('', $nonce, $key), '');
is($crypto_stream->chacha20_ietf_xor('', $nonce, $key), '');
is($crypto_stream->chacha20_ietf_xor_ic('', $nonce, 1, $key), '');

$msg = "\x42" x 160;

$msg = $crypto_stream->chacha20_ietf_xor($msg, $nonce, $key);
$xored_hex = bin2hex($msg);
is($xored_hex, "c89ed3bfddb6b2b7594def12bd579475a64cbfe0448e1085c1e50042127e57c08fda71743f4816973f7edcdbcd0b4ca4dee10e5dbbab7be517c6876f2b48779652b3a5a693791b57124d9f5de16233868593b68571822a414660e8d881962e0c90c0260445dde84b568095479bc940e0f750de939c540cfb8992c1aae0127e0c48cac1357b95fd0cba8eeef2a869fb94df1481d6e8775fbfe7fd07dd486cddaa", "chacha20_ietf_xor returns expected: $xored_hex");

$msg = $crypto_stream->chacha20_ietf_xor_ic($msg, $nonce, 0, $key);
$xored_hex = bin2hex($msg);
is($xored_hex,
    "42424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242424242",
    "chacha20_ietf_xor_ic returns expected: $xored_hex");


$msg = $crypto_stream->chacha20_ietf_xor_ic($msg, $nonce, 1, $key);
$xored_hex = bin2hex($msg);
is($xored_hex,
    "52b3a5a693791b57124d9f5de16233868593b68571822a414660e8d881962e0c90c0260445dde84b568095479bc940e0f750de939c540cfb8992c1aae0127e0c48cac1357b95fd0cba8eeef2a869fb94df1481d6e8775fbfe7fd07dd486cddaaa563bad017bb86c4fd6325de2a7f0dde1eb0b865c4176442194488750ec4ed799efdff89c1fc27c46c97804cec1801665f28d0982f88d85729a010d5b75e655a",
    "chacha20_ietf_xor_ic returns expected: $xored_hex");

done_testing();

