######################################################################
#
# KEIS83_by_Unicode_CultiCoLtd.pl
#
# Copyright (c) 2018 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################

# KEIS code book, Culti Co.,Ltd.
# http://www.culti.co.jp/2016/02/01/%e3%82%ab%e3%83%ab%e3%83%81%e7%99%ba%e8%a1%8c%e6%9b%b8%e7%b1%8d/

use strict;

my %KEIS83_by_Unicode_CultiCoLtd = ();
my %Unicode_by_KEIS83_CultiCoLtd = ();

while (<DATA>) {
    chomp;
    my($keis83, $bushu, $kakusu, $menkuten, $jis, $unicode) = split(/\t/, $_);

# print STDERR "($keis83)($bushu)($kakusu)($menkuten)($jis)($unicode)\n";

    if ($keis83  !~ /^[0123456789ABCDEF]+$/) {
        next;
    }
    if ($unicode !~ /^[0123456789ABCDEF]+$/) {
        next;
    }

    # ignore Private Use Area
    if (('E000' le $unicode) and ($unicode le 'F8FF')) {
        next;
    }

    if (defined $KEIS83_by_Unicode_CultiCoLtd{$unicode}) {
        die;
    }
    if (defined $Unicode_by_KEIS83_CultiCoLtd{$keis83}) {
        die;
    }

    $KEIS83_by_Unicode_CultiCoLtd{$unicode} = $keis83;
    $Unicode_by_KEIS83_CultiCoLtd{$keis83}  = $unicode;
}

sub KEIS83_by_Unicode_CultiCoLtd {
    my($unicode) = @_;
    return $KEIS83_by_Unicode_CultiCoLtd{$unicode};
}

sub keys_of_KEIS83_by_Unicode_CultiCoLtd {
    return keys %KEIS83_by_Unicode_CultiCoLtd;
}

sub values_of_KEIS83_by_Unicode_CultiCoLtd {
    return values %KEIS83_by_Unicode_CultiCoLtd;
}

1;

# KEISR[h	ԍ	搔	ʋ_ԍ	JISR[h	Unicode
__DATA__
A1A1			01-01-01	2121	3000
A1A2			01-01-02	2122	3001
A1A3			01-01-03	2123	3002
A1A4			01-01-04	2124	FF0C
A1A5			01-01-05	2125	FF0E
A1A6			01-01-06	2126	30FB
A1A7			01-01-07	2127	FF1A
A1A8			01-01-08	2128	FF1B
A1A9			01-01-09	2129	FF1F
A1AA			01-01-10	212A	FF01
A1AB			01-01-11	212B	309B
A1AC			01-01-12	212C	309C
A1AD			01-01-13	212D	00B4
A1AE			01-01-14	212E	FF40
A1AF			01-01-15	212F	00A8
A1B0			01-01-16	2130	FF3E
A1B1			01-01-17	2131	FFE3
A1B2			01-01-18	2132	FF3F
A1B3			01-01-19	2133	30FD
A1B4			01-01-20	2134	30FE
A1B5			01-01-21	2135	309D
A1B6			01-01-22	2136	309E
A1B7			01-01-23	2137	3003
A1B8			01-01-24	2138	4EDD
A1B9			01-01-25	2139	3005
A1BA			01-01-26	213A	3006
A1BB			01-01-27	213B	3007
A1BC			01-01-28	213C	30FC
A1BD			01-01-29	213D	2015
A1BE			01-01-30	213E	2010
A1BF			01-01-31	213F	FF0F
A1C0			01-01-32	2140	FF3C
A1C1			01-01-33	2141	301C
A1C2			01-01-34	2142	2016
A1C3			01-01-35	2143	FF5C
A1C4			01-01-36	2144	2026
A1C5			01-01-37	2145	2025
A1C6			01-01-38	2146	2018
A1C7			01-01-39	2147	2019
A1C8			01-01-40	2148	201C
A1C9			01-01-41	2149	201D
A1CA			01-01-42	214A	FF08
A1CB			01-01-43	214B	FF09
A1CC			01-01-44	214C	3014
A1CD			01-01-45	214D	3015
A1CE			01-01-46	214E	FF3B
A1CF			01-01-47	214F	FF3D
A1D0			01-01-48	2150	FF5B
A1D1			01-01-49	2151	FF5D
A1D2			01-01-50	2152	3008
A1D3			01-01-51	2153	3009
A1D4			01-01-52	2154	300A
A1D5			01-01-53	2155	300B
A1D6			01-01-54	2156	300C
A1D7			01-01-55	2157	300D
A1D8			01-01-56	2158	300E
A1D9			01-01-57	2159	300F
A1DA			01-01-58	215A	3010
A1DB			01-01-59	215B	3011
A1DC			01-01-60	215C	FF0B
A1DD			01-01-61	215D	2212
A1DE			01-01-62	215E	00B1
A1DF			01-01-63	215F	00D7
A1E0			01-01-64	2160	00F7
A1E1			01-01-65	2161	FF1D
A1E2			01-01-66	2162	2260
A1E3			01-01-67	2163	FF1C
A1E4			01-01-68	2164	FF1E
A1E5			01-01-69	2165	2266
A1E6			01-01-70	2166	2267
A1E7			01-01-71	2167	221E
A1E8			01-01-72	2168	2234
A1E9			01-01-73	2169	2642
A1EA			01-01-74	216A	2640
A1EB			01-01-75	216B	00B0
A1EC			01-01-76	216C	2032
A1ED			01-01-77	216D	2033
A1EE			01-01-78	216E	2103
A1EF			01-01-79	216F	FFE5
A1F0			01-01-80	2170	FF04
A1F1			01-01-81	2171	FFE0
A1F2			01-01-82	2172	FFE1
A1F3			01-01-83	2173	FF05
A1F4			01-01-84	2174	FF03
A1F5			01-01-85	2175	FF06
A1F6			01-01-86	2176	FF0A
A1F7			01-01-87	2177	FF20
A1F8			01-01-88	2178	00A7
A1F9			01-01-89	2179	2606
A1FA			01-01-90	217A	2605
A1FB			01-01-91	217B	25CB
A1FC			01-01-92	217C	25CF
A1FD			01-01-93	217D	25CE
A1FE			01-01-94	217E	25C7
A2A1			01-02-01	2221	25C6
A2A2			01-02-02	2222	25A1
A2A3			01-02-03	2223	25A0
A2A4			01-02-04	2224	25B3
A2A5			01-02-05	2225	25B2
A2A6			01-02-06	2226	25BD
A2A7			01-02-07	2227	25BC
A2A8			01-02-08	2228	203B
A2A9			01-02-09	2229	3012
A2AA			01-02-10	222A	2192
A2AB			01-02-11	222B	2190
A2AC			01-02-12	222C	2191
A2AD			01-02-13	222D	2193
A2AE			01-02-14	222E	3013
A2BA			01-02-26	223A	2208
A2BB			01-02-27	223B	220B
A2BC			01-02-28	223C	2286
A2BD			01-02-29	223D	2287
A2BE			01-02-30	223E	2282
A2BF			01-02-31	223F	2283
A2C0			01-02-32	2240	222A
A2C1			01-02-33	2241	2229
A2CA			01-02-42	224A	2227
A2CB			01-02-43	224B	2228
A2CC			01-02-44	224C	FFE2
A2CD			01-02-45	224D	21D2
A2CE			01-02-46	224E	21D4
A2CF			01-02-47	224F	2200
A2D0			01-02-48	2250	2203
A2DC			01-02-60	225C	2220
A2DD			01-02-61	225D	22A5
A2DE			01-02-62	225E	2312
A2DF			01-02-63	225F	2202
A2E0			01-02-64	2260	2207
A2E1			01-02-65	2261	2261
A2E2			01-02-66	2262	2252
A2E3			01-02-67	2263	226A
A2E4			01-02-68	2264	226B
A2E5			01-02-69	2265	221A
A2E6			01-02-70	2266	223D
A2E7			01-02-71	2267	221D
A2E8			01-02-72	2268	2235
A2E9			01-02-73	2269	222B
A2EA			01-02-74	226A	222C
A2F2			01-02-82	2272	212B
A2F3			01-02-83	2273	2030
A2F4			01-02-84	2274	266F
A2F5			01-02-85	2275	266D
A2F6			01-02-86	2276	266A
A2F7			01-02-87	2277	2020
A2F8			01-02-88	2278	2021
A2F9			01-02-89	2279	00B6
A2FE			01-02-94	227E	25EF
A3B0			01-03-16	2330	FF10
A3B1			01-03-17	2331	FF11
A3B2			01-03-18	2332	FF12
A3B3			01-03-19	2333	FF13
A3B4			01-03-20	2334	FF14
A3B5			01-03-21	2335	FF15
A3B6			01-03-22	2336	FF16
A3B7			01-03-23	2337	FF17
A3B8			01-03-24	2338	FF18
A3B9			01-03-25	2339	FF19
A3C1			01-03-33	2341	FF21
A3C2			01-03-34	2342	FF22
A3C3			01-03-35	2343	FF23
A3C4			01-03-36	2344	FF24
A3C5			01-03-37	2345	FF25
A3C6			01-03-38	2346	FF26
A3C7			01-03-39	2347	FF27
A3C8			01-03-40	2348	FF28
A3C9			01-03-41	2349	FF29
A3CA			01-03-42	234A	FF2A
A3CB			01-03-43	234B	FF2B
A3CC			01-03-44	234C	FF2C
A3CD			01-03-45	234D	FF2D
A3CE			01-03-46	234E	FF2E
A3CF			01-03-47	234F	FF2F
A3D0			01-03-48	2350	FF30
A3D1			01-03-49	2351	FF31
A3D2			01-03-50	2352	FF32
A3D3			01-03-51	2353	FF33
A3D4			01-03-52	2354	FF34
A3D5			01-03-53	2355	FF35
A3D6			01-03-54	2356	FF36
A3D7			01-03-55	2357	FF37
A3D8			01-03-56	2358	FF38
A3D9			01-03-57	2359	FF39
A3DA			01-03-58	235A	FF3A
A3E1			01-03-65	2361	FF41
A3E2			01-03-66	2362	FF42
A3E3			01-03-67	2363	FF43
A3E4			01-03-68	2364	FF44
A3E5			01-03-69	2365	FF45
A3E6			01-03-70	2366	FF46
A3E7			01-03-71	2367	FF47
A3E8			01-03-72	2368	FF48
A3E9			01-03-73	2369	FF49
A3EA			01-03-74	236A	FF4A
A3EB			01-03-75	236B	FF4B
A3EC			01-03-76	236C	FF4C
A3ED			01-03-77	236D	FF4D
A3EE			01-03-78	236E	FF4E
A3EF			01-03-79	236F	FF4F
A3F0			01-03-80	2370	FF50
A3F1			01-03-81	2371	FF51
A3F2			01-03-82	2372	FF52
A3F3			01-03-83	2373	FF53
A3F4			01-03-84	2374	FF54
A3F5			01-03-85	2375	FF55
A3F6			01-03-86	2376	FF56
A3F7			01-03-87	2377	FF57
A3F8			01-03-88	2378	FF58
A3F9			01-03-89	2379	FF59
A3FA			01-03-90	237A	FF5A
A4A1			01-04-01	2421	3041
A4A2			01-04-02	2422	3042
A4A3			01-04-03	2423	3043
A4A4			01-04-04	2424	3044
A4A5			01-04-05	2425	3045
A4A6			01-04-06	2426	3046
A4A7			01-04-07	2427	3047
A4A8			01-04-08	2428	3048
A4A9			01-04-09	2429	3049
A4AA			01-04-10	242A	304A
A4AB			01-04-11	242B	304B
A4AC			01-04-12	242C	304C
A4AD			01-04-13	242D	304D
A4AE			01-04-14	242E	304E
A4AF			01-04-15	242F	304F
A4B0			01-04-16	2430	3050
A4B1			01-04-17	2431	3051
A4B2			01-04-18	2432	3052
A4B3			01-04-19	2433	3053
A4B4			01-04-20	2434	3054
A4B5			01-04-21	2435	3055
A4B6			01-04-22	2436	3056
A4B7			01-04-23	2437	3057
A4B8			01-04-24	2438	3058
A4B9			01-04-25	2439	3059
A4BA			01-04-26	243A	305A
A4BB			01-04-27	243B	305B
A4BC			01-04-28	243C	305C
A4BD			01-04-29	243D	305D
A4BE			01-04-30	243E	305E
A4BF			01-04-31	243F	305F
A4C0			01-04-32	2440	3060
A4C1			01-04-33	2441	3061
A4C2			01-04-34	2442	3062
A4C3			01-04-35	2443	3063
A4C4			01-04-36	2444	3064
A4C5			01-04-37	2445	3065
A4C6			01-04-38	2446	3066
A4C7			01-04-39	2447	3067
A4C8			01-04-40	2448	3068
A4C9			01-04-41	2449	3069
A4CA			01-04-42	244A	306A
A4CB			01-04-43	244B	306B
A4CC			01-04-44	244C	306C
A4CD			01-04-45	244D	306D
A4CE			01-04-46	244E	306E
A4CF			01-04-47	244F	306F
A4D0			01-04-48	2450	3070
A4D1			01-04-49	2451	3071
A4D2			01-04-50	2452	3072
A4D3			01-04-51	2453	3073
A4D4			01-04-52	2454	3074
A4D5			01-04-53	2455	3075
A4D6			01-04-54	2456	3076
A4D7			01-04-55	2457	3077
A4D8			01-04-56	2458	3078
A4D9			01-04-57	2459	3079
A4DA			01-04-58	245A	307A
A4DB			01-04-59	245B	307B
A4DC			01-04-60	245C	307C
A4DD			01-04-61	245D	307D
A4DE			01-04-62	245E	307E
A4DF			01-04-63	245F	307F
A4E0			01-04-64	2460	3080
A4E1			01-04-65	2461	3081
A4E2			01-04-66	2462	3082
A4E3			01-04-67	2463	3083
A4E4			01-04-68	2464	3084
A4E5			01-04-69	2465	3085
A4E6			01-04-70	2466	3086
A4E7			01-04-71	2467	3087
A4E8			01-04-72	2468	3088
A4E9			01-04-73	2469	3089
A4EA			01-04-74	246A	308A
A4EB			01-04-75	246B	308B
A4EC			01-04-76	246C	308C
A4ED			01-04-77	246D	308D
A4EE			01-04-78	246E	308E
A4EF			01-04-79	246F	308F
A4F0			01-04-80	2470	3090
A4F1			01-04-81	2471	3091
A4F2			01-04-82	2472	3092
A4F3			01-04-83	2473	3093
A5A1			01-05-01	2521	30A1
A5A2			01-05-02	2522	30A2
A5A3			01-05-03	2523	30A3
A5A4			01-05-04	2524	30A4
A5A5			01-05-05	2525	30A5
A5A6			01-05-06	2526	30A6
A5A7			01-05-07	2527	30A7
A5A8			01-05-08	2528	30A8
A5A9			01-05-09	2529	30A9
A5AA			01-05-10	252A	30AA
A5AB			01-05-11	252B	30AB
A5AC			01-05-12	252C	30AC
A5AD			01-05-13	252D	30AD
A5AE			01-05-14	252E	30AE
A5AF			01-05-15	252F	30AF
A5B0			01-05-16	2530	30B0
A5B1			01-05-17	2531	30B1
A5B2			01-05-18	2532	30B2
A5B3			01-05-19	2533	30B3
A5B4			01-05-20	2534	30B4
A5B5			01-05-21	2535	30B5
A5B6			01-05-22	2536	30B6
A5B7			01-05-23	2537	30B7
A5B8			01-05-24	2538	30B8
A5B9			01-05-25	2539	30B9
A5BA			01-05-26	253A	30BA
A5BB			01-05-27	253B	30BB
A5BC			01-05-28	253C	30BC
A5BD			01-05-29	253D	30BD
A5BE			01-05-30	253E	30BE
A5BF			01-05-31	253F	30BF
A5C0			01-05-32	2540	30C0
A5C1			01-05-33	2541	30C1
A5C2			01-05-34	2542	30C2
A5C3			01-05-35	2543	30C3
A5C4			01-05-36	2544	30C4
A5C5			01-05-37	2545	30C5
A5C6			01-05-38	2546	30C6
A5C7			01-05-39	2547	30C7
A5C8			01-05-40	2548	30C8
A5C9			01-05-41	2549	30C9
A5CA			01-05-42	254A	30CA
A5CB			01-05-43	254B	30CB
A5CC			01-05-44	254C	30CC
A5CD			01-05-45	254D	30CD
A5CE			01-05-46	254E	30CE
A5CF			01-05-47	254F	30CF
A5D0			01-05-48	2550	30D0
A5D1			01-05-49	2551	30D1
A5D2			01-05-50	2552	30D2
A5D3			01-05-51	2553	30D3
A5D4			01-05-52	2554	30D4
A5D5			01-05-53	2555	30D5
A5D6			01-05-54	2556	30D6
A5D7			01-05-55	2557	30D7
A5D8			01-05-56	2558	30D8
A5D9			01-05-57	2559	30D9
A5DA			01-05-58	255A	30DA
A5DB			01-05-59	255B	30DB
A5DC			01-05-60	255C	30DC
A5DD			01-05-61	255D	30DD
A5DE			01-05-62	255E	30DE
A5DF			01-05-63	255F	30DF
A5E0			01-05-64	2560	30E0
A5E1			01-05-65	2561	30E1
A5E2			01-05-66	2562	30E2
A5E3			01-05-67	2563	30E3
A5E4			01-05-68	2564	30E4
A5E5			01-05-69	2565	30E5
A5E6			01-05-70	2566	30E6
A5E7			01-05-71	2567	30E7
A5E8			01-05-72	2568	30E8
A5E9			01-05-73	2569	30E9
A5EA			01-05-74	256A	30EA
A5EB			01-05-75	256B	30EB
A5EC			01-05-76	256C	30EC
A5ED			01-05-77	256D	30ED
A5EE			01-05-78	256E	30EE
A5EF			01-05-79	256F	30EF
A5F0			01-05-80	2570	30F0
A5F1			01-05-81	2571	30F1
A5F2			01-05-82	2572	30F2
A5F3			01-05-83	2573	30F3
A5F4			01-05-84	2574	30F4
A5F5			01-05-85	2575	30F5
A5F6			01-05-86	2576	30F6
A6A1			01-06-01	2621	0391
A6A2			01-06-02	2622	0392
A6A3			01-06-03	2623	0393
A6A4			01-06-04	2624	0394
A6A5			01-06-05	2625	0395
A6A6			01-06-06	2626	0396
A6A7			01-06-07	2627	0397
A6A8			01-06-08	2628	0398
A6A9			01-06-09	2629	0399
A6AA			01-06-10	262A	039A
A6AB			01-06-11	262B	039B
A6AC			01-06-12	262C	039C
A6AD			01-06-13	262D	039D
A6AE			01-06-14	262E	039E
A6AF			01-06-15	262F	039F
A6B0			01-06-16	2630	03A0
A6B1			01-06-17	2631	03A1
A6B2			01-06-18	2632	03A3
A6B3			01-06-19	2633	03A4
A6B4			01-06-20	2634	03A5
A6B5			01-06-21	2635	03A6
A6B6			01-06-22	2636	03A7
A6B7			01-06-23	2637	03A8
A6B8			01-06-24	2638	03A9
A6C1			01-06-33	2641	03B1
A6C2			01-06-34	2642	03B2
A6C3			01-06-35	2643	03B3
A6C4			01-06-36	2644	03B4
A6C5			01-06-37	2645	03B5
A6C6			01-06-38	2646	03B6
A6C7			01-06-39	2647	03B7
A6C8			01-06-40	2648	03B8
A6C9			01-06-41	2649	03B9
A6CA			01-06-42	264A	03BA
A6CB			01-06-43	264B	03BB
A6CC			01-06-44	264C	03BC
A6CD			01-06-45	264D	03BD
A6CE			01-06-46	264E	03BE
A6CF			01-06-47	264F	03BF
A6D0			01-06-48	2650	03C0
A6D1			01-06-49	2651	03C1
A6D2			01-06-50	2652	03C3
A6D3			01-06-51	2653	03C4
A6D4			01-06-52	2654	03C5
A6D5			01-06-53	2655	03C6
A6D6			01-06-54	2656	03C7
A6D7			01-06-55	2657	03C8
A6D8			01-06-56	2658	03C9
A7A1			01-07-01	2721	0410
A7A2			01-07-02	2722	0411
A7A3			01-07-03	2723	0412
A7A4			01-07-04	2724	0413
A7A5			01-07-05	2725	0414
A7A6			01-07-06	2726	0415
A7A7			01-07-07	2727	0401
A7A8			01-07-08	2728	0416
A7A9			01-07-09	2729	0417
A7AA			01-07-10	272A	0418
A7AB			01-07-11	272B	0419
A7AC			01-07-12	272C	041A
A7AD			01-07-13	272D	041B
A7AE			01-07-14	272E	041C
A7AF			01-07-15	272F	041D
A7B0			01-07-16	2730	041E
A7B1			01-07-17	2731	041F
A7B2			01-07-18	2732	0420
A7B3			01-07-19	2733	0421
A7B4			01-07-20	2734	0422
A7B5			01-07-21	2735	0423
A7B6			01-07-22	2736	0424
A7B7			01-07-23	2737	0425
A7B8			01-07-24	2738	0426
A7B9			01-07-25	2739	0427
A7BA			01-07-26	273A	0428
A7BB			01-07-27	273B	0429
A7BC			01-07-28	273C	042A
A7BD			01-07-29	273D	042B
A7BE			01-07-30	273E	042C
A7BF			01-07-31	273F	042D
A7C0			01-07-32	2740	042E
A7C1			01-07-33	2741	042F
A7D1			01-07-49	2751	0430
A7D2			01-07-50	2752	0431
A7D3			01-07-51	2753	0432
A7D4			01-07-52	2754	0433
A7D5			01-07-53	2755	0434
A7D6			01-07-54	2756	0435
A7D7			01-07-55	2757	0451
A7D8			01-07-56	2758	0436
A7D9			01-07-57	2759	0437
A7DA			01-07-58	275A	0438
A7DB			01-07-59	275B	0439
A7DC			01-07-60	275C	043A
A7DD			01-07-61	275D	043B
A7DE			01-07-62	275E	043C
A7DF			01-07-63	275F	043D
A7E0			01-07-64	2760	043E
A7E1			01-07-65	2761	043F
A7E2			01-07-66	2762	0440
A7E3			01-07-67	2763	0441
A7E4			01-07-68	2764	0442
A7E5			01-07-69	2765	0443
A7E6			01-07-70	2766	0444
A7E7			01-07-71	2767	0445
A7E8			01-07-72	2768	0446
A7E9			01-07-73	2769	0447
A7EA			01-07-74	276A	0448
A7EB			01-07-75	276B	0449
A7EC			01-07-76	276C	044A
A7ED			01-07-77	276D	044B
A7EE			01-07-78	276E	044C
A7EF			01-07-79	276F	044D
A7F0			01-07-80	2770	044E
A7F1			01-07-81	2771	044F
A8A1			01-08-01	2821	2500
A8A2			01-08-02	2822	2502
A8A3			01-08-03	2823	250C
A8A4			01-08-04	2824	2510
A8A5			01-08-05	2825	2518
A8A6			01-08-06	2826	2514
A8A7			01-08-07	2827	251C
A8A8			01-08-08	2828	252C
A8A9			01-08-09	2829	2524
A8AA			01-08-10	282A	2534
A8AB			01-08-11	282B	253C
A8AC			01-08-12	282C	2501
A8AD			01-08-13	282D	2503
A8AE			01-08-14	282E	250F
A8AF			01-08-15	282F	2513
A8B0			01-08-16	2830	251B
A8B1			01-08-17	2831	2517
A8B2			01-08-18	2832	2523
A8B3			01-08-19	2833	2533
A8B4			01-08-20	2834	252B
A8B5			01-08-21	2835	253B
A8B6			01-08-22	2836	254B
A8B7			01-08-23	2837	2520
A8B8			01-08-24	2838	252F
A8B9			01-08-25	2839	2528
A8BA			01-08-26	283A	2537
A8BB			01-08-27	283B	253F
A8BC			01-08-28	283C	251D
A8BD			01-08-29	283D	2530
A8BE			01-08-30	283E	2525
A8BF			01-08-31	283F	2538
A8C0			01-08-32	2840	2542
B0A1	7	7	01-16-01	3021	4E9C
B0A2	30	10	01-16-02	3022	5516
B0A3	38	9	01-16-03	3023	5A03
B0A4	170	8	01-16-04	3024	963F
B0A5	30	9	01-16-05	3025	54C0
B0A6	61	13	01-16-06	3026	611B
B0A7	64	10	01-16-07	3027	6328
B0A8	38	9	01-16-08	3028	59F6
B0A9	162	10	01-16-09	3029	9022
B0AA	140	12	01-16-10	302A	8475
B0AB	140	9	01-16-11	302B	831C
B0AC	115	16	01-16-12	302C	7A50
B0AD	61	11	01-16-13	302D	60AA
B0AE	64	12	01-16-14	302E	63E1
B0AF	85	12	01-16-15	302F	6E25
B0B0	72	6	01-16-16	3030	65ED
B0B1	140	12	01-16-17	3031	8466
B0B2	140	7	01-16-18	3032	82A6
B0B3	195	19	01-16-19	3033	9BF5
B0B4	75	11	01-16-20	3034	6893
B0B5	32	5	01-16-21	3035	5727
B0B6	68	14	01-16-22	3036	65A1
B0B7	64	6	01-16-23	3037	6271
B0B8	40	8	01-16-24	3038	5B9B
B0B9	38	8	01-16-25	3039	59D0
B0BA	142	9	01-16-26	303A	867B
B0BB	184	13	01-16-27	303B	98F4
B0BC	120	12	01-16-28	303C	7D62
B0BD	120	14	01-16-29	303D	7DBE
B0BE	195	16	01-16-30	303E	9B8E
B0BF	62	8	01-16-31	303F	6216
B0C0	119	12	01-16-32	3040	7C9F
B0C1	145	11	01-16-33	3041	88B7
B0C2	40	6	01-16-34	3042	5B89
B0C3	53	11	01-16-35	3043	5EB5
B0C4	64	9	01-16-36	3044	6309
B0C5	72	13	01-16-37	3045	6697
B0C6	75	10	01-16-38	3046	6848
B0C7	169	17	01-16-39	3047	95C7
B0C8	177	15	01-16-40	3048	978D
B0C9	75	7	01-16-41	3049	674F
B0CA	9	5	01-16-42	304A	4EE5
B0CB	9	6	01-16-43	304B	4F0A
B0CC	9	7	01-16-44	304C	4F4D
B0CD	9	8	01-16-45	304D	4F9D
B0CE	9	12	01-16-46	304E	5049
B0CF	31	7	01-16-47	304F	56F2
B0D0	37	6	01-16-48	3050	5937
B0D1	38	8	01-16-49	3051	59D4
B0D2	38	9	01-16-50	3052	5A01
B0D3	41	11	01-16-51	3053	5C09
B0D4	61	11	01-16-52	3054	60DF
B0D5	61	13	01-16-53	3055	610F
B0D6	61	15	01-16-54	3056	6170
B0D7	72	8	01-16-55	3057	6613
B0D8	75	12	01-16-56	3058	6905
B0D9	87	9	01-16-57	3059	70BA
B0DA	102	9	01-16-58	305A	754F
B0DB	102	11	01-16-59	305B	7570
B0DC	115	11	01-16-60	305C	79FB
B0DD	120	14	01-16-61	305D	7DAD
B0DE	120	16	01-16-62	305E	7DEF
B0DF	130	9	01-16-63	305F	80C3
B0E0	140	11	01-16-64	3060	840E
B0E1	145	6	01-16-65	3061	8863
B0E2	149	16	01-16-66	3062	8B02
B0E3	162	13	01-16-67	3063	9055
B0E4	162	15	01-16-68	3064	907A
B0E5	22	7	01-16-69	3065	533B
B0E6	7	4	01-16-70	3066	4E95
B0E7	8	6	01-16-71	3067	4EA5
B0E8	32	11	01-16-72	3068	57DF
B0E9	130	8	01-16-73	3069	80B2
B0EA	163	9	01-16-74	306A	90C1
B0EB	112	17	01-16-75	306B	78EF
B0EC	1	1	01-16-76	306C	4E00
B0ED	33	7	01-16-77	306D	58F1
B0EE	85	13	01-16-78	306E	6EA2
B0EF	162	11	01-16-79	306F	9038
B0F0	115	14	01-16-80	3070	7A32
B0F1	140	9	01-16-81	3071	8328
B0F2	140	6	01-16-82	3072	828B
B0F3	195	21	01-16-83	3073	9C2F
B0F4	10	4	01-16-84	3074	5141
B0F5	26	6	01-16-85	3075	5370
B0F6	30	9	01-16-86	3076	54BD
B0F7	30	10	01-16-87	3077	54E1
B0F8	31	6	01-16-88	3078	56E0
B0F9	38	9	01-16-89	3079	59FB
B0FA	57	4	01-16-90	307A	5F15
B0FB	76	12	01-16-91	307B	98F2
B0FC	85	11	01-16-92	307C	6DEB
B0FD	130	9	01-16-93	307D	80E4
B0FE	140	14	01-16-94	307E	852D
B1A1	170	10	01-17-01	3121	9662
B1A2	170	11	01-17-02	3122	9670
B1A3	170	14	01-17-03	3123	96A0
B1A4	180	19	01-17-04	3124	97FB
B1A5	30	6	01-17-05	3125	540B
B1A6	30	5	01-17-06	3126	53F3
B1A7	40	6	01-17-07	3127	5B87
B1A8	86	10	01-17-08	3128	70CF
B1A9	124	6	01-17-09	3129	7FBD
B1AA	162	6	01-17-10	312A	8FC2
B1AB	173	8	01-17-11	312B	96E8
B1AC	26	5	01-17-12	312C	536F
B1AD	196	18	01-17-13	312D	9D5C
B1AE	116	16	01-17-14	312E	7ABA
B1AF	1	4	01-17-15	312F	4E11
B1B0	112	13	01-17-16	3130	7893
B1B1	134	6	01-17-17	3131	81FC
B1B2	85	12	01-17-18	3132	6E26
B1B3	30	14	01-17-19	3133	5618
B1B4	30	10	01-17-20	3134	5504
B1B5	75	25	01-17-21	3135	6B1D
B1B6	140	14	01-17-22	3136	851A
B1B7	195	22	01-17-23	3137	9C3B
B1B8	38	9	01-17-24	3138	59E5
B1B9	27	12	01-17-25	3139	53A9
B1BA	85	10	01-17-26	313A	6D66
B1BB	97	5	01-17-27	313B	74DC
B1BC	169	12	01-17-28	313C	958F
B1BD	30	15	01-17-29	313D	5642
B1BE	7	4	01-17-30	313E	4E91
B1BF	162	12	01-17-31	313F	904B
B1C0	173	12	01-17-32	3140	96F2
B1C1	140	9	01-17-33	3141	834F
B1C2	184	14	01-17-34	3142	990C
B1C3	29	16	01-17-35	3143	53E1
B1C4	30	12	01-17-36	3144	55B6
B1C5	38	17	01-17-37	3145	5B30
B1C6	59	15	01-17-38	3146	5F71
B1C7	72	9	01-17-39	3147	6620
B1C8	73	6	01-17-40	3148	66F3
B1C9	75	9	01-17-41	3149	6804
B1CA	85	5	01-17-42	314A	6C38
B1CB	85	8	01-17-43	314B	6CF3
B1CC	85	9	01-17-44	314C	6D29
B1CD	96	12	01-17-45	314D	745B
B1CE	108	9	01-17-46	314E	76C8
B1CF	115	16	01-17-47	314F	7A4E
B1D0	181	16	01-17-48	3150	9834
B1D1	140	8	01-17-49	3151	82F1
B1D2	144	16	01-17-50	3152	885B
B1D3	149	12	01-17-51	3153	8A60
B1D4	167	15	01-17-52	3154	92ED
B1D5	85	11	01-17-53	3155	6DB2
B1D6	104	9	01-17-54	3156	75AB
B1D7	108	10	01-17-55	3157	76CA
B1D8	187	14	01-17-56	3158	99C5
B1D9	61	10	01-17-57	3159	60A6
B1DA	149	15	01-17-58	315A	8B01
B1DB	156	12	01-17-59	315B	8D8A
B1DC	169	15	01-17-60	315C	95B2
B1DD	75	14	01-17-61	315D	698E
B1DE	27	14	01-17-62	315E	53AD
B1DF	13	4	01-17-63	315F	5186
B1E0	31	13	01-17-64	3160	5712
B1E1	32	12	01-17-65	3161	5830
B1E2	37	8	01-17-66	3162	5944
B1E3	40	10	01-17-67	3163	5BB4
B1E4	54	8	01-17-68	3164	5EF6
B1E5	61	9	01-17-69	3165	6028
B1E6	64	11	01-17-70	3166	63A9
B1E7	64	12	01-17-71	3167	63F4
B1E8	85	8	01-17-72	3168	6CBF
B1E9	85	14	01-17-73	3169	6F14
B1EA	86	8	01-17-74	316A	708E
B1EB	86	11	01-17-75	316B	7114
B1EC	86	13	01-17-76	316C	7159
B1ED	86	16	01-17-77	316D	71D5
B1EE	94	13	01-17-78	316E	733F
B1EF	120	15	01-17-79	316F	7E01
B1F0	139	19	01-17-80	3170	8276
B1F1	140	8	01-17-81	3171	82D1
B1F2	140	16	01-17-82	3172	8597
B1F3	162	13	01-17-83	3173	9060
B1F4	167	13	01-17-84	3174	925B
B1F5	196	16	01-17-85	3175	9D1B
B1F6	32	13	01-17-86	3176	5869
B1F7	70	8	01-17-87	3177	65BC
B1F8	85	6	01-17-88	3178	6C5A
B1F9	100	12	01-17-89	3179	7525
B1FA	17	5	01-17-90	317A	51F9
B1FB	37	5	01-17-91	317B	592E
B1FC	37	12	01-17-92	317C	5965
B1FD	60	8	01-17-93	317D	5F80
B1FE	61	7	01-17-94	317E	5FDC
B2A1	64	8	01-18-01	3221	62BC
B2A2	72	8	01-18-02	3222	65FA
B2A3	75	15	01-18-03	3223	6A2A
B2A4	76	8	01-18-04	3224	6B27
B2A5	79	8	01-18-05	3225	6BB4
B2A6	96	4	01-18-06	3226	738B
B2A7	124	10	01-18-07	3227	7FC1
B2A8	145	17	01-18-08	3228	8956
B2A9	196	16	01-18-09	3229	9D2C
B2AA	196	15	01-18-10	322A	9D0E
B2AB	201	11	01-18-11	322B	9EC4
B2AC	46	8	01-18-12	322C	5CA1
B2AD	85	7	01-18-13	322D	6C96
B2AE	140	10	01-18-14	322E	837B
B2AF	9	15	01-18-15	322F	5104
B2B0	44	9	01-18-16	3230	5C4B
B2B1	61	16	01-18-17	3231	61B6
B2B2	130	17	01-18-18	3232	81C6
B2B3	75	11	01-18-19	3233	6876
B2B4	93	7	01-18-20	3234	7261
B2B5	5	1	01-18-21	3235	4E59
B2B6	9	10	01-18-22	3236	4FFA
B2B7	26	9	01-18-23	3237	5378
B2B8	61	10	01-18-24	3238	6069
B2B9	85	12	01-18-25	3239	6E29
B2BA	115	16	01-18-26	323A	7A4F
B2BB	180	9	01-18-27	323B	97F3
B2BC	1	3	01-18-28	323C	4E0B
B2BD	9	4	01-18-29	323D	5316
B2BE	9	6	01-18-30	323E	4EEE
B2BF	9	7	01-18-31	323F	4F55
B2C0	9	7	01-18-32	3240	4F3D
B2C1	9	8	01-18-33	3241	4FA1
B2C2	9	8	01-18-34	3242	4F73
B2C3	19	5	01-18-35	3243	52A0
B2C4	30	5	01-18-36	3244	53EF
B2C5	30	14	01-18-37	3245	5609
B2C6	35	10	01-18-38	3246	590F
B2C7	38	13	01-18-39	3247	5AC1
B2C8	40	10	01-18-40	3248	5BB6
B2C9	40	14	01-18-41	3249	5BE1
B2CA	68	9	01-18-42	324A	79D1
B2CB	72	13	01-18-43	324B	6687
B2CC	75	8	01-18-44	324C	679C
B2CD	75	9	01-18-45	324D	67B6
B2CE	76	14	01-18-46	324E	6B4C
B2CF	85	8	01-18-47	324F	6CB3
B2D0	86	4	01-18-48	3250	706B
B2D1	96	9	01-18-49	3251	73C2
B2D2	113	13	01-18-50	3252	798D
B2D3	115	5	01-18-51	3253	79BE
B2D4	115	15	01-18-52	3254	7A3C
B2D5	118	14	01-18-53	3255	7B87
B2D6	140	7	01-18-54	3256	82B1
B2D7	140	8	01-18-55	3257	82DB
B2D8	140	8	01-18-56	3258	8304
B2D9	140	10	01-18-57	3259	8377
B2DA	140	10	01-18-58	325A	83EF
B2DB	140	11	01-18-59	325B	83D3
B2DC	142	15	01-18-60	325C	8766
B2DD	149	15	01-18-61	325D	8AB2
B2DE	30	13	01-18-62	325E	5629
B2DF	154	11	01-18-63	325F	8CA8
B2E0	162	8	01-18-64	3260	8FE6
B2E1	162	12	01-18-65	3261	904E
B2E2	173	17	01-18-66	3262	971E
B2E3	142	10	01-18-67	3263	868A
B2E4	9	9	01-18-68	3264	4FC4
B2E5	46	10	01-18-69	3265	5CE8
B2E6	62	7	01-18-70	3266	6211
B2E7	92	4	01-18-71	3267	7259
B2E8	102	8	01-18-72	3268	753B
B2E9	131	9	01-18-73	3269	81E5
B2EA	140	8	01-18-74	326A	82BD
B2EB	142	13	01-18-75	326B	86FE
B2EC	154	12	01-18-76	326C	8CC0
B2ED	172	13	01-18-77	326D	96C5
B2EE	184	15	01-18-78	326E	9913
B2EF	187	15	01-18-79	326F	99D5
B2F0	9	4	01-18-80	3270	4ECB
B2F1	9	6	01-18-81	3271	4F1A
B2F2	18	13	01-18-82	3272	89E3
B2F3	31	6	01-18-83	3273	56DE
B2F4	32	13	01-18-84	3274	584A
B2F5	32	16	01-18-85	3275	58CA
B2F6	54	9	01-18-86	3276	5EFB
B2F7	61	7	01-18-87	3277	5FEB
B2F8	61	8	01-18-88	3278	602A
B2F9	61	9	01-18-89	3279	6094
B2FA	61	9	01-18-90	327A	6062
B2FB	61	16	01-18-91	327B	61D0
B2FC	62	7	01-18-92	327C	6212
B2FD	64	8	01-18-93	327D	62D0
B2FE	66	7	01-18-94	327E	6539
B3A1	68	14	01-19-01	3321	9B41
B3A2	72	10	01-19-02	3322	6666
B3A3	75	11	01-19-03	3323	68B0
B3A4	85	9	01-19-04	3324	6D77
B3A5	86	6	01-19-05	3325	7070
B3A6	102	9	01-19-06	3326	754C
B3A7	106	9	01-19-07	3327	7686
B3A8	120	12	01-19-08	3328	7D75
B3A9	140	7	01-19-09	3329	82A5
B3AA	142	19	01-19-10	332A	87F9
B3AB	169	12	01-19-11	332B	958B
B3AC	170	12	01-19-12	332C	968E
B3AD	154	7	01-19-13	332D	8C9D
B3AE	16	12	01-19-14	332E	51F1
B3AF	19	8	01-19-15	332F	52BE
B3B0	25	5	01-19-16	3330	5916
B3B1	30	9	01-19-17	3331	54B3
B3B2	40	10	01-19-18	3332	5BB3
B3B3	46	11	01-19-19	3333	5D16
B3B4	61	13	01-19-20	3334	6168
B3B5	75	14	01-19-21	3335	6982
B3B6	85	11	01-19-22	3336	6DAF
B3B7	112	13	01-19-23	3337	788D
B3B8	140	13	01-19-24	3338	84CB
B3B9	144	12	01-19-25	3339	8857
B3BA	149	13	01-19-26	333A	8A72
B3BB	167	18	01-19-27	333B	93A7
B3BC	188	16	01-19-28	333C	9AB8
B3BD	85	10	01-19-29	333D	6D6C
B3BE	186	20	01-19-30	333E	99A8
B3BF	142	12	01-19-31	333F	86D9
B3C0	32	9	01-19-32	3340	57A3
B3C1	75	9	01-19-33	3341	67FF
B3C2	142	11	01-19-34	3342	86CE
B3C3	167	12	01-19-35	3343	920E
B3C4	18	14	01-19-36	3344	5283
B3C5	30	17	01-19-37	3345	5687
B3C6	34	6	01-19-38	3346	5404
B3C7	53	14	01-19-39	3347	5ED3
B3C8	64	8	01-19-40	3348	62E1
B3C9	64	15	01-19-41	3349	64B9
B3CA	75	10	01-19-42	334A	683C
B3CB	75	10	01-19-43	334B	6838
B3CC	79	11	01-19-44	334C	6BBB
B3CD	94	16	01-19-45	334D	7372
B3CE	112	15	01-19-46	334E	78BA
B3CF	115	18	01-19-47	334F	7A6B
B3D0	147	12	01-19-48	3350	899A
B3D1	148	7	01-19-49	3351	89D2
B3D2	155	14	01-19-50	3352	8D6B
B3D3	159	13	01-19-51	3353	8F03
B3D4	163	11	01-19-52	3354	90ED
B3D5	169	14	01-19-53	3355	95A3
B3D6	170	13	01-19-54	3356	9694
B3D7	177	9	01-19-55	3357	9769
B3D8	39	8	01-19-56	3358	5B66
B3D9	46	8	01-19-57	3359	5CB3
B3DA	75	13	01-19-58	335A	697D
B3DB	181	18	01-19-59	335B	984D
B3DC	181	18	01-19-60	335C	984E
B3DD	64	11	01-19-61	335D	639B
B3DE	118	11	01-19-62	335E	7B20
B3DF	75	16	01-19-63	335F	6A2B
B3E0	75	17	01-19-64	3360	6A7F
B3E1	75	11	01-19-65	3361	68B6
B3E2	195	20	01-19-66	3362	9C0D
B3E3	85	15	01-19-67	3363	6F5F
B3E4	18	12	01-19-68	3364	5272
B3E5	30	11	01-19-69	3365	559D
B3E6	61	9	01-19-70	3366	6070
B3E7	64	9	01-19-71	3367	62EC
B3E8	85	9	01-19-72	3368	6D3B
B3E9	85	11	01-19-73	3369	6E07
B3EA	85	13	01-19-74	336A	6ED1
B3EB	140	11	01-19-75	336B	845B
B3EC	145	13	01-19-76	336C	8910
B3ED	159	17	01-19-77	336D	8F44
B3EE	1	5	01-19-78	336E	4E14
B3EF	195	23	01-19-79	336F	9C39
B3F0	30	5	01-19-80	3370	53F6
B3F1	75	11	01-19-81	3371	691B
B3F2	75	14	01-19-82	3372	6A3A
B3F3	177	14	01-19-83	3373	9784
B3F4	75	10	01-19-84	3374	682A
B3F5	10	11	01-19-85	3375	515C
B3F6	116	17	01-19-86	3376	7AC3
B3F7	140	13	01-19-87	3377	84B2
B3F8	167	10	01-19-88	3378	91DC
B3F9	167	18	01-19-89	3379	938C
B3FA	30	15	01-19-90	337A	565B
B3FB	196	16	01-19-91	337B	9D28
B3FC	75	10	01-19-92	337C	6822
B3FD	140	8	01-19-93	337D	8305
B3FE	140	12	01-19-94	337E	8431
B4A1	119	12	01-20-01	3421	7CA5
B4A2	18	4	01-20-02	3422	5208
B4A3	18	7	01-20-03	3423	82C5
B4A4	98	5	01-20-04	3424	74E6
B4A5	5	11	01-20-05	3425	4E7E
B4A6	9	8	01-20-06	3426	4F83
B4A7	14	9	01-20-07	3427	51A0
B4A8	15	12	01-20-08	3428	5BD2
B4A9	18	5	01-20-09	3429	520A
B4AA	19	11	01-20-10	342A	52D8
B4AB	19	13	01-20-11	342B	52E7
B4AC	49	9	01-20-12	342C	5DFB
B4AD	30	12	01-20-13	342D	559A
B4AE	32	12	01-20-14	342E	582A
B4AF	38	9	01-20-15	342F	59E6
B4B0	40	7	01-20-16	3430	5B8C
B4B1	40	8	01-20-17	3431	5B98
B4B2	40	13	01-20-18	3432	5BDB
B4B3	51	3	01-20-19	3433	5E72
B4B4	51	13	01-20-20	3434	5E79
B4B5	61	11	01-20-21	3435	60A3
B4B6	61	13	01-20-22	3436	611F
B4B7	61	14	01-20-23	3437	6163
B4B8	61	16	01-20-24	3438	61BE
B4B9	64	12	01-20-25	3439	63DB
B4BA	66	12	01-20-26	343A	6562
B4BB	75	9	01-20-27	343B	67D1
B4BC	75	10	01-20-28	343C	6853
B4BD	75	12	01-20-29	343D	68FA
B4BE	76	12	01-20-30	343E	6B3E
B4BF	76	15	01-20-31	343F	6B53
B4C0	85	6	01-20-32	3440	6C57
B4C1	85	13	01-20-33	3441	6F22
B4C2	85	15	01-20-34	3442	6F97
B4C3	85	14	01-20-35	3443	6F45
B4C4	96	17	01-20-36	3444	74B0
B4C5	99	5	01-20-37	3445	7518
B4C6	108	15	01-20-38	3446	76E3
B4C7	109	9	01-20-39	3447	770B
B4C8	118	9	01-20-40	3448	7AFF
B4C9	118	14	01-20-41	3449	7BA1
B4CA	118	18	01-20-42	344A	7C21
B4CB	120	15	01-20-43	344B	7DE9
B4CC	121	6	01-20-44	344C	7F36
B4CD	124	16	01-20-45	344D	7FF0
B4CE	130	7	01-20-46	344E	809D
B4CF	137	21	01-20-47	344F	8266
B4D0	140	10	01-20-48	3450	839E
B4D1	147	18	01-20-49	3451	89B3
B4D2	149	15	01-20-50	3452	8ACC
B4D3	154	11	01-20-51	3453	8CAB
B4D4	162	16	01-20-52	3454	9084
B4D5	167	23	01-20-53	3455	9451
B4D6	169	12	01-20-54	3456	9593
B4D7	169	12	01-20-55	3457	9591
B4D8	169	14	01-20-56	3458	95A2
B4D9	170	10	01-20-57	3459	9665
B4DA	178	17	01-20-58	345A	97D3
B4DB	184	16	01-20-59	345B	9928
B4DC	135	16	01-20-60	345C	8218
B4DD	3	3	01-20-61	345D	4E38
B4DE	30	7	01-20-62	345E	542B
B4DF	46	8	01-20-63	345F	5CB8
B4E0	46	20	01-20-64	3460	5DCC
B4E1	96	8	01-20-65	3461	73A9
B4E2	104	17	01-20-66	3462	764C
B4E3	109	11	01-20-67	3463	773C
B4E4	112	8	01-20-68	3464	5CA9
B4E5	124	15	01-20-69	3465	7FEB
B4E6	154	19	01-20-70	3466	8D0B
B4E7	172	12	01-20-71	3467	96C1
B4E8	181	13	01-20-72	3468	9811
B4E9	181	18	01-20-73	3469	9854
B4EA	181	19	01-20-74	346A	9858
B4EB	9	6	01-20-75	346B	4F01
B4EC	9	6	01-20-76	346C	4F0E
B4ED	26	6	01-20-77	346D	5371
B4EE	30	12	01-20-78	346E	559C
B4EF	30	15	01-20-79	346F	5668
B4F0	32	11	01-20-80	3470	57FA
B4F1	37	8	01-20-81	3471	5947
B4F2	38	15	01-20-82	3472	5B09
B4F3	40	11	01-20-83	3473	5BC4
B4F4	46	7	01-20-84	3474	5C90
B4F5	50	7	01-20-85	3475	5E0C
B4F6	52	12	01-20-86	3476	5E7E
B4F7	61	7	01-20-87	3477	5FCC
B4F8	64	12	01-20-88	3478	63EE
B4F9	75	6	01-20-89	3479	673A
B4FA	70	14	01-20-90	347A	65D7
B4FB	71	10	01-20-91	347B	65E2
B4FC	74	12	01-20-92	347C	671F
B4FD	75	12	01-20-93	347D	68CB
B4FE	75	13	01-20-94	347E	68C4
B5A1	75	16	01-21-01	3521	6A5F
B5A2	77	10	01-21-02	3522	5E30
B5A3	79	15	01-21-03	3523	6BC5
B5A4	84	6	01-21-04	3524	6C17
B5A5	85	7	01-21-05	3525	6C7D
B5A6	102	15	01-21-06	3526	757F
B5A7	113	8	01-21-07	3527	7948
B5A8	115	8	01-21-08	3528	5B63
B5A9	115	12	01-21-09	3529	7A00
B5AA	120	9	01-21-10	352A	7D00
B5AB	120	16	01-21-11	352B	5FBD
B5AC	147	11	01-21-12	352C	898F
B5AD	149	10	01-21-13	352D	8A18
B5AE	154	12	01-21-14	352E	8CB4
B5AF	156	10	01-21-15	352F	8D77
B5B0	159	9	01-21-16	3530	8ECC
B5B1	159	15	01-21-17	3531	8F1D
B5B2	184	10	01-21-18	3532	98E2
B5B3	187	18	01-21-19	3533	9A0E
B5B4	194	10	01-21-20	3534	9B3C
B5B5	213	11	01-21-21	3535	4E80
B5B6	9	11	01-21-22	3536	507D
B5B7	9	15	01-21-23	3537	5100
B5B8	38	7	01-21-24	3538	5993
B5B9	40	8	01-21-25	3539	5B9C
B5BA	62	15	01-21-26	353A	622F
B5BB	64	7	01-21-27	353B	6280
B5BC	64	17	01-21-28	353C	64EC
B5BD	76	12	01-21-29	353D	6B3A
B5BE	93	17	01-21-30	353E	72A0
B5BF	103	14	01-21-31	353F	7591
B5C0	113	8	01-21-32	3540	7947
B5C1	123	13	01-21-33	3541	7FA9
B5C2	142	19	01-21-34	3542	87FB
B5C3	149	15	01-21-35	3543	8ABC
B5C4	149	20	01-21-36	3544	8B70
B5C5	64	11	01-21-37	3545	63AC
B5C6	140	11	01-21-38	3546	83CA
B5C7	177	17	01-21-39	3547	97A0
B5C8	30	6	01-21-40	3548	5409
B5C9	30	6	01-21-41	3549	5403
B5CA	30	12	01-21-42	354A	55AB
B5CB	75	10	01-21-43	354B	6854
B5CC	75	16	01-21-44	354C	6A58
B5CD	149	13	01-21-45	354D	8A70
B5CE	112	10	01-21-46	354E	7827
B5CF	75	8	01-21-47	354F	6775
B5D0	202	12	01-21-48	3550	9ECD
B5D1	26	7	01-21-49	3551	5374
B5D2	40	9	01-21-50	3552	5BA2
B5D3	130	11	01-21-51	3553	811A
B5D4	141	9	01-21-52	3554	8650
B5D5	162	9	01-21-53	3555	9006
B5D6	1	5	01-21-54	3556	4E18
B5D7	4	3	01-21-55	3557	4E45
B5D8	9	4	01-21-56	3558	4EC7
B5D9	9	6	01-21-57	3559	4F11
B5DA	29	3	01-21-58	355A	53CA
B5DB	30	6	01-21-59	355B	5438
B5DC	40	10	01-21-60	355C	5BAE
B5DD	57	3	01-21-61	355D	5F13
B5DE	61	9	01-21-62	355E	6025
B5DF	66	11	01-21-63	355F	6551
B5E0	75	6	01-21-64	3560	673D
B5E1	85	7	01-21-65	3561	6C42
B5E2	85	7	01-21-66	3562	6C72
B5E3	85	8	01-21-67	3563	6CE3
B5E4	86	7	01-21-68	3564	7078
B5E5	96	11	01-21-69	3565	7403
B5E6	116	7	01-21-70	3566	7A76
B5E7	116	15	01-21-71	3567	7AAE
B5E8	118	10	01-21-72	3568	7B08
B5E9	120	9	01-21-73	3569	7D1A
B5EA	120	9	01-21-74	356A	7CFE
B5EB	120	12	01-21-75	356B	7D66
B5EC	72	5	01-21-76	356C	65E7
B5ED	93	4	01-21-77	356D	725B
B5EE	28	5	01-21-78	356E	53BB
B5EF	44	8	01-21-79	356F	5C45
B5F0	48	5	01-21-80	3570	5DE8
B5F1	64	8	01-21-81	3571	62D2
B5F2	64	8	01-21-82	3572	62E0
B5F3	64	10	01-21-83	3573	6319
B5F4	85	12	01-21-84	3574	6E20
B5F5	141	11	01-21-85	3575	865A
B5F6	149	11	01-21-86	3576	8A31
B5F7	157	12	01-21-87	3577	8DDD
B5F8	167	16	01-21-88	3578	92F8
B5F9	85	14	01-21-89	3579	6F01
B5FA	113	16	01-21-90	357A	79A6
B5FB	195	11	01-21-91	357B	9B5A
B5FC	8	7	01-21-92	357C	4EA8
B5FD	8	8	01-21-93	357D	4EAB
B5FE	8	8	01-21-94	357E	4EAC
B6A1	9	8	01-22-01	3621	4F9B
B6A2	9	8	01-22-02	3622	4FA0
B6A3	9	14	01-22-03	3623	50D1
B6A4	10	6	01-22-04	3624	5147
B6A5	10	20	01-22-05	3625	7AF6
B6A6	12	6	01-22-06	3626	5171
B6A7	17	4	01-22-07	3627	51F6
B6A8	19	8	01-22-08	3628	5354
B6A9	22	6	01-22-09	3629	5321
B6AA	26	10	01-22-10	362A	537F
B6AB	30	6	01-22-11	362B	53EB
B6AC	30	12	01-22-12	362C	55AC
B6AD	32	14	01-22-13	362D	5883
B6AE	46	9	01-22-14	362E	5CE1
B6AF	57	11	01-22-15	362F	5F37
B6B0	57	16	01-22-16	3630	5F4A
B6B1	61	8	01-22-17	3631	602F
B6B2	61	10	01-22-18	3632	6050
B6B3	61	10	01-22-19	3633	606D
B6B4	64	9	01-22-20	3634	631F
B6B5	66	11	01-22-21	3635	6559
B6B6	75	16	01-22-22	3636	6A4B
B6B7	85	8	01-22-23	3637	6CC1
B6B8	94	7	01-22-24	3638	72C2
B6B9	94	9	01-22-25	3639	72ED
B6BA	111	17	01-22-26	363A	77EF
B6BB	130	10	01-22-27	363B	80F8
B6BC	130	10	01-22-28	363C	8105
B6BD	134	16	01-22-29	363D	8208
B6BE	140	15	01-22-30	363E	854E
B6BF	163	11	01-22-31	363F	90F7
B6C0	167	19	01-22-32	3640	93E1
B6C1	180	20	01-22-33	3641	97FF
B6C2	184	20	01-22-34	3642	9957
B6C3	187	22	01-22-35	3643	9A5A
B6C4	9	6	01-22-36	3644	4EF0
B6C5	15	16	01-22-37	3645	51DD
B6C6	32	8	01-22-38	3646	5C2D
B6C7	72	12	01-22-39	3647	6681
B6C8	75	13	01-22-40	3648	696D
B6C9	44	7	01-22-41	3649	5C40
B6CA	73	6	01-22-42	364A	66F2
B6CB	75	12	01-22-43	364B	6975
B6CC	96	5	01-22-44	364C	7389
B6CD	75	10	01-22-45	364D	6850
B6CE	119	9	01-22-46	364E	7C81
B6CF	9	12	01-22-47	364F	50C5
B6D0	19	12	01-22-48	3650	52E4
B6D1	32	7	01-22-49	3651	5747
B6D2	50	3	01-22-50	3652	5DFE
B6D3	50	16	01-22-51	3653	9326
B6D4	69	4	01-22-52	3654	65A4
B6D5	76	8	01-22-53	3655	6B23
B6D6	76	12	01-22-54	3656	6B3D
B6D7	96	12	01-22-55	3657	7434
B6D8	113	13	01-22-56	3658	7981
B6D9	114	13	01-22-57	3659	79BD
B6DA	118	12	01-22-58	365A	7B4B
B6DB	120	15	01-22-59	365B	7DCA
B6DC	140	7	01-22-60	365C	82B9
B6DD	140	11	01-22-61	365D	83CC
B6DE	145	9	01-22-62	365E	887F
B6DF	145	18	01-22-63	365F	895F
B6E0	149	17	01-22-64	3660	8B39
B6E1	162	7	01-22-65	3661	8FD1
B6E2	167	8	01-22-66	3662	91D1
B6E3	30	7	01-22-67	3663	541F
B6E4	167	14	01-22-68	3664	9280
B6E5	5	2	01-22-69	3665	4E5D
B6E6	9	10	01-22-70	3666	5036
B6E7	30	5	01-22-71	3667	53E5
B6E8	22	4	01-22-72	3668	533A
B6E9	94	8	01-22-73	3669	72D7
B6EA	96	7	01-22-74	366A	7396
B6EB	111	10	01-22-75	366B	77E9
B6EC	140	8	01-22-76	366C	82E6
B6ED	158	11	01-22-77	366D	8EAF
B6EE	187	14	01-22-78	366E	99C6
B6EF	187	15	01-22-79	366F	99C8
B6F0	187	15	01-22-80	3670	99D2
B6F1	12	8	01-22-81	3671	5177
B6F2	61	13	01-22-82	3672	611A
B6F3	141	13	01-22-83	3673	865E
B6F4	30	12	01-22-84	3674	55B0
B6F5	116	8	01-22-85	3675	7A7A
B6F6	9	11	01-22-86	3676	5076
B6F7	40	12	01-22-87	3677	5BD3
B6F8	162	12	01-22-88	3678	9047
B6F9	170	12	01-22-89	3679	9685
B6FA	2	7	01-22-90	367A	4E32
B6FB	75	17	01-22-91	367B	6ADB
B6FC	167	11	01-22-92	367C	91E7
B6FD	44	10	01-22-93	367D	5C51
B6FE	44	8	01-22-94	367E	5C48
B7A1	64	11	01-23-01	3721	6398
B7A2	116	13	01-23-02	3722	7A9F
B7A3	73	8	01-23-03	3723	6C93
B7A4	177	13	01-23-04	3724	9774
B7A5	120	22	01-23-05	3725	8F61
B7A6	85	14	01-23-06	3726	7AAA
B7A7	86	14	01-23-07	3727	718A
B7A8	170	12	01-23-08	3728	9688
B7A9	119	9	01-23-09	3729	7C82
B7AA	75	10	01-23-10	372A	6817
B7AB	120	19	01-23-11	372B	7E70
B7AC	75	10	01-23-12	372C	6851
B7AD	167	17	01-23-13	372D	936C
B7AE	19	15	01-23-14	372E	52F2
B7AF	30	7	01-23-15	372F	541B
B7B0	140	16	01-23-16	3730	85AB
B7B1	149	10	01-23-17	3731	8A13
B7B2	123	13	01-23-18	3732	7FA4
B7B3	159	9	01-23-19	3733	8ECD
B7B4	163	10	01-23-20	3734	90E1
B7B5	25	8	01-23-21	3735	5366
B7B6	145	11	01-23-22	3736	8888
B7B7	163	7	01-23-23	3737	7941
B7B8	9	9	01-23-24	3738	4FC2
B7B9	9	13	01-23-25	3739	50BE
B7BA	18	6	01-23-26	373A	5211
B7BB	10	5	01-23-27	373B	5144
B7BC	30	11	01-23-28	373C	5553
B7BD	32	6	01-23-29	373D	572D
B7BE	32	10	01-23-30	373E	73EA
B7BF	32	9	01-23-31	373F	578B
B7C0	37	9	01-23-32	3740	5951
B7C1	59	7	01-23-33	3741	5F62
B7C2	60	8	01-23-34	3742	5F84
B7C3	61	10	01-23-35	3743	6075
B7C4	61	15	01-23-36	3744	6176
B7C5	61	15	01-23-37	3745	6167
B7C6	61	16	01-23-38	3746	61A9
B7C7	64	11	01-23-39	3747	63B2
B7C8	64	13	01-23-40	3748	643A
B7C9	66	12	01-23-41	3749	656C
B7CA	72	12	01-23-42	374A	666F
B7CB	75	10	01-23-43	374B	6842
B7CC	85	11	01-23-44	374C	6E13
B7CD	102	11	01-23-45	374D	7566
B7CE	115	15	01-23-46	374E	7A3D
B7CF	120	7	01-23-47	374F	7CFB
B7D0	120	11	01-23-48	3750	7D4C
B7D1	120	13	01-23-49	3751	7D99
B7D2	120	17	01-23-50	3752	7E4B
B7D3	122	13	01-23-51	3753	7F6B
B7D4	140	8	01-23-52	3754	830E
B7D5	140	9	01-23-53	3755	834A
B7D6	142	11	01-23-54	3756	86CD
B7D7	149	9	01-23-55	3757	8A08
B7D8	149	13	01-23-56	3758	8A63
B7D9	149	19	01-23-57	3759	8B66
B7DA	159	12	01-23-58	375A	8EFD
B7DB	181	14	01-23-59	375B	981A
B7DC	196	19	01-23-60	375C	9D8F
B7DD	140	7	01-23-61	375D	82B8
B7DE	162	7	01-23-62	375E	8FCE
B7DF	195	19	01-23-63	375F	9BE8
B7E0	18	15	01-23-64	3760	5287
B7E1	62	12	01-23-65	3761	621F
B7E2	64	15	01-23-66	3762	6483
B7E3	85	16	01-23-67	3763	6FC0
B7E4	170	13	01-23-68	3764	9699
B7E5	75	10	01-23-69	3765	6841
B7E6	9	13	01-23-70	3766	5091
B7E7	76	4	01-23-71	3767	6B20
B7E8	85	7	01-23-72	3768	6C7A
B7E9	85	15	01-23-73	3769	6F54
B7EA	116	5	01-23-74	376A	7A74
B7EB	120	12	01-23-75	376B	7D50
B7EC	143	6	01-23-76	376C	8840
B7ED	149	11	01-23-77	376D	8A23
B7EE	74	4	01-23-78	376E	6708
B7EF	9	6	01-23-79	376F	4EF6
B7F0	9	10	01-23-80	3770	5039
B7F1	9	11	01-23-81	3771	5026
B7F2	9	11	01-23-82	3772	5065
B7F3	12	10	01-23-83	3773	517C
B7F4	18	8	01-23-84	3774	5238
B7F5	18	10	01-23-85	3775	5263
B7F6	30	12	01-23-86	3776	55A7
B7F7	31	12	01-23-87	3777	570F
B7F8	32	12	01-23-88	3778	5805
B7F9	38	13	01-23-89	3779	5ACC
B7FA	54	9	01-23-90	377A	5EFA
B7FB	61	16	01-23-91	377B	61B2
B7FC	61	20	01-23-92	377C	61F8
B7FD	64	10	01-23-93	377D	62F3
B7FE	64	12	01-23-94	377E	6372
B8A1	75	12	01-24-01	3821	691C
B8A2	75	15	01-24-02	3822	6A29
B8A3	93	11	01-24-03	3823	727D
B8A4	94	4	01-24-04	3824	72AC
B8A5	94	13	01-24-05	3825	732E
B8A6	112	9	01-24-06	3826	7814
B8A7	112	12	01-24-07	3827	786F
B8A8	120	13	01-24-08	3828	7D79
B8A9	109	9	01-24-09	3829	770C
B8AA	130	8	01-24-10	382A	80A9
B8AB	147	7	01-24-11	382B	898B
B8AC	149	17	01-24-12	382C	8B19
B8AD	154	16	01-24-13	382D	8CE2
B8AE	159	10	01-24-14	382E	8ED2
B8AF	162	13	01-24-15	382F	9063
B8B0	167	17	01-24-16	3830	9375
B8B1	170	11	01-24-17	3831	967A
B8B2	181	18	01-24-18	3832	9855
B8B3	187	18	01-24-19	3833	9A13
B8B4	197	19	01-24-20	3834	9E78
B8B5	10	4	01-24-21	3835	5143
B8B6	27	10	01-24-22	3836	539F
B8B7	30	17	01-24-23	3837	53B3
B8B8	52	4	01-24-24	3838	5E7B
B8B9	57	8	01-24-25	3839	5F26
B8BA	85	12	01-24-26	383A	6E1B
B8BB	85	13	01-24-27	383B	6E90
B8BC	95	5	01-24-28	383C	7384
B8BD	96	11	01-24-29	383D	73FE
B8BE	120	11	01-24-30	383E	7D43
B8BF	137	11	01-24-31	383F	8237
B8C0	149	7	01-24-32	3840	8A00
B8C1	149	16	01-24-33	3841	8AFA
B8C2	170	9	01-24-34	3842	9650
B8C3	4	5	01-24-35	3843	4E4E
B8C4	9	10	01-24-36	3844	500B
B8C5	30	5	01-24-37	3845	53E4
B8C6	30	8	01-24-38	3846	547C
B8C7	31	8	01-24-39	3847	56FA
B8C8	38	8	01-24-40	3848	59D1
B8C9	39	9	01-24-41	3849	5B64
B8CA	49	3	01-24-42	384A	5DF1
B8CB	53	10	01-24-43	384B	5EAB
B8CC	57	9	01-24-44	384C	5F27
B8CD	63	4	01-24-45	384D	6238
B8CE	66	9	01-24-46	384E	6545
B8CF	75	9	01-24-47	384F	67AF
B8D0	85	12	01-24-48	3850	6E56
B8D1	94	8	01-24-49	3851	72D0
B8D2	119	15	01-24-50	3852	7CCA
B8D3	145	11	01-24-51	3853	88B4
B8D4	130	8	01-24-52	3854	80A1
B8D5	130	9	01-24-53	3855	80E1
B8D6	140	11	01-24-54	3856	83F0
B8D7	141	8	01-24-55	3857	864E
B8D8	149	13	01-24-56	3858	8A87
B8D9	157	13	01-24-57	3859	8DE8
B8DA	167	13	01-24-58	385A	9237
B8DB	172	12	01-24-59	385B	96C7
B8DC	181	21	01-24-60	385C	9867
B8DD	207	13	01-24-61	385D	9F13
B8DE	7	4	01-24-62	385E	4E94
B8DF	7	4	01-24-63	385F	4E92
B8E0	9	6	01-24-64	3860	4F0D
B8E1	24	4	01-24-65	3861	5348
B8E2	30	7	01-24-66	3862	5449
B8E3	30	7	01-24-67	3863	543E
B8E4	38	10	01-24-68	3864	5A2F
B8E5	60	9	01-24-69	3865	5F8C
B8E6	60	12	01-24-70	3866	5FA1
B8E7	61	10	01-24-71	3867	609F
B8E8	75	11	01-24-72	3868	68A7
B8E9	75	17	01-24-73	3869	6A8E
B8EA	96	13	01-24-74	386A	745A
B8EB	112	13	01-24-75	386B	7881
B8EC	149	14	01-24-76	386C	8A9E
B8ED	149	14	01-24-77	386D	8AA4
B8EE	149	20	01-24-78	386E	8B77
B8EF	164	16	01-24-79	386F	9190
B8F0	5	3	01-24-80	3870	4E5E
B8F1	195	18	01-24-81	3871	9BC9
B8F2	8	6	01-24-82	3872	4EA4
B8F3	9	8	01-24-83	3873	4F7C
B8F4	9	9	01-24-84	3874	4FAF
B8F5	9	10	01-24-85	3875	5019
B8F6	9	10	01-24-86	3876	5016
B8F7	10	6	01-24-87	3877	5149
B8F8	12	4	01-24-88	3878	516C
B8F9	19	5	01-24-89	3879	529F
B8FA	19	8	01-24-90	387A	52B9
B8FB	20	4	01-24-91	387B	52FE
B8FC	27	9	01-24-92	387C	539A
B8FD	30	3	01-24-93	387D	53E3
B8FE	30	6	01-24-94	387E	5411
B9A1	30	6	01-25-01	3921	540E
B9A2	30	12	01-25-02	3922	5589
B9A3	32	7	01-25-03	3923	5751
B9A4	32	9	01-25-04	3924	57A2
B9A5	38	6	01-25-05	3925	597D
B9A6	39	4	01-25-06	3926	5B54
B9A7	39	7	01-25-07	3927	5B5D
B9A8	40	7	01-25-08	3928	5B8F
B9A9	48	3	01-25-09	3929	5DE5
B9AA	48	5	01-25-10	392A	5DE7
B9AB	49	9	01-25-11	392B	5DF7
B9AC	51	8	01-25-12	392C	5E78
B9AD	53	5	01-25-13	392D	5E83
B9AE	53	8	01-25-14	392E	5E9A
B9AF	53	11	01-25-15	392F	5EB7
B9B0	57	5	01-25-16	3930	5F18
B9B1	61	9	01-25-17	3931	6052
B9B2	61	12	01-25-18	3932	614C
B9B3	64	7	01-25-19	3933	6297
B9B4	64	8	01-25-20	3934	62D8
B9B5	64	11	01-25-21	3935	63A7
B9B6	66	7	01-25-22	3936	653B
B9B7	72	8	01-25-23	3937	6602
B9B8	72	10	01-25-24	3938	6643
B9B9	73	7	01-25-25	3939	66F4
B9BA	75	8	01-25-26	393A	676D
B9BB	75	10	01-25-27	393B	6821
B9BC	75	11	01-25-28	393C	6897
B9BD	75	14	01-25-29	393D	69CB
B9BE	85	6	01-25-30	393E	6C5F
B9BF	85	9	01-25-31	393F	6D2A
B9C0	85	10	01-25-32	3940	6D69
B9C1	85	12	01-25-33	3941	6E2F
B9C2	85	13	01-25-34	3942	6E9D
B9C3	102	5	01-25-35	3943	7532
B9C4	106	9	01-25-36	3944	7687
B9C5	112	12	01-25-37	3945	786C
B9C6	115	15	01-25-38	3946	7A3F
B9C7	115	17	01-25-39	3947	7CE0
B9C8	120	9	01-25-40	3948	7D05
B9C9	120	10	01-25-41	3949	7D18
B9CA	120	12	01-25-42	394A	7D5E
B9CB	120	14	01-25-43	394B	7DB1
B9CC	127	10	01-25-44	394C	8015
B9CD	125	6	01-25-45	394D	8003
B9CE	130	8	01-25-46	394E	80AF
B9CF	130	8	01-25-47	394F	80B1
B9D0	130	12	01-25-48	3950	8154
B9D1	130	14	01-25-49	3951	818F
B9D2	137	10	01-25-50	3952	822A
B9D3	140	9	01-25-51	3953	8352
B9D4	144	6	01-25-52	3954	884C
B9D5	144	16	01-25-53	3955	8861
B9D6	149	17	01-25-54	3956	8B1B
B9D7	154	10	01-25-55	3957	8CA2
B9D8	154	17	01-25-56	3958	8CFC
B9D9	163	9	01-25-57	3959	90CA
B9DA	164	14	01-25-58	395A	9175
B9DB	167	13	01-25-59	395B	9271
B9DC	112	10	01-25-60	395C	783F
B9DD	167	16	01-25-61	395D	92FC
B9DE	169	14	01-25-62	395E	95A4
B9DF	170	10	01-25-63	395F	964D
B9E0	181	12	01-25-64	3960	9805
B9E1	186	9	01-25-65	3961	9999
B9E2	189	10	01-25-66	3962	9AD8
B9E3	196	17	01-25-67	3963	9D3B
B9E4	18	10	01-25-68	3964	525B
B9E5	19	7	01-25-69	3965	52AB
B9E6	30	5	01-25-70	3966	53F7
B9E7	30	6	01-25-71	3967	5408
B9E8	32	17	01-25-72	3968	58D5
B9E9	64	9	01-25-73	3969	62F7
B9EA	85	17	01-25-74	396A	6FE0
B9EB	152	14	01-25-75	396B	8C6A
B9EC	159	21	01-25-76	396C	8F5F
B9ED	199	15	01-25-77	396D	9EB9
B9EE	10	7	01-25-78	396E	514B
B9EF	18	8	01-25-79	396F	523B
B9F0	30	7	01-25-80	3970	544A
B9F1	31	8	01-25-81	3971	56FD
B9F2	115	14	01-25-82	3972	7A40
B9F3	164	14	01-25-83	3973	9177
B9F4	196	18	01-25-84	3974	9D60
B9F5	203	11	01-25-85	3975	9ED2
B9F6	149	14	01-25-86	3976	7344
B9F7	85	14	01-25-87	3977	6F09
B9F8	130	13	01-25-88	3978	8170
B9F9	98	16	01-25-89	3979	7511
B9FA	61	8	01-25-90	397A	5FFD
B9FB	61	11	01-25-91	397B	60DA
B9FC	188	10	01-25-92	397C	9AA8
B9FD	94	8	01-25-93	397D	72DB
B9FE	162	5	01-25-94	397E	8FBC
BAA1	77	6	01-26-01	3A21	6B64
BAA2	181	11	01-26-02	3A22	9803
BAA3	9	4	01-26-03	3A23	4ECA
BAA4	31	7	01-26-04	3A24	56F0
BAA5	32	8	01-26-05	3A25	5764
BAA6	32	16	01-26-06	3A26	58BE
BAA7	38	11	01-26-07	3A27	5A5A
BAA8	61	9	01-26-08	3A28	6068
BAA9	61	17	01-26-09	3A29	61C7
BAAA	72	8	01-26-10	3A2A	660F
BAAB	72	8	01-26-11	3A2B	6606
BAAC	75	10	01-26-12	3A2C	6839
BAAD	75	11	01-26-13	3A2D	68B1
BAAE	85	11	01-26-14	3A2E	6DF7
BAAF	104	11	01-26-15	3A2F	75D5
BAB0	120	11	01-26-16	3A30	7D3A
BAB1	138	6	01-26-17	3A31	826E
BAB2	194	14	01-26-18	3A32	9B42
BAB3	7	8	01-26-19	3A33	4E9B
BAB4	9	7	01-26-20	3A34	4F50
BAB5	29	3	01-26-21	3A35	53C9
BAB6	30	10	01-26-22	3A36	5506
BAB7	46	13	01-26-23	3A37	5D6F
BAB8	48	5	01-26-24	3A38	5DE6
BAB9	48	10	01-26-25	3A39	5DEE
BABA	75	9	01-26-26	3A3A	67FB
BABB	85	7	01-26-27	3A3B	6C99
BABC	96	14	01-26-28	3A3C	7473
BABD	112	9	01-26-29	3A3D	7802
BABE	149	12	01-26-30	3A3E	8A50
BABF	167	18	01-26-31	3A3F	9396
BAC0	145	13	01-26-32	3A40	88DF
BAC1	32	7	01-26-33	3A41	5750
BAC2	53	10	01-26-34	3A42	5EA7
BAC3	64	10	01-26-35	3A43	632B
BAC4	9	13	01-26-36	3A44	50B5
BAC5	9	13	01-26-37	3A45	50AC
BAC6	13	6	01-26-38	3A46	518D
BAC7	29	12	01-26-39	3A47	6700
BAC8	30	9	01-26-40	3A48	54C9
BAC9	32	13	01-26-41	3A49	585E
BACA	38	8	01-26-42	3A4A	59BB
BACB	40	10	01-26-43	3A4B	5BB0
BACC	59	11	01-26-44	3A4C	5F69
BACD	64	3	01-26-45	3A4D	624D
BACE	64	11	01-26-46	3A4E	63A1
BACF	75	10	01-26-47	3A4F	683D
BAD0	77	13	01-26-48	3A50	6B73
BAD1	85	11	01-26-49	3A51	6E08
BAD2	86	7	01-26-50	3A52	707D
BAD3	87	8	01-26-51	3A53	91C7
BAD4	93	12	01-26-52	3A54	7280
BAD5	112	9	01-26-53	3A55	7815
BAD6	112	11	01-26-54	3A56	7826
BAD7	113	11	01-26-55	3A57	796D
BAD8	113	11	01-26-56	3A58	658E
BAD9	120	11	01-26-57	3A59	7D30
BADA	140	11	01-26-58	3A5A	83DC
BADB	145	12	01-26-59	3A5B	88C1
BADC	159	13	01-26-60	3A5C	8F09
BADD	170	14	01-26-61	3A5D	969B
BADE	18	10	01-26-62	3A5E	5264
BADF	32	6	01-26-63	3A5F	5728
BAE0	75	7	01-26-64	3A60	6750
BAE1	122	13	01-26-65	3A61	7F6A
BAE2	154	10	01-26-66	3A62	8CA1
BAE3	15	7	01-26-67	3A63	51B4
BAE4	32	7	01-26-68	3A64	5742
BAE5	32	7	01-26-69	3A65	962A
BAE6	102	12	01-26-70	3A66	583A
BAE7	75	13	01-26-71	3A67	698A
BAE8	130	8	01-26-72	3A68	80B4
BAE9	30	9	01-26-73	3A69	54B2
BAEA	46	11	01-26-74	3A6A	5D0E
BAEB	32	11	01-26-75	3A6B	57FC
BAEC	112	13	01-26-76	3A6C	7895
BAED	196	24	01-26-77	3A6D	9DFA
BAEE	9	7	01-26-78	3A6E	4F5C
BAEF	18	9	01-26-79	3A6F	524A
BAF0	30	8	01-26-80	3A70	548B
BAF1	64	13	01-26-81	3A71	643E
BAF2	72	9	01-26-82	3A72	6628
BAF3	74	10	01-26-83	3A73	6714
BAF4	75	9	01-26-84	3A74	67F5
BAF5	116	10	01-26-85	3A75	7A84
BAF6	118	12	01-26-86	3A76	7B56
BAF7	120	10	01-26-87	3A77	7D22
BAF8	167	16	01-26-88	3A78	932F
BAF9	75	10	01-26-89	3A79	685C
BAFA	195	17	01-26-90	3A7A	9BAD
BAFB	118	11	01-26-91	3A7B	7B39
BAFC	21	11	01-26-92	3A7C	5319
BAFD	13	5	01-26-93	3A7D	518A
BAFE	18	8	01-26-94	3A7E	5237
BBA1	40	14	01-27-01	3B21	5BDF
BBA2	64	9	01-27-02	3B22	62F6
BBA3	64	15	01-27-03	3B23	64AE
BBA4	64	17	01-27-04	3B24	64E6
BBA5	75	5	01-27-05	3B25	672D
BBA6	79	10	01-27-06	3B26	6BBA
BBA7	140	17	01-27-07	3B27	85A9
BBA8	172	14	01-27-08	3B28	96D1
BBA9	106	11	01-27-09	3B29	7690
BBAA	195	19	01-27-10	3B2A	9BD6
BBAB	64	10	01-27-11	3B2B	634C
BBAC	167	16	01-27-12	3B2C	9306
BBAD	195	17	01-27-13	3B2D	9BAB
BBAE	108	5	01-27-14	3B2E	76BF
BBAF	72	10	01-27-15	3B2F	6652
BBB0	1	3	01-27-16	3B30	4E09
BBB1	9	12	01-27-17	3B31	5098
BBB2	28	8	01-27-18	3B32	53C2
BBB3	46	3	01-27-19	3B33	5C71
BBB4	61	11	01-27-20	3B34	60E8
BBB5	64	15	01-27-21	3B35	6492
BBB6	66	12	01-27-22	3B36	6563
BBB7	75	10	01-27-23	3B37	685F
BBB8	86	17	01-27-24	3B38	71E6
BBB9	96	9	01-27-25	3B39	73CA
BBBA	100	11	01-27-26	3B3A	7523
BBBB	118	14	01-27-27	3B3B	7B97
BBBC	120	20	01-27-28	3B3C	7E82
BBBD	142	10	01-27-29	3B3D	8695
BBBE	149	22	01-27-30	3B3E	8B83
BBBF	154	15	01-27-31	3B3F	8CDB
BBC0	164	14	01-27-32	3B40	9178
BBC1	184	16	01-27-33	3B41	9910
BBC2	69	11	01-27-34	3B42	65AC
BBC3	72	15	01-27-35	3B43	66AB
BBC4	78	10	01-27-36	3B44	6B8B
BBC5	9	5	01-27-37	3B45	4ED5
BBC6	9	5	01-27-38	3B46	4ED4
BBC7	9	7	01-27-39	3B47	4F3A
BBC8	9	8	01-27-40	3B48	4F7F
BBC9	18	8	01-27-41	3B49	523A
BBCA	30	5	01-27-42	3B4A	53F8
BBCB	30	5	01-27-43	3B4B	53F2
BBCC	30	13	01-27-44	3B4C	55E3
BBCD	31	5	01-27-45	3B4D	56DB
BBCE	33	3	01-27-46	3B4E	58EB
BBCF	38	8	01-27-47	3B4F	59CB
BBD0	38	8	01-27-48	3B50	59C9
BBD1	38	9	01-27-49	3B51	59FF
BBD2	39	3	01-27-50	3B52	5B50
BBD3	44	9	01-27-51	3B53	5C4D
BBD4	50	5	01-27-52	3B54	5E02
BBD5	50	10	01-27-53	3B55	5E2B
BBD6	61	7	01-27-54	3B56	5FD7
BBD7	61	9	01-27-55	3B57	601D
BBD8	64	9	01-27-56	3B58	6307
BBD9	65	4	01-27-57	3B59	652F
BBDA	66	7	01-27-58	3B5A	5B5C
BBDB	69	12	01-27-59	3B5B	65AF
BBDC	70	9	01-27-60	3B5C	65BD
BBDD	72	6	01-27-61	3B5D	65E8
BBDE	75	8	01-27-62	3B5E	679D
BBDF	77	4	01-27-63	3B5F	6B62
BBE0	78	6	01-27-64	3B60	6B7B
BBE1	83	4	01-27-65	3B61	6C0F
BBE2	94	13	01-27-66	3B62	7345
BBE3	113	8	01-27-67	3B63	7949
BBE4	115	7	01-27-68	3B64	79C1
BBE5	120	6	01-27-69	3B65	7CF8
BBE6	120	10	01-27-70	3B66	7D19
BBE7	120	12	01-27-71	3B67	7D2B
BBE8	130	8	01-27-72	3B68	80A2
BBE9	130	10	01-27-73	3B69	8102
BBEA	133	6	01-27-74	3B6A	81F3
BBEB	147	11	01-27-75	3B6B	8996
BBEC	149	12	01-27-76	3B6C	8A5E
BBED	149	13	01-27-77	3B6D	8A69
BBEE	149	13	01-27-78	3B6E	8A66
BBEF	149	14	01-27-79	3B6F	8A8C
BBF0	149	16	01-27-80	3B70	8AEE
BBF1	154	13	01-27-81	3B71	8CC7
BBF2	154	15	01-27-82	3B72	8CDC
BBF3	172	14	01-27-83	3B73	96CC
BBF4	184	13	01-27-84	3B74	98FC
BBF5	211	12	01-27-85	3B75	6B6F
BBF6	6	8	01-27-86	3B76	4E8B
BBF7	9	7	01-27-87	3B77	4F3C
BBF8	9	8	01-27-88	3B78	4F8D
BBF9	10	7	01-27-89	3B79	5150
BBFA	39	6	01-27-90	3B7A	5B57
BBFB	41	6	01-27-91	3B7B	5BFA
BBFC	61	13	01-27-92	3B7C	6148
BBFD	64	9	01-27-93	3B7D	6301
BBFE	72	10	01-27-94	3B7E	6642
BCA1	76	6	01-28-01	3C21	6B21
BCA2	85	12	01-28-02	3C22	6ECB
BCA3	85	8	01-28-03	3C23	6CBB
BCA4	89	14	01-28-04	3C24	723E
BCA5	96	19	01-28-05	3C25	74BD
BCA6	104	11	01-28-06	3C26	75D4
BCA7	112	14	01-28-07	3C27	78C1
BCA8	113	5	01-28-08	3C28	793A
BCA9	126	6	01-28-09	3C29	800C
BCAA	128	6	01-28-10	3C2A	8033
BCAB	132	6	01-28-11	3C2B	81EA
BCAC	140	13	01-28-12	3C2C	8494
BCAD	160	13	01-28-13	3C2D	8F9E
BCAE	85	6	01-28-14	3C2E	6C50
BCAF	198	11	01-28-15	3C2F	9E7F
BCB0	48	6	01-28-16	3C30	5F0F
BCB1	149	19	01-28-17	3C31	8B58
BCB2	196	16	01-28-18	3C32	9D2B
BCB3	7	8	01-28-19	3C33	7AFA
BCB4	159	12	01-28-20	3C34	8EF8
BCB5	40	7	01-28-21	3C35	5B8D
BCB6	173	11	01-28-22	3C36	96EB
BCB7	1	2	01-28-23	3C37	4E03
BCB8	30	5	01-28-24	3C38	53F1
BCB9	32	11	01-28-25	3C39	57F7
BCBA	37	5	01-28-26	3C3A	5931
BCBB	38	13	01-28-27	3C3B	5AC9
BCBC	40	9	01-28-28	3C3C	5BA4
BCBD	61	11	01-28-29	3C3D	6089
BCBE	85	12	01-28-30	3C3E	6E7F
BCBF	85	14	01-28-31	3C3F	6F06
BCC0	104	10	01-28-32	3C40	75BE
BCC1	154	15	01-28-33	3C41	8CEA
BCC2	40	8	01-28-34	3C42	5B9F
BCC3	140	14	01-28-35	3C43	8500
BCC4	118	16	01-28-36	3C44	7BE0
BCC5	9	11	01-28-37	3C45	5072
BCC6	75	10	01-28-38	3C46	67F4
BCC7	140	6	01-28-39	3C47	829D
BCC8	44	12	01-28-40	3C48	5C61
BCC9	140	15	01-28-41	3C49	854A
BCCA	120	16	01-28-42	3C4A	7E1E
BCCB	30	8	01-28-43	3C4B	820E
BCCC	14	5	01-28-44	3C4C	5199
BCCD	41	10	01-28-45	3C4D	5C04
BCCE	64	11	01-28-46	3C4E	6368
BCCF	66	11	01-28-47	3C4F	8D66
BCD0	68	11	01-28-48	3C50	659C
BCD1	86	12	01-28-49	3C51	716E
BCD2	113	7	01-28-50	3C52	793E
BCD3	120	10	01-28-51	3C53	7D17
BCD4	125	8	01-28-52	3C54	8005
BCD5	149	17	01-28-53	3C55	8B1D
BCD6	159	7	01-28-54	3C56	8ECA
BCD7	162	14	01-28-55	3C57	906E
BCD8	142	11	01-28-56	3C58	86C7
BCD9	163	8	01-28-57	3C59	90AA
BCDA	9	10	01-28-58	3C5A	501F
BCDB	20	3	01-28-59	3C5B	52FA
BCDC	44	4	01-28-60	3C5C	5C3A
BCDD	75	7	01-28-61	3C5D	6753
BCDE	86	7	01-28-62	3C5E	707C
BCDF	87	17	01-28-63	3C5F	7235
BCE0	164	10	01-28-64	3C60	914C
BCE1	165	11	01-28-65	3C61	91C8
BCE2	167	16	01-28-66	3C62	932B
BCE3	30	8	01-28-67	3C63	82E5
BCE4	40	11	01-28-68	3C64	5BC2
BCE5	57	10	01-28-69	3C65	5F31
BCE6	61	12	01-28-70	3C66	60F9
BCE7	3	5	01-28-71	3C67	4E3B
BCE8	29	8	01-28-72	3C68	53D6
BCE9	40	6	01-28-73	3C69	5B88
BCEA	64	4	01-28-74	3C6A	624B
BCEB	75	6	01-28-75	3C6B	6731
BCEC	78	10	01-28-76	3C6C	6B8A
BCED	94	9	01-28-77	3C6D	72E9
BCEE	96	10	01-28-78	3C6E	73E0
BCEF	115	14	01-28-79	3C6F	7A2E
BCF0	130	13	01-28-80	3C70	816B
BCF1	156	15	01-28-81	3C71	8DA3
BCF2	164	10	01-28-82	3C72	9152
BCF3	185	9	01-28-83	3C73	9996
BCF4	9	16	01-28-84	3C74	5112
BCF5	29	8	01-28-85	3C75	53D7
BCF6	30	8	01-28-86	3C76	546A
BCF7	33	7	01-28-87	3C77	5BFF
BCF8	64	11	01-28-88	3C78	6388
BCF9	75	16	01-28-89	3C79	6A39
BCFA	120	14	01-28-90	3C7A	7DAC
BCFB	173	14	01-28-91	3C7B	9700
BCFC	9	5	01-28-92	3C7C	56DA
BCFD	29	4	01-28-93	3C7D	53CE
BCFE	30	8	01-28-94	3C7E	5468
BDA1	40	8	01-29-01	3D21	5B97
BDA2	43	12	01-29-02	3D22	5C31
BDA3	47	6	01-29-03	3D23	5DDE
BDA4	59	10	01-29-04	3D24	4FEE
BDA5	61	13	01-29-05	3D25	6101
BDA6	64	9	01-29-06	3D26	62FE
BDA7	85	9	01-29-07	3D27	6D32
BDA8	115	7	01-29-08	3D28	79C0
BDA9	115	9	01-29-09	3D29	79CB
BDAA	120	11	01-29-10	3D2A	7D42
BDAB	120	17	01-29-11	3D2B	7E4D
BDAC	124	11	01-29-12	3D2C	7FD2
BDAD	132	9	01-29-13	3D2D	81ED
BDAE	137	6	01-29-14	3D2E	821F
BDAF	140	13	01-29-15	3D2F	8490
BDB0	143	12	01-29-16	3D30	8846
BDB1	145	22	01-29-17	3D31	8972
BDB2	149	23	01-29-18	3D32	8B90
BDB3	157	19	01-29-19	3D33	8E74
BDB4	159	16	01-29-20	3D34	8F2F
BDB5	162	11	01-29-21	3D35	9031
BDB6	164	9	01-29-22	3D36	914B
BDB7	164	13	01-29-23	3D37	916C
BDB8	172	12	01-29-24	3D38	96C6
BDB9	194	17	01-29-25	3D39	919C
BDBA	9	4	01-29-26	3D3A	4EC0
BDBB	9	7	01-29-27	3D3B	4F4F
BDBC	10	6	01-29-28	3D3C	5145
BDBD	24	2	01-29-29	3D3D	5341
BDBE	60	10	01-29-30	3D3E	5F93
BDBF	62	6	01-29-31	3D3F	620E
BDC0	75	9	01-29-32	3D40	67D4
BDC1	85	5	01-29-33	3D41	6C41
BDC2	85	11	01-29-34	3D42	6E0B
BDC3	94	16	01-29-35	3D43	7363
BDC4	120	16	01-29-36	3D44	7E26
BDC5	166	9	01-29-37	3D45	91CD
BDC6	167	14	01-29-38	3D46	9283
BDC7	29	8	01-29-39	3D47	53D4
BDC8	36	6	01-29-40	3D48	5919
BDC9	40	11	01-29-41	3D49	5BBF
BDCA	85	11	01-29-42	3D4A	6DD1
BDCB	113	9	01-29-43	3D4B	795D
BDCC	120	17	01-29-44	3D4C	7E2E
BDCD	129	11	01-29-45	3D4D	7C9B
BDCE	32	14	01-29-46	3D4E	587E
BDCF	86	15	01-29-47	3D4F	719F
BDD0	17	5	01-29-48	3D50	51FA
BDD1	144	11	01-29-49	3D51	8853
BDD2	162	8	01-29-50	3D52	8FF0
BDD3	9	9	01-29-51	3D53	4FCA
BDD4	46	10	01-29-52	3D54	5CFB
BDD5	72	9	01-29-53	3D55	6625
BDD6	109	18	01-29-54	3D56	77AC
BDD7	117	12	01-29-55	3D57	7AE3
BDD8	136	12	01-29-56	3D58	821C
BDD9	187	17	01-29-57	3D59	99FF
BDDA	15	10	01-29-58	3D5A	51C6
BDDB	60	12	01-29-59	3D5B	5FAA
BDDC	72	6	01-29-60	3D5C	65EC
BDDD	75	13	01-29-61	3D5D	696F
BDDE	78	10	01-29-62	3D5E	6B89
BDDF	85	11	01-29-63	3D5F	6DF3
BDE0	85	13	01-29-64	3D60	6E96
BDE1	85	15	01-29-65	3D61	6F64
BDE2	109	9	01-29-66	3D62	76FE
BDE3	120	10	01-29-67	3D63	7D14
BDE4	162	6	01-29-68	3D64	5DE1
BDE5	162	15	01-29-69	3D65	9075
BDE6	164	15	01-29-70	3D66	9187
BDE7	181	12	01-29-71	3D67	9806
BDE8	16	5	01-29-72	3D68	51E6
BDE9	18	7	01-29-73	3D69	521D
BDEA	69	8	01-29-74	3D6A	6240
BDEB	72	12	01-29-75	3D6B	6691
BDEC	72	17	01-29-76	3D6C	66D9
BDED	85	11	01-29-77	3D6D	6E1A
BDEE	86	11	01-29-78	3D6E	5EB6
BDEF	120	14	01-29-79	3D6F	7DD2
BDF0	122	13	01-29-80	3D70	7F72
BDF1	129	10	01-29-81	3D71	66F8
BDF2	140	16	01-29-82	3D72	85AF
BDF3	140	18	01-29-83	3D73	85F7
BDF4	149	15	01-29-84	3D74	8AF8
BDF5	19	7	01-29-85	3D75	52A9
BDF6	29	9	01-29-86	3D76	53D9
BDF7	38	3	01-29-87	3D77	5973
BDF8	53	7	01-29-88	3D78	5E8F
BDF9	60	10	01-29-89	3D79	5F90
BDFA	61	10	01-29-90	3D7A	6055
BDFB	167	15	01-29-91	3D7B	92E4
BDFC	170	10	01-29-92	3D7C	9664
BDFD	9	13	01-29-93	3D7D	50B7
BDFE	9	17	01-29-94	3D7E	511F
BEA1	19	12	01-30-01	3E21	52DD
BEA2	22	6	01-30-02	3E22	5320
BEA3	24	4	01-30-03	3E23	5347
BEA4	30	5	01-30-04	3E24	53EC
BEA5	30	10	01-30-05	3E25	54E8
BEA6	30	11	01-30-06	3E26	5546
BEA7	30	11	01-30-07	3E27	5531
BEA8	30	14	01-30-08	3E28	5617
BEA9	37	13	01-30-09	3E29	5968
BEAA	38	8	01-30-10	3E2A	59BE
BEAB	38	11	01-30-11	3E2B	5A3C
BEAC	40	10	01-30-12	3E2C	5BB5
BEAD	41	10	01-30-13	3E2D	5C06
BEAE	42	3	01-30-14	3E2E	5C0F
BEAF	42	4	01-30-15	3E2F	5C11
BEB0	42	8	01-30-16	3E30	5C1A
BEB1	53	6	01-30-17	3E31	5E84
BEB2	53	7	01-30-18	3E32	5E8A
BEB3	53	15	01-30-19	3E33	5EE0
BEB4	59	14	01-30-20	3E34	5F70
BEB5	64	8	01-30-21	3E35	627F
BEB6	64	7	01-30-22	3E36	6284
BEB7	64	8	01-30-23	3E37	62DB
BEB8	64	12	01-30-24	3E38	638C
BEB9	64	11	01-30-25	3E39	6377
BEBA	72	8	01-30-26	3E3A	6607
BEBB	72	8	01-30-27	3E3B	660C
BEBC	72	9	01-30-28	3E3C	662D
BEBD	72	12	01-30-29	3E3D	6676
BEBE	75	8	01-30-30	3E3E	677E
BEBF	75	11	01-30-31	3E3F	68A2
BEC0	75	15	01-30-32	3E40	6A1F
BEC1	75	16	01-30-33	3E41	6A35
BEC2	85	8	01-30-34	3E42	6CBC
BEC3	85	10	01-30-35	3E43	6D88
BEC4	85	11	01-30-36	3E44	6E09
BEC5	85	12	01-30-37	3E45	6E58
BEC6	86	12	01-30-38	3E46	713C
BEC7	86	12	01-30-39	3E47	7126
BEC8	86	13	01-30-40	3E48	7167
BEC9	104	10	01-30-41	3E49	75C7
BECA	109	9	01-30-42	3E4A	7701
BECB	112	12	01-30-43	3E4B	785D
BECC	112	17	01-30-44	3E4C	7901
BECD	113	10	01-30-45	3E4D	7965
BECE	115	10	01-30-46	3E4E	79F0
BECF	117	11	01-30-47	3E4F	7AE0
BED0	118	10	01-30-48	3E50	7B11
BED1	119	12	01-30-49	3E51	7CA7
BED2	120	11	01-30-50	3E52	7D39
BED3	130	7	01-30-51	3E53	8096
BED4	140	11	01-30-52	3E54	83D6
BED5	140	13	01-30-53	3E55	848B
BED6	140	15	01-30-54	3E56	8549
BED7	144	15	01-30-55	3E57	885D
BED8	145	14	01-30-56	3E58	88F3
BED9	149	11	01-30-57	3E59	8A1F
BEDA	149	12	01-30-58	3E5A	8A3C
BEDB	149	12	01-30-59	3E5B	8A54
BEDC	149	13	01-30-60	3E5C	8A73
BEDD	152	12	01-30-61	3E5D	8C61
BEDE	154	15	01-30-62	3E5E	8CDE
BEDF	164	17	01-30-63	3E5F	91A4
BEE0	167	13	01-30-64	3E60	9266
BEE1	167	17	01-30-65	3E61	937E
BEE2	167	20	01-30-66	3E62	9418
BEE3	170	14	01-30-67	3E63	969C
BEE4	177	16	01-30-68	3E64	9798
BEE5	1	3	01-30-69	3E65	4E0A
BEE6	1	3	01-30-70	3E66	4E08
BEE7	1	6	01-30-71	3E67	4E1E
BEE8	4	9	01-30-72	3E68	4E57
BEE9	14	4	01-30-73	3E69	5197
BEEA	18	11	01-30-74	3E6A	5270
BEEB	32	9	01-30-75	3E6B	57CE
BEEC	32	12	01-30-76	3E6C	5834
BEED	32	16	01-30-77	3E6D	58CC
BEEE	38	16	01-30-78	3E6E	5B22
BEEF	50	11	01-30-79	3E6F	5E38
BEF0	61	11	01-30-80	3E70	60C5
BEF1	64	18	01-30-81	3E71	64FE
BEF2	75	7	01-30-82	3E72	6761
BEF3	75	7	01-30-83	3E73	6756
BEF4	85	9	01-30-84	3E74	6D44
BEF5	94	7	01-30-85	3E75	72B6
BEF6	102	12	01-30-86	3E76	7573
BEF7	115	18	01-30-87	3E77	7A63
BEF8	140	13	01-30-88	3E78	84B8
BEF9	149	20	01-30-89	3E79	8B72
BEFA	164	20	01-30-90	3E7A	91B8
BEFB	167	16	01-30-91	3E7B	9320
BEFC	30	15	01-30-92	3E7C	5631
BEFD	32	11	01-30-93	3E7D	57F4
BEFE	50	13	01-30-94	3E7E	98FE
BFA1	64	9	01-31-01	3F21	62ED
BFA2	75	12	01-31-02	3F22	690D
BFA3	78	12	01-31-03	3F23	6B96
BFA4	86	17	01-31-04	3F24	71ED
BFA5	120	18	01-31-05	3F25	7E54
BFA6	128	18	01-31-06	3F26	8077
BFA7	139	6	01-31-07	3F27	8272
BFA8	148	13	01-31-08	3F28	89E6
BFA9	184	9	01-31-09	3F29	98DF
BFAA	142	14	01-31-10	3F2A	8755
BFAB	41	10	01-31-11	3F2B	8FB1
BFAC	44	5	01-31-12	3F2C	5C3B
BFAD	9	7	01-31-13	3F2D	4F38
BFAE	9	9	01-31-14	3F2E	4FE1
BFAF	9	9	01-31-15	3F2F	4FB5
BFB0	30	10	01-31-16	3F30	5507
BFB1	38	10	01-31-17	3F31	5A20
BFB2	40	13	01-31-18	3F32	5BDD
BFB3	40	15	01-31-19	3F33	5BE9
BFB4	61	4	01-31-20	3F34	5FC3
BFB5	61	13	01-31-21	3F35	614E
BFB6	64	10	01-31-22	3F36	632F
BFB7	69	13	01-31-23	3F37	65B0
BFB8	72	10	01-31-24	3F38	664B
BFB9	75	12	01-31-25	3F39	68EE
BFBA	75	14	01-31-26	3F3A	699B
BFBB	85	10	01-31-27	3F3B	6D78
BFBC	85	11	01-31-28	3F3C	6DF1
BFBD	102	5	01-31-29	3F3D	7533
BFBE	104	10	01-31-30	3F3E	75B9
BFBF	109	10	01-31-31	3F3F	771F
BFC0	113	9	01-31-32	3F40	795E
BFC1	115	10	01-31-33	3F41	79E6
BFC2	120	11	01-31-34	3F42	7D33
BFC3	131	7	01-31-35	3F43	81E3
BFC4	140	7	01-31-36	3F44	82AF
BFC5	140	16	01-31-37	3F45	85AA
BFC6	147	16	01-31-38	3F46	89AA
BFC7	149	12	01-31-39	3F47	8A3A
BFC8	158	7	01-31-40	3F48	8EAB
BFC9	160	7	01-31-41	3F49	8F9B
BFCA	162	11	01-31-42	3F4A	9032
BFCB	167	10	01-31-43	3F4B	91DD
BFCC	173	15	01-31-44	3F4C	9707
BFCD	9	2	01-31-45	3F4D	4EBA
BFCE	9	4	01-31-46	3F4E	4EC1
BFCF	18	3	01-31-47	3F4F	5203
BFD0	32	14	01-31-48	3F50	5875
BFD1	33	4	01-31-49	3F51	58EC
BFD2	41	12	01-31-50	3F52	5C0B
BFD3	99	9	01-31-51	3F53	751A
BFD4	44	6	01-31-52	3F54	5C3D
BFD5	130	13	01-31-53	3F55	814E
BFD6	149	10	01-31-54	3F56	8A0A
BFD7	162	6	01-31-55	3F57	8FC5
BFD8	170	10	01-31-56	3F58	9663
BFD9	177	12	01-31-57	3F59	976D
BFDA	118	11	01-31-58	3F5A	7B25
BFDB	149	15	01-31-59	3F5B	8ACF
BFDC	181	12	01-31-60	3F5C	9808
BFDD	164	12	01-31-61	3F5D	9162
BFDE	31	7	01-31-62	3F5E	56F3
BFDF	27	12	01-31-63	3F5F	53A8
BFE0	162	10	01-31-64	3F60	9017
BFE1	30	7	01-31-65	3F61	5439
BFE2	32	8	01-31-66	3F62	5782
BFE3	50	9	01-31-67	3F63	5E25
BFE4	64	11	01-31-68	3F64	63A8
BFE5	85	4	01-31-69	3F65	6C34
BFE6	86	8	01-31-70	3F66	708A
BFE7	109	13	01-31-71	3F67	7761
BFE8	119	10	01-31-72	3F68	7C8B
BFE9	124	14	01-31-73	3F69	7FE0
BFEA	145	10	01-31-74	3F6A	8870
BFEB	162	12	01-31-75	3F6B	9042
BFEC	164	11	01-31-76	3F6C	9154
BFED	167	16	01-31-77	3F6D	9310
BFEE	167	16	01-31-78	3F6E	9318
BFEF	162	12	01-31-79	3F6F	968F
BFF0	96	13	01-31-80	3F70	745E
BFF1	188	19	01-31-81	3F71	9AC4
BFF2	46	11	01-31-82	3F72	5D07
BFF3	46	13	01-31-83	3F73	5D69
BFF4	66	13	01-31-84	3F74	6570
BFF5	75	8	01-31-85	3F75	67A2
BFF6	156	17	01-31-86	3F76	8DA8
BFF7	172	18	01-31-87	3F77	96DB
BFF8	64	11	01-31-88	3F78	636E
BFF9	75	7	01-31-89	3F79	6749
BFFA	75	12	01-31-90	3F7A	6919
BFFB	140	11	01-31-91	3F7B	83C5
BFFC	181	14	01-31-92	3F7C	9817
BFFD	172	11	01-31-93	3F7D	96C0
BFFE	145	13	01-31-94	3F7E	88FE
C0A1	85	15	01-32-01	4021	6F84
C0A2	64	14	01-32-02	4022	647A
C0A3	41	3	01-32-03	4023	5BF8
C0A4	1	5	01-32-04	4024	4E16
C0A5	85	19	01-32-05	4025	702C
C0A6	102	10	01-32-06	4026	755D
C0A7	72	9	01-32-07	4027	662F
C0A8	15	10	01-32-08	4028	51C4
C0A9	18	8	01-32-09	4029	5236
C0AA	19	13	01-32-10	402A	52E2
C0AB	38	8	01-32-11	402B	59D3
C0AC	60	8	01-32-12	402C	5F81
C0AD	61	8	01-32-13	402D	6027
C0AE	62	6	01-32-14	402E	6210
C0AF	66	9	01-32-15	402F	653F
C0B0	66	16	01-32-16	4030	6574
C0B1	72	9	01-32-17	4031	661F
C0B2	72	12	01-32-18	4032	6674
C0B3	75	12	01-32-19	4033	68F2
C0B4	75	10	01-32-20	4034	6816
C0B5	77	5	01-32-21	4035	6B63
C0B6	85	11	01-32-22	4036	6E05
C0B7	93	9	01-32-23	4037	7272
C0B8	100	5	01-32-24	4038	751F
C0B9	108	11	01-32-25	4039	76DB
C0BA	119	14	01-32-26	403A	7CBE
C0BB	128	13	01-32-27	403B	8056
C0BC	33	7	01-32-28	403C	58F0
C0BD	145	14	01-32-29	403D	88FD
C0BE	146	6	01-32-30	403E	897F
C0BF	149	13	01-32-31	403F	8AA0
C0C0	149	14	01-32-32	4040	8A93
C0C1	149	15	01-32-33	4041	8ACB
C0C2	162	10	01-32-34	4042	901D
C0C3	164	16	01-32-35	4043	9192
C0C4	174	8	01-32-36	4044	9752
C0C5	174	14	01-32-37	4045	9759
C0C6	210	8	01-32-38	4046	6589
C0C7	115	12	01-32-39	4047	7A0E
C0C8	130	10	01-32-40	4048	8106
C0C9	29	10	01-32-41	4049	96BB
C0CA	50	10	01-32-42	404A	5E2D
C0CB	61	11	01-32-43	404B	60DC
C0CC	62	11	01-32-44	404C	621A
C0CD	69	5	01-32-45	404D	65A5
C0CE	72	8	01-32-46	404E	6614
C0CF	75	8	01-32-47	404F	6790
C0D0	112	5	01-32-48	4050	77F3
C0D1	115	16	01-32-49	4051	7A4D
C0D2	118	20	01-32-50	4052	7C4D
C0D3	120	17	01-32-51	4053	7E3E
C0D4	130	10	01-32-52	4054	810A
C0D5	154	11	01-32-53	4055	8CAC
C0D6	155	7	01-32-54	4056	8D64
C0D7	157	13	01-32-55	4057	8DE1
C0D8	157	18	01-32-56	4058	8E5F
C0D9	181	14	01-32-57	4059	78A9
C0DA	18	4	01-32-58	405A	5207
C0DB	64	8	01-32-59	405B	62D9
C0DC	64	11	01-32-60	405C	63A5
C0DD	64	13	01-32-61	405D	6442
C0DE	69	7	01-32-62	405E	6298
C0DF	79	11	01-32-63	405F	8A2D
C0E0	116	9	01-32-64	4060	7A83
C0E1	118	13	01-32-65	4061	7BC0
C0E2	149	14	01-32-66	4062	8AAC
C0E3	173	11	01-32-67	4063	96EA
C0E4	120	12	01-32-68	4064	7D76
C0E5	135	6	01-32-69	4065	820C
C0E6	142	15	01-32-70	4066	8749
C0E7	9	5	01-32-71	4067	4ED9
C0E8	10	6	01-32-72	4068	5148
C0E9	24	3	01-32-73	4069	5343
C0EA	25	5	01-32-74	406A	5360
C0EB	40	9	01-32-75	406B	5BA3
C0EC	41	9	01-32-76	406C	5C02
C0ED	42	6	01-32-77	406D	5C16
C0EE	47	3	01-32-78	406E	5DDD
C0EF	62	13	01-32-79	406F	6226
C0F0	63	10	01-32-80	4070	6247
C0F1	64	15	01-32-81	4071	64B0
C0F2	75	10	01-32-82	4072	6813
C0F3	75	10	01-32-83	4073	6834
C0F4	85	9	01-32-84	4074	6CC9
C0F5	85	9	01-32-85	4075	6D45
C0F6	85	9	01-32-86	4076	6D17
C0F7	85	9	01-32-87	4077	67D3
C0F8	85	15	01-32-88	4078	6F5C
C0F9	86	13	01-32-89	4079	714E
C0FA	86	14	01-32-90	407A	717D
C0FB	103	11	01-32-91	407B	65CB
C0FC	116	9	01-32-92	407C	7A7F
C0FD	118	15	01-32-93	407D	7BAD
C0FE	120	15	01-32-94	407E	7DDA
C1A1	120	17	01-33-01	4121	7E4A
C1A2	123	13	01-33-02	4122	7FA8
C1A3	130	13	01-33-03	4123	817A
C1A4	136	6	01-33-04	4124	821B
C1A5	137	11	01-33-05	4125	8239
C1A6	140	16	01-33-06	4126	85A6
C1A7	149	13	01-33-07	4127	8A6E
C1A8	154	13	01-33-08	4128	8CCE
C1A9	157	13	01-33-09	4129	8DF5
C1AA	162	15	01-33-10	412A	9078
C1AB	162	15	01-33-11	412B	9077
C1AC	167	14	01-33-12	412C	92AD
C1AD	167	14	01-33-13	412D	9291
C1AE	169	10	01-33-14	412E	9583
C1AF	195	17	01-33-15	412F	9BAE
C1B0	18	9	01-33-16	4130	524D
C1B1	30	12	01-33-17	4131	5584
C1B2	85	14	01-33-18	4132	6F38
C1B3	86	12	01-33-19	4133	7136
C1B4	96	6	01-33-20	4134	5168
C1B5	113	13	01-33-21	4135	7985
C1B6	120	18	01-33-22	4136	7E55
C1B7	130	16	01-33-23	4137	81B3
C1B8	119	15	01-33-24	4138	7CCE
C1B9	30	14	01-33-25	4139	564C
C1BA	32	13	01-33-26	413A	5851
C1BB	46	8	01-33-27	413B	5CA8
C1BC	64	11	01-33-28	413C	63AA
C1BD	73	12	01-33-29	413D	66FE
C1BE	73	11	01-33-30	413E	66FD
C1BF	75	13	01-33-31	413F	695A
C1C0	94	8	01-33-32	4140	72D9
C1C1	103	12	01-33-33	4141	758F
C1C2	103	12	01-33-34	4142	758E
C1C3	112	18	01-33-35	4143	790E
C1C4	113	9	01-33-36	4144	7956
C1C5	115	10	01-33-37	4145	79DF
C1C6	119	11	01-33-38	4146	7C97
C1C7	120	10	01-33-39	4147	7D20
C1C8	120	11	01-33-40	4148	7D44
C1C9	140	19	01-33-41	4149	8607
C1CA	149	12	01-33-42	414A	8A34
C1CB	170	8	01-33-43	414B	963B
C1CC	162	13	01-33-44	414C	9061
C1CD	208	13	01-33-45	414D	9F20
C1CE	9	13	01-33-46	414E	50E7
C1CF	18	12	01-33-47	414F	5275
C1D0	29	4	01-33-48	4150	53CC
C1D1	29	18	01-33-49	4151	53E2
C1D2	30	10	01-33-50	4152	5009
C1D3	30	12	01-33-51	4153	55AA
C1D4	33	6	01-33-52	4154	58EE
C1D5	37	9	01-33-53	4155	594F
C1D6	37	11	01-33-54	4156	723D
C1D7	40	7	01-33-55	4157	5B8B
C1D8	44	14	01-33-56	4158	5C64
C1D9	50	5	01-33-57	4159	531D
C1DA	61	12	01-33-58	415A	60E3
C1DB	61	13	01-33-59	415B	60F3
C1DC	64	10	01-33-60	415C	635C
C1DD	64	11	01-33-61	415D	6383
C1DE	64	10	01-33-62	415E	633F
C1DF	64	11	01-33-63	415F	63BB
C1E0	64	16	01-33-64	4160	64CD
C1E1	72	6	01-33-65	4161	65E9
C1E2	73	11	01-33-66	4162	66F9
C1E3	75	11	01-33-67	4163	5DE3
C1E4	75	14	01-33-68	4164	69CD
C1E5	75	15	01-33-69	4165	69FD
C1E6	85	14	01-33-70	4166	6F15
C1E7	86	17	01-33-71	4167	71E5
C1E8	87	6	01-33-72	4168	4E89
C1E9	104	12	01-33-73	4169	75E9
C1EA	109	9	01-33-74	416A	76F8
C1EB	116	11	01-33-75	416B	7A93
C1EC	119	17	01-33-76	416C	7CDF
C1ED	120	14	01-33-77	416D	7DCF
C1EE	120	14	01-33-78	416E	7D9C
C1EF	128	14	01-33-79	416F	8061
C1F0	140	9	01-33-80	4170	8349
C1F1	140	9	01-33-81	4171	8358
C1F2	140	12	01-33-82	4172	846C
C1F3	140	13	01-33-83	4173	84BC
C1F4	140	19	01-33-84	4174	85FB
C1F5	145	12	01-33-85	4175	88C5
C1F6	156	7	01-33-86	4176	8D70
C1F7	162	9	01-33-87	4177	9001
C1F8	162	14	01-33-88	4178	906D
C1F9	167	18	01-33-89	4179	9397
C1FA	173	17	01-33-90	417A	971C
C1FB	187	18	01-33-91	417B	9A12
C1FC	9	14	01-33-92	417C	50CF
C1FD	32	14	01-33-93	417D	5897
C1FE	61	14	01-33-94	417E	618E
C2A1	130	19	01-34-01	4221	81D3
C2A2	140	15	01-34-02	4222	8535
C2A3	154	18	01-34-03	4223	8D08
C2A4	162	10	01-34-04	4224	9020
C2A5	9	9	01-34-05	4225	4FC3
C2A6	9	11	01-34-06	4226	5074
C2A7	18	9	01-34-07	4227	5247
C2A8	26	7	01-34-08	4228	5373
C2A9	61	10	01-34-09	4229	606F
C2AA	64	10	01-34-10	422A	6349
C2AB	75	7	01-34-11	422B	675F
C2AC	85	12	01-34-12	422C	6E2C
C2AD	157	7	01-34-13	422D	8DB3
C2AE	162	10	01-34-14	422E	901F
C2AF	9	9	01-34-15	422F	4FD7
C2B0	44	12	01-34-16	4230	5C5E
C2B1	62	13	01-34-17	4231	8CCA
C2B2	70	11	01-34-18	4232	65CF
C2B3	120	13	01-34-19	4233	7D9A
C2B4	24	8	01-34-20	4234	5352
C2B5	145	10	01-34-21	4235	8896
C2B6	12	8	01-34-22	4236	5176
C2B7	64	12	01-34-23	4237	63C3
C2B8	39	6	01-34-24	4238	5B58
C2B9	39	10	01-34-25	4239	5B6B
C2BA	41	12	01-34-26	423A	5C0A
C2BB	64	13	01-34-27	423B	640D
C2BC	75	7	01-34-28	423C	6751
C2BD	162	13	01-34-29	423D	905C
C2BE	9	5	01-34-30	423E	4ED6
C2BF	36	6	01-34-31	423F	591A
C2C0	37	4	01-34-32	4240	592A
C2C1	85	7	01-34-33	4241	6C70
C2C2	149	12	01-34-34	4242	8A51
C2C3	30	11	01-34-35	4243	553E
C2C4	32	12	01-34-36	4244	5815
C2C5	38	7	01-34-37	4245	59A5
C2C6	61	12	01-34-38	4246	60F0
C2C7	64	5	01-34-39	4247	6253
C2C8	75	9	01-34-40	4248	67C1
C2C9	75	11	01-34-41	4249	8235
C2CA	75	13	01-34-42	424A	6955
C2CB	170	8	01-34-43	424B	9640
C2CC	187	14	01-34-44	424C	99C4
C2CD	187	19	01-34-45	424D	9A28
C2CE	9	7	01-34-46	424E	4F53
C2CF	32	11	01-34-47	424F	5806
C2D0	41	7	01-34-48	4250	5BFE
C2D1	41	9	01-34-49	4251	8010
C2D2	46	8	01-34-50	4252	5CB1
C2D3	50	10	01-34-51	4253	5E2F
C2D4	60	9	01-34-52	4254	5F85
C2D5	61	9	01-34-53	4255	6020
C2D6	61	14	01-34-54	4256	614B
C2D7	62	17	01-34-55	4257	6234
C2D8	73	12	01-34-56	4258	66FF
C2D9	85	10	01-34-57	4259	6CF0
C2DA	85	13	01-34-58	425A	6EDE
C2DB	130	9	01-34-59	425B	80CE
C2DC	130	13	01-34-60	425C	817F
C2DD	140	8	01-34-61	425D	82D4
C2DE	145	11	01-34-62	425E	888B
C2DF	154	12	01-34-63	425F	8CB8
C2E0	162	9	01-34-64	4260	9000
C2E1	162	11	01-34-65	4261	902E
C2E2	170	12	01-34-66	4262	968A
C2E3	203	16	01-34-67	4263	9EDB
C2E4	195	19	01-34-68	4264	9BDB
C2E5	9	5	01-34-69	4265	4EE3
C2E6	30	5	01-34-70	4266	53F0
C2E7	37	3	01-34-71	4267	5927
C2E8	118	11	01-34-72	4268	7B2C
C2E9	164	16	01-34-73	4269	918D
C2EA	181	18	01-34-74	426A	984C
C2EB	196	24	01-34-75	426B	9DF9
C2EC	85	13	01-34-76	426C	6EDD
C2ED	85	19	01-34-77	426D	7027
C2EE	24	8	01-34-78	426E	5353
C2EF	30	10	01-34-79	426F	5544
C2F0	40	6	01-34-80	4270	5B85
C2F1	64	6	01-34-81	4271	6258
C2F2	64	7	01-34-82	4272	629E
C2F3	64	8	01-34-83	4273	62D3
C2F4	85	7	01-34-84	4274	6CA2
C2F5	85	17	01-34-85	4275	6FEF
C2F6	96	11	01-34-86	4276	7422
C2F7	149	10	01-34-87	4277	8A17
C2F8	167	21	01-34-88	4278	9438
C2F9	85	16	01-34-89	4279	6FC1
C2FA	149	15	01-34-90	427A	8AFE
C2FB	140	9	01-34-91	427B	8338
C2FC	16	5	01-34-92	427C	51E7
C2FD	142	13	01-34-93	427D	86F8
C2FE	30	5	01-34-94	427E	53EA
C3A1	26	5	01-35-01	4321	53E9
C3A2	9	7	01-35-02	4322	4F46
C3A3	162	12	01-35-03	4323	9054
C3A4	161	7	01-35-04	4324	8FB0
C3A5	41	14	01-35-05	4325	596A
C3A6	130	11	01-35-06	4326	8131
C3A7	12	12	01-35-07	4327	5DFD
C3A8	117	14	01-35-08	4328	7AEA
C3A9	162	6	01-35-09	4329	8FBF
C3AA	75	12	01-35-10	432A	68DA
C3AB	150	7	01-35-11	432B	8C37
C3AC	94	10	01-35-12	432C	72F8
C3AD	195	22	01-35-13	432D	9C48
C3AE	75	16	01-35-14	432E	6A3D
C3AF	149	15	01-35-15	432F	8AB0
C3B0	3	4	01-35-16	4330	4E39
C3B1	30	9	01-35-17	4331	5358
C3B2	30	13	01-35-18	4332	5606
C3B3	32	8	01-35-19	4333	5766
C3B4	64	8	01-35-20	4334	62C5
C3B5	64	11	01-35-21	4335	63A2
C3B6	72	5	01-35-22	4336	65E6
C3B7	76	15	01-35-23	4337	6B4E
C3B8	85	11	01-35-24	4338	6DE1
C3B9	85	12	01-35-25	4339	6E5B
C3BA	86	9	01-35-26	433A	70AD
C3BB	111	12	01-35-27	433B	77ED
C3BC	117	14	01-35-28	433C	7AEF
C3BD	118	15	01-35-29	433D	7BAA
C3BE	120	14	01-35-30	433E	7DBB
C3BF	128	10	01-35-31	433F	803D
C3C0	130	9	01-35-32	4340	80C6
C3C1	142	11	01-35-33	4341	86CB
C3C2	149	15	01-35-34	4342	8A95
C3C3	167	17	01-35-35	4343	935B
C3C4	31	6	01-35-36	4344	56E3
C3C5	32	16	01-35-37	4345	58C7
C3C6	57	12	01-35-38	4346	5F3E
C3C7	69	11	01-35-39	4347	65AD
C3C8	72	13	01-35-40	4348	6696
C3C9	75	17	01-35-41	4349	6A80
C3CA	79	9	01-35-42	434A	6BB5
C3CB	102	7	01-35-43	434B	7537
C3CC	149	15	01-35-44	434C	8AC7
C3CD	9	10	01-35-45	434D	5024
C3CE	30	8	01-35-46	434E	77E5
C3CF	32	6	01-35-47	434F	5730
C3D0	57	6	01-35-48	4350	5F1B
C3D1	61	10	01-35-49	4351	6065
C3D2	72	12	01-35-50	4352	667A
C3D3	85	6	01-35-51	4353	6C60
C3D4	104	13	01-35-52	4354	75F4
C3D5	115	13	01-35-53	4355	7A1A
C3D6	122	13	01-35-54	4356	7F6E
C3D7	133	10	01-35-55	4357	81F4
C3D8	142	14	01-35-56	4358	8718
C3D9	162	12	01-35-57	4359	9045
C3DA	187	13	01-35-58	435A	99B3
C3DB	75	16	01-35-59	435B	7BC9
C3DC	102	10	01-35-60	435C	755C
C3DD	118	6	01-35-61	435D	7AF9
C3DE	118	12	01-35-62	435E	7B51
C3DF	140	13	01-35-63	435F	84C4
C3E0	162	10	01-35-64	4360	9010
C3E1	115	10	01-35-65	4361	79E9
C3E2	116	11	01-35-66	4362	7A92
C3E3	140	9	01-35-67	4363	8336
C3E4	38	14	01-35-68	4364	5AE1
C3E5	109	12	01-35-69	4365	7740
C3E6	2	4	01-35-70	4366	4E2D
C3E7	9	6	01-35-71	4367	4EF2
C3E8	40	8	01-35-72	4368	5B99
C3E9	61	8	01-35-73	4369	5FE0
C3EA	64	8	01-35-74	436A	62BD
C3EB	72	9	01-35-75	436B	663C
C3EC	75	9	01-35-76	436C	67F1
C3ED	85	8	01-35-77	436D	6CE8
C3EE	142	6	01-35-78	436E	866B
C3EF	145	9	01-35-79	436F	8877
C3F0	149	12	01-35-80	4370	8A3B
C3F1	164	10	01-35-81	4371	914E
C3F2	167	15	01-35-82	4372	92F3
C3F3	187	15	01-35-83	4373	99D0
C3F4	75	15	01-35-84	4374	6A17
C3F5	85	18	01-35-85	4375	7026
C3F6	94	11	01-35-86	4376	732A
C3F7	140	8	01-35-87	4377	82E7
C3F8	140	11	01-35-88	4378	8457
C3F9	154	12	01-35-89	4379	8CAF
C3FA	1	2	01-35-90	437A	4E01
C3FB	10	6	01-35-91	437B	5146
C3FC	15	10	01-35-92	437C	51CB
C3FD	30	12	01-35-93	437D	558B
C3FE	40	19	01-35-94	437E	5BF5
C4A1	50	8	01-36-01	4421	5E16
C4A2	50	11	01-36-02	4422	5E33
C4A3	53	5	01-36-03	4423	5E81
C4A4	57	4	01-36-04	4424	5F14
C4A5	57	11	01-36-05	4425	5F35
C4A6	59	11	01-36-06	4426	5F6B
C4A7	60	14	01-36-07	4427	5FB4
C4A8	61	18	01-36-08	4428	61F2
C4A9	64	9	01-36-09	4429	6311
C4AA	72	14	01-36-10	442A	66A2
C4AB	74	12	01-36-11	442B	671D
C4AC	85	15	01-36-12	442C	6F6E
C4AD	91	13	01-36-13	442D	7252
C4AE	102	7	01-36-14	442E	753A
C4AF	109	11	01-36-15	442F	773A
C4B0	128	17	01-36-16	4430	8074
C4B1	130	12	01-36-17	4431	8139
C4B2	130	13	01-36-18	4432	8178
C4B3	142	15	01-36-19	4433	8776
C4B4	149	15	01-36-20	4434	8ABF
C4B5	149	16	01-36-21	4435	8ADC
C4B6	156	12	01-36-22	4436	8D85
C4B7	157	13	01-36-23	4437	8DF3
C4B8	167	14	01-36-24	4438	929A
C4B9	168	8	01-36-25	4439	9577
C4BA	181	11	01-36-26	443A	9802
C4BB	196	11	01-36-27	443B	9CE5
C4BC	19	9	01-36-28	443C	52C5
C4BD	64	11	01-36-29	443D	6357
C4BE	109	8	01-36-30	443E	76F4
C4BF	74	10	01-36-31	443F	6715
C4C0	85	7	01-36-32	4440	6C88
C4C1	96	9	01-36-33	4441	73CD
C4C2	154	13	01-36-34	4442	8CC3
C4C3	167	18	01-36-35	4443	93AE
C4C4	170	11	01-36-36	4444	9673
C4C5	85	9	01-36-37	4445	6D25
C4C6	32	15	01-36-38	4446	589C
C4C7	75	12	01-36-39	4447	690E
C4C8	75	13	01-36-40	4448	69CC
C4C9	162	9	01-36-41	4449	8FFD
C4CA	167	17	01-36-42	444A	939A
C4CB	104	12	01-36-43	444B	75DB
C4CC	162	10	01-36-44	444C	901A
C4CD	32	12	01-36-45	444D	585A
C4CE	75	9	01-36-46	444E	6802
C4CF	64	11	01-36-47	444F	63B4
C4D0	75	15	01-36-48	4450	69FB
C4D1	9	7	01-36-49	4451	4F43
C4D2	85	14	01-36-50	4452	6F2C
C4D3	75	9	01-36-51	4453	67D8
C4D4	162	5	01-36-52	4454	8FBB
C4D5	140	14	01-36-53	4455	8526
C4D6	120	14	01-36-54	4456	7DB4
C4D7	167	17	01-36-55	4457	9354
C4D8	75	13	01-36-56	4458	693F
C4D9	85	15	01-36-57	4459	6F70
C4DA	32	8	01-36-58	445A	576A
C4DB	33	11	01-36-59	445B	58F7
C4DC	38	17	01-36-60	445C	5B2C
C4DD	120	11	01-36-61	445D	7D2C
C4DE	87	4	01-36-62	445E	722A
C4DF	30	6	01-36-63	445F	540A
C4E0	167	11	01-36-64	4460	91E3
C4E1	196	21	01-36-65	4461	9DB4
C4E2	8	9	01-36-66	4462	4EAD
C4E3	9	7	01-36-67	4463	4F4E
C4E4	9	11	01-36-68	4464	505C
C4E5	9	11	01-36-69	4465	5075
C4E6	18	9	01-36-70	4466	5243
C4E7	25	9	01-36-71	4467	8C9E
C4E8	30	7	01-36-72	4468	5448
C4E9	32	12	01-36-73	4469	5824
C4EA	40	8	01-36-74	446A	5B9A
C4EB	50	9	01-36-75	446B	5E1D
C4EC	53	8	01-36-76	446C	5E95
C4ED	53	10	01-36-77	446D	5EAD
C4EE	54	7	01-36-78	446E	5EF7
C4EF	57	7	01-36-79	446F	5F1F
C4F0	61	10	01-36-80	4470	608C
C4F1	64	8	01-36-81	4471	62B5
C4F2	64	10	01-36-82	4472	633A
C4F3	64	12	01-36-83	4473	63D0
C4F4	75	11	01-36-84	4474	68AF
C4F5	85	5	01-36-85	4475	6C40
C4F6	112	13	01-36-86	4476	7887
C4F7	113	13	01-36-87	4477	798E
C4F8	115	12	01-36-88	4478	7A0B
C4F9	120	15	01-36-89	4479	7DE0
C4FA	137	13	01-36-90	447A	8247
C4FB	149	9	01-36-91	447B	8A02
C4FC	149	16	01-36-92	447C	8AE6
C4FD	157	16	01-36-93	447D	8E44
C4FE	162	10	01-36-94	447E	9013
C5A1	163	8	01-37-01	4521	90B8
C5A2	163	15	01-37-02	4522	912D
C5A3	167	10	01-37-03	4523	91D8
C5A4	206	13	01-37-04	4524	9F0E
C5A5	85	8	01-37-05	4525	6CE5
C5A6	64	14	01-37-06	4526	6458
C5A7	64	17	01-37-07	4527	64E2
C5A8	66	15	01-37-08	4528	6575
C5A9	85	14	01-37-09	4529	6EF4
C5AA	106	8	01-37-10	452A	7684
C5AB	118	11	01-37-11	452B	7B1B
C5AC	162	14	01-37-12	452C	9069
C5AD	167	19	01-37-13	452D	93D1
C5AE	85	13	01-37-14	452E	6EBA
C5AF	30	10	01-37-15	452F	54F2
C5B0	60	15	01-37-16	4530	5FB9
C5B1	64	15	01-37-17	4531	64A4
C5B2	159	19	01-37-18	4532	8F4D
C5B3	162	8	01-37-19	4533	8FED
C5B4	167	13	01-37-20	4534	9244
C5B5	12	8	01-37-21	4535	5178
C5B6	32	13	01-37-22	4536	586B
C5B7	37	4	01-37-23	4537	5929
C5B8	44	10	01-37-24	4538	5C55
C5B9	53	8	01-37-25	4539	5E97
C5BA	85	11	01-37-26	453A	6DFB
C5BB	120	21	01-37-27	453B	7E8F
C5BC	135	11	01-37-28	453C	751C
C5BD	154	12	01-37-29	453D	8CBC
C5BE	159	11	01-37-30	453E	8EE2
C5BF	181	19	01-37-31	453F	985B
C5C0	86	9	01-37-32	4540	70B9
C5C1	9	6	01-37-33	4541	4F1D
C5C2	79	13	01-37-34	4542	6BBF
C5C3	85	16	01-37-35	4543	6FB1
C5C4	102	5	01-37-36	4544	7530
C5C5	173	13	01-37-37	4545	96FB
C5C6	10	8	01-37-38	4546	514E
C5C7	30	6	01-37-39	4547	5410
C5C8	32	11	01-37-40	4548	5835
C5C9	32	13	01-37-41	4549	5857
C5CA	38	8	01-37-42	454A	59AC
C5CB	44	11	01-37-43	454B	5C60
C5CC	60	10	01-37-44	454C	5F92
C5CD	68	4	01-37-45	454D	6597
C5CE	75	7	01-37-46	454E	675C
C5CF	85	12	01-37-47	454F	6E21
C5D0	105	12	01-37-48	4550	767B
C5D1	140	12	01-37-49	4551	83DF
C5D2	154	15	01-37-50	4552	8CED
C5D3	162	10	01-37-51	4553	9014
C5D4	163	11	01-37-52	4554	90FD
C5D5	167	17	01-37-53	4555	934D
C5D6	112	10	01-37-54	4556	7825
C5D7	112	10	01-37-55	4557	783A
C5D8	19	7	01-37-56	4558	52AA
C5D9	29	9	01-37-57	4559	5EA6
C5DA	32	3	01-37-58	455A	571F
C5DB	38	5	01-37-59	455B	5974
C5DC	61	9	01-37-60	455C	6012
C5DD	9	10	01-37-61	455D	5012
C5DE	10	10	01-37-62	455E	515A
C5DF	15	5	01-37-63	455F	51AC
C5E0	15	10	01-37-64	4560	51CD
C5E1	18	2	01-37-65	4561	5200
C5E2	30	10	01-37-66	4562	5510
C5E3	32	12	01-37-67	4563	5854
C5E4	32	13	01-37-68	4564	5858
C5E5	37	10	01-37-69	4565	5957
C5E6	40	8	01-37-70	4566	5B95
C5E7	46	10	01-37-71	4567	5CF6
C5E8	46	14	01-37-72	4568	5D8B
C5E9	61	11	01-37-73	4569	60BC
C5EA	64	7	01-37-74	456A	6295
C5EB	64	12	01-37-75	456B	642D
C5EC	75	8	01-37-76	456C	6771
C5ED	75	10	01-37-77	456D	6843
C5EE	75	11	01-37-78	456E	68BC
C5EF	75	12	01-37-79	456F	68DF
C5F0	76	11	01-37-80	4570	76D7
C5F1	85	11	01-37-81	4571	6DD8
C5F2	85	12	01-37-82	4572	6E6F
C5F3	85	11	01-37-83	4573	6D9B
C5F4	86	6	01-37-84	4574	706F
C5F5	86	16	01-37-85	4575	71C8
C5F6	102	6	01-37-86	4576	5F53
C5F7	104	12	01-37-87	4577	75D8
C5F8	113	11	01-37-88	4578	7977
C5F9	118	12	01-37-89	4579	7B49
C5FA	118	12	01-37-90	457A	7B54
C5FB	118	12	01-37-91	457B	7B52
C5FC	119	16	01-37-92	457C	7CD6
C5FD	120	12	01-37-93	457D	7D71
C5FE	133	8	01-37-94	457E	5230
C6A1	140	12	01-38-01	4621	8463
C6A2	85	15	01-38-02	4622	8569
C6A3	140	18	01-38-03	4623	85E4
C6A4	149	10	01-38-04	4624	8A0E
C6A5	149	17	01-38-05	4625	8B04
C6A6	151	7	01-38-06	4626	8C46
C6A7	157	15	01-38-07	4627	8E0F
C6A8	162	9	01-38-08	4628	9003
C6A9	162	10	01-38-09	4629	900F
C6AA	167	20	01-38-10	462A	9419
C6AB	170	11	01-38-11	462B	9676
C6AC	181	16	01-38-12	462C	982D
C6AD	187	20	01-38-13	462D	9A30
C6AE	191	18	01-38-14	462E	95D8
C6AF	9	13	01-38-15	462F	50CD
C6B0	19	11	01-38-16	4630	52D5
C6B1	30	6	01-38-17	4631	540C
C6B2	32	11	01-38-18	4632	5802
C6B3	41	15	01-38-19	4633	5C0E
C6B4	61	15	01-38-20	4634	61A7
C6B5	64	15	01-38-21	4635	649E
C6B6	85	9	01-38-22	4636	6D1E
C6B7	109	17	01-38-23	4637	77B3
C6B8	117	12	01-38-24	4638	7AE5
C6B9	130	10	01-38-25	4639	80F4
C6BA	140	11	01-38-26	463A	8404
C6BB	162	12	01-38-27	463B	9053
C6BC	167	14	01-38-28	463C	9285
C6BD	46	9	01-38-29	463D	5CE0
C6BE	196	15	01-38-30	463E	9D07
C6BF	22	10	01-38-31	463F	533F
C6C0	60	11	01-38-32	4640	5F97
C6C1	60	14	01-38-33	4641	5FB3
C6C2	85	10	01-38-34	4642	6D9C
C6C3	93	10	01-38-35	4643	7279
C6C4	109	13	01-38-36	4644	7763
C6C5	115	7	01-38-37	4645	79BF
C6C6	187	16	01-38-38	4646	7BE4
C6C7	80	8	01-38-39	4647	6BD2
C6C8	94	9	01-38-40	4648	72EC
C6C9	149	14	01-38-41	4649	8AAD
C6CA	75	9	01-38-42	464A	6803
C6CB	75	16	01-38-43	464B	6A61
C6CC	17	5	01-38-44	464C	51F8
C6CD	116	8	01-38-45	464D	7A81
C6CE	75	13	01-38-46	464E	6934
C6CF	44	8	01-38-47	464F	5C4A
C6D0	196	14	01-38-48	4650	9CF6
C6D1	140	8	01-38-49	4651	82EB
C6D2	40	11	01-38-50	4652	5BC5
C6D3	164	7	01-38-51	4653	9149
C6D4	85	17	01-38-52	4654	701E
C6D5	30	16	01-38-53	4655	5678
C6D6	45	4	01-38-54	4656	5C6F
C6D7	61	11	01-38-55	4657	60C7
C6D8	66	12	01-38-56	4658	6566
C6D9	85	7	01-38-57	4659	6C8C
C6DA	152	11	01-38-58	465A	8C5A
C6DB	162	12	01-38-59	465B	9041
C6DC	181	13	01-38-60	465C	9813
C6DD	30	7	01-38-61	465D	5451
C6DE	72	16	01-38-62	465E	66C7
C6DF	167	12	01-38-63	465F	920D
C6E0	37	8	01-38-64	4660	5948
C6E1	163	7	01-38-65	4661	90A3
C6E2	11	4	01-38-66	4662	5185
C6E3	4	5	01-38-67	4663	4E4D
C6E4	16	6	01-38-68	4664	51EA
C6E5	140	16	01-38-69	4665	8599
C6E6	149	16	01-38-70	4666	8B0E
C6E7	85	21	01-38-71	4667	7058
C6E8	64	11	01-38-72	4668	637A
C6E9	167	17	01-38-73	4669	934B
C6EA	75	13	01-38-74	466A	6962
C6EB	187	13	01-38-75	466B	99B4
C6EC	120	15	01-38-76	466C	7E04
C6ED	102	13	01-38-77	466D	7577
C6EE	24	9	01-38-78	466E	5357
C6EF	75	13	01-38-79	466F	6960
C6F0	159	11	01-38-80	4670	8EDF
C6F1	172	18	01-38-81	4671	96E3
C6F2	85	6	01-38-82	4672	6C5D
C6F3	7	2	01-38-83	4673	4E8C
C6F4	44	5	01-38-84	4674	5C3C
C6F5	56	6	01-38-85	4675	5F10
C6F6	162	8	01-38-86	4676	8FE9
C6F7	20	4	01-38-87	4677	5302
C6F8	154	14	01-38-88	4678	8CD1
C6F9	130	6	01-38-89	4679	8089
C6FA	142	9	01-38-90	467A	8679
C6FB	24	4	01-38-91	467B	5EFF
C6FC	72	4	01-38-92	467C	65E5
C6FD	5	8	01-38-93	467D	4E73
C6FE	11	2	01-38-94	467E	5165
C7A1	30	6	01-39-01	4721	5982
C7A2	44	7	01-39-02	4722	5C3F
C7A3	179	12	01-39-03	4723	97EE
C7A4	9	6	01-39-04	4724	4EFB
C7A5	38	7	01-39-05	4725	598A
C7A6	61	7	01-39-06	4726	5FCD
C7A7	149	14	01-39-07	4727	8A8D
C7A8	85	17	01-39-08	4728	6FE1
C7A9	113	18	01-39-09	4729	79B0
C7AA	113	9	01-39-10	472A	7962
C7AB	40	14	01-39-11	472B	5BE7
C7AC	140	12	01-39-12	472C	8471
C7AD	94	11	01-39-13	472D	732B
C7AE	86	15	01-39-14	472E	71B1
C7AF	51	6	01-39-15	472F	5E74
C7B0	61	8	01-39-16	4730	5FF5
C7B1	64	11	01-39-17	4731	637B
C7B2	64	15	01-39-18	4732	649A
C7B3	86	16	01-39-19	4733	71C3
C7B4	119	11	01-39-20	4734	7C98
C7B5	4	2	01-39-21	4735	4E43
C7B6	54	9	01-39-22	4736	5EFC
C7B7	4	3	01-39-23	4737	4E4B
C7B8	32	11	01-39-24	4738	57DC
C7B9	30	18	01-39-25	4739	56A2
C7BA	61	10	01-39-26	473A	60A9
C7BB	85	16	01-39-27	473B	6FC3
C7BC	120	10	01-39-28	473C	7D0D
C7BD	130	10	01-39-29	473D	80FD
C7BE	130	11	01-39-30	473E	8133
C7BF	130	17	01-39-31	473F	81BF
C7C0	161	13	01-39-32	4740	8FB2
C7C1	147	12	01-39-33	4741	8997
C7C2	142	10	01-39-34	4742	86A4
C7C3	49	4	01-39-35	4743	5DF4
C7C4	64	7	01-39-36	4744	628A
C7C5	64	15	01-39-37	4745	64AD
C7C6	74	19	01-39-38	4746	8987
C7C7	75	8	01-39-39	4747	6777
C7C8	85	8	01-39-40	4748	6CE2
C7C9	85	9	01-39-41	4749	6D3E
C7CA	96	12	01-39-42	474A	7436
C7CB	112	10	01-39-43	474B	7834
C7CC	38	11	01-39-44	474C	5A46
C7CD	122	15	01-39-45	474D	7F75
C7CE	140	7	01-39-46	474E	82AD
C7CF	187	10	01-39-47	474F	99AC
C7D0	9	10	01-39-48	4750	4FF3
C7D1	53	12	01-39-49	4751	5EC3
C7D2	64	8	01-39-50	4752	62DD
C7D3	64	11	01-39-51	4753	6392
C7D4	66	11	01-39-52	4754	6557
C7D5	75	8	01-39-53	4755	676F
C7D6	75	9	01-39-54	4756	76C3
C7D7	91	13	01-39-55	4757	724C
C7D8	130	9	01-39-56	4758	80CC
C7D9	130	9	01-39-57	4759	80BA
C7DA	159	15	01-39-58	475A	8F29
C7DB	164	10	01-39-59	475B	914D
C7DC	9	10	01-39-60	475C	500D
C7DD	32	11	01-39-61	475D	57F9
C7DE	38	12	01-39-62	475E	5A92
C7DF	75	10	01-39-63	475F	6885
C7E0	75	13	01-39-64	4760	6973
C7E1	86	13	01-39-65	4761	7164
C7E2	94	10	01-39-66	4762	72FD
C7E3	154	12	01-39-67	4763	8CB7
C7E4	33	7	01-39-68	4764	58F2
C7E5	154	15	01-39-69	4765	8CE0
C7E6	170	11	01-39-70	4766	966A
C7E7	162	10	01-39-71	4767	9019
C7E8	142	15	01-39-72	4768	877F
C7E9	115	10	01-39-73	4769	79E4
C7EA	111	9	01-39-74	476A	77E7
C7EB	140	12	01-39-75	476B	8429
C7EC	9	7	01-39-76	476C	4F2F
C7ED	18	10	01-39-77	476D	5265
C7EE	24	12	01-39-78	476E	535A
C7EF	64	8	01-39-79	476F	62CD
C7F0	75	9	01-39-80	4770	67CF
C7F1	85	8	01-39-81	4771	6CCA
C7F2	106	5	01-39-82	4772	767D
C7F3	118	14	01-39-83	4773	7B94
C7F4	119	11	01-39-84	4774	7C95
C7F5	137	11	01-39-85	4775	8236
C7F6	140	16	01-39-86	4776	8584
C7F7	162	8	01-39-87	4777	8FEB
C7F8	72	19	01-39-88	4778	66DD
C7F9	85	13	01-39-89	4779	6F20
C7FA	86	19	01-39-90	477A	7206
C7FB	120	16	01-39-91	477B	7E1B
C7FC	140	10	01-39-92	477C	83AB
C7FD	187	14	01-39-93	477D	99C1
C7FE	199	7	01-39-94	477E	9EA6
C8A1	17	8	01-40-01	4821	51FD
C8A2	118	15	01-40-02	4822	7BB1
C8A3	112	12	01-40-03	4823	7872
C8A4	118	14	01-40-04	4824	7BB8
C8A5	129	14	01-40-05	4825	8087
C8A6	118	12	01-40-06	4826	7B48
C8A7	75	20	01-40-07	4827	6AE8
C8A8	50	15	01-40-08	4828	5E61
C8A9	130	6	01-40-09	4829	808C
C8AA	102	9	01-40-10	482A	7551
C8AB	102	10	01-40-11	482B	7560
C8AC	12	2	01-40-12	482C	516B
C8AD	167	13	01-40-13	482D	9262
C8AE	85	12	01-40-14	482E	6E8C
C8AF	105	9	01-40-15	482F	767A
C8B0	164	16	01-40-16	4830	9197
C8B1	190	14	01-40-17	4831	9AEA
C8B2	9	6	01-40-18	4832	4F10
C8B3	18	14	01-40-19	4833	7F70
C8B4	64	7	01-40-20	4834	629C
C8B5	118	12	01-40-21	4835	7B4F
C8B6	169	14	01-40-22	4836	95A5
C8B7	196	13	01-40-23	4837	9CE9
C8B8	30	16	01-40-24	4838	567A
C8B9	32	13	01-40-25	4839	5859
C8BA	142	12	01-40-26	483A	86E4
C8BB	172	10	01-40-27	483B	96BC
C8BC	9	7	01-40-28	483C	4F34
C8BD	18	7	01-40-29	483D	5224
C8BE	24	5	01-40-30	483E	534A
C8BF	29	4	01-40-31	483F	53CD
C8C0	29	9	01-40-32	4840	53DB
C8C1	50	6	01-40-33	4841	5E06
C8C2	64	13	01-40-34	4842	642C
C8C3	67	12	01-40-35	4843	6591
C8C4	75	8	01-40-36	4844	677F
C8C5	85	5	01-40-37	4845	6C3E
C8C6	85	6	01-40-38	4846	6C4E
C8C7	91	8	01-40-39	4847	7248
C8C8	94	5	01-40-40	4848	72AF
C8C9	96	10	01-40-41	4849	73ED
C8CA	102	10	01-40-42	484A	7554
C8CB	120	16	01-40-43	484B	7E41
C8CC	137	10	01-40-44	484C	822C
C8CD	140	18	01-40-45	484D	85E9
C8CE	154	11	01-40-46	484E	8CA9
C8CF	159	15	01-40-47	484F	7BC4
C8D0	165	7	01-40-48	4850	91C6
C8D1	181	13	01-40-49	4851	7169
C8D2	181	13	01-40-50	4852	9812
C8D3	184	12	01-40-51	4853	98EF
C8D4	64	11	01-40-52	4854	633D
C8D5	72	12	01-40-53	4855	6669
C8D6	102	12	01-40-54	4856	756A
C8D7	108	15	01-40-55	4857	76E4
C8D8	112	15	01-40-56	4858	78D0
C8D9	140	15	01-40-57	4859	8543
C8DA	142	12	01-40-58	485A	86EE
C8DB	22	10	01-40-59	485B	532A
C8DC	24	9	01-40-60	485C	5351
C8DD	30	7	01-40-61	485D	5426
C8DE	38	6	01-40-62	485E	5983
C8DF	53	7	01-40-63	485F	5E87
C8E0	60	8	01-40-64	4860	5F7C
C8E1	61	12	01-40-65	4861	60B2
C8E2	63	12	01-40-66	4862	6249
C8E3	64	7	01-40-67	4863	6279
C8E4	64	8	01-40-68	4864	62AB
C8E5	67	12	01-40-69	4865	6590
C8E6	81	4	01-40-70	4866	6BD4
C8E7	85	8	01-40-71	4867	6CCC
C8E8	104	10	01-40-72	4868	75B2
C8E9	107	5	01-40-73	4869	76AE
C8EA	112	14	01-40-74	486A	7891
C8EB	115	10	01-40-75	486B	79D8
C8EC	120	14	01-40-76	486C	7DCB
C8ED	122	15	01-40-77	486D	7F77
C8EE	130	8	01-40-78	486E	80A5
C8EF	145	10	01-40-79	486F	88AB
C8F0	149	15	01-40-80	4870	8AB9
C8F1	154	12	01-40-81	4871	8CBB
C8F2	162	16	01-40-82	4872	907F
C8F3	175	8	01-40-83	4873	975E
C8F4	183	9	01-40-84	4874	98DB
C8F5	75	14	01-40-85	4875	6A0B
C8F6	118	19	01-40-86	4876	7C38
C8F7	9	12	01-40-87	4877	5099
C8F8	44	7	01-40-88	4878	5C3E
C8F9	60	13	01-40-89	4879	5FAE
C8FA	75	8	01-40-90	487A	6787
C8FB	81	9	01-40-91	487B	6BD8
C8FC	96	12	01-40-92	487C	7435
C8FD	109	9	01-40-93	487D	7709
C8FE	123	9	01-40-94	487E	7F8E
C9A1	209	14	01-41-01	4921	9F3B
C9A2	75	9	01-41-02	4922	67CA
C9A3	115	14	01-41-03	4923	7A17
C9A4	22	4	01-41-04	4924	5339
C9A5	103	5	01-41-05	4925	758B
C9A6	190	16	01-41-06	4926	9AED
C9A7	59	9	01-41-07	4927	5F66
C9A8	130	15	01-41-08	4928	819D
C9A9	140	11	01-41-09	4929	83F1
C9AA	130	7	01-41-10	492A	8098
C9AB	57	12	01-41-11	492B	5F3C
C9AC	61	5	01-41-12	492C	5FC5
C9AD	102	11	01-41-13	492D	7562
C9AE	118	12	01-41-14	492E	7B46
C9AF	162	12	01-41-15	492F	903C
C9B0	75	10	01-41-16	4930	6867
C9B1	38	10	01-41-17	4931	59EB
C9B2	38	12	01-41-18	4932	5A9B
C9B3	120	10	01-41-19	4933	7D10
C9B4	1	6	01-41-20	4934	767E
C9B5	149	18	01-41-21	4935	8B2C
C9B6	9	10	01-41-22	4936	4FF5
C9B7	59	11	01-41-23	4937	5F6A
C9B8	75	15	01-41-24	4938	6A19
C9B9	85	5	01-41-25	4939	6C37
C9BA	85	14	01-41-26	493A	6F02
C9BB	97	16	01-41-27	493B	74E2
C9BC	113	11	01-41-28	493C	7968
C9BD	145	8	01-41-29	493D	8868
C9BE	149	12	01-41-30	493E	8A55
C9BF	153	10	01-41-31	493F	8C79
C9C0	53	15	01-41-32	4940	5EDF
C9C1	64	11	01-41-33	4941	63CF
C9C2	104	10	01-41-34	4942	75C5
C9C3	115	9	01-41-35	4943	79D2
C9C4	140	8	01-41-36	4944	82D7
C9C5	167	16	01-41-37	4945	9328
C9C6	167	15	01-41-38	4946	92F2
C9C7	140	13	01-41-39	4947	849C
C9C8	142	12	01-41-40	4948	86ED
C9C9	195	21	01-41-41	4949	9C2D
C9CA	30	9	01-41-42	494A	54C1
C9CB	59	11	01-41-43	494B	5F6C
C9CC	67	12	01-41-44	494C	658C
C9CD	85	10	01-41-45	494D	6D5C
C9CE	85	20	01-41-46	494E	7015
C9CF	154	11	01-41-47	494F	8CA7
C9D0	154	15	01-41-48	4950	8CD3
C9D1	181	17	01-41-49	4951	983B
C9D2	66	10	01-41-50	4952	654F
C9D3	98	11	01-41-51	4953	74F6
C9D4	1	4	01-41-52	4954	4E0D
C9D5	9	5	01-41-53	4955	4ED8
C9D6	32	11	01-41-54	4956	57E0
C9D7	37	4	01-41-55	4957	592B
C9D8	38	11	01-41-56	4958	5A66
C9D9	40	12	01-41-57	4959	5BCC
C9DA	14	11	01-41-58	495A	51A8
C9DB	50	5	01-41-59	495B	5E03
C9DC	53	8	01-41-60	495C	5E9C
C9DD	61	8	01-41-61	495D	6016
C9DE	64	7	01-41-62	495E	6276
C9DF	66	15	01-41-63	495F	6577
C9E0	69	8	01-41-64	4960	65A7
C9E1	72	12	01-41-65	4961	666E
C9E2	85	10	01-41-66	4962	6D6E
C9E3	88	4	01-41-67	4963	7236
C9E4	118	11	01-41-68	4964	7B26
C9E5	130	14	01-41-69	4965	8150
C9E6	130	15	01-41-70	4966	819A
C9E7	140	7	01-41-71	4967	8299
C9E8	149	19	01-41-72	4968	8B5C
C9E9	154	9	01-41-73	4969	8CA0
C9EA	154	15	01-41-74	496A	8CE6
C9EB	156	9	01-41-75	496B	8D74
C9EC	170	8	01-41-76	496C	961C
C9ED	170	8	01-41-77	496D	9644
C9EE	9	8	01-41-78	496E	4FAE
C9EF	64	15	01-41-79	496F	64AB
C9F0	77	8	01-41-80	4970	6B66
C9F1	136	15	01-41-81	4971	821E
C9F2	140	12	01-41-82	4972	8461
C9F3	140	15	01-41-83	4973	856A
C9F4	163	11	01-41-84	4974	90E8
C9F5	41	9	01-41-85	4975	5C01
C9F6	75	13	01-41-86	4976	6953
C9F7	182	9	01-41-87	4977	98A8
C9F8	140	12	01-41-88	4978	847A
C9F9	140	16	01-41-89	4979	8557
C9FA	9	6	01-41-90	497A	4F0F
C9FB	18	11	01-41-91	497B	526F
C9FC	60	12	01-41-92	497C	5FA9
C9FD	50	12	01-41-93	497D	5E45
C9FE	74	8	01-41-94	497E	670D
CAA1	113	13	01-42-01	4A21	798F
CAA2	130	13	01-42-02	4A22	8179
CAA3	145	14	01-42-03	4A23	8907
CAA4	146	18	01-42-04	4A24	8986
CAA5	85	11	01-42-05	4A25	6DF5
CAA6	57	5	01-42-06	4A26	5F17
CAA7	64	5	01-42-07	4A27	6255
CAA8	85	8	01-42-08	4A28	6CB8
CAA9	9	4	01-42-09	4A29	4ECF
CAAA	93	8	01-42-10	4A2A	7269
CAAB	195	16	01-42-11	4A2B	9B92
CAAC	18	4	01-42-12	4A2C	5206
CAAD	30	7	01-42-13	4A2D	543B
CAAE	30	15	01-42-14	4A2E	5674
CAAF	32	15	01-42-15	4A2F	58B3
CAB0	61	15	01-42-16	4A30	61A4
CAB1	64	7	01-42-17	4A31	626E
CAB2	86	12	01-42-18	4A32	711A
CAB3	102	16	01-42-19	4A33	596E
CAB4	119	10	01-42-20	4A34	7C89
CAB5	119	17	01-42-21	4A35	7CDE
CAB6	120	10	01-42-22	4A36	7D1B
CAB7	173	12	01-42-23	4A37	96F0
CAB8	67	4	01-42-24	4A38	6587
CAB9	128	14	01-42-25	4A39	805E
CABA	1	5	01-42-26	4A3A	4E19
CABB	9	8	01-42-27	4A3B	4F75
CABC	12	7	01-42-28	4A3C	5175
CABD	32	12	01-42-29	4A3D	5840
CABE	50	15	01-42-30	4A3E	5E63
CABF	51	5	01-42-31	4A3F	5E73
CAC0	55	15	01-42-32	4A40	5F0A
CAC1	75	9	01-42-33	4A41	67C4
CAC2	117	8	01-42-34	4A42	4E26
CAC3	140	15	01-42-35	4A43	853D
CAC4	169	11	01-42-36	4A44	9589
CAC5	170	10	01-42-37	4A45	965B
CAC6	119	6	01-42-38	4A46	7C73
CAC7	181	9	01-42-39	4A47	9801
CAC8	9	15	01-42-40	4A48	50FB
CAC9	32	16	01-42-41	4A49	58C1
CACA	104	18	01-42-42	4A4A	7656
CACB	112	14	01-42-43	4A4B	78A7
CACC	18	7	01-42-44	4A4C	5225
CACD	109	17	01-42-45	4A4D	77A5
CACE	140	14	01-42-46	4A4E	8511
CACF	118	14	01-42-47	4A4F	7B86
CAD0	9	11	01-42-48	4A50	504F
CAD1	66	9	01-42-49	4A51	5909
CAD2	91	4	01-42-50	4A52	7247
CAD3	118	15	01-42-51	4A53	7BC7
CAD4	120	15	01-42-52	4A54	7DE8
CAD5	162	5	01-42-53	4A55	8FBA
CAD6	162	7	01-42-54	4A56	8FD4
CAD7	162	12	01-42-55	4A57	904D
CAD8	9	9	01-42-56	4A58	4FBF
CAD9	19	10	01-42-57	4A59	52C9
CADA	38	11	01-42-58	4A5A	5A29
CADB	55	5	01-42-59	4A5B	5F01
CADC	177	18	01-42-60	4A5C	97AD
CADD	9	9	01-42-61	4A5D	4FDD
CADE	30	15	01-42-62	4A5E	8217
CADF	167	15	01-42-63	4A5F	92EA
CAE0	31	10	01-42-64	4A60	5703
CAE1	64	10	01-42-65	4A61	6355
CAE2	77	8	01-42-66	4A62	6B69
CAE3	101	7	01-42-67	4A63	752B
CAE4	145	12	01-42-68	4A64	88DC
CAE5	159	14	01-42-69	4A65	8F14
CAE6	115	15	01-42-70	4A66	7A42
CAE7	19	12	01-42-71	4A67	52DF
CAE8	32	13	01-42-72	4A68	5893
CAE9	61	14	01-42-73	4A69	6155
CAEA	62	5	01-42-74	4A6A	620A
CAEB	72	14	01-42-75	4A6B	66AE
CAEC	80	5	01-42-76	4A6C	6BCD
CAED	118	19	01-42-77	4A6D	7C3F
CAEE	140	11	01-42-78	4A6E	83E9
CAEF	9	10	01-42-79	4A6F	5023
CAF0	9	10	01-42-80	4A70	4FF8
CAF1	20	5	01-42-81	4A71	5305
CAF2	30	7	01-42-82	4A72	5446
CAF3	32	12	01-42-83	4A73	5831
CAF4	37	8	01-42-84	4A74	5949
CAF5	40	8	01-42-85	4A75	5B9D
CAF6	46	10	01-42-86	4A76	5CF0
CAF7	46	10	01-42-87	4A77	5CEF
CAF8	46	11	01-42-88	4A78	5D29
CAF9	53	8	01-42-89	4A79	5E96
CAFA	64	8	01-42-90	4A7A	62B1
CAFB	64	11	01-42-91	4A7B	6367
CAFC	66	8	01-42-92	4A7C	653E
CAFD	70	4	01-42-93	4A7D	65B9
CAFE	74	8	01-42-94	4A7E	670B
CBA1	85	8	01-43-01	4B21	6CD5
CBA2	85	8	01-43-02	4B22	6CE1
CBA3	86	11	01-43-03	4B23	70F9
CBA4	112	10	01-43-04	4B24	7832
CBA5	120	16	01-43-05	4B25	7E2B
CBA6	130	9	01-43-06	4B26	80DE
CBA7	140	7	01-43-07	4B27	82B3
CBA8	140	11	01-43-08	4B28	840C
CBA9	140	13	01-43-09	4B29	84EC
CBAA	142	13	01-43-10	4B2A	8702
CBAB	145	15	01-43-11	4B2B	8912
CBAC	149	11	01-43-12	4B2C	8A2A
CBAD	151	13	01-43-13	4B2D	8C4A
CBAE	163	7	01-43-14	4B2E	90A6
CBAF	167	15	01-43-15	4B2F	92D2
CBB0	184	13	01-43-16	4B30	98FD
CBB1	196	14	01-43-17	4B31	9CF3
CBB2	196	19	01-43-18	4B32	9D6C
CBB3	4	4	01-43-19	4B33	4E4F
CBB4	8	3	01-43-20	4B34	4EA1
CBB5	9	12	01-43-21	4B35	508D
CBB6	18	10	01-43-22	4B36	5256
CBB7	32	7	01-43-23	4B37	574A
CBB8	38	7	01-43-24	4B38	59A8
CBB9	50	12	01-43-25	4B39	5E3D
CBBA	61	7	01-43-26	4B3A	5FD8
CBBB	61	6	01-43-27	4B3B	5FD9
CBBC	63	8	01-43-28	4B3C	623F
CBBD	72	15	01-43-29	4B3D	66B4
CBBE	74	11	01-43-30	4B3E	671B
CBBF	75	9	01-43-31	4B3F	67D0
CBC0	75	12	01-43-32	4B40	68D2
CBC1	109	9	01-43-33	4B41	5192
CBC2	120	10	01-43-34	4B42	7D21
CBC3	130	8	01-43-35	4B43	80AA
CBC4	130	16	01-43-36	4B44	81A8
CBC5	149	16	01-43-37	4B45	8B00
CBC6	153	14	01-43-38	4B46	8C8C
CBC7	154	12	01-43-39	4B47	8CBF
CBC8	167	14	01-43-40	4B48	927E
CBC9	170	7	01-43-41	4B49	9632
CBCA	30	7	01-43-42	4B4A	5420
CBCB	181	15	01-43-43	4B4B	982C
CBCC	21	5	01-43-44	4B4C	5317
CBCD	9	14	01-43-45	4B4D	50D5
CBCE	25	2	01-43-46	4B4E	535C
CBCF	32	14	01-43-47	4B4F	58A8
CBD0	64	15	01-43-48	4B50	64B2
CBD1	75	6	01-43-49	4B51	6734
CBD2	93	8	01-43-50	4B52	7267
CBD3	109	13	01-43-51	4B53	7766
CBD4	115	16	01-43-52	4B54	7A46
CBD5	167	11	01-43-53	4B55	91E6
CBD6	19	9	01-43-54	4B56	52C3
CBD7	85	7	01-43-55	4B57	6CA1
CBD8	78	9	01-43-56	4B58	6B86
CBD9	32	11	01-43-57	4B59	5800
CBDA	50	13	01-43-58	4B5A	5E4C
CBDB	37	8	01-43-59	4B5B	5954
CBDC	75	5	01-43-60	4B5C	672C
CBDD	124	18	01-43-61	4B5D	7FFB
CBDE	16	3	01-43-62	4B5E	51E1
CBDF	108	9	01-43-63	4B5F	76C6
CBE0	64	15	01-43-64	4B60	6469
CBE1	112	16	01-43-65	4B61	78E8
CBE2	194	21	01-43-66	4B62	9B54
CBE3	200	11	01-43-67	4B63	9EBB
CBE4	32	10	01-43-68	4B64	57CB
CBE5	38	8	01-43-69	4B65	59B9
CBE6	72	9	01-43-70	4B66	6627
CBE7	75	8	01-43-71	4B67	679A
CBE8	80	6	01-43-72	4B68	6BCE
CBE9	30	10	01-43-73	4B69	54E9
CBEA	75	14	01-43-74	4B6A	69D9
CBEB	50	13	01-43-75	4B6B	5E55
CBEC	130	14	01-43-76	4B6C	819C
CBED	75	8	01-43-77	4B6D	6795
CBEE	195	17	01-43-78	4B6E	9BAA
CBEF	75	9	01-43-79	4B6F	67FE
CBF0	195	23	01-43-80	4B70	9C52
CBF1	75	10	01-43-81	4B71	685D
CBF2	8	6	01-43-82	4B72	4EA6
CBF3	9	9	01-43-83	4B73	4FE3
CBF4	29	2	01-43-84	4B74	53C8
CBF5	64	8	01-43-85	4B75	62B9
CBF6	75	5	01-43-86	4B76	672B
CBF7	85	8	01-43-87	4B77	6CAB
CBF8	162	6	01-43-88	4B78	8FC4
CBF9	9	8	01-43-89	4B79	4FAD
CBFA	120	18	01-43-90	4B7A	7E6D
CBFB	200	18	01-43-91	4B7B	9EBF
CBFC	1	3	01-43-92	4B7C	4E07
CBFD	61	14	01-43-93	4B7D	6162
CBFE	85	12	01-43-94	4B7E	6E80
CCA1	85	14	01-44-01	4C21	6F2B
CCA2	140	14	01-44-02	4C22	8513
CCA3	30	8	01-44-03	4C23	5473
CCA4	75	5	01-44-04	4C24	672A
CCA5	194	15	01-44-05	4C25	9B45
CCA6	49	3	01-44-06	4C26	5DF3
CCA7	118	14	01-44-07	4C27	7B95
CCA8	46	8	01-44-08	4C28	5CAC
CCA9	46	11	01-44-09	4C29	5BC6
CCAA	142	14	01-44-10	4C2A	871C
CCAB	85	12	01-44-11	4C2B	6E4A
CCAC	140	13	01-44-12	4C2C	84D1
CCAD	115	13	01-44-13	4C2D	7A14
CCAE	130	10	01-44-14	4C2E	8108
CCAF	38	7	01-44-15	4C2F	5999
CCB0	119	10	01-44-16	4C30	7C8D
CCB1	83	5	01-44-17	4C31	6C11
CCB2	109	10	01-44-18	4C32	7720
CCB3	19	11	01-44-19	4C33	52D9
CCB4	36	13	01-44-20	4C34	5922
CCB5	86	12	01-44-21	4C35	7121
CCB6	93	6	01-44-22	4C36	725F
CCB7	110	5	01-44-23	4C37	77DB
CCB8	173	19	01-44-24	4C38	9727
CCB9	196	19	01-44-25	4C39	9D61
CCBA	75	12	01-44-26	4C3A	690B
CCBB	38	12	01-44-27	4C3B	5A7F
CCBC	38	10	01-44-28	4C3C	5A18
CCBD	14	10	01-44-29	4C3D	51A5
CCBE	30	6	01-44-30	4C3E	540D
CCBF	30	8	01-44-31	4C3F	547D
CCC0	72	8	01-44-32	4C40	660E
CCC1	108	13	01-44-33	4C41	76DF
CCC2	162	9	01-44-34	4C42	8FF7
CCC3	167	14	01-44-35	4C43	9298
CCC4	196	14	01-44-36	4C44	9CF4
CCC5	38	9	01-44-37	4C45	59EA
CCC6	93	6	01-44-38	4C46	725D
CCC7	85	13	01-44-39	4C47	6EC5
CCC8	10	8	01-44-40	4C48	514D
CCC9	75	12	01-44-41	4C49	68C9
CCCA	120	14	01-44-42	4C4A	7DBF
CCCB	120	15	01-44-43	4C4B	7DEC
CCCC	176	9	01-44-44	4C4C	9762
CCCD	199	16	01-44-45	4C4D	9EBA
CCCE	64	13	01-44-46	4C4E	6478
CCCF	75	14	01-44-47	4C4F	6A21
CCD0	140	8	01-44-48	4C50	8302
CCD1	38	6	01-44-49	4C51	5984
CCD2	39	8	01-44-50	4C52	5B5F
CCD3	82	4	01-44-51	4C53	6BDB
CCD4	94	11	01-44-52	4C54	731B
CCD5	109	8	01-44-53	4C55	76F2
CCD6	120	14	01-44-54	4C56	7DB2
CCD7	127	10	01-44-55	4C57	8017
CCD8	140	13	01-44-56	4C58	8499
CCD9	9	17	01-44-57	4C59	5132
CCDA	75	4	01-44-58	4C5A	6728
CCDB	94	15	01-44-59	4C5B	9ED9
CCDC	109	5	01-44-60	4C5C	76EE
CCDD	75	7	01-44-61	4C5D	6762
CCDE	20	4	01-44-62	4C5E	52FF
CCDF	184	14	01-44-63	4C5F	9905
CCE0	43	4	01-44-64	4C60	5C24
CCE1	63	7	01-44-65	4C61	623B
CCE2	119	9	01-44-66	4C62	7C7E
CCE3	154	12	01-44-67	4C63	8CB0
CCE4	30	11	01-44-68	4C64	554F
CCE5	61	12	01-44-69	4C65	60B6
CCE6	120	10	01-44-70	4C66	7D0B
CCE7	169	8	01-44-71	4C67	9580
CCE8	20	4	01-44-72	4C68	5301
CCE9	5	3	01-44-73	4C69	4E5F
CCEA	15	7	01-44-74	4C6A	51B6
CCEB	36	8	01-44-75	4C6B	591C
CCEC	88	13	01-44-76	4C6C	723A
CCED	163	9	01-44-77	4C6D	8036
CCEE	166	11	01-44-78	4C6E	91CE
CCEF	57	8	01-44-79	4C6F	5F25
CCF0	111	5	01-44-80	4C70	77E2
CCF1	27	4	01-44-81	4C71	5384
CCF2	60	7	01-44-82	4C72	5F79
CCF3	120	9	01-44-83	4C73	7D04
CCF4	140	16	01-44-84	4C74	85AC
CCF5	149	11	01-44-85	4C75	8A33
CCF6	157	21	01-44-86	4C76	8E8D
CCF7	117	13	01-44-87	4C77	9756
CCF8	75	9	01-44-88	4C78	67F3
CCF9	140	16	01-44-89	4C79	85AE
CCFA	167	21	01-44-90	4C7A	9453
CCFB	61	12	01-44-91	4C7B	6109
CCFC	61	13	01-44-92	4C7C	6108
CCFD	85	8	01-44-93	4C7D	6CB9
CCFE	104	18	01-44-94	4C7E	7652
CDA1	149	16	01-45-01	4D21	8AED
CDA2	159	16	01-45-02	4D22	8F38
CDA3	30	11	01-45-03	4D23	552F
CDA4	9	7	01-45-04	4D24	4F51
CDA5	9	17	01-45-05	4D25	512A
CDA6	19	9	01-45-06	4D26	52C7
CDA7	29	4	01-45-07	4D27	53CB
CDA8	40	9	01-45-08	4D28	5BA5
CDA9	52	9	01-45-09	4D29	5E7D
CDAA	61	11	01-45-10	4D2A	60A0
CDAB	61	15	01-45-11	4D2B	6182
CDAC	64	12	01-45-12	4D2C	63D6
CDAD	74	6	01-45-13	4D2D	6709
CDAE	75	9	01-45-14	4D2E	67DA
CDAF	85	12	01-45-15	4D2F	6E67
CDB0	85	10	01-45-16	4D30	6D8C
CDB1	94	12	01-45-17	4D31	7336
CDB2	94	13	01-45-18	4D32	7337
CDB3	102	5	01-45-19	4D33	7531
CDB4	113	9	01-45-20	4D34	7950
CDB5	145	12	01-45-21	4D35	88D5
CDB6	149	14	01-45-22	4D36	8A98
CDB7	162	12	01-45-23	4D37	904A
CDB8	163	7	01-45-24	4D38	9091
CDB9	163	11	01-45-25	4D39	90F5
CDBA	172	12	01-45-26	4D3A	96C4
CDBB	193	16	01-45-27	4D3B	878D
CDBC	36	3	01-45-28	4D3C	5915
CDBD	6	4	01-45-29	4D3D	4E88
CDBE	9	7	01-45-30	4D3E	4F59
CDBF	1	3	01-45-31	4D3F	4E0E
CDC0	149	13	01-45-32	4D40	8A89
CDC1	159	17	01-45-33	4D41	8F3F
CDC2	181	13	01-45-34	4D42	9810
CDC3	9	13	01-45-35	4D43	50AD
CDC4	19	5	01-45-36	4D44	5E7C
CDC5	38	7	01-45-37	4D45	5996
CDC6	40	10	01-45-38	4D46	5BB9
CDC7	53	11	01-45-39	4D47	5EB8
CDC8	64	12	01-45-40	4D48	63DA
CDC9	64	12	01-45-41	4D49	63FA
CDCA	64	16	01-45-42	4D4A	64C1
CDCB	72	18	01-45-43	4D4B	66DC
CDCC	75	13	01-45-44	4D4C	694A
CDCD	75	14	01-45-45	4D4D	69D8
CDCE	85	9	01-45-46	4D4E	6D0B
CDCF	85	13	01-45-47	4D4F	6EB6
CDD0	167	14	01-45-48	4D50	7194
CDD1	101	5	01-45-49	4D51	7528
CDD2	116	15	01-45-50	4D52	7AAF
CDD3	123	6	01-45-51	4D53	7F8A
CDD4	124	20	01-45-52	4D54	8000
CDD5	140	12	01-45-53	4D55	8449
CDD6	140	13	01-45-54	4D56	84C9
CDD7	146	9	01-45-55	4D57	8981
CDD8	149	16	01-45-56	4D58	8B21
CDD9	157	14	01-45-57	4D59	8E0A
CDDA	162	12	01-45-58	4D5A	9065
CDDB	170	12	01-45-59	4D5B	967D
CDDC	184	15	01-45-60	4D5C	990A
CDDD	61	15	01-45-61	4D5D	617E
CDDE	64	7	01-45-62	4D5E	6291
CDDF	76	11	01-45-63	4D5F	6B32
CDE0	85	7	01-45-64	4D60	6C83
CDE1	85	10	01-45-65	4D61	6D74
CDE2	124	11	01-45-66	4D62	7FCC
CDE3	124	17	01-45-67	4D63	7FFC
CDE4	85	11	01-45-68	4D64	6DC0
CDE5	122	19	01-45-69	4D65	7F85
CDE6	142	17	01-45-70	4D66	87BA
CDE7	145	13	01-45-71	4D67	88F8
CDE8	9	7	01-45-72	4D68	6765
CDE9	140	10	01-45-73	4D69	83B1
CDEA	154	16	01-45-74	4D6A	983C
CDEB	173	13	01-45-75	4D6B	96F7
CDEC	85	9	01-45-76	4D6C	6D1B
CDED	120	12	01-45-77	4D6D	7D61
CDEE	140	12	01-45-78	4D6E	843D
CDEF	164	13	01-45-79	4D6F	916A
CDF0	5	7	01-45-80	4D70	4E71
CDF1	26	7	01-45-81	4D71	5375
CDF2	46	12	01-45-82	4D72	5D50
CDF3	75	20	01-45-83	4D73	6B04
CDF4	85	18	01-45-84	4D74	6FEB
CDF5	140	18	01-45-85	4D75	85CD
CDF6	140	19	01-45-86	4D76	862D
CDF7	147	17	01-45-87	4D77	89A7
CDF8	18	7	01-45-88	4D78	5229
CDF9	30	6	01-45-89	4D79	540F
CDFA	44	15	01-45-90	4D7A	5C65
CDFB	75	7	01-45-91	4D7B	674E
CDFC	75	11	01-45-92	4D7C	68A8
CDFD	96	11	01-45-93	4D7D	7406
CDFE	96	15	01-45-94	4D7E	7483
CEA1	104	12	01-46-01	4E21	75E2
CEA2	145	13	01-46-02	4E22	88CF
CEA3	145	12	01-46-03	4E23	88E1
CEA4	166	7	01-46-04	4E24	91CC
CEA5	172	19	01-46-05	4E25	96E2
CEA6	170	11	01-46-06	4E26	9678
CEA7	60	9	01-46-07	4E27	5F8B
CEA8	95	11	01-46-08	4E28	7387
CEA9	117	5	01-46-09	4E29	7ACB
CEAA	140	12	01-46-10	4E2A	844E
CEAB	64	11	01-46-11	4E2B	63A0
CEAC	102	11	01-46-12	4E2C	7565
CEAD	18	15	01-46-13	4E2D	5289
CEAE	85	10	01-46-14	4E2E	6D41
CEAF	85	13	01-46-15	4E2F	6E9C
CEB0	96	11	01-46-16	4E30	7409
CEB1	102	10	01-46-17	4E31	7559
CEB2	112	12	01-46-18	4E32	786B
CEB3	119	11	01-46-19	4E33	7C92
CEB4	170	11	01-46-20	4E34	9686
CEB5	212	10	01-46-21	4E35	7ADC
CEB6	212	16	01-46-22	4E36	9F8D
CEB7	9	9	01-46-23	4E37	4FB6
CEB8	61	15	01-46-24	4E38	616E
CEB9	70	10	01-46-25	4E39	65C5
CEBA	141	13	01-46-26	4E3A	865C
CEBB	6	2	01-46-27	4E3B	4E86
CEBC	8	9	01-46-28	4E3C	4EAE
CEBD	9	14	01-46-29	4E3D	50DA
CEBE	11	6	01-46-30	4E3E	4E21
CEBF	15	10	01-46-31	4E3F	51CC
CEC0	40	15	01-46-32	4E40	5BEE
CEC1	68	10	01-46-33	4E41	6599
CEC2	75	11	01-46-34	4E42	6881
CEC3	85	11	01-46-35	4E43	6DBC
CEC4	94	11	01-46-36	4E44	731F
CEC5	104	17	01-46-37	4E45	7642
CEC6	109	17	01-46-38	4E46	77AD
CEC7	115	13	01-46-39	4E47	7A1C
CEC8	119	18	01-46-40	4E48	7CE7
CEC9	138	7	01-46-41	4E49	826F
CECA	149	15	01-46-42	4E4A	8AD2
CECB	162	15	01-46-43	4E4B	907C
CECC	166	12	01-46-44	4E4C	91CF
CECD	170	11	01-46-45	4E4D	9675
CECE	181	14	01-46-46	4E4E	9818
CECF	19	2	01-46-47	4E4F	529B
CED0	120	14	01-46-48	4E50	7DD1
CED1	9	10	01-46-49	4E51	502B
CED2	27	9	01-46-50	4E52	5398
CED3	75	8	01-46-51	4E53	6797
CED4	85	11	01-46-52	4E54	6DCB
CED5	86	16	01-46-53	4E55	71D0
CED6	96	12	01-46-54	4E56	7433
CED7	131	18	01-46-55	4E57	81E8
CED8	159	15	01-46-56	4E58	8F2A
CED9	170	16	01-46-57	4E59	96A3
CEDA	195	23	01-46-58	4E5A	9C57
CEDB	198	23	01-46-59	4E5B	9E9F
CEDC	96	14	01-46-60	4E5C	7460
CEDD	32	12	01-46-61	4E5D	5841
CEDE	85	10	01-46-62	4E5E	6D99
CEDF	120	11	01-46-63	4E5F	7D2F
CEE0	181	18	01-46-64	4E60	985E
CEE1	9	5	01-46-65	4E61	4EE4
CEE2	9	7	01-46-66	4E62	4F36
CEE3	9	8	01-46-67	4E63	4F8B
CEE4	15	7	01-46-68	4E64	51B7
CEE5	19	7	01-46-69	4E65	52B1
CEE6	46	17	01-46-70	4E66	5DBA
CEE7	61	8	01-46-71	4E67	601C
CEE8	96	9	01-46-72	4E68	73B2
CEE9	113	5	01-46-73	4E69	793C
CEEA	140	8	01-46-74	4E6A	82D3
CEEB	167	13	01-46-75	4E6B	9234
CEEC	171	16	01-46-76	4E6C	96B7
CEED	173	13	01-46-77	4E6D	96F6
CEEE	173	15	01-46-78	4E6E	970A
CEEF	198	19	01-46-79	4E6F	9E97
CEF0	211	17	01-46-80	4E70	9F62
CEF1	72	14	01-46-81	4E71	66A6
CEF2	77	14	01-46-82	4E72	6B74
CEF3	18	6	01-46-83	4E73	5217
CEF4	19	6	01-46-84	4E74	52A3
CEF5	86	10	01-46-85	4E75	70C8
CEF6	145	12	01-46-86	4E76	88C2
CEF7	53	13	01-46-87	4E77	5EC9
CEF8	61	10	01-46-88	4E78	604B
CEF9	61	15	01-46-89	4E79	6190
CEFA	85	13	01-46-90	4E7A	6F23
CEFB	86	12	01-46-91	4E7B	7149
CEFC	118	19	01-46-92	4E7C	7C3E
CEFD	120	14	01-46-93	4E7D	7DF4
CEFE	128	17	01-46-94	4E7E	806F
CFA1	140	13	01-47-01	4F21	84EE
CFA2	162	10	01-47-02	4F22	9023
CFA3	167	16	01-47-03	4F23	932C
CFA4	30	7	01-47-04	4F24	5442
CFA5	73	15	01-47-05	4F25	9B6F
CFA6	75	19	01-47-06	4F26	6AD3
CFA7	86	8	01-47-07	4F27	7089
CFA8	154	13	01-47-08	4F28	8CC2
CFA9	157	13	01-47-09	4F29	8DEF
CFAA	173	21	01-47-10	4F2A	9732
CFAB	19	7	01-47-11	4F2B	52B4
CFAC	38	11	01-47-12	4F2C	5A41
CFAD	53	12	01-47-13	4F2D	5ECA
CFAE	55	7	01-47-14	4F2E	5F04
CFAF	74	10	01-47-15	4F2F	6717
CFB0	75	13	01-47-16	4F30	697C
CFB1	75	13	01-47-17	4F31	6994
CFB2	85	10	01-47-18	4F32	6D6A
CFB3	85	14	01-47-19	4F33	6F0F
CFB4	93	7	01-47-20	4F34	7262
CFB5	94	10	01-47-21	4F35	72FC
CFB6	118	16	01-47-22	4F36	7BED
CFB7	125	6	01-47-23	4F37	8001
CFB8	128	22	01-47-24	4F38	807E
CFB9	142	14	01-47-25	4F39	874B
CFBA	163	9	01-47-26	4F3A	90CE
CFBB	12	4	01-47-27	4F3B	516D
CFBC	75	19	01-47-28	4F3C	9E93
CFBD	113	12	01-47-29	4F3D	7984
CFBE	130	6	01-47-30	4F3E	808B
CFBF	167	16	01-47-31	4F3F	9332
CFC0	149	15	01-47-32	4F40	8AD6
CFC1	9	10	01-47-33	4F41	502D
CFC2	30	8	01-47-34	4F42	548C
CFC3	149	13	01-47-35	4F43	8A71
CFC4	77	9	01-47-36	4F44	6B6A
CFC5	154	13	01-47-37	4F45	8CC4
CFC6	130	10	01-47-38	4F46	8107
CFC7	61	12	01-47-39	4F47	60D1
CFC8	75	8	01-47-40	4F48	67A0
CFC9	196	23	01-47-41	4F49	9DF2
CFCA	7	6	01-47-42	4F4A	4E99
CFCB	7	6	01-47-43	4F4B	4E98
CFCC	195	20	01-47-44	4F4C	9C10
CFCD	149	13	01-47-45	4F4D	8A6B
CFCE	140	17	01-47-46	4F4E	85C1
CFCF	140	15	01-47-47	4F4F	8568
CFD0	75	12	01-47-48	4F50	6900
CFD1	85	12	01-47-49	4F51	6E7E
CFD2	108	13	01-47-50	4F52	7897
CFD3	130	12	01-47-51	4F53	8155
D0A1	1	4	01-48-01	5021	5F0C
D0A2	1	4	01-48-02	5022	4E10
D0A3	1	5	01-48-03	5023	4E15
D0A4	2	3	01-48-04	5024	4E2A
D0A5	2	5	01-48-05	5025	4E31
D0A6	3	1	01-48-06	5026	4E36
D0A7	3	5	01-48-07	5027	4E3C
D0A8	4	1	01-48-08	5028	4E3F
D0A9	4	2	01-48-09	5029	4E42
D0AA	4	8	01-48-10	502A	4E56
D0AB	4	10	01-48-11	502B	4E58
D0AC	5	13	01-48-12	502C	4E82
D0AD	6	1	01-48-13	502D	4E85
D0AE	152	16	01-48-14	502E	8C6B
D0AF	6	7	01-48-15	502F	4E8A
D0B0	6	12	01-48-16	5030	8212
D0B1	7	5	01-48-17	5031	5F0D
D0B2	7	3	01-48-18	5032	4E8E
D0B3	7	8	01-48-19	5033	4E9E
D0B4	7	8	01-48-20	5034	4E9F
D0B5	8	2	01-48-21	5035	4EA0
D0B6	8	4	01-48-22	5036	4EA2
D0B7	8	9	01-48-23	5037	4EB0
D0B8	8	10	01-48-24	5038	4EB3
D0B9	8	13	01-48-25	5039	4EB6
D0BA	9	4	01-48-26	503A	4ECE
D0BB	9	4	01-48-27	503B	4ECD
D0BC	9	4	01-48-28	503C	4EC4
D0BD	9	4	01-48-29	503D	4EC6
D0BE	9	4	01-48-30	503E	4EC2
D0BF	9	5	01-48-31	503F	4ED7
D0C0	9	5	01-48-32	5040	4EDE
D0C1	9	5	01-48-33	5041	4EED
D0C2	9	5	01-48-34	5042	4EDF
D0C3	9	6	01-48-35	5043	4EF7
D0C4	9	6	01-48-36	5044	4F09
D0C5	9	7	01-48-37	5045	4F5A
D0C6	9	7	01-48-38	5046	4F30
D0C7	9	7	01-48-39	5047	4F5B
D0C8	9	7	01-48-40	5048	4F5D
D0C9	9	7	01-48-41	5049	4F57
D0CA	9	7	01-48-42	504A	4F47
D0CB	9	8	01-48-43	504B	4F76
D0CC	9	8	01-48-44	504C	4F88
D0CD	9	8	01-48-45	504D	4F8F
D0CE	9	8	01-48-46	504E	4F98
D0CF	9	8	01-48-47	504F	4F7B
D0D0	9	8	01-48-48	5050	4F69
D0D1	9	8	01-48-49	5051	4F70
D0D2	9	8	01-48-50	5052	4F91
D0D3	9	8	01-48-51	5053	4F6F
D0D4	9	8	01-48-52	5054	4F86
D0D5	9	8	01-48-53	5055	4F96
D0D6	9	16	01-48-54	5056	5118
D0D7	9	9	01-48-55	5057	4FD4
D0D8	9	9	01-48-56	5058	4FDF
D0D9	9	9	01-48-57	5059	4FCE
D0DA	9	9	01-48-58	505A	4FD8
D0DB	9	9	01-48-59	505B	4FDB
D0DC	9	9	01-48-60	505C	4FD1
D0DD	9	9	01-48-61	505D	4FDA
D0DE	9	9	01-48-62	505E	4FD0
D0DF	9	9	01-48-63	505F	4FE4
D0E0	9	9	01-48-64	5060	4FE5
D0E1	9	10	01-48-65	5061	501A
D0E2	9	10	01-48-66	5062	5028
D0E3	9	10	01-48-67	5063	5014
D0E4	9	10	01-48-68	5064	502A
D0E5	9	10	01-48-69	5065	5025
D0E6	9	10	01-48-70	5066	5005
D0E7	9	6	01-48-71	5067	4F1C
D0E8	9	10	01-48-72	5068	4FF6
D0E9	9	10	01-48-73	5069	5021
D0EA	9	10	01-48-74	506A	5029
D0EB	9	10	01-48-75	506B	502C
D0EC	9	10	01-48-76	506C	4FFE
D0ED	9	10	01-48-77	506D	4FEF
D0EE	9	10	01-48-78	506E	5011
D0EF	9	10	01-48-79	506F	5006
D0F0	9	11	01-48-80	5070	5043
D0F1	9	11	01-48-81	5071	5047
D0F2	9	13	01-48-82	5072	6703
D0F3	9	11	01-48-83	5073	5055
D0F4	9	11	01-48-84	5074	5050
D0F5	9	11	01-48-85	5075	5048
D0F6	9	11	01-48-86	5076	505A
D0F7	9	11	01-48-87	5077	5056
D0F8	9	11	01-48-88	5078	506C
D0F9	9	11	01-48-89	5079	5078
D0FA	9	12	01-48-90	507A	5080
D0FB	9	12	01-48-91	507B	509A
D0FC	9	12	01-48-92	507C	5085
D0FD	9	13	01-48-93	507D	50B4
D0FE	9	13	01-48-94	507E	50B2
D1A1	9	13	01-49-01	5121	50C9
D1A2	9	14	01-49-02	5122	50CA
D1A3	9	13	01-49-03	5123	50B3
D1A4	9	13	01-49-04	5124	50C2
D1A5	9	14	01-49-05	5125	50D6
D1A6	9	14	01-49-06	5126	50DE
D1A7	9	14	01-49-07	5127	50E5
D1A8	9	14	01-49-08	5128	50ED
D1A9	9	14	01-49-09	5129	50E3
D1AA	9	14	01-49-10	512A	50EE
D1AB	9	15	01-49-11	512B	50F9
D1AC	9	15	01-49-12	512C	50F5
D1AD	9	15	01-49-13	512D	5109
D1AE	9	15	01-49-14	512E	5101
D1AF	9	15	01-49-15	512F	5102
D1B0	9	17	01-49-16	5130	5116
D1B1	9	16	01-49-17	5131	5115
D1B2	9	16	01-49-18	5132	5114
D1B3	9	15	01-49-19	5133	511A
D1B4	9	17	01-49-20	5134	5121
D1B5	9	21	01-49-21	5135	513A
D1B6	9	21	01-49-22	5136	5137
D1B7	9	22	01-49-23	5137	513C
D1B8	9	22	01-49-24	5138	513B
D1B9	10	2	01-49-25	5139	513F
D1BA	10	3	01-49-26	513A	5140
D1BB	10	8	01-49-27	513B	5152
D1BC	10	7	01-49-28	513C	514C
D1BD	10	8	01-49-29	513D	5154
D1BE	10	14	01-49-30	513E	5162
D1BF	10	22	01-49-31	513F	7AF8
D1C0	11	8	01-49-32	5140	5169
D1C1	11	9	01-49-33	5141	516A
D1C2	12	4	01-49-34	5142	516E
D1C3	12	16	01-49-35	5143	5180
D1C4	13	2	01-49-36	5144	5182
D1C5	13	5	01-49-37	5145	56D8
D1C6	13	5	01-49-38	5146	518C
D1C7	13	5	01-49-39	5147	5189
D1C8	13	7	01-49-40	5148	518F
D1C9	13	9	01-49-41	5149	5191
D1CA	13	10	01-49-42	514A	5193
D1CB	13	11	01-49-43	514B	5195
D1CC	14	2	01-49-44	514C	5196
D1CD	14	10	01-49-45	514D	51A4
D1CE	14	10	01-49-46	514E	51A6
D1CF	14	10	01-49-47	514F	51A2
D1D0	14	14	01-49-48	5150	51A9
D1D1	14	15	01-49-49	5151	51AA
D1D2	15	2	01-49-50	5152	51AB
D1D3	15	6	01-49-51	5153	51B3
D1D4	15	6	01-49-52	5154	51B1
D1D5	15	6	01-49-53	5155	51B2
D1D6	15	6	01-49-54	5156	51B0
D1D7	15	7	01-49-55	5157	51B5
D1D8	15	8	01-49-56	5158	51BD
D1D9	15	10	01-49-57	5159	51C5
D1DA	15	10	01-49-58	515A	51C9
D1DB	15	15	01-49-59	515B	51DB
D1DC	16	2	01-49-60	515C	51E0
D1DD	141	11	01-49-61	515D	8655
D1DE	16	6	01-49-62	515E	51E9
D1DF	16	8	01-49-63	515F	51ED
D1E0	16	11	01-49-64	5160	51F0
D1E1	17	2	01-49-65	5161	51F5
D1E2	17	9	01-49-66	5162	51FE
D1E3	18	3	01-49-67	5163	5204
D1E4	18	5	01-49-68	5164	520B
D1E5	18	6	01-49-69	5165	5214
D1E6	18	6	01-49-70	5166	520E
D1E7	18	7	01-49-71	5167	5227
D1E8	18	7	01-49-72	5168	522A
D1E9	18	8	01-49-73	5169	522E
D1EA	18	8	01-49-74	516A	5233
D1EB	18	8	01-49-75	516B	5239
D1EC	18	9	01-49-76	516C	524F
D1ED	18	9	01-49-77	516D	5244
D1EE	18	9	01-49-78	516E	524B
D1EF	18	9	01-49-79	516F	524C
D1F0	18	10	01-49-80	5170	525E
D1F1	18	10	01-49-81	5171	5254
D1F2	18	11	01-49-82	5172	526A
D1F3	18	12	01-49-83	5173	5274
D1F4	18	12	01-49-84	5174	5269
D1F5	18	11	01-49-85	5175	5273
D1F6	18	13	01-49-86	5176	527F
D1F7	18	13	01-49-87	5177	527D
D1F8	18	15	01-49-88	5178	528D
D1F9	18	16	01-49-89	5179	5294
D1FA	18	16	01-49-90	517A	5292
D1FB	18	11	01-49-91	517B	5271
D1FC	18	15	01-49-92	517C	5288
D1FD	18	16	01-49-93	517D	5291
D1FE	18	16	01-49-94	517E	8FA8
D2A1	160	16	01-50-01	5221	8FA7
D2A2	19	7	01-50-02	5222	52AC
D2A3	19	7	01-50-03	5223	52AD
D2A4	19	8	01-50-04	5224	52BC
D2A5	19	8	01-50-05	5225	52B5
D2A6	19	9	01-50-06	5226	52C1
D2A7	19	10	01-50-07	5227	52CD
D2A8	19	11	01-50-08	5228	52D7
D2A9	19	12	01-50-09	5229	52DE
D2AA	19	13	01-50-10	522A	52E3
D2AB	19	13	01-50-11	522B	52E6
D2AC	19	13	01-50-12	522C	98ED
D2AD	19	13	01-50-13	522D	52E0
D2AE	19	16	01-50-14	522E	52F3
D2AF	19	16	01-50-15	522F	52F5
D2B0	19	19	01-50-16	5230	52F8
D2B1	20	2	01-50-17	5231	52F9
D2B2	20	5	01-50-18	5232	5306
D2B3	20	6	01-50-19	5233	5308
D2B4	20	7	01-50-20	5234	7538
D2B5	20	9	01-50-21	5235	530D
D2B6	20	11	01-50-22	5236	5310
D2B7	20	11	01-50-23	5237	530F
D2B8	21	2	01-50-24	5238	5315
D2B9	22	2	01-50-25	5239	531A
D2BA	22	7	01-50-26	523A	5323
D2BB	22	13	01-50-27	523B	532F
D2BC	22	14	01-50-28	523C	5331
D2BD	22	15	01-50-29	523D	5333
D2BE	23	2	01-50-30	523E	5338
D2BF	22	11	01-50-31	523F	5340
D2C0	24	4	01-50-32	5240	5346
D2C1	24	4	01-50-33	5241	5345
D2C2	1	5	01-50-34	5242	4E17
D2C3	24	5	01-50-35	5243	5349
D2C4	24	6	01-50-36	5244	534D
D2C5	15	12	01-50-37	5245	51D6
D2C6	25	4	01-50-38	5246	535E
D2C7	26	2	01-50-39	5247	5369
D2C8	26	5	01-50-40	5248	536E
D2C9	26	5	01-50-41	5249	5918
D2CA	26	9	01-50-42	524A	537B
D2CB	26	8	01-50-43	524B	5377
D2CC	27	2	01-50-44	524C	5382
D2CD	27	9	01-50-45	524D	5396
D2CE	27	11	01-50-46	524E	53A0
D2CF	27	12	01-50-47	524F	53A6
D2D0	27	12	01-50-48	5250	53A5
D2D1	27	14	01-50-49	5251	53AE
D2D2	27	14	01-50-50	5252	53B0
D2D3	28	2	01-50-51	5253	53B6
D2D4	28	11	01-50-52	5254	53C3
D2D5	28	17	01-50-53	5255	7C12
D2D6	172	18	01-50-54	5256	96D9
D2D7	29	9	01-50-55	5257	53DF
D2D8	29	11	01-50-56	5258	66FC
D2D9	29	17	01-50-57	5259	71EE
D2DA	30	5	01-50-58	525A	53EE
D2DB	30	5	01-50-59	525B	53E8
D2DC	30	5	01-50-60	525C	53ED
D2DD	30	5	01-50-61	525D	53FA
D2DE	30	6	01-50-62	525E	5401
D2DF	30	7	01-50-63	525F	543D
D2E0	30	7	01-50-64	5260	5440
D2E1	30	7	01-50-65	5261	542C
D2E2	30	7	01-50-66	5262	542D
D2E3	30	7	01-50-67	5263	543C
D2E4	30	7	01-50-68	5264	542E
D2E5	30	7	01-50-69	5265	5436
D2E6	30	7	01-50-70	5266	5429
D2E7	30	7	01-50-71	5267	541D
D2E8	30	7	01-50-72	5268	544E
D2E9	30	8	01-50-73	5269	548F
D2EA	30	8	01-50-74	526A	5475
D2EB	30	8	01-50-75	526B	548E
D2EC	30	8	01-50-76	526C	545F
D2ED	30	8	01-50-77	526D	5471
D2EE	30	8	01-50-78	526E	5477
D2EF	30	9	01-50-79	526F	5470
D2F0	30	8	01-50-80	5270	5492
D2F1	30	8	01-50-81	5271	547B
D2F2	30	8	01-50-82	5272	5480
D2F3	30	8	01-50-83	5273	5476
D2F4	30	8	01-50-84	5274	5484
D2F5	30	8	01-50-85	5275	5490
D2F6	30	8	01-50-86	5276	5486
D2F7	30	9	01-50-87	5277	54C7
D2F8	30	9	01-50-88	5278	54A2
D2F9	30	9	01-50-89	5279	54B8
D2FA	30	9	01-50-90	527A	54A5
D2FB	30	9	01-50-91	527B	54AC
D2FC	30	9	01-50-92	527C	54C4
D2FD	30	9	01-50-93	527D	54C8
D2FE	30	9	01-50-94	527E	54A8
D3A1	30	9	01-51-01	5321	54AB
D3A2	30	9	01-51-02	5322	54C2
D3A3	30	9	01-51-03	5323	54A4
D3A4	30	9	01-51-04	5324	54BE
D3A5	30	9	01-51-05	5325	54BC
D3A6	30	9	01-51-06	5326	54D8
D3A7	30	10	01-51-07	5327	54E5
D3A8	30	10	01-51-08	5328	54E6
D3A9	30	10	01-51-09	5329	550F
D3AA	30	10	01-51-10	532A	5514
D3AB	30	10	01-51-11	532B	54FD
D3AC	30	10	01-51-12	532C	54EE
D3AD	30	10	01-51-13	532D	54ED
D3AE	30	10	01-51-14	532E	54FA
D3AF	30	10	01-51-15	532F	54E2
D3B0	30	11	01-51-16	5330	5539
D3B1	30	11	01-51-17	5331	5540
D3B2	30	12	01-51-18	5332	5563
D3B3	30	11	01-51-19	5333	554C
D3B4	30	11	01-51-20	5334	552E
D3B5	30	11	01-51-21	5335	555C
D3B6	30	11	01-51-22	5336	5545
D3B7	30	11	01-51-23	5337	5556
D3B8	30	11	01-51-24	5338	5557
D3B9	30	11	01-51-25	5339	5538
D3BA	30	10	01-51-26	533A	5533
D3BB	30	11	01-51-27	533B	555D
D3BC	30	12	01-51-28	533C	5599
D3BD	30	12	01-51-29	533D	5580
D3BE	30	9	01-51-30	533E	54AF
D3BF	30	12	01-51-31	533F	558A
D3C0	30	12	01-51-32	5340	559F
D3C1	30	12	01-51-33	5341	557B
D3C2	30	12	01-51-34	5342	557E
D3C3	30	12	01-51-35	5343	5598
D3C4	30	12	01-51-36	5344	559E
D3C5	30	12	01-51-37	5345	55AE
D3C6	30	12	01-51-38	5346	557C
D3C7	30	12	01-51-39	5347	5583
D3C8	30	12	01-51-40	5348	55A9
D3C9	30	12	01-51-41	5349	5587
D3CA	30	12	01-51-42	534A	55A8
D3CB	30	13	01-51-43	534B	55DA
D3CC	30	13	01-51-44	534C	55C5
D3CD	30	13	01-51-45	534D	55DF
D3CE	30	13	01-51-46	534E	55C4
D3CF	30	13	01-51-47	534F	55DC
D3D0	30	13	01-51-48	5350	55E4
D3D1	30	13	01-51-49	5351	55D4
D3D2	30	14	01-51-50	5352	5614
D3D3	30	14	01-51-51	5353	55F7
D3D4	30	14	01-51-52	5354	5616
D3D5	30	14	01-51-53	5355	55FE
D3D6	30	14	01-51-54	5356	55FD
D3D7	30	14	01-51-55	5357	561B
D3D8	30	14	01-51-56	5358	55F9
D3D9	30	15	01-51-57	5359	564E
D3DA	30	15	01-51-58	535A	5650
D3DB	30	17	01-51-59	535B	71DF
D3DC	30	16	01-51-60	535C	5634
D3DD	30	15	01-51-61	535D	5636
D3DE	30	15	01-51-62	535E	5632
D3DF	30	15	01-51-63	535F	5638
D3E0	30	16	01-51-64	5360	566B
D3E1	30	16	01-51-65	5361	5664
D3E2	30	15	01-51-66	5362	562F
D3E3	30	16	01-51-67	5363	566C
D3E4	30	16	01-51-68	5364	566A
D3E5	30	16	01-51-69	5365	5686
D3E6	30	17	01-51-70	5366	5680
D3E7	30	17	01-51-71	5367	568A
D3E8	30	18	01-51-72	5368	56A0
D3E9	30	18	01-51-73	5369	5694
D3EA	30	17	01-51-74	536A	568F
D3EB	30	19	01-51-75	536B	56A5
D3EC	30	19	01-51-76	536C	56AE
D3ED	30	20	01-51-77	536D	56B6
D3EE	30	20	01-51-78	536E	56B4
D3EF	30	21	01-51-79	536F	56C2
D3F0	30	21	01-51-80	5370	56BC
D3F1	30	21	01-51-81	5371	56C1
D3F2	30	21	01-51-82	5372	56C3
D3F3	30	21	01-51-83	5373	56C0
D3F4	30	21	01-51-84	5374	56C8
D3F5	30	22	01-51-85	5375	56CE
D3F6	30	24	01-51-86	5376	56D1
D3F7	211	24	01-51-87	5377	56D3
D3F8	31	3	01-51-88	5378	56D7
D3F9	31	7	01-51-89	5379	56EE
D3FA	31	8	01-51-90	537A	56F9
D3FB	31	9	01-51-91	537B	5700
D3FC	31	9	01-51-92	537C	56FF
D3FD	31	10	01-51-93	537D	5704
D3FE	31	11	01-51-94	537E	5709
D4A1	31	11	01-52-01	5421	5708
D4A2	31	11	01-52-02	5422	570B
D4A3	31	12	01-52-03	5423	570D
D4A4	31	13	01-52-04	5424	5713
D4A5	31	14	01-52-05	5425	5718
D4A6	31	14	01-52-06	5426	5716
D4A7	31	13	01-52-07	5427	55C7
D4A8	31	16	01-52-08	5428	571C
D4A9	32	5	01-52-09	5429	5726
D4AA	32	6	01-52-10	542A	5737
D4AB	32	6	01-52-11	542B	5738
D4AC	32	7	01-52-12	542C	574E
D4AD	32	7	01-52-13	542D	573B
D4AE	32	7	01-52-14	542E	5740
D4AF	32	7	01-52-15	542F	574F
D4B0	32	8	01-52-16	5430	5769
D4B1	32	10	01-52-17	5431	57C0
D4B2	32	8	01-52-18	5432	5788
D4B3	32	8	01-52-19	5433	5761
D4B4	32	8	01-52-20	5434	577F
D4B5	32	8	01-52-21	5435	5789
D4B6	32	9	01-52-22	5436	5793
D4B7	32	9	01-52-23	5437	57A0
D4B8	32	9	01-52-24	5438	57B3
D4B9	32	9	01-52-25	5439	57A4
D4BA	32	9	01-52-26	543A	57AA
D4BB	32	9	01-52-27	543B	57B0
D4BC	32	10	01-52-28	543C	57C3
D4BD	112	10	01-52-29	543D	57C6
D4BE	32	10	01-52-30	543E	57D4
D4BF	32	10	01-52-31	543F	57D2
D4C0	32	10	01-52-32	5440	57D3
D4C1	32	11	01-52-33	5441	580A
D4C2	32	10	01-52-34	5442	57D6
D4C3	32	11	01-52-35	5443	57E3
D4C4	32	11	01-52-36	5444	580B
D4C5	32	12	01-52-37	5445	5819
D4C6	32	12	01-52-38	5446	581D
D4C7	32	14	01-52-39	5447	5872
D4C8	32	12	01-52-40	5448	5821
D4C9	32	13	01-52-41	5449	5862
D4CA	32	13	01-52-42	544A	584B
D4CB	32	13	01-52-43	544B	5870
D4CC	32	13	01-52-44	544C	6BC0
D4CD	32	13	01-52-45	544D	5852
D4CE	32	13	01-52-46	544E	583D
D4CF	32	14	01-52-47	544F	5879
D4D0	32	14	01-52-48	5450	5885
D4D1	32	15	01-52-49	5451	58B9
D4D2	32	15	01-52-50	5452	589F
D4D3	32	15	01-52-51	5453	58AB
D4D4	32	16	01-52-52	5454	58BA
D4D5	32	19	01-52-53	5455	58DE
D4D6	32	16	01-52-54	5456	58BB
D4D7	32	15	01-52-55	5457	58B8
D4D8	32	15	01-52-56	5458	58AE
D4D9	32	16	01-52-57	5459	58C5
D4DA	32	17	01-52-58	545A	58D3
D4DB	32	17	01-52-59	545B	58D1
D4DC	32	17	01-52-60	545C	58D7
D4DD	32	18	01-52-61	545D	58D9
D4DE	32	18	01-52-62	545E	58D8
D4DF	32	20	01-52-63	545F	58E5
D4E0	32	19	01-52-64	5460	58DC
D4E1	32	20	01-52-65	5461	58E4
D4E2	32	19	01-52-66	5462	58DF
D4E3	33	7	01-52-67	5463	58EF
D4E4	33	12	01-52-68	5464	58FA
D4E5	33	12	01-52-69	5465	58F9
D4E6	33	12	01-52-70	5466	58FB
D4E7	33	13	01-52-71	5467	58FC
D4E8	33	14	01-52-72	5468	58FD
D4E9	34	3	01-52-73	5469	5902
D4EA	35	3	01-52-74	546A	590A
D4EB	35	14	01-52-75	546B	5910
D4EC	36	6	01-52-76	546C	591B
D4ED	36	11	01-52-77	546D	68A6
D4EE	36	14	01-52-78	546E	5925
D4EF	37	4	01-52-79	546F	592C
D4F0	37	4	01-52-80	5470	592D
D4F1	37	5	01-52-81	5471	5932
D4F2	37	6	01-52-82	5472	5938
D4F3	37	7	01-52-83	5473	593E
D4F4	117	9	01-52-84	5474	7AD2
D4F5	37	9	01-52-85	5475	5955
D4F6	37	9	01-52-86	5476	5950
D4F7	37	9	01-52-87	5477	594E
D4F8	37	10	01-52-88	5478	595A
D4F9	37	10	01-52-89	5479	5958
D4FA	37	12	01-52-90	547A	5962
D4FB	37	12	01-52-91	547B	5960
D4FC	37	13	01-52-92	547C	5967
D4FD	37	14	01-52-93	547D	596C
D4FE	37	14	01-52-94	547E	5969
D5A1	38	6	01-53-01	5521	5978
D5A2	38	6	01-53-02	5522	5981
D5A3	38	7	01-53-03	5523	599D
D5A4	38	7	01-53-04	5524	4F5E
D5A5	38	8	01-53-05	5525	4FAB
D5A6	38	7	01-53-06	5526	59A3
D5A7	38	8	01-53-07	5527	59B2
D5A8	38	8	01-53-08	5528	59C6
D5A9	38	9	01-53-09	5529	59E8
D5AA	38	9	01-53-10	552A	59DC
D5AB	38	7	01-53-11	552B	598D
D5AC	38	9	01-53-12	552C	59D9
D5AD	38	9	01-53-13	552D	59DA
D5AE	38	10	01-53-14	552E	5A25
D5AF	38	10	01-53-15	552F	5A1F
D5B0	38	10	01-53-16	5530	5A11
D5B1	38	10	01-53-17	5531	5A1C
D5B2	38	10	01-53-18	5532	5A09
D5B3	38	10	01-53-19	5533	5A1A
D5B4	38	11	01-53-20	5534	5A40
D5B5	38	11	01-53-21	5535	5A6C
D5B6	38	11	01-53-22	5536	5A49
D5B7	38	11	01-53-23	5537	5A35
D5B8	38	11	01-53-24	5538	5A36
D5B9	38	11	01-53-25	5539	5A62
D5BA	38	11	01-53-26	553A	5A6A
D5BB	38	12	01-53-27	553B	5A9A
D5BC	38	13	01-53-28	553C	5ABC
D5BD	38	13	01-53-29	553D	5ABE
D5BE	38	13	01-53-30	553E	5ACB
D5BF	38	12	01-53-31	553F	5AC2
D5C0	38	13	01-53-32	5540	5ABD
D5C1	38	14	01-53-33	5541	5AE3
D5C2	38	14	01-53-34	5542	5AD7
D5C3	38	14	01-53-35	5543	5AE6
D5C4	38	14	01-53-36	5544	5AE9
D5C5	38	14	01-53-37	5545	5AD6
D5C6	38	15	01-53-38	5546	5AFA
D5C7	38	15	01-53-39	5547	5AFB
D5C8	38	15	01-53-40	5548	5B0C
D5C9	38	15	01-53-41	5549	5B0B
D5CA	38	16	01-53-42	554A	5B16
D5CB	38	17	01-53-43	554B	5B32
D5CC	38	13	01-53-44	554C	5AD0
D5CD	38	17	01-53-45	554D	5B2A
D5CE	38	17	01-53-46	554E	5B36
D5CF	38	19	01-53-47	554F	5B3E
D5D0	38	20	01-53-48	5550	5B43
D5D1	38	20	01-53-49	5551	5B45
D5D2	38	20	01-53-50	5552	5B40
D5D3	39	3	01-53-51	5553	5B51
D5D4	39	5	01-53-52	5554	5B55
D5D5	39	7	01-53-53	5555	5B5A
D5D6	39	7	01-53-54	5556	5B5B
D5D7	39	8	01-53-55	5557	5B65
D5D8	39	9	01-53-56	5558	5B69
D5D9	39	11	01-53-57	5559	5B70
D5DA	39	12	01-53-58	555A	5B73
D5DB	39	14	01-53-59	555B	5B75
D5DC	39	16	01-53-60	555C	5B78
D5DD	67	7	01-53-61	555D	6588
D5DE	39	17	01-53-62	555E	5B7A
D5DF	40	3	01-53-63	555F	5B80
D5E0	40	5	01-53-64	5560	5B83
D5E1	40	10	01-53-65	5561	5BA6
D5E2	40	10	01-53-66	5562	5BB8
D5E3	40	11	01-53-67	5563	5BC3
D5E4	66	11	01-53-68	5564	5BC7
D5E5	40	11	01-53-69	5565	5BC9
D5E6	40	12	01-53-70	5566	5BD4
D5E7	40	12	01-53-71	5567	5BD0
D5E8	40	14	01-53-72	5568	5BE4
D5E9	40	14	01-53-73	5569	5BE6
D5EA	40	14	01-53-74	556A	5BE2
D5EB	40	13	01-53-75	556B	5BDE
D5EC	40	14	01-53-76	556C	5BE5
D5ED	40	15	01-53-77	556D	5BEB
D5EE	40	16	01-53-78	556E	5BF0
D5EF	40	20	01-53-79	556F	5BF6
D5F0	40	19	01-53-80	5570	5BF3
D5F1	41	10	01-53-81	5571	5C05
D5F2	41	11	01-53-82	5572	5C07
D5F3	41	11	01-53-83	5573	5C08
D5F4	41	14	01-53-84	5574	5C0D
D5F5	42	5	01-53-85	5575	5C13
D5F6	42	13	01-53-86	5576	5C20
D5F7	43	3	01-53-87	5577	5C22
D5F8	43	7	01-53-88	5578	5C28
D5F9	44	3	01-53-89	5579	5C38
D5FA	44	4	01-53-90	557A	5C39
D5FB	44	7	01-53-91	557B	5C41
D5FC	44	8	01-53-92	557C	5C46
D5FD	44	9	01-53-93	557D	5C4E
D5FE	44	10	01-53-94	557E	5C53
D6A1	44	10	01-54-01	5621	5C50
D6A2	44	9	01-54-02	5622	5C4F
D6A3	44	12	01-54-03	5623	5B71
D6A4	44	21	01-54-04	5624	5C6C
D6A5	45	3	01-54-05	5625	5C6E
D6A6	46	4	01-54-06	5626	4E62
D6A7	46	5	01-54-07	5627	5C76
D6A8	46	6	01-54-08	5628	5C79
D6A9	46	7	01-54-09	5629	5C8C
D6AA	46	7	01-54-10	562A	5C91
D6AB	46	7	01-54-11	562B	5C94
D6AC	46	7	01-54-12	562C	599B
D6AD	46	8	01-54-13	562D	5CAB
D6AE	46	8	01-54-14	562E	5CBB
D6AF	46	8	01-54-15	562F	5CB6
D6B0	46	8	01-54-16	5630	5CBC
D6B1	46	8	01-54-17	5631	5CB7
D6B2	46	8	01-54-18	5632	5CC5
D6B3	46	8	01-54-19	5633	5CBE
D6B4	46	9	01-54-20	5634	5CC7
D6B5	46	9	01-54-21	5635	5CD9
D6B6	46	10	01-54-22	5636	5CE9
D6B7	46	10	01-54-23	5637	5CFD
D6B8	46	10	01-54-24	5638	5CFA
D6B9	46	10	01-54-25	5639	5CED
D6BA	46	14	01-54-26	563A	5D8C
D6BB	46	10	01-54-27	563B	5CEA
D6BC	46	10	01-54-28	563C	5D0B
D6BD	46	11	01-54-29	563D	5D15
D6BE	46	11	01-54-30	563E	5D17
D6BF	46	12	01-54-31	563F	5D5C
D6C0	46	11	01-54-32	5640	5D1F
D6C1	46	11	01-54-33	5641	5D1B
D6C2	46	11	01-54-34	5642	5D11
D6C3	46	11	01-54-35	5643	5D14
D6C4	46	11	01-54-36	5644	5D22
D6C5	46	11	01-54-37	5645	5D1A
D6C6	46	11	01-54-38	5646	5D19
D6C7	46	11	01-54-39	5647	5D18
D6C8	46	12	01-54-40	5648	5D4C
D6C9	46	12	01-54-41	5649	5D52
D6CA	46	12	01-54-42	564A	5D4E
D6CB	46	12	01-54-43	564B	5D4B
D6CC	46	13	01-54-44	564C	5D6C
D6CD	46	13	01-54-45	564D	5D73
D6CE	46	13	01-54-46	564E	5D76
D6CF	46	14	01-54-47	564F	5D87
D6D0	46	14	01-54-48	5650	5D84
D6D1	46	14	01-54-49	5651	5D82
D6D2	46	15	01-54-50	5652	5DA2
D6D3	46	15	01-54-51	5653	5D9D
D6D4	46	16	01-54-52	5654	5DAC
D6D5	46	16	01-54-53	5655	5DAE
D6D6	46	17	01-54-54	5656	5DBD
D6D7	46	15	01-54-55	5657	5D90
D6D8	46	17	01-54-56	5658	5DB7
D6D9	46	17	01-54-57	5659	5DBC
D6DA	46	20	01-54-58	565A	5DC9
D6DB	46	21	01-54-59	565B	5DCD
D6DC	46	22	01-54-60	565C	5DD3
D6DD	46	22	01-54-61	565D	5DD2
D6DE	46	23	01-54-62	565E	5DD6
D6DF	47	3	01-54-63	565F	5DDB
D6E0	48	7	01-54-64	5660	5DEB
D6E1	49	3	01-54-65	5661	5DF2
D6E2	26	7	01-54-66	5662	5DF5
D6E3	50	7	01-54-67	5663	5E0B
D6E4	50	8	01-54-68	5664	5E1A
D6E5	50	8	01-54-69	5665	5E19
D6E6	50	8	01-54-70	5666	5E11
D6E7	50	8	01-54-71	5667	5E1B
D6E8	50	11	01-54-72	5668	5E36
D6E9	50	11	01-54-73	5669	5E37
D6EA	50	12	01-54-74	566A	5E44
D6EB	50	12	01-54-75	566B	5E43
D6EC	50	12	01-54-76	566C	5E40
D6ED	50	13	01-54-77	566D	5E4E
D6EE	50	14	01-54-78	566E	5E57
D6EF	50	14	01-54-79	566F	5E54
D6F0	50	15	01-54-80	5670	5E5F
D6F1	50	15	01-54-81	5671	5E62
D6F2	50	15	01-54-82	5672	5E64
D6F3	50	12	01-54-83	5673	5E47
D6F4	51	6	01-54-84	5674	5E75
D6F5	51	6	01-54-85	5675	5E76
D6F6	52	3	01-54-86	5676	5E7A
D6F7	52	14	01-54-87	5677	9EBC
D6F8	53	3	01-54-88	5678	5E7F
D6F9	53	9	01-54-89	5679	5EA0
D6FA	53	12	01-54-90	567A	5EC1
D6FB	53	12	01-54-91	567B	5EC2
D6FC	53	13	01-54-92	567C	5EC8
D6FD	53	14	01-54-93	567D	5ED0
D6FE	53	14	01-54-94	567E	5ECF
D7A1	53	14	01-55-01	5721	5ED6
D7A2	53	15	01-55-02	5722	5EE3
D7A3	53	15	01-55-03	5723	5EDD
D7A4	53	15	01-55-04	5724	5EDA
D7A5	53	15	01-55-05	5725	5EDB
D7A6	53	15	01-55-06	5726	5EE2
D7A7	53	15	01-55-07	5727	5EE1
D7A8	53	16	01-55-08	5728	5EE8
D7A9	53	16	01-55-09	5729	5EE9
D7AA	53	19	01-55-10	572A	5EEC
D7AB	53	21	01-55-11	572B	5EF1
D7AC	53	25	01-55-12	572C	5EF3
D7AD	53	20	01-55-13	572D	5EF0
D7AE	54	3	01-55-14	572E	5EF4
D7AF	54	8	01-55-15	572F	5EF8
D7B0	55	3	01-55-16	5730	5EFE
D7B1	75	7	01-55-17	5731	5F03
D7B2	37	10	01-55-18	5732	5F09
D7B3	55	18	01-55-19	5733	5F5D
D7B4	55	16	01-55-20	5734	5F5C
D7B5	56	3	01-55-21	5735	5F0B
D7B6	56	12	01-55-22	5736	5F11
D7B7	57	4	01-55-23	5737	5F16
D7B8	57	8	01-55-24	5738	5F29
D7B9	57	9	01-55-25	5739	5F2D
D7BA	57	11	01-55-26	573A	5F38
D7BB	57	13	01-55-27	573B	5F41
D7BC	57	15	01-55-28	573C	5F48
D7BD	57	17	01-55-29	573D	5F4C
D7BE	57	22	01-55-30	573E	5F4E
D7BF	57	9	01-55-31	573F	5F2F
D7C0	58	3	01-55-32	5740	5F51
D7C1	58	9	01-55-33	5741	5F56
D7C2	58	11	01-55-34	5742	5F57
D7C3	58	13	01-55-35	5743	5F59
D7C4	59	3	01-55-36	5744	5F61
D7C5	59	12	01-55-37	5745	5F6D
D7C6	60	3	01-55-38	5746	5F73
D7C7	60	7	01-55-39	5747	5F77
D7C8	60	8	01-55-40	5748	5F83
D7C9	60	8	01-55-41	5749	5F82
D7CA	60	8	01-55-42	574A	5F7F
D7CB	60	9	01-55-43	574B	5F8A
D7CC	60	9	01-55-44	574C	5F88
D7CD	60	10	01-55-45	574D	5F91
D7CE	60	9	01-55-46	574E	5F87
D7CF	60	11	01-55-47	574F	5F9E
D7D0	60	11	01-55-48	5750	5F99
D7D1	60	11	01-55-49	5751	5F98
D7D2	60	11	01-55-50	5752	5FA0
D7D3	60	12	01-55-51	5753	5FA8
D7D4	60	13	01-55-52	5754	5FAD
D7D5	60	16	01-55-53	5755	5FBC
D7D6	61	6	01-55-54	5756	5FD6
D7D7	61	7	01-55-55	5757	5FFB
D7D8	61	7	01-55-56	5758	5FE4
D7D9	61	7	01-55-57	5759	5FF8
D7DA	61	7	01-55-58	575A	5FF1
D7DB	61	8	01-55-59	575B	5FDD
D7DC	61	12	01-55-60	575C	60B3
D7DD	61	8	01-55-61	575D	5FFF
D7DE	61	8	01-55-62	575E	6021
D7DF	61	9	01-55-63	575F	6060
D7E0	61	8	01-55-64	5760	6019
D7E1	61	8	01-55-65	5761	6010
D7E2	61	8	01-55-66	5762	6029
D7E3	61	9	01-55-67	5763	600E
D7E4	61	9	01-55-68	5764	6031
D7E5	61	8	01-55-69	5765	601B
D7E6	61	8	01-55-70	5766	6015
D7E7	61	8	01-55-71	5767	602B
D7E8	61	8	01-55-72	5768	6026
D7E9	61	8	01-55-73	5769	600F
D7EA	61	8	01-55-74	576A	603A
D7EB	61	10	01-55-75	576B	605A
D7EC	61	10	01-55-76	576C	6041
D7ED	61	9	01-55-77	576D	606A
D7EE	61	10	01-55-78	576E	6077
D7EF	61	9	01-55-79	576F	605F
D7F0	61	9	01-55-80	5770	604A
D7F1	61	9	01-55-81	5771	6046
D7F2	61	9	01-55-82	5772	604D
D7F3	61	10	01-55-83	5773	6063
D7F4	61	9	01-55-84	5774	6043
D7F5	61	9	01-55-85	5775	6064
D7F6	61	9	01-55-86	5776	6042
D7F7	61	9	01-55-87	5777	606C
D7F8	61	9	01-55-88	5778	606B
D7F9	61	10	01-55-89	5779	6059
D7FA	61	10	01-55-90	577A	6081
D7FB	61	10	01-55-91	577B	608D
D7FC	61	11	01-55-92	577C	60E7
D7FD	61	10	01-55-93	577D	6083
D7FE	61	10	01-55-94	577E	609A
D8A1	61	10	01-56-01	5821	6084
D8A2	61	10	01-56-02	5822	609B
D8A3	61	10	01-56-03	5823	6096
D8A4	61	10	01-56-04	5824	6097
D8A5	61	10	01-56-05	5825	6092
D8A6	61	10	01-56-06	5826	60A7
D8A7	61	10	01-56-07	5827	608B
D8A8	61	12	01-56-08	5828	60E1
D8A9	61	11	01-56-09	5829	60B8
D8AA	61	12	01-56-10	582A	60E0
D8AB	61	11	01-56-11	582B	60D3
D8AC	61	11	01-56-12	582C	60B4
D8AD	61	7	01-56-13	582D	5FF0
D8AE	61	11	01-56-14	582E	60BD
D8AF	61	11	01-56-15	582F	60C6
D8B0	61	11	01-56-16	5830	60B5
D8B1	61	11	01-56-17	5831	60D8
D8B2	61	13	01-56-18	5832	614D
D8B3	61	12	01-56-19	5833	6115
D8B4	61	13	01-56-20	5834	6106
D8B5	61	12	01-56-21	5835	60F6
D8B6	61	13	01-56-22	5836	60F7
D8B7	61	12	01-56-23	5837	6100
D8B8	61	12	01-56-24	5838	60F4
D8B9	61	12	01-56-25	5839	60FA
D8BA	61	12	01-56-26	583A	6103
D8BB	61	12	01-56-27	583B	6121
D8BC	61	12	01-56-28	583C	60FB
D8BD	61	12	01-56-29	583D	60F1
D8BE	61	13	01-56-30	583E	610D
D8BF	61	12	01-56-31	583F	610E
D8C0	61	14	01-56-32	5840	6147
D8C1	61	13	01-56-33	5841	613E
D8C2	61	14	01-56-34	5842	6128
D8C3	61	13	01-56-35	5843	6127
D8C4	61	13	01-56-36	5844	614A
D8C5	61	14	01-56-37	5845	613F
D8C6	61	13	01-56-38	5846	613C
D8C7	61	14	01-56-39	5847	612C
D8C8	61	13	01-56-40	5848	6134
D8C9	61	13	01-56-41	5849	613D
D8CA	61	14	01-56-42	584A	6142
D8CB	61	13	01-56-43	584B	6144
D8CC	61	15	01-56-44	584C	6173
D8CD	61	14	01-56-45	584D	6177
D8CE	61	14	01-56-46	584E	6158
D8CF	61	15	01-56-47	584F	6159
D8D0	61	14	01-56-48	5850	615A
D8D1	61	15	01-56-49	5851	616B
D8D2	61	14	01-56-50	5852	6174
D8D3	61	14	01-56-51	5853	616F
D8D4	61	14	01-56-52	5854	6165
D8D5	61	14	01-56-53	5855	6171
D8D6	61	14	01-56-54	5856	615F
D8D7	61	14	01-56-55	5857	615D
D8D8	61	14	01-56-56	5858	6153
D8D9	61	14	01-56-57	5859	6175
D8DA	61	16	01-56-58	585A	6199
D8DB	61	16	01-56-59	585B	6196
D8DC	61	15	01-56-60	585C	6187
D8DD	61	15	01-56-61	585D	61AC
D8DE	61	15	01-56-62	585E	6194
D8DF	61	15	01-56-63	585F	619A
D8E0	61	16	01-56-64	5860	618A
D8E1	61	16	01-56-65	5861	6191
D8E2	61	15	01-56-66	5862	61AB
D8E3	61	15	01-56-67	5863	61AE
D8E4	61	16	01-56-68	5864	61CC
D8E5	61	16	01-56-69	5865	61CA
D8E6	61	17	01-56-70	5866	61C9
D8E7	61	19	01-56-71	5867	61F7
D8E8	61	16	01-56-72	5868	61C8
D8E9	61	17	01-56-73	5869	61C3
D8EA	61	16	01-56-74	586A	61C6
D8EB	61	16	01-56-75	586B	61BA
D8EC	61	17	01-56-76	586C	61CB
D8ED	61	16	01-56-77	586D	7F79
D8EE	61	16	01-56-78	586E	61CD
D8EF	61	17	01-56-79	586F	61E6
D8F0	61	18	01-56-80	5870	61E3
D8F1	61	19	01-56-81	5871	61F6
D8F2	61	20	01-56-82	5872	61FA
D8F3	61	18	01-56-83	5873	61F4
D8F4	61	22	01-56-84	5874	61FF
D8F5	61	20	01-56-85	5875	61FD
D8F6	61	21	01-56-86	5876	61FC
D8F7	61	21	01-56-87	5877	61FE
D8F8	61	23	01-56-88	5878	6200
D8F9	62	4	01-56-89	5879	6208
D8FA	62	5	01-56-90	587A	6209
D8FB	62	6	01-56-91	587B	620D
D8FC	62	6	01-56-92	587C	620C
D8FD	62	8	01-56-93	587D	6214
D8FE	62	11	01-56-94	587E	621B
D9A1	62	12	01-57-01	5921	621E
D9A2	62	13	01-57-02	5922	6221
D9A3	62	14	01-57-03	5923	622A
D9A4	62	15	01-57-04	5924	622E
D9A5	62	16	01-57-05	5925	6230
D9A6	62	17	01-57-06	5926	6232
D9A7	62	18	01-57-07	5927	6233
D9A8	63	9	01-57-08	5928	6241
D9A9	64	4	01-57-09	5929	624E
D9AA	64	6	01-57-10	592A	625E
D9AB	64	6	01-57-11	592B	6263
D9AC	64	6	01-57-12	592C	625B
D9AD	64	6	01-57-13	592D	6260
D9AE	64	6	01-57-14	592E	6268
D9AF	64	7	01-57-15	592F	627C
D9B0	64	7	01-57-16	5930	6282
D9B1	64	7	01-57-17	5931	6289
D9B2	64	7	01-57-18	5932	627E
D9B3	64	7	01-57-19	5933	6292
D9B4	64	7	01-57-20	5934	6293
D9B5	64	7	01-57-21	5935	6296
D9B6	64	8	01-57-22	5936	62D4
D9B7	64	7	01-57-23	5937	6283
D9B8	64	7	01-57-24	5938	6294
D9B9	64	8	01-57-25	5939	62D7
D9BA	64	8	01-57-26	593A	62D1
D9BB	64	8	01-57-27	593B	62BB
D9BC	64	9	01-57-28	593C	62CF
D9BD	64	10	01-57-29	593D	62FF
D9BE	64	8	01-57-30	593E	62C6
D9BF	64	16	01-57-31	593F	64D4
D9C0	64	8	01-57-32	5940	62C8
D9C1	64	9	01-57-33	5941	62DC
D9C2	64	8	01-57-34	5942	62CC
D9C3	64	8	01-57-35	5943	62CA
D9C4	64	8	01-57-36	5944	62C2
D9C5	64	8	01-57-37	5945	62C7
D9C6	64	7	01-57-38	5946	629B
D9C7	64	8	01-57-39	5947	62C9
D9C8	64	9	01-57-40	5948	630C
D9C9	64	9	01-57-41	5949	62EE
D9CA	64	9	01-57-42	594A	62F1
D9CB	64	9	01-57-43	594B	6327
D9CC	64	9	01-57-44	594C	6302
D9CD	64	10	01-57-45	594D	6308
D9CE	64	9	01-57-46	594E	62EF
D9CF	64	9	01-57-47	594F	62F5
D9D0	64	10	01-57-48	5950	6350
D9D1	64	10	01-57-49	5951	633E
D9D2	64	10	01-57-50	5952	634D
D9D3	64	12	01-57-51	5953	641C
D9D4	64	10	01-57-52	5954	634F
D9D5	64	11	01-57-53	5955	6396
D9D6	64	11	01-57-54	5956	638E
D9D7	64	11	01-57-55	5957	6380
D9D8	64	11	01-57-56	5958	63AB
D9D9	64	11	01-57-57	5959	6376
D9DA	64	12	01-57-58	595A	63A3
D9DB	64	11	01-57-59	595B	638F
D9DC	64	11	01-57-60	595C	6389
D9DD	64	11	01-57-61	595D	639F
D9DE	64	11	01-57-62	595E	63B5
D9DF	64	11	01-57-63	595F	636B
D9E0	64	10	01-57-64	5960	6369
D9E1	64	12	01-57-65	5961	63BE
D9E2	64	12	01-57-66	5962	63E9
D9E3	64	12	01-57-67	5963	63C0
D9E4	64	12	01-57-68	5964	63C6
D9E5	64	12	01-57-69	5965	63E3
D9E6	64	12	01-57-70	5966	63C9
D9E7	64	12	01-57-71	5967	63D2
D9E8	64	12	01-57-72	5968	63F6
D9E9	64	12	01-57-73	5969	63C4
D9EA	64	13	01-57-74	596A	6416
D9EB	64	14	01-57-75	596B	6434
D9EC	64	13	01-57-76	596C	6406
D9ED	64	13	01-57-77	596D	6413
D9EE	64	13	01-57-78	596E	6426
D9EF	64	13	01-57-79	596F	6436
D9F0	64	21	01-57-80	5970	651D
D9F1	64	13	01-57-81	5971	6417
D9F2	64	13	01-57-82	5972	6428
D9F3	64	13	01-57-83	5973	640F
D9F4	64	14	01-57-84	5974	6467
D9F5	64	15	01-57-85	5975	646F
D9F6	64	14	01-57-86	5976	6476
D9F7	64	14	01-57-87	5977	644E
D9F8	64	23	01-57-88	5978	652A
D9F9	64	15	01-57-89	5979	6495
D9FA	64	15	01-57-90	597A	6493
D9FB	64	15	01-57-91	597B	64A5
D9FC	64	15	01-57-92	597C	64A9
D9FD	64	15	01-57-93	597D	6488
D9FE	64	16	01-57-94	597E	64BC
DAA1	64	16	01-58-01	5A21	64DA
DAA2	64	16	01-58-02	5A22	64D2
DAA3	64	16	01-58-03	5A23	64C5
DAA4	64	16	01-58-04	5A24	64C7
DAA5	64	16	01-58-05	5A25	64BB
DAA6	64	17	01-58-06	5A26	64D8
DAA7	64	16	01-58-07	5A27	64C2
DAA8	64	17	01-58-08	5A28	64F1
DAA9	64	18	01-58-09	5A29	64E7
DAAA	134	17	01-58-10	5A2A	8209
DAAB	64	17	01-58-11	5A2B	64E0
DAAC	64	17	01-58-12	5A2C	64E1
DAAD	64	8	01-58-13	5A2D	62AC
DAAE	64	17	01-58-14	5A2E	64E3
DAAF	64	17	01-58-15	5A2F	64EF
DAB0	64	25	01-58-16	5A30	652C
DAB1	64	18	01-58-17	5A31	64F6
DAB2	64	18	01-58-18	5A32	64F4
DAB3	64	18	01-58-19	5A33	64F2
DAB4	64	18	01-58-20	5A34	64FA
DAB5	64	19	01-58-21	5A35	6500
DAB6	64	18	01-58-22	5A36	64FD
DAB7	64	20	01-58-23	5A37	6518
DAB8	64	21	01-58-24	5A38	651C
DAB9	64	18	01-58-25	5A39	6505
DABA	64	22	01-58-26	5A3A	6524
DABB	64	23	01-58-27	5A3B	6523
DABC	64	23	01-58-28	5A3C	652B
DABD	66	4	01-58-29	5A3D	6534
DABE	66	4	01-58-30	5A3E	6535
DABF	66	6	01-58-31	5A3F	6537
DAC0	66	6	01-58-32	5A40	6536
DAC1	66	7	01-58-33	5A41	6538
DAC2	66	9	01-58-34	5A42	754B
DAC3	66	10	01-58-35	5A43	6548
DAC4	66	11	01-58-36	5A44	6556
DAC5	66	11	01-58-37	5A45	6555
DAC6	66	11	01-58-38	5A46	654D
DAC7	66	11	01-58-39	5A47	6558
DAC8	66	12	01-58-40	5A48	655E
DAC9	66	12	01-58-41	5A49	655D
DACA	66	14	01-58-42	5A4A	6572
DACB	66	15	01-58-43	5A4B	6578
DACC	66	17	01-58-44	5A4C	6582
DACD	66	18	01-58-45	5A4D	6583
DACE	66	23	01-58-46	5A4E	8B8A
DACF	68	11	01-58-47	5A4F	659B
DAD0	68	13	01-58-48	5A50	659F
DAD1	69	9	01-58-49	5A51	65AB
DAD2	69	18	01-58-50	5A52	65B7
DAD3	70	10	01-58-51	5A53	65C3
DAD4	70	10	01-58-52	5A54	65C6
DAD5	70	10	01-58-53	5A55	65C1
DAD6	70	10	01-58-54	5A56	65C4
DAD7	70	11	01-58-55	5A57	65CC
DAD8	70	13	01-58-56	5A58	65D2
DAD9	70	18	01-58-57	5A59	65DB
DADA	70	16	01-58-58	5A5A	65D9
DADB	71	4	01-58-59	5A5B	65E0
DADC	71	4	01-58-60	5A5C	65E1
DADD	72	7	01-58-61	5A5D	65F1
DADE	72	8	01-58-62	5A5E	6772
DADF	72	8	01-58-63	5A5F	660A
DAE0	72	8	01-58-64	5A60	6603
DAE1	72	8	01-58-65	5A61	65FB
DAE2	72	8	01-58-66	5A62	6773
DAE3	72	9	01-58-67	5A63	6635
DAE4	72	9	01-58-68	5A64	6636
DAE5	72	9	01-58-69	5A65	6634
DAE6	72	9	01-58-70	5A66	661C
DAE7	72	10	01-58-71	5A67	664F
DAE8	72	10	01-58-72	5A68	6644
DAE9	72	10	01-58-73	5A69	6649
DAEA	72	10	01-58-74	5A6A	6641
DAEB	72	11	01-58-75	5A6B	665E
DAEC	72	11	01-58-76	5A6C	665D
DAED	72	11	01-58-77	5A6D	6664
DAEE	72	11	01-58-78	5A6E	6667
DAEF	72	11	01-58-79	5A6F	6668
DAF0	72	11	01-58-80	5A70	665F
DAF1	72	11	01-58-81	5A71	6662
DAF2	72	12	01-58-82	5A72	6670
DAF3	72	12	01-58-83	5A73	6683
DAF4	72	13	01-58-84	5A74	6688
DAF5	72	12	01-58-85	5A75	668E
DAF6	72	13	01-58-86	5A76	6689
DAF7	72	13	01-58-87	5A77	6684
DAF8	72	13	01-58-88	5A78	6698
DAF9	72	14	01-58-89	5A79	669D
DAFA	72	16	01-58-90	5A7A	66C1
DAFB	72	16	01-58-91	5A7B	66B9
DAFC	72	16	01-58-92	5A7C	66C9
DAFD	72	16	01-58-93	5A7D	66BE
DAFE	72	16	01-58-94	5A7E	66BC
DBA1	72	14	01-59-01	5B21	66C4
DBA2	72	16	01-59-02	5B22	66B8
DBA3	72	17	01-59-03	5B23	66D6
DBA4	72	17	01-59-04	5B24	66DA
DBA5	72	19	01-59-05	5B25	66E0
DBA6	72	9	01-59-06	5B26	663F
DBA7	72	20	01-59-07	5B27	66E6
DBA8	72	21	01-59-08	5B28	66E9
DBA9	73	4	01-59-09	5B29	66F0
DBAA	73	7	01-59-10	5B2A	66F5
DBAB	73	9	01-59-11	5B2B	66F7
DBAC	74	9	01-59-12	5B2C	670F
DBAD	74	11	01-59-13	5B2D	6716
DBAE	74	12	01-59-14	5B2E	671E
DBAF	74	17	01-59-15	5B2F	6726
DBB0	74	20	01-59-16	5B30	6727
DBB1	74	21	01-59-17	5B31	9738
DBB2	75	5	01-59-18	5B32	672E
DBB3	75	6	01-59-19	5B33	673F
DBB4	75	6	01-59-20	5B34	6736
DBB5	75	6	01-59-21	5B35	6741
DBB6	75	6	01-59-22	5B36	6738
DBB7	75	6	01-59-23	5B37	6737
DBB8	75	7	01-59-24	5B38	6746
DBB9	75	7	01-59-25	5B39	675E
DBBA	75	7	01-59-26	5B3A	6760
DBBB	75	7	01-59-27	5B3B	6759
DBBC	75	7	01-59-28	5B3C	6763
DBBD	75	7	01-59-29	5B3D	6764
DBBE	75	8	01-59-30	5B3E	6789
DBBF	75	8	01-59-31	5B3F	6770
DBC0	75	8	01-59-32	5B40	67A9
DBC1	75	8	01-59-33	5B41	677C
DBC2	75	8	01-59-34	5B42	676A
DBC3	75	8	01-59-35	5B43	678C
DBC4	75	8	01-59-36	5B44	678B
DBC5	75	8	01-59-37	5B45	67A6
DBC6	75	8	01-59-38	5B46	67A1
DBC7	75	8	01-59-39	5B47	6785
DBC8	75	9	01-59-40	5B48	67B7
DBC9	75	9	01-59-41	5B49	67EF
DBCA	75	9	01-59-42	5B4A	67B4
DBCB	75	9	01-59-43	5B4B	67EC
DBCC	75	9	01-59-44	5B4C	67B3
DBCD	75	9	01-59-45	5B4D	67E9
DBCE	75	9	01-59-46	5B4E	67B8
DBCF	75	9	01-59-47	5B4F	67E4
DBD0	75	9	01-59-48	5B50	67DE
DBD1	75	9	01-59-49	5B51	67DD
DBD2	75	9	01-59-50	5B52	67E2
DBD3	75	9	01-59-51	5B53	67EE
DBD4	75	9	01-59-52	5B54	67B9
DBD5	75	9	01-59-53	5B55	67CE
DBD6	75	9	01-59-54	5B56	67C6
DBD7	75	9	01-59-55	5B57	67E7
DBD8	75	17	01-59-56	5B58	6A9C
DBD9	75	10	01-59-57	5B59	681E
DBDA	75	10	01-59-58	5B5A	6846
DBDB	75	10	01-59-59	5B5B	6829
DBDC	75	10	01-59-60	5B5C	6840
DBDD	75	10	01-59-61	5B5D	684D
DBDE	75	10	01-59-62	5B5E	6832
DBDF	75	10	01-59-63	5B5F	684E
DBE0	75	11	01-59-64	5B60	68B3
DBE1	75	10	01-59-65	5B61	682B
DBE2	75	10	01-59-66	5B62	6859
DBE3	75	10	01-59-67	5B63	6863
DBE4	75	11	01-59-68	5B64	6877
DBE5	75	11	01-59-69	5B65	687F
DBE6	75	11	01-59-70	5B66	689F
DBE7	75	11	01-59-71	5B67	688F
DBE8	75	11	01-59-72	5B68	68AD
DBE9	75	11	01-59-73	5B69	6894
DBEA	75	11	01-59-74	5B6A	689D
DBEB	75	11	01-59-75	5B6B	689B
DBEC	75	11	01-59-76	5B6C	6883
DBED	75	18	01-59-77	5B6D	6AAE
DBEE	75	11	01-59-78	5B6E	68B9
DBEF	75	11	01-59-79	5B6F	6874
DBF0	75	11	01-59-80	5B70	68B5
DBF1	75	11	01-59-81	5B71	68A0
DBF2	75	11	01-59-82	5B72	68BA
DBF3	75	12	01-59-83	5B73	690F
DBF4	75	11	01-59-84	5B74	688D
DBF5	75	11	01-59-85	5B75	687E
DBF6	75	12	01-59-86	5B76	6901
DBF7	75	12	01-59-87	5B77	68CA
DBF8	75	12	01-59-88	5B78	6908
DBF9	75	12	01-59-89	5B79	68D8
DBFA	75	12	01-59-90	5B7A	6922
DBFB	75	12	01-59-91	5B7B	6926
DBFC	75	12	01-59-92	5B7C	68E1
DBFD	75	12	01-59-93	5B7D	690C
DBFE	75	12	01-59-94	5B7E	68CD
DCA1	75	12	01-60-01	5C21	68D4
DCA2	75	12	01-60-02	5C22	68E7
DCA3	75	12	01-60-03	5C23	68D5
DCA4	75	13	01-60-04	5C24	6936
DCA5	75	12	01-60-05	5C25	6912
DCA6	75	12	01-60-06	5C26	6904
DCA7	75	12	01-60-07	5C27	68D7
DCA8	75	12	01-60-08	5C28	68E3
DCA9	75	12	01-60-09	5C29	6925
DCAA	75	12	01-60-10	5C2A	68F9
DCAB	75	12	01-60-11	5C2B	68E0
DCAC	75	12	01-60-12	5C2C	68EF
DCAD	75	12	01-60-13	5C2D	6928
DCAE	75	12	01-60-14	5C2E	692A
DCAF	75	12	01-60-15	5C2F	691A
DCB0	75	12	01-60-16	5C30	6923
DCB1	75	12	01-60-17	5C31	6921
DCB2	75	12	01-60-18	5C32	68C6
DCB3	75	13	01-60-19	5C33	6979
DCB4	75	13	01-60-20	5C34	6977
DCB5	75	13	01-60-21	5C35	695C
DCB6	75	13	01-60-22	5C36	6978
DCB7	75	13	01-60-23	5C37	696B
DCB8	75	13	01-60-24	5C38	6954
DCB9	75	13	01-60-25	5C39	697E
DCBA	75	13	01-60-26	5C3A	696E
DCBB	75	13	01-60-27	5C3B	6939
DCBC	75	13	01-60-28	5C3C	6974
DCBD	75	13	01-60-29	5C3D	693D
DCBE	75	13	01-60-30	5C3E	6959
DCBF	75	13	01-60-31	5C3F	6930
DCC0	75	13	01-60-32	5C40	6961
DCC1	75	13	01-60-33	5C41	695E
DCC2	75	13	01-60-34	5C42	695D
DCC3	75	13	01-60-35	5C43	6981
DCC4	75	13	01-60-36	5C44	696A
DCC5	75	14	01-60-37	5C45	69B2
DCC6	75	14	01-60-38	5C46	69AE
DCC7	75	14	01-60-39	5C47	69D0
DCC8	75	14	01-60-40	5C48	69BF
DCC9	75	14	01-60-41	5C49	69C1
DCCA	75	14	01-60-42	5C4A	69D3
DCCB	75	14	01-60-43	5C4B	69BE
DCCC	75	14	01-60-44	5C4C	69CE
DCCD	75	14	01-60-45	5C4D	5BE8
DCCE	75	14	01-60-46	5C4E	69CA
DCCF	75	14	01-60-47	5C4F	69DD
DCD0	75	14	01-60-48	5C50	69BB
DCD1	75	14	01-60-49	5C51	69C3
DCD2	75	14	01-60-50	5C52	69A7
DCD3	75	14	01-60-51	5C53	6A2E
DCD4	75	14	01-60-52	5C54	6991
DCD5	75	14	01-60-53	5C55	69A0
DCD6	75	14	01-60-54	5C56	699C
DCD7	75	14	01-60-55	5C57	6995
DCD8	75	14	01-60-56	5C58	69B4
DCD9	75	14	01-60-57	5C59	69DE
DCDA	75	15	01-60-58	5C5A	69E8
DCDB	75	15	01-60-59	5C5B	6A02
DCDC	75	15	01-60-60	5C5C	6A1B
DCDD	75	15	01-60-61	5C5D	69FF
DCDE	75	21	01-60-62	5C5E	6B0A
DCDF	75	15	01-60-63	5C5F	69F9
DCE0	75	15	01-60-64	5C60	69F2
DCE1	75	15	01-60-65	5C61	69E7
DCE2	75	15	01-60-66	5C62	6A05
DCE3	75	14	01-60-67	5C63	69B1
DCE4	75	15	01-60-68	5C64	6A1E
DCE5	75	15	01-60-69	5C65	69ED
DCE6	75	15	01-60-70	5C66	6A14
DCE7	75	15	01-60-71	5C67	69EB
DCE8	75	15	01-60-72	5C68	6A0A
DCE9	75	15	01-60-73	5C69	6A12
DCEA	75	18	01-60-74	5C6A	6AC1
DCEB	75	15	01-60-75	5C6B	6A23
DCEC	75	15	01-60-76	5C6C	6A13
DCED	75	16	01-60-77	5C6D	6A44
DCEE	75	15	01-60-78	5C6E	6A0C
DCEF	75	16	01-60-79	5C6F	6A72
DCF0	75	16	01-60-80	5C70	6A36
DCF1	75	16	01-60-81	5C71	6A78
DCF2	75	16	01-60-82	5C72	6A47
DCF3	75	16	01-60-83	5C73	6A62
DCF4	75	16	01-60-84	5C74	6A59
DCF5	75	16	01-60-85	5C75	6A66
DCF6	75	16	01-60-86	5C76	6A48
DCF7	75	16	01-60-87	5C77	6A38
DCF8	75	15	01-60-88	5C78	6A22
DCF9	75	17	01-60-89	5C79	6A90
DCFA	75	17	01-60-90	5C7A	6A8D
DCFB	75	16	01-60-91	5C7B	6AA0
DCFC	75	17	01-60-92	5C7C	6A84
DCFD	75	17	01-60-93	5C7D	6AA2
DCFE	75	17	01-60-94	5C7E	6AA3
DDA1	75	17	01-61-01	5D21	6A97
DDA2	140	20	01-61-02	5D22	8617
DDA3	75	19	01-61-03	5D23	6ABB
DDA4	75	18	01-61-04	5D24	6AC3
DDA5	75	18	01-61-05	5D25	6AC2
DDA6	75	18	01-61-06	5D26	6AB8
DDA7	75	18	01-61-07	5D27	6AB3
DDA8	75	17	01-61-08	5D28	6AAC
DDA9	75	19	01-61-09	5D29	6ADE
DDAA	75	19	01-61-10	5D2A	6AD1
DDAB	75	19	01-61-11	5D2B	6ADF
DDAC	75	17	01-61-12	5D2C	6AAA
DDAD	75	19	01-61-13	5D2D	6ADA
DDAE	75	20	01-61-14	5D2E	6AEA
DDAF	75	21	01-61-15	5D2F	6AFB
DDB0	75	22	01-61-16	5D30	6B05
DDB1	75	20	01-61-17	5D31	8616
DDB2	75	21	01-61-18	5D32	6AFA
DDB3	75	23	01-61-19	5D33	6B12
DDB4	75	26	01-61-20	5D34	6B16
DDB5	75	29	01-61-21	5D35	9B31
DDB6	75	28	01-61-22	5D36	6B1F
DDB7	76	11	01-61-23	5D37	6B38
DDB8	76	11	01-61-24	5D38	6B37
DDB9	76	12	01-61-25	5D39	76DC
DDBA	76	12	01-61-26	5D3A	6B39
DDBB	76	13	01-61-27	5D3B	98EE
DDBC	76	13	01-61-28	5D3C	6B47
DDBD	76	13	01-61-29	5D3D	6B43
DDBE	76	14	01-61-30	5D3E	6B49
DDBF	76	15	01-61-31	5D3F	6B50
DDC0	76	16	01-61-32	5D40	6B59
DDC1	76	16	01-61-33	5D41	6B54
DDC2	76	17	01-61-34	5D42	6B5B
DDC3	76	18	01-61-35	5D43	6B5F
DDC4	76	21	01-61-36	5D44	6B61
DDC5	77	18	01-61-37	5D45	6B78
DDC6	78	4	01-61-38	5D46	6B79
DDC7	78	8	01-61-39	5D47	6B7F
DDC8	78	8	01-61-40	5D48	6B80
DDC9	78	9	01-61-41	5D49	6B84
DDCA	78	9	01-61-42	5D4A	6B83
DDCB	78	11	01-61-43	5D4B	6B8D
DDCC	78	12	01-61-44	5D4C	6B98
DDCD	78	12	01-61-45	5D4D	6B95
DDCE	78	14	01-61-46	5D4E	6B9E
DDCF	78	15	01-61-47	5D4F	6BA4
DDD0	78	16	01-61-48	5D50	6BAA
DDD1	78	16	01-61-49	5D51	6BAB
DDD2	78	18	01-61-50	5D52	6BAF
DDD3	78	21	01-61-51	5D53	6BB2
DDD4	78	19	01-61-52	5D54	6BB1
DDD5	79	4	01-61-53	5D55	6BB3
DDD6	79	10	01-61-54	5D56	6BB7
DDD7	79	12	01-61-55	5D57	6BBC
DDD8	79	15	01-61-56	5D58	6BC6
DDD9	80	4	01-61-57	5D59	6BCB
DDDA	80	14	01-61-58	5D5A	6BD3
DDDB	82	8	01-61-59	5D5B	6BDF
DDDC	82	11	01-61-60	5D5C	6BEC
DDDD	82	11	01-61-61	5D5D	6BEB
DDDE	82	12	01-61-62	5D5E	6BF3
DDDF	82	12	01-61-63	5D5F	6BEF
DDE0	82	15	01-61-64	5D60	9EBE
DDE1	82	17	01-61-65	5D61	6C08
DDE2	83	8	01-61-66	5D62	6C13
DDE3	84	4	01-61-67	5D63	6C14
DDE4	84	8	01-61-68	5D64	6C1B
DDE5	84	10	01-61-69	5D65	6C24
DDE6	84	10	01-61-70	5D66	6C23
DDE7	85	7	01-61-71	5D67	6C5E
DDE8	85	6	01-61-72	5D68	6C55
DDE9	85	6	01-61-73	5D69	6C62
DDEA	85	7	01-61-74	5D6A	6C6A
DDEB	85	7	01-61-75	5D6B	6C82
DDEC	85	7	01-61-76	5D6C	6C8D
DDED	85	7	01-61-77	5D6D	6C9A
DDEE	85	7	01-61-78	5D6E	6C81
DDEF	85	7	01-61-79	5D6F	6C9B
DDF0	85	7	01-61-80	5D70	6C7E
DDF1	85	7	01-61-81	5D71	6C68
DDF2	85	7	01-61-82	5D72	6C73
DDF3	85	7	01-61-83	5D73	6C92
DDF4	85	7	01-61-84	5D74	6C90
DDF5	85	8	01-61-85	5D75	6CC4
DDF6	85	8	01-61-86	5D76	6CF1
DDF7	85	8	01-61-87	5D77	6CD3
DDF8	85	8	01-61-88	5D78	6CBD
DDF9	85	8	01-61-89	5D79	6CD7
DDFA	85	8	01-61-90	5D7A	6CC5
DDFB	85	8	01-61-91	5D7B	6CDD
DDFC	85	8	01-61-92	5D7C	6CAE
DDFD	85	8	01-61-93	5D7D	6CB1
DDFE	85	8	01-61-94	5D7E	6CBE
DEA1	85	8	01-62-01	5E21	6CBA
DEA2	85	7	01-62-02	5E22	6CDB
DEA3	85	8	01-62-03	5E23	6CEF
DEA4	85	8	01-62-04	5E24	6CD9
DEA5	85	8	01-62-05	5E25	6CEA
DEA6	85	9	01-62-06	5E26	6D1F
DEA7	85	9	01-62-07	5E27	884D
DEA8	85	9	01-62-08	5E28	6D36
DEA9	85	9	01-62-09	5E29	6D2B
DEAA	85	9	01-62-10	5E2A	6D3D
DEAB	85	9	01-62-11	5E2B	6D38
DEAC	85	9	01-62-12	5E2C	6D19
DEAD	85	9	01-62-13	5E2D	6D35
DEAE	85	9	01-62-14	5E2E	6D33
DEAF	85	9	01-62-15	5E2F	6D12
DEB0	85	9	01-62-16	5E30	6D0C
DEB1	85	10	01-62-17	5E31	6D63
DEB2	85	10	01-62-18	5E32	6D93
DEB3	85	10	01-62-19	5E33	6D64
DEB4	85	10	01-62-20	5E34	6D5A
DEB5	85	10	01-62-21	5E35	6D79
DEB6	85	10	01-62-22	5E36	6D59
DEB7	85	10	01-62-23	5E37	6D8E
DEB8	85	10	01-62-24	5E38	6D95
DEB9	85	17	01-62-25	5E39	6FE4
DEBA	85	10	01-62-26	5E3A	6D85
DEBB	85	11	01-62-27	5E3B	6DF9
DEBC	85	11	01-62-28	5E3C	6E15
DEBD	85	11	01-62-29	5E3D	6E0A
DEBE	85	11	01-62-30	5E3E	6DB5
DEBF	85	11	01-62-31	5E3F	6DC7
DEC0	85	11	01-62-32	5E40	6DE6
DEC1	85	11	01-62-33	5E41	6DB8
DEC2	85	11	01-62-34	5E42	6DC6
DEC3	85	11	01-62-35	5E43	6DEC
DEC4	85	11	01-62-36	5E44	6DDE
DEC5	85	11	01-62-37	5E45	6DCC
DEC6	85	11	01-62-38	5E46	6DE8
DEC7	85	11	01-62-39	5E47	6DD2
DEC8	85	11	01-62-40	5E48	6DC5
DEC9	85	11	01-62-41	5E49	6DFA
DECA	85	11	01-62-42	5E4A	6DD9
DECB	85	11	01-62-43	5E4B	6DE4
DECC	85	11	01-62-44	5E4C	6DD5
DECD	85	11	01-62-45	5E4D	6DEA
DECE	85	11	01-62-46	5E4E	6DEE
DECF	85	12	01-62-47	5E4F	6E2D
DED0	85	12	01-62-48	5E50	6E6E
DED1	85	11	01-62-49	5E51	6E2E
DED2	85	12	01-62-50	5E52	6E19
DED3	85	12	01-62-51	5E53	6E72
DED4	85	12	01-62-52	5E54	6E5F
DED5	85	12	01-62-53	5E55	6E3E
DED6	85	12	01-62-54	5E56	6E23
DED7	85	12	01-62-55	5E57	6E6B
DED8	85	12	01-62-56	5E58	6E2B
DED9	85	12	01-62-57	5E59	6E76
DEDA	85	12	01-62-58	5E5A	6E4D
DEDB	85	12	01-62-59	5E5B	6E1F
DEDC	85	12	01-62-60	5E5C	6E43
DEDD	85	12	01-62-61	5E5D	6E3A
DEDE	85	12	01-62-62	5E5E	6E4E
DEDF	85	12	01-62-63	5E5F	6E24
DEE0	85	14	01-62-64	5E60	6EFF
DEE1	85	12	01-62-65	5E61	6E1D
DEE2	85	12	01-62-66	5E62	6E38
DEE3	85	12	01-62-67	5E63	6E82
DEE4	85	13	01-62-68	5E64	6EAA
DEE5	85	13	01-62-69	5E65	6E98
DEE6	85	13	01-62-70	5E66	6EC9
DEE7	85	13	01-62-71	5E67	6EB7
DEE8	85	13	01-62-72	5E68	6ED3
DEE9	85	13	01-62-73	5E69	6EBD
DEEA	85	13	01-62-74	5E6A	6EAF
DEEB	85	13	01-62-75	5E6B	6EC4
DEEC	85	13	01-62-76	5E6C	6EB2
DEED	85	13	01-62-77	5E6D	6ED4
DEEE	85	15	01-62-78	5E6E	6ED5
DEEF	85	13	01-62-79	5E6F	6E8F
DEF0	85	13	01-62-80	5E70	6EA5
DEF1	85	13	01-62-81	5E71	6EC2
DEF2	85	13	01-62-82	5E72	6E9F
DEF3	85	15	01-62-83	5E73	6F41
DEF4	85	14	01-62-84	5E74	6F11
DEF5	85	20	01-62-85	5E75	704C
DEF6	85	14	01-62-86	5E76	6EEC
DEF7	85	14	01-62-87	5E77	6EF8
DEF8	85	14	01-62-88	5E78	6EFE
DEF9	85	15	01-62-89	5E79	6F3F
DEFA	85	14	01-62-90	5E7A	6EF2
DEFB	85	14	01-62-91	5E7B	6F31
DEFC	85	14	01-62-92	5E7C	6EEF
DEFD	85	14	01-62-93	5E7D	6F32
DEFE	85	14	01-62-94	5E7E	6ECC
DFA1	85	14	01-63-01	5F21	6F3E
DFA2	85	14	01-63-02	5F22	6F13
DFA3	85	14	01-63-03	5F23	6EF7
DFA4	85	15	01-63-04	5F24	6F86
DFA5	85	15	01-63-05	5F25	6F7A
DFA6	85	15	01-63-06	5F26	6F78
DFA7	85	15	01-63-07	5F27	6F81
DFA8	85	17	01-63-08	5F28	6F80
DFA9	85	15	01-63-09	5F29	6F6F
DFAA	85	15	01-63-10	5F2A	6F5B
DFAB	85	19	01-63-11	5F2B	6FF3
DFAC	85	15	01-63-12	5F2C	6F6D
DFAD	85	15	01-63-13	5F2D	6F82
DFAE	85	15	01-63-14	5F2E	6F7C
DFAF	85	15	01-63-15	5F2F	6F58
DFB0	85	15	01-63-16	5F30	6F8E
DFB1	85	15	01-63-17	5F31	6F91
DFB2	85	16	01-63-18	5F32	6FC2
DFB3	85	15	01-63-19	5F33	6F66
DFB4	85	16	01-63-20	5F34	6FB3
DFB5	85	16	01-63-21	5F35	6FA3
DFB6	85	16	01-63-22	5F36	6FA1
DFB7	85	16	01-63-23	5F37	6FA4
DFB8	85	16	01-63-24	5F38	6FB9
DFB9	85	15	01-63-25	5F39	6FC6
DFBA	85	16	01-63-26	5F3A	6FAA
DFBB	85	17	01-63-27	5F3B	6FDF
DFBC	85	17	01-63-28	5F3C	6FD5
DFBD	85	17	01-63-29	5F3D	6FEC
DFBE	85	17	01-63-30	5F3E	6FD4
DFBF	85	17	01-63-31	5F3F	6FD8
DFC0	85	17	01-63-32	5F40	6FF1
DFC1	85	17	01-63-33	5F41	6FEE
DFC2	85	16	01-63-34	5F42	6FDB
DFC3	85	18	01-63-35	5F43	7009
DFC4	85	18	01-63-36	5F44	700B
DFC5	85	18	01-63-37	5F45	6FFA
DFC6	85	18	01-63-38	5F46	7011
DFC7	85	18	01-63-39	5F47	7001
DFC8	85	18	01-63-40	5F48	700F
DFC9	85	18	01-63-41	5F49	6FFE
DFCA	85	19	01-63-42	5F4A	701B
DFCB	85	19	01-63-43	5F4B	701A
DFCC	85	15	01-63-44	5F4C	6F74
DFCD	85	19	01-63-45	5F4D	701D
DFCE	85	19	01-63-46	5F4E	7018
DFCF	85	18	01-63-47	5F4F	701F
DFD0	85	20	01-63-48	5F50	7030
DFD1	85	20	01-63-49	5F51	703E
DFD2	85	20	01-63-50	5F52	7032
DFD3	85	22	01-63-51	5F53	7051
DFD4	85	25	01-63-52	5F54	7063
DFD5	86	8	01-63-53	5F55	7099
DFD6	86	8	01-63-54	5F56	7092
DFD7	86	9	01-63-55	5F57	70AF
DFD8	86	11	01-63-56	5F58	70F1
DFD9	86	9	01-63-57	5F59	70AC
DFDA	86	9	01-63-58	5F5A	70B8
DFDB	86	9	01-63-59	5F5B	70B3
DFDC	86	9	01-63-60	5F5C	70AE
DFDD	86	10	01-63-61	5F5D	70DF
DFDE	86	10	01-63-62	5F5E	70CB
DFDF	86	10	01-63-63	5F5F	70DD
DFE0	86	10	01-63-64	5F60	70D9
DFE1	86	11	01-63-65	5F61	7109
DFE2	86	11	01-63-66	5F62	70FD
DFE3	86	12	01-63-67	5F63	711C
DFE4	86	12	01-63-68	5F64	7119
DFE5	86	13	01-63-69	5F65	7165
DFE6	86	13	01-63-70	5F66	7155
DFE7	86	15	01-63-71	5F67	7188
DFE8	86	13	01-63-72	5F68	7166
DFE9	86	13	01-63-73	5F69	7162
DFEA	86	13	01-63-74	5F6A	714C
DFEB	86	13	01-63-75	5F6B	7156
DFEC	86	13	01-63-76	5F6C	716C
DFED	86	14	01-63-77	5F6D	718F
DFEE	86	18	01-63-78	5F6E	71FB
DFEF	86	14	01-63-79	5F6F	7184
DFF0	86	14	01-63-80	5F70	7195
DFF1	86	15	01-63-81	5F71	71A8
DFF2	86	15	01-63-82	5F72	71AC
DFF3	86	16	01-63-83	5F73	71D7
DFF4	86	16	01-63-84	5F74	71B9
DFF5	86	16	01-63-85	5F75	71BE
DFF6	86	16	01-63-86	5F76	71D2
DFF7	86	16	01-63-87	5F77	71C9
DFF8	86	16	01-63-88	5F78	71D4
DFF9	86	16	01-63-89	5F79	71CE
DFFA	86	17	01-63-90	5F7A	71E0
DFFB	86	17	01-63-91	5F7B	71EC
DFFC	86	17	01-63-92	5F7C	71E7
DFFD	86	17	01-63-93	5F7D	71F5
DFFE	86	18	01-63-94	5F7E	71FC
E0A1	86	18	01-64-01	6021	71F9
E0A2	86	18	01-64-02	6022	71FF
E0A3	86	19	01-64-03	6023	720D
E0A4	86	20	01-64-04	6024	7210
E0A5	86	21	01-64-05	6025	721B
E0A6	86	29	01-64-06	6026	7228
E0A7	87	8	01-64-07	6027	722D
E0A8	87	8	01-64-08	6028	722C
E0A9	87	9	01-64-09	6029	7230
E0AA	87	12	01-64-10	602A	7232
E0AB	89	4	01-64-11	602B	723B
E0AC	89	9	01-64-12	602C	723C
E0AD	90	4	01-64-13	602D	723F
E0AE	90	8	01-64-14	602E	7240
E0AF	90	17	01-64-15	602F	7246
E0B0	91	12	01-64-16	6030	724B
E0B1	91	19	01-64-17	6031	7258
E0B2	93	9	01-64-18	6032	7274
E0B3	93	11	01-64-19	6033	727E
E0B4	93	12	01-64-20	6034	7282
E0B5	93	11	01-64-21	6035	7281
E0B6	93	12	01-64-22	6036	7287
E0B7	93	14	01-64-23	6037	7292
E0B8	93	14	01-64-24	6038	7296
E0B9	93	19	01-64-25	6039	72A2
E0BA	93	20	01-64-26	603A	72A7
E0BB	94	7	01-64-27	603B	72B9
E0BC	94	6	01-64-28	603C	72B2
E0BD	94	7	01-64-29	603D	72C3
E0BE	94	7	01-64-30	603E	72C6
E0BF	94	7	01-64-31	603F	72C4
E0C0	94	8	01-64-32	6040	72CE
E0C1	94	8	01-64-33	6041	72D2
E0C2	94	9	01-64-34	6042	72E2
E0C3	94	9	01-64-35	6043	72E0
E0C4	94	9	01-64-36	6044	72E1
E0C5	94	10	01-64-37	6045	72F9
E0C6	94	10	01-64-38	6046	72F7
E0C7	94	10	01-64-39	6047	500F
E0C8	94	11	01-64-40	6048	7317
E0C9	94	11	01-64-41	6049	730A
E0CA	94	11	01-64-42	604A	731C
E0CB	94	11	01-64-43	604B	7316
E0CC	94	11	01-64-44	604C	731D
E0CD	94	12	01-64-45	604D	7334
E0CE	94	12	01-64-46	604E	732F
E0CF	94	12	01-64-47	604F	7329
E0D0	94	12	01-64-48	6050	7325
E0D1	94	13	01-64-49	6051	733E
E0D2	37	15	01-64-50	6052	734E
E0D3	94	13	01-64-51	6053	734F
E0D4	94	16	01-64-52	6054	9ED8
E0D5	94	15	01-64-53	6055	7357
E0D6	94	16	01-64-54	6056	736A
E0D7	94	16	01-64-55	6057	7368
E0D8	94	17	01-64-56	6058	7370
E0D9	94	19	01-64-57	6059	7378
E0DA	94	18	01-64-58	605A	7375
E0DB	94	20	01-64-59	605B	737B
E0DC	94	19	01-64-60	605C	737A
E0DD	96	9	01-64-61	605D	73C8
E0DE	96	9	01-64-62	605E	73B3
E0DF	96	9	01-64-63	605F	73CE
E0E0	96	9	01-64-64	6060	73BB
E0E1	96	9	01-64-65	6061	73C0
E0E2	96	10	01-64-66	6062	73E5
E0E3	96	10	01-64-67	6063	73EE
E0E4	96	10	01-64-68	6064	73DE
E0E5	96	16	01-64-69	6065	74A2
E0E6	96	11	01-64-70	6066	7405
E0E7	96	14	01-64-71	6067	746F
E0E8	96	12	01-64-72	6068	7425
E0E9	96	11	01-64-73	6069	73F8
E0EA	96	12	01-64-74	606A	7432
E0EB	96	12	01-64-75	606B	743A
E0EC	96	13	01-64-76	606C	7455
E0ED	96	13	01-64-77	606D	743F
E0EE	96	13	01-64-78	606E	745F
E0EF	96	13	01-64-79	606F	7459
E0F0	96	13	01-64-80	6070	7441
E0F1	96	13	01-64-81	6071	745C
E0F2	96	15	01-64-82	6072	7469
E0F3	96	14	01-64-83	6073	7470
E0F4	96	14	01-64-84	6074	7463
E0F5	96	14	01-64-85	6075	746A
E0F6	96	13	01-64-86	6076	7476
E0F7	96	15	01-64-87	6077	747E
E0F8	96	15	01-64-88	6078	748B
E0F9	96	16	01-64-89	6079	749E
E0FA	96	18	01-64-90	607A	74A7
E0FB	96	19	01-64-91	607B	74CA
E0FC	96	20	01-64-92	607C	74CF
E0FD	96	21	01-64-93	607D	74D4
E0FE	96	10	01-64-94	607E	73F1
E1A1	97	11	01-65-01	6121	74E0
E1A2	97	19	01-65-02	6122	74E3
E1A3	98	7	01-65-03	6123	74E7
E1A4	98	8	01-65-04	6124	74E9
E1A5	98	9	01-65-05	6125	74EE
E1A6	98	9	01-65-06	6126	74F2
E1A7	98	9	01-65-07	6127	74F0
E1A8	98	9	01-65-08	6128	74F1
E1A9	98	11	01-65-09	6129	74F8
E1AA	98	11	01-65-10	612A	74F7
E1AB	98	14	01-65-11	612B	7504
E1AC	98	14	01-65-12	612C	7503
E1AD	98	14	01-65-13	612D	7505
E1AE	98	16	01-65-14	612E	750C
E1AF	98	16	01-65-15	612F	750E
E1B0	98	15	01-65-16	6130	750D
E1B1	98	18	01-65-17	6131	7515
E1B2	98	18	01-65-18	6132	7513
E1B3	30	13	01-65-19	6133	751E
E1B4	100	12	01-65-20	6134	7526
E1B5	101	7	01-65-21	6135	752C
E1B6	102	7	01-65-22	6136	753C
E1B7	102	8	01-65-23	6137	7544
E1B8	102	9	01-65-24	6138	754D
E1B9	127	9	01-65-25	6139	754A
E1BA	102	9	01-65-26	613A	7549
E1BB	102	10	01-65-27	613B	755B
E1BC	102	9	01-65-28	613C	7546
E1BD	102	10	01-65-29	613D	755A
E1BE	102	11	01-65-30	613E	7569
E1BF	102	11	01-65-31	613F	7564
E1C0	102	11	01-65-32	6140	7567
E1C1	102	12	01-65-33	6141	756B
E1C2	102	12	01-65-34	6142	756D
E1C3	102	13	01-65-35	6143	7578
E1C4	102	13	01-65-36	6144	7576
E1C5	102	19	01-65-37	6145	7586
E1C6	102	19	01-65-38	6146	7587
E1C7	102	12	01-65-39	6147	7574
E1C8	102	22	01-65-40	6148	758A
E1C9	102	22	01-65-41	6149	7589
E1CA	102	16	01-65-42	614A	7582
E1CB	104	7	01-65-43	614B	7594
E1CC	104	8	01-65-44	614C	759A
E1CD	104	8	01-65-45	614D	759D
E1CE	104	9	01-65-46	614E	75A5
E1CF	104	9	01-65-47	614F	75A3
E1D0	104	10	01-65-48	6150	75C2
E1D1	104	10	01-65-49	6151	75B3
E1D2	104	10	01-65-50	6152	75C3
E1D3	104	11	01-65-51	6153	75B5
E1D4	104	10	01-65-52	6154	75BD
E1D5	104	10	01-65-53	6155	75B8
E1D6	104	10	01-65-54	6156	75BC
E1D7	104	10	01-65-55	6157	75B1
E1D8	104	11	01-65-56	6158	75CD
E1D9	104	11	01-65-57	6159	75CA
E1DA	104	11	01-65-58	615A	75D2
E1DB	104	12	01-65-59	615B	75D9
E1DC	104	12	01-65-60	615C	75E3
E1DD	104	12	01-65-61	615D	75DE
E1DE	104	13	01-65-62	615E	75FE
E1DF	104	13	01-65-63	615F	75FF
E1E0	104	13	01-65-64	6160	75FC
E1E1	104	13	01-65-65	6161	7601
E1E2	104	13	01-65-66	6162	75F0
E1E3	104	13	01-65-67	6163	75FA
E1E4	104	13	01-65-68	6164	75F2
E1E5	104	13	01-65-69	6165	75F3
E1E6	104	14	01-65-70	6166	760B
E1E7	104	14	01-65-71	6167	760D
E1E8	104	14	01-65-72	6168	7609
E1E9	104	15	01-65-73	6169	761F
E1EA	104	15	01-65-74	616A	7627
E1EB	104	15	01-65-75	616B	7620
E1EC	104	15	01-65-76	616C	7621
E1ED	104	15	01-65-77	616D	7622
E1EE	104	15	01-65-78	616E	7624
E1EF	104	16	01-65-79	616F	7634
E1F0	104	16	01-65-80	6170	7630
E1F1	104	16	01-65-81	6171	763B
E1F2	104	17	01-65-82	6172	7647
E1F3	104	17	01-65-83	6173	7648
E1F4	104	17	01-65-84	6174	7646
E1F5	104	18	01-65-85	6175	765C
E1F6	104	17	01-65-86	6176	7658
E1F7	104	19	01-65-87	6177	7661
E1F8	104	20	01-65-88	6178	7662
E1F9	104	21	01-65-89	6179	7668
E1FA	104	21	01-65-90	617A	7669
E1FB	104	21	01-65-91	617B	766A
E1FC	104	21	01-65-92	617C	7667
E1FD	104	22	01-65-93	617D	766C
E1FE	104	23	01-65-94	617E	7670
E2A1	104	24	01-66-01	6221	7672
E2A2	105	5	01-66-02	6222	7676
E2A3	105	9	01-66-03	6223	7678
E2A4	105	12	01-66-04	6224	767C
E2A5	106	7	01-66-05	6225	7680
E2A6	106	7	01-66-06	6226	7683
E2A7	106	9	01-66-07	6227	7688
E2A8	106	10	01-66-08	6228	768B
E2A9	106	11	01-66-09	6229	768E
E2AA	106	12	01-66-10	622A	7696
E2AB	106	12	01-66-11	622B	7693
E2AC	106	13	01-66-12	622C	7699
E2AD	106	15	01-66-13	622D	769A
E2AE	107	10	01-66-14	622E	76B0
E2AF	107	12	01-66-15	622F	76B4
E2B0	107	14	01-66-16	6230	76B8
E2B1	107	14	01-66-17	6231	76B9
E2B2	107	15	01-66-18	6232	76BA
E2B3	108	8	01-66-19	6233	76C2
E2B4	108	10	01-66-20	6234	76CD
E2B5	140	11	01-66-21	6235	76D6
E2B6	108	11	01-66-22	6236	76D2
E2B7	108	13	01-66-23	6237	76DE
E2B8	108	14	01-66-24	6238	76E1
E2B9	108	16	01-66-25	6239	76E5
E2BA	108	16	01-66-26	623A	76E7
E2BB	108	17	01-66-27	623B	76EA
E2BC	108	20	01-66-28	623C	862F
E2BD	109	9	01-66-29	623D	76FB
E2BE	109	9	01-66-30	623E	7708
E2BF	109	9	01-66-31	623F	7707
E2C0	109	9	01-66-32	6240	7704
E2C1	109	10	01-66-33	6241	7729
E2C2	109	10	01-66-34	6242	7724
E2C3	109	10	01-66-35	6243	771E
E2C4	109	11	01-66-36	6244	7725
E2C5	109	11	01-66-37	6245	7726
E2C6	109	10	01-66-38	6246	771B
E2C7	109	11	01-66-39	6247	7737
E2C8	109	11	01-66-40	6248	7738
E2C9	109	12	01-66-41	6249	7747
E2CA	109	13	01-66-42	624A	775A
E2CB	109	13	01-66-43	624B	7768
E2CC	109	13	01-66-44	624C	776B
E2CD	109	13	01-66-45	624D	775B
E2CE	109	13	01-66-46	624E	7765
E2CF	109	14	01-66-47	624F	777F
E2D0	109	14	01-66-48	6250	777E
E2D1	109	14	01-66-49	6251	7779
E2D2	109	15	01-66-50	6252	778E
E2D3	109	15	01-66-51	6253	778B
E2D4	109	15	01-66-52	6254	7791
E2D5	109	16	01-66-53	6255	77A0
E2D6	109	16	01-66-54	6256	779E
E2D7	109	17	01-66-55	6257	77B0
E2D8	109	17	01-66-56	6258	77B6
E2D9	109	18	01-66-57	6259	77B9
E2DA	109	18	01-66-58	625A	77BF
E2DB	109	18	01-66-59	625B	77BC
E2DC	109	18	01-66-60	625C	77BD
E2DD	109	18	01-66-61	625D	77BB
E2DE	109	18	01-66-62	625E	77C7
E2DF	109	20	01-66-63	625F	77CD
E2E0	109	24	01-66-64	6260	77D7
E2E1	109	26	01-66-65	6261	77DA
E2E2	110	9	01-66-66	6262	77DC
E2E3	111	7	01-66-67	6263	77E3
E2E4	111	13	01-66-68	6264	77EE
E2E5	112	8	01-66-69	6265	77FC
E2E6	112	9	01-66-70	6266	780C
E2E7	112	9	01-66-71	6267	7812
E2E8	112	20	01-66-72	6268	7926
E2E9	112	10	01-66-73	6269	7820
E2EA	112	19	01-66-74	626A	792A
E2EB	112	11	01-66-75	626B	7845
E2EC	112	13	01-66-76	626C	788E
E2ED	112	12	01-66-77	626D	7874
E2EE	112	13	01-66-78	626E	7886
E2EF	112	13	01-66-79	626F	787C
E2F0	112	13	01-66-80	6270	789A
E2F1	112	13	01-66-81	6271	788C
E2F2	112	14	01-66-82	6272	78A3
E2F3	112	14	01-66-83	6273	78B5
E2F4	112	14	01-66-84	6274	78AA
E2F5	112	14	01-66-85	6275	78AF
E2F6	112	15	01-66-86	6276	78D1
E2F7	112	15	01-66-87	6277	78C6
E2F8	112	15	01-66-88	6278	78CB
E2F9	112	15	01-66-89	6279	78D4
E2FA	112	15	01-66-90	627A	78BE
E2FB	112	15	01-66-91	627B	78BC
E2FC	112	15	01-66-92	627C	78C5
E2FD	112	15	01-66-93	627D	78CA
E2FE	112	16	01-66-94	627E	78EC
E3A1	112	16	01-67-01	6321	78E7
E3A2	112	16	01-67-02	6322	78DA
E3A3	112	17	01-67-03	6323	78FD
E3A4	112	17	01-67-04	6324	78F4
E3A5	112	18	01-67-05	6325	7907
E3A6	112	18	01-67-06	6326	7912
E3A7	112	18	01-67-07	6327	7911
E3A8	112	19	01-67-08	6328	7919
E3A9	112	20	01-67-09	6329	792C
E3AA	112	20	01-67-10	632A	792B
E3AB	113	8	01-67-11	632B	7940
E3AC	113	10	01-67-12	632C	7960
E3AD	113	10	01-67-13	632D	7957
E3AE	113	10	01-67-14	632E	795F
E3AF	113	10	01-67-15	632F	795A
E3B0	113	10	01-67-16	6330	7955
E3B1	113	10	01-67-17	6331	7953
E3B2	113	13	01-67-18	6332	797A
E3B3	113	13	01-67-19	6333	797F
E3B4	113	14	01-67-20	6334	798A
E3B5	113	15	01-67-21	6335	799D
E3B6	113	17	01-67-22	6336	79A7
E3B7	113	17	01-67-23	6337	9F4B
E3B8	113	17	01-67-24	6338	79AA
E3B9	113	18	01-67-25	6339	79AE
E3BA	113	22	01-67-26	633A	79B3
E3BB	114	9	01-67-27	633B	79B9
E3BC	114	9	01-67-28	633C	79BA
E3BD	115	8	01-67-29	633D	79C9
E3BE	115	9	01-67-30	633E	79D5
E3BF	115	10	01-67-31	633F	79E7
E3C0	115	10	01-67-32	6340	79EC
E3C1	115	10	01-67-33	6341	79E1
E3C2	115	10	01-67-34	6342	79E3
E3C3	115	12	01-67-35	6343	7A08
E3C4	115	12	01-67-36	6344	7A0D
E3C5	115	13	01-67-37	6345	7A18
E3C6	115	13	01-67-38	6346	7A19
E3C7	115	13	01-67-39	6347	7A20
E3C8	115	13	01-67-40	6348	7A1F
E3C9	113	13	01-67-41	6349	7980
E3CA	115	14	01-67-42	634A	7A31
E3CB	115	15	01-67-43	634B	7A3B
E3CC	115	15	01-67-44	634C	7A3E
E3CD	115	15	01-67-45	634D	7A37
E3CE	115	15	01-67-46	634E	7A43
E3CF	115	17	01-67-47	634F	7A57
E3D0	115	17	01-67-48	6350	7A49
E3D1	115	18	01-67-49	6351	7A61
E3D2	115	18	01-67-50	6352	7A62
E3D3	115	19	01-67-51	6353	7A69
E3D4	115	21	01-67-52	6354	9F9D
E3D5	115	22	01-67-53	6355	7A70
E3D6	116	8	01-67-54	6356	7A79
E3D7	116	9	01-67-55	6357	7A7D
E3D8	116	10	01-67-56	6358	7A88
E3D9	116	12	01-67-57	6359	7A97
E3DA	116	11	01-67-58	635A	7A95
E3DB	116	12	01-67-59	635B	7A98
E3DC	116	12	01-67-60	635C	7A96
E3DD	116	14	01-67-61	635D	7AA9
E3DE	116	21	01-67-62	635E	7AC8
E3DF	116	15	01-67-63	635F	7AB0
E3E0	116	16	01-67-64	6360	7AB6
E3E1	116	18	01-67-65	6361	7AC5
E3E2	116	18	01-67-66	6362	7AC4
E3E3	116	17	01-67-67	6363	7ABF
E3E4	116	18	01-67-68	6364	9083
E3E5	116	20	01-67-69	6365	7AC7
E3E6	116	22	01-67-70	6366	7ACA
E3E7	117	7	01-67-71	6367	7ACD
E3E8	117	8	01-67-72	6368	7ACF
E3E9	117	9	01-67-73	6369	7AD5
E3EA	117	9	01-67-74	636A	7AD3
E3EB	117	10	01-67-75	636B	7AD9
E3EC	117	10	01-67-76	636C	7ADA
E3ED	117	10	01-67-77	636D	7ADD
E3EE	117	11	01-67-78	636E	7AE1
E3EF	117	12	01-67-79	636F	7AE2
E3F0	117	12	01-67-80	6370	7AE6
E3F1	117	14	01-67-81	6371	7AED
E3F2	117	14	01-67-82	6372	7AF0
E3F3	118	9	01-67-83	6373	7B02
E3F4	118	10	01-67-84	6374	7B0F
E3F5	118	10	01-67-85	6375	7B0A
E3F6	118	10	01-67-86	6376	7B06
E3F7	118	11	01-67-87	6377	7B33
E3F8	118	11	01-67-88	6378	7B18
E3F9	118	11	01-67-89	6379	7B19
E3FA	118	11	01-67-90	637A	7B1E
E3FB	118	11	01-67-91	637B	7B35
E3FC	118	11	01-67-92	637C	7B28
E3FD	118	11	01-67-93	637D	7B36
E3FE	118	12	01-67-94	637E	7B50
E4A1	118	13	01-68-01	6421	7B7A
E4A2	118	10	01-68-02	6422	7B04
E4A3	118	12	01-68-03	6423	7B4D
E4A4	118	10	01-68-04	6424	7B0B
E4A5	118	12	01-68-05	6425	7B4C
E4A6	118	12	01-68-06	6426	7B45
E4A7	118	13	01-68-07	6427	7B75
E4A8	118	13	01-68-08	6428	7B65
E4A9	118	13	01-68-09	6429	7B74
E4AA	118	13	01-68-10	642A	7B67
E4AB	118	13	01-68-11	642B	7B70
E4AC	118	13	01-68-12	642C	7B71
E4AD	118	12	01-68-13	642D	7B6C
E4AE	118	13	01-68-14	642E	7B6E
E4AF	118	14	01-68-15	642F	7B9D
E4B0	118	14	01-68-16	6430	7B98
E4B1	118	14	01-68-17	6431	7B9F
E4B2	118	14	01-68-18	6432	7B8D
E4B3	118	14	01-68-19	6433	7B9C
E4B4	118	14	01-68-20	6434	7B9A
E4B5	118	14	01-68-21	6435	7B8B
E4B6	118	14	01-68-22	6436	7B92
E4B7	118	14	01-68-23	6437	7B8F
E4B8	118	12	01-68-24	6438	7B5D
E4B9	118	14	01-68-25	6439	7B99
E4BA	118	15	01-68-26	643A	7BCB
E4BB	118	15	01-68-27	643B	7BC1
E4BC	118	15	01-68-28	643C	7BCC
E4BD	118	15	01-68-29	643D	7BCF
E4BE	118	15	01-68-30	643E	7BB4
E4BF	118	15	01-68-31	643F	7BC6
E4C0	118	16	01-68-32	6440	7BDD
E4C1	118	16	01-68-33	6441	7BE9
E4C2	118	16	01-68-34	6442	7C11
E4C3	118	17	01-68-35	6443	7C14
E4C4	118	16	01-68-36	6444	7BE6
E4C5	118	16	01-68-37	6445	7BE5
E4C6	118	22	01-68-38	6446	7C60
E4C7	118	17	01-68-39	6447	7C00
E4C8	118	17	01-68-40	6448	7C07
E4C9	118	17	01-68-41	6449	7C13
E4CA	118	17	01-68-42	644A	7BF3
E4CB	118	17	01-68-43	644B	7BF7
E4CC	118	17	01-68-44	644C	7C17
E4CD	118	17	01-68-45	644D	7C0D
E4CE	118	17	01-68-46	644E	7BF6
E4CF	118	18	01-68-47	644F	7C23
E4D0	118	18	01-68-48	6450	7C27
E4D1	118	18	01-68-49	6451	7C2A
E4D2	118	18	01-68-50	6452	7C1F
E4D3	118	19	01-68-51	6453	7C37
E4D4	118	18	01-68-52	6454	7C2B
E4D5	118	19	01-68-53	6455	7C3D
E4D6	118	20	01-68-54	6456	7C4C
E4D7	118	21	01-68-55	6457	7C43
E4D8	118	21	01-68-56	6458	7C54
E4D9	118	20	01-68-57	6459	7C4F
E4DA	118	19	01-68-58	645A	7C40
E4DB	118	21	01-68-59	645B	7C50
E4DC	118	22	01-68-60	645C	7C58
E4DD	118	22	01-68-61	645D	7C5F
E4DE	118	23	01-68-62	645E	7C64
E4DF	118	21	01-68-63	645F	7C56
E4E0	214	23	01-68-64	6460	7C65
E4E1	118	25	01-68-65	6461	7C6C
E4E2	119	8	01-68-66	6462	7C75
E4E3	119	10	01-68-67	6463	7C83
E4E4	119	10	01-68-68	6464	7C90
E4E5	119	12	01-68-69	6465	7CA4
E4E6	119	12	01-68-70	6466	7CAD
E4E7	119	12	01-68-71	6467	7CA2
E4E8	119	12	01-68-72	6468	7CAB
E4E9	119	12	01-68-73	6469	7CA1
E4EA	119	12	01-68-74	646A	7CA8
E4EB	119	13	01-68-75	646B	7CB3
E4EC	119	13	01-68-76	646C	7CB2
E4ED	119	13	01-68-77	646D	7CB1
E4EE	119	13	01-68-78	646E	7CAE
E4EF	119	14	01-68-79	646F	7CB9
E4F0	119	14	01-68-80	6470	7CBD
E4F1	119	13	01-68-81	6471	7CC0
E4F2	119	15	01-68-82	6472	7CC5
E4F3	119	15	01-68-83	6473	7CC2
E4F4	119	16	01-68-84	6474	7CD8
E4F5	119	16	01-68-85	6475	7CD2
E4F6	119	17	01-68-86	6476	7CDC
E4F7	119	16	01-68-87	6477	7CE2
E4F8	119	22	01-68-88	6478	9B3B
E4F9	119	20	01-68-89	6479	7CEF
E4FA	119	20	01-68-90	647A	7CF2
E4FB	119	22	01-68-91	647B	7CF4
E4FC	119	25	01-68-92	647C	7CF6
E4FD	120	7	01-68-93	647D	7CFA
E4FE	120	9	01-68-94	647E	7D06
E5A1	120	9	01-69-01	6521	7D02
E5A2	120	10	01-69-02	6522	7D1C
E5A3	120	10	01-69-03	6523	7D15
E5A4	120	10	01-69-04	6524	7D0A
E5A5	120	11	01-69-05	6525	7D45
E5A6	120	11	01-69-06	6526	7D4B
E5A7	120	11	01-69-07	6527	7D2E
E5A8	120	11	01-69-08	6528	7D32
E5A9	120	11	01-69-09	6529	7D3F
E5AA	120	11	01-69-10	652A	7D35
E5AB	120	11	01-69-11	652B	7D46
E5AC	120	12	01-69-12	652C	7D73
E5AD	120	12	01-69-13	652D	7D56
E5AE	120	12	01-69-14	652E	7D4E
E5AF	120	12	01-69-15	652F	7D72
E5B0	120	12	01-69-16	6530	7D68
E5B1	120	12	01-69-17	6531	7D6E
E5B2	120	12	01-69-18	6532	7D4F
E5B3	120	12	01-69-19	6533	7D63
E5B4	120	13	01-69-20	6534	7D93
E5B5	120	13	01-69-21	6535	7D89
E5B6	120	13	01-69-22	6536	7D5B
E5B7	120	13	01-69-23	6537	7D8F
E5B8	120	13	01-69-24	6538	7D7D
E5B9	120	13	01-69-25	6539	7D9B
E5BA	120	14	01-69-26	653A	7DBA
E5BB	120	14	01-69-27	653B	7DAE
E5BC	120	14	01-69-28	653C	7DA3
E5BD	120	14	01-69-29	653D	7DB5
E5BE	120	14	01-69-30	653E	7DC7
E5BF	120	14	01-69-31	653F	7DBD
E5C0	120	14	01-69-32	6540	7DAB
E5C1	120	17	01-69-33	6541	7E3D
E5C2	120	14	01-69-34	6542	7DA2
E5C3	120	14	01-69-35	6543	7DAF
E5C4	120	15	01-69-36	6544	7DDC
E5C5	120	14	01-69-37	6545	7DB8
E5C6	120	13	01-69-38	6546	7D9F
E5C7	120	14	01-69-39	6547	7DB0
E5C8	120	15	01-69-40	6548	7DD8
E5C9	120	15	01-69-41	6549	7DDD
E5CA	120	15	01-69-42	654A	7DE4
E5CB	120	15	01-69-43	654B	7DDE
E5CC	120	16	01-69-44	654C	7DFB
E5CD	120	15	01-69-45	654D	7DF2
E5CE	120	15	01-69-46	654E	7DE1
E5CF	120	15	01-69-47	654F	7E05
E5D0	120	16	01-69-48	6550	7E0A
E5D1	120	16	01-69-49	6551	7E23
E5D2	120	16	01-69-50	6552	7E21
E5D3	120	16	01-69-51	6553	7E12
E5D4	120	17	01-69-52	6554	7E31
E5D5	120	16	01-69-53	6555	7E1F
E5D6	120	16	01-69-54	6556	7E09
E5D7	120	16	01-69-55	6557	7E0B
E5D8	120	16	01-69-56	6558	7E22
E5D9	120	17	01-69-57	6559	7E46
E5DA	120	18	01-69-58	655A	7E66
E5DB	120	17	01-69-59	655B	7E3B
E5DC	120	17	01-69-60	655C	7E35
E5DD	120	17	01-69-61	655D	7E39
E5DE	120	17	01-69-62	655E	7E43
E5DF	120	17	01-69-63	655F	7E37
E5E0	120	17	01-69-64	6560	7E32
E5E1	120	17	01-69-65	6561	7E3A
E5E2	120	18	01-69-66	6562	7E67
E5E3	120	18	01-69-67	6563	7E5D
E5E4	120	18	01-69-68	6564	7E56
E5E5	120	18	01-69-69	6565	7E5E
E5E6	120	18	01-69-70	6566	7E59
E5E7	120	18	01-69-71	6567	7E5A
E5E8	120	19	01-69-72	6568	7E79
E5E9	120	19	01-69-73	6569	7E6A
E5EA	120	19	01-69-74	656A	7E69
E5EB	120	20	01-69-75	656B	7E7C
E5EC	120	20	01-69-76	656C	7E7B
E5ED	120	20	01-69-77	656D	7E83
E5EE	120	14	01-69-78	656E	7DD5
E5EF	120	20	01-69-79	656F	7E7D
E5F0	120	20	01-69-80	6570	8FAE
E5F1	120	21	01-69-81	6571	7E7F
E5F2	120	21	01-69-82	6572	7E88
E5F3	120	21	01-69-83	6573	7E89
E5F4	120	21	01-69-84	6574	7E8C
E5F5	120	22	01-69-85	6575	7E92
E5F6	120	21	01-69-86	6576	7E90
E5F7	120	23	01-69-87	6577	7E93
E5F8	120	23	01-69-88	6578	7E94
E5F9	120	23	01-69-89	6579	7E96
E5FA	120	21	01-69-90	657A	7E8E
E5FB	120	24	01-69-91	657B	7E9B
E5FC	120	28	01-69-92	657C	7E9C
E5FD	121	9	01-69-93	657D	7F38
E5FE	121	10	01-69-94	657E	7F3A
E6A1	121	17	01-70-01	6621	7F45
E6A2	121	20	01-70-02	6622	7F4C
E6A3	121	21	01-70-03	6623	7F4D
E6A4	32	22	01-70-04	6624	7F4E
E6A5	121	23	01-70-05	6625	7F50
E6A6	122	6	01-70-06	6626	7F51
E6A7	122	7	01-70-07	6627	7F55
E6A8	122	8	01-70-08	6628	7F54
E6A9	122	9	01-70-09	6629	7F58
E6AA	122	10	01-70-10	662A	7F5F
E6AB	122	10	01-70-11	662B	7F60
E6AC	122	13	01-70-12	662C	7F68
E6AD	122	13	01-70-13	662D	7F69
E6AE	122	13	01-70-14	662E	7F67
E6AF	122	15	01-70-15	662F	7F78
E6B0	122	18	01-70-16	6630	7F82
E6B1	122	19	01-70-17	6631	7F86
E6B2	14	18	01-70-18	6632	7F83
E6B3	122	24	01-70-19	6633	7F88
E6B4	122	22	01-70-20	6634	7F87
E6B5	123	8	01-70-21	6635	7F8C
E6B6	123	10	01-70-22	6636	7F94
E6B7	123	11	01-70-23	6637	7F9E
E6B8	123	11	01-70-24	6638	7F9D
E6B9	123	11	01-70-25	6639	7F9A
E6BA	123	13	01-70-26	663A	7FA3
E6BB	123	15	01-70-27	663B	7FAF
E6BC	123	16	01-70-28	663C	7FB2
E6BD	123	19	01-70-29	663D	7FB9
E6BE	123	15	01-70-30	663E	7FAE
E6BF	123	19	01-70-31	663F	7FB6
E6C0	123	19	01-70-32	6640	7FB8
E6C1	30	20	01-70-33	6641	8B71
E6C2	124	10	01-70-34	6642	7FC5
E6C3	124	10	01-70-35	6643	7FC6
E6C4	124	11	01-70-36	6644	7FCA
E6C5	124	12	01-70-37	6645	7FD5
E6C6	124	12	01-70-38	6646	7FD4
E6C7	124	14	01-70-39	6647	7FE1
E6C8	124	15	01-70-40	6648	7FE6
E6C9	124	15	01-70-41	6649	7FE9
E6CA	124	17	01-70-42	664A	7FF3
E6CB	124	18	01-70-43	664B	7FF9
E6CC	124	21	01-70-44	664C	98DC
E6CD	125	10	01-70-45	664D	8006
E6CE	125	10	01-70-46	664E	8004
E6CF	125	12	01-70-47	664F	800B
E6D0	127	6	01-70-48	6650	8012
E6D1	127	10	01-70-49	6651	8018
E6D2	127	10	01-70-50	6652	8019
E6D3	127	11	01-70-51	6653	801C
E6D4	127	13	01-70-52	6654	8021
E6D5	127	16	01-70-53	6655	8028
E6D6	128	10	01-70-54	6656	803F
E6D7	61	10	01-70-55	6657	803B
E6D8	128	11	01-70-56	6658	804A
E6D9	128	11	01-70-57	6659	8046
E6DA	128	12	01-70-58	665A	8052
E6DB	128	13	01-70-59	665B	8058
E6DC	128	14	01-70-60	665C	805A
E6DD	128	14	01-70-61	665D	805F
E6DE	128	14	01-70-62	665E	8062
E6DF	128	15	01-70-63	665F	8068
E6E0	128	17	01-70-64	6660	8073
E6E1	128	17	01-70-65	6661	8072
E6E2	128	17	01-70-66	6662	8070
E6E3	128	18	01-70-67	6663	8076
E6E4	128	20	01-70-68	6664	8079
E6E5	128	22	01-70-69	6665	807D
E6E6	129	6	01-70-70	6666	807F
E6E7	129	13	01-70-71	6667	8084
E6E8	129	13	01-70-72	6668	8086
E6E9	129	12	01-70-73	6669	8085
E6EA	130	7	01-70-74	666A	809B
E6EB	130	7	01-70-75	666B	8093
E6EC	130	7	01-70-76	666C	809A
E6ED	130	8	01-70-77	666D	80AD
E6EE	109	8	01-70-78	666E	5190
E6EF	130	8	01-70-79	666F	80AC
E6F0	130	9	01-70-80	6670	80DB
E6F1	130	9	01-70-81	6671	80E5
E6F2	130	9	01-70-82	6672	80D9
E6F3	130	9	01-70-83	6673	80DD
E6F4	130	9	01-70-84	6674	80C4
E6F5	130	9	01-70-85	6675	80DA
E6F6	130	9	01-70-86	6676	80D6
E6F7	130	9	01-70-87	6677	8109
E6F8	130	10	01-70-88	6678	80EF
E6F9	130	10	01-70-89	6679	80F1
E6FA	130	11	01-70-90	667A	811B
E6FB	130	11	01-70-91	667B	8129
E6FC	130	11	01-70-92	667C	8123
E6FD	130	11	01-70-93	667D	812F
E6FE	130	12	01-70-94	667E	814B
E7A1	130	12	01-71-01	6721	968B
E7A2	130	12	01-71-02	6722	8146
E7A3	130	12	01-71-03	6723	813E
E7A4	130	12	01-71-04	6724	8153
E7A5	130	12	01-71-05	6725	8151
E7A6	130	10	01-71-06	6726	80FC
E7A7	130	13	01-71-07	6727	8171
E7A8	181	13	01-71-08	6728	816E
E7A9	130	13	01-71-09	6729	8165
E7AA	130	13	01-71-10	672A	8166
E7AB	130	13	01-71-11	672B	8174
E7AC	130	14	01-71-12	672C	8183
E7AD	130	14	01-71-13	672D	8188
E7AE	130	14	01-71-14	672E	818A
E7AF	130	14	01-71-15	672F	8180
E7B0	130	14	01-71-16	6730	8182
E7B1	130	15	01-71-17	6731	81A0
E7B2	130	15	01-71-18	6732	8195
E7B3	130	15	01-71-19	6733	81A4
E7B4	130	15	01-71-20	6734	81A3
E7B5	130	13	01-71-21	6735	815F
E7B6	130	15	01-71-22	6736	8193
E7B7	130	16	01-71-23	6737	81A9
E7B8	130	16	01-71-24	6738	81B0
E7B9	130	15	01-71-25	6739	81B5
E7BA	130	17	01-71-26	673A	81BE
E7BB	130	17	01-71-27	673B	81B8
E7BC	130	17	01-71-28	673C	81BD
E7BD	130	17	01-71-29	673D	81C0
E7BE	130	17	01-71-30	673E	81C2
E7BF	130	17	01-71-31	673F	81BA
E7C0	130	17	01-71-32	6740	81C9
E7C1	130	18	01-71-33	6741	81CD
E7C2	130	18	01-71-34	6742	81D1
E7C3	130	20	01-71-35	6743	81D9
E7C4	130	19	01-71-36	6744	81D8
E7C5	130	16	01-71-37	6745	81C8
E7C6	130	20	01-71-38	6746	81DA
E7C7	130	22	01-71-39	6747	81DF
E7C8	130	25	01-71-40	6748	81E0
E7C9	131	15	01-71-41	6749	81E7
E7CA	133	14	01-71-42	674A	81FA
E7CB	133	16	01-71-43	674B	81FB
E7CC	134	9	01-71-44	674C	81FE
E7CD	134	9	01-71-45	674D	8201
E7CE	134	11	01-71-46	674E	8202
E7CF	134	13	01-71-47	674F	8205
E7D0	134	14	01-71-48	6750	8207
E7D1	172	17	01-71-49	6751	820A
E7D2	30	8	01-71-50	6752	820D
E7D3	135	10	01-71-51	6753	8210
E7D4	30	15	01-71-52	6754	8216
E7D5	137	10	01-71-53	6755	8229
E7D6	137	10	01-71-54	6756	822B
E7D7	137	11	01-71-55	6757	8238
E7D8	137	11	01-71-56	6758	8233
E7D9	137	13	01-71-57	6759	8240
E7DA	137	16	01-71-58	675A	8259
E7DB	137	15	01-71-59	675B	8258
E7DC	137	17	01-71-60	675C	825D
E7DD	137	17	01-71-61	675D	825A
E7DE	137	18	01-71-62	675E	825F
E7DF	137	19	01-71-63	675F	8264
E7E0	137	19	01-71-64	6760	8262
E7E1	137	19	01-71-65	6761	8268
E7E2	137	21	01-71-66	6762	826A
E7E3	137	22	01-71-67	6763	826B
E7E4	137	10	01-71-68	6764	822E
E7E5	138	17	01-71-69	6765	8271
E7E6	139	24	01-71-70	6766	8277
E7E7	140	6	01-71-71	6767	8278
E7E8	140	5	01-71-72	6768	827E
E7E9	140	6	01-71-73	6769	828D
E7EA	140	6	01-71-74	676A	8292
E7EB	140	7	01-71-75	676B	82AB
E7EC	140	7	01-71-76	676C	829F
E7ED	20	10	01-71-77	676D	82BB
E7EE	140	7	01-71-78	676E	82AC
E7EF	140	8	01-71-79	676F	82E1
E7F0	140	8	01-71-80	6770	82E3
E7F1	140	8	01-71-81	6771	82DF
E7F2	140	8	01-71-82	6772	82D2
E7F3	140	8	01-71-83	6773	82F4
E7F4	140	8	01-71-84	6774	82F3
E7F5	140	8	01-71-85	6775	82FA
E7F6	140	10	01-71-86	6776	8393
E7F7	140	8	01-71-87	6777	8303
E7F8	140	8	01-71-88	6778	82FB
E7F9	140	8	01-71-89	6779	82F9
E7FA	140	8	01-71-90	677A	82DE
E7FB	140	8	01-71-91	677B	8306
E7FC	140	8	01-71-92	677C	82DC
E7FD	140	8	01-71-93	677D	8309
E7FE	140	8	01-71-94	677E	82D9
E8A1	140	9	01-72-01	6821	8335
E8A2	140	9	01-72-02	6822	8334
E8A3	140	9	01-72-03	6823	8316
E8A4	140	9	01-72-04	6824	8332
E8A5	140	9	01-72-05	6825	8331
E8A6	140	9	01-72-06	6826	8340
E8A7	140	9	01-72-07	6827	8339
E8A8	140	9	01-72-08	6828	8350
E8A9	140	9	01-72-09	6829	8345
E8AA	140	9	01-72-10	682A	832F
E8AB	140	9	01-72-11	682B	832B
E8AC	140	9	01-72-12	682C	8317
E8AD	140	9	01-72-13	682D	8318
E8AE	140	10	01-72-14	682E	8385
E8AF	140	10	01-72-15	682F	839A
E8B0	140	10	01-72-16	6830	83AA
E8B1	140	10	01-72-17	6831	839F
E8B2	140	10	01-72-18	6832	83A2
E8B3	140	10	01-72-19	6833	8396
E8B4	140	10	01-72-20	6834	8323
E8B5	140	10	01-72-21	6835	838E
E8B6	140	10	01-72-22	6836	8387
E8B7	140	10	01-72-23	6837	838A
E8B8	140	10	01-72-24	6838	837C
E8B9	140	10	01-72-25	6839	83B5
E8BA	140	10	01-72-26	683A	8373
E8BB	140	10	01-72-27	683B	8375
E8BC	140	10	01-72-28	683C	83A0
E8BD	140	10	01-72-29	683D	8389
E8BE	140	10	01-72-30	683E	83A8
E8BF	140	11	01-72-31	683F	83F4
E8C0	140	11	01-72-32	6840	8413
E8C1	140	11	01-72-33	6841	83EB
E8C2	140	11	01-72-34	6842	83CE
E8C3	140	11	01-72-35	6843	83FD
E8C4	140	11	01-72-36	6844	8403
E8C5	140	11	01-72-37	6845	83D8
E8C6	140	11	01-72-38	6846	840B
E8C7	140	11	01-72-39	6847	83C1
E8C8	140	11	01-72-40	6848	83F7
E8C9	140	11	01-72-41	6849	8407
E8CA	140	11	01-72-42	684A	83E0
E8CB	140	11	01-72-43	684B	83F2
E8CC	140	11	01-72-44	684C	840D
E8CD	140	11	01-72-45	684D	8422
E8CE	140	11	01-72-46	684E	8420
E8CF	140	9	01-72-47	684F	83BD
E8D0	140	12	01-72-48	6850	8438
E8D1	140	14	01-72-49	6851	8506
E8D2	140	11	01-72-50	6852	83FB
E8D3	140	12	01-72-51	6853	846D
E8D4	140	12	01-72-52	6854	842A
E8D5	140	12	01-72-53	6855	843C
E8D6	140	15	01-72-54	6856	855A
E8D7	140	12	01-72-55	6857	8484
E8D8	140	12	01-72-56	6858	8477
E8D9	140	12	01-72-57	6859	846B
E8DA	140	13	01-72-58	685A	84AD
E8DB	140	12	01-72-59	685B	846E
E8DC	140	12	01-72-60	685C	8482
E8DD	140	12	01-72-61	685D	8469
E8DE	140	12	01-72-62	685E	8446
E8DF	140	12	01-72-63	685F	842C
E8E0	140	12	01-72-64	6860	846F
E8E1	140	12	01-72-65	6861	8479
E8E2	140	12	01-72-66	6862	8435
E8E3	140	13	01-72-67	6863	84CA
E8E4	140	12	01-72-68	6864	8462
E8E5	140	13	01-72-69	6865	84B9
E8E6	140	13	01-72-70	6866	84BF
E8E7	140	13	01-72-71	6867	849F
E8E8	140	13	01-72-72	6868	84D9
E8E9	140	13	01-72-73	6869	84CD
E8EA	140	13	01-72-74	686A	84BB
E8EB	140	13	01-72-75	686B	84DA
E8EC	140	13	01-72-76	686C	84D0
E8ED	140	13	01-72-77	686D	84C1
E8EE	140	13	01-72-78	686E	84C6
E8EF	140	13	01-72-79	686F	84D6
E8F0	140	13	01-72-80	6870	84A1
E8F1	140	14	01-72-81	6871	8521
E8F2	140	14	01-72-82	6872	84FF
E8F3	140	14	01-72-83	6873	84F4
E8F4	140	14	01-72-84	6874	8517
E8F5	140	14	01-72-85	6875	8518
E8F6	140	15	01-72-86	6876	852C
E8F7	140	14	01-72-87	6877	851F
E8F8	140	14	01-72-88	6878	8515
E8F9	140	14	01-72-89	6879	8514
E8FA	140	14	01-72-90	687A	84FC
E8FB	140	15	01-72-91	687B	8540
E8FC	140	15	01-72-92	687C	8563
E8FD	140	15	01-72-93	687D	8558
E8FE	140	15	01-72-94	687E	8548
E9A1	140	15	01-73-01	6921	8541
E9A2	140	19	01-73-02	6922	8602
E9A3	140	15	01-73-03	6923	854B
E9A4	140	15	01-73-04	6924	8555
E9A5	140	16	01-73-05	6925	8580
E9A6	140	16	01-73-06	6926	85A4
E9A7	140	16	01-73-07	6927	8588
E9A8	140	16	01-73-08	6928	8591
E9A9	140	16	01-73-09	6929	858A
E9AA	140	16	01-73-10	692A	85A8
E9AB	140	15	01-73-11	692B	856D
E9AC	140	16	01-73-12	692C	8594
E9AD	140	16	01-73-13	692D	859B
E9AE	140	18	01-73-14	692E	85EA
E9AF	140	16	01-73-15	692F	8587
E9B0	140	16	01-73-16	6930	859C
E9B1	140	16	01-73-17	6931	8577
E9B2	140	16	01-73-18	6932	857E
E9B3	140	16	01-73-19	6933	8590
E9B4	140	17	01-73-20	6934	85C9
E9B5	140	17	01-73-21	6935	85BA
E9B6	140	18	01-73-22	6936	85CF
E9B7	140	17	01-73-23	6937	85B9
E9B8	140	17	01-73-24	6938	85D0
E9B9	140	18	01-73-25	6939	85D5
E9BA	140	18	01-73-26	693A	85DD
E9BB	140	18	01-73-27	693B	85E5
E9BC	140	18	01-73-28	693C	85DC
E9BD	140	19	01-73-29	693D	85F9
E9BE	140	19	01-73-30	693E	860A
E9BF	140	19	01-73-31	693F	8613
E9C0	140	19	01-73-32	6940	860B
E9C1	140	19	01-73-33	6941	85FE
E9C2	140	19	01-73-34	6942	85FA
E9C3	140	19	01-73-35	6943	8606
E9C4	140	19	01-73-36	6944	8622
E9C5	140	20	01-73-37	6945	861A
E9C6	140	20	01-73-38	6946	8630
E9C7	140	22	01-73-39	6947	863F
E9C8	141	6	01-73-40	6948	864D
E9C9	4	7	01-73-41	6949	4E55
E9CA	141	10	01-73-42	694A	8654
E9CB	141	13	01-73-43	694B	865F
E9CC	141	17	01-73-44	694C	8667
E9CD	142	8	01-73-45	694D	8671
E9CE	142	10	01-73-46	694E	8693
E9CF	142	10	01-73-47	694F	86A3
E9D0	142	10	01-73-48	6950	86A9
E9D1	142	10	01-73-49	6951	86AA
E9D2	142	10	01-73-50	6952	868B
E9D3	142	10	01-73-51	6953	868C
E9D4	142	11	01-73-52	6954	86B6
E9D5	142	11	01-73-53	6955	86AF
E9D6	142	11	01-73-54	6956	86C4
E9D7	142	11	01-73-55	6957	86C6
E9D8	142	11	01-73-56	6958	86B0
E9D9	142	11	01-73-57	6959	86C9
E9DA	142	20	01-73-58	695A	8823
E9DB	142	11	01-73-59	695B	86AB
E9DC	142	12	01-73-60	695C	86D4
E9DD	142	12	01-73-61	695D	86DE
E9DE	142	12	01-73-62	695E	86E9
E9DF	142	12	01-73-63	695F	86EC
E9E0	142	12	01-73-64	6960	86DF
E9E1	142	12	01-73-65	6961	86DB
E9E2	142	12	01-73-66	6962	86EF
E9E3	142	13	01-73-67	6963	8712
E9E4	142	13	01-73-68	6964	8706
E9E5	142	13	01-73-69	6965	8708
E9E6	142	13	01-73-70	6966	8700
E9E7	142	13	01-73-71	6967	8703
E9E8	142	13	01-73-72	6968	86FB
E9E9	142	13	01-73-73	6969	8711
E9EA	142	13	01-73-74	696A	8709
E9EB	142	13	01-73-75	696B	870D
E9EC	142	13	01-73-76	696C	86F9
E9ED	142	13	01-73-77	696D	870A
E9EE	142	14	01-73-78	696E	8734
E9EF	142	14	01-73-79	696F	873F
E9F0	142	14	01-73-80	6970	8737
E9F1	142	14	01-73-81	6971	873B
E9F2	142	14	01-73-82	6972	8725
E9F3	142	14	01-73-83	6973	8729
E9F4	142	14	01-73-84	6974	871A
E9F5	142	15	01-73-85	6975	8760
E9F6	142	15	01-73-86	6976	875F
E9F7	142	15	01-73-87	6977	8778
E9F8	142	15	01-73-88	6978	874C
E9F9	142	15	01-73-89	6979	874E
E9FA	142	15	01-73-90	697A	8774
E9FB	142	15	01-73-91	697B	8757
E9FC	142	15	01-73-92	697C	8768
E9FD	142	15	01-73-93	697D	876E
E9FE	142	15	01-73-94	697E	8759
EAA1	142	15	01-74-01	6A21	8753
EAA2	142	15	01-74-02	6A22	8763
EAA3	142	15	01-74-03	6A23	876A
EAA4	142	19	01-74-04	6A24	8805
EAA5	142	16	01-74-05	6A25	87A2
EAA6	142	16	01-74-06	6A26	879F
EAA7	142	16	01-74-07	6A27	8782
EAA8	142	17	01-74-08	6A28	87AF
EAA9	142	17	01-74-09	6A29	87CB
EAAA	142	17	01-74-10	6A2A	87BD
EAAB	142	17	01-74-11	6A2B	87C0
EAAC	142	17	01-74-12	6A2C	87D0
EAAD	142	17	01-74-13	6A2D	96D6
EAAE	142	17	01-74-14	6A2E	87AB
EAAF	142	17	01-74-15	6A2F	87C4
EAB0	142	17	01-74-16	6A30	87B3
EAB1	142	16	01-74-17	6A31	87C7
EAB2	142	16	01-74-18	6A32	87C6
EAB3	142	17	01-74-19	6A33	87BB
EAB4	142	18	01-74-20	6A34	87EF
EAB5	142	18	01-74-21	6A35	87F2
EAB6	142	18	01-74-22	6A36	87E0
EAB7	142	19	01-74-23	6A37	880F
EAB8	142	19	01-74-24	6A38	880D
EAB9	142	19	01-74-25	6A39	87FE
EABA	142	19	01-74-26	6A3A	87F6
EABB	142	19	01-74-27	6A3B	87F7
EABC	142	17	01-74-28	6A3C	880E
EABD	142	16	01-74-29	6A3D	87D2
EABE	142	20	01-74-30	6A3E	8811
EABF	142	19	01-74-31	6A3F	8816
EAC0	142	20	01-74-32	6A40	8815
EAC1	142	21	01-74-33	6A41	8822
EAC2	142	21	01-74-34	6A42	8821
EAC3	142	23	01-74-35	6A43	8831
EAC4	142	24	01-74-36	6A44	8836
EAC5	142	24	01-74-37	6A45	8839
EAC6	142	22	01-74-38	6A46	8827
EAC7	142	25	01-74-39	6A47	883B
EAC8	143	10	01-74-40	6A48	8844
EAC9	143	9	01-74-41	6A49	8842
EACA	144	11	01-74-42	6A4A	8852
EACB	144	13	01-74-43	6A4B	8859
EACC	144	16	01-74-44	6A4C	885E
EACD	144	24	01-74-45	6A4D	8862
EACE	145	8	01-74-46	6A4E	886B
EACF	145	10	01-74-47	6A4F	8881
EAD0	145	10	01-74-48	6A50	887E
EAD1	145	11	01-74-49	6A51	889E
EAD2	145	9	01-74-50	6A52	8875
EAD3	145	9	01-74-51	6A53	887D
EAD4	145	11	01-74-52	6A54	88B5
EAD5	145	9	01-74-53	6A55	8872
EAD6	145	9	01-74-54	6A56	8882
EAD7	145	10	01-74-55	6A57	8897
EAD8	145	10	01-74-56	6A58	8892
EAD9	145	10	01-74-57	6A59	88AE
EADA	145	10	01-74-58	6A5A	8899
EADB	145	10	01-74-59	6A5B	88A2
EADC	145	10	01-74-60	6A5C	888D
EADD	145	11	01-74-61	6A5D	88A4
EADE	145	11	01-74-62	6A5E	88B0
EADF	145	11	01-74-63	6A5F	88BF
EAE0	145	11	01-74-64	6A60	88B1
EAE1	145	11	01-74-65	6A61	88C3
EAE2	145	11	01-74-66	6A62	88C4
EAE3	145	13	01-74-67	6A63	88D4
EAE4	145	13	01-74-68	6A64	88D8
EAE5	145	12	01-74-69	6A65	88D9
EAE6	145	13	01-74-70	6A66	88DD
EAE7	145	14	01-74-71	6A67	88F9
EAE8	145	13	01-74-72	6A68	8902
EAE9	145	13	01-74-73	6A69	88FC
EAEA	145	14	01-74-74	6A6A	88F4
EAEB	145	13	01-74-75	6A6B	88E8
EAEC	145	13	01-74-76	6A6C	88F2
EAED	145	13	01-74-77	6A6D	8904
EAEE	145	13	01-74-78	6A6E	890C
EAEF	145	14	01-74-79	6A6F	890A
EAF0	145	14	01-74-80	6A70	8913
EAF1	145	17	01-74-81	6A71	8943
EAF2	145	15	01-74-82	6A72	891E
EAF3	145	15	01-74-83	6A73	8925
EAF4	145	15	01-74-84	6A74	892A
EAF5	145	15	01-74-85	6A75	892B
EAF6	145	17	01-74-86	6A76	8941
EAF7	145	17	01-74-87	6A77	8944
EAF8	145	17	01-74-88	6A78	893B
EAF9	145	16	01-74-89	6A79	8936
EAFA	145	16	01-74-90	6A7A	8938
EAFB	145	17	01-74-91	6A7B	894C
EAFC	145	14	01-74-92	6A7C	891D
EAFD	145	18	01-74-93	6A7D	8960
EAFE	145	19	01-74-94	6A7E	895E
EBA1	145	19	01-75-01	6B21	8966
EBA2	145	20	01-75-02	6B22	8964
EBA3	145	20	01-75-03	6B23	896D
EBA4	145	19	01-75-04	6B24	896A
EBA5	145	21	01-75-05	6B25	896F
EBA6	145	22	01-75-06	6B26	8974
EBA7	145	23	01-75-07	6B27	8977
EBA8	146	6	01-75-08	6B28	897E
EBA9	146	12	01-75-09	6B29	8983
EBAA	146	19	01-75-10	6B2A	8988
EBAB	146	25	01-75-11	6B2B	898A
EBAC	147	11	01-75-12	6B2C	8993
EBAD	147	12	01-75-13	6B2D	8998
EBAE	147	14	01-75-14	6B2E	89A1
EBAF	109	16	01-75-15	6B2F	89A9
EBB0	147	16	01-75-16	6B30	89A6
EBB1	147	17	01-75-17	6B31	89AC
EBB2	147	17	01-75-18	6B32	89AF
EBB3	147	18	01-75-19	6B33	89B2
EBB4	147	20	01-75-20	6B34	89BA
EBB5	147	22	01-75-21	6B35	89BD
EBB6	147	22	01-75-22	6B36	89BF
EBB7	147	24	01-75-23	6B37	89C0
EBB8	148	12	01-75-24	6B38	89DA
EBB9	148	13	01-75-25	6B39	89DC
EBBA	148	12	01-75-26	6B3A	89DD
EBBB	148	13	01-75-27	6B3B	89E7
EBBC	148	18	01-75-28	6B3C	89F4
EBBD	148	20	01-75-29	6B3D	89F8
EBBE	149	9	01-75-30	6B3E	8A03
EBBF	149	10	01-75-31	6B3F	8A16
EBC0	149	10	01-75-32	6B40	8A10
EBC1	149	10	01-75-33	6B41	8A0C
EBC2	149	11	01-75-34	6B42	8A1B
EBC3	149	11	01-75-35	6B43	8A1D
EBC4	149	11	01-75-36	6B44	8A25
EBC5	149	12	01-75-37	6B45	8A36
EBC6	149	12	01-75-38	6B46	8A41
EBC7	149	12	01-75-39	6B47	8A5B
EBC8	149	12	01-75-40	6B48	8A52
EBC9	149	12	01-75-41	6B49	8A46
EBCA	149	12	01-75-42	6B4A	8A48
EBCB	149	13	01-75-43	6B4B	8A7C
EBCC	149	13	01-75-44	6B4C	8A6D
EBCD	149	13	01-75-45	6B4D	8A6C
EBCE	149	13	01-75-46	6B4E	8A62
EBCF	149	13	01-75-47	6B4F	8A85
EBD0	149	13	01-75-48	6B50	8A82
EBD1	149	13	01-75-49	6B51	8A84
EBD2	149	14	01-75-50	6B52	8AA8
EBD3	149	14	01-75-51	6B53	8AA1
EBD4	149	14	01-75-52	6B54	8A91
EBD5	149	14	01-75-53	6B55	8AA5
EBD6	149	14	01-75-54	6B56	8AA6
EBD7	149	14	01-75-55	6B57	8A9A
EBD8	149	14	01-75-56	6B58	8AA3
EBD9	149	15	01-75-57	6B59	8AC4
EBDA	149	15	01-75-58	6B5A	8ACD
EBDB	149	15	01-75-59	6B5B	8AC2
EBDC	149	15	01-75-60	6B5C	8ADA
EBDD	149	16	01-75-61	6B5D	8AEB
EBDE	149	16	01-75-62	6B5E	8AF3
EBDF	149	16	01-75-63	6B5F	8AE7
EBE0	30	16	01-75-64	6B60	8AE4
EBE1	149	16	01-75-65	6B61	8AF1
EBE2	149	17	01-75-66	6B62	8B14
EBE3	149	16	01-75-67	6B63	8AE0
EBE4	149	16	01-75-68	6B64	8AE2
EBE5	149	16	01-75-69	6B65	8AF7
EBE6	149	16	01-75-70	6B66	8ADE
EBE7	149	15	01-75-71	6B67	8ADB
EBE8	149	17	01-75-72	6B68	8B0C
EBE9	149	17	01-75-73	6B69	8B07
EBEA	149	17	01-75-74	6B6A	8B1A
EBEB	149	16	01-75-75	6B6B	8AE1
EBEC	149	17	01-75-76	6B6C	8B16
EBED	149	17	01-75-77	6B6D	8B10
EBEE	149	17	01-75-78	6B6E	8B17
EBEF	149	17	01-75-79	6B6F	8B20
EBF0	149	18	01-75-80	6B70	8B33
EBF1	149	18	01-75-81	6B71	97AB
EBF2	149	18	01-75-82	6B72	8B26
EBF3	149	18	01-75-83	6B73	8B2B
EBF4	149	18	01-75-84	6B74	8B3E
EBF5	149	17	01-75-85	6B75	8B28
EBF6	149	17	01-75-86	6B76	8B41
EBF7	149	19	01-75-87	6B77	8B4C
EBF8	149	19	01-75-88	6B78	8B4F
EBF9	149	19	01-75-89	6B79	8B4E
EBFA	149	19	01-75-90	6B7A	8B49
EBFB	149	19	01-75-91	6B7B	8B56
EBFC	149	19	01-75-92	6B7C	8B5B
EBFD	149	19	01-75-93	6B7D	8B5A
EBFE	149	20	01-75-94	6B7E	8B6B
ECA1	149	20	01-76-01	6C21	8B5F
ECA2	149	20	01-76-02	6C22	8B6C
ECA3	149	20	01-76-03	6C23	8B6F
ECA4	149	21	01-76-04	6C24	8B74
ECA5	149	21	01-76-05	6C25	8B7D
ECA6	149	22	01-76-06	6C26	8B80
ECA7	149	23	01-76-07	6C27	8B8C
ECA8	149	23	01-76-08	6C28	8B8E
ECA9	149	24	01-76-09	6C29	8B92
ECAA	149	24	01-76-10	6C2A	8B93
ECAB	149	24	01-76-11	6C2B	8B96
ECAC	149	24	01-76-12	6C2C	8B99
ECAD	149	26	01-76-13	6C2D	8B9A
ECAE	150	11	01-76-14	6C2E	8C3A
ECAF	150	17	01-76-15	6C2F	8C41
ECB0	150	17	01-76-16	6C30	8C3F
ECB1	151	10	01-76-17	6C31	8C48
ECB2	151	15	01-76-18	6C32	8C4C
ECB3	151	16	01-76-19	6C33	8C4E
ECB4	151	18	01-76-20	6C34	8C50
ECB5	152	7	01-76-21	6C35	8C55
ECB6	152	13	01-76-22	6C36	8C62
ECB7	94	16	01-76-23	6C37	8C6C
ECB8	153	7	01-76-24	6C38	8C78
ECB9	153	10	01-76-25	6C39	8C7A
ECBA	153	12	01-76-26	6C3A	8C82
ECBB	153	13	01-76-27	6C3B	8C89
ECBC	153	13	01-76-28	6C3C	8C85
ECBD	153	13	01-76-29	6C3D	8C8A
ECBE	94	14	01-76-30	6C3E	8C8D
ECBF	94	15	01-76-31	6C3F	8C8E
ECC0	153	17	01-76-32	6C40	8C94
ECC1	153	11	01-76-33	6C41	8C7C
ECC2	153	17	01-76-34	6C42	8C98
ECC3	154	11	01-76-35	6C43	621D
ECC4	154	11	01-76-36	6C44	8CAD
ECC5	154	11	01-76-37	6C45	8CAA
ECC6	154	12	01-76-38	6C46	8CBD
ECC7	154	13	01-76-39	6C47	8CB2
ECC8	154	12	01-76-40	6C48	8CB3
ECC9	154	11	01-76-41	6C49	8CAE
ECCA	154	11	01-76-42	6C4A	8CB6
ECCB	154	13	01-76-43	6C4B	8CC8
ECCC	154	12	01-76-44	6C4C	8CC1
ECCD	154	15	01-76-45	6C4D	8CE4
ECCE	154	15	01-76-46	6C4E	8CE3
ECCF	154	15	01-76-47	6C4F	8CDA
ECD0	154	17	01-76-48	6C50	8CFD
ECD1	154	17	01-76-49	6C51	8CFA
ECD2	154	17	01-76-50	6C52	8CFB
ECD3	154	18	01-76-51	6C53	8D04
ECD4	154	18	01-76-52	6C54	8D05
ECD5	154	19	01-76-53	6C55	8D0A
ECD6	154	19	01-76-54	6C56	8D07
ECD7	154	20	01-76-55	6C57	8D0F
ECD8	154	20	01-76-56	6C58	8D0D
ECD9	154	21	01-76-57	6C59	8D10
ECDA	154	21	01-76-58	6C5A	9F4E
ECDB	154	22	01-76-59	6C5B	8D13
ECDC	154	13	01-76-60	6C5C	8CCD
ECDD	154	21	01-76-61	6C5D	8D14
ECDE	154	22	01-76-62	6C5E	8D16
ECDF	155	12	01-76-63	6C5F	8D67
ECE0	155	16	01-76-64	6C60	8D6D
ECE1	156	6	01-76-65	6C61	8D71
ECE2	156	10	01-76-66	6C62	8D73
ECE3	156	12	01-76-67	6C63	8D81
ECE4	156	14	01-76-68	6C64	8D99
ECE5	157	11	01-76-69	6C65	8DC2
ECE6	157	11	01-76-70	6C66	8DBE
ECE7	157	11	01-76-71	6C67	8DBA
ECE8	157	12	01-76-72	6C68	8DCF
ECE9	157	12	01-76-73	6C69	8DDA
ECEA	157	12	01-76-74	6C6A	8DD6
ECEB	157	12	01-76-75	6C6B	8DCC
ECEC	157	12	01-76-76	6C6C	8DDB
ECED	157	12	01-76-77	6C6D	8DCB
ECEE	157	13	01-76-78	6C6E	8DEA
ECEF	157	13	01-76-79	6C6F	8DEB
ECF0	157	13	01-76-80	6C70	8DDF
ECF1	157	13	01-76-81	6C71	8DE3
ECF2	157	14	01-76-82	6C72	8DFC
ECF3	157	14	01-76-83	6C73	8E08
ECF4	157	14	01-76-84	6C74	8E09
ECF5	157	14	01-76-85	6C75	8DFF
ECF6	157	15	01-76-86	6C76	8E1D
ECF7	157	15	01-76-87	6C77	8E1E
ECF8	157	15	01-76-88	6C78	8E10
ECF9	157	15	01-76-89	6C79	8E1F
ECFA	157	16	01-76-90	6C7A	8E42
ECFB	157	16	01-76-91	6C7B	8E35
ECFC	157	16	01-76-92	6C7C	8E30
ECFD	157	16	01-76-93	6C7D	8E34
ECFE	60	17	01-76-94	6C7E	8E4A
EDA1	157	17	01-77-01	6D21	8E47
EDA2	157	17	01-77-02	6D22	8E49
EDA3	157	17	01-77-03	6D23	8E4C
EDA4	157	17	01-77-04	6D24	8E50
EDA5	157	17	01-77-05	6D25	8E48
EDA6	157	18	01-77-06	6D26	8E59
EDA7	157	18	01-77-07	6D27	8E64
EDA8	157	18	01-77-08	6D28	8E60
EDA9	157	15	01-77-09	6D29	8E2A
EDAA	157	18	01-77-10	6D2A	8E63
EDAB	157	18	01-77-11	6D2B	8E55
EDAC	157	19	01-77-12	6D2C	8E76
EDAD	157	19	01-77-13	6D2D	8E72
EDAE	157	19	01-77-14	6D2E	8E7C
EDAF	157	20	01-77-15	6D2F	8E81
EDB0	157	19	01-77-16	6D30	8E87
EDB1	157	20	01-77-17	6D31	8E85
EDB2	157	20	01-77-18	6D32	8E84
EDB3	157	21	01-77-19	6D33	8E8B
EDB4	157	21	01-77-20	6D34	8E8A
EDB5	157	22	01-77-21	6D35	8E93
EDB6	157	22	01-77-22	6D36	8E91
EDB7	157	22	01-77-23	6D37	8E94
EDB8	157	23	01-77-24	6D38	8E99
EDB9	157	26	01-77-25	6D39	8EAA
EDBA	157	25	01-77-26	6D3A	8EA1
EDBB	158	10	01-77-27	6D3B	8EAC
EDBC	158	12	01-77-28	6D3C	8EB0
EDBD	158	20	01-77-29	6D3D	8EC6
EDBE	158	13	01-77-30	6D3E	8EB1
EDBF	158	16	01-77-31	6D3F	8EBE
EDC0	158	19	01-77-32	6D40	8EC5
EDC1	158	24	01-77-33	6D41	8EC8
EDC2	159	8	01-77-34	6D42	8ECB
EDC3	159	11	01-77-35	6D43	8EDB
EDC4	159	11	01-77-36	6D44	8EE3
EDC5	159	12	01-77-37	6D45	8EFC
EDC6	159	12	01-77-38	6D46	8EFB
EDC7	159	12	01-77-39	6D47	8EEB
EDC8	159	13	01-77-40	6D48	8EFE
EDC9	159	13	01-77-41	6D49	8F0A
EDCA	159	13	01-77-42	6D4A	8F05
EDCB	159	14	01-77-43	6D4B	8F15
EDCC	159	14	01-77-44	6D4C	8F12
EDCD	159	15	01-77-45	6D4D	8F19
EDCE	159	14	01-77-46	6D4E	8F13
EDCF	159	15	01-77-47	6D4F	8F1C
EDD0	159	15	01-77-48	6D50	8F1F
EDD1	159	15	01-77-49	6D51	8F1B
EDD2	159	13	01-77-50	6D52	8F0C
EDD3	159	15	01-77-51	6D53	8F26
EDD4	159	16	01-77-52	6D54	8F33
EDD5	159	16	01-77-53	6D55	8F3B
EDD6	159	16	01-77-54	6D56	8F39
EDD7	159	17	01-77-55	6D57	8F45
EDD8	159	17	01-77-56	6D58	8F42
EDD9	159	17	01-77-57	6D59	8F3E
EDDA	159	18	01-77-58	6D5A	8F4C
EDDB	159	18	01-77-59	6D5B	8F49
EDDC	159	18	01-77-60	6D5C	8F46
EDDD	159	19	01-77-61	6D5D	8F4E
EDDE	159	20	01-77-62	6D5E	8F57
EDDF	159	21	01-77-63	6D5F	8F5C
EDE0	159	22	01-77-64	6D60	8F62
EDE1	159	23	01-77-65	6D61	8F63
EDE2	159	23	01-77-66	6D62	8F64
EDE3	160	12	01-77-67	6D63	8F9C
EDE4	160	13	01-77-68	6D64	8F9F
EDE5	160	14	01-77-69	6D65	8FA3
EDE6	160	19	01-77-70	6D66	8FAD
EDE7	149	21	01-77-71	6D67	8FAF
EDE8	162	5	01-77-72	6D68	8FB7
EDE9	162	8	01-77-73	6D69	8FDA
EDEA	162	9	01-77-74	6D6A	8FE5
EDEB	162	9	01-77-75	6D6B	8FE2
EDEC	162	8	01-77-76	6D6C	8FEA
EDED	162	9	01-77-77	6D6D	8FEF
EDEE	162	18	01-77-78	6D6E	9087
EDEF	162	10	01-77-79	6D6F	8FF4
EDF0	162	10	01-77-80	6D70	9005
EDF1	157	10	01-77-81	6D71	8FF9
EDF2	162	10	01-77-82	6D72	8FFA
EDF3	162	11	01-77-83	6D73	9011
EDF4	60	11	01-77-84	6D74	9015
EDF5	162	11	01-77-85	6D75	9021
EDF6	162	11	01-77-86	6D76	900D
EDF7	162	11	01-77-87	6D77	901E
EDF8	162	11	01-77-88	6D78	9016
EDF9	162	11	01-77-89	6D79	900B
EDFA	162	11	01-77-90	6D7A	9027
EDFB	162	12	01-77-91	6D7B	9036
EDFC	162	12	01-77-92	6D7C	9035
EDFD	162	12	01-77-93	6D7D	9039
EDFE	162	10	01-77-94	6D7E	8FF8
EEA1	162	13	01-78-01	6E21	904F
EEA2	162	13	01-78-02	6E22	9050
EEA3	162	13	01-78-03	6E23	9051
EEA4	162	13	01-78-04	6E24	9052
EEA5	162	11	01-78-05	6E25	900E
EEA6	162	13	01-78-06	6E26	9049
EEA7	162	13	01-78-07	6E27	903E
EEA8	162	13	01-78-08	6E28	9056
EEA9	162	14	01-78-09	6E29	9058
EEAA	162	14	01-78-10	6E2A	905E
EEAB	162	15	01-78-11	6E2B	9068
EEAC	162	15	01-78-12	6E2C	906F
EEAD	162	16	01-78-13	6E2D	9076
EEAE	162	16	01-78-14	6E2E	96A8
EEAF	162	16	01-78-15	6E2F	9072
EEB0	162	17	01-78-16	6E30	9082
EEB1	162	17	01-78-17	6E31	907D
EEB2	162	16	01-78-18	6E32	9081
EEB3	162	17	01-78-19	6E33	9080
EEB4	162	19	01-78-20	6E34	908A
EEB5	162	17	01-78-21	6E35	9089
EEB6	162	23	01-78-22	6E36	908F
EEB7	75	7	01-78-23	6E37	90A8
EEB8	163	8	01-78-24	6E38	90AF
EEB9	163	8	01-78-25	6E39	90B1
EEBA	163	8	01-78-26	6E3A	90B5
EEBB	163	10	01-78-27	6E3B	90E2
EEBC	163	10	01-78-28	6E3C	90E4
EEBD	163	11	01-78-29	6E3D	6248
EEBE	163	10	01-78-30	6E3E	90DB
EEBF	163	12	01-78-31	6E3F	9102
EEC0	163	13	01-78-32	6E40	9112
EEC1	163	14	01-78-33	6E41	9119
EEC2	163	15	01-78-34	6E42	9132
EEC3	170	15	01-78-35	6E43	9130
EEC4	164	9	01-78-36	6E44	914A
EEC5	164	11	01-78-37	6E45	9156
EEC6	164	11	01-78-38	6E46	9158
EEC7	164	12	01-78-39	6E47	9163
EEC8	164	12	01-78-40	6E48	9165
EEC9	164	13	01-78-41	6E49	9169
EECA	164	14	01-78-42	6E4A	9173
EECB	164	14	01-78-43	6E4B	9172
EECC	164	15	01-78-44	6E4C	918B
EECD	164	15	01-78-45	6E4D	9189
EECE	164	15	01-78-46	6E4E	9182
EECF	164	17	01-78-47	6E4F	91A2
EED0	164	18	01-78-48	6E50	91AB
EED1	164	19	01-78-49	6E51	91AF
EED2	164	18	01-78-50	6E52	91AA
EED3	164	20	01-78-51	6E53	91B5
EED4	164	20	01-78-52	6E54	91B4
EED5	164	21	01-78-53	6E55	91BA
EED6	164	24	01-78-54	6E56	91C0
EED7	164	25	01-78-55	6E57	91C1
EED8	165	12	01-78-56	6E58	91C9
EED9	165	20	01-78-57	6E59	91CB
EEDA	166	18	01-78-58	6E5A	91D0
EEDB	167	10	01-78-59	6E5B	91D6
EEDC	167	10	01-78-60	6E5C	91DF
EEDD	167	10	01-78-61	6E5D	91E1
EEDE	167	10	01-78-62	6E5E	91DB
EEDF	167	11	01-78-63	6E5F	91FC
EEE0	167	11	01-78-64	6E60	91F5
EEE1	167	11	01-78-65	6E61	91F6
EEE2	167	12	01-78-66	6E62	921E
EEE3	167	12	01-78-67	6E63	91FF
EEE4	167	12	01-78-68	6E64	9214
EEE5	167	12	01-78-69	6E65	922C
EEE6	167	12	01-78-70	6E66	9215
EEE7	167	12	01-78-71	6E67	9211
EEE8	167	13	01-78-72	6E68	925E
EEE9	167	13	01-78-73	6E69	9257
EEEA	167	13	01-78-74	6E6A	9245
EEEB	167	13	01-78-75	6E6B	9249
EEEC	167	13	01-78-76	6E6C	9264
EEED	167	13	01-78-77	6E6D	9248
EEEE	167	14	01-78-78	6E6E	9295
EEEF	167	13	01-78-79	6E6F	923F
EEF0	167	13	01-78-80	6E70	924B
EEF1	167	13	01-78-81	6E71	9250
EEF2	167	14	01-78-82	6E72	929C
EEF3	167	14	01-78-83	6E73	9296
EEF4	167	14	01-78-84	6E74	9293
EEF5	167	14	01-78-85	6E75	929B
EEF6	167	13	01-78-86	6E76	925A
EEF7	167	15	01-78-87	6E77	92CF
EEF8	167	15	01-78-88	6E78	92B9
EEF9	167	15	01-78-89	6E79	92B7
EEFA	167	14	01-78-90	6E7A	92E9
EEFB	167	16	01-78-91	6E7B	930F
EEFC	167	16	01-78-92	6E7C	92FA
EEFD	167	16	01-78-93	6E7D	9344
EEFE	167	16	01-78-94	6E7E	932E
EFA1	167	16	01-79-01	6F21	9319
EFA2	167	16	01-79-02	6F22	9322
EFA3	167	16	01-79-03	6F23	931A
EFA4	167	16	01-79-04	6F24	9323
EFA5	167	15	01-79-05	6F25	933A
EFA6	167	15	01-79-06	6F26	9335
EFA7	167	16	01-79-07	6F27	933B
EFA8	167	17	01-79-08	6F28	935C
EFA9	167	17	01-79-09	6F29	9360
EFAA	167	17	01-79-10	6F2A	937C
EFAB	167	17	01-79-11	6F2B	936E
EFAC	167	17	01-79-12	6F2C	9356
EFAD	167	18	01-79-13	6F2D	93B0
EFAE	167	18	01-79-14	6F2E	93AC
EFAF	167	18	01-79-15	6F2F	93AD
EFB0	167	18	01-79-16	6F30	9394
EFB1	167	18	01-79-17	6F31	93B9
EFB2	167	19	01-79-18	6F32	93D6
EFB3	167	20	01-79-19	6F33	93D7
EFB4	167	19	01-79-20	6F34	93E8
EFB5	167	19	01-79-21	6F35	93E5
EFB6	167	19	01-79-22	6F36	93D8
EFB7	167	19	01-79-23	6F37	93C3
EFB8	167	19	01-79-24	6F38	93DD
EFB9	167	19	01-79-25	6F39	93D0
EFBA	167	19	01-79-26	6F3A	93C8
EFBB	167	19	01-79-27	6F3B	93E4
EFBC	167	20	01-79-28	6F3C	941A
EFBD	167	20	01-79-29	6F3D	9414
EFBE	167	20	01-79-30	6F3E	9413
EFBF	167	20	01-79-31	6F3F	9403
EFC0	167	20	01-79-32	6F40	9407
EFC1	167	20	01-79-33	6F41	9410
EFC2	167	21	01-79-34	6F42	9436
EFC3	167	21	01-79-35	6F43	942B
EFC4	167	21	01-79-36	6F44	9435
EFC5	167	20	01-79-37	6F45	9421
EFC6	167	21	01-79-38	6F46	943A
EFC7	167	21	01-79-39	6F47	9441
EFC8	167	23	01-79-40	6F48	9452
EFC9	167	22	01-79-41	6F49	9444
EFCA	167	23	01-79-42	6F4A	945B
EFCB	167	23	01-79-43	6F4B	9460
EFCC	167	23	01-79-44	6F4C	9462
EFCD	167	23	01-79-45	6F4D	945E
EFCE	167	24	01-79-46	6F4E	946A
EFCF	167	12	01-79-47	6F4F	9229
EFD0	167	25	01-79-48	6F50	9470
EFD1	167	25	01-79-49	6F51	9475
EFD2	167	26	01-79-50	6F52	9477
EFD3	167	27	01-79-51	6F53	947D
EFD4	167	23	01-79-52	6F54	945A
EFD5	167	27	01-79-53	6F55	947C
EFD6	167	27	01-79-54	6F56	947E
EFD7	167	28	01-79-55	6F57	9481
EFD8	167	28	01-79-56	6F58	947F
EFD9	169	9	01-79-57	6F59	9582
EFDA	169	11	01-79-58	6F5A	9587
EFDB	169	11	01-79-59	6F5B	958A
EFDC	169	12	01-79-60	6F5C	9594
EFDD	169	12	01-79-61	6F5D	9596
EFDE	169	13	01-79-62	6F5E	9598
EFDF	169	13	01-79-63	6F5F	9599
EFE0	169	13	01-79-64	6F60	95A0
EFE1	169	14	01-79-65	6F61	95A8
EFE2	169	14	01-79-66	6F62	95A7
EFE3	169	15	01-79-67	6F63	95AD
EFE4	169	16	01-79-68	6F64	95BC
EFE5	169	16	01-79-69	6F65	95BB
EFE6	169	16	01-79-70	6F66	95B9
EFE7	169	16	01-79-71	6F67	95BE
EFE8	169	17	01-79-72	6F68	95CA
EFE9	169	17	01-79-73	6F69	6FF6
EFEA	169	17	01-79-74	6F6A	95C3
EFEB	169	17	01-79-75	6F6B	95CD
EFEC	169	17	01-79-76	6F6C	95CC
EFED	169	18	01-79-77	6F6D	95D5
EFEE	169	18	01-79-78	6F6E	95D4
EFEF	169	18	01-79-79	6F6F	95D6
EFF0	169	19	01-79-80	6F70	95DC
EFF1	169	20	01-79-81	6F71	95E1
EFF2	169	21	01-79-82	6F72	95E5
EFF3	169	21	01-79-83	6F73	95E2
EFF4	170	6	01-79-84	6F74	9621
EFF5	170	7	01-79-85	6F75	9628
EFF6	170	7	01-79-86	6F76	962E
EFF7	32	7	01-79-87	6F77	962F
EFF8	32	8	01-79-88	6F78	9642
EFF9	170	9	01-79-89	6F79	964C
EFFA	170	9	01-79-90	6F7A	964F
EFFB	170	9	01-79-91	6F7B	964B
EFFC	170	11	01-79-92	6F7C	9677
EFFD	170	10	01-79-93	6F7D	965C
EFFE	170	10	01-79-94	6F7E	965E
F0A1	170	10	01-80-01	7021	965D
F0A2	170	10	01-80-02	7022	965F
F0A3	170	10	01-80-03	7023	9666
F0A4	170	11	01-80-04	7024	9672
F0A5	170	11	01-80-05	7025	966C
F0A6	170	12	01-80-06	7026	968D
F0A7	170	13	01-80-07	7027	9698
F0A8	170	13	01-80-08	7028	9695
F0A9	170	13	01-80-09	7029	9697
F0AA	170	16	01-80-10	702A	96AA
F0AB	170	16	01-80-11	702B	96A7
F0AC	170	17	01-80-12	702C	96B1
F0AD	187	17	01-80-13	702D	96B2
F0AE	170	17	01-80-14	702E	96B0
F0AF	170	19	01-80-15	702F	96B4
F0B0	171	8	01-80-16	7030	96B6
F0B1	171	17	01-80-17	7031	96B8
F0B2	172	8	01-80-18	7032	96B9
F0B3	172	13	01-80-19	7033	96CE
F0B4	172	13	01-80-20	7034	96CB
F0B5	172	13	01-80-21	7035	96C9
F0B6	172	13	01-80-22	7036	96CD
F0B7	172	17	01-80-23	7037	894D
F0B8	172	18	01-80-24	7038	96DC
F0B9	172	16	01-80-25	7039	970D
F0BA	172	16	01-80-26	703A	96D5
F0BB	173	13	01-80-27	703B	96F9
F0BC	173	15	01-80-28	703C	9704
F0BD	173	15	01-80-29	703D	9706
F0BE	173	15	01-80-30	703E	9708
F0BF	173	16	01-80-31	703F	9713
F0C0	173	16	01-80-32	7040	970E
F0C1	173	16	01-80-33	7041	9711
F0C2	173	16	01-80-34	7042	970F
F0C3	173	16	01-80-35	7043	9716
F0C4	173	16	01-80-36	7044	9719
F0C5	173	18	01-80-37	7045	9724
F0C6	173	19	01-80-38	7046	972A
F0C7	173	20	01-80-39	7047	9730
F0C8	173	21	01-80-40	7048	9739
F0C9	173	22	01-80-41	7049	973D
F0CA	173	22	01-80-42	704A	973E
F0CB	173	24	01-80-43	704B	9744
F0CC	173	24	01-80-44	704C	9746
F0CD	173	24	01-80-45	704D	9748
F0CE	173	24	01-80-46	704E	9742
F0CF	173	25	01-80-47	704F	9749
F0D0	174	16	01-80-48	7050	975C
F0D1	175	15	01-80-49	7051	9760
F0D2	176	14	01-80-50	7052	9764
F0D3	176	16	01-80-51	7053	9766
F0D4	176	23	01-80-52	7054	9768
F0D5	177	11	01-80-53	7055	52D2
F0D6	177	12	01-80-54	7056	976B
F0D7	177	12	01-80-55	7057	9771
F0D8	177	13	01-80-56	7058	9779
F0D9	177	14	01-80-57	7059	9785
F0DA	177	14	01-80-58	705A	977C
F0DB	177	14	01-80-59	705B	9781
F0DC	177	14	01-80-60	705C	977A
F0DD	177	14	01-80-61	705D	9786
F0DE	177	15	01-80-62	705E	978B
F0DF	177	15	01-80-63	705F	978F
F0E0	177	15	01-80-64	7060	9790
F0E1	177	17	01-80-65	7061	979C
F0E2	177	18	01-80-66	7062	97A8
F0E3	177	18	01-80-67	7063	97A6
F0E4	177	18	01-80-68	7064	97A3
F0E5	177	18	01-80-69	7065	97B3
F0E6	177	19	01-80-70	7066	97B4
F0E7	177	22	01-80-71	7067	97C3
F0E8	177	24	01-80-72	7068	97C6
F0E9	177	23	01-80-73	7069	97C8
F0EA	178	9	01-80-74	706A	97CB
F0EB	178	19	01-80-75	706B	97DC
F0EC	179	9	01-80-76	706C	97ED
F0ED	179	23	01-80-77	706D	9F4F
F0EE	179	19	01-80-78	706E	97F2
F0EF	180	11	01-80-79	706F	7ADF
F0F0	180	14	01-80-80	7070	97F6
F0F1	180	13	01-80-81	7071	97F5
F0F2	181	13	01-80-82	7072	980F
F0F3	181	13	01-80-83	7073	980C
F0F4	181	16	01-80-84	7074	9838
F0F5	181	16	01-80-85	7075	9824
F0F6	181	15	01-80-86	7076	9821
F0F7	181	16	01-80-87	7077	9837
F0F8	181	16	01-80-88	7078	983D
F0F9	181	17	01-80-89	7079	9846
F0FA	181	18	01-80-90	707A	984F
F0FB	181	18	01-80-91	707B	984B
F0FC	181	22	01-80-92	707C	986B
F0FD	181	23	01-80-93	707D	986F
F0FE	181	24	01-80-94	707E	9870
F1A1	181	25	01-81-01	7121	9871
F1A2	181	26	01-81-02	7122	9874
F1A3	181	27	01-81-03	7123	9873
F1A4	182	12	01-81-04	7124	98AA
F1A5	182	14	01-81-05	7125	98AF
F1A6	182	14	01-81-06	7126	98B1
F1A7	182	17	01-81-07	7127	98B6
F1A8	182	20	01-81-08	7128	98C4
F1A9	182	20	01-81-09	7129	98C3
F1AA	182	21	01-81-10	712A	98C6
F1AB	184	13	01-81-11	712B	98E9
F1AC	184	13	01-81-12	712C	98EB
F1AD	184	15	01-81-13	712D	9903
F1AE	184	15	01-81-14	712E	9909
F1AF	184	16	01-81-15	712F	9912
F1B0	184	16	01-81-16	7130	9914
F1B1	184	16	01-81-17	7131	9918
F1B2	184	17	01-81-18	7132	9921
F1B3	50	16	01-81-19	7133	991D
F1B4	184	17	01-81-20	7134	991E
F1B5	184	17	01-81-21	7135	9924
F1B6	184	17	01-81-22	7136	9920
F1B7	184	18	01-81-23	7137	992C
F1B8	184	18	01-81-24	7138	992E
F1B9	184	19	01-81-25	7139	993D
F1BA	184	19	01-81-26	713A	993E
F1BB	184	19	01-81-27	713B	9942
F1BC	184	20	01-81-28	713C	9949
F1BD	184	20	01-81-29	713D	9945
F1BE	184	21	01-81-30	713E	9950
F1BF	184	21	01-81-31	713F	994B
F1C0	184	21	01-81-32	7140	9951
F1C1	184	21	01-81-33	7141	9952
F1C2	184	21	01-81-34	7142	994C
F1C3	184	22	01-81-35	7143	9955
F1C4	185	11	01-81-36	7144	9997
F1C5	185	17	01-81-37	7145	9998
F1C6	186	18	01-81-38	7146	99A5
F1C7	187	12	01-81-39	7147	99AD
F1C8	187	12	01-81-40	7148	99AE
F1C9	187	14	01-81-41	7149	99BC
F1CA	187	15	01-81-42	714A	99DF
F1CB	187	15	01-81-43	714B	99DB
F1CC	187	15	01-81-44	714C	99DD
F1CD	187	15	01-81-45	714D	99D8
F1CE	187	15	01-81-46	714E	99D1
F1CF	187	16	01-81-47	714F	99ED
F1D0	187	16	01-81-48	7150	99EE
F1D1	187	16	01-81-49	7151	99F1
F1D2	187	16	01-81-50	7152	99F2
F1D3	187	17	01-81-51	7153	99FB
F1D4	187	17	01-81-52	7154	99F8
F1D5	187	17	01-81-53	7155	9A01
F1D6	187	18	01-81-54	7156	9A0F
F1D7	187	18	01-81-55	7157	9A05
F1D8	187	16	01-81-56	7158	99E2
F1D9	187	19	01-81-57	7159	9A19
F1DA	187	20	01-81-58	715A	9A2B
F1DB	187	20	01-81-59	715B	9A37
F1DC	187	21	01-81-60	715C	9A45
F1DD	187	21	01-81-61	715D	9A42
F1DE	187	20	01-81-62	715E	9A40
F1DF	187	21	01-81-63	715F	9A43
F1E0	187	21	01-81-64	7160	9A3E
F1E1	187	22	01-81-65	7161	9A55
F1E2	187	22	01-81-66	7162	9A4D
F1E3	187	23	01-81-67	7163	9A5B
F1E4	187	23	01-81-68	7164	9A57
F1E5	187	24	01-81-69	7165	9A5F
F1E6	187	26	01-81-70	7166	9A62
F1E7	187	26	01-81-71	7167	9A65
F1E8	187	27	01-81-72	7168	9A64
F1E9	187	27	01-81-73	7169	9A69
F1EA	187	30	01-81-74	716A	9A6B
F1EB	187	29	01-81-75	716B	9A6A
F1EC	188	13	01-81-76	716C	9AAD
F1ED	188	14	01-81-77	716D	9AB0
F1EE	188	16	01-81-78	716E	9ABC
F1EF	188	18	01-81-79	716F	9AC0
F1F0	188	21	01-81-80	7170	9ACF
F1F1	188	23	01-81-81	7171	9AD1
F1F2	188	23	01-81-82	7172	9AD3
F1F3	188	23	01-81-83	7173	9AD4
F1F4	189	23	01-81-84	7174	9ADE
F1F5	190	10	01-81-85	7175	9ADF
F1F6	190	13	01-81-86	7176	9AE2
F1F7	190	14	01-81-87	7177	9AE3
F1F8	190	14	01-81-88	7178	9AE6
F1F9	190	15	01-81-89	7179	9AEF
F1FA	190	15	01-81-90	717A	9AEB
F1FB	190	15	01-81-91	717B	9AEE
F1FC	190	15	01-81-92	717C	9AF4
F1FD	190	15	01-81-93	717D	9AF1
F1FE	190	16	01-81-94	717E	9AF7
F2A1	190	16	01-82-01	7221	9AFB
F2A2	190	18	01-82-02	7222	9B06
F2A3	190	21	01-82-03	7223	9B18
F2A4	190	22	01-82-04	7224	9B1A
F2A5	190	23	01-82-05	7225	9B1F
F2A6	190	24	01-82-06	7226	9B22
F2A7	190	25	01-82-07	7227	9B23
F2A8	191	10	01-82-08	7228	9B25
F2A9	191	15	01-82-09	7229	9B27
F2AA	191	16	01-82-10	722A	9B28
F2AB	191	18	01-82-11	722B	9B29
F2AC	191	20	01-82-12	722C	9B2A
F2AD	191	26	01-82-13	722D	9B2E
F2AE	192	10	01-82-14	722E	9B2F
F2AF	193	10	01-82-15	722F	9B32
F2B0	194	15	01-82-16	7230	9B44
F2B1	194	15	01-82-17	7231	9B43
F2B2	194	18	01-82-18	7232	9B4F
F2B3	194	18	01-82-19	7233	9B4D
F2B4	194	18	01-82-20	7234	9B4E
F2B5	194	21	01-82-21	7235	9B51
F2B6	194	24	01-82-22	7236	9B58
F2B7	195	15	01-82-23	7237	9B74
F2B8	195	16	01-82-24	7238	9B93
F2B9	195	16	01-82-25	7239	9B83
F2BA	195	16	01-82-26	723A	9B91
F2BB	195	16	01-82-27	723B	9B96
F2BC	195	16	01-82-28	723C	9B97
F2BD	195	17	01-82-29	723D	9B9F
F2BE	195	17	01-82-30	723E	9BA0
F2BF	195	17	01-82-31	723F	9BA8
F2C0	195	17	01-82-32	7240	9BB4
F2C1	195	18	01-82-33	7241	9BC0
F2C2	195	18	01-82-34	7242	9BCA
F2C3	195	18	01-82-35	7243	9BB9
F2C4	195	18	01-82-36	7244	9BC6
F2C5	195	18	01-82-37	7245	9BCF
F2C6	195	18	01-82-38	7246	9BD1
F2C7	195	18	01-82-39	7247	9BD2
F2C8	195	19	01-82-40	7248	9BE3
F2C9	195	19	01-82-41	7249	9BE2
F2CA	195	19	01-82-42	724A	9BE4
F2CB	195	19	01-82-43	724B	9BD4
F2CC	195	19	01-82-44	724C	9BE1
F2CD	195	22	01-82-45	724D	9C3A
F2CE	195	19	01-82-46	724E	9BF2
F2CF	195	19	01-82-47	724F	9BF1
F2D0	195	19	01-82-48	7250	9BF0
F2D1	195	20	01-82-49	7251	9C15
F2D2	195	20	01-82-50	7252	9C14
F2D3	195	20	01-82-51	7253	9C09
F2D4	195	20	01-82-52	7254	9C13
F2D5	195	20	01-82-53	7255	9C0C
F2D6	195	20	01-82-54	7256	9C06
F2D7	195	20	01-82-55	7257	9C08
F2D8	195	20	01-82-56	7258	9C12
F2D9	195	20	01-82-57	7259	9C0A
F2DA	195	20	01-82-58	725A	9C04
F2DB	195	21	01-82-59	725B	9C2E
F2DC	195	20	01-82-60	725C	9C1B
F2DD	195	21	01-82-61	725D	9C25
F2DE	195	21	01-82-62	725E	9C24
F2DF	195	21	01-82-63	725F	9C21
F2E0	195	21	01-82-64	7260	9C30
F2E1	195	22	01-82-65	7261	9C47
F2E2	205	22	01-82-66	7262	9C32
F2E3	195	22	01-82-67	7263	9C46
F2E4	195	22	01-82-68	7264	9C3E
F2E5	195	23	01-82-69	7265	9C5A
F2E6	130	24	01-82-70	7266	9C60
F2E7	195	24	01-82-71	7267	9C67
F2E8	195	26	01-82-72	7268	9C76
F2E9	195	27	01-82-73	7269	9C78
F2EA	196	13	01-82-74	726A	9CE7
F2EB	196	9	01-82-75	726B	9CEC
F2EC	196	13	01-82-76	726C	9CF0
F2ED	196	15	01-82-77	726D	9D09
F2EE	172	15	01-82-78	726E	9D08
F2EF	196	13	01-82-79	726F	9CEB
F2F0	196	15	01-82-80	7270	9D03
F2F1	196	15	01-82-81	7271	9D06
F2F2	196	16	01-82-82	7272	9D2A
F2F3	196	16	01-82-83	7273	9D26
F2F4	196	21	01-82-84	7274	9DAF
F2F5	196	16	01-82-85	7275	9D23
F2F6	196	16	01-82-86	7276	9D1F
F2F7	196	17	01-82-87	7277	9D44
F2F8	196	16	01-82-88	7278	9D15
F2F9	196	16	01-82-89	7279	9D12
F2FA	196	17	01-82-90	727A	9D41
F2FB	196	17	01-82-91	727B	9D3F
F2FC	196	17	01-82-92	727C	9D3E
F2FD	196	17	01-82-93	727D	9D46
F2FE	196	17	01-82-94	727E	9D48
F3A1	196	18	01-83-01	7321	9D5D
F3A2	196	18	01-83-02	7322	9D5E
F3A3	196	18	01-83-03	7323	9D64
F3A4	196	18	01-83-04	7324	9D51
F3A5	196	18	01-83-05	7325	9D50
F3A6	196	18	01-83-06	7326	9D59
F3A7	196	19	01-83-07	7327	9D72
F3A8	196	19	01-83-08	7328	9D89
F3A9	196	19	01-83-09	7329	9D87
F3AA	196	20	01-83-10	732A	9DAB
F3AB	196	19	01-83-11	732B	9D6F
F3AC	196	19	01-83-12	732C	9D7A
F3AD	196	20	01-83-13	732D	9D9A
F3AE	196	20	01-83-14	732E	9DA4
F3AF	196	20	01-83-15	732F	9DA9
F3B0	196	21	01-83-16	7330	9DB2
F3B1	196	21	01-83-17	7331	9DC4
F3B2	196	21	01-83-18	7332	9DC1
F3B3	196	21	01-83-19	7333	9DBB
F3B4	196	21	01-83-20	7334	9DB8
F3B5	196	21	01-83-21	7335	9DBA
F3B6	196	21	01-83-22	7336	9DC6
F3B7	196	21	01-83-23	7337	9DCF
F3B8	196	21	01-83-24	7338	9DC2
F3B9	196	22	01-83-25	7339	9DD9
F3BA	196	22	01-83-26	733A	9DD3
F3BB	196	23	01-83-27	733B	9DF8
F3BC	196	23	01-83-28	733C	9DE6
F3BD	196	23	01-83-29	733D	9DED
F3BE	196	23	01-83-30	733E	9DEF
F3BF	196	24	01-83-31	733F	9DFD
F3C0	196	28	01-83-32	7340	9E1A
F3C1	196	28	01-83-33	7341	9E1B
F3C2	196	30	01-83-34	7342	9E1E
F3C3	197	11	01-83-35	7343	9E75
F3C4	197	20	01-83-36	7344	9E79
F3C5	197	25	01-83-37	7345	9E7D
F3C6	198	13	01-83-38	7346	9E81
F3C7	198	16	01-83-39	7347	9E88
F3C8	198	17	01-83-40	7348	9E8B
F3C9	198	18	01-83-41	7349	9E8C
F3CA	198	19	01-83-42	734A	9E92
F3CB	198	19	01-83-43	734B	9E95
F3CC	198	19	01-83-44	734C	9E91
F3CD	198	21	01-83-45	734D	9E9D
F3CE	199	11	01-83-46	734E	9EA5
F3CF	199	15	01-83-47	734F	9EA9
F3D0	199	11	01-83-48	7350	9EB8
F3D1	199	15	01-83-49	7351	9EAA
F3D2	199	16	01-83-50	7352	9EAD
F3D3	200	19	01-83-51	7353	9761
F3D4	201	25	01-83-52	7354	9ECC
F3D5	202	15	01-83-53	7355	9ECE
F3D6	202	17	01-83-54	7356	9ECF
F3D7	202	23	01-83-55	7357	9ED0
F3D8	203	16	01-83-56	7358	9ED4
F3D9	203	17	01-83-57	7359	9EDC
F3DA	203	17	01-83-58	735A	9EDE
F3DB	203	17	01-83-59	735B	9EDD
F3DC	203	18	01-83-60	735C	9EE0
F3DD	203	20	01-83-61	735D	9EE5
F3DE	203	20	01-83-62	735E	9EE8
F3DF	203	21	01-83-63	735F	9EEF
F3E0	203	23	01-83-64	7360	9EF4
F3E1	203	26	01-83-65	7361	9EF6
F3E2	203	27	01-83-66	7362	9EF7
F3E3	204	12	01-83-67	7363	9EF9
F3E4	204	17	01-83-68	7364	9EFB
F3E5	204	19	01-83-69	7365	9EFC
F3E6	205	13	01-83-70	7366	9EFD
F3E7	205	24	01-83-71	7367	9F07
F3E8	205	25	01-83-72	7368	9F08
F3E9	207	14	01-83-73	7369	76B7
F3EA	207	18	01-83-74	736A	9F15
F3EB	208	8	01-83-75	736B	9F21
F3EC	208	18	01-83-76	736C	9F2C
F3ED	209	17	01-83-77	736D	9F3E
F3EE	210	14	01-83-78	736E	9F4A
F3EF	211	15	01-83-79	736F	9F52
F3F0	211	17	01-83-80	7370	9F54
F3F1	211	20	01-83-81	7371	9F63
F3F2	211	20	01-83-82	7372	9F5F
F3F3	211	20	01-83-83	7373	9F60
F3F4	211	20	01-83-84	7374	9F61
F3F5	211	21	01-83-85	7375	9F66
F3F6	211	21	01-83-86	7376	9F67
F3F7	211	22	01-83-87	7377	9F6C
F3F8	211	22	01-83-88	7378	9F6A
F3F9	211	24	01-83-89	7379	9F77
F3FA	211	24	01-83-90	737A	9F72
F3FB	211	24	01-83-91	737B	9F76
F3FC	212	22	01-83-92	737C	9F95
F3FD	213	16	01-83-93	737D	9F9C
F3FE	214	17	01-83-94	737E	9FA0
F4A1	32	12	01-84-01	7421	582F
F4A2	75	14	01-84-02	7422	69C7
F4A3	162	14	01-84-03	7423	9059
F4A4	96	14	01-84-04	7424	7464
59A1	1	2	02-01-02	2122	4E02
59A2	1	3			
59A3	1	4			
59A4	1	4	01-14-02	2E22	2000B
59A5	1	4	02-01-04	2124	4E12
59A6	1	4			
59A7	1	5			
59A8	1	6	01-14-03	2E23	3402
59A9	1	6	02-88-37	7845	8980
59AA	1	7			
59AB	1	9			
59AC	2	1	01-14-04	2E24	4E28
59AD	2	4			
59AE	3	3			
59AF	4	2			
59B0	4	2			
59B1	4	3	02-01-09	2129	4E47
59B2	5	8			
59B3	7	5			
59B4	8	6			
59B5	8	7			
59B6	8	10			
59B7	8	10			
59B8	8	13			
59B9	8	13			
59BA	8	14			
59BB	8	15			
59BC	8	16			
59BD	8	17			
59BE	8	20			
59BF	8	21	02-01-20	2134	4EB9
59C0	9	5			
59C1	9	5	01-14-08	2E28	4EE1
59C2	9	5			
59C3	9	5			
59C4	9	6			
59C5	9	6	01-14-09	2E29	4EFD
59C6	9	6			
59C7	9	6	02-01-31	213F	4F00
59C8	9	6			
59C9	9	6	02-01-29	213D	4EF1
59CA	9	6			
59CB	9	7	01-14-16	2E30	4F56
59CC	9	7			
59CD	9	7			
59CE	9	7			
59CF	9	7			
59D0	9	7			
59D1	9	7			
59D2	9	7			
59D3	9	7			
59D4	9	7	02-01-35	2143	4F3E
59D5	9	8			
59D6	9	8			
59D7	9	8	01-14-20	2E34	4F7E
59D8	9	8	01-14-23	2E37	4F97
59D9	9	8	01-14-22	2E36	4F94
59DA	9	8	02-01-46	214E	4F9A
59DB	9	8	02-01-45	214D	4F92
59DC	9	8			
59DD	9	8			
59DE	9	8			
59DF	9	8			
59E0	9	8	01-14-21	2E35	4F8A
59E1	9	8			
59E2	9	9	01-14-24	2E38	FA30
59E3	9	9	02-01-47	214F	4FE6
59E4	9	9	01-14-25	2E39	4FC9
59E5	9	9			
59E6	9	9			
59E7	9	9	02-01-48	2150	4FB2
59E8	9	9			
59E9	9	9			
59EA	9	9	02-01-51	2153	4FCB
59EB	9	9			
59EC	9	9	02-01-49	2151	4FBE
59ED	9	9			
59EE	9	9			
59EF	9	10			
59F0	9	10	01-14-30	2E3E	5018
59F1	9	10	02-01-61	215D	5022
59F2	9	10	02-01-59	215B	501C
59F3	9	10	02-01-60	215C	501E
59F4	9	10	01-14-31	2E3F	5027
59F5	9	10			
59F6	9	10			
59F7	9	10			
59F8	9	10	01-14-29	2E3D	500E
59F9	9	10			
59FA	9	10			
59FB	9	10			
59FC	9	10	01-14-01	2E21	4FF1
59FD	9	10			
59FE	9	10	02-01-62	215E	3468
5AA1	9	10	01-14-33	2E41	5040
5AA2	9	11			
5AA3	9	11			
5AA4	9	11			
5AA5	9	11			
5AA6	9	11	02-01-63	215F	5042
5AA7	9	11	02-01-64	2160	5046
5AA8	9	11	02-01-67	2163	5057
5AA9	9	11	02-01-68	2164	5063
5AAA	9	11	02-01-69	2165	5066
5AAB	9	11			
5AAC	9	11	02-01-71	2167	5070
5AAD	9	12			
5AAE	9	12			
5AAF	9	12			
5AB0	9	12	01-14-36	2E44	5094
5AB1	9	12			
5AB2	9	12			
5AB3	9	12	02-01-75	216B	5093
5AB4	9	12			
5AB5	9	12			
5AB6	9	12			
5AB7	9	12			
5AB8	9	13			
5AB9	9	13			
5ABA	9	13			
5ABB	9	13			
5ABC	9	13	01-14-37	2E45	50CC
5ABD	9	14	01-14-41	2E49	FA31
5ABE	9	14			
5ABF	9	14	02-01-92	217C	50E1
5AC0	9	14			
5AC1	9	14			
5AC2	9	14			
5AC3	9	14			
5AC4	9	14			
5AC5	9	15			
5AC6	9	15			
5AC7	9	16			
5AC8	9	17			
5AC9	9	18			
5ACA	10	6			
5ACB	10	6	01-14-47	2E4F	514A
5ACC	10	6			
5ACD	10	7			
5ACE	10	7	01-14-48	2E50	FA32
5ACF	10	8			
5AD0	10	9	01-14-50	2E52	5157
5AD1	10	12			
5AD2	10	20			
5AD3	10	21			
5AD4	11	5	02-01-28	213C	4EDA
5AD5	12	5			
5AD6	12	7			
5AD7	12	8			
5AD8	12	9			
5AD9	12	10			
5ADA	12	11			
5ADB	49	12			
5ADC	12	12			
5ADD	13	5			
5ADE	13	8			
5ADF	13	14			
5AE0	13	14			
5AE1	14	7	01-14-52	2E54	519D
5AE2	14	10			
5AE3	15	5			
5AE4	15	8			
5AE5	15	8			
5AE6	15	8	02-03-16	2330	51BC
5AE7	15	10			
5AE8	16	3	01-14-56	2E58	51E2
5AE9	16	4			
5AEA	16	8	01-14-57	2E59	51EE
5AEB	18	5			
5AEC	18	7			
5AED	18	7			
5AEE	18	15			
5AEF	18	16			
5AF0	19	3			
5AF1	19	6	02-03-35	2343	52A6
5AF2	19	6	02-03-34	2342	52A4
5AF3	19	7			
5AF4	19	8			
5AF5	19	8			
5AF6	19	9			
5AF7	19	9			
5AF8	19	9	02-03-39	2347	52CA
5AF9	19	9	01-14-67	2E63	FA33
5AFA	19	10	02-03-41	2349	52D1
5AFB	19	10	02-03-40	2348	351F
5AFC	19	11			
5AFD	19	11			
5AFE	19	11	01-14-70	2E66	52D6
5BA1	19	12	01-14-71	2E67	52DB
5BA2	19	12			
5BA3	19	13	01-14-72	2E68	FA34
5BA4	19	13			
5BA5	19	14			
5BA6	19	14			
5BA7	19	16	01-92-50	7C52	8FA6
5BA8	20	4	01-14-74	2E6A	52FB
5BA9	20	4	01-14-75	2E6B	5300
5BAA	20	5			
5BAB	20	5	01-14-76	2E6C	5307
5BAC	20	12			
5BAD	21	2			
5BAE	21	4			
5BAF	22	7	02-03-46	234E	5324
5BB0	23	9			
5BB1	24	3	02-03-49	2351	5342
5BB2	24	5			
5BB3	24	6			
5BB4	24	8			
5BB5	24	8			
5BB6	24	8			
5BB7	24	10	02-14-66	2E62	684C
5BB8	24	11			
5BB9	24	11			
5BBA	24	13			
5BBB	26	4	02-03-53	2355	536C
5BBC	26	7			
5BBD	26	8			
5BBE	26	8			
5BBF	26	9	01-14-81	2E71	537D
5BC0	27	5			
5BC1	27	6			
5BC2	27	7			
5BC3	27	7			
5BC4	27	8	01-14-82	2E72	5393
5BC5	27	9			
5BC6	27	12			
5BC7	27	16	01-86-37	7645	6B77
5BC8	27	18			
5BC9	27	14	01-14-84	2E74	53B2
5BCA	28	4			
5BCB	28	4			
5BCC	28	6			
5BCD	28	7	02-05-26	253A	590B
5BCE	28	9			
5BCF	29	4			
5BD0	29	9	02-03-64	2360	53DA
5BD1	29	9			
5BD2	29	10			
5BD3	29	14			
5BD4	29	18			
5BD5	30	5	01-47-52	4F54	20B9F
5BD6	30	7			
5BD7	30	7			
5BD8	30	7			
5BD9	30	6			
5BDA	30	8	02-03-76	236C	548A
5BDB	30	8	01-14-88	2E78	549C
5BDC	30	8			
5BDD	30	8	01-14-93	2E7D	5496
5BDE	30	8			
5BDF	30	9	01-15-02	2F22	54C6
5BE0	30	9	01-14-94	2E7E	54A1
5BE1	30	9	01-15-01	2F21	54A9
5BE2	30	9			
5BE3	30	10	01-15-03	2F23	54FF
5BE4	30	10			
5BE5	30	10			
5BE6	30	11			
5BE7	30	11	02-04-08	2428	5561
5BE8	30	10	02-03-92	237C	5500
5BE9	30	11			
5BEA	30	11			
5BEB	30	12			
5BEC	30	12			
5BED	30	12	01-15-10	2F2A	5586
5BEE	30	12			
5BEF	30	12	01-15-12	2F2C	FA36
5BF0	30	12			
5BF1	30	12	01-15-09	2F29	5581
5BF2	30	12			
5BF3	30	12			
5BF4	30	12			
5BF5	30	13			
5BF6	30	13			
5BF7	30	13			
5BF8	30	14			
5BF9	30	14	01-15-15	2F2F	FA37
5BFA	30	14	02-04-31	243F	5610
5BFB	30	15	01-15-18	2F32	563B
5BFC	30	15			
5BFD	30	16	01-15-23	2F37	566F
5BFE	30	16	01-15-25	2F39	5672
5CA1	30	17			
5CA2	30	16	01-15-22	2F36	FA38
5CA3	30	18	02-04-47	244F	569D
5CA4	30	18	01-15-27	2F3B	569E
5CA5	30	18			
5CA6	30	19	02-04-49	2451	56AD
5CA7	30	24			
5CA8	31	7			
5CA9	31	12			
5CAA	31	13			
5CAB	31	15			
5CAC	31	15			
5CAD	32	4	01-15-34	2F42	2123D
5CAE	32	4	01-15-35	2F43	5721
5CAF	32	5	02-04-59	245B	5723
5CB0	32	6	01-15-38	2F46	5734
5CB1	32	7	02-04-64	2460	5746
5CB2	32	7			
5CB3	32	7			
5CB4	32	8			
5CB5	32	8			
5CB6	32	8			
5CB7	32	8			
5CB8	32	9	02-83-83	7373	25E2E
5CB9	32	9	02-04-76	246C	57AC
5CBA	32	10	01-15-44	2F4C	2131B
5CBB	32	10	01-15-43	2F4B	FA0F
5CBC	32	10			
5CBD	32	10	01-15-47	2F4F	57C8
5CBE	32	10	02-04-83	2473	57CC
5CBF	32	11	01-15-51	2F53	57F5
5CC0	32	11	01-15-50	2F52	57ED
5CC1	32	11	01-15-54	2F56	5809
5CC2	32	12			
5CC4	32	12			
5CC5	32	12			
5CC6	32	13			
5CC7	32	13	01-15-55	2F57	FA10
5CC8	32	13			
5CC9	32	13			
5CCA	32	14			
5CCB	32	14			
5CCC	32	14			
5CCD	32	14	02-05-09	2529	588A
5CCE	32	14	02-05-12	252C	588F
5CCF	32	14			
5CD0	32	14			
5CD1	32	15	01-15-61	2F5D	589E
5CD2	32	15	01-15-62	2F5E	FA3A
5CD3	32	15			
5CD4	32	15			
5CD5	32	17	01-15-67	2F63	58D4
5CD6	32	18			
5CD7	33	12			
5CD8	33	13			
5CD9	33	14			
5CDA	33	16			
5CDB	34	7	02-05-25	2539	5906
5CDC	34	8	01-15-71	2F67	590C
5CDD	36	5			
5CDE	36	6			
5CDF	36	6			
5CE0	36	6			
5CE1	36	7			
5CE2	37	5			
5CE3	37	8			
5CE4	37	9			
5CE5	37	9			
5CE6	37	11	02-05-37	2545	595B
5CE7	37	11	02-05-38	2546	595F
5CE8	37	11	01-15-73	2F69	595D
5CE9	37	12			
5CEA	37	13			
5CEB	37	13			
5CEC	37	13			
5CED	37	15	01-15-74	2F6A	596D
5CEE	37	23			
5CEF	38	5			
5CF0	38	5	02-05-40	2548	5975
5CF1	38	7			
5CF2	38	7			
5CF3	38	7	02-05-43	254B	599F
5CF4	38	7	01-15-77	2F6D	59A4
5CF5	38	8	02-05-46	254E	59C8
5CF6	38	8	01-15-78	2F6E	59C3
5CF7	38	8			
5CF8	38	9	02-05-50	2552	59E4
5CF9	38	9	01-15-80	2F70	59DD
5CFA	38	9	02-05-48	2550	59DE
5CFB	38	10			
5CFC	38	10	01-15-82	2F72	5A23
5CFD	38	10			
5CFE	38	10			
5DA1	38	11			
5DA2	38	11			
5DA3	38	11			
5DA4	38	11			
5DA5	38	11			
5DA6	38	11	01-15-83	2F73	5A67
5DA7	38	12	02-05-66	2562	5A9F
5DA8	38	12	01-15-88	2F78	5A9E
5DA9	38	12	01-15-87	2F77	5A84
5DAA	38	12			
5DAB	38	12			
5DAC	38	13			
5DAD	38	13	02-05-70	2566	5AB3
5DAE	38	13			
5DAF	38	13			
5DB0	38	14	02-05-78	256E	5AF0
5DB1	38	14			
5DB2	38	22			
5DB3	39	6	01-47-54	4F56	5B56
5DB4	39	10			
5DB5	39	13			
5DB6	39	12			
5DB7	39	14			
5DB8	39	14			
5DB9	39	20	02-13-72	2D68	6585
5DBA	40	6	02-05-94	257E	219C3
5DBB	40	10			
5DBC	40	10			
5DBD	40	10			
5DBE	40	10			
5DBF	40	10			
5DC0	40	10			
5DC1	40	11			
5DC2	40	11	02-08-04	2824	5BC0
5DC3	40	11	02-08-03	2823	3761
5DC4	40	11			
5DC5	40	12	01-88-41	7849	752F
5DC6	40	13	01-47-57	4F59	5BD8
5DC7	40	14	01-47-58	4F5A	5BEC
5DC8	40	14			
5DC9	40	15			
5DCA	40	15			
5DCB	40	16			
5DCC	40	18			
5DCD	41	11			
5DCE	41	11			
5DCF	41	12			
5DD0	41	12			
5DD1	42	8			
5DD2	42	12	01-47-60	4F5C	5C1E
5DD3	10	9			
5DD4	44	6	01-47-63	4F5F	378D
5DD5	44	9			
5DD6	44	13			
5DD7	44	15	01-47-65	4F61	FA3B
5DD8	46	7	01-47-69	4F65	5C8F
5DD9	46	7	02-08-29	283D	5C8A
5DDA	46	7			
5DDB	46	7			
5DDC	46	7	02-08-28	283C	5C88
5DDD	46	8	01-47-73	4F69	5CBA
5DDE	46	8	02-08-35	2843	5CA6
5DDF	46	8	02-08-39	2847	5CB5
5DE0	46	8			
5DE1	46	8	02-08-38	2846	5CAD
5DE2	46	9			
5DE3	46	9	01-47-74	4F6A	5CCB
5DE4	46	9	02-08-41	2849	5CC9
5DE5	46	9			
5DE6	46	10			
5DE7	46	10	01-47-77	4F6D	5CF4
5DE8	46	11	01-47-81	4F71	5D27
5DE9	46	11			
5DEA	46	11	02-08-46	284E	5D2B
5DEB	46	11			
5DEC	46	12	01-47-82	4F72	FA11
5DED	46	12			
5DEE	46	12			
5DEF	46	12	02-08-53	2855	5D42
5DF0	46	12	01-47-85	4F75	5D53
5DF1	46	13	02-08-55	2857	5D61
5DF2	46	13			
5DF3	46	13			
5DF4	46	13			
5DF5	46	14	02-08-62	285E	3800
5DF6	46	14			
5DF7	46	14			
5DF8	46	14			
5DF9	46	15	01-47-89	4F79	5DA0
5DFA	46	15	02-08-64	2860	5D94
5DFB	46	15	02-08-63	285F	5D92
5DFC	46	15			
5DFD	46	15			
5DFE	46	15			
5EA1	46	16	01-47-91	4F7B	5DA7
5EA2	46	16			
5EA3	46	17	01-47-92	4F7C	5DB8
5EA4	46	17			
5EA5	46	17	02-08-71	2867	5DB9
5EA6	46	20	01-47-93	4F7D	5DCB
5EA7	46	20			
5EA8	46	21			
5EA9	48	5	01-14-59	2E5B	34DB
5EAA	49	9			
5EAB	49	9			
5EAC	50	7			
5EAD	50	7			
5EAE	50	7			
5EAF	50	7			
5EB0	50	11			
5EB1	50	12			
5EB2	50	14	01-84-11	742B	5E58
5EB3	50	15			
5EB4	51	5			
5EB5	51	6			
5EB6	52	9			
5EB7	53	5			
5EB8	53	7			
5EB9	53	10			
5EBA	53	10	02-12-05	2C25	5EAC
5EBB	53	11			
5EBC	53	11			
5EBD	53	11			
5EBE	53	11			
5EBF	53	12	01-84-13	742D	5EBE
5EC0	53	12			
5EC1	53	12			
5EC2	53	13	01-84-14	742E	F928
5EC3	53	14	02-94-57	7E59	9EBD
5EC4	53	14			
5EC5	53	14			
5EC6	53	14			
5EC7	53	16			
5EC8	53	16			
5EC9	54	7			
5ECA	54	8	01-84-16	7430	5EF9
5ECB	54	10	02-12-12	2C2C	5EFD
5ECC	55	9	01-84-19	7433	5F07
5ECD	55	18			
5ECE	56	5			
5ECF	56	6	02-12-14	2C2E	5F0E
5ED0	57	3			
5ED1	57	6	02-12-15	2C2F	5F1C
5ED2	57	8			
5ED3	57	11	01-84-22	7436	5F34
5ED4	57	12			
5ED5	57	12			
5ED6	57	12			
5ED7	57	12			
5ED8	57	14	01-84-26	743A	5F45
5ED9	57	15			
5EDA	59	7	02-12-19	2C33	5F63
5EDB	59	9			
5EDC	59	14			
5EDD	60	6			
5EDE	60	7			
5EDF	60	8			
5EE0	60	9			
5EE1	60	9	01-84-32	7440	5F89
5EE2	60	11			
5EE3	60	12	02-12-24	2C38	5FA4
5EE4	60	15	01-84-36	7444	5FB5
5EE5	60	15			
5EE6	60	15	01-84-37	7445	5FB7
5EE7	60	15			
5EE8	61	5	01-84-38	7446	5FC9
5EE9	61	5	02-12-28	2C3C	5FC7
5EEA	61	7			
5EEB	61	7			
5EEC	61	8	01-84-41	7449	5FE9
5EED	61	8	01-84-45	744D	6033
5EEE	61	8	01-84-39	7447	5FDE
5EEF	61	8			
5EF0	61	8	02-12-33	2C41	5FE2
5EF1	61	9	02-12-38	2C46	3917
5EF2	61	9			
5EF3	61	9			
5EF4	61	9	02-12-42	2C4A	391A
5EF5	61	9			
5EF6	61	11	02-12-45	2C4D	608A
5EF7	61	10	01-84-48	7450	FA3D
5EF8	61	10			
5EF9	61	10			
5EFA	61	10			
5EFB	61	10			
5EFC	61	10			
5EFD	61	11			
5EFE	61	11	02-12-44	2C4C	607F
5FA1	61	11			
5FA2	61	11			
5FA3	61	11	01-84-53	7455	60D5
5FA4	61	11	02-12-49	2C51	60B0
5FA5	61	11			
5FA6	61	11			
5FA7	61	11			
5FA8	61	12			
5FA9	61	12			
5FAA	61	12			
5FAB	61	12	02-12-60	2C5C	6113
5FAC	61	12			
5FAD	61	12	02-12-56	2C58	60F2
5FAE	61	12			
5FAF	61	12			
5FB0	61	12			
5FB1	61	13	01-84-59	745B	6137
5FB2	61	13			
5FB3	61	13	01-84-58	745A	6130
5FB4	61	13			
5FB5	61	13			
5FB6	61	13			
5FB7	61	13			
5FB8	61	13	01-84-57	7459	612B
5FB9	61	14			
5FBA	61	14			
5FBB	61	14			
5FBC	61	14			
5FBD	61	14			
5FBE	61	15	02-12-73	2C69	6198
5FBF	61	15			
5FC0	61	15			
5FC1	61	15	01-84-62	745E	FA3F
5FC2	61	15	02-12-76	2C6C	61AD
5FC3	61	16	01-84-63	745F	61BC
5FC4	61	15	02-12-71	2C67	6193
5FC5	61	17	02-12-79	2C6F	61DD
5FC6	61	18			
5FC7	61	19	01-84-65	7461	FA40
5FC8	62	8			
5FC9	62	8			
5FCA	62	10	01-84-30	743E	5F67
5FCB	62	10			
5FCC	62	11			
5FCD	62	13	01-84-66	7462	6222
5FCE	62	14	02-12-85	2C75	6229
5FCF	62	14			
5FD0	63	4			
5FD1	63	8			
5FD2	1	4			
5FD3	64	7	02-12-94	2C7E	6273
5FD4	64	8			
5FD5	64	8			
5FD6	64	8	02-13-02	2D22	62A6
5FD7	64	9			
5FD8	64	10			
5FD9	64	10	01-84-78	746E	6339
5FDA	64	10			
5FDB	64	10			
5FDC	64	11			
5FDD	64	11			
5FDE	64	11			
5FDF	64	12	01-84-83	7473	63ED
5FE0	64	12			
5FE1	64	12			
5FE2	64	12	01-84-84	7474	63F5
5FE3	64	12			
5FE4	64	12			
5FE5	64	12			
5FE6	64	12			
5FE7	64	13	02-13-36	2D44	6409
5FE8	64	13			
5FE9	64	13			
5FEA	64	14			
5FEB	64	14	01-84-90	747A	6460
5FEC	64	14			
5FED	64	14			
5FEE	64	15	02-13-48	2D50	649D
5FEF	64	15			
5FF0	64	15			
5FF1	64	16			
5FF2	64	16	01-84-92	747C	64CE
5FF3	64	17	01-85-02	7522	64CA
5FF4	64	23			
5FF5	34	10			
5FF6	66	10			
5FF7	66	11	01-85-08	7528	FA41
5FF8	66	11			
5FF9	66	11			
5FFA	66	12			
5FFB	66	12			
5FFC	66	14			
5FFD	66	14			
5FFE	66	18			
60A1	66	20			
60A2	67	9			
60A3	67	10	02-13-73	2D69	658A
60A4	68	12			
60A5	69	8			
60A6	70	10			
60A7	71	5			
60A8	72	7	02-13-81	2D71	3AD6
60A9	72	7			
60AA	72	7			
60AB	72	7			
60AC	72	7	02-13-82	2D72	65F2
60AD	72	8			
60AE	72	8			
60AF	72	8	01-85-13	752D	6609
60B0	72	8			
60B1	72	8	01-85-14	752E	6615
60B3	72	8	01-85-12	752C	6600
60B4	72	8			
60B5	72	8	02-13-86	2D76	6608
60B6	72	9			
60B7	72	9			
60B8	72	9	01-85-20	7534	6630
60B9	72	9	01-85-18	7532	6624
60BA	72	9			
60BB	72	9			
60BC	72	9			
60BD	72	9			
60BE	72	9			
60BF	72	9	01-85-17	7531	6622
60C0	72	9	02-13-87	2D77	6621
60C1	72	9			
60C2	72	9	01-85-22	7536	6633
60C3	72	9			
60C4	72	9	01-85-15	752F	661E
60C5	72	9	01-85-16	7530	663A
60C6	72	9			
60C7	72	10	01-85-44	754C	66FA
60C8	72	10			
60C9	72	10	01-85-24	7538	6648
60CA	72	10			
60CB	72	10	01-85-25	7539	664C
60CC	72	10	02-13-90	2D7A	6651
60CD	72	10			
60CE	72	10			
60CF	72	10			
60D0	72	10	02-13-91	2D7B	664E
60D1	72	11	01-85-30	753E	6665
60D2	72	11	02-14-02	2E22	6663
60D3	72	11	01-85-23	7537	66FB
60D4	72	11	02-14-01	2E21	665B
60D5	72	11			
60D6	72	11	02-13-94	2D7E	6657
60D7	72	11	01-85-27	753B	6659
60D8	72	11	01-85-26	753A	231C4
60D9	72	11			
60DA	72	11			
60DB	72	11	01-85-28	753C	665A
60DC	72	12	01-85-32	7540	6677
60DD	72	12			
60DE	72	12			
60DF	72	12			
60E0	72	12	02-14-06	2E26	666B
60E1	72	12	01-85-31	753F	6673
60E2	72	12	02-14-03	2E23	231F5
60E3	72	12	02-14-05	2E25	666A
60E4	72	13	01-85-35	7543	FA43
60E5	72	13			
60E6	72	13	02-14-13	2E2D	6699
60E7	72	13			
60E8	72	13	02-14-12	2E2C	6692
60E9	72	14			
60EA	72	14	01-85-36	7544	66A0
60EB	72	14			
60EC	72	14			
60ED	72	14			
60EE	72	15	01-85-37	7545	66B2
60EF	72	16	02-14-19	2E33	66BF
60F0	72	16	01-85-39	7547	66C6
60F1	72	16	01-85-38	7546	66BB
60F2	72	16	01-85-40	7548	66C8
60F3	72	16	02-14-18	2E32	3B1A
60F4	72	17	01-85-41	7549	3B22
60F5	72	17			
60F6	72	18			
60F7	72	18			
60F8	73	9	01-85-21	7535	6631
60F9	73	12			
60FA	74	9	01-90-43	7A4B	80D7
60FB	74	9	01-90-44	7A4C	80E0
60FC	74	9			
60FD	74	9			
60FE	74	9			
61A1	74	10	02-85-30	753E	43EE
61A2	74	10			
61A3	74	10	02-85-26	753A	266B0
61A4	74	10	01-85-45	754D	6713
61A5	74	11	02-14-27	2E3B	6719
61A6	74	11			
61A7	74	11	01-85-46	754E	F929
61A8	74	11	02-85-31	753F	8135
61A9	74	11			
61AA	74	11			
61AB	74	11			
61AC	74	11			
61AD	74	14	02-15-26	2F3A	69BA
61AE	75	6			
61AF	75	6			
61B0	75	7	01-85-49	7551	6747
61B1	75	7			
61B2	75	7	01-85-48	7550	6766
61B3	75	7	01-85-52	7554	6781
61B4	75	7			
61B5	75	7			
61B6	75	8			
61B7	75	8	01-85-54	7556	6798
61B8	75	8	02-14-28	2E3C	233D3
61B9	75	8			
61BA	75	9	01-85-59	755B	67D7
61BB	75	9	01-85-56	7558	67BB
61BC	75	9	02-14-49	2E51	67D9
61BD	75	9			
61BE	75	9	01-85-62	755E	6852
61BF	75	9			
61C0	75	9			
61C1	75	9			
61C2	75	10			
61C3	75	10			
61C4	75	10			
61C5	75	10			
61C6	75	10	02-14-63	2E5F	6844
61C7	75	10			
61C8	75	10			
61C9	75	10	02-14-65	2E61	6849
61CA	75	9	01-85-57	7559	67F9
61CB	75	11	02-14-72	2E68	687A
61CC	75	9	01-85-61	755D	6801
61CD	75	11	01-85-69	7565	FA44
61CE	75	11			
61CF	75	11	01-84-08	7428	5DE2
61D0	75	11			
61D1	75	19			
61D2	75	11			
61D3	75	11			
61D4	75	11	01-85-68	7564	6875
61D5	75	11			
61D6	75	11	01-85-72	7568	68B2
61D7	75	11			
61D8	75	11	02-14-73	2E69	687C
61D9	75	12			
61DA	75	12			
61DB	75	12			
61DC	75	12	01-85-73	7569	68C8
61DD	75	12			
61DE	75	12	01-85-81	7571	6949
61DF	75	12	02-14-93	2E7D	68CF
61E0	75	12	02-14-89	2E79	FA13
61E1	75	12			
61E2	75	12			
61E3	75	12			
61E4	75	12	02-15-09	2F29	3B88
61E5	75	12	01-85-80	7570	690A
61E6	75	11	02-14-80	2E70	689A
61E7	75	12	01-85-77	756D	68F0
61E8	75	13	01-85-85	7575	6957
61E9	75	13	01-85-88	7578	6968
61EA	75	14			
61EB	75	13			
61EC	75	14			
61ED	75	13			
61EE	75	13			
61EF	75	15	01-86-02	7622	69E2
61F0	75	13			
61F1	75	13			
61F2	75	13			
61F3	75	14	02-15-25	2F39	69B7
61F4	75	14	02-15-28	2F3C	69C0
61F5	75	14	01-85-92	757C	69AD
61F6	75	14			
61F7	75	14			
61F8	75	14	01-85-90	757A	FA14
61F9	75	15			
61FA	75	14			
61FB	75	14			
61FC	75	14			
61FD	75	15	02-15-35	2F43	2363A
61FE	75	15			
62A1	75	15			
62A2	75	15			
62A3	75	15	02-15-33	2F41	6A30
62A4	75	15			
62A5	75	15	02-15-43	2F4B	6A11
62A6	75	15	02-15-36	2F44	69E3
62A7	75	15			
62A8	75	15			
62A9	75	16	01-86-16	7630	6A6B
62AA	75	16	02-15-54	2F56	6A4E
62AB	75	16			
62AC	75	16			
62AD	75	16			
62AE	75	16	02-15-51	2F53	6A46
62AF	75	16	02-15-55	2F57	6A52
62B0	75	16	01-86-12	762C	6A45
62B1	75	16			
62B2	75	16	01-86-17	7631	6A73
62B3	75	17	02-15-58	2F5A	6A7E
62B4	75	18			
62B5	75	18	02-15-69	2F65	6AD4
62B6	75	18			
62B7	75	18			
62B8	75	19	01-86-24	7638	6AE4
62B9	75	19			
62BA	75	19			
62BB	75	19			
62BC	75	19			
62BD	75	19			
62BE	75	21	01-86-27	763B	F91D
62BF	75	21			
62C0	75	21			
62C1	76	11	01-86-31	763F	6B35
62C2	76	11			
62C3	76	14	02-15-89	2F79	6B4A
62C4	76	16	01-86-33	7641	6B56
62C5	77	7	01-86-35	7643	6B65
62C6	77	13			
62C7	77	13			
62C8	77	13			
62C9	77	14			
62CA	78	9			
62CB	78	19			
62CC	79	10			
62CD	79	13	02-78-05	6E25	6BBE
62CE	79	17			
62CF	80	7	01-86-42	764A	6BCF
62D0	81	9	01-86-43	764B	6BD6
62D1	82	12	02-78-13	6E2D	6BF1
62D2	83	5	01-86-47	764F	6C10
62D3	83	6			
62D4	83	9			
62D5	85	5	02-78-21	6E35	6C3F
62D6	85	6	01-86-50	7652	6C5C
62D7	85	6			
62D8	85	7	01-86-55	7657	6C86
62D9	85	7			
62DA	85	7			
62DB	85	7			
62DC	85	7	01-86-53	7655	6C76
62DD	85	8	01-86-62	765E	6CEB
62DE	85	8	01-86-61	765D	6CE0
62DF	85	8	02-78-38	6E46	6CE9
62E0	85	8	02-78-32	6E40	6CC2
62E1	85	7	02-78-24	6E38	6C6D
62E2	85	9	02-78-36	6E44	6CDA
62E3	85	8	02-78-35	6E43	6CD6
62E4	85	9	01-86-65	7661	6D04
62E5	85	9	01-86-67	7663	6D2E
62E6	85	9	01-86-69	7665	6D39
62E7	85	8	01-86-58	765A	6CFB
62E8	85	9			
62E9	85	10			
62EA	85	10	01-86-73	7669	FA45
62EB	85	10	01-86-75	766B	6D87
62EC	85	10	01-86-74	766A	6D82
62ED	85	10	01-86-72	7668	6D65
62EE	85	10			
62EF	85	11	01-86-83	7673	6DDA
62F0	85	10	01-86-76	766C	6D89
62F1	85	10			
62F2	85	10			
62F3	85	10	02-78-57	6E59	6D8D
62F4	85	10			
62F5	85	10	02-78-55	6E57	6D81
62F6	85	10			
62F7	85	10	02-78-48	6E50	6D2F
62F8	85	10	02-78-53	6E55	6D70
62F9	85	10			
62FA	85	11	01-86-64	7660	23CFE
62FB	85	13	02-79-07	6F27	6ED9
62FC	85	11	01-86-79	766F	6DAC
62FD	85	11			
62FE	85	11			
63A1	85	11	02-78-75	6E6B	6DE9
63A2	85	11			
63A3	85	11			
63A4	85	11	02-78-72	6E68	6DCF
63A5	85	11			
63A6	85	11			
63A7	85	11	02-78-73	6E69	6DD0
63A8	85	11			
63A9	85	11	02-78-71	6E67	6DCE
63AA	85	11	02-78-77	6E6D	6E36
63AB	85	11			
63AC	85	12	01-86-88	7678	6E34
63AD	85	12			
63AE	85	12	02-78-80	6E70	6E27
63AF	85	12			
63B0	85	12	01-86-87	7677	FA46
63B1	85	12	02-78-89	6E79	6E51
63B2	85	12	02-78-90	6E7A	6E53
63B3	85	12			
63B4	85	12	01-86-90	767A	6E5C
63B5	85	12			
63B6	85	12			
63B7	85	12			
63B8	85	12	02-78-83	6E73	6E3C
63B9	85	12			
63BA	85	12	01-86-91	767B	6E5E
63BB	85	12			
63BC	85	6			
63BD	85	12	01-86-86	7676	6DFC
63BE	85	12			
63BF	85	12			
63C0	85	12			
63C1	85	13	01-86-92	767C	6EAB
63C2	85	13			
63C3	85	13	01-86-93	767D	6EB1
63C4	85	13	01-87-01	7721	6EC7
63C5	85	13			
63C6	85	13			
63C7	85	13			
63C8	85	13			
63C9	85	13			
63CA	85	13			
63CB	85	14	01-87-05	7725	FA47
63CC	85	14	02-79-15	6F2F	6F25
63CD	85	14			
63CE	85	14			
63CF	85	14			
63D0	85	14			
63D1	85	14			
63D2	85	14			
63D3	85	14	01-87-08	7728	6F33
63D4	85	14			
63D5	85	14			
63D6	85	15			
63D7	85	15			
63D8	85	15	01-87-15	772F	6F88
63D9	85	15			
63DA	85	15	01-87-12	772C	6F61
63DB	85	15			
63DC	85	15			
63DD	85	15			
63DE	85	16	01-87-20	7734	6FA7
63DF	85	16	02-79-33	6F41	6FB5
63E0	85	16	02-79-31	6F3F	6FAF
63E1	85	16			
63E2	85	16			
63E3	85	17			
63E4	85	17	01-87-27	773B	6FF5
63E5	85	17			
63E6	85	17	02-79-36	6F44	6FDA
63E7	85	17			
63E8	85	18	02-79-42	6F4A	7007
63E9	85	18			
63EA	85	18			
63EB	85	18	01-87-28	773C	7005
63EC	85	18			
63ED	85	18			
63EE	85	18			
63EF	85	19			
63F0	85	19			
63F1	85	19	01-87-30	773E	7028
63F2	85	19			
63F3	85	19			
63F4	85	19			
63F5	85	20			
63F6	85	24	01-87-32	7740	705D
63F7	85	26	01-87-34	7742	704E
63F8	86	8	01-87-37	7745	7085
63F9	86	8			
63FA	86	8			
63FB	86	8	02-79-61	6F5D	7086
63FC	86	9	01-87-38	7746	70A4
63FD	86	9	01-87-39	7747	70AB
63FE	86	9	02-79-64	6F60	70BB
64A1	86	10	02-79-73	6F69	70DC
64A2	86	10	01-87-42	774A	70D8
64A3	86	10	02-79-72	6F68	70D3
64A4	86	11			
64A5	86	11	02-79-75	6F6B	7104
64A6	86	11			
64A7	86	11			
64A8	86	12	01-87-46	774E	711E
64A9	86	12			
64AA	86	12	02-79-81	6F71	712F
64AB	86	12			
64AC	86	13	01-87-53	7755	FA48
64AD	86	13	01-87-55	7757	715C
64AE	86	13	01-87-50	7752	7146
64AF	86	13	01-87-51	7753	7147
64B0	86	13			
64B1	86	14			
64B2	86	14	01-87-54	7756	7152
64B3	86	14	01-87-61	775D	7192
64B4	86	14	01-87-62	775E	71C1
64B5	86	14	02-79-89	6F79	7180
64B6	86	14			
64B7	86	14			
64B8	86	15	01-84-06	7426	7199
64B9	86	15	01-87-58	775A	FA15
64BA	86	15			
64BB	86	15			
64BC	86	15			
64BD	86	15			
64BE	86	16			
64BF	86	16	02-80-03	7023	71CB
64C0	86	16			
64C1	86	16	01-87-63	775F	71BA
64C2	86	18	01-87-65	7761	71FE
64C3	86	18	01-87-66	7762	7200
64C4	86	21			
64C5	87	8			
64C6	90	7			
64C7	90	8	01-87-74	776A	72C0
64C8	90	10			
64C9	90	11			
64CA	91	5			
64CB	91	5			
64CC	94	6	02-80-29	703D	72B1
64CD	94	11			
64CE	94	12	01-87-79	776F	FA16
64CF	94	12			
64D0	94	12			
64D1	94	14	01-87-80	7770	7350
64D2	94	18	02-80-55	7057	7377
64D3	95	9	02-80-57	7059	7385
64D4	95	10			
64D5	96	6			
64D6	96	8	02-80-62	705E	739E
64D7	96	8	02-80-64	7060	73A8
64D8	96	8	02-80-63	705F	73A5
64D9	96	8	01-87-84	7774	739F
64DA	96	8	02-80-61	705D	739C
64DB	96	9	01-87-89	7779	73C9
64DC	96	9	01-87-90	777A	73CF
64DD	96	9	02-80-71	7067	73CB
64DE	96	9			
64DF	96	9			
64E0	96	9			
64E1	96	9	02-80-69	7065	73BF
64E2	96	10	01-87-92	777C	73D9
64E3	96	10	01-87-94	777E	73E9
64E4	96	10	01-87-91	777B	73D6
64E5	96	10	01-87-93	777D	73E3
64E6	96	10	02-80-73	7069	73E7
64E7	96	10	02-80-74	706A	73F9
64E8	96	10			
64E9	96	10			
64EA	96	10			
64EB	96	11	01-88-01	7821	7407
64EC	96	11			
64ED	96	11			
64EE	96	11	02-80-77	706D	7401
64EF	96	11			
64F0	96	11			
64F1	96	11			
64F2	96	11			
64F4	96	12	01-88-05	7825	FA4A
64F5	96	12	01-88-06	7826	7426
64F6	96	12	01-88-08	7828	742A
64F7	96	12	02-80-80	7070	7439
64F8	96	12	01-88-11	782B	742E
64F9	96	12	01-88-04	7824	741B
64FA	96	12	01-88-03	7823	741A
64FB	96	12	01-88-09	7829	742B
64FC	96	12			
64FD	96	12			
64FE	96	12			
65A1	96	12			
65A2	96	13			
65A3	96	13	02-80-83	7073	7443
65A4	96	13	01-88-18	7832	7457
65A5	96	13			
65A6	96	14	01-88-17	7831	744B
65A8	96	14	01-88-19	7833	7462
65A9	96	14			
65AA	96	14			
65AB	96	14	02-80-87	7077	7471
65AC	96	15			
65AD	96	15	01-88-23	7837	7487
65AE	96	15	01-88-24	7838	7489
65AF	96	15			
65B0	96	16	01-88-28	783C	74A3
65B1	96	16	01-88-25	7839	7498
65B2	96	16	01-88-27	783B	749F
65B3	96	16	02-80-94	707E	7499
65B4	96	16			
65B5	96	18	01-88-34	7842	74BF
65B6	97	11			
65B7	97	19			
65B8	98	12	02-81-15	712F	74FA
65B9	98	13	01-88-39	7847	7501
65BA	98	16			
65BB	99	9			
65BC	100	10			
65BD	100	11			
65BE	101	9			
65BF	102	8	02-81-24	7138	7540
65C0	102	10			
65C1	102	8	02-81-23	7137	753E
65C2	102	11			
65C3	102	12	01-88-42	784A	756F
65C4	104	12	01-88-47	784F	75E4
65C5	104	15	01-88-53	7855	7619
65C6	104	16	01-88-55	7857	762D
65C7	104	20	01-88-59	785B	7664
65C8	104	22	01-88-61	785D	766D
65C9	106	7	01-14-51	2E53	34B5
65CA	106	7			
65CB	106	12	02-81-78	716E	7695
65CC	106	12			
65CD	106	12			
65CE	106	15	02-81-79	716F	769C
65CF	106	15	01-88-67	7863	769E
65D0	106	15			
65D1	106	15	01-88-65	7861	769B
65D2	106	18	01-88-68	7864	76A6
65D3	106	18	02-81-82	7172	76A7
65D4	106	19			
65D5	107	14	01-88-70	7866	76B6
65D6	108	7			
65D7	108	8			
65D8	108	10			
65D9	108	13			
65DA	108	15			
65DB	109	8			
65DC	109	9			
65DD	109	10			
65DE	109	10			
65DF	109	10			
65E0	109	10			
65E1	109	10			
65E2	109	11			
65E3	109	11	02-81-93	717D	772D
65E4	109	11			
65E5	109	11			
65E6	109	12	01-88-82	7872	7746
65E7	109	12	01-88-83	7873	774D
65E8	109	12			
65E9	109	13	01-88-85	7875	775C
65EA	109	13			
65EB	109	14			
65EC	109	14			
65ED	109	14	02-82-09	7229	7772
65EE	109	15			
65EF	109	17			
65F0	109	20			
65F1	110	12	02-82-19	7233	77DE
65F2	111	12	02-82-25	7239	4093
65F3	111	12			
65F4	111	14	02-15-22	2F36	6998
65F5	112	6			
65F6	112	7	02-03-36	2344	52AF
65F7	112	10	02-82-35	7243	7821
65F8	112	11	01-89-03	7923	784F
65F9	112	11			
65FA	112	11	01-89-02	7922	784E
65FB	112	11	01-89-04	7924	7851
65FC	112	12	01-89-06	7926	786E
65FD	112	12			
65FE	112	12			
66A1	112	12	02-82-38	7246	7864
66A2	112	12			
66A4	112	13			
66A5	112	13			
66A6	112	13			
66A7	112	14	02-82-45	724D	789E
66A8	112	15			
66A9	112	16	02-82-54	7256	78E6
66AA	112	16			
66AB	112	17	02-82-57	7259	78FB
66AC	112	17	02-82-56	7258	78FA
66AD	112	19			
66AE	112	21	02-82-62	725E	7930
66AF	112	21	02-82-63	725F	7925
66B0	113	6			
66B1	113	6			
66B2	113	7			
66B3	113	8			
66B4	113	8	01-89-19	7933	FA4C
66B5	113	9	01-89-23	7937	FA4E
66B6	113	9	01-89-20	7934	FA4D
66B7	113	9			
66B8	113	9			
66B9	113	10	01-89-26	793A	795C
66BA	113	10	01-89-27	793B	FA51
66BB	113	10	01-89-28	793C	FA19
66BC	113	10	01-89-25	7939	FA50
66BD	113	10	01-89-24	7938	FA4F
66BE	113	10			
66BF	113	10			
66C0	113	10	02-82-66	7262	7958
66C1	113	10			
66C2	113	11	01-89-29	793D	FA1A
66C3	113	11			
66C4	113	11	02-82-68	7264	4105
66C5	113	11			
66C6	113	11			
66C7	113	12	01-91-89	7B79	FA61
66C8	113	12			
66C9	113	12			
66CB	113	13			
66CC	133	13	01-90-57	7A59	4453
66CD	113	13			
66CE	113	13	01-89-30	793E	7979
66CF	113	14	01-89-31	793F	FA52
66D0	113	14	02-82-71	7267	7994
66D1	113	14	01-89-32	7940	FA53
66D2	113	14			
66D3	113	14	01-89-33	7941	FA1B
66D4	113	15	02-82-74	726A	799B
66D5	113	16			
66D6	113	17			
66D7	113	19			
66D8	113	18			
66D9	113	21			
66DA	115	8	01-89-38	7946	79CA
66DB	115	9	02-82-81	7271	79CC
66DC	115	9	02-82-83	7273	79D6
66DD	115	10	01-89-41	7949	79DE
66DE	115	10			
66DF	115	10			
66E0	115	11			
66E1	115	11			
66E2	115	11	02-82-84	7274	4148
66E3	115	11			
66E4	115	12			
66E5	115	13	02-82-89	7279	7A11
66E6	115	14			
66E7	115	14			
66E8	115	14			
66E9	115	14	02-82-93	727D	4163
66EA	115	14			
66EB	115	15	01-89-45	794D	FA54
66EC	115	15	01-89-46	794E	7A39
66ED	115	16			
66EE	115	17	01-94-89	7E79	9FA2
66EF	115	17			
66F0	115	18	02-83-08	7328	7A60
66F1	115	18	02-83-07	7327	7A5F
66F2	115	19	02-83-10	732A	7A6A
66F3	115	19	02-83-09	7329	7A67
66F4	115	21			
66F5	116	5			
66F6	116	8			
66F7	116	9	01-89-49	7951	FA55
66F8	116	11			
66F9	116	12			
66FA	116	12			
66FB	116	13	01-89-51	7953	7AA0
66FC	116	14	01-89-52	7954	259C4
66FD	116	17			
66FE	117	9	02-83-25	7339	7AD1
67A1	117	9			
67A2	117	12	02-83-26	733A	7AE7
67A3	117	12			
67A4	117	13	01-89-56	7958	7AEB
67A5	117	13			
67A6	117	13			
67A7	117	13			
67A8	117	13	02-83-27	733B	7AE8
67A9	117	16			
67AA	117	16			
67AB	118	9	01-89-57	7959	7AFD
67AC	118	10			
67AD	118	10			
67AE	118	11	02-83-34	7342	7B27
67AF	118	12			
67B0	118	12			
67B1	118	12			
67B2	118	13	01-89-64	7960	7B6D
67B3	118	13	02-83-46	734E	7B64
67B4	118	14			
67B5	118	14			
67B6	118	15	01-89-68	7964	FA56
67B7	118	15	01-89-67	7963	7B9E
67B8	118	15			
67B9	118	15			
67BA	118	16			
67BB	118	16	01-89-70	7966	7BD9
67BC	118	16	02-83-61	735D	25C4B
67BD	118	16			
67BE	118	16			
67BF	118	16			
67C0	118	16			
67C1	118	17	02-83-71	7367	7C0F
67C2	118	17			
67C3	118	17			
67C4	118	17			
67C5	118	18	01-89-72	7968	7C31
67C6	118	18			
67C7	118	19	01-89-77	796D	4264
67C8	118	22			
67C9	119	10			
67CA	119	10			
67CB	119	11			
67CC	119	14			
67CD	119	14	02-83-89	7379	7CB7
67CE	119	16			
67CF	119	21			
67D0	120	9	01-89-90	797A	7D08
67D1	120	9	02-84-09	7429	7D03
67D2	120	9	02-84-10	742A	7D09
67D3	120	9			
67D4	120	10	01-89-92	797C	7D1D
67D5	120	10			
67D6	120	10			
67D7	120	10			
67D8	120	10			
67D9	120	10	01-89-91	797B	7D13
67DA	120	11	02-84-19	7433	7D47
67DB	120	11			
67DC	120	11			
67DD	120	11	01-90-02	7A22	7D48
67DE	120	11	02-84-17	7431	7D3E
67DF	120	12			
67E0	120	12			
67E1	120	12	01-90-04	7A24	7D5C
67E2	120	12			
67E3	120	12			
67E4	120	12	01-90-03	7A23	7D53
67E5	120	12			
67E6	120	13	01-90-07	7A27	7D8B
67E7	120	13			
67E8	120	13			
67E9	120	13			
67EA	120	14			
67EB	120	14	01-90-08	7A28	7DA0
67EC	120	14	02-84-39	7447	7DB7
67ED	120	14			
67EE	120	14			
67EF	120	14	02-84-38	7446	7DB6
67F0	120	14			
67F1	120	15	01-90-12	7A2C	7DD6
67F2	120	15	01-90-14	7A2E	FA57
67F3	120	15			
67F4	120	15			
67F5	120	15	01-90-13	7A2D	7DE3
67F6	120	15			
67F7	120	16	01-90-17	7A31	7E11
67F8	120	17	01-90-19	7A33	FA59
67F9	120	17			
67FA	120	18			
67FB	120	18	01-90-21	7A35	7E52
67FC	120	18			
67FD	120	20			
67FE	120	21	01-90-23	7A37	7E8A
68A1	120	21			
68A2	120	22			
68A3	120	22			
68A4	121	8	02-03-45	234D	530B
68A5	121	18	01-90-25	7A39	7F47
68A6	121	17	02-84-70	7466	7F44
68A7	13	8			
68A8	122	14	01-90-26	7A3A	FA5A
68A9	123	9			
68AA	123	9			
68AB	123	9	02-84-84	7474	517B
68AC	123	9			
68AD	123	10			
68AE	123	12			
68AF	123	14			
68B0	123	15			
68B1	123	17			
68B2	124	6			
68B3	124	10			
68B4	124	12			
68B5	124	14			
68B6	124	15			
68B7	124	16	01-90-34	7A42	7FEE
68B8	124	17			
68B9	124	18			
68BA	125	9	01-90-36	7A44	FA5B
68BB	127	9	01-90-37	7A45	8014
68BC	127	12			
68BD	127	14	02-85-08	7528	8024
68BE	128	10			
68BF	128	12			
68C0	128	12			
68C1	128	13			
68C2	128	16			
68C3	128	17			
68C4	131	8	02-03-52	2354	5367
68C5	131	8	02-85-55	7557	81E4
68C6	132	10	01-90-56	7A58	FA5C
68C7	134	11			
68C8	134	13			
68C9	137	13			
68CA	138	6			
68CB	140	5	02-85-89	7579	827F
68CC	140	6			
68CD	140	6	02-85-90	757A	8283
68CE	140	6	02-85-91	757B	828A
68CF	140	7	02-86-03	7623	82BA
68D0	140	7	02-85-93	757D	82A7
68D1	140	7	02-85-94	757E	82A8
68D2	140	7	01-90-69	7A65	82BE
68D3	140	8	02-86-08	7628	8307
68D4	140	8			
68D5	140	8			
68D6	140	8			
68D7	140	8			
68D8	140	8			
68D9	140	8			
68DA	140	8	01-90-76	7A6C	8301
68DB	140	8			
68DC	140	9	01-90-81	7A71	8343
68DD	140	9	01-90-77	7A6D	8362
68DE	140	9			
68DF	140	9			
68E0	140	9			
68E1	140	9			
68E2	140	9			
68E3	140	9			
68E4	140	9			
68E5	140	9			
68E6	140	9			
68E7	140	9			
68E9	140	9	02-86-18	7632	44BE
68EA	140	9			
68EB	140	9			
68EC	140	9			
68ED	140	10			
68EE	140	10			
68EF	140	10			
68F0	140	10			
68F1	140	10			
68F2	140	10	02-86-19	7633	837F
68F3	140	11			
68F4	140	10	01-90-88	7A78	8398
68F5	140	11			
68F6	140	11			
68F7	140	11			
68F8	140	11			
68F9	140	13			
68FA	140	11			
68FB	140	11			
68FC	140	11	01-90-93	7A7D	83C7
68FD	140	11			
68FE	140	11			
69A1	140	12			
69A2	140	11	01-90-92	7A7C	83C0
69A3	140	12	01-91-08	7B28	8448
69A4	140	12	02-86-40	7648	844F
69A5	140	12			
69A6	140	12			
69A7	140	12			
69A8	140	12	01-91-09	7B29	845F
69A9	140	12			
69AA	140	12			
69AB	140	12	01-91-10	7B2A	8470
69AC	140	12	01-91-07	7B27	FA5F
69AD	140	15			
69AE	140	13	02-86-53	7655	84DC
69AF	140	13	01-91-15	7B2F	84B4
69B0	140	13			
69B1	140	13			
69B2	140	13			
69B3	140	13			
69B4	140	13			
69B5	140	14	02-86-55	7657	84A6
69B6	140	14			
69B7	140	14			
69B8	140	14			
69B9	140	15			
69BA	140	15			
69BB	140	15			
69BC	140	15	02-86-75	766B	854F
69BD	140	15	02-86-78	766E	4543
69BE	140	15	02-86-81	7671	8553
69BF	140	15	01-91-24	7B38	8559
69C0	140	16	01-91-30	7B3E	858F
69C1	140	16	01-91-26	7B3A	FA1F
69C2	140	16			
69C3	140	17	02-87-05	7725	85C7
69C4	140	16	02-86-90	767A	8586
69C5	140	17			
69C6	140	17			
69C7	140	17			
69C8	140	17			
69C9	140	17	01-91-32	7B40	85B0
69CA	140	18			
69CB	140	18			
69CC	140	18			
69CD	140	19	01-91-36	7B44	8612
69CE	140	19	01-91-38	7B46	8604
69CF	140	20	01-91-42	7B4A	8618
69D0	140	24	02-87-24	7738	FA20
69D1	141	10	02-87-25	7739	8653
69D2	141	12	01-91-46	7B4E	865B
69D3	141	12	01-91-47	7B4F	F936
69D4	141	15			
69D5	142	8	01-91-49	7B51	459D
69D6	142	10			
69D7	142	11	02-87-37	7745	FA21
69D8	142	14	02-87-59	775B	873E
69D9	142	18			
69DA	142	19			
69DB	142	19			
69DC	142	21			
69DD	144	12			
69DE	144	15			
69DF	144	16			
69E0	145	9			
69E1	145	10			
69E2	145	13			
69E3	145	13	02-01-19	2133	342E
69E4	145	14	01-91-79	7B6F	FA60
69E5	145	14			
69E6	145	14	01-91-77	7B6D	88F5
69E7	145	14			
69E8	145	15	01-91-78	7B6E	891C
69E9	145	15			
69EA	145	16			
69EB	145	17			
69EC	145	18			
69ED	146	11			
69EF	146	19			
69F0	146	25			
69F1	147	11	02-88-39	7847	8990
69F2	147	14			
69F3	147	14			
69F4	147	20			
69F5	147	20			
69F6	147	21			
69F7	148	7			
69F8	148	7			
69F9	148	9	01-91-90	7B7A	89D4
69FA	148	13	01-91-91	7B7B	89E5
69FB	148	12			
69FC	148	17			
69FD	149	10	01-91-94	7B7E	8A15
69FE	149	10	01-91-93	7B7D	8A12
6AA1	149	10	02-88-52	7854	8A11
6AA2	149	11	01-92-01	7C21	8A22
6AA3	149	12	01-92-04	7C24	8A4E
6AA4	149	12			
6AA5	149	12	01-92-02	7C22	8A37
6AA6	149	12	01-92-03	7C23	8A47
6AA7	149	12			
6AA8	149	13			
6AA9	149	13			
6AAA	149	13	01-92-08	7C28	8A79
6AAB	149	14	01-92-09	7C29	8AA7
6AAC	149	14			
6AAD	149	15	02-88-64	7860	8ABE
6AAE	149	15	02-88-65	7861	8AD7
6AAF	149	15			
6AB0	149	15			
6AB1	149	16	01-92-15	7C2F	FA62
6AB2	149	16	01-92-14	7C2E	FA22
6AB3	149	16	01-92-11	7C2B	8ADF
6AB4	149	16	01-92-13	7C2D	8AF6
6AB5	149	16			
6AB6	149	16			
6AB7	149	16			
6AB8	149	16			
6AB9	149	18	01-92-16	7C30	FA63
6ABA	149	17	02-88-71	7867	8B1C
6ABB	149	19	01-92-17	7C31	8B46
6ABC	149	19			
6ABD	149	19			
6ABE	149	19			
6ABF	149	19	01-92-18	7C32	8B54
6AC0	149	20			
6AC1	149	20			
6AC2	149	22			
6AC3	149	22	02-88-79	786F	8B7F
6AC4	149	23			
6AC5	151	28	02-88-94	787E	8C54
6AC6	152	11			
6AC7	152	12			
6AC8	152	15			
6AC9	154	9			
6ACA	154	10	02-89-11	792B	8CA4
6ACB	154	11			
6ACC	154	12			
6ACD	154	12			
6ACE	154	14	01-92-24	7C38	FA64
6ACF	154	16	01-92-26	7C3A	8CF4
6AD0	154	16	02-89-16	7930	8CF0
6AD1	154	16			
6AD2	154	16			
6AD3	154	17			
6AD4	154	19	01-92-29	7C3D	FA65
6AD5	154	21	01-92-30	7C3E	8D12
6AD6	154	24			
6AD7	156	10			
6AD8	156	10			
6AD9	156	11			
6ADA	156	19			
6ADB	157	12			
6ADC	157	17			
6ADD	157	19			
6ADE	157	21			
6ADF	158	17			
6AE0	158	20			
6AE1	159	10			
6AE2	159	11			
6AE3	159	15			
6AE4	159	16			
6AE5	159	20	01-92-48	7C50	8F54
6AE6	160	14			
6AE7	162	6			
6AE8	162	6			
6AE9	162	6			
6AEA	162	7			
6AEB	162	7	02-89-78	796E	FA24
6AEC	162	8			
6AED	162	8			
6AEE	162	9			
6AEF	162	9	01-92-54	7C56	8FEE
6AF0	162	9	01-92-53	7C55	8FE8
6AF1	162	10			
6AF2	162	11			
6AF3	162	10			
6AF4	162	10			
6AF5	162	11			
6AF6	162	11	02-89-88	7978	900C
6AF7	162	11			
6AF8	162	11			
6AF9	162	12			
6AFA	162	12			
6AFB	162	12			
6AFC	162	12			
6AFD	162	13			
6AFE	162	13	02-89-92	797C	9044
6BA1	162	14			
6BA3	162	14			
6BA4	162	14			
6BA5	162	15			
6BA6	162	15			
6BA7	162	15			
6BA8	162	15			
6BA9	162	15			
6BAA	162	15			
6BAB	162	15			
6BAC	162	16			
6BAD	162	16			
6BAE	162	16			
6BAF	162	16			
6BB0	162	16			
6BB1	162	16			
6BB2	162	16			
6BB3	162	16			
6BB4	162	16			
6BB5	162	16			
6BB6	162	17			
6BB7	162	17			
6BB8	162	17			
6BB9	162	17			
6BBA	162	17			
6BBB	162	18			
6BBC	162	18			
6BBD	163	3	02-90-05	7A25	961D
6BBE	163	7			
6BBF	163	7			
6BC0	163	7			
6BC1	163	7			
6BC2	163	7			
6BC3	163	7			
6BC4	163	7	01-92-63	7C5F	90A2
6BC5	163	8	02-90-08	7A28	90B0
6BC6	163	9	01-92-66	7C62	90C4
6BC7	163	10	01-92-71	7C67	90DE
6BC8	163	10			
6BC9	163	11			
6BCA	163	12	01-92-74	7C6A	FA26
6BCB	163	12			
6BCC	163	12			
6BCD	163	13	01-92-76	7C6C	9115
6BCE	163	13			
6BCF	163	13			
6BD0	163	13			
6BD1	163	15	01-92-80	7C70	9127
6BD2	163	15			
6BD3	163	16	02-90-27	7A3B	9137
6BD4	163	18	02-90-29	7A3D	913A
6BD5	164	20			
6BD6	166	9			
6BD7	167	10			
6BD8	167	10	01-92-92	7C7C	91D7
6BD9	167	10	02-90-46	7A4E	91DA
6BDA	167	10			
6BDB	167	11	02-90-49	7A51	91EC
6BDC	167	11	01-93-02	7D22	91ED
6BDD	167	11	01-93-03	7D23	91F1
6BDE	167	11			
6BDF	167	11	01-93-01	7D21	91E5
6BE0	167	11	01-92-94	7C7E	91E4
6BE1	167	11	02-90-50	7A52	91EE
6BE2	167	11			
6BE3	167	11			
6BE4	167	12	02-90-51	7A53	9201
6BE5	167	12			
6BE6	167	12			
6BE7	167	12	02-90-52	7A54	920A
6BE8	167	12			
6BE9	167	12	01-93-05	7D25	9210
6BEA	167	12	02-90-53	7A55	9216
6BEB	167	12			
6BEC	167	13	02-90-61	7A5D	9251
6BED	167	13	01-93-07	7D27	9239
6BEE	167	13	01-93-10	7D2A	9240
6BEF	167	13	02-90-60	7A5C	924E
6BF0	167	13	02-90-59	7A5B	924A
6BF1	167	13	02-90-63	7A5F	9259
6BF2	167	13	01-93-08	7D28	923A
6BF3	167	13	01-93-09	7D29	923C
6BF4	167	13	02-90-57	7A59	9242
6BF5	167	13	02-90-62	7A5E	9256
6BF6	167	13	02-90-67	7A63	9267
6BF7	167	13	02-90-64	7A60	9260
6BF8	167	13	02-90-56	7A58	9233
6BF9	167	13			
6BFA	167	13			
6BFB	167	13	01-93-06	7D26	9238
6BFC	167	13			
6BFD	167	13	02-90-66	7A62	9265
6BFE	167	13			
6CA1	167	13			
6CA2	167	13			
6CA3	167	14	01-93-14	7D2E	9288
6CA4	167	14			
6CA5	167	14	01-93-13	7D2D	9278
6CA6	167	14	02-90-79	7A6F	92A7
6CA7	167	14	02-90-72	7A68	927D
6CA8	167	14			
6CA9	167	14	02-90-74	7A6A	9289
6CAA	167	14			
6CAB	167	15			
6CAC	167	15	02-90-89	7A79	92D7
6CAD	167	15	02-90-91	7A7B	92E5
6CAE	167	15			
6CAF	167	15			
6CB0	167	15	01-93-19	7D33	92E0
6CB1	167	15	02-90-92	7A7C	92E7
6CB2	167	15	02-90-87	7A77	92CE
6CB3	167	15	02-90-88	7A78	92D0
6CB4	167	15			
6CB5	167	15			
6CB6	167	15			
6CB7	167	15			
6CB8	167	15	02-90-90	7A7A	92D9
6CB9	167	15	01-93-16	7D30	92CB
6CBA	167	15			
6CBB	167	16	02-91-03	7B23	92F9
6CBC	167	16	01-93-22	7D36	931F
6CBD	167	16	02-91-10	7B2A	9327
6CBE	167	16	01-93-23	7D37	9321
6CBF	167	16	02-91-09	7B29	931E
6CC0	167	16	01-93-21	7D35	9304
6CC1	167	16	01-93-20	7D34	92FF
6CC2	167	16			
6CC3	167	16	01-93-24	7D38	9325
6CC4	167	16	02-91-08	7B28	931D
6CC5	167	16	02-91-05	7B25	9302
6CC6	167	16	02-91-11	7B2B	9329
6CC7	167	16	01-93-25	7D39	9348
6CC8	167	17			
6CC9	167	17	01-93-27	7D3B	934A
6CCA	167	17	01-93-31	7D3F	9370
6CCB	167	17	02-91-18	7B32	936B
6CCC	167	17	02-91-16	7B30	9357
6CCD	167	17	02-91-04	7B24	92FB
6CCE	167	18			
6CCF	167	17	01-93-26	7D3A	9349
6CD0	167	17	02-91-17	7B31	935A
6CD1	167	17			
6CD2	167	17			
6CD3	167	17			
6CD4	167	17			
6CD5	167	18	02-91-20	7B34	9373
6CD6	167	18			
6CD7	167	18			
6CD8	167	18			
6CD9	167	18	02-91-24	7B38	9388
6CDA	167	18			
6CDB	167	18			
6CDC	167	18			
6CDD	167	18			
6CDE	167	19	01-93-37	7D45	93DF
6CDF	167	19	01-93-36	7D44	93DE
6CE0	167	19	01-93-35	7D43	93C6
6CE1	167	19			
6CE2	167	19			
6CE3	167	19			
6CE4	167	19			
6CE5	167	20	02-91-38	7B46	940F
6CE6	167	20			
6CE7	167	20			
6CE8	167	20			
6CE9	167	20			
6CEA	167	20			
6CEB	167	20	02-91-39	7B47	9416
6CEC	167	20			
6CED	167	21	02-91-41	7B49	93FB
6CEE	167	21	01-93-40	7D48	9433
6CEF	167	21			
6CF0	167	22	02-91-45	7B4D	9445
6CF1	167	22			
6CF2	167	22			
6CF3	167	23			
6CF4	167	23			
6CF5	167	24	01-93-43	7D4B	946B
6CF6	167	25	02-91-48	7B50	946D
6CF7	169	12	01-93-46	7D4E	958E
6CF8	169	12			
6CF9	169	15	01-93-50	7D52	95AC
6CFA	169	15			
6CFB	169	16			
6CFC	169	17			
6CFD	169	19	01-93-56	7D58	49B0
6CFE	170	7	02-91-63	7B5F	9633
6DA1	170	12	02-91-70	7B66	9682
6DA2	170	12	01-93-61	7D5D	F9DC
6DA3	170	13			
6DA4	170	13			
6DA5	170	14	01-93-62	7D5E	969D
6DA6	170	17	02-91-76	7B6C	96AF
6DA7	172	12			
6DA8	172	16	02-91-81	7B71	28FF8
6DA9	172	17			
6DAA	172	18	01-93-66	7D62	96DE
6DAB	172	19	01-93-67	7D63	FA68
6DAC	173	11			
6DAD	173	12	01-93-69	7D65	96EF
6DAE	173	24			
6DAF	173	25			
6DB0	173	27	01-93-72	7D68	974D
6DB1	174	8			
6DB2	174	13	02-91-92	7B7C	9755
6DB3	174	15			
6DB4	174	15			
6DB5	175	15			
6DB6	177	18			
6DB7	177	20			
6DB8	177	21			
6DB9	177	23			
6DBA	180	22	01-93-86	7D76	FA69
6DBB	181	13	02-92-20	7C34	980E
6DBC	181	13	01-93-87	7D77	980A
6DBD	181	14			
6DBE	181	16			
6DBF	181	18			
6DC0	181	19	01-94-04	7E24	F9D0
6DC1	181	19	01-94-01	7E21	9857
6DC2	181	21	01-94-05	7E25	9865
6DC3	182	7			
6DC4	182	19	02-92-39	7C47	98BF
6DC5	184	13			
6DC6	184	13			
6DC7	184	15			
6DC8	184	17			
6DC9	184	17			
6DCA	184	19	02-92-68	7C64	9940
6DCB	184	22			
6DCC	186	16	01-94-11	7E2B	999E
6DCD	186	16			
6DCE	186	17	02-92-76	7C6C	99A3
6DCF	187	7			
6DD0	187	14	02-92-78	7C6E	99B9
6DD1	187	14			
6DD2	187	16			
6DD3	187	18			
6DD4	189	11			
6DD5	187	18			
6DD6	187	19	01-94-14	7E2E	9A24
6DD7	187	19			
6DD8	187	20	01-94-18	7E32	9A4A
6DD9	187	22	01-94-19	7E33	9A4E
6DDA	188	26			
6DDB	189	13	02-93-17	7D31	4BE8
6DDC	189	18	02-93-18	7D32	9ADC
6DDD	190	10			
6DDE	190	15			
6DDF	195	10			
6DE0	195	15	01-94-35	7E43	9B72
6DE1	195	15			
6DE2	195	16	01-94-39	7E47	9B8F
6DE3	195	16			
6DE4	195	16			
6DE5	195	17	02-93-42	7D4A	9BB1
6DE6	195	17			
6DE7	195	18			
6DE8	195	19			
6DE9	195	19			
6DEA	195	20			
6DEB	195	22			
6DEC	196	8			
6DED	196	15			
6DEE	196	15	02-94-03	7E23	9D0B
6DEF	196	15			
6DF0	196	16			
6DF1	196	19			
6DF2	196	19	02-94-20	7E34	9D6B
6DF3	196	21			
6DF4	196	22			
6DF5	196	29			
6DF6	198	13			
6DF7	198	20			
6DF8	199	15			
6DF9	199	15			
6DFA	199	16			
6DFB	200	14			
6DFC	200	15			
6DFD	200	18			
6DFE	201	12			
6EA1	203	12	01-94-82	7E72	9ED1
6EA2	203	17			
6EA3	203	20			
6EA4	205	13			
6EA5	210	15			
6EA6	212	15			
6EA7	212	16			
6EA8	212	16			
6EA9	212	18			
6EAA	212	20			
6EAB	212	22	01-94-87	7E77	9F94
6EAC	214	28	02-94-47	7E4F	9E19
6EB0	9	9	01-14-26	2E3A	4FE0
6EB1	9	10			
6EB2	9	13			
6EB3	9	13			
6EB4	9	18			
6EB5	10	8			
6EB6	14	10			
6EB7	15	10			
6EB8	18	10	01-15-94	2F7E	525D
6EB9	26	12			
6EBA	27	13			
6EBB	29	11			
6EBC	30	11	01-15-08	2F28	555E
6EBD	30	11			
6EBE	30	11			
6EBF	30	15	01-84-07	7427	5653
6EC0	30	15			
6EC1	30	15			
6EC2	30	15			
6EC3	30	18	01-15-26	2F3A	5699
6EC4	30	20			
6EC5	30	21			
6EC6	30	22	01-15-32	2F40	56CA
6EC7	32	11			
6EC8	32	12			
6EC9	32	13	01-15-56	2F58	5861
6ECA	32	13			
6ECB	40	11			
6ECC	43	7			
6ECD	44	11	01-94-91	7E7B	5C5B
6ECE	44	12			
6ECF	44	14	01-47-64	4F60	5C62
6ED0	49	9			
6ED1	50	14			
6ED2	53	8			
6ED3	53	14			
6ED4	53	15			
6ED5	57	11			
6ED6	61	9			
6ED7	61	10			
6ED8	61	13			
6ED9	64	10			
6EDA	64	11			
6EDB	64	11			
6EDC	64	12			
6EDD	64	13	01-84-86	7476	6414
6EDE	64	13			
6EDF	64	14	01-84-89	7479	6451
6EE0	64	15			
6EE1	64	21			
6EE2	64	22	01-85-06	7526	6522
6EE3	72	16			
6EE4	72	18			
6EE5	75	9			
6EE6	75	9			
6EE7	75	11			
6EE8	75	11			
6EE9	75	13			
6EEA	75	14			
6EEB	75	14			
6EEC	75	15	01-86-06	7626	69F6
6EED	75	17	01-86-20	7634	6A94
6EEE	75	20	02-15-78	2F6E	6AF3
6EEF	75	26			
6EF0	85	12			
6EF1	85	14			
6EF2	85	14			
6EF3	85	15			
6EF4	85	15	01-87-09	7729	6F51
6EF5	85	18	01-87-29	773D	7006
6EF6	85	19			
6EF7	85	19			
6EF8	85	19			
6EF9	85	22			
6EFA	86	12	01-87-49	7751	7130
6EFB	86	13			
6EFC	86	13			
6EFD	94	13			
6EFE	96	9			
6FA1					
6FA2					
6FA3					
6FA4					
6FA5			01-02-52	2254	2225
6FA6					
6FA7					
6FA8					
6FA9					
6FAA					
6FAB					
6FAC			01-02-35	2243	2285
6FAD					
6FAE					
6FAF			01-02-76	226C	2243
6FB0			01-02-77	226D	2245
6FB1			01-03-59	235B	2213
6FB2			01-02-80	2270	2277
6FB3			01-07-87	2777	22DB
6FB4					
6FB5			01-06-77	266D	25B1
6FB6			01-02-78	226E	2248
6FB7			01-13-88	2D78	221F
6FB8			01-13-83	2D73	222E
6FB9					
6FBA			01-13-89	2D79	22BF
6FBB					
6FBC					
6FBD					
6FBE					
6FBF					
6FC0					
6FC1					
6FC2					
6FC3					
6FC4					
6FC5					
6FC6					
6FC7					
6FC8					
6FC9					
6FCA			01-13-50	2D52	339E
6FCB			01-13-49	2D51	339D
6FCC			01-13-48	2D50	339C
6FCD					
6FCE					
6FCF					
6FD0			01-13-54	2D56	33A1
6FD1					
6FD2					
6FD3					
6FD4					
6FD5					
6FD6					
6FD7					
6FD8					
6FD9			01-13-52	2D54	338F
6FDA			01-13-51	2D53	338E
6FDB					
6FDC					
6FDD					
6FDE					
6FDF					
6FE0					
6FE1					
6FE2					
6FE3					
6FE4					
6FE5					
6FE6					
6FE7					
6FE8					
6FE9					
6FEA					
6FEB					
6FEC					
6FED					
6FEE					
6FEF					
6FF0					
6FF1					
6FF2					
6FF3					
6FF4					
6FF5					
6FF6					
6FF7					
6FF8					
6FF9					
6FFA			01-13-53	2D55	33C4
6FFB					
6FFC					
6FFD					
6FFE					
70A1					
70A2					
70A3					
70A4					
70A5					
70A6					
70A7			01-03-62	235E	33CB
70A8					
70A9					
70AA					
70AB					
70AC					
70AD					
70AE					
70AF					
70B0					
70B1					
70B2					
70B3					
70B4					
70B5					
70B6					
70B7					
70B8					
70B9					
70BA					
70BB					
70BC					
70BD					
70BE					
70BF					
70C0					
70C1					
70C2					
70C3					
70C4					
70C5					
70C6					
70C7					
70C8					
70C9					
70CA					
70CB					
70CC					
70CD					
70CE					
70CF			01-13-32	2D40	3349
70D0			01-13-34	2D42	3322
70D1			01-13-33	2D41	3314
70D2					
70D3			01-13-35	2D43	334D
70D4					
70D5			01-13-36	2D44	3318
70D6					
70D7			01-13-37	2D45	3327
70D8			01-13-38	2D46	3303
70D9			01-13-39	2D47	3336
70DA			01-13-46	2D4E	334A
70DB			01-13-41	2D49	3357
70DC			01-13-42	2D4A	330D
70DD					
70DE					
70DF			01-13-47	2D4F	333B
70E0			01-13-45	2D4D	332B
70E1			01-13-44	2D4C	3323
70E2			01-13-43	2D4B	3326
70E3					
70E4					
70E5					
70E6					
70E7					
70E8					
70E9			01-03-09	2329	21C4
70EA			01-02-81	2271	2194
70EB					
70EC			01-03-08	2328	2199
70ED			01-03-05	2325	2197
70EE			01-03-06	2326	2198
70EF			01-03-07	2327	2196
70F0					
70F1			01-03-28	233C	303D
70F2			01-06-31	263F	2667
70F3			01-06-29	263D	2661
70F4			01-06-25	2639	2664
70F5			01-06-27	263B	2662
70F6			01-06-32	2640	2663
70F7			01-06-30	263E	2665
70F8			01-06-26	263A	2660
70F9			01-06-28	263C	2666
70FA			01-03-03	2323	25C1
70FB			01-03-01	2321	25B7
70FC			01-03-04	2324	25C0
70FD			01-03-02	2322	25B6
70FE					
71A1					
71A2					
71A3					
71A4			01-06-70	2666	3020
71A5					
71A6					
71A7					
71A8			01-03-10	232A	21E8
71A9			01-03-11	232B	21E6
71AA			01-03-12	232C	21E7
71AB			01-03-13	232D	21E9
71AC					
71AD					
71AE					
71AF					
71B0					
71B1					
71B2					
71B3			01-02-51	2253	2297
71B4			01-02-49	2251	2295
71B5			01-02-50	2252	2296
71B6					
71B7			01-02-39	2247	2205
71B8			01-03-27	233B	25C9
71B9					
71BA					
71BB					
71BC					
71BD					
71BE					
71C1					
71C2					
71C3					
71C4					
71C5					
71C6					
71C7					
71C8					
71C9					
71CA					
71CB					
71CC					
71CD					
71CE					
71CF					
71D0			01-03-63	235F	2113
72A1					
72A2					
72A3					
72A4					
72A5					
72A6					
72A7					
72A8					
72A9					
72AA					
72AB					
72AC					
72AD					
72AE					
72AF					
72B0					
72B1					
72B2					
72B3					
72B4					
72B5					
72B6			01-13-67	2D63	33CD
72B7					
72B8					
72B9					
72BA					
72BB					
72BC					
72BD					
72BE			01-13-66	2D62	2116
72BF					
72C0					
72C1					
72C2					
72C3					
72C4					
72C5					
72C6					
72C7					
72C8					
72C9					
72CA					
72CB					
72CC					
72CD					
72CE					
72CF					
72D0					
72D1					
72D2					
72D3					
72D4					
72D5					
72D6					
72D7					
72D8					
72D9					
72DA					
72DB					
72DC					
72DD					
72DE					
72DF					
72E0					
72E1					
72E2					
72E3					
72E4					
72E5					
72E6					
72E7					
72E8					
72E9					
72EA					
72EB					
72EC					
72ED					
72EE					
72EF					
72F0					
72F1					
72F2					
72F3					
72F4					
72F5			01-13-77	2D6D	337E
72F6			01-13-78	2D6E	337D
72F7			01-13-79	2D6F	337C
72F8					
72F9					
72FA					
72FB					
72FC					
72FD					
72FE					
73A1					
73A2					
73A3					
73A4					
73A5					
73A6					
73A7					
73A8					
73A9					
73AA					
73AB					
73AC					
73AD					
73AE					
73AF					
73B0					
73B1			01-13-75	2D6B	3232
73B2					
73B3					
73B4					
73B5					
73B6					
73B7					
73B8					
73B9					
73BA					
73BB					
73BC					
73BD			01-13-76	2D6C	3239
73BE					
73BF					
73C0					
73C1					
73C2					
73C3					
73C4					
73C5					
73C6					
73C7					
73C8					
73C9					
73CA					
73CB					
73CC					
73CD					
73CE					
73CF					
73D0			01-13-74	2D6A	3231
73D1					
73D2					
73D3					
73D4					
73D5					
73D6					
73D7					
73D8					
73D9					
73DA					
73DB					
73DC					
73DD					
73DE					
73DF					
73E0					
73E1					
73E2					
73E3					
73E4					
73E5					
73E6					
73E7					
73E8					
73E9					
73EA			01-13-69	2D65	32A4
73EB			01-13-70	2D66	32A5
73EC			01-13-71	2D67	32A6
73ED			01-13-72	2D68	32A7
73EE			01-13-73	2D69	32A8
73EF					
73F0					
73F1					
73F2					
73F3					
73F4					
73F5					
73F6					
73F7					
73F8			01-13-63	2D5F	337B
73F9					
74B1			01-13-21	2D35	2160
74B2			01-13-22	2D36	2161
74B3			01-13-23	2D37	2162
74B4			01-13-24	2D38	2163
74B5			01-13-25	2D39	2164
74B6			01-13-26	2D3A	2165
74B7			01-13-27	2D3B	2166
74B8			01-13-28	2D3C	2167
74B9			01-13-29	2D3D	2168
74BA			01-13-30	2D3E	2169
74BB			01-13-31	2D3F	216A
74BC			01-13-55	2D57	216B
74C1			01-12-21	2C35	2170
74C2			01-12-22	2C36	2171
74C3			01-12-23	2C37	2172
74C4			01-12-24	2C38	2173
74C5			01-12-25	2C39	2174
74C6			01-12-26	2C3A	2175
74C7			01-12-27	2C3B	2176
74C8			01-12-28	2C3C	2177
74C9			01-12-29	2C3D	2178
74CA			01-12-30	2C3E	2179
74CB			01-12-31	2C3F	217A
74CC			01-12-32	2C40	217B
74D1					
74D2					
74D3					
74D4					
74D5					
74D6					
74D7					
74D8					
74D9					
74DA					
74E1					
74E2					
74E3					
74E4					
74E5					
74E6					
74E7					
74E8					
74E9					
74EA					
74F1					
74F2					
74F3					
74F4					
74F5					
74F6					
74F7					
74F8					
74F9					
74FA					
75B1					
75B2					
75B3					
75B4					
75B5					
75B6					
75B7					
75B8					
75B9					
75BA					
75C1					
75C2					
75C3					
75C4					
75C5					
75C6					
75C7					
75C8					
75C9					
75CA					
75CB					
75CC					
75CD					
75CE					
75CF					
75D0					
75D1					
75D2					
75D3					
75D4					
75D5					
75D6					
75D7					
75D8					
75D9					
75DA					
75E1					
75E2					
75E3					
75E4					
75E5					
75E6					
75E7					
75E8					
75E9					
75EA					
75EB					
75EC					
75ED					
75EE					
75EF					
75F0					
75F1					
75F2					
75F3					
75F4					
75F5					
75F6					
75F7					
75F8					
75F9					
75FA					
76B1			01-13-01	2D21	2460
76B2			01-13-02	2D22	2461
76B3			01-13-03	2D23	2462
76B4			01-13-04	2D24	2463
76B5			01-13-05	2D25	2464
76B6			01-13-06	2D26	2465
76B7			01-13-07	2D27	2466
76B8			01-13-08	2D28	2467
76B9			01-13-09	2D29	2468
76BA			01-13-10	2D2A	2469
76C1					
76C2					
76C3					
76C4					
76C5					
76C6					
76C7					
76C8					
76C9					
76CA					
76CB					
76CC					
76CD					
76CE					
76CF					
76D0					
76D1					
76D2					
76D3					
76D4					
76D5					
76D6					
76D7					
76D8					
76D9					
76DA					
76E1			01-12-33	2C41	24D0
76E2			01-12-34	2C42	24D1
76E3			01-12-35	2C43	24D2
76E4			01-12-36	2C44	24D3
76E5			01-12-37	2C45	24D4
76E6			01-12-38	2C46	24D5
76E7			01-12-39	2C47	24D6
76E8			01-12-40	2C48	24D7
76E9			01-12-41	2C49	24D8
76EA			01-12-42	2C4A	24D9
76EB			01-12-43	2C4B	24DA
76EC			01-12-44	2C4C	24DB
76ED			01-12-45	2C4D	24DC
76EE			01-12-46	2C4E	24DD
76EF			01-12-47	2C4F	24DE
76F0			01-12-48	2C50	24DF
76F1			01-12-49	2C51	24E0
76F2			01-12-50	2C52	24E1
76F3			01-12-51	2C53	24E2
76F4			01-12-52	2C54	24E3
76F5			01-12-53	2C55	24E4
76F6			01-12-54	2C56	24E5
76F7			01-12-55	2C57	24E6
76F8			01-12-56	2C58	24E7
76F9			01-12-57	2C59	24E8
76FA			01-12-58	2C5A	24E9
77A1			01-13-11	2D2B	246A
77A2			01-13-12	2D2C	246B
77A3			01-13-13	2D2D	246C
77A4			01-13-14	2D2E	246D
77A5			01-13-15	2D2F	246E
77A6			01-13-16	2D30	246F
77A7			01-13-17	2D31	2470
77A8			01-13-18	2D32	2471
77A9			01-13-19	2D33	2472
77AA			01-13-20	2D34	2473
77AB			01-08-33	2841	3251
77AC			01-08-34	2842	3252
77AD			01-08-35	2843	3253
77AE			01-08-36	2844	3254
77AF			01-08-37	2845	3255
77B0			01-08-38	2846	3256
77B1			01-08-39	2847	3257
77B2			01-08-40	2848	3258
77B3			01-08-41	2849	3259
77B4			01-08-42	284A	325A
77B5			01-08-43	284B	325B
77B6			01-08-44	284C	325C
77B7			01-08-45	284D	325D
77B8			01-08-46	284E	325E
77B9			01-08-47	284F	325F
77BA			01-08-48	2850	32B1
77BB			01-08-49	2851	32B2
77BC			01-08-50	2852	32B3
77BD			01-08-51	2853	32B4
77BE			01-08-52	2854	32B5
77BF			01-08-53	2855	32B6
77C0			01-08-54	2856	32B7
77C1			01-08-55	2857	32B8
77C2			01-08-56	2858	32B9
77C3			01-08-57	2859	32BA
77C4			01-08-58	285A	32BB
77C5			01-08-59	285B	32BC
77C6			01-08-60	285C	32BD
77C7			01-08-61	285D	32BE
77C8			01-08-62	285E	32BF
77C9					
77CA					
77CB					
77CC					
77CD					
77CE					
77CF					
77D0					
77D1					
77D2					
77D3					
77D4					
77D5					
77D6					
77D7					
77D8					
77D9					
77DA					
77DB					
77DC					
77DD					
77DE					
77DF					
77E0					
77E1					
77E2					
77E3					
77E4					
77E5					
77E6					
77E7					
77E8					
77E9					
77EA					
77EB					
77EC					
77ED					
77EE					
77EF					
77F0					
77F1					
77F2					
77F3					
77F4					
77F5					
77F6					
77F7					
77F8					
77F9					
78A1					
78A2					
78A3					
78A4					
78A5					
78A6					
78A7					
78A8					
78A9					
78AA					
78AB					
78AC					
78AD					
78AE					
78AF					
78B0					
78B1					
78B2					
78B3					
78B4					
78B5					
78B6					
78B7					
78B8					
78B9					
78BA					
78BB					
78BC					
78BD					
78BE					
78BF					
78C0					
78C1					
78C2					
78C3					
78C4					
78C5					
78C6					
78C7					
78C8					
78C9					
78CA					
78CB					
78CC					
78CD					
78CE					
78CF					
78D0					
79A1			01-12-59	2C5B	32D0
79A2			01-12-60	2C5C	32D1
79A3			01-12-61	2C5D	32D2
79A4			01-12-62	2C5E	32D3
79A5			01-12-63	2C5F	32D4
79A6			01-12-64	2C60	32D5
79A7			01-12-65	2C61	32D6
79A8			01-12-66	2C62	32D7
79A9			01-12-67	2C63	32D8
79AA			01-12-68	2C64	32D9
79AB			01-12-69	2C65	32DA
79AC			01-12-70	2C66	32DB
79AD			01-12-71	2C67	32DC
79AE			01-12-72	2C68	32DD
79AF			01-12-73	2C69	32DE
79B0			01-12-74	2C6A	32DF
79B1			01-12-75	2C6B	32E0
79B2			01-12-76	2C6C	32E1
79B3			01-12-77	2C6D	32E2
79B4			01-12-78	2C6E	32E3
79B5					
79B6			01-12-81	2C71	32E5
79B7					
79B8					
79B9					
79BA			01-12-80	2C70	32E9
79BB					
79BC					
79BD			01-12-83	2C73	32EC
79BE			01-12-82	2C72	32ED
79BF					
79C0					
79C1					
79C2					
79C3					
79C4					
79C5					
79C6					
79C7					
79C8					
79C9					
79CA					
79CB			01-12-79	2C6F	32FA
79CC					
79CD					
79CE					
79CF					
79D0					
7AA2					
7AA4					
7AA6					
7AA8					
7AAA					
7AAB					
7AAD					
7AAF					
7AB1					
7AB3					
7AB5					
7AB7					
7AB9					
7ABB					
7ABD					
7ABF					
7AC1					
7AC4					
7AC6					
7AC8					
7ACA					
7ACB					
7ACC					
7ACD					
7ACE					
7ACF					
7AD2					
7AD5					
7AD8					
7ADB					
7ADE					
7ADF					
7AE0					
7AE1					
7AE2					
7AE4					
7AE6					
7AE8					
7AE9					
7AEA					
7AEB					
7AEC					
7AED					
7AEF					
7AF0					
7AF1					
7AF2					
7AF3					
7BA2					
7BA4					
7BA6					
7BA8					
7BAA					
7BAB					
7BAD					
7BAF					
7BB1					
7BB3					
7BB5					
7BB7					
7BB9					
7BBB					
7BBD					
7BBF					
7BC1					
7BC4					
7BC6					
7BC8					
7BCA					
7BCB					
7BCC					
7BCD					
7BCE					
7BCF					
7BD2					
7BD5					
7BD8					
7BDB					
7BDE					
7BDF					
7BE0					
7BE1					
7BE2					
7BE4					
7BE6					
7BE8					
7BE9					
7BEA					
7BEB					
7BEC					
7BED					
7BEF					
7BF0					
7BF1					
7BF2					
7BF3					
7CA2					
7CA4					
7CA6					
7CA8					
7CAB					
7CAD					
7CAF					
7CB1					
7CB3					
7CB5					
7CB7					
7CB9					
7CBB					
7CBD					
7CBF					
7CC1					
7CC4					
7CC6					
7CC8					
7CCA					
7CCB					
7CCC					
7CCD					
7CCE					
7CCF					
7CD2					
7CD5					
7CD8					
7CDB					
7CDE					
7CDF					
7CE0					
7CE1					
7CE2					
7CE4					
7CE8					
7CE9					
7CEA					
7CEB					
7CEC					
7CED					
7CEF					
7CF0					
7CF2					
7EAB					
7EB1					
7EB9					
7EC1					
7ECA					
7ECB					
7ECE					
7ECF					
7ED2					
7ED8					
7EDB					
7EDE					
80A1	98	16			
80A2	98	17			
80A3	104	15	01-94-93	7E7D	7626
80A4	112	13			
80A5	113	9			
80A6	113	19	01-89-35	7943	79B1
80A7	115	14			
80A8	115	21			
80A9	116	12			
80AA	116	18			
80AB	118	14			
80AC	118	15			
80AD	118	18	01-89-73	7969	7C1E
80AE	118	19			
80AF	119	9			
80B0	120	13			
80B1	120	14			
80B2	120	17			
80B3	120	17			
80B4	120	19	01-94-94	7E7E	7E6B
80B5	120	19	01-90-22	7A36	7E61
80B6	121	24			
80B7	127	16			
80B8	128	14			
80B9	128	17			
80BA	128	17			
80BB	128	17			
80BC	128	22			
80BD	130	12			
80BE	130	14			
80BF	140	9			
80C0	140	10			
80C1	140	11			
80C2	140	11			
80C3	140	12			
80C4	140	11	01-91-06	7B26	840A
80C5	140	13			
80C6	140	14	01-91-22	7B36	8523
80C7	140	15			
80C8	140	15			
80C9	140	18			
80CA	140	18			
80CB	140	20			
80CC	142	15			
80CD	142	16			
80CE	142	18	01-91-66	7B62	87EC
80CF	142	19			
80D0	142	21	01-91-71	7B67	881F
80D1	145	18			
80D2	147	17			
80D3	149	16			
80D4	149	17			
80D5	154	16			
80D6	158	18	01-92-42	7C4A	8EC0
80D7	162	7			
80D8	162	7			
80D9	162	9			
80DA	162	9			
80DB	162	11			
80DC	162	11			
80DD	162	13			
80DE	162	13			
80DF	162	14			
80E0	162	14			
80E1	113	8			
80E2	163	15			
80E3	164	9			
80E4	164	18	01-92-89	7C79	91AC
80E5	164	19	01-92-90	7C7A	91B1
80E6	167	16			
80E7	167	22			
80E8	170	14			
80E9	177	12			
80EA	177	12			
80EB	177	14			
80EC	181	16	01-93-90	7D7A	9830
80ED	181	19	01-94-03	7E23	985A
80EE	184	14			
80EF	184	15			
80F0	187	22	01-94-20	7E34	9A52
80F1	195	19			
80F2	195	23			
80F3	196	18			
80F4	196	22	01-94-69	7E65	9DD7
80F5	197	24	01-94-74	7E6A	9E7C
80F6	199	15			
80F7	199	19	01-94-79	7E6F	9EB4
80F8	199	20	01-94-80	7E70	9EB5
9FA1	9	6	01-14-11	2E2B	4F03
9FA2	9	6	01-14-12	2E2C	4F0B
9FA3	18	6	01-14-61	2E5D	5215
9FA4	18	11			
9FA5	32	7	02-04-67	2463	21274
9FA6	32	10	01-15-46	2F4E	57C7
9FA7	32	11	02-04-86	2476	57DE
9FA8	46	11	01-47-79	4F6F	37E2
9FA9	46	13	01-47-87	4F77	5D6D
9FAA	64	6	01-84-70	7466	625A
9FAB	75	8	01-85-51	7553	677B
9FAC	75	9	01-85-58	755A	67C0
9FAD	75	9			
9FAE	75	13	01-85-83	7573	6935
9FAF	75	13	01-85-89	7579	6980
9FB0	75	14	02-15-19	2F33	6992
9FB1	75	16			
9FB2	75	17	01-86-22	7636	6A9E
9FB3	75	19			
9FB4	85	9	02-78-30	6E3E	6C97
9FB5	85	12	02-78-92	6E7C	6E57
9FB6	85	13	02-79-04	6F24	6EBF
9FB7	85	15	01-87-17	7731	6F8D
9FB8	85	15			
9FB9	86	11	01-87-44	774C	710F
9FBA	94	7	01-87-73	7769	72BE
9FBB	96	9			
9FBC	102	15	02-81-34	7142	757E
9FBD	106	7			
9FBE	112	10			
9FBF	112	13			
9FC0	115	17	01-89-47	794F	7A5D
9FC1	119	12	01-89-84	7974	7CA0
9FC2	119	17	01-89-87	7977	7CD9
9FC3	133	13			
9FC4	140	6	01-90-62	7A5E	828E
9FC5	140	9	02-86-14	762E	8330
9FC6	140	12	02-86-44	764C	845A
9FC7	140	16	01-91-27	7B3B	85AD
9FC8	142	19	02-87-92	777C	880B
9FC9	146	12			
9FCA	167	15	01-93-18	7D32	92D3
9FCB	167	18	01-93-34	7D42	93BA
9FCC	169	12	02-91-54	7B56	958D
9FCD	173	20	01-93-70	7D66	9733
9FCE	173	21	01-93-71	7D67	973B
9FCF	173	29	01-93-74	7D6A	974F
9FD0	191	20			
9FD1	195	15	01-94-34	7E42	9B66
9FD2	195	15	01-94-36	7E44	9B75
9FD3	195	20	01-94-47	7E4F	9C00
9FD4	195	24	01-94-53	7E55	9C63
9FD5	196	20	01-94-65	7E61	9D96
9FD6	196	16	01-94-57	7E59	9D1E
9FD7	196	18	02-94-17	7E31	9D62
9FD8	196	19	01-94-62	7E5E	9D70
AFA1					
AFA2					
AFA3					
AFA4					
AFA5					
AFA6					
AFA7					
AFA8					
AFA9					
AFAA					
AFAB					
AFAC					
AFAD					
AFAE					
AFAF					
AFB0					
AFB1					
AFB2					
AFB3					
AFB4					
AFB5					
AFB6					
AFB7					
AFB8					
AFB9					
AFBA					
AFBB					
AFBC					
AFBD					
AFBE					
AFBF					
AFC0					
AFC1					
AFC2					
AFC3					
AFC4					
