package Check::GlobalPhase;

use strict;
use warnings;

# ABSTRACT: Check::GlobalPhase fast XS checker for Perl GlobalPhase

our $VERSION = '0.001'; # VERSION: generated by DZP::OurPkgVersion

use XSLoader;
XSLoader::load(__PACKAGE__);


1;

__END__

=pod

=encoding utf-8

=head1 NAME

Check::GlobalPhase - Check::GlobalPhase fast XS checker for Perl GlobalPhase

=head1 VERSION

version 0.001

=head1 SYNOPSIS

 #!perl
 
 use strict;
 use warnings;
 
 use Test::More;    # just used for illustration purpose
 
 use Check::GlobalPhase;
 
 INIT {
 
     # instead of using a string comparison
     ok ${^GLOBAL_PHASE} eq 'INIT';
 
     # you can use the boolean helpers
     #   to check if you are in one of the current Perl Phase
     ok Check::GlobalPhase::in_global_phase_init();
 
     # other helpers
     ok !Check::GlobalPhase::in_global_phase_construct();
     ok !Check::GlobalPhase::in_global_phase_start();
     ok !Check::GlobalPhase::in_global_phase_check();
     ok !Check::GlobalPhase::in_global_phase_run();
     ok !Check::GlobalPhase::in_global_phase_end();
     ok !Check::GlobalPhase::in_global_phase_destruct();
 }
 
 # if you need to check more than one phase at the same time
 #   you can use bitmask like this
 ok Check::GlobalPhase::current_phase()
     & ( Check::GlobalPhase::PERL_PHASE_INIT | Check::GlobalPhase::PERL_PHASE_RUN );
 
 # using one ore more of the available constants
 Check::GlobalPhase::PERL_PHASE_CONSTRUCT;
 Check::GlobalPhase::PERL_PHASE_START;
 Check::GlobalPhase::PERL_PHASE_CHECK;
 Check::GlobalPhase::PERL_PHASE_INIT;
 Check::GlobalPhase::PERL_PHASE_RUN;
 Check::GlobalPhase::PERL_PHASE_END;
 Check::GlobalPhase::PERL_PHASE_DESTRUCT;
 
 done_testing;

=head1 DESCRIPTION

Check::GlobalPhase provides some fast helpers to check the current Perl Phase.
This is avoiding the creation of useless C<SvPV> to perform a string comparison
using C<${^GLOBAL_PHASE}>

View C<${^GLOBAL_PHASE}> from L<perlvar> for more details.

=head1 FUNCTIONS

=head2 in_global_phase_construct()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'CONSTRUCT'

=head2 in_global_phase_start()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'START'

=head2 in_global_phase_check()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'CHECK'

=head2 in_global_phase_init()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'INIT'

=head2 in_global_phase_run()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'RUN'

=head2 in_global_phase_end()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'END'

=head2 in_global_phase_destruct()

Return a boolean value. True if ${^GLOBAL_PHASE} eq 'DESTRUCT'

=head1 CONSTANTS

You can use this constant to perform some bitmask check with the current Perl_Phase
returned from L<current_phase>. (view synopsis)

=head2 PERL_PHASE_CONSTRUCT

=head2 PERL_PHASE_START

=head2 PERL_PHASE_CHECK

=head2 PERL_PHASE_INIT

=head2 PERL_PHASE_RUN

=head2 PERL_PHASE_END

=head2 PERL_PHASE_DESTRUCT

=head1 AUTHOR

Nicolas R. <atoomic@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2019 by cPanel L.L.C.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
