#!/usr/bin/perl -w

###############################################################################
#
# A test for Spreadsheet::WriteExcelXML.
#
# Tests merged cells.
#
# reverse(''), July 2004, John McNamara, jmcnamara@cpan.org
#


use strict;
use Spreadsheet::WriteExcelXML;
use Test::More tests => 25;



##############################################################################
#
# Create a new Excel XML file with row data set.
#
my $test_file  = "temp_test_file.xml";
my $workbook   = Spreadsheet::WriteExcelXML->new($test_file);
my $worksheet  = $workbook->add_worksheet();


my $text = 'M01';

my $format = $workbook->add_format(
                                    border  => 6,
                                    bold    => 1,
                                    color   => 'red',
                                    valign  => 'vcenter',
                                    align   => 'center',
                                   );

#
# Tests for simple merges
#

# Horizontal
$worksheet->merge_range('A1:D1', $text++, $format); # M01

# Vertical
$worksheet->merge_range('F1:F4', $text++, $format); # M02

# Range
$worksheet->merge_range('H1:K4', $text++, $format); # M03

#
# Tests for merges followed by merges
#
$worksheet->merge_range('A6:D6', $text++, $format); # M04
$worksheet->merge_range('E6:E7', $text++, $format); # M05
$worksheet->merge_range('F7:G7', $text++, $format); # M06

#
# Tests for merges followed by data cells
#
$worksheet->merge_range('A9:D9', $text++, $format); # M07
$worksheet->write      ('E9',    'test'          );
$worksheet->write      ('A10',   10              );

#
# Tests for reversed range
#
$worksheet->merge_range('A14:D12',    $text++, $format); # M08
$worksheet->merge_range(15, 6, 14, 4, $text++, $format); # M09

#
# Test for data written to part of a merged range. This should be overwritten.
#
$worksheet->write('D9', 'Inside range M07', $format);


#
# Tests for merges followed by merges (2)
#
$worksheet->merge_range('B18:B21', $text++, $format); # M10
$worksheet->merge_range('C18:D19', $text++, $format); # M11
$worksheet->merge_range('C20:D21', $text++, $format); # M12

#
# Tests for merges followed by merges (2)
#
$worksheet->merge_range('B23:B26', $text++, $format); # M13
$worksheet->merge_range('C23:D23', $text++, $format); # M14
$worksheet->merge_range('C24:D24', $text++, $format); # M15
$worksheet->merge_range('C25:D25', $text++, $format); # M16
$worksheet->merge_range('C26:D26', $text++, $format); # M17

#
# Tests for merges followed by text
#
$worksheet->merge_range('B28:B31', $text++, $format); # M18
$worksheet->write      ('C28',     $text++, $format); # M19
$worksheet->write      ('C29',     $text++, $format); # M20
$worksheet->write      ('C30',     $text++, $format); # M21
$worksheet->write      ('C31',     $text++, $format); # M22


$workbook->close();


##############################################################################
#
# Re-open and reread the Excel file.
#
open XML, $test_file or die "Couldn't open $test_file: $!\n";
my @swex_data = extract_cells(*XML);
close XML;
unlink $test_file;


##############################################################################
#
# Read the data from the Excel file in the __DATA__ section
#
my @test_data = extract_cells(*DATA);


##############################################################################
#
# Check for the same number of elements.
#

is(@swex_data, @test_data, " \tCheck for data size");


##############################################################################
#
# Test that the SWEX elements and Excel are the same.
#

# Pad the SWEX data if necessary.
push @swex_data, ('') x (@test_data -@swex_data);

for my $i (0 .. @test_data -1) {
    is($swex_data[$i],$test_data[$i], " \tTesting ss:Index attribute");

}



##############################################################################
#
# Extract <Cell> elements from a given filehandle.
#
sub extract_cells {

    my $fh     = $_[0];
    my $in_row = 0;
    my $row    = '';
    my @rows;

    while (<$fh>) {
        s/^\s+([<| ])/$1/;
        s/\s+$//;

        $in_row = 1 if m/<Cell/;

        $row .= $_ if $in_row;

        if (m[</Cell>]) {
            $in_row  = 0;
            push @rows, $row;
            $row     = '';
        }
    }

    return @rows;
}


# The following data was generated by Excel.
# The merge styles have been renamed from "m\d+" to "s21".
__DATA__
<?xml version="1.0"?>
<?mso-application progid="Excel.Sheet"?>
<Workbook xmlns="urn:schemas-microsoft-com:office:spreadsheet"
 xmlns:o="urn:schemas-microsoft-com:office:office"
 xmlns:x="urn:schemas-microsoft-com:office:excel"
 xmlns:ss="urn:schemas-microsoft-com:office:spreadsheet"
 xmlns:html="http://www.w3.org/TR/REC-html40">
 <Styles>
  <Style ss:ID="Default" ss:Name="Normal">
   <Alignment ss:Vertical="Bottom"/>
   <Borders/>
   <Font/>
   <Interior/>
   <NumberFormat/>
   <Protection/>
  </Style>
  <Style ss:ID="s21">
   <Alignment ss:Horizontal="Center" ss:Vertical="Center"/>
   <Borders>
    <Border ss:Position="Bottom" ss:LineStyle="Double" ss:Weight="3"/>
    <Border ss:Position="Left" ss:LineStyle="Double" ss:Weight="3"/>
    <Border ss:Position="Right" ss:LineStyle="Double" ss:Weight="3"/>
    <Border ss:Position="Top" ss:LineStyle="Double" ss:Weight="3"/>
   </Borders>
   <Font ss:Color="#FF0000" ss:Bold="1"/>
  </Style>
 </Styles>
 <Worksheet ss:Name="Sheet1">
  <Table ss:ExpandedColumnCount="11" ss:ExpandedRowCount="31" x:FullColumns="1"
   x:FullRows="1">
   <Row>
    <Cell ss:MergeAcross="3" ss:StyleID="s21"><Data ss:Type="String">M01</Data></Cell>
    <Cell ss:Index="6" ss:MergeDown="3" ss:StyleID="s21"><Data ss:Type="String">M02</Data></Cell>
    <Cell ss:Index="8" ss:MergeAcross="3" ss:MergeDown="3" ss:StyleID="s21"><Data
      ss:Type="String">M03</Data></Cell>
   </Row>
   <Row ss:Index="6">
    <Cell ss:MergeAcross="3" ss:StyleID="s21"><Data ss:Type="String">M04</Data></Cell>
    <Cell ss:MergeDown="1" ss:StyleID="s21"><Data ss:Type="String">M05</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="6" ss:MergeAcross="1" ss:StyleID="s21"><Data ss:Type="String">M06</Data></Cell>
   </Row>
   <Row ss:Index="9">
    <Cell ss:MergeAcross="3" ss:StyleID="s21"><Data ss:Type="String">M07</Data></Cell>
    <Cell><Data ss:Type="String">test</Data></Cell>
   </Row>
   <Row>
    <Cell><Data ss:Type="Number">10</Data></Cell>
   </Row>
   <Row ss:Index="12">
    <Cell ss:MergeAcross="3" ss:MergeDown="2" ss:StyleID="s21"><Data
      ss:Type="String">M08</Data></Cell>
   </Row>
   <Row ss:Index="15">
    <Cell ss:Index="5" ss:MergeAcross="2" ss:MergeDown="1" ss:StyleID="s21"><Data
      ss:Type="String">M09</Data></Cell>
   </Row>
   <Row ss:Index="18">
    <Cell ss:Index="2" ss:MergeDown="3" ss:StyleID="s21"><Data ss:Type="String">M10</Data></Cell>
    <Cell ss:MergeAcross="1" ss:MergeDown="1" ss:StyleID="s21"><Data
      ss:Type="String">M11</Data></Cell>
   </Row>
   <Row ss:Index="20">
    <Cell ss:Index="3" ss:MergeAcross="1" ss:MergeDown="1" ss:StyleID="s21"><Data
      ss:Type="String">M12</Data></Cell>
   </Row>
   <Row ss:Index="23">
    <Cell ss:Index="2" ss:MergeDown="3" ss:StyleID="s21"><Data ss:Type="String">M13</Data></Cell>
    <Cell ss:MergeAcross="1" ss:StyleID="s21"><Data ss:Type="String">M14</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:MergeAcross="1" ss:StyleID="s21"><Data ss:Type="String">M15</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:MergeAcross="1" ss:StyleID="s21"><Data ss:Type="String">M16</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:MergeAcross="1" ss:StyleID="s21"><Data ss:Type="String">M17</Data></Cell>
   </Row>
   <Row ss:Index="28">
    <Cell ss:Index="2" ss:MergeDown="3" ss:StyleID="s21"><Data ss:Type="String">M18</Data></Cell>
    <Cell ss:StyleID="s21"><Data ss:Type="String">M19</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:StyleID="s21"><Data ss:Type="String">M20</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:StyleID="s21"><Data ss:Type="String">M21</Data></Cell>
   </Row>
   <Row>
    <Cell ss:Index="3" ss:StyleID="s21"><Data ss:Type="String">M22</Data></Cell>
   </Row>
  </Table>
  <WorksheetOptions xmlns="urn:schemas-microsoft-com:office:excel">
   <Print>
    <FitWidth>0</FitWidth>
    <FitHeight>0</FitHeight>
    <ValidPrinterInfo/>
    <PaperSizeIndex>0</PaperSizeIndex>
    <HorizontalResolution>600</HorizontalResolution>
    <VerticalResolution>600</VerticalResolution>
    <Gridlines/>
   </Print>
   <Selected/>
   <Panes>
    <Pane>
     <Number>3</Number>
     <RangeSelection>R1C1:R1C4</RangeSelection>
    </Pane>
   </Panes>
   <ProtectObjects>False</ProtectObjects>
   <ProtectScenarios>False</ProtectScenarios>
  </WorksheetOptions>
 </Worksheet>
</Workbook>
