package FusionInventory::Agent::Task::Inventory::Win32::CPU;

use strict;
use warnings;

use English qw(-no_match_vars);

use FusionInventory::Agent::Tools;
use FusionInventory::Agent::Tools::Win32;
use FusionInventory::Agent::Tools::Generic;

sub isEnabled {
    return 1;
}

sub doInventory {
    my (%params) = @_;

    my $inventory = $params{inventory};
    my $logger    = $params{logger};

    my @cpus = _getCPUs($logger);

    foreach my $cpu (@cpus) {
        $inventory->addEntry(
            section => 'CPUS',
            entry   => $cpu
        );
    }

    if (any { $_->{NAME} =~ /QEMU/i } @cpus) {
        $inventory->setHardware ({
            VMSYSTEM => 'QEMU'
        });
    }
}

sub _getCPUs {
    my ($logger) = @_;

    my @dmidecodeInfos = getCpusFromDmidecode();
    # the CPU description in WMI is false, we use the registry instead
    my $registryInfos = getRegistryKey(
        path   => "HKEY_LOCAL_MACHINE/Hardware/Description/System/CentralProcessor",
        logger => $logger
    );

    my $cpuId = 0;
    my @cpus;

    foreach my $object (getWMIObjects(
        class      => 'Win32_Processor',
        properties => [ qw/NumberOfCores ProcessorId MaxClockSpeed/ ]
    )) {

        my $dmidecodeInfo = $dmidecodeInfos[$cpuId];
        my $registryInfo  = $registryInfos->{"$cpuId/"};

        # Split CPUID from its value inside registry
        my @splitted_identifier = split(/ |\n/ ,$registryInfo->{'/Identifier'});

        my $cpu = {
            CORE         => $dmidecodeInfo->{CORE} || $object->{NumberOfCores},
            THREAD       => $dmidecodeInfo->{THREAD},
            DESCRIPTION  => $registryInfo->{'/Identifier'},
            NAME         => $registryInfo->{'/ProcessorNameString'},
            MANUFACTURER => $registryInfo->{'/VendorIdentifier'},
            SERIAL       => $dmidecodeInfo->{SERIAL},
            SPEED        => $dmidecodeInfo->{SPEED} || $object->{MaxClockSpeed},
            FAMILYNUMBER => $splitted_identifier[2],
            MODEL        => $splitted_identifier[4],
            STEPPING     => $splitted_identifier[6],
            ID           => $dmidecodeInfo->{ID} || $object->{ProcessorId}
        };

        # Some information are missing on Win2000
        if (!$cpu->{NAME}) {
            $cpu->{NAME} = $ENV{PROCESSOR_IDENTIFIER};
            if ($cpu->{NAME} =~ s/,\s(\S+)$//) {
                $cpu->{MANUFACTURER} = $1;
            }
        }

        if ($cpu->{MANUFACTURER}) {
            $cpu->{MANUFACTURER} =~ s/Genuine//;
            $cpu->{MANUFACTURER} =~ s/(TMx86|TransmetaCPU)/Transmeta/;
            $cpu->{MANUFACTURER} =~ s/CyrixInstead/Cyrix/;
            $cpu->{MANUFACTURER} =~ s/CentaurHauls/VIA/;
        }

        if ($cpu->{SERIAL}) {
            $cpu->{SERIAL} =~ s/\s//g;
        }

        if ($cpu->{NAME}) {
            $cpu->{NAME} =~ s/^\s+//;
            $cpu->{NAME} =~ s/\s+$//;


            if ($cpu->{NAME} =~ /([\d\.]+)s*(GHZ)/i) {
                $cpu->{SPEED} = {
                    ghz => 1000,
                    mhz => 1,
                }->{lc($2)} * $1;
            }
        }

        push @cpus, $cpu;

        $cpuId++;
    }

    return @cpus;
}

1;
