#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;

BEGIN {
  if ( eval { require Convert::PEM; 1; } ) {
    plan tests => 10;
  } else {
    plan skip_all => 'Requires Convert::PEM';
  }
}

use Crypt::DSA::GMP;
use Crypt::DSA::GMP::Key;

my $dsa = Crypt::DSA::GMP->new;
ok($dsa, 'Created Crypt::DSA::GMP object');

# A key generated with openssl 1.0.1
my $key = Crypt::DSA::GMP::Key->new( Content => <<KEY, Type => 'PEM' );
-----BEGIN DSA PRIVATE KEY-----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-----END DSA PRIVATE KEY-----
KEY
ok($key, 'Parsed key');
is($key->p, '803098586964298685190360843300601795180489076133516144132542787535280592092309949598642457471283903034639666549271591463009392114824746002949805782762865296615179437445966891197191685052771953036260872329636190842909634115568583006883592251910072902673871923641578240206217335870298463981719908960349004312511236524537862300219963221867148939530715225461652918904587958593942969213483028782627986095171835029234290038846490919866728297918195059869450753996934275189661453357817760613916975731813805317439351608226873774906230309459976403709326027836921280546421569340919608332814049506955372999232646794986154291616813147281991737815084155403063810166939422397036594841211195899917096577816343975198638717144643385250614078962025178772794423108038973430878320774299600888615207007580150262134746982542938895193900436747411277045535024125389587788333012190592423084381541967206537050732655110729134537392868108764388436783477511760727709034682407082242976517261584444303696265308053862146390815790024676532397337520172179630413832579980929526689200002753776184543104103231126926647685857736507652283124439872846539635552533537721613564388605910652178773119740662099678336511129338860505914688754116349837426233445958271648341212294253', 'key->p is correct');
is($key->q, '110482413320846881735854050105291420852995371215222113868753732254953370285067', 'key->q is correct');
is($key->g, '703589231571878111740025805897685677103249105638541452644463057796315613128378608103168853508951966506519321237535368989180739198411702438040297868091385880879893460839905203842930857809298413449006770853648564586801031566539609645560673444519921292259366067270365924478527058734717408490586129647807163996857071705636874388560121096618524819562257519751048982003644929470819331036266686712031897279341413848936695794157756311661706880753596660799679881004042570331935603282381511330685519291653307074074791344591923320989839482331430538886524865355411133175280748246593415932561533205706593332977695953772753738869591550943799842679163807907693713165787855906012588010677478536108807258505718847640320453246788184391239175930954386838452527173710905363630471082015706957542620122466752996229779060721931128218893956804522255489336808157321592152206221925325534072004599349149673695812900326806873933602961732612075036048047150763860257018181532498432867608980876429347522660216080730283728998446450281434034747645738965438461958883916563329886653688073061584189915761173650816600890775384370599431236133298817667905971462161835847058834513186012109674292967319908973665287378615685574178116678885887357058310677207895553161660673829', 'key->g is correct');
is($key->pub_key, '65618409102585939455167077952214172987613070601099405053142609126221548696490446984848841257943943316466276610728734152404659147317435021606214446483718494070847010380008439560047801539340360549837883558041487523178266197731335597909715611204399961234966502754368258064522050222508996132420229206124735779350167107319507993927427281199227059880260078410252248585019175952989207728447306687120062350919564030001566897399539879772859278542966396949735612749744635000275010188092107688668688783703425727365007193860929642312897194874286082595511475579109145831601009163762232018596686546403172301505999997559928059906995488284892664619426806662109527976507523837101605166461521342864697655760661652208911547503697184238613969848435943309091408556010461218508716750948322576572102458544766146167870030523662527098143889965052388290611878160313100422069970631250005583489200315791142396247424651723247770301261952224672646680527577268139554648877361069035718069320396037453051751369063207966840416322781155646021925543629439652630914350445695477308199418597584321235492415923764456634599859880390542693520999755440668657621796329773449056244868808821142579385851270241275872244007614755265620580992231188870274242571833562546514403316248', 'key->pub_key is correct');

my $message = "This prime validation algorithm is used to validate that the integers p and q were generated by the prime generation algorithm given in Appendix A.1.1.2.  The validation algorithm requires the values of p, q, domain_parameter_seed and counter, which were output from the prime generation algorithm.  Let Hash() be the hash function used to generate p and q, and let outlen be its output block length.";
{
  my $sig = $dsa->sign(
        Message => $message,
        Key => $key,
  );
  ok($sig, 'generated default signature');
  my $verified = $dsa->verify(
        Key       => $key,
        Message   => $message,
        Signature => $sig,
  );
  ok($verified, 'verified default signature');
}
{
  my $sig = $dsa->sign(
        Message => $message,
        Key => $key,
        Standard => "FIPS 186-4",
  );
  ok($sig, 'generated FIPS 186-4 signature');
  my $verified = $dsa->verify(
        Key       => $key,
        Message   => $message,
        Signature => $sig,
        Standard => "FIPS 186-4",
  );
  ok($verified, 'verified FIPS 186-4 signature');
}
#$key->write( Type => 'PEM', Filename => 'test-pem-dsa.pem' );
