use utf8;
use strict;

package XML::Pastor::Type;
use base qw(Class::Accessor Class::Data::Inheritable);

XML::Pastor::Type->mk_classdata('XmlSchemaType');

#----------------------------------------------
sub new {
	my $proto 	= shift;
	my $class	= ref($proto) || $proto;
	my $self = {@_};
	
	return bless $self, $class;
}

#----------------------------------------------
sub is_xml_valid {
	my $self	= shift;
	my $result;
	
	eval { $result = $self->xml_validate(@_); };
		
	return 0 if ($@);	# if we died along the way...
	return $result;	
}

#----------------------------------------------
sub xml_validate {
	return xml_validate_futher(@_);	# to be overriden!
}

#----------------------------------------------
sub xml_validate_futher {
	return 1;	# to be overriden!
}

#-----------------------------------------------------------------------------
# CLASS METHOD. Obtain the ancestors of this class. 
#-----------------------------------------------------------------------------
sub get_ancestors {
	my $self	= shift;
	my @ancestors;
	
	{
		no strict 'refs';
		my $cls	=  ref ($self) || $self;		
		@ancestors = @{ $cls . '::ISA' };
	}
	return (@ancestors);
}

1;

__END__

=head1 NAME

B<XML::Pastor::Type> - Ancestor of L<XML::Pastor::ComplexType> and L<XML::Pastor::SimpleType>.

=head1 WARNING

This module is used internally by L<XML::Pastor>. You do not normally know much about this module to actually use L<XML::Pastor>.  It is 
documented here for completeness and for L<XML::Pastor> developers. Do not count on the interface of this module. It may change in 
any of the subsequent releases. You have been warned. 

=head1 ISA

This class descends from L<Class::Data::Inheritable>. 

=head1 DESCRIPTION

B<XML::Pastor::Type> is an B<abstract> ancestor of L<XML::Pastor::ComplexType> and L<XML::Pastor::SimpleType> and 
therefore indirectly of all simple and complex classes generated by L<XML::Pastor> which is a Perl code 
generator from W3C XSD schemas. For an introduction, please refer to the
documentation of L<XML::Pastor>.

B<XML::Pastor::SimpleType> defines a class data accessor called L</XmlSchemaType()> 
with the help of L<Class::Data::Inheritable>. This accessor is normally used by many other methods to access the W3C schema meta information 
related to the class at hand. But at this stage, L</XmlSchemaType()> does not contain any information. 

The generated subclasses set L</XmlSchemaType()> to information specific to the W3C schema type. It is then used for the XML binding and validation methods. 

=head1 METHODS

=head2 CONSTRUCTORS
 
=head4 new() 

  $class->new(%fields)

B<CONSTRUCTOR>.

The new() constructor method instantiates a new B<XML::Pastor::Type> object. It is inheritable, and indeed inherited,
by the generated decsendant classes. Normally, you do not call the B<new> method on B<XML::Pastor::Type>. You rather
call it on your generated subclasses.
  
Any -named- fields that are passed as parameters are initialized to those values within
the newly created object. 

.

=head2 CLASS DATA ACCESSORS

=head4 XmlSchemaType()

  my $type = $class->XmlSchemaType()

B<CLASS METHOD>, but may also be called directly on an B<OBJECT>. 

B<XML::Pastor::Type> defines (thanks to L<Class::Data::Inheritable>) 
a class data acessor B<XmlSchemaType> which returns B<undef>. 

This data accessor is set by each generated simple class to the meta information coming from your B<W3C Schema>. 
This data is of class L<XML::Pastor::Schema::ComplexType> or L<XML::Pastor::Schema::SimpleType>. 

You don't really need to know much about B<XmlSchemaType>. It's used internally by Pastor's XML binding and validation 
methods as meta information about the generated class. 


=head2 OTHER METHODS

=head4 is_xml_valid()

  $bool = $object->is_xml_valid();

B<OBJECT METHOD>.

'B<is_xml_valid>' is similar to L</xml_validate> except that it will not B<die> on failure. 
Instead, it will just return FALSE (0). 

The implementation of this method is very simple. Currently,
it just calls L</xml_validate> in an B<eval> block and will return FALSE (0) if L</xml_validate> dies.  
Otherwise, it will just return the same value as L</xml_validate>.

In case of failure, the contents of the special variable C<$@> will be left untouched in case you would like to access the 
error message that resulted from the death of L</xml_validate>.

.

=head4 xml_validate()
 
	$object->xml_validate();	# Will die on failure

B<OBJECT METHOD>.
 
When overriden by the descendants, 'B<xml_validate>' validates a Pastor XML object (of a generated class) with respect to the META information that
had originally be extracted from your original B<W3C XSD Schema>.

On sucess, B<xml_validate> returns TRUE (1). On failure, it will B<die> on you on validation errors. 

At this stage, B<xml_validate> simply returns the vaue returned by L</xml_validate_further> 
which should perform extra checks. For B<XML::Pastor::Type> this always returns TRUE, but some builtin types 
actually perform some extra validation during this call. 

B<xml_validate> is normally overriden by descendant classes suchs as L<XML::Pastor::SimpleType> and
L<XML::Pastor::ComplexType>.


=head4 xml_validate_further()
 
	$object->xml_validate_further();	# Never called directly.

B<OBJECT METHOD>.
 
'B<xml_validate_further>' should perform extra validation on a Pastor XML object (of a generated class).

It is called by L</xml_validate> after performing rutine validations.  

This method should return TRUE(1) on success, and I<die> on failure with an error message.

For B<XML::Pastor::Type>, this method simple returns TRUE(1).

This method may be, and is indeed, overriden by subclasses. Several builtin classes like
like L<XML::Pastor::Builtin::date> and L<XML::Pastor::Builtin::dateTime> override this method.

=head1 BUGS & CAVEATS

There no known bugs at this time, but this doesn't mean there are aren't any. 
Note that, although some testing was done prior to releasing the module, this should still be considered alpha code. 
So use it at your own risk.

Note that there may be other bugs or limitations that the author is not aware of.

=head1 AUTHOR

Ayhan Ulusoy <dev@ulusoy.name>


=head1 COPYRIGHT

  Copyright (C) 2006-2007 Ayhan Ulusoy. All Rights Reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.


=head1 SEE ALSO

See also L<XML::Pastor>, L<XML::Pastor::ComplexType>, L<XML::Pastor::SimpleType>

If you are curious about the implementation, see L<XML::Pastor::Schema::Parser>,
L<XML::Pastor::Schema::Model>, L<XML::Pastor::Generator>.

If you really want to dig in, see L<XML::Pastor::Schema::Attribute>, L<XML::Pastor::Schema::AttributeGroup>,
L<XML::Pastor::Schema::ComplexType>, L<XML::Pastor::Schema::Element>, L<XML::Pastor::Schema::Group>,
L<XML::Pastor::Schema::List>, L<XML::Pastor::Schema::SimpleType>, L<XML::Pastor::Schema::Type>, 
L<XML::Pastor::Schema::Object>

=cut
