/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Kanji Extensions
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Adding for Japanese language by <fujita@ainix.isac.co.jp> 1994.9.5
     and extend coding system to EUC/SJIS/JIS/HEX at 1994.10.11
     and add all jis codes sequence type at 1995.8.16
     Notes: Hexadecimal code by <ohki@gssm.otuka.tsukuba.ac.jp>
   Adding features about Machine dependent codes and User Defined Codes
     by Hiroshi MIURA <miura@samba.gr.jp> 2000.3.19
*/

#define _KANJI_C_
#include "includes.h"

static char *sj_to_rsj(char *to, const char *from);
static char *no_conversion(char *to, const char *from);
static size_t skip_kanji_multibyte_char(char);
static BOOL is_kanji_multibyte_char_1(char);
static BOOL is_constant_false(char);
static size_t skip_non_multibyte_char(char);
static BOOL not_multibyte_char_1(char);

/*
 * Function pointers that get overridden when multi-byte code pages
 * are loaded.
 */

const char *(*multibyte_strchr)(const char *, int ) = (const char *(*)(const char *, int )) strchr;
const char *(*multibyte_strrchr)(const char *, int ) = (const char *(*)(const char *, int )) strrchr;
const char *(*multibyte_strstr)(const char *, const char *) = (const char *(*)(const char *, const char *)) strstr;
char *(*multibyte_strtok)(char *, const char *) = (char *(*)(char *, const char *)) strtok;

/*
 * Kanji is treated differently here due to historical accident of
 * it being the first non-English codepage added to Samba.
 * The define 'KANJI' is being overloaded to mean 'use kanji codepage
 * by default' and also 'this is the filename-to-disk conversion 
 * method to use'. This really should be removed and all control
 * over this left in the smb.conf parameters 'client codepage'
 * and 'coding system'.
 */

#ifndef KANJI

/*
 * Set the default conversion to be the functions in
 * charcnv.c.
 */

char *(*_dos_to_unix)(char *, const char *) = dos2unix_format;
char *(*_unix_to_dos)(char *, const char *) = unix2dos_format;
BOOL (*_is_unix_sjis)(char) = is_constant_false;
size_t (*_skip_multibyte_char)(char) = skip_non_multibyte_char;
BOOL (*is_multibyte_char_1)(char) = not_multibyte_char_1;

#else /* KANJI */

/*
 * Set the default conversion to be the function
 * sj_to_rsj in this file.
 */

char *(*_dos_to_unix)(char *, BOOL) = sj_to_rsj;
char *(*_unix_to_dos)(char *, BOOL) = no_conversion;
BOOL (*_is_unix_sjis)(char) = is_kanji_multibyte_char_1;
size_t (*_skip_multibyte_char)(char) = skip_kanji_multibyte_char;
BOOL (*is_multibyte_char_1)(char) = is_kanji_multibyte_char_1;

#endif /* KANJI */

BOOL global_is_multibyte_codepage = False;

/* jis si/so sequence */
static char jis_kso = JIS_KSO;
static char jis_ksi = JIS_KSI;
static char hex_tag = HEXTAG;
static int utf8_normalization = NF_NONE;

#define	MAXUCS2STR	1024

/*******************************************************************
  SHIFT JIS functions
********************************************************************/

/*******************************************************************
 search token from S1 separated any char of S2
 S1 contains SHIFT JIS chars.
********************************************************************/

static char *sj_strtok(char *s1, const char *s2)
{
  static char *s = NULL;
  char *q;
  if (!s1) {
    if (!s) {
      return NULL;
    }
    s1 = s;
  }
  for (q = s1; *s1; ) {
    if (is_shift_jis_str (s1)) {
      s1 += 2;
    } else if (is_kana (*s1)) {
      s1++;
    } else {
      char *p = strchr (s2, *s1);
      if (p) {
        if (s1 != q) {
          s = s1 + 1;
          *s1 = '\0';
          return q;
        }
        q = s1 + 1;
      }
      s1++;
    }
  }
  s = NULL;
  if (*q) {
    return q;
  }
  return NULL;
}

/*******************************************************************
 search string S2 from S1
 S1 contains SHIFT JIS chars.
********************************************************************/

static const char *sj_strstr(const char *s1, const char *s2)
{
  size_t len = strlen (s2);
  if (!*s2) 
    return (const char *) s1;
  for (;*s1;) {
    if (*s1 == *s2) {
      if (strncmp (s1, s2, len) == 0)
        return (const char *) s1;
    }
    if (is_shift_jis_str (s1)) {
      s1 += 2;
    } else {
      s1++;
    }
  }
  return NULL;
}

/*******************************************************************
 Search char C from beginning of S.
 S contains SHIFT JIS chars.
********************************************************************/

static const char *sj_strchr (const char *s, int c)
{
  for (; *s; ) {
    if (*s == c)
      return s;
    if (is_shift_jis_str (s)) {
      s += 2;
    } else {
      s++;
    }
  }
  return NULL;
}

/*******************************************************************
 Search char C end of S.
 S contains SHIFT JIS chars.
********************************************************************/

static const char *sj_strrchr(const char *s, int c)
{
  const char *q;

  for (q = NULL; *s; ) {
    if (*s == c) {
      q = s;
    }
    if (is_shift_jis_str (s)) {
      s += 2;
    } else {
      s++;
    }
  }
  return q;
}

/*******************************************************************
 Kanji multibyte char skip function.
*******************************************************************/
   
static size_t skip_kanji_multibyte_char(char c)
{
  if(is_shift_jis(c)) {
    return 2;
  } else if (is_kana(c)) {
    return 1;
  }
  return 0;
}

/*******************************************************************
 Kanji multibyte char identification.
*******************************************************************/
   
static BOOL is_kanji_multibyte_char_1(char c)
{
  return is_shift_jis(c);
}

/*******************************************************************
 Dummy function.
*******************************************************************/
   
/*ARGSUSED*/
static BOOL is_constant_false(char c)
{
  return 0;
}

/*******************************************************************
 The following functions are the only ones needed to do multibyte
 support for Hangul, Big5 and Simplified Chinese. Most of the
 real work for these codepages is done in the generic multibyte
 functions. The only reason these functions are needed at all
 is that the is_xxx(c) calls are really preprocessor macros.
********************************************************************/

/*******************************************************************
  Hangul (Korean - code page 949) function.
********************************************************************/

static BOOL hangul_is_multibyte_char_1(char c)
{
  return is_hangul(c);
}

/*******************************************************************
  Big5 Traditional Chinese (code page 950) function.
********************************************************************/

static BOOL big5_is_multibyte_char_1(char c)
{
  return is_big5_c1(c);
}

/*******************************************************************
  Simplified Chinese (code page 936) function.
********************************************************************/

static BOOL simpch_is_multibyte_char_1(char c)
{
  return is_simpch_c1(c);
}

/*******************************************************************
  Generic multibyte functions - used by Hangul, Big5 and Simplified
  Chinese codepages.
********************************************************************/

/*******************************************************************
 search token from S1 separated any char of S2
 S1 contains generic multibyte chars.
********************************************************************/

static char *generic_multibyte_strtok(char *s1, const char *s2)
{
  static char *s = NULL;
  char *q;
  if (!s1) {
    if (!s) {
      return NULL;
    }
    s1 = s;
  }
  for (q = s1; *s1; ) {
    if ((*is_multibyte_char_1)(*s1)) {
        s1 += 2;
    } else {
      char *p = strchr (s2, *s1);
      if (p) {
        if (s1 != q) {
          s = s1 + 1;
          *s1 = '\0';
          return q;
        }
        q = s1 + 1;
      }
    s1++;
    }
  }
  s = NULL;
  if (*q) {
    return q;
  }
  return NULL;
}

/*******************************************************************
 search string S2 from S1
 S1 contains generic multibyte chars.
********************************************************************/

static const char *generic_multibyte_strstr(const char *s1, const char *s2)
{
  size_t len = strlen (s2);
  if (!*s2)
    return (const char *) s1;
  for (;*s1;) {
    if (*s1 == *s2) {
      if (strncmp (s1, s2, len) == 0)
        return (const char *) s1;
    }
    if ((*is_multibyte_char_1)(*s1)) {
      s1 += 2;
    } else {
      s1++;
    }
  }
  return NULL;
}

/*******************************************************************
 Search char C from beginning of S.
 S contains generic multibyte chars.
********************************************************************/

static const char *generic_multibyte_strchr(const char *s, int c)
{
  for (; *s; ) {
    if (*s == c)
      return (const char *) s;
    if ((*is_multibyte_char_1)(*s)) {
      s += 2;
    } else {
      s++;
    }
  }
  return NULL;
}

/*******************************************************************
 Search char C end of S.
 S contains generic multibyte chars.
********************************************************************/

static const char *generic_multibyte_strrchr(const char *s, int c)
{
  const char *q;
 
  for (q = 0; *s; ) {
    if (*s == c) {
      q = (const char *) s;
    }
    if ((*is_multibyte_char_1)(*s)) {
      s += 2;
    } else {
      s++;
    }
  }
  return q;
}

/*******************************************************************
 Generic multibyte char skip function.
*******************************************************************/

static size_t skip_generic_multibyte_char(char c)
{
  if( (*is_multibyte_char_1)(c)) {
    return 2;
  }
  return 0;
}

/*******************************************************************
  Code conversion
********************************************************************/

int regularize_sjis (int w)
{
  int i, minidx, maxidx;

  if ( (sconv_reg[0].start <= w) && (w <= sconv_reg[SCONVREGSIZ-1].end) ) {
    minidx = -1;
    maxidx = SCONVREGSIZ;
    i = SCONVREGSIZ / 2;
    while (i > minidx && i < maxidx) {
      if ( sconv_reg[i].start > w ) {
	maxidx = i;
      } else if (w > sconv_reg[i].end) {
	minidx = i;
      } else {
	w -= sconv_reg[i].start;
	w += sconv_reg[i].rstart;
	break;
      }
      i = ( minidx + maxidx ) / 2;
    }
  }
  return w;
}

static int sjisconv_jis (int w)
{
  int i, minidx, maxidx;

  if ( (sconv_jis[0].start <= w) && (w <= sconv_jis[SCONVJISSIZ-1].end) ) {
    minidx = -1;
    maxidx = SCONVJISSIZ;
    i = SCONVJISSIZ / 2;
    while (i > minidx && i < maxidx) {
      if ( sconv_jis[i].start > w ) {
	maxidx = i;
      } else if (w > sconv_jis[i].end) {
	minidx = i;
      } else {
	w -= sconv_jis[i].start;
	w += sconv_jis[i].rstart;
	break;
      }
      i = ( minidx + maxidx ) / 2;
    }
  }
  return w;
}

static int sjisconv_nec2ibm (int w)
{
  int i, minidx, maxidx;

  if ( (sconv_nec2ibm[0].start <= w)
      && (w <= sconv_nec2ibm[SCONVNEC2IBMSIZ-1].end) ) {
    minidx = -1;
    maxidx = SCONVNEC2IBMSIZ;
    i = SCONVNEC2IBMSIZ / 2;
    while (i > minidx && i < maxidx) {
      if ( sconv_nec2ibm[i].start > w ) {
	maxidx = i;
      } else if (w > sconv_nec2ibm[i].end) {
	minidx = i;
      } else {
	w -= sconv_nec2ibm[i].start;
	w += sconv_nec2ibm[i].rstart;
	break;
      }
      i = ( minidx + maxidx ) / 2;
    }
  }
  return w;
}

/*******************************************************************
  EUC <-> SJIS
********************************************************************/

static int j2sj (int hi, int lo)
{
  return regularize_sjis(((int)(jis2sjis_table1[hi]) << 8) |
    ((hi & 1) ? jis2sjis_table2[lo] : (lo + 0x7e)));
}

static int sj2euc (int hi, int lo)
{
  int w = (int)((hi << 8) | lo);

  w = sjisconv_jis(w);
  if (w < 0xf000) {
    hi = (int) ((w >> 8) & 0xff);
    lo = (int) (w & 0xff);
  } else {
    hi = GETAHI;
    lo = GETALO;
  }
  return (((int)(sjis2jis_table1[hi] | 0x80) << 8) +
    (sjis2jis_table2[lo] | 0x80));
}

/*******************************************************************
 Convert FROM contain SHIFT JIS codes to EUC codes
 return converted buffer
********************************************************************/

static char *sj_to_euc(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_shift_jis_str (from)) {
      int code = sj2euc ((unsigned char)from[0], (unsigned char)from[1]);
      *out++ = (code >> 8) & 0xff;
      *out++ = code & 0xff;
      from += 2;
    } else if (is_kana (*from)) {
      *out++ = (char)euc_kana;
      *out++ = *from++;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain EUC codes to SHIFT JIS codes
 return converted buffer
********************************************************************/

static char *euc_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3); ) {
    if (is_euc (*from)) {
      int code = j2sj ((int) from[0] & 0x7f, (int) from[1] & 0x7f);
      *out++ = (code >> 8) & 0xff;
      *out++ = code & 0xff;
      from += 2;
    } else if (is_euc_kana (*from)) {
      *out++ = from[1];
      from += 2;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
  EUC3 <-> SJIS
********************************************************************/
static int sjis3euc (int hi, int lo, int *len)
{
  int i, w, minidx, maxidx;

  w = (int)((hi << 8) | lo);

  /* no sjis */
  if (!is_shift_jis2(lo)) {
    w = (GETAHI << 8) | GETALO;

  /* IBM Extended Kanji */
  } else if (( w == 0xfa54 )||( w == 0x81ca )) {
    *len = 2;
    return (0xa2cc);

  } else if (( w ==  0xfa5b )||( w == 0x81e6)) {
    *len = 2;
    return (0xa2e8);

  } else if (is_sjis_ibm(hi)) {
    i = w - 0xfa40 - ( hi - 0xfa )*( 0xfb40 - 0xfafc) - ((lo < 0x7f)? 0 : 1 );
    if ( i < EUC3CONVTBLSIZ ){
      *len = 3;
      return euc3conv[i];
    }  

/* NEC selected IBM Extend Kanji */
    /* there are 3 code that is not good for conv */
  } else if (( 0x8754 <= w ) && ( w <= 0x878a)) {
    minidx = -1;
    maxidx = EUC3CONV2TBLSIZ;
    i = EUC3CONV2TBLSIZ / 2;
    while ( i > minidx && i < maxidx ) {
      if ( euc3conv2[i].sjis > w ) {
	maxidx = i;
      } else if ( w > euc3conv2[i].sjis ) {
	minidx = i;
      } else {
	*len = 3;
	return euc3conv2[i].euc;
      }
      i = ( minidx + maxidx ) / 2;
    }
    /* else normal EUC */

  } else if (( w == 0xeef9 ) || ( w == 0x81ca )) {  
    *len = 2; 
    return (0xa2cc);

  } else if (is_sjis_nec(hi)) {
    w = sjisconv_nec2ibm(w);
    if ( w >= 0xfa40 ) {
      hi = (int) ((w >> 8) & 0xff);
      lo = (int) (w & 0xff);
      i = w - 0xfa40 - ( hi - 0xfa )*( 0xfb40 - 0xfafc) - ((lo < 0x7f)? 0 : 1 );
      if ( i < EUC3CONVTBLSIZ ){
	*len = 3;
	return euc3conv[i];
      } else {
	w = (GETAHI << 8) | GETALO;
      }
    }
    /* else normal EUC */

/* UDC half low*/
/* this area maps to the G1 UDC area: 0xf5a1 -- 0xfefe */
  } else if (is_sjis_g1_udc(hi)) {
    *len = 2;
    return (((hi * 2 - 0xeb) << 8) + (sjis2jis_table2[lo] | 0x80));

/* UDC half high*/
/* this area maps to the G3 UDC area: 0xf8f5a1 -- 0xf8fefe */
  } else if (is_sjis_g3_udc(hi)) {  
    *len = 3;
    return (((hi * 2 - 0xf5) << 8) + (sjis2jis_table2[lo] | 0x80));
    /* ....checked all special case */
  }

  /*  These Normal 2 byte EUC */
  *len = 2;
  w = sjisconv_jis(w);
  if (w < 0xf000) {
    hi = (int) ((w >> 8) & 0xff);
    lo = (int) (w & 0xff);
  } else {
    hi = GETAHI;
    lo = GETALO;
  }
  return (((int)(sjis2jis_table1[hi] | 0x80) << 8) +
    (sjis2jis_table2[lo] | 0x80));
}

static int  euc3sjis (int hi, int lo, BOOL is_3byte)
{
  int w;

  w = (int)((hi << 8) | lo) | 0x8080;
  if (is_3byte) {
    if (is_jis_g1_udc(hi)) {
     /* UDC half high*/
     /* this area maps to the G3 UDC area */
     /* 0xf8f5a1 -- 0xf8fefe --> 0xf540 -- 0xf9fc */
      if (hi & 1) {
	return (((hi / 2 + 0xbb) << 8) | jis2sjis_table2[lo]);
      } else {
	return (((hi / 2 + 0xba) << 8) | (lo + 0x7e));
      }
    } else {
      /* Using map table */
      int i, minidx, maxidx;

      minidx = -1;
      maxidx = EUC3REVTBLSIZ;
      i = EUC3REVTBLSIZ / 2;
      while ( i > minidx && i < maxidx) {
	if ( euc3rev[i].euc > w ) {
	  maxidx = i;
	} else if (euc3rev[i].euc < w) {
	  minidx = i;
	} else {
	  return euc3rev[i].sjis;
	}
	i = ( minidx + maxidx ) / 2;
      }
    }
  } else { /* is_2byte */
    if (is_jis_g3_udc(hi)) {
      /* UDC half low*/
      /* this area maps to the G1 UDC area */
      /* 0xf5a1 -- 0xfefe  --> 0xf040 -- 0xf4fc */
      if (hi & 1) {
	return (((hi / 2 + 0xb6) << 8) | jis2sjis_table2[lo]);
      } else {
	return (((hi / 2 + 0xb5) << 8) | (lo + 0x7e));
      }
    } else { /* Normal EUC */
      return (((int)(jis2sjis_table1[hi]) << 8) |
        ((hi & 1) ? jis2sjis_table2[lo] : (lo + 0x7e)));
    }
  }
  return ((GETAHI << 8) | GETALO);
}

/*******************************************************************
 Convert FROM contain SHIFT JIS codes to EUC codes (with SS2)
 return converted buffer
********************************************************************/

static char *sj_to_euc3(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int len;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-4);) {
    if (is_shift_jis_str (from)) {
      int code = sjis3euc ((unsigned char)from[0],
		           (unsigned char)from[1], &len);
      if (len == 3) {
	*out++ = (char)euc_sup;
      }
      *out++ = (code >> 8) & 0xff;
      *out++ = code & 0xff;
      from += 2;
    } else if (is_kana (*from)) {
      *out++ = (char)euc_kana;
      *out++ = *from++;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain EUC codes (with Sup-Kanji) to SHIFT JIS codes
 return converted buffer
********************************************************************/
static char *euc3_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3); ) {
    if (is_euc_sup (*from)) {
      int code = euc3sjis((int) from[1] & 0x7f, (int) from[2] & 0x7f, True);
      *out++ = (code >> 8) & 0xff;
      *out++ = code & 0xff;
      from += 3;
    } else if (is_euc (*from)) {
      int code = euc3sjis ((int) from[0] & 0x7f, (int) from[1] & 0x7f, False);
      *out++ = (code >> 8) & 0xff;
      *out++ = code & 0xff;
      from += 2;
    } else if (is_euc_kana (*from)) {
      *out++ = from[1];
      from += 2;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
  JIS7,JIS8,JUNET <-> SJIS
********************************************************************/

static int sj2j(int hi, int lo)
{
  int w = (int)((hi << 8) | lo);

  w = sjisconv_jis(w);
  if (w < 0xf000) {
    hi = (int) ((w >> 8) & 0xff);
    lo = (int) (w & 0xff);
  } else {
    hi = GETAHI;
    lo = GETALO;
  }
  return (((int)(sjis2jis_table1[hi]) << 8) + sjis2jis_table2[lo]);
}

/*******************************************************************
 Convert FROM contain SHIFT JIS codes to JIS codes
 return converted buffer
********************************************************************/

static char *sj_to_jis8(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-9); ) {
    if (is_shift_jis_str (from)) {
      int code;
      switch (shifted) {
      case _KJ_ROMAN: /* to KANJI */
        *out++ = jis_esc;
        *out++ = jis_so1;
        *out++ = jis_kso;
        shifted = _KJ_KANJI;
        break;
      }
      code = sj2j ((unsigned char)from[0], (unsigned char)from[1]);
      *out++ = (code >> 8) & 0xff;
      if ((code &= 0xff) == '/') code = ' ';
      *out++ = code;
      from += 2;
    } else {
      switch (shifted) {
      case _KJ_KANJI: /* to ROMAN/KANA */
        *out++ = jis_esc;
        *out++ = jis_si1;
        *out++ = jis_ksi;
        shifted = _KJ_ROMAN;
        break;
      }
      *out++ = *from++;
    }
  }

  switch (shifted) {
  case _KJ_KANJI: /* to ROMAN/KANA */
    *out++ = jis_esc;
    *out++ = jis_si1;
    *out++ = jis_ksi;
    shifted = _KJ_ROMAN;
    break;
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain JIS codes to SHIFT JIS codes
 return converted buffer
********************************************************************/

static char *jis8_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_esc (*from)) {
      if (is_so1 (from[1]) && is_so2 (from[2])) {
        shifted = _KJ_KANJI;
        from += 3;
      } else if (is_si1 (from[1]) && is_si2 (from[2])) {
        shifted = _KJ_ROMAN;
        from += 3;
      } else { /* sequence error */
        goto normal;
      }
    } else {
normal:

      switch (shifted) {
      default:
      case _KJ_ROMAN:
        *out++ = *from++;
        break;
      case _KJ_KANJI:
        {
	  int code;

	  if (from[1] == ' ')
	    code = j2sj ((unsigned char)from[0], '/');
	  else
	    code = j2sj ((unsigned char)from[0], (unsigned char)from[1]);
          *out++ = (code >> 8) & 0xff;
          *out++ = code;
          from += 2;
          break;
        }
      }
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain SHIFT JIS codes to 7 bits JIS codes
 return converted buffer
********************************************************************/
static char *sj_to_jis7(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-11); ) {
    if (is_shift_jis_str (from)) {
      int code;

      if (shifted & _KJ_KANA) {
	*out++ = jis_si;	/* to ROMAN */
	shifted &= ~_KJ_KANA;
      }
      if (!(shifted & _KJ_KANJI)) {
	*out++ = jis_esc;	/* to KANJI */
	*out++ = jis_so1;
	*out++ = jis_kso;
	shifted |= _KJ_KANJI;
      }
      code = sj2j ((unsigned char)from[0], (unsigned char)from[1]);
      *out++ = (code >> 8) & 0xff;
      if ((code &= 0xff) == '/') code = ' ';
      *out++ = code;
      from += 2;
    } else if (is_kana (from[0])) {
      if (!(shifted & _KJ_KANA)) {
	*out++ = jis_so;	/* to KANA */
	shifted |= _KJ_KANA;
      }
      if ((unsigned char)*from == '/' + 0x80)
	*out++ = ' ';
      else
	*out++ = ((unsigned char)*from) - 0x80;
      from++;
    } else {
      if (shifted & _KJ_KANA) {
	*out++ = jis_si;	/* to ROMAN */
	shifted &= ~_KJ_KANA;
      }
      if (shifted & _KJ_KANJI) {
	*out++ = jis_esc;	/* to ROMAN */
	*out++ = jis_si1;
	*out++ = jis_ksi;
	shifted &= ~_KJ_KANJI;
      }
      *out++ = *from++;
    }
  }

  if (shifted & _KJ_KANA)
    *out++ = jis_si;		/* to ROMAN */
  if (shifted & _KJ_KANJI) {
    *out++ = jis_esc;		/* to ROMAN */
    *out++ = jis_si1;
    *out++ = jis_ksi;
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain 7 bits JIS codes to SHIFT JIS codes
 return converted buffer
********************************************************************/
static char *jis7_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_esc (*from)) {
      if (is_so1 (from[1]) && is_so2 (from[2])) {
	shifted = _KJ_KANJI;
	from += 3;
      } else if (is_si1 (from[1]) && is_si2 (from[2])) {
	shifted = _KJ_ROMAN;
	from += 3;
      } else {			/* sequence error */
	goto normal;
      }
    } else if (is_so (*from)) {
      shifted = _KJ_KANA;		/* to KANA */
      from++;
    } else if (is_si (*from)) {
      shifted = _KJ_ROMAN;	/* to ROMAN */
      from++;
    } else {
normal:

      switch (shifted) {
      default:
      case _KJ_ROMAN:
	*out++ = *from++;
	break;
      case _KJ_KANJI:
	{
	  int code;

	  if (from[1] == ' ')
	    code = j2sj ((unsigned char)from[0], '/');
	  else
	    code = j2sj ((unsigned char)from[0], (unsigned char)from[1]);
	  *out++ = (code >> 8) & 0xff;
	  *out++ = code;
	  from += 2;
	}
	break;
      case _KJ_KANA:
	if (*from == ' ')
	  *out++ = '/' + 0x80;
	else
	  *out++ = ((unsigned char)*from) + 0x80;
	from++;
	break;
      }
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain SHIFT JIS codes to 7 bits JIS(junet) codes
 return converted buffer
********************************************************************/
static char *sj_to_junet(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-9); ) {
    if (is_shift_jis_str (from)) {
      int code;
      switch (shifted) {
      case _KJ_KANA:
      case _KJ_ROMAN:		/* to KANJI */
	*out++ = jis_esc;
	*out++ = jis_so1;
	*out++ = jis_so2;
	shifted = _KJ_KANJI;
	break;
      }
      code = sj2j ((unsigned char)from[0], (unsigned char)from[1]);
      *out++ = (code >> 8) & 0xff;
      if ((code &= 0xff) == '/') code = ' ';
      *out++ = code;
      from += 2;
    } else if (is_kana (from[0])) {
      switch (shifted) {
      case _KJ_KANJI:		/* to ROMAN */
      case _KJ_ROMAN:		/* to KANA */
	*out++ = jis_esc;
	*out++ = junet_kana1;
	*out++ = junet_kana2;
	shifted = _KJ_KANA;
	break;
      }
      if ((unsigned char)*from == '/' + 0x80) 
	*out++ = ' ';
      else
	*out++ = ((unsigned char)*from) - 0x80;
      from++;
    } else {
      switch (shifted) {
      case _KJ_KANA:
      case _KJ_KANJI:		/* to ROMAN */
	*out++ = jis_esc;
	*out++ = jis_si1;
	*out++ = jis_si2;
	shifted = _KJ_ROMAN;
	break;
      }
      *out++ = *from++;
    }
  }

  switch (shifted) {
  case _KJ_KANA:
  case _KJ_KANJI:			/* to ROMAN */
    *out++ = jis_esc;
    *out++ = jis_si1;
    *out++ = jis_si2;
    break;
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 Convert FROM contain 7 bits JIS(junet) codes to SHIFT JIS codes
 return converted buffer
********************************************************************/

static char *junet_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int shifted;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  shifted = _KJ_ROMAN;
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_esc (*from)) {
      if (is_so1 (from[1]) && is_so2 (from[2])) {
	shifted = _KJ_KANJI;
	from += 3;
      } else if (is_si1 (from[1]) && is_si2 (from[2])) {
	shifted = _KJ_ROMAN;
	from += 3;
      } else if (is_juk1(from[1]) && is_juk2 (from[2])) {
	shifted = _KJ_KANA;
	from += 3;
      } else {			/* sequence error */
	goto normal;
      }
    } else {
normal:

      switch (shifted) {
      default:
      case _KJ_ROMAN:
	*out++ = *from++;
	break;
      case _KJ_KANJI:
	{
	  int code;

	  if (from[1] == ' ')
	    code = j2sj ((unsigned char)from[0], '/');
	  else
	    code = j2sj ((unsigned char)from[0], (unsigned char)from[1]);
	  *out++ = (code >> 8) & 0xff;
	  *out++ = code;
	  from += 2;
	}
	break;
      case _KJ_KANA:
	if (*from == ' ')
	  *out++ = '/' + 0x80;
	else
	  *out++ = ((unsigned char)*from) + 0x80;
	from++;
	break;
      }
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
  kanji/kana -> ":xx" 
********************************************************************/
static char *sj_to_hex(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-7);) {
    if (is_kana(*from)) {
      *out++ = hex_tag;
      *out++ = bin2hex (((*from)>>4)&0x0f);
      *out++ = bin2hex ((*from)&0x0f);
      from++;
    } else if (is_shift_jis_str (from)) {
      int w;

      w = (int)(((unsigned char)from[0] << 8) | (unsigned char)from[1]);
      from += 2;
      w = regularize_sjis(w);
      *out++ = hex_tag;
      *out++ = bin2hex ((w>>12)&0x0f);
      *out++ = bin2hex ((w>>8)&0x0f);
      *out++ = hex_tag;
      *out++ = bin2hex ((w>>4)&0x0f);
      *out++ = bin2hex (w&0x0f);
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
  HEX <-> SJIS
********************************************************************/
/* ":xx" -> a byte */
static char *hex_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_hex(from)) {
      *out++ = (hex2bin (from[1])<<4) | (hex2bin (from[2]));
      from += 3;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}
 
/*******************************************************************
  kanji/kana -> ":xx" - CAP format.
********************************************************************/
static char *sj_to_cap(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-7);) {
    int w;

    if (is_shift_jis_str (from)) {
      w = (int)(((unsigned char)from[0] << 8) | (unsigned char)from[1]);
      from += 2;
      w = regularize_sjis(w);
      *out++ = hex_tag;
      *out++ = bin2hex ((w>>12)&0x0f);
      *out++ = bin2hex ((w>>8)&0x0f);
      w &= 0xff;
    } else {
      w = (unsigned char)*from++;
    }
    if (w >= 0x80) {
      *out++ = hex_tag;
      *out++ = bin2hex ((w>>4)&0x0f);
      *out++ = bin2hex (w&0x0f);
    } else {
      *out++ = w;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
  CAP <-> SJIS
********************************************************************/
/* ":xx" CAP -> a byte */
static char *cap_to_sj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    /*
     * The only change between this and hex_to_sj is here. sj_to_cap only
     * translates characters greater or equal to 0x80 - make sure that here
     * we only do the reverse (that's why the strchr is used rather than
     * isxdigit. Based on fix from ado@elsie.nci.nih.gov (Arthur David Olson).
     */
    if (is_cap(from)) {
      *out++ = (hex2bin (from[1])<<4) | (hex2bin (from[2]));
      from += 3;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 sj to rsj
********************************************************************/
static char *sj_to_rsj(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  int w;

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    if (is_shift_jis_str (from)) {
      w = (int)(((unsigned char)from[0] << 8) | (unsigned char)from[1]);
      from += 2;
      w = regularize_sjis(w);
      *out++ = (w >> 8) & 0xff;
      *out++ = w & 0xff;
    } else {
      *out++ = *from++;
    }
  }
  *out = '\0';
  return to;
}

/*******************************************************************
 cp to utf8
********************************************************************/
static char *cp_to_utf8(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  smb_ucs2_t val, src[MAXUCS2STR], dest[MAXUCS2STR];
  int i, w;

  if (utf8_normalization != NF_NONE) {
    for (i = 0; *from && i < MAXUCS2STR-1; i++) {
      if (is_shift_jis_str (from)) {
	w = (int)(((unsigned char)from[0] << 8) | (unsigned char)from[1]);
	from += 2;
      } else {
	w = (unsigned char)*from++;
      }
      src[i] = doscp2ucs2(w);
    }
    src[i] = 0;

    normalize_ucs2(dest, src, MAXUCS2STR);
    for (i = 0, out = to; dest[i] && (out - to < sizeof(pstring)-4); i++) {
      if ( dest[i] <= 0x7f ) {
	*out++ = (char)(dest[i] & 0xff);
      } else if ( dest[i] <= 0x7ff ){
	*out++ = (char)( 0xc0 | ((dest[i] >> 6) & 0xff)); 
	*out++ = (char)( 0x80 | ( dest[i] & 0x3f ));
      } else {
	*out++ = (char)( 0xe0 | ((dest[i] >> 12) & 0x0f));
	*out++ = (char)( 0x80 | ((dest[i] >> 6)  & 0x3f));
	*out++ = (char)( 0x80 | (dest[i] & 0x3f));
      }
    }
    *out = '\0';
    return to;
  }

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-4);) {
    if (is_shift_jis_str (from)) {
      w = (int)(((unsigned char)from[0] << 8) | (unsigned char)from[1]);
      from += 2;
    } else {
      w = (unsigned char)*from++;
    }
    val = doscp2ucs2(w);

    if ( val <= 0x7f ) {
      *out++ = (char)(val & 0xff);
    } else if ( val <= 0x7ff ){
      *out++ = (char)( 0xc0 | ((val >> 6) & 0xff)); 
      *out++ = (char)( 0x80 | ( val & 0x3f ));
    } else {
      *out++ = (char)( 0xe0 | ((val >> 12) & 0x0f));
      *out++ = (char)( 0x80 | ((val >> 6)  & 0x3f));
      *out++ = (char)( 0x80 | (val & 0x3f));
    }

  }
  *out = '\0';
  return to;
}

/*******************************************************************
 utf8 to cp
********************************************************************/
static char *utf8_to_cp(char *to, const char *from)
{
  pstring cvtbuf;
  char *out;
  smb_ucs2_t val, src[MAXUCS2STR], dest[MAXUCS2STR];
  int i, w;

  if (utf8_normalization != NF_NONE) {
    for (i = 0; *from && i < MAXUCS2STR-1; i++) {
      val = (unsigned char)*from++;
      if (val < 0x80) {
	src[i] = (val & 0x7f);
      }
      else if (0xc0 <= val && val <= 0xdf
	    && 0x80 <= (unsigned char)*from && (unsigned char)*from <= 0xbf) {
	src[i] = ( ((val & 31) << 6)  | (*from++ & 63 ));
      }
      else {
	val  = (val & 0x0f) << 12;
	val |= ((*from++ & 0x3f) << 6);
	val |= (*from++ & 0x3f);
	src[i] = val;
      }
    }
    src[i] = 0;

    denormalize_ucs2(dest, src, MAXUCS2STR);
    for (i = 0, out = to; dest[i] && (out - to < sizeof(pstring)-3); i++) {
      w = ucs2doscp(dest[i]);
      if (w <= 0xff) {
	*out++ = (char)(w & 0xff);
      } else {
	*out++ = (char)((w >> 8) & 0xff);
	*out++ = (char)(w & 0xff);
      }
    }
    *out = '\0';
    return to;
  }

  if (to == from) {
    from = pstrcpy ((char *) cvtbuf, from);
  }
  for (out = to; *from && (out - to < sizeof(pstring)-3);) {
    val = (unsigned char)*from++;
    if (val < 0x80) {
      w = ucs2doscp(val & 0x7f); 
    } else if (0xc0 <= val && val <= 0xdf
	    && 0x80 <= (unsigned char)*from && (unsigned char)*from <= 0xbf) {
      w = ucs2doscp( ((val & 31) << 6)  | (*from++ & 63 ));
    } else {
      val  = (val & 0x0f) << 12;
      val |= ((*from++ & 0x3f) << 6);
      val |= (*from++ & 0x3f);
      w = ucs2doscp(val);
    }
    if (w <= 0xff) {
      *out++ = (char)(w & 0xff);
    } else {
      *out++ = (char)((w >> 8) & 0xff);
      *out++ = (char)(w & 0xff);
    }
  }
  *out = '\0';
  return to;
}

/************************************************************************
 conversion:
 _dos_to_unix		_unix_to_dos
************************************************************************/

static void setup_string_function(int codes)
{
  switch (codes) {
  default:
    _dos_to_unix = dos2unix_format;
    _unix_to_dos = unix2dos_format;
    break;
  case SJIS_CODE:
    _dos_to_unix = sj_to_rsj;
    _unix_to_dos = no_conversion;
    break;
  case EUC_CODE:
    _dos_to_unix = sj_to_euc;
    _unix_to_dos = euc_to_sj;
    break;
  case JIS7_CODE:
    _dos_to_unix = sj_to_jis7;
    _unix_to_dos = jis7_to_sj;
    break;
  case JIS8_CODE:
    _dos_to_unix = sj_to_jis8;
    _unix_to_dos = jis8_to_sj;
    break;
  case JUNET_CODE:
    _dos_to_unix = sj_to_junet;
    _unix_to_dos = junet_to_sj;
    break;
  case HEX_CODE:
    _dos_to_unix = sj_to_hex;
    _unix_to_dos = hex_to_sj;
    break;
  case CAP_CODE:
    _dos_to_unix = sj_to_cap;
    _unix_to_dos = cap_to_sj;
    break;
  case UTF8_CODE:
    _dos_to_unix = cp_to_utf8;
    _unix_to_dos = utf8_to_cp;
    load_normalization_form(-1, utf8_normalization);
#if 0 
    /* Try and load the unicode map for safety */
    load_dos_unicode_map(lp_client_code_page());
#endif
    break;
  case EUC3_CODE:
    _dos_to_unix = sj_to_euc3;
    _unix_to_dos = euc3_to_sj;
    break;
  }
  _is_unix_sjis =
	(codes == SJIS_CODE) ? is_kanji_multibyte_char_1 : is_constant_false;
}

/************************************************************************
 Interpret coding system.
************************************************************************/

void interpret_coding_system(const char *str)
{
  int codes = UNKNOWN_CODE;
    
  if (strequal (str, "sjis")) {
    codes = SJIS_CODE;
  } else if (strequal (str, "euc")) {
    codes = EUC_CODE;
  } else if (strequal (str, "cap")) {
    codes = CAP_CODE;
    hex_tag = HEXTAG;
  } else if (strequal (str, "hex")) {
    codes = HEX_CODE;
    hex_tag = HEXTAG;
  } else if (!strncasecmp (str, "hex", 3)) {
    codes = HEX_CODE;
    hex_tag = (str[3] ? str[3] : HEXTAG);
  } else if (strequal (str, "j8bb")) {
    codes = JIS8_CODE;
    jis_kso = 'B';
    jis_ksi = 'B';
  } else if (strequal (str, "j8bj") || strequal (str, "jis8")) {
    codes = JIS8_CODE;
    jis_kso = 'B';
    jis_ksi = 'J';
  } else if (strequal (str, "j8bh")) {
    codes = JIS8_CODE;
    jis_kso = 'B';
    jis_ksi = 'H';
  } else if (strequal (str, "j8@b")) {
    codes = JIS8_CODE;
    jis_kso = '@';
    jis_ksi = 'B';
  } else if (strequal (str, "j8@j")) {
    codes = JIS8_CODE;
    jis_kso = '@';
    jis_ksi = 'J';
  } else if (strequal (str, "j8@h")) {
    codes = JIS8_CODE;
    jis_kso = '@';
    jis_ksi = 'H';
  } else if (strequal (str, "j7bb")) {
    codes = JIS7_CODE;
    jis_kso = 'B';
    jis_ksi = 'B';
  } else if (strequal (str, "j7bj") || strequal (str, "jis7")) {
    codes = JIS7_CODE;
    jis_kso = 'B';
    jis_ksi = 'J';
  } else if (strequal (str, "j7bh")) {
    codes = JIS7_CODE;
    jis_kso = 'B';
    jis_ksi = 'H';
  } else if (strequal (str, "j7@b")) {
    codes = JIS7_CODE;
    jis_kso = '@';
    jis_ksi = 'B';
  } else if (strequal (str, "j7@j")) {
    codes = JIS7_CODE;
    jis_kso = '@';
    jis_ksi = 'J';
  } else if (strequal (str, "j7@h")) {
    codes = JIS7_CODE;
    jis_kso = '@';
    jis_ksi = 'H';
  } else if (strequal (str, "jubb")) {
    codes = JUNET_CODE;
    jis_kso = 'B';
    jis_ksi = 'B';
  } else if (strequal (str, "jubj") || strequal (str, "junet")) {
    codes = JUNET_CODE;
    jis_kso = 'B';
    jis_ksi = 'J';
  } else if (strequal (str, "jubh")) {
    codes = JUNET_CODE;
    jis_kso = 'B';
    jis_ksi = 'H';
  } else if (strequal (str, "ju@b")) {
    codes = JUNET_CODE;
    jis_kso = '@';
    jis_ksi = 'B';
  } else if (strequal (str, "ju@j")) {
    codes = JUNET_CODE;
    jis_kso = '@';
    jis_ksi = 'J';
  } else if (strequal (str, "ju@h")) {
    codes = JUNET_CODE;
    jis_kso = '@';
    jis_ksi = 'H';
  } else if (strequal (str, "utf8")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_NONE;
  } else if (strequal (str, "utf8-nfc")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_NFC;
  } else if (strequal (str, "utf8-nfd")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_NFD;
  } else if (strequal (str, "utf8-nfkc")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_NFKC;
  } else if (strequal (str, "utf8-nfkd")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_NFKD;
  } else if (strequal (str, "utf8-mac")) {
    codes = UTF8_CODE;
    utf8_normalization = NF_MAC;
  } else if (strequal (str, "euc3")) {
    codes = EUC3_CODE;
  }	
  setup_string_function (codes);
}

/*******************************************************************
 Non multibyte char function.
*******************************************************************/
   
static size_t skip_non_multibyte_char(char c)
{
  return 0;
}

/*******************************************************************
 Function that always says a character isn't multibyte.
*******************************************************************/

static BOOL not_multibyte_char_1(char c)
{
  return False;
}

/*******************************************************************
 Setup the function pointers for the functions that are replaced
 when multi-byte codepages are used.

 The dos_to_unix and unix_to_dos function pointers are only
 replaced by setup_string_function called by interpret_coding_system
 above.
*******************************************************************/

void initialize_multibyte_vectors( int client_codepage)
{
  switch( client_codepage )
  {
  case KANJI_CODEPAGE:
    multibyte_strchr = sj_strchr;
    multibyte_strrchr = sj_strrchr;
    multibyte_strstr = sj_strstr;
    multibyte_strtok = sj_strtok;
    _skip_multibyte_char = skip_kanji_multibyte_char;
    is_multibyte_char_1 = is_kanji_multibyte_char_1;
    global_is_multibyte_codepage = True;
    break;
  case HANGUL_CODEPAGE:
    multibyte_strchr = generic_multibyte_strchr;
    multibyte_strrchr = generic_multibyte_strrchr;
    multibyte_strstr = generic_multibyte_strstr;
    multibyte_strtok = generic_multibyte_strtok;
    _skip_multibyte_char = skip_generic_multibyte_char;
    is_multibyte_char_1 = hangul_is_multibyte_char_1;
    global_is_multibyte_codepage = True;
    break;
  case BIG5_CODEPAGE:
    multibyte_strchr = generic_multibyte_strchr;
    multibyte_strrchr = generic_multibyte_strrchr;
    multibyte_strstr = generic_multibyte_strstr;
    multibyte_strtok = generic_multibyte_strtok;
    _skip_multibyte_char = skip_generic_multibyte_char;
    is_multibyte_char_1 = big5_is_multibyte_char_1;
    global_is_multibyte_codepage = True;
    break;
  case SIMPLIFIED_CHINESE_CODEPAGE:
    multibyte_strchr = generic_multibyte_strchr;
    multibyte_strrchr = generic_multibyte_strrchr;
    multibyte_strstr = generic_multibyte_strstr;
    multibyte_strtok = generic_multibyte_strtok;
    _skip_multibyte_char = skip_generic_multibyte_char;
    is_multibyte_char_1 = simpch_is_multibyte_char_1;
    global_is_multibyte_codepage = True;
    break;
  /*
   * Single char size code page.
   */
  default:
    multibyte_strchr = (const char *(*)(const char *, int )) strchr;
    multibyte_strrchr = (const char *(*)(const char *, int )) strrchr;
    multibyte_strstr = (const char *(*)(const char *, const char *)) strstr;
    multibyte_strtok = (char *(*)(char *, const char *)) strtok;
    _skip_multibyte_char = skip_non_multibyte_char;
    is_multibyte_char_1 = not_multibyte_char_1;
    global_is_multibyte_codepage = False;
    break; 
  }
}

/* *******************************************************
   function(s) for "dynamic" encoding of SWAT output.
   in this version, only dos_to_dos, dos_to_unix, unix_to_dos
   are used for bug fix. conversion to web encoding
   (to catalog file encoding) is not needed because
   they are using same character codes.
   **************************************************** */
static char *no_conversion(char *to, const char *from)
{
  if (to != from) {
    pstrcpy (to, from);
  }
  return to;
}
char *(*dos_to_dos)(char *, const char *) = no_conversion;
