/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming.async;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.util.concurrent.Future;
import java.io.IOError;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.channels.ClosedByInterruptException;
import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.concurrent.ExecutorPlus;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.streaming.StreamDeserializingTask;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.StreamingChannel;
import org.apache.cassandra.streaming.StreamingDataOutputPlus;
import org.apache.cassandra.streaming.async.NettyStreamingChannel;
import org.apache.cassandra.streaming.messages.KeepAliveMessage;
import org.apache.cassandra.streaming.messages.OutgoingStreamMessage;
import org.apache.cassandra.streaming.messages.StreamMessage;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.concurrent.BlockingQueues;
import org.apache.cassandra.utils.concurrent.ImmediateFuture;
import org.apache.cassandra.utils.concurrent.Semaphore;
import org.apache.cassandra.utils.concurrent.UncheckedInterruptedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamingMultiplexedChannel {
    private static final Logger logger = LoggerFactory.getLogger(StreamingMultiplexedChannel.class);
    private static final int DEFAULT_MAX_PARALLEL_TRANSFERS = FBUtilities.getAvailableProcessors();
    private static final int MAX_PARALLEL_TRANSFERS = CassandraRelevantProperties.STREAMING_SESSION_PARALLELTRANSFERS.getInt(DEFAULT_MAX_PARALLEL_TRANSFERS);
    private static final Semaphore fileTransferSemaphore = Semaphore.newFairSemaphore(MAX_PARALLEL_TRANSFERS);
    private final StreamingChannel.Factory factory;
    private final InetAddressAndPort to;
    private final StreamSession session;
    private final int messagingVersion;
    private volatile boolean closed;
    private volatile StreamingChannel controlChannel;
    private final Collection<ScheduledFuture<?>> channelKeepAlives = BlockingQueues.newBlockingQueue();
    private final ExecutorPlus fileTransferExecutor;
    private final ConcurrentMap<Thread, StreamingChannel> threadToChannelMap = new ConcurrentHashMap<Thread, StreamingChannel>();

    public StreamingMultiplexedChannel(StreamSession session, StreamingChannel.Factory factory, InetAddressAndPort to, @Nullable StreamingChannel controlChannel, int messagingVersion) {
        this.session = session;
        this.factory = factory;
        this.to = to;
        assert (messagingVersion >= 12);
        this.messagingVersion = messagingVersion;
        this.controlChannel = controlChannel;
        String name = session.peer.toString().replace(':', '.');
        this.fileTransferExecutor = (ExecutorPlus)ExecutorFactory.Global.executorFactory().configurePooled("NettyStreaming-Outbound-" + name, MAX_PARALLEL_TRANSFERS).withKeepAlive(1L, TimeUnit.SECONDS).build();
    }

    public InetAddressAndPort peer() {
        return this.to;
    }

    public InetSocketAddress connectedTo() {
        return this.controlChannel == null ? this.to : this.controlChannel.connectedTo();
    }

    private void setupControlMessageChannel() throws IOException {
        if (this.controlChannel == null) {
            this.controlChannel = this.createControlChannel();
        }
    }

    private StreamingChannel createControlChannel() throws IOException {
        logger.debug("Creating stream session to {} as {}", (Object)this.to, (Object)(this.session.isFollower() ? "follower" : "initiator"));
        StreamingChannel channel = this.factory.create(this.to, this.messagingVersion, StreamingChannel.Kind.CONTROL);
        ExecutorFactory.Global.executorFactory().startThread(String.format("Stream-Deserializer-%s-%s", this.to.toString(), channel.id()), new StreamDeserializingTask(this.session, channel, this.messagingVersion));
        this.session.attachInbound(channel);
        this.session.attachOutbound(channel);
        this.scheduleKeepAliveTask(channel);
        logger.debug("Creating control {}", (Object)channel.description());
        return channel;
    }

    private StreamingChannel createFileChannel(InetAddressAndPort connectTo) throws IOException {
        logger.debug("Creating stream session to {} as {}", (Object)this.to, (Object)(this.session.isFollower() ? "follower" : "initiator"));
        StreamingChannel channel = this.factory.create(this.to, connectTo, this.messagingVersion, StreamingChannel.Kind.FILE);
        this.session.attachOutbound(channel);
        logger.debug("Creating file {}", (Object)channel.description());
        return channel;
    }

    public Future<?> sendControlMessage(StreamMessage message) {
        try {
            this.setupControlMessageChannel();
            return this.sendMessage(this.controlChannel, message);
        }
        catch (Exception e) {
            this.close();
            this.session.onError(e);
            return ImmediateFuture.failure(e);
        }
    }

    public Future<?> sendMessage(StreamingChannel channel, StreamMessage message) {
        if (this.closed) {
            throw new RuntimeException("stream has been closed, cannot send " + message);
        }
        if (message instanceof OutgoingStreamMessage) {
            if (this.session.isPreview()) {
                throw new RuntimeException("Cannot send stream data messages for preview streaming sessions");
            }
            if (logger.isDebugEnabled()) {
                logger.debug("{} Sending {}", (Object)StreamSession.createLogTag(this.session), (Object)message);
            }
            InetAddressAndPort connectTo = this.factory.supportsPreferredIp() ? SystemKeyspace.getPreferredIP(this.to) : this.to;
            return this.fileTransferExecutor.submit(new FileStreamTask((OutgoingStreamMessage)message, connectTo));
        }
        try {
            Future<?> promise = channel.send(outSupplier -> {
                long messageSize = StreamMessage.serializedSize(message, this.messagingVersion);
                if (messageSize > 0x40000000L) {
                    throw new IllegalStateException(String.format("%s something is seriously wrong with the calculated stream control message's size: %d bytes, type is %s", new Object[]{StreamSession.createLogTag(this.session, this.controlChannel.id()), messageSize, message.type}));
                }
                try (StreamingDataOutputPlus out = (StreamingDataOutputPlus)outSupplier.apply((int)messageSize);){
                    StreamMessage.serialize(message, out, this.messagingVersion, this.session);
                }
            });
            promise.addListener(future -> this.onMessageComplete(future, message));
            return promise;
        }
        catch (Exception e) {
            this.close();
            this.session.onError(e);
            return ImmediateFuture.failure(e);
        }
    }

    Future<?> onMessageComplete(Future<?> future, StreamMessage msg) {
        Throwable cause = future.cause();
        if (cause == null) {
            return null;
        }
        Channel channel = future instanceof ChannelFuture ? ((ChannelFuture)future).channel() : null;
        logger.error("{} failed to send a stream message/data to peer {}: msg = {}", new Object[]{StreamSession.createLogTag(this.session, channel), this.to, msg, future.cause()});
        return this.session.onError(cause);
    }

    private void scheduleKeepAliveTask(StreamingChannel channel) {
        io.netty.util.concurrent.ScheduledFuture scheduledFuture;
        if (!(channel instanceof NettyStreamingChannel)) {
            return;
        }
        int keepAlivePeriod = DatabaseDescriptor.getStreamingKeepAlivePeriod();
        if (keepAlivePeriod <= 0) {
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("{} Scheduling keep-alive task with {}s period.", (Object)StreamSession.createLogTag(this.session, channel), (Object)keepAlivePeriod);
        }
        KeepAliveTask task = new KeepAliveTask(channel);
        task.future = scheduledFuture = ((NettyStreamingChannel)channel).channel.eventLoop().scheduleAtFixedRate((Runnable)task, (long)keepAlivePeriod, (long)keepAlivePeriod, TimeUnit.SECONDS);
        this.channelKeepAlives.add((ScheduledFuture<?>)scheduledFuture);
    }

    public void setClosed() {
        this.closed = true;
    }

    void setControlChannel(NettyStreamingChannel channel) {
        this.controlChannel = channel;
    }

    int semaphoreAvailablePermits() {
        return fileTransferSemaphore.permits();
    }

    public boolean connected() {
        return !this.closed && (this.controlChannel == null || this.controlChannel.connected());
    }

    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        if (logger.isDebugEnabled()) {
            logger.debug("{} Closing stream connection channels on {}", (Object)StreamSession.createLogTag(this.session), (Object)this.to);
        }
        for (ScheduledFuture<?> future : this.channelKeepAlives) {
            future.cancel(false);
        }
        this.channelKeepAlives.clear();
        this.threadToChannelMap.values().forEach(StreamingChannel::close);
        this.threadToChannelMap.clear();
        this.fileTransferExecutor.shutdownNow();
    }

    @VisibleForTesting
    public void unsafeCloseControlChannel() {
        logger.warn("Unsafe close of control channel");
        this.controlChannel.close().awaitUninterruptibly();
    }

    class KeepAliveTask
    implements Runnable {
        private final StreamingChannel channel;
        ScheduledFuture<?> future;

        KeepAliveTask(StreamingChannel channel) {
            this.channel = channel;
        }

        @Override
        public void run() {
            if (!this.channel.connected() || StreamingMultiplexedChannel.this.closed) {
                if (null != this.future) {
                    this.future.cancel(false);
                }
                return;
            }
            if (logger.isTraceEnabled()) {
                logger.trace("{} Sending keep-alive to {}.", (Object)StreamSession.createLogTag(StreamingMultiplexedChannel.this.session, this.channel), (Object)StreamingMultiplexedChannel.this.session.peer);
            }
            StreamingMultiplexedChannel.this.sendControlMessage(new KeepAliveMessage()).addListener(f -> {
                if (f.isSuccess() || f.isCancelled()) {
                    return;
                }
                if (logger.isDebugEnabled()) {
                    logger.debug("{} Could not send keep-alive message (perhaps stream session is finished?).", (Object)StreamSession.createLogTag(StreamingMultiplexedChannel.this.session, this.channel), (Object)f.cause());
                }
            });
        }
    }

    class FileStreamTask
    implements Runnable {
        private static final int SEMAPHORE_UNAVAILABLE_LOG_INTERVAL = 3;
        private final StreamMessage msg;
        private final InetAddressAndPort connectTo;

        FileStreamTask(OutgoingStreamMessage ofm, InetAddressAndPort connectTo) {
            this.msg = ofm;
            this.connectTo = connectTo;
        }

        FileStreamTask(StreamMessage msg) {
            this.msg = msg;
            this.connectTo = null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (!this.acquirePermit(3)) {
                return;
            }
            StreamingChannel channel = null;
            try {
                channel = this.getOrCreateFileChannel(this.connectTo);
                try (StreamingDataOutputPlus out = channel.acquireOut();){
                    StreamMessage.serialize(this.msg, out, StreamingMultiplexedChannel.this.messagingVersion, StreamingMultiplexedChannel.this.session);
                }
            }
            catch (Exception e) {
                StreamingMultiplexedChannel.this.session.onError(e);
            }
            catch (Throwable t) {
                if (StreamingMultiplexedChannel.this.closed && Throwables.getRootCause((Throwable)t) instanceof ClosedByInterruptException && StreamingMultiplexedChannel.this.fileTransferExecutor.isShutdown()) {
                    logger.debug("{} Streaming channel was closed due to the executor pool being shutdown", (Object)StreamSession.createLogTag(StreamingMultiplexedChannel.this.session, channel));
                } else {
                    JVMStabilityInspector.inspectThrowable(t);
                    if (!StreamingMultiplexedChannel.this.session.state().isFinalState()) {
                        StreamingMultiplexedChannel.this.session.onError(t);
                    }
                }
            }
            finally {
                fileTransferSemaphore.release(1);
            }
        }

        boolean acquirePermit(int logInterval) {
            long logIntervalNanos = TimeUnit.MINUTES.toNanos(logInterval);
            long timeOfLastLogging = Clock.Global.nanoTime();
            while (!StreamingMultiplexedChannel.this.closed) {
                try {
                    if (fileTransferSemaphore.tryAcquire(1, 1L, TimeUnit.SECONDS)) {
                        return true;
                    }
                    long now = Clock.Global.nanoTime();
                    if (now - timeOfLastLogging <= logIntervalNanos) continue;
                    timeOfLastLogging = now;
                    OutgoingStreamMessage ofm = (OutgoingStreamMessage)this.msg;
                    if (!logger.isInfoEnabled()) continue;
                    logger.info("{} waiting to acquire a permit to begin streaming {}. This message logs every {} minutes", new Object[]{StreamSession.createLogTag(StreamingMultiplexedChannel.this.session), ofm.getName(), logInterval});
                }
                catch (InterruptedException e) {
                    throw new UncheckedInterruptedException(e);
                }
            }
            return false;
        }

        private StreamingChannel getOrCreateFileChannel(InetAddressAndPort connectTo) {
            Thread currentThread = Thread.currentThread();
            try {
                StreamingChannel channel = (StreamingChannel)StreamingMultiplexedChannel.this.threadToChannelMap.get(currentThread);
                if (channel != null) {
                    return channel;
                }
                channel = StreamingMultiplexedChannel.this.createFileChannel(connectTo);
                StreamingMultiplexedChannel.this.threadToChannelMap.put(currentThread, channel);
                return channel;
            }
            catch (Exception e) {
                throw new IOError(e);
            }
        }

        void injectChannel(StreamingChannel channel) {
            Thread currentThread = Thread.currentThread();
            if (StreamingMultiplexedChannel.this.threadToChannelMap.get(currentThread) != null) {
                throw new IllegalStateException("previous channel already set");
            }
            StreamingMultiplexedChannel.this.threadToChannelMap.put(currentThread, channel);
        }

        void unsetChannel() {
            StreamingMultiplexedChannel.this.threadToChannelMap.remove(Thread.currentThread());
        }
    }
}

