/* These routines are used instead of the standard library time routines to force
** correct handling of "timestamps" in "UNIX time", instead of "ANSI time".
** The library routines time() gmtime() and mktime() are either ANSI or UNIX.
** The routine initJwtime attempts to find out which by passing in a UNIX time
** for 1/1/92 and finding out if the year coming from gmtime() is 1992. If the
** year is 1992, then we know the library is UNIX. Otherwise, we know it is ANSI.
** The constant offset variable "ansi_offset" is then set accordingly, either to
** 0 for UNIX libraries, or to the ANSI timestamp of 01/01/70 if the library is
** ANSI. This offset is then used to adjust serial numbers going into and coming
** out of the library routines.
**
** The only problem is how to get a program written with calls to gmtime() time()
** and mktime() to use the routines unix_gmtime() unix_time() and unix_mktime().
** This is achieved in the file unixtime.h, which is included in the program
** instead of time.h.
*/
#include <time.h>
#define T_010170 0x83abd000 /* count from ANSI time base - 0101 1900 to 01/01/70 */
#define T_010192 0x29610480 /* count from 01/01/70 to 01/01/92 */
int jwtimeInited = 0;       /* Set when inited */
unsigned long ansi_offset;  /* Set 0 if time(), gmtime() and mktime() are 01/01/70 	*/
                            /* based. Otherwise set to offset of 01/01/70 from ansi */
                            /* time base of 01/01 1900                             	*/

void initJwtime(void){
	time_t t;
	t = T_010192;
	if (gmtime(&t)->tm_year != 92)
		ansi_offset = T_010170;
	else
		ansi_offset = 0l;
	jwtimeInited = 1;
}

time_t unix_time(time_t * arg){
	if (! jwtimeInited)
		initJwtime();

 	if (arg != NULL)
 		*arg = time(NULL) - ansi_offset;

	return time(NULL)-ansi_offset;
}

struct tm * unix_gmtime(time_t * arg){
	time_t t;

	if (! jwtimeInited)
		initJwtime();

	t = *arg + ansi_offset;
	return gmtime(&t);
}

time_t unix_mktime(struct tm * arg){
	time_t t;

	if (! jwtimeInited)
		initJwtime();

	return (mktime(arg) - ansi_offset);
}

/* Convert a 1970 epoc second count to a 1900 epoc day count */
/* Use these values in EXCEL and other microsoft products.	 */
/* Note that although day 1 in the time() world is the second */
/* day, day 1.00 in the Microsoft floating point world is the */
/* first day, Jan 1 1900. So 1.5 is Noon on Jan 1 1900. 			*/
#define DATEVAL010170 25569.0
double TimeVal(time_t t){
	double temp;
	temp = (double) t;
	temp /= (24.0 * 60.0 * 60.0);
	temp += DATEVAL010170;
	return temp;
}
