/*
 * $Id: Xconv.c,v 1.1 1997/06/11 02:10:46 morgan Exp $
 *
 * This file deals with the piped input to the X-based function.
 * This code knows nothing of PAM.  It simply takes requests from
 * a pipe and provides responses through a second pipe.
 *
 * Copyright (c) 1997 Andrew Morgan <morgan@parc.power.net>
 * Copyright (c) 1997 Danny Sung
 */

#undef FOR_EDITRES
#if defined(FOR_EDITRES)
extern void _XEditResCheckMessages();
#endif

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include "Auth/Auth.h"
#include "Xconv.h"
#include "pipe.h"

/*
 * There are two externally callable functions:
 *
 *       StartGraphicsInterface()
 *         - this establishes a pipe and forks a child to deal
 *           with PAM driven requests.
 *
 *       EndGraphicsInterface()
 *         - this closes down the X window and kills the graphics
 *           processing child.
 */

/*
 * SUPPORT ROUTINES
 * ================
 */

/*
 *       (xpipe)
 *  +-+ --[out]--> +---+
 *  |X|            +PAM+
 *  +-+ <--[in]--- +---+
 */

struct pipeio_t xpipe;                                          /* X in/out */

/* XXX - what does this do? */

static void okNotify(Widget w, Prompt *prompt, int type)
{
    void *data=NULL;
    int len;
    enum PIPEIO epipe;

    D(("called."));

    /* Signal to PAM process that the user has completed a form */
    outmsg(&xpipe, OK_NOTIFY, NULL, 0);

    /* Loop through all of the PAM related queries about completed form */
    while ( (epipe=inmsg(&xpipe, &data, &len)) != QUERY_DONE ) {
	int nothing=TRUE;
	Prompt *p;

	switch(epipe) {
	case QUERY_RESPONSE:
	    /* Loop through the list of items (Prompts) */
	    for (p=prompt; p; p=(Prompt *)p->next) {
		if ( !strcmp(data, p->labeltext) ) {
		    /* send contents of requested item (Prompt) */
		    outmsg(&xpipe, SEND_RESPONSE, p->entrytext,
			   strlen(p->entrytext)+1);
		    nothing=FALSE;
		}
            }
            if ( nothing ) {
		/* empty - we have nothing to report */
		outmsg(&xpipe, SEND_RESPONSE, NULL, 0);
	    }
            break;
	default:
            D(("protocol error: bad query response"));
	}
	/* clean up */
	_pam_overwrite(data);
	_pam_drop(data);
    }

    D(("end."));
}

/* XXX - what does this do? */

static void abortNotify(Widget w, Prompt *prompt, int type)
{
    D(("called."));
    outmsg(&xpipe, ABORT_NOTIFY, 0, 0);
    D(("end."));
}

static void xauth_ipcb(XtPointer client_data, int *source, XtInputId *id)
{
    Widget wauthWidget = (Widget) client_data;
    void *data;
    int len;

    switch(inmsg(&xpipe, &data, &len)) {
    case CLEARMSGS:
	D(("got clear message enum"));
	XtVaSetValues(wauthWidget, XtNclearMessage, True, NULL);
	break;
    case CLEARPROMPTS:
	D(("got clear prompts"));
	XtVaSetValues(wauthWidget, XtNclearPrompts, True, NULL);
	break;
    case PROMPT_ECHO_ON:
	D(("Got prompt echo on: '%s'", (char *)data));
	XtVaSetValues(wauthWidget, XtNaddEchoString,
		      (XtPointer)data, NULL);
	break;
    case PROMPT_ECHO_OFF:
	D(("Got prompt echo off: '%s'", (char *)data));
	XtVaSetValues(wauthWidget, XtNaddNoEchoString,
		      (XtPointer)data, NULL);
	break;
    case ERROR_MSG:
	D(("Got error message: '%s'", (char *)data));
	XtVaSetValues(wauthWidget, XtNmessageString,
		      (XtPointer)data, NULL);
	break;
    case TEXT_INFO:
	D(("Got text info: '%s'", (char *)data));
	XtVaSetValues(wauthWidget, XtNaddLabelString,
		      (XtPointer)data, NULL);
	break;
    case POPUP:
	D(("make widget visible"));
	XtRealizeWidget(XtParent(wauthWidget));
	XtManageChild(wauthWidget);
	break;
    case POPDOWN:
	D(("hide widget"));
	XtUnmanageChild(wauthWidget);
	break;
    case HIDE_WINDOW:
	D(("hide window"));
	XtUnrealizeWidget(XtParent(wauthWidget));
	break;
    case FINAL_SUCCESS:
	D(("Authentication ok."));
	D(("Place rest of X program here."));
	break;
    case FINAL_FAIL:
	D(("Authentication failure."));
	exit(1);
	break;
    case ABORT_NOTIFY:
	D(("request to terminate."));
	XtDestroyWidget(wauthWidget);
	exit(0);
    default:
	D(("invalid message type"));
	exit(1);
    }
}

static void BeginX(int argc, char **argv)
{
    Widget toplevel, wauthWidget;
    XtAppContext appcontext;

    /*
     * Child process (knows about X)
     */

    /* Initialize X */
    toplevel = XtVaAppInitialize(&appcontext, "XAuth", NULL, 0,
				 &argc, argv, NULL,
				 XtNwidth, 350, XtNheight, 250, NULL, 0 );

    wauthWidget = XtVaCreateManagedWidget("auth",
					  authWidgetClass, toplevel,
					  XtNwidth, 350, XtNheight, 250,
					  XtNbannerString,
					  (XtPointer) "Welcome to PAM", NULL);

    XtVaSetValues(wauthWidget, XtNokNotify, (XtPointer)okNotify, NULL);
    XtVaSetValues(wauthWidget, XtNabortNotify, (XtPointer)abortNotify,NULL);

#if defined(FOR_EDITRES)
    XtAddEventHandler(toplevel, (EventMask)0, True,
		      _XEditResCheckMessages, NULL);
#endif

    /* Input on X-pipe is an event that the auth widget should notice */
    XtAppAddInput( appcontext, xpipe.infd, (XtPointer) XtInputReadMask,
		   xauth_ipcb, (XtPointer) wauthWidget );

    /* Infinite loop */
    XtAppMainLoop(appcontext);
}

/*
 * Initialize X-process
 */

int StartGraphicsInterface(int argc, char **argv, ConversingStruct **appdata)
{
    ConversingStruct *talker;
    int xfd[2], cfd[2];

    /* Initialize the application pointer */
    if ( (talker = calloc(1, sizeof(ConversingStruct))) == NULL ) {
	D(("no memory to create pipe"));
	return FALSE;
    }

    /* Initialize pipes */
    if ( pipe(xfd) < 0 || pipe(cfd) < 0 ) {
	D(("pipe creation error."));
	return FALSE;
    }

    talker->cpipe.infd  = cfd[0];
    talker->cpipe.outfd = xfd[1];
    xpipe.infd  = xfd[0];
    xpipe.outfd = cfd[1];

    cfd[0] = cfd[1] = xfd[0] = xfd[1] = 0;

    /* Create processes */
    if ( (talker->child=fork()) == -1 ) {
	D(("xconv: fork() error"));

	return FALSE;
    } else if ( talker->child != 0 ) {
	*appdata = talker;

	return TRUE;
    }

    BeginX(argc, argv);                                  /* never returns */

    /* exit */
    D(("Shouldn't get here"));
    exit(1);
}

/*
 * Sever link to X-process
 */

int EndGraphicsInterface(ConversingStruct **appdata, int success)
{
    ConversingStruct *talker = *appdata;

    *appdata = NULL;       /* calling function does not need this anymore */

    /*
     * We need to signal that the child should understand that PAM
     * is done with requests success indicates if we should proceed
     * or not.
     */

    if (talker->child) {           /* There is still an X-process running */
	if (success) {
	    D(("indicate that PAM says success"));
	} else {
	    D(("indicate that PAM says failed"));
	}
    }

    /*
     * Tidy up.
     */

    memset(talker, 0, sizeof(*talker));
    _pam_drop(talker);

    /*
     * Return.
     */

    return TRUE;
}
