
/***************************************************************************
*
*   Copyright (c) 1998, 1999 Jeff V. Merkey
*   895 West Center Street
*   Orem, Utah  84057
*   jmerkey@utah-nac.org
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation, version 2, or any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You are free to modify and re-distribute this program in accordance
*   with the terms specified in the GNU Public License.  The copyright
*   contained in this code is required to be present in any derivative
*   works and you are required to provide the source code for this
*   program as part of any commercial or non-commercial distribution.
*   You are required to respect the rights of the Copyright holders
*   named within this code.
*
*   jmerkey@utah-nac.org is the official maintainer of
*   this code.  You are encouraged to report any bugs, problems, fixes,
*   suggestions, and comments about this software to jmerkey@utah-nac.org
*   or linux-kernel@vger.kernel.org.  New releases, patches, bug fixes, and
*   technical documentation can be found at www.kernel.org.  We will
*   periodically post new releases of this software to www.kernel.org
*   that contain bug fixes and enhanced capabilities.
*
*   Original Authorship      :
*      source code written by Jeff V. Merkey
*
*   Original Contributors    :
*      Jeff V. Merkey
*      Darren Major
*      
*
****************************************************************************
*
*
*   AUTHORS  :  Jeff V. Merkey (jmerkey@utah-nac.org)
*   FILE     :  BLOCK.C
*   DESCRIP  :  NWFS Block Device Interface Module
*   DATE     :  December 2, 1998
*
*
***************************************************************************/

#include "globals.h"

BYTE *ZeroBuffer = 0;

#if (FILE_DISK_EMULATION)

void SyncDevice(ULONG disk)
{
    return;
}

uint32	pReadDiskSectors(
	uint32	disk_id,
	uint32	sector_offset,
	uint8	*buffer,
	uint32	sector_count,
	uint32	extra)
{
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
	if (extra) {};
	if (SystemDisk[disk_id] != 0)
	{
		fseek((FILE *) SystemDisk[disk_id]->PhysicalDiskHandle, sector_offset * 512, 0);
		return (fread(buffer, 512, sector_count, (FILE *) SystemDisk[disk_id]->PhysicalDiskHandle));
	}
	return(0);
}

uint32	pWriteDiskSectors(
	uint32	disk_id,
	uint32	sector_offset,
	uint8	*buffer,
	uint32	sector_count,
	uint32	extra)
{
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
	if (extra) {};
	if (SystemDisk[disk_id] != 0)
	{
		fseek((FILE *) SystemDisk[disk_id]->PhysicalDiskHandle, sector_offset * 512, 0);
		return (fwrite(buffer, 512, sector_count, (FILE *) SystemDisk[disk_id]->PhysicalDiskHandle));

	}
	return(0);
}


ULONG pZeroFillDiskSectors(
	uint32	disk_id,
	uint32	sector_offset,
	uint32	sector_count,
	uint32	extra)
{
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
	if (extra) {};
	if (SystemDisk[disk_id] != 0)
	{
		fseek((FILE *) SystemDisk[disk_id]->PhysicalDiskHandle, sector_offset * 512, 0);
		return (fwrite(ZeroBuffer, 512, sector_count, (FILE *) SystemDisk[disk_id]->PhysicalDiskHandle));

	}
	return(0);
}

ULONG ReadDiskSectors(ULONG disk, ULONG LBA, BYTE *Sector,
				 ULONG sectors, ULONG readAhead)
{
	return (pReadDiskSectors(disk, LBA, Sector, sectors, readAhead));
}

ULONG WriteDiskSectors(ULONG disk, ULONG LBA, BYTE *Sector,
				  ULONG sectors, ULONG readAhead)
{
	return (pWriteDiskSectors(disk, LBA, Sector, sectors, readAhead));
}

ULONG ZeroFillDiskSectors(ULONG disk, ULONG StartingLBA,
				 ULONG sectors, ULONG readAhead)
{
	return (pZeroFillDiskSectors(disk, StartingLBA, sectors, readAhead));
}

#else  // FILE_DISK_EMULATION


#if (LINUX_20 | LINUX_22 | LINUX_24)

extern ULONG insert_io(ULONG disk, ASYNCH_IO *io);
extern void RunAsynchIOQueue(ULONG disk);
extern void profile_complete(void);
void insert_callback(ASYNCH_IO *io);

#if (LINUX_24)
DECLARE_WAIT_QUEUE_HEAD(wait_for_bh);
#elif (LINUX_20 | LINUX_22)
struct wait_queue *wait_for_bh = NULL;
#endif

#if (LINUX_22 | LINUX_24)
kmem_cache_t *bh_p = 0;
#endif

//
//  low level disk sector interface routines
//

#define AIO_SIGNATURE  (ULONG)"AIOD"

struct buffer_head *bh_head = 0;
struct buffer_head *bh_tail = 0;
ULONG bh_count = 0;
ULONG bh_waiters = 0;

#if (LINUX_SLEEP)
#if (LINUX_SPIN)
spinlock_t bh_spinlock = SPIN_LOCK_UNLOCKED;
long bh_flags = 0;
#else
NWFSInitMutex(bh_semaphore);
#endif
#endif

void lock_bh_free(void)
{
#if (LINUX_SLEEP)
#if (LINUX_SPIN)
    spin_lock_irqsave(&bh_spinlock, bh_flags);
#else
    if (WaitOnSemaphore(&bh_semaphore) == -EINTR)
       NWFSPrint("lock bh was interrupted\n");
#endif
#endif
}

void unlock_bh_free(void)
{
#if (LINUX_SLEEP)
#if (LINUX_SPIN)
    spin_unlock_irqrestore(&bh_spinlock, bh_flags);
#else
    SignalSemaphore(&bh_semaphore);
#endif
#endif
}


void nwfs_put_bh(struct buffer_head *bh)
{
    lock_bh_free();
    if (!bh_head)
    {
       bh_head = bh_tail = bh;
       bh->b_next_free = bh->b_prev_free = 0;
    }
    else
    {
       bh_tail->b_next_free = bh;
       bh->b_next_free = 0;
       bh->b_prev_free = bh_tail;
       bh_tail = bh;
    }
    unlock_bh_free();

    // wake up any waiters
    wake_up(&wait_for_bh);
    
    return;
}

struct buffer_head *int_get_bh(void)
{
    struct buffer_head *bh;

    lock_bh_free();
    if (bh_head)
    {
       bh = bh_head;
       bh_head = bh->b_next_free;
       if (bh_head)
	  bh_head->b_prev_free = NULL;
       else
	  bh_tail = NULL;

       bh->b_next_free = bh->b_prev_free = 0;
       unlock_bh_free();
       return bh;
    }
    else 
    if (bh_count < MAX_BUFFER_HEADS)
    {
       unlock_bh_free();

#if (LINUX_20)
       bh = (struct buffer_head *)kmalloc(sizeof(struct buffer_head), 
		                          GFP_ATOMIC);
       
       BH_TRACKING.count += sizeof(struct buffer_head);
       BH_TRACKING.units++;
       MemoryInUse += sizeof(struct buffer_head);

#elif (LINUX_22 | LINUX_24)
       bh = (struct buffer_head *)kmem_cache_alloc(bh_p, SLAB_KERNEL);
       
       BH_TRACKING.count += sizeof(struct buffer_head);
       BH_TRACKING.units++;
       MemoryInUse += sizeof(struct buffer_head);

#else
       bh = (struct buffer_head *)NWFSIOAlloc(sizeof(struct buffer_head), BH_TAG);
#endif
       if (!bh)
	  return 0;

       bh_count++; 

       NWFSSet(bh, 0, sizeof(struct buffer_head));

#if (LINUX_24)
       init_waitqueue_head(&bh->b_wait); 
#endif
       return bh;
    }
    unlock_bh_free();
    return 0;
}

void free_bh_list(void)
{
    struct buffer_head *bh;

    lock_bh_free();
    while (bh_head)
    {
       bh = bh_head;
       bh_head = bh->b_next_free;

#if (LINUX_20)
       kfree(bh);

       BH_TRACKING.count -= sizeof(struct buffer_head);
       BH_TRACKING.units--;
       MemoryInUse -= sizeof(struct buffer_head);

#elif (LINUX_22 | LINUX_24)
       if (bh_p)
       {
          kmem_cache_free(bh_p, bh);

	  BH_TRACKING.count -= sizeof(struct buffer_head);
          BH_TRACKING.units--;
          MemoryInUse -= sizeof(struct buffer_head);
       }
#else
       NWFSFree(bh);
#endif
    }
    bh_head = bh_tail = 0;

#if (LINUX_22 | LINUX_24)
    if (bh_p)
    {
       kmem_cache_shrink(bh_p);
#if (LINUX_24)
       kmem_cache_destroy(bh_p);
#endif
    }
    bh_p = 0;
#endif

    bh_count = 0;
    unlock_bh_free();
    return;
}

struct buffer_head *__get_bh_wait(void)
{
    struct buffer_head *bh;
#if (LINUX_20 | LINUX_22)   
    struct wait_queue wait = { current, NULL };
#elif (LINUX_24)
    DECLARE_WAITQUEUE(wait, current);
#endif
    
#if (LINUX_20 | LINUX_22)
    add_wait_queue(&wait_for_bh, &wait);
#elif (LINUX_24)
    add_wait_queue_exclusive(&wait_for_bh, &wait);
#endif
    
    for (;;)
    {
       current->state = TASK_UNINTERRUPTIBLE;
       bh = int_get_bh();
       if (bh)
	  break;
#if (!POST_IMMEDIATE)

#if (LINUX_22)
       lock_kernel();
#endif       
       run_task_queue(&tq_disk);
#if (LINUX_22)
       unlock_kernel();
#endif       

#endif
       bh_waiters++;
       schedule();
       bh_waiters--;
    }
    remove_wait_queue(&wait_for_bh, &wait);
    current->state = TASK_RUNNING;
    return bh;
}

struct buffer_head *nwfs_get_bh(void)
{
    struct buffer_head *bh;

#if (LINUX_22 | LINUX_24)
    if (!bh_p)
    {
       bh_p = kmem_cache_create("nwfs_buffer_head", sizeof(struct buffer_head),
			        0, SLAB_HWCACHE_ALIGN, NULL, NULL);
       if (!bh_p)
       {
          NWFSPrint("Cannot create buffer head SLAB cache\n");
          unlock_bh_free();
          return 0;
       }
    }
#endif
    
    bh = int_get_bh();
    if (bh)
       return bh;
    return __get_bh_wait();
}

#if (LINUX_BUFFER_CACHE)

//
//  The 2.2.X buffer cache is not SMP thread safe,
//  so lock the kernel when we enter it to prevent memory corruption
//  from occurring on SMP systems.
//

ULONG pReadDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		      ULONG sectors, ULONG readAhead)
{
    register ULONG i, bytesRead = 0;
    register ULONG bps, lba;
    register NWDISK *NWDisk;
    register ULONG read_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    struct buffer_head *bh[sectors];

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;
    
#if (VERBOSE)
    NWFSPrint("read disk-%d lba-%d for %d sectors\n", 
	     (int)disk, (int)StartingLBA, (int)sectors);
#endif

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;

    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read-bc)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }
    
#if (LINUX_22)
    lock_kernel();
#endif

    for (i=0; i < blocks; i++)
    {
       if (!(bh[i] = getblk((ULONG)NWDisk->PhysicalDiskHandle,
			     lba + i, blocksize)))
       {
	  for (i=0; i < blocks; i++)
	     if (bh[i])
		brelse(bh[i]);
#if (LINUX_22)
          unlock_kernel();
#endif
	  return 0;
       }
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
  	  NWFSPrint("getblk buffer head (read) %d-0x%08X\n", (int)i, 
		        (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
#if (LINUX_22)
       unlock_kernel();
#endif
       return 0;
    }
#endif
    
    ll_rw_block(READ, blocks, bh);
    for(i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (buffer_uptodate(bh[i]))
       {
	  NWFSCopy(&Sector[i * blocksize], bh[i]->b_data, blocksize);
	  bytesRead += blocksize;
       }
       else
       {
          NWFSPrint("read error %d of %d  bytes read %d blksize-%d\n", 
		   (int)i, (int)blocks, (int)bytesRead,
		   (int)blocksize);
          read_error = 1;
       }
       brelse(bh[i]);
    }
#if (LINUX_22)
    unlock_kernel();
#endif

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (read_error ? 0 : bytesRead);

}

ULONG pWriteDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		       ULONG sectors, ULONG readAhead)
{
    register ULONG i, bytesWritten = 0;
    register ULONG bps, lba;
    struct buffer_head *bh[sectors];
    register ULONG write_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    register NWDISK *NWDisk;

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

#if (VERBOSE)
    NWFSPrint("write disk-%d lba-%d for %d sectors\n",
	     (int)disk, (int)StartingLBA, (int)sectors);
#endif

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (write-bc)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

#if (LINUX_22)
    lock_kernel();
#endif
    for (i=0; i < blocks; i++)
    {
       if (!(bh[i] = getblk((ULONG)NWDisk->PhysicalDiskHandle,
			    lba + i, blocksize)))
       {
	  for (i=0; i < blocks; i++)
	     if (bh[i])
		brelse(bh[i]);
#if (LINUX_22)
          unlock_kernel();
#endif
	  return 0;
       }
       NWFSCopy(bh[i]->b_data, &Sector[i * blocksize], blocksize);
       mark_buffer_uptodate(bh[i], 1);

#if (LINUX_24)
       mark_buffer_dirty(bh[i]);
#else
       mark_buffer_dirty(bh[i], 0);
#endif
       bytesWritten += blocksize;
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
     	  NWFSPrint("getblk buffer head (write) %d-0x%08X\n", (int)i, 
		    (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
#if (LINUX_22)
       unlock_kernel();
#endif
       return 0;
    }
#endif
    
    ll_rw_block(WRITE, blocks, bh);
    for (i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (!buffer_uptodate(bh[i]))
	  write_error = 1;
       brelse(bh[i]);
    }
#if (LINUX_22)
    unlock_kernel();
#endif

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (write_error ? 0 : bytesWritten);

}

ULONG pZeroFillDiskSectors(ULONG disk, ULONG StartingLBA, ULONG sectors,
			  ULONG readAhead)
{
    register ULONG i, bytesWritten = 0;
    register ULONG bps, lba;
    struct buffer_head *bh[sectors];
    register ULONG write_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    register NWDISK *NWDisk;

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

#if (VERBOSE)
    NWFSPrint("zero disk-%d lba-%d for %d sectors\n", 
	     (int)disk, (int)StartingLBA, (int)sectors);
#endif
    
    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;

    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {

       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (fill-bc)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }
    
#if (LINUX_22)
    lock_kernel();
#endif
    for (i=0; i < blocks; i++)
    {
       if (!(bh[i] = getblk((ULONG)NWDisk->PhysicalDiskHandle,
			  lba + i, blocksize)))
       {
	  for (i=0; i < blocks; i++)
	     if (bh[i])
		brelse(bh[i]);
#if (LINUX_22)
          unlock_kernel();
#endif
	  return 0;
       }
       NWFSSet(bh[i]->b_data, 0, blocksize);
       mark_buffer_uptodate(bh[i], 1);
#if (LINUX_24)
       mark_buffer_dirty(bh[i]);
#else
       mark_buffer_dirty(bh[i], 0);
#endif
       bytesWritten += blocksize;
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
     	  NWFSPrint("getblk buffer head (fill) %d-0x%08X\n", (int)i, 
		    (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
#if (LINUX_22)
       unlock_kernel();
#endif
       return 0;
    }
#endif
    
    ll_rw_block(WRITE, blocks, bh);
    for (i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (!buffer_uptodate(bh[i]))
	  write_error = 1;
       brelse(bh[i]);
    }
#if (LINUX_22)
    unlock_kernel();
#endif

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (write_error ? 0 : bytesWritten);
}

#else

//
// this case assumes we will use the NWFS buffer cache and not allow
// cache sharing with the linux buffer cache.
//

//
//  Linux 2.4
//

#if (LINUX_24)

void end_io(struct buffer_head *bh, int uptodate)
{
    mark_buffer_uptodate(bh, uptodate);
    unlock_buffer(bh);
    return;
}

ULONG pReadDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		      ULONG sectors, ULONG readAhead)
{
    register ULONG i, j, bytesRead = 0;
    register ULONG bps, lba;
    register NWDISK *NWDisk;
    register ULONG read_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    struct buffer_head *bh[sectors];
    register struct page *page = virt_to_page(Sector);

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       bh[i] = nwfs_get_bh();
       if (!bh[i])
       {
	  for (j=0; j < i; j++)
	     if (bh[j])
		nwfs_put_bh(bh[j]);
	  return 0;
       }
    }

    for (i=0; i < blocks; i++)
    {
       bh[i]->b_this_page = bh[(i + 1) % blocks]; // create circular list
       bh[i]->b_state = 0;
       bh[i]->b_next_free = (struct buffer_head *)NULL;
       bh[i]->b_size = blocksize;
       bh[i]->b_data = (char *)&Sector[i * blocksize];
       bh[i]->b_list = BUF_CLEAN;
       bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       bh[i]->b_blocknr = lba + i;
       bh[i]->b_count.counter = 1;
       bh[i]->b_flushtime = 0;
       bh[i]->b_end_io = end_io;
       bh[i]->b_private = NULL;
       bh[i]->b_page = page;
       set_bit(BH_Mapped, &bh[i]->b_state);
       set_bit(BH_Lock, &bh[i]->b_state);
       clear_bit(BH_Uptodate, &bh[i]->b_state);
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
          NWFSPrint("nwfs buffer head (read) %d-0x%08X\n", (int)i, 
		 (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
       return 0;
    }
#endif
    
    for (i=0; i < blocks; i++)
       submit_bh(READ, bh[i]);

    for (i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (buffer_uptodate(bh[i]))
	  bytesRead += blocksize;
       else
	  read_error = 1;
       nwfs_put_bh(bh[i]);
    }

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (read_error ? 0 : bytesRead);

}

ULONG pWriteDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		       ULONG sectors, ULONG readAhead)
{
    register ULONG i, j, bytesWritten = 0;
    register ULONG bps, lba;
    register ULONG write_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    register NWDISK *NWDisk;
    struct buffer_head *bh[sectors];
    register struct page *page = virt_to_page(Sector);

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       bh[i] = nwfs_get_bh();
       if (!bh[i])
       {
	  for (j=0; j < i; j++)
	     if (bh[j])
		nwfs_put_bh(bh[j]);
	  return 0;
       }
    }

    for (i=0; i < blocks; i++)
    {
       bh[i]->b_this_page = bh[(i + 1) % blocks]; // create circular list
       bh[i]->b_state = 0;
       bh[i]->b_next_free = (struct buffer_head *) NULL;
       bh[i]->b_size = blocksize;
       bh[i]->b_data = (char *)&Sector[i * blocksize];
       bh[i]->b_list = BUF_CLEAN;
       bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       bh[i]->b_blocknr = lba + i;
       bh[i]->b_count.counter = 1;
       bh[i]->b_flushtime = 0;
       bh[i]->b_end_io = end_io;
       bh[i]->b_private = NULL;
       bh[i]->b_page = page;
       set_bit(BH_Req, &bh[i]->b_state);
       set_bit(BH_Mapped, &bh[i]->b_state);
       set_bit(BH_Uptodate, &bh[i]->b_state);
       set_bit(BH_Dirty, &bh[i]->b_state);
       set_bit(BH_Lock, &bh[i]->b_state);
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
          NWFSPrint("nwfs buffer head (write) %d-0x%08X\n", (int)i, 
		 (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
       return 0;
    }
#endif
    
    for (i=0; i < blocks; i++)
       submit_bh(WRITE, bh[i]);

    for (i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (buffer_uptodate(bh[i]))
	  bytesWritten += blocksize;
       else
	  write_error = 1;
       nwfs_put_bh(bh[i]);
    }

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (write_error ? 0 : bytesWritten);

}

ULONG pZeroFillDiskSectors(ULONG disk, ULONG StartingLBA, ULONG sectors,
			  ULONG readAhead)
{
    register ULONG i, j, bytesWritten = 0;
    register ULONG bps, lba;
    register ULONG write_error = 0;
    register ULONG rsize, blocks, blocksize, spb;
    register NWDISK *NWDisk;
    struct buffer_head *bh[sectors];
    register struct page *page = virt_to_page(ZeroBuffer);

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       bh[i] = nwfs_get_bh();
       if (!bh[i])
       {
	  for (j=0; j < i; j++)
	     if (bh[j])
		nwfs_put_bh(bh[j]);
	  return 0;
       }
    }

    for (i=0; i < blocks; i++)
    {
       bh[i]->b_this_page = bh[(i + 1) % blocks]; // create circular list
       bh[i]->b_state = 0;
       bh[i]->b_next_free = (struct buffer_head *) NULL;
       bh[i]->b_size = blocksize;
       bh[i]->b_data = (char *) ZeroBuffer;
       bh[i]->b_list = BUF_CLEAN;
       bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       bh[i]->b_blocknr = lba + i;
       bh[i]->b_count.counter = 1;
       bh[i]->b_flushtime = 0;
       bh[i]->b_end_io = end_io;
       bh[i]->b_private = NULL;
       bh[i]->b_page = page;
       set_bit(BH_Req, &bh[i]->b_state);
       set_bit(BH_Mapped, &bh[i]->b_state);
       set_bit(BH_Uptodate, &bh[i]->b_state);
       set_bit(BH_Dirty, &bh[i]->b_state);
       set_bit(BH_Lock, &bh[i]->b_state);
    }

#if (DUMP_BUFFER_HEADS)
    if (1)
    {
       for (i=0; i < blocks; i++)
       {
          NWFSPrint("nwfs buffer head (fill) %d-0x%08X\n", (int)i, 
		 (unsigned)bh[i]);
          dumpRecordBytes(bh[i], sizeof(struct buffer_head));
          dumpRecord(bh[i], sizeof(struct buffer_head));
       }
       return 0;
    }
#endif
    
    for (i=0; i < blocks; i++)
       submit_bh(WRITE, bh[i]);

    for (i=0; i < blocks; i++)
    {
       wait_on_buffer(bh[i]);
       if (buffer_uptodate(bh[i]))
	  bytesWritten += blocksize;
       else
	  write_error = 1;
       nwfs_put_bh(bh[i]);
    }

#if (PROFILE_AIO)
    profile_complete();
#endif

    return (write_error ? 0 : bytesWritten);
}

//
//
//

void end_asynch_io(struct buffer_head *bh, int uptodate)
{
    register int i;
    ASYNCH_IO *io = (ASYNCH_IO *)bh->b_private;
    bh->b_private = 0;

    mark_buffer_uptodate(bh, uptodate);
    unlock_buffer(bh);

    if (!io)
    {
       NWFSPrint("nwfs:  aio callback has NULL handle\n");
       return;
    }

    if (!test_bit(BH_Uptodate, &bh->b_state))
       io->ccode = ASIO_IO_ERROR;

    atomic_inc((atomic_t *)&io->complete);
    if (io->complete == io->count)
    {
       for (i=0; i < io->count; i++)
       {
          nwfs_put_bh(io->bh[i]);
          io->bh[i] = 0;
       }

       insert_callback(io);

#if (PROFILE_AIO)
       profile_complete();
#endif
    }
    return;
}

ULONG aReadDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		      ULONG sectors, ULONG readAhead, ASYNCH_IO *io)
{
    register ULONG i, j, bytesRead = 0;
    register ULONG bps, lba;
    register NWDISK *NWDisk;
    register struct page *page = virt_to_page(Sector);
    register ULONG rsize, blocks, blocksize, spb;

    if (!page)
    {
       NWFSPrint("nwfs:  page context was NULL in aReadDiskSectors\n");
       return 0;
    }

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       io->bh[i] = nwfs_get_bh();
       if (!io->bh[i])
       {
	  for (j=0; j < i; j++)
	     if (io->bh[j])
		nwfs_put_bh(io->bh[j]);
	  return 0;
       }
    }

    io->ccode = 0;
    io->count = blocks;
    io->complete = 0;
    for (i=0; i < blocks; i++)
    {
       io->bh[i]->b_this_page = io->bh[(i + 1) % blocks]; 
       io->bh[i]->b_state = 0;
       io->bh[i]->b_next_free = (struct buffer_head *)NULL;
       io->bh[i]->b_size = blocksize;
       io->bh[i]->b_data = (char *)&Sector[i * blocksize];
       io->bh[i]->b_list = BUF_CLEAN;
       io->bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       io->bh[i]->b_blocknr = lba + i;
       io->bh[i]->b_count.counter = 1;
       io->bh[i]->b_flushtime = 0;
       io->bh[i]->b_end_io = end_asynch_io;
       io->bh[i]->b_private = io;
       io->bh[i]->b_page = page;
       set_bit(BH_Mapped, &io->bh[i]->b_state);
       set_bit(BH_Lock, &io->bh[i]->b_state);
       clear_bit(BH_Uptodate, &io->bh[i]->b_state);
    }

    for (i=0; i < blocks; i++)
       submit_bh(READ, io->bh[i]);

#if (POST_IMMEDIATE)
    run_task_queue(&tq_disk);
#endif

    bytesRead = (sectors * bps);
    return (bytesRead);

}

ULONG aWriteDiskSectors(ULONG disk, ULONG StartingLBA, BYTE *Sector,
		       ULONG sectors, ULONG readAhead, ASYNCH_IO *io)
{
    register ULONG i, j, bytesWritten = 0;
    register ULONG bps, lba;
    register NWDISK *NWDisk;
    register struct page *page = virt_to_page(Sector);
    register ULONG rsize, blocks, blocksize, spb;

    if (!page)
    {
       NWFSPrint("nwfs:  page context was NULL in aWriteDiskSectors\n");
       return 0;
    }

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       io->bh[i] = nwfs_get_bh();
       if (!io->bh[i])
       {
	  for (j=0; j < i; j++)
	     if (io->bh[j])
		nwfs_put_bh(io->bh[j]);
	  return 0;
       }
    }

    io->ccode = 0;
    io->count = blocks;
    io->complete = 0;
    for (i=0; i < blocks; i++)
    {
       io->bh[i]->b_this_page = io->bh[(i + 1) % blocks]; 
       io->bh[i]->b_state = 0;
       io->bh[i]->b_next_free = (struct buffer_head *) NULL;
       io->bh[i]->b_size = blocksize;
       io->bh[i]->b_data = (char *)&Sector[i * blocksize];
       io->bh[i]->b_list = BUF_CLEAN;
       io->bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       io->bh[i]->b_blocknr = lba + i;
       io->bh[i]->b_count.counter = 1;
       io->bh[i]->b_flushtime = 0;
       io->bh[i]->b_end_io = end_asynch_io;
       io->bh[i]->b_private = io;
       io->bh[i]->b_page = page;
       set_bit(BH_New, &io->bh[i]->b_state);
       set_bit(BH_Mapped, &io->bh[i]->b_state);
       set_bit(BH_Uptodate, &io->bh[i]->b_state);
       set_bit(BH_Dirty, &io->bh[i]->b_state);
       set_bit(BH_Lock, &io->bh[i]->b_state);
    }

    for (i=0; i < blocks; i++)
       submit_bh(WRITE, io->bh[i]);

#if (POST_IMMEDIATE)
    run_task_queue(&tq_disk);
#endif
    
    bytesWritten = (sectors * bps);
    return (bytesWritten);

}

ULONG aZeroFillDiskSectors(ULONG disk, ULONG StartingLBA, ULONG sectors,
			  ULONG readAhead, ASYNCH_IO *io)
{
    register ULONG i, j, bytesWritten = 0;
    register ULONG bps, lba;
    register NWDISK *NWDisk;
    register struct page *page = virt_to_page(ZeroBuffer);
    register ULONG rsize, blocks, blocksize, spb;

    if (!page)
    {
       NWFSPrint("nwfs:  page context was NULL in aZeroFillDiskSectors\n");
       return 0;
    }

    NWDisk = SystemDisk[disk];
    bps = NWDisk->BytesPerSector;
    blocksize = NWDisk->DeviceBlockSize;

    rsize = sectors * bps;
    blocks = rsize / blocksize;
    if (!blocks)
       return 0;
    spb = blocksize / bps;
    
    lba = StartingLBA / spb;
    if (StartingLBA % spb)
    {
       NWFSPrint("request not %d block aligned (%d) sectors-%d lba-%d (read)\n",
	       (int)blocksize, (int)(StartingLBA % spb), (int)sectors,
	       (int)StartingLBA);
       return 0;
    }

    for (i=0; i < blocks; i++)
    {
       io->bh[i] = nwfs_get_bh();
       if (!io->bh[i])
       {
	  for (j=0; j < i; j++)
	     if (io->bh[j])
		nwfs_put_bh(io->bh[j]);
	  return 0;
       }
    }

    io->ccode = 0;
    io->count = blocks;
    io->complete = 0;
    for (i=0; i < blocks; i++)
    {
       io->bh[i]->b_this_page = io->bh[(i + 1) % blocks]; 
       io->bh[i]->b_state = 0;
       io->bh[i]->b_next_free = (struct buffer_head *) NULL;
       io->bh[i]->b_size = blocksize;
       io->bh[i]->b_data = (char *) ZeroBuffer;
       io->bh[i]->b_list = BUF_CLEAN;
       io->bh[i]->b_dev = (int)NWDisk->PhysicalDiskHandle;
       io->bh[i]->b_blocknr = lba + i;
       io->bh[i]->b_count.counter = 1;
       io->bh[i]->b_flushtime = 0;
       io->bh[i]->b_end_io = end_asynch_io;
       io->bh[i]->b_private = io;
       io->bh[i]->b_page = page;
       set_bit(BH_New, &io->bh[i]->b_state);
       set_bit(BH_Mapped, &io->bh[i]->b_state);
       set_bit(BH_Uptodate, &io->bh[i]->b_state);
       set_bit(BH_Dirty, &io->bh[i]->b_state);
       set_bit(BH_Lock, &io->bh[i]->b_state);
    }

    for (i=0; i < blocks; i++)
       submit_bh(WRITE, io->bh[i]);

#if (POST_IMMEDIATE)
    run_task_queue(&tq_disk);
#endif

    bytesWritten = (sectors * bps);
    return (bytesWritten);
}

#endif

#endif

void SyncDevice(ULONG disk)
{
    sync_dev((int)disk);
    return;
}

#if (DO_ASYNCH_IO)
ULONG release_aio(ASYNCH_IO *io)
{
#if (LINUX_SLEEP)
    SignalSemaphore((struct semaphore *)io->call_back_parameter);
#endif
    return 0;
}
#endif

ULONG ReadDiskSectors(ULONG disk, ULONG LBA, BYTE *Sector,
			     ULONG sectors, ULONG readAhead)
{
#if (DO_ASYNCH_IO)
    ASYNCH_IO io;
#if (LINUX_SLEEP)
    NWFSInitSemaphore(raw_semaphore);
#endif

    NWFSSet(&io, 0, sizeof(ASYNCH_IO));

    io.command = ASYNCH_READ;
    io.disk = disk;
    io.flags = ASIO_INTR_CALLBACK;
    io.sector_offset = LBA;
    io.sector_count = sectors;
    io.buffer = Sector;
    io.call_back_routine = release_aio;
#if (LINUX_SLEEP)
    io.call_back_parameter = (ULONG) &raw_semaphore;
#endif

    insert_io(disk, &io);

    RunAsynchIOQueue(disk);

#if (LINUX_SLEEP)
    WaitOnSemaphore(&raw_semaphore);
#endif

    return (ULONG)(io.return_code);
#else
    return (pReadDiskSectors(disk, LBA, Sector, sectors, readAhead));
#endif
}

ULONG WriteDiskSectors(ULONG disk, ULONG LBA, BYTE *Sector,
			      ULONG sectors, ULONG readAhead)
{
#if (DO_ASYNCH_IO)
    ASYNCH_IO io;
#if (LINUX_SLEEP)
    NWFSInitSemaphore(raw_semaphore);
#endif

    NWFSSet(&io, 0, sizeof(ASYNCH_IO));

    io.command = ASYNCH_WRITE;
    io.disk = disk;
    io.flags = ASIO_INTR_CALLBACK;
    io.sector_offset = LBA;
    io.sector_count = sectors;
    io.buffer = Sector;
    io.call_back_routine = release_aio;
#if (LINUX_SLEEP)
    io.call_back_parameter = (ULONG) &raw_semaphore;
#endif

    insert_io(disk, &io);

    RunAsynchIOQueue(disk);

#if (LINUX_SLEEP)
    WaitOnSemaphore(&raw_semaphore);
#endif

    return (ULONG)(io.return_code);
#else
    return (pWriteDiskSectors(disk, LBA, Sector, sectors, readAhead));
#endif
}

ULONG ZeroFillDiskSectors(ULONG disk, ULONG StartingLBA,
				 ULONG sectors, ULONG readAhead)
{
#if (DO_ASYNCH_IO)
    ASYNCH_IO io;
#if (LINUX_SLEEP)
    NWFSInitSemaphore(raw_semaphore);
#endif

    NWFSSet(&io, 0, sizeof(ASYNCH_IO));

    io.command = ASYNCH_FILL;
    io.disk = disk;
    io.flags = ASIO_INTR_CALLBACK;
    io.sector_offset = StartingLBA;
    io.sector_count = sectors;
    io.buffer = NULL;
    io.call_back_routine = release_aio;
#if (LINUX_SLEEP)
    io.call_back_parameter = (ULONG) &raw_semaphore;
#endif

    insert_io(disk, &io);

    RunAsynchIOQueue(disk);

#if (LINUX_SLEEP)
    WaitOnSemaphore(&raw_semaphore);
#endif

    return (ULONG)(io.return_code);
#else
    return (pZeroFillDiskSectors(disk, StartingLBA, sectors, readAhead));
#endif
}

#endif
#endif

