#ident "$Id: mount_bind.c,v 1.1 2000/06/01 19:29:31 hpa Exp $"
/* ----------------------------------------------------------------------- *
 *   
 *  mount_bind.c      - module to mount a local filesystem if possible;
 *			otherwise create a symlink.
 *
 *   Copyright 2000 Transmeta Corporation - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
 *   USA; either version 2 of the License, or (at your option) any later
 *   version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>

#define MODULE_MOUNT
#include "automount.h"

#define MODPREFIX "mount(bind): "
int mount_version = AUTOFS_MOUNT_VERSION; /* Required by protocol */

static int bind_works = 0;

int mount_init(void **context)
{
  char *tmp = tempnam(NULL, "auto");
  int err;
  struct stat st1, st2;

  if (tmp == NULL)
    return 0;

  if (mkdir(tmp, 0700) == -1) {
    free(tmp);
    return 0;
  }

  err = spawnl(LOG_DEBUG, PATH_MOUNT, PATH_MOUNT, "--bind", "/", tmp, NULL);

  if (err == 0 &&
      stat("/", &st1) == 0 &&
      stat(tmp, &st2) == 0 &&
      st1.st_dev == st2.st_dev &&
      st1.st_ino == st2.st_ino) {
    bind_works = 1;
  }
  syslog(LOG_DEBUG, MODPREFIX "bind_works = %d\n", bind_works);
  spawnl(LOG_DEBUG, PATH_UMOUNT, PATH_UMOUNT, tmp, NULL);

  rmdir(tmp);
  free(tmp);
  return 0;
}

int mount_mount(const char *root, const char *name, int name_len,
		const char *what, const char *fstype, const char *options,
		void *context)
{
  char *fullpath;
  int err;
  int i;

  fullpath = alloca(strlen(root)+name_len+2);
  if ( !fullpath ) {
    syslog(LOG_ERR, MODPREFIX "alloca: %m");
    return 1;
  }
  sprintf(fullpath, "%s/%s", root, name);
  i = strlen(fullpath);
  while(--i > 0 && fullpath[i] == '/')
    fullpath[i] = '\0';

  if (bind_works) {
    syslog(LOG_DEBUG, MODPREFIX "calling mkdir_path %s", fullpath);
    if ( mkdir_path(fullpath, 0555) && errno != EEXIST ) {
      syslog(LOG_NOTICE, MODPREFIX "mkdir_path %s failed: %m", fullpath);
      return 1;
    }
   
    syslog(LOG_DEBUG, MODPREFIX "calling mount --bind %s %s",
	   what, fullpath);
    err = spawnl(LOG_NOTICE, PATH_MOUNT, PATH_MOUNT, "--bind",
		 what, fullpath, NULL);

    if ( err ) {
      rmdir_path(fullpath);
      return 1;
    } else {
      syslog(LOG_DEBUG, MODPREFIX "mounted %s type %s on %s",
	     what, fstype, fullpath);
      return 0;
    }
  } else {
    char *cp;
    char *basepath = alloca(strlen(fullpath) + 1);

    strcpy(basepath, fullpath);
    cp = strrchr(basepath, '/');
    if (cp != NULL && cp != basepath)
      *cp = '\0';

    syslog(LOG_DEBUG, MODPREFIX "calling mkdir_path %s", basepath);
    if ( mkdir_path(basepath, 0555) ) {
      syslog(LOG_NOTICE, MODPREFIX "mkdir_path %s failed: %m", basepath);
      return 1;
    }
    
    if ( symlink(what, fullpath) && errno != EEXIST ) {
      syslog(LOG_NOTICE, MODPREFIX "failed to create local mount %s -> %s", fullpath, what);
      rmdir_path(fullpath);
      
      return 1;
    } else {
      syslog(LOG_DEBUG, MODPREFIX "symlinked %s -> %s", fullpath, what);
      return 0;
    }
  }
}

int mount_done(void *context)
{
  return 0;
}
