##########################################################################
#
# $Id: XMLReport.pm,v 1.7 2006/09/17 19:18:35 pkaluski Exp $
# $Name: Stable_0_16 $
# 
# Object of this class creates an XML report from csv dumps.
#
# $Log: XMLReport.pm,v $
# Revision 1.7  2006/09/17 19:18:35  pkaluski
# Adjusted command line options to new key definitions
#
# Revision 1.6  2006/04/09 15:42:19  pkaluski
# Small code clean-up. Each module has comprehensive POD
#
# Revision 1.5  2006/02/10 22:32:16  pkaluski
# Major redesign in progress. Updated POD. Works.
#
# Revision 1.4  2006/01/21 20:39:47  pkaluski
# Improved lcsvdiff output format
#
# Revision 1.3  2006/01/14 12:52:33  pkaluski
# New tool design in progress
#
# Revision 1.2  2006/01/07 22:56:17  pkaluski
# Redesigned simple reporting works. Diff reports still to be implemented.
#
# Revision 1.1  2006/01/07 21:59:28  pkaluski
# Initial revision. Works.
#
#
##########################################################################
package LRpt::XMLReport;

use strict;
use Getopt::Long;
use LRpt::Config;
use LRpt::Collection;
use LRpt::JarReader;
use LRpt::RKeysRdr;
use File::Basename;

use vars qw( @EXPORT @ISA );
@ISA = qw( Exporter );
@EXPORT = qw( create_report );

our %all_rows = ();
our %metadata = ();
our $rldf     = "";
our @report   = ();

our $stmt_fname = "";
our $diffs = "";

our $rkeys_rdr  = "";

=head1 NAME

LRpt::XMLReport - A module for converting outputs generated by 
LReport tools to XML format.

=head1 SYNOPSIS

    lrptxml.pl --selects=selects.txt --keys_file=keys.txt data 
  
    lrptxml.pl --diffs --selects=selects.txt --keys_file=rkeys.txt diffs.txt 

=head1 DESCRIPTION

C<LRpt::XMLReport> is a part of L<C<LRpt>|LRpt> (B<LReport>) library.
It used to convert outputs generated by other tools from B<LReport> suite to
I<XML> format. You should not call this class directly from your code. 
Instead you should use B<lrptxml.pl> tool, which is a simple wrapper around 
this module. It looks like this:

  use strict;
  use LRpt::XMLReport;
  
  create_report( @ARGV );
  

=head1 COMMAND LINE SWITCHES

=over 4

=item --selects=file

Optional. Name of a selects file (see doc). They are included in the 
report for a select of corresponding name. If selects are not given, report
will not contain selects.

=item --diffs

Switches working mode to read output of B<lcsvdiff.pl>. If not given
the working mode is reading I<csv> files from a directory

=item --keys_file=keys_file

File name containing row key definitions.

=back

=head1 HOW IS THIS MANUAL ORGANIZED

On this page you will find details on C<LRpt::XMLReport> package. You also find
details on how to use B<LReport> tools for reporting. 

Firstly you will find a list all methods provided by the class.

Then I will show you a step by step example, in which I will explain you
what can be done and how it can be done. 

Then I will give some additional details, which I did not include in
the example to avoid making it too complex.

On the end I will describe existing XSLT stylesheets for converting I<XML>
reports to other formats. Currently only conversion to RTF is available.

=head1 METHODS

In this sections you will find a more or less complete listing of all
methods provided by the package. 

=cut

########################################################################

=head2 C<create_report>

  create_report( @ARGV );

Main public function. The only one exported. See 
L</COMMAND LINE SWITCHES> for meaning of parameters.

=cut

########################################################################
sub create_report
{
    local ( @ARGV ) = @_;
    my @key       = ();
    my @key_cols  = ();
    my $keys_file;
    my $global_keys;
    my $help = "";
    GetOptions( "selects=s" => \$stmt_fname,
                "diffs" => \$diffs,
                "help" => \$help,
                "key=s"      => \@key,
                "key_cols=s"  => \@key_cols,
                "keys_file=s" => \$keys_file,
                "global_keys" =>  \$global_keys );

    if( $help ){
        print_usage();
    }

    if( @key or $keys_file or @key_cols ){
        $rkeys_rdr = LRpt::RKeysRdr->new( 'fname'    => $keys_file,
                                      'key'      => \@key,
                                      'key_cols' => \@key_cols,
                                      'global_keys' => $global_keys );
    }

    if( $diffs ){
        load_diffs();
    }else{
        load_csvs();
    }
    generate_report();
    print "\n";
}

########################################################################

=head2 C<load_diffs>

  load_diffs();

Loads input data from files generated by L<C<LRpt::CSVDiff>|LRpt::CSVDiff>
module.

=cut

########################################################################
sub load_diffs
{
    my $line = undef;
    my $file = "";
    my $config = LRpt::Config->new();
    my $ext = $config->get_value( 'ext' );
    MAIN: while( $line = <> ){
        if( $line =~ /lcsvdiff (\S+) (\S+)/ ){
            $file = $1;
            my $name = basename( $file );
            $name =~ s/\.$ext$//; 
            $line = load_one_select( $name );
            redo MAIN;
        }elsif( !$line ){
            return;
        }
        else{
            die "Unexpected line $line";
        }
    }
}

########################################################################

=head2 C<load_one_select>

  load_one_select();

Loads one select from the input diff file.

=cut

########################################################################
sub load_one_select
{
    my $file = shift;
    my $line = <>;
    my $cols_string = ( $line =~ /^SCHEMA: (.*)$/ )[ 0 ];
    if( not $cols_string ) {
        die "Expected SCHEMA line not given";
    }else{
        $LRpt::XMLReport::metadata{ $file } = [ split( /\t/, $cols_string ) ];
    }
    LINE: while( $line = <> ){
        if( $line =~ /^ROW\( .* \): .*$/ ){
            $line = load_ROW_entry( $line, $file );
            if( not defined $line ){
                return undef;
            }else{
                redo LINE;
            }
#        }elsif( $line =~ /^UPD\( .* \): \w+: .*$/ ){
#            $line = load_UPD_entry( $line, $file );
        }elsif( $line =~ /^DEL\( .* \): .*$/ ){
            load_DEL_entry( $line, $file );
        }elsif( $line =~ /^INS\( .* \): .*$/ ){
            load_INS_entry( $line, $file );
        }else{
            return $line;
        }
    }
    return undef;
}

########################################################################

=head2 C<load_ROW_entry>

  load_ROW_entry( $line, $file );

Loads one ROW type entry from the diff file.

=cut

########################################################################
sub load_ROW_entry
{
    my $line = shift; 
    my $file = shift;
    my $row  = {};

    my ( $rkey, $values_string ) =
                      ( $line =~ /^ROW\( (.*) \): (.*)$/ )[ 0, 1 ];
     
    my @values = split( /\t/, $values_string );
    my @cols = @{ $LRpt::XMLReport::metadata{ $file } };
    @$row{ @cols } = @values;
    
    my $col; 
    my $value;
    my $diff_found = 0;
    while( $line = <> ){
        ( $rkey, $col, $value ) =
                      ( $line =~ /^UPD\( (.*) \): (\w+): (.*)$/ )[ 0, 1, 2 ];
        if( $rkey ){
            $diff_found = 1;
            my ( $old_value, $new_value ) = 
                                 ( $value =~ /^(.*) ==#> (.*)$/ )[ 0, 1 ];
                delete $row->{ $col };
                $row->{ $col }->{ 'old_value' } = $old_value;
                $row->{ $col }->{ 'new_value' } = $new_value;
            next;
        }else{
            if( $diff_found ){
                include_in_report( $file, $row, "different" );
            }else{
                include_in_report( $file, $row, "equal" );
            }
            return $line;
        }
    }
    
    if( $diff_found ){
        include_in_report( $file, $row, "different" );
    }else{
        include_in_report( $file, $row, "equal" );
    }
    return undef;
}

########################################################################

=head2 C<load_DEL_entry>

  load_DEL_entry( $line, $file );

Loads one DEL type entry from the diff file.

=cut

########################################################################
sub load_DEL_entry
{
    my $line = shift; 
    my $file = shift;
    my $row  = {};
    my $stored_rkey = "";

    my ( $rkey, $values_string ) =
                      ( $line =~ /^DEL\( (.*) \): (.*)$/ )[ 0, 1 ];
     
    my @values = split( /\t/, $values_string );
    my @cols = @{ $LRpt::XMLReport::metadata{ $file } };
    @$row{ @cols } = @values;

    include_in_report( $file, $row, "missing" );
}

########################################################################

=head2 C<load_INS_entry>

  load_INS_entry( $line, $file );

Loads one INS type entry from the diff file.

=cut

########################################################################
sub load_INS_entry
{
    my $line = shift; 
    my $file = shift;
    my $row  = {};
    my $stored_rkey = "";

    my ( $rkey, $values_string ) =
                      ( $line =~ /^INS\( (.*) \): (.*)$/ )[ 0, 1 ];
     
    my @values = split( /\t/, $values_string );
    my @cols = @{ $LRpt::XMLReport::metadata{ $file } };
    @$row{ @cols } = @values;

    include_in_report( $file, $row, "additional" );
}

########################################################################

=head2 C<load_csvs>

  load_csvs();

Loads data from csv files

=cut

########################################################################
sub load_csvs
{
    my $file = "";
    my @files = ();
    if( !@ARGV ){
        die "CSV files names not given";
    }else{
        $file = shift( @ARGV );
        if( -d $file ){
            if( @ARGV ){
                die "When directory name given, other files " .
                    "must not be specified";
            }else{
                my $config = LRpt::Config->new();
                my $ext = $config->get_value( 'ext' );
                @files = glob( "$file/*.$ext" );
            }
        }else{
            while( $file = shift( @ARGV ) ){
                if( -f $file ){
                    die "No such file $file";
                }
                push( @files, $file );
            }
        }
    }
    foreach $file ( @files ){
        load_csv_file( $file );
    }      
}

########################################################################

=head2 C<load_csv_file>

  load_csv_file();

Loads data from one csv file

=cut

########################################################################
sub load_csv_file
{
    my $file = shift;
    my $config = LRpt::Config->new();
    my $ext = $config->get_value( 'ext' );
    my $name = basename( $file );
    $name =~ s/\.$ext$//; 
    my $coll = "";
    my $rkey = "";
    if( $rkeys_rdr and $rkey = $rkeys_rdr->find_key( $name ) ){
        $coll = LRpt::Collection->new_from_csv( 'name'      => $name,
                                                'data_file' => $file,
                                                'key'       => $rkey );
    }else{
        $coll = LRpt::Collection->new_from_csv( 'name'      => $name,
                                                'data_file' => $file );
    }
        
    $metadata{ $name } = $coll->get_columns();
    if( $rkey ){
        my @rkey_values = sort $coll->get_key_values();
        if( ! @rkey_values ){
            $all_rows{ $name } = [];
        }else{        
            foreach my $value ( @rkey_values ){
                include_in_report( $name, $coll->get_row( $value ), 'equal' );
            }
        }
    }else{
        my $rows = $coll->get_all_rows();
        if( ! @$rows ){
            $all_rows{ $name } = [];
        }else{        
            foreach my $row ( @$rows ){
                include_in_report( $name, $row, 'equal' );
            }
        }
    }    
}

########################################################################

=head2 C<include_in_report>

  include_in_report( $name, $row, $type );

Adds a row to a list, which will be used for report generation

=cut

########################################################################
sub include_in_report
{
    my $name = shift;
    my $row  = shift;
    my $type = shift;

    if( not exists $all_rows{ $name } ){
        $all_rows{ $name } = [];
    }
    if( $type eq "equal" ){
        push( @{ $all_rows{ $name } }, [ $row, 'equal' ] );
    }
    if( $type eq "different" ){
        push( @{ $all_rows{ $name } }, [ $row, 'different' ] );
    }
    if( $type eq "additional" ){
        push( @{ $all_rows{ $name } }, [ $row, 'additional' ] );
    }
    if( $type eq "missing" ){
        push( @{ $all_rows{ $name } }, [ $row, 'missing' ] );
    }
}

########################################################################

=head2 C<generate_report>

  generate_report();

Generate XML report.

=cut

########################################################################
sub generate_report
{
    foreach my $select ( sort keys %all_rows ){
        process_data( $select );
    }
    print_report();
}

########################################################################

=head2 C<process_data>

  process_data( $select );

Creates report for one select

=cut

########################################################################
sub process_data
{
    my $select = shift;
    my $data = $all_rows{ $select };

    my @rows = ();
    foreach my $item ( @$data ){
        my ( $row, $type ) = @$item;
        push( @rows, process_row( $select, $item ) );
    }
    push( @report, [ $select, \@rows ] );
}

########################################################################

=head2 C<process_row>

  process_row( $select, $item );

Creates entry for one row

=cut

########################################################################
sub process_row
{
    my $select = shift;
    my $item   = shift;

    my ( $row, $type ) = @$item;

    my @cols = @{ $metadata{ $select } }; 
    my @rpt_row = ();
    if( $type eq "equal" ){
        foreach my $col ( @cols ){
            push( @rpt_row, [ $col, $row->{ $col } ] );
        }
        return [ 'equal', \@rpt_row ];
    } 
    if( $type eq "different" ){
        foreach my $col ( @cols ){
            if( ref( $row->{ $col } ) eq "HASH" ){
                push( @rpt_row, [ $col,  {  
                                    'old' => $row->{ $col }->{ 'old_value' },
                                    'new' => $row->{ $col }->{ 'new_value' }
                                         }
                                ] );
            }else{
                push( @rpt_row, [ $col, $row->{ $col } ] );
            }
        }
        return [ 'different', \@rpt_row ]; 
    } 
    if( $type eq "additional" ){
        foreach my $col ( @cols ){
            push( @rpt_row, [ $col, $row->{ $col } ] );
        }
        return [ 'additional', \@rpt_row ];
    } 
    if( $type eq "missing" ){
        foreach my $col ( @cols ){
            push( @rpt_row, [ $col, $row->{ $col } ] );
        }
        return [ 'missing', \@rpt_row ];
    } 
}

########################################################################

=head2 C<print_report>

  print_report();

Prints report to the standard output.

=cut

########################################################################
sub print_report
{
    print "<report>\n";
    foreach my $select ( @report ){
        print_select( $select );
    }
    print "</report>\n";
}

########################################################################

=head2 C<print_select>

  print_select();

Formats one select for XML

=cut

########################################################################
sub print_select
{
    my $select = shift;
    print "    <" . $select->[ 0 ] . ">\n";
    print_statement( $select->[ 0 ] );
    print_header( $select->[ 0 ] );
    print_data( $select );
    print "    </" . $select->[ 0 ] . ">\n";
}


########################################################################

=head2 C<print_header>

  print_header();

Formats one header for XML

=cut

########################################################################
sub print_header
{
    my $select = shift;
    print "        <header>\n";
    foreach my $col ( @{ $metadata{ $select } } ){
        print " " x 12 . "<$col/>\n";
    }
    print "        </header>\n";
}

########################################################################

=head2 C<print_data>

  print_data();

Formats one data section for XML

=cut

########################################################################
sub print_data
{
    my $select = shift;
    print "        <data>\n";
    foreach my $row ( @{ $select->[ 1 ] } ){
        if( $row->[ 0 ] eq 'equal' ){
            print_row( $row, 'equal' );
        }
        if( $row->[ 0 ] eq 'additional' ){
            print_row( $row, 'additional' );
        }
        if( $row->[ 0 ] eq 'missing' ){
            print_row( $row, 'missing' );
        }
        if( $row->[ 0 ] eq 'different' ){
            print_different_row( $row );
        }
    }
    print "        </data>\n";
}

########################################################################

=head2 C<print_row>

  print_row();

Formats one row for XML

=cut

########################################################################
sub print_row
{
    my $row = shift;
    my $type = shift;
    print  " " x 12 . "<$type>\n";
    foreach my $col ( @{ $row->[ 1 ] } ){
        print " " x 16 . "<" . $col->[ 0 ] . ">";
        print_value( $col->[ 1 ] );
        print  "</" . $col->[ 0 ] . ">\n";
    }
    print  " " x 12 . "</$type>\n";
}

########################################################################

=head2 C<print_value>

  print_value();

Formats one value for XML

=cut

########################################################################
sub print_value
{
    my $value = shift;
    if( $value =~ /\&|</ ){
        print "<![CDATA[" . $value . "]]>";
    }else{
        print $value;
    }
}    
    

########################################################################

=head2 C<print_different_row>

  print_different_row();

Formats one different row for XML

=cut

########################################################################
sub print_different_row
{
    my $row = shift;
    print  " " x 12 . "<different>\n";
    foreach my $col ( @{ $row->[ 1 ] } ){
        print " " x 16 . "<" . $col->[ 0 ] . ">";
        if( ref( $col->[ 1 ] ) ne "HASH" ){
            print_value( $col->[ 1 ] );
            print "</" . $col->[ 0 ] . ">\n";
        }else{
            print "\n" . " " x 20 . "<old_value>";
            print_value( $col->[ 1 ]->{ 'old' } );
            print "</old_value>\n";
            print " " x 20 . "<new_value>"; 
            print_value( $col->[ 1 ]->{ 'new' } );
            print "</new_value>\n";
            print " " x 16 . "</" . $col->[ 0 ] . ">\n";
        }
    }
    print  " " x 12 . "</different>\n";
}


########################################################################

=head2 C<print_statement>

  print_statement();

Adds details of select statement to the report

=cut

########################################################################
sub print_statement
{
    my $select = shift;
    my @stmt_rules = ( { 'name' => 'name' },
                       { 'name' => 'select' } );

    if( !$stmt_fname ){
        return;
    }else{
        my $jr = LRpt::JarReader->new( 'rules'    => \@stmt_rules, 
                                       'filename' => $stmt_fname ); 
        $jr->read_all();
        my $sect = $jr->get_section_with( 'name' => $select );
        print " " x 8 . "<statement><![CDATA[" . 
              $sect->{ 'select' } . "]]></statement>\n";
    }
}

######################################################################

=head2 C<print_usage>

  print_usage();

Prints usage text.

=cut

######################################################################
sub print_usage
{
    print "Usage:  $0 --help \n";
    print "             --selects=<name> --rkeys_file=<name>" .
          " --diffs file_name\n";
    print "\n";
    print "  --diffs         - Sets input file mode to diffs\n";
    print "  --help          - prints this help screen\n";
    print "  --rkeys_file    - Name of the file containing row keys " . 
                               "definitions\n";
    print "  --selects       - Name of a file containing selects to be\n" .
          "                    executed.\n";
    print "  file_name       - Name of a file or a folder containing data\n" .
          "                    to be reported.\n";
    exit( 0 );
}

=head1 REPORTING AND FORMATING IN LREPORT

This chapter is supposed to help you to understand how B<LReport> helps
you in generating nicely formated reports of database contents and 
differences.

We will start by an example

=head2 EXAMPLE 1

We will continue using an example from manual page for L<C<LRpt>|LRpt>. 
Our goal would be to generate
an RTF document, which looks like the one given at http:/xxxx.

As you remember, we finished with the output of B<lcsvdiff.pm>

We will create an XML document and then we will process it with
XSLT processor to get a final RTF document. 

One of possible inputs for B<lrptxml.pm> is output generated by 
B<lcsvdiff.pm>. However, this is not exactly the format generated in the 
example shown in L<C<LRpt>|LRpt> manual page. The format used there, contained
only information about changes. In order to generate a full report we will
have to use I<--all> switch when calling B<lcsvdiff.pm>. This switch
will generate the following output:

  lcsvdiff.pl before/customer.txt after/customer.txt 
  SCHEMA: customer_id name  last_name  address
  ROW( 1234 ): 1234  Jan Nowak   Warszawa
  lcsvdiff.pl before/service.txt after/service.txt 
  SCHEMA: customer_id service_type price status
  INS( 1234#GPRS ): 1234  GPRS 2.05  ACTIVE
  DEL( 1234#MAIL ): 1234  MAIL 1.30  ACTIVE
  ROW( 1234#VOICE ): 1234 VOICE   0.34  DEACTIVATED
  UPD( 1234#VOICE ): status: ACTIVE =#=> DEACTIVATED

As you can see, differences are reported together with information about
all rows returned by selects.

You run B<lrptxml.pl> in the following way

  lrptxml.pl --diffs --selects=selects.txt --rkeys_file=rkeys.txt diffs.txt

Meaning of parameters:

=over 4

=item --diffs

A switch indicating that a XML report should be generated from output of
B<lcsvdiff.pl> tool.

=item --selects

A file containing selects. This is the same file as used by B<lcsvdmp.pl>. 
It is used here to include in XML report the full text of select, which 
returned presented rows

=item --rkeys_file

File with a definiton of row keys. This is exactly the same type of file
as used by B<lcsvdiff.pl> tool. It is used here to order rows in the report
properly.

=item diffs.txt

Output generated by B<lcsvdiff.pl> ran with I<--all> switch.

=back

The output of B<lrptxml.pl> should look like this:

  <report>
      <customer>
          <statement><![CDATA[select * from customer where customer_id = 1234]]>
          </statement>
          <header>
              <customer_id/>
              <name/>
              <last_name/>
              <address/>
          </header>
          <data>
              <equal>
                  <customer_id>1234</customer_id>
                  <name>Jan</name>
                  <last_name>Nowak</last_name>
                  <address>Warszawa</address>
              </equal>
          </data>
      </customer>
      <service>
          <statement><![CDATA[select * from service where customer_id = 1234]]>
          </statement>
          <header>
              <customer_id/>
              <service_type/>
              <price/>
              <status/>
          </header>
          <data>
              <additional>
                  <customer_id>1234</customer_id>
                  <service_type>GPRS</service_type>
                  <price>2.05</price>
                  <status>ACTIVE</status>
              </additional>
              <missing>
                  <customer_id>1234</customer_id>
                  <service_type>MAIL</service_type>
                  <price>1.30</price>
                  <status>ACTIVE</status>
              </missing>
              <different>
                  <customer_id>1234</customer_id>
                  <service_type>VOICE</service_type>
                  <price>0.34</price>
                  <status>
                      <old_value>ACTIVE</old_value>
                      <new_value>DEACTIVATED</new_value>
                  </status>
              <different>
          </data>
      </service>
  </report>

Let's have a closer look at it.

The C<report> tag is a root element. Each of its children is a select. 
The example above has 2 selects: I<customer> and I<service>.
Each select element has 3 children:

=over 4

=item statement

Contains a full text of a select, which returned given rows.

=item header

Contains all columns' names returned by the select

=item data

Contains all rows (with all columns) returned by the select. 

=back

Direct children on I<data> element are:

=over 4

=item equal

No differences found. 

=item additional

The row was not present in the I<before> collection but is present in an
I<after> collection.

=item missing

The row was not present in the I<after> collection but is present in a
I<before> collection.

=item different

The row is present in the I<after> and I<before> collection. There are some
differences found in columns' values.

=back

You can read about I<XML> report format in more details later in xxx.

Now you can get XSLT stylesheets for RTF from source forge and process
the generated report. There is some fuzziness around XSLT specification so
I can't guarantee that it will work for any XSLT processor. But it worked
for me with Saxon.

After getting RTF stylesheet and installing Saxon, you can run the
conversion. On my Windows machine it look like this:

  java net.sf.saxon.Transform report.xml rld_rtf.xsl > report.rtf


=head1 WORKING MODES

B<lrptxml.pl> can work is 2 modes - csv and diffs. 

In csv mode (default one)
B<lrptxmp.pl> will pick all
files with expected extension found is this directory. Since there would
be no comparison information, all rows will be reported as equal.

Second mode is I<diffs> mode, when B<lrptxml.pl> accepts on input a file
produced by B<lcsvdiff.pl> tool ran with B<--all> switch. 
In this mode differences are reflected in the output report. 
In order to switch to diffs mode, B<--diffs> switch has to be used.

The default mode is reading I<csv> files. 

=head1 XML REPORT FORMAT

XML report has to be well formed. Its root element is I<report>. 
Its direct children are elements dedicated to results of single selects.
Name of each of this elements is a name of a select.

The general structure of an XML report is given below:

  <report>
      <select_name1>
          <statement>select * from ..... </statement>
          <header>
              <col1/>
              <col2/>
              <col3/>
              <colN/>
          </header>
          <data>
              <equal>
                 <col1>value11</col1>
                 <col2>value12</col2>
                 <col3>value13</col3>
                 <colN>value14</colN>
              </equal>
              <different>
                 <col1>value21</col1>
                 <col2>value22</col2>
                 <col3>
                     <old_value>old_value23</old_value>
                     <new_value>new_value23</new_value>
                 </col3>
                 <colN>value24</colN>
              </different>
              <missing>
                 <col1>value31</col1>
                 <col2>value32</col2>
                 <col3>value33</col3>
                 <colN>value34</colN>
              </missing>
              <additional>
                 <col1>value41</col1>
                 <col2>value42</col2>
                 <col3>value43</col3>
                 <colN>value44</colN>
              </additional>
          </data>
      </select_name1>
      <select_name2>
      ........
      </select_nameN>
  </report>

Each select is reported in an element, which name is equal to select name.
It contains the following children: statement, header and data.

=head2 statement

Contains a full text of select which produced reported rows.

=head2 header

Contains elements for all columns in a row. Order of elements is
similar to order of columns in result returned by a select.

=head2 data

Contains actual rows. It may have 4 types of child elements:

=over 4

=item equal

No differences reported for a given row. Child elements are columns, their
value are values from columns.

=item different

Row existed in I<after> and I<before> collections. Values in some
columns were different. Child elements are columns, their values are values
from columns for all columns except columns where differences were found.
For those columns, column element contains to child elements: I<old_value> and
I<new_value> which contains values from I<before> and I<after> rows.

=item additional

I<before> row does not exist, I<after> row does. 
Child elements are columns, their
value are values from columns.

=item missing

I<before> row exists, I<after> row doesn't. 
Child elements are columns, their
values are values from columns.

=back

=head1 XSLT STYLESHEET

Currently there is only one stylesheet available, for conversion to RTF
documents.

=head2 RTF converter.

RTF converter create an RTF document. An example of such a document
can be found at xxx.
Section containing results of one
select consists of select statement text and then rows. Rows are not 
presented in the tables. Instead each row is one paragraph, with fields
separated by tabs. This is good for presenting multi row selects. 
Using tables would make columns very narrow and hardly readable.
Header, containg all columns to be reported is 
printed in bold on a gray background. Every second row is shaded, so it's
easier to see which values belongs to which row.

Additional rows (see before-after comparison) are printed using italics, bold
and double underline.

Missing rows are (see before-after comparison) are printed using stroke through

Different rows (see before-after comparison) are presented in standard font.
Only columns that differ are printed in bold, italics and underline.

We can choose to highlight some columns with some colors. It is helpful when
viewing reports with many columns. Highlighting can be used to point readers
attention to most important data.

=head3 FILES

The stylesheet is composed of 3 files:

=over 4

=item rtf.xsl

Contains the main processing logic. Not to be modified (unless you know 
what you are doing)

=item rtf_header.xsl

Some headers and constants related to RTF format. Not to be modified.

=item rldf_rtf.xsl

To be edited by you. Contains details of formating in RTF documents. 
I<rldf> stands for I<Report Layout Definition File>.

=back

An example of I<rldf> is given below:

It contains some standard stuff related to XSLT. The only part, which should
be modified by you is contents of I<rld> variable element. You can read
details on I<rld> in next chapter.

=head3 RLD

I<rld> is an XML tree, used to specify details on how are select results
shown in RTF document. It allows to specify, which columns should be on 
the report and with what color it should be highlighted.

Its structure is as follows:

  <rld>
      <select_name1 all="1" >
          <col1 color="red" />
      </select_name1>
      <select_name2>
          <col1 color="blue" />
          <col2 color="yellow" />
          <col3 color="yellow" />
      </select_name2>
  </rld>

Element named as a select name defines details for a given select. This element
has one attribute: I<all>. If this attributes value is I<1> than all columns
from the select are shown in RTF document.

Each select element has children elements for columns. Column element may 
have one attribute, I<color>, which defines color of column's highlighting.
Possible values are xxxx.
If I<all> attribute is not set to I<1> then only those columns are present
in RTF document, for which column elements are present in I<rld>.
If I<all> attribute is set to I<1> then all columns are present
in RTF document. If column child elements are present, their I<color>
attribute are used for defining color of column highlighting.

=head1 TO DO

More stylesheets from other formats should be available. I am not planning
to do any in the nearest future.

The current RTF stylesheet is poorly documented, and possibly poorly
structured. I am not an XSLT champion.

=head1 SEE ALSO

The project is maintained on Source Forge L<http://lreport.sourceforge.net>. 
You can find there links to some helpful documentation like tutorial.

=head1 AUTHORS

Piotr Kaluski E<lt>pkaluski@piotrkaluski.comE<gt>

=head1 COPYRIGHT

Copyright (c) 2004-2006 Piotr Kaluski. Poland. All rights reserved.

You may distribute under the terms of either the GNU General Public License 
or the Artistic License, as specified in the Perl README file. 

=cut

1;

