use strict;
use warnings;
use Test::More;

use lib "t/lib";

use ManocTest::Schema;

my $schema = ManocTest::Schema->connect();
ok( $schema, "Create schema" );

my $rs = $schema->resultset("HWAsset");
ok( $rs, 'Resultset' );

my %fields = (
    type   => App::Manoc::DB::Result::HWAsset->TYPE_DEVICE,
    vendor => 'Moon',
    model  => 'ShinyBlade',
);

foreach my $attr ( keys %fields ) {
    my %test_fields = %fields;
    delete $test_fields{$attr};
    eval { my $hw = $rs->create( \%test_fields ); };
    ok( $@, "$attr is required" );
}

my $hw = $rs->create( \%fields );
ok( $hw,                       "Create a new HWAsset" );
ok( defined( $hw->inventory ), "Autogenerated inventory" );
ok( $hw->delete,               "Delete" );

ok( $hw = $rs->new_result( {} ), "New result with empty args" );
foreach my $attr ( keys %fields ) {
    ok( $hw->$attr( $fields{$attr} ), "set column $attr" );
}
ok( $hw->insert(),             "Insert" );
ok( defined( $hw->inventory ), "Autogenerated inventory" );
$hw->delete;

ok( $hw = $rs->new_result( { %fields, inventory => 'Manoc0001' } ),
    "New result with inventory" );
ok( $hw->insert(), "Insert" );
is( $hw->inventory, 'Manoc0001', "Preserve povided inventory" );
is( $hw->label, 'Manoc0001 (Moon - ShinyBlade)', "label" );

$hw->discard_changes();

# tests on location
ok( $hw->is_in_warehouse,    "default location is warehouse" );
ok( !$hw->is_decommissioned, "default not decommissioned" );

my $building = $schema->resultset("Building")->create(
    {
        name        => 'B01',
        description => 'Test'
    }
);
my $rack = $schema->resultset("Rack")->create(
    {
        name     => 'R01',
        floor    => 1,
        room     => '00',
        building => $building,
    }
);
my $warehouse = $schema->resultset("Warehouse")->create(
    {
        name => 'W01',
    }
);
$hw->move_to_rack($rack);
ok( $hw->is_in_rack,         "Move to rack" );
ok( !$hw->is_decommissioned, "if in rack not decommissioned" );
ok( !$hw->is_in_warehouse,   "if in rack not in warehouse" );
is( $hw->display_location, "Rack R01 (B01)", "Display location rack" );

$hw->move_to_room( $building, "1", "R01" );
ok( !$hw->is_decommissioned, "if in room not decommissioned" );
ok( !$hw->is_in_warehouse,   "if in room not in warehouse" );
ok( !$hw->is_in_rack,        "if in room not in rack" );
is( $hw->display_location, "B01 (Test) - 1 - R01", "Display location room" );

$hw->move_to_warehouse();
ok( $hw->is_in_warehouse,    "move to warehouse" );
ok( !$hw->is_decommissioned, "if in w/h not decommissioned" );
ok( !$hw->is_in_rack,        "if in room not in rack" );
is( $hw->display_location, "Warehouse", "Display location Warehouse" );

$hw->move_to_warehouse($warehouse);
is( $hw->display_location, "Warehouse - W01", "Display location Warehouse" );
ok( $warehouse->hwassets, "Warehouse has associated assets" );

$hw->decommission();
ok( $hw->is_decommissioned, "decommissioned" );
ok( !$hw->is_in_warehouse,  "if decommissioned not in warehouse" );
ok( !$hw->is_in_rack,       "if decommissioned not in rack" );
is( $hw->display_location, "Decommissioned", "Display location decommissioned" );

$hw->delete;

# test unused devices
$rs->populate(
    [
        {
            type   => App::Manoc::DB::Result::HWAsset->TYPE_DEVICE,
            vendor => 'IQ',
            model  => 'MegaPort 48',
            serial => 'Test01',
            device => {
                name        => 'D0001',
                mng_address => '1.1.1.1',
            },
        },
        {
            type     => App::Manoc::DB::Result::HWAsset->TYPE_DEVICE,
            vendor   => 'IQ',
            model    => 'MegaPort 48',
            serial   => 'Test02',
            location => App::Manoc::DB::Result::HWAsset->LOCATION_DECOMMISSIONED,
        },
        {
            type   => App::Manoc::DB::Result::HWAsset->TYPE_SERVER,
            vendor => 'HAL',
            model  => 'MyServer',
            serial => 'Test03',
        },
        {
            type   => App::Manoc::DB::Result::HWAsset->TYPE_DEVICE,
            vendor => 'IQ',
            model  => 'MegaPort 24',
            serial => 'Test04',
        },
    ]
);

my $unused_devices = $rs->unused_devices;
is( $unused_devices->count,        1,        "No extra result in unused device" );
is( $unused_devices->next->serial, 'Test04', "Unused device is the right one" );

done_testing;
