;;;; 	Copyright (C) 1998 Lauri Alanko <la@iki.fi>
;;;; 
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2, or (at your option)
;;;; any later version.
;;;; 
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License
;;;; along with this software; see the file COPYING.  If not, write to
;;;; the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
;;;; Boston, MA 02111-1307 USA
;;;; 
;;;; The GPL is also available at http://www.gnu.org/copyleft/gpl.html


; script-fu-compat.scm
(use-modules (gtk gtk))
(use-modules (gtk gdk))


(define (dbg . args)
  (for-each (lambda (x) (display x (current-error-port)))
            `("Debug: " ,@args #\nl)))

(enum
 SF-IMAGE
 SF-DRAWABLE
 SF-LAYER
 SF-CHANNEL
 SF-COLOR
 SF-TOGGLE
 SF-VALUE)

(enum
 RGB_IMAGE
 RGBA_IMAGE
 GRAY_IMAGE
 GRAYA_IMAGE
 INDEXEDA_IMAGE)

(define (color-widget default)
  (define adjs (map (lambda (val) (gtk-adjustment-new val 0 255 1 1 0))
		    default))
  (define scales (map gtk-hscale-new adjs))
  (define labels (map gtk-label-new '("R" "G" "B")))
  (define table (gtk-table-new 3 2 #f))
  (define i 0)
  (for-each (lambda (label scale)
	      (gtk-table-attach table label 0 1 i (1+ i) '() '())
	      (gtk-widget-show label)
	      (gtk-table-attach table scale 1 2 i (1+ i))
	      (gtk-scale-set-digits scale 0)
	      (gtk-widget-show scale)
	      (set! i (1+ i)))
	    labels
	    scales)
  (define (getter)
    (map (lambda (adj)
	   (inexact->exact (gtk-adjustment-value adj)))
	 adjs))
  (cons table getter))


(define (value-widget default)
  (define entry (gtk-entry-new))
  (gtk-entry-set-text entry default)
  (define (getter)
    (eval-string (gtk-entry-get-text entry)))
  (cons entry getter))


(define (toggle-widget default)
  (define toggle (gtk-check-button-new))
  (gtk-toggle-button-set-state toggle (not (zero? default)))
  (define (getter)
    (if (gtk-toggle-button-active toggle)
	TRUE
	FALSE))
  (cons toggle getter))

(define (error-widget default)
  (error "Sorry, drawable menus are not yet supported."))

(define argt
  `((,SF-COLOR ,color-widget)
    (,SF-VALUE ,value-widget)
    (,SF-TOGGLE ,toggle-widget)
    (,SF-IMAGE ,error-widget)
    (,SF-DRAWABLE ,error-widget)
    (,SF-CHANNEL ,error-widget)
    (,SF-LAYER ,error-widget)))


(define (run-script-interactive procname args)
  (define procinfo (cddr (assoc procname sf-scriptdb)))
  (define image-based (car procinfo))
  (define arginfo (if image-based
		      (cdddr procinfo)
		      (cdr procinfo)))
  (define imgargs
    (if image-based
	(params-to-sfargs (list-head args 2) (list-head (cdr procinfo) 2))
	'()))
  (define title (string-append "GimpleSF: " procname))
  (define dlg (gtk-dialog-new))
  (define (return) (gtk-main-quit)) 
  (gtk-window-set-title dlg title)
  (gtk-signal-connect dlg "destroy" return)

  (define button (gtk-button-new-with-label "OK"))
  (gtk-signal-connect button "clicked" 
		      (lambda ()
			(call-script procname
				     (append imgargs
					     (map (lambda (f) (f)) getters)))
			(return)))
  (gtk-box-pack-start (gtk-dialog-action-area dlg) button)
  (gtk-widget-show button)

  (define button (gtk-button-new-with-label "Cancel"))
  (gtk-signal-connect button "clicked" return)
  (gtk-box-pack-start (gtk-dialog-action-area dlg)  button)
  (gtk-widget-show button)
  
  (define frame (gtk-frame-new "Script Arguments"))
  (gtk-box-pack-start (gtk-dialog-vbox dlg) frame)
  (gtk-widget-show frame)

  (define table (gtk-table-new (length arginfo) 2 #f))
  (gtk-container-add frame table)
  (gtk-widget-show table)

  (define inputters
    (map (lambda (arg)
	    ((cadr (assq (car arg) argt)) (caddr arg)))
	 arginfo))
  (define descs (map cadr arginfo))
  (define getters (map cdr inputters))
  (define widgets (map car inputters))
 
  (let ((row 0))
    (for-each (lambda (desc widget)
		(define label (gtk-label-new desc))
		(gtk-table-attach table label 0 1 row (+ row 1) '() '())
		(gtk-widget-show label)
		(gtk-table-attach table widget 1 2 row (+ row 1))
		(gtk-widget-show widget)
		(set! row (+ row 1)))
	      descs
	      widgets))


  (define frame (gtk-frame-new "Current command:"))
  (gtk-box-pack-start (gtk-dialog-vbox dlg) frame #f #f 0)
  (gtk-widget-show frame)

  (define current-command (gtk-entry-new))
  (gtk-entry-set-editable current-command #f)
  (gtk-container-add frame current-command)
  (set! pdb-cb (lambda (procname args)
		 (gtk-entry-set-text current-command procname)
		 (gdk-flush)))

  (gtk-widget-show current-command)

  (gtk-widget-show dlg)
  (gtk-main)
  (gtk-widget-destroy dlg)
  (gdk-flush)
  (list (cons PARAM-STATUS STATUS-SUCCESS )))




(define (assert expr)
  (if (not expr)
      (error "Assert error: "))) 



; This is an alist whose elements are of the form:
; (scriptname proc img-based (type desc default) (type desc default) ... ) 

(define sf-scriptdb '())


(define pdb-cb list)

(define (sf-parse-param type desc default)
  (let ((ti (assq type sfargtype-alist)))
    (if (not (string? desc))
	(error "Invalid description"))
    (if (not ti)
	(error "Invalid type"))
    (if (not ((cadddr ti) default))
	(error "Invalid default"))
    (list (cadr ti) (caddr ti) desc)))
	

(define (sf-parse-params-iter params parsed)
  (if (null? params)
       (reverse parsed)    ;paramdef
      (sf-parse-params-iter
       (cdr params)
       (cons (apply sf-parse-param (car params)) parsed))))


(define (group-params-iter params accum)
  (if (null? params)
      (reverse accum)
      (group-params-iter (cdddr params) (cons (list (car params)
						    (cadr params)
						    (caddr params))
					      accum))))
(define (script-fu-register name
			    desc
			    help
			    author
			    copyright
			    date
			    img-types
			    . params)
  (define paramgrp (group-params-iter params '()))
  (define parsed (sf-parse-params-iter paramgrp '()))
  (define paramdefs (cons (list PARAM-INT32
				"run_mode"
				"Interactive, non-interactive")
			  parsed))
  (define dbname (string-append gimplesf-prefix name))
  (set! sf-scriptdb (cons (append (list dbname
					(symbol-binding #f (string->symbol name))
					(not (string-match "^<Toolbox>" desc)))
				  paramgrp)
			  sf-scriptdb))

  (define menu-path
    (regexp-substitute #f (string-match "Script-Fu" desc) 'pre "GimpleSF" 'post))
  (gimple-install-proc PROC-TEMPORARY
		       paramdefs
		       '()
		       dbname
		       "A GimpleSF script"
		       help
		       author
		       copyright
		       date
		       menu-path
		       img-types))


(define (conv-default arg) arg)

(define (conv-string arg)
  (if (string? arg)
      arg
      (error "argument must be a string" arg)))

(define (conv-int arg)
  (if (number? arg)
      (if (integer? arg)
	  arg
	  (inexact->exact (floor arg)))
      (error "argument must be a number" arg)))

(define (conv-drawable arg)
  (if (number? arg)
      arg
      (error "argument must be an integer identifier")))

(define (conv-float arg)
  (if (number? arg)
      arg
      (error "argument must be a number")))

(define (conv-color arg)
  (if (equal? (map number? arg) '(#t #t #t))
      (list->vector (map conv-int arg))
      (error "argument must be a color (a list of three integers)")))

(define (conv-stringarray arg)
  (if (list? arg)
      arg
      (error "argument must be a stringarray (a list of strings)")))




(define sfargtype-alist
  (list
   (list SF-IMAGE PARAM-IMAGE "Image" integer? id)  
   (list SF-DRAWABLE PARAM-DRAWABLE "Drawable" integer? id)  
   (list SF-LAYER PARAM-LAYER "Layer" integer? id)
   (list SF-CHANNEL PARAM-CHANNEL "Channel" integer? id)
   (list SF-COLOR PARAM-COLOR "Color"
	 (lambda (def)
	   (equal? (map integer? def) '(#t #t #t)))
	 vector->list)
   (list SF-VALUE PARAM-STRING "Value" string?
	 (lambda (arg) (call-with-input-string arg read)))
   (list SF-TOGGLE PARAM-INT32 "Toggle"
	 (lambda (def)
	   (or (zero? def) (= def 1))) id)))

(define gparamtype-alist
  (list
   (cons PARAM-INT32 conv-int)
   (cons PARAM-INT16 conv-int)
   (cons PARAM-INT8 conv-int)
   (cons PARAM-FLOAT conv-float)
   (cons PARAM-STRING conv-string)
   (cons PARAM-STRINGARRAY conv-stringarray)
   (cons PARAM-COLOR conv-color)
   (cons PARAM-DRAWABLE conv-drawable)
   (cons PARAM-IMAGE conv-drawable)
   (cons PARAM-LAYER conv-drawable)
   (cons PARAM-CHANNEL conv-drawable)
   (cons PARAM-DISPLAY conv-drawable)
   ))

(define (convert_ str)
  (let ((i (string-index str #\_)))
    (if i (begin
	    (string-set! str i #\-)
	    (convert_ str))
	str)))


(define (add-procedure procname)
  (define sfname (symbol (convert_ (string-copy procname))))
  (intern-symbol #f sfname)
  (symbol-set! #f sfname
	       (lambda args
		 (apply proc-run-check procname args))))



(define (listify-path-iter path idx accum)
  (define i (string-index path #\: idx))
  (if i
      (listify-path-iter path
			 (+ i 1)
			 (cons (substring path idx i) accum))
      (reverse (cons (substring path idx (string-length path)) accum))))

      
(define (read-scripts dirname)
  (if (access? dirname (logior R_OK X_OK))
  (let ((dir (opendir dirname)))
    (do ((file (readdir dir) (readdir dir)))
	((eof-object? file) #t)
      (if (string-match "\.scm$" file)
	  (catch #t
		 (lambda ()
                   (dbg file)
		   (load (string-append dirname "/" file)))
		 (lambda (key . args)
		   (warn "Error" key args "processing file" file))))))
  #f))
  

(define (find-scripts)
  (define path (listify-path-iter
		(car (gimp-gimprc-query "gimplesf-path")) 0 '()))
  (dbg path)
  ;Just a temp thingy
  (define omap map)
  (define ofor-each for-each)
;  (define oset! set!)
;  (oset! set! define)
  (for-each read-scripts path)
;  (oset! set! oset!)
  (set! for-each ofor-each)
  (set! map omap))
  

  
(define (proc-param-type proc npar)
  (cdr (cadr (gimple-run-pdb
		      "gimp_procedural_db_proc_arg"
		      proc
		      (cons PARAM-INT32 npar)))))

(define (proc-param-types-iter proc idx accum)
  (if (= idx -1)
      accum
      (proc-param-types-iter proc (- idx 1)
			     (cons
			      (proc-param-type proc idx)
			      accum))))
       
(define (proc-param-types proc)
  (proc-param-types-iter 
   (cons PARAM-STRING proc)
   (- (cdr (list-ref (gimple-run-pdb "gimp_procedural_db_proc_info"
				     (cons PARAM-STRING proc))
		     7))
      1)
   '()))



(define (params-to-sfargs params argtypes)
  (map (lambda (param argtype)
	 ((list-ref (assq (car argtype) sfargtype-alist) 4) (cdr param))); Would be good to check the param type
       params
       argtypes))

(define (proc-run procname . args)
  (define argtypes (proc-param-types procname))
  (map cdr (apply gimple-run-pdb procname (map cons argtypes args))))

(define (revconv gparam)
  (if (eq? (car gparam) PARAM-COLOR)
      (vector->list (cdr gparam))
      (cdr gparam)))

(define (proc-run-check procname . args)
  (if (not (string? procname))
      (error "procname must be a string"))
  (define argtypes (proc-param-types procname))
  (if (not (= (length args) (length argtypes)))
      (error "wrong number of args"))
  (define pdbargs
    (map (lambda (argtype arg)
	   (cons argtype ((cdr (assq argtype gparamtype-alist)) arg)))
	 argtypes
	 args))
  (pdb-cb procname args)
  (define retvals (apply gimple-run-pdb procname pdbargs))
  (if (equal? (car retvals) (cons PARAM-STATUS STATUS-SUCCESS ))
      (map revconv (cdr retvals))
      (error "Error at pdb execution")))
      
(define gimp-proc-db-call proc-run-check)

(define (call-script procname sfargs)
  (define oset! set!)
  (set! set! define)
  (apply
   (cadr (assoc procname sf-scriptdb))
   sfargs)
  (oset! set! oset!))


(define (sf-dispatcher procname . args)
  (catch #t
	 (lambda ()
	   (define run-type (car args))
	   (if (eq? (car run-type) PARAM-INT32)
	       (case (cdr run-type)
		 ;RUN-INTERACTIVE
		 ((0) (run-script-interactive procname (cdr args)))
		 ;RUN-NONINTERACTIVE
		 ((1)
		  (set! pdb-cb list)
		  (let* ((argtypes (cdddr (assoc procname sf-scriptdb)))
			     
			 ; Convert the GParams to SFArgs
			 (sfargs (params-to-sfargs (cdr args);discard run_type
						       argtypes)))
		    ; Call the script
		    (call-script procname sfargs)
		    (list (cons PARAM-STATUS STATUS-SUCCESS)))))
	       (list (cons PARAM-STATUS STATUS-CALLING-ERROR))))
	 (lambda err
	   (warn err)
	   (list (cons PARAM-STATUS STATUS-EXECUTION-ERROR)))))



;SIOD compatibility stuff

(define (fmod nom den) (- nom (* (truncate (/ nom den)) den)))
(define (nth i l) (list-ref l i))
(define *pi* (* 4 (atan 1)))
(define cons-array make-vector)
(define aset vector-set!)
(define aref vector-ref)
(define pow expt)
(defmacro prog1 (first . rest)
  (define tmp (gensym))
  `(begin (define ,tmp ,first)
	  ,@rest
	  ,tmp))



(debug-set! depth 50)

(define (sf-init)
  (define x (cons PARAM-STRING ".*"))
  (define allprocs (gimple-run-pdb "gimp_procedural_db_query" x x x x x x x))
  (if (not (equal? (car allprocs) (cons PARAM-STATUS STATUS-SUCCESS)))
      (error "initial pdb query failed!"))
  (define proclist (vector->list (cdaddr allprocs)))
  (for-each add-procedure proclist)
  )

(dbg 1)
(sf-init)
(define gimp-data-dir (car (gimp-gimprc-query "gimp_data_dir")))
(define gimp-plugin-dir (car (gimp-gimprc-query "gimp_plugin_dir")))
(define gimplesf-prefix (car (gimp-gimprc-query "gimplesf-prefix")))

(dbg 2)
(find-scripts)
;maybe this should be done earlier.. gimp blocks until this..
(dbg 3)
(gimple-extension-ack)
(dbg 4)

;(top-repl)

(do ()
    (#f)
  (gimple-extension-process sf-dispatcher 0))

