/*************************************************************************
* TOOL:          emfplus.cpp
*
* VERSION:       1.00
*
* AUTHOR:        tidine
*
* SUPPORT:       tidine
*
* CREATION DATE: 10Feb2012
*
* SYNTAX:        emfplus [-format <value>][-dpi <value>][-debug] <infile> <outfile>
*                -format  : list of images to convert to. Defaults RAW.
*                -dpi  : image resolution
*                -debug   : adds printout information.
* 
* DESCRIPTION:   Delegate used to convert EMF+ images to raw image data.
*                This tool can be used as a stand-alone tool to convert
*                EMF+ images to other windows images formats, such as
*                bmp, gif, jpg, png or tif.
*
* NOTE:
*
* REVISION HISTORY:
*                Revision 1.1  2012/02/10 11:29:44  tidine
*                Initial revision
*
*************************************************************************/
#include <windows.h>
#include <gdiplus.h>
#include <stdio.h>
#include <string>
using namespace Gdiplus;

// Output type to encode
enum OutputType {
  OT_RAW = 0,
  OT_BMP,
  OT_GIF,
  OT_JPG,
  OT_PNG,
  OT_TIF
};

// Encode values
#define OT_RAW_ENCODE  L""
#define OT_BMP_ENCODE  L"image/bmp"
#define OT_GIF_ENCODE  L"image/gif"
#define OT_JPG_ENCODE  L"image/jpg"
#define OT_PNG_ENCODE  L"image/png"
#define OT_TIF_ENCODE  L"image/tif"

#define SYNTAX "\nSYNTAX: emfplus [-format <value>][-debug][-dpi <value>] <infile> <outfile>.\n"

GdiplusStartupInput gdiplusStartupInput;
ULONG_PTR           gdiplusToken;

FILE                *oFile = NULL; 
Bitmap              *image = NULL;
WCHAR               *wcsInFile = NULL;
WCHAR               *wcsOutFile = NULL;
WCHAR               *EncoderType = OT_RAW_ENCODE;
char                *strInFile = NULL;
char                *strOutFile = NULL;
errno_t	            eNo = 0;
BOOLEAN             isDebug = FALSE;
enum OutputType     outputType;

// Standard exit routine
void std_exit(
   int    rc )
{
   if ( oFile ) fclose( oFile );
   if ( wcsInFile ) delete wcsInFile;
   if ( wcsOutFile ) delete wcsOutFile;
   if ( strInFile ) delete strInFile;
   if ( strOutFile ) delete strOutFile;
   if ( image ) delete image;

   oFile = NULL;

   GdiplusShutdown( gdiplusToken );

   exit( rc );
}


// Print Out debug info
void print_debug(
   char    *msg )
{
   if ( isDebug && msg )
   {
      printf( msg );
   }
}


// Get class id for the encoder.
int GetEncoderClsid(
   const WCHAR *format,
   CLSID       *pClsid )
{
   UINT  num = 0;          // number of image encoders
   UINT  size = 0;         // size of the image encoder array in bytes

   ImageCodecInfo* pImageCodecInfo = NULL;

   GetImageEncodersSize( &num, &size );
   if ( size == 0 ) return -1;  // Failure

   pImageCodecInfo = new ImageCodecInfo[size];
   if( pImageCodecInfo == NULL ) return -1;  // Failure

   GetImageEncoders( num, size, pImageCodecInfo );

   for( UINT j = 0; j < num; ++j )
   {
      if( wcscmp( pImageCodecInfo[j].MimeType, format ) == 0 )
      {
         *pClsid = pImageCodecInfo[j].Clsid;
         delete pImageCodecInfo;
         return j;  // Success
      }
   }

   delete pImageCodecInfo;
   return -1;  // Failure
}


// Convert files to a wchar_t*
size_t toWchar(
   char    *chrStr,
   WCHAR   **wcsStr )
{
   size_t cChars = 0;
   size_t origSize = strlen( chrStr ) + 1;
   WCHAR  *tmpWCS = NULL;

   if ( wcsStr )
   {
      *wcsStr = NULL;
      if ( chrStr )
      {
         tmpWCS = new WCHAR[origSize];
         if ( tmpWCS )
         {
            mbstowcs_s( &cChars, tmpWCS, origSize, chrStr, _TRUNCATE);
         }
         if ( cChars > 0 )
         {
            *wcsStr = tmpWCS;
         }
      }
   }
   return cChars;
}

void loadImage(
   REAL dpiX,
   REAL dpiY )
{
   Image    *source = NULL;
   Graphics *graphics = NULL;
   INT      width, height;

   if (dpiX < 1 || dpiY < 1)
   {
      image = new Bitmap( wcsInFile );
      return;
   }

   source = Image::FromFile( wcsInFile );
   if ( !source )
      return;

   width = (int)floor( (REAL)source->GetWidth() / source->GetHorizontalResolution() * dpiX + 0.5 );
   height = (int)floor( (REAL)source->GetHeight() / source->GetVerticalResolution() * dpiY + 0.5 );
   image = new Bitmap( width, height );

   graphics = Graphics::FromImage(image);
   graphics->DrawImage( source, 0, 0, width, height );

   delete graphics;
   delete source;

   image->SetResolution( dpiX, dpiY );
}

// main
int main(int argc, char **argv)
{
   char         msg[256];
   std::string  dpi;
   int          arg = 0, pos = 0;
   REAL         dpiX = 0, dpiY = 0;

   outputType = OT_RAW;

   GdiplusStartup( &gdiplusToken, &gdiplusStartupInput, NULL );

   // Test arguments.
   if ( argc < 3 )
   {
      printf ( SYNTAX );
      std_exit( 2 );
   }

   // Parse the arguments.
   while (++arg < argc)
   {
      if      ( !_stricmp( argv[arg], "-debug" ) ) isDebug = TRUE;
      else if ( !_stricmp( argv[arg], "-format" ) )
      {
         if ( arg + 1 > argc )
         {
            fprintf( stderr, "ERROR: A parameter is required for -format.  Exiting.\n" );
            printf ( SYNTAX );
            std_exit( 2 );
         }
         ++arg;

         if      ( !_stricmp( argv[arg], "bmp" ) )
         {
            outputType = OT_BMP;
            EncoderType = OT_BMP_ENCODE;
         }
         else if ( !_stricmp( argv[arg], "gif" ) )
         {
            outputType = OT_GIF;
            EncoderType = OT_GIF_ENCODE;
         }
         else if ( !_stricmp( argv[arg], "jpg" ) )
         {
            outputType = OT_JPG;
            EncoderType = OT_JPG_ENCODE;
         }
         else if ( !_stricmp( argv[arg], "png" ) )
         {
            outputType = OT_PNG;
            EncoderType = OT_PNG_ENCODE;
         }
         else if ( !_stricmp( argv[arg], "tif" ) )
         {
            outputType = OT_TIF;
            EncoderType = OT_TIF_ENCODE;
         }
         else
         {
            fprintf( stderr, "WARNING: Requested format value (" );
            fprintf( stderr, argv[arg] );
            fprintf( stderr, ") is unknown.  Ignoring.\n" );
         }
      }
      else if ( !_stricmp( argv[arg], "-dpi" ) )
      {
         if ( arg + 1 > argc )
         {
            fprintf( stderr, "ERROR: A parameter is required for -dpi.  Exiting.\n" );
            printf ( SYNTAX );
            std_exit( 2 );
         }
         ++arg;

        dpi = std::string( argv[arg] );
        pos = dpi.find( "x" );
        if ( pos == -1 )
        {
          dpiX = (REAL)std::stoi( dpi );
          dpiY = dpiX;
        }
        else if ( pos != 0 && pos != dpi.length() - 1 )
        {
           dpiX = (REAL)std::stoi( dpi.substr( 0, pos ) );
           dpiY = (REAL)std::stoi( dpi.substr( ++pos ) );
        }

        if ( dpiX < 1 || dpiY < 1 )
        {
            fprintf( stderr, "WARNING: Requested dpi value (" );
            fprintf( stderr, argv[arg] );
            fprintf( stderr, ") is invalid.  Ignoring.\n" );
        }
      }
      else
      {
         if ( argv[arg][0] == '-' ) 
         {
            fprintf( stderr, "WARNING: Unknown option (" );
            fprintf( stderr, argv[arg] );
            fprintf( stderr, ").  Ignoring.\n" );
         }
         else
         {
            break;
         }
      }
   }

   // Open input file and store in a bitmap
   if ( toWchar( argv[arg], &wcsInFile ) == 0 )
   {
      printf ( "ERROR: problem reading input file %s.\n", argv[arg] );
      std_exit( 51 );
   }

   loadImage( dpiX, dpiY );

   if ( !image )
   {
      printf ( "ERROR: problem reading input file %s.\n", argv[arg] );
      std_exit( 51 );
   }

   if ( outputType == OT_RAW )
   {
      unsigned int wMax, w;
      unsigned int hMax, h;

      // Open output file to write.
      if ( ( eNo = fopen_s( &oFile, argv[arg + 1], "wb" ) ) )
      {
         printf ( "ERROR: output file %s could not be created (%d).\n", argv[arg + 1], eNo );
         std_exit( 52 );
      }

      // Get image info for debug printing
      wMax = image->GetWidth( );
      hMax = image->GetHeight( );
      sprintf_s( msg, "The width of the image is %u.\n", wMax );
      print_debug( msg );
      sprintf_s( msg, "The height of the image is %u.\n", hMax ); 
      print_debug( msg );

      ARGB  *pix = new ARGB[wMax];
      for ( h = 0; h < hMax; h++ )
      {
         for ( w = 0; w < wMax; w++ )
         {
            Color color;
            image->GetPixel( w, h, &color );
            pix[w] = color.GetValue( );
         }
         fwrite( pix, sizeof( ARGB ), wMax, oFile );
      }
      delete pix;
   }
   else
   {
      // Save image to external bitmap file
      if ( toWchar( argv[arg + 1], &wcsOutFile ) == 0 )
      {
         printf ( "ERROR: problem writing output file %s.\n", argv[arg + 1] );
         std_exit( 52 );
      }

      CLSID imgClsid;

      GetEncoderClsid( EncoderType, &imgClsid );

      if ( image->Save( wcsOutFile, &imgClsid, NULL ) != Ok )
      {
         printf ( "ERROR: problem writing output file %s.\n", argv[arg + 1] );
         std_exit( 52 );
      }
   }
   std_exit( 0 );
}

